module HAppS.MACID.Util
    (-- * Generic utilities
     addSideEffect, backgroundIOandRetry, persistentEventChannel,
     -- * Random numbers
     getRandom, getRandomR,
     -- * Modifying mutable references
     ModRef(..), modifyRefPure_
    ) where

import Control.Concurrent
import Control.Concurrent.STM
import System.IO.Unsafe(unsafeInterleaveIO)
import System.Random

import HAppS.MACID.Monad
import HAppS.MACID.Types
import HAppS.MACID.Var

-- Random numbers

-- | Get a random number.
getRandom :: Random a => AnyEv a
getRandom = do r <- sel evRandoms
               g <- readRef r
               let (x,g') = random g
               writeRef r g'
               return x

-- | Get a random number inside the range.
getRandomR :: Random a => (a,a) -> AnyEv a
getRandomR z = do r <- sel evRandoms
                  g <- readRef r
                  let (x,g') = randomR z g
                  writeRef r g'
                  return x

-- Modifiable references

class Ref ref => ModRef ref where
    modifyRef_ :: ref t -> (t -> Ev st ev t) -> Ev st ev ()
    modifyRef_ ref fun = modifyRef ref (\t -> Ev $ \e -> do x <- unEv (fun t) e;  return (x,()))
    modifyRef  :: ref t -> (t -> Ev st ev (t,a)) -> Ev st ev a


modifyRefPure_ :: ModRef ref => ref t -> (t -> t) -> AnyEv ()
modifyRefPure_ ref fun = modifyRef_ ref (return . fun)

instance ModRef MutVar where
    modifyRef (MV r) x = do a     <- unsafeSTMToEv $ readTVar r
                            (b,c) <- x a
                            unsafeSTMToEv $ writeTVar r b
                            return c


-- Special Actions in the Ev monad

-- | Run the IO action in the background and retry the current computation.
backgroundIOandRetry :: IO () -> AnyEv a
backgroundIOandRetry io = do
  done <- sel evBackgroundIOCompletion
  unsafeIOToEv $ forkIO (io >> done)
  unsafeSTMToEv retry

-- | Add a side-effect that is run once after the transactions has succesfully completed.
addSideEffect :: Seconds -> IO () -> AnyEv ()
addSideEffect t c = do var <- sel evSideEffects
                       modifyRefPure_ var ((t,c):)


-- Persistent event channel. This is racy. I think this has bug
-- with the GHC implementation. Revisit before using.
persistentEventChannel :: (a -> b) -> TChan a -> Ev st ev (IO [b])
persistentEventChannel fun raw = return $ unsafeInterleaveIO work
    where work     = do atomically (dupTChan raw) >>= loop
          loop chs = do s  <- atomically $ readTChan chs
                        ss <- unsafeInterleaveIO $ loop chs
                        return (fun s:ss)
