module Main where	-- HatStack main program

import LowLevel           (openHatFile,FileNode(..),nil,getParentNode
                          ,getErrorLoc,getErrorMessage
                          ,getSrcRef)
import SrcRef		  (SrcRef(..),readSrcRef)
import SExp               (SExp(..),Label,fileNode2SExp,sExp2Doc,prune)
import PrettyLibHighlight (Doc,pretty,nest,text,(<>),parens)
import HighlightStyle     (getTerminalSize)
import System             (getArgs,getProgName,exitWith,ExitCode(..))
import FFIExtensions      (withCString)
import IO                 (hPutStrLn,stderr)
import List               (isSuffixOf)
import Monad              (when)

main = do
    args    <- System.getArgs
    prog    <- System.getProgName
    hatfile <- case args of (f:_) -> return (rectify f)
                            _     -> do hPutStrLn stderr
                                                  ("hat-stack: no trace file")
                                        exitWith (ExitFailure 1)
    withCString prog (\p-> withCString hatfile (openHatFile p))
    errloc <- getErrorLoc
    errmsg <- getErrorMessage
    when (errloc==nil)
         (do hPutStrLn stderr ("Tracefile \""++hatfile
                               ++"\" contains no reference to a program error.")
             exitWith (ExitFailure 1))
    putStrLn ("Program terminated with error:\n\t"++errmsg)
    putStrLn ("Virtual stack trace:")
    (width,lines) <- getTerminalSize
    (mapM_ (putStrLn . paint width . toSExp)
      . takeWhile (/=nil)
      . iterate getParentNode)
      errloc

rectify :: FilePath -> FilePath
rectify f | ".hat" `isSuffixOf` f = f
          | otherwise = f ++ ".hat"

toSExp :: FileNode -> (SExp Label, Maybe (String, Int))
toSExp node = let srcref = getSrcRef node
                  sr = readSrcRef srcref in
              ( fileNode2SExp 10 False True True ("l",node)
              , if (srcref==nil) then Nothing
                else Just (SrcRef.filename sr, SrcRef.line sr)
              )

paint :: Int -> (SExp Label, Maybe (String, Int)) -> String
paint width (sexp, srcpos) =
    let doc = sExp2Doc False True False (\_->id) (prune 10 sexp) in
        pretty width
               (parens (maybe (text "unknown")
                              (\(mod,line)-> text mod <> text ":"
                                             <> text (show line))
                              srcpos)
               <> text "\t" <> nest 3 doc)

