/********************************************************************
 *                                                                  *
 * THIS FILE IS PART OF THE OGG VORBIS PROJECT SOURCE CODE.         *
 *                                                                  *
 * THE OGG VORBIS PROJECT SOURCE CODE IS (C) COPYRIGHT 1994-2001    *
 * by the XIPHOPHORUS Company http://www.xiph.org/                  *

 ********************************************************************

 function: implementation of the file format plugin for RealSystem

********************************************************************/
/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: oggfformat.cpp,v 1.3.2.5 2004/07/09 01:58:26 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#define INITGUID

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ogg/ogg.h>

#include "hxtypes.h"
#include "hxcom.h"
#include "hxcomm.h"
#include "ihxpckts.h"
#include "hxplugn.h"
#include "hxfiles.h"
#include "hxformt.h"
#include "chxpckts.h" // CHXHeader::mergeHeaders

#include "oggfformat.h"

#include "debug.h" // DPRINTF()
#define D_OGG_FF 0 //0x4000000

// static variables
const char *COggFileFormat::zm_pDescription = DESCRIPTION;
const char *COggFileFormat::zm_pCopyright = COPYRIGHT;
const char *COggFileFormat::zm_pMoreInfoURL = MORE_INFO_URL;
const char *COggFileFormat::zm_pFileMimeTypes[] = FILE_MIME_TYPES;
const char *COggFileFormat::zm_pFileExtensions[] = FILE_EXTENSIONS;
const char *COggFileFormat::zm_pFileOpenNames[] = FILE_OPEN_NAMES;

const ULONG32 EndSearchSize = 4096;
const ULONG32 EndSearchLimit = 32 * EndSearchSize;

int COggFileFormat::instno = 0;

STDAPI RMACreateInstance(IUnknown **ppExFileFormatObj)
{
#ifdef _DEBUG
    debug_level() |= D_OGG_FF;
#endif /* _DEBUG */

    DPRINTF(D_OGG_FF, ("RMACreateInstance() called\n"));

    *ppExFileFormatObj = (IUnknown *)(IHXPlugin *)new COggFileFormat();
    if (*ppExFileFormatObj != NULL) {
        (*ppExFileFormatObj)->AddRef();
        return HXR_OK;  
    }

    return HXR_OUTOFMEMORY;
}

// constructor
COggFileFormat::COggFileFormat(void)
    :   m_RefCount(0),
        m_pClassFactory(NULL),
        m_pResponse(NULL),
        m_pPageReader(NULL),
        m_State(Start),
        m_pGetPacketPending(NULL),
        m_ulLastSeekOffset(0),
        m_ulSeekTime(0),
        m_ulLowerOffset(0),
        m_ulUpperOffset(0),
        m_ulLastValidPageOff(0),
        m_pLastValidPage(NULL),
        m_pCurrentGroup(NULL),
        m_bInDispatchPendingRequest(FALSE)
{
    DPRINTF(D_OGG_FF, ("Entering COggFileFormat()\n"));
    m_instno = instno++;
    DPRINTF(D_OGG_FF, ("Exiting COggFileFormat(%d)\n", m_instno));
}

STDMETHODIMP COggFileFormat::GetPluginInfo(
    REF(BOOL) bLoadMultiple,
    REF(const char *) pDescription,
    REF(const char *) pCopyright,
    REF(const char *) pMoreInfoURL,
    REF(UINT32) versionNumber
    )
{
    DPRINTF(D_OGG_FF, ("Entering GetPluginInfo(%d)\n", m_instno));

    // File Format plugins must be able to be multiply instantiated
    bLoadMultiple = TRUE;

    pDescription = zm_pDescription;
    pCopyright = zm_pCopyright;
    pMoreInfoURL = zm_pMoreInfoURL;
    versionNumber = PLUGIN_VERSION_NUM;

    DPRINTF(D_OGG_FF, ("Exiting GetPluginInfo(%d)\n", m_instno));

    return HXR_OK;
}

STDMETHODIMP COggFileFormat::GetFileFormatInfo(
    REF(const char **) pFileMimeTypes,
    REF(const char **) pFileExtensions,
    REF(const char **) pFileOpenNames
    )
{
    DPRINTF(D_OGG_FF, ("Entering GetFileFormatInfo(%d)\n", m_instno));

    pFileMimeTypes = zm_pFileMimeTypes;
    pFileExtensions = zm_pFileExtensions;
    pFileOpenNames = zm_pFileOpenNames;

    DPRINTF(D_OGG_FF, ("Exiting GetFileFormatInfo(%d)\n", m_instno));

    return HXR_OK;
}

STDMETHODIMP COggFileFormat::InitPlugin(IUnknown* pContext)
{
    DPRINTF(D_OGG_FF, ("Entering InitPlugin(%d)\n", m_instno));

    HX_RESULT res = HXR_FAILED;

    if (pContext)
    {
        // store a reference to the IHXCommonClassFactory interface
        res = pContext->QueryInterface(IID_IHXCommonClassFactory, 
                                       (void **)&m_pClassFactory);
    }

    DPRINTF(D_OGG_FF, ("Exiting InitPlugin(%d)\n", m_instno));

    return res;
}

STDMETHODIMP COggFileFormat::InitFileFormat(
    IHXRequest *pRequest,
    IHXFormatResponse *pFormatResponse,
    IHXFileObject *pFileObject
    )
{
    HX_RESULT res = HXR_FAILED;

    if (pRequest && pFormatResponse && pFileObject)
    {
        DPRINTF(D_OGG_FF, ("Entering InitFileFormat(%d)\n", m_instno));

        // the format response object is used to notify rma core of our status
        m_pResponse = pFormatResponse;
        HX_ADDREF(m_pResponse);

        HX_RELEASE(m_pPageReader);
        m_pPageReader = new COggPageReader();

        if (m_pPageReader)
        {
            ChangeState(InitPending);

            m_pPageReader->AddRef();

            res = m_pPageReader->Init(this, pFileObject);

            if (HXR_OK != res)
            {
                ChangeState(Start);
            }
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
    } 

    DPRINTF(D_OGG_FF, ("Exiting InitFileFormat(%d)\n", m_instno));

    return HXR_OK;
}

STDMETHODIMP COggFileFormat::GetFileHeader(void)
{
    DPRINTF(D_OGG_FF, ("Entering GetFileHeader(%d)\n", m_instno));

    HX_RESULT res = HXR_UNEXPECTED;

    if (Ready == m_State)
    {
        ChangeState(NeedFirstStream);
        res = m_pPageReader->ReadNextPage();
    }

    DPRINTF(D_OGG_FF, ("Exiting GetFileHeader(%d)\n", m_instno));

    return res;
}

STDMETHODIMP COggFileFormat::GetStreamHeader(UINT16 streamNo)
{
    DPRINTF(D_OGG_FF, ("Entering GetStreamHeader(%d, %u)\n", 
                       m_instno, streamNo));

    HX_RESULT res = HXR_UNEXPECTED;
    if ((m_State == Ready) && m_pResponse && m_pCurrentGroup)
    {
        IHXValues* pHeader = NULL;
        
        res = m_pCurrentGroup->GetStreamHeader(streamNo, pHeader);
        
        if (HXR_OK == res) 
        {
            IHXBuffer* pTmp = NULL;

            // these properties are required

            // "StreamNumber" the stream number
            pHeader->SetPropertyULONG32("StreamNumber", streamNo);

            ULONG32 ulDuration = m_pCurrentGroup->Duration();
            if (ulDuration)
            {
                // "Duration" length in milliseconds
                pHeader->SetPropertyULONG32("Duration", ulDuration);
            }

            // Send the stream header to the core
            m_pResponse->StreamHeaderReady(HX_STATUS_OK, pHeader);

            HX_RELEASE(pHeader);
        }
    }


    DPRINTF(D_OGG_FF, ("Exiting GetStreamHeader(%d)\n", m_instno));

    return res;
}

HX_RESULT COggFileFormat::CreateFileHeader(void)
{
    HX_RESULT res = HXR_FAILED;

    DPRINTF(D_OGG_FF, ("Entering CreateFileHeaderObj(%d)\n", m_instno));

    if (m_pClassFactory && m_pResponse && m_pCurrentGroup)
    {
        ChangeState(Ready);

        UINT16 ulStreamCount = m_pCurrentGroup->StreamCount();

        if (ulStreamCount > 0)
        {
            IHXValues *pHeaderObj = NULL;

            // Allocate GetPacket status array

            HX_VECTOR_DELETE(m_pGetPacketPending);      
            m_pGetPacketPending = new BOOL[ulStreamCount];

            if (m_pGetPacketPending)
            {
                // Clear all the booleans
                memset(m_pGetPacketPending, 0, sizeof(BOOL) * ulStreamCount);

                res = m_pClassFactory->CreateInstance(CLSID_IHXValues, 
                                                      (void **)&pHeaderObj);
            }
            else
            {
                res = HXR_OUTOFMEMORY;
            }

            if (HXR_OK == res)
            {
                pHeaderObj->SetPropertyULONG32("StreamCount", ulStreamCount);

                for (UINT16 i = 0; i < ulStreamCount; i++)
                {
                    IHXValues* pTAC = NULL;

                    if (HXR_OK == m_pCurrentGroup->GetTACInfo(i, pTAC))
                    {
                        CHXHeader::mergeHeaders(pHeaderObj, pTAC);
                        HX_RELEASE(pTAC);
                    }
                }
                
                if (HXR_OK == res)
                {
                    // "LiveStream" stream is an http or otherwise "live" 
                    // stream if it is not seekable
                    if (!m_pPageReader->IsSeekable())
                    {
                        pHeaderObj->SetPropertyULONG32("LiveStream", 1);
                    }

                    m_pResponse->FileHeaderReady(HX_STATUS_OK, pHeaderObj);
                }
            }
            HX_RELEASE(pHeaderObj);
        }
        else
        {
            res = HXR_UNEXPECTED;
        }
    }

    DPRINTF(D_OGG_FF, ("Exiting CreateFileHeaderObj(%d) : %08x\n", 
                       m_instno, res));

    return res;
}

STDMETHODIMP COggFileFormat::GetPacket(UINT16 streamNo)
{
    DPRINTF(D_OGG_FF, ("Entering GetPacket(%d, %u)\n", m_instno, streamNo));
    HX_RESULT res = HXR_UNEXPECTED;

    if (m_pPageReader && m_pGetPacketPending && m_pCurrentGroup)
    {   
        if (m_pCurrentGroup->EndOfStream(streamNo))
        {
            DPRINTF(D_OGG_FF, 
                    ("COggFileFormat::GetPacket() : calling StreamDone(%u)\n", 
                     streamNo));

            m_pResponse->StreamDone(streamNo);
        }
        else
        {
            m_pGetPacketPending[streamNo] = TRUE;

            res = DispatchPendingRequests();

            if (HXR_OK != res)
            {
                if (m_State == Ready)
                {
                    // Request the next page
                    ChangeState(GetPacketReadPending);
                    res = m_pPageReader->ReadNextPage();
                }
                else if ((FindSeekPointPending == m_State) ||
                         (GetPacketReadPending == m_State))
                {
                    // A file operation is already pending.
                    // This request will get serviced when the 
                    // pending operation completes.
                    res = HXR_OK;
                }
            }
        }
    }

    DPRINTF(D_OGG_FF, ("Exiting GetPacket(%d)\n", m_instno));

    return res;
}

STDMETHODIMP COggFileFormat::Seek(UINT32 requestedTime)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::Seek(%u)\n", requestedTime));

    HX_RESULT res = HXR_FAILED;

    if (m_pPageReader && m_pPageReader->IsSeekable() && m_pCurrentGroup)
    {
        res = m_pCurrentGroup->OnSeek(requestedTime);

        if (HXR_OK == res)
        {
            m_ulSeekTime = requestedTime;
            m_ulLowerOffset = m_pCurrentGroup->StartOffset();
            m_ulLowerPageSize = m_pCurrentGroup->StartPageSize();
            m_ulUpperOffset = m_pCurrentGroup->EndOffset();
            
            ChangeState(FindSeekPointPending);
            res = SeekToTheMiddle();

            if (HXR_OK != res)
            {
                ChangeState(Ready);
            }
        }
    }

    DPRINTF(D_OGG_FF, ("COggFileFormat::Seek() : done %08x\n", res));
    return res;
}

STDMETHODIMP COggFileFormat::Close(void)
{
    DPRINTF(D_OGG_FF, ("Entering Close(%d)\n", m_instno));

    DestroyLastValidPage();

    HX_RELEASE(m_pClassFactory);
    HX_RELEASE(m_pResponse);

    if (m_pPageReader)
    {
        m_pPageReader->Close();
        HX_RELEASE(m_pPageReader);
    }

    HX_VECTOR_DELETE(m_pGetPacketPending);
    HX_DELETE(m_pCurrentGroup);

    DPRINTF(D_OGG_FF, ("Exiting Close(%d)\n", m_instno));

    return HXR_OK;
}

COggFileFormat::~COggFileFormat(void)
{
    DPRINTF(D_OGG_FF, ("Entering ~COggFileFormat(%d)\n", m_instno));
    Close();
    DPRINTF(D_OGG_FF, ("Exiting ~COggFileFormat(%d)\n", m_instno));
}

STDMETHODIMP_(UINT32) COggFileFormat::AddRef(void)
{
    return InterlockedIncrement(&m_RefCount);
}

STDMETHODIMP_(UINT32) COggFileFormat::Release(void)
{
    if (InterlockedDecrement(&m_RefCount) > 0)
        return m_RefCount;

    delete this;
    return 0;
}

STDMETHODIMP COggFileFormat::QueryInterface(REFIID interfaceID, 
                                            void **ppInterfaceObj)
{
    if (IsEqualIID(interfaceID, IID_IUnknown)) {
        AddRef();
        *ppInterfaceObj = (IUnknown *)(IHXPlugin *)this;
        return HXR_OK;
    } else if (IsEqualIID(interfaceID, IID_IHXPlugin)) {
        AddRef();
        *ppInterfaceObj = (IHXPlugin *)this;
        return HXR_OK;
    } else if (IsEqualIID(interfaceID, IID_IHXFileFormatObject)) {
        AddRef();
        *ppInterfaceObj = (IHXFileFormatObject *)this;
        return HXR_OK;
    }

    *ppInterfaceObj = NULL;
    return HXR_NOINTERFACE;
}

STDMETHODIMP COggFileFormat::PageReaderInitDone(THIS_ HX_RESULT status)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::PageReaderInitDone(%08x)\n", status));

    HX_RESULT res = HXR_UNEXPECTED;

    if (m_State == InitPending)
    {
        ChangeState((HXR_OK == status) ? Ready : Start);

        m_pResponse->InitDone(status);
    }

    DPRINTF(D_OGG_FF, ("COggFileFormat::PageReaderInitDone() exiting\n"));

    return HXR_OK;
}

STDMETHODIMP COggFileFormat::ReadNextPageDone(THIS_ HX_RESULT status,
                                              ULONG32 ulFileOffset,
                                              UINT32 ulPageSize,
                                              ogg_page* pPage)
{
    DPRINTF(D_OGG_FF, 
            ("COggFileFormat::ReadNextPageDone(%08x, (%u, %08x), %u)\n", 
             status, ulFileOffset, ulFileOffset, ulPageSize));

    switch(m_State) {
    case NeedFirstStream:
    case GetIdentHeaders:
    case NeedStartTime:
        if (HXR_OK == status)
        {
            status = HandleFileHeaderPage(ulFileOffset, ulPageSize, pPage);
        }
        break;
    case GFHEndSearching:
        status = HandleEndSearch(status, ulFileOffset, ulPageSize, pPage);
        break;
    case GFHSeekBeginPending:
        if (HXR_OK == status)
        {
            status = CreateFileHeader();
        }
        break;
    case GetPacketReadPending:
        if (HXR_OK == status)
        {
            status = HandlePacketPage(pPage);
        }
        else if (m_pCurrentGroup)
        {
            status = m_pCurrentGroup->OnEndOfFile();

            if (HXR_OK == status)
            {
                status = DispatchPendingRequests();

                if (HXR_OK == status)
                {
                    ChangeState(Ready);
                }
            }
        }
        break;
    case FindSeekPointPending:
        if (HXR_OK == status)
        {
            status = HandlePlaybackSeekPage(ulFileOffset, ulPageSize, pPage);
        }
        break;
    default:
        status = HXR_UNEXPECTED;
        break;
    };

    if (HXR_OK != status)
    {
        OnError(status);
    }

    DPRINTF(D_OGG_FF, ("COggFileFormat::ReadNextPageDone() : exiting\n"));

    return HXR_OK;
}

#ifdef _DEBUG
static const char* z_pStateNames[] = {
    "Start",
    "InitPending",
    "Ready",
    "NeedFirstStream",
    "GetIdentHeaders",
    "NeedStartTime",
    "GFHEndSearching",
    "GFHSeekBeginPending",
    "GetPacketReadPending",
    "FindSeekPointPending"
};
#endif /* _DEBUG */

void COggFileFormat::ChangeState(PluginState newState)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::ChangeState() : %s -> %s\n", 
                       z_pStateNames[m_State], 
                       z_pStateNames[newState]));
    m_State = newState;
}

void COggFileFormat::OnError(HX_RESULT status)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::OnError(%08x)\n", status));
    if (m_pResponse)
    {
        switch(m_State) {
        case NeedFirstStream:
        case GetIdentHeaders:
        case NeedStartTime:
        case GFHEndSearching:
        case GFHSeekBeginPending:
            ChangeState(Ready);
            m_pResponse->FileHeaderReady(status, NULL);
            break;
        case GetPacketReadPending:
        {
            ChangeState(Ready);

            for (UINT16 i = 0; i < m_pCurrentGroup->StreamCount(); i++)
            {
                if (m_pGetPacketPending[i])
                {
                    m_pGetPacketPending[i] = FALSE;
                    m_pResponse->PacketReady(status, NULL);
                }
            }
        }break;
        case FindSeekPointPending:
            ChangeState(Ready);
            m_pResponse->SeekDone(status);
            break;

        };
    }
}

HX_RESULT COggFileFormat::HandleFileHeaderPage(ULONG32 ulFileOffset,
                                               UINT32 ulPageSize,
                                               ogg_page* pPage)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::HandleFileHeaderPage()\n"));
    HX_RESULT res = HXR_OK;

    if (ogg_page_bos(pPage))
    {
        res = HandleIdentPage(ulFileOffset, ulPageSize, pPage);
    }
    else if (GetIdentHeaders == m_State)
    {
        ChangeState(NeedStartTime);
    }

    if (HXR_OK == res)
    {
        if (!m_pCurrentGroup)
        {
            res = HXR_UNEXPECTED;
        }
        else 
        {
            res = m_pCurrentGroup->OnPage(pPage);
            
            if (HXR_NO_DATA == res)
            {
                // We still need more data for the stream
                // associated with pPage.
                res = m_pPageReader->ReadNextPage();
            }
            else if (HXR_OK == res)
            {
                if ((NeedStartTime == m_State) &&
                    m_pCurrentGroup->HaveAllStartTimes())
                {
                    res = HaveFileHeaders(ulFileOffset);
                }
                else
                {
                    // We still need more data for at least
                    // 1 of the streams
                    res = m_pPageReader->ReadNextPage();
                }
            }
        }
    }

    return res;
}

HX_RESULT COggFileFormat::HandleIdentPage(ULONG32 ulFileOffset,
                                          UINT32 ulPageSize,
                                          ogg_page*  pPage)
{
    HX_RESULT res = HXR_FAILED;

    switch(m_State) {
    case NeedFirstStream:
        HX_DELETE(m_pCurrentGroup);

        m_pCurrentGroup = new COggStreamGroup();
        
        if (m_pCurrentGroup)
        {
            res = m_pCurrentGroup->Init(m_pClassFactory);
            
            if (HXR_OK == res)
            {
                res = m_pCurrentGroup->AddStream(ulFileOffset, ulPageSize,
                                                 pPage);

                if (HXR_OK == res)
                {
                    ChangeState(GetIdentHeaders);
                }
            }
        }
        else
        {
            res = HXR_OUTOFMEMORY;
        }
        break;

    case GetIdentHeaders:
        res = m_pCurrentGroup->AddStream(ulFileOffset, ulPageSize, pPage);
        break;
    default:
        res = HXR_INVALID_FILE;
        break;
    };
    
    return res;
}

HX_RESULT COggFileFormat::HaveFileHeaders(ULONG32 ulFileOffset)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::HaveFileHeaders(%u)\n", ulFileOffset));
    HX_RESULT res = HXR_FAILED;

    if (m_pPageReader->IsSeekable())
    {
        /* Start looking for the last page in the file so
         * we can determine the clip duration.
         */
        ChangeState(GFHEndSearching);
        m_ulLowerOffset = ulFileOffset; // Store current offset before
                                        // searching for the end

        res = DoFileSeek(m_pPageReader->FileSize() - EndSearchSize);
    }
    else
    {
        /* We can't seek so we create the file header now */
        res = CreateFileHeader();
    }

    return res;
}

HX_RESULT COggFileFormat::HandleEndSearch(HX_RESULT status, 
                                          ULONG32 ulFileOffset,
                                          UINT32 ulPageSize,
                                          ogg_page* pPage)
{
    HX_RESULT res = HXR_FAILED;

    if (HXR_OK == status)
    {
        /* We found a page */
        if ((ulFileOffset + ulPageSize) == m_pPageReader->FileSize())
        {
            /* This is the last page.*/
            res = FoundLastPage(ulFileOffset, pPage);
        }
        else
        {
            /* This isn't the last page. Cache this page and read 
             * the next one. */
            UpdateLastValidPage(ulFileOffset, pPage);

            res = m_pPageReader->ReadNextPage();
        }
    }
    else if (m_pLastValidPage)
    {
        /* We've reached the end of the file, but we have
         * seen a page. This can happen with truncated files.
         * Use the last valid page we've seen as the last page of
         * the file.
         */
        res = FoundLastPage(m_ulLastValidPageOff, m_pLastValidPage);
    }
    else
    {
        /* We didn't find a page */

        /* Compute a new offset */
        ULONG32 ulNewSeekOffset = m_ulLastSeekOffset - EndSearchSize;

        if ((ulNewSeekOffset + EndSearchLimit) > m_pPageReader->FileSize())
        {
            /* Seek back a little farther and try again */
            res = DoFileSeek(ulNewSeekOffset);
        }
        else
        {
            /* We've gone too far from the end of the file. We should
             * have found a page by now
             */
            res = HXR_INVALID_FILE;
        }
    }

    return res;
}

HX_RESULT COggFileFormat::HandlePacketPage(ogg_page* pPage)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::HandlePacketPage()\n"));

    HX_RESULT res = HXR_FAILED;
    
    if (m_pCurrentGroup && m_pGetPacketPending)
    {
        BOOL bNeedMorePages = FALSE;

        res = m_pCurrentGroup->OnPage(pPage);

        if (HXR_OK == res)
        {
            if (HXR_OK != DispatchPendingRequests())
            {
                bNeedMorePages = TRUE;
            }
        }
        else if (HXR_NO_DATA == res)
        {
            bNeedMorePages = TRUE;
        }

        if (bNeedMorePages)
        {
            res = m_pPageReader->ReadNextPage();
        }
        else
        {
            ChangeState(Ready);
        }
    }

    return res;
}

HX_RESULT COggFileFormat::DispatchPendingRequests()
{
    HX_RESULT res = HXR_OK;

    DPRINTF(D_OGG_FF, 
            ("COggFileFormat::DispatchPendingRequests()\n"));
    
    if (m_pCurrentGroup && m_pResponse && m_pGetPacketPending)
    {
        if (!m_bInDispatchPendingRequest)
        {
            BOOL bDone = FALSE;

            m_bInDispatchPendingRequest = TRUE;

            while (!bDone)
            {
                UINT32 ulNextStreamID;

                // Assume this will be the last loop
                bDone = TRUE;

                res = m_pCurrentGroup->NextPacketStreamID(ulNextStreamID);

                if ((HXR_OK == res) && m_pGetPacketPending[ulNextStreamID])
                {
                    // We got the streamID for the next packet
                    // and we have a GetPacket() request pending.

                    IHXPacket* pPkt = NULL;
                    
                    // Get the next packet
                    res = m_pCurrentGroup->GetNextPacket(pPkt);
                    
                    if (HXR_OK == res)
                    {
                        m_pGetPacketPending[ulNextStreamID] = FALSE;
                        
                        DPRINTF(D_OGG_FF, 
                                ("COggFileFormat::DispatchPendingRequests() : PacketReady(%u, %u)\n",
                                 pPkt->GetStreamNumber(),
                                 pPkt->GetTime()));
                        m_pResponse->PacketReady(HXR_OK, pPkt);
                        HX_RELEASE(pPkt);
                        
                        // We dispatched a packet so
                        // we should loop again
                        bDone = FALSE;
                    }
                }
            }
        
            if (HXR_STREAM_DONE == res)
            {
                UINT16 ulStreamCount = m_pCurrentGroup->StreamCount();

                // All the streams are done.
                // Handle any pending GetPacket() requests
                for (UINT16 i = 0; i < ulStreamCount; i++)
                {
                    if (m_pGetPacketPending[i])
                    {
                        m_pGetPacketPending[i] = FALSE;

                        DPRINTF(D_OGG_FF, 
                                ("COggFileFormat::DispatchPendingRequests() : StreamDone(%u)\n",
                                 i));
                        m_pResponse->StreamDone(i);
                    }
                }
                
                res = HXR_OK;
            }

            m_bInDispatchPendingRequest = FALSE;
        }
        else
        {
            // We are already in a DispatchPendingRequests() call
            // so we don't want to do anything here
            DPRINTF(D_OGG_FF, 
                    ("COggFileFormat::DispatchPendingRequests() : deferred\n", res));
        }
    }
    else
    {
        res = HXR_FAILED;
    }

    DPRINTF(D_OGG_FF, 
            ("COggFileFormat::DispatchPendingRequests() : res %08x\n", res));
    return res;
}

HX_RESULT COggFileFormat::DoFileSeek(ULONG32 ulFileOffset)
{
    HX_RESULT res = HXR_FAILED;

    if (m_pPageReader)
    {
        m_ulLastSeekOffset = ulFileOffset;
        res = m_pPageReader->Seek(ulFileOffset);
    }

    return res;
}

HX_RESULT COggFileFormat::SeekToTheMiddle()
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::SeekToTheMiddle()\n"));

    HX_RESULT res = HXR_FAILED;

    if (m_ulUpperOffset > m_ulLowerOffset)
    {
        ULONG32 ulDelta = m_ulUpperOffset - m_ulLowerOffset;
        ULONG32 ulSeekOffset = m_ulLowerOffset + ulDelta / 2;

        DPRINTF(D_OGG_FF, 
                ("COggFileFormat::SeekToTheMiddle() : ulSeekOffset %u\n",
                 ulSeekOffset));

        res = DoFileSeek(ulSeekOffset);
    }

    DPRINTF(D_OGG_FF, ("COggFileFormat::SeekToTheMiddle() : done %08x\n", 
                       res));
    return res;
}

HX_RESULT COggFileFormat::HandlePlaybackSeekPage(ULONG32 ulFileOffset,
                                                 UINT32 ulPageSize, 
                                                 ogg_page* pPage)
{
    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP(%u, %u)\n",
                       ulFileOffset, ulPageSize));

    HX_RESULT res = HXR_FAILED;

    if (pPage && m_pCurrentGroup)
    {
        UINT32 ulTimestamp = 0;
        res = m_pCurrentGroup->GetTimestamp(pPage, ulTimestamp);

        if (HXR_OK == res)
        {
            DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : Timestamp %u\n", 
                               ulTimestamp));

            if (ulFileOffset >= m_ulUpperOffset)
            {
                // The last seek offset resulted in the same
                // upper page. Update the upper offset and
                // seek to the middle again,
                m_ulUpperOffset = m_ulLastSeekOffset;

                DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : Updating upper offset 1\n", res));

                res = SeekToTheMiddle();
            }
            else if ((ulFileOffset + ulPageSize) >= m_ulUpperOffset)
            {
                // The current page is the one before the upper page
                BOOL bSeekPointInNextPage = FALSE;
                BOOL bSeekPointInThisPage = FALSE;

                ULONG32 ulSecondPageOffset = 
                    (m_ulLowerOffset + m_ulLowerPageSize);

                if (ulTimestamp < m_ulSeekTime)
                {
                    // We've found a page that is lower
                    // than the seek time. We know that
                    // the next page is >= the seek time.
                    bSeekPointInNextPage = TRUE;
                }
                else if (ulFileOffset == m_ulLowerOffset)
                {
                    // We are at lower offset page
                    if (ulTimestamp < m_ulSeekTime)
                    {
                        bSeekPointInNextPage = TRUE;
                    }
                    else
                    {
                        bSeekPointInThisPage = TRUE;
                    }
                }
                else if (ulFileOffset == ulSecondPageOffset)
                {
                    // We are at the page after the lower offset
                    // page and it's timestamp >= the seek time.
                    // Seek to the lower bound page
                    
                    m_ulUpperOffset = ulSecondPageOffset;
                    res = DoFileSeek(m_ulLowerOffset);
                }
                else
                {
                    // There must still be pages between the lower
                    // offset and this page. 
                    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : Updating upper offset 2\n", res));
                    m_ulUpperOffset = ulFileOffset;
                    
                    res = SeekToTheMiddle();
                }

                if (bSeekPointInNextPage)
                {
                    // The seek point is in the next page

                    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : seek point is in the next page\n"));
                    ChangeState(Ready);

                    m_pResponse->SeekDone(HXR_OK);

                    res = HXR_OK;
                }
                else if (bSeekPointInThisPage)
                {
                    // The seek point is in this page
                    
                    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : seek point is in this page\n"));
                    res = m_pCurrentGroup->OnPage(pPage);
                    
                    if ((HXR_OK == res) || (HXR_NO_DATA == res))
                    {
                        ChangeState(Ready);
                        m_pResponse->SeekDone(HXR_OK);

                        res = HXR_OK;
                    }
                }
            }
            else 
            {
                if (ulTimestamp < m_ulSeekTime)
                {
                    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : Updating lower offset\n", res));
                    m_ulLowerOffset = ulFileOffset;
                    m_ulLowerPageSize = ulPageSize;
                }
                else
                {
                    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : Updating upper offset 3\n", res));
                    m_ulUpperOffset = ulFileOffset;
                }
                res = SeekToTheMiddle();
            }
        }
    }

    DPRINTF(D_OGG_FF, ("COggFileFormat::HPSP() : done %08x\n", res));

    return res;
}

HX_RESULT COggFileFormat::FoundLastPage(ULONG32 ulFileOffset, ogg_page* pPage)
{
    HX_RESULT res = HXR_FAILED;

    if (!m_pCurrentGroup->SerialInGroup(ogg_page_serialno(pPage)))
    {
        /* This is a chained file. Chained files are not
         * supported yet.
         */
        res = HXR_INVALID_FILE;
    }
    else
    {
        /* This is the last page of the group*/
        
        m_pCurrentGroup->SetEndPage(ulFileOffset, pPage);
        
        /* Seek back to where we were before we started
         * seeking for the end
         */
        ChangeState(GFHSeekBeginPending);
        res = DoFileSeek(m_ulLowerOffset);
    }

    return res;
}

HX_RESULT COggFileFormat::UpdateLastValidPage(ULONG32 ulFileOffset, 
                                              ogg_page* pPage)
{
    HX_RESULT res = HXR_OK;

    if (pPage)
    {
        DestroyLastValidPage();

        m_ulLastValidPageOff = ulFileOffset;
        m_pLastValidPage = new ogg_page;

        if (m_pLastValidPage)
        {
            *m_pLastValidPage = *pPage;

            m_pLastValidPage->header = new unsigned char[pPage->header_len];
            m_pLastValidPage->body = new unsigned char[pPage->body_len];

            if (m_pLastValidPage->header && m_pLastValidPage->body)
            {
                memcpy(m_pLastValidPage->header, pPage->header, 
                       pPage->header_len);
                memcpy(m_pLastValidPage->body, pPage->body, 
                       pPage->body_len);
            }
            else
            {
                delete [] m_pLastValidPage->header;
                delete [] m_pLastValidPage->body;
                delete m_pLastValidPage;
                m_pLastValidPage = NULL;
            }
        }
    }

    return res;
}

void COggFileFormat::DestroyLastValidPage()
{
    if (m_pLastValidPage)
    {
        delete [] m_pLastValidPage->header;
        delete [] m_pLastValidPage->body;
        delete m_pLastValidPage;
        m_pLastValidPage = NULL;
    }
}
