/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: strm_group.cpp,v 1.2.2.3 2004/07/09 01:58:26 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
#include "strm_group.h"

#include "vorbis_page2pkt.h"
#include "theora_page2pkt.h"

#include "debug.h" // DPRINTF()
#define D_OGG_STRGRP 0 //0x4000000

const COggStreamGroup::BuildFunc COggStreamGroup::zm_buildFuncs[] = {
    &VorbisPageToPacket::Build,
    &TheoraPageToPacket::Build,
};

COggStream::COggStream(UINT32 ulSerialNum, COggPageToPacket* pPage2Pkt) :
    m_ulSerialNum(ulSerialNum),
    m_pPage2Pkt(pPage2Pkt)
{
#ifdef _DEBUG
    debug_level() |= D_OGG_STRGRP;
#endif

    DPRINTF(D_OGG_STRGRP, ("COggStream::COggStream(%u)\n", ulSerialNum));
}

COggStream::~COggStream()
{
    DPRINTF(D_OGG_STRGRP, ("COggStream::~COggStream() : serial %u\n", 
                           m_ulSerialNum));

    HX_DELETE(m_pPage2Pkt);
}

UINT32 COggStream::SerialNum() const
{
    return m_ulSerialNum;
}

COggPageToPacket* COggStream::GetPage2Packet() const
{
    return m_pPage2Pkt;
}

COggStreamGroup::COggStreamGroup() :
    m_pContext(NULL),
    m_uStreamCount(0),
    m_ppStreams(NULL),
    m_ulStartOffset(0),
    m_ulEndOffset(0),
    m_bHaveAllStartTimes(FALSE)
{
#ifdef _DEBUG
    debug_level() |= D_OGG_STRGRP;
#endif
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::COggStreamGroup()\n"));
}

COggStreamGroup::~COggStreamGroup()
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::~COggStreamGroup()\n"));

    HX_RELEASE(m_pContext);

    for (UINT16 i = 0; i < m_uStreamCount; i++)
    {
        HX_DELETE(m_ppStreams[i]);
    }
    HX_VECTOR_DELETE(m_ppStreams);
}

HX_RESULT COggStreamGroup::Init(IUnknown* pContext)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::Init()\n"));

    HX_RESULT res = HXR_FAILED;
    
    if (pContext)
    {
        HX_RELEASE(m_pContext);

        m_pContext = pContext;
        m_pContext->AddRef();
        res = HXR_OK;
    }

    return res;
}

HX_RESULT COggStreamGroup::AddStream(ULONG32 ulFileOffset, 
                                     UINT32 uPageSize,
                                     ogg_page* pPage)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::AddStream(%lu, %u)\n", 
                           ulFileOffset, uPageSize));

    HX_RESULT res = HXR_FAILED;

    ogg_stream_state os;
    ogg_packet op;
    ogg_stream_init(&os, 0);

    if (pPage && m_pContext && 
        !SerialInGroup(ogg_page_serialno(pPage)) &&
        (HXR_OK == (res = GetIdentPacket(&os, pPage, &op))))
    {
        COggPageToPacket* pPage2Pkt = NULL;
        
        // Try all of the BuildFunc's to see if one will
        // build a COggPageToPacket object for this ident packet
        UINT32 ulBuildFuncCount = sizeof(zm_buildFuncs) / sizeof(BuildFunc);
        for (UINT32 i = 0; !pPage2Pkt && (i < ulBuildFuncCount); i++)
        {
            pPage2Pkt = zm_buildFuncs[i](&op);
        }

        if (!pPage2Pkt)
        {
            DPRINTF(D_OGG_STRGRP, 
                    ("COggStreamGroup::AddStream() : failed to build Page2Pkt\n"));
            // We don't know how to handle this type of stream
            res = HXR_MISSING_COMPONENTS;
        }
        else if (HXR_OK == (res = pPage2Pkt->Init(m_pContext)))
        {
            // pPage2Pkt ownership is passed here
            res = AddStream(ogg_page_serialno(pPage), pPage2Pkt);

            if ((HXR_OK == res) && (m_uStreamCount == 1))
            {
                // Store the file offset if this is the first stream
                m_ulStartOffset = ulFileOffset;
                m_uStartPageSize = uPageSize;
            }
        }
    }

    ogg_stream_clear(&os);

    return res;
}

HX_RESULT COggStreamGroup::GetTACInfo(UINT32 uStream, REF(IHXValues*)pHeader)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::GetTACInfo(%u)\n", uStream));

    HX_RESULT res = HXR_FAILED;
    
    COggPageToPacket* pPage2Pkt = Page2PktByStreamID(uStream);

    if (pPage2Pkt)
    {
        res = pPage2Pkt->GetTACInfo(pHeader);
    }

    return res;
}

HX_RESULT COggStreamGroup::GetStreamHeader(UINT32 uStream, 
                                           REF(IHXValues*)pHeader)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::GetStreamHeader(%u)\n", uStream));

    HX_RESULT res = HXR_FAILED;
    
    COggPageToPacket* pPage2Pkt = Page2PktByStreamID(uStream);

    if (pPage2Pkt)
    {
        res = pPage2Pkt->GetStreamHeader(pHeader);
    }

    return res;
}

HX_RESULT COggStreamGroup::GetPacket(UINT16 uStream, REF(IHXPacket*) pPkt)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::GetPacket(%u)\n", uStream));

    HX_RESULT res = HXR_FAILED;

    COggPageToPacket* pPage2Pkt = Page2PktByStreamID(uStream);

    if (pPage2Pkt)
    {
        res = pPage2Pkt->GetPacket(pPkt);
    }

    return res;
}

HX_RESULT COggStreamGroup::NextPacketStreamID(REF(UINT32) uStreamID)
{
    HX_RESULT res = HXR_OK;

    if (!NextStreamIDValid())
    {
        res = FindNextPacket();
    }

    if (HXR_OK == res)
    {
        uStreamID = m_uNextStreamID;
    }

    return res;
}

HX_RESULT COggStreamGroup::GetNextPacket(REF(IHXPacket*) pPkt)
{
    HX_RESULT res = HXR_OK;

    if (!NextStreamIDValid())
    {
        res = FindNextPacket();
    }

    if (HXR_OK == res)
    {
        COggPageToPacket* pPage2Pkt = Page2PktByStreamID(m_uNextStreamID);

        if (pPage2Pkt)
        {
            res = pPage2Pkt->GetPacket(pPkt);
        }

        InvalidateNextStreamID();
    }

    return res;
}


HX_RESULT COggStreamGroup::OnPage(ogg_page* pPage)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::OnPage()\n"));

    HX_RESULT res = HXR_FAILED;

    COggPageToPacket* pPage2Pkt = Page2PktByPage(pPage);

    if (pPage2Pkt)
    {
        res = pPage2Pkt->OnPage(pPage);
    }

    return res;
}

HX_RESULT COggStreamGroup::OnEndOfFile()
{
    HX_RESULT res = HXR_OK;

    for (UINT16 i = 0; (HXR_OK == res) && (i < m_uStreamCount); i++)
    {
        COggPageToPacket* pPage2Pkt = Page2PktByStreamID(i);
        
        if (pPage2Pkt)
        {
            res = pPage2Pkt->OnEndOfFile();
        }
        else
        {
            res = HXR_UNEXPECTED;
        }
    }

    return res;
}

HX_RESULT COggStreamGroup::SetEndPage(ULONG32 ulFileOffset, ogg_page* pPage)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::SetEndPage(%u)\n", ulFileOffset));

    HX_RESULT res = HXR_FAILED;

    COggPageToPacket* pPage2Pkt = Page2PktByPage(pPage);

    if (pPage2Pkt)
    {
        res = pPage2Pkt->SetEndTime(ogg_page_granulepos(pPage));
        
        if ((HXR_OK == res) && (ulFileOffset > m_ulEndOffset))
        {
            m_ulEndOffset = ulFileOffset;
        }
    }

    return res;
}

HX_RESULT COggStreamGroup::OnSeek(UINT32 ulSeekPoint)
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::OnSeek(%u)\n",ulSeekPoint));

    HX_RESULT res = HXR_OK;

    InvalidateNextStreamID();

    for (UINT16 i = 0; (HXR_OK == res) && (i < m_uStreamCount); i++)
    {
        COggPageToPacket* pPage2Pkt = Page2PktByStreamID(i);
        
        if (pPage2Pkt)
        {
            res = pPage2Pkt->OnSeek(ulSeekPoint);
        }
        else
        {
            res = HXR_UNEXPECTED;
        }
    }

    return res;
}

BOOL COggStreamGroup::SerialInGroup(UINT32 ulSerialNum) const
{
    BOOL bRet = (StreamBySerial(ulSerialNum) != NULL) ? TRUE : FALSE;

    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::SerialInGroup(%u) : %d\n",
                           ulSerialNum, bRet));

    return bRet;
}

UINT16 COggStreamGroup::StreamCount() const
{
    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::StreamCount() : %u\n", 
                           m_uStreamCount));

    return m_uStreamCount;
}

UINT32 COggStreamGroup::Duration() const
{
    UINT32 ulRet = 0;

    for (UINT16 i = 0; i < m_uStreamCount; i++)
    {
        COggPageToPacket* pPage2Pkt = Page2PktByStreamID(i);

        if (pPage2Pkt)
        {
            ULONG32 ulDuration = pPage2Pkt->Duration();

            if (ulRet < ulDuration)
            {
                ulRet = ulDuration;
            }
        }
    }

    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::Duration() : %u\n", ulRet));

    return ulRet;
}

ULONG32 COggStreamGroup::StartOffset() const
{
    return m_ulStartOffset;
}

UINT32 COggStreamGroup::StartPageSize() const
{
    return m_uStartPageSize;
}

ULONG32 COggStreamGroup::EndOffset() const
{
    return m_ulEndOffset;
}

BOOL COggStreamGroup::HaveAllStartTimes()
{
    if (!m_bHaveAllStartTimes)
    {
        // Recompute m_bHaveAllStartTimes by scanning all the
        // COggPageToPacket objects.

        // Initially assume we have all the start times
        m_bHaveAllStartTimes = TRUE;

        for (UINT16 i = 0; m_bHaveAllStartTimes && (i < m_uStreamCount); i++)
        {
            COggPageToPacket* pPage2Pkt = Page2PktByStreamID(i);

            if (pPage2Pkt && !pPage2Pkt->HasStartTime())
            {
                // We've found a COggPageToPacket that doesn't have
                // a start time
                m_bHaveAllStartTimes = FALSE;
            }
        }
    }

    DPRINTF(D_OGG_STRGRP, ("COggStreamGroup::HaveAllStartTimes() : %d\n",
                           m_bHaveAllStartTimes));

    return m_bHaveAllStartTimes;
}

BOOL COggStreamGroup::EndOfStream(UINT32 uStream) const
{
    BOOL bRet = TRUE;

    COggPageToPacket* pPage2Pkt = Page2PktByStreamID(uStream);

    if (pPage2Pkt)
    {
        bRet = pPage2Pkt->EndOfStream();
    }

    return bRet;
}

HX_RESULT COggStreamGroup::GetTimestamp(ogg_page* pPage, 
                                        REF(UINT32) ulTimestamp) const
{
    HX_RESULT res = HXR_FAILED;

    COggPageToPacket* pPage2Pkt = Page2PktByPage(pPage);

    if (pPage2Pkt)
    {
        res = pPage2Pkt->GetTimestamp(ogg_page_granulepos(pPage), ulTimestamp);
    }
    
    return res;
}

COggStream* COggStreamGroup::StreamBySerial(UINT32 ulSerialNum) const
{
    COggStream* pRet = NULL;

    for (UINT16 i = 0; !pRet && (i < m_uStreamCount); i++)
    {
        if (m_ppStreams[i]->SerialNum() == ulSerialNum)
        {
            pRet = m_ppStreams[i];
        }
    }

    return pRet;
}

COggStream* COggStreamGroup::StreamByPage(ogg_page* pPage) const
{
    return StreamBySerial(ogg_page_serialno(pPage));
}

COggPageToPacket* COggStreamGroup::Page2PktByStreamID(UINT16 uStream) const
{
    COggPageToPacket* pRet = NULL;

    if ((uStream < m_uStreamCount) && (m_ppStreams[uStream]))
    {
        pRet = m_ppStreams[uStream]->GetPage2Packet();
    }

    return pRet;
}

COggPageToPacket* COggStreamGroup::Page2PktByPage(ogg_page* pPage) const
{
    COggPageToPacket* pRet = NULL;

    COggStream* pStream = StreamByPage(pPage);

    if (pStream)
    {
        pRet = pStream->GetPage2Packet();
    }

    return pRet;
}

HX_RESULT COggStreamGroup::AddStream(UINT32 ulSerialNum, 
                                     COggPageToPacket* pPage2Pkt)
{
    HX_RESULT res = HXR_OUTOFMEMORY;

    COggStream** ppNewStreams = new COggStream*[m_uStreamCount + 1];
    COggStream* pStream = new COggStream(ulSerialNum, pPage2Pkt);

    if (ppNewStreams && pStream)
    {
        // Copy existing stream pointers
        for (UINT16 i = 0; i < m_uStreamCount; i++)
        {
            ppNewStreams[i] = m_ppStreams[i];
        }
        
        // Add new stream pointer
        pPage2Pkt->SetStreamID(m_uStreamCount);
        ppNewStreams[m_uStreamCount] = pStream;
        
        // Replace m_ppStreams
        HX_VECTOR_DELETE(m_ppStreams);
        m_ppStreams = ppNewStreams;
        
        // Update stream count
        m_uStreamCount++;

        InvalidateNextStreamID();

        res = HXR_OK;
    }
    else
    {
        HX_VECTOR_DELETE(ppNewStreams);

        if (pStream)
        {
            // If the stream object was created then
            // pPage2Pkt will get destroyed with
            // pStream
            HX_DELETE(pStream);
        }
        else
        {
            // We have to destroy pPage2Pkt because
            // ownership was not passed to pStream
            HX_DELETE(pPage2Pkt);
        }
    }
        
    return res;
}

HX_RESULT COggStreamGroup::GetIdentPacket(ogg_stream_state* pOs, 
                                          ogg_page* pPage, 
                                          ogg_packet* pPkt)
{
    HX_RESULT res = HXR_FAILED;

    if (pOs && pPage && pPkt)
    {
        ogg_stream_clear(pOs);
        ogg_stream_init(pOs, ogg_page_serialno(pPage));

        if (ogg_page_bos(pPage) && // must be a start page
            (ogg_page_packets(pPage) == 1) && // must only have 1 packet
            (ogg_stream_pagein(pOs, pPage) == 0) &&
            (ogg_stream_packetout(pOs, pPkt) == 1))
        {
            res = HXR_OK;
        }
    }

    return res;
}

HX_RESULT COggStreamGroup::FindNextPacket()
{
    HX_RESULT res = HXR_OK;

    if (!NextStreamIDValid())
    {
        BOOL bTimestampSet = FALSE;
        UINT32 ulLowTS = 0;
        UINT32 ulLowStream = 0;

        for (UINT16 i = 0; (HXR_OK == res) && (i < m_uStreamCount); i++)
        {
            COggPageToPacket* pPage2Pkt = Page2PktByStreamID(i);
            
            if (pPage2Pkt)
            {
                UINT32 ulTmpTS;
                HX_RESULT tmpRes = pPage2Pkt->GetNextTimestamp(ulTmpTS);

                if (HXR_OK == tmpRes)
                {
                    if (!bTimestampSet || (ulTmpTS < ulLowTS))
                    {
                        ulLowTS = ulTmpTS;
                        ulLowStream = i;
                        bTimestampSet = TRUE;
                    }
                }
                else if (HXR_STREAM_DONE != tmpRes)
                {
                    // We want to report all errors
                    // other than HXR_STREAM_DONE
                    res = tmpRes;
                }
            }
        }

        if (HXR_OK == res)
        {
            if (bTimestampSet)
            {
                m_uNextStreamID = ulLowStream;
            }
            else
            {
                // The lowest timestamp was never
                // and we didn't get an error
                // so that must mean that all the
                // streams reported HXR_STREAM_DONE.
                res = HXR_STREAM_DONE;
            }
        }

    }

    return res;
}
