/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: vorbis_page2pkt.cpp,v 1.3.2.4 2004/07/09 01:58:26 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */
#include "vorbis_page2pkt.h"
#include "oggutil.h"

#include "debug.h" // DPRINTF()
#define D_VORBIS_P2P 0 //0x4000000

#define VORBIS_MIME_TYPE "application/ogg"

const Ogg2HelixEntry VorbisPageToPacket::zm_pOgg2HelixMap[] = {
    {"title", "Title"},
    {"artist", "Author"},
    {"album", "Abstract"},
    {"copyright", "Copyright"}  
};


COggPageToPacket* VorbisPageToPacket::Build(ogg_packet* pPacket)
{
#ifdef _DEBUG
    debug_level() |= D_VORBIS_P2P;
#endif /* _DEBUG */

    COggPageToPacket* pRet = NULL;
    
    if (IsVorbisHeader(pPacket) &&   // Is it a vorbis header?
        (pPacket->packet[0] == 0x1)) // Is it the ident header?
    {
        pRet = new VorbisPageToPacket;
    }

    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::Build() : %p\n", pRet));

    return pRet;
}

VorbisPageToPacket::VorbisPageToPacket() :
    m_bHasStreamID(FALSE),
    m_uStreamID(0),
    m_bHasStartTime(FALSE),
    m_startTime(0),
    m_endTime(0),
    m_granuleCount(0),
    m_lastBlockSize(0),
    m_ulSeekTime(0),
    m_bEndOfStream(FALSE),
    m_state(stStart),
    m_bSentHeaders(FALSE),
    m_pCCF(NULL)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::VorbisPageToPacket()\n"));

    ogg_stream_init(&m_os, 0);
    vorbis_info_init(&m_vi);
    vorbis_comment_init(&m_vc);
}

VorbisPageToPacket::~VorbisPageToPacket()
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::~VorbisPageToPacket()\n"));
    HX_RELEASE(m_pCCF);

    ogg_stream_clear(&m_os);
    vorbis_info_clear(&m_vi);
    vorbis_comment_clear(&m_vc);
    
    FlushQueues();
    FlushHeaders();
}

STDMETHODIMP VorbisPageToPacket::Init(IUnknown* pContext)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::Init()\n"));
    HX_RESULT res = HXR_FAILED;

    if (pContext)
    {
        HX_RELEASE(m_pCCF);
        res = pContext->QueryInterface(IID_IHXCommonClassFactory,
                                       (void**)&m_pCCF);

        if (HXR_OK == res)
        {
            ChangeState(stNeedIdent);
        }
    }

    return res;
}

STDMETHODIMP VorbisPageToPacket::SetStreamID(UINT16 uStream)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::SetStreamID(%u)\n", uStream));

    m_bHasStreamID = TRUE;
    m_uStreamID = uStream;

    return HXR_OK;
}


STDMETHODIMP VorbisPageToPacket::OnPage(ogg_page* pPage)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::OnPage()\n"));

    HX_RESULT res = HXR_FAILED;

    if (pPage)
    {
        if (ogg_page_bos(pPage))
        {
            ogg_stream_clear(&m_os);
            vorbis_info_clear(&m_vi);
            vorbis_comment_clear(&m_vc);

            ogg_stream_init(&m_os, ogg_page_serialno(pPage));
            vorbis_info_init(&m_vi);
            vorbis_comment_init(&m_vc);

            ChangeState(stNeedIdent);
        }

        if (ogg_page_eos(pPage))
        {
            m_bEndOfStream = TRUE;
        }

        ogg_int64_t startTime = m_granuleCount;

        DPRINTF(D_VORBIS_P2P, 
                ("VorbisPageToPacket::OnPage() : startTime %lld\n",
                 startTime));

        BOOL bHasHeader = FALSE;

        if (ogg_stream_pagein(&m_os, pPage) == 0)
        {
            ogg_packet op;
            BOOL bDone = FALSE;

            // Assume we are going to need
            // more data
            res = HXR_NO_DATA;

            while(!bDone)
            {
                int poErr = ogg_stream_packetout(&m_os, &op);

                if (poErr > 0)
                {
                    switch(m_state) {
                    case stNeedIdent:
                        if (vorbis_synthesis_headerin(&m_vi, &m_vc, &op) < 0)
                        {
                            bDone = TRUE;
                            res = HXR_INVALID_FILE;
                        }
                        else
                        {
                            bHasHeader = TRUE;
                            ChangeState(stNeedComment);
                        }
                        break;
                    case stNeedComment:
                    {
                        int err = vorbis_synthesis_headerin(&m_vi, &m_vc, &op);
                        if (err < 0)
                        {
                            bDone = TRUE;
                            res = HXR_INVALID_FILE;
                        }
                        else
                        {
                            bHasHeader = TRUE;
                            ChangeState(stNeedCodebook);
                        }
                    } break;
                    case stNeedCodebook:
                    {
                        int err = vorbis_synthesis_headerin(&m_vi, &m_vc, &op);
                        if (err < 0)
                        {
                            bDone = TRUE;
                            res = HXR_INVALID_FILE;
                        }
                        else
                        {
                            bHasHeader = TRUE;
                            m_granuleCount = 0;
                            m_lastBlockSize = 0;
                            ChangeState(stNeedStartTime);
                        }
                    } break;
                    case stNeedStartTime:
                    case stReady:
                    case stSeeking:
                    {
                        HX_ASSERT(!bHasHeader);

                        int currentSize = vorbis_packet_blocksize(&m_vi, &op);

                        if (m_lastBlockSize)
                        {
                            int outputSize = 
                                currentSize / 4 + m_lastBlockSize / 4;

                            m_granuleCount += outputSize;
                        }
                        m_lastBlockSize = currentSize;
                    }break;
                    default:
                        bDone = TRUE;
                        res = HXR_FAILED;
                        break;
                    };
                }
                else if (poErr == 0)
                {
                    bDone = TRUE;
                }
                else if (stNeedStartTime != m_state)
                {
                    // This indicates that loss has occured.
                    // Loss is only ok if we haven't received
                    // the start time yet
                    res = HXR_UNEXPECTED;
                }
            }

            if ((ogg_page_granulepos(pPage) != -1) && (m_granuleCount > 0))
            {
                ogg_int64_t endTime = ogg_page_granulepos(pPage);
                
                DPRINTF(D_VORBIS_P2P, 
                        ("VorbisPageToPacket::OnPage() : gc %lld end %lld\n",
                         m_granuleCount,
                         endTime));

                if (startTime == 0)
                {
                    startTime = endTime - m_granuleCount;

                    m_granuleCount = endTime;

                    DPRINTF(D_VORBIS_P2P, 
                            ("VorbisPageToPacket::OnPage() : new st %lld gc %lld\n",
                             startTime, m_granuleCount));

                }
                
                if ((m_granuleCount > endTime) && !m_bEndOfStream)
                {
                    // Currently we don't support files
                    // that have a negative starting offset
                    res = HXR_INVALID_FILE;
                }
                else if (stNeedStartTime == m_state)
                {
                    m_startTime = startTime;
                    m_bHasStartTime = TRUE;

                    res = ProcessQueuedPages(m_startTime);
                    
                    if ( HXR_OK == res)
                    {
                        ChangeState(stReady);
                    }
                }
                else if (stReady == m_state)
                {
                    res = HXR_OK;
                }
            }
        }

        if ((HXR_OK == res) || (HXR_NO_DATA == res))
        {
            HX_RESULT tmpRes = HXR_OK;

            if (bHasHeader)
            {
                CacheHeader(pPage);
            }

            if (HasStartTime())
            {
                if (stSeeking == m_state)
                {
                    UINT32 ulEndTS;
                    GetTimestamp(ogg_page_granulepos(pPage), ulEndTS);

                    if (ulEndTS >= m_ulSeekTime)
                    {
                        ChangeState(stReady);
                    }
                }

                if (stReady == m_state)
                {
                    UINT32 ulTimestamp = 0;
                    GetTimestamp(startTime, ulTimestamp);

                    tmpRes = CreatePackets(ulTimestamp, pPage, FALSE);
                }
                else if (!bHasHeader)
                {
                    tmpRes = QueuePage(pPage);
                }
            }
            else if (!bHasHeader)
            {
                tmpRes = QueuePage(pPage);
            }

            if (HXR_OK != tmpRes)
            {
                res = tmpRes;
            }
        }
    }

    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::OnPage() : res %08x\n", res));
    return res;
}

STDMETHODIMP VorbisPageToPacket::OnEndOfFile()
{
    m_bEndOfStream = TRUE;

    return HXR_OK;
}

STDMETHODIMP VorbisPageToPacket::GetPacket(REF(IHXPacket*) pPkt)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::GetPacket()\n"));

    HX_RESULT res = HXR_NO_DATA;

    if (!m_timedPackets.IsEmpty())
    {
        if (!m_bSentHeaders)
        {
            res = CreateHeaderPackets();
        }
        else
        {
            res = HXR_OK;
        }

        if (HXR_OK == res)
        {
            pPkt = (IHXPacket*)m_timedPackets.RemoveHead();
        }
    }
    else if (m_bEndOfStream)
    {
        res = HXR_STREAM_DONE;
    }

    return res;
}

STDMETHODIMP VorbisPageToPacket::GetNextTimestamp(REF(UINT32) ulTimestamp)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::GetNextTimestamp()\n"));

    HX_RESULT res = HXR_NO_DATA;

    if (!m_timedPackets.IsEmpty())
    {
        IHXPacket* pPkt = (IHXPacket*)m_timedPackets.GetHead();

        ulTimestamp = pPkt->GetTime();

        res = HXR_OK;
    }
    else if (m_bEndOfStream)
    {
        res = HXR_STREAM_DONE;
    }

    return res;
}

STDMETHODIMP VorbisPageToPacket::GetTACInfo(REF(IHXValues*) pHeader)
{
    HX_RESULT res = HXR_FAILED;

    if ((stReady == m_state) && m_pCCF)
    {
        res = m_pCCF->CreateInstance(CLSID_IHXValues, (void **)&pHeader);
        
        if (HXR_OK == res) 
        {
            int hdrCount = sizeof(zm_pOgg2HelixMap) / sizeof(Ogg2HelixEntry);
    
            for (int i = 0; i < m_vc.comments; i++) 
            {
                BOOL bFoundHdr = FALSE;
                
                for (int j = 0; !bFoundHdr && (j < hdrCount); j++)
                {
                    if (HXR_OK == AddOggFieldToHeader(zm_pOgg2HelixMap[j].m_pOggHdr,
                                                      m_vc.user_comments[i],
                                                      zm_pOgg2HelixMap[j].m_pHelixHdr,
                                                      pHeader))
                    {
                        bFoundHdr = TRUE;
                    }
                }
            }
        }
    }

    return res;
}

STDMETHODIMP VorbisPageToPacket::GetStreamHeader(REF(IHXValues*) pHeader)
{
    HX_RESULT res = HXR_FAILED;

    if ((stReady == m_state) && m_pCCF)
    {
        res = m_pCCF->CreateInstance(CLSID_IHXValues, (void **)&pHeader);
        
        if (HXR_OK == res) 
        {
            // 2 seconds of preroll should be plenty
            pHeader->SetPropertyULONG32("Preroll", 2000);

            if (m_vi.bitrate_nominal > 0)
            {
                pHeader->SetPropertyULONG32("AvgBitRate", 
                                            m_vi.bitrate_nominal);
            }

            if (m_vi.bitrate_upper > 0)
            {
                pHeader->SetPropertyULONG32("MaxBitRate", m_vi.bitrate_upper);
            }

            IHXBuffer* pMimeType = NULL;

            res = m_pCCF->CreateInstance(CLSID_IHXBuffer,(void**)&pMimeType);

            if (HXR_OK == res)
            {
                res = pMimeType->Set((UINT8*)VORBIS_MIME_TYPE,
                                     strlen(VORBIS_MIME_TYPE) + 1);

                if (HXR_OK == res)
                {
                    res = pHeader->SetPropertyCString("MimeType", pMimeType);
                }

                HX_RELEASE(pMimeType);
            }
        }

        if (HXR_OK != res)
        {
            HX_RELEASE(pHeader);
        }
    }

    return res;
}

STDMETHODIMP VorbisPageToPacket::OnSeek(UINT32 ulSeekPoint)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::OnSeek(%u)\n", ulSeekPoint));
    HX_RESULT res = HXR_FAILED;

    if (stReady == m_state)
    {
        FlushQueues();

        ogg_stream_reset(&m_os);
        m_granuleCount = 0;
        m_lastBlockSize = 0;
        
        m_bEndOfStream = FALSE;

        m_ulSeekTime = ulSeekPoint;
        ChangeState(stSeeking);

        res = HXR_OK;
    }

    return res;
}

STDMETHODIMP VorbisPageToPacket::SetEndTime(ogg_int64_t endTime)
{
    m_endTime = endTime;

    return HXR_OK;
}

STDMETHODIMP_(BOOL) VorbisPageToPacket::HasStartTime() const
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::HasStartTime() : %d\n",
                           m_bHasStartTime));

    return m_bHasStartTime;
}

STDMETHODIMP_(UINT32) VorbisPageToPacket::StartTime() const
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::StartTime() : %lld\n", 
                           m_startTime));
    return INT64_TO_UINT32(m_startTime);
}

STDMETHODIMP_(UINT32) VorbisPageToPacket::Duration() const
{
    UINT32 ulRet = 0;

    GetTimestamp(m_endTime, ulRet);
    
    return ulRet;
}

STDMETHODIMP_(BOOL) VorbisPageToPacket::EndOfStream() const
{
    return m_bEndOfStream && m_timedPackets.IsEmpty();
}

STDMETHODIMP VorbisPageToPacket::GetTimestamp(ogg_int64_t granulepos, 
                                              REF(UINT32) ulTimestamp) const
{
    HX_RESULT res = HXR_FAILED;
    
    if (m_bHasStartTime && (granulepos >= m_startTime))
    {
        ogg_int64_t delta = granulepos - m_startTime;
        ogg_int64_t q =  delta / m_vi.rate;
        ogg_int64_t r = delta - q * m_vi.rate;

        ulTimestamp = (UINT32)(1000 * q + ((1000 * r) / m_vi.rate));

        DPRINTF(D_VORBIS_P2P, 
                ("VorbisPageToPacket::GetTimestamp(%lld) : %u\n", 
                 granulepos, ulTimestamp));
        
        res = HXR_OK;
    }

    return res;
}

void VorbisPageToPacket::ChangeState(State newState)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::ChangeState() : %d -> %d\n", 
                           m_state, newState));
    m_state = newState;
}

HX_RESULT VorbisPageToPacket::QueuePage(ogg_page* pPage)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::QueuePage()\n"));

    HX_RESULT res = HXR_OUTOFMEMORY;

    if (pPage)
    {
        ogg_page* pNewPage = OggUtil::CopyPage(pPage);

        if (pNewPage)
        {
            if (m_untimedPages.AddTail(pNewPage))
            {
                res = HXR_OK;
            }
            else
            {
                OggUtil::DestroyPage(pPage);
            }
        }
    }
    else
    {
        res = HXR_UNEXPECTED;
    }

    return res;
}

HX_RESULT VorbisPageToPacket::ProcessQueuedPages(ogg_int64_t startTime)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::ProcessQueuedPages(%lld)\n",
                           startTime));

    HX_RESULT res = HXR_OK;
    
    while((HXR_OK == res) && !m_untimedPages.IsEmpty())
    {
        ogg_page* pPage = (ogg_page*)m_untimedPages.RemoveHead();

        if (pPage)
        {
            UINT32 ulTimestamp = 0;
            GetTimestamp(startTime, ulTimestamp);

            res = CreatePackets(ulTimestamp, pPage, FALSE);

            OggUtil::DestroyPage(pPage);
        }
    }

    return res;
}

HX_RESULT VorbisPageToPacket::CreatePackets(UINT32 ulTimestamp, 
                                            ogg_page* pPage,
                                            BOOL bIsHeader)
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::CreatePackets(%lu)\n",
                           ulTimestamp));

    HX_RESULT res = HXR_FAILED;
    
    if (m_pCCF && pPage && m_bHasStreamID)
    {
        IHXPacket* pPacket = NULL;
        res = m_pCCF->CreateInstance(CLSID_IHXPacket, (void**)&pPacket);

        if (HXR_OK == res)
        {
            IHXBuffer* pPayload = NULL;
            
            res = m_pCCF->CreateInstance(CLSID_IHXBuffer, (void**)&pPayload);

            if (HXR_OK == res)
            {
                res = pPayload->SetSize(pPage->header_len + pPage->body_len);

                if (HXR_OK == res)
                {
                    memcpy(pPayload->GetBuffer(), 
                           pPage->header, pPage->header_len);
                    memcpy(pPayload->GetBuffer() + pPage->header_len, 
                           pPage->body, pPage->body_len);

                    res = pPacket->Set(pPayload,
                                       ulTimestamp,
                                       m_uStreamID, // streamID
                                       0, // flags
                                       0); // rule
                    
                    if (HXR_OK == res)
                    {
                        pPacket->AddRef();

                        BOOL bAddFailed = FALSE;

                        if (bIsHeader)
                        {
                            bAddFailed = !m_timedPackets.AddHead(pPacket);
                        }
                        else
                        {
                            bAddFailed = !m_timedPackets.AddTail(pPacket);
                        }

                        if (bAddFailed)
                        {
                            HX_RELEASE(pPacket);

                            res = HXR_OUTOFMEMORY;
                        }
                    }
                }
            }

            HX_RELEASE(pPayload);
        }
        HX_RELEASE(pPacket);
    }

    return res;
}

HX_RESULT VorbisPageToPacket::CacheHeader(ogg_page* pPage)
{    
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::CacheHeader()\n"));

    HX_RESULT res = HXR_OUTOFMEMORY;

    if (HaveThisHeader(pPage))
    {
        res = HXR_OK;
    }
    else
    {
        ogg_page* pNewPage = OggUtil::CopyPage(pPage);

        // We add the pages to the head so that they are in
        // the right order when we insert them into the head
        // of the timed packet list
        if (pNewPage && 
            m_headerPages.AddHead(pNewPage))
        {
            DPRINTF(D_VORBIS_P2P, 
                    ("VorbisPageToPacket::CacheHeader() : hdr cached\n"));
            res = HXR_OK;
        }
        else
        {
            OggUtil::DestroyPage(pNewPage);
        }
    }

    return res;
}

BOOL VorbisPageToPacket::HaveThisHeader(ogg_page* pPage)
{
    BOOL bRet = FALSE;
    
    CHXSimpleList::Iterator i;

    for (i = m_headerPages.Begin(); !bRet && i != m_headerPages.End(); ++i)
    {
        ogg_page* pHdr = (ogg_page*)(*i);

        if (OggUtil::ComparePages(pHdr, pPage))
        {
            bRet = TRUE;
        }
    }

    DPRINTF(D_VORBIS_P2P, 
            ("VorbisPageToPacket::HaveThisHeader() : bRet %d\n", bRet));

    return bRet;
}

HX_RESULT VorbisPageToPacket::CreateHeaderPackets()
{
    DPRINTF(D_VORBIS_P2P, ("VorbisPageToPacket::CreateHeaderPackets()\n"));

    HX_RESULT res = HXR_UNEXPECTED;

    if (!m_headerPages.IsEmpty())
    {
        // Get the timestamp of the first data packet
        UINT32 ulTimestamp;
        res = GetNextTimestamp(ulTimestamp);

        if (HXR_OK == res)
        {
            // Create packets for these header pages and
            // enqueue them before the first data packet.
            CHXSimpleList::Iterator i;
            for (i = m_headerPages.Begin(); 
                 (HXR_OK == res) && i != m_headerPages.End(); ++i)
            {
                ogg_page* pHdr = (ogg_page*)(*i);
                
                res = CreatePackets(ulTimestamp, pHdr, TRUE);
            }

            if (HXR_OK == res)
            {
                m_bSentHeaders = TRUE;
            }
        }
    }

    return res;
}

void VorbisPageToPacket::FlushQueues()
{
    while(!m_untimedPages.IsEmpty())
    {
        ogg_page* pPage = (ogg_page*)m_untimedPages.RemoveHead();

        OggUtil::DestroyPage(pPage);
    }

    while(!m_timedPackets.IsEmpty())
    {
        IHXPacket* pPkt = (IHXPacket*)m_timedPackets.RemoveHead();

        HX_RELEASE(pPkt);
    }
}

void VorbisPageToPacket::FlushHeaders()
{
    while(!m_headerPages.IsEmpty())
    {
        ogg_page* pPage = (ogg_page*)m_headerPages.RemoveHead();

        OggUtil::DestroyPage(pPage);
    }
}

BOOL VorbisPageToPacket::IsVorbisHeader(ogg_packet* pPkt)
{
    BOOL bRet = FALSE;

    if (pPkt && pPkt->packet && (pPkt->bytes > 7) &&
        ((pPkt->packet[0] & 0x1) == 0x1) &&
        !memcmp(pPkt->packet + 1, "vorbis", 6))
        
    {
        bRet = TRUE;
    }

    return bRet;
}

HX_RESULT VorbisPageToPacket::AddOggFieldToHeader(const char* pOggFieldName,
                                                  const char* pOggData,
                                                  const char* pHelixFieldName,
                                                  IHXValues* pHeader) const
{
    HX_RESULT res = HXR_FAILED;
    const char* pos = NULL;
    IHXBuffer* pBuf = NULL;

    if (!strncasecmp(pOggFieldName, pOggData, strlen(pOggFieldName)) &&
        ((pos = strchr(pOggData, '=')) != NULL) &&
        ((pBuf = CreateBuffer((const UCHAR*)pos + 1, 
                              strlen(pos + 1) + 1)) != NULL))
    {
        res = pHeader->SetPropertyBuffer(pHelixFieldName, pBuf);
    }
    HX_RELEASE(pBuf);

    return res;
}

IHXBuffer* VorbisPageToPacket::CreateBuffer(const UCHAR* pData, 
                                            UINT32 ulSize) const
{
    IHXBuffer* pRet = NULL;

    if (m_pCCF &&
        (HXR_OK == m_pCCF->CreateInstance(CLSID_IHXBuffer, (void **)&pRet)) &&
        (HXR_OK != pRet->Set(pData, ulSize)))
    {
        HX_RELEASE(pRet);
    }

    return pRet;
}
