/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: CHXClientEngineWin.cpp,v 1.1.14.3 2004/07/09 01:49:44 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "CHXClientEngineWin.h"
#include "hxassert.h"
#include "safestring.h"
#include "dllpath.h"
#include "hxwintyp.h"

/*!
  @const kCreateEngine
  @abstract symbol name for the client engine creation func
*/
const char* const kCreateEngine = "CreateEngine";


/*!
  @const kCloseEngine
  @abstract symbol name for the client engine destruction func
*/
const char* const kCloseEngine = "CloseEngine";

/*!
  @const kClientCoreDll
  @abstract the name of the helix client core shared library on unix
*/
const char* const kClientCoreDll = "clntcore.dll";

/*!
  @const kSetDll
  @abstract symbol name for the client engine's dll search path function
*/
const char* const kSetDll = "SetDLLAccessPath";

/*!
  @const kDllTypes
  @abstract names of common dll locations used by SetDLLAccessPath
*/
const char* const kDllTypes[] =
{
    "DT_Common", "DT_Plugins", "DT_Codecs"
};

/*!
  @const kDllDirs
  @abstract names of common dll directories used by SetDLLAccessPath
*/
const char* const kDllDirs[] =
{
    "common", "plugins", "codecs"
};

/*!
  @const kNumDllTypes
  @abstract number of dll location names used by SetDLLAccessPath
*/
const UINT32 kNumDllTypes = sizeof(kDllTypes) / sizeof(*kDllTypes);


ENABLE_DLLACCESS_PATHS(hxclient);


/*!
  @function CreatePlatformClientEngine
  @abstract create and return a new CHXClientEngineWin
*/
CHXClientEngine* CreatePlatformClientEngine(void)
{
    return new CHXClientEngineWin;
}

//typedef HX_RESULT (HXEXPORT_PTR FPRMSETDLLACCESSPATH) (const char*);


/*!
  @function ~CHXClientEngineWin
  @abstract destructor
  @discussion unloads the client core
*/
CHXClientEngineWin::~CHXClientEngineWin(void)
{
    UnloadClientCore();
}


/*!
  @function CHXClientEngineWin
  @abstract constructor
*/
CHXClientEngineWin::CHXClientEngineWin(void)
    : m_pICoreEngine(NULL)
{
}


/*!
  @function LoadClientCore
  @abstract load and initialize the helix client engine
*/
HX_RESULT
CHXClientEngineWin::LoadClientCore(void)
{
    HX_RESULT res = HXR_OK;

    if (!m_dllHXCore.isOpen())
    {
	// See if the user has set their HELIX_LIBS env var
        char dllhome[MAX_PATH] = ".";
        const char* pszHelixLibs = getenv("HELIX_LIBS");
        if (pszHelixLibs)
            SafeStrCpy(dllhome, pszHelixLibs, MAX_PATH);
 
        char dllpath[MAX_PATH];
        SafeSprintf(dllpath, MAX_PATH, "%s/common/%s", dllhome, kClientCoreDll);

	if (m_dllHXCore.open(dllpath) != DLLAccess::DLL_OK)
	{
	    res = HXR_FAIL;
	}
    }
    
    return res;
}


/*!
  @function InitDLLAccessPaths
  @abstract initialize DLLAccess paths
  @discussion DLLAccess is a class used in Helix to easily load dlls and
  access their symbols.  The client engine typically expects to find certain
  types of dlls (eg: codecs, plugins) in certain locations.  The
  SetDLLAccessPath function is called to set these locations.
  InitDLLAccessPaths sets these locations to some reasonable defaults.
*/
HX_RESULT
CHXClientEngineWin::InitDLLAccessPaths(void)
{
    HX_RESULT res = LoadClientCore();

    if (SUCCEEDED(res))
    {
	FPSETDLLACCESSPATH pfnSetDll;
	pfnSetDll = (FPSETDLLACCESSPATH)m_dllHXCore.getSymbol(kSetDll);
	if (pfnSetDll)
	{
	    // See if the user has set their HELIX_LIBS env var
	    char dllhome[MAX_PATH] = ".";
	    const char* pszHelixLibs = getenv("HELIX_LIBS");
	    if (pszHelixLibs)
		SafeStrCpy(dllhome, pszHelixLibs, MAX_PATH);
	    
	    char paths[MAX_PATH * kNumDllTypes];
	    int nLen = sizeof(paths) / sizeof(char);
	    char* pPath = paths;
	    for (int i = 0; i != kNumDllTypes; ++i)
	    {
		int nSize = SafeSprintf(pPath, nLen, "%s=%s/%s",
                                        kDllTypes[i], dllhome, kDllDirs[i]);

                nSize++; // Skip over the '\0'                
		pPath += nSize;
		nLen -= nSize;
		
		HX_ASSERT(nLen >= 0);

            }
	    pPath++;
	    *pPath = 0;
	    
	    pfnSetDll(paths);
	}
    }
    return res;
}


/*!
  @function OnCreateClientEngine
  @abstract create and return an IHXClientEngine
*/
IHXClientEngine*
CHXClientEngineWin::OnCreateClientEngine(HX_RESULT* pResult)
{
    *pResult = HXR_OK;
    if (!m_pICoreEngine)
    {
	*pResult = InitDLLAccessPaths();
	if (SUCCEEDED(*pResult))
	{
	    FPRMCREATEENGINE pCreateEngine =
		(FPRMCREATEENGINE)m_dllHXCore.getSymbol(kCreateEngine);
	    if (pCreateEngine)
	    {
		*pResult = pCreateEngine(&m_pICoreEngine);
		// If engine was not created, we should receive a reason.
		HX_ASSERT((NULL != m_pICoreEngine) || (FAILED(*pResult)));
	    }
	    else
	    {
		*pResult = HXR_COULDNOTINITCORE;
	    }
	}
    }
    return m_pICoreEngine;
}


/*!
  @function GetClientEngine
  @abstract return the IHXClientEngine
*/
IHXClientEngine*
CHXClientEngineWin::GetClientEngine(void)
{
    return m_pICoreEngine;
}


/*!
  @function OnCloseClientEngine
  @abstract destroy the IHXClientEngine and unload the client core
*/
void
CHXClientEngineWin::OnCloseClientEngine(void)
{
    if (m_pICoreEngine)
    {
	HX_ASSERT(m_dllHXCore.isOpen());
	FPRMCLOSEENGINE pCloseEngine =
	    (FPRMCLOSEENGINE)m_dllHXCore.getSymbol(kCloseEngine);
	if (pCloseEngine)
	{
	    pCloseEngine(m_pICoreEngine);
	}
	m_pICoreEngine = NULL;
    }
}


/*!
  @function UnloadClientCore
  @abstract shutdown the engine and unload the client core dll
*/
void
CHXClientEngineWin::UnloadClientCore(void)
{
    if (m_dllHXCore.isOpen())
    {
	OnCloseClientEngine();
	
	m_dllHXCore.close();
    }
}
