/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: drawline.cpp,v 1.1.1.1.42.1 2004/07/09 01:59:28 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include <stdlib.h>
#include <math.h>
#include "hxtypes.h"
#include "drawline.h"

#ifdef _DEBUG
#include "region.h" //just for _DumpString
#include <stdio.h>
#endif

#define BLURALPHA(x,color) \
           {\
              int   alphap = 255-(((x)&0xff000000)>>24);\
              int     alpha  = 255-(((color)&0xff000000)>>24);\
              ULONG32 newalp = (255-((alphap*alpha)/255))<<24;\
              (x) = ((x)&0x00ffffff)|newalp;\
           }

void WriteBits(UCHAR *pBuffer, UINT8 bpp, UINT8 red, UINT8 green, UINT8 blue);
BOOL DrawOffColor(UINT32 ulStyle, UINT32 nPixelCount);

//color is now interpreted as a color if it is has no alpha channel.
//if the color does have an alpha channel value(non-zero) then we
//are blending borders and will only write the alphavalue into the
//destination pixels. If the destination has alpha values then we
//will 
void DrawLine( int* bits,
               int  width,
               int  x0,
               int  y0,
               int  x1,
               int  y1,
               ULONG32 color
               )
{

    int temp, adjUp, adjDown, errorTerm, xAdvance, xDelta, yDelta,
        wholeStep, initialPixelCount, finalPixelCount, i, j;
    BOOL bBlended = FALSE;
    
    if( color & 0xff000000)
    {
       //This is a blended border line. Just mess with the alpha channel.
       bBlended = TRUE;
    }
    
    // We'll always draw top to bottom to reduce the number of cases we have to
    // handle, and to make lines between the same endpoints draw the same pixels
    if (y0 > y1)
    {
        temp = y0;
        y0 = y1;
        y1 = temp;
        temp = x0;
        x0 = x1;
        x1 = temp;
    }

    int* p = bits + x0 + y0 * width;

    // Figure out if we're going left or right, and how far we're going horizontally
    if ((xDelta = x1 - x0) < 0)
    {
        xAdvance = -1;
        xDelta = -xDelta;
    }
    else
    {
        xAdvance = 1;
    }

    // Figure out how far we're going vertically
    yDelta = y1 - y0;

    // Special-case horizontal, vertical and diagonal lines, for speed and to avoid
    // boundary conditions and divide by 0
    if (!xDelta)
    {
        // vertical line
        for (i = 0; i <= yDelta; ++i)
        {
           if( !bBlended )
              *p = color;
           else
              BLURALPHA(*p, color);
           
           p += width;
        }
        return;
    }
    if (!yDelta)
    {
        // horizontal line
        for (i = 0; i <= xDelta; ++i)
        {
           if( !bBlended )
              *p = color;
           else
              BLURALPHA(*p, color);
           p += xAdvance;
        }
        return;
    }
    if (xDelta == yDelta)
    {
        // diagonal line
        for (i = 0; i <= xDelta; ++i)
        {
           if( !bBlended )
              *p = color;
           else
              BLURALPHA(*p, color);
           p += width + xAdvance;
        }
        return;
    }

    // determins whether line is X or Y major
    if (xDelta > yDelta)
    {
        // X major line
        // minimum number of pixels in a run
        wholeStep = xDelta / yDelta;

        // Error term adjust each time Y steps by 1; used to tell when
        // one extra pixel should be drawn as part of a run, to account for
        // fractional steps along the X axis per 1 pixel steps along Y
        adjUp = (xDelta % yDelta) * 2;

        // Error term adjust when the error term turns over, used to factor
        // out the X step made at that time
        adjDown = yDelta * 2;

        // Initial error term reflects an initial step of 0.5 along the Y axis
        errorTerm = (xDelta % yDelta) - (yDelta * 2);

        // The initial and last runs are partial, because Y advances only 0.5
        // for these runs, rather than 1.  Divide one full run, plus the
        // initial pixel, between the initail and last runs
        initialPixelCount = (wholeStep / 2) + 1;
        finalPixelCount = initialPixelCount;

        // If the basic run length is even and there's no fractional
        // advance, we have one pixel that could go to either the initial
        // or last partial run, which we'll arbitrarily allocate to the last run
        if ((!adjUp) && ((wholeStep & 1) == 0))
        {
            --initialPixelCount;
        }

        // If ther are an odd number of pixels per run, we have 1 pixel that can't
        // be allocated to either the initial or last partial run, so we'll add 0.5
        // to the error term so this pixel will be handled by the normal full-run loop
        if (wholeStep & 1)
        {
            errorTerm += yDelta;
        }

        // Draw the first partial run of pixels
        for (j = 0; j < initialPixelCount; ++j)
        {
            if( !bBlended )
               *p = color;
            else
               BLURALPHA(*p, color);
            p += xAdvance;
        }
        p += width;

        // Draw all full runs
        for (i = 0; i < yDelta - 1; ++i)
        {
            // advance the error term and add an extra pixel if indicated
            if ((errorTerm += adjUp) > 0)
            {
                errorTerm -= adjDown;

                // Draw this scan lines run
                for (j = 0; j < wholeStep + 1; ++j)
                {
                   if( !bBlended )
                      *p = color;
                   else
                      BLURALPHA(*p, color);
                   p += xAdvance;
                }
            }
            else
            {
                // Draw this scan lines run
                for (j = 0; j < wholeStep; ++j)
                {
                   if( !bBlended )
                      *p = color;
                   else
                      BLURALPHA(*p, color);
                   p += xAdvance;
                }
            }
            p += width;
        }
        // Draw final run
        for (j = 0; j < finalPixelCount; ++j)
        {
           if( !bBlended )
              *p = color;
           else
              BLURALPHA(*p, color);
           p += xAdvance;
        }
        return;
    }
    else
    {
        // Y major line 
        // for comments see above section
        wholeStep = yDelta / xDelta;
        adjUp = (yDelta % xDelta) * 2;
        adjDown = xDelta * 2;
        errorTerm = (yDelta % xDelta) - (xDelta * 2);
        initialPixelCount = (wholeStep / 2) + 1;
        finalPixelCount = initialPixelCount;

        if ((!adjUp) && ((wholeStep & 1) == 0))
        {
            --initialPixelCount;
        }

        if (wholeStep & 1)
        {
            errorTerm += xDelta;
        }

        for (j = 0; j < initialPixelCount; ++j)
        {
           if( !bBlended )
              *p = color;
           else
              BLURALPHA(*p, color);
           p += width;
        }
        p += xAdvance;

        for (i = 0; i < xDelta - 1; ++i)
        {
            if ((errorTerm += adjUp) > 0)
            {
                errorTerm -= adjDown;
                for (j = 0; j < wholeStep + 1; ++j)
                {
                   if( !bBlended )
                      *p = color;
                   else
                      BLURALPHA(*p, color);
                   p += width;
                }
            }
            else
            {
                for (j = 0; j < wholeStep; ++j)
                {
                   if( !bBlended )
                      *p = color;
                   else
                      BLURALPHA(*p, color);
                   p += width;
                }
            }
            p += xAdvance;
        }
        for (j = 0; j < finalPixelCount; ++j)
        {
           if( !bBlended )
              *p = color;
           else
              BLURALPHA(*p, color);
           p += width;
        }
        return;
    }
}


BOOL DrawOffColor(INT32 *pPattern,
                  INT32 nEntries,
                  INT32 nPixelsInPattern,
                  UINT32 nPixelCount)
{
    BOOL bRet = FALSE;

    INT32 nMod = nPixelCount % nPixelsInPattern;
    INT32 nPixelSum = 0;

    for (int i=0; i<nEntries; i++)
    {
        nPixelSum += pPattern[i];
        if (nPixelSum > nMod)
        {
            bRet = i % 2;
            break;
        }
    }

    return bRet;
}

void WriteBits(UCHAR *pBuffer, UINT8 bpp, UINT8 red, UINT8 green, UINT8 blue)
{
    if (bpp == 32)
    {
        pBuffer[0] = blue;
        pBuffer[1] = green;
        pBuffer[2] = red;
        pBuffer[3] = 0;
    }
    else if (bpp == 24)
    {
        pBuffer[0] = blue;
        pBuffer[1] = green;
        pBuffer[2] = red;
    }
    // Use most significant bits when "dithering" to RGB565
    else if (bpp == 16)
    {
        pBuffer[0] = 0;
        pBuffer[1] = 0;

        pBuffer[0] |= blue & 0xF8;
        pBuffer[0] |= red>>5;
        pBuffer[1] |= (red & 0x1C)<<3;
        pBuffer[1] |= green>>3;
    }
    // Use most significant bits when "dithering" to RGB555 (NEED TO TEST!!!)
    else if (bpp == 15)
    {
        pBuffer[0] = 0;
        pBuffer[1] = 0;

        pBuffer[0] |= blue & 0xF8;
        pBuffer[0] |= red>>5;
        pBuffer[1] |= (red & 0x18)<<3;
        pBuffer[1] |= (green & 0xF8)>>3;
    }
}

void StraightLine(void *pImage,
                  INT32 nPitch,
                  INT32 bpp,
                  INT32 nImageWidth, INT32 nImageHeight,
                  INT32 x0, INT32 y0,
                  INT32 x1, INT32 y1,
                  UINT8 lineWidth,
                  PixelProps *pPrimaryPixels,
                  PixelProps *pSecondaryPixels,
                  INT32 ulLineSytle,
                  INT32 *pCustomPattern, INT32 nCustomEntries)
{
    // Assume image is alwas draw top-down - cross platform??? yikes
    if (nPitch < 0)
        nPitch *= -1;

    UINT8 *pBuffer = (UINT8*)pImage;
    INT32 aPattern[2];

    PixelProps *pColor = pPrimaryPixels;

    if (ulLineSytle != CUSTOM_LINE)
    {
        // Build our own custom pattern list
        if (ulLineSytle == SOLID_LINE)
        {
            aPattern[0] = 2;
            aPattern[1] = 0;
        }
        else if (ulLineSytle == DOTTED_LINE)
        {
            aPattern[0] = 2;
            aPattern[1] = 2;
        }
        else if (ulLineSytle == DASHED_LINE)
        {
            aPattern[0] = 16;
            aPattern[1] = 8;
        }

        pCustomPattern = aPattern;
        nCustomEntries = 2;
    }

    INT32 nPatternPixelSum = 0;
    for (int i=0; i<nCustomEntries; i++)
        nPatternPixelSum += pCustomPattern[i];
        
    INT32 x0orig = x0;
    INT32 y0orig = y0;
    INT32 x1orig = x1;
    INT32 y1orig = y1;

    // Draw a line for each pixel width
    for (int j=0; j<lineWidth; j++)
    {        
        INT32 nPixelCount = 0;

        // Line length for x and y line components
        INT32 yDelta = y1 - y0;
        INT32 xDelta = x1 - x0;
    
        // Bytes to move for the next xy pixel
        INT32 nNextXPixel = bpp>>3,
              nNextYPixel = nPitch;
    
        // Handle negative xy deltas
        if (yDelta < 0)
        { 
            yDelta = -yDelta;
            nNextYPixel = -nNextYPixel;
        }
    
        if (xDelta < 0)
        {
            xDelta = -xDelta;
            nNextXPixel = -nNextXPixel;
        }

        // Move x and y points to their corresponding byte offset in the image
        y0 *= nPitch;
        y1 *= nPitch;

        x0 *= abs(nNextXPixel);
        x1 *= abs(nNextXPixel);

        pColor = pPrimaryPixels;

        // Write the first pixel in the line
        if (DrawOffColor(pCustomPattern,
                         nCustomEntries,
                         nPatternPixelSum,
                         nPixelCount))
        {
            pColor = pSecondaryPixels;
        }
        
        if (pColor->bHasColors)
        {
            WriteBits(&pBuffer[x0+y0], (UINT8)bpp, 
                      pColor->red,
                      pColor->green,
                      pColor->blue);
        }

        ++nPixelCount;

        if (xDelta > yDelta)
        {
            // Slope more horizontal, so incrment the x value by 1 and 
            // calculate a new y value.
            float slope = (float)(yDelta / xDelta * nNextYPixel);
            float newY = (float)(.5 + slope + y0);
        
            // Set points for next line
            ++y0orig;
            ++y1orig;
        
            while (x0 != x1)
            {
                x0 += nNextXPixel;
                newY += slope;
            
                pColor = pPrimaryPixels;
                if (DrawOffColor(pCustomPattern,
                                 nCustomEntries,
                                 nPatternPixelSum,
                                 nPixelCount))
                {
                    pColor = pSecondaryPixels;
                }
        
                if (pColor->bHasColors)
                {
                    WriteBits(&pBuffer[x0+(INT32)newY], (UINT8)bpp, 
                              pColor->red,
                              pColor->green,
                              pColor->blue);
                }
            
                ++nPixelCount;
            }
        }
        else
        {
            // Slope more vertical, so incrment the y value by 1 and 
            // calculate a new x value.
            float slope = (float)(xDelta / yDelta * nNextXPixel);
            float newX = (float)(.5 + slope + x0);

            // Set points for next line
            ++x0orig;
            ++x1orig;
        
            while (y0 != y1)
            {
                y0 += nNextYPixel;
                newX += slope;

                pColor = pPrimaryPixels;
                if (DrawOffColor(pCustomPattern,
                                 nCustomEntries,
                                 nPatternPixelSum,
                                 nPixelCount))
                {
                    pColor = pSecondaryPixels;
                }
        
                if (pColor->bHasColors)
                {
                    WriteBits(&pBuffer[(INT32)newX+y0], (UINT8)bpp, 
                              pColor->red,
                              pColor->green,
                              pColor->blue);
                }
            
                ++nPixelCount;
            }
        }

        // Reset points to pixels instead of byte offsets
        x0 = x0orig;
        x1 = x1orig;

        y0 = y0orig;
        y1 = y1orig;
    }
}

