/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: brushff.cpp,v 1.3.2.1 2004/07/09 01:57:45 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

// system
#include <stdio.h>
// include
#include "hxcom.h"
#include "hxtypes.h"
#include "hxcomm.h"
#include "ihxpckts.h"
#include "hxfiles.h"
#include "hxformt.h"
#include "hxplugn.h"
#include "hxver.h"
// pnmisc
#include "baseobj.h"
// pxcomlib
#include "buffutil.h"
// brushff
#include "brushff.h"
#include "brushff.ver"
// pndebug
#include "errdbg.h"
#include "hxheap.h"
#ifdef _DEBUG
#undef HX_THIS_FILE             
static const char HX_THIS_FILE[] = __FILE__;
#endif

const char* const CBrushFileFormat::m_pszDescription       = "RealNetworks Brush File Format Plugin";
const char* const CBrushFileFormat::m_pszCopyright         = HXVER_COPYRIGHT;
const char* const CBrushFileFormat::m_pszMoreInfoURL       = HXVER_MOREINFO;
const char* const CBrushFileFormat::m_ppszFileMimeTypes[]  = {"text/brush", NULL};
const char* const CBrushFileFormat::m_ppszFileExtensions[] = {"bsh", NULL};
const char* const CBrushFileFormat::m_ppszFileOpenNames[]  = {"Brush Files (*.bsh)", NULL};
const char* const CBrushFileFormat::m_pszStreamMimeType    = "application/vnd.rn-brushstream";
const UINT32      CBrushFileFormat::m_ulContentVersion     = HX_ENCODE_PROD_VERSION(0, 0, 0, 0);
const UINT32      CBrushFileFormat::m_ulStreamVersion      = HX_ENCODE_PROD_VERSION(0, 0, 0, 0);

CBrushFileFormat::CBrushFileFormat()
{
    m_lRefCount = 0;
    Reset();
};

CBrushFileFormat::~CBrushFileFormat()
{
    Deallocate();
};

STDMETHODIMP CBrushFileFormat::QueryInterface(REFIID riid, void** ppvObj)
{
    HX_RESULT retVal = HXR_OK;

    if (IsEqualIID(riid, IID_IUnknown))
    {
        AddRef();
        *ppvObj = (IUnknown*) (IHXPlugin*) this;
    }
    else if (IsEqualIID(riid, IID_IHXPlugin))
    {
        AddRef();
        *ppvObj = (IHXPlugin*) this;
    }
    else if (IsEqualIID(riid, IID_IHXFileFormatObject))
    {
        AddRef();
        *ppvObj = (IHXFileFormatObject*) this;
    }
    else if (IsEqualIID(riid, IID_IHXFileResponse))
    {
        AddRef();
        *ppvObj = (IHXFileResponse*) this;
    }
    else
    {
        *ppvObj = NULL;
        retVal  = HXR_NOINTERFACE;
    }

    return retVal;
}

STDMETHODIMP_(UINT32) CBrushFileFormat::AddRef()
{
    return InterlockedIncrement(&m_lRefCount);
}

STDMETHODIMP_(UINT32) CBrushFileFormat::Release()
{
    if (InterlockedDecrement(&m_lRefCount) > 0)
    {
        return m_lRefCount;
    }

    delete this;

    return 0;
}

STDMETHODIMP CBrushFileFormat::GetPluginInfo(REF(BOOL)        rbLoadMultiple,
                                             REF(const char*) rpszDescription,
                                             REF(const char*) rpszCopyright,
                                             REF(const char*) rpszMoreInfoURL,
                                             REF(UINT32)      rulVersionNumber)
{
    rbLoadMultiple   = TRUE;
    rpszDescription  = (const char*) m_pszDescription;
    rpszCopyright    = (const char*) m_pszCopyright;
    rpszMoreInfoURL  = (const char*) m_pszMoreInfoURL;
    rulVersionNumber = TARVER_ULONG32_VERSION;

    return HXR_OK;
}

STDMETHODIMP CBrushFileFormat::InitPlugin(IUnknown* pContext)
{
    HX_RESULT retVal = HXR_FAIL;

    if (pContext)
    {
        // Clear out everything
        Deallocate();
        // Save a copy of the calling context
        m_pContext = pContext;
        m_pContext->AddRef();
        // Get an interface to the common class factory
        retVal = m_pContext->QueryInterface(IID_IHXCommonClassFactory,
                                            (void**) &m_pCommonClassFactory);
        if (SUCCEEDED(retVal))
        {
            // Set the state
            m_ulState = kStatePluginInitialized;
        }
    }

    return retVal;
}


STDMETHODIMP CBrushFileFormat::GetFileFormatInfo(REF(const char**) rppszFileMimeTypes,
                                                 REF(const char**) rppszFileExtensions,  
                                                 REF(const char**) rppszFileOpenNames)
{
    rppszFileMimeTypes  = (const char**) m_ppszFileMimeTypes;
    rppszFileExtensions = (const char**) m_ppszFileExtensions;
    rppszFileOpenNames  = (const char**) m_ppszFileOpenNames;

    return HXR_OK;
}

STDMETHODIMP CBrushFileFormat::InitFileFormat(IHXRequest*        pRequest, 
                                              IHXFormatResponse* pFormatResponse,
                                              IHXFileObject*     pFileObject)
{
    HX_RESULT retVal = HXR_OK;

    if (pRequest && pFormatResponse && pFileObject)
    {
        if (m_ulState == kStatePluginInitialized)
        {
            // Find out from the 822 headers if this
            // is a NULL brush
            m_bNullBrush = IsNullBrush(pRequest);
            // Save members
            HX_RELEASE(m_pFormatResponse);
            m_pFormatResponse = pFormatResponse;
            m_pFormatResponse->AddRef();
            HX_RELEASE(m_pFileObject);
            m_pFileObject = pFileObject;
            m_pFileObject->AddRef();
            // Set the state
            m_ulState = kStateFileInitPending;
            // Init the file object
            m_pFileObject->Init(HX_FILE_READ | HX_FILE_BINARY, this);
        }
        else
        {
            retVal = HXR_UNEXPECTED;
        }
    }
    else
    {
        retVal = HXR_FAIL;
    }

    if (FAILED(retVal) && pFormatResponse)
    {
        pFormatResponse->InitDone(retVal);
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::GetFileHeader()
{
    HX_RESULT retVal = HXR_OK;

    if (m_ulState == kStateFileFormatInitialized)
    {
        // Get an IHXValues object
        IHXValues* pFileHeader = NULL;
        retVal                  = m_pCommonClassFactory->CreateInstance(CLSID_IHXValues,
                                                                        (void**) &pFileHeader);
        if (SUCCEEDED(retVal))
        {
            // Set the stream count
            pFileHeader->SetPropertyULONG32("StreamCount",    1);
            pFileHeader->SetPropertyULONG32("IsRealDataType", 1);
            // Set the new state
            m_ulState = kStateFileHeaderSent;
            // Call the response interface
            m_pFormatResponse->FileHeaderReady(HXR_OK, pFileHeader);
        }
        HX_RELEASE(pFileHeader);

        if (FAILED(retVal))
        {
            m_pFormatResponse->FileHeaderReady(retVal, NULL);
        }
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::GetStreamHeader(UINT16 usStreamNum)
{
    HX_RESULT retVal = HXR_OK;

    if (m_ulState == kStateFileHeaderSent)
    {
        // Create an IHXValues object
        IHXValues* pStreamHeader = NULL;
        retVal                    = m_pCommonClassFactory->CreateInstance(CLSID_IHXValues,
                                                                          (void**) &pStreamHeader);
        if (SUCCEEDED(retVal))
        {
            // Create mime type IHXBuffer string
            IHXBuffer* pMimeStr = NULL;
            retVal = PXUtilities::CreateStringBuffer((const char*) m_pszStreamMimeType,
                                                     m_pContext,
                                                     pMimeStr);
            if (SUCCEEDED(retVal))
            {
                // Create intrinsic duration type string buffer
                IHXBuffer* pIntrin = NULL;
                retVal = PXUtilities::CreateStringBuffer("intrinsicDurationDiscrete",
                                                         m_pContext,
                                                         pIntrin);
                if (SUCCEEDED(retVal))
                {
                    // Create ASM rulebook string buffer
                    char szASM[64]; /* Flawfinder: ignore */
                    sprintf(szASM, "AverageBandwidth=%lu,Priority=5;", kDefaultBitrate); /* Flawfinder: ignore */
                    IHXBuffer* pASM = NULL;
                    retVal = PXUtilities::CreateStringBuffer((const char*) szASM,
                                                             m_pContext, pASM);
                    if (SUCCEEDED(retVal))
                    {
                        // Set the properties
                        pStreamHeader->SetPropertyULONG32("StreamNumber",          0);
                        pStreamHeader->SetPropertyULONG32("MaxBitRate",            kDefaultBitrate);
                        pStreamHeader->SetPropertyULONG32("AvgBitRate",            kDefaultBitrate);
                        pStreamHeader->SetPropertyULONG32("MaxPacketSize",         kDefaultPacketSize);
                        pStreamHeader->SetPropertyULONG32("AvgPacketSize",         kDefaultPacketSize);
                        pStreamHeader->SetPropertyULONG32("Preroll",               kDefaultPreroll);
                        pStreamHeader->SetPropertyULONG32("Duration",              kDefaultDuration);
                        pStreamHeader->SetPropertyCString("MimeType",              pMimeStr);
                        pStreamHeader->SetPropertyULONG32("ContentVersion",        m_ulContentVersion);
                        pStreamHeader->SetPropertyULONG32("StreamVersion",         m_ulStreamVersion);
                        pStreamHeader->SetPropertyCString("ASMRuleBook",           pASM);
                        pStreamHeader->SetPropertyCString("intrinsicDurationType", pIntrin);
                        if (m_pFileBuffer)
                        {
                            pStreamHeader->SetPropertyBuffer("OpaqueData", m_pFileBuffer);
                        }
                        if (m_bNullBrush)
                        {
                            pStreamHeader->SetPropertyULONG32("NullBrush", 1);
                        }
                        // Set the new state
                        m_ulState = kStateReadyForGetPacket;
                        // Pass the stream header back to the server
                        m_pFormatResponse->StreamHeaderReady(HXR_OK, pStreamHeader);
                    }
                    HX_RELEASE(pASM);
                }
                HX_RELEASE(pIntrin);
            }
            HX_RELEASE(pMimeStr);
        }
        HX_RELEASE(pStreamHeader);
        // Now we can release the file buffer
        HX_RELEASE(m_pFileBuffer);
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    if (FAILED(retVal))
    {
        m_pFormatResponse->StreamHeaderReady(retVal, NULL);
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::GetPacket(UINT16 usStreamNum)
{
    HX_RESULT retVal = HXR_OK;

    if (m_ulState == kStateReadyForGetPacket)
    {
        if (usStreamNum == 0)
        {
            // Set the state
            m_ulState = kStateStreamDoneSent;
            // Read from the file
            m_pFormatResponse->StreamDone(0);
        }
        else
        {
            retVal = HXR_INVALID_PARAMETER;
        }
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::Seek(UINT32 ulOffset)
{
    HX_RESULT retVal = HXR_OK;

    if (m_pFormatResponse)
    {
        // Set the state
        m_ulState = kStateReadyForGetPacket;
        // Tell the format response the seek is done
        m_pFormatResponse->SeekDone(HXR_OK);
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::Close()
{
    HX_RESULT retVal = HXR_OK;

    Deallocate();

    return retVal;
}

STDMETHODIMP CBrushFileFormat::InitDone(HX_RESULT status)
{
    HX_RESULT retVal = HXR_OK;

    if (m_ulState == kStateFileInitPending)
    {
        if (SUCCEEDED(status))
        {
            // Set the new state
            m_ulState = kStateFileReadPending;
            // Read the first kDefaultPacketSize bytes of the file
            m_pFileObject->Read(kDefaultPacketSize);
        }
        else
        {
            // Set the state
            m_ulState = kStateError;
            // Inform the response interface of the error
            m_pFormatResponse->InitDone(status);
        }
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::SeekDone(HX_RESULT status)
{
    return HXR_UNEXPECTED;
}

STDMETHODIMP CBrushFileFormat::ReadDone(HX_RESULT status, IHXBuffer* pBuffer)
{
    HX_RESULT retVal = HXR_OK;

    if (m_ulState == kStateFileReadPending)
    {
        if (SUCCEEDED(status))
        {
            // If we already have a buffer, then we'll need to
            // copy this buffer into the bigger one. In practical
            // use, this should never happen, since the SMIL-generated
            // "brush file" should only be a few bytes, well within
            // kDefaultPacketSize bytes.
            if (m_pFileBuffer)
            {
                IHXBuffer* pTmp = NULL;
                m_pCommonClassFactory->CreateInstance(CLSID_IHXBuffer, (void**) &pTmp);
                if (pTmp)
                {
                    // Set the size
                    pTmp->SetSize(m_pFileBuffer->GetSize() + pBuffer->GetSize());
                    // Copy in the old buffer
                    memcpy(pTmp->GetBuffer(), m_pFileBuffer->GetBuffer(), m_pFileBuffer->GetSize()); /* Flawfinder: ignore */
                    // Copy in the new buffer
                    memcpy(pTmp->GetBuffer() + m_pFileBuffer->GetSize(), /* Flawfinder: ignore */
                           pBuffer->GetBuffer(), pBuffer->GetSize());
                    // Now get rid of the old buffer
                    HX_RELEASE(m_pFileBuffer);
                    m_pFileBuffer = pTmp;
                    m_pFileBuffer->AddRef();
                }
                HX_RELEASE(pTmp);
            }
            else
            {
                m_pFileBuffer = pBuffer;
                m_pFileBuffer->AddRef();
            }
            // Set the new state
            m_ulState = kStateFileReadPending;
            // Read the first kDefaultPacketSize bytes of the file
            m_pFileObject->Read(kDefaultPacketSize);
        }
        else
        {
            // Set the state
            m_ulState = kStateFileClosePending;
            // Close the file
            m_pFileObject->Close();
        }
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    return retVal;
}

STDMETHODIMP CBrushFileFormat::WriteDone(HX_RESULT status)
{
    // We don't ever write, so we don't expect to get this...
    return HXR_UNEXPECTED;
}

STDMETHODIMP CBrushFileFormat::CloseDone(HX_RESULT status)
{
    HX_RESULT retVal = HXR_OK;

    if (m_ulState == kStateFileClosePending)
    {
        // We can't be initialized successfully if we
        // don't have a file buffer
        if (!m_pFileBuffer)
        {
            status = HXR_FAIL;
        }
        // Set the state
        m_ulState = (SUCCEEDED(status) ? kStateFileFormatInitialized : kStateError);
        // Send a stream done
        m_pFormatResponse->InitDone(status);
    }
    else
    {
        retVal = HXR_UNEXPECTED;
    }

    return retVal;
}

HX_RESULT STDAPICALLTYPE CBrushFileFormat::HXCreateInstance(IUnknown** ppIUnknown)
{
    HX_RESULT retVal = HXR_FAIL;

    if (ppIUnknown)
    {
        // Create the object
        CBrushFileFormat* pObj = new CBrushFileFormat();
        if (pObj)
        {
            // QI for IUnknown
            retVal = pObj->QueryInterface(IID_IUnknown, (void**) ppIUnknown);
        }
    }

    return retVal;
}

HX_RESULT STDAPICALLTYPE CBrushFileFormat::CanUnload2()
{
    return ((CHXBaseCountingObject::ObjectsActive() > 0) ? HXR_FAIL : HXR_OK );
}

void CBrushFileFormat::Deallocate()
{
    HX_RELEASE(m_pContext);
    HX_RELEASE(m_pFileObject);
    HX_RELEASE(m_pFormatResponse);
    HX_RELEASE(m_pCommonClassFactory);
    HX_RELEASE(m_pFileBuffer);
    Reset();
}

void CBrushFileFormat::Reset()
{
    m_pContext            = NULL;
    m_pFileObject         = NULL;
    m_pFormatResponse     = NULL;
    m_pCommonClassFactory = NULL;
    m_pFileBuffer         = NULL;
    m_ulState             = kStateConstructed;
    m_bNullBrush          = FALSE;
}

BOOL CBrushFileFormat::IsNullBrush(IHXRequest* pRequest)
{
    BOOL bRet = FALSE;

    if (pRequest)
    {
        // Get the request headers
        IHXValues* pReqHdrs = NULL;
        pRequest->GetRequestHeaders(pReqHdrs);
        if (pReqHdrs)
        {
            // See if there was a "NullBrush" flag
            IHXBuffer* pTmp = NULL;
            pReqHdrs->GetPropertyCString("NullBrush", pTmp);
            if (pTmp)
            {
                // Yes, there WAS a "NullBrush" flag
                bRet = TRUE;
            }
            HX_RELEASE(pTmp);
        }
        HX_RELEASE(pReqHdrs);
    }

    return bRet;
}

