/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: shmhelp.cpp,v 1.2.4.2 2004/07/09 12:48:45 pankajgupta Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

// for shared memory
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/utsname.h>

#include "hlxclib/string.h"
#include "shmhelp.h"
#include "hxassert.h"

//Work around AIX problem.
#ifdef _AIX
#  define MAX_SHARED_REGIONS ((UINT32)1)
#else
#  define MAX_SHARED_REGIONS ((UINT32)9999)
#endif

//Keeps track of how many shared memory segments we have used.
int      ShmHelp::zm_nSegment = 0;
Display* ShmHelp::zm_pDisplay = NULL;
BOOL     ShmHelp::zm_bUseShm  = FALSE;

void ShmHelp::Init(Display* pDisplay)
{
   HX_ASSERT( pDisplay );

   zm_pDisplay = pDisplay;
   zm_bUseShm  = FALSE;
   
   //See if our XServer supports ShmMemory....
#if defined(_AIX)  
   int majorVersion, minorVersion;
   Bool sharedPixmaps;
   // doesn't seem to work with multiple presentation panes...
   XLockDisplay(zm_pDisplay);
   zm_bUseShm = XShmQueryVersion( zm_pDisplay,
                                  &majorVersion,
                                  &minorVersion, &sharedPixmaps); 
   XUnlockDisplay(zm_pDisplay);
   if (zm_nSegment > AIX_MAX_SHARED_REGIONS) 
      zm_bUseShm = FALSE;
#else
#  ifdef _SOLARIS
   zm_bUseShm = FALSE;
#  else
   XLockDisplay(zm_pDisplay);   
   zm_bUseShm = XShmQueryExtension(zm_pDisplay);
   XUnlockDisplay(zm_pDisplay);
#  endif    
#endif    

   //Make sure we only use it if we are displaying locally.  The
   //DISPLAY string can start with ':', '0' or the machine name.
   if(zm_bUseShm)
   {
      char *disp = getenv("DISPLAY");
      HX_ASSERT( disp != NULL && "Your display variable isn't set!" );
      if(!disp)
      {
         zm_bUseShm = FALSE;
      }
      else if( disp[0]!=':' && disp[0]!='0' )
      {
         //See if our machine name is in the dipsplay string.
         struct utsname inf;
         uname(&inf);
         if( (strlen(disp)<=strlen(inf.machine)) ||
             (strncmp(disp, inf.machine, strlen(inf.machine))) )
         {
            zm_bUseShm = FALSE;
         }
      }
   }
}

//This ctor implies someone has called the one that sets the
//display var.
ShmHelp::ShmHelp()
{
   HX_ASSERT( zm_pDisplay );
}

BOOL ShmHelp::ShmAvailable()
{
   return zm_bUseShm;
}

HX_RESULT ShmHelp::CreateSharedRegion(INT32 nSize, UCHAR**ppMem, int* pnShmID, XShmSegmentInfo* pInfo ) 
{
    HX_RESULT theErr   = HXR_OK;
    int       nRetCode = 0;

    HX_ASSERT( ppMem );
    HX_ASSERT( pnShmID );

    //Make sure we were inited...
    HX_ASSERT(zm_pDisplay);
    if( !zm_pDisplay )
       return HXR_UNEXPECTED;

    *ppMem   = NULL;
    *pnShmID = -1;
    
    //get the shared memory segment, etc.
    if( !zm_bUseShm ) 
    {
	theErr = HXR_UNEXPECTED;
    }
    
#ifdef _AIX
    if (zm_nSegment > AIX_MAX_SHARED_REGIONS)
    {
	theErr = HXR_UNEXPECTED;
    }
#endif

    if( theErr == HXR_OK )
    {
        // Request a new shared memory ID
        *pnShmID = ::shmget(0, nSize, IPC_CREAT | 0777);
    
        if( *pnShmID == -1 ) 
        {
#ifdef _DEBUG
            perror("shmget");
#endif            
            theErr = HXR_FAIL;
        }
        else
        {
            // Attach shared memory segment to process's address space
            *ppMem = (UCHAR*)::shmat(*pnShmID, 0, 0);
            if( ((int)*ppMem) == -1) 
            {
                theErr = HXR_FAIL;
#ifdef _DEBUG            
                perror("shmat");
#endif            
            }
            else
            {
                //Request that segment be removed after last reference to it 
                //is released
                pInfo->shmseg   = ++zm_nSegment;
                pInfo->shmid    = *pnShmID;
                pInfo->shmaddr  = (char*)*ppMem;
                pInfo->readOnly = True;
		XLockDisplay(zm_pDisplay);
                nRetCode = XShmAttach(zm_pDisplay, pInfo);
                XSync(zm_pDisplay, False);
		XUnlockDisplay(zm_pDisplay);
                ::shmctl(*pnShmID, IPC_RMID, NULL); 
    
                if( !nRetCode )
                {
                    theErr = HXR_FAIL;
                }
                else
                {
                    theErr = HXR_OK;
                }
            }
        }

        if( HXR_OK != theErr )
        {
            // in case of failure, make a best case effort to detach the region
            ::shmctl(*pnShmID, IPC_RMID, NULL);
            zm_bUseShm = FALSE;
        }
    }
    return theErr;
}

HX_RESULT ShmHelp::DetachSharedRegion(UCHAR**ppMem, XShmSegmentInfo* pInfo ) 
{
   HX_ASSERT(zm_bUseShm);
   HX_ASSERT(pInfo);
   HX_ASSERT(*ppMem);
   
   //Make sure we were inited...
   HX_ASSERT(zm_pDisplay);
   if( !zm_pDisplay )
      return HXR_UNEXPECTED;
   
   if(!zm_bUseShm)
      return HXR_UNEXPECTED;
   XLockDisplay(zm_pDisplay);
   XShmDetach(zm_pDisplay, pInfo);
   XUnlockDisplay(zm_pDisplay);
   
   ::shmdt((char*)*ppMem);
   *ppMem = NULL;
   
   --zm_nSegment;
   
   return HXR_OK;
}
