/***************************************************************************
                          codeparser.h  -  description
                             -------------------
    begin                : Die Jul 9 2002
    copyright            : (C) 2002 by André Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CODEPARSER_H
#define CODEPARSER_H

#include <iostream>
#include <sstream>
#include <string>
#include <iomanip>
#include <cctype>

#include "languagedefinition.h"
#include "documentstyle.h"
#include "ASFormatter.h"
#include "linewrapper.h"
#include "enums.h"

#define NUMBER_OF_STYLES 12

#define LINE_NUMBER_WIDTH 5

namespace highlight {

/** Class which contains parsing methods. Works similar to a Turing machine.
   @author André Simon
  */

class CodeParser
  {

  public:
    CodeParser();

    //! CodeParser Konstruktor
    /** \param langDef language definition which contains parsing information 
               about the programming language
        \param printLineNumbers Test if line numbers should be printed
        \param colourTheme Name of coloring style being used
        \param numSpaces Number of spaces that replace a tab
        \param lineWrappingStyle Line wrapping mode (WRAP_DISABLED, WRAP_SIMPLE, WRAP_DEFAULT)
        \param fragment Test if header and footer of document should be omitted
    */
    CodeParser(bool printLineNumbers,
               const string &colourTheme, 
               int numSpaces, 
               WrapMode lineWrappingStyle, 
               bool fragment);    

    virtual ~CodeParser();

    
    /**
     Generates output
     \param inFileName Path of input file (if empty use stdin)
     \param outFileName Path of input file (if empty use stdout)
     \return ParseError
    */
    ParseError printOutput (const string & inFileName, const string &outFileName);
       
    /** Resets parser to origin state*/
    void reset();
    
    /** \return True if document style was found*/
    bool styleValid(); 
    
    /** \param langDefPath Absolute path to language definition
        \return  Failure: LOAD_FAILED; Reload necessary: LOAD_NEW, 
                 no reload necessary:  LOAD_NONE */
    LoadResult setLanguage(const string& langDefPath); 
    
    /** \return Language definition*/
    LanguageDefinition &getLanguage();
    
    /** \param style Formatting style
        \param suffix File suffix
        \return True if formatting is enabled  */
    bool enableReformatting(FormatStyle style, const string&suffix);

protected:    

    /** \param c Character to be masked
        \return Escape sequence of output format */
    virtual string maskCharacter(unsigned char c) = 0;

    /* \param s string 
       \return Copy of s with all escaped characters */
    string maskString(const string &s ) ;

    /** \param s Symbol string
        \param searchPos Position where search starts
        \return Found state (integer value)  */
    State getState(const string &s, unsigned int searchPos);

    /** \return Next identifier in current line of code */
    string getIdentifier();

    /** \return Next number in current line of code */
    string getNumber();   

    /** Insert line number at the beginning of current output line */
    virtual void insertLineNumber(bool insertNewLine=true);

    /** Prints document footer*/
    virtual string getFooter() = 0;

    /** Prints document body*/
    virtual void printBody() = 0;

    /** prints document header
       \param  title Title of the document
    */
    virtual string getHeader(const string &title) = 0;    

    /** Get current line number
      \return line number  */
    unsigned int getLineNumber();
    
    


    /** Tag Delimiters for every colour style*/
    string styleTagOpen[NUMBER_OF_STYLES],
           styleTagClose[NUMBER_OF_STYLES];

    /** Description of document colour style*/
    DocumentStyle docStyle;

    /** Language definition*/
    LanguageDefinition langInfo;

    /** Tag for inserting line feeds*/
    string newLineTag;

    /** String that represents a white space in output */
    string spacer;

    /** In- and output streams*/
    istream *in;
    ostream *out;

    /** Tags which enclose white space indentation blocks */
    string maskWsBegin, maskWsEnd;

    /** Test if maskWsBegin and maskWsEnd should be applied*/
    bool maskWs;
    bool excludeWs;

    /** Test if header and footer should be omitted*/
    bool fragmentOutput;

    /* Test if line numbers should be printed*/
    bool showLineNumbers;   

    /** Current line of input file*/
    string line;
    
    /** enthält header unf footer zur schnelleren Erzeugung*/
    string headerCache, footerCache; 

    /** Curent number of lines*/
    unsigned int lineNumber;

    // Zeigt den aktuellen Zustand an
    // wird nicht in getCurrentState gesetzt, da nur Zustände interessant
    // sind, die als Index auf die styleCloseTags und styleOpenTags verwendet
    // werden können
    /** Current state*/
    State currentState;

    /** Processes origin state */
    void processStandardState();
    
    /** gibt String zum Zeilenumbruch aus*/  
    virtual string getNewLine();

private:
    
    //Name des FarbStils
    string  styleName;

    // Anzahl von Leerzeichen, die ein Tab ersetzen sollen
    int numberSpaces;

    // Zaehler, der aktuelle Position in der geparsten Codezeile enthaelt
    unsigned int lineIndex;         
    
    /**letztes Zeichen der letzten gelesenen Zeile*/
    unsigned char terminatingChar;

    /** AStyle Reformatter*/
    astyle::ASFormatter *formatter;
    
    /** Klasse zum Umbrechen langer Zeilen*/
    LineWrapper *lineWrapper;      

    /** enthalt das aktuell geparste Token*/
    string token;       

    /** liest Zeile von InputStream ein*/
    bool readNewLine(string &newLine);
    
    /** liefert den n?hsten Character aus dem Inputstream */
    unsigned char getInputChar();    

    /** liefert den letzten gelesenen Character */
    unsigned char getLastChar();    

   /** liefert den Zustand zurueck, der durch das aktuelle Token eingeleitet 
       wird. Ist token leer, wird in der aktuellen Eingabezeile gesucht, sonst 
       wird der interne Zeilenindex zurueckgesetzt, um das Token erneut zu 
       parsen.
       Parameter: bool lastStateWasNumber:
                  true, falls der letzte Zustand gleich Number war (zum Erkennen
                  von Termen wie a-b)
       Return:    naechsten Zustand
                  token: Erkanntes Token, das den neuen Zustand einleitet */
    State getCurrentState ( bool lastStateWasNumber=false);

    /** Methoden zur Verarbeitung der einzelnen Zustaende*/
    bool processKeywordState(State myState=KEYWORD) ;
    bool processTypeState(State myState=TYPE) ;
    bool processNumberState() ;
    bool processMultiLineCommentState();
    bool processSingleLineCommentState();
    bool processStringState(State oldState);
    bool processEscapeCharState();
    bool processDirectiveState();
    bool processTagState();
    bool processSymbolState();
    void processWsState();

    /** gibt true zurck, falls c ein erlaubter Character innerhalb von Keyword 
        oder Typbezeichner ist */
    bool isAllowedChar(char c) ;

    /** gibt true zurck, das aktuelle Token das erste in der Zeile und kein 
        Whitespace ist */
    bool isFirstNonWsChar() ;

    /** gibt maskierten Token aus und setzt token auf ""*/
    void printMaskedToken();

    /** gibt Escapesequenz aus*/
    void skipEscapeSequence();
    
    void openTag(State s);

    void closeTag(State s);
    
   /** Sets formatter to new format style
       \param style New formatting style */
    void setFormatter(FormatStyle style);
    
    
  };
}

#endif
