/////////////////////////////////////////////////////////////////////////////
//	file		:	dirstream.inl
//  copyright	:	(C) 2002 by Benjamin Kaufmann
//  email		:	hume@c-plusplus.de
//	internet	:	http://bens.c-plusplus.info/
//
//	Implementationsdatei fr den Dirstream.
//
//	Die POSIX-Wrapper fr die Win32-Verzeichnisfunktionen sind nicht von mir!
//	Bitte beachten Sie die Copyright-Hinweise von Kevlin Henney!
//
//	Weitere Informationen zu den Wrapper-Funktionen gibt es hier:
//	http://www.two-sdg.demon.co.uk/curbralan/code/dirent/dirent.html
//
/////////////////////////////////////////////////////////////////////////////
//
/****************************************************************************
 *                                                                         	*
 *	This program is free software; you can redistribute it and/or modify  	*
 *	it under the terms of the GNU General Public License as published by  	*
 *	the Free Software Foundation; either version 2 of the License, or     	*
 *	(at your option) any later version.                                   	*
 *																			*
 * 	This program is distributed in the hope that it will be useful,			*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of			*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *	GNU General Public License for more details.                           	*
 *																			*
 ****************************************************************************/

#include <stack>

#ifdef _WIN32

	#include <sys/types.h>
	#include <sys/stat.h>

	// Abbildung der bentigten Funktionen/Strukturen auf die
	// entsprechenden Windows-Geschichten.
	#define stat _stat

	inline  int lstat(const char *file_name, struct stat *buf)
	{
		return _stat(file_name,buf);
	}

	// einige Compiler definieren bereits ein Makro S_ISDIR.
	// Falls dem nicht so ist, wird hier eine inline-Funktion
	// definiert.
	#ifndef S_ISDIR
	inline bool S_ISDIR(unsigned short Mode)
	{
		return (Mode & _S_IFDIR) == _S_IFDIR;
	}
	#endif


	namespace DIRSTREAM
	{
		const char DIR_DELIMITER = '\\';
	}

	/*

    Implementation of POSIX directory browsing functions and types for Win32.

    Kevlin Henney (mailto:kevlin@acm.org), March 1997.

    Copyright Kevlin Henney, 1997. All rights reserved.

    Permission to use, copy, modify, and distribute this software and its
    documentation for any purpose is hereby granted without fee, provided
    that this copyright and permissions notice appear in all copies and
    derivatives, and that no charge may be made for the software and its
    documentation except to cover cost of distribution.

    This software is supplied "as is" without express or implied warranty.

    But that said, if there are any problems please get in touch.
	*/

	// ============================= Hinweis =============================
	//
	// Alle Anweisungen die innerhalb eines USE_SDK_FIND_FUNCTIONS
	// Blocks stehen, wurden von mir hinzugefgt!
	//
	// Ist USE_SDK_FIND_FUNCTIONS definiert, so werden statt der Funktionen
	// _findfirst, _findnext und _findclose die Funktionen FindFirstFile,
	// FindNextFile und FindClose verwendet. Dadurch kann
	// auf den sehr teuren stat-Aufruf verzichtet werden. Diese Variante ist
	// deutlich schneller (bei mir bis zu Faktor 40), hat aber den Nachteil,
	// dass nicht alle Verzeichnisse bercksichtigt werden (wie z.B.
	// WINDOWS\Temporary Internet Files)
	//
	// ===================================================================

	#include <errno.h>
	#ifdef USE_SDK_FIND_FUNCTIONS
		#include <windows.h>
	#else
		#include <io.h>
	#endif
	#include <stdlib.h>
	#include <string.h>
	typedef struct DIR DIR;

	struct dirent
	{
		char *d_name;
	};

	DIR*			opendir(const char *);
	int				closedir(DIR *);
	struct dirent*	readdir(DIR *);

	struct DIR
	{
		#ifdef USE_SDK_FIND_FUNCTIONS
			HANDLE			handle;
			WIN32_FIND_DATA	info;
		#else
			long                handle;		// -1 for failed rewind
			struct _finddata_t  info;
		#endif
		struct dirent   result;		// d_name null iff first time
		char*			name;		// NTBS
	};

	DIR* opendir(const char* name)
	{
		DIR *dir = 0;

		if(name && name[0])
		{
			size_t base_length = strlen(name);
			const char* all = // the root directory is a special case...
				strchr("/\\", name[base_length - 1]) ? "*" : "/*";

			if((dir = (DIR *) malloc(sizeof *dir)) != 0 &&
			   (dir->name = (char *) malloc(base_length + strlen(all) + 1)) != 0)
			{
				strcat(strcpy(dir->name, name), all);
				#ifdef USE_SDK_FIND_FUNCTIONS
				if((dir->handle = FindFirstFile(dir->name, &dir->info)) != INVALID_HANDLE_VALUE)
				{
					dir->result.d_name = 0;
				}
				#else
				if((dir->handle = _findfirst(dir->name, &dir->info)) != -1)
				{
					dir->result.d_name = 0;
				}
				#endif
				else // rollback
				{
					free(dir->name);
					free(dir);
					dir = 0;
				}
			}
			else // rollback
			{
				free(dir);
				dir   = 0;
				errno = ENOMEM;
			}
		}
		else
		{
			errno = EINVAL;
		}

		return dir;
	}

	int closedir(DIR *dir)
	{
		int result = -1;

		if(dir)
		{
			#ifdef USE_SDK_FIND_FUNCTIONS
			if(dir->handle != INVALID_HANDLE_VALUE)
			{
				result = FindClose(dir->handle);
			}
			#else
			if(dir->handle != -1)
			{
				result = _findclose(dir->handle);
			}
			#endif
			free(dir->name);
			free(dir);
		}

		if(result == -1) // map all errors to EBADF
		{
			errno = EBADF;
		}

		return result;
	}

	struct dirent *readdir(DIR *dir)
	{
		struct dirent *result = 0;
		#ifdef USE_SDK_FIND_FUNCTIONS
		if(dir && dir->handle != INVALID_HANDLE_VALUE)
		{
			if( !dir->result.d_name || FindNextFile(dir->handle, &dir->info))
			{
				result         = &dir->result;
				result->d_name = dir->info.cFileName;
			}
		}
		#else
		if(dir && dir->handle != -1)
		{
			if(!dir->result.d_name || _findnext(dir->handle, &dir->info) != -1)
			{
				result         = &dir->result;
				result->d_name = dir->info.name;
			}
		}
		#endif
		else
		{
			errno = EBADF;
		}

		return result;
	}

	/* END Implementation of POSIX directory browsing functions */

#else
	// Wenn vorhanden, POSIX Funktionen verwenden.
	#include <dirent.h>
	#include <sys/stat.h>
	namespace DIRSTREAM
	{
		const char DIR_DELIMITER = '/';
	}
#endif

namespace DIRSTREAM
{
	template <class FSelect, class DSelect>
	class DirStreamImpl
	{
		public:
			DirStreamImpl(const char* Dn, bool Re, const FSelect& Fs, const DSelect& Ds ) :
					m_FileSelect(Fs), m_DirSelect(Ds), m_Recurse(Re),
					m_DirName(Dn), m_TheDir(opendir(Dn)), m_CurrEntry(0),
					m_Info(new struct stat)
			{
				if (!m_DirName.empty() && *m_DirName.rbegin() != DIR_DELIMITER)
				{
					m_DirName += DIR_DELIMITER;
				}

			}
			~DirStreamImpl()
			{
				delete m_Info;
			}

			operator const void* () const
			{
				return m_TheDir;
			}

			DirStreamImpl& operator >> (std::string& Dest);
			bool IsDirectory() const;
			bool OpenDir(	const char* DirName, bool Rec,
							const FSelect& Fs, const DSelect& Ds
						);
			bool IsOpen() const {return m_TheDir != 0;}

		private:
			FSelect					m_FileSelect;
			DSelect					m_DirSelect;
			bool 					m_Recurse;
			std::stack<std::string>	m_Dirs;
			std::string				m_DirName;
			std::string				m_RecurseDirName;
			DIR*					m_TheDir;
			dirent*					m_CurrEntry;
			struct stat*			m_Info;

			void CloseDir()
			{
				if (m_TheDir)
				{
					closedir(m_TheDir);
					m_TheDir = 0;
					m_CurrEntry = 0;
				}
			}
	};

	template <class FS, class DS>
	DirStreamImpl<FS, DS>& DirStreamImpl<FS, DS>::operator >> (std::string& Dest)
	{
		// Es muss ja nicht immer Rekursion sein ;-)
		while (true)
		{
			if ( !m_TheDir || !(m_CurrEntry  = readdir(m_TheDir)) )
			{	// Kein weiterer Eintrag im Verzeichnis gefunden
				// Verzeichnis schlieen
				CloseDir();
				if (m_Recurse && !m_Dirs.empty())
				{	// nchstes Unterverzeichnis durchbrowsen
					// Der neue Verzeichnisname setzt sich aus dem
					// Basisverzeichnis und dem Unterverzeichnis zusammen.
					std::string NewDir(m_DirName);

					NewDir += m_Dirs.top();
					m_RecurseDirName = m_Dirs.top();
					m_Dirs.pop();
					m_TheDir = opendir(NewDir.c_str());
				}
				else
				{	// keine weiteren Verzeichnisse mehr
					return *this;
				}
			}
			else
			{	// Verzeichnis enthlt einen Eintrag.
				// Dieser wird nun berprft.
				const char* CurrEntry = m_CurrEntry->d_name;
				bool IsDir = IsDirectory();

				if ( IsDir && m_Recurse && std::string(".") !=  CurrEntry &&
					 std::string("..") != CurrEntry && m_DirSelect(CurrEntry)
					)
				{	// Unterverzeichnis hinzufgen
					std::string NewDir(m_RecurseDirName);
					if (!NewDir.empty() && *NewDir.rbegin() != DIR_DELIMITER)
					{
						NewDir += DIR_DELIMITER ;
					}
					NewDir += m_CurrEntry->d_name;
					NewDir += DIR_DELIMITER ;
					m_Dirs.push(NewDir);
				}

				if (m_FileSelect(m_CurrEntry->d_name, IsDir))
				{	// Wir wollen den aktuellen Eintrag
					// Datei-/Verzeichnisname zurckliefern.
					// Relativ zum Startverteichnis.
					Dest = m_RecurseDirName;
					Dest += m_CurrEntry->d_name;
					break;

				}
			}

		}
		return *this;
	}

	template <class FS, class DS>
	bool DirStreamImpl<FS, DS>::IsDirectory() const
	{
		if (!m_CurrEntry) return false;

		#ifdef _WIN32
			#ifdef USE_SDK_FIND_FUNCTIONS
				return	m_TheDir->info.dwFileAttributes == FILE_ATTRIBUTE_DIRECTORY ||
						m_TheDir->info.dwFileAttributes == 17;
			#endif
		#endif
		std::string Temp(m_DirName);
		Temp += m_RecurseDirName;
		char ch = *Temp.rbegin();

		if ( ch != DIR_DELIMITER) Temp += DIR_DELIMITER ;
		Temp += m_CurrEntry->d_name ;
		lstat (Temp.c_str() , m_Info);
		return S_ISDIR(m_Info->st_mode);
	}

	template <class FS, class DS>
	bool DirStreamImpl<FS, DS>::OpenDir(const char* DirName, bool Rec,
										const FS& Fs, const DS& Ds)
	{
		CloseDir();
		if (! (m_TheDir = opendir(DirName)) ) return false;
		while (!m_Dirs.empty()) m_Dirs.pop();
		m_DirName = DirName;

		if (!m_DirName.empty() && *m_DirName.rbegin() != DIR_DELIMITER)
		{
			m_DirName += DIR_DELIMITER;
		}

		m_RecurseDirName = "";
		m_FileSelect = Fs;
		m_DirSelect  = Ds;
		m_Recurse = Rec;
		return true;
	}


	template <class FS, class DS>
	DirStream_t<FS, DS>::DirStream_t(const char* Name,  bool Rec, const FS& Fs, const DS& Ds) :
		m_pImpl(new DirStreamImpl<FS, DS>(Name, Rec, Fs, Ds))
	{}

	template <class FS, class DS>
	DirStream_t<FS, DS>::~DirStream_t()
	{
		delete m_pImpl;
	}

	template <class FS, class DS>
	DirStream_t<FS, DS>::operator const void* () const
	{
		return m_pImpl->operator const void*();
	}

	template <class FS, class DS>
	DirStream_t<FS, DS>& DirStream_t<FS, DS>::operator >> (std::string& Source)
	{
		*m_pImpl >> Source;
		return *this;
	}

	template <class FS, class DS>
	bool DirStream_t<FS, DS>::directory() const
	{
		return m_pImpl->IsDirectory();
	}

	template <class FS, class DS>
	bool DirStream_t<FS, DS>::open(const char* DirName, bool Rec, const FS& Fs, const DS& Ds)
	{
		return m_pImpl->OpenDir(DirName, Rec, Fs, Ds);
	}

	template <class FS, class DS>
	bool DirStream_t<FS, DS>::is_open() const
	{
		return m_pImpl->IsOpen();
	}
}	// Ende des Namespaces DIRSTREAM
