/////////////////////////////////////////////////////////////////////////////
//	file		:	cmdline.cpp
//  copyright	:	(C) 2002 by Benjamin Kaufmann
//  email		:	hume@c-plusplus.de
//	internet	:	http://bens.c-plusplus.info/?page=cpp2html
//
//	implementation file for the command line parser
//
// 08.08.2003: fixed typo in line 225,Andre Simon andre.simon@gmx.de
/////////////////////////////////////////////////////////////////////////////
//
/****************************************************************************
 *                                                                         	*
 *	This program is free software; you can redistribute it and/or modify  	*
 *	it under the terms of the GNU General Public License as published by  	*
 *	the Free Software Foundation; either version 2 of the License, or     	*
 *	(at your option) any later version.                                   	*
 *										*
 * 	This program is distributed in the hope that it will be useful,		*
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the		*
 *	GNU General Public License for more details.                           	*
 *									*
 ****************************************************************************/

#include "win32cmdline.h"

using namespace std;

namespace
{
	bool isPrefix(const std::string& str, const std::string& prefix)
	{
		if (str.length() < prefix.length()) return false;
		string substr(str, 0, prefix.length());
		return substr == prefix;
	}
}

CommandLineParser::CommandLineParser():
argCount(0){
}

CommandLineParser::Option::Option(char Short, const string& Long,
								  const string& DefVal, bool WantVal)
	: expectsValue_(WantVal)
	, longName_(Long)
	, defaultValue_(DefVal)
	, value_(DefVal)
{
	shortName_ += Short;
	if (Long == "") longName_ = shortName_;
}


string& CommandLineParser::Option::setValue(string& Str)
{
	if (Str == "")
	{
		string Msg("Option: '");
		Msg += getLongName();
		Msg += "' expects a value!";
		throw ArgumentParseException(Msg);
	}
	value_ = Str;
	return Str;
}

CommandLineParser::IntOption::IntOption(char Short, const string& Long,
										const string& DefVal)
	: Option(Short, Long, DefVal, true)
{}

string& CommandLineParser::IntOption::setValue(std::string& Str)
{
	if (Str.find_first_not_of("1234567890") != string::npos)
	{
		string Msg("Option: '");
		Msg += getLongName();
		Msg += "' expects an integer value!";
		throw ArgumentParseException(Msg);
	}
	Option::setValue(Str);
	return Str;
}


CommandLineParser::~CommandLineParser()
{
	OptionMap::iterator Beg = options_.begin();
	OptionMap::iterator End = options_.end();
	while (Beg != End)
	{
		if (Beg->first.find("--") != string::npos)
		{
			delete Beg->second;
			Beg->second = 0;
		}
		++Beg;
	}
	options_.clear();
}

void CommandLineParser::addOption(Option* p)
{
	string Short(p->getShortName());
	string Long(p->getLongName());
	if (Short[0] != '-') Short.insert(0, "-");
	if (Long.find("--") != 0) Long.insert(0, "--");
	options_.insert(OptionMap::value_type(Short, p));
	options_.insert(OptionMap::value_type(Long, p));
}
void CommandLineParser::addIntOption(char ShortName,
									 const string& LongName,
									 const string& DefVal /* = "-1" */)
{
	Option* p = new IntOption(ShortName, LongName, DefVal);
	addOption(p);

}

void CommandLineParser::addStringOption(char ShortName,
										const string& LongName,
										const string& DefVal /* = "" */)
{
	Option* p = new Option(ShortName, LongName, DefVal, true);
	addOption(p);

}

void CommandLineParser::addSwitchOption(char ShortName, const std::string& LongName)
{
	Option* p = new Option(ShortName, LongName, "false", false);
	addOption(p);
}

bool CommandLineParser::hasOption(char ShortOptionName) const {
        std::string option;
        option+=ShortOptionName;
        return hasOption(option);
}


bool CommandLineParser::hasOption(const string& OptionName) const
{
	string Op(OptionName);
	if (Op.length() > 1)
		Op.insert(0, "--");
	else
		Op.insert(0, "-");

	OptionMap::const_iterator It = options_.find(Op);
	if (It == options_.end())
	{
		string Msg("'");
		Msg += OptionName;
		Msg += "'";
		Msg += " is not a valid argument for HasOption";
		Msg += "\nNo such option defined!";
		throw invalid_argument(Msg);
	}

	Option* p = It->second;
	if (p->getValue() != p->getDefValue())
	{
		return true;
	}
	return false;
}


const string& CommandLineParser::getAt(const string& OptionName) const
{
	string Op(OptionName);
	if (Op.length() > 1)
		Op.insert(0, "--");
	else
		Op.insert(0, "-");

	OptionMap::const_iterator It = options_.find(Op);
	if (It == options_.end())
	{
		string Msg("'");
		Msg += OptionName;
		Msg += "'";
		Msg += " is not a valid argument for operator[]";
		Msg += "\nNo such option defined!";
		throw invalid_argument(Msg);
	}

	Option* p = It->second;
	if (p->getValue() != p->getDefValue())
	{
		return p->getValue();
	}
	return It->second->getDefValue();
}

ValueProxy CommandLineParser::operator[] (const string& OptionName) const
{
	return ValueProxy(getAt(OptionName));
}

ValueProxy CommandLineParser::operator[] (char ShortOptionName) const
{
        std::string option;
        option+=ShortOptionName;
	return ValueProxy(getAt(option));
}

 unsigned int CommandLineParser::getArgCount(){
  return argCount;
 }

void CommandLineParser::parse(int& argc, char** argv)
{
	for (int Run = 0 ; Run < argc ; )
	{
		bool WasOption = false;
		bool NoSpaceBetweenOptAndVal = false;
		string Current(argv[Run]);

		if (isPrefix(Current, "-"))
		{	// Option gefunden
			string Value("");

			if (isPrefix(Current, "--"))
			{	// Eine Option die mit -- eingeleitet wird wurde gefunden.
				if (Current == "--")
				{	// Endezeichen fr Optionen gefunden. Entfernen und raus
					int TempRun = Run + 1;
					// <= damit auch das terminierende 0-Array kopiert wird
					for (int j = Run ; TempRun <= argc;)  argv[j++] = argv[TempRun++];
					--argc;
					return;
				}
				string::size_type PosEq = Current.find("=");
				if (PosEq != string::npos)
				{	// Die Option hat die Form --arg=value
					// Die Option heit also --arg. Der Wert ist der String
					// nach dem Gleichheitszeichen.
					Value = Current.substr(PosEq + 1);
					Current = Current.substr(0, PosEq);
					NoSpaceBetweenOptAndVal = true;
					
				}
			}
			// Prfen ob die gefundene Option gewnscht ist
			OptionMap::iterator It = options_.find(Current);
			if (It == options_.end())
			{	// Gefunde Option ist unerwartet
				string Msg("Unexpected option: ");
				Msg += Current;
				throw ArgumentParseException(Msg);
			}
			bool WantsValue = It->second->expectsValue();
			if (WantsValue)
			{
				if (Value == "")
				{
					if ( Run + 1 < argc) Value = argv[Run + 1];
				}
				It->second->setValue(Value);				
			}
			else
			{	// gefundene Aktion bentigt keinen Wert. Das true
				// markiert, dass sie gefunden wurde.
				string True("true");
				It->second->setValue(True);
			}

			// Die Option wurde berarbeitet. Sie wird nun aus der
			// Kommandozeile entfernt.
			int TempRun = Run;
			if (!WantsValue || NoSpaceBetweenOptAndVal)
   				TempRun = Run + 1;	// Option bestand aus einem Teil				
			else
				TempRun = Run + 2;	// Option bestand aus Name Wert
			// <= damit auch das terminierende 0-Array kopiert wird
			for (int j = Run ; TempRun <= argc;) 
                 argv[j++] = argv[TempRun++];

			if (!WantsValue || NoSpaceBetweenOptAndVal)
				--argc;
			else
				argc -= 2;
			WasOption = true;
		}
		if (!WasOption) ++Run;
	}

}

