/***************************************************************************
                          cmdlineoptions.cpp  -  description
                             -------------------
    begin                : Sun Nov 25 2001
    copyright            : (C) 2001 by Andr Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmdlineoptions.h"

using namespace std;

/* Siehe man getopt (3)
   Konstruktor legt Optionen und Argumente fest
*/
CmdLineOptions::CmdLineOptions(int argc, char *argv[]):
    numberSpaces(0),
    wrappingStyle(highlight::WRAP_DISABLED),
    outputType (highlight::HTML),
    cssOutFilename("highlight.css"),
    opt_language (false),
    opt_include_css (false),
    opt_help (false),
    opt_version (false),
    opt_verbose (false),
    opt_linenumbers (false),
    opt_css_style (false),
    opt_batch_mode (false),
    opt_fragment (false) ,
    opt_attach_line_anchors (false),
    opt_show_themes (false),
    opt_show_langdefs (false),
    opt_printindex(false),
    opt_quiet(false),
    opt_xslfo_fop(false),
    opt_replacequotes(false),
    opt_print_progress(false),
    opt_fill_zeroes(false),
    configFileRead(false),
    helpLang("en")
{
/*
  #if defined (_WIN32)
   char pathAndName[MAX_PATH], path[MAX_PATH], drive[3];
   ::GetModuleFileName(NULL, pathAndName, MAX_PATH);
    _splitpath(pathAndName, drive, path, 0, 0);
    appPath = std::string(drive)+path;

  appPath= string(argv[0]);
  string::size_type Pos = appPath.find_last_of('\\');
  appPath = appPath.substr(0, Pos+1);

  #endif
*/
  loadConfigurationFile();

  int c, option_index = 0;
  static struct option long_options[] =
      {
        {OPT_OUT, 1, 0, S_OPT_OUT},
        {OPT_IN, 1, 0, S_OPT_IN},
        {OPT_SYNTAX, 1, 0, S_OPT_SYNTAX},
        {OPT_VERBOSE, 0, 0, S_OPT_VERBOSE},
        {OPT_INCLUDECSS, 0, 0, S_OPT_INCLUDECSS},
        {OPT_HELP, 0, 0, S_OPT_HELP},
        {OPT_HELPINT, 1, 0, S_OPT_HELPINT},
        {OPT_LINENO,0,0,S_OPT_LINENO},
        {OPT_STYLE, 1,0,S_OPT_STYLE},
        {OPT_CSSOUT, 1, 0,S_OPT_CSSOUT},
        {OPT_CSSIN, 1, 0,S_OPT_CSSIN},
        {OPT_DELTABS,1,0,S_OPT_DELTABS},
        {OPT_XHTML, 0,0,S_OPT_XHTML},
        {OPT_RTF, 0,0,S_OPT_RTF},
        {OPT_TEX,0, 0,S_OPT_TEX},
        {OPT_LATEX,0, 0,S_OPT_LATEX},
        {OPT_XSLFO,0, 0,S_OPT_XSLFO},
        {OPT_ANSI,0, 0,S_OPT_ANSI},
        {OPT_BATCHREC,1,0,S_OPT_BATCHREC},
        {OPT_FRAGMENT,0,0,S_OPT_FRAGMENT},
        {OPT_ANCHORS, 0,0,S_OPT_ANCHORS },
        {OPT_LISTTHEMES, 0,0,S_OPT_LISTTHEMES },
        {OPT_LISTLANGS, 0,0,S_OPT_LISTLANGS },
        {OPT_OUTDIR,1,0,S_OPT_OUTDIR},
        {OPT_VERSION,0,0,0},
        {OPT_FORMATSTYLE,1,0,S_OPT_FORMATSTYLE},
        {OPT_DATADIR,1,0,S_OPT_DATADIR},
        {OPT_ADDDATADIR,1,0,S_OPT_ADDDATADIR},
        {OPT_INDEXFILE,0,0,S_OPT_INDEXFILE},
        {OPT_WRAP,0,0,S_OPT_WRAP},
        {OPT_WRAPSIMPLE,0,0,S_OPT_WRAPSIMPLE},
        {OPT_QUIET,0,0,S_OPT_QUIET},
        //remove as soon as APAche fixes the bug in FOP (0.20.5)
        {OPT_FOP,0,0,S_OPT_FOP},
        {OPT_REPLACE_QUOTES,0,0,S_OPT_REPLACE_QUOTES},
        {OPT_PROGRESSBAR,0,0,S_OPT_PROGRESSBAR},
        {OPT_FILLZEROES,0,0,S_OPT_FILLZEROES},
        {0, 0, 0, 0}
      };

  while (1)
    {
      c = getopt_long (argc, argv,S_OPTIONS_STRING,long_options, &option_index);
      if (c == -1)
        break;

      switch (c)
        {
        case 0:   //nur long options
          if (long_options[option_index].name==OPT_VERSION)
            {
              opt_version = true;
            }
          break;
        case S_OPT_OUT:
          outFilename=string(optarg);
          break;
        case S_OPT_IN:
          inputFileNames.push_back(string(optarg));
          break;
        case S_OPT_CSSOUT:
          cssOutFilename=string(optarg);
          break;
        case S_OPT_CSSIN:
          cssInFilename=string(optarg);
          break;
        case S_OPT_VERBOSE:
          opt_verbose = true;
          break;
        case S_OPT_QUIET:
          opt_quiet = true;
          break;
        case S_OPT_INCLUDECSS:
          opt_include_css = true;
          break;
        case S_OPT_HELPINT:
          helpLang=string(optarg);
        case S_OPT_HELP:
          opt_help = true;
          break;
        case S_OPT_LINENO:
          opt_linenumbers = true;
          break;
        case '?':
          opt_help = true;
          break;
        case S_OPT_STYLE:
          styleName=string(optarg);
          opt_css_style = true;
          break;
        case S_OPT_SYNTAX:
          language=string(optarg);
          opt_language = true;
          break;
        case S_OPT_DELTABS:
          numberSpaces = StringTools::str2int (string(optarg));
          break;
        case S_OPT_XHTML:
          outputType=highlight::XHTML;
          break;
        case S_OPT_RTF:
          outputType=highlight::RTF;
          break;
        case S_OPT_TEX:
          outputType=highlight::TEX;
          break;
        case S_OPT_LATEX:
          outputType=highlight::LATEX;
          break;
        case S_OPT_XSLFO:
          outputType=highlight::XSLFO;
          break;
        case S_OPT_ANSI:
          outputType=highlight::ANSI;
          break;
        case S_OPT_BATCHREC:
          opt_batch_mode = true;
          readDirectory(string(optarg));
          break;
        case S_OPT_FRAGMENT:
          opt_fragment = true;
          break;
        case S_OPT_ANCHORS:
          opt_attach_line_anchors = true;
          break;
        case S_OPT_LISTTHEMES:
          opt_show_themes = true;
          break;
        case S_OPT_LISTLANGS:
          opt_show_langdefs = true;
          break;
        case S_OPT_OUTDIR:
          outDirectory = validateDirPath(string(optarg));
          break;
        case S_OPT_FORMATSTYLE:
          indentScheme =string(optarg);
          break;
        case S_OPT_DATADIR:
          dataDir=validateDirPath(string(optarg));
          break;
        case S_OPT_ADDDATADIR:
          additionalDataDir=validateDirPath(string(optarg));
          break;
         case S_OPT_INDEXFILE:
          opt_printindex=true;
          break;
         case S_OPT_WRAPSIMPLE:
          wrappingStyle = highlight::WRAP_SIMPLE;
          break;
         case S_OPT_WRAP:
          wrappingStyle = highlight::WRAP_DEFAULT;
          break;
         case S_OPT_FOP:
          opt_xslfo_fop=true;
          break;
         case S_OPT_REPLACE_QUOTES:
          opt_replacequotes=true;
          break;
         case S_OPT_PROGRESSBAR:
          opt_print_progress=true;
          break;
         case S_OPT_FILLZEROES:
          opt_fill_zeroes=true;
          break;

        default:
          cerr <<"higlight: Unknown option " <<c<< endl;
        }
    }

  if (optind < argc)   //still args left
    {
      if  (inputFileNames.empty()){
        while (optind < argc){
          inputFileNames.push_back(string(argv[optind++]));
         }
      }
    } else if (inputFileNames.empty()){
       inputFileNames.push_back("");
    }
  if (printDebugInfo() && configFileRead) {
    cout << "Configuration file \""<<configFilePath<<"\" was read.\n";
  }
}

CmdLineOptions::~CmdLineOptions(){
}

const string &CmdLineOptions::getSingleOutFilename()
  {
   if (!inputFileNames.empty() && !outDirectory.empty()){

      if (outFilename.empty()) {
        outFilename = outDirectory;
        int delim = getSingleInFilename().find_last_of(Platform::pathSeparator)+1;
        outFilename += getSingleInFilename().substr((delim>-1)?delim:0)
                       + getOutFileSuffix();
      } else {
         outFilename = outDirectory+outFilename;
      }
   }
   return outFilename;
  }

const string &CmdLineOptions::getSingleInFilename()  const
  {
  return inputFileNames[0];
  }

const string &CmdLineOptions::getOutDirectory()
  {
    if (!outFilename.empty() && !enableBatchMode()){
      outDirectory=getDirName(outFilename);
    }
    return outDirectory;
  }

const string &CmdLineOptions::getCssOutFilename() const
  {
    return cssOutFilename;
  }
const string &CmdLineOptions::getCssInFilename() const
  {
    return cssInFilename;
  }
int CmdLineOptions::getNumberSpaces() const
  {
    return numberSpaces;
  }
bool CmdLineOptions::printVersion()const
  {
    return opt_version;
  }
bool CmdLineOptions::printHelp()const
  {
    return opt_help;
  }
bool CmdLineOptions::printDebugInfo()const
  {
    return opt_verbose;
  }
bool CmdLineOptions::quietMode()const
  {
    return opt_quiet;
  }
bool CmdLineOptions::includeCssDefinition()const
  {
    return opt_include_css && (outputType==highlight::HTML ||
                                outputType==highlight::XHTML);
  }
bool CmdLineOptions::printLineNumbers()const
  {
    return opt_linenumbers;
  }

string CmdLineOptions::getStyleName()const
  {
    return ( ( opt_css_style) ? styleName+".style" : "kwrite.style" );
  }
bool CmdLineOptions::enableBatchMode()const{
    return inputFileNames.size()>1 || opt_batch_mode;
}
bool CmdLineOptions::fragmentOutput()const{
    return opt_fragment;
}
string CmdLineOptions::getOutFileSuffix()const{
    switch (outputType){
      case highlight::XHTML: return ".xhtml";
      case highlight::RTF:   return ".rtf";
      case highlight::TEX:
      case highlight::LATEX: return ".tex";
      case highlight::XSLFO: return ".fo";
      default:    return ".html";
    }
}
string CmdLineOptions::getDirName(const string & path) {
  size_t dirNameLength=path.rfind(Platform::pathSeparator);
  return (dirNameLength==string::npos)?string():path.substr(0, dirNameLength);
}
bool CmdLineOptions::attachLineAnchors()const{
    return opt_attach_line_anchors;
}
bool CmdLineOptions::showThemes()const{
    return opt_show_themes;
}
bool CmdLineOptions::showLangdefs()const{
    return opt_show_langdefs;
}
bool CmdLineOptions::outDirGiven()const{
    return !outFilename.empty();
}
bool CmdLineOptions::fopCompatible() const {
   return opt_xslfo_fop;
}
bool CmdLineOptions::replaceQuotes() const {
   return opt_replacequotes;
}
bool CmdLineOptions::getFlag( const string& paramVal){
   return StringTools::lowerCase(paramVal)=="true";
}
bool CmdLineOptions::formattingEnabled(){
    return !indentScheme.empty();
}
bool CmdLineOptions::dataDirGiven()const {
  return !dataDir.empty();
}
bool CmdLineOptions::additionalDataDirGiven()const {
  return !additionalDataDir.empty();
}
const string &CmdLineOptions::getDataDir() const {
 return dataDir;
}
const string &CmdLineOptions::getIndentScheme() const {
 return indentScheme;
}
const string &CmdLineOptions::getAdditionalDataDir()const{
 return additionalDataDir;
}
const string &CmdLineOptions::getLanguage()  {
 return language;
}
bool CmdLineOptions::printIndexFile() const{
 return opt_printindex && (outputType==highlight::HTML ||
                           outputType==highlight::XHTML);
}
bool CmdLineOptions::printProgress() const{
 return opt_print_progress;
}
bool CmdLineOptions::fillLineNrZeroes() const{
  return opt_fill_zeroes;
}
bool CmdLineOptions::syntaxGiven() const{
  return opt_language;
}
const string  CmdLineOptions::getHelpLang()const{
  return helpLang+".help";
}
highlight::WrapMode CmdLineOptions::getWrappingStyle() const {
  return wrappingStyle;
}
const vector <string> & CmdLineOptions::getInputFileNames() const{
  return inputFileNames;
}
void CmdLineOptions::readDirectory(const string & wildcard){
  // get matching files, use  recursive search
  bool directoryOK=Platform::getDirectoryEntries(inputFileNames, wildcard, true);
  if (!directoryOK)
    {
       cerr << "highlight: No files matched the pattern \""
            << wildcard << "\"."<< endl;
    }
}
void CmdLineOptions::loadConfigurationFile()
{
 string paramVal;
  #ifndef _WIN32
    #ifdef CONFIG_FILE_PATH
      string configFilePath=CONFIG_FILE_PATH;
    #else
      char* homeEnv=getenv("HOME");
      if (homeEnv==NULL) return;
      string configFilePath=string(homeEnv)+"/.highlightrc";
    #endif
  #else
    string configFilePath = Platform::getAppPath() + "highlight.conf";
  #endif
  ConfigurationReader presets(configFilePath);

  if (presets.found())
    {
      configFileRead=true;
      paramVal = presets.getParameter(OPT_CSSOUT);
      if (!paramVal.empty())
        {
          cssOutFilename=paramVal;
        }
      paramVal = presets.getParameter(OPT_CSSIN);
      if (!paramVal.empty())
        {
          cssInFilename=paramVal;
        }
      paramVal = presets.getParameter(OPT_STYLE);
      if (!paramVal.empty())
        {
          styleName=paramVal;
          opt_css_style = true;
        }
      paramVal = presets.getParameter(OPT_DELTABS);
      if (!paramVal.empty())
        {
          numberSpaces = StringTools::str2int(paramVal);
        }
      paramVal = presets.getParameter(OPT_FORMATSTYLE);
      if (!paramVal.empty())
        {
          indentScheme = paramVal;
        }
      paramVal = presets.getParameter(OPT_DATADIR);
      if (!paramVal.empty())
        {
          dataDir=validateDirPath(paramVal);
        }
      paramVal = presets.getParameter(OPT_ADDDATADIR);
      if (!paramVal.empty())
        {
          additionalDataDir=validateDirPath(paramVal);
        }
      paramVal = presets.getParameter(OPT_OUTDIR);
      if (!paramVal.empty())
        {
          outDirectory=validateDirPath(paramVal);
        }

      opt_include_css=getFlag(presets.getParameter(OPT_INCLUDECSS));
      opt_verbose=getFlag(presets.getParameter(OPT_VERBOSE));
      opt_linenumbers=getFlag(presets.getParameter(OPT_LINENO));
      opt_fragment=getFlag(presets.getParameter(OPT_FRAGMENT));
      opt_attach_line_anchors=getFlag(presets.getParameter(OPT_ANCHORS));
      opt_printindex=getFlag(presets.getParameter(OPT_INDEXFILE));
      opt_quiet=getFlag(presets.getParameter(OPT_QUIET));
      opt_xslfo_fop=getFlag(presets.getParameter(OPT_FOP));
      opt_replacequotes=getFlag(presets.getParameter(OPT_REPLACE_QUOTES));
      opt_print_progress=getFlag(presets.getParameter(OPT_PROGRESSBAR));
      opt_fill_zeroes=getFlag(presets.getParameter(OPT_FILLZEROES));

      if (getFlag(presets.getParameter(OPT_WRAP)))
        {
          wrappingStyle=highlight::WRAP_DEFAULT;
        }
      if (getFlag(presets.getParameter(OPT_WRAPSIMPLE)))
        {
          wrappingStyle=highlight::WRAP_SIMPLE;
        }
      if (getFlag(presets.getParameter(OPT_XHTML)))
        {
          outputType=highlight::XHTML;
        }
      else if (getFlag(presets.getParameter(OPT_RTF)))
        {
          outputType=highlight::RTF;
        }
      else if (getFlag(presets.getParameter(OPT_TEX)))
        {
          outputType=highlight::TEX;
        }
      else if (getFlag(presets.getParameter(OPT_LATEX)))
        {
          outputType=highlight::LATEX;
        }
      else if (getFlag(presets.getParameter(OPT_XSLFO)))
        {
          outputType=highlight::XSLFO;
        }
      else if (getFlag(presets.getParameter(OPT_ANSI)))
        {
          outputType=highlight::ANSI;
        }

    }
}

string CmdLineOptions::validateDirPath(const string & path){
   return (path[path.length()-1] !=Platform::pathSeparator)?
              path+Platform::pathSeparator : path;
}

highlight::OutputType CmdLineOptions::getOutputType() const {
  return outputType;
}
