/***************************************************************************
                          languagedefinition.cpp  -  description
                             -------------------
    begin                : Wed Nov 28 2001
    copyright            : (C) 2001 by Andre imon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "languagedefinition.h"

using namespace std;

namespace highlight {

LanguageDefinition::LanguageDefinition():
    ignoreCase(false),
    disableHighlighting(false),
    vhdl_mode(false),
    java_mode(false),
    allowNestedComments(true),
    fullLineComment(false),
    reformatCode(false)
{}

int LanguageDefinition::isKeyword(const string &s)
{
  if (s.length())
    {
      if (keywords.count(s)){
        return keywords[s];
      }
      else if (prefixes.count(s[0])){
        return prefixes[s[0]];
      }
    }
  return 0;
}

bool LanguageDefinition::isPrefix(unsigned char c)
{
  return ( prefixes.count(c));
}

void LanguageDefinition::addSimpleSymbol(stringstream& symbolStream,
                                        State state,
                                        const string& paramValues ) {
  istringstream valueStream(paramValues);
  bool valExists=false;
  string value;
   while (valueStream >> value)
    {
      symbolStream << " " << value;
      valExists = true;
    }
  if (valExists)
    {
      symbolStream << " " << state;
    }
}

void LanguageDefinition::addDelimiterSymbol(stringstream& symbolStream,
                                        State stateBegin, State stateEnd,
                                        const string& paramValues,
                                        unsigned int classID) {
  istringstream valueStream(paramValues);
  string delimPrefix, delimSuffix;
  while (valueStream>>delimPrefix){
    valueStream >> delimSuffix;
    symbolStream << " "<<delimPrefix <<" " << stateBegin;
    symbolStream <<" "<< delimSuffix<<" "<< stateEnd;
    delimiterPrefixes.insert(make_pair(delimPrefix, classID));
  };
}

bool LanguageDefinition::getFlag( string& paramValue){
      return (StringTools::lowerCase(paramValue)=="true");
}

unsigned char LanguageDefinition::getSymbol(const string& paramValue){
      istringstream valueStream(paramValue);
      unsigned char symbol;
      valueStream >> symbol;
      return symbol;
}

void LanguageDefinition::addKeywords(const string &kwList,
                                     int classID){
  istringstream valueStream(kwList);
  string keyword;
  while (valueStream >> keyword){
    keywords.insert(make_pair(keyword, classID));
  }
}

unsigned int LanguageDefinition::generateNewKWClass(const string& newClassName){
  unsigned int newClassID=0;
  bool found=false;
  while (newClassID<keywordClasses.size() && !found){
   found= (newClassName==keywordClasses[newClassID++]);
  }
  if (!found){
   newClassID++;
   keywordClasses.push_back(newClassName);
  }
  return newClassID;
}

unsigned int LanguageDefinition::getDelimPrefixClassID(const string& prefix){
  if (delimiterPrefixes.count(prefix)){
    return delimiterPrefixes[prefix];
  }
  return 0;
}

bool LanguageDefinition::load(const string& langDefPath, bool clear)
{
  if (clear)  reset();

  ConfigurationReader langDef(langDefPath);
  if (langDef.found())
    {
      currentPath=langDefPath;
      disableHighlighting=false;
      string token;
      stringstream symbolStrStream;

      //Stringstream zum Einlesen der Token:
      istringstream valueStream;

      addDelimiterSymbol(symbolStrStream, ML_COMMENT_BEGIN, ML_COMMENT_END,
                         langDef.getParameter("ml_comment"));

      addSimpleSymbol(symbolStrStream, SL_COMMENT,
                      langDef.getParameter("sl_comment"));

      addSimpleSymbol(symbolStrStream, ESC_CHAR,
                      langDef.getParameter("escchar"));

      addSimpleSymbol(symbolStrStream, ESC_CHAR_EXT,
                      langDef.getParameter("escchar_ext"));


      addSimpleSymbol(symbolStrStream, DIRECTIVE_LINE,
                      langDef.getParameter("directive"));

      addSimpleSymbol(symbolStrStream, DIRECTIVE_LINE_END,
                      langDef.getParameter("directiveend"));

      addSimpleSymbol(symbolStrStream, STRING,
                      langDef.getParameter("stringdelimiters"));

      ignoreCase=getFlag(langDef.getParameter("ignorecase"));
      allowNestedComments=getFlag(langDef.getParameter("allownestedcomments"));
      vhdl_mode=getFlag(langDef.getParameter("vhdl_mode"));
      java_mode=getFlag(langDef.getParameter("java_mode"));
      disableHighlighting=getFlag(langDef.getParameter("disablehighlighting"));
      fullLineComment=getFlag(langDef.getParameter("fl_comment"));
      reformatCode=getFlag(langDef.getParameter("reformatting"));
      rawStringPrefix=getSymbol(langDef.getParameter("rawstringprefix"));
      continuationChar=getSymbol(langDef.getParameter("continuationsymbol"));

      string paramName, className, classValue;
      vector<string> paramNames=langDef.getParameterNames();
      for (unsigned int i=0;i<paramNames.size();i++){
         paramName=paramNames[i];
         className=StringTools::getParantheseVal(paramName);
         classValue=langDef.getParameter(paramName);
         if (paramName.find("kw_list") != string::npos ){
           addKeywords(classValue, generateNewKWClass(className));
         }
         if (paramName.find("kw_prefix") != string::npos){
           prefixes.insert(make_pair(classValue[0], generateNewKWClass(className)));
         }
         if (paramName.find("kw_delim") != string::npos ){
          addDelimiterSymbol(symbolStrStream, KEYWORD_BEGIN, KEYWORD_END,
                             classValue, generateNewKWClass(className));
        }
         if (paramName.find("tag_delim") != string::npos ){
          addDelimiterSymbol(symbolStrStream, TAG_BEGIN, TAG_END,
                             classValue, generateNewKWClass(className));
        }
      }

      // zuletzt einlesen, um Probleme mit Delimitern, die Zeichen der
      // Symbolliste enthalten, zu vermeiden
      addSimpleSymbol(symbolStrStream, SYMBOL, langDef.getParameter("symbols"));

      valueStream.str(langDef.getParameter("allowedchars"));
      while (valueStream >> token )
        {
          allowedChars += token;
        }

      symbolString = symbolStrStream.str();

      string fileToInclude=langDef.getParameter("include");
      if (!fileToInclude.empty()){
        string::size_type Pos = langDefPath.find_last_of(Platform::pathSeparator);
        string includeLangDefPath = langDefPath.substr(0, Pos+1) + fileToInclude;
        load(includeLangDefPath, false);
      }
      return true;
    }
  else
    {
      currentPath.clear();
      return false;
    }
}

void LanguageDefinition::reset()
{
  keywords.clear();
  keywordClasses.clear();
  delimiterPrefixes.clear();;
  prefixes.clear();
  allowedChars.clear();
  ignoreCase= false;
  java_mode= vhdl_mode= false;
  allowNestedComments= reformatCode = false;
  rawStringPrefix = continuationChar = '\0';
  disableHighlighting=false;
  fullLineComment=false;
}

bool LanguageDefinition::isVHDL()
{
  return vhdl_mode;
}

bool LanguageDefinition::isJava()
{
  return java_mode;
}

bool LanguageDefinition::allowNestedMLComments(){
  return allowNestedComments;
}

bool LanguageDefinition::highlightingDisabled(){
  return disableHighlighting;
}

bool LanguageDefinition::isFullLineComment(){
  return fullLineComment;
}

bool LanguageDefinition::needsReload(const string &langDefPath){
  return currentPath!=langDefPath;
}

bool LanguageDefinition::enableReformatting(){
  return reformatCode;
}

const KeywordMap& LanguageDefinition::getKeywords() const{
  return keywords;
}


string &LanguageDefinition::getSymbolString() {
  return symbolString;
}


unsigned char LanguageDefinition::getRawStringPrefix(){
  return rawStringPrefix;
}

unsigned char LanguageDefinition::getContinuationChar(){
  return continuationChar;
}

string &LanguageDefinition::getAllowedChars() {
  return allowedChars;
}

bool LanguageDefinition::getSyntaxHighlight() {
  return !disableHighlighting;
}

bool LanguageDefinition::isIgnoreCase() {
  return ignoreCase;
}

const vector<string>&LanguageDefinition::getKeywordClasses() const{
  return keywordClasses;
}

}
