/* libhpojip -- HP OfficeJet image-processing library. */

/* Copyright (C) 1995-2002 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */

/* Original author: Mark Overton and others.
 *
 * Ported to Linux by David Paschal.
 * This file is not currently functional due to its reliance on
 * MS-Windows functionality.
 */

/*****************************************************************************\
 *
 * xbmp.c - encoder and decoder for BMP bitmap files for image processor
 *
 *****************************************************************************
 *
 * Mark Overton, Jan 1998
 *
 * Name of Global Jump-Table:
 *
 *    bmpEncodeTbl = the encoder,
 *    bmpDecodeTbl = the decoder.
 *
 * Items in aXformInfo array passed into setXformSpec:
 *
 *    aXformInfo[IP_BMP_NEGATIVE_HEIGHT]:
 *          0=output positive height, else=output negative height,
 *          only applies to encoder; decoder ignores this.
 *          A negative height means the raster rows go from top to
 *          bottom, the usual way.  A positive height means the rows
 *          go from bottom to top.  While a negative height implies
 *          the 'correct' row-order, some app's can't handle it.
 *
 * Capabilities and Limitations:
 *
 *    Handles 1, 4, 8 and 24 bits per pixel.
 *    Decoder discards the palette in the BMP file (but see todo below).
 *    Encoder assumes input is internal raw format, which means that
 *    1, 4, and 8 bits/pixel are assumed to be grayscales, where
 *    1 bit/pixel is [0=white, 1=black], 4 bits/pixel is [0=black, 15=white]
 *    8 bits/pixel is [0=black, 255=white].  Encoder outputs a palette for
 *    the preceding gray-ranges.
 *    24 bits/pixel is assumed to have r-g-b input color space.
 *    If # rows is not known in input traits (negative), the encoder will
 *    re-write the header when # rows is known at the end of the page. So
 *    the output file will have a valid row-count when conversion is done.
 *
 * Default Input Traits, and Output Traits:
 *
 *    For decoder:
 *
 *          trait             default input             output
 *    -------------------  ---------------------  ------------------------
 *    iPixelsPerRow           ignored              based on header
 *    iBitsPerPixel           ignored              based on header
 *    iComponentsPerPixel     ignored              1 or 3, based on header
 *    lHorizDPI               ignored              based on header
 *    lVertDPI                ignored              based on header
 *    lNumRows                ignored              based on header
 *    iNumPages               passed into output   same as default input
 *    iPageNum                passed into output   same as default input
 *
 *    For encoder:
 *
 *          trait             default input             output
 *    -------------------  ---------------------  ------------------------
 *    iPixelsPerRow         * passed into output   same as default input
 *    iBitsPerPixel         * passed into output   same as default input
 *    iComponentsPerPixel     passed into output   same as default input
 *    lHorizDPI               passed into output   same as default input
 *    lVertDPI                passed into output   same as default input
 *    lNumRows                passed into output   same as default input
 *    iNumPages               passed into output   same as default input
 *    iPageNum                passed into output   same as default input
 *
 *    Above, a "*" by an item indicates it must be valid (not negative).
 *
 *
 * todo: MS docs imply that for 24-bit color, the palette should contain
 * exactly *one* b-g-r entry, which is set to NULL.  Is this true?
 * the code below has *no* palette entries for 24-bit color.
 * a big todo: include palette in 'traits' structure.
 *
\*****************************************************************************/

#include "hpojip.h"
#include "ipdefs.h"
#include "string.h"    /* for memset and memcpy */


#if 0
    #include "stdio.h"
    #include <tchar.h>
    #define PRINT(msg,arg1,arg2) \
        _ftprintf(stderr, msg, (int)arg1, (int)arg2)
#else
    #define PRINT(msg,arg1,arg2)
#endif

/* TODO: Make this work for big and little endian. */
#define LITTLE_ENDIAN  (! (defined SNAKES))

#define MIN_BMP_HEADER_SIZE  \
    (sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER))
#define MAX_BMP_HEADER_SIZE  (MIN_BMP_HEADER_SIZE + 256*sizeof(RGBQUAD))

#define CHECK_VALUE 0x1ce5ca7e



/* BMP_INST - our instance variables */

typedef struct {
    IP_IMAGE_TRAITS traits;   /* traits of the image */
    DWORD    dwInRowBytes;    /* bytes per input row */
    DWORD    dwOutRowBytes;   /* includes pad bytes */
    DWORD    dwRowsDone;      /* number of rows converted so far */
    WORD     nPalette;        /* number of entries in palette */
    RGBQUAD *pPalette;        /* the palette (decode only) */
    DWORD    dwValidChk;      /* struct validity check value */
    DWORD    dwInNextPos;     /* file pos for subsequent input */
    DWORD    dwOutNextPos;    /* file pos for subsequent output */
    BOOL     fDidHeader;      /* already processed the header? */
    BOOL     fSendNegHeight;  /* output negative height? */
} BMP_INST, *PBMP_INST;



// swapRB - swaps the R and B bytes in RGB triples
//
// Input and output of this xform is RGB triples, but the BMP file
// is stored as BGR triples.
//
static void swapRB (
    PBYTE pbInputBuf,
    PBYTE pbOutputBuf,
    int   iPixels)
{
    for ( ; iPixels>0; iPixels-=2)
    {
        pbOutputBuf[0] = pbInputBuf[2];
        pbOutputBuf[1] = pbInputBuf[1];
        pbOutputBuf[2] = pbInputBuf[0];

        pbOutputBuf[3+0] = pbInputBuf[3+2];
        pbOutputBuf[3+1] = pbInputBuf[3+1];
        pbOutputBuf[3+2] = pbInputBuf[3+0];

        pbInputBuf  += 6;
        pbOutputBuf += 6;
    }
} // swapRB



/*
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@                                                                          @@
@@                                                                          @@
@@                           E  N  C  O  D  E  R                            @@
@@                                                                          @@
@@                                                                          @@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
*/



/*****************************************************************************\
 *
 * bmpEncode_openXform - Creates a new instance of the transformer
 *
 *****************************************************************************
 *
 * This returns a handle for the new instance to be passed into
 * all subsequent calls.
 *
 * Return value: IP_DONE=success; IP_FATAL_ERROR=misc error.
 *
\*****************************************************************************/

static WORD bmpEncode_openXform (
    IP_XFORM_HANDLE *pXform)   /* out: returned handle */
{
    PBMP_INST g;

    #if ! defined LITTLE_ENDIAN  /* header must be in little-endian */
        INSURE (0);
    #endif

    INSURE (pXform != NULL);
    IP_MEM_ALLOC (sizeof(BMP_INST), g);
    *pXform = g;
    memset (g, 0, sizeof(BMP_INST));
    g->dwValidChk = CHECK_VALUE;
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpEncode_setDefaultInputTraits - Specifies default input image traits
 *
 *****************************************************************************
 *
 * The header of the file-type handled by the transform probably does
 * not include *all* the image traits we'd like to know.  Those not
 * specified in the file-header are filled in from info provided by
 * this routine.
 *
 * Return value: IP_DONE=success; IP_FATAL_ERROR=misc error.
 *
\*****************************************************************************/

static WORD bmpEncode_setDefaultInputTraits (
    IP_XFORM_HANDLE  hXform,     /* in: handle for xform */
    PIP_IMAGE_TRAITS pTraits)    /* in: default image traits */
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);

    /* Insure that values we actually use are known */
    INSURE (pTraits->iPixelsPerRow > 0);
    INSURE (pTraits->iBitsPerPixel > 0);

    g->traits = *pTraits;   /* a structure copy */

    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpEncode_setXformSpec - Provides xform-specific information
 *
\*****************************************************************************/

static WORD bmpEncode_setXformSpec (
    IP_XFORM_HANDLE  hXform,         /* in: handle for xform */
    DWORD_OR_PVOID   aXformInfo[])   /* in: xform information */
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);
    g->fSendNegHeight = (aXformInfo[IP_BMP_NEGATIVE_HEIGHT].dword != 0);
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpEncode_getHeaderBufSize- Returns size of input buf needed to hold header
 *
\*****************************************************************************/

static WORD bmpEncode_getHeaderBufSize (
    IP_XFORM_HANDLE   hXform,         /* in:  handle for xform */
    DWORD           *pdwInBufLen)     /* out: buf size for parsing header */
{
    /* since input is raw pixels, there is no header, so set it to zero */
    *pdwInBufLen = 0;
    return IP_DONE;
}



/*****************************************************************************\
 *
 * bmpEncode_getActualTraits - Parses header, and returns input & output traits
 *
 *****************************************************************************
 *
 * If depth is 1, a bilevel BMP will be encoded/decoded from/to a
 * RASTER_BITMAP image.
 *
 * If depth is 4, a 16-level BMP will be encoded/decoded from/to gray data.
 * The 4-bit gray value is in the upper nibble of each byte in the raw gray
 * data.
 *
 * If depth is 8, a 256-level BMP will be encoded/decoded from/to gray data.
 *
 * If depth is 24, the BMP will be encoded/decoded from/to RGB data.
 *
\*****************************************************************************/

static WORD bmpEncode_getActualTraits (
    IP_XFORM_HANDLE  hXform,         /* in:  handle for xform */
    DWORD            dwInputAvail,   /* in:  # avail bytes in input buf */
    PBYTE            pbInputBuf,     /* in:  ptr to input buffer */
    PDWORD           pdwInputUsed,   /* out: # bytes used from input buf */
    PDWORD           pdwInputNextPos,/* out: file-pos to read from next */
    PIP_IMAGE_TRAITS pInTraits,      /* out: input image traits */
    PIP_IMAGE_TRAITS pOutTraits)     /* out: output image traits */
{
    PBMP_INST g;
    DWORD     packed_n_bytes;
    WORD      bpp;

    HANDLE_TO_PTR (hXform, g);

    /* Since there is no header, we'll report no usage of input */
    *pdwInputUsed    = 0;
    *pdwInputNextPos = 0;

    /* Since we don't change traits, just copy out the default traits */
    *pInTraits  = g->traits;
    *pOutTraits = g->traits;

    /**** Compute Some Variables ****/

    bpp = g->traits.iBitsPerPixel;
    INSURE (bpp==1 || bpp==4 || bpp==8 || bpp==24);
    packed_n_bytes   = (g->traits.iPixelsPerRow*bpp + 7) / 8;
    g->dwInRowBytes  = packed_n_bytes;
    g->dwOutRowBytes = (packed_n_bytes + 3) & ~3;  /* boost to multiple of 4 */
    g->nPalette      = bpp==24 ? 0 : (1 << bpp);

    PRINT (_T("bmp_encode_output_header: pixels/row=%d, n_rows=%d\n"),
        g->traits.iPixelsPerRow, g->traits.lNumRows);
    PRINT (_T("bmp_encode_output_header: depth=%d, bytes/row=%d\n"),
        g->traits.iBitsPerPixel, g->dwOutRowBytes);

    return IP_DONE | IP_READY_FOR_DATA;

    fatal_error:
    return IP_FATAL_ERROR;
}



/****************************************************************************\
 *
 * bmpEncode_getActualBufSizes - Returns buf sizes needed for remainder of job
 *
\****************************************************************************/

static WORD bmpEncode_getActualBufSizes (
    IP_XFORM_HANDLE  hXform,           /* in:  handle for xform */
    PDWORD           pdwMinInBufLen,   /* out: min input buf size */
    PDWORD           pdwMinOutBufLen)  /* out: min output buf size */
{
    PBMP_INST g;
    DWORD     len;

    HANDLE_TO_PTR (hXform, g);
    *pdwMinInBufLen = g->dwInRowBytes;

    len = MIN_BMP_HEADER_SIZE + g->nPalette*sizeof(RGBQUAD);
    if (len < g->dwOutRowBytes)
        len = g->dwOutRowBytes;
    *pdwMinOutBufLen = len;

    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/****************************************************************************\
 *
 * outputHeader - Only called by bmpEncode_convert
 *
\****************************************************************************/

static WORD outputHeader (
    PBMP_INST g,                /* in:  ptr to instance structure */
    DWORD     dwOutputAvail,    /* in:  # avail bytes in out-buf */
    PBYTE     pbOutputBuf,      /* in:  ptr to out-buffer */
    PDWORD    pdwOutputUsed,    /* out: # bytes written in out-buf */
    PDWORD    pdwOutputThisPos) /* out: file-pos to write the data */
{
    LPBITMAPFILEHEADER  file_hdr_p;
    LPBITMAPINFOHEADER  info_hdr_p;
    RGBQUAD            *pal_p;
    int                 i;
    #define  PELFAC  (39.37 / 65536.0)

    *pdwOutputThisPos = 0;
    *pdwOutputUsed = MIN_BMP_HEADER_SIZE + g->nPalette*sizeof(RGBQUAD);
    INSURE (dwOutputAvail >= *pdwOutputUsed);
    g->dwOutNextPos = *pdwOutputUsed;

      /**************************/
     /* Output the file header */
    /**************************/

    file_hdr_p = (LPBITMAPFILEHEADER)pbOutputBuf;

    file_hdr_p->bfType      = *(WORD *)"BM";
    file_hdr_p->bfOffBits   = *pdwOutputUsed;
    file_hdr_p->bfSize      = file_hdr_p->bfOffBits
                              + g->dwOutRowBytes*g->traits.lNumRows;
    file_hdr_p->bfReserved1 = 0;
    file_hdr_p->bfReserved2 = 0;

      /**************************/
     /* Output the info header */
    /**************************/

    info_hdr_p =
        (LPBITMAPINFOHEADER) ((PBYTE)file_hdr_p + sizeof(BITMAPFILEHEADER));

    info_hdr_p->biSize          = sizeof (BITMAPINFOHEADER);
    info_hdr_p->biWidth         = g->traits.iPixelsPerRow;
    info_hdr_p->biHeight        = g->traits.lNumRows;
    info_hdr_p->biPlanes        = 1;
    info_hdr_p->biBitCount      = g->traits.iBitsPerPixel;
    info_hdr_p->biCompression   = BI_RGB;  /* No compression */
    info_hdr_p->biSizeImage     = g->dwOutRowBytes * g->traits.lNumRows;
    info_hdr_p->biXPelsPerMeter = (long) (g->traits.lHorizDPI*PELFAC + 0.5f);
    info_hdr_p->biYPelsPerMeter = (long) (g->traits.lVertDPI *PELFAC + 0.5f);
    info_hdr_p->biClrUsed       = 0;
    info_hdr_p->biClrImportant  = 0;

    if (g->fSendNegHeight)
        info_hdr_p->biHeight = - info_hdr_p->biHeight;

      /**********************/
     /* Output the palette */
    /**********************/

    pal_p = (RGBQUAD*) ((PBYTE)info_hdr_p + sizeof(BITMAPINFOHEADER));

    if (g->traits.iBitsPerPixel == 1) {
        /* bilevel: 0=white, 1=black */
        pal_p->rgbRed = pal_p->rgbGreen = pal_p->rgbBlue = 255;
        pal_p += 1;
        pal_p->rgbRed = pal_p->rgbGreen = pal_p->rgbBlue = 0;
    } else {
        /* 8-bit gray: 0=black, 255=white */
        for (i=0; i<g->nPalette; i++) {
            pal_p->rgbRed = pal_p->rgbGreen = pal_p->rgbBlue =
                i * 255 / (g->nPalette-1);
            pal_p += 1;
        }
    }

    return IP_READY_FOR_DATA;

    fatal_error:
    return IP_FATAL_ERROR;

    #undef PELFAC
}



/*****************************************************************************\
 *
 * bmpEncode_convert - the work-horse routine
 *
\*****************************************************************************/

static WORD bmpEncode_convert (
    IP_XFORM_HANDLE hXform,
    DWORD           dwInputAvail,      /* in:  # avail bytes in in-buf */
    PBYTE           pbInputBuf,        /* in:  ptr to in-buffer */
    PDWORD          pdwInputUsed,      /* out: # bytes used from in-buf */
    PDWORD          pdwInputNextPos,   /* out: file-pos to read from next */
    DWORD           dwOutputAvail,     /* in:  # avail bytes in out-buf */
    PBYTE           pbOutputBuf,       /* in:  ptr to out-buffer */
    PDWORD          pdwOutputUsed,     /* out: # bytes written in out-buf */
    PDWORD          pdwOutputThisPos)  /* out: file-pos to write the data */
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);

    /**** Output the Header if we haven't already ****/

    if (! g->fDidHeader) {
        g->fDidHeader = TRUE;
        *pdwInputUsed    = 0;
        *pdwInputNextPos = 0;
        return outputHeader (g, dwOutputAvail, pbOutputBuf,
                             pdwOutputUsed, pdwOutputThisPos);
    }

    /**** Check if we were told to flush ****/

    if (pbInputBuf == NULL) {
        PRINT (_T("bmp_encode_convert_row: Told to flush.\n"), 0, 0);
        if (g->traits.lNumRows < 0) {
            /* # rows wasn't known at first, so output header again
             * now that we know the number of rows */
            g->traits.lNumRows = g->dwRowsDone;
            *pdwInputUsed    = 0;
            *pdwInputNextPos = g->dwInNextPos;
            return outputHeader (g, dwOutputAvail, pbOutputBuf,
                                 pdwOutputUsed, pdwOutputThisPos);
        }

        *pdwInputUsed     = *pdwOutputUsed = 0;
        *pdwInputNextPos  = g->dwInNextPos;
        *pdwOutputThisPos = g->dwOutNextPos;
        return IP_DONE;
    }

    /**** Output a Row ****/

    INSURE (dwInputAvail  >= g->dwInRowBytes);
    INSURE (dwOutputAvail >= g->dwOutRowBytes);

    if (g->traits.iBitsPerPixel != 24)
        memcpy (pbOutputBuf, pbInputBuf, g->dwOutRowBytes);
    else
        swapRB (pbInputBuf, pbOutputBuf, g->traits.iPixelsPerRow);

    g->dwInNextPos   += g->dwInRowBytes;
    *pdwInputNextPos  = g->dwInNextPos;
    *pdwInputUsed     = g->dwInRowBytes;
    *pdwOutputUsed    = g->dwOutRowBytes;
    *pdwOutputThisPos = g->dwOutNextPos;
    g->dwOutNextPos  += g->dwOutRowBytes;

    g->dwRowsDone += 1;

    return IP_CONSUMED_ROW | IP_PRODUCED_ROW | IP_READY_FOR_DATA;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpEncode_insertedData - client inserted into our output stream
 *
\*****************************************************************************/

static WORD bmpEncode_insertedData (
    IP_XFORM_HANDLE hXform,
    DWORD           dwNumBytes)
{
    fatalBreakPoint ();
    return IP_FATAL_ERROR;   /* must never be called (can't insert data) */
}



/*****************************************************************************\
 *
 * bmpEncode_newPage - Tells us to flush this page, and start a new page
 *
\*****************************************************************************/

static WORD bmpEncode_newPage (
    IP_XFORM_HANDLE hXform)
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);
    /* todo: return fatal error if convert is called again? */
    return IP_DONE;   /* can't insert page-breaks, so ignore this call */

    fatal_error:
    return IP_FATAL_ERROR;

}



/*****************************************************************************\
 *
 * bmpEncode_closeXform - Destroys this instance
 *
\*****************************************************************************/

static WORD bmpEncode_closeXform (IP_XFORM_HANDLE hXform)
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);
    g->dwValidChk = 0;
    IP_MEM_FREE (g);       /* free memory for the instance */
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpEncodeTbl - Jump-table for encoder
 *
\*****************************************************************************/

IP_XFORM_TBL bmpEncodeTbl = {
    bmpEncode_openXform,
    bmpEncode_setDefaultInputTraits,
    bmpEncode_setXformSpec,
    bmpEncode_getHeaderBufSize,
    bmpEncode_getActualTraits,
    bmpEncode_getActualBufSizes,
    bmpEncode_convert,
    bmpEncode_newPage,
    bmpEncode_insertedData,
    bmpEncode_closeXform
};



/*
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@                                                                          @@
@@                                                                          @@
@@                           D  E  C  O  D  E  R                            @@
@@                                                                          @@
@@                                                                          @@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
*/



/*****************************************************************************\
 *
 * bmpDecode_openXform - Creates a new instance of the transformer
 *
 *****************************************************************************
 *
 * This returns a handle for the new instance to be passed into
 * all subsequent calls.
 *
 * Return value: IP_DONE=success; IP_FATAL_ERROR=misc error.
 *
\*****************************************************************************/

static WORD bmpDecode_openXform (
    IP_XFORM_HANDLE *pXform)   /* out: returned handle */
{
    return bmpEncode_openXform(pXform);   /* allocs & zeroes a new instance */
}



/*****************************************************************************\
 *
 * bmpDecode_setDefaultInputTraits - Specifies default input image traits
 *
 *****************************************************************************
 *
 * The header of the file-type handled by the transform probably does
 * not include *all* the image traits we'd like to know.  Those not
 * specified in the file-header are filled in from info provided by
 * this routine.
 *
 * Return value: IP_DONE=success; IP_FATAL_ERROR=misc error.
 *
\*****************************************************************************/

static WORD bmpDecode_setDefaultInputTraits (
    IP_XFORM_HANDLE  hXform,     /* in: handle for xform */
    PIP_IMAGE_TRAITS pTraits)    /* in: default image traits */
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);
    /* the BMP header will overwrite most if not all items in traits below */
    g->traits = *pTraits;   /* a structure copy */
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpDecode_setXformSpec - Provides xform-specific information
 *
\*****************************************************************************/

static WORD bmpDecode_setXformSpec (
    IP_XFORM_HANDLE  hXform,         /* in: handle for xform */
    DWORD_OR_PVOID   aXformInfo[])   /* in: xform information */
{
    /* file header tells us all we need, so do nothing here */
    return IP_DONE;
}



/*****************************************************************************\
 *
 * bmpDecode_getHeaderBufSize- Returns size of input buf needed to hold header
 *
\*****************************************************************************/

static WORD bmpDecode_getHeaderBufSize (
    IP_XFORM_HANDLE  hXform,          /* in:  handle for xform */
    DWORD           *pdwInBufLen)     /* out: buf size for parsing header */
{
    /* Assume max-size header, which means max-size palette (256 entries) */
    *pdwInBufLen = MIN_BMP_HEADER_SIZE + 256*sizeof(RGBQUAD);
    return IP_DONE;
}



/*****************************************************************************\
 *
 * bmpDecode_getActualTraits - Parses header, and returns input & output traits
 *
 *****************************************************************************
 *
 * If depth is 1, a bilevel BMP will be encoded/decoded from/to a
 * RASTER_BITMAP image.
 *
 * If depth is 4, a 16-level BMP will be encoded/decoded from/to gray data.
 * The 4-bit gray value is in the upper nibble of each byte in the raw gray
 * data.
 *
 * If depth is 8, a 256-level BMP will be encoded/decoded from/to gray data.
 *
 * If depth is 24, the BMP will be encoded/decoded from/to RGB data.
 *
\*****************************************************************************/

static WORD bmpDecode_getActualTraits (
    IP_XFORM_HANDLE  hXform,          /* in:  handle for xform */
    DWORD            dwInputAvail,    /* in:  # avail bytes in input buf */
    PBYTE            pbInputBuf,      /* in:  ptr to input buffer */
    PDWORD           pdwInputUsed,    /* out: # bytes used from input buf */
    PDWORD           pdwInputNextPos, /* out: file-pos to read from next */
    PIP_IMAGE_TRAITS pInTraits,       /* out: input image traits */
    PIP_IMAGE_TRAITS pOutTraits)      /* out: output image traits */
{
    PBMP_INST           g;
    LPBITMAPFILEHEADER  file_hdr_p;
    LPBITMAPINFOHEADER  info_hdr_p;
    RGBQUAD            *pal_p;
    int                 bpp;
    int                 packed_n_bytes;
    int                 bytes_in_pal;
    unsigned            ret_val;
    #define  PELFAC  (65536.0 / 39.37)

    ret_val = IP_DONE;
    HANDLE_TO_PTR (hXform, g);

    /**** Set up pointers ****/

    file_hdr_p = (LPBITMAPFILEHEADER) pbInputBuf;
    info_hdr_p =
        (LPBITMAPINFOHEADER) ((PBYTE)file_hdr_p + sizeof(BITMAPFILEHEADER));
    pal_p = (RGBQUAD*) ((PBYTE)info_hdr_p + sizeof(BITMAPINFOHEADER));

    /**** Parse the file- and info-headers ****/

    g->traits.iPixelsPerRow = info_hdr_p->biWidth;
    g->traits.iBitsPerPixel = bpp = (int)info_hdr_p->biBitCount;
    g->traits.lNumRows      = info_hdr_p->biHeight;
    g->traits.lHorizDPI     = (long)(info_hdr_p->biXPelsPerMeter*PELFAC + 0.5);
    g->traits.lVertDPI      = (long)(info_hdr_p->biYPelsPerMeter*PELFAC + 0.5);
    g->traits.iComponentsPerPixel = (bpp==24) ? 3 : 1;

    g->dwOutRowBytes = packed_n_bytes = (g->traits.iPixelsPerRow*bpp + 7) / 8;
    g->dwInRowBytes  = (packed_n_bytes + 3) & ~3;  /* make multiple of 4 */
    g->nPalette = bpp>8 ? 0 : (1 << bpp);
    if (g->traits.lNumRows < 0)  /* neg height: rows go from top to bottom */
        g->traits.lNumRows = - g->traits.lNumRows;
    /* g->traits.lBytesPerPixel =
     *   ((ULONG)packed_n_bytes<<16) / g->traits.iPixelsPerRow;
     */

    /**** Get the palette ****/

    bytes_in_pal = g->nPalette * sizeof(RGBQUAD);

    if (g->nPalette > 0) {
        IP_MEM_ALLOC (bytes_in_pal, g->pPalette);
        INSURE (g->pPalette != NULL);
        memcpy (g->pPalette, pal_p, bytes_in_pal);
    }

    /**** Misc ****/

    if (   file_hdr_p->bfType != *(PWORD)"BM" 
        || (bpp!=1 && bpp!=4 && bpp!=8 && bpp!=24)
        || g->traits.iPixelsPerRow <= 0)
    {
        ret_val |= IP_INPUT_ERROR;
    }

    *pdwInputNextPos = g->dwInNextPos = *pdwInputUsed = 
        MIN_BMP_HEADER_SIZE + bytes_in_pal;
    INSURE (dwInputAvail >= *pdwInputUsed);
    *pInTraits = *pOutTraits = g->traits;   /* structure copy */

    PRINT (_T("bmp_decode_parse_header: depth=%d, n_rows=%d\n"),
           g->traits.iBitsPerPixel, g->traits.lNumRows);

    return ret_val | IP_READY_FOR_DATA;

    fatal_error:
    return IP_FATAL_ERROR;

    #undef PELFAC
}



/****************************************************************************\
 *
 * bmpDecode_getActualBufSizes - Returns buf sizes needed for remainder of job
 *
\****************************************************************************/

static WORD bmpDecode_getActualBufSizes (
    IP_XFORM_HANDLE  hXform,           /* in:  handle for xform */
    PDWORD           pdwMinInBufLen,   /* out: min input buf size */
    PDWORD           pdwMinOutBufLen)  /* out: min output buf size */
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);
    *pdwMinInBufLen  = g->dwInRowBytes;
    *pdwMinOutBufLen = g->dwOutRowBytes;
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpDecode_convert - the work-horse routine
 *
\*****************************************************************************/

static WORD bmpDecode_convert (
    IP_XFORM_HANDLE hXform,
    DWORD           dwInputAvail,      /* in:  # avail bytes in in-buf */
    PBYTE           pbInputBuf,        /* in:  ptr to in-buffer */
    PDWORD          pdwInputUsed,      /* out: # bytes used from in-buf */
    PDWORD          pdwInputNextPos,   /* out: file-pos to read from next */
    DWORD           dwOutputAvail,     /* in:  # avail bytes in out-buf */
    PBYTE           pbOutputBuf,       /* in:  ptr to out-buffer */
    PDWORD          pdwOutputUsed,     /* out: # bytes written in out-buf */
    PDWORD          pdwOutputThisPos)  /* out: file-pos to write the data */
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);

    if (pbInputBuf == NULL) {
        /* we are being told to flush */
        PRINT (_T("bmp_decode_convert_row: Told to flush; doing nothing.\n"), 0, 0);
        *pdwInputUsed     = *pdwOutputUsed = 0;
        *pdwInputNextPos  = g->dwInNextPos;
        *pdwOutputThisPos = g->dwOutNextPos;
        return IP_DONE;
    }

    if (g->traits.iBitsPerPixel != 24)
        memcpy (pbOutputBuf, pbInputBuf, g->dwInRowBytes);
    else
        swapRB (pbInputBuf, pbOutputBuf, g->traits.iPixelsPerRow);

    g->dwInNextPos   += g->dwInRowBytes;
    *pdwInputNextPos  = g->dwInNextPos;
    *pdwInputUsed     = g->dwInRowBytes;
    *pdwOutputUsed    = g->dwOutRowBytes;
    *pdwOutputThisPos = g->dwOutNextPos;
    g->dwOutNextPos  += g->dwOutRowBytes;

    g->dwRowsDone += 1;

    PRINT (_T("bmp_encode_convert_row: Returning, out used = %d\n"), *pdwOutputUsed, 0);
    return IP_CONSUMED_ROW | IP_PRODUCED_ROW | IP_READY_FOR_DATA;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpDecode_insertedData - client inserted into our output stream
 *
\*****************************************************************************/

static WORD bmpDecode_insertedData (
    IP_XFORM_HANDLE hXform,
    DWORD           dwNumBytes)
{
    fatalBreakPoint ();
    return IP_FATAL_ERROR;   /* must never be called (can't insert data) */
}



/*****************************************************************************\
 *
 * bmpDecode_newPage - Tells us to flush this page, and start a new page
 *
\*****************************************************************************/

static WORD bmpDecode_newPage (
    IP_XFORM_HANDLE hXform)
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);
    /* todo: return fatal error if convert is called again? */
    return IP_DONE;   /* can't insert page-breaks, so ignore this call */

    fatal_error:
    return IP_FATAL_ERROR;

}



/*****************************************************************************\
 *
 * bmpDecode_closeXform - Destroys this instance
 *
\*****************************************************************************/

static WORD bmpDecode_closeXform (IP_XFORM_HANDLE hXform)
{
    PBMP_INST g;

    HANDLE_TO_PTR (hXform, g);

    if (g->pPalette != NULL)
        IP_MEM_FREE (g->pPalette);

    g->dwValidChk = 0;
    IP_MEM_FREE (g);       /* free memory for the instance */
    return IP_DONE;

    fatal_error:
    return IP_FATAL_ERROR;
}



/*****************************************************************************\
 *
 * bmpDecodeTbl - Jump-table for Decoder
 *
\*****************************************************************************/

IP_XFORM_TBL bmpDecodeTbl = {
    bmpDecode_openXform,
    bmpDecode_setDefaultInputTraits,
    bmpDecode_setXformSpec,
    bmpDecode_getHeaderBufSize,
    bmpDecode_getActualTraits,
    bmpDecode_getActualBufSizes,
    bmpDecode_convert,
    bmpDecode_newPage,
    bmpDecode_insertedData,
    bmpDecode_closeXform
};

/* End of File */
