/*
 * Routines to open a URL instead of a local file
 *
 * int openurl(const char *path)
 * FILE *fopenurl(const char *path)
 *
 * ToDo: Add arguments for PUT, POST; parse and return headers.
 *
 * Uses http_proxy and ftp_proxy environment variables.
 *
 * Copyright  1994-2008 World Wide Web Consortium
 * See http://www.w3.org/Consortium/Legal/copyright-software
 *
 * Author: Bert Bos <bert@w3.org>
 * Created: 7 March 1999
 */
#include "config.h"
#include <assert.h>
#include <stdlib.h>
#if HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#if HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif
#if HAVE_FCNTL_H
#  include <fcntl.h>
#endif
#if HAVE_ERRNO_H
#  include <errno.h>
#endif
#include <sys/errno.h>
#include <stdio.h>
#if HAVE_SYS_SOCKET_H
#  include <sys/socket.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#include <ctype.h>
#include "export.h"
#include "heap.e"
#include "types.e"
#include "url.e"
#include "connectsock.e"
#include "dict.e"
#include "headers.e"

#define BUFLEN 4096		/* Max len of header lines */
#define MAXREDIRECTS 10		/* Maximum # or 30x redirects to follow */

static URL http_proxy = NULL, ftp_proxy = NULL;
static int http_proxy_init = 0, ftp_proxy_init = 0;



/* Forward declaration */
FILE *fopenurl2(const conststring path, const conststring mode,
		const Dictionary headers, Dictionary response);


/* open_http2 -- open resource via HTTP; return file pointer or NULL */
static FILE *open_http2(const URL server, const conststring url,
			Dictionary headers, Dictionary response)
{
  Boolean delete_response = !response, delete_headers = !headers;
  int fd, n, redircount;
  conststring h, v;
  char buf[BUFLEN];
  string s;
  FILE *f;
  conststring machine = server->machine;
  conststring port = server->port ? server->port : "80";

  /* Connect */
  fd = connectTCP(machine, port);
  if (fd == -1) return NULL;

  /* Send request */
  newarray(s, strlen(url) + strlen(machine) + strlen(port) + 25);
  if (eq(port, "80"))
    n = sprintf(s, "GET %s HTTP/1.1\r\nHost: %s\r\n", url, machine);
  else
    n = sprintf(s, "GET %s HTTP/1.1\r\nHost: %s:%s\r\n", url, machine, port);
  if (write(fd, s, n) != n) {dispose(s); close(fd); return NULL;}

  /* Send request headers */
  if (headers) {
    for (h = dict_next(headers, NULL); h; h = dict_next(headers, h)) {
      v = dict_find(headers, h);
      assert(v);
      renewarray(s, strlen(h) + strlen(v) + 5);
      n = sprintf(s, "%s: %s\r\n", h, v);
      if (write(fd, s, n) != n) {dispose(s); close(fd); return NULL;}
    }
  }
  if (write(fd, "\r\n", 2) != 2) {close(fd); return NULL;}
  dispose(s);

  shutdown(fd, 1);				/* No more output to server */

  f = fdopen(fd, "r");

  /* Read status code */
  if (!fgets(buf, sizeof(buf), f) ||
      !(hasprefix(buf, "HTTP/1.1 ") || hasprefix(buf, "HTTP/1.0 "))) {
    fclose(f);
    return NULL;
  }

  /* Read response headers */
  if (!response) response = dict_create(50);
  if (!read_mail_headers(f, response)) {
    fclose(f);
    f = NULL;
  } else if (hasprefix(buf+9, "301") || hasprefix(buf+9, "302") ||
	     hasprefix(buf+9, "303") || hasprefix(buf+9, "307")) {
    fclose(f);
    if (!(v = dict_find(response, "location"))) {
      f = NULL;			/* Redirect without a location!? */
    } else {
      s = newstring(v);		/* Because we'll delete the response dict. */
      if (!headers) headers = dict_create(2);
      if ((h = dict_find(headers, " redirects")) &&
	  (redircount = atoi(h)) >= MAXREDIRECTS) {
	f = NULL;		/* Too many redirects */
      } else {
	(void) sprintf(buf, "%d", h ? ++redircount : 1);
	if (!dict_add(headers, " redirects", buf)) {
	  f = NULL;		/* Out of memory */
	} else {
	  dict_destroy_all(response);
	  f = fopenurl2(s, "r", headers, response); /* Redirect */
	  dispose(s);
	}
      }
    }
  } else if (!hasprefix(buf + 9, "200")) {
    if (hasprefix(buf + 9, "404")) errno = EFAULT;
    else if (hasprefix(buf + 9, "4")) errno = EACCES;
    fclose(f);
    f = NULL;
  }
  /* To do:handle 305 Use Proxy */

  /* Return the body of the stream */
  if (delete_response) dict_delete(response);
  if (delete_headers) dict_delete(headers);
  return f;
}


/* open_http -- open resource via HTTP; return file pointer or NULL */
static FILE *open_http(const URL url, Dictionary headers, Dictionary response)
{
  string proxy;

  /* To do: use the response dictionary to pass back information about
   * any errors and the received status code?
   *
   * To do:authentication, use relevant fields in url.
   */

  /* Initialize proxy from environment variable */
  if (! http_proxy_init) {
    proxy = getenv("http_proxy");
    if (proxy) http_proxy = URL_new(proxy);
    http_proxy_init = 1;
  }

  return open_http2(http_proxy?http_proxy:url, url->full, headers, response);
}


/* open_ftp -- open resource via FTP; return file pointer or NULL */
static FILE *open_ftp(const URL url, Dictionary headers, Dictionary response)
{
  string proxy;

  if (! ftp_proxy_init) {
    if ((proxy = getenv("ftp_proxy"))) ftp_proxy = URL_new(proxy);
    ftp_proxy_init = 1;
  }
  if (ftp_proxy) return open_http2(ftp_proxy, url->full, headers, response);

  /* Can only work via proxy for now... */
  errno = ENOSYS;
  return NULL;
}


/* open_file -- open resource as local file or FTP; return file ptr or NULL */
static FILE *open_file(const URL url, const conststring mode,
		       Dictionary headers, Dictionary response)
{
  FILE *f = NULL;

  if (! url->machine || eq(url->machine, "localhost")) {
    f = fopen(url->path, mode);
  }
  if (! f) {
    if (! eq(mode, "r")) errno = EACCES;	/* Not yet supported */
    else f = open_ftp(url, headers, response);
  }
  return f;
}


/* fopenurl2 -- like fopenurl, but sends and returns HTTP headers */
EXPORT FILE *fopenurl2(const conststring path, const conststring mode,
		       const Dictionary headers, Dictionary response)
{
  URL url;
  FILE *f = NULL;

  url = URL_new(path);
  if (! url) {
    errno = EACCES;				/* Invalid URL */
  } else if (! url->proto) {
    f = fopen(path, mode);			/* Assume it's a local file */
  } else if (eq(url->proto, "http")) {
    if (! eq(mode, "r")) errno = ENOSYS;	/* Not yet supported */
    else f = open_http(url, headers, response);
  } else if (eq(url->proto, "ftp")) {
    if (! eq(mode, "r")) errno = ENOSYS;	/* Not yet supported */
    else f = open_ftp(url, headers, response);
  } else if (eq(url->proto, "file")) {
    f = open_file(url, mode, headers, response);
  } else {
    errno = EACCES;				/* Unimplemented protocol */
  }
  URL_dispose(url);
  return f;
}


/* fopenurl -- like fopen, but takes a URL; HTTP headers are parsed */
EXPORT FILE *fopenurl(const string path, const string mode)
{
  return fopenurl2(path, mode, NULL, NULL);
}
