/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// bool-proc.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "bool-proc.h"

#include "defs.h"
#include "htp-files.h"

BOOL DiscardConditionalBlock(TASK *task)
{
    char *plaintext;
    HTML_MARKUP htmlMarkup;
    BOOL result;
    uint embeddedConditionals;
    uint markupType;

    /* discard the block, looking for the matching ELSE or /IF statement */
    embeddedConditionals = 0;
    for(;;)
    {
        result = ReadHtmlFile(task->infile, NULL, &plaintext, &markupType);
        if(result == ERROR)
        {
            return FALSE;
        }
        else if(result == FALSE)
        {
            /* end-of-file before end-of-conditional ... error */
            HtpMsg(MSG_ERROR, task->infile, "EOF encountered before conditional closed");
            return FALSE;
        }

        if(PlaintextToMarkup(plaintext, &htmlMarkup) == FALSE)
        {
            /* memory alloc error */
            HtpMsg(MSG_ERROR, task->infile, "could not parse markup tag (out of memory?)");
            FreeMemory(plaintext);
            return FALSE;
        }

        /* FreeMemory the plaintext buffer, not needed any longer */
        FreeMemory(plaintext);
        plaintext = NULL;

        /* another conditional started? */
        if(IsMarkupTag(&htmlMarkup, "IF"))
        {
            embeddedConditionals++;
        }
        else if(IsMarkupTag(&htmlMarkup, "/IF"))
        {
            /* end of the conditional? */
            if(embeddedConditionals == 0)
            {
                DestroyMarkupStruct(&htmlMarkup);
                task->conditionalLevel--;
                break;
            }
            embeddedConditionals--;
        }
        else if(IsMarkupTag(&htmlMarkup, "ELSE"))
        {
            /* start of TRUE block? */
            if(embeddedConditionals == 0)
            {
                DestroyMarkupStruct(&htmlMarkup);
                break;
            }
        }

        /* destroy and continue */
        DestroyMarkupStruct(&htmlMarkup);
    }

    return TRUE;
}   

uint BooleanProcessor(TASK *task, HTML_MARKUP *htmlMarkup, char **newPlaintext)
{
    const char *value;
    uint type;
    BOOL condTrue;
    HTML_ATTRIBUTE *attrib;
    BOOL notTagFound;

    UNREF_PARAM(newPlaintext);

    condTrue = FALSE;

    /* conditionalLevel keeps track of boolean depth */
    if(task->conditionalLevel == 0)
    {
        if((IsMarkupTag(htmlMarkup, "/IF")) || (IsMarkupTag(htmlMarkup, "ELSE")))
        {
            HtpMsg(MSG_ERROR, task->infile, "<ELSE> or </IF> without matching IF tag");
            return MARKUP_ERROR;
        }
    }

    if(IsMarkupTag(htmlMarkup, "IF"))
    {
        task->conditionalLevel++;

        /* this is an ugly way to handle the IF-IF NOT test, but will need */
        /* be cleaned up in the future */

        /* should either be one or two attributes in markup */
        if(htmlMarkup->attribCount == 0)
        {
            HtpMsg(MSG_ERROR, task->infile, "no conditional to test");
            return MARKUP_ERROR;
        }

        if(htmlMarkup->attribCount > 2)
        {
            HtpMsg(MSG_ERROR, task->infile, "too many items in conditional expression");
            return MARKUP_ERROR;
        }

        /* find the attribute to evaluate and search for NOT attribute */
        notTagFound = FALSE;
        attrib = NULL;
        if(stricmp(htmlMarkup->attrib[0].name, "NOT") == 0)
        {
            /* check to make sure the second attribute is present */
            if(htmlMarkup->attribCount == 1)
            {
                HtpMsg(MSG_ERROR, task->infile, "NOT listed, no conditional to test");
                return MARKUP_ERROR;
            }

            notTagFound = TRUE;
            attrib = &htmlMarkup->attrib[1];
        }
        else if(htmlMarkup->attribCount == 2)
        {
            if(stricmp(htmlMarkup->attrib[1].name, "NOT") == 0)
            {
                notTagFound = TRUE;
                attrib = &htmlMarkup->attrib[0];
            }
            else
            {
                /* this should have been the NOT expression */
                HtpMsg(MSG_ERROR, task->infile, "too many conditionals to test");
                return MARKUP_ERROR;
            }
        }
        else
        {
            attrib = &htmlMarkup->attrib[0];
        }

        /* get the macros associated type (UNKNOWN if macro is not defined) */
        type = GetVariableType(task->varstore, attrib->name);

        /* if only a name is specified, only care if macro is defined */
        if(attrib->value == NULL)
        {
            condTrue = (type != VAR_TYPE_UNKNOWN) ? TRUE : FALSE;
        }
        else
        {
            /* macro value comparison */
            if(type == VAR_TYPE_SET_MACRO)
            {
                /* macro comparison (case-sensitive) */
                value = GetVariableValue(task->varstore, attrib->name);
                condTrue = (value && strcmp(value, attrib->value) == 0)
                    ? TRUE : FALSE;
            }
            else
            {
                /* block macro, comparisons not allowed */
                condTrue = FALSE;
            }
        }

        /* reverse conditional if NOT attribute found */
        if(notTagFound == TRUE)
        {
            condTrue = (condTrue == TRUE) ? FALSE : TRUE;
        }

        if(condTrue == TRUE)
        {
            /* simply discard the markup and let the file processor continue */
            return DISCARD_MARKUP;
        }

        /* discard the rest of the conditional block since this portion has */
        /* evaluated false */
        if(DiscardConditionalBlock(task) == FALSE)
        {
            return MARKUP_ERROR;
        }
    }
    else if(IsMarkupTag(htmlMarkup, "ELSE"))
    {
        /* this can only occur if the associated conditional statement */
        /* evaluated TRUE, so the remaining block must be discarded */
        if(DiscardConditionalBlock(task) == FALSE)
        {
            return MARKUP_ERROR;
        }
    }
    else
    {
        /* end of conditional */
        assert(task->conditionalLevel > 0);
        task->conditionalLevel--;
    }

    return DISCARD_MARKUP;
}   
