/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include <xpl.h>
#include <memmgr.h>
#include <logger.h>
#include <hulautil.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>
#include <msgapi.h>
#include <libical.h>
#include <streamio.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <connio.h>

#include "antispam.h"

#if defined(RELEASE_BUILD)
#define ASpamClientAlloc() MemPrivatePoolGetEntry(ASpam.nmap.pool)
#else
#define ASpamClientAlloc() MemPrivatePoolGetEntryDebug(ASpam.nmap.pool, __FILE__, __LINE__)
#endif

#define QUEUE_WORK_TO_DO(c, id, r) \
        { \
            XplWaitOnLocalSemaphore(ASpam.nmap.semaphore); \
            if (XplSafeRead(ASpam.nmap.worker.idle)) { \
                (c)->queue.previous = NULL; \
                if (((c)->queue.next = ASpam.nmap.worker.head) != NULL) { \
                    (c)->queue.next->queue.previous = (c); \
                } else { \
                    ASpam.nmap.worker.tail = (c); \
                } \
                ASpam.nmap.worker.head = (c); \
                (r) = 0; \
            } else { \
                XplSafeIncrement(ASpam.nmap.worker.active); \
                XplBeginThread(&(id), HandleConnection, 24 * 1024, XplSafeRead(ASpam.nmap.worker.active), (r)); \
                if (!(r)) { \
                    (c)->queue.previous = NULL; \
                    if (((c)->queue.next = ASpam.nmap.worker.head) != NULL) { \
                        (c)->queue.next->queue.previous = (c); \
                    } else { \
                        ASpam.nmap.worker.tail = (c); \
                    } \
                    ASpam.nmap.worker.head = (c); \
                } else { \
                    XplSafeDecrement(ASpam.nmap.worker.active); \
                    (r) = -1; \
                } \
            } \
            XplSignalLocalSemaphore(ASpam.nmap.semaphore); \
        }


ASpamGlobals ASpam;

static BOOL 
ASpamClientAllocCB(void *buffer, void *data)
{
    register ASpamClient *c = (ASpamClient *)buffer;

    memset(c, 0, sizeof(ASpamClient));

    return(TRUE);
}

static void 
ASpamClientFree(ASpamClient *client)
{
    register ASpamClient *c = client;

    if (c->conn) {
        ConnClose(c->conn, 1);
        ConnFree(c->conn);
        c->conn = NULL;
    }

    MemPrivatePoolReturnEntry(c);

    return;
}

static void 
FreeClientData(ASpamClient *client)
{
    if (client && !(client->flags & ASPAM_CLIENT_FLAG_EXITING)) {
        client->flags |= ASPAM_CLIENT_FLAG_EXITING;

        if (client->conn) {
            ConnClose(client->conn, 1);
            ConnFree(client->conn);
            client->conn = NULL;
        }

        if (client->envelope) {
            MemFree(client->envelope);
        }
    }

    return;
}

static int 
CmpAddr(const void *c, const void *d)
{
    int r = 0;
    int clen;
    int dlen;
    int ci;
    int di;
    unsigned char cc;
    unsigned char dc;
    const unsigned char *candidate = *(unsigned char**)c;
    const unsigned char *domain = *(unsigned char**)d;

    if (candidate && domain) {
        ci = clen = strlen(candidate) - 1;
        di = dlen = strlen(domain) - 1;

        while ((ci >= 0) && (di >= 0) && (r == 0)) {
            cc = toupper(candidate[ci]);
            dc = toupper(domain[di]);
            if (cc == dc) {
                --ci;
                --di;
            } else if (cc < dc) {
                r = -1;
            } else {
                r = 1;
            }
        }

        if (r == 0) {
            if (clen < dlen) {
                r = -1;
            } else if ((clen > dlen) && (candidate[ci] != '.') && (candidate[ci] != '@')) {
                r = 1;
            } else {
                r = 1;
            }
        }

        return(r);
    }

    return(0);
}

static int 
MatchAddr(unsigned char *candidate, unsigned char *domain)
{
    int r = 0;
    int clen;
    int dlen;
    int ci;
    int di;
    unsigned char cc;
    unsigned char dc;

    if (candidate && domain) {
        ci = clen = strlen(candidate) - 1;
        di = dlen = strlen(domain) - 1;

        while ((ci >= 0) && (di >= 0) && (r == 0)) {
            cc = toupper(candidate[ci]);
            dc = toupper(domain[di]);
            if (cc == dc) {
                --ci;
                --di;
            } else if(cc < dc) {
                r = -1;
            } else {
                r = 1;
            }
        }

        if ((r == 0) && (clen != dlen)) {
            if (clen < dlen) {
                r = -1;
            } else if ((candidate[ci] != '.') && (candidate[ci] != '@')) {
                r = 1;
            }
        }

        return(r);
    }

    return(0);
}

static unsigned char 
*IsSpammer(unsigned char *from)
{
    int cmp;
    int start;
    int end;
    int middle = 0;
    BOOL matched = FALSE;

    start = 0;
    end = ASpam.disallow.used - 1;

    while ((end >= start) && !matched) {
        middle = (end - start) / 2 + start;
        cmp = MatchAddr(from, ASpam.disallow.list->Value[middle]);
        if (cmp == 0) {
            matched = TRUE;
        } else if (cmp < 0) {
            end = middle - 1;
        } else {
            start = middle + 1;
        }
    }

    if(matched) {
        return(ASpam.disallow.list->Value[middle]);
    }

    return(NULL);
}

static __inline int 
ProcessConnection(ASpamClient *client)
{
    int ccode;
    int length;
    unsigned long source = 0;
    unsigned char *ptr;
    unsigned char *cur;
    unsigned char *line;
    unsigned char *blockedAddr = NULL;
    unsigned char qID[16];
    BOOL copy;
    BOOL blocked = FALSE;

    if (((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
            && (ccode == 6020) 
            && ((ptr = strchr(client->line, ' ')) != NULL)) {
        *ptr++ = '\0';

        strcpy(qID, client->line);

        length = atoi(ptr);
        client->envelope = MemMalloc(length + 3);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if (client->envelope) {
        sprintf(client->line, "ASpam: %s", qID);
        XplRenameThread(XplGetThreadID(), client->line);

        ccode = ConnRead(client->conn, client->envelope, length);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 6021)) {
        client->envelope[length] = '\0';

        cur = client->envelope;
    } else {
        MemFree(client->envelope);
        client->envelope = NULL;

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    while (*cur) {
        copy = TRUE;
        line = strchr(cur, 0x0A);
        if (line) {
            if (line[-1] == 0x0D) {
                line[-1] = '\0';
            } else {
                *line = '\0';
            }

            line++;
        } else {
            line = cur + strlen(cur);
        }

        switch (cur[0]) {
            case QUEUE_FROM: {
                ptr = strchr(cur + 1, ' ');
                if (ptr) {
                    *ptr = '\0';

                    blockedAddr = IsSpammer(cur + 1);
                    if (!blockedAddr) {
                        *ptr = ' ';
                    } else {
                        blocked = TRUE;

                        LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_QUEUE, LOGGER_EVENT_SPAM_BLOCKED, LOG_NOTICE, 0, cur + 1, NULL, source, 0, NULL, 0);

                        if (ASpam.flags & ASPAM_FLAG_RETURN_TO_SENDER) {
                            ccode = NMAPSendCommandF(client->conn, "QRTS %s %s %lu %d Mail from user or domain %s is blocked at this site.\r\n", 
                                      cur + 1, cur + 1, (long unsigned int)(DSN_HEADER|DSN_FAILURE), DELIVER_BLOCKED, blockedAddr);
                        }

                        if ((ccode != -1) && (ASpam.flags & ASPAM_FLAG_NOTIFY_POSTMASTER)) {
                            if (((ccode = NMAPSendCommandF(client->conn, "QCOPY %s\r\n", qID)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommand(client->conn, "QSTOR FROM - -\r\n", 16)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommandF(client->conn, "QSTOR LOCAL %s %s 0\r\n", ASpam.postmaster, ASpam.postmaster)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE)) != -1) 
                                    && ((ccode = NMAPSendCommand(client->conn, "QRUN\r\n", 6)) != -1)) {
                                ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
                            } else {
                                NMAPSendCommand(client->conn, "QABRT\r\n", 7);
                            }
                        }

                        *ptr = ' ';
                    }
                }

                break;
            }

            case QUEUE_RECIP_REMOTE:
            case QUEUE_RECIP_LOCAL:
            case QUEUE_RECIP_MBOX_LOCAL: {
                if (blocked) {
                    copy = FALSE;
                }

                break;
            }

            case QUEUE_ADDRESS: {
                source = atol(cur + 1);
                break;
            }
        }

        if (copy && (ccode != -1)) {
            ccode = NMAPSendCommandF(client->conn, "QMOD RAW %s\r\n", cur);
        }

        cur = line;
    }

    if ((ccode != -1) 
            && ((ccode = NMAPSendCommand(client->conn, "QDONE\r\n", 7)) != -1)) {
        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
    }

    if (client->envelope) {
        MemFree(client->envelope);
        client->envelope = NULL;
    }

    return(0);
}

static void 
HandleConnection(void *param)
{
    int ccode;
    long threadNumber = (long)param;
    time_t sleep = time(NULL);
    time_t wokeup;
    ASpamClient *client;

    XplSignalBlock();

    if ((client = ASpamClientAlloc()) == NULL) {
        XplConsolePrintf("hulaantispam: New worker failed to startup; out of memory.\r\n");

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);

        XplSafeDecrement(ASpam.nmap.worker.active);

        return;
    }

    do {
        XplRenameThread(XplGetThreadID(), "ASpam Worker");

        XplSafeIncrement(ASpam.nmap.worker.idle);

        XplWaitOnLocalSemaphore(ASpam.nmap.worker.todo);

        XplSafeDecrement(ASpam.nmap.worker.idle);

        wokeup = time(NULL);

        XplWaitOnLocalSemaphore(ASpam.nmap.semaphore);

        client->conn = ASpam.nmap.worker.tail;
        if (client->conn) {
            ASpam.nmap.worker.tail = client->conn->queue.previous;
            if (ASpam.nmap.worker.tail) {
                ASpam.nmap.worker.tail->queue.next = NULL;
            } else {
                ASpam.nmap.worker.head = NULL;
            }
        }

        XplSignalLocalSemaphore(ASpam.nmap.semaphore);

        if (client->conn) {
            if (ConnNegotiate(client->conn, ASpam.nmap.ssl.context)) {
                ccode = ProcessConnection(client);
            } else {
                NMAPSendCommand(client->conn, "QDONE\r\n", 7);
                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
            }
        }

        if (client->conn) {
            ConnFlush(client->conn);
        }

        FreeClientData(client);

        /* Live or die? */
        if (threadNumber == XplSafeRead(ASpam.nmap.worker.active)) {
            if ((wokeup - sleep) > ASpam.nmap.sleepTime) {
                break;
            }
        }

        sleep = time(NULL);

        ASpamClientAllocCB(client, NULL);
    } while (ASpam.state == ASPAM_STATE_RUNNING);

    FreeClientData(client);

    ASpamClientFree(client);

    XplSafeDecrement(ASpam.nmap.worker.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void 
AntiSpamServer(void *ignored)
{
    int i;
    int ccode;
    XplThreadID id;
    Connection *conn;

    XplSafeIncrement(ASpam.server.active);

    XplRenameThread(XplGetThreadID(), "Anti-Spam Server");

    while (ASpam.state < ASPAM_STATE_STOPPING) {
        if (ConnAccept(ASpam.nmap.conn, &conn) != -1) {
            if (ASpam.state < ASPAM_STATE_STOPPING) {
                conn->ssl.enable = FALSE;

                QUEUE_WORK_TO_DO(conn, id, ccode);
                if (!ccode) {
                    XplSignalLocalSemaphore(ASpam.nmap.worker.todo);

                    continue;
                }
            }

            ConnWrite(conn, "QDONE\r\n", 7);
            ConnClose(conn, 0);

            ConnFree(conn);
            conn = NULL;

            continue;
        }

        switch (errno) {
            case ECONNABORTED:
#ifdef EPROTO
            case EPROTO: 
#endif
            case EINTR: {
                if (ASpam.state < ASPAM_STATE_STOPPING) {
                    LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);
                }

                continue;
            }

            default: {
                if (ASpam.state < ASPAM_STATE_STOPPING) {
                    XplConsolePrintf("hulaantispam: Exiting after an accept() failure; error %d\r\n", errno);

                    LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);

                    ASpam.state = ASPAM_STATE_STOPPING;
                }

                break;
            }
        }

        break;
    }

    /* Shutting down */
    ASpam.state = ASPAM_STATE_UNLOADING;

    XplConsolePrintf("hulaantispam: Shutting down.\r\n");

    id = XplSetThreadGroupID(ASpam.id.group);

    if (ASpam.nmap.conn) {
        ConnClose(ASpam.nmap.conn, 1);
        ASpam.nmap.conn = NULL;
    }

    if (ASpam.nmap.ssl.enable) {
        ASpam.nmap.ssl.enable = FALSE;

        if (ASpam.nmap.ssl.conn) {
            ConnClose(ASpam.nmap.ssl.conn, 1);
            ASpam.nmap.ssl.conn = NULL;
        }

        if (ASpam.nmap.ssl.context) {
            ConnSSLContextFree(ASpam.nmap.ssl.context);
            ASpam.nmap.ssl.context = NULL;
        }
    }

    ConnCloseAll(1);

    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (i = 0; (XplSafeRead(ASpam.server.active) > 1) && (i < 60); i++) {
        XplDelay(1000);
    }

    for (i = 0; (ManagementState() != MANAGEMENT_STOPPED) && (i < 60); i++) {
        XplDelay(1000);
    }

    XplConsolePrintf("hulaantispam: Shutting down %d queue threads\r\n", XplSafeRead(ASpam.nmap.worker.active));

    XplWaitOnLocalSemaphore(ASpam.nmap.semaphore);

    ccode = XplSafeRead(ASpam.nmap.worker.idle);
    while (ccode--) {
        XplSignalLocalSemaphore(ASpam.nmap.worker.todo);
    }

    XplSignalLocalSemaphore(ASpam.nmap.semaphore);

    for (i = 0; XplSafeRead(ASpam.nmap.worker.active) && (i < 60); i++) {
        XplDelay(1000);
    }

    if (XplSafeRead(ASpam.server.active) > 1) {
        XplConsolePrintf("hulaantispam: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(ASpam.server.active) - 1);
    }

    if (XplSafeRead(ASpam.nmap.worker.active)) {
        XplConsolePrintf("hulaantispam: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(ASpam.nmap.worker.active));
    }

    LoggerClose(ASpam.handle.logging);
    ASpam.handle.logging = NULL;

    /* shutdown the scanning engine */

    XplCloseLocalSemaphore(ASpam.nmap.worker.todo);
    XplCloseLocalSemaphore(ASpam.nmap.semaphore);

    if (ASpam.allow.list) {
        MDBDestroyValueStruct(ASpam.allow.list);
        ASpam.allow.list = NULL;
        ASpam.allow.used = 0;
    }

    if (ASpam.disallow.list) {
        MDBDestroyValueStruct(ASpam.disallow.list);
        ASpam.disallow.list = NULL;
        ASpam.disallow.used = 0;
    }

    MsgShutdown();

    ConnShutdown();

    MemPrivatePoolFree(ASpam.nmap.pool);

    MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

    XplConsolePrintf("hulaantispam: Shutdown complete\r\n");

    XplSignalLocalSemaphore(ASpam.sem.main);
    XplWaitOnLocalSemaphore(ASpam.sem.shutdown);

    XplCloseLocalSemaphore(ASpam.sem.shutdown);
    XplCloseLocalSemaphore(ASpam.sem.main);

    XplSetThreadGroupID(id);

    return;
}

static BOOL 
ReadConfiguration(void)
{
    unsigned long used;
    unsigned char *ptr;
    MDBValueStruct *config;

    ASpam.allow.list = MDBCreateValueStruct(ASpam.handle.directory, MsgGetServerDN(NULL));
    ASpam.disallow.list = MDBCreateValueStruct(ASpam.handle.directory, MsgGetServerDN(NULL));
    config = MDBCreateValueStruct(ASpam.handle.directory, MsgGetServerDN(NULL));
    if (ASpam.allow.list && ASpam.disallow.list && config) {
        ASpam.allow.used = 0;
        ASpam.disallow.used = 0;
    } else {
        if (config) {
            MDBDestroyValueStruct(config);
        }

        return(FALSE);
    }

    if (MDBRead(MSGSRV_AGENT_ANTISPAM, MSGSRV_A_ACTION, config) > 0) {
        LoggerEvent(ASpam.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_ACTION", config->Value[0], 0, 0, NULL, 0);

        if (atoi(config->Value[0]) & 0x01) {
            ASpam.flags |= ASPAM_FLAG_RETURN_TO_SENDER;
        }

        if (atoi(config->Value[0]) & 0x02) {
            ASpam.flags |= ASPAM_FLAG_NOTIFY_POSTMASTER;
        }

        MDBFreeValues(config);
    }

    if (MDBRead(MSGSRV_AGENT_ANTISPAM, MSGSRV_A_CONFIGURATION, config) > 0) {
        for (used = 0; used < config->Used; used++) {
            if (XplStrNCaseCmp(config->Value[used], "Queue: ", 7) == 0) {
                ASpam.nmap.queue = atol(config->Value[used] + 7);
            }
        }
        
        MDBFreeValues(config);
    }

    if (MDBRead(MSGSRV_SELECTED_CONTEXT, MSGSRV_A_POSTMASTER, config) > 0) {
        ptr = strrchr(config->Value[0], '\\');
        if (ptr) {
            ptr++;
        } else {
            ptr = config->Value[0];
        }

        strcpy(ASpam.postmaster, ptr);

        MDBFreeValues(config);
    } else {
        ASpam.flags &= ~ASPAM_FLAG_NOTIFY_POSTMASTER;
    }

    if ((ASpam.disallow.used = MDBRead(MSGSRV_AGENT_ANTISPAM, MSGSRV_A_EMAIL_ADDRESS, ASpam.disallow.list)) > 0) {
        qsort(ASpam.disallow.list->Value, ASpam.disallow.used, sizeof(unsigned char*), CmpAddr);

        MDBFreeValues(config);
    }

    MDBSetValueStructContext(NULL, config);
    if (MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, config)>0) { 
        HashCredential(MsgGetServerDN(NULL), config->Value[0], ASpam.nmap.hash);
    }

    MDBDestroyValueStruct(config);

    return(TRUE);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
static int 
_NonAppCheckUnload(void)
{
    static BOOL    checked = FALSE;
    XplThreadID    id;

    if (!checked) {
        checked = TRUE;
        ASpam.state = ASPAM_STATE_UNLOADING;

        XplWaitOnLocalSemaphore(ASpam.sem.shutdown);

        id = XplSetThreadGroupID(ASpam.id.group);
        ConnClose(ASpam.nmap.conn, 1);
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(ASpam.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (ASpam.state < ASPAM_STATE_UNLOADING) {
                ASpam.state = ASPAM_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (ASpam.state == ASPAM_STATE_STOPPING) {
                XplUnloadApp(getpid());
            } else if (ASpam.state < ASPAM_STATE_STOPPING) {
                ASpam.state = ASPAM_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

static int 
QueueSocketInit(void)
{
    ASpam.nmap.conn = ConnAlloc(FALSE);
    if (ASpam.nmap.conn) {
        memset(&(ASpam.nmap.conn->socketAddress), 0, sizeof(ASpam.nmap.conn->socketAddress));

        ASpam.nmap.conn->socketAddress.sin_family = AF_INET;
        ASpam.nmap.conn->socketAddress.sin_addr.s_addr = MsgGetAgentBindIPAddress();

        /* Get root privs back for the bind.  It's ok if this fails -
        * the user might not need to be root to bind to the port */
        XplSetEffectiveUserId(0);

        ASpam.nmap.conn->socket = ConnServerSocket(ASpam.nmap.conn, 2048);
        if (XplSetEffectiveUser(MsgGetUnpriviledgedUser()) < 0) {
            XplConsolePrintf("hulaantispam: Could not drop to unpriviledged user '%s'\r\n", MsgGetUnpriviledgedUser());
            ConnFree(ASpam.nmap.conn);
            ASpam.nmap.conn = NULL;
            return(-1);
        }

        if (ASpam.nmap.conn->socket == -1) {
            XplConsolePrintf("hulaantispam: Could not bind to dynamic port\r\n");
            ConnFree(ASpam.nmap.conn);
            ASpam.nmap.conn = NULL;
            return(-1);
        }

        if (NMAPRegister(MSGSRV_AGENT_ANTISPAM, ASpam.nmap.queue, ASpam.nmap.conn->socketAddress.sin_port) != REGISTRATION_COMPLETED) {
            XplConsolePrintf("hulaantispam: Could not register with hulanmap\r\n");
            ConnFree(ASpam.nmap.conn);
            ASpam.nmap.conn = NULL;
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaantispam: Could not allocate connection.\r\n");
        return(-1);
    }

    return(0);
}

XplServiceCode(SignalHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int                ccode;
    XplThreadID        id;

    if (XplSetEffectiveUser(MsgGetUnpriviledgedUser()) < 0) {
        XplConsolePrintf("hulaantispam: Could not drop to unpriviledged user '%s', exiting.\n", MsgGetUnpriviledgedUser());
        return(1);
    }

    XplSignalHandler(SignalHandler);

    ASpam.id.main = XplGetThreadID();
    ASpam.id.group = XplGetThreadGroupID();

    ASpam.state = ASPAM_STATE_INITIALIZING;
    ASpam.flags = 0;

    ASpam.nmap.conn = NULL;
    ASpam.nmap.queue = Q_INCOMING;
    ASpam.nmap.pool = NULL;
    ASpam.nmap.sleepTime = (5 * 60);
    ASpam.nmap.ssl.conn = NULL;
    ASpam.nmap.ssl.enable = FALSE;
    ASpam.nmap.ssl.context = NULL;
    ASpam.nmap.ssl.config.options = 0;

    ASpam.handle.directory = NULL;
    ASpam.handle.logging = NULL;

    strcpy(ASpam.nmap.address, "127.0.0.1");

    XplSafeWrite(ASpam.server.active, 0);

    XplSafeWrite(ASpam.nmap.worker.idle, 0);
    XplSafeWrite(ASpam.nmap.worker.active, 0);
    XplSafeWrite(ASpam.nmap.worker.maximum, 100000);

    if (MemoryManagerOpen(MSGSRV_AGENT_ANTISPAM) == TRUE) {
        ASpam.nmap.pool = MemPrivatePoolAlloc("AntiSpam Connections", sizeof(ASpamClient), 0, 3072, TRUE, FALSE, ASpamClientAllocCB, NULL, NULL);
        if (ASpam.nmap.pool != NULL) {
            XplOpenLocalSemaphore(ASpam.sem.main, 0);
            XplOpenLocalSemaphore(ASpam.sem.shutdown, 1);
            XplOpenLocalSemaphore(ASpam.nmap.semaphore, 1);
            XplOpenLocalSemaphore(ASpam.nmap.worker.todo, 1);
        } else {
            MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

            XplConsolePrintf("hulaantispam: Unable to create connection pool; shutting down.\r\n");
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaantispam: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }

    ConnStartup(CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    ASpam.handle.directory = (MDBHandle)MsgInit();
    if (ASpam.handle.directory == NULL) {
        XplBell();
        XplConsolePrintf("hulaantispam: Invalid directory credentials; exiting!\r\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

        return(-1);
    }

    NMAPInitialize(ASpam.handle.directory);

    SetCurrentNameSpace(NWOS2_NAME_SPACE);
    SetTargetNameSpace(NWOS2_NAME_SPACE);

	ASpam.handle.logging = LoggerOpen("hulaantispam");
	if (!ASpam.handle.logging) {
		XplConsolePrintf("hulaantispam: Unable to initialize logging; disabled.\r\n");
	}

    ReadConfiguration();

    if (ASpam.allow.used || ASpam.disallow.used) {
        if (QueueSocketInit() < 0) {
            XplConsolePrintf("hulaantispam: Exiting.\r\n");

            MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

            return -1;
        }

        /* initialize scanning engine here */


        if (XplSetRealUser(MsgGetUnpriviledgedUser()) < 0) {
            XplConsolePrintf("hulaantispam: Could not drop to unpriviledged user '%s', exiting.\r\n", MsgGetUnpriviledgedUser());

            MemoryManagerClose(MSGSRV_AGENT_ANTISPAM);

            return 1;
        }

        if ((ManagementInit(MSGSRV_AGENT_ANTISPAM, ASpam.handle.directory)) 
                && (ManagementSetVariables(GetASpamManagementVariables(), GetASpamManagementVariablesCount())) 
                && (ManagementSetCommands(GetASpamManagementCommands(), GetASpamManagementCommandsCount()))) {
            XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
        }


        if (ccode) {
            XplConsolePrintf("hulaantispam: Unable to startup the management interface.\r\n");
        }

        ASpam.state = ASPAM_STATE_RUNNING;
    } else {
        XplConsolePrintf("hulaantispam: No hosts allowed or disallowed; unloading\r\n");

        /* Linux Advanced Server 2.1 has a race condition if we exit too early. */
        XplDelay(30 * 1000);

        ASpam.state = ASPAM_STATE_STOPPING;
    }

    XplStartMainThread(PRODUCT_SHORT_NAME, &id, AntiSpamServer, 8192, NULL, ccode);
    
    XplUnloadApp(XplGetThreadID());
    return(0);
}
