/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <openssl/ssl.h>
#include <openssl/err.h>

#include <msgapi.h>
#include <nmap.h>

#include "modwebp.h"

unsigned char	*ModWebTVersion = "$Revision: 1.2 $";

/* Public "portal" template */
TemplateStruct	*PublicTemplate		= NULL;
unsigned char	***PublicTemplateImages= NULL;

/* Session-based templates */
TemplateStruct	**Templates				= NULL;
int				TemplateCount			= 0;
unsigned char	***TemplateImages		= NULL;
int				TemplateMaxLanguages	= 0;

#if defined(MODWEB_DEBUG_BUILD)
int		TemplateScreen;
#define	TemplateDebugInit()	TemplateScreen=XplCreateScreen("MW Template Debug Screen", 0)
#else
#define	TemplateDebugInit()
#endif


static void
AddTemplate(TemplateStruct *Template)
{
	Templates=MemRealloc(Templates, (TemplateCount+1) * sizeof(TemplateStruct *));
	if (!Templates) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, ((TemplateCount+1) * sizeof(TemplateStruct *)), __LINE__, NULL, 0);
		return;
	}
	Templates[TemplateCount]=Template;
	TemplateCount++;
}

void
FreeTemplates(void)
{
	int i;

	TemplateDebug("Freeing all templates\n");

	for (i=0; i<TemplateCount; i++) {
		MemFree(Templates[i]);
	}

	if (Templates) {
		MemFree(Templates);
	}

	if (TemplateImages) {
		MemFree(TemplateImages);
	}

	if (PublicTemplateImages) {
		MemFree(PublicTemplateImages);
	}

	if (PublicTemplate) {
		MemFree(PublicTemplate);
	}

	TemplateImages=NULL;
	Templates=NULL;
	TemplateCount=0;
}

#if 0
unsigned long
FindLanguage(unsigned long LanguageID, long Template)
{
	return(0);
}
#endif


long
MWFindTemplate(unsigned char *TemplateName)
{
	int	i;

	TemplateDebug("FindTemplate(%s)\n", TemplateName);

	for (i=0; i<TemplateCount; i++) {
		if (MWQuickCmp(TemplateName, Templates[i]->Name)) {
			TemplateDebug("FindTemplate(%s) found:%d\n", TemplateName, i);
			return(i);
		}
	}

	return(-1);
}

long
MWFindTemplatePage(unsigned char *Name, unsigned long Template)
{
	NameLookupStruct	**NLS;
	unsigned char		*ptr;
	unsigned long		i;

	ptr=(unsigned char *)Templates[Template] +
		sizeof(TemplateStruct) +
		sizeof(unsigned long)*Templates[Template]->LanguageCount+
		( 
			sizeof(unsigned char *) *
			(
				Templates[Template]->StringCount+
				Templates[Template]->ImageCount
			)*
			Templates[Template]->LanguageCount 
		);

	NLS=(NameLookupStruct **)ptr;
	for(i=0; i<Templates[Template]->NameCount; i++) {
		if (MWQuickCmp(Name, NLS[i]->Name)) {
			return(NLS[i]->PageID);
		}
	}
	return(-1);
}


static BOOL
FixupTemplate(TemplateStruct *Template)
{
    TemplateObjectOverlay **localObject;
	unsigned long	*TemplateAddr=(unsigned long *)Template;
	unsigned long	*AddrPtr;
	unsigned long	i;
	unsigned long	FixupCount=Template->FixupCount;

	TemplateDebug("FixupTemplate: %s; Languages:%d, Strings:%d Images:%d Fixups:%d\n", Template->Name, (int)Template->LanguageCount, (int)Template->StringCount, (int)Template->ImageCount, (int)Template->FixupCount);

	AddrPtr=(unsigned long *)((unsigned char *)TemplateAddr+sizeof(TemplateStruct)+sizeof(unsigned long)*Template->LanguageCount);
	for (i=0; i<FixupCount; i++) {
		AddrPtr[i]+=(unsigned long)TemplateAddr;
	}

	/* Fixup dynamic object pointers */
	localObject = (TemplateObjectOverlay **)(unsigned char **)((unsigned char *)Template + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Template->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Template->StringCount+
									Template->ImageCount
								)*
								Template->LanguageCount
							) +
							sizeof(unsigned char *) * Template->NameCount);

	for (i = 0; i < Template->TemplateCount; i++) {
	    localObject[i]->FileName = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->FileName);
	    localObject[i]->ContentType = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->ContentType);
	    localObject[i]->Object = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->Object);
	}

	/* Fixup static object pointers */
	localObject = (TemplateObjectOverlay **)((unsigned char **)((unsigned char *)Template + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Template->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Template->StringCount+
									Template->ImageCount
								)*
								Template->LanguageCount
							) +
							sizeof(unsigned char *) * Template->NameCount +
							sizeof(unsigned char *) * Template->TemplateCount ));

	for (i = 0; i < Template->CommonStaticObjectCount; i++) {
	    localObject[i]->FileName = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->FileName);
	    localObject[i]->ContentType = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->ContentType);
	    localObject[i]->Object = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->Object);
	}

	return(TRUE);
}

BOOL
SetPublicSessionTemplate(ConnectionStruct *Client, SessionStruct *Session)
{
	unsigned long *LanguageList = (unsigned long *)((unsigned char *)PublicTemplate + sizeof(TemplateStruct));
	unsigned long i;
	signed long Language = -1;


	/* Determine language; check Client->Language */
	FindLanguage(Client->Language, Language);
	
	for (i = 0; i < PublicTemplate->LanguageCount;i++) {
		if (LanguageList[i] == Language) {
			Language = i;
			break;
		}
	}

	if (i == PublicTemplate->LanguageCount) {
		Language = 0;
	}

	Session->Language=Language;

	Session->Strings=(unsigned char **)((unsigned char *)PublicTemplate + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									PublicTemplate->StringCount+
									PublicTemplate->ImageCount
								)*
								Session->Language
							));
	Session->Images = (unsigned char **)((unsigned char *)Session->Strings + PublicTemplate->StringCount * sizeof(unsigned char *));

	Session->Pages = (TemplateObjectOverlay **)((unsigned char *)PublicTemplate + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									PublicTemplate->StringCount+
									PublicTemplate->ImageCount
								)*
								PublicTemplate->LanguageCount
							) +
							sizeof(unsigned char *) * PublicTemplate->NameCount);

	Session->StaticObjects = (TemplateObjectOverlay **)((unsigned char *)PublicTemplate + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									PublicTemplate->StringCount+
									PublicTemplate->ImageCount
								)*
								PublicTemplate->LanguageCount
							) +
							sizeof(unsigned char *) * PublicTemplate->NameCount +
							sizeof(unsigned char *) * PublicTemplate->TemplateCount );

	return(TRUE);
}

static BOOL
GeneratePublicImages(void)
{
	unsigned long	Language;

	TemplateDebug("GeneratePublicImages() called\n");

	PublicTemplateImages=MemMalloc(sizeof(unsigned char *)*PublicTemplate->LanguageCount);
	if (!PublicTemplateImages) {
		return(FALSE);
	}

	for (Language=0; Language<PublicTemplate->LanguageCount; Language++) {
		PublicTemplateImages[Language]=(unsigned char **)((unsigned char *)PublicTemplate + 
					sizeof(TemplateStruct) + 
					sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
					(
						sizeof(unsigned char *) *
						(
							PublicTemplate->StringCount+
							PublicTemplate->ImageCount
						)*
						Language
						+ sizeof(unsigned char *) *
						PublicTemplate->StringCount
					));
	}

	return(TRUE);
}

static BOOL
GenerateTemplateImages(void)
{
	unsigned long	Template;
	unsigned long	Language;

	TemplateDebug("GenerateTemplateImages() called\n");

	for (Template=0; Template<TemplateCount; Template++) {
		if (Templates[Template]->LanguageCount>TemplateMaxLanguages) {
			TemplateMaxLanguages=Templates[Template]->LanguageCount;
		}
	}

	TemplateImages=MemMalloc(sizeof(unsigned char *)*TemplateCount*TemplateMaxLanguages);
	if (!TemplateImages) {
		return(FALSE);
	}

	for (Template=0; Template<TemplateCount; Template++) {
		for (Language=0; Language<TemplateMaxLanguages; Language++) {
			if (Language<Templates[Template]->LanguageCount) {
				TemplateImages[Template*TemplateMaxLanguages+Language]=(unsigned char **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Language
								+ sizeof(unsigned char *) *
								Templates[Template]->StringCount
							));
			} else {
				TemplateImages[Template*TemplateMaxLanguages+Language]=NULL;
			}
		}
	}
	return(TRUE);
}

BOOL
LoadTemplates(unsigned char *TemplatePath)
{
	unsigned char	Path[XPL_MAX_PATH+1];
	XplDir			*Directory;
	XplDir			*DirectoryEntry;
	FILE				*TemplateFile;
	struct stat		sb;
	unsigned char	*Template;
	MDBValueStruct	*TemplateList;
	MDBValueStruct	*DefaultTemplateDN;
	unsigned char	*ptr;
	unsigned char	*DefaultTemplateName="";
	unsigned long	i;
	TemplateStruct	CheckTemplate;

	TemplateDebugInit();

	/* We read what templates are supposed to be there... */
	TemplateList = MDBCreateValueStruct(ModWebDirectoryHandle, MsgGetServerDN(NULL));
	DefaultTemplateDN = MDBCreateValueStruct(ModWebDirectoryHandle, MsgGetServerDN(NULL));

	MDBReadDN(MSGSRV_AGENT_MODWEB, MSGSRV_A_TEMPLATE, TemplateList);
	if (MDBReadDN(MSGSRV_AGENT_MODWEB, MSGSRV_A_DEFAULT_TEMPLATE, DefaultTemplateDN)>0) {
		DefaultTemplateName=strrchr(DefaultTemplateDN->Value[0], '\\');
		if (!DefaultTemplateName) {
			DefaultTemplateName=DefaultTemplateDN->Value[0];
		} else {
			DefaultTemplateName++;
		}
	}
	

	/* We load precompiled templates only, they need to be compiled separately */
	snprintf(Path, sizeof(Path), "%s", TemplatePath);
	if ((Directory=XplOpenDir(Path))!=NULL) {
		while ((DirectoryEntry=XplReadDir(Directory))!=NULL) {
			if ((strlen(DirectoryEntry->d_nameDOS)<5) || (MWQuickNCmp(DirectoryEntry->d_nameDOS+strlen(DirectoryEntry->d_nameDOS)-4, TEMPLATE_EXTENSION, strlen(TEMPLATE_EXTENSION))==FALSE)) {
				continue;
			}
			snprintf(Path, sizeof(Path), "%s/%s", TemplatePath, DirectoryEntry->d_nameDOS);
			if (stat(Path, &sb)!=0) {
				continue;
			}

			TemplateFile=fopen(Path, "rb");
			if (!TemplateFile) {
				continue;
			}

			/* Let's find out if we're supposed to be using/loading this template */
			if (fgets((unsigned char *)&CheckTemplate, sizeof(TemplateStruct), TemplateFile)!=NULL) {
				XplConsolePrintf("\rif tree\n");
				/* We handle the case if DefaultTemplate is not part of supported templates */
				if (CheckTemplate.VersionID!=TEMPLATE_CURRENT_VERSION) {
					fclose(TemplateFile);
					XplConsolePrintf("\rMODWEBD: Template %s has wrong version number\n", DirectoryEntry->d_nameDOS);
					XplConsolePrintf("\r         Expected %x, found %x\n", (unsigned int)TEMPLATE_CURRENT_VERSION, (unsigned int)CheckTemplate.VersionID);
					continue;
				}

				/* Try to find a public template */
				if (!PublicTemplate && MWQuickCmp(CheckTemplate.Name, PUBLIC_TEMPLATE_NAME)) {
					PublicTemplate=MemMalloc(sb.st_size);
					if (!PublicTemplate) {
						fclose(TemplateFile);
						continue;
					}
					
					fseek(TemplateFile, 0, SEEK_SET);
					fread(PublicTemplate, sb.st_size, 1, TemplateFile);
					fclose(TemplateFile);

					FixupTemplate((TemplateStruct *)PublicTemplate);

					if (!GeneratePublicImages()) {
						MemFree(PublicTemplate);
						PublicTemplate=NULL;
					}
					continue;
				}

				XplConsolePrintf("\rstill there\n");
				if (MWQuickCmp(CheckTemplate.Name, DefaultTemplateName)) {
					XplConsolePrintf("\rmwquickcmp\n");
					for (i=0; i<TemplateList->Used; i++) {
						ptr=strrchr(TemplateList->Value[i], '\\');
						if (!ptr) {
							ptr=TemplateList->Value[i];
						} else {
							ptr++;
						}
						if (MWQuickCmp(CheckTemplate.Name, ptr)) {
							i=TemplateList->Used+2;
						}
					}

					if (i==TemplateList->Used) {
						MDBAddValue(CheckTemplate.Name, TemplateList);
					}
				}


				for (i=0; i<TemplateList->Used; i++) {
					ptr=strrchr(TemplateList->Value[i], '\\');
					if (!ptr) {
						ptr=TemplateList->Value[i];
					} else {
						ptr++;
					}

					if (MWQuickCmp(CheckTemplate.Name, ptr)) {
						if (i<TemplateList->Used) {
							TemplateList->Value[i][0]='\0';
						}
						i=TemplateList->Used+2;
					}
				}
				if (i==TemplateList->Used) {
					fclose(TemplateFile);
					XplConsolePrintf("\rcontinue\n");
					continue;
				}
			}

			Template=MemMalloc(sb.st_size);
			if (!Template) {
				fclose(TemplateFile);
				XplConsolePrintf("\rlast one\n");
				continue;
			}

			fseek(TemplateFile, 0, SEEK_SET);
			fread(Template, sb.st_size, 1, TemplateFile);
			fclose(TemplateFile);
			/* Scan for the related strings and associate them with the template */
			TemplateDebug("Adding template %s\n", DirectoryEntry->d_nameDOS);
			AddTemplate((TemplateStruct *)Template);

			/* Try to set our "default" template */
			if (MWQuickCmp(Templates[TemplateCount-1]->Name, DefaultTemplateName)) {
				DefaultTemplate=TemplateCount-1;
			}
			FixupTemplate((TemplateStruct *)Template);
		}
		XplCloseDir(Directory);
	}

	/* Do some nice error reporting */
	for (i=0; i<TemplateList->Used; i++) {
		if (TemplateList->Value[i][0]!='\0') {
			ptr=strrchr(TemplateList->Value[i], '\\');
			if (!ptr) {
				ptr=TemplateList->Value[i];
			} else {
				ptr++;
			}
			XplConsolePrintf("\rMODWEBD: Could not find or load template %s\n", ptr);
			XplBell();
			XplDelay(1500);
		}
	}
	MDBDestroyValueStruct(TemplateList);
	MDBDestroyValueStruct(DefaultTemplateDN);

	if (!GenerateTemplateImages()) {
		return(FALSE);
	}

	return(TRUE);
}

BOOL
MWSetSessionTemplate(unsigned long Template, unsigned long Language, SessionStruct *Session)
{
	unsigned long	*LanguageList=(unsigned long *)((unsigned char *)Templates[Template] + sizeof(TemplateStruct));
	unsigned long	i;

	if (Template>=TemplateCount) {
		Template=DefaultTemplate;
	}

	TemplateDebug("MWSetSessionTemplate: Template %d (%s), Language %d, Session %x)\n", (int)Template, Templates[Template]->Name, (int)Language, (unsigned int)Session);

	for (i=0; i<Templates[Template]->LanguageCount; i++) {
		if (LanguageList[i]==Language) {
			Session->Language=i;
			break;
		}
	}
	if (i==Templates[Template]->LanguageCount) {
		TemplateDebug("MWSetSessionTemplate: Requested language %d not found\n", (int)Language);
		Session->Language=0;
	}

	Session->TemplateID=Template;

/*
	Format of compiled template file

	[ Template Name			]
	[ Fixed Data Area			]		-> sizeof(TemplateStruct)

	[ Language List			]			-> sizeof(unsigned long)*Languages
		[ Strings Table Lang 0	]		-> sizeof(unsigned char*)*Strings
		[ Images Table Lang 0	]		-> sizeof(unsigned char*)*Images
	
		[ Strings Table Lang  1	]
		[ Images Table Lang 1	]

		[ Strings Table Lang n	]
		[ Images Table Lang n	]

	[ Names table				]		-> sizeof(unsigned char *)*NameCount
	[ Templates table			]		-> sizeof(unsigned char *)*PageCount
   [ StaticObject table      ]     -> sizeof(unsigned char *)*CommonStaticObjectCount;
	[ Template Data Area		]

*/

	/* Check if language exists */
	Session->Strings=(unsigned char **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Session->Language
							));
	Session->Images = (unsigned char **)((unsigned char *)Session->Strings + Templates[Template]->StringCount * sizeof(unsigned char *));

	Session->Pages = (TemplateObjectOverlay **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Templates[Template]->LanguageCount
							) +
							sizeof(unsigned char *) * Templates[Template]->NameCount);

	Session->StaticObjects = (TemplateObjectOverlay **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Templates[Template]->LanguageCount
							) +
							sizeof(unsigned char *) * Templates[Template]->NameCount +
							sizeof(unsigned char *) * Templates[Template]->TemplateCount );

	TemplateDebug("MWSetSessionTemplate: Session %x configured\n", (unsigned int)Session);
	return(TRUE);
}
