/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

#include <mdb.h>

#include "webadminp.h"

/* Public "portal" template */
TemplateStruct	*PublicTemplate			= NULL;
unsigned char	***PublicTemplateImages	= NULL;

/* Session-based templates */
TemplateStruct	**Templates					= NULL;

int				TemplateCount				= 0;
unsigned char	***TemplateImages			= NULL;
int				TemplateMaxLanguages		= 0;

#if defined(DEBUG)
int		TemplateScreen;
#define	TemplateDebugInit()	TemplateScreen=XplCreateScreen("WA Template Debug Screen", 0)
#else
#define	TemplateDebugInit()
#endif


static void
AddTemplate(TemplateStruct *Template, unsigned long Priority, unsigned long *TemplateOrder)
{
	unsigned long		i;
	unsigned long		c;

	if (TemplateCount == 0) {
		memset(TemplateOrder, 0, sizeof(TemplateOrder));
	}

	/* Make sure that we don't already have a template by that name */
	for (i = 0; i < TemplateCount; i++) {
		if (WAQuickCmp(Template->Name, Templates[i]->Name)) {
			TemplateDebug("Template %s is already loaded.", Template->Name);
			return;
		}
	}

	Templates=realloc(Templates, (TemplateCount+1) * sizeof(TemplateStruct *));
	if (!Templates) {
		TemplateDebug("Could not add template, out of memory.");
		// syslog(LOG_ERR, "Could not add template, out of memory.");
		return;
	}

	/* Put it in the right order based on priority.  0 is highest, 9 is lowest */
	for (i = 0; i < TemplateCount; i++) {
		if (TemplateOrder[i] > Priority) {
			break;
		}
	}

	for (c = TemplateCount; c > i; c--) {
		Templates[c] = Templates[c - 1];
		TemplateOrder[c] = TemplateOrder[c - 1];
	}

 	Templates[i] = Template;
	TemplateOrder[i] = Priority;

	TemplateCount++;
	return;
}

void
FreeTemplates(void)
{
	int i;

	TemplateDebug("Freeing all templates\n");

	for (i=0; i<TemplateCount; i++) {
		free(Templates[i]);
	}

	if (Templates) {
		free(Templates);
	}

	if (TemplateImages) {
		free(TemplateImages);
	}

	if (PublicTemplateImages) {
		free(PublicTemplateImages);
	}

	if (PublicTemplate) {
		free(PublicTemplate);
	}

	TemplateImages=NULL;
	Templates=NULL;
	TemplateCount=0;
}


unsigned long
FindLanguage(unsigned long LanguageID, long Template)
{
	return(0);
}


long
WAFindTemplate(unsigned char *TemplateName)
{
	int	i;

	TemplateDebug("FindTemplate(%s)\n", TemplateName);

	for (i=0; i<TemplateCount; i++) {
		if (WAQuickCmp(TemplateName, Templates[i]->Name)) {
			TemplateDebug("FindTemplate(%s) found:%d\n", TemplateName, i);
			return(i);
		}
	}

	return(-1);
}

long
WAFindTemplatePage(unsigned char *Name, unsigned long Template)
{
	NameLookupStruct	**NLS;
	unsigned char		*ptr;
	unsigned long		i;

	ptr = (unsigned char *)Templates[Template] +
		sizeof(TemplateStruct) +
		sizeof(unsigned long)*Templates[Template]->LanguageCount +
		( 
			sizeof(unsigned char *) *
			(
				Templates[Template]->StringCount +
				Templates[Template]->ImageCount
			)*
			Templates[Template]->LanguageCount 
		);

	NLS = (NameLookupStruct **)ptr;
	for(i = 0; i < Templates[Template]->NameCount; i++) {
		if (WAQuickCmp(Name, NLS[i]->Name)) {
			return(NLS[i]->PageID);
		}
	}
	return(-1);
}

long
WAFindTaskPageEx(unsigned char *Name, unsigned long Length, TaskStruct *Task)
{
	NameLookupStruct	**NLS;

	if (!Task) {
		return(-1);
	}

	Task->ResultPage = 0;

	if (!Task->Pages) {
		Task->Pages = (unsigned char *)Templates[Task->Template] +
			sizeof(TemplateStruct) +
			sizeof(unsigned long)*Templates[Task->Template]->LanguageCount+
			( 
				sizeof(unsigned char *) *
				(
					Templates[Task->Template]->StringCount+
					Templates[Task->Template]->ImageCount
				)*
				Templates[Task->Template]->LanguageCount 
			);
	} else {
		/*
			If we have pages that means that we already found something, so Count
			is pointing to the task we already found.  Make sense?
		*/
		Task->Count++;
	}

	NLS = (NameLookupStruct **)Task->Pages;
	for(; Task->Count < Templates[Task->Template]->NameCount; Task->Count++) {
		if (WAQuickNCmp(Name, NLS[Task->Count]->Name, Length)) {
			Task->ResultPage = NLS[Task->Count]->PageID;
			HulaStrNCpy(Task->Name, NLS[Task->Count]->Name + Length, sizeof(Task->Name));

			return(NLS[Task->Count]->PageID);
		}
	}
	Task->Count = 0;
	Task->Pages = NULL;

	return(-1);
}

static BOOL
FixupTemplate(TemplateStruct *Template)
{
    TemplateObjectOverlay **localObject;
	unsigned long	*TemplateAddr=(unsigned long *)Template;
	unsigned long	*AddrPtr;
	unsigned long	i;
	unsigned long	FixupCount=Template->FixupCount;

	AddrPtr=(unsigned long *)((unsigned char *)TemplateAddr+sizeof(TemplateStruct)+sizeof(unsigned long)*Template->LanguageCount);
	for (i=0; i<FixupCount; i++) {
		AddrPtr[i]+=(unsigned long)TemplateAddr;
	}

	localObject = (TemplateObjectOverlay **)((unsigned char *)Template + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Template->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Template->StringCount+
									Template->ImageCount
								)*
								Template->LanguageCount
							) +
							sizeof(unsigned char *) * Template->NameCount);

	for (i = 0; i < Template->TemplateCount; i++) {
	    localObject[i]->FileName = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->FileName);
	    localObject[i]->ContentType = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->ContentType);
	    localObject[i]->Object = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->Object);
	}

	localObject = (TemplateObjectOverlay **)((unsigned char *)Template + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Template->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Template->StringCount+
									Template->ImageCount
								)*
								Template->LanguageCount
							) +
							sizeof(unsigned char *) * Template->NameCount +
							sizeof(unsigned char *) * Template->TemplateCount );

	for (i = 0; i < Template->CommonStaticObjectCount; i++) {
	    localObject[i]->FileName = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->FileName);
	    localObject[i]->ContentType = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->ContentType);
	    localObject[i]->Object = (unsigned char *)localObject[i] + (unsigned long)(localObject[i]->Object);
	}

	return(TRUE);
}

BOOL
SetPublicSessionTemplate(ConnectionStruct *Client, SessionStruct *Session)
{
	unsigned long			*LanguageList = (unsigned long *)((unsigned char *)PublicTemplate + sizeof(TemplateStruct));
	unsigned long			i;
	signed long				Language = -1;

	/* Determine language; check Client->Language */
	WAFindLanguage(Client->Language, Language);

	for (i = 0; i < PublicTemplate->LanguageCount; i++) {
		if (LanguageList[i] == Language) {
			Language = i;
			break;
		}
	}

	if (i == PublicTemplate->LanguageCount) {
		Language = 0;
	}

	Session->Strings=(unsigned char **)((unsigned char *)PublicTemplate + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									PublicTemplate->StringCount+
									PublicTemplate->ImageCount
								)*
								Language
							));
	Session->Images = (unsigned char **)((unsigned char *)Session->Strings + PublicTemplate->StringCount * sizeof(unsigned char *));

	Session->Pages = (TemplateObjectOverlay **)((unsigned char *)PublicTemplate + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									PublicTemplate->StringCount+
									PublicTemplate->ImageCount
								)*
								PublicTemplate->LanguageCount
							) +
							sizeof(unsigned char *) * PublicTemplate->NameCount);

	Session->StaticPages = (TemplateObjectOverlay **)((unsigned char *)PublicTemplate + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									PublicTemplate->StringCount+
									PublicTemplate->ImageCount
								)*
								PublicTemplate->LanguageCount
							) +
							sizeof(unsigned char *) * PublicTemplate->NameCount +
							sizeof(unsigned char *) * PublicTemplate->TemplateCount );
	return(TRUE);
}

static BOOL
GeneratePublicImages(void)
{
	unsigned long	Language;

	TemplateDebug("GeneratePublicImages() called\n");

	PublicTemplateImages=malloc(sizeof(unsigned char *)*PublicTemplate->LanguageCount);
	if (!PublicTemplateImages) {
		return(FALSE);
	}

	for (Language=0; Language<PublicTemplate->LanguageCount; Language++) {
		PublicTemplateImages[Language]=(unsigned char **)((unsigned char *)PublicTemplate + 
					sizeof(TemplateStruct) + 
					sizeof(unsigned long)*PublicTemplate->LanguageCount+ 
					(
						sizeof(unsigned char *) *
						(
							PublicTemplate->StringCount+
							PublicTemplate->ImageCount
						)*
						Language
						+ sizeof(unsigned char *) *
						PublicTemplate->StringCount
					));
	}

	return(TRUE);
}

static BOOL
GenerateTemplateImages(void)
{
	unsigned long	Template;
	unsigned long	Language;

	TemplateDebug("GenerateTemplateImages() called\n");

	for (Template=0; Template<TemplateCount; Template++) {
		if (Templates[Template]->LanguageCount>TemplateMaxLanguages) {
			TemplateMaxLanguages=Templates[Template]->LanguageCount;
		}
	}

	TemplateImages=malloc(sizeof(unsigned char *)*TemplateCount*TemplateMaxLanguages);
	if (!TemplateImages) {
		return(FALSE);
	}

	for (Template=0; Template<TemplateCount; Template++) {
		for (Language=0; Language<TemplateMaxLanguages; Language++) {
			if (Language<Templates[Template]->LanguageCount) {
				TemplateImages[Template*TemplateMaxLanguages+Language]=(unsigned char **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Language
								+ sizeof(unsigned char *) *
								Templates[Template]->StringCount
							));
			} else {
				TemplateImages[Template*TemplateMaxLanguages+Language]=NULL;
			}
		}
	}
	return(TRUE);
}

BOOL
LoadTemplates(unsigned char *TemplatePath)
{
	unsigned char		Path[XPL_MAX_PATH+1];
	XplDir				*Directory;
	XplDir				*DirectoryEntry;
	FILE					*TemplateFile;
	struct stat			sb;
	unsigned char		*Template;
	unsigned char		*DefaultTemplateName="Default";
	TemplateStruct		CheckTemplate;
	unsigned long		i;
	unsigned long		TemplateOrder[1000];

	TemplateDebugInit();

	/* We load precompiled templates only, they need to be compiled separately */
	snprintf(Path, sizeof(Path), "%s", TemplatePath);
	if ((Directory=XplOpenDir(Path))!=NULL) {
		while ((DirectoryEntry=XplReadDir(Directory))!=NULL) {
			if ((strlen(DirectoryEntry->d_nameDOS)<5) || (WAQuickNCmp(DirectoryEntry->d_nameDOS+strlen(DirectoryEntry->d_nameDOS)-4, TEMPLATE_EXTENSION, strlen(TEMPLATE_EXTENSION))==FALSE)) {
				continue;
			}
			snprintf(Path, sizeof(Path), "%s/%s", TemplatePath, DirectoryEntry->d_nameDOS);
			if (stat(Path, &sb)!=0) {
				continue;
			}

			TemplateFile=fopen(Path, "rb");
			if (!TemplateFile) {
				continue;
			}

			/* Let's find out if we're supposed to be using/loading this template */
			if (fgets((unsigned char *)&CheckTemplate, sizeof(TemplateStruct), TemplateFile)!=NULL) {
				/* We handle the case if DefaultTemplate is not part of supported templates */
				if (CheckTemplate.VersionID!=TEMPLATE_CURRENT_VERSION) {
					fclose(TemplateFile);
					XplConsolePrintf("\rWebAdmin: Template %s has wrong version number\n", DirectoryEntry->d_nameDOS);
					XplConsolePrintf("\r         Expected %x, found %lux\n", TEMPLATE_CURRENT_VERSION, CheckTemplate.VersionID);
					continue;
				}

				/* Try to find a public template */
				if (!PublicTemplate && WAQuickCmp(CheckTemplate.Name, PUBLIC_TEMPLATE_NAME)) {
					PublicTemplate=malloc(sb.st_size);
					if (!PublicTemplate) {
						fclose(TemplateFile);
						continue;
					}
					
					fseek(TemplateFile, 0, SEEK_SET);
					fread(PublicTemplate, sb.st_size, 1, TemplateFile);
					fclose(TemplateFile);

					FixupTemplate((TemplateStruct *)PublicTemplate);

					if (!GeneratePublicImages()) {
						free(PublicTemplate);
						PublicTemplate=NULL;
					}
					continue;
				}
			}

			Template=malloc(sb.st_size);
			if (!Template) {
				fclose(TemplateFile);
				continue;
			}

			fseek(TemplateFile, 0, SEEK_SET);
			fread(Template, sb.st_size, 1, TemplateFile);
			fclose(TemplateFile);
			/* Scan for the related strings and associate them with the template */
			TemplateDebug("Adding template %s\n", DirectoryEntry->d_nameDOS);
			AddTemplate((TemplateStruct *)Template, atol(DirectoryEntry->d_nameDOS), TemplateOrder);

			FixupTemplate((TemplateStruct *)Template);
		}
		XplCloseDir(Directory);
	}

	/* Try to set our "default" template */
	for (i = 0; i < TemplateCount; i++) {
		if (WAQuickCmp(Templates[i]->Name, DefaultTemplateName)) {
			DefaultTemplate=i;
		}
	}


	if (!GenerateTemplateImages()) {
		return(FALSE);
	}

	return(TRUE);
}

BOOL
WASetSessionTemplate(unsigned long Template, unsigned long Language, SessionStruct *Session)
{
	unsigned long	*LanguageList=(unsigned long *)((unsigned char *)Templates[Template] + sizeof(TemplateStruct));
	unsigned long	i;

	if (Template >= TemplateCount) {
		Template=DefaultTemplate;
	}

	for (i = 0; i < Templates[Template]->LanguageCount; i++) {
		if (LanguageList[i] == Language) {
			Language = i;
			break;
		}
	}

	if (i == Templates[Template]->LanguageCount) {
		Language = 0;
	}

	Session->TemplateID = Template;

/*
	Format of compiled template file

	[ Template Name			]
	[ Fixed Data Area			]		-> sizeof(TemplateStruct)
	[ Language List			]			-> sizeof(unsigned long)*Languages
		[ Strings Table Lang 0	]		-> sizeof(unsigned char*)*Strings
		[ Images Table Lang 0	]		-> sizeof(unsigned char*)*Images
	
		[ Strings Table Lang  1	]
		[ Images Table Lang 1	]

		[ Strings Table Lang n	]
		[ Images Table Lang n	]

	[ Names table				]		-> sizeof(unsigned char *)*NameCount
	[ Templates table			]		-> sizeof(unsigned char *)*PageCount
   [ StaticPage table      ]     -> sizeof(unsigned char *)*CommonStaticObjectCount;

	[ Template Data Area		]

*/

	/* Check if language exists */
	Session->Strings = (unsigned char **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Language
							));

	Session->Images = (unsigned char **)((unsigned char *)Session->Strings + Templates[Template]->StringCount * sizeof(unsigned char *));

	Session->Pages = (TemplateObjectOverlay **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Templates[Template]->LanguageCount
							) +
							sizeof(unsigned char *) * Templates[Template]->NameCount);

	Session->StaticPages = (TemplateObjectOverlay **)((unsigned char *)Templates[Template] + 
							sizeof(TemplateStruct) + 
							sizeof(unsigned long)*Templates[Template]->LanguageCount+ 
							(
								sizeof(unsigned char *) *
								(
									Templates[Template]->StringCount+
									Templates[Template]->ImageCount
								)*
								Templates[Template]->LanguageCount
							) +
							sizeof(unsigned char *) * Templates[Template]->NameCount +
							sizeof(unsigned char *) * Templates[Template]->TemplateCount );

	return(TRUE);
}
