/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#if defined(WIN32)
#include <config.h>

#include <xpl.h>
#include <xplutil.h>

#include <errno.h>

#include "conniop.h"

#include <openssl/bio.h>
#include <openssl/ssl.h>

#define IPerror()           WSAGetLastError()
#define IPclear()           WSASetLastError(0)

#define CONNECTION_TIMEOUT  (30*60)

int sock_write(BIO *h, const char *buf, int num);
int sock_read(BIO *h, char *buf, int size);
int sock_puts(BIO *h, const char *str);
long sock_ctrl(BIO *h, int cmd, long arg1, void *arg2);
int sock_new(BIO *h);
int sock_free(BIO *data);

static BIO_METHOD methods_bsdsockp =
{
	BIO_TYPE_SOCKET,
	"bsdsocket",
	sock_write,
	sock_read,
	sock_puts,
	NULL, /* sock_gets, */
	sock_ctrl,
	sock_new,
	sock_free,
	NULL
};

unsigned long
XplGetHostIPAddress(void)
{
	unsigned long	Address;
	unsigned char	Name[256];
	struct hostent	*he;

	gethostname(Name, sizeof(Name));

	he = gethostbyname(Name);
	if (he) {
		memcpy(&Address, he->h_addr_list[0], sizeof(Address));
		return(Address);
	}

	return(inet_addr("127.0.0.1"));
}

BOOL 
XplIsLocalIPAddress(unsigned long Address)
{
	int				i;
	unsigned char	name[256];
	struct hostent	*he;

	gethostname(name, sizeof(name));
	he = gethostbyname(name);
	if (he) {
		for (i = 0; he->h_addr_list[i] != NULL; i++) {
			if (Address == *((unsigned long *)(he->h_addr_list[i]))) {
				return(TRUE);
			}
		}
	}

	return((Address == inet_addr("127.0.0.1")));
}

static int 
IPshouldretry(int i)
{
	int err;

	if ((i == 0) || (i == -1))	{
		err = IPerror();

		switch (err) {
			case WSAEINTR:
			case WSAEINPROGRESS: {
				return(1);
				/* break; */
			}

			default: {
				return(0);
			}
		}
	}
	return(0);
}

BIO_METHOD *
BIO_s_bsdsocket(void)
{
	return(&methods_bsdsockp);
}

BIO *
BIO_new_bsdsocket(int fd, int close_flag)
{
	BIO *ret;

	ret = BIO_new(BIO_s_bsdsocket());
	if (ret) {
		BIO_set_fd(ret,fd,close_flag);
		return(ret);
	}

	return(NULL);
}

int 
sock_new(BIO *bi)
{
	bi->init=0;
	bi->num=0;
	bi->ptr=NULL;
	bi->flags=0;
	return(1);
}

static int 
sock_free(BIO *a)
{
	if (a) {
		if (a->shutdown) {
			if (a->init) {
				IPshutdown((a->num),2); 
				IPclose(a->num);
			}

			a->init = 0;
			a->flags = 0;
		}

		a->num = -1;
		return(1);
	}

	return(0);
}
	

#define	SocketReadyTimeout(Socket, Timeout)					\
	{																					\
		int                  ret;												\
		fd_set               readfds;											\
		struct timeval       timeout;											\
																						\
		FD_ZERO(&readfds);														\
		FD_SET((Socket), &readfds);											\
		timeout.tv_usec = 0;														\
		timeout.tv_sec = (Timeout);											\
		ret = select(FD_SETSIZE, &readfds, NULL, NULL, &timeout);	\
																						\
		if (ret < 1) {																\
			return(-1);																\
		}																				\
	}

int 
sock_read(BIO *b, char *out, int outl)
{
	int ret;

	SocketReadyTimeout(b->num, CONNECTION_TIMEOUT);

	IPclear();
	BIO_clear_retry_flags(b);
	ret = IPrecv(b->num, out, outl, 0);
	
	if (ret >= 0){
		return(ret);
	}

	if (IPshouldretry(ret))	{
		BIO_set_retry_read(b);
	}

	return(-1);
}

int 
sock_write(BIO *b, const char *in, int inl)
{
	int	ret;

	IPclear();
	ret = IPsend(b->num, (void *)in, inl, 0);
	BIO_clear_retry_flags(b);
	if (ret <= 0) {
		if (IPshouldretry(ret))	{
			BIO_set_retry_write(b);
		}
	}

	return(ret);
}

long 
sock_ctrl(BIO *b, int cmd, long num, void *ptr)
{
	long ret;
	int *ip;

	switch (cmd) {
		case BIO_CTRL_RESET: {
			num = 0;
			/*	Fall through	*/
		}

		case BIO_C_FILE_SEEK: {
			ret = 0;
			break;
		}

		case BIO_C_FILE_TELL:
		case BIO_CTRL_INFO: {
			ret = 0;
			break;
		}

		case BIO_C_SET_FD: {
			sock_free(b);
			b->num = *((int *)ptr);
			b->shutdown = (int)num;
			b->init = 1;
			ret = 1;
			break;
		}

		case BIO_C_GET_FD: {
			if (b->init) {
				ip = (int *)ptr;
				if (ip) {
					*ip = b->num;
				}

				ret = b->num;
			} else {
				ret = -1;
			}

			break;
		}

		case BIO_CTRL_GET_CLOSE: {
			ret = b->shutdown;
			break;
		}

		case BIO_CTRL_SET_CLOSE: {
			b->shutdown = (int)num;
			ret = 1;
			break;
		}

		case BIO_CTRL_PENDING:
		case BIO_CTRL_WPENDING: {
			ret = 0;
			break;
		}

		case BIO_CTRL_DUP:
		case BIO_CTRL_FLUSH: {
			ret = 1;
			break;
		}

		default: {
			ret = 0;
			break;
		}
	}

	return(ret);
}

int 
sock_puts(BIO *bp, const char *str)
{
	int	n;
	int	ret;

	n = strlen(str);
	ret = sock_write(bp,str,n);
	return(ret);
}

int 
SSL_set_bsdfd(SSL *s,int fd)
{
	BIO *bio;

	bio = BIO_new(BIO_s_bsdsocket());

	if (bio) {
		BIO_set_fd(bio,fd,BIO_NOCLOSE);
		SSL_set_bio(s,bio,bio);
		return(1);
	}

//	SSLerr(SSL_F_SSL_SET_FD,ERR_R_BUF_LIB);

	return(0);
}

/*  fixme - deprecated
*/
int
XplIPRead(void * socket, unsigned char *Buf, int Len, int readTimeout)
{
	int				rc;
	fd_set			readfds;
	struct timeval	timeout;

	FD_ZERO(&readfds);
	FD_SET( (int)socket,&readfds);
	timeout.tv_usec = 0;
	timeout.tv_sec = readTimeout;

	/* All sockets are closed when exiting.  */
	/* Closed sockets will error on read. */
	rc = select(FD_SETSIZE, &readfds, NULL,NULL,&timeout);
	if (rc > 0) {
		/*success - no special handling of read */
		return(recv((int)socket, Buf, Len, 0));
	}

	/* return -1 on timeout or select error  */
	return(-1);
}

int XplIPWrite(void * socket, unsigned char *Buf, int Len)
{
	return(send((int)socket, Buf, Len,0));
}

int 
XplIPReadSSL(void *sslctx, unsigned char *Buf, int Len, int timeout)
{
	return(SSL_read((SSL*)sslctx, (void*)Buf, Len));
}


int
XplIPWriteSSL(void *sslctx, unsigned char *Buf, int Len)
{
	return(SSL_write((SSL*)sslctx, (void*)Buf, Len));
}
#endif	/*	defined(WIN32)	*/
