/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <memmgr.h>
#include <hulautil.h>

#include <openssl/md5.h>
#include <openssl/rand.h>
#include <openssl/err.h>


/**** Begin HulaKeyword Code ***/


static BOOL
HulaKeywordAddWordToBitRows(HulaKeywordIndex **table, unsigned long *tableRows, unsigned long startRow, unsigned char *stringValue, unsigned long stringNum)
{
    unsigned long i;
    unsigned long j;
    HulaKeywordIndex *localTable;
    HulaKeywordIndex *tmp;

    localTable = *table;
    i = 0;

    do {
	if ((startRow + i) >= *tableRows) {

	    tmp = (HulaKeywordIndex *)realloc(localTable, sizeof(HulaKeywordIndex) * (*tableRows + 1));
	    if (!tmp) {
		*table = localTable;
		return(FALSE);
	    }

	    localTable = tmp;
	    for (j = 0; j < HULA_KEYWORD_VALUES_PER_BYTE; j++) {
		localTable[*tableRows][j] = 0;
	    }
	    (*tableRows)++;
	}
	
	localTable[startRow + i][toupper(stringValue[i])] |= (1 << stringNum);
	localTable[startRow + i][tolower(stringValue[i])] |= (1 << stringNum);


	if (stringValue[i] != '\0') {
	    i++;
	    continue;
	}
		
	break;
    } while (TRUE);	

    *table = localTable;

    return(TRUE);
}

static long
HulaKeywordIndexCreateIndexRow(HulaKeywordIndex **table, unsigned long *tableRows, unsigned long column, unsigned char **stringList, unsigned long stringStart, unsigned long stringCount)
{
    HulaKeywordIndex *tmp;
    HulaKeywordIndex *localTable;
    unsigned long localTableRow;
    unsigned long allocatedTableRows;
    unsigned long stringEnd;

    unsigned long nextTableRow;

    unsigned long blockStart;
    unsigned long blockOffset = 0;
    unsigned char lastChar = 0;
    unsigned long blockEnd = 0;
	
    long i; 

    stringEnd = stringStart + stringCount;
    localTable = *table;
    allocatedTableRows = *tableRows;
    localTableRow = *tableRows;

    /* this function adds one column that it writes to */

    tmp = (HulaKeywordIndex *)realloc(localTable, sizeof(HulaKeywordIndex) * (allocatedTableRows + 1));
    if (!tmp) {
	/* Fixme: clean up here */
	*table = localTable;
	*tableRows = allocatedTableRows;
	return(-1);
    }

    /* initialize row */
    localTable = tmp;
    for (i = 0; i < HULA_KEYWORD_VALUES_PER_BYTE; i++) {
	localTable[allocatedTableRows][i] = HULA_KEYWORD_HIGH_BIT_MASK;
    }

    allocatedTableRows++;
    blockStart = stringStart;

    do {

	/* find a block of strings that fits in a block or all has the same character in this position */
	blockEnd = blockStart;
	blockOffset = 0;
	lastChar = 0;

	do {
	    if (stringList[blockStart + blockOffset][column] > lastChar) {
		blockEnd = blockStart + blockOffset;
		lastChar = stringList[blockStart + blockOffset][column];
	    }

	    blockOffset++;

	    if ((blockStart + blockOffset) < stringEnd) {
		if (blockOffset < HULA_KEYWORD_BITS_PER_WORD) {
		    continue;
		}
		if (blockEnd == blockStart) {
		    continue;
		}
	    } else {
		if (blockEnd == blockStart) {
		    blockEnd = stringEnd;
		}
	    }
	    break;
	} while (TRUE);

	nextTableRow = allocatedTableRows;

	if ((blockEnd - blockStart) < HULA_KEYWORD_BITS_PER_WORD) {
	    /* We have a block that can be process, with out increasing column */
			
	    for (i = 0; (unsigned long)i < (blockEnd - blockStart); i++) {
		if (!HulaKeywordAddWordToBitRows(&localTable, &allocatedTableRows, nextTableRow, stringList[blockStart + i] + column, i)) {
		    /* Fixme: clean up here */
		    *table = localTable;
		    *tableRows = allocatedTableRows;
		    return(-1);
		}

		localTable[localTableRow][toupper(stringList[blockStart + i][column])] |= ((blockStart << HULA_KEYWORD_BTIM) | nextTableRow);
		localTable[localTableRow][tolower(stringList[blockStart + i][column])] |= ((blockStart << HULA_KEYWORD_BTIM) | nextTableRow);
	    }

	    /* initialize a dead row */
	    tmp = (HulaKeywordIndex *)realloc(localTable, sizeof(HulaKeywordIndex) * (allocatedTableRows + 1));
	    if (!tmp) {
		/* Fixme: clean up here */
		*table = localTable;
		*tableRows = allocatedTableRows;
		return(-1);
	    }

	    /* initialize row */
	    localTable = tmp;
	    for (i = 0; i < HULA_KEYWORD_VALUES_PER_BYTE; i++) {
		localTable[allocatedTableRows][i] = 0;
	    }

	    allocatedTableRows++;

	} else {
	    /* there are more than 31 string that have the same letter in the current column; got to dive deeper */
	    localTable[localTableRow][toupper(stringList[blockStart][column])] |= nextTableRow;
	    localTable[localTableRow][tolower(stringList[blockStart][column])] |= nextTableRow;

	    if (HulaKeywordIndexCreateIndexRow(&localTable, &allocatedTableRows, column + 1, stringList, blockStart, blockEnd - blockStart) != (long)(blockEnd - blockStart)) {
		/* Fixme: clean up here */
		*table = localTable;
		*tableRows = allocatedTableRows;
		return(-1);
	    }
	}

	blockStart = blockEnd;

    } while (blockStart < stringEnd);

    *table = localTable;
    *tableRows = allocatedTableRows;

    /* initialize row */
    for (i = 0; i < HULA_KEYWORD_VALUES_PER_BYTE; i++) {
    }


    return(stringCount);
}

/* HulaKeywordIndexCreate expects the stringList passes in to have a NULL passed in at the end of the list */
/* It also expects the list to be sorted in an ascii case insensitive order */

HulaKeywordIndex *
HulaKeywordIndexCreate(unsigned char **stringList)
{
    HulaKeywordIndex *table = NULL;
    HulaKeywordIndex *tmp;
    long count = 0;
    unsigned long tableRows = 0;
    unsigned long stringNum = 0;
    unsigned long i;

    /* We need to break the list into groups of (BITS_PER_WORD - 1) or less. */

    /* count members in the list */
    while (stringList[count] != NULL) {
	count++;
    }

    if ((unsigned long)count < HULA_KEYWORD_BITS_PER_WORD) {
	/* it already fits */
	do {
	    if (!HulaKeywordAddWordToBitRows(&table, &tableRows, 0, stringList[stringNum], stringNum)) {
		/* Fixme: clean up here */
		return(NULL);
	    }
	    stringNum++;
	} while (stringList[stringNum] != NULL);

	/* initialize a dead row */
	tmp = (HulaKeywordIndex *)realloc(table, sizeof(HulaKeywordIndex) * (tableRows + 1));
	if (!tmp) {
	    /* Fixme: clean up here */
	    return(NULL);
	}

	/* initialize row */
	table = tmp;
	for (i = 0; i < HULA_KEYWORD_VALUES_PER_BYTE; i++) {
	    table[tableRows][i] = 0;
	}
				
	return(table);
    }

    if (HulaKeywordIndexCreateIndexRow(&table, &tableRows, 0, stringList, 0, count) != count) {
	return(NULL);
    }
	
    return(table);
}

long
HulaKeywordFind(HulaKeywordIndex *keywordTable, unsigned char *searchString)
{
    unsigned long tableRow;
    unsigned long matches;
    unsigned long nextMatches;
    unsigned long baseID;
    unsigned long count;
 
    tableRow = 0;
    baseID = 0;
    count = 0;

    do {
	if (keywordTable[tableRow][*searchString] & HULA_KEYWORD_HIGH_BIT_MASK) {
	    baseID += ((keywordTable[tableRow][*searchString] & HULA_KEYWORD_STRING_INDEX_MASK) >> HULA_KEYWORD_BTIM);
	    tableRow = keywordTable[tableRow][*searchString] & HULA_KEYWORD_TABLE_INDEX_MASK;
	    searchString++;
	    continue;
	}

	break;
    } while (TRUE);

    searchString--;
    matches = HULA_KEYWORD_FULL_MASK;

    do {
	nextMatches = matches & keywordTable[tableRow][*searchString];
	if (nextMatches) {
	    tableRow++;
	    searchString++;
	    matches = nextMatches;
	    continue;
	}

	break;
    } while (TRUE);

    nextMatches = matches;
    do {
	if ((matches & 0x1) == 0) {
	    matches >>= 1;
	    count++;
	    continue;
	}
			
	break;					
    } while(TRUE);

    matches >>= 1;

    if (!matches) {
	if (keywordTable[tableRow][0] & nextMatches) {  /* begins with case */
	    return(count + baseID);
	}

	tableRow--;
	if (keywordTable[tableRow][0] & nextMatches) { /* matches exactly */
	    return(count + baseID);
	}
    }

    return(-1);
}

void
HulaKeywordIndexFree(HulaKeywordIndex *Index)
{
    free(Index);
}

/**** End HulaKeyword Code ***/

const unsigned char *Base64Chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";

BOOL
QuickNCmp(unsigned char *str1, unsigned char *str2, int len)
{
	while (--len && *str1 && *str2 && toupper(*str1) == toupper(*str2)) {
		str1++;
		str2++;
	}

	return(toupper(*str1) == toupper(*str2));
}

BOOL
QuickCmp(unsigned char *str1, unsigned char *str2)
{
	while (*str1 && *str2 && toupper(*str1) == toupper(*str2)) {
		str1++;
		str2++;
	}

	return(toupper(*str1) == toupper(*str2));
}

static void
ProtocolCommandTreeRotateLeft(ProtocolCommandTree *tree, ProtocolCommand *command)
{
    register ProtocolCommand *x = command;
    register ProtocolCommand *y;
                                                                                                                                                                            
    y = x->right;
                                                                                                                                                                            
    x->right = y->left;
                                                                                                                                                                            
    if (y->left != tree->sentinel) {
        y->left->parent = x;
    }
                                                                                                                                                                            
    y->parent = x->parent;
                                                                                                                                                                            
    if (x->parent != tree->sentinel) {
        if (x == x->parent->left) {
            x->parent->left = y;
        } else {
            x->parent->right = y;
        }
    } else {
        tree->root = y;
    }
                                                                                                                                                                            
    y->left = x;
                                                                                                                                                                            
    x->parent = y;
                                                                                                                                                                            
    return;
}
                                                                                                                                                                            
static void
ProtocolCommandTreeRotateRight(ProtocolCommandTree *tree, ProtocolCommand *command)
{
    ProtocolCommand *x = command;
    ProtocolCommand *y;
                                                                                                                                                                            
    y = x->left;
                                                                                                                                                                            
    x->left = y->right;
                                                                                                                                                                            
    if (y->right != tree->sentinel) {
        y->right->parent = x;
    }
                                                                                                                                                                            
    y->parent = x->parent;
                                                                                                                                                                            
    if (x->parent != tree->sentinel) {
        if (x == x->parent->right) {
            x->parent->right = y;
        } else {
            x->parent->left = y;
        }
    } else {
        tree->root = y;
    }
                                                                                                                                                                            
    y->right = x;
                                                                                                                                                                            
    x->parent = y;
                                                                                                                                                                            
    return;
}
                                                                                                                                                                            
ProtocolCommand *
ProtocolCommandTreeSearch(ProtocolCommandTree *tree, const unsigned char *command)
{
    const unsigned char *k1;
    const unsigned char *k2;
    ProtocolCommand *x;
    ProtocolCommand *y;
                                                                                                                                                                            
    if (tree && command) {
        y = tree->sentinel;
        x = tree->root;
                                                                                                                                                                            
        k1 = x->name;
        k2 = command;
        while (x != tree->sentinel) {
            y = x;
                                                                                                                                                                            
            if (*k1 < toupper(*k2)) {
                x = x->left;
                                                                                                                                                                            
                k1 = x->name;
                k2 = command;

                continue;
            }

            if (*k1 > toupper(*k2)) {
                x = x->right;
                                                                                                                                                                            
                k1 = x->name;
                k2 = command;

                continue;
            }

            if (*(++k1)) {
                k2++;
                continue;
            }

            if (!*(++k2) || (*k2 == ' ')) {
                return(x);
            }
        }
    }
                                                                                                                                                                            
    return(NULL);
}
                                                                                                                                                                            
static void
ProtocolCommandTreeInsert(ProtocolCommandTree *tree, ProtocolCommand *command)
{
    const unsigned char *k1;
    const unsigned char *k2;
    BOOL l;
    ProtocolCommand *x;
    ProtocolCommand *y;
                                                                                                                                                                            
    if (tree && command) {
        y = tree->sentinel;
        x = tree->root;
                                                                                                                                                                            
        k1 = x->name;
        k2 = command->name;
        while (x != tree->sentinel) {
            y = x;
                                                                                                                                                                            
            if (*k1 < *k2) {
                l = TRUE;
                                                                                                                                                                            
                x = x->left;
                                                                                                                                                                            
                k1 = x->name;
                k2 = command->name;

                continue;
            }

            if (*k1 > *k2) {
                l = FALSE;
                                                                                                                                                                            
                x = x->right;
                                                                                                                                                                            
                k1 = x->name;
                k2 = command->name;

                continue;
            }

            if (*k1) {
                k1++;
                k2++;

                continue;
            }

            return;
        }
                                                                                                                                                                            
        XplSafeIncrement(tree->nodes);
                                                                                                                                                                            
        x = command;
                                                                                                                                                                            
        x->parent = y;
                                                                                                                                                                            
        if (y != tree->sentinel) {
            if (l == TRUE) {
                y->left = x;
            } else {
                y->right = x;
            }
        } else {
            tree->root = x;
        }
                                                                                                                                                                            
        x->left = tree->sentinel;
        x->right = tree->sentinel;
                                                                                                                                                                            
        x->color = RedCommand;
                                                                                                                                                                            
        while ((x != tree->root) && (x->parent->color == RedCommand)) {
            if (x->parent == x->parent->parent->left) {
                y = x->parent->parent->right;
                if (y->color == RedCommand) {
                    x->parent->color = BlackCommand;
                                                                                                                                                                            
                    y->color = BlackCommand;
                                                                                                                                                                            
                    x->parent->parent->color = RedCommand;
                                                                                                                                                                            
                    x = x->parent->parent;
                } else {
                    if (x == x->parent->right) {
                        x = x->parent;
                                                                                                                                                                            
                        ProtocolCommandTreeRotateLeft(tree, x);
                    }
                                                                                                                                                                            
                    x->parent->color = BlackCommand;
                                                                                                                                                                            
                    x->parent->parent->color = RedCommand;
                                                                                                                                                                            
                    ProtocolCommandTreeRotateRight(tree, x->parent->parent);
                }
            } else {
                y = x->parent->parent->left;
                if (y->color == RedCommand) {
                    x->parent->color = BlackCommand;
                                                                                                                                                                            
                    y->color = BlackCommand;
                                                                                                                                                                            
                    x->parent->parent->color = RedCommand;
                                                                                                                                                                            
                    x = x->parent->parent;
                } else {
                    if (x == x->parent->left) {
                        x = x->parent;
                                                                                                                                                                            
                        ProtocolCommandTreeRotateRight(tree, x);
                    }
                                                                                                                                                                            
                    x->parent->color = BlackCommand;
                                                                                                                                                                            
                    x->parent->parent->color = RedCommand;
                                                                                                                                                                            
                    ProtocolCommandTreeRotateLeft(tree, x->parent->parent);
                }
            }
        }
                                                                                                                                                                            
        tree->root->color = BlackCommand;
    }
                                                                                                                                                                            
    return;
}
                                                                                                                                                                            
void
LoadProtocolCommandTree(ProtocolCommandTree *tree, ProtocolCommand *commands)
{
    ProtocolCommand *cmd;
                                                                                                                                                                            
    if (tree) {
        cmd = &(tree->proxy);
                                                                                                                                                                            
        cmd->color = BlackCommand;
                                                                                                                                                                            
        cmd->parent = NULL;
                                                                                                                                                                            
        cmd->left = NULL;
        cmd->right = NULL;
                                                                                                                                                                            
        cmd->name = NULL;
        cmd->help = NULL;
                                                                                                                                                                            
        cmd->length = 0;
                                                                                                                                                                            
        cmd->handler = NULL;
                                                                                                                                                                            
        tree->sentinel = cmd;
        tree->root = cmd;
                                                                                                                                                                            
        XplSafeWrite(tree->nodes, 0);
                                                                                                                                                                            
        cmd = commands;
                                                                                                                                                                            
        while (cmd->name) {
            ProtocolCommandTreeInsert(tree, cmd);
                                                                                                                                                                            
            cmd++;
        }
    }
                                                                                                                                                                            
                                                                                                                                                                            
    return;
}
                                                                                                                                                                            
unsigned char *
DecodeBase64(unsigned char *EncodedString)
{
    int i;
    int j;
    int length = strlen(EncodedString);
    unsigned char *buffer = MemStrdup(EncodedString);
    unsigned char *ptr = buffer;
    unsigned char table[257];
    unsigned char *ptr2;

    memset(table, 127, 256);
    table[256] = '\0';

    for (i = 0; i < 65; i++) {
        table[Base64Chars[i]] = i;
    }

    for (i=0; i<length; i++) {
        if ((*ptr = table[(unsigned char)buffer[i]]) <= 64) {
            ptr++;
        }
    }

    length = ((int)(ptr - buffer) & ~3);

    ptr = ptr2 = buffer;

    for (i = 0; i < length; i += 4) {
        if (ptr[3] == 64) {
            if (ptr[2] == 64) {
                j  = *ptr++ << 2;
                j += *ptr++ >> 4;

                *ptr2++ = (unsigned char)(j & 255);
    
                break;
            }

            j  = (*ptr++ << 10);
            j += (*ptr++ <<  4);
            j += (*ptr++ >>  2);

            *ptr2++ = (unsigned char)(j >> 8);
            *ptr2++ = (unsigned char)(j &  255);

            break;
        }

        j  = (*ptr++ << 18);
        j += (*ptr++ << 12);
        j += (*ptr++ <<  6);
        j += (*ptr++);

        *ptr2++ = (unsigned char)((j>>16) & 255);
        *ptr2++ = (unsigned char)((j>>8)  & 255);
        *ptr2++ = (unsigned char)( j      & 255);
    }

    strncpy(EncodedString, buffer, (int)(ptr2 - buffer));

    EncodedString[(int)(ptr2 - buffer)]='\0';

    MemFree(buffer);

    return(EncodedString);
}

unsigned char *
EncodeBase64(const unsigned char *UnencodedString)
{
    int length;
    int lengthOut;
    int groups;
    int i;
    int lines;
    int width = 0;
    int na;
    int nb;
    int nc;
    int remaining;
    unsigned char *buffer;
    unsigned char *out;
    unsigned char cha;
    unsigned char chb;
    unsigned char chc;
    unsigned char chd;
    unsigned char *ptr;
    const unsigned char *in;

    if (UnencodedString) {
        length = strlen(UnencodedString);
        if (!length) {
            return(MemStrdup("\r\n"));
        }
    } else {
        return(NULL);
    }
    
    lengthOut = ((length+2)/3)*4;
    groups = length / 3;
	lines = (lengthOut + 75) / 76;

	lengthOut += 2 * lines;

    buffer = MemMalloc(lengthOut);
    out = buffer;
    in = UnencodedString;

    for (i = 0; i < groups; i++)
    {
        na = (unsigned char)(*in++);
        nb = (unsigned char)(*in++);
        nc = (unsigned char)(*in++);

        cha = (unsigned char)(na >> 2);
        chb = (unsigned char)(((na << 4) + (nb >> 4)) & 63);
        chc = (unsigned char)(((nb << 2) + (nc >> 6)) & 63);
        chd = (unsigned char)(nc & 63);

        *out++ = Base64Chars[cha];
        *out++ = Base64Chars[chb];
        *out++ = Base64Chars[chc];
        *out++ = Base64Chars[chd];

		/* insert a CRLF every 76 characters */
		if (((width += 4) == 76) && lines) {
			*out++ = '\r';
			*out++ = '\n';

			width = 0;
			lines--;
		}
    }

    remaining = length - groups * 3;

    if (remaining == 1) {
        na = (unsigned char)(*in++);

        *out++ = Base64Chars[na >> 2];
        *out++ = Base64Chars[(na & 3) << 4];
        *out++ = '=';
        *out++ = '=';
    } else if (remaining == 2) {
        na = (unsigned char)(*in++);
        nb = (unsigned char)(*in++);

        *out++ = Base64Chars[na>>2];
        *out++ = Base64Chars[((na&3)<<4)+(nb>>4)];
        *out++ = Base64Chars[((nb&15)<<2)];
        *out++ = '=';
    }


	if (lines) {
		*out++ = '\r';
		*out++ = '\n';
	}

    ptr = malloc(lengthOut + 1);
    if (ptr) {
        strncpy(ptr, buffer, lengthOut);
        ptr[lengthOut] = '\0';
    }

    MemFree(buffer);

    return(ptr);
}

BOOL 
HashCredential(const unsigned char *DN, unsigned char *Credential, unsigned char *Hash)
{
    unsigned char *delim;
    unsigned char *srcPtr;
    unsigned char *dstPtr;
    unsigned char *dstEnd;
    unsigned char digest[NMAP_CRED_DIGEST_SIZE];
    unsigned long i;
    unsigned long len;
    MD5_CTX mdContext;

    if (Credential && DN && (DN[0] == '\\') && ((delim = strchr(DN + 1, '\\')) != NULL)) {
        len = strlen(Credential);
        if (len >= NMAP_CRED_STORE_SIZE) {
            srcPtr = Credential;
            dstPtr = Hash;
            dstEnd = Hash + NMAP_HASH_SIZE;
            
            MD5_Init(&mdContext);
            MD5_Update(&mdContext, srcPtr,  NMAP_CRED_CHUNK_SIZE);    
            MD5_Update(&mdContext, DN, delim - DN);
            MD5_Final(digest, &mdContext);

            srcPtr += NMAP_CRED_CHUNK_SIZE;

            for (i = 0; i < NMAP_CRED_DIGEST_SIZE; i++) {
                if (dstPtr < dstEnd) {
                    *dstPtr = digest[i];
                    dstPtr++;
                }
            }

             do {
                MD5_Init(&mdContext);
                MD5_Update(&mdContext, srcPtr,  NMAP_CRED_CHUNK_SIZE);    
                MD5_Update(&mdContext, Hash, dstPtr - Hash);
                MD5_Final(digest, &mdContext);

                srcPtr += NMAP_CRED_CHUNK_SIZE;

                for (i = 0; i < NMAP_CRED_DIGEST_SIZE; i++) {
                    if (dstPtr < dstEnd) {
                        *dstPtr = digest[i];
                        dstPtr++;
                    }
                }
            } while (dstPtr < dstEnd);

            /* Hash now contains a non-terminated 128 byte octet string */

            return(TRUE);
        }
    }

    return(FALSE);
}
