/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

/* Calendaring Notes 

1. In create appointment and view appointment, the Organizer, Required, Optional, and Not Attending fields changed.

Old name	New Name
Organizer	From
Required	To
Optional	Cc
Not Attending	Bc
 
The code behind the From and To forms did not change. Bc has been updated to not send attendee status information regarding
the Bc recipient to other attendees. Only the organizer receives status updates for Bc attendee(s). -- This still conforms to the ICAL RFC. 

2. The Organizer(From recipient) of a calendar object has been separated from the list of recipients. This means that he can now choose
   to not be a part of the appointment, task, or note. The From recipient still requires a calendar entry in his own calendar in order to 
   recieve status updates(according to ICAL RFC). If he chooses to be listed as a recipient then a message will be delivered to his INBOX 
   similar to other recipients. He can then accept or decline the message from either the INBOX or MAIN calendar. 

3. To support #2 accept button has been added to calendar day view, and to objects opened within the calendar. Buttons are context sensitive 
   to only display when appropriate. 

4. Four new icons have been added to the Calendar view appointment screen which allow calendar users to toggle between showing their appointments as free 
   and showing their appointments as busy during a free busy search.  A grey'ed and selected button for Show Appointment as Busy and a grey'ed and 
   seleted button for show appointment as free along with the following following two help strings:
   - "Show Appointment as Busy" 
   - "Show Appointment as Free"

   The method used to conduct a free/busy search has not changed. 

   Here are the defaults for showing free or busy on calendar items.
   to recipients - Show as busy
   cc recipients - Show as busy
   Bc recipients - Show time as free
   From a to, or cc recipient - Show as busy 
   From a bc recipient, or not a to, cc, or bc recipient - Show as free 

   note:(check has been added to make sure at least one recipient is listed in form before allowing calendar item to be sent.)

5. New icons in recipient's view of his day calendar have been added to indicate visual status on whether he is free or busy, 
   according to the rules outlined in above. 

6. New icon for task completed (as distinguished from task not completed).

7. New icons for calendar items that need attention have also been added. 

8. Icon summary:
   Appointments:
   - Appointment needs action (unopened appointment)
   - Appointment inactive, shows up free in busy search (greyed open book)
   - Appointment active, shows up busy in search (open book with clock)

   Notes:
   - only one icon, to show all states 

   Tasks:
   - Task needs action (unopened notepad )
   - Task is active, has been accepted (opened notepad with checks)
   - Task is complete, user has marked it as complete, or has nothing left to act on (notepad with green check mark)

9. Tasks have been changed to only be displayed on today's calendar day or on the day view of the day the task was completed. 
   If the task needs action or has been accepted, but has not yet been completed then the task will show on today's date 
   only. If the task has been completed then the task will show up on the date it was completed and not on any other day. 


*/

#include <config.h>
#define MSGDATE_H_NEED_DAYS_PER_MONTH
#define NMLOGID_H_NEED_LOGGING_KEY
#define NMLOGID_H_NEED_LOGGING_CERT
#include <xpl.h>
#include <connio.h>

#include <mdb.h>
#include <msgapi.h>
#include <msgdate.h>
#include <nmap.h>
#include <msgaddr.h>

#include <modweb.h>
#include <mwtempl.h>
#include <hulautil.h>

MWAPIDefine;
#if 1
#include "mwcal.h"
#include "mwcaltz.h"


#if !defined(A_INTERNET_EMAIL_ADDRESS)
#define	A_INTERNET_EMAIL_ADDRESS	"Internet EMail Address"
#endif

unsigned char		*COMPOSE_EXT_TO_LIST[]	= { "to", "cc", "bcc" };
unsigned long		COMPOSE_TYPE_LIST[]	= {ICAL_ROLE_REQUIRED_PARTICIPANT, ICAL_ROLE_OPTIONAL_PARTICIPANT, ICAL_ROLE_NON_PARTICIPANT};
MwCalGlobal MwCal;


#if defined(MODWEB_DEBUG_BUILD)
int				ModScreen;
#endif

#define	ProcessMemoryStream(Length)									\
	MemoryInCodec.StreamLength=(Length);								\
	MemoryInCodec.Codec(&MemoryInCodec, MemoryInCodec.Next);		\
	MWResetStreamChain(&MemoryInCodec);

#define	ProcessFileStream(File)											\
	FileInCodec.StreamData=(File);										\
	FileInCodec.Codec(&FileInCodec, FileInCodec.Next);				\
	MWResetStreamChain(&FileInCodec);

#define	MWCalStreams														\
	StreamStruct			*RFC1522Codec		= NULL;					\
	StreamStruct			*Base64Codec		= NULL;					\
	StreamStruct			*BodyCharsetCodec = NULL;					\
	StreamStruct			*BodyQPCodec		= NULL;					\
	StreamStruct			*MessageOutCodec	= NULL;					\
	StreamStruct			*CRLFCodec			= NULL;					\
	StreamStruct			*RFC2445Codec		= NULL;					\
	StreamStruct			*RFC2445MLCodec	= NULL;

#define	ReleaseStreams																								\
	if (RFC1522Codec) {		MWReleaseStream(RFC1522Codec);		RFC1522Codec = NULL;			}	\
	if (Base64Codec) {		MWReleaseStream(Base64Codec);			Base64Codec = NULL;			}	\
	if (BodyCharsetCodec) {	MWReleaseStream(BodyCharsetCodec);	BodyCharsetCodec = NULL;		}	\
	if (BodyQPCodec) {		MWReleaseStream(BodyQPCodec);			BodyQPCodec = NULL;			}	\
	if (RFC2445Codec) {		MWReleaseStream(RFC2445Codec);		RFC2445Codec = NULL;			}	\
	if (RFC2445MLCodec) {	MWReleaseStream(RFC2445MLCodec);		RFC2445MLCodec = NULL;		}	\
	if (MessageOutCodec) {	MWReleaseStream(MessageOutCodec);	MessageOutCodec = NULL;			}	\
	if (CRLFCodec) {		MWReleaseStream(CRLFCodec);			CRLFCodec = NULL;				}

#define	RestoreUserNMAP													\
	if (ClientNMAPs) {														\
		MWSendNMAPServer(Session, "QUIT\r\n", 6);						\
		MWGetNMAPAnswer(Session, Buffer, BUFSIZE, FALSE);			\
		IPclose(Session->NMAPs);											\
		/* "flush" our input buffer */									\
		Session->NBufferPtr=0;												\
		Session->NMAPs=ClientNMAPs;										\
		Session->NMAPAddress.s_addr=NMAPCAddress;						\
	}

#define	SendWithBreaks(String, Len)									\
	{																				\
		unsigned char	*Begin = String;									\
		unsigned char	*Current;											\
		unsigned char	*End = Begin + Len;								\
																					\
		Current = (Begin);													\
		while (Current < (End)) {											\
			if (Current[0]=='\\') {											\
				MWHTMLSendClient(Client, (Begin), (Current - (Begin)));\
				if (toupper(Current[1]) =='N') {							\
					MWSendClient(Client, "<BR>", 4);						\
					Current++;													\
				}																	\
				Begin = Current+1;											\
			}																		\
			Current++;															\
		}																			\
																					\
		MWSendClient(Client, Begin, (End - Begin));					\
	}

#define	WriteWithBreaks(String, Len, File)							\
	{																				\
		unsigned char	*Begin = String;									\
		unsigned char	*Current;											\
		unsigned char	*End = Begin + Len;								\
																					\
		Current = (Begin);													\
		while (Current < (End)) {											\
			if (Current[0]=='\\') {											\
				fwrite((Begin), (Current - (Begin)), 1, (File));	\
				if (toupper(Current[1]) =='N') {							\
					fwrite("\r\n", 2, 1, (File));							\
					Current++;													\
				}																	\
				Begin = Current+1;											\
			}																		\
			Current++;															\
		}																			\
																					\
		fwrite((Begin), (End - (Begin)), 1, (File));					\
	}

/*
	These are the functions this module implements
*/
BOOL
MWCALInitSession(SessionStruct *Session, void **ModuleData)
{
	CalendarSessionStruct	*CalendarSession;

	MsgGetDMY(time(NULL)+Session->TimezoneOffset, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear, 0, 0 ,0);
	Session->CurrentRataDie = MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);

	CalendarSession = MemMalloc(sizeof(CalendarSessionStruct));
	if (!CalendarSession) {
		XplConsolePrintf("NWCAL Module out of memory!\n");
		return(FALSE);
	}

	memset(CalendarSession, 0, sizeof(CalendarSessionStruct));
	/* FIXME - Read the preferences... */

	*ModuleData = (void *)CalendarSession;
	if (MsgGetUserFeature(Session->UserDN, FEATURE_CALENDAR, NULL, NULL)) {
		CalendarSession->CalendarEnabled=TRUE;
	} else {
		CalendarSession->CalendarEnabled=FALSE;
	}

	CalendarSession->CopyValues = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);

	ModDebug("MWCALInitSession() called\n");

	return(TRUE);
}

BOOL
MWCALDestroySession(SessionStruct *Session, void *ModuleData)
{
	unsigned char				Path[XPL_MAX_PATH+1];
	CalendarSessionStruct	*CalendarSession=(CalendarSessionStruct *)ModuleData;

	ModDebug("MWCALDestroySession() called\n");

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_RRULE, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	if (CalendarSession) {
		FreeCalendarEntryDetails(Session, CalendarSession);
		FreeCalendarView(Session, CalendarSession);
		if (CalendarSession->IDList) {
			MemFree(CalendarSession->IDList);
			CalendarSession->IDList = NULL;
		}

		if (CalendarSession->CopyValues) {
			MDBDestroyValueStruct(CalendarSession->CopyValues);
		}

		MemFree(CalendarSession);
	}
	return(TRUE);
}

static BOOL
GetReadableDate(long Offset, long UTCTime, unsigned char *DateBuffer)
{
	struct tm		TimeStruct;
	unsigned char	TempBuffer[80];

	if (UTCTime == 0) {
		UTCTime = time(NULL);
	}

	UTCTime += Offset;

	gmtime_r(&UTCTime, &TimeStruct);
	strftime(TempBuffer, 80, "%A, %d of %B at %l:%M %p %%+04.4ld", &TimeStruct);

	snprintf(DateBuffer, BUFSIZE, TempBuffer, (long)(Offset / 36));

	return(TRUE);
}

static int
GetFreeBusyBusyBit(unsigned long *BitField, unsigned long BitNumber)
{
	unsigned long			Remainder = BitNumber % (8 * sizeof(unsigned long));
	unsigned long			Divedend = (BitNumber - Remainder) / (8 * sizeof(unsigned long));

	if (BitField[Divedend] & (1 << Remainder)) {
		return(TRUE);
	} else {
		return(FALSE);
	}
}

static int
SetFreeBusyBusyBit(unsigned long *BitField, unsigned long BitNumber, BOOL Value)
{
	unsigned long			Remainder = BitNumber % (8 * sizeof(unsigned long));
	unsigned long			Divedend = (BitNumber - Remainder) / (8 * sizeof(unsigned long));

	if (Value) {
		BitField[Divedend] |= (1 << Remainder);
	} else {
		BitField[Divedend] &= ~(1 << Remainder);
	}
	return(TRUE);
}

static BOOL
CalComposeCleanUp(SessionStruct *Session, void *ModuleData)
{
	CalendarSessionStruct	*CalendarSession=(CalendarSessionStruct *)ModuleData;
	unsigned char	Path[XPL_MAX_PATH+1];

	snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_BODY, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_SUBJECT, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_LOCATION, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);
								
	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_MESSAGE, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_RRULE, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_ICAL, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_TEMP, MwCal.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	/* Reset begin and end times */
	CalendarSession->BeginTime = 0;
	CalendarSession->EndTime = 0;

#if 0
	/* Clean up attachments */
	AttachNum = 0;
	do {
		snprintf(Path, sizeof(Path), "%s/%x.%d", MwCal.WorkDir, Session->SessionID, AttachNum);
		AttachNum++;
	} while (unlink(Path) != -1);
#endif

	return(TRUE);
}

// BOOL MsgGetEmailAddressAndFullName(char *inName, char **emailAddress, char **FullName, SessionStruct *Session, int flags)
// 
// objtain emailAddress and or fullname based on flags which are or'd together
// 
// inName is username or address entered during compose. 

#define MSG_GET_EMAIL    1
#define MSG_GET_FULLNAME 2

static BOOL 
MsgGetEmailAddressAndFullName(char *inName, char **emailAddress, char **FullName, SessionStruct *Session, int flags)
{
	char *ptr = NULL;
	char Answer[BUFSIZE];
	MDBValueStruct *User;

	/* Sanity check inputs */
	if (((flags & MSG_GET_EMAIL) && !emailAddress) || ((flags & MSG_GET_FULLNAME) && !FullName))
	{
		return(FALSE);
	}
		

	User = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);

	ptr = strchr(inName, '@');

	if (ptr){

		if (flags & MSG_GET_EMAIL){
			*emailAddress = MemStrdup(inName);
		}
		
		if (flags & MSG_GET_FULLNAME){

			unsigned char		UserDN[MDB_MAX_OBJECT_CHARS+1];

			/* stamp out domain portion temporarily */
			*ptr = '\0';


			/*
			  Hula doesn't require a full eEmail address for sending a message, but iCal does. So
			  if the current address doesn't have an '@' in it we need to get the full address.
			*/

			if (MsgFindObject(inName, UserDN, NULL, NULL, User)) {

				/* restore domain portion of original address (inName)*/
				*ptr = '@';

				MDBFreeValues(User);

				*FullName = MsgGetUserDisplayName(UserDN, User);

				if (!*FullName) {

				/* we couldn't find DisplayName on object in DS */
				/* use his email address in inName */					
					*FullName = MemStrdup(inName);
				}

			}
			else {
				/* restore domain portion of original address (inName)*/
				*ptr = '@';

				/* we couldn't find this user in DS */
				/* use his email address in inName */
				*FullName = MemStrdup(inName);				
			}

			
		}
			

	}
	else {
		/* inName does not have '@' it may be a user name we know about */

		if ((flags & MSG_GET_EMAIL) || (flags & MSG_GET_FULLNAME))	{

			unsigned char		UserDN[MDB_MAX_OBJECT_CHARS+1];

			/*
			  Hula doesn't require a full eEmail address for sending a message, but iCal does. So
			  if the current address doesn't have an '@' in it we need to get the full address.
			*/

			if (MsgFindObject(inName, UserDN, NULL, NULL, User)) {

				MDBFreeValues(User);


				if (flags & MSG_GET_EMAIL)	{

					*emailAddress = MsgGetUserEmailAddress(UserDN, User, NULL, 0);

					if (!*emailAddress){
						/* Error */
						MDBDestroyValueStruct(User);
						return(FALSE);
					}
				}

				if (flags & MSG_GET_FULLNAME) {
					
					*FullName = MsgGetUserDisplayName(UserDN, User);

					MDBFreeValues(User);

					if (!*FullName)	{

						if (*emailAddress) {
							*FullName = MemStrdup(*emailAddress);
						}
						else {
							*FullName = MsgGetUserEmailAddress(UserDN, User, NULL, 0);

							if (!*FullName) {
								MDBDestroyValueStruct(User);
								return(FALSE);
							}
						}

					}

				}

			}
			else { 
				if (flags & MSG_GET_EMAIL){
					
					/* If mail address doesn't have @ and is not a local email address */
					/* assume that it is an alias and tack on local domain */
					snprintf(Answer, sizeof(Answer), "%s@%s", inName, Session->OfficialDomain);
					*emailAddress = MemStrdup(Answer);
				}

				if (flags & MSG_GET_FULLNAME) {
					/* use same rules for display name */
					snprintf(Answer, sizeof(Answer), "%s@%s", inName, Session->OfficialDomain);
					*FullName = MemStrdup(Answer);
				}

			}


		}

		
	}


	MDBDestroyValueStruct(User);
	return(TRUE);

}

static BOOL 
CreateICalFileForAppointment(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, 
			     unsigned long Action, char *fileName, char *Buffer, char *FullName, 
			     unsigned long *iCalStart ,unsigned long *iCalEnd, BOOL *FreeBusyNeeded, unsigned int *UIDTime,
			     int addrType, char *attendee)
{
	unsigned long			Day;
	unsigned long			Month;
	unsigned long			Year;
	unsigned long			Hour;
	unsigned long			Minute;
	unsigned long			i;
	FILE						*FileHandle;
	unsigned char			Boundary[BUFSIZE + 1];
	MDBValueStruct			*User;							/* Used to read user information such as the full name */
	long						retVal;
	FILE *ICalFile = fopen(fileName, "wb");
	StreamStruct			FileInCodec;
	StreamStruct			MemoryInCodec;
	unsigned long			ReplyInt;
	char NameWithoutQuotes[MAXEMAILNAMESIZE + 1];
	char UID[BUFSIZE];
	
	MWCalStreams;

	if (!ICalFile) {
		return(FALSE);
	}

	/* UID */
	*UIDTime = time(NULL);

	snprintf(UID, BUFSIZE, "%d%s%d", (int)*UIDTime, Session->EMailAddress, (int)Session->SessionID);

	/* Prepare all required streams */
	memset(&FileInCodec, 0, sizeof(StreamStruct));
	memset(&MemoryInCodec, 0, sizeof(StreamStruct));
	RFC1522Codec = MWGetStream(&MemoryInCodec, "RFC1522", TRUE);
	RFC2445Codec = MWGetStream(NULL, "RFC2445", TRUE);
	RFC2445MLCodec = MWGetStream(NULL, "RFC2445ML", TRUE);
	MessageOutCodec = MWGetStream(RFC1522Codec, NULL, TRUE);

	BodyCharsetCodec = MWGetStream(NULL, Session->Charset, TRUE);
	if (BodyCharsetCodec) {
		BodyQPCodec = MWGetStream(BodyCharsetCodec, "quoted-printable", TRUE);
	} else {
		BodyQPCodec = MWGetStream(NULL, "quoted-printable", TRUE);
	}
	BodyQPCodec->Next = MessageOutCodec;

	/* Set up the input & output */
	MemoryInCodec.StreamData = Buffer;
	MemoryInCodec.Codec = MWStreamFromMemory;
	MemoryInCodec.Next = RFC1522Codec;
	FileInCodec.Codec = MWStreamFromFile;
	FileInCodec.Next = RFC1522Codec;
	RFC2445MLCodec->Next = RFC2445Codec;
	RFC2445Codec->Next = MessageOutCodec;
	RFC1522Codec->Charset = Session->Charset;
	MessageOutCodec->StreamData = ICalFile;
	MessageOutCodec->Codec = MWStreamToFile;

	/* Start building the message */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	i=sizeof(Client->cs);
	getpeername(Client->s, (struct sockaddr *)&(Client->cs), (socklen_t *)&i);
	fprintf(ICalFile, "Received: from %s [%d.%d.%d.%d] by %s\r\n\twith %s; %s\r\n", 
						Session->User,
						Client->cs.sin_addr.s_net,
						Client->cs.sin_addr.s_host,
						Client->cs.sin_addr.s_lh,
						Client->cs.sin_addr.s_impno,
						Session->OfficialDomain, 
						PRODUCT_NAME,
						Buffer);

	/* Subject */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_SUBJECT);
	FileHandle=fopen(Buffer,"rb");
	if (FileHandle) {
		fprintf(ICalFile, "Subject: ");
		ProcessFileStream(FileHandle);
		fclose(FileHandle);	
	} 
	User = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);
	/* Reply-To:*/
	if (MDBRead(Session->UserDN, MSGSRV_A_EMAIL_ADDRESS, User) > 0) {
		fprintf(ICalFile, "Reply-To: %s\r\n", User->Value[0]);
	} 														
	MDBFreeValues(User);  	

	/* From */
	fprintf(ICalFile, "From: ");
	retVal = MWGetQuotedString(Session->DisplayName, FullName, BUFSIZE + 1);
	if (retVal == QP_STRING) {
		fputs(FullName, ICalFile);
		fprintf(ICalFile, " <%s>\r\n", Session->EMailAddress);
	} else if (retVal == HAS_EXTENDED_CHARS) {
		HulaStrNCpy(Buffer, Session->DisplayName, BUFSIZE);
		ProcessMemoryStream(strlen(Buffer));
		fprintf(ICalFile, " <%s>\r\n", Session->EMailAddress);
	} else {
		fprintf(ICalFile, "<%s>\r\n", Session->EMailAddress);
	}

	switch(addrType) {
	case COMPOSE_TO:
	    fprintf(ICalFile, "To: %s\r\n", attendee);
	    break;
	case COMPOSE_CC:
	    fprintf(ICalFile, "CC: %s\r\n", attendee);
	    break;
	case COMPOSE_BCC:
	    fprintf(ICalFile, "BCC: %s\r\n", attendee);
	    break;
	}

	/* Date */	
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	fprintf(ICalFile, "Date: %s\r\n", Buffer);

	/* Prorietary information */	
	fprintf(ICalFile, "X-Mailer: %s\r\n", PRODUCT_NAME);
	fprintf(ICalFile, "X-Sender: %s\r\n", Session->EMailAddress);

	/* MIME */
	snprintf(Boundary, sizeof(Boundary), "------=_ModWebBOUNDARY_%x_%lu", (unsigned int)Session->User, time(NULL));
	fprintf(ICalFile, "Content-Type: multipart/alternative;\r\n\tboundary=\"%s\"\r\n\r\n", Boundary);

	fprintf(ICalFile, "This is a multi-part message in MIME format.\r\n\r\n--%s\r\nContent-Type: text/plain; charset=\"%s\"\r\nContent-Transfer-Encoding: quoted-printable\r\n\r\n", Boundary, Session->Charset);

	/* Stream from file to file */
	CRLFCodec = MWGetStream(NULL, "CRLF", TRUE);
	if (CRLFCodec) {
		FileInCodec.Next = CRLFCodec;
		if (BodyCharsetCodec) {
			CRLFCodec->Next = BodyCharsetCodec;
		} else {
			CRLFCodec->Next = BodyQPCodec;
		}
	} else {
		if (BodyCharsetCodec) {
			FileInCodec.Next = BodyCharsetCodec;
		} else {
			FileInCodec.Next = BodyQPCodec;
		}
	}

	/* Send a text version of the iCal object */
	switch (Action) {
		case ACTION_APPOINTMENT: {
			fprintf(ICalFile, "Item Type: Appointment\r\n");
			break;
		}

		case ACTION_TASK:	{
			fprintf(ICalFile, "Item Type: Task\r\n");
			break;
		}

		case ACTION_NOTE:	{
			fprintf(ICalFile, "Item Type: Note\r\n");
			break;
		}
	}

	/* Don't add the attendee's because they can already see the list of people who received the message */

	GetReadableDate(Session->TimezoneOffset, CalendarSession->BeginTime, Buffer);
	fprintf(ICalFile, "Begin time: %s\r\n", Buffer);

	if (CalendarSession->EndTime > CalendarSession->BeginTime) {
		unsigned long		duration = CalendarSession->EndTime - CalendarSession->BeginTime;
		unsigned long		hours = duration / (60 * 60);
		unsigned long		minutes = (duration % (60 * 60)) / 60;

		if (hours < 24) {

			/* If the duration is less than a day show duration.  If not show end time */
			if (minutes == 0) {
				/* Just hours */

				fprintf(ICalFile, "Duration:   %lu Hour%s\r\n", hours, (hours > 1) ? "s" : "");
			} else if (hours != 0) {
				/* Hours and minutes */

				fprintf(ICalFile, "Duration:   %lu Hour%s and %lu Minutes\r\n", hours, (hours > 1) ? "s" : "", minutes);
			} else {
				/* Just minutes */

				fprintf(ICalFile, "Duration:   %lu Minutes\r\n", minutes);
			}
		} else {
			GetReadableDate(Session->TimezoneOffset, CalendarSession->EndTime, Buffer);
			fprintf(ICalFile, "End time:   %s\r\n", Buffer);
		}
	}

	/* LOCATION */
	if (Action != ACTION_NOTE) {
		struct stat				sb;

		snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_LOCATION);
		if ((stat(Buffer, &sb) != -1) && (sb.st_size > 0)) {
			FileHandle = fopen(Buffer,"rb");

			if (FileHandle) {
				fprintf(ICalFile, "Location:   ");

				ProcessFileStream(FileHandle);

				fclose(FileHandle);	
				fprintf(ICalFile, "\r\n");
			}
		}
	}

	/* DESCRIPTION */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);
	FileHandle = fopen(Buffer,"rb");

	if (FileHandle) {
		ProcessFileStream(FileHandle);

		fclose(FileHandle);	
		fprintf(ICalFile, "\r\n");
	} 

	/* We don't need to send the summary, because it "should" match the subject of the email */

	fprintf(ICalFile, "\r\n--%s\r\nContent-Type: text/html; charset=utf-8\r\n\r\n", Boundary);

	fprintf(ICalFile,
		"<html>"
		"<body>"
		"<table width=\"100\%\" border=\"0\">"
		"<tr>"
		" <td width=\"100\%\" height=\"5\" bgcolor=\"#8ba2bf\" border=0>"
		" </td>"
		"</tr>"
		"<tr>"
		" <td width=\"100\%\" height=\"15\" bgcolor=\"#d2dbe7\" border=0>"
		"   <h2>Invitation to ");

	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_SUBJECT);
	FileHandle=fopen(Buffer,"rb");

	if (FileHandle) {
	    ProcessFileStream(FileHandle);
	    fclose(FileHandle);	
	}

	if (FullName[0] == '"') {
	    int last;
	    
	    strcpy(NameWithoutQuotes, FullName + 1);
	    last = strlen(NameWithoutQuotes) - 1;
	    
	    if (NameWithoutQuotes[last] == '"') {
		NameWithoutQuotes[last] = '\0';
	    }
	    
	} else {
	    strcpy(NameWithoutQuotes, FullName);
	}

	fprintf(ICalFile, 
		"</h2>"
		" </td>"
		"</tr>"
		"<tr>"
		"  <td>"
		"    <p>"
		"    %s has invited you to attend this event:"
		"    <p>"
		"    <table>"
		"      <tr>", 
		NameWithoutQuotes);
	/* Don't add the attendee's because they can already see the list of people who received the message */

	GetReadableDate(Session->TimezoneOffset, CalendarSession->BeginTime, Buffer);
	fprintf(ICalFile,
		"      <tr>"
		"        <td><b>Begins:</b></td>"
		"        <td>%s", Buffer);

	if (CalendarSession->EndTime > CalendarSession->BeginTime) {
		unsigned long		duration = CalendarSession->EndTime - CalendarSession->BeginTime;
		unsigned long		hours = duration / (60 * 60);
		unsigned long		minutes = (duration % (60 * 60)) / 60;

		if (hours < 24) {
		    fprintf(ICalFile,
			    "</td>"
			    "      </tr>"
			    "      <tr>"
			    "        <td><b>Duration:</b></td>"
			    "        <td>");
			/* If the duration is less than a day show duration.  If not show end time */
			if (minutes == 0) {
				/* Just hours */

				fprintf(ICalFile, "%lu Hour%s\r\n", hours, (hours > 1) ? "s" : "");
			} else if (hours != 0) {
				/* Hours and minutes */

				fprintf(ICalFile, "%lu Hour%s and %lu Minutes\r\n", hours, (hours > 1) ? "s" : "", minutes);
			} else {
				/* Just minutes */

				fprintf(ICalFile, "%lu Minutes\r\n", minutes);
			}
		} else {
		    GetReadableDate(Session->TimezoneOffset, CalendarSession->EndTime, Buffer);
		    fprintf(ICalFile,
			    "</td>"
			    "      </tr>"
			    "      <tr>"
			    "        <td><b>Ends:</b></td>"
			    "        <td>%s", Buffer);
		}
	}

	fprintf(ICalFile,
		"        </td>"
		"      </tr>"
		"    </table><p>");
	/* DESCRIPTION */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);
	FileHandle = fopen(Buffer,"rb");

	if (FileHandle) {
		ProcessFileStream(FileHandle);

		fclose(FileHandle);	
		fprintf(ICalFile, "\r\n");
	}	

	fprintf(ICalFile, 
		"  </td>"
		"</tr>"
		"<tr>"
		"  <td height=\"15\" bgcolor=\"#eeeeee\" border=0 align=\"left\"><b>RSVP:</b>");
	
	fprintf(ICalFile, 
		"<a href=\"http://%s/status/%s/MAIN/%s/%d/accept?email=%s\">Accept</a>", Client->URLHost, Session->User, UID, 0, attendee);
	fprintf(ICalFile, 
		"| <a href=\"http://%s/status/%s/MAIN/%s/%d/decline?email=%s\">Decline</a>", Client->URLHost, Session->User, UID, 0, attendee);

	fprintf(ICalFile,
		"  </td>"
		"</tr>"
		"</table>"
		"</body>"
		"</html>");

	fprintf(ICalFile, "\r\n--%s\r\nContent-Type: text/calendar; method=REQUEST; charset=utf-8\r\n\r\n", Boundary);

	FileInCodec.Next = RFC2445Codec;

	/* Now for the iCal object */
	*iCalStart = ftell(ICalFile);
	fprintf(ICalFile, "BEGIN:VCALENDAR\r\nPRODID:-//Novell Inc//NetMail ModWeb//\r\nVERSION:2.0\r\nMETHOD:REQUEST\r\n");

	/* Throw in TimeZone Stuff if there is a reoccuring rule */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_RRULE);
	FileHandle = fopen(Buffer, "rb");
	if (FileHandle) {
		fclose(FileHandle);
		if (Session->TimezoneID > 74) {
			Session->TimezoneID = TZ_UTC;
		}
		fprintf(ICalFile, ICalTZRules[Session->TimezoneID].FullTZ);
	}

	switch (Action) {
		case ACTION_APPOINTMENT: {
			fprintf(ICalFile, "BEGIN:VEVENT\r\n");
			break;
		}

		case ACTION_TASK:	{
			fprintf(ICalFile, "BEGIN:VTODO\r\n");
			break;
		}

		case ACTION_NOTE:	{
			fprintf(ICalFile, "BEGIN:VJOURNAL\r\n");
			break;
		}
	}

	/* List Attendees */
	/* This also means that we do not automatically accept the appointment. If we are a recipient */
	/* We become the CHAIR. This may change if/when we get an interface to support setting a CHAIR. */

	for (i = 0; i < 3; i++) {
		snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
		FileHandle = fopen(Buffer, "rb");	

		if (!FileHandle) {
			continue;
		}

		while (!feof(FileHandle) && !ferror(FileHandle)) {
			if (fgets(Buffer, MAXEMAILNAMESIZE, FileHandle)) {
				unsigned char *attendeeEmail = NULL;
				unsigned char *attendeeFullname = NULL;

				ChopNL(Buffer);
				if (Buffer[0] == '\0') {
					continue;
				}

				/* get the email address and full name of attendee */
				if (MsgGetEmailAddressAndFullName(Buffer, (char **)&attendeeEmail, (char **)&attendeeFullname, Session, MSG_GET_EMAIL|MSG_GET_FULLNAME)){

					switch (i) {
						case COMPOSE_TO: {	/* They are required */
							/* If it matches the sender/originator then for now we set him up as the chair. */
							if (!MWQuickCmp(attendeeEmail, Session->EMailAddress)) {
								fprintf(ICalFile, "ATTENDEE;CN=%s;ROLE=REQ-PARTICIPANT;RSVP=TRUE:MAILTO:%s\r\n", attendeeFullname, attendeeEmail);
							}
							else {
								/* Organizer is attendee */
								*FreeBusyNeeded = TRUE;
								fprintf(ICalFile, "ATTENDEE;CN=%s;ROLE=CHAIR;PARTSTAT=ACCEPTED:MAILTO:%s\r\n", attendeeFullname, attendeeEmail);
							}
							break;
						}

						case COMPOSE_CC: {	/* They are not required */
							if (!MWQuickCmp(attendeeEmail, Session->EMailAddress)) {
								;
							}
							else {
								/* Organizer is attendee */
								*FreeBusyNeeded = TRUE;
							}

							fprintf(ICalFile, "ATTENDEE;CN=%s;ROLE=OPT-PARTICIPANT;RSVP=TRUE:MAILTO:%s\r\n", attendeeFullname, attendeeEmail);
							break;
						}

						case COMPOSE_BCC: {	/* They aren't supposed to show up - just need to know about it */
							fprintf(ICalFile, "ATTENDEE;CN=%s;ROLE=NON-PARTICIPANT;RSVP=TRUE:MAILTO:%s\r\n", attendeeFullname, attendeeEmail);

							break;
						}
					}

				}

				if (attendeeEmail) {
					MemFree(attendeeEmail);
				}

				if (attendeeFullname){
					MemFree(attendeeFullname);
				}

			} 
		}
		fclose(FileHandle);	
	}

	MDBDestroyValueStruct(User);
	fprintf(ICalFile, "ORGANIZER;CN=%s:MAILTO:%s\r\n", FullName, Session->EMailAddress);

	/* DTSTART */
	/* 
		If we have a .rul file we need to calculate the DTSTART based on
		that rule.  The DTSTART needs to be the first actual occurance of
		the rule.  If we parse the RRULE with the DTSTART we got from the
		form and take the second rule instance we will have the correct
		DTSTART.  This does mean that we have to provide an 'valid' iCal
		object to the parser.  It doesn't understand just the RRULE line.

		This entire section is done in local time if we have a rule.

		If we don't have a .rul file we simply send the DTSTART in UTC
	*/

	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_RRULE);
	FileHandle = fopen(Buffer, "rb");
	if (FileHandle) {
		ICalObject		*ICal=NULL;
		ICalVRuleIterator	Iterator;
		unsigned char	*RRule;
		unsigned long	EndYear;
		unsigned long	EndMonth;
		unsigned long	EndDay;
		unsigned long	EndHour;
		unsigned long	EndMinute;

		/* Send DTStart based on the RRULE and send the RRULE */
		MsgGetDMY(CalendarSession->BeginTime, &Day, &Month, &Year, &Hour, &Minute, NULL);
		MsgGetDMY(CalendarSession->EndTime, &EndDay, &EndMonth, &EndYear, &EndHour, &EndMinute, NULL);
		ReplyInt = snprintf(Buffer, BUFSIZE, "%sBEGIN:VEVENT\r\nDTSTART;TZID=\"%s\":%04d%02d%02dT%02d%02d00\r\nDTEND;TZID=\"%s\":%04d%02d%02dT%02d%02d00\r\n",
			ICalTZRules[Session->TimezoneID].FullTZ, ICalTZRules[Session->TimezoneID].TZName, (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute,
			ICalTZRules[Session->TimezoneID].TZName, (int)EndYear, (int)EndMonth, (int)EndDay, (int)EndHour, (int)EndMinute);

		if (fgets((Buffer + ReplyInt), (BUFSIZE - ReplyInt), FileHandle)) {		/* Read the rrule into the buffer */
			RRule = MemStrdup(Buffer + ReplyInt);	/* ICalPraseObject is destructive */
  			ICal=ICalParseObject(NULL, Buffer, strlen(Buffer));
			if (ICal && ICalHasRule(ICal)) {
				if (ICalFirstRuleInstance(ICal, &Iterator)) {
					fprintf(ICalFile, "DTSTART;TZID=\"%s\":%04d%02d%02dT%02d%02d00\r\n", ICalTZRules[Session->TimezoneID].TZName, 
							  (int)Iterator.Year, (int)Iterator.Month, (int)Iterator.Day, (int)Iterator.Hour, (int)Iterator.Minute);
					if (RRule) {
						fprintf(ICalFile, RRule);		/* The RRULE */
					}
					/* DTEND */
					CalendarSession->BeginTime = MsgGetUTC(Iterator.Day, Iterator.Month, Iterator.Year, Iterator.Hour, Iterator.Minute, Iterator.Second);
					CalendarSession->EndTime = CalendarSession->BeginTime + Iterator.Duration;
					if (CalendarSession->BeginTime < CalendarSession->EndTime && Action != ACTION_NOTE) {
						MsgGetDMY(CalendarSession->EndTime, &Day, &Month, &Year, &Hour, &Minute, NULL);
						if (Action == ACTION_APPOINTMENT) {
							fprintf(ICalFile, "DTEND;TZID=\"%s\":%04d%02d%02dT%02d%02d00\r\n", ICalTZRules[Session->TimezoneID].TZName, 
									  (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);
						} else if (Action == ACTION_TASK) {
							fprintf(ICalFile, "DUE;TZID=\"%s\":%04d%02d%02dT%02d%02d00\r\n", ICalTZRules[Session->TimezoneID].TZName, 
									  (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);
						}
					}
				}
			}
			if (ICal) {
				ICalFreeObjects(ICal);
			}
			if (RRule) {
				MemFree(RRule);
			}
		}
		fclose(FileHandle);
	} else {
		/* Just send DTStart */
		MsgGetDMY(CalendarSession->BeginTime, &Day, &Month, &Year, &Hour, &Minute, NULL);
		MsgGetDMY(CalendarSession->BeginTime-MsgGetUTCOffsetByDate(Session->TimezoneID, Day, Month, Year, Hour),
			&Day, &Month, &Year, &Hour, &Minute, NULL);
		fprintf(ICalFile, "DTSTART:%04d%02d%02dT%02d%02d00Z\r\n", (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);
		/* DTEND */
		if (CalendarSession->BeginTime < CalendarSession->EndTime && Action != ACTION_NOTE) {
			MsgGetDMY(CalendarSession->EndTime, &Day, &Month, &Year, &Hour, &Minute, NULL);
			MsgGetDMY(CalendarSession->EndTime-MsgGetUTCOffsetByDate(Session->TimezoneID, Day, Month, Year, Hour),
				&Day, &Month, &Year, &Hour, &Minute, NULL);
			fprintf(ICalFile, "DTEND:%04d%02d%02dT%02d%02d00Z\r\n", (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);
		}
	}

	/* LOCATION */
	if (Action != ACTION_NOTE) {
		snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_LOCATION);
		FileHandle=fopen(Buffer,"rb");

		if (FileHandle) {
			fprintf(ICalFile, "LOCATION:");
			/* Switch codec to "MultiLine RFC2445" codec; preset state to contain the chars we already sent for this line */
			RFC2445Codec->State=9;
			FileInCodec.Next=RFC2445MLCodec;
			ProcessFileStream(FileHandle);

			/* Reset to use regular RFC2445 codec */
			FileInCodec.Next=RFC2445Codec;

			fclose(FileHandle);	
			fprintf(ICalFile, "\r\n");
		}
	}

	/* TRANSP */
	fprintf(ICalFile, "TRANSP:OPAQUE\r\n");

	/* SEQUENCE */
	fprintf(ICalFile, "SEQUENCE:0\r\n");

	fprintf(ICalFile, "UID:%s\r\n", UID);
	
	/* DTSTAMP */
	MsgGetDMY(time(NULL), &Day, &Month, &Year, &Hour, &Minute, NULL);
	fprintf(ICalFile, "DTSTAMP:%04d%02d%02dT%02d%02d00Z\r\n", (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);

	/* DESCRIPTION */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);
	FileHandle=fopen(Buffer,"rb");

	if (FileHandle) {
		fprintf(ICalFile, "DESCRIPTION:");

		/* Switch codec to "MultiLine RFC2445" codec; preset state to contain the chars we already sent for this line */
		RFC2445Codec->State=12;
		FileInCodec.Next=RFC2445MLCodec;
		ProcessFileStream(FileHandle);

		/* Reset to use regular RFC2445 codec */
		FileInCodec.Next=RFC2445Codec;
		fclose(FileHandle);	
	} 

	fprintf(ICalFile, "\r\n");

	/* SUMMARY */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_SUBJECT);
	FileHandle=fopen(Buffer,"rb");

	if (FileHandle) {
		fprintf(ICalFile, "SUMMARY:");
		RFC2445Codec->State=8;
		ProcessFileStream(FileHandle);
		fclose(FileHandle);	
	} 

	
	/* PRIORITY */
	/* CLASS */

	/* Close it off */

	switch (Action) {
		case ACTION_APPOINTMENT: {
			fprintf(ICalFile, "END:VEVENT\r\nEND:VCALENDAR\r\n");
			break;
		}

		case ACTION_TASK:	{
			fprintf(ICalFile, "END:VTODO\r\nEND:VCALENDAR\r\n");
			break;
		}

		case ACTION_NOTE:	{
			fprintf(ICalFile, "END:VJOURNAL\r\nEND:VCALENDAR\r\n");
			break;
		}
	}

	*iCalEnd = ftell(ICalFile);
	fprintf(ICalFile, "\r\n--%s--\r\n\r\n", Boundary);
	fclose(ICalFile);
	ReleaseStreams;

	if (*iCalEnd < *iCalStart) {
		return(FALSE);
	}

	return(TRUE);

}

static BOOL 
SendAppointmentToNMAP(ConnectionStruct *Client, SessionStruct *Session, char *Buffer, char *Answer, char *FullName, char *toAddr)
{
	int ReplyInt;
	int i = 0;
	FILE *ICalTempFile;
	struct stat FileInfo;
	FILE *ICalFile = NULL;

	/* We can handle this with one message */
	/* Create Queue Entry */
	MWSendNMAPServer(Session, "QCREA\r\n", 7);
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
	    return(FALSE);
	}
	ReplyInt = snprintf(Answer, BUFSIZE, "QSTOR FROM %s %s\r\n", Session->EMailAddress, Session->User);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
	    return(FALSE);
	}

	ReplyInt = snprintf(Answer, BUFSIZE, "QSTOR TO %s %s %d\r\n", toAddr, toAddr, DSN_DEFAULT);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
	    return(FALSE);
	}
	
	/* Send the sucker to NMAP */
	snprintf(Buffer, BUFSIZE, "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	if ((stat(Buffer, &FileInfo)==-1) || (ICalFile=fopen(Buffer, "rb"))==NULL) {
	    fclose(ICalFile);
	    return(FALSE);
	}	
	
	/* Tell NMAP the Message is coming */
	ReplyInt=snprintf(Answer, BUFSIZE, "QSTOR MESSAGE %d\r\n", (int)FileInfo.st_size);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	while (!feof(ICalFile) && !ferror(ICalFile)) {
	    if ((ReplyInt=fread(Answer, 1, BUFSIZE, ICalFile))>0) {
		MWSendNMAPServer(Session, Answer, ReplyInt);
	    } 
	}
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)!=1000) {
	    fclose(ICalFile);
	    return(FALSE);
	}
	
	/* Kick it */
	MWSendNMAPServer(Session, "QRUN\r\n", 6);
	
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
	    return(FALSE);
	}
	return(TRUE);
}

static BOOL
SendAppointment(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long Action)
{
    FILE						*ICalFile;
    FILE                 *FileHandle;
    int i;
    FILE						*TempFileHandle;
    unsigned long			RecipCount = 0;				/* Number of recipients we are sending to */
    unsigned long			ReplyInt;
    int						ClientNMAPs			= 0;
    unsigned long			NMAPCAddress		= 0;
    unsigned long        iCalStart,iCalEnd;
    unsigned int         flags = 0;
    struct stat				FileInfo;
    unsigned char			FullName[BUFSIZE + 1];		/* Stores the user's fullname */
    unsigned char			Buffer[BUFSIZE + 1];
    unsigned char			Answer[BUFSIZE + 1];
    BOOL FreeBusyNeeded = FALSE; /* easier to use negative logic*/
    unsigned int UIDTime; /* needed to recreate UID */
    int numBCC = 0;

    if (Session->ReadOnly) {
	return(FALSE);
    }

    /* rewrite addresses to make sure they are full email addresses */
    for (i=0; i<3; i++) {
	BOOL	First;

	First=TRUE;
	snprintf(Answer, sizeof(Answer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
	FileHandle=fopen(Answer, "rb");	
	if (!FileHandle) {
	    continue;
	}

	/* we are abusing FullName to prevent getting extra buffer */
	snprintf(FullName, sizeof(FullName), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TEMP);
	TempFileHandle=fopen(FullName,"wb");
		
	if (!TempFileHandle) {
	    fclose(FileHandle);
	    return(FALSE);
	}

	while (!feof(FileHandle) && !ferror(FileHandle)) {
	    if (fgets(Buffer, MAXEMAILNAMESIZE, FileHandle)) {
		unsigned char *attendeeEmail = NULL;

		ChopNL(Buffer);
		if (Buffer[0]=='\0') {
		    continue;
		}

		/* get the email address and full name of attendee */
		if (MsgGetEmailAddressAndFullName(Buffer, (char **)&attendeeEmail, NULL, Session, MSG_GET_EMAIL) == TRUE) {
		    fprintf(TempFileHandle, "%s\r\n", attendeeEmail);
		    RecipCount +=1;
		    if (i == COMPOSE_BCC) {
			numBCC++;
		    }
		}					

		if (attendeeEmail) {
		    MemFree(attendeeEmail);
		}
 
	    }
	}

	fclose(FileHandle);	
	fclose(TempFileHandle);
	unlink(Answer);
	rename(FullName,Answer);
	  
    }

    /* must have at least one recipient for apt/task/note */
    if (RecipCount > numBCC) {
	;
    } else { 
	CalendarSession->Error = ERROR_NO_RECIPIENTS;
	return(FALSE);
    }

    /* Return an error if the dates are smegged up */
    if (CalendarSession->BeginTime > CalendarSession->EndTime && Action != ACTION_NOTE) {
	CalendarSession->Error = END_BEFORE_BEGIN;
	return(FALSE);
    }

    /* We know that we have at least one recipient for this appointment */
    /* Connect to NMAP server */
    if ((MwCal.NmapQAddress != 0) && (MwCal.NmapQAddress != Session->NMAPAddress.s_addr)) {
	unsigned char	*User;
		
	ClientNMAPs = Session->NMAPs;
	Session->NMAPs = -1;
	Session->NBufferPtr = 0;
	NMAPCAddress = Session->NMAPAddress.s_addr;
	Session->NMAPAddress.s_addr = MwCal.NmapQAddress;
	User = Session->User;
	Session->User = NULL;
	if (!MWConnectUserToNMAPServer(Session)) {
	    Session->User = User;
	    RestoreUserNMAP;
	    return(FALSE);
	}
	Session->User = User;
    }

    /* send the message to everyone */
    for (i=0; i<3; i++) {
	BOOL	First;

	First=TRUE;
	snprintf(Answer, sizeof(Answer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
	FileHandle=fopen(Answer, "rb");	
	if (!FileHandle) {
	    continue;
	}

	while (!feof(FileHandle) && !ferror(FileHandle)) {
	    if (fgets(Buffer, MAXEMAILNAMESIZE, FileHandle)) {
		unsigned char *attendeeEmail = NULL;

		ChopNL(Buffer);
		if (Buffer[0]=='\0') {
		    continue;
		}

		/* get the email address and full name of attendee */
		if (MsgGetEmailAddressAndFullName(Buffer, (char **)&attendeeEmail, NULL, Session, MSG_GET_EMAIL) == TRUE) {
		    /* Start creating Header in file */
		    snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
		    if (!CreateICalFileForAppointment(Client, Session, CalendarSession, Action, Buffer, Answer, FullName, &iCalStart, &iCalEnd, &FreeBusyNeeded, &UIDTime, i, attendeeEmail)) {
			RestoreUserNMAP;
			return(FALSE);
		    }

		    if (!strcmp(attendeeEmail, Session->EMailAddress)) {
			/*
			  Now we have the whole message, so we do a CSSTOR to put it in our own
			  calendar, and then switch NMAP connections to throw it in the queue for
			  everyone else, if there is anyone else.
			*/

			snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
			if ((stat(Buffer, &FileInfo) == -1) || (ICalFile = fopen(Buffer, "rb")) == NULL) {
			    RestoreUserNMAP;
			    return(FALSE);
			}

			fseek(ICalFile, iCalStart, SEEK_SET);

			if (!Session->ReadOnly) {
			    int remaining;
			    MWProtectNMAP(Client, TRUE);

			    remaining = (iCalEnd - iCalStart);
			    /* HOLE: should use FreeBusyNeeded on CSSTOR, but not implemented */
			    MWSendNMAPServer(Session, "CSSTOR MAIN\r\n", 13);
			    if (MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE) != 2002) {
				fclose(ICalFile);
				RestoreUserNMAP;
				return(FALSE);
			    }	

			    while (remaining > 0 && !feof(ICalFile) && !ferror(ICalFile)) {
				if ((ReplyInt = fread(Answer, 1, (remaining > BUFSIZE) ? BUFSIZE : remaining, ICalFile)) > 0) {
				    MWSendNMAPServer(Session, Answer, ReplyInt);

				    remaining -= ReplyInt;
				} 
			    }

			    MWSendNMAPServer(Session, "\r\n.\r\n", 5);
			    MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE);

			    if (!FreeBusyNeeded) {
				/* find new calendar entry to add NOFREEBUSY flag to*/
				MWSendNMAPServer(Session, "CSUPDA\r\n", 8);
				do {
				    ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE,TRUE);
				}while(ReplyInt == 6001);

				/* find new calendar entry to add NOFREEBUSY flag to*/
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSFIND 0 %d%s%d\r\n",  (int)UIDTime, Session->EMailAddress, (int)Session->SessionID);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

				if (ReplyInt == 1000) {
				    long unsigned int sequence = atol(Client->Temp);

				    /* now set the NOFREEBUSY flag on the object */
				    ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSAFLG %lu %d\r\n", sequence, MSG_STATE_NOFREEBUSY);
				    MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				    ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
				
				    if (ReplyInt == 1000) {
					/* set the flag */
					CalendarSession->EntryState |= MSG_STATE_NOFREEBUSY;
				    }
				}
			    }
			    MWProtectNMAP(Client, FALSE);
			}

			/* Make sure it gets parsed right away */
			UpdateCalendar(Session, CalendarSession);
			fclose(ICalFile);

		    } else {			
			/* We have borrowed session, ICalFile and attendee start offset within file, send the appointment to recipients. */ 
			if (!SendAppointmentToNMAP(Client, Session, Buffer, Answer, FullName, attendeeEmail)) {
			    RestoreUserNMAP;
			    return(FALSE);
			}
		    }
		    
		}

		if (attendeeEmail) {
		    MemFree(attendeeEmail);
		}
	    }
	}
    }

    RestoreUserNMAP;

    /* Whack our temp files */
    CalComposeCleanUp(Session, CalendarSession);
    return(TRUE);
}

static BOOL
SendTextMessage(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned char			Answer[BUFSIZE + 1];
	unsigned char			Buffer[BUFSIZE + 1];
	int						ReplyInt;
	FILE						*FileHandle;
	FILE						*Message;
	unsigned long			NumRecipients = 0;
	unsigned long			i;
	struct stat				sb;
	MDBValueStruct			*V;
	int						ClientNMAPs			= 0;
	unsigned long			NMAPCAddress		= 0;
	long						retVal;
	StreamStruct			FileInCodec;
	StreamStruct			MemoryInCodec;
	MWCalStreams;

	if (Session->ReadOnly) {
		return(FALSE);
	}

	/*
		If there is no message we don't want to do anything.
	*/
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);
	if ((stat(Buffer, &sb) == -1) || (sb.st_size == 0)) {
		/* No message file, so just clean up and bail */
		CalComposeCleanUp(Session, CalendarSession);

		return(FALSE);
	}	

	V = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);	

	/* Connect to NMAP server */
	if ((MwCal.NmapQAddress != 0) && (MwCal.NmapQAddress != Session->NMAPAddress.s_addr)) {
		unsigned char	*User;

		ClientNMAPs = Session->NMAPs;
		Session->NMAPs = -1;
		Session->NBufferPtr = 0;
		NMAPCAddress = Session->NMAPAddress.s_addr;
		Session->NMAPAddress.s_addr = MwCal.NmapQAddress;
		User = Session->User;
		Session->User = NULL;
		if (!MWConnectUserToNMAPServer(Session)) {
			Session->User = User;
			RestoreUserNMAP;
			return(FALSE);
		}
		Session->User = User;
	}

	/* Create Queue Entry */
	MWSendNMAPServer(Session, "QCREA\r\n", 7);
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
		MDBDestroyValueStruct(V);
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Start creating Header in file */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	Message = fopen(Buffer, "wb");
	if (!Message) {
		MDBDestroyValueStruct(V);
		RestoreUserNMAP;
		return(FALSE);
	}
	
	/* Prepare all required streams */
	memset(&FileInCodec, 0, sizeof(StreamStruct));
	memset(&MemoryInCodec, 0, sizeof(StreamStruct));

	RFC1522Codec = MWGetStream(&MemoryInCodec, "RFC1522", TRUE);
	MessageOutCodec = MWGetStream(RFC1522Codec, NULL, TRUE);

	BodyCharsetCodec = MWGetStream(NULL, Session->Charset, TRUE);
	if (BodyCharsetCodec) {
		BodyQPCodec = MWGetStream(BodyCharsetCodec, "quoted-printable", TRUE);
	} else {
		BodyQPCodec = MWGetStream(NULL, "quoted-printable", TRUE);
	}
	BodyQPCodec->Next = MessageOutCodec;

	/* Set up the input & output */
	MemoryInCodec.StreamData = Buffer;
	MemoryInCodec.Codec = MWStreamFromMemory;
	MemoryInCodec.Next = RFC1522Codec;

	FileInCodec.Codec = MWStreamFromFile;
	FileInCodec.Next = RFC1522Codec;

	RFC1522Codec->Charset = Session->Charset;

	MessageOutCodec->StreamData = Message;
	MessageOutCodec->Codec = MWStreamToFile;	

	/* Start building the message */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);

	i = sizeof(Client->cs);
	getpeername(Client->s, (struct sockaddr *)&(Client->cs), (socklen_t *)&i);
	fprintf(Message, "Received: from %s [%d.%d.%d.%d] by %s\r\n\twith %s; %s\r\n", 
						Session->User,
						Client->cs.sin_addr.s_net,
						Client->cs.sin_addr.s_host,
						Client->cs.sin_addr.s_lh,
						Client->cs.sin_addr.s_impno,
						Session->OfficialDomain, 
						PRODUCT_NAME,
						Buffer);

	/* Subject */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_SUBJECT);
	FileHandle = fopen(Buffer,"rb");

	if (FileHandle) {
		fprintf(Message, "Subject: ");

		fgets(Buffer, sizeof(Buffer), FileHandle);
		ProcessMemoryStream(strlen(Buffer));
		fclose(FileHandle);
		FileHandle = NULL;
	} else {
		fprintf(Message, "Subject: \r\n");
	}

	/* Reply-To:*/
	if (MDBRead(Session->UserDN, MSGSRV_A_EMAIL_ADDRESS, V) > 0) {
		fprintf(Message, "Reply-To: %s\r\n", V->Value[0]);
	} 														
	MDBFreeValues(V);

	/* From */
	fprintf(Message, "From: ");
	retVal = MWGetQuotedString(Session->DisplayName, Client->Temp, sizeof(Client->Temp));
	if (retVal == QP_STRING) {
		fputs(Client->Temp, Message);
		fprintf(Message, " <%s>\r\n", Session->EMailAddress);
	} else if (retVal == HAS_EXTENDED_CHARS) {
		HulaStrNCpy(Buffer, Session->DisplayName, sizeof(Buffer));
		ProcessMemoryStream(strlen(Buffer));
		fprintf(Message, " <%s>\r\n", Session->EMailAddress);
	} else {
		fprintf(Message, "<%s>\r\n", Session->EMailAddress);
	}

	/* The envelope needs to have the real ID and Session->EmailAddess does not necessarily contain it */
	ReplyInt = snprintf(Answer, sizeof(Answer), "QSTOR FROM %s %s\r\n", Session->EMailAddress, Session->User);
	MWSendNMAPServer(Session, Answer, ReplyInt);

	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
		fclose(Message);
		ReleaseStreams;
		RestoreUserNMAP;
		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	/* Add recipients from the To:, CC: and BCC fields */
	for (i = 0; i < 3; i++) {
		BOOL	First;

		First = TRUE;

		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
		FileHandle = fopen(Buffer, "rb");

		if (FileHandle) {
			while (!feof(FileHandle) && !ferror(FileHandle)) {
				if (fgets(Buffer, MAXEMAILNAMESIZE, FileHandle)) {
					ChopNL(Buffer);
					if (Buffer[0] == '\0') {
						continue;
					}

					switch (i) {
						case COMPOSE_TO: {
							if (First) {
								First = FALSE;
								fprintf(Message, "To: %s", Buffer);
							} else {
								fprintf(Message, ",\r\n\t%s", Buffer);
							}
							break;
						}

						case COMPOSE_CC: {
							if (First) {
								First = FALSE;
								fprintf(Message, "CC: %s", Buffer);
							} else {
								fprintf(Message, ",\r\n\t%s", Buffer);
							}
							break;
						}

						case COMPOSE_BCC: {
							break;
						}
					}

					ReplyInt = snprintf(Answer, sizeof(Answer), 
							    "QSTOR TO %s %s %d\r\n",
										Buffer,
										Buffer,
										DSN_DEFAULT);
					MWSendNMAPServer(Session, Answer, ReplyInt);

					if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
						fclose(FileHandle);
						fclose(Message);
						ReleaseStreams;
						MDBDestroyValueStruct(V);
						return(FALSE);
					} else {
						NumRecipients++;
					}
				} 
			}

			if (!First) {
				fprintf(Message, "\r\n");
			}

			fclose(FileHandle);
		} 
	}

	if (NumRecipients == 0) {
		fclose(Message);
		ReleaseStreams;
		RestoreUserNMAP;
		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	/* Date */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	fprintf(Message, "Date: %s\r\n", Buffer);

	/* Prorietary information */
	fprintf(Message, "X-Mailer: %s\r\n", PRODUCT_NAME);
	fprintf(Message, "X-Sender: %s\r\n", Session->User);

	/* MIME */
	fprintf(Message, "MIME-Version: 1.0\r\n");

	/* Message-ID */
	fprintf(Message, "Message-ID: <%lu.%x%s>\r\n", time(NULL), (unsigned int)Session->User, Session->EMailAddress);

	fprintf(Message, "Content-Type: text/plain; charset=\"%s\"\r\nContent-Transfer-Encoding: quoted-printable\r\n\r\n", Session->Charset);

	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);

	FileHandle = fopen(Buffer,"rb");
	if (!FileHandle) {
		fclose(Message);
		ReleaseStreams;
		RestoreUserNMAP;
		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	/* Stream from file to file */
	CRLFCodec = MWGetStream(NULL, "CRLF", TRUE);
	if (CRLFCodec) {
		FileInCodec.Next=CRLFCodec;
		if (BodyCharsetCodec) {
			CRLFCodec->Next = BodyCharsetCodec;
		} else {
			CRLFCodec->Next = BodyQPCodec;
		}
	} else {
		if (BodyCharsetCodec) {
			FileInCodec.Next = BodyCharsetCodec;
		} else {
			FileInCodec.Next = BodyQPCodec;
		}
	}

	ProcessFileStream(FileHandle);
	fclose(FileHandle);
	fprintf(Message, "\r\n");

	MDBDestroyValueStruct(V);

	fclose(Message);

	ReleaseStreams;

	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	if ((stat(Buffer, &sb) == -1) || (FileHandle = fopen(Buffer, "rb")) == NULL) {
		RestoreUserNMAP;
		return(FALSE);
	}	

	/* Tell NMAP the Message is coming */
	ReplyInt = snprintf(Answer, sizeof(Answer), "QSTOR MESSAGE %d\r\n", (int)sb.st_size);
	MWSendNMAPServer(Session, Answer, ReplyInt);

	while (!feof(FileHandle) && !ferror(FileHandle)) {
		if ((ReplyInt = fread(Answer, 1, BUFSIZE, FileHandle)) > 0) {
			MWSendNMAPServer(Session, Answer, ReplyInt);
		} 
	}
	fclose(FileHandle);
	
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)!=1000) {
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Kick it */
	MWSendNMAPServer(Session, "QRUN\r\n", 6);

	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
		RestoreUserNMAP;
		return(FALSE);
	}

	RestoreUserNMAP;

	/* Whack our temp files */
	CalComposeCleanUp(Session, CalendarSession);
	return(TRUE);
}

BOOL
SendICalReply(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long Action)
{
	unsigned char			Buffer[BUFSIZE + 1];
	FILE						*ICalFile;
	struct stat				FileInfo;
	ICalObject				*ICal = CalendarSession->ICal;
	MDBValueStruct			*User;
	unsigned long			i;
	unsigned char			*FullName			= NULL;
	int						ClientNMAPs			= 0;
	unsigned long			NMAPCAddress		= 0;
	unsigned long			Day;
	unsigned long			Month;
	unsigned long			Year;
	unsigned long			Hour;
	unsigned long			Minute;
	long						retVal;
	StreamStruct			MemoryInCodec;
	MWCalStreams;

	if (!ICal || Session->ReadOnly) {
		return(FALSE);
	}

	if (Action == AA_DELETE || Action == AA_DELETEALL) {
		/* Are we the organizer?  If so we send a cancel.  If not we treat it like a decline */
		if (!ICal->Organizer || !ICal->Organizer->Address || !MWQuickCmp(ICal->Organizer->Address, Session->EMailAddress)) {
			Action = AA_DECLINE;
		}
	}

	/* Connect to queue NMAP server */
	if ((MwCal.NmapQAddress != 0) && (MwCal.NmapQAddress != Session->NMAPAddress.s_addr)) {
		unsigned char	*User;

		ClientNMAPs=Session->NMAPs;
		Session->NMAPs=-1;
		Session->NBufferPtr=0;
		NMAPCAddress=Session->NMAPAddress.s_addr;
		Session->NMAPAddress.s_addr=MwCal.NmapQAddress;
		User=Session->User;
		Session->User=NULL;
		if (!MWConnectUserToNMAPServer(Session)) {
			Session->User=User;
			RestoreUserNMAP;
			return(FALSE);
		}
		Session->User=User;
	}

	/* Create Queue Entry */
	MWSendNMAPServer(Session, "QCREA\r\n", 7);
	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Start creating Header in file */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	ICalFile=fopen(Buffer, "wb");
	if (!ICalFile) {
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Prepare all required streams */
	memset(&MemoryInCodec, 0, sizeof(StreamStruct));
	RFC1522Codec=MWGetStream(&MemoryInCodec, "RFC1522", TRUE);
	RFC2445Codec=MWGetStream(NULL, "RFC2445", TRUE);
	RFC2445MLCodec=MWGetStream(NULL, "RFC2445ML", TRUE);
	MessageOutCodec=MWGetStream(RFC1522Codec, NULL, TRUE);

	/* Set up the input & output */
	MemoryInCodec.StreamData=Buffer;
	MemoryInCodec.Codec=MWStreamFromMemory;
	MemoryInCodec.Next=RFC1522Codec;

	RFC2445MLCodec->Next=RFC2445Codec;
	RFC2445Codec->Next=MessageOutCodec;

	RFC1522Codec->Charset="UTF-8";

	MessageOutCodec->StreamData=ICalFile;
	MessageOutCodec->Codec=MWStreamToFile;

	/* Start building the message */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	i=sizeof(Client->cs);
	getpeername(Client->s, (struct sockaddr *)&(Client->cs), (socklen_t *)&i);
	fprintf(ICalFile, "Received: from %s [%d.%d.%d.%d] by %s\r\n\twith %s; %s\r\n", 
						Session->User,
						Client->cs.sin_addr.s_net,
						Client->cs.sin_addr.s_host,
						Client->cs.sin_addr.s_lh,
						Client->cs.sin_addr.s_impno,
						Session->OfficialDomain, 
						PRODUCT_NAME,
						Buffer);

	/* Subject */
	fprintf(ICalFile, "Subject: ");

	if (ICal->Summary) {
		HulaStrNCpy(Buffer, ICal->Summary, sizeof(Buffer));
		ProcessMemoryStream(strlen(Buffer));
	}

	switch (Action) {
		case AA_ACCEPT: {
			fprintf(ICalFile, " (Accepted)\r\n");
			break;
		}
		case AA_COMPLETE: {
			fprintf(ICalFile, " (Completed)\r\n");
			break;
		}
		case AA_DECLINE: {
			fprintf(ICalFile, " (Declined)\r\n");
			break;
		}
		case AA_TENTATIVE: {
			fprintf(ICalFile, " (Tentative)\r\n");
			break;
		}
		case AA_DELEGATE: {
			fprintf(ICalFile, " (Delegated)\r\n");
			break;
		}
		case AA_DELETE:
		case AA_DELETEALL: {
			fprintf(ICalFile, " (Cancel)\r\n");
			break;
		}
	}

	User = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);
	/* Reply-To:*/
	if (MDBRead(Session->UserDN, MSGSRV_A_EMAIL_ADDRESS, User) > 0) {
		fprintf(ICalFile, "Reply-To: %s\r\n", User->Value[0]);
	} 														
	MDBDestroyValueStruct(User);  	

	/* From */
	fprintf(ICalFile, "From: ");
	retVal = MWGetQuotedString(Session->DisplayName, Client->Temp, sizeof(Client->Temp));
	if (retVal == QP_STRING) {
		fputs(Client->Temp, ICalFile);
		fprintf(ICalFile, " <%s>\r\n", Session->EMailAddress);
	} else if (retVal == HAS_EXTENDED_CHARS) {
		HulaStrNCpy(Buffer, Session->DisplayName, sizeof(Buffer));
		ProcessMemoryStream(strlen(Buffer));
		fprintf(ICalFile, " <%s>\r\n", Session->EMailAddress);
	} else {
		fprintf(ICalFile, "<%s>\r\n", Session->EMailAddress);
	}

	FullName = MemStrdup(Client->Temp);

	i = snprintf(Buffer, sizeof(Buffer), "QSTOR FROM %s %s\r\n", Session->EMailAddress, Session->User);
	MWSendNMAPServer(Session, Buffer, i);

	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
		if (FullName) {
			MemFree(FullName);
		}
		ReleaseStreams;
		RestoreUserNMAP;
		fclose(ICalFile);
		return(FALSE);
	}	

	if (Action != AA_DELETE && Action != AA_DELETEALL) {
		/* Add Organizer to the To: field */
		if (ICal->Organizer && ICal->Organizer->Address) {
			fprintf(ICalFile, "To: %s, ", ICal->Organizer->Address);
			i = snprintf(Buffer, sizeof(Buffer), "QSTOR TO %s %s %d\r\n",
							ICal->Organizer->Address, 
							ICal->Organizer->Address,
							DSN_DEFAULT);
			MWSendNMAPServer(Session, Buffer, i);

			if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
				if (FullName) {
					MemFree(FullName);
				}
				ReleaseStreams;
				RestoreUserNMAP;
				fclose(ICalFile);
				return(FALSE);
			}
		} else {
			/* No one to send to so whats the point? */
			if (FullName) {
				MemFree(FullName);
			}
			ReleaseStreams;
			RestoreUserNMAP;
			fclose(ICalFile);
			return(FALSE);
		}
	} else {
		ICalVAttendee			*Attendee;

		/* If its a cancel add every one else to the TO: field */
		Attendee = ICal->Attendee;

		fprintf(ICalFile, "To: ");

		while (Attendee) {
			if (MWQuickCmp(Attendee->Address, Session->EMailAddress)==FALSE) {
				fprintf(ICalFile, "%s, ", Attendee->Address);
				i = snprintf(Buffer, sizeof(Buffer), "QSTOR TO %s %s %d\r\n",
								Attendee->Address, 
								Attendee->Address,
								DSN_DEFAULT);
				MWSendNMAPServer(Session, Buffer, i);

				if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
					if (FullName) {
						MemFree(FullName);
					}
					ReleaseStreams;
					RestoreUserNMAP;
					fclose(ICalFile);
					return(FALSE);
				}
			}

			Attendee = Attendee->Next;
		}
	}
	fprintf(ICalFile, "\r\n");

	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	fprintf(ICalFile, "Date: %s\r\n", Buffer);

	/* Prorietary information */	
	fprintf(ICalFile, "X-Mailer: %s\r\n", PRODUCT_NAME);
	fprintf(ICalFile, "X-Sender: %s\r\n", Session->EMailAddress);

	/* MIME */
	if (Action == AA_DELETE || Action == AA_DELETEALL) {
		fprintf(ICalFile, "MIME-Version: 1.0\r\nContent-Type: text/calendar; method=CANCEL; charset=utf-8\r\n\r\n");
	} else {
		fprintf(ICalFile, "MIME-Version: 1.0\r\nContent-Type: text/calendar; method=REPLY; charset=utf-8\r\n\r\n");
	}

	MemoryInCodec.Next = RFC2445Codec;

	/* Now for the iCal object */
	if (Action == AA_DELETE || Action == AA_DELETEALL) {
		fprintf(ICalFile, "BEGIN:VCALENDAR\r\nPRODID:-//Novell Inc//NetMail ModWeb//\r\nVERSION:2.0\r\nMETHOD:CANCEL\r\n");
	} else {
		fprintf(ICalFile, "BEGIN:VCALENDAR\r\nPRODID:-//Novell Inc//NetMail ModWeb//\r\nVERSION:2.0\r\nMETHOD:REPLY\r\n");
	}

	if (ICal->Type == ICAL_VEVENT) {
		fprintf(ICalFile, "BEGIN:VEVENT\r\n");
	} else if (ICal->Type == ICAL_VTODO) {
		fprintf(ICalFile, "BEGIN:VTODO\r\n");
	} else if (ICal->Type == ICAL_VJOURNAL) {
		fprintf(ICalFile, "BEGIN:VJOURNAL\r\n");
	}

	/* On a cancel we MUST list all attendees */
	if (Action == AA_DELETE || Action == AA_DELETEALL) {
		ICalVAttendee	*Attendee;

		Attendee = ICal->Attendee;

		while (Attendee) {
			/* HOLE: if I am a non-participant Attendee do I need to send this */
			snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s:MAILTO:%s\r\n", Attendee->Name, Attendee->Address);
			ProcessMemoryStream(strlen(Buffer));
			Attendee = Attendee->Next;
		}
	} else {
		/* Just send our own status */
		switch (Action) {
			case AA_ACCEPT: {
				snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;PARTSTAT=ACCEPTED:MAILTO:%s\r\n", FullName, Session->EMailAddress);
				break;
			}

			case AA_DECLINE: {
				snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;PARTSTAT=DECLINED:MAILTO:%s\r\n", FullName, Session->EMailAddress);
				break;
			}

			case AA_COMPLETE: {
				snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;PARTSTAT=COMPLETED:MAILTO:%s\r\n", FullName, Session->EMailAddress);
				break;
			}

			case AA_TENTATIVE: {
				snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;PARTSTAT=TENTATIVE:MAILTO:%s\r\n", FullName, Session->EMailAddress);
				break;
			}

			case AA_DELEGATE: {
				/* Send a Reply to the organizer with the following updates:
					1- The Delegators's ATTENDEE property partstat paraemter set to delegated and the delegated-to parameter is set to the address of the delegate
					2- Add and additional ATTENDEE property for the Delegate with the delegated-from property parameter set to the Delegator (Session->EmailAddress)
					3- Indicate whether we want to continue recieving updates (RSVP = FALSE)
					4- Original request should have already been sent to Delegate.
				*/
				FILE						*detailsFile;
				unsigned char			delegatee[BUFSIZE + 1];
				unsigned char        *DelegateesEmail = NULL;
				unsigned char        *DelegateesFullName = NULL;

				delegatee[0] = '\0';

				snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[0]);
				detailsFile = fopen(Buffer, "rb");

				if (detailsFile) {
					while (!feof(detailsFile) && !ferror(detailsFile)) {
						if (fgets(delegatee, MAXEMAILNAMESIZE, detailsFile)) {
							ChopNL(delegatee);
						}
					}

					fclose(detailsFile);
				}

				/* get the email address and full name of delegatee */
				if (MsgGetEmailAddressAndFullName(delegatee, (char **)&DelegateesEmail, (char **)&DelegateesFullName, Session, MSG_GET_EMAIL|MSG_GET_FULLNAME)){
					snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;PARTSTAT=DELEGATED;DELEGATED-TO=\"Mailto:%s\":MAILTO:%s\r\n", FullName, DelegateesEmail, Session->EMailAddress);
					ProcessMemoryStream(strlen(Buffer));
					snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;ROLE=REQ-PARTICIPANT;RSVP=TRUE;DELEGATED-FROM=\"Mailto:%s\":MAILTO:%s\r\n", DelegateesFullName ? (char *)DelegateesFullName : "", Session->EMailAddress, DelegateesEmail);

				}

				if (DelegateesEmail)
					MemFree(DelegateesEmail);
				if (DelegateesFullName)
					MemFree(DelegateesFullName);
				
				break;
			}

			default: {
				if (FullName) {
					MemFree(FullName);
				}
				ReleaseStreams;
				RestoreUserNMAP;
				fclose(ICalFile);
				return(FALSE);
			}
		}
		ProcessMemoryStream(strlen(Buffer));
	}

	if (FullName) {
		MemFree(FullName);
	}

	/* SEQUENCE */
	fprintf(ICalFile, "SEQUENCE:%d\r\n", (int)ICal->Sequence);

	/* UID */
	if (ICal->UID) {
		fprintf(ICalFile, "UID:%s\r\n", ICal->UID);
	}

	/* Recurrence ID */
	if (Action == AA_DELETE) {
		/* If it is not preset it deletes all reccurences so we do not supply it for a delete all */
		fprintf(ICalFile, "RECURRENCE-ID:%04d%02d%02dT%02d%02d00Z\r\n", (int)ICal->Start.Year, 
				  (int)ICal->Start.Month, (int)ICal->Start.Day, (int)ICal->Start.Hour, (int)ICal->Start.Minute);
	}

	if (Action == AA_DELETE || Action == AA_DELETEALL) {
		fprintf(ICalFile, "STATUS:CANCELLED\r\n");
	}
	
	/* DTSTAMP */
	MsgGetDMY(time(NULL), &Day, &Month, &Year, &Hour, &Minute, NULL);
	fprintf(ICalFile, "DTSTAMP:%04d%02d%02dT%02d%02d00Z\r\n", (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);

	/* Close it off */

	if (ICal->Type == ICAL_VEVENT) {
		fprintf(ICalFile, "END:VEVENT\r\nEND:VCALENDAR\r\n");
	} else if (ICal->Type == ICAL_VTODO) {
		fprintf(ICalFile, "END:VTODO\r\nEND:VCALENDAR\r\n");
	} else if (ICal->Type == ICAL_VJOURNAL) {
		fprintf(ICalFile, "END:VJOURNAL\r\nEND:VCALENDAR\r\n");
	}

	/* Done - Clean up */
	fclose(ICalFile);

	/* Send the sucker to NMAP */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	if ((stat(Buffer, &FileInfo)==-1) || (ICalFile=fopen(Buffer, "rb"))==NULL) {
		ReleaseStreams;
		RestoreUserNMAP;
		return(FALSE);
	}	

	/* Tell NMAP the Message is coming */
	i=snprintf(Buffer, sizeof(Buffer), "QSTOR MESSAGE %d\r\n", (int)FileInfo.st_size);
	MWSendNMAPServer(Session, Buffer, i);

	while (!feof(ICalFile) && !ferror(ICalFile)) {
		if ((i=fread(Buffer, 1, BUFSIZE, ICalFile))>0) {
			MWSendNMAPServer(Session, Buffer, i);
		} 
	}
	fclose(ICalFile);
	
	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE)!=1000) {
		ReleaseStreams;
		RestoreUserNMAP;
		return(FALSE);
	}
						  
	/* Kick it */
	MWSendNMAPServer(Session, "QRUN\r\n", 6);

	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE)!=1000) {
		ReleaseStreams;
		RestoreUserNMAP;
		return(FALSE);
	}

	ReleaseStreams;
	RestoreUserNMAP;

	/* Whack our temp file */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	unlink(Buffer);

	return(TRUE);
}


static void 
PrintICalAttendee(char *Buffer, ICalVAttendee *Attendee)
{

	/* start with Attendee name CN */
	snprintf(Buffer, BUFSIZE, "ATTENDEE;CN=%s;",Attendee->Name);

	/* calendar user type */
	switch(Attendee->Type) {
		case ICAL_CUTYPE_INDIVIDUAL:
			/* default unnecessary */
			break;
		case ICAL_CUTYPE_GROUP:
			strncat(Buffer,"CUTYPE=GROUP;", BUFSIZE);
			break; 
		case ICAL_CUTYPE_RESOURCE:
			strncat(Buffer,"CUTYPE=RESOURCE;", BUFSIZE);
			break; 
		case ICAL_CUTYPE_ROOM:
			strncat(Buffer,"CUTYPE=ROOM;", BUFSIZE);
			break; 
		case ICAL_CUTYPE_UNKNOWN:
			strncat(Buffer,"CUTYPE=UNKNOWN;", BUFSIZE);
			break; 


		default:
			break;
	}

	/* participation status */
	switch(Attendee->State){
		case	ICAL_PARTSTAT_NEEDS_ACTION:
			/* default unnecessary */
			break;
		case ICAL_PARTSTAT_ACCEPTED:
			strncat(Buffer,"PARTSTAT=ACCEPTED;", BUFSIZE);
			break;
		case ICAL_PARTSTAT_DECLINED:
			strncat(Buffer,"PARTSTAT=DECLINED;", BUFSIZE);
			break;

		case ICAL_PARTSTAT_TENTATIVE:
			strncat(Buffer,"PARTSTAT=TENTATIVE;", BUFSIZE);
			break;

		case ICAL_PARTSTAT_DELEGATED:
			strncat(Buffer,"PARTSTAT=DELEGATED;", BUFSIZE);
			break;

		case ICAL_PARTSTAT_COMPLETED:
			strncat(Buffer,"PARTSTAT=COMPLETED;", BUFSIZE);
			break;

		case ICAL_PARTSTAT_IN_PROCESS:
			strncat(Buffer,"PARTSTAT=IN-PROCESS;", BUFSIZE);
			break;
		default:
			break;

	}

	/* role */
	switch(Attendee->Role){
		case ICAL_ROLE_CHAIR:
			strncat(Buffer,"ROLE=CHAIR;", BUFSIZE);
			break;

		case ICAL_ROLE_REQUIRED_PARTICIPANT:
			strncat(Buffer,"ROLE=REQ-PARTICIPANT;", BUFSIZE);
			break;

		case ICAL_ROLE_OPTIONAL_PARTICIPANT:
			strncat(Buffer,"ROLE=OPT-PARTICIPANT;", BUFSIZE);
			break;

		case ICAL_ROLE_NON_PARTICIPANT:
			strncat(Buffer,"ROLE=NON-PARTICIPANT;", BUFSIZE);
			break;

		default:
			break;
	}


	/* check for RSVP */
	if (Attendee->RSVP)
		strncat(Buffer, "RSVP=TRUE;", BUFSIZE);

	/* check for delegated from and delegated to */
	if (Attendee->Delegated)
		snprintf(Buffer + strlen(Buffer), BUFSIZE - strlen(Buffer), "DELEGATED-TO=\"Mailto:%s\"",Attendee->Delegated);

	if (Attendee->DelegatedFrom)
		snprintf(Buffer + strlen(Buffer), BUFSIZE - strlen(Buffer), "DELEGATED-FROM=\"Mailto:%s\"",Attendee->Delegated);
		
	/* finally add the attendees address */
	snprintf(Buffer + strlen(Buffer), BUFSIZE - strlen(Buffer), ":MAILTO:%s\r\n", Attendee->Address);

	return;


}


static int 
IsOrganizer(SessionStruct *Session, char *Address)
{
	if (MWQuickCmp(Address, Session->User) || MWQuickCmp(Address, Session->EMailAddress)) {
		return(TRUE);
	}

	return(FALSE);

}

static int 
IsAttendee(ICalObject *ICal, char *Address) 
{
	ICalVAttendee *Attendee;

	for (Attendee = ICal->Attendee; Attendee; Attendee = Attendee->Next) {
		if ((Attendee->Address && MWQuickCmp(Attendee->Address, Address)) || 
			 (Attendee->Name && MWQuickCmp(Attendee->Name, Address))){
			return(TRUE);
		}
	}
	
	return(FALSE);
		
}


BOOL
SendICalDelegation(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned char			Buffer[BUFSIZE + 1];
	unsigned char			innerBoundary[BUFSIZE + 1];
	unsigned char			outerBoundary[BUFSIZE + 1];
	unsigned char			Delegatee[BUFSIZE + 1];
	FILE						*iCalFile;
	FILE						*detailsFile;
	struct stat				FileInfo;
	ICalObject				*iCal					= CalendarSession->ICal;
	MDBValueStruct			*User;
	unsigned long			i;
	unsigned char			*FullName			= NULL;
	unsigned char			*DelegateesFullName	= NULL;
	unsigned char        *DelegateesEmail = NULL;
	
	int						ClientNMAPs			= 0;
	unsigned long			NMAPCAddress		= 0;
	long						retVal;
	BOOL                 FoundMyself = FALSE; 
	ICalVAttendee				*Attendee;
	StreamStruct			FileInCodec;
	StreamStruct			MemoryInCodec;
	MWCalStreams;

	if (!iCal || Session->ReadOnly) {
		return(FALSE);
	}



	/*
		The sequence number must NOT be incremented when sending a delegation "REQUEST"

		To Delegate:

		-	Delegator forwards the existing iCal object as a REQUEST to the Delegatee.  The REQUEST method MUST include
			an ATTENDEE property with the calendar address of the delegatee.
		-	The delegator MUST also send a REPLY method to the organizer with the delegator's ATTENDEE property partstat
			parameter value set to "delegated", and the "delegated-to" parameter MUST be included with the calendar
			address of the delegatee.


		When accepting a delegated REQUEST:

		-	The REPLY method to the organizer SHOULD include the ATTENDEE property with the "delegated-from" parameter
			value of the delegator's calendar address.

	*/

	/* Connect to queue NMAP server */
	if ((MwCal.NmapQAddress != 0) && (MwCal.NmapQAddress != Session->NMAPAddress.s_addr)) {
		unsigned char	*User;

		ClientNMAPs = Session->NMAPs;
		Session->NMAPs = -1;
		Session->NBufferPtr = 0;
		NMAPCAddress = Session->NMAPAddress.s_addr;
		Session->NMAPAddress.s_addr = MwCal.NmapQAddress;
		User = Session->User;
		Session->User = NULL;

		if (!MWConnectUserToNMAPServer(Session)) {
			Session->User = User;
			RestoreUserNMAP;
			return(FALSE);
		}
		Session->User = User;
	}

	/* Create Queue Entry */
	MWSendNMAPServer(Session, "QCREA\r\n", 7);
	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Start creating Header in file */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	iCalFile = fopen(Buffer, "wb");
	if (!iCalFile) {
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Prepare all required streams */
	memset(&FileInCodec, 0, sizeof(StreamStruct));
	memset(&MemoryInCodec, 0, sizeof(StreamStruct));
	RFC1522Codec = MWGetStream(&MemoryInCodec, "RFC1522", TRUE);
	RFC2445Codec = MWGetStream(NULL, "RFC2445", TRUE);
	RFC2445MLCodec = MWGetStream(NULL, "RFC2445ML", TRUE);
	MessageOutCodec = MWGetStream(RFC1522Codec, NULL, TRUE);

	/* Set up the input & output */
	MemoryInCodec.StreamData = Buffer;
	MemoryInCodec.Codec = MWStreamFromMemory;
	MemoryInCodec.Next = RFC1522Codec;

	FileInCodec.Codec = MWStreamFromFile;
	FileInCodec.Next = RFC1522Codec;

	RFC2445MLCodec->Next = RFC2445Codec;
	RFC2445Codec->Next = MessageOutCodec;

	RFC1522Codec->Charset = "UTF-8";

	MessageOutCodec->StreamData = iCalFile;
	MessageOutCodec->Codec = MWStreamToFile;

	BodyCharsetCodec = MWGetStream(NULL, Session->Charset, TRUE);
	if (BodyCharsetCodec) {
		BodyQPCodec = MWGetStream(BodyCharsetCodec, "quoted-printable", TRUE);
	} else {
		BodyQPCodec = MWGetStream(NULL, "quoted-printable", TRUE);
	}
	BodyQPCodec->Next = MessageOutCodec;

	/* Start building the message */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	i = sizeof(Client->cs);

	getpeername(Client->s, (struct sockaddr *)&(Client->cs), (socklen_t *)&i);
	fprintf(iCalFile, "Received: from %s [%d.%d.%d.%d] by %s\r\n\twith %s; %s\r\n", 
						Session->User,
						Client->cs.sin_addr.s_net,
						Client->cs.sin_addr.s_host,
						Client->cs.sin_addr.s_lh,
						Client->cs.sin_addr.s_impno,
						Session->OfficialDomain, 
						PRODUCT_NAME,
						Buffer);

	/* Subject */
	fprintf(iCalFile, "Subject: ");

	if (iCal->Summary) {
		HulaStrNCpy(Buffer, iCal->Summary, sizeof(Buffer));
		ProcessMemoryStream(strlen(Buffer));
	}

	fprintf(iCalFile, " (Delegated)\r\n");

	User = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);
	/* Reply-To:*/
	if (MDBRead(Session->UserDN, MSGSRV_A_EMAIL_ADDRESS, User) > 0) {
		fprintf(iCalFile, "Reply-To: %s\r\n", User->Value[0]);
	} 														
	MDBDestroyValueStruct(User);

	/* From */
	fprintf(iCalFile, "From: ");
	retVal = MWGetQuotedString(Session->DisplayName, Client->Temp, sizeof(Client->Temp));
	if (retVal == QP_STRING) {
		fputs(Client->Temp, iCalFile);
		fprintf(iCalFile, " <%s>\r\n", Session->EMailAddress);
	} else if (retVal == HAS_EXTENDED_CHARS) {
		HulaStrNCpy(Buffer, Session->DisplayName, sizeof(Buffer));
		ProcessMemoryStream(strlen(Buffer));
		fprintf(iCalFile, " <%s>\r\n", Session->EMailAddress);
	} else {
		fprintf(iCalFile, "<%s>\r\n", Session->EMailAddress);
	}

	FullName = MemStrdup(Client->Temp);

	i = snprintf(Buffer, sizeof(Buffer), "QSTOR FROM %s %s\r\n", Session->EMailAddress, Session->User);
	MWSendNMAPServer(Session, Buffer, i);

	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
		if (FullName) {
			MemFree(FullName);
		}

		ReleaseStreams;
		RestoreUserNMAP;
		fclose(iCalFile);
		return(FALSE);
	}	

	/* Add the delegatee */

	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[0]);
	detailsFile = fopen(Buffer, "rb");

	Delegatee[0] = '\0';
	if (detailsFile) {
		BOOL Delegated = FALSE;
		while (!feof(detailsFile) && !ferror(detailsFile)) {
			if (fgets(Delegatee, MAXEMAILNAMESIZE, detailsFile)) {
				ChopNL(Delegatee);
				if (!Delegated && Delegatee[0] != '\0' && !IsOrganizer(Session, &Delegatee[0]) && !IsAttendee(iCal, &Delegatee[0])) {
					/* OK */
					Delegated = TRUE;
					fprintf(iCalFile, "To: %s\r\n", Delegatee);
					MWSendNMAPServer(Session, Buffer, snprintf(Buffer, sizeof(Buffer), "QSTOR TO %s %s %d\r\n", Delegatee, Delegatee, DSN_DEFAULT));

					if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
						fclose(detailsFile);

						if (FullName) {
							MemFree(FullName);
						}

						ReleaseStreams;
						RestoreUserNMAP;
						fclose(iCalFile);
						return(FALSE);
					}

				} else {
					/* we currently only support delegating to one recipient at a time */
					if (Delegated) {
						CalendarSession->Error = ERROR_TOO_MANY_RECIPIENTS;
					} else if (Delegatee[0] == '\0'){
						CalendarSession->Error = ERROR_NO_RECIPIENTS;
					} else {
						CalendarSession->Error = ERROR_BAD_TO_ADDR;
					}
					

					fclose(detailsFile);
					
					if (FullName) {
						MemFree(FullName);
					}
					
					ReleaseStreams;
					RestoreUserNMAP;
					fclose(iCalFile);
					return(FALSE);
				}
			}
		}

		fclose(detailsFile);

		if (!Delegated) {
			CalendarSession->Error = ERROR_NO_RECIPIENTS;
			if (FullName) {
				MemFree(FullName);
			}
					
			ReleaseStreams;
			RestoreUserNMAP;
			return(FALSE);
		}

	}

	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	fprintf(iCalFile, "Date: %s\r\n", Buffer);

	/* Prorietary information */	
	fprintf(iCalFile, "X-Mailer: %s\r\n", PRODUCT_NAME);
	fprintf(iCalFile, "X-Sender: %s\r\n", Session->EMailAddress);

	/* MIME */
	snprintf(outerBoundary, sizeof(outerBoundary), "------=_ModWebBOUNDARY_%x_o%lu", (unsigned int)Session->User, (long)time(NULL));
	snprintf(innerBoundary, sizeof(innerBoundary), "------=_ModWebBOUNDARY_%x_i%lu", (unsigned int)Session->User, (long)time(NULL));

	fprintf(iCalFile, "Content-Type: multipart/mixed;\r\n\tboundary=\"%s\"\r\n\r\n", outerBoundary);
	fprintf(iCalFile, "This is a multi-part message in MIME format.\r\n\r\n--%s\r\nContent-Type: text/plain; charset=\"%s\"\r\nContent-Transfer-Encoding: quoted-printable\r\n\r\n", outerBoundary, Session->Charset);

	/* Stream from file to file */
	CRLFCodec = MWGetStream(NULL, "CRLF", TRUE);
	if (CRLFCodec) {
		FileInCodec.Next = CRLFCodec;
		if (BodyCharsetCodec) {
			CRLFCodec->Next = BodyCharsetCodec;
		} else {
			CRLFCodec->Next = BodyQPCodec;
		}
	} else {
		if (BodyCharsetCodec) {
			FileInCodec.Next = BodyCharsetCodec;
		} else {
			FileInCodec.Next = BodyQPCodec;
		}
	}

	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);

	detailsFile = fopen(Buffer,"rb");
	if (detailsFile) {

		fprintf(iCalFile, "\r\n");
		ProcessFileStream(detailsFile);

		fclose(detailsFile);
	}

	fprintf(iCalFile, "\r\n\r\n--%s\r\nContent-Type: multipart/alternative;\r\n\tboundary=\"%s\"\r\n\r\n", outerBoundary, innerBoundary);

	fprintf(iCalFile, "This is a multi-part message in MIME format.\r\n\r\n--%s\r\nContent-Type: text/plain; charset=\"%s\"\r\nContent-Transfer-Encoding: quoted-printable\r\n\r\n", innerBoundary, Session->Charset);


	/* Stream from memory to file */
	CRLFCodec = MWGetStream(NULL, "CRLF", TRUE);
	if (CRLFCodec) {
		MemoryInCodec.Next = CRLFCodec;
		if (BodyCharsetCodec) {
			CRLFCodec->Next = BodyCharsetCodec;
		} else {
			CRLFCodec->Next = BodyQPCodec;
		}
	} else {
		if (BodyCharsetCodec) {
			MemoryInCodec.Next = BodyCharsetCodec;
		} else {
			MemoryInCodec.Next = BodyQPCodec;
		}
	}

	/* Send a text version of the iCal object */
	switch (iCal->Type) {
		case ACTION_APPOINTMENT: {
			fprintf(iCalFile, "Item Type: Appointment\r\n");
			break;
		}

		case ACTION_TASK:	{
			fprintf(iCalFile, "Item Type: Task\r\n");
			break;
		}

		case ACTION_NOTE:	{
			fprintf(iCalFile, "Item Type: Note\r\n");
			break;
		}
	}

	/* Don't add the attendee's because they can already see the list of people who received the message */

	GetReadableDate(Session->TimezoneOffset, iCal->Start.UTC, Buffer);
	fprintf(iCalFile, "Begin time: %s\r\n", Buffer);

	if (iCal->End.UTC > iCal->Start.UTC) {
		unsigned long		duration = iCal->End.UTC - iCal->Start.UTC;
		unsigned long		hours = duration / (60 * 60);
		unsigned long		minutes = (duration % (60 * 60)) / 60;

		if (hours < 24) {

			/* If the duration is less than a day show duration.  If not show end time */
			if (minutes == 0) {
				/* Just hours */

				fprintf(iCalFile, "Duration:   %lu Hour%s\r\n", hours, (hours > 1) ? "s" : "");
			} else if (hours != 0) {
				/* Hours and minutes */

				fprintf(iCalFile, "Duration:   %lu Hour%s and %lu Minutes\r\n", hours, (hours > 1) ? "s" : "", minutes);
			} else {
				/* Just minutes */

				fprintf(iCalFile, "Duration:   %lu Minutes\r\n", minutes);
			}
		} else {
			GetReadableDate(Session->TimezoneOffset, iCal->End.UTC, Buffer);
			fprintf(iCalFile, "End time:   %s\r\n", Buffer);
		}
	}

	/* LOCATION */
	if (iCal->Type != ACTION_NOTE && iCal->Location) {
		fprintf(iCalFile, "Location:   ");

		MemoryInCodec.StreamData = iCal->Location;
		ProcessMemoryStream(strlen(iCal->Location));
	}

	/* DESCRIPTION */
	if (iCal->Description) {
		fprintf(iCalFile, "Location:   ");

		MemoryInCodec.StreamData = iCal->Description;
		ProcessMemoryStream(strlen(iCal->Description));
	}

	fprintf(iCalFile, "\r\n");

	/* We don't need to send the summary, because it "should" match the subject of the email */

	fprintf(iCalFile, "\r\n--%s\r\nContent-Type: text/calendar; method=REQUEST; charset=utf-8\r\n\r\n", innerBoundary);

	/* Reset to use regular RFC2445 codec */
	MemoryInCodec.StreamData = Buffer;
	MemoryInCodec.Next = RFC2445Codec;

	/*
		Now for the iCal object

		We need to recreate the original iCal object based on the parsed iCal object we have in memory.  We are not
		currently going to support delegating a recurring item (If you try it will only delegate one recurrence).
		When we send the attendee's we will add the delegatee and modify our propert's appropriatly.
	*/
	fprintf(iCalFile, "BEGIN:VCALENDAR\r\nPRODID:-//Novell Inc//NetMail ModWeb//\r\nVERSION:2.0\r\nMETHOD:REQUEST\r\n");

	if (iCal->Type == ICAL_VEVENT) {
		fprintf(iCalFile, "BEGIN:VEVENT\r\n");
	} else if (iCal->Type == ICAL_VTODO) {
		fprintf(iCalFile, "BEGIN:VTODO\r\n");
	} else if (iCal->Type == ICAL_VJOURNAL) {
		fprintf(iCalFile, "BEGIN:VJOURNAL\r\n");
	}

	/* Add the organizer */
	if (iCal->Organizer && iCal->Organizer->Address) {
		fprintf(iCalFile, "ORGANIZER;CN=%s:MAILTO:%s\r\n", iCal->Organizer->Name ? (char *)iCal->Organizer->Name : "", (char *)iCal->Organizer->Address);
	}

	/* get the email address and full name of delegatee */
	if (!MsgGetEmailAddressAndFullName(Delegatee, (char **)&DelegateesEmail, (char **)&DelegateesFullName, Session, MSG_GET_EMAIL|MSG_GET_FULLNAME))	{
		/* error */
		if (FullName)
			MemFree(FullName);

		fclose(iCalFile);
		ReleaseStreams;
		RestoreUserNMAP;
		return(FALSE);
	}

	/* We must list all attendee's */
	for (Attendee = iCal->Attendee;
		  Attendee != NULL;
		  Attendee = Attendee->Next){

		/* Skip my Entry here and add it below */
		/* This gaurantees my entry will be in list even if its not in the original object (ALIAS)*/
		if (!MWQuickCmp(Attendee->Address, Session->EMailAddress)) {
			PrintICalAttendee(Buffer, Attendee);
			ProcessMemoryStream(strlen(Buffer));
		}
		else	{
			/* make a copy of the attendee info */
			ICalVAttendee me = *Attendee;
			
			/* fill in the Delegated field */
			me.Delegated = DelegateesEmail;
			
			/* update the status which is now delegated */
			me.State = ICAL_PARTSTAT_DELEGATED;
			PrintICalAttendee(Buffer,&me);
			ProcessMemoryStream(strlen(Buffer));
			FoundMyself = TRUE;

		}		


	}

	if (!FoundMyself)	{
		/* Add my entry here */
		snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;ROLE=REQ-PARTICIPANT;PARTSTAT=DELEGATED;DELEGATED-TO=\"Mailto:%s\":MAILTO:%s\r\n", FullName ? (char *)FullName : "", DelegateesEmail, Session->EMailAddress);
		ProcessMemoryStream(strlen(Buffer));
	}
	
	/* Add the delegatee */
	snprintf(Buffer, sizeof(Buffer), "ATTENDEE;CN=%s;ROLE=REQ-PARTICIPANT;RSVP=TRUE;DELEGATED-FROM=\"Mailto:%s\":MAILTO:%s\r\n", DelegateesFullName ? (char *)DelegateesFullName : "", Session->EMailAddress, DelegateesEmail);
	ProcessMemoryStream(strlen(Buffer));

	if (DelegateesFullName){
		MemFree(DelegateesFullName);
	}

	if (FullName) {
		MemFree(FullName);
	}

	/* DTSTART */
	fprintf(iCalFile, "DTSTART:%04d%02d%02dT%02d%02d00Z\r\n", (int)iCal->Start.Year,(int)iCal->Start.Month, (int)iCal->Start.Day, (int)iCal->Start.Hour, (int)iCal->Start.Minute);

	/* DTEND */
	fprintf(iCalFile, "DTEND:%04d%02d%02dT%02d%02d00Z\r\n", (int)iCal->End.Year, (int)iCal->End.Month, (int)iCal->End.Day, (int)iCal->End.Hour, (int)iCal->End.Minute);

	/* Switch codec to "MultiLine RFC2445" codec, for sending the Summary, Location, and Description */
	MemoryInCodec.Next = RFC2445MLCodec;

	/* SUMMARY */
	if (iCal->Summary) {
		fprintf(iCalFile, "SUMMARY:");

		/* preset state to contain the chars we already sent for this line */
		RFC2445Codec->State = 8;
		MemoryInCodec.StreamData = iCal->Summary;

		ProcessMemoryStream(strlen(iCal->Summary));
		fprintf(iCalFile, "\r\n");
	}

	/* DESCRIPTION */
	if (iCal->Description) {
		fprintf(iCalFile, "DESCRIPTION:");
		/* preset state to contain the chars we already sent for this line */
		RFC2445Codec->State = 12;
		MemoryInCodec.StreamData = iCal->Description;

		ProcessMemoryStream(strlen(iCal->Description));
		fprintf(iCalFile, "\r\n");
	}

	/* LOCATION */
	if (iCal->Location) {
		fprintf(iCalFile, "LOCATION:");

		/* preset state to contain the chars we already sent for this line */
		RFC2445Codec->State = 9;
		MemoryInCodec.StreamData = iCal->Location;

		ProcessMemoryStream(strlen(iCal->Location));
		fprintf(iCalFile, "\r\n");
	}

	/* Reset to use regular RFC2445 codec */
	MemoryInCodec.StreamData = Buffer;
	MemoryInCodec.Next = RFC2445Codec;

	/* SEQUENCE */
	fprintf(iCalFile, "SEQUENCE:%lu\r\n", iCal->Sequence);

	/* UID */
	if (iCal->UID) {
		fprintf(iCalFile, "UID:%s\r\n", iCal->UID);
	}

	/* DTSTAMP */
	fprintf(iCalFile, "DTSTAMP:%04d%02d%02dT%02d%02d00Z\r\n", (int)iCal->Stamp.Year, (int)iCal->Stamp.Month, (int)iCal->Stamp.Day, (int)iCal->Stamp.Hour, (int)iCal->Stamp.Minute);

	/* Close it off */
	if (iCal->Type == ICAL_VEVENT) {
		fprintf(iCalFile, "END:VEVENT\r\nEND:VCALENDAR\r\n");
	} else if (iCal->Type == ICAL_VTODO) {
		fprintf(iCalFile, "END:VTODO\r\nEND:VCALENDAR\r\n");
	} else if (iCal->Type == ICAL_VJOURNAL) {
		fprintf(iCalFile, "END:VJOURNAL\r\nEND:VCALENDAR\r\n");
	}

	fprintf(iCalFile, "\r\n--%s--\r\n\r\n", innerBoundary);
	fprintf(iCalFile, "\r\n--%s--\r\n\r\n", outerBoundary);

	/* Done - Clean up */
	fclose(iCalFile);

	/* Send the sucker to NMAP */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	if ((stat(Buffer, &FileInfo)==-1) || (iCalFile=fopen(Buffer, "rb"))==NULL) {
		ReleaseStreams;
		RestoreUserNMAP;
		if (DelegateesEmail) {
			MemFree(DelegateesEmail);
		}
		return(FALSE);
	}	

	/* Tell NMAP the Message is coming */
	MWSendNMAPServer(Session, Buffer, snprintf(Buffer, sizeof(Buffer), "QSTOR MESSAGE %d\r\n", (int)FileInfo.st_size));

	while (!feof(iCalFile) && !ferror(iCalFile)) {
		if ((i = fread(Buffer, 1, BUFSIZE, iCalFile)) > 0) {
			MWSendNMAPServer(Session, Buffer, i);
		} 
	}
	fclose(iCalFile);
	
	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
		ReleaseStreams;
		RestoreUserNMAP;
		if (DelegateesEmail) {
			MemFree(DelegateesEmail);
		}
		return(FALSE);
	}
						  
	/* Kick it */
	MWSendNMAPServer(Session, "QRUN\r\n", 6);

	if (MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE) != 1000) {
		ReleaseStreams;
		RestoreUserNMAP;
		if (DelegateesEmail) {
			MemFree(DelegateesEmail);
		}
		return(FALSE);
	}

	ReleaseStreams;
	RestoreUserNMAP;

	/* Whack our temp file */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	unlink(Buffer);

	/*		Are we the organizer? If so update my attendee status to delegated and put the new address in as an attendee */
	/*    If we are not then we must rely on the ICal reply to update  this through calendar agent */
	if (iCal->Organizer && iCal->Organizer->Address && DelegateesEmail && MWQuickCmp(iCal->Organizer->Address, Session->EMailAddress)) {
		int ReplyInt;
		/* As organizer we want to add the new recipient as an attendee and myself as a new attendee and update our status as appropriate. */
		/* update my attendee status to DELEGATED, I don't need to do this if I'm not organizer because item will be deleted  */
		ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_DELEGATED, Session->EMailAddress);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);
		ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_NEED_ACT, DelegateesEmail);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);
		
	}

	/* We especially saved this so we could update attendee status if we were organizer */
	if (DelegateesEmail) {
		MemFree(DelegateesEmail);
	}

	return(TRUE);
}

BOOL
ICalMailReply(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long Action)
{
	FILE		*FileHandle=NULL;
	unsigned char	*OriginalMessageString="-----Original Message-----";
	unsigned char	*FromString="From";
//JNT	unsigned char	*ToString="To";
//JNT	unsigned char	*DateString="Date";
	unsigned char	*SubjectString="Subject";
	ICalObject		*ICal;

	ICal = CalendarSession->ICal;

	if (!ICal) {
		return(FALSE);
	}

	if (Action == AA_REPLY) {
		snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
		FileHandle=fopen(Client->Temp, "wb");
		if (FileHandle && ICal->Organizer && ICal->Organizer->Address) {
			fprintf(FileHandle, "%s\r\n", ICal->Organizer->Address);
			fclose(FileHandle);
		}
	} else if (Action == AA_REPLYALL) {
		unsigned long		i;
		ICalVAttendee			*Attendee;

		for (i = 0; i < 3; i++) {
			Attendee = ICal->Attendee;

			snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
			FileHandle=fopen(Client->Temp, "wb");
			if (FileHandle) {
				if (COMPOSE_TYPE_LIST[i] == AA_REQUIRED && ICal->Organizer && ICal->Organizer->Address) {
					/* Add the organizer */
					fprintf(FileHandle, "%s\r\n", ICal->Organizer->Address);
				}

				while (Attendee) {
					if (Attendee->Role == COMPOSE_TYPE_LIST[i]) {
						fprintf(FileHandle, "%s\r\n", Attendee->Address);
					}
					Attendee = Attendee->Next;
				}
				fclose(FileHandle);
			}
		}
	}

	snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_SUBJECT, MwCal.WorkDir, (unsigned int)Session->SessionID);
	FileHandle=fopen(Client->Temp, "wb");
	if (FileHandle && ICal->Summary) {
		fwrite("Re: ", 4, 1, FileHandle);
		WriteWithBreaks(ICal->Summary, strlen(ICal->Summary), FileHandle);
	}
	if (FileHandle) {
		fclose(FileHandle);
	}

	snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_BODY, MwCal.WorkDir, (unsigned int)Session->SessionID);
	FileHandle=fopen(Client->Temp, "wb");
	if (FileHandle && ICal->Description) {
		unsigned long		StartTime;

		fprintf(FileHandle, "\r\n\r\n\r\n%s\r\n", OriginalMessageString);

		/* From */
		if (ICal->Organizer && ICal->Organizer->Address) {
			if (ICal->Organizer->Name) {
				fprintf(FileHandle, "%s: %s <%s>\r\n", FromString, ICal->Organizer->Name, ICal->Organizer->Address);
			} else {
				fprintf(FileHandle, "%s: <%s>\r\n", FromString, ICal->Organizer->Address);
			}
		}

		/* Date */
		StartTime = MsgGetUTC(ICal->Start.Day, ICal->Start.Month, ICal->Start.Year, ICal->Start.Hour, ICal->Start.Minute, ICal->Start.Second);
		if (StartTime == (unsigned long)-1) {
			StartTime = MsgGetUTC(ICal->Stamp.Day, ICal->Stamp.Month, ICal->Stamp.Year, ICal->Stamp.Hour, ICal->Stamp.Minute, ICal->Stamp.Second);
		}

		MsgGetRFC822Date(MsgGetUTCOffsetByUTC(Session->TimezoneID, StartTime), StartTime, Client->Temp);
		fprintf(FileHandle, "Date: %s\r\n", Client->Temp);

		/* Subject */
		if (ICal->Summary) {
			fprintf(FileHandle, "%s: ", SubjectString);
			WriteWithBreaks(ICal->Summary, strlen(ICal->Summary), FileHandle);
		}

		/* Body */
		fprintf(FileHandle, "\r\n\r\n");
		if (ICal->Description) {
			WriteWithBreaks(ICal->Description, strlen(ICal->Description), FileHandle);
		}

	}
	if (FileHandle) {
		fclose(FileHandle);
	}

	return(TRUE);
}

static int 
StoreICalObjectForCalCompose(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long Action)
{
	FILE					*iCalFile;
	unsigned long		i;
	int ReplyInt;

	snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_ICAL, MwCal.WorkDir, (unsigned int)Session->SessionID);
	iCalFile = fopen(Client->Temp, "wb");
	if (!iCalFile) {
		return(FALSE);
	}

	/* Request the object */
	ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSLIST %lu\r\n", CalendarSession->IDList[CalendarSession->DetailID - 1]);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);

	ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
	if (ReplyInt == 2023) {
		i = atol(Client->Temp);

		if (Session->NBufferPtr > 0) {
			if (strlen(Session->NBuffer) > i) {
				/* We already have it in our buffer, so just write it */
				fwrite(Session->NBuffer, sizeof(unsigned char), i, iCalFile);

				Session->NBufferPtr = strlen(Session->NBuffer + i);
				memmove(Session->NBuffer, Session->NBuffer + i, Session->NBufferPtr + i);

				i = 0;
			} else {
				/* We have part of it in our buffer */
				fwrite(Session->NBuffer, sizeof(unsigned char), Session->NBufferPtr, iCalFile);

				i -= Session->NBufferPtr;
				Session->NBufferPtr = 0;
				Session->NBuffer[0] = '\0';
			}
		}

		while (i > 0) {
			unsigned long		len;

			if (i < sizeof(Client->Temp)) {
				len = MWDirectNMAPRead(Session, Client->Temp, i);
			} else {
				len = MWDirectNMAPRead(Session, Client->Temp, sizeof(Client->Temp));
			}

			if (len > 0) {
				fwrite(Client->Temp, sizeof(unsigned char), len, iCalFile);
				i -= len;
			} else {
				ModDebug("Lost NMAP connection\n");
				if (iCalFile) {
					fclose(iCalFile);
				}
				return(FALSE);
			}
		}
	} else if (ReplyInt != 1000) {
		if (iCalFile) {
			fclose(iCalFile);
		}
		return(FALSE);
	}
	fclose(iCalFile);

	/* Grab confirmation of CSLIST */
	if (MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE) == 1000) {
		if (CalendarSession->ICal) {
			if (Action != AA_DELEGATE) {
				/* Build To: file.  Luckily we have the iCal object cached right now. */
				snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
				iCalFile = fopen(Client->Temp, "ab");
				if (iCalFile) {
					if (CalendarSession->ICal && CalendarSession->ICal->Organizer && CalendarSession->ICal->Organizer->Address) {
						fprintf(iCalFile, "%s\r\n", CalendarSession->ICal->Organizer->Address);
					}
					fclose(iCalFile);
				}
			}

			/* Create the subject line */
			snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_SUBJECT, MwCal.WorkDir, (unsigned int)Session->SessionID);
			iCalFile = fopen(Client->Temp, "wb");
			if (iCalFile) {
				if (Action == AA_DECLINECOMMENT) {
					fprintf(iCalFile, "Decline: %s", CalendarSession->ICal->Summary);
				} else if (Action == AA_DELEGATE) {
					fprintf(iCalFile, "Delegated: %s", CalendarSession->ICal->Summary);
				} else if (Action == AA_COMPLETECOMMENT) {
					fprintf(iCalFile, "Completed: %s", CalendarSession->ICal->Summary);
				}
				fclose(iCalFile);
			}
		}

		/* used if we need to delete the message later */
		/* cheaper to just set them then to check if we are originator etc. */
		Session->DirtyMessageID = CalendarSession->DetailID;
		Session->DirtyMessageFolder = Session->CurrentCalendar;

	} else {
		/* We failed to get the iCal object, so lets delete the file */
		snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_ICAL, MwCal.WorkDir, (unsigned int)Session->SessionID);
		unlink(Client->Temp);
	}

	return(TRUE);

}


static BOOL
ICalAction(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long Action)
{
	unsigned long		ReplyInt = 0;
	ICalObject *iCal = CalendarSession->ICal;

	if (Session->ReadOnly) {
	  return(FALSE);
	}

	/* Send ICal Reply to originator and update our own calendar according to state */
	switch (Action){
		case AA_ACCEPT:{
			SendICalReply(Client, Session, CalendarSession, Action);
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_ACCEPTED, Session->EMailAddress);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

			if (iCal->Type == ICAL_VTODO) {
				/* If we are task then clear the completed flag */
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDFLG %lu %d\r\n", 
										 CalendarSession->IDList[CalendarSession->DetailID - 1], MSG_STATE_COMPLETED);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
				if (ReplyInt == 1000) {
					/* clear the flag */
					CalendarSession->EntryState &= ~MSG_STATE_COMPLETED;
				}
			}

			break;
		}
		case AA_DECLINE:{
			SendICalReply(Client, Session, CalendarSession, Action);

			/* if there is no ical object or organizer or no recipients or if I am the organizer and there are not any others recipients then delete from calendar */
			if (!iCal || !iCal->Organizer || !MWQuickNCmp(iCal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress)) ||
				 !iCal->Attendee  || (!iCal->Attendee->Next && MWQuickNCmp(iCal->Organizer->Address, iCal->Attendee->Address, strlen(iCal->Organizer->Address)))) {
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu%s\r\n", CalendarSession->IDList[CalendarSession->DetailID - 1], (Action == AA_DELETEALL) ? " RECURRING" : "");
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);

				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
				MWSendNMAPServer(Session, "CSPURG\r\n", 9);
				ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
			
			} else {
				/* otherwise update attendee status to declined */
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp),"CSATND %lu %c %s\r\n",CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_DECLINED, Session->EMailAddress);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);

				if (iCal->Type == ICAL_VTODO) {
					/* If we are task then clear the completed flag */
					ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDFLG %lu %d\r\n", 
											 CalendarSession->IDList[CalendarSession->DetailID - 1], MSG_STATE_COMPLETED);
					MWSendNMAPServer(Session, Client->Temp, ReplyInt);
					/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
					ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
					if (ReplyInt == 1000) {
						/* clear the flag */
						CalendarSession->EntryState &= ~MSG_STATE_COMPLETED;
					}
				}

			}

			break;
		}
		case AA_DECLINECOMMENT:{
			if (iCal && iCal->Organizer && MWQuickNCmp(iCal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress))) {
				ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_DECLINED, Session->EMailAddress);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);
			}

			/* if there is an ical object and organizer and I am not the organizer then fall through and send decline  */
			/* otherwise treat this like a normal decline. The organizer does not need to talk to himself. */
				/* Store the iCal object for decline compose */
			return(StoreICalObjectForCalCompose(Client,Session,CalendarSession,Action));

		}
		case AA_COMPLETECOMMENT:{
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_COMPLETED, Session->EMailAddress);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);

			/* Store the iCal object for decline compose if needed */

			/* if there is an ical object and organizer and I am not the organizer then fall through and send decline  */
			/* otherwise treat this like a normal decline. Talking to yourself is too weird. */
			if ( iCal && iCal->Organizer && !MWQuickNCmp(iCal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress))) {
				/* Store the iCal object for decline compose */
				return(StoreICalObjectForCalCompose(Client,Session,CalendarSession,Action));
			} else {
				/* I am the organizer don't talk to myself and don't delete item just update attendee status to completed */
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp),"CSATND %lu C %s\r\n",CalendarSession->IDList[CalendarSession->DetailID-1], Session->EMailAddress);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);

				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSCOMP %lu %lu\r\n", CalendarSession->IDList[CalendarSession->DetailID - 1], time(NULL));
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

				/* now set the completed flag on the object */
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSAFLG %lu %d\r\n", 
										 CalendarSession->IDList[CalendarSession->DetailID - 1], MSG_STATE_COMPLETED);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);
				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
				if (ReplyInt == 1000) {
					/* set the flag */
					CalendarSession->EntryState |= MSG_STATE_COMPLETED;
				}
			}
			return(FALSE);			
		}
		case AA_COMPLETE:{
			SendICalReply(Client, Session, CalendarSession, Action);
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_COMPLETED, Session->EMailAddress);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSCOMP %lu %lu\r\n", CalendarSession->IDList[CalendarSession->DetailID - 1], time(NULL));
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

			/* now set the completed flag on the object */
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSAFLG %lu %d\r\n", 
									 CalendarSession->IDList[CalendarSession->DetailID - 1], MSG_STATE_COMPLETED);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
			if (ReplyInt == 1000) {
				/* set the flag */
				CalendarSession->EntryState |= MSG_STATE_COMPLETED;
			}

			UpdateCalendar(Session, CalendarSession);
			break;
		}
		case AA_TENTATIVE:{
			SendICalReply(Client, Session, CalendarSession, Action);
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", CalendarSession->IDList[CalendarSession->DetailID-1], NMAP_CAL_STATE_TENTATIVE, Session->EMailAddress);
			break;
		}
		case AA_DELETE:{
			SendICalReply(Client, Session, CalendarSession, Action);
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu\r\n", CalendarSession->IDList[CalendarSession->DetailID - 1]);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			if ((ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE)) == 1000){
			  MWSendNMAPServer(Session, "CSPURG\r\n", 9);
			  if ((ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE)) == 1000) {
			    break;
			  }
			}
			
			CalendarSession->Error = ERROR_DELETING_MESSAGE__AFLG;
			return(FALSE);
		}
		case AA_DELETEALL:{
			SendICalReply(Client, Session, CalendarSession, Action);
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu RECURRING\r\n", CalendarSession->IDList[CalendarSession->DetailID - 1]);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			if ((ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE)) == 1000) {
			  MWSendNMAPServer(Session, "CSPURG\r\n", 9);
			  if ((ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE)) == 1000) {
			    break;
			  }
			}
			CalendarSession->Error = ERROR_DELETING_MESSAGE__AFLG;
			return(FALSE);
		}

		case AA_DELEGATE:{ 
			/* Store the iCal object for decline compose */
			return(StoreICalObjectForCalCompose(Client,Session,CalendarSession,Action));
		}
		case AA_SHOWBUSY:{
			/* user wants appointment to show up as busy, so clear the NOFREEBUSY flag*/
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDFLG %lu %d\r\n",
									 CalendarSession->IDList[CalendarSession->DetailID - 1], MSG_STATE_NOFREEBUSY);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
			if (ReplyInt == 1000){
				/* clear the flag */
				CalendarSession->EntryState &= ~MSG_STATE_NOFREEBUSY;
			}
			UpdateCalendar(Session, CalendarSession);
			break;
		}
		case AA_SHOWFREE: {
			/* user wants appointment to show up as free, so set the NOFREEBUSY flag*/
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSAFLG %lu %d\r\n", 
									 CalendarSession->IDList[CalendarSession->DetailID - 1], MSG_STATE_NOFREEBUSY);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

			if (ReplyInt == 1000) {
				/* set the flag */
				CalendarSession->EntryState |= MSG_STATE_NOFREEBUSY;
			}
			UpdateCalendar(Session, CalendarSession);
			break;
			
		}

		case AA_CALCOPY: 
		case AA_CALMOVE:{
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSCOPY %lu %s\r\n", 
									 CalendarSession->IDList[CalendarSession->DetailID - 1],
									 Session->FolderList->Value[CalendarSession->CurrentCalendar - 1] + 3);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
			break;
		}
		case AA_REPLY:
		case AA_REPLYALL:{
			/* Fill out compose files and open compose page if needed */
			ICalMailReply(Client, Session, CalendarSession, Action);
		}
		default: 
			break;
	}

	return(TRUE);
}

/* This function completes the calendar item stored in CalendarSession->ICal    */
/* It also sets the attendee status and completed flag. */
/* Since it may be called when calendar MAIN is not currently selection, MAIN */
/* is first selected and then a CSFIND command is issued to find the associated */
/* Entry. Only if the entry is found is it then completed */
static BOOL
CompleteICalMessageInCalendar(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession) 
{
	ICalObject *ICal;
	int currentCalendar;
	int ReplyInt;
	/* remember which calendar we had selected if any */
	currentCalendar = Session->CurrentCalendar;

	/* select Main */
	SelectCalendar(0, Session, CalendarSession);

	/* select calendar main */
	ICal = CalendarSession->ICal;

	/* find calender entry to complete */
	ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSFIND %lu %s\r\n", ICal->Sequence, ICal->UID);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

	if (ReplyInt == 1000) {
		long unsigned int sequence = atol(Client->Temp);

		/* complete the calendar entry */
		ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", sequence, NMAP_CAL_STATE_COMPLETED, Session->EMailAddress);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);
		ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSCOMP %lu %lu\r\n", sequence, time(NULL));
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);

		/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
		ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

		/* now set the completed flag on the object */
		ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSAFLG %lu %d\r\n", sequence, MSG_STATE_COMPLETED);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

		if (ReplyInt == 1000) {
			/* set the flag */
			CalendarSession->EntryState |= MSG_STATE_COMPLETED;
		}
		
		/* restore original calendar */
		CalendarSession->CurrentCalendar = currentCalendar;
		SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);
		return(TRUE);
	}

	/* restore original calendar */
	CalendarSession->CurrentCalendar = currentCalendar;
	SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);
	return(FALSE);
}

/* This function sets the attendee status and clears the completed flag. */
/* Since it may be called when calendar MAIN is not currently selection, MAIN */
/* is first selected and then a CSFIND command is issued to find the associated */
/* Entry. Only if the entry is found is it then declined */
static BOOL
DeclineICalMessageInCalendar(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession) 
{
	ICalObject *ICal;
	int currentCalendar;
	int ReplyInt;
	/* remember which calendar we had selected if any */
	currentCalendar = Session->CurrentCalendar;

	/* select Main */
	SelectCalendar(0, Session, CalendarSession);

	/* select calendar main */
	ICal = CalendarSession->ICal;

	/* find calender entry to complete */
	ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSFIND %lu %s\r\n", ICal->Sequence, ICal->UID);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

	if (ReplyInt == 1000) {
		long unsigned int sequence = atol(Client->Temp);

		/* complete the calendar entry */
		ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu %c %s\r\n", sequence, NMAP_CAL_STATE_DECLINED, Session->EMailAddress);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);

		/* now clear the completed flag on the object */
		ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDFLG %lu %d\r\n", sequence, MSG_STATE_COMPLETED);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

		if (ReplyInt == 1000) {
			/* set the flag */
			CalendarSession->EntryState &= ~MSG_STATE_COMPLETED;
		}
		
		/* restore original calendar */
		CalendarSession->CurrentCalendar = currentCalendar;
		SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);
		return(TRUE);
	}

	/* restore original calendar */
	CalendarSession->CurrentCalendar = currentCalendar;
	SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);
	return(FALSE);
}


/* This function deletes the calendar item stored in CalendarSession->ICal    */
/* Since it may be called when calendar MAIN is not currently selection, MAIN */
/* is first selected and then a CSFIND command is issued to find the associated */
/* Entry. Only if the entry is found is it then deleted */
static BOOL
DeleteICalMessageFromCalendar(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession) 
{
	ICalObject *ICal;
	int currentCalendar;
	int ReplyInt;
	/* remember which calendar we had selected if any */
	currentCalendar = Session->CurrentCalendar;

	/* select Main */
	SelectCalendar(0, Session, CalendarSession);

	/* select calendar main */
	ICal = CalendarSession->ICal;

	/* find calender entry to delete */
	ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSFIND %lu %s\r\n", ICal->Sequence, ICal->UID);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

	if (ReplyInt == 1000) {
		long unsigned int sequence = atol(Client->Temp);
		/* delete calendar entry */
		ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu\r\n", sequence);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);
		/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
		ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

		MWSendNMAPServer(Session, "CSPURG\r\n", 9);
		ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
		
		/* restore original calendar */
		CalendarSession->CurrentCalendar = currentCalendar;
		SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);
		return(TRUE);
	}

	/* restore original calendar */
	CalendarSession->CurrentCalendar = currentCalendar;
	SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);
	return(FALSE);

		

}




static BOOL
ProcessCalComposeForm(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long *CurrentPage, unsigned long SendPage)
{
	FILE				*FileHandle = NULL;
	unsigned char	Path[XPL_MAX_PATH+1];
	unsigned long	ValueSize;
	unsigned char	FieldName[128];
	unsigned long	Offset = 0;
	unsigned long	RetVal;
	unsigned char	*ptr;
	unsigned char	*StartPtr;
	unsigned char	*EndPtr;
	unsigned long	DefaultPage;
	unsigned long	Action = 0;
	unsigned long	FileSize;
	BOOL				SendComment = FALSE;
	unsigned long	BeginMinutes = 0;
	unsigned long	BeginDay = 0;
	unsigned long	BeginMonth = 0;
	unsigned long	BeginYear = 0;
	unsigned long	EndMinutes = 0;
	unsigned long	EndDay = 0;
	unsigned long	EndMonth = 0;
	unsigned long	EndYear = 0;
	unsigned long	Duration = 0;
	const char *mode;
	
	DefaultPage = *CurrentPage;

	MWProtectNMAP(Client, TRUE);

	/* Make sure the TO file is clean, as we'll be appending to it */
	snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
	unlink(Path);

	while (MWGetFormName(Client, FieldName, sizeof(FieldName))) {
	        mode = "wb";
		ModDebug("Got Name:%s\n", FieldName);
		ValueSize=BUFSIZE-Offset;
		FileSize=0;

		while ((RetVal=MWGetFormValue(Client, Client->Temp+Offset, &ValueSize))!=FORMFIELD_NEXT) {
			ValueSize+=Offset;
			switch (toupper(FieldName[0])) {
				/* Address fields & Load Page */
				case 'A': {
					switch (toupper(FieldName[4])) {
						/* actiOnloadpage */
						case 'O': {
							unsigned long i;
							long retVal;

							i=strlen(FieldName);
							if (FieldName[i-2]=='.') {
								if (toupper(FieldName[i-1])=='X' || toupper(FieldName[i-1])=='Y') {
									FieldName[i-2]='\0';
								}
							}
							retVal = MWFindTemplatePage(FieldName+14, Session->TemplateID);
							if (retVal >= 0) {
							    *CurrentPage = (unsigned long)retVal;
							} else {
							    *CurrentPage = DefaultPage;
							}

							Action = ACTION_CANCEL;
							goto SkipWriteRecipients;
							break;
						}

                                                /* addrSelf */    
					        case 'S':
						/* addrTo */
						case 'T': {
						    mode = "ab";
							if (!FileHandle) {
								snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
							}
							break;
						}

						/* addrCc */
						case 'C': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
							}
							break;
						}

						/* addrBcc */
						case 'B': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
							}
							break;
						} 

						default: {
							/* Unknown form element; Ignore it */
							goto SkipWriteRecipients;
							break;
						}
					}

					if (!FileHandle) {
						FileHandle=fopen(Path, mode);
						if (!FileHandle) {
							return(FALSE);
						}
					}

					ptr=Client->Temp;
					StartPtr=Client->Temp;
					EndPtr=Client->Temp+ValueSize;

					while (ptr<EndPtr) {
						switch (*ptr) {
							case ',':
							case ';':
							case ' ': {
								fwrite(StartPtr, ptr-StartPtr, 1, FileHandle);

								/* validate email address is RFC compliant */
								if (MsgIsAddress(StartPtr,ptr-StartPtr, ",; ", NULL)){
									;
								} else {
									if (FieldName[4] == 'T'){
										CalendarSession->Error = ERROR_BAD_TO_ADDR;
									}else if (FieldName[4] == 'C') {
										CalendarSession->Error = ERROR_BAD_CC_ADDR;
									}
									else if (FieldName[4] == 'B') {
										CalendarSession->Error = ERROR_BAD_BCC_ADDR;
									}
								}

								fwrite("\r\n", 2, 1, FileHandle);
								while (ptr<EndPtr && !isalnum(*ptr)) {
									ptr++;
								}
								StartPtr=ptr;
								break;
							}

							default: {
								ptr++;
								break;
							}
						}
					}

					if (ptr-StartPtr) {
						if (RetVal==FORMFIELD_DONE) {
							fwrite(StartPtr, ptr-StartPtr, 1, FileHandle);

							/* validate email address is RFC compliant */
							if (MsgIsAddress(StartPtr,ptr-StartPtr, ",; ", NULL)){
								;
							} else {
								if (FieldName[4] == 'T'){
									CalendarSession->Error = ERROR_BAD_TO_ADDR;
								}else if (FieldName[4] == 'C') {
									CalendarSession->Error = ERROR_BAD_CC_ADDR;
								}
								else if (FieldName[4] == 'B') {
									CalendarSession->Error = ERROR_BAD_BCC_ADDR;
								}
							}

							fwrite("\r\n", 2, 1, FileHandle);
							Offset=0;
						} else {
							/*
								An address cannot be larger than MAXEMAILNAMESIZE, if we get something bigger it's some 
								!%@#%&* is trying to break us. We ignore data longer than that.
							*/
							if (strlen(StartPtr)<MAXEMAILNAMESIZE) {
								Offset=ptr-StartPtr;
								memmove(Client->Temp, StartPtr, Offset);
							} else {
								Offset=0;
							}
						}
					}

SkipWriteRecipients:

					break;
				}

				/* Date fields */
				case 'D': {
					switch (toupper(FieldName[1])) {
						/* Date */
						case 'A': {
							switch (toupper(FieldName[4])) {
								/* DateDy */
								case 'D': {
									if (FieldName[6] == 'E') {
										EndDay = atol(Client->Temp);
									} else {
										BeginDay = atol(Client->Temp);
									}

									break;
								}

								/* DateMn */
								case 'M': {
									if (FieldName[6] == 'E') {
										EndMonth = atol(Client->Temp);
									} else {
										BeginMonth = atol(Client->Temp);
									}
									break;
								}

								/* DateYr */
								case 'Y': {
									if (FieldName[6] == 'E') {
										EndYear = atol(Client->Temp);
									} else {
										BeginYear = atol(Client->Temp);
									}
									break;
								}

								/* DateTm */
								case 'T': {
									if (FieldName[6] == 'E') {
										EndMinutes = atol(Client->Temp);
									} else {
										BeginMinutes = atol(Client->Temp);
									}
									break;
								}

								default: {

									break;
								}
							}
							break;
						}
						/* Duration */
						case 'U': {
							switch (toupper(FieldName[8])) {
								/* DurationHours */
								case 'H': {
									Duration += atol(Client->Temp);
									break;
								}
								/* DurationDays */
								case 'D': {
									Duration += atol(Client->Temp) * 24 * 60;
									break;
								}
							}
							break;
						}
					}
					break;
				}

				/* Send */
				case 'S': {
					switch (toupper(FieldName[4])) {
						/* SendAppointment */
						case 'A': {
							Action = ACTION_APPOINTMENT;
							*CurrentPage = SendPage;
							break;
						}

						/* SendNote */
						case 'N': {
							Action = ACTION_NOTE;
							*CurrentPage = SendPage;
							break;
						}

						/* SendTask */
						case 'T': {
							Action = ACTION_TASK;
							*CurrentPage = SendPage;
							break;
						}

						/* SendDecline or SendDelegate */
						case 'D': {
							switch (toupper(FieldName[6])) {
								/* SendDeCline */
								case 'C': {
									SendComment = TRUE;

									Action = ACTION_DECLINE;
									*CurrentPage = SendPage;
									break;
								}

								/* SendDeLegate */
								case 'L': {
									Action = ACTION_DELEGATE;
									*CurrentPage = SendPage;
									break;
								}
							}
							break;
						}

						/* SendComplete */
						case 'C': {
							SendComment = TRUE;

							Action = ACTION_COMPLETE;
							*CurrentPage = SendPage;
							break;
						}

						default: {

							break;
						}
					}
					break;
				}

				/* Text Fields */
				case 'T': {
					switch (toupper(FieldName[4])) {
						/* TextBody */
						case 'B': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_BODY, MwCal.WorkDir, (unsigned int)Session->SessionID);
								FileHandle=fopen(Path,"wb");
								if (!FileHandle) {
									return(FALSE);
								}
							}
							fwrite(Client->Temp, ValueSize, 1, FileHandle);
							if (RetVal==FORMFIELD_DONE) {
								fwrite("\r\n", 2, 1, FileHandle);
							}
							break;
						}

						/* TextSubject */
						case 'S': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_SUBJECT, MwCal.WorkDir, (unsigned int)Session->SessionID);
								FileHandle=fopen(Path,"wb");
								if (!FileHandle) {
									return(FALSE);
								}
							}
							fwrite(Client->Temp, ValueSize, 1, FileHandle);
							if (RetVal==FORMFIELD_DONE) {
								fwrite("\r\n", 2, 1, FileHandle);
							}
							break;
						}

						/* TextLocation */
						case 'L': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path), "%s/%x."COMPOSE_EXT_LOCATION, MwCal.WorkDir, (unsigned int)Session->SessionID);
								FileHandle=fopen(Path,"wb");
								if (!FileHandle) {
									return(FALSE);
								}
							}
							fwrite(Client->Temp, ValueSize, 1, FileHandle);
							if (RetVal==FORMFIELD_DONE) {
								fwrite("\r\n", 2, 1, FileHandle);
							}
							break;
						}

						/* TextPriority */
						case 'P': {

							break;
						}		 

						default: {

							break;
						}
					}
					break;	
				}

				default: {
					break;
				} 
			}
			ValueSize = BUFSIZE - Offset;
		}
		Offset = 0;

		if (FileHandle) {
			fclose(FileHandle);
			FileHandle = NULL;
		}
	}

	/* If there was an error detected when reading in the compose form return here */
	if (CalendarSession->Error) {
		*CurrentPage = DefaultPage;
		return(FALSE);
	}

	MWProtectNMAP(Client, FALSE);

	if (BeginDay) {
		CalendarSession->BeginTime = MsgGetUTC(BeginDay, BeginMonth, BeginYear, BeginMinutes/60, BeginMinutes%60, 0);
	}

	if (Duration != 0) {
		CalendarSession->EndTime = CalendarSession->BeginTime + (Duration * 60);
	} else if (EndDay) {
		CalendarSession->EndTime = MsgGetUTC(EndDay, EndMonth, EndYear, EndMinutes/60, EndMinutes%60, 0);
	}

	if (Action == ACTION_APPOINTMENT || Action == ACTION_TASK || Action == ACTION_NOTE) {
		if (!SendAppointment(Client, Session, CalendarSession, Action)) {
			*CurrentPage = DefaultPage;
			return(FALSE);
		}
	} else if (SendComment && (Action == ACTION_DECLINE || Action == ACTION_COMPLETE)) {
		FILE						*iCalFile;
		struct stat				fileInfo;
		ICalObject				*iCal;

		iCal = CalendarSession->ICal;

		/* We must load the iCal object into Session->ICal before calling SendICalReply */
		snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_ICAL, MwCal.WorkDir, (unsigned int)Session->SessionID);
		if (stat(Client->Temp, &fileInfo) != -1) {

			iCalFile = fopen(Client->Temp, "rb");
			if (iCalFile) {

				CalendarSession->ICal = ICalParseObject(iCalFile, NULL, fileInfo.st_size);

				/* Close it right away */
				fclose(iCalFile);

				if (CalendarSession->ICal) {
					ICalObject *iCal2 = CalendarSession->ICal;

					/* remove from our calendar if there is no organizer or if we are not the organizer or if */
					/* there are not recipients or we are the only recipient for message */
					if (!iCal2->Organizer){

						/* This function finds and deletes the calendar message stored in CalendarSession->ICal according to UID */
						DeleteICalMessageFromCalendar(Client, Session, CalendarSession);

					}

					/* send a decline response if not the organizer */
					if (iCal2->Organizer) { 
						/* Send ICal Reply to update the organizers state */
						/* Set the dirty message bit to allow message to be deleted */
						switch (Action) {
							default:
							case ACTION_DECLINE: {
								/* remove from our calendar if there is no organizer or if we are not the organizer or if */
								/* there are not recipients or we are the only recipient for message */
								if (!MWQuickNCmp(iCal2->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress)) ||
									 !iCal2->Attendee  || (!iCal2->Attendee->Next && MWQuickNCmp(iCal2->Organizer->Address, iCal2->Attendee->Address, strlen(iCal2->Organizer->Address)))) {

									/* This function finds and deletes the calendar message stored in CalendarSession->ICal according to UID */
									DeleteICalMessageFromCalendar(Client, Session, CalendarSession);
								} else {
									DeclineICalMessageInCalendar(Client, Session, CalendarSession);
								}
								SendICalReply(Client, Session, CalendarSession, AA_DECLINE);
								break;
							}

							case ACTION_COMPLETE: {
								/* Complete, set the attendee status, and set complete flag in our own calendar */
								CompleteICalMessageInCalendar(Client, Session, CalendarSession);
								SendICalReply(Client, Session, CalendarSession, AA_COMPLETE);
								break;
							}
						}
						if (!MWQuickNCmp(iCal2->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress))) { 									
							/* there is an organizer and we are not it, send a text message telling the organizer why we declined or completed it. */
							SendTextMessage(Client, Session, CalendarSession);
						}
					}
 
					/* Set the dirty message bit so that upper layer will delete message indicating we have dealt with it */
					Session->DirtyMessage = TRUE;
					if (CalendarSession->ICal) {
						/* if DeleteICalMessageFromCalendar() is called this object has already been freed */
						ICalFreeObjects(CalendarSession->ICal);
					}
				}

				CalendarSession->ICal = iCal;
			}
		}

	} else if (Action == ACTION_DELEGATE) {
		FILE						*iCalFile;
		struct stat				fileInfo;
		ICalObject				*iCal;

		iCal = CalendarSession->ICal;

		/* Load the iCal object into Session-ICal before calling SendICalDelegation */
		snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_ICAL, MwCal.WorkDir, (unsigned int)Session->SessionID);
		if (stat(Client->Temp, &fileInfo) != -1) {

			iCalFile = fopen(Client->Temp, "rb");
			if (iCalFile) {
				CalendarSession->ICal = ICalParseObject(iCalFile, NULL, fileInfo.st_size);

				fclose(iCalFile);

				if (CalendarSession->ICal) {
					if (SendICalDelegation(Client, Session, CalendarSession) == TRUE) {
						
						/* We want to only delete the message if we are  not the originator */
						if (CalendarSession->ICal->Organizer && !MWQuickNCmp(CalendarSession->ICal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress))) { 									

							Session->DirtyMessage = TRUE;
						}


						/* send reply to update organizer with new participant. */
						/* We shouldn't have to send this to the organizer but since CSATND command does not allow us to */
						/* set delegatedto and delegatedfrom fields etc. we will have problems if we don't */
						SendICalReply(Client, Session, CalendarSession, AA_DELEGATE);

					} else {
						*CurrentPage = DefaultPage;
						return(FALSE);
					}
					
					ICalFreeObjects(CalendarSession->ICal);
				}

				CalendarSession->ICal = iCal;
			}
		}

		CalComposeCleanUp(Session, CalendarSession);
	}

	return(TRUE);
}

static BOOL
ProcessRecurForm(ConnectionStruct *Client, SessionStruct *Session)
{
	unsigned char	FieldName[128];
	unsigned long	ValueSize;		
	unsigned long	i;
	unsigned	char	*ptr;
	FILE				*RRuleFile;

	unsigned long	EndType = NOEND;		/* End date type */
	unsigned long	EndDay = 1;				/* End date information - Required if EndType == DATEEND */
	unsigned long	EndMonth = 1;
	unsigned long	EndYear = 1;
	unsigned long	EndHour = 0;
	unsigned long	EndMinute = 0;
	unsigned long	Count = 1;				/* Count - Required if EndType == COUNTEND */

	unsigned char	*FreqTypes[] = {
		"DAILY",
		"WEEKLY",
		"MONTHLY",
		"YEARLY"
	};

	unsigned long	Freq = DAILY;			/* Rule Frequency */
	unsigned long	Interval = 1;			/* Rule Interval */
	unsigned long	Type = 0;				/* Used to distinguish between different 'types' with the same frequency */
	signed long		Position = 0;			/* Sets first, second, last, etc.. */

	BOOL				SetDays = FALSE;
	unsigned char	Days[25];				/* List of specific days, Example: "TU,TH,SA" */
	unsigned long	ByMonthDay = 0;		/* Specific day in a month */
	unsigned long	ByMonth = 0;			/* Month for Yearly rules */

	Days[0] = '\0';

	while (MWGetFormName(Client, FieldName, 128)) {
		ModDebug("Got Name:%s\n", FieldName);
		ValueSize=BUFSIZE;
		while ((i=MWGetFormValue(Client, Client->Temp, &ValueSize))!=FORMFIELD_NEXT) {
			switch (toupper(FieldName[0])) {
				case 'F': {	/* FREQ */
					if (MWQuickNCmp(FieldName, "FREQ", 4)) {
						switch (toupper(Client->Temp[0])) {
							case 'D': { /* Daily */
								Freq = DAILY;
								break;
							}
							case 'W': { /* Weekly */
								Freq = WEEKLY;
								break;
							}
							case 'M': { /* Monthly */
								Freq = MONTHLY;
								break;
							}
							case 'Y': { /* Yearly */
								Freq = YEARLY;
								break;
							}
						}
					}
					break;
				}
				case 'T': {
					if (MWQuickNCmp(FieldName, "Type", 4)) {
						Type = atol(Client->Temp);
					} else if (MWQuickNCmp(FieldName, "Time", 5)) {		/* End Time */
						unsigned long	Time;

						Time = atol(Client->Temp);
						if (Time > 0) {
							EndHour = Time/60;
							EndMinute = Time - ((Time/60)*60);
						}
					}
					break;
				}
				case 'E': {
					if (MWQuickNCmp(FieldName, "EndType", 7)) {
						switch (toupper(Client->Temp[0])) {
							case 'N': { /* NONE */
								EndType = NOEND;
								break;
							}
							case 'C': { /* Count */
								EndType = COUNTEND;
								break;
							}
							case 'D': { /* Date */
								EndType = DATEEND;
								break;
							}
						}
					}
					break;
				}
				case 'C': {
					if (MWQuickNCmp(FieldName, "Count", 5)) {
						Count = atol(Client->Temp);
						if (Count == 0) {
							Count = 1;
						}
					}
					break;
				}
				case 'I': {
					if (MWQuickNCmp(FieldName, "Interval", 8)) {
						Interval = atol(Client->Temp);
						if (Interval == 0) {
							Interval = 1;
						}
					}
					break;
				}
				case 'D': {
					if (MWQuickNCmp(FieldName, "DayOfWeek", 9)) {
						unsigned long		Len;

						Len = strlen(Days);
						ptr = Days + Len;

						if (Len <= 21) {
							if (Len != 0) {
								*ptr = ',';
								ptr++;
							}
							*ptr = Client->Temp[0];
							ptr++;
							*ptr = Client->Temp[1];
							ptr++;
							*ptr = '\0';
						}
					} else if (MWQuickNCmp(FieldName, "Day", 4)) {	/* For End day */
						EndDay = atol(Client->Temp);
					}
					break;
				}
				case 'P': {
					if (MWQuickNCmp(FieldName, "Position", 8)) {
						Position = atol(Client->Temp);
					}
					break;
				}
				case 'M': {
					if (MWQuickNCmp(FieldName, "Month", 5)) {
						EndMonth = atol(Client->Temp);		/* For End Month */
						if (EndMonth > 12) {
							EndMonth = 0;
						}
						break;
					}
					break;
				}
				case 'Y': {
					if (MWQuickNCmp(FieldName, "Year", 5)) {		/* For End Year */
						EndYear = atol(Client->Temp);
					}
					break;
				}
				case 'B': {
					if (MWQuickNCmp(FieldName, "ByMonthDay", 10)) {
						ByMonthDay = atol(Client->Temp);
					} else if (MWQuickNCmp(FieldName, "ByMonth", 7)) {
						ByMonth = atol(Client->Temp);
						if (ByMonth > 12) {
							ByMonth = 0;
						}
					}
					break;
				}
			}
			ValueSize=BUFSIZE;
		}
	}

	if ((Freq == MONTHLY || Freq == YEARLY) && Type == 1) {
		SetDays = TRUE;
		ByMonthDay = 0;
	} else if (Freq == WEEKLY) {
		SetDays = TRUE;
	}

	/* Open file */
	snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_RRULE);
	RRuleFile = fopen(Client->Temp, "wb");

	if (!RRuleFile) {
		return(FALSE);
	} 

	/* Generate the actual rule string */
	fprintf(RRuleFile, "RRULE:FREQ=%s;INTERVAL=%d", FreqTypes[Freq], (int)Interval);			/* Frequency and Interval */
	if ((Days[0] != '\0') && SetDays) {																		/* BYDAY */
		fprintf(RRuleFile, ";BYDAY=");
		if (Position != 0 && Freq != YEARLY) {
			fprintf(RRuleFile, "%d", (int)Position);
		}
		fprintf(RRuleFile, Days);
	} else if (ByMonthDay != 0) {
		fprintf(RRuleFile, ";BYMONTHDAY=%d", (int)ByMonthDay);											/* BYMONTHDAY */
	}
	if (ByMonth != 0) {
		fprintf(RRuleFile, ";BYMONTH=%d", (int)ByMonth);
	}
	if (Position != 0 && Freq == YEARLY && Type == 1) {											/* BYSETPOS */
		fprintf(RRuleFile, ";BYSETPOS=%d", (int)Position);
	}
	if (EndType == COUNTEND && Count != 0) {															/* EndType */
		fprintf(RRuleFile, ";COUNT=%d", (int)Count);
	} else if (EndType == DATEEND && EndDay != 0 && EndMonth != 0 && EndYear != 0) {
		/* Until date has to be in UTC */
		
		MsgGetDMY(MsgGetUTC(EndDay, EndMonth, EndYear, EndHour, EndMinute, 0) - Session->TimezoneOffset,
			&EndDay, &EndMonth, &EndYear, &EndHour, &EndMinute, NULL);
		fprintf(RRuleFile, ";UNTIL=%04d%02d%02dT%02d%02d00Z", (int)EndYear, (int)EndMonth, (int)EndDay, (int)EndHour, (int)EndMinute);
	}
	fprintf(RRuleFile, "\r\n");

	/* 
		Figure out the begin date.  It must be the first day that the rule acctually occurs,
		so we have to date the date the user gave us and figure out when the first occurance of the rule
		after (or on) that date is.
	*/
	fclose(RRuleFile);

	return(TRUE);
}

static unsigned long
ProcessCalListForm(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned char	FieldName[128];
	unsigned long	ValueSize;
	unsigned long	i = 0;
	unsigned char	ID[500];
	unsigned long	Type = 0;
	unsigned long	retVal = 0;

	MDBFreeValues(CalendarSession->CopyValues);

	while (MWGetFormName(Client, FieldName, 128)) {
		ModDebug("Got Name:%s\n", FieldName);

		ValueSize = BUFSIZE;
		while ((MWGetFormValue(Client, Client->Temp, &ValueSize)) != FORMFIELD_NEXT) {
			switch (toupper(FieldName[0])) {
				case 'I': {	/* ItemID */
					if (MWQuickNCmp(FieldName, "ItemID", 6)) {
						ID[i] = atoi(Client->Temp);
						i++;
					}
					break;
				}

				case 'A': { /* Accept */
					Type = AA_ACCEPT;
					break;
				}

				case 'C': {
					if (toupper(FieldName[2]) == 'M') { /* Complete */
						Type = AA_COMPLETE;
					} else if (toupper(FieldName[2]) == 'P') { /* Copy */
						retVal = 1;
					}
					break;
				}

				case 'D': { /* Delete & Decline */
					if (MWQuickNCmp(FieldName, "DeleteAll", 9)) {
						Type = AA_DELETEALL;
					} else if (MWQuickNCmp(FieldName, "Delete", 6)) {
						Type = AA_DELETE;
					} else if (MWQuickNCmp(FieldName, "Decline", 7)) {
						Type = AA_DECLINE;
					}
					break;
				}

				case 'M': { /* Move */
					retVal = 2;
					break;
				}

				case 'R': { /* Reply & Reply ALl */
					if (MWQuickNCmp(FieldName, "ReplyAll", 8)) {
						Type = AA_REPLYALL;
					} else if (MWQuickNCmp(FieldName, "Reply", 5)) {
						Type = AA_REPLY;
					}
					break;
				}

				case 'T': { /* Tentative */
					Type = AA_TENTATIVE;
					break;
				}

			}
			ValueSize = BUFSIZE;
		}
	}

	while (i > 0) {
		i--;

		if (Type != 0) {
			/* Need to select the message first */
			if (LoadCalendarEntryDetails(ID[i], Session, CalendarSession, FALSE)) {
				ICalAction(Client, Session, CalendarSession, Type);
			}
		} else {
			/* Copy and Move */
			snprintf(FieldName, sizeof(FieldName), "%lu", (long unsigned int)ID[i]);
			MDBAddValue(FieldName, CalendarSession->CopyValues);
		}
	}
	UpdateCalendar(Session, CalendarSession);

	return(retVal);
}

static BOOL
ProcessCalendarMessageMove(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession, BOOL Move)
{
	unsigned long	ValueSize;
	unsigned char	FieldName[128];
	unsigned long	Offset = 0;
	unsigned long	i;
	unsigned long	ReplyInt;
	unsigned long	SelectedFolder = 0;
	unsigned long	ID;

	if ((CalendarSession->CopyValues->Used == 0) || (Session->ReadOnly)) {
		return(FALSE);
	}

	/* First, read which folder we're moving stuff to */
	while (MWGetFormName(Client, FieldName, 128)) {
		ValueSize = BUFSIZE - Offset;

		while (MWGetFormValue(Client, Client->Temp + Offset, &ValueSize) != FORMFIELD_NEXT) {
			ValueSize += Offset;
			if (MWQuickCmp("FolderName", FieldName)) {
				SelectedFolder = atol(Client->Temp);
			}
			ValueSize = BUFSIZE - Offset;
		}
		Offset = 0;
	}

	if (SelectedFolder == 0) {
		CalendarSession->Error = ERROR_NO_FOLDER_SELECTED;
		return(FALSE);
	}
	CalendarSession->Error = 0;

	if (SelectedFolder > Session->FolderList->Used) {
		CalendarSession->Error = ERROR_NO_FOLDER_SELECTED;
		return(FALSE);
	}

	for (i = 0; i < CalendarSession->CopyValues->Used; i++) {
		ID = CalendarSession->IDList[atol(CalendarSession->CopyValues->Value[i]) - 1];

		ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSCOPY %lu %s\r\n", ID, Session->FolderList->Value[SelectedFolder - 1] + 3);
		MWSendNMAPServer(Session, Client->Temp, ReplyInt);

		ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		if (ReplyInt == 1000 && Move) {
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu\r\n", ID);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);

			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
			if (ReplyInt != 1000) {
				CalendarSession->Error = ERROR_DELETING_MESSAGE__AFLG;
			}

			MWSendNMAPServer(Session, "CSPURG\r\n", 9);
			ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
		} else {
			if (ReplyInt != 1000) {
				CalendarSession->Error = ERROR_COPYING_MESSAGE__COPY;
			}
		}
	}
	MDBFreeValues(CalendarSession->CopyValues);

	UpdateCalendar(Session, CalendarSession);
	return(TRUE);
}

static BOOL
ProcessDateSelection(ConnectionStruct *Client, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned long	ValueSize;
	unsigned char	FieldName[128];
	BOOL				Changed=FALSE;

	while (MWGetFormName(Client, FieldName, 128)) {
		ValueSize=BUFSIZE;

		ModDebug("Got Name:%s\n", FieldName);
		MWGetFormValue(Client, Client->Temp, &ValueSize);
		if (ValueSize>0) {
			if (MWQuickCmp(FieldName, "Day")) {
				Session->CurrentDay=atol(Client->Temp);
				Changed=TRUE;
				ModDebug("Got value:%s\n", Client->Temp);
			} else if (MWQuickCmp(FieldName, "Month")) {
				Session->CurrentMonth=atol(Client->Temp);
				Changed=TRUE;
				ModDebug("Got value:%s\n", Client->Temp);
			} else if (MWQuickCmp(FieldName, "Year")) {
				Session->CurrentYear=atol(Client->Temp);
				Changed=TRUE;
				ModDebug("Got value:%s\n", Client->Temp);
			}
		}

		/* Clean out the rest; should not be anything, but just in case */
		while (MWGetFormValue(Client, Client->Temp, &ValueSize)) {
			ValueSize=BUFSIZE;
		}
	}

	if (Session->CurrentDay > (long)MsgDaysPerMonth[Session->CurrentMonth - 1]) {
		Session->CurrentDay = MsgDaysPerMonth[Session->CurrentMonth-1];
	}

	if (Changed) {
		Session->CurrentRataDie=MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);
	}
	return(TRUE);
}

/* print start date and or time */
static int  
mwCalSPrintDTStart(char *buf, int bufLen, char *dateFormat, ICalObject *ICal, SessionStruct *Session)
{
	unsigned long ReplyInt;
	unsigned long StartTime;
	int timeZoneOffset;

	StartTime = MsgGetUTC(ICal->Start.Day, ICal->Start.Month, ICal->Start.Year, ICal->Start.Hour, ICal->Start.Minute, ICal->Start.Second);
	if (StartTime!=(unsigned long)-1) {
		/* To get correct local time adjust ICalObject->Start to GMT then GMT to Session->TimezoneID */
		timeZoneOffset = ( MsgGetUTCOffsetByUTC(Session->TimezoneID, StartTime) - MsgGetUTCOffsetByUTC(ICal->Start.TimezoneID, StartTime));
	} else {
		StartTime = MsgGetUTC(ICal->Stamp.Day, ICal->Stamp.Month, ICal->Stamp.Year, ICal->Stamp.Hour, ICal->Stamp.Minute, ICal->Stamp.Second);
		/* To get correct local time adjust ICalObject->Stamp to GMT then GMT to Session->TimezoneID */
		timeZoneOffset = ( MsgGetUTCOffsetByUTC(Session->TimezoneID, StartTime) - MsgGetUTCOffsetByUTC(ICal->Stamp.TimezoneID, StartTime));
	}

	ReplyInt=MsgPrint(buf, bufLen, dateFormat, StartTime+timeZoneOffset, &Session->DateFormat);
	return(ReplyInt);

}

static int
mwCalSPrintDTStartViewCache(char *buf, int bufLen, char *dateFormat, ICalObject *ICal, CalendarSessionStruct *CalendarSession, SessionStruct *Session)
{
	unsigned long ReplyInt;
	int timeZoneOffset;

	if (ICal) {
		timeZoneOffset = (MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCStart) - 
							MsgGetUTCOffsetByUTC(ICal->Start.TimezoneID,CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCStart ));
	} else {
		timeZoneOffset = MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCStart);
	}
	
	ReplyInt=MsgPrint(buf, bufLen, dateFormat, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCStart + timeZoneOffset, &Session->DateFormat);
	return(ReplyInt);

}

static int
mwCalSPrintDTEndViewCache(char *buf, int bufLen, char *dateFormat, ICalObject *ICal, CalendarSessionStruct *CalendarSession, SessionStruct *Session)
{
	unsigned long ReplyInt;
	int timeZoneOffset; 

	if (!CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCDue) {
		if (ICal) {
			timeZoneOffset = (MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCEnd) -
									MsgGetUTCOffsetByUTC(ICal->End.TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCEnd));
		} else {
			timeZoneOffset = MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCEnd);
		}
		ReplyInt=MsgPrint(buf, bufLen, Session->DateFormatShort, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCEnd + timeZoneOffset, &Session->DateFormat);
	} else {
		if (ICal) {
			timeZoneOffset = (MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCDue) -
									MsgGetUTCOffsetByUTC(ICal->End.TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCDue));
		} else {
			timeZoneOffset = MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCDue);
		}
		
		ReplyInt=MsgPrint(buf, bufLen, Session->DateFormatShort, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCDue + timeZoneOffset, &Session->DateFormat);
	}

	return(ReplyInt);

}




static int
mwCalSPrintDTEnd(char *buf, int bufLen, char *dateFormat, ICalObject *ICal, SessionStruct *Session)
{
	unsigned long	ReplyInt;
	unsigned long	EndTime;
	int timeZoneOffset = 0;
	
	EndTime = MsgGetUTC(ICal->End.Day, ICal->End.Month, ICal->End.Year, ICal->End.Hour, ICal->End.Minute, ICal->End.Second);
	if (EndTime!=(unsigned long)-1) {
		/* To get correct local time adjust ICalObject->Start to GMT then GMT to Session->TimezoneID */
		timeZoneOffset = (MsgGetUTCOffsetByUTC(Session->TimezoneID, EndTime) - MsgGetUTCOffsetByUTC(ICal->End.TimezoneID, EndTime));
		
	} else {
		EndTime=MsgGetUTC(ICal->Start.Day, ICal->Start.Month, ICal->Start.Year, ICal->Start.Hour, ICal->Start.Minute, ICal->Start.Second);
		/* To get correct local time adjust ICalObject->Stamp to GMT then GMT to Session->TimezoneID */
		timeZoneOffset = (MsgGetUTCOffsetByUTC(Session->TimezoneID, EndTime) - MsgGetUTCOffsetByUTC(ICal->Start.TimezoneID, EndTime));
		
	}
	
	ReplyInt=MsgPrint(buf, bufLen, dateFormat, EndTime+timeZoneOffset, &Session->DateFormat);
	
	return(ReplyInt);
	
}

static BOOL
IsUserAttending(char *answer, SessionStruct *Session, char *User, long unsigned int seqNumber)
{
	char *attendeeEmail;
	int ReplyInt;
	int rc = FALSE;

	/* get the email address and full name of attendee */
	if (MsgGetEmailAddressAndFullName(User, (char **)&attendeeEmail, NULL, Session, MSG_GET_EMAIL) == TRUE){
		ReplyInt=snprintf(answer, BUFSIZE, "CSATND %lu\r\n", seqNumber);
		MWSendNMAPServer(Session,answer, ReplyInt);
		ReplyInt=MWGetNMAPAnswer(Session, answer, BUFSIZ, TRUE);	/* List comming up */
		if (ReplyInt==2002) {
			while ((ReplyInt=MWGetNMAPAnswer(Session, answer, BUFSIZ, TRUE)) == 2002) {
				unsigned char		*Address;
				unsigned char     *Role;

				/* Find the e-mail address */
				Address = strchr(answer + 8, ' ');
				if (Address) {
					*Address = '\0';	/* We don't need anything after the address */
				}

				Address = answer + 8;
				Role = answer + 2;

				if ((*Role != 'N') && (MWQuickCmp(attendeeEmail, Address) || MWQuickCmp(User, Address))){
					/* we can't bail because we need to drain connection */
					rc = TRUE;
				}

			}
		}

		if (attendeeEmail) {
			MemFree(attendeeEmail);
		}

	}					

	return(rc);

}




/* This function returns true if Session email address is attendee of ICal object. */
/* This function is intended to be used in conjunction with Validate Object */
/* It does not verify the existence of the attendee in the actual calendar object. */
static BOOL
AMNMAPRecipient(CalendarSessionStruct *CalendarSession, SessionStruct *Session, int viewCacheID, char *answer)
{

	BOOL rc = FALSE;

	if (CalendarSession){
	
		/* we need to make another NMAP connection here */
		/* so we can determine if this guy is really a recipient */
		int NMAPsOld = Session->NMAPs; /* save the socket a second time so that we can steal the structure */
		int NBufferPtrOld	= Session->NBufferPtr;
		char NBufferOld[BUFSIZE + 1]; /* save the nmap buffer too */
		int ReplyInt;

		/* If we don't have an ICal object then we are listing calendar objects */
		/* and we only care to grey out icon if we are not the organizer */
		if (CalendarSession->EntryOrganizer && MWQuickCmp(Session->User, CalendarSession->EntryOrganizer)) {
			return(TRUE);
		}
		
		/* save off the old buffer for below */
		memcpy(NBufferOld, Session->NBuffer, Session->NBufferPtr);


		/* set up a new connection to use to validate user as attendee */
		/* Save off old connection info and establish new connection to NMAP server */
		Session->NMAPs = -1;
		Session->NBufferPtr = 0;

		if (!MWConnectUserToNMAPServer(Session)) {
			/* restore the first connection layer */
			IPclose(Session->NMAPs);
			Session->NMAPs = NMAPsOld;
			memcpy(Session->NBuffer, NBufferOld, NBufferPtrOld);
			Session->NBuffer[NBufferPtrOld] = 0;
			Session->NBufferPtr = NBufferPtrOld;
				
			return(FALSE);
		}

		ReplyInt = snprintf(answer, BUFSIZE, "CSOPEN MAIN\r\n");
		MWSendNMAPServer(Session, answer, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session, answer, BUFSIZE, TRUE);
		
		if ((ReplyInt != 1000) && (ReplyInt != 1020)) {
			/* Failed to open */
			/* we must bail */
			/* need to close new connection */
			IPclose(Session->NMAPs);
			Session->NMAPs = NMAPsOld;
			memcpy(Session->NBuffer, NBufferOld, NBufferPtrOld);
			Session->NBuffer[NBufferPtrOld] = 0;
			Session->NBufferPtr = NBufferPtrOld;
			return(FALSE);

		}

		/* Verify that this user is an attendee - he may be an organizer but not an attendee */
		rc = IsUserAttending(answer, Session, Session->User, CalendarSession->IDList[viewCacheID - 1]);

		/* need to swap socket and buffers back */
		IPclose(Session->NMAPs);
		Session->NMAPs = NMAPsOld;
		memcpy(Session->NBuffer, NBufferOld, NBufferPtrOld);
		Session->NBuffer[NBufferPtrOld] = 0;
		Session->NBufferPtr = NBufferPtrOld;

	}
		
	return(rc);
	

}

static BOOL
IsBCCAttendee(ICalObject *ICal, SessionStruct *Session) 
{
	ICalVAttendee *Attendee;
	
	if (ICal && ICal->Organizer) {
		
		for (Attendee = ICal->Attendee; Attendee; Attendee = Attendee->Next){
			if (Attendee->Role == ICAL_ROLE_NON_PARTICIPANT){
				return(TRUE);
			}
		}
	}

	return(FALSE);

}





/* This function returns true if Session email address is attendee of ICal object. */
/* This function is intended to be used in conjunction with Validate Object */
/* It does not verify the existence of the attendee in the actual calendar object. */
static BOOL
AMICalObjectAttendee(SessionStruct *Session, ICalObject *ICal)
{
	ICalVAttendee		*Attendee;

	/* If we have a good ICal object use it */
	/* otherwise try the view cache id */
	if (ICal){
		Attendee = ICal->Attendee;

		for (Attendee = ICal->Attendee; Attendee != NULL; Attendee = Attendee->Next) {
			if (MWQuickCmp(Session->EMailAddress, Attendee->Address)) {
				return(TRUE);
			}
		}
	} 
		
	return(FALSE);
	

}

static BOOL
AmBCCAttendee(ICalObject *ICal, SessionStruct *Session) 
{
	ICalVAttendee *Attendee;
	
	if (ICal && ICal->Organizer) {
		
		for (Attendee = ICal->Attendee; Attendee; Attendee = Attendee->Next){
			if ((MWQuickCmp(Session->EMailAddress, Attendee->Address)) && Attendee->Role == ICAL_ROLE_NON_PARTICIPANT){
				return(TRUE);
			}
		}
	}

	return(FALSE);

}



/* This function verifies that the calendar entry is in state acceptable to accept, decline or delegate from. */
/* First determines if this is a local or remote ICal object */
/* It determines whether the originator exists within the Hula system. */
/* If so it connects to the originators calendar and verifies the message exists, and has not been deleted. */
/* If the message looks bad then we send an error description to the client. */
/* Note: if the originator is outside our mail system then we allow it to pass. */
/* Note: the Session structure is borrowed because MWNMAP calls rely on the private NMAP structure within modwebd.  */
 
static BOOL
ValidateICalObject(ConnectionStruct *Client, SessionStruct *RealSession, ICalObject *ICal) 
{
	char organizer[256];
	int ReplyInt;
	char *ptr;
	struct sockaddr_in	saddr;
	MDBValueStruct		   *UserInfo;
	unsigned char		   UserDN[MDB_MAX_OBJECT_CHARS+1];

	int					   ClientNMAPs			= 0;
	unsigned char        *User;
	unsigned long		   NMAPCAddress		= 0;
	SessionStruct *Session = RealSession;
	unsigned int         NMAPBufferPtr;
	char Answer[BUFSIZE +1];

	unsigned char		   Buffer[BUFSIZE + 1];
	unsigned long		   RecurrenceID;
	unsigned long        EntryID;
	char *OrganizerEmailAddress = NULL;

	if (ICal && ICal->Organizer) {

		if (ICal->ProdID != ICAL_PROD_MODWEB) {
			/* this came from outside modwebd we must assume it is ok to allow accept to occur */
			return(TRUE);
		}

		HulaStrNCpy(organizer, ICal->Organizer->Address, 256);
		ptr = strchr(organizer,'@');

		if (ptr) {
			*ptr = '\0';
		}

		UserInfo = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);

		/* Find the user & get his full DN */
		if (!MsgFindObject(&organizer[0], UserDN, NULL, &saddr.sin_addr, UserInfo)) {
			/* Try once to see if this is a hosted domain of ours */
			/* ??? would it be less expensive to check SMTP and parent objects */
			if (!MsgFindObject(ICal->Organizer->Address, UserDN, NULL, &saddr.sin_addr, UserInfo)) {
				MDBDestroyValueStruct(UserInfo);
				/* This user may be from a remote system */
				return(TRUE);
			} else {
			  *ptr = '@';
			}
		} else {

			MDBFreeValues(UserInfo);

			/* validate email address */
			if (!(OrganizerEmailAddress = MsgGetUserEmailAddress(UserDN, UserInfo, Buffer, BUFSIZE + 1))|| 
				 !MWQuickCmp(OrganizerEmailAddress, ICal->Organizer->Address)) {
				MDBDestroyValueStruct(UserInfo);
				return(TRUE);
			}
		}

		MDBDestroyValueStruct(UserInfo);

		/* Ok, the organizer looks to be within our Email system */
 

		/*
		  Switch out our NMAP connection temporarily, and connect to the
		  NMAP store of the user we are searching against.
		  
		  We have to be very careful to reset our Client information when
		  we are done.
		*/

		/* Connect to NMAP server */
		ClientNMAPs = Session->NMAPs;
		Session->NMAPs = -1;

		/* save whats in the buffer */
		memcpy(Answer, Session->NBuffer, Session->NBufferPtr);
		NMAPBufferPtr = Session->NBufferPtr;
		Session->NBufferPtr = 0;
		NMAPCAddress = Session->NMAPAddress.s_addr;
		Session->NMAPAddress.s_addr = saddr.sin_addr.s_addr;
		User = Session->User;
		Session->User = &organizer[0];


		if (!MWConnectUserToNMAPServer(Session)) {
			Session->User = User;
			MWSendNMAPServer(Session, "QUIT\r\n", 6);
			MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
			IPclose(Session->NMAPs);
			/* "flush" our input buffer */
			Session->NBufferPtr=NMAPBufferPtr;
			memcpy(Session->NBuffer, Answer, Session->NBufferPtr);
			Session->NMAPs=ClientNMAPs;
			Session->NMAPAddress.s_addr=NMAPCAddress;
			return(FALSE);
		}

		Session->User = User;
		ReplyInt = snprintf(Buffer, sizeof(Buffer), "CSOPEN MAIN\r\n");
		MWSendNMAPServer(Session, Buffer, ReplyInt);
		ReplyInt = MWGetNMAPAnswer(Session, Buffer, sizeof(Buffer), TRUE);
		if ((ReplyInt != 1000) && (ReplyInt != 1020)) {
			/* Failed to open */
			MWSendNMAPServer(Session, "QUIT\r\n", 6);
			MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
			IPclose(Session->NMAPs);
			/* "flush" our input buffer */
			Session->NBufferPtr=NMAPBufferPtr;
			memcpy(Session->NBuffer, Answer, Session->NBufferPtr);
			Session->NMAPs=ClientNMAPs;
			Session->NMAPAddress.s_addr=NMAPCAddress;
			return(FALSE);
		}


		/* We've got the calendar selected, now search for the entry */
		RecurrenceID = MsgGetUTC(ICal->RecurrenceID.Day, ICal->RecurrenceID.Month, ICal->RecurrenceID.Year, ICal->RecurrenceID.Hour, ICal->RecurrenceID.Minute, ICal->RecurrenceID.Second);
		if (RecurrenceID == (unsigned long)-1) {
			RecurrenceID = 0;
		}
		if (RecurrenceID) {
			ReplyInt=snprintf(Buffer, sizeof(Buffer), "CSFIND %lu %s %lu\r\n", ICal->Sequence, ICal->UID, RecurrenceID);
		} else {
			ReplyInt=snprintf(Buffer, sizeof(Buffer), "CSFIND %lu %s\r\n", ICal->Sequence, ICal->UID);
		}
		MWSendNMAPServer(Session, Buffer, ReplyInt);
		
		ReplyInt=MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
		if (ReplyInt!=1000) {
			MWSendNMAPServer(Session, "QUIT\r\n", 6);
			MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
			IPclose(Session->NMAPs);
			/* "flush" our input buffer */
			Session->NBufferPtr=NMAPBufferPtr;
			memcpy(Session->NBuffer, Answer, Session->NBufferPtr);
			Session->NMAPs=ClientNMAPs;
			Session->NMAPAddress.s_addr=NMAPCAddress;
			return(FALSE);
		}
		
		EntryID=atol(Buffer);
		
		/* Verify the orgnizer */
		ReplyInt=snprintf(Buffer, sizeof(Buffer), "CSORG %lu\r\n", EntryID);
		MWSendNMAPServer(Session, Buffer, ReplyInt);
		ReplyInt=MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
		if (ReplyInt != 2001) {
			MWSendNMAPServer(Session, "QUIT\r\n", 6);
			MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
			IPclose(Session->NMAPs);
			/* "flush" our input buffer */
			Session->NBufferPtr=NMAPBufferPtr;
			memcpy(Session->NBuffer, Answer, Session->NBufferPtr);
			Session->NMAPs=ClientNMAPs;
			Session->NMAPAddress.s_addr=NMAPCAddress;
			return(FALSE);
		}

		ptr = strchr(Buffer, ' ');
		if (ptr) {
			*ptr = '\0';
		}

		if (!MWQuickCmp(Buffer, ICal->Organizer->Address) && !MWQuickCmp(Buffer, organizer)) {
			MWSendNMAPServer(Session, "QUIT\r\n", 6);
			MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
			IPclose(Session->NMAPs);
			/* "flush" our input buffer */
			Session->NBufferPtr=NMAPBufferPtr;
			memcpy(Session->NBuffer, Answer, Session->NBufferPtr);
			Session->NMAPs=ClientNMAPs;
			Session->NMAPAddress.s_addr=NMAPCAddress;
			return(FALSE);
		}

		if (ClientNMAPs) {
			MWSendNMAPServer(Session, "QUIT\r\n", 6);
			MWGetNMAPAnswer(Session, Buffer, BUFSIZE, TRUE);
			IPclose(Session->NMAPs);
			/* "flush" our input buffer */
			Session->NBufferPtr=NMAPBufferPtr;
			memcpy(Session->NBuffer, Answer, Session->NBufferPtr);
			Session->NMAPs=ClientNMAPs;
			Session->NMAPAddress.s_addr=NMAPCAddress;

		}

		return(TRUE);

	} else {
		return(FALSE);
	}


}
static BOOL
FolderNotShared(unsigned long CalendarID, SessionStruct *Session) 
{
	char SharedFlag;

	if (CalendarID == 0) {
		/*
			Special case

			We know that 0 will always be INBOX, so we are going to select MAIN instead.
		*/

		do {
			CalendarID++;
		} while (CalendarID < Session->FolderList->Used && !MWQuickCmp(Session->FolderList->Value[CalendarID] + 3, "MAIN"));
	}

	if (CalendarID >= Session->FolderList->Used){
		return(TRUE);
	}

	SharedFlag = *(Session->FolderList->Value[CalendarID] + 1);

    /* fixme - 0 = resource and not shared; 1 = resource and shared; 2 = share */
	if (SharedFlag != '2')
		return(TRUE);

	return(FALSE);

}


/* This is for the mail view */
static BOOL
MWCALHandleCalendarTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData)
{
	ICalObject *ICal=(ICalObject *)ModuleData;

	if (!ICal) {
		return(FALSE);
	}

	ModDebug("MWCALHandleCalendarTemplate(%lu) called\n", Token->TokenID);
	switch(Token->TokenID) {
		case	T_CAL_DETAILS_VALUE: {
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_ORGANIZER: {	/* Sender */
					if (ICal->Organizer && ICal->Organizer->Address && ICal->Organizer->Address != '\0') {
						if (ICal->Organizer->Name && ICal->Organizer->Name[0] != '\0') {
							MWSendClient(Client, ICal->Organizer->Name, strlen(ICal->Organizer->Name));
						} else {
							MWSendClient(Client, ICal->Organizer->Address, strlen(ICal->Organizer->Address));
						}
					} else {
						/* If there is no organizer and only one attendee we can use the attendee */
						if (ICal->Attendee && !ICal->Attendee->Next && ICal->Attendee->Address) {
							if (ICal->Attendee->Name) {
								MWSendClient(Client, ICal->Attendee->Name, strlen(ICal->Attendee->Name));
							} else {
								MWSendClient(Client, ICal->Attendee->Address, strlen(ICal->Attendee->Address));
							}
						}
					}
					return(TRUE);
				}

				case AA_SUMMARY: {	/* Summary */
					if (ICal->Summary) {
						SendWithBreaks(ICal->Summary, strlen(ICal->Summary));
					}
					return(TRUE);
				}

				case AA_DESCRIPTION: {	/* Description */
					if (ICal->Description) {
						SendWithBreaks(ICal->Description, strlen(ICal->Description));
					}
					return(TRUE);
				}

				case AA_LOC: {	/* Location */
					if (ICal->Location) {
						MWSendClient(Client, ICal->Location, strlen(ICal->Location));
					}
					return(TRUE);
				}


				case AA_STARTDATE: {	/* Date */
					int ReplyInt;
					ReplyInt = mwCalSPrintDTStart(Client->Temp, sizeof (Client->Temp),Session->DateFormatShort, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ENDDATE: {	/* End-Date */
					int ReplyInt;
					ReplyInt = mwCalSPrintDTEnd(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_STARTTIME: {	/* Start Time */
					int ReplyInt;
					ReplyInt = mwCalSPrintDTStart(Client->Temp, sizeof (Client->Temp), Session->TimeFormat, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ENDTIME: {	/* End-Time */
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTEnd(Client->Temp, sizeof (Client->Temp), Session->TimeFormat, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_STATE: {		/* Calendar state (Free/Busy) */
					return(TRUE);
				}

				case AA_TYPE: {		/* Type of entry (Event/Journal/Note) */
					unsigned char	*ptr=Session->Strings[Token->ArgumentOffsetOrID[1]];

					switch(ICal->Type) {
						case ICAL_VTODO: ptr=Session->Strings[Token->ArgumentOffsetOrID[2]]; break;
						case ICAL_VJOURNAL: ptr=Session->Strings[Token->ArgumentOffsetOrID[3]]; break;
					}
					MWSendClient(Client, ptr, strlen(ptr));
					return(TRUE);
				}
			}
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_LIST_BEGIN: {
			ICalVAttendee		*Attendee=ICal->UserData;
			unsigned long	Role;

			if (!ICal->Attendee) {
				ICal->UserLong=0;
				*GotoToken=T_CAL_ATTENDEE_LIST_END;
				return(TOKEN_MOVE_FORWARD);
			}
			if (ICal->UserLong!=T_CAL_ATTENDEE_LIST_BEGIN) {
				ICal->UserLong=T_CAL_ATTENDEE_LIST_BEGIN;
				ICal->UserData=(void *)ICal->Attendee;
				Attendee=ICal->Attendee;
			}

			switch(Token->ArgumentOffsetOrID[0]) {
				case AA_CHAIR:			Role=ICAL_ROLE_CHAIR; break;
				case AA_REQUIRED:		Role=ICAL_ROLE_REQUIRED_PARTICIPANT; break;
				case AA_OPTIONAL:		Role=ICAL_ROLE_OPTIONAL_PARTICIPANT; break;
				default:					Role=ICAL_ROLE_NON_PARTICIPANT; break;
			}

			while (Attendee && Attendee->Role!=Role) {
				Attendee=Attendee->Next;
			}
			if (!Attendee) {
				ICal->UserLong=0;
				*GotoToken=T_CAL_ATTENDEE_LIST_END;
				return(TOKEN_MOVE_FORWARD);
			}
			ICal->UserData=(void *)Attendee;
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_LIST_END: {
			ICalVAttendee		*Attendee=ICal->UserData;
			unsigned long	Role;

			if (ICal->UserLong!=T_CAL_ATTENDEE_LIST_BEGIN) {
				ICal->UserLong=0;
				return(TRUE);
			}

			if (Attendee) {
				Role = Attendee->Role;
				Attendee=Attendee->Next;
			} else {
				Role=ICAL_ROLE_NON_PARTICIPANT;
			}

			while (Attendee && Attendee->Role!=Role) {
				Attendee=Attendee->Next;
			}

			if (Attendee) {
				ICal->UserData=(void *)Attendee;
				*GotoToken=T_CAL_ATTENDEE_LIST_BEGIN;
				return(TOKEN_MOVE_BACKWARD);
			}
			ICal->UserLong=0;
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_LIST_DISPLAY: {
			ICalVAttendee	*Attendee=ICal->UserData;

			if (!Attendee || ICal->UserLong!=T_CAL_ATTENDEE_LIST_BEGIN) {
				return(TRUE);
			}

			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_NAME:{
					if (Attendee->Name) {
						MWHTMLSendClient(Client, Attendee->Name, strlen(Attendee->Name));
						return(TRUE);
					}
					/* Fall through */
				}

				case AA_ADDRESS:{
					if (Attendee->Address) {
						MWHTMLSendClient(Client, Attendee->Address, strlen(Attendee->Address));
					}
					return(TRUE);
				}

				case AA_STATUS: {
					unsigned char *ptr = NULL;

					/* Only print out pending status if it belongs to us or we are the originator */
					if ( /* (Attendee->Role != ICAL_ROLE_NON_PARTICIPANT) && */ (MWQuickCmp(Session->EMailAddress, Attendee->Address) || (ICal && ICal->Organizer && MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address)))) {

						switch(Attendee->State) {
							case ICAL_PARTSTAT_NEEDS_ACTION:		ptr=Session->Strings[Token->ArgumentOffsetOrID[1]]; break;
							case ICAL_PARTSTAT_ACCEPTED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+1];	break;
							case ICAL_PARTSTAT_DECLINED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+2];	break;
							case ICAL_PARTSTAT_TENTATIVE:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+3];	break;
							case ICAL_PARTSTAT_DELEGATED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+4];	break;
							case ICAL_PARTSTAT_COMPLETED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+5];	break;
							case ICAL_PARTSTAT_IN_PROCESS:		ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+6];	break;
								/* HOLE: do we need a default */

						}
						if (ptr)	{
							MWSendClient(Client, ptr, strlen(ptr));

							if ((Attendee->State == ICAL_PARTSTAT_DELEGATED) && Attendee->Delegated) {
								MWSendClient(Client, " - ", 3);
								MWHTMLSendClient(Client, Attendee->Delegated, strlen(Attendee->Delegated));
							} 

						}
					}


					return(TRUE);
				}


			}
			return(TRUE);
		}
		case	T_CAL_ATTENDEE_VALIDATE_STATUS_BEGIN: {
			ICalVAttendee		*Attendee=ICal->UserData;

			/* see if there is a problem with the attendee */
			/* or I am not the attendee  I am not the organizer */
			if (!Attendee || !ICal->Attendee || !Attendee->Address || 
				 (!MWQuickCmp(Session->EMailAddress, Attendee->Address) && ( !ICal->Organizer ||  !MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address))))
			{
				/* common case */
				*GotoToken=T_CAL_ATTENDEE_VALIDATE_STATUS_END;
				return(TOKEN_MOVE_FORWARD);
			} else {
				/* we are ok to display status */
				return(TRUE);			
			} 

		}

		case	T_CAL_ATTENDEE_VALIDATE_STATUS_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case T_CAL_OBJECT_VALIDATE_BEGIN: {
			/* This token either just returns to allow token between begin and end to comlete */
			/* or returns a description string of type of calendar item */
			
			/* If message claims that object organizer is local to hula and */
			/* we cannot find the object ( organizer deleted message ). Then message is invalid and we */
			/* need to move to end of list. */
			if (ValidateICalObject(Client, Session, ICal)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				
				/* move token to T_CAL_OBJECT_VALIDATE_ELSE */
				/* disply error message was sent to client in ValidateICalObject */ 
				*GotoToken = T_CAL_OBJECT_VALIDATE_ELSE;
				return(TOKEN_MOVE_FORWARD);
			}
		}

		case T_CAL_OBJECT_VALIDATE_ELSE: {
			/* This token must either return TRUE if ValidateICalObject returns FALSE or move to end token */
			if (!ValidateICalObject(Client, Session, ICal)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				
				/* move token to T_CAL_OBJECT_VALIDATE_END */
				/* disply error message was sent to client in ValidateICalObject */ 
				*GotoToken = T_CAL_OBJECT_VALIDATE_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
			
		case T_CAL_OBJECT_VALIDATE_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case T_CAL_OBJECT_AMRECIPIENT_BEGIN: {
			/* This token either just returns to allow token between begin and end to complete */
			/* or returns a description string of type of calendar item */
			
			/* If our email address is in Attendee list then return TRUE */
			/* otherwise need to move to end of list. */
			if (AMICalObjectAttendee(Session, ICal)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_OBJECT_AMRECIPIENT_ELSE */
				/* disply error message was sent to client in AmrecipientICalObject */ 
				*GotoToken = T_CAL_OBJECT_AMRECIPIENT_ELSE;
				return(TOKEN_MOVE_FORWARD);
			}
		}

		case T_CAL_OBJECT_AMRECIPIENT_ELSE: {
			/* This token must either return TRUE if AmrecipientICalObject returns FALSE or move to end token */
			if (!AMICalObjectAttendee(Session, ICal)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_OBJECT_AMRECIPIENT_END */
				/* disply error message was sent to client in AmrecipientICalObject */ 
				*GotoToken = T_CAL_OBJECT_AMRECIPIENT_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
			
		case T_CAL_OBJECT_AMRECIPIENT_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case	T_CAL_ISBCC_BEGIN: {

			/* Ways to be BCC */
			/* 1. Originator places himself in BCC field. */
			/*   State - IAMAttendee == TRUE, role == NOT-PARTICIPANTG; */
			/* 2. Originator places another recipient in BCC field. */
			/*    State at originator IAMAttendee == TRUE, role == NOT-PARTICIPATING; */
			/*    State at recipient  IAMAttendee == FALSE, role == NOT-PARTICIPATING; */
			/* 3. Originator doesn't place himself in any field. */
         /*    State at originator IAMAttendee == FALSE role == '\0x' */ 

			/* This token must return true if I am a bcc */
			/* This is determined by not being in the attendee list of my object. */
			if (IsBCCAttendee(ICal, Session) || 
				 (ICal && ICal->Organizer && !AMICalObjectAttendee(Session, ICal) && !MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address))){
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_ISBCC_END */
				*GotoToken = T_CAL_ISBCC_END;
				return(TOKEN_MOVE_FORWARD);
			}
			
		}

		case T_CAL_ISBCC_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case	T_CAL_ISBCC_ATTENDEE_BEGIN: {

			/* This token must either return TRUE if AmrecipientICalObject returns FALSE or move to end token */
			if (IsBCCAttendee(ICal,Session)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_ISBCC_ATTENDEE_END */
				*GotoToken = T_CAL_ISBCC_ATTENDEE_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
		case T_CAL_ISBCC_ATTENDEE_END: {
			return(TRUE);
		}

		case	T_CAL_ISBCC_RECIP_BEGIN: {

			/* This token must either return TRUE if AmrecipientICalObject returns FALSE or move to end token */
			if (ICal && ICal->Organizer && !AMICalObjectAttendee(Session, ICal) && !MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address)){
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_ISBCC_RECIP_END */
				*GotoToken = T_CAL_ISBCC_RECIP_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
		case T_CAL_ISBCC_RECIP_END: {
			return(TRUE);
		}

		case T_CAL_BUTTON_BEGIN: {
			/* Since this comes from message list view, We can only rely on ICal object details which may or may not be true.*/
			/* The only information we have is the type of object note, apt, task - and the attendee information the organizer sent to us.*/
			/* We look for our own status as set by organizer. We then exclude any buttons that don't make sense */
			/* If we don't have any information about the button we let it pass. */
			ICalVAttendee		*Attendee = NULL;

			if (ICal) {
				for (Attendee = ICal->Attendee;Attendee; Attendee = Attendee->Next) {
					if (MWQuickCmp(Session->EMailAddress, Attendee->Address)) {
						break; 
					}
				}
			}

			if (Attendee) {

				/* This token must return TRUE if ICal object is in correct state for Button to be present */
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_ACCEPT:
						switch(Attendee->State) {
							case ICAL_PARTSTAT_DECLINED: /* Declined */
							case ICAL_PARTSTAT_DELEGATED: /* Delegated */
							case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
								return(TRUE);
							default: /* inactive icon */
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
						}
						break;
					case AA_DECLINE:
						switch(Attendee->State) {
							case ICAL_PARTSTAT_ACCEPTED: /* Accepted */
							case ICAL_PARTSTAT_COMPLETED: /* Completed */
							case ICAL_PARTSTAT_DELEGATED: /* Delegated */
							case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
								return(TRUE);
							default: /* inactive icon */
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
						}
						break;

					case AA_DELEGATE:
						if ((ICal->Type == ICAL_VEVENT) ||
							 (ICal->Type == ICAL_VTODO)) {
							switch(Attendee->State) {
								case ICAL_PARTSTAT_ACCEPTED: /* Accepted */
								case ICAL_PARTSTAT_COMPLETED: /* Completed */
								case ICAL_PARTSTAT_DECLINED: /* Declined */
								case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
									return(TRUE);
								default: /* inactive icon */
									*GotoToken = T_CAL_BUTTON_END;
									return(TOKEN_MOVE_FORWARD);
							}
							break;
						} else {
							*GotoToken = T_CAL_BUTTON_END;
							return(TOKEN_MOVE_FORWARD);
						}


					case AA_COMPLETE:{
						if (ICal->Type == ICAL_VTODO) {
							switch(Attendee->State) {
								case ICAL_PARTSTAT_ACCEPTED: /* Accepted */
								case ICAL_PARTSTAT_DECLINED: /* Declined */
								case ICAL_PARTSTAT_DELEGATED: /* Delegated */
								case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
									return(TRUE);
								default: /* inactive icon */
									*GotoToken = T_CAL_BUTTON_END;
									return(TOKEN_MOVE_FORWARD);
							}

						} else {
							*GotoToken = T_CAL_BUTTON_END;
							return(TOKEN_MOVE_FORWARD);
						}
					}

					default:{
						/* let unknown buttons pass */
						break;
					}
				}
			}

		}


		case T_CAL_BUTTON_END: {
			/* placeholder only */
			return(TRUE);
		}



	}
	return(TRUE);
}



BOOL
MWCALHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData)
{
	unsigned char				Answer[BUFSIZE+1];
	CalendarSessionStruct	*CalendarSession=(CalendarSessionStruct *)ModuleData;
	int							ReplyInt;
	unsigned char				*ptr = NULL;
	unsigned char				URL[256];

	/* This will just return if it is already selected */
	if (SelectCalendar(Session->CurrentCalendar, Session, CalendarSession) == TRUE) {
		/* Is there a dirty message that we need to take care of? */
		if (Session->DirtyMessage && (Session->DirtyMessageFolder < Session->FolderList->Used) && (Session->FolderList->Value[Session->DirtyMessageFolder][0] == 'C')) {
			/*
				Yup, there is an item to be deleted, and it is indeed a Calendar item
			*/

			if (!Session->ReadOnly && (SelectCalendar(Session->DirtyMessageFolder, Session, CalendarSession))) {
				ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu\r\n", CalendarSession->IDList[Session->DirtyMessageID - 1]);
				MWSendNMAPServer(Session, Client->Temp, ReplyInt);

				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
			}

			Session->DirtyMessage = FALSE;
		}
	} else {
		return(FALSE);
	}

	switch(Token->TokenID) {
		case T_CAL_FEATURE_START: {
			if (CalendarSession->CalendarEnabled) {
				return(TRUE);
			} else {
				*GotoToken=T_CAL_FEATURE_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}

		case T_CAL_FEATURE_END: {
			return(TRUE);
		}

		case T_CAL_FOLDERS_REFRESH: {

			return(TRUE);
		}

		case T_CAL_NOTSHARED_BEGIN:
			if (FolderNotShared(CalendarSession->CurrentCalendar, Session) == TRUE){
				break;
			} else {
				/* folder is shared */
				/* Move to the end token */
				*GotoToken=T_CAL_NOTSHARED_END;
				return(TOKEN_MOVE_FORWARD);

			}
			break;
		case T_CAL_NOTSHARED_END:
			break;


		case T_CAL_LIST_BEGIN: {
			BOOL					HaveEntry = FALSE;

			if (CalendarSession->InView == 0) {
				unsigned long		BeginTime;
				unsigned long		EndTime;

				/* Must prepare cache data first */
				ReplyInt = SelectCalendar(CalendarSession->CurrentCalendar, Session, CalendarSession);

				/* Cache the current month + 6 days on each end for the week view */
				BeginTime = MsgGetUTC(1, Session->CurrentMonth, Session->CurrentYear, 0, 0, 0) - (6*24*60*60);
				EndTime = MsgGetUTC(MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear), Session->CurrentMonth, Session->CurrentYear, 0, 0, 0)  + (6*24*60*60);
				if (ReplyInt==0 || LoadCalendarView(BeginTime, EndTime, Session, CalendarSession)==0) {
					*GotoToken=T_CAL_LIST_END;
					return(TOKEN_MOVE_FORWARD);
				}

				CalendarSession->InView=Token->ArgumentOffsetOrID[0];
				CalendarSession->ListUTCStart = MsgGetUTC(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear, 0, 0, 0);
				CalendarSession->ListUTCStart-=MsgGetUTCOffsetByUTC(Session->TimezoneID, CalendarSession->ListUTCStart);
				CalendarSession->ListUTCEnd = CalendarSession->ListUTCStart + 86399;
			}
			
			/* Uncompleted tasks float - JNT ? what about tasks that are originated by me but I'm not attending ??? */
			/* If entry is marked floating and begin day is not the current day we skip to the next one */

			while (!HaveEntry) {
				if (!FindCalendarEntry(CalendarSession->InView, CalendarSession->ListUTCStart, CalendarSession->ListUTCEnd, Session, CalendarSession)) {
					CalendarSession->InView=0;
					*GotoToken=T_CAL_LIST_END;
					return(TOKEN_MOVE_FORWARD);
				}

				HaveEntry = TRUE;

#if 0
				if (CalendarSession->HideFloatingTasks && CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCDue) {
					if ((CalendarSession->ListUTCStart<=CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCStart) &&
						(CalendarSession->ViewCache[CalendarSession->ViewPointer-1].UTCStart<=CalendarSession->ListUTCEnd)) {
							HaveEntry = TRUE;
					}
				} else {
					HaveEntry = TRUE;
				}
#else 
#endif
			}

			if (!LoadCalendarEntry(CalendarSession->ViewCache[CalendarSession->ViewPointer-1].ID, Session, CalendarSession)) {
				CalendarSession->InView=0;
				*GotoToken=T_CAL_LIST_END;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case	T_CAL_LIST_END: {
			if (CalendarSession->InView!=0) {
				*GotoToken=T_CAL_LIST_BEGIN;
				return(TOKEN_MOVE_BACKWARD);
			}
			return(TRUE);
		}

		case	T_CAL_LIST_CUR: {
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_SUMMARY: {	/* Summary */
					MWSendClient(Client, CalendarSession->EntrySummary, strlen(CalendarSession->EntrySummary));
					break;
				}

				case AA_STARTDATE: {	/* Date */
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTStartViewCache(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, CalendarSession->ICal, CalendarSession, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ENDDATE: {	/* End-Date */
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTEndViewCache(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, CalendarSession->ICal, CalendarSession, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_STARTTIME: {	/* Start Time */
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTStartViewCache(Client->Temp, sizeof (Client->Temp), Session->TimeFormat, CalendarSession->ICal, CalendarSession, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ENDTIME: {	/* End-Time */
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTEndViewCache(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, CalendarSession->ICal, CalendarSession, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ORGANIZER: {	/* Sender */
					if (CalendarSession->EntryOrganizer[0] != '\0') {
						MWSendClient(Client, CalendarSession->EntryOrganizer, strlen(CalendarSession->EntryOrganizer));
					} else {
						/* If we don't have an orginizer we can assume that the user is the organizer */
						MWSendClient(Client, Session->User, strlen(Session->User));
					}
					break;
				}

				case AA_CURRENTMONTHDAY: { /* The current day of the month */
					unsigned long		Len;

					Len = snprintf(Client->Temp, sizeof(Client->Temp), "%d", (int)Session->CurrentDay);
					MWSendClient(Client, Client->Temp, Len);
					break;
				}

				case AA_CURRENTWEEKDAY: { /* The current day of the week */
					unsigned long	CurrentDayOfWeek;

					CurrentDayOfWeek = MSG_DAY_OF_WEEK(MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear));
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[1] + CurrentDayOfWeek], strlen(Session->Strings[Token->ArgumentOffsetOrID[1] + CurrentDayOfWeek]));
					return(TRUE);
				}

				case AA_ID: { /* ID For the current calendar object */
					unsigned long		Len;

					Len = snprintf(Client->Temp, sizeof(Client->Temp), "%d", (int)CalendarSession->ViewCache[CalendarSession->ViewPointer-1].ID);
					MWSendClient(Client, Client->Temp, Len);
					return(TRUE);
				}
			}

			return(TRUE);
		}

		case	T_CAL_CUR_VIEW: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_VIEW_OBJECT, CalendarSession->ViewCache[CalendarSession->ViewPointer-1].ID, CalendarSession->CurrentCalendar, Token->ArgumentOffsetOrID[0], 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case	T_CAL_CUR_SELECT: {
		    MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_SET_CURRENT_DATE, Token->ArgumentOffsetOrID[0], MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear), 0, 0);
		    MWSendClient(Client, URL, strlen(URL));
		    return(TRUE);
		}

		case T_CAL_LIST_STATEIMAGE:{
			/* We need to be able to distinguish between the following icons */
         /* Arg - Explanation */

			/*       APPOINTMENT        */
			/* 0)    Needs action Icon - state of appointment is Needs action */
			/*       This occurs when the organizer of a appointment has not yet accepted/declined the appointment. */
			/* 1)    Inactive Icon - This occurs when the appointment is set to Show Appointment as Free, as */
			/*                       indicated by MSG_STATE_NOFREEBUSY flag being set. */
			/*       Default: Active, unless the organizer of a appointment is not a recipient of the Appointment. */
			/*                Set to inactive when organizer declines the appointment, for which other recipients */
			/*                are still a part of, or when the Show Appointment as Free Button is activated.  */
			/* 2)    Active Icon - This is used when MSG_STATE_NOFREEBUSY is not set. Which is true when the */
			/*                     Appointment has been accepted, Or when the Show Appointment */
			/*                     as Busy button has been pressed. */

			/*       TASK        */
			/* 0)    Needs Action Icon - This happens if the organizer of a task is a recipient and has not yet accepted/declined the task. */
			/* 2)    Organizer Icon - assign this to , because the task is completed which respect to the organizer. */
			/*       - This occurs when the organizer of a task is not a recipient of the task, or has declined or delegated the task, for which other recipients are still a part of. */
			/* 1)    Active Icon - This is when the task has been accepted. */
			/* 2)    Completed Icon - This is when the task has been marked completed.*/

			/*       NOTE        */
			/* 0)    Note needs action Icon - This occurs when the organizer of a note has not yet accepted/declined the note, for which other recipients are a part of. */
			/* 0)    Note inactive Icon - This occurs when the organizer of a note is not a recipient of the note, or has declined the note. */
			/* 0)    Note active Icon - This is when the note has been accepted. */
         /*       Since it is not important to distinguich these we just use the same icon. A note is a note for crying out load! */

			char Role = CalendarSession->EntryRole;
			char Status = CalendarSession->EntryStatus;

			/* Handle Originator differently if he is not participating (to: cc: bcc:) */ 
			if (Role != '\0') {

				switch (CalendarSession->ViewCache[CalendarSession->ViewPointer-1].Type) {
					case NMAP_CAL_EVENT:{
						switch(Status) {
							case 'N': /* Needs attention */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);									
								break;

							case 'D': /* Declined */
							case 'G': /* Delegated */
								if (CalendarSession->EntryState & MSG_STATE_NOFREEBUSY) {
									MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
								} else {
									MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
								}

								break;

							case 'A': /* Accepted */
								if (!(CalendarSession->EntryState & MSG_STATE_NOFREEBUSY)) {
									MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
								} else {
									MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
								}

								break;

							case 'T': /* Tentative */
							case 'I': /* In Process */
							default: /* inactive icon */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
								break;
						
						}
						break;
					}
					case NMAP_CAL_TODO: {
						switch(Status) {
							case 'N': /* Needs attention */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
								break;

							case 'D': /* Declined */
							case 'G': /* Delegated */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[3], 0);
								break;

							case 'A': /* Accepted */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
								break;

							case 'C': /* Completed */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
								break;

							case 'T': /* Tentative */
							case 'I': /* In Process */
							default: /* inactive icon */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
								break;
						
						}
						break;
					}
					case NMAP_CAL_JOURNAL:{
						switch(Status) {
							case 'N': /* Needs attention */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
								break;

							case 'D': /* Declined */
							case 'G': /* Delegated */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
								break;

							case 'A': /* Accepted */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
								break;

							case 'T': /* Tentative */
							case 'I': /* In Process */
							default: /* inactive icon */
								MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
								break;
						
						}
						break;
					}
					default: {
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
						break;
					}
				}
			} else { 
				/* handle originator not in to: cc: or bcc: fields differently */
				switch (CalendarSession->ViewCache[CalendarSession->ViewPointer-1].Type) {
					case NMAP_CAL_EVENT:{
						/* look at free busy since appointments can be marked with free busy */
						if (CalendarSession->EntryState & MSG_STATE_NOFREEBUSY) {
							/* We are marked free show it as inactive */
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
						} else {
							/* We are marked busy show it as active */
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
						}
						break;
					}
					case NMAP_CAL_TODO: {
						/* Tasks show up for organizer benefit only - otherwise we have nothing to do with them */
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[3], 0);
						break;
					}
					case NMAP_CAL_JOURNAL:{
						{
							/* Journals show up the same all the time anyway */
							MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
						}
						break;
					}

					default: {
						MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
						break;
					}
				}
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}
		case T_CAL_LIST_STATESTRING:{
			/* WARNING ! this must remain exactly parallel to T_CAL_LIST_STATEIMAGE*/
			char Role = CalendarSession->EntryRole;
			char Status = CalendarSession->EntryStatus;
			int Index = 0;

			/* Handle Originator differently if he is not participating (to: cc: bcc:) */ 
			if (Role != '\0') {

				switch (CalendarSession->ViewCache[CalendarSession->ViewPointer-1].Type) {
					case NMAP_CAL_EVENT:{
						switch(Status) {
							case 'N': /* Needs attention */
								Index = 0;
								break;

							case 'D': /* Declined */
							case 'G': /* Delegated */
								if (CalendarSession->EntryState & MSG_STATE_NOFREEBUSY) {
									Index = 1;
								} else {
									Index = 2;
								}

								break;

							case 'A': /* Accepted */
								if (!(CalendarSession->EntryState & MSG_STATE_NOFREEBUSY)) {
									Index = 2;
								} else {
									Index = 1;
								}

								break;

							case 'T': /* Tentative */
							case 'I': /* In Process */
							default: /* inactive icon */
								Index = 1;
								break;
						
						}
						break;
					}
					case NMAP_CAL_TODO: {
						switch(Status) {
							case 'N': /* Needs attention */
								Index = 0;
								break;

							case 'D': /* Declined */
							case 'G': /* Delegated */
								Index = 3;
								break;

							case 'A': /* Accepted */
								Index = 1;
								break;

							case 'C': /* Completed */
								Index = 2;
								break;

							case 'T': /* Tentative */
							case 'I': /* In Process */
							default: /* inactive icon */
								Index = 1;
								break;
						
						}
						break;
					}
					case NMAP_CAL_JOURNAL:{
						switch(Status) {
							case 'N': /* Needs attention */
								Index = 0;
								break;

							case 'D': /* Declined */
							case 'G': /* Delegated */
								Index = 0;
								break;

							case 'A': /* Accepted */
								Index = 0;
								break;

							case 'T': /* Tentative */
							case 'I': /* In Process */
							default: /* inactive icon */
								Index = 0;
								break;
						
						}
						break;
					}
					default: {
						Index = 0;
						break;
					}
				}
			} else { 
				/* handle originator not in to: cc: or bcc: fields differently */
				switch (CalendarSession->ViewCache[CalendarSession->ViewPointer-1].Type) {
					case NMAP_CAL_EVENT:{
						/* look at free busy since appointments can be marked with free busy */
						if (CalendarSession->EntryState & MSG_STATE_NOFREEBUSY) {
							/* We are marked free show it as inactive */
							Index = 1;
						} else {
							/* We are marked busy show it as active */
							Index = 2;
						}
						break;
					}
					case NMAP_CAL_TODO: {
						/* Tasks show up for organizer benefit only - otherwise we have nothing to do with them */
						Index = 3;
						break;
					}
					case NMAP_CAL_JOURNAL:{
						{
							/* Journals show up the same all the time anyway */
							Index = 0;
						}
						break;
					}

					default: {
						Index = 0;
						break;
					}
				}
			}

			MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[Index]] , strlen(Session->Strings[Token->ArgumentOffsetOrID[Index]]));
			return(TRUE);
		}



		case	T_CAL_DETAILS_VALUE: {
			/* Make sure that the calendar and calendar item we think we have cached are in fact cached */
			if (CalendarSession->CurrentCalendar != CalendarSession->DetailCalendar) {
				SelectCalendar(CalendarSession->DetailCalendar, Session, CalendarSession);
			}

			LoadCalendarEntryDetails(CalendarSession->DetailID, Session, CalendarSession, FALSE);
			if (!CalendarSession->ICal) {
				return(TRUE);
			}

			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_SUMMARY: {	/* Summary */
					if (CalendarSession->ICal->Summary) {
						SendWithBreaks(CalendarSession->ICal->Summary, strlen(CalendarSession->ICal->Summary));
					}
					break;
				}

				case AA_DESCRIPTION: {	/* Description */
					if (CalendarSession->ICal->Description) {
						SendWithBreaks(CalendarSession->ICal->Description, strlen(CalendarSession->ICal->Description));
					}
					break;
				}

				case AA_LOC: {	/* Location */
					if (CalendarSession->ICal->Location) {
						MWSendClient(Client, CalendarSession->ICal->Location, strlen(CalendarSession->ICal->Location));
					}
					break;
				}

				case AA_STARTDATE: {	/* Date */
					ICalObject		*ICal = CalendarSession->ICal;
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTStart(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ENDDATE: {	/* End-Date */
					ICalObject		*ICal = CalendarSession->ICal;
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTEnd(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_STARTTIME: {	/* Start Time */
					ICalObject		*ICal = CalendarSession->ICal;
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTStart(Client->Temp, sizeof (Client->Temp), Session->TimeFormat, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ENDTIME: {	/* End-Time */
					ICalObject		*ICal = CalendarSession->ICal;
					unsigned long	ReplyInt;
					ReplyInt = mwCalSPrintDTEnd(Client->Temp, sizeof (Client->Temp), Session->TimeFormat, ICal, Session);
					MWSendClient(Client, Client->Temp, ReplyInt);
					return(TRUE);
				}

				case AA_ORGANIZER: {	/* Sender */
					ICalObject		*ICal = CalendarSession->ICal;

					if (ICal->Organizer && ICal->Organizer->Address && ICal->Organizer->Address != '\0') {
						if (ICal->Organizer->Name && ICal->Organizer->Name[0] != '\0') {
							MWSendClient(Client, ICal->Organizer->Name, strlen(ICal->Organizer->Name));
						} else {
							MWSendClient(Client, ICal->Organizer->Address, strlen(ICal->Organizer->Address));
						}
					} else {
						/* If there is no organizer and only one attendee we can use the attendee */
						if (ICal->Attendee && !ICal->Attendee->Next && ICal->Attendee->Address) {
							if (ICal->Attendee->Name) {
								MWSendClient(Client, ICal->Attendee->Name, strlen(ICal->Attendee->Name));
							} else {
								MWSendClient(Client, ICal->Attendee->Address, strlen(ICal->Attendee->Address));
							}
						}
					}
					return(TRUE);
				}

				case AA_RECURSTRING: { /* Reccurence string in human readable format */
					if (CalendarSession->ICal->Rule) {
						//MWSendClient(Client, CalendarSession->ICal->Rule, strlen(CalendarSession->ICal->Rule));
					}
					return(TRUE);
				}
			}

			return(TRUE);
		}

		case	T_CAL_DETAILS_ACTION: {
			ICalObject		*ICal = CalendarSession->ICal;
			/* If there is not a return page value or we are not the organizer for this appointment ? */
			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_DECLINECOMMENT:{
					if (!Token->ArgumentSize[2] || 
						 (!ICal || !ICal->Organizer || 
						  !MWQuickNCmp(ICal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress)))){
						/* Token arguments - 0 AA_DECLINECOMMENT, 1 - next page for declinecomment,  2 next page decline */
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_REPLY, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], CalendarSession->DetailID, CalendarSession->CurrentCalendar);
					} else {
						/* Treat this like a normal decline */
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_REPLY, AA_DECLINE, Token->ArgumentOffsetOrID[2], CalendarSession->DetailID, CalendarSession->CurrentCalendar);
					}
					break;
				}
				case AA_COMPLETECOMMENT:{
					if (!Token->ArgumentSize[1] || 
						 (!ICal || !ICal->Organizer || 
						  !MWQuickNCmp(ICal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress)))){
						/* Token arguments - 0 AA_COMPLETECOMMENT, 1 - next page for completecomment,  2 next page complete */
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_REPLY, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], CalendarSession->DetailID, CalendarSession->CurrentCalendar);
					} else {
						/* Treat this like a normal complete */
						MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_REPLY, AA_COMPLETE, Token->ArgumentOffsetOrID[2], CalendarSession->DetailID, CalendarSession->CurrentCalendar);
					}
					break;
				}
				default:
					MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_REPLY, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], CalendarSession->DetailID, CalendarSession->CurrentCalendar);
					break;
					}

			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_LIST_BEGIN: {
			unsigned long	Role;

			/* Make sure that the calendar and calendar item we think we have cached are in fact cached */
			if (CalendarSession->CurrentCalendar != CalendarSession->DetailCalendar) {
				SelectCalendar(CalendarSession->DetailCalendar, Session, CalendarSession);
			}

			LoadCalendarEntryDetails(CalendarSession->DetailID, Session, CalendarSession, FALSE);
			if (!CalendarSession->ICal) {
				return(TRUE);
			}

			if (!CalendarSession->CurrentAttendee) {
				CalendarSession->CurrentAttendee = CalendarSession->ICal->Attendee;
			} else {
				CalendarSession->CurrentAttendee = CalendarSession->CurrentAttendee->Next;
			}

			switch(Token->ArgumentOffsetOrID[0]) {
				case AA_CHAIR:			Role=ICAL_ROLE_CHAIR; break;
				case AA_REQUIRED:		Role=ICAL_ROLE_REQUIRED_PARTICIPANT; break;
				case AA_OPTIONAL:		Role=ICAL_ROLE_OPTIONAL_PARTICIPANT; break;
				default:					Role=ICAL_ROLE_NON_PARTICIPANT; break;
			}

			while (CalendarSession->CurrentAttendee && CalendarSession->CurrentAttendee->Role != Role) {
				CalendarSession->CurrentAttendee = CalendarSession->CurrentAttendee->Next;
			}
			if (!CalendarSession->CurrentAttendee) {
				*GotoToken = T_CAL_ATTENDEE_LIST_END;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_LIST_END: {
			if (CalendarSession->CurrentAttendee) {
				*GotoToken = T_CAL_ATTENDEE_LIST_BEGIN;
				return(TOKEN_MOVE_BACKWARD);
			}
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_LIST_DISPLAY: {
			if (!CalendarSession->CurrentAttendee) {
				*GotoToken = T_CAL_ATTENDEE_LIST_END;
				return(TOKEN_MOVE_FORWARD);
			}

			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_NAME:{
					if (CalendarSession->CurrentAttendee->Name) {
						MWHTMLSendClient(Client, CalendarSession->CurrentAttendee->Name, strlen(CalendarSession->CurrentAttendee->Name));
						return(TRUE);
					}
					/* Fall through */
				}

				case AA_ADDRESS:{
					if (CalendarSession->CurrentAttendee->Address) {
						MWHTMLSendClient(Client, CalendarSession->CurrentAttendee->Address, strlen(CalendarSession->CurrentAttendee->Address));
					}
					return(TRUE);
				}

				case AA_STATUS: {
					/* Only print out status if we are the organizer or if we are printing out our own status and we are not non-participant  */
					if (/* JNT (CalendarSession->CurrentAttendee->Role != ICAL_ROLE_NON_PARTICIPANT) && */ ((CalendarSession->CurrentAttendee->Address && MWQuickCmp(Session->EMailAddress, CalendarSession->CurrentAttendee->Address)) || 
						 (CalendarSession->ICal && CalendarSession->ICal->Organizer && MWQuickCmp(Session->EMailAddress, CalendarSession->ICal->Organizer->Address)))) {
						ptr = NULL;

						switch(CalendarSession->CurrentAttendee->State) {
							case ICAL_PARTSTAT_NEEDS_ACTION:	  ptr=Session->Strings[Token->ArgumentOffsetOrID[1]];	break;
							case ICAL_PARTSTAT_ACCEPTED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+1];	break;
							case ICAL_PARTSTAT_DECLINED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+2];	break;
							case ICAL_PARTSTAT_TENTATIVE:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+3];	break;
							case ICAL_PARTSTAT_DELEGATED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+4];	break;
							case ICAL_PARTSTAT_COMPLETED:			ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+5];	break;
							case ICAL_PARTSTAT_IN_PROCESS:		ptr=Session->Strings[Token->ArgumentOffsetOrID[1]+6];	break;
								/* HOLE: do we need a default */
						}

						if (ptr) {

							MWSendClient(Client, ptr, strlen(ptr));

							if ((CalendarSession->CurrentAttendee->State == ICAL_PARTSTAT_DELEGATED) && CalendarSession->CurrentAttendee->Delegated) {
								MWSendClient(Client, " - ", 3);
								MWHTMLSendClient(Client, CalendarSession->CurrentAttendee->Delegated, strlen(CalendarSession->CurrentAttendee->Delegated));
							} 

						}
					}

					return(TRUE);
				}
			}
			return(TRUE);
		}

		case	T_CAL_ATTENDEE_VALIDATE_STATUS_BEGIN: {
			ICalVAttendee		*Attendee=CalendarSession->CurrentAttendee;
			ICalObject *ICal = CalendarSession->ICal;

			/* see if there is a problem with the attendee */
			/* or I am not the attendee and I am not the organizer */
			if (!Attendee || !ICal->Attendee || !Attendee->Address || /* JNT (Attendee->Role == ICAL_ROLE_NON_PARTICIPANT) || */
				 (!MWQuickCmp(Session->EMailAddress, Attendee->Address) && ( !ICal->Organizer ||  !MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address))))
			{
				/* common case */
				*GotoToken=T_CAL_ATTENDEE_VALIDATE_STATUS_END;
				return(TOKEN_MOVE_FORWARD);
			} else {
				/* we are ok to display status */
				return(TRUE);			
			} 

		}

		case	T_CAL_ATTENDEE_VALIDATE_STATUS_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case T_CAL_OBJECT_VALIDATE_BEGIN: {
			ICalObject *ICal = CalendarSession->ICal;
			/* This token either just returns to allow token between begin and end to comlete */
			/* or returns a description string of type of calendar item */
			
			/* If message claims that object organizer is local to hula and */
			/* we cannot find the object ( organizer deleted message ). Then message is invalid and we */
			/* need to move to end of list. */
			if (ValidateICalObject(Client, Session, ICal)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				
				/* move token to T_CAL_OBJECT_VALIDATE_ELSE */
				/* disply error message was sent to client in ValidateICalObject */ 
				*GotoToken = T_CAL_OBJECT_VALIDATE_ELSE;
				return(TOKEN_MOVE_FORWARD);
			}
		}

		case T_CAL_OBJECT_VALIDATE_ELSE: {
			ICalObject *ICal = CalendarSession->ICal;
			/* This token must either return TRUE if ValidateICalObject returns FALSE or move to end token */
			if (!ValidateICalObject(Client, Session, ICal)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				
				/* move token to T_CAL_OBJECT_VALIDATE_END */
				/* disply error message was sent to client in ValidateICalObject */ 
				*GotoToken = T_CAL_OBJECT_VALIDATE_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
			
		case T_CAL_OBJECT_VALIDATE_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case T_CAL_OBJECT_AMRECIPIENT_BEGIN: {
			/* This token either just returns to allow token between begin and end to comlete */
			/* or returns a description string of type of calendar item */
			
			/* If our email address is in Attendee list then return TRUE */
			/* otherwise need to move to end of list. */
			if (CalendarSession->IAmAttendee){
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_OBJECT_AMRECIPIENT_ELSE */
				/* disply error message was sent to client in AmrecipientICalObject */ 
				*GotoToken = T_CAL_OBJECT_AMRECIPIENT_ELSE;
				return(TOKEN_MOVE_FORWARD);
			}
		}

		case T_CAL_OBJECT_AMRECIPIENT_ELSE: {
			/* This token must either return TRUE if AmrecipientICalObject returns FALSE or move to end token */
			if (!CalendarSession->IAmAttendee){
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				
				/* move token to T_CAL_OBJECT_AMRECIPIENT_END */
				/* disply error message was sent to client in AmrecipientICalObject */ 
				*GotoToken = T_CAL_OBJECT_AMRECIPIENT_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
			
		case T_CAL_OBJECT_AMRECIPIENT_END: {
			/* This token is placeholder only */
			return(TRUE);
		}


		case T_CHANGE_DATE: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_CHANGE_DATE, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2], 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_SELECT_TODAY: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_SELECT_TODAY, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CURRENTMONTH: {
			ptr=Session->Strings[Token->ArgumentOffsetOrID[0]+Session->CurrentMonth-1];
			MWSendClient(Client, ptr, strlen(ptr));
			return(TRUE);
		}

		case T_CURRENTYEAR: {
			ReplyInt=snprintf(Answer, sizeof(Answer), "%d", (int)Session->CurrentYear);
			MWSendClient(Client, Answer, ReplyInt);
			return(TRUE);
		}

		case T_MONTH_BEGIN_WEEK: {
			if (CalendarSession->SavedRataDie == 0) {
				CalendarSession->SavedRataDie = MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);
				Session->CurrentDay = 1;
				CalendarSession->CurrentWeekDay = 0;
				CalendarSession->HideFloatingTasks = TRUE;
			}
			return(TRUE);
		}

		case T_MONTH_BEGIN_DAY:	{
			unsigned long	CurrentDayOfWeek;

			CurrentDayOfWeek = MSG_DAY_OF_WEEK(MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear));
			if ((CurrentDayOfWeek != ((CalendarSession->CurrentWeekDay + Session->StartWeekday+7) % 7)) || 
					(Session->CurrentDay > MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear))) {
				*GotoToken=T_MONTH_ELSE_DAY;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_MONTH_ELSE_DAY: {
			unsigned long	CurrentDayOfWeek;

			CurrentDayOfWeek = MSG_DAY_OF_WEEK(MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear));
			if ((CurrentDayOfWeek == ((CalendarSession->CurrentWeekDay + Session->StartWeekday+7) % 7)) && 
					(Session->CurrentDay <= MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear))) {
				*GotoToken=T_MONTH_END_DAY;
				return(TOKEN_MOVE_FORWARD);
			}
			return(TRUE);
		}

		case T_MONTH_END_DAY:	{
			unsigned long	CurrentDayOfWeek;

			CurrentDayOfWeek = MSG_DAY_OF_WEEK(MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear));
			if (CurrentDayOfWeek == ((CalendarSession->CurrentWeekDay + Session->StartWeekday+7) % 7)) {
				Session->CurrentDay++;
			}

			CalendarSession->CurrentWeekDay++;
			if (CalendarSession->CurrentWeekDay > 6) {
				CalendarSession->CurrentWeekDay = 0; 
				return(TRUE);
			} else {
				*GotoToken=T_MONTH_BEGIN_DAY;
				return(TOKEN_MOVE_BACKWARD);
			}
		}


		case T_MONTH_END_WEEK: {
			if (Session->CurrentDay <= MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear)) {
				*GotoToken=T_MONTH_BEGIN_WEEK;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				Session->CurrentRataDie = CalendarSession->SavedRataDie;
				MsgGetDate(Session->CurrentRataDie, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear);
				CalendarSession->SavedRataDie = 0;
				CalendarSession->HideFloatingTasks = FALSE;

				return(TRUE);
			}
		}

		case T_WEEK_BEGIN: {
			unsigned long	CurrentDayOfWeek;
			unsigned long	RD;

			if (CalendarSession->SavedRataDie == 0) {
				CalendarSession->CurrentWeekDay = 0;
				RD = MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);
				CalendarSession->SavedRataDie = RD;

				/* Move to the begining of the week */
				CurrentDayOfWeek = MSG_DAY_OF_WEEK(MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear));
				RD -= CurrentDayOfWeek;
				RD += Session->StartWeekday;
				MsgGetDate(RD, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear);
				CalendarSession->HideFloatingTasks = TRUE;
			} else {
				RD = MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);
				RD++;
				MsgGetDate(RD, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear);
				CalendarSession->CurrentWeekDay++;
			}

			return(TRUE);
		}

		case T_WEEK_END: {
			if (CalendarSession->CurrentWeekDay < 6) {
				*GotoToken=T_WEEK_BEGIN;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				Session->CurrentRataDie = CalendarSession->SavedRataDie;
				MsgGetDate(Session->CurrentRataDie, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear);
				CalendarSession->SavedRataDie = 0;
				CalendarSession->HideFloatingTasks = FALSE;
				return(TRUE);
			}
		}

		case T_MINICALENDAR: {
			unsigned long	Rows;
			unsigned long	i;
			unsigned long	Days;
			unsigned long	Day;
			unsigned long   StartCol;
			unsigned long	Cells;
			unsigned long	RD1;

			RD1 = MsgGetRataDie(1, Session->CurrentMonth, Session->CurrentYear);
			StartCol = (MSG_DAY_OF_WEEK(RD1)-Session->StartWeekday+7) % 7;
			Days = MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear);
			Rows = MSG_CAL_ROWS(Session->StartWeekday, Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear, Session->CurrentRataDie);
			Cells = Rows * 7;

			/* Send table headers */
			MWSendClient(Client, "<tr>", 4);
			for (i = 0; i < 7; i++) {
				MWSendClient(Client, "<th>", 4);

				if (Token->ArgumentSize[3] > 0) {
					MWSendClient(Client, "<font face=\"", 12);
					MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
					MWSendClient(Client, "\" size=\"", 8);
					MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[4], Token->ArgumentSize[4]);
					MWSendClient(Client, "\">", 2);
				}

				ptr = Session->Strings[Token->ArgumentOffsetOrID[2]+ ((i+Session->StartWeekday) % 7)];
				MWSendClient(Client, ptr, strlen(ptr));

				if (Token->ArgumentSize[3] > 0) {
					MWSendClient(Client, "</font>", 7);
				}

				MWSendClient(Client, "</th>\r\n", 7);
			}
			MWSendClient(Client, "</tr><tr>", 9);

			for (i = 0; i < StartCol; i++) {
				MWSendClient(Client, "<td></td>", 9);
			}

			Day = 0;
			for (i = StartCol; i < (Days + StartCol); i++) {
				if ((i % 7) == 0) {
					MWSendClient(Client, "</tr><tr>", 9);
				}

				if ((Day + 1) != Session->CurrentDay) {
					MWSendClient(Client, "<td align=center>", 17);
				} else {
					MWSendClient(Client, "<td align=center bgcolor=white>", 31);
				}

				if (Token->ArgumentSize[3] > 0) {
					MWSendClient(Client, "<font face=\"", 12);
					MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
					MWSendClient(Client, "\" size=\"", 8);
					MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[4], Token->ArgumentSize[4]);
					MWSendClient(Client, "\">", 2);
				}

				/* Insert anchor to select day */
				MWSendClient(Client, "<a href=\"", 9);
				MWEncodeURL(Session, Answer, URL_TYPE_LINK, REQUEST_SET_CURRENT_DATE, Token->ArgumentOffsetOrID[0], RD1 + Day, 0, 0);
				MWSendClient(Client, Answer, strlen(Answer));
				MWSendClient(Client, "\" target=\"", 10);
				MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
				MWSendClient(Client, "\" alt=\"", 7);
				Day++;
				ReplyInt = snprintf(Answer, sizeof(Answer), "%d", (int)Day);
				/* Day for the alt tag */
				MWSendClient(Client, Answer, ReplyInt);
				MWSendClient(Client, "\">", 2);

				/* Now send the actual day */
				MWSendClient(Client, Answer, ReplyInt);
				MWSendClient(Client, "</a>", 4);

				if (Token->ArgumentSize[3] > 0) {
					MWSendClient(Client, "</font>", 7);
				}

				MWSendClient(Client, "</td>\r\n", 7);
			}

			for (i = StartCol + Days; i < Cells; i++) {
				MWSendClient(Client, "<td></td>", 9);
			}

			MWSendClient(Client, "</tr>", 5);
			return(TRUE);
		}

		case T_MINIMONTH_CAL: {
			unsigned long	x;
			unsigned long	y;
			unsigned long	CurrentMonth = 0;
			unsigned long	RD;

			/* Send table headers */
			MWSendClient(Client, "<tr>", 4);

			for (x = 0; x < 3; x++) {
				MWSendClient(Client, "</tr><tr>", 9);

				for (y = 0; y < 4; y++) {

					if ((CurrentMonth + 1) != Session->CurrentMonth) {
						MWSendClient(Client, "<td align=center>", 17);
					} else {
						MWSendClient(Client, "<td align=center bgcolor=white>", 31);
					}

					if (Token->ArgumentSize[3] > 0) {
						MWSendClient(Client, "<font face=\"", 12);
						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
						MWSendClient(Client, "\" size=\"", 8);
						MWSendClient(Client, Token->Data + Token->ArgumentOffsetOrID[4], Token->ArgumentSize[4]);
						MWSendClient(Client, "\">", 2);
					}

					/* Insert anchor to select day */
					MWSendClient(Client, "<a href=\"", 9);
					RD = MsgGetRataDie((Session->CurrentDay <= MSG_DAYS_IN_MONTH(CurrentMonth + 1, Session->CurrentYear)) ? Session->CurrentDay : MSG_DAYS_IN_MONTH(CurrentMonth + 1, Session->CurrentYear),
						CurrentMonth + 1, Session->CurrentYear);

					MWEncodeURL(Session, Answer, URL_TYPE_LINK, REQUEST_SET_CURRENT_DATE, Token->ArgumentOffsetOrID[0], RD, 0, 0);
					MWSendClient(Client, Answer, strlen(Answer));

					MWSendClient(Client, "\" target=\"", 10);
					MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
					MWSendClient(Client, "\" alt=\"", 7);

					/* Month Name for the alt tag */
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[2] + CurrentMonth], strlen(Session->Strings[Token->ArgumentOffsetOrID[2] + CurrentMonth]));

					MWSendClient(Client, "\">", 2);

					/* Now send the actual Month Name */
					MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[2] + CurrentMonth], strlen(Session->Strings[Token->ArgumentOffsetOrID[2] + CurrentMonth]));

					MWSendClient(Client, "</a>", 4);

					if (Token->ArgumentSize[3] > 0) {
						MWSendClient(Client, "</font>", 7);
					}

					MWSendClient(Client, "</td>\r\n", 7);
					CurrentMonth++;
				}
			}
			MWSendClient(Client, "</tr>", 5);
			return(TRUE);
		}

		case T_TODAY: {
			if (Token->ArgumentOffsetOrID[0]==AA_LONG) {
				ReplyInt=MsgPrint(Client->Temp, sizeof (Client->Temp), Session->DateFormatLong, time(NULL) + Session->TimezoneOffset, &Session->DateFormat);
			} else {
				ReplyInt=MsgPrint(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, time(NULL) + Session->TimezoneOffset, &Session->DateFormat);
			}
			MWSendClient(Client, Client->Temp, ReplyInt);
			return(TRUE);
		}

		case T_DATE_SELECTED: {	
			if (Token->ArgumentOffsetOrID[0]==AA_LONG) {
				ReplyInt=MsgPrint(Client->Temp, sizeof (Client->Temp), Session->DateFormatLong, MsgGetUTC(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear, 0, 0, 0), &Session->DateFormat);
			} else {
				ReplyInt=MsgPrint(Client->Temp, sizeof (Client->Temp), Session->DateFormatShort, MsgGetUTC(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear, 0, 0, 0), &Session->DateFormat);
			}
			MWSendClient(Client, Client->Temp, ReplyInt);
			return(TRUE);
		}

		case T_INSERTDAYSEL: {
			unsigned long	i = 1;
			unsigned long	Day = 0;
		
			if (Token->ArgumentOffsetOrID[0]==AA_TODAY) {
				MsgGetDMY(time(NULL) + Session->TimezoneOffset, &Day, NULL, NULL, NULL, NULL, NULL);
			} else if (Token->ArgumentOffsetOrID[0]==AA_DURATION) {
				if (CalendarSession->EndTime && CalendarSession->BeginTime) {
					Day = (CalendarSession->EndTime - CalendarSession->BeginTime) / (1440 * 60);
				}
				i = 0;	/* Start at 0 instead of 1 */
			} else {
				Day = Session->CurrentDay;
			}

			for (; i<32; i++) {
				if (i != Day) {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
				} else {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
				}
				MWSendClient(Client, Answer, ReplyInt);
			}

			return(TRUE);
		}

		case T_INSERTMONTHSEL: {
			unsigned long	i;
			unsigned long	Month;
		
			if (Token->ArgumentOffsetOrID[0]==AA_TODAY) {
				MsgGetDMY(time(NULL) + Session->TimezoneOffset, NULL, &Month, NULL, NULL, NULL, NULL);
			} else {
				Month=Session->CurrentMonth;
			}

			for (i=1; i<13; i++) {
				if (i != Month) {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%s</OPTION>", (int)i, Session->DateFormat.monthLong[i-1]);
				} else {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%s</OPTION>", (int)i, Session->DateFormat.monthLong[i-1]);
				}
				MWSendClient(Client, Answer, ReplyInt);
			}

			return(TRUE);
		}

		case T_INSERTYEARSEL: {
			unsigned long	i;
			unsigned long	Year;
		
			if (Token->ArgumentOffsetOrID[0]==AA_TODAY) {
				MsgGetDMY(time(NULL) + Session->TimezoneOffset, NULL, NULL, &Year, NULL, NULL, NULL);
			} else {
				Year=Session->CurrentYear;
			}

			for (i=2000; i<2031; i++) {
				if (i != Year) {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
				} else {
					ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
				}
				MWSendClient(Client, Answer, ReplyInt);
			}

			return(TRUE);
		}

		case T_INSERTDATESEL: {
			unsigned long	i;
			unsigned char	*ptr;
			unsigned long	Day;
			unsigned long	Month;
			unsigned long	Year;		

			switch (Token->ArgumentOffsetOrID[0]) {
				case AA_TODAY: {
					MsgGetDMY(time(NULL) + Session->TimezoneOffset, &Day, &Month, &Year, NULL, NULL, NULL);
					break;
				}

				case AA_BEGIN: {
					if (CalendarSession->BeginTime != 0) {
						MsgGetDMY(CalendarSession->BeginTime, &Day, &Month, &Year, NULL, NULL, NULL);
					} else {
						MsgGetDMY(time(NULL) + Session->TimezoneOffset, &Day, &Month, &Year, NULL, NULL, NULL);
					}
					break;
				}

				case AA_END: {
					if (CalendarSession->EndTime != 0) {
						MsgGetDMY(CalendarSession->EndTime, &Day, &Month, &Year, NULL, NULL, NULL);
					} else {
						MsgGetDMY(time(NULL) + Session->TimezoneOffset, &Day, &Month, &Year, NULL, NULL, NULL);
					}
					break;
				}

				default: {
					Day=Session->CurrentDay;
					Month=Session->CurrentMonth;
					Year=Session->CurrentYear;
					break;
				}
			}

			ptr = Session->DateFormatLong;			

			ModDebug("T_INSERTDATESEL: Date Format String = %s\n", ptr);

			while (*ptr != '\0') {
				switch (*ptr) {
					case '%': {								
						ptr++;
						switch (*ptr) {
							case 'm': {
								MWSendClient(Client, "<SELECT NAME=\"", 14);
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
								MWSendClient(Client, "\">", 2);

								for (i=1; i<13; i++) {
									if (i != Month) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%02d</OPTION>", (int)i, (int)i);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%02d</OPTION>", (int)i, (int)i);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}

								MWSendClient(Client, "</SELECT>", 9);
							
								break;
							}

							case 'B':
							case 'b': {

								MWSendClient(Client, "<SELECT NAME=\"", 14);
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
								MWSendClient(Client, "\">", 2);
								for (i=1; i<13; i++) {
									if (i != Month) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%s</OPTION>", (int)i, Session->DateFormat.monthLong[i-1]);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%s</OPTION>", (int)i, Session->DateFormat.monthLong[i-1]);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}

								MWSendClient(Client, "</SELECT>", 9);

								break;
							}

							case 'd': {
								MWSendClient(Client, "<SELECT NAME=\"", 14);
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
								MWSendClient(Client, "\">", 2);

								for (i=1; i<32; i++) {
									if (i != Day) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%d</OPTION>", (int)i, (int)i);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}
								MWSendClient(Client, "</SELECT>", 9);

								break;
							}

							case 'y': {

								MWSendClient(Client, "<SELECT NAME=\"", 14);
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
								MWSendClient(Client, "\">", 2);

								for (i=2000; i<2031; i++) {
									if (i != Year) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%02d</OPTION>", (int)i, (int)(i-2000));
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%02d</OPTION>", (int)i, (int)(i-2000));
									}
									MWSendClient(Client, Answer, ReplyInt);
								}
								MWSendClient(Client, "</SELECT>", 9);
							
								break;
							}

							case 'Y': {
								MWSendClient(Client, "<SELECT NAME=\"", 14);
								MWSendClient(Client, Token->Data+Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
								MWSendClient(Client, "\">", 2);

								for (i=2000; i<2031; i++) {
									if (i != Year) {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">%04d</OPTION>", (int)i, (int)i);
									} else {
										ReplyInt=snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">%04d</OPTION>", (int)i, (int)i);
									}
									MWSendClient(Client, Answer, ReplyInt);
								}
								MWSendClient(Client, "</SELECT>", 9);
						
								break;
							}

							case '%': {
								MWSendClient(Client, ptr, 1);
								break;
							}
						}	
						break;
					}

					default: {
						// MWSendClient(Client, ptr, 1);
						break;
					}	
				}
				ptr++;	
			}	
			return(TRUE);
		}

		case T_INSERTTIMESEL: {
			unsigned long	i;
			BOOL				Selected = FALSE;		
			long				Hour;
			long				Minute;

			ModDebug("T_INSERTTIMESEL: Time Format String = %s\n", Session->TimeFormat);

			switch (Token->ArgumentOffsetOrID[1]) {
				case AA_BEGIN: {
					if (CalendarSession->BeginTime != 0) {
						MsgGetDMY(CalendarSession->BeginTime, NULL, NULL, NULL, &Hour, &Minute, NULL);
					} else {
						MsgGetDMY(time(NULL) + Session->TimezoneOffset, NULL, NULL, NULL, &Hour, &Minute, NULL);
					}
					break;
				}

				case AA_END: {
					if (CalendarSession->EndTime != 0) {
						MsgGetDMY(CalendarSession->EndTime, NULL, NULL, NULL, &Hour, &Minute, NULL);
					} else {
						MsgGetDMY(time(NULL) + Session->TimezoneOffset, NULL, NULL, NULL, &Hour, &Minute, NULL);
					}
					break;
				}

				case AA_DURATION: {
					unsigned long			Length = 60;

					if (CalendarSession->BeginTime != 0 && CalendarSession->EndTime != 0) {
						Length = (CalendarSession->EndTime - CalendarSession->BeginTime) / 60;
						Length = Length % 1440;
					}

					/* Logic is different for duration so do it here */
					for (i = 15; i < 1440; i += Token->ArgumentOffsetOrID[0]) {
						
						if (!Selected && (i >= Length)) {
							ReplyInt = snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">", (int)i);
							Selected = TRUE;
						} else {
							ReplyInt = snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">", (int)i);
						}
						MWSendClient(Client, Answer, ReplyInt);

						ReplyInt = snprintf(Answer, sizeof(Answer), "%lu:%02lu", (i / 60), (i % 60));
						MWSendClient(Client, Answer, ReplyInt);

						MWSendClient(Client, "</OPTION>", 9);
					}
					
					return(TRUE);
				}

				case AA_TODAY:
				default: {
					MsgGetDMY(time(NULL) + Session->TimezoneOffset, NULL, NULL, NULL, &Hour, NULL, NULL);
					break;
				}
			}

			Minute += Hour * 60;

			for (i = 0; i < 1440; i += Token->ArgumentOffsetOrID[0]) {
				if (!Selected && i >= Minute) {
					ReplyInt = snprintf(Answer, sizeof(Answer), "<OPTION SELECTED VALUE=\"%d\">", (int)i);
					Selected = TRUE;
				} else {
					ReplyInt = snprintf(Answer, sizeof(Answer), "<OPTION VALUE=\"%d\">", (int)i);
				}
				MWSendClient(Client, Answer, ReplyInt);

				ReplyInt = MsgPrint(Answer, BUFSIZE + 1, Session->TimeFormat, i*60, &Session->DateFormat);
				MWSendClient(Client, Answer, ReplyInt);

				MWSendClient(Client, "</OPTION>", 9);
			}
			
			return(TRUE);
		}

		case T_PROCESSDATESEL: {
			MWEncodeURL(Session, Client->Temp, URL_TYPE_LINK, PROCESS_DATE_SELECTION, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			ModDebug("Sending PROCESSDATESEL to page id %d\n", (int)Token->ArgumentOffsetOrID[0]);
			MWSendClient(Client, Client->Temp, strlen(Client->Temp));
			return(TRUE);
		}

		/* Compose Handler */

		case T_CAL_COMPOSE_FORM: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, COMPOSE_FORM, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2], 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CAL_PROCESS_LIST: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, LIST_FORM, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[2], Session->CurrentCalendar);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CAL_PROCESS_COPY: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, COPY_FORM, Token->ArgumentOffsetOrID[1], Token->ArgumentOffsetOrID[0], 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		/*	Recurrence Tokens */

		case T_CAL_RECUR_FORM: {
			MWEncodeURL(Session, URL, URL_TYPE_LINK, RECUR_FORM, Token->ArgumentOffsetOrID[0], 0, 0, 0);
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CAL_ICS_LINK: {
            /* fixme - temporary hack for supressing webcal protocol from being returned to Windows hosted browsers. */
            if (Session->FolderList->Value[CalendarSession->CurrentCalendar][1] != '3') {
			    MWEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_ICS, 0, 0, 0, 0);
            } else if (MWQuickCmp(Session->FolderList->Value[CalendarSession->CurrentCalendar] + 3, "MAIN")) {
                sprintf(URL, "webcal://%s/Calendar/%s.ics", Client->URLHost, Session->User);
            } else {
                sprintf(URL, "webcal://%s/Calendar/%s/%s.ics", Client->URLHost, Session->User, Session->FolderList->Value[CalendarSession->CurrentCalendar] + 3);
            }

            MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CAL_REFRESH: {
			if (Token->ArgumentOffsetOrID[0] != 0) {
				MWEncodeURL(Session, URL, URL_TYPE_LINK, CAL_REFRESH, Token->ArgumentOffsetOrID[0], 0, 0, 0);
				MWSendClient(Client, URL, strlen(URL));
			} else {
				UpdateCalendar(Session, CalendarSession);
			}
			return(TRUE);
		}

		case T_CAL_ERROR: {
			if (CalendarSession->Error) {
				MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0]+CalendarSession->Error], strlen(Session->Strings[Token->ArgumentOffsetOrID[0]+CalendarSession->Error]));
				ReplyInt = snprintf(Answer, sizeof(Answer), " (%lu)", CalendarSession->Error); 
				MWSendClient(Client, Answer, ReplyInt);
				CalendarSession->Error=0;
			}
			return(TRUE);
		}

		/* Busy Search Tokens */
		case T_CAL_BUSY_DAYLIST_START: {
			/* Do nothing.  This token is just here for looping to */

			return(TRUE);
		}

		case T_CAL_BUSY_DAYLIST_END: {
			CalendarSession->DayCounter++;

			if (CalendarSession->DayCounter < 7) {
				/* We aren't done, so return to T_CAL_BUSY_DAYLIST_START */

				*GotoToken = T_CAL_BUSY_DAYLIST_START;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				CalendarSession->DayCounter = 0;

				return(TRUE);
			}
		}

		case T_CAL_BUSY_DAYLIST_NAME: {
			unsigned long			Day;
			unsigned long			Month;
			unsigned long			Year;

			MsgGetDMY(CalendarSession->BeginTime, &Day, &Month, &Year, NULL, NULL, NULL);
			MWSendClient(Client, Session->Strings[Token->ArgumentOffsetOrID[0] + ((MSG_DAY_OF_WEEK(MsgGetRataDie(Day, Month, Year)) + CalendarSession->DayCounter) % 7)], strlen(Session->Strings[Token->ArgumentOffsetOrID[0] + ((MSG_DAY_OF_WEEK(MsgGetRataDie(Day, Month, Year)) + CalendarSession->DayCounter) % 7)]));

			return(TRUE);
		}

		case T_CAL_BUSY_TIMELIST_START: {
			/* Do nothing.  This token is just here for looping to */

			if (CalendarSession->HourCounter == 0) {
				CalendarSession->HourCounter = 0; /* Plug in the "Begining of business hours" time here */
			}

			return(TRUE);
		}

		case T_CAL_BUSY_TIMELIST_END: {
			CalendarSession->HourCounter++;

			if (CalendarSession->HourCounter < 24 /* Plug in the "End of business hours" time here */) {
				/* We aren't done, so return to T_CAL_BUSY_TIMELIST_START */

				*GotoToken = T_CAL_BUSY_TIMELIST_START;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				CalendarSession->HourCounter = 0;

				return(TRUE);
			}
			return(TRUE);
		}

		case T_CAL_BUSY_TIMELIST_NAME: {
			unsigned long		DisplayTime = (CalendarSession->HourCounter > 12) ? (CalendarSession->HourCounter - 12) : CalendarSession->HourCounter;

			if (DisplayTime == 0) {
				DisplayTime = 12;
			}

			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", DisplayTime);
			MWSendClient(Client, Client->Temp, ReplyInt);

			return(TRUE);
		}

		case T_CAL_BUSY_TIMELIST_AMPM: {
			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "%s", (CalendarSession->HourCounter > 11) ? Session->DateFormat.AmPm[1] : Session->DateFormat.AmPm[0]);
			MWSendClient(Client, Client->Temp, ReplyInt);
			return(TRUE);
		}

		case T_CAL_BUSY_USERS_START: {
			int					ClientNMAPs			= 0;
			unsigned long		NMAPCAddress		= 0;
			unsigned long		UserNMAPAddress	= 0;
			unsigned char		Buffer[BUFSIZE + 1];
			unsigned long		BeginTime			= (CalendarSession->BeginTime == 0) ? (time(NULL) + Session->TimezoneOffset) : CalendarSession->BeginTime;
			unsigned long		Day;
			unsigned long		Month;
			unsigned long		Year;


			/*
				CalendarSession->BeginTime is in the local timezone.  We need a
				value in UTC, and we want it to be the begining of the day.  We
				strip this off while in the local timezone.  We then remove the
				timezone offset to get the UTC time.
			*/
			MsgGetDMY(BeginTime, &Day, &Month, &Year, NULL, NULL, NULL);
			BeginTime = MsgGetUTC(Day, Month, Year, 0, 0, 0);
			BeginTime -= MsgGetUTCOffsetByUTC(Session->TimezoneID, BeginTime);

			/*
				Do the search now
			*/
			if (!CalendarSession->FreeBusyData) {
				/* Allocate the cache */

				CalendarSession->FreeBusyData = MemMalloc(SearchCacheSlotSize);

				if (!CalendarSession->FreeBusyData) {
					/* Oops.  Lets just leave */
					*GotoToken = T_CAL_BUSY_USERS_END;
					return(TOKEN_MOVE_FORWARD);
				}
			}
			memset(CalendarSession->FreeBusyData, 0, SearchCacheSlotSize);

			if (!CalendarSession->FreeBusyTotalData) {
				/* Allocate the cache */

				CalendarSession->FreeBusyTotalData = MemMalloc(SearchCacheSlotSize);

				if (!CalendarSession->FreeBusyTotalData) {
					MemFree(CalendarSession->FreeBusyData);
					CalendarSession->FreeBusyData = NULL;

					/* Oops.  Lets just leave */

					*GotoToken = T_CAL_BUSY_USERS_END;
					return(TOKEN_MOVE_FORWARD);
				}
				memset(CalendarSession->FreeBusyTotalData, 0, SearchCacheSlotSize);
			}

			if (!CalendarSession->FreeBusyUsers) {
				unsigned long		i;
				FILE					*Users;

				CalendarSession->FreeBusyUsers = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);

				/*
					Find all the users to search

					Always put ourself in.
				*/

//				MDBAddValue(Session->User, CalendarSession->FreeBusyUsers);
				for (i = 0; i < 3; i++) {
					snprintf(Answer, sizeof(Answer), "%s/%x.%s", MwCal.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
					Users = fopen(Answer, "rb");

					if (!Users) {
						return(FALSE);
					}

					while (!feof(Users) && !ferror(Users)) {
						if (fgets(Answer, MAXEMAILNAMESIZE, Users)) {
							ChopNL(Answer);
							if (Answer[0]=='\0') {
								continue;
							}
							MDBAddValue(Answer, CalendarSession->FreeBusyUsers);
						} 
					}
					fclose(Users);
				}
			}

			while (CalendarSession->FreeBusyUsers->Used > 0 && UserNMAPAddress == 0) {
				unsigned char		*AtPtr = NULL;

				/*
					Find the local domain they belong to, and strip off the domain.  If we
					don't find a match then we have to assume that they are a hosted user,
					and do the MsgFindObject with the domain.  If that fails as well then
					we know its not one of our users and we give up on them.

					Eventually we will be putting some method of looking up remote users here.
				*/
				AtPtr = strchr(CalendarSession->FreeBusyUsers->Value[0], '@');
				if (AtPtr) {
					unsigned char		i;

					for (i = 0; i < MwCal.Domains->Used && AtPtr; i++) {
						if (MWQuickCmp(AtPtr + 1, MwCal.Domains->Value[i])) {
							/*
								Found our domain
							*/
							*AtPtr = '\0';
						}
					}
				}

				MsgFindObject(CalendarSession->FreeBusyUsers->Value[0], NULL, NULL, (struct in_addr *)&UserNMAPAddress, NULL);
				if (UserNMAPAddress == 0) {
					MDBFreeValue(0, CalendarSession->FreeBusyUsers);
				}
			}

			if (CalendarSession->FreeBusyUsers->Used == 0) {
				/*
					Guess there are no local users after all so lets just bail
				*/
				*GotoToken = T_CAL_BUSY_USERS_END;
				return(TOKEN_MOVE_FORWARD);
			}

			/*
				Switch out our NMAP connection temporarily, and connect to the
				NMAP store of the user we are searching against.

				We have to be very careful to reset our Client information when
				we are done.
			*/

			/* Connect to NMAP server */
			if (UserNMAPAddress != 0) {
				unsigned char	*User;

				ClientNMAPs = Session->NMAPs;
				Session->NMAPs = -1;
				Session->NBufferPtr = 0;
				NMAPCAddress = Session->NMAPAddress.s_addr;
				Session->NMAPAddress.s_addr = UserNMAPAddress;
				User = Session->User;
				Session->User = CalendarSession->FreeBusyUsers->Value[0];
				if (!MWConnectUserToNMAPServer(Session)) {
					Session->User = User;
					RestoreUserNMAP;
					*GotoToken = T_CAL_BUSY_USERS_END;
					return(TOKEN_MOVE_FORWARD);
				}
				Session->User = User;
			}

			ReplyInt = snprintf(Answer, sizeof(Answer), "CSOPEN MAIN\r\n");
			MWSendNMAPServer(Session, Answer, ReplyInt);
			ReplyInt = MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
			if ((ReplyInt != 1000) && (ReplyInt != 1020)) {
				/* Failed to open */
				RestoreUserNMAP;

				*GotoToken = T_CAL_BUSY_USERS_END;
				return(TOKEN_MOVE_FORWARD);
			}

			ReplyInt = snprintf(Answer, sizeof(Answer), "CSFILT %lu %lu\r\n", BeginTime, BeginTime + (60 * 24 * 7 * 60));
			MWSendNMAPServer(Session, Answer, ReplyInt);

			while (MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE) == 2002) {
				unsigned long        seqNumber;
				unsigned long			Type;
				unsigned long			Start;
				unsigned long			End;
				unsigned long			i;
				long unsigned int State;

				sscanf(Answer, "%lu %lu %lu %lu %*u %*u %lu", &seqNumber, &Type, &Start, &End, &State);

				if (State & MSG_STATE_NOFREEBUSY) {
					/* Skip this entry because NO_FREEBUSY is set */
					continue;
				}

				/* we are an attendee */
				if (Start > BeginTime) {
					Start -= BeginTime;
				} else {
					Start = 0;
				}

				End -= BeginTime;
				if (End > (60 * 24 * 7 * 60)) {
					End = (60 * 24 * 7 * 60);
				}

				Start = Start / 60;
				End = End / 60;
					
				/* Type 1 == Appointment.  We don't care about tasks and notes */
				if (Type == 1) {
					for (i = Start; i < End && i < (60 * 24 * 7); i++) {
						SetFreeBusyBusyBit(CalendarSession->FreeBusyData, i, TRUE);
						SetFreeBusyBusyBit(CalendarSession->FreeBusyTotalData, i, TRUE);
					}
				}
			}

			/* This takes care of closing 1st connection*/
			RestoreUserNMAP;
			return(TRUE);
		}

		case T_CAL_BUSY_USERS_END: {
			if (CalendarSession->FreeBusyUsers->Used > 0) {
				MDBFreeValue(0, CalendarSession->FreeBusyUsers);
			}

			if (CalendarSession->FreeBusyUsers->Used > 0) {
				/* We have user's left */

				*GotoToken = T_CAL_BUSY_USERS_START;
				return(TOKEN_MOVE_BACKWARD);
			}

			if (CalendarSession->FreeBusyData) {
				MemFree(CalendarSession->FreeBusyData);
				CalendarSession->FreeBusyData = NULL;				
			}

			if (CalendarSession->FreeBusyUsers) {
				MDBDestroyValueStruct(CalendarSession->FreeBusyUsers);
				CalendarSession->FreeBusyUsers = NULL;
			}

			return(TRUE);
		}

		case T_CAL_BUSY_USERS_NAME: {
			if (CalendarSession->FreeBusyUsers->Used > 0) {
				MWSendClient(Client, CalendarSession->FreeBusyUsers->Value[0], strlen(CalendarSession->FreeBusyUsers->Value[0]));
			}
			return(TRUE);
		}

		case T_CAL_BUSY_DATA_START: {

			return(TRUE);
		}

		case T_CAL_BUSY_DATA_END: {
			CalendarSession->FreeBusySegmentStart = CalendarSession->FreeBusySegmentEnd + 1;

			if (CalendarSession->FreeBusySegmentStart < (60 * 24 * 7)) {
				/* We aren't to the end of the week.  Lets keep going */

				*GotoToken = T_CAL_BUSY_DATA_START;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				CalendarSession->FreeBusySegmentStart = 0;
				CalendarSession->FreeBusySegmentEnd = 0;

				return(TRUE);
			}
		}

		case T_CAL_BUSY_DATA_DURATION: {
			unsigned long		i = CalendarSession->FreeBusySegmentStart;
			BOOL					Busy;
			BOOL					inAppointment;
			unsigned long		beginDay;
			unsigned long		beginHour;
			unsigned long		beginMinute;
			unsigned long		endDay;
			unsigned long		endHour;
			unsigned long		endMinute;

			Busy = GetFreeBusyBusyBit(CalendarSession->FreeBusyData, CalendarSession->FreeBusySegmentStart);
			/*
				We want to show the times for the appointment in different colours.
			*/
			MsgGetDMY(CalendarSession->BeginTime, &beginDay, NULL, NULL, &beginHour, &beginMinute, NULL);
			MsgGetDMY(CalendarSession->EndTime, &endDay, NULL, NULL, &endHour, &endMinute, NULL);

			/* The search always starts on the same day */
			beginMinute += (beginHour * 60);
			endMinute += (endHour * 60) + ((endDay - beginDay) * 24 * 60);

			inAppointment = ((i >= beginMinute) && (i < endMinute));

			while ((i < (60 * 24 * 7)) &&
					(GetFreeBusyBusyBit(CalendarSession->FreeBusyData, i) == Busy) &&
					(((i >= beginMinute) && (i < endMinute)) == inAppointment)
					) {
				i++;
			}
			CalendarSession->FreeBusySegmentEnd = i - 1;

			i -= CalendarSession->FreeBusySegmentStart;

			if (Token->ArgumentOffsetOrID[0] > 0) {
				i = i / Token->ArgumentOffsetOrID[0];
			}

			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", i);
			MWSendClient(Client, Client->Temp, ReplyInt);
			return(TRUE);
		}

		case T_CAL_BUSY_DATA_FBTEXT: {

			return(TRUE);
		}
		
		case T_CAL_BUSY_DATA_FBIMAGE: {
			unsigned long		beginDay;
			unsigned long		beginHour;
			unsigned long		beginMinute;
			unsigned long		endDay;
			unsigned long		endHour;
			unsigned long		endMinute;

			/*
				We want to show the times for the appointment in different colours.
			*/
			MsgGetDMY(CalendarSession->BeginTime, &beginDay, NULL, NULL, &beginHour, &beginMinute, NULL);
			MsgGetDMY(CalendarSession->EndTime, &endDay, NULL, NULL, &endHour, &endMinute, NULL);

			/* The search always starts on the same day */
			beginMinute += (beginHour * 60);
			endMinute += (endHour * 60) + ((endDay - beginDay) * 24 * 60);

			if ((CalendarSession->FreeBusySegmentStart >= beginMinute) && (CalendarSession->FreeBusySegmentStart < endMinute)) {
				if (GetFreeBusyBusyBit(CalendarSession->FreeBusyData, CalendarSession->FreeBusySegmentStart)) {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[3], 0);
				} else {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
				}
			} else {
				if (GetFreeBusyBusyBit(CalendarSession->FreeBusyData, CalendarSession->FreeBusySegmentStart)) {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
				} else {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
				}
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}


		case T_CAL_BUSY_TOTALDATA_START: {

			return(TRUE);
		}

		case T_CAL_BUSY_TOTALDATA_END: {
			CalendarSession->FreeBusySegmentStart = CalendarSession->FreeBusySegmentEnd + 1;

			if (CalendarSession->FreeBusySegmentStart < (60 * 24 * 7)) {
				/* We aren't to the end of the week.  Lets keep going */

				*GotoToken = T_CAL_BUSY_TOTALDATA_START;
				return(TOKEN_MOVE_BACKWARD);
			} else {
				CalendarSession->FreeBusySegmentStart = 0;
				CalendarSession->FreeBusySegmentEnd = 0;

				if (CalendarSession->FreeBusyTotalData) {
					MemFree(CalendarSession->FreeBusyTotalData);
					CalendarSession->FreeBusyTotalData = NULL;
				}

				return(TRUE);
			}
		}

		case T_CAL_BUSY_TOTALDATA_DURATION: {
			unsigned long		i = CalendarSession->FreeBusySegmentStart;
			BOOL					Busy;
			BOOL					inAppointment;
			unsigned long		beginDay;
			unsigned long		beginHour;
			unsigned long		beginMinute;
			unsigned long		endDay;
			unsigned long		endHour;
			unsigned long		endMinute;

			Busy = GetFreeBusyBusyBit(CalendarSession->FreeBusyTotalData, CalendarSession->FreeBusySegmentStart);
			/*
				We want to show the times for the appointment in different colours.
			*/
			MsgGetDMY(CalendarSession->BeginTime, &beginDay, NULL, NULL, &beginHour, &beginMinute, NULL);
			MsgGetDMY(CalendarSession->EndTime, &endDay, NULL, NULL, &endHour, &endMinute, NULL);

			/* The search always starts on the same day */
			beginMinute += (beginHour * 60);
			endMinute += (endHour * 60) + ((endDay - beginDay) * 24 * 60);

			inAppointment = ((i >= beginMinute) && (i < endMinute));

			while ((i < (60 * 24 * 7)) &&
					(GetFreeBusyBusyBit(CalendarSession->FreeBusyTotalData, i) == Busy) &&
					(((i >= beginMinute) && (i < endMinute)) == inAppointment)
					) {
				i++;
			}
			CalendarSession->FreeBusySegmentEnd = i - 1;

			i -= CalendarSession->FreeBusySegmentStart;

			if (Token->ArgumentOffsetOrID[0] > 0) {
				i = i / Token->ArgumentOffsetOrID[0];
			}

			ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", i);
			MWSendClient(Client, Client->Temp, ReplyInt);
			return(TRUE);
		}

		case T_CAL_BUSY_TOTALDATA_FBTEXT: {

			return(TRUE);
		}
		
		case T_CAL_BUSY_TOTALDATA_FBIMAGE: {
			unsigned long		beginDay;
			unsigned long		beginHour;
			unsigned long		beginMinute;
			unsigned long		endDay;
			unsigned long		endHour;
			unsigned long		endMinute;

			/*
				We want to show the times for the appointment in different colours.
			*/
			MsgGetDMY(CalendarSession->BeginTime, &beginDay, NULL, NULL, &beginHour, &beginMinute, NULL);
			MsgGetDMY(CalendarSession->EndTime, &endDay, NULL, NULL, &endHour, &endMinute, NULL);

			/* The search always starts on the same day */
			beginMinute += (beginHour * 60);
			endMinute += (endHour * 60) + ((endDay - beginDay) * 24 * 60);

			if ((CalendarSession->FreeBusySegmentStart >= beginMinute) && (CalendarSession->FreeBusySegmentStart < endMinute)) {
				if (GetFreeBusyBusyBit(CalendarSession->FreeBusyTotalData, CalendarSession->FreeBusySegmentStart)) {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[3], 0);
				} else {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[2], 0);
				}
			} else {
				if (GetFreeBusyBusyBit(CalendarSession->FreeBusyTotalData, CalendarSession->FreeBusySegmentStart)) {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
				} else {
					MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
				}
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);

			if (GetFreeBusyBusyBit(CalendarSession->FreeBusyTotalData, CalendarSession->FreeBusySegmentStart)) {
				MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[1], 0);
			} else {
				MWEncodeURL(Session, URL, URL_TYPE_IMAGE, DISPLAY_IMAGE, Session->TemplateID, Session->Language, Token->ArgumentOffsetOrID[0], 0);
			}
			MWSendClient(Client, URL, strlen(URL));
			return(TRUE);
		}

		case T_CAL_BUTTON_BEGIN: {
			ICalVAttendee		*Attendee = NULL;
			ICalObject     *ICal = NULL;
			/* Make sure that the calendar and calendar item we think we have cached are in fact cached */
			if (CalendarSession->CurrentCalendar != CalendarSession->DetailCalendar) {
				SelectCalendar(CalendarSession->DetailCalendar, Session, CalendarSession);
			}
			LoadCalendarEntryDetails(CalendarSession->DetailID, Session, CalendarSession, FALSE);
			if (!(ICal = CalendarSession->ICal)) {
				*GotoToken = T_CAL_BUTTON_END;
				return(TOKEN_MOVE_FORWARD);
			}
			if (ICal) {
				for (Attendee = ICal->Attendee;Attendee; Attendee = Attendee->Next) {
					if (MWQuickCmp(Session->EMailAddress, Attendee->Address)) {
						break; 
					}
				}
			}

			if (Attendee) {

				/* This token must either return TRUE if ICal object is in correct state for AcceptButton to be present */
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_SHOWBUSY:
						/* Show Appointment as Busy */
						/* We must not currently be busy */
						if (CalendarSession->EntryState & MSG_STATE_NOFREEBUSY)
							{
								return(TRUE);
							} else {
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
							}
						break;
					case AA_SHOWFREE:
						/* Show Appointment as Free */
						/* We must currently be busy */
						if (!(CalendarSession->EntryState & MSG_STATE_NOFREEBUSY))
							{
								return(TRUE);
							} else {
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
							}
						break;

					case AA_ACCEPT:
						switch(Attendee->State) {
							case ICAL_PARTSTAT_DECLINED: /* Declined */
							case ICAL_PARTSTAT_DELEGATED: /* Delegated */
							case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
								return(TRUE);
							default: /* inactive icon */
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
						}
						break;
					case AA_DECLINE:
						switch(Attendee->State) {
							case ICAL_PARTSTAT_ACCEPTED: /* Accepted */
							case ICAL_PARTSTAT_COMPLETED: /* Completed */
							case ICAL_PARTSTAT_DELEGATED: /* Delegated */
							case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
								return(TRUE);
							default: /* inactive icon */
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
						}
						break;

					case AA_DELEGATE:
						if ((ICal->Type == ICAL_VEVENT) ||
							 (ICal->Type == ICAL_VTODO)) {
							switch(Attendee->State) {
								case ICAL_PARTSTAT_ACCEPTED: /* Accepted */
								case ICAL_PARTSTAT_COMPLETED: /* Completed */
								case ICAL_PARTSTAT_DECLINED: /* Declined */
								case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
									return(TRUE);
								default: /* inactive icon */
									*GotoToken = T_CAL_BUTTON_END;
									return(TOKEN_MOVE_FORWARD);
							}
							break;
						} else {
							*GotoToken = T_CAL_BUTTON_END;
							return(TOKEN_MOVE_FORWARD);
						}


					case AA_COMPLETE:{
						if (ICal->Type == ICAL_VTODO) {
							switch(Attendee->State) {
								case ICAL_PARTSTAT_ACCEPTED: /* Accepted */
								case ICAL_PARTSTAT_DECLINED: /* Declined */
								case ICAL_PARTSTAT_DELEGATED: /* Delegated */
								case ICAL_PARTSTAT_NEEDS_ACTION: /* Needs attention */
									return(TRUE);
								default: /* inactive icon */
									*GotoToken = T_CAL_BUTTON_END;
									return(TOKEN_MOVE_FORWARD);
							}

						} else {
							*GotoToken = T_CAL_BUTTON_END;
							return(TOKEN_MOVE_FORWARD);
						}
					}

					default:{
						*GotoToken = T_CAL_BUTTON_END;
						return(TOKEN_MOVE_FORWARD);
					}
				}
			}else {
				/* handle appointments differently is originator is not attendee */
				/* only allow show, busy and show free buttons to be active */
				switch (Token->ArgumentOffsetOrID[0]) {
					case AA_SHOWBUSY:
						/* Show Appointment as Busy */
						/* We must not currently be busy */
						if (CalendarSession->EntryState & MSG_STATE_NOFREEBUSY){
								return(TRUE);
							} else {
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
							}
						break;
					case AA_SHOWFREE:
						/* Show Appointment as Free */
						/* We must currently be busy */
						if (!(CalendarSession->EntryState & MSG_STATE_NOFREEBUSY)){
							return(TRUE);
						} else {
								*GotoToken = T_CAL_BUTTON_END;
								return(TOKEN_MOVE_FORWARD);
							}
						break;
					default:{
						*GotoToken = T_CAL_BUTTON_END;
						return(TOKEN_MOVE_FORWARD);
					}
				}

			}
		}


		case T_CAL_BUTTON_END: {
			/* placeholder only */
			return(TRUE);
		}
			
		case	T_CAL_ISBCC_BEGIN: {

			/* Ways to be BCC */
			/* 1. Originator places himself in BCC field. */
			/*   State - IAMAttendee == TRUE, role == NOT-PARTICIPANTG; */
			/* 2. Originator places another recipient in BCC field. */
			/*    State at originator IAMAttendee == TRUE, role == NOT-PARTICIPATING; */
			/*    State at recipient  IAMAttendee == FALSE, role == NOT-PARTICIPATING; */
			/* 3. Originator doesn't place himself in any field. */
         /*    State at originator IAMAttendee == FALSE role == '\0x' */ 

			ICalObject *ICal = CalendarSession->ICal;
			
			/* This token must return true if I am a bcc */
			/* This is determined by not being in the attendee list of my object. */
			if (IsBCCAttendee(ICal, Session) || 
				 (ICal && ICal->Organizer && !CalendarSession->IAmAttendee && !MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address))){
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_ISBCC_END */
				*GotoToken = T_CAL_ISBCC_END;
				return(TOKEN_MOVE_FORWARD);
			}
			
		}

		case T_CAL_ISBCC_END: {
			/* This token is placeholder only */
			return(TRUE);
		}

		case	T_CAL_ISBCC_ATTENDEE_BEGIN: {
			ICalObject *ICal = CalendarSession->ICal;

			/* This token must either return TRUE if AmrecipientICalObject returns FALSE or move to end token */
			if (IsBCCAttendee(ICal,Session)) {
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_ISBCC_ATTENDEE_END */
				*GotoToken = T_CAL_ISBCC_ATTENDEE_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
		case T_CAL_ISBCC_ATTENDEE_END: {
			return(TRUE);
		}

		case	T_CAL_ISBCC_RECIP_BEGIN: {
			ICalObject *ICal = CalendarSession->ICal;

			/* This token must either return TRUE if AmrecipientICalObject returns FALSE or move to end token */
			if (ICal && ICal->Organizer && !CalendarSession->IAmAttendee && !MWQuickCmp(Session->EMailAddress, ICal->Organizer->Address)){
				/* just return and let intermediate tokens execute  */
				return(TRUE);
			} else {
				/* move token to T_CAL_ISBCC_RECIP_END */
				*GotoToken = T_CAL_ISBCC_RECIP_END;
				return(TOKEN_MOVE_FORWARD);
			}
		}
		case T_CAL_ISBCC_RECIP_END: {
			return(TRUE);
		}

		case T_CAL_BCC_RECIP_STATUS: {
			/* This token assumes that T_CAL_ISBCC_RECIP_BEGIN  has been called to verify that we are a BCC recipient */
			/* It then returns the status cached on the CalendarSession for this user */
			ptr = NULL;

			switch(CalendarSession->EntryState) {
				case 'N': ptr=Session->Strings[Token->ArgumentOffsetOrID[0]]; break;
				case 'A': ptr=Session->Strings[Token->ArgumentOffsetOrID[0]+1]; break;
				case 'D': ptr=Session->Strings[Token->ArgumentOffsetOrID[0]+2]; break;
				case 'C': ptr=Session->Strings[Token->ArgumentOffsetOrID[0]+5]; break;
					
					/* Delegated, Tentative, Inprocess don't make sense */
				case 'T':
				case 'G':
				case 'I':
				default: 
					ptr=Session->Strings[Token->ArgumentOffsetOrID[0]];
					break;
					
			}

			if (ptr)	{
				MWSendClient(Client, ptr, strlen(ptr));
			}

			return(TRUE);
		}

	}


	return(FALSE);
}



BOOL
MWCALHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData)
{
	CalendarSessionStruct	*CalendarSession=(CalendarSessionStruct	*)ModuleData;

	ModDebug("MWCALHandleURL called\n");
	switch(URL->Request) {
		case PROCESS_DATE_SELECTION: {		/* Read the form data containing the new date/time and display the template again */
			ModDebug("Received PROCESS_DATE_SELECTION Request, URL->arg[0]:%d\n", (int)URL->Argument[0]);
			ProcessDateSelection(Client, Session, (CalendarSessionStruct *)ModuleData);
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case COMPOSE_FORM: {						/* Handles compose form for iCal objects */
			ModDebug("Received COMPOSE_FORM Request\n");
			ProcessCalComposeForm(Client, Session, CalendarSession, &URL->Argument[0], URL->Argument[1]);		  /* New Template returned in &URL->Argument[0] */
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case LIST_FORM: {							/* Allows accept/decline/delete/complete from day/week view */
			SelectCalendar(URL->Argument[3], Session, CalendarSession);
		
			MWHandleTemplate(Client, Session, URL->Argument[ProcessCalListForm(Client, Session, CalendarSession)]);
			return(TRUE);
		}

		case COPY_FORM: {
			ProcessCalendarMessageMove(Client, Session, CalendarSession, URL->Argument[1]);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case RECUR_FORM: {						/* Creates RRULE for Ical object */
			ModDebug("Received RECUR_FORM Request\n");
			ProcessRecurForm(Client, Session);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case CAL_REFRESH: {
			UpdateCalendar(Session, CalendarSession);
			MWHandleTemplate(Client, Session, URL->Argument[0]);	 
			return(TRUE);
		}

		case REQUEST_CHANGE_DATE: {
			signed long		Day=(signed long)URL->Argument[1];
			signed long		Month=(signed long)URL->Argument[2];

			Session->CurrentMonth += Month;
			if (Session->CurrentMonth<1) {
				Session->CurrentYear+=((Session->CurrentMonth - 1)/12)-1;
				Session->CurrentMonth = 12 - (((Session->CurrentMonth-1)*-1) - 1) % 12;
			} else if (Session->CurrentMonth>12) {
				Session->CurrentYear+=(Session->CurrentMonth - 1)/12;
				Session->CurrentMonth = ((Session->CurrentMonth - 1) % 12) + 1;
			}

			if (Session->CurrentDay > MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear)) {
				Session->CurrentDay = MSG_DAYS_IN_MONTH(Session->CurrentMonth, Session->CurrentYear);
			}

			Session->CurrentRataDie=MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);

			if (Day!=0) {
				Session->CurrentRataDie+=Day;
				MsgGetDate(Session->CurrentRataDie, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear);
			}

			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_SELECT_TODAY: {
			MsgGetDMY(time(NULL)+Session->TimezoneOffset, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear, 0, 0 ,0);
			Session->CurrentRataDie=MsgGetRataDie(Session->CurrentDay, Session->CurrentMonth, Session->CurrentYear);
			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_SET_CURRENT_DATE: {
			Session->CurrentRataDie=URL->Argument[1];

			MsgGetDate(Session->CurrentRataDie, &Session->CurrentDay, &Session->CurrentMonth, &Session->CurrentYear);

			MWHandleTemplate(Client, Session, URL->Argument[0]);
			return(TRUE);
		}

		case REQUEST_VIEW_OBJECT: {
			CalendarSession->InView = 0;

			/* Make sure that the calendar and calendar item we think we have cached are in fact cached */
			if (CalendarSession->CurrentCalendar != URL->Argument[1]) {
				SelectCalendar(URL->Argument[1], Session, CalendarSession);
			}

			LoadCalendarEntryDetails(URL->Argument[0], Session, CalendarSession, TRUE);
			MWHandleTemplate(Client, Session, URL->Argument[2]);
			return(TRUE);
		}

		case REQUEST_DISPLAY_CALENDAR: {
			SelectCalendar(URL->Argument[0], Session, CalendarSession);

			MWHandleTemplate(Client, Session, URL->Argument[1]);
			return(TRUE);
		}

		case REQUEST_REPLY: {
			/* Make sure that the calendar and calendar item we think we have cached are in fact cached */
			if (CalendarSession->CurrentCalendar != URL->Argument[3]) {
				SelectCalendar(URL->Argument[3], Session, CalendarSession);
			}
		
			/* Send a reply for the message */
			LoadCalendarEntryDetails(URL->Argument[2], Session, CalendarSession, FALSE);

			if (ICalAction(Client, Session, CalendarSession, URL->Argument[0])) {
				UpdateCalendar(Session, CalendarSession);
				MWHandleTemplate(Client, Session, URL->Argument[1]);

			} 
			return(TRUE);
		}

		case REFRESH_CALENDAR:{
			UpdateCalendar(Session, CalendarSession);
			return(TRUE);
		}

		case REQUEST_ICS:{
		    if (CalendarSession->CalendarSelected) {
			MWSendIcs(Client, Session);
			return(TRUE);
		    }

		    SelectCalendar(0, Session, CalendarSession);
		    MWSendIcs(Client, Session);
		    return(TRUE);
		}
	}

	return(FALSE);
}


static BOOL 
ReadCalendarConfiguration(void)
{
	MDBValueStruct	*V;
	XplDir			*Dir;
	XplDir			*DirEnt;
	unsigned char	Path[XPL_MAX_PATH + 1];
	unsigned char	*ptr;

	V = MDBCreateValueStruct(MwCal.DirectoryHandle, MsgGetServerDN(NULL));

	/* Queue server */
	if (MsgReadIP(MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWCAL, MSGSRV_A_NMAP_QUEUE_SERVER, V)) {
		MwCal.NmapQAddress = inet_addr(V->Value[0]);
		if (MwCal.NmapQAddress == (unsigned long)-1) {
			MwCal.NmapQAddress = 0;
		}
	} else {
		MwCal.NmapQAddress = 0;
	}
	MDBFreeValues(V);

	/* Maximum Number of recipients */
	if (MDBRead(MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWCAL, MSGSRV_A_RECIPIENT_LIMIT, V)) {
		MwCal.MaxRecipients = atol(V->Value[0]);
	}

	MDBFreeValues(V);

	MsgGetWorkDir(MwCal.WorkDir);
	strncat(MwCal.WorkDir, "/modweb", sizeof(MwCal.WorkDir));
	MsgMakePath(MwCal.WorkDir);

	/* Clean up our temp dir */
	DirEnt = XplOpenDir(MwCal.WorkDir);
	if (DirEnt) {
		while((Dir = XplReadDir(DirEnt)) != NULL) {
			ptr=strchr(Dir->d_name, '.');
			if (ptr) {
				ptr++;
				if (MWQuickCmp(ptr, COMPOSE_EXT_SUBJECT) || MWQuickCmp(ptr, COMPOSE_EXT_BODY) || MWQuickCmp(ptr, COMPOSE_EXT_MESSAGE) || MWQuickCmp(ptr, COMPOSE_EXT_RRULE) ||
				MWQuickCmp(ptr, COMPOSE_EXT_TO_LIST[COMPOSE_TO]) || MWQuickCmp(ptr, COMPOSE_EXT_TO_LIST[COMPOSE_CC]) || MWQuickCmp(ptr, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]) ||
				isdigit(*ptr)) {
					snprintf(Path, sizeof(Path), "%s/%s", MwCal.WorkDir, Dir->d_name);
					unlink(Path);
				}
					
			}
		}
		XplCloseDir(DirEnt);
	}

	/*
		Read a list of the system domains so that we can tell if a user is local or not
	*/
	MDBSetValueStructContext(NULL, V);
	MwCal.Domains = MDBCreateValueStruct(MwCal.DirectoryHandle, NULL);

	MDBEnumerateObjects(MSGSRV_ROOT, MSGSRV_C_SERVER, NULL, V);
	while (V->Used) {
		MDBSetValueStructContext(V->Value[0], MwCal.Domains);
		MDBRead(MSGSRV_AGENT_SMTP, MSGSRV_A_DOMAIN, MwCal.Domains);

		MDBFreeValue(0, V);
	}

	MDBDestroyValueStruct(V);

	return(TRUE);
}

#define NUM_MWCAL_NAMED_URLS 1
NAMED_URL_ID MWCALNamedURLs[NUM_MWCAL_NAMED_URLS] = { {NM_REFRESH_CALENDAR, REFRESH_CALENDAR}};


/*
	This are the two required functions for a module. The naming
	must be the binary name plus "Init" / "Shutdown"
	Init is supposed to call ModWeb and register any modules that
	are contained within the binary.
	Shutdown must clear the module for unloading.
*/

EXPORT BOOL 
MWCALInit(MWAPIArg)
{
	ModuleRegisterStruct	Register;

	/* This relies on globals (MwCal.Running) being initialized to 0 implicitly */
	if (!MwCal.Running) {                            
	    MwCal.LogHandle = NULL;
	    MwCal.MaxRecipients = 0;
	    MwCal.Domains = NULL;

		XplSafeWrite(MwCal.LibraryUserCount, 0);

		MwCal.DirectoryHandle = (MDBHandle)MsgInit();
		if (MwCal.DirectoryHandle
) {
			MwCal.Running = TRUE;

  			MwCal.LogHandle = LoggerOpen("mwcal");
			if (MwCal.LogHandle == NULL) {
				XplConsolePrintf("MWCAL: Unable to initialize Nsure Audit.  Logging disabled.\r\n");
			}

			MWAPISet;

			Register.ModuleType = MODULE_TEMPLATE;
			Register.Module.Template.InitSession = MWCALInitSession;
			Register.Module.Template.DestroySession = MWCALDestroySession;
			Register.Module.Template.HandleURL = MWCALHandleURL;
			Register.Module.Template.HandleTemplate = MWCALHandleTemplate;
			Register.Module.Template.TokenRangeStart = MWCAL_TOKEN_START;
			Register.Module.Template.TokenRangeEnd = MWCAL_TOKEN_END;

			MWRegisterModule(&Register);

			MWRegisterNamedURLs(&MWCALNamedURLs[0], NUM_MWCAL_NAMED_URLS);

			MWAddStreamTemplateHandler("text/calendar", (TemplateHandlerFunc)MWCALHandleCalendarTemplate);

			ReadCalendarConfiguration();

			XplSafeIncrement(MwCal.LibraryUserCount);

			return(TRUE);
		}
	}

	return(FALSE);
}

EXPORT BOOL 
MWCALShutdown(void)
{
	XplSafeDecrement(MwCal.LibraryUserCount);

	if (MwCal.Running) {
		MwCal.Running = FALSE;

		/*	Make sure the library users are gone before beginning to 
			shutdown.	*/
		while (XplSafeRead(MwCal.LibraryUserCount)) {
			XplDelay(33);
		}

		/* Do any required cleanup */
		LoggerClose(MwCal.LogHandle);

		if (MwCal.Domains) {
			MDBDestroyValueStruct(MwCal.Domains);
			MwCal.Domains = NULL;
		}

#if defined(NETWARE) || defined(LIBC)
		XplSignalLocalSemaphore(MwCal.ShutdownSempaphore);	/* The signal will release main() */
		XplWaitOnLocalSemaphore(MwCal.ShutdownSempaphore);	/* The wait will wait until main() is gone */

		XplCloseLocalSemaphore(MwCal.ShutdownSempaphore);
#endif

		return(TRUE);
	}

	return(FALSE);
}


/*
	Below are "stock" functions that are basically infrastructure.
	However, one might want to add initialization code to main
	and takedown code to the signal handler
*/
void 
MWCALShutdownSigHandler(int Signal)
{
	int	oldTgid;

	oldTgid = XplSetThreadGroupID(MwCal.Tgid);

	/*	Make sure the library users are gone before beginning to 
		shutdown.	*/
	while (XplSafeRead(MwCal.LibraryUserCount) > 0) {
		XplDelay(33);
	}

	if (MwCal.Running) {
		MwCal.Running = FALSE;

		/* Do any required cleanup */
		LoggerClose(MwCal.LogHandle);

		if (MwCal.Domains) {
			MDBDestroyValueStruct(MwCal.Domains);
			MwCal.Domains = NULL;
		}

#if defined(NETWARE) || defined(LIBC)
		XplSignalLocalSemaphore(MwCal.ShutdownSempaphore);	/* The signal will release main() */
		XplWaitOnLocalSemaphore(MwCal.ShutdownSempaphore);	/* The wait will wait until main() is gone */

		XplCloseLocalSemaphore(MwCal.ShutdownSempaphore);
#endif
	}

	XplSetThreadGroupID(oldTgid);

	return;
}

int 
_NonAppCheckUnload(void)
{
	if (MwCal.Running) {
		XplConsolePrintf("\rThis NLM will automatically be unloaded by the thread that loaded it.\n");
		XplConsolePrintf("\rIt does not allow manual unloading.\n");
#if defined(NETWARE)	/* fixme - NetWare requirement */
		SetCurrentScreen(CreateScreen("System Console", 0));
#endif

		return(1);
	}

	return(0);
}

int
main(int argc, char *argv[])
{
	MwCal.Tgid = XplGetThreadGroupID();

	XplRenameThread(XplGetThreadID(), "ModWeb Calendar Module");

	XplOpenLocalSemaphore(MwCal.ShutdownSemaphore, 0);

	XplSignalHandler(MWCALShutdownSigHandler);

	/*
		This will "park" the module 'til we get unloaded; 
		it would not be neccessary to do this on NetWare, 
		but to prevent from automatically exiting on Unix
		we need to keep main around...
	*/
	XplWaitOnLocalSemaphore(MwCal.ShutdownSemaphore);
	XplSignalLocalSemaphore(MwCal.ShutdownSemaphore);

	return(0);
}
#else
BOOL
MWCALInitSession(SessionStruct *Session, void **ModuleData)
{
    return(TRUE);
}

BOOL
MWCALDestroySession(SessionStruct *Session, void *ModuleData)
{
    return(TRUE);
}

BOOL
MWCALHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData)
{
    return(TRUE);
}

BOOL
MWCALHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData)
{
    return(TRUE);
}

EXPORT BOOL MWCALInit(MWAPIArg)
{
    return(TRUE);
}

EXPORT BOOL MWCALShutdown(void)
{
    return(TRUE);
}

#endif
