/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>
#include <xplresolve.h>
#include <hulautil.h>

#if defined(USING_OPENLDAP)
#include <ldap/lber.h>
#include <ldap/ldap.h>
#endif

#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */

#include "mwmail.h"

#define StrDiv(Buffer)			{unsigned char *ptr; unsigned long field; ptr=Buffer; field=0; while (*ptr != '\0' && field<11) {if (*ptr == '\r') {*ptr ='\0';field++;} ptr++;}}


static int
CompareAddress(const void *f, const void *s)
{
	int						r				= 0;
	const unsigned char	*first		= *(unsigned char**)f;
	const unsigned char	*second		= *(unsigned char**)s;
	int						flen;
	int						slen;
	int						fi				= 0;
	int						si				= 0;
	unsigned char			fc;
	int						sc;
	const unsigned char	*ptr;
	#ifdef ADDRBOOK_SORT_DEBUG
		unsigned char			*debugp1;
		unsigned char			*debugp2;
	#endif

	if(!first || !second) {
		return(0);
	}

	ptr=first;
	while (ptr[0]!='\0') {			/* skipping unique number */
		ptr++;
	}
	ptr++;

	while (ptr[0]!='\0') {			/* skipping email address */
		ptr++;
	}
	ptr++;

	/* Calculate locations for first name */
	if (MwMail.SortFirstName) {
		first=ptr;
	}
	while (ptr[0]!='\0') {			/* skipping first name */
		ptr++;
	}
	#ifdef ADDRBOOK_SORT_DEBUG
		debugp1=ptr;
		*debugp1='-';
	#endif
	ptr++;

	if (!MwMail.SortFirstName) {
		first=ptr;
	}

	while (ptr[0]!='\0') {			/* skipping last name */
		ptr++;
	}
	flen=ptr-first;

	/* Calculate locations for second name */
	ptr=second;
	while (ptr[0]!='\0') {			/* skipping unique number */
		ptr++;
	}
	ptr++;

	while (ptr[0]!='\0') {			/* skipping email address */
		ptr++;
	}
	ptr++;

	/* Calculate locations for first name */
	if (MwMail.SortFirstName) {
		second=ptr;
	}

	while (ptr[0]!='\0') {			/* skipping first name */
		ptr++;
	}
	#ifdef ADDRBOOK_SORT_DEBUG
		debugp2=ptr;
		*debugp2='-';
	#endif
	ptr++;

	if (!MwMail.SortFirstName) {
		second=ptr;
	}

	while (ptr[0]!='\0') {			/* skipping last name */
		ptr++;
	}
	/* Now points to end of lastname */
	slen=ptr-second;

	#ifdef ADDRBOOK_SORT_DEBUG
		XplConsolePrintf("First  string:%s (%d)\n", first, flen);
		XplConsolePrintf("Second string:%s (%d)\n", second, slen);
		*debugp1='\0';
		*debugp2='\0';
	#endif

	while(fi < flen && si < slen && r == 0) {
		fc = toupper(first[fi]);
		sc = toupper(second[si]);
		if(fc == sc) {
			fi++;
			si++;
		} else if(fc < sc) {
			r = -1;
		} else {
			r = 1;
		}
	}

	if(r == 0) {
		if(flen < slen) {
			r = -1;
		} else if (flen > slen) {
			r = 1;
		}
	}

	return(r);
}


BOOL 
MwMailAddrBookSearch(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession) 
{
	MDBValueStruct	*LDAPConfig;
	unsigned long	i;
	unsigned long	FirstEntry;
	BOOL				DoingPublicSearch;
#if defined(USING_OPENLDAP)
	const unsigned char *LDAPAttrList[] = {"cn", "mail", "givenName", "surname", "sn", "telephoneNumber", "o", NULL};
#endif
	/* Search has been executed, Clean up addressbook artifacts in MailSession */
	MailSession->Index				= 0;
	MailSession->InAddressList		= NO;
	MailSession->FirstResultInList= 0;

	/* Clean up anything in the list structure */
	MDBFreeValues(MailSession->Value);

	/* Mark the ValueStruct as Address Book territory */		
	MailSession->ValueType = SESSION_VALUE_TYPE_ADDRESSES;

	if ((MailSession->SearchScope & (SEARCH_SYSTEM | SEARCH_PUBLIC)) && MailSession->SearchString) {
		LDAPConfig = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

		/* Loop for System & Public */
		if (!(MailSession->SearchScope & SEARCH_SYSTEM)) {
			DoingPublicSearch=TRUE;
		} else {
			DoingPublicSearch=FALSE;
		}

		SearchLDAPAgain:

		FirstEntry = MailSession->Value->Used;

		/* First, find which ldap servers we'll be using */
		if (!DoingPublicSearch) {
			ModDebug("Searching system addressbook\n");
			if (MsgGetUserFeature(Session->UserDN, FEATURE_MWMAIL_ADDRBOOK_S, MSGSRV_A_ADDRESSBOOK_URL_SYSTEM, LDAPConfig)==0) {
				MDBSetValueStructContext(MsgGetServerDN(NULL), LDAPConfig);
				HulaStrNCpy(Client->Temp, MsgGetServerDN(NULL), sizeof(Client->Temp));
				
				strncat(Client->Temp, "\\"MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWMAIL, sizeof(Client->Temp));
				MsgGetUserFeature(Client->Temp, FEATURE_MWMAIL_ADDRBOOK_S, MSGSRV_A_ADDRESSBOOK_URL_SYSTEM, LDAPConfig);
			}
		} else {
			ModDebug("Searching public addressbook\n");
			if (MsgGetUserFeature(Session->UserDN, FEATURE_MWMAIL_ADDRBOOK_G, MSGSRV_A_ADDRESSBOOK_URL_PUBLIC, LDAPConfig)==0) {
				MDBSetValueStructContext(MsgGetServerDN(NULL), LDAPConfig);
				HulaStrNCpy(Client->Temp, MsgGetServerDN(NULL), sizeof(Client->Temp));
				strcat(Client->Temp,  "\\"MSGSRV_AGENT_MODWEB"\\"MSGSRV_AGENT_MWMAIL);
				MsgGetUserFeature(Client->Temp, FEATURE_MWMAIL_ADDRBOOK_G, MSGSRV_A_ADDRESSBOOK_URL_PUBLIC, LDAPConfig);
			}
		}

#if defined(USING_OPENLDAP)
		if (LDAPConfig->Used>0) {
			MsgDNSRecord		*DNSARec		= NULL;
			LDAP				*LDAPServer	= NULL;
			LDAPMessage		*LDAPResult	= NULL;
			LDAPMessage		*LDAPEntry	= NULL;
			BerElement		*Ber;
			unsigned char	*LDAPAttribute;
			unsigned char	**LDAPValue;
			unsigned char	*BaseDN		= NULL;
			unsigned char	*Username	= "";
			unsigned char	*Password	= "";
			unsigned char	*Port;
			unsigned char	*Server;
			unsigned char	*prevptr;
			unsigned char	*Dest;
			int				Status;
			unsigned char	IPAddress[20];
			unsigned long	Address;
			int destSize;
			int count;

			for (i=0; i<LDAPConfig->Used; i++) {
				/* First, figure out the proper address; we're using our own resolver, it should be faster... */
				Server=LDAPConfig->Value[i];

				if (MWQuickNCmp(Server, "ldap://", 7)) {
					Server+=7;
				} else if (MWQuickNCmp(Server, "ldap:", 5)) {
					Server+=5;
				}

				/* Check if there's a username/password specified */
				if ((ptr=strchr(Server, '@'))!=NULL) {
					*ptr='\0';
					Username=Server;
					Server=ptr+1;
					if ((ptr=strchr(Username, ':'))!=NULL) {
						*ptr='\0';
						Password=ptr+1;
					}
				}

				if ((BaseDN=strchr(Server, '?'))!=NULL) {
					BaseDN[0]='\0';
					BaseDN++;
				} else {
					BaseDN=NULL;
				}

				if ((Port=strchr(Server, ':'))!=NULL) {
					Port[0]='\0';
					Port++;
				} else {
					Port=NULL;
				}

				/* Any trailing slashes? */
				if ((ptr=strchr(Server, '/'))!=NULL) {
					*ptr='\0';
				}

				if ((Address=inet_addr(Server))==-1) {
					Status=XplDnsResolve(Server, &DNSARec, XPL_RR_A);
					if (Status==XPL_DNS_SUCCESS) {
						memcpy(&Address,&DNSARec[0].A.addr, sizeof(struct in_addr));
						Address=htonl(Address);
						MemFree(DNSARec);
					}
				} else {
					Address=htonl(Address);
					Status=DNS_SUCCESS;
				}

				if (Status!=DNS_SUCCESS) {
					continue;
				}

				XplPrintIPAddress(IPAddress, sizeof(IPAddress), Address);
				ModDebug("Connecting to %s, port >%s< name:>%s<, password:>%s<\n", IPAddress, Port ? (char *)Port : "389", (char *)Username, Password);
				if ((LDAPServer=ldap_open(IPAddress, Port ? atol(Port) : LDAP_PORT))==NULL) {
					continue;
				}

				if (ldap_simple_bind_s(LDAPServer, Username, Password) != LDAP_SUCCESS) {
					ldap_unbind(LDAPServer);
					continue;
				}
				ModDebug("Have successful LDAP bind\n");

				/* Build our ldap query string in Client->Temp */
				strcpy(Client->Temp, "(|");
				prevptr=MailSession->SearchString;
				ptr=MailSession->SearchString;
				destSize = BUFSIZE-2;
				Dest=Client->Temp+2;
				
				while (*ptr) {
					switch (*ptr) {
						case ' ':
						case ';':
						case ',': {
							if (prevptr==ptr) {
								prevptr++;
								ptr++;
								continue;
							}

							/* We've got a token, pull it and build a search string */
							*ptr='\0';
							count=snprintf(Dest, destSize, "(cn=%s*)(sn=%s*)(givenName=%s*)(mail=%s*)", prevptr, prevptr, prevptr, prevptr);
							destSize -= count;
							Dest += count;
							
							*ptr=' ';
							ptr++;
							prevptr=ptr;
							continue;
						}

						default: {
							ptr++;
							continue;
						}
					}
				}

				if (prevptr!=ptr) {
					count=snprintf(Dest, destSize, "(cn=%s*)(sn=%s*)(givenName=%s*)(mail=%s*)", prevptr, prevptr, prevptr, prevptr);
					Dest += count;
					destSize -= count;
				}
				count = snprintf(Dest, destSize, ")");
				Dest += count;
				destSize -= count;

				ModDebug("Built search string %s\n", Client->Temp);
				/* Now perform the search */
				Status=ldap_search_s(LDAPServer, BaseDN, LDAP_SCOPE_SUBTREE, Client->Temp, (char **)LDAPAttrList, 0, &LDAPResult);
				if (Status != LDAP_SUCCESS && Status!=LDAP_SIZELIMIT_EXCEEDED && Status!=LDAP_TIMELIMIT_EXCEEDED) {
					ModDebug("Query failed\n");
					ldap_unbind(LDAPServer);
					LDAPServer=NULL;
					continue;
				}

				for (LDAPEntry=ldap_first_entry(LDAPServer, LDAPResult); LDAPEntry!=NULL; LDAPEntry=ldap_next_entry(LDAPServer, LDAPEntry)) {
					unsigned char	FirstName[65]		= "";
					unsigned char	LastName[65]		= "";
					unsigned char	FullName[131]		= "";
					unsigned char	EMailAddr[129]		= "";
					unsigned char	Phone[33]			= "";
					unsigned char	Organization[65]	= "";

					for (LDAPAttribute=ldap_first_attribute(LDAPServer, LDAPEntry, &Ber); LDAPAttribute!=NULL; LDAPAttribute=ldap_next_attribute(LDAPServer, LDAPEntry, Ber)) {
						LDAPValue=(unsigned char **)ldap_get_values(LDAPServer, LDAPEntry, LDAPAttribute);
						if (LDAPValue && LDAPValue[0]) {
							ModDebug("Found:%s=%s\n", LDAPAttribute, LDAPValue[0]);
							if (MWQuickCmp(LDAPAttribute, "cn")) {
							    HulaStrNCpy(FullName, LDAPValue[0], sizeof(FullName));
							} else if (MWQuickCmp(LDAPAttribute, "mail")) {
							    HulaStrNCpy(EMailAddr, LDAPValue[0], sizeof(EMailAddr));
							} else if (MWQuickCmp(LDAPAttribute, "givenName")) {
							    HulaStrNCpy(FirstName, LDAPValue[0], sizeof(FirstName));
							} else if (MWQuickCmp(LDAPAttribute, "sn") || MWQuickCmp(LDAPAttribute, "surname")) {
							    HulaStrNCpy(LastName, LDAPValue[0], sizeof(LastName));
							} else if (MWQuickCmp(LDAPAttribute, "telephoneNumber")) {
							    HulaStrNCpy(Phone, LDAPValue[0], sizeof(Phone));
							} else if (MWQuickCmp(LDAPAttribute, "o")) {
							    HulaStrNCpy(Organization, LDAPValue[0], sizeof(Organization));
							}
						}
						ldap_value_free((char **)LDAPValue);
					}
					/* Make up the first/last name or cn if none provided */
#if 0
					// We don't use Fullname...
					if (FullName[0]=='\0') {
					    snprintf(FullName, sizeof(FullName), "%s%s%s", FirstName, FirstName[0] != '\0' ? ", " : "", LastName);
					} else 
#endif
					if ((FirstName[0]=='\0') || (LastName[0]=='\0')) {
						ptr=strchr(FullName, ' ');
						if (ptr) {
							*ptr='\0';
							if (FirstName[0]=='\0') {
								HulaStrNCpy(FirstName, FullName, sizeof(FirstName));
							}
							if (LastName[0]=='\0') {
								HulaStrNCpy(LastName, ptr+1, sizeof(LastName));
							}
							*ptr=' ';
						}
					}
					/* Build the response string; we should add the organization */
					if (!DoingPublicSearch) {
						snprintf(Client->Temp, sizeof(Client->Temp), "Y-%s\r%s\r%s\r%s\r%s", Organization, EMailAddr, FirstName, LastName, Phone);
					} else {
						snprintf(Client->Temp, sizeof(Client->Temp), "U-%s\r%s\r%s\r%s\r%s", Organization, EMailAddr, FirstName, LastName, Phone);
					}
					MDBAddValue(Client->Temp, MailSession->Value);
					StrDiv(MailSession->Value->Value[MailSession->Value->Used-1]);
				}

				if (LDAPResult) {
					ldap_msgfree(LDAPResult);
					LDAPResult=NULL;
				}
				ldap_unbind(LDAPServer);
				LDAPServer=NULL;
			}							
		}
#endif

		/* Sort the results */
		if ((MailSession->Value->Used > FirstEntry) && MwMail.SortAddressBook) {
			qsort(MailSession->Value->Value + FirstEntry, MailSession->Value->Used - FirstEntry, sizeof(unsigned char *), CompareAddress);
		}

		if (!DoingPublicSearch && (MailSession->SearchScope & SEARCH_PUBLIC)) {
			DoingPublicSearch=TRUE;
			MDBFreeValues(LDAPConfig);
			goto SearchLDAPAgain;
		}

		MDBDestroyValueStruct(LDAPConfig);
	}

	/* Personal Address Book Search */
	if (MailSession->SearchScope & SEARCH_PERSONAL) {
		MDBValueStruct		*V;
		BOOL					Match;
		unsigned long		j;
		unsigned long		k;
		unsigned long		len;
		unsigned long		Vlen;
		unsigned long		Field;
		unsigned char		*TempBuffer = NULL;
		int bufLen;

		V = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

		MsgGetUserFeature(Session->UserDN, FEATURE_MWMAIL_ADDRBOOK_P, MSGSRV_A_ADDRESSBOOK, V);

		FirstEntry = MailSession->Value->Used;

		/* Show entire Personal Address Book if no search string is given */
		Match = MailSession->SearchString ? FALSE : TRUE;
		for (i = 0; i < V->Used; i++) { 
			if (!Match) {	 /* Don't enter this loop is search string is NULL */
				/* Search only the begining of strings, consistant with our LDAP search */
				len=strlen(MailSession->SearchString);
				Vlen=strlen(V->Value[i]);
				Field=0;
				for (j=0; j<Vlen; j++) {
					if (V->Value[i][j]=='\r') {
						Field++;
						if (Field<9) {
							/* Search regular field */
							k=0;
							while (k<len && k<Vlen && tolower(V->Value[i][j+1+k])==MailSession->SearchString[k]) {
								k++;
							}
							if (k==len) {
								/* Have match */
								goto FoundPersonal;
							}
						}
						continue;
					} else {
						if (Field>8) {
							/* Search comment */
							k=0;
							while (k<len && k<Vlen && tolower(V->Value[i][j+k])==MailSession->SearchString[k]) {
								k++;
							}
							if (k==len) {
								/* Have match */
								goto FoundPersonal;
							}
						}
					}
				}
				continue;
			}
			
FoundPersonal:
			bufLen = strlen(V->Value[i]) + 3;
			TempBuffer = MemRealloc(TempBuffer, bufLen);
			snprintf(TempBuffer, bufLen, "%c-%s", PERSONAL, V->Value[i]);
			MDBAddValue(TempBuffer, MailSession->Value);

			StrDiv(MailSession->Value->Value[MailSession->Value->Used-1]);
		}
		if (TempBuffer) {
			MemFree(TempBuffer);
		}

		MDBDestroyValueStruct(V);
		/* Sort the results */
		if ((MailSession->Value->Used > FirstEntry) && MwMail.SortAddressBook) {
			qsort(MailSession->Value->Value + FirstEntry, MailSession->Value->Used - FirstEntry, sizeof(unsigned char *), CompareAddress);
		}
	}
	return(TRUE);
}

BOOL
MwMailGetPersonalAddressBookEntry(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long ListIdx, unsigned long PersonalEntryID)
{
	unsigned char	PersonalEntryIDStr[ADDR_ENTRY_ID_LEN+1];
	unsigned long	len;

	snprintf(PersonalEntryIDStr, sizeof(PersonalEntryIDStr), "%x", (unsigned int)PersonalEntryID);

	if ((MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES) && (ListIdx < MailSession->Value->Used) && MWQuickNCmp(PersonalEntryIDStr, MailSession->Value->Value[ListIdx]+2, ADDR_ENTRY_ID_LEN)) {
		/* We still have it in the ValueStruct */
		MailSession->Index = ListIdx;				  
		return(TRUE);
	} else {
		MDBValueStruct					*V;
		unsigned long					i;
		BOOL								Match = FALSE;

		/* Clean up anything in the list structure */
		MDBFreeValues(MailSession->Value);

		/* Mark the ValueStruct as Address Book territory */		
		MailSession->ValueType	= SESSION_VALUE_TYPE_ADDRESSES;
		MailSession->Index		= 0;

		V = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);
		MsgGetUserFeature(Session->UserDN, FEATURE_MWMAIL_ADDRBOOK_P, MSGSRV_A_ADDRESSBOOK, V);

		/* Find the Entry */
		for (i = 0; i < V->Used; i++) {
			if (MWQuickNCmp(V->Value[i], PersonalEntryIDStr, ADDR_ENTRY_ID_LEN)) {
				Match = TRUE;	
				break;				
			}
		} 

		if (!Match) {
			MDBDestroyValueStruct(V);
			MailSession->Error =	ERROR_NO_ENTRY_FOUND;
			return(FALSE);
		}

		len=strlen(V->Value[i]);
		if (len > (BUFSIZE-3)) {
			unsigned char *TempBuffer;

			TempBuffer = MemMalloc(len+3);
			if (!TempBuffer) {
				MDBDestroyValueStruct(V);
				MailSession->Error =	ERROR_NOT_ENOUGH_MEMORY_TO_DISPLAY_ENTRY;
				return(FALSE);
			}
			snprintf(TempBuffer, len + 3, "%c-%s", PERSONAL, V->Value[i]);
			MDBAddValue(TempBuffer, MailSession->Value);			
		} else {
			snprintf(Client->Temp, sizeof(Client->Temp), "%c-%s", PERSONAL, V->Value[i]);
			MDBAddValue(Client->Temp, MailSession->Value);			
		}	

		StrDiv(MailSession->Value->Value[MailSession->Value->Used-1]);
		MDBDestroyValueStruct(V);
	}
	return(TRUE);
}



#define	MIN_VALUE_SIZE		512

BOOL
MwMailProcessAddrPersonalForm(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long PersonalEntryID, unsigned long *CurrentPage, unsigned long NextPage)
{
	unsigned char	FieldName[128];
	unsigned long	ValueSize;		
	unsigned long	PreviousPage;
	unsigned long	i;
	unsigned char	*FinalString;
	unsigned char	*TempArray[ADDR_MAX_FIELD_PERSONAL];
	unsigned long	ValueAllocated		= MIN_VALUE_SIZE;
	unsigned long	ValueUsed			= 0;
	unsigned char	*Value;
	MDBValueStruct	*V;	

	PreviousPage=*CurrentPage;

	memset(TempArray, 0, ADDR_MAX_FIELD_PERSONAL*sizeof(unsigned char *));

	Value = MemMalloc(sizeof(unsigned char)*(ValueAllocated+1));				/* One extra for terminating NULL */
	if (!Value) {
		return(FALSE);
	}

	while (MWGetFormName(Client, FieldName, 128)) {
		ValueUsed=0;
		do {
			/* Determine how much space is in the buffer */
			ValueSize = ValueAllocated-ValueUsed;

			/* Enough for us? */
			if (ValueSize<MIN_VALUE_SIZE) {
				Value=MemRealloc(Value, ValueAllocated+MIN_VALUE_SIZE);
				if (!Value) {
					return(FALSE);
				}
				ValueAllocated+=MIN_VALUE_SIZE;
				ValueSize=ValueAllocated-ValueUsed;
			}

			/* Read the data from the form */
			i=MWGetFormValue(Client, Value + ValueUsed, &ValueSize);
			ValueUsed+=ValueSize;

			/* Make sure we don't get more than we allow */
         if (ValueUsed >= MDB_MAX_ATTRIBUTE_VALUE_CHARS) {
            /* Toss remaining data */
            unsigned char	DummyBuffer[MIN_VALUE_SIZE];
            unsigned long	DummySize = MIN_VALUE_SIZE;
            do {
               i=MWGetFormValue(Client, DummyBuffer, &DummySize);
            } while (i == FORMFIELD_PARTIAL);
         }
      } while (i == FORMFIELD_PARTIAL);
      
		switch (toupper(FieldName[0])) {
			case 'B': {
				switch (toupper(FieldName[5])) {
					/* BirthDay */
					case 'D': {
						TempArray[AA_BIRTHDAY] = MemStrdup(Value);
						break;
					}

					/* BirthMonth */
					case 'M': {
						TempArray[AA_BIRTHMONTH] = MemStrdup(Value);
						break;
					}

					/* BirthYear */	
					case 'Y': {
						TempArray[AA_BIRTHYEAR] = MemStrdup(Value);
						break;
					}

					default: {
						break;
					}
				}
				break;
			} 

			/* Comment */
			case 'C': {
				TempArray[AA_COMMENT] = MemStrdup(Value);
				break;
			}

			/* EmailAddress */
			case 'E': {
				TempArray[AA_EMAILADDRESS] = MemStrdup(Value);
				break;
			}

			/* FirstName */
			case 'F': {
				TempArray[AA_FIRSTNAME] = MemStrdup(Value);
				break;
			}

			/* LastName */
			case 'L': {
				TempArray[AA_LASTNAME] = MemStrdup(Value);
				break;
			}

			case 'P': {
				switch (toupper(FieldName[5])) {
					/* Phone1 */
					case '1': {
						TempArray[AA_PHONE1] = MemStrdup(Value);
						break;
					}

					/* Phone2 */
					case '2': {
						TempArray[AA_PHONE2] = MemStrdup(Value);
						break;
					}

					case 'T': {
						switch (toupper(FieldName[9])) {
							/* PhoneType1 */
							case '1': {
								TempArray[AA_PHONETYPE1] = MemStrdup(Value);
								break;
							}

							/* PhoneType2 */
							case '2': {
								TempArray[AA_PHONETYPE2] = MemStrdup(Value);
								break;
							}
						}
					}

					default: {
						break;
					}
				}
				break;
			} 

			/* Save */
			case 'S': {
				*CurrentPage = NextPage;
				break;
			}

			/* Unknown FieldName */
			default: {
				break;
			}
		}
	}

	MemFree(Value);

	/* Build the string; first calculate how much space we need */
	ValueSize=9+ADDR_MAX_FIELD_PERSONAL;	/* for the ID\r and the \r for every field */
	for (i=0; i<ADDR_MAX_FIELD_PERSONAL; i++) {
		if (TempArray[i]) {
			ValueSize+=strlen(TempArray[i]);
		} else {
			ValueSize++;
		}
	}
   
	if (ValueSize > MDB_MAX_ATTRIBUTE_VALUE_CHARS) {
		MailSession->Error = ERROR_ADDR_BOOK_ENTRY_EXCEEDED_SIZE_LIMIT;
		while (i < ADDR_MAX_FIELD_PERSONAL) {
			if (TempArray[i]) {
				MemFree(TempArray[i]);
			}
			i++;
		}
		*CurrentPage = PreviousPage;
		return(FALSE);
	}


	FinalString = MemMalloc(ValueSize+1);

	snprintf(FinalString, ValueSize + 1, "%08x\r%s\r%s\r%s\r%s\r%s\r%s\r%s\r%s\r%s\r%s\r%s", 
		PersonalEntryID ? (unsigned int)PersonalEntryID : (unsigned int)time(NULL), 
		TempArray[AA_EMAILADDRESS] ? (char *)TempArray[AA_EMAILADDRESS] : "",
		TempArray[AA_FIRSTNAME] ? (char *)TempArray[AA_FIRSTNAME] : "",
		TempArray[AA_LASTNAME] ? (char *)TempArray[AA_LASTNAME] : "",
		TempArray[AA_PHONE1] ? (char *)TempArray[AA_PHONE1] : "",
		TempArray[AA_PHONETYPE1] ? (char *)TempArray[AA_PHONETYPE1] : "",
		TempArray[AA_PHONE2] ? (char *)TempArray[AA_PHONE2] : "",
		TempArray[AA_PHONETYPE2] ? (char *)TempArray[AA_PHONETYPE2] : "",
      TempArray[AA_BIRTHMONTH] ? (char *)TempArray[AA_BIRTHMONTH] : "",
		TempArray[AA_BIRTHDAY] ? (char *)TempArray[AA_BIRTHDAY] : "",
		TempArray[AA_BIRTHYEAR] ? (char *)TempArray[AA_BIRTHYEAR] : "",
		TempArray[AA_COMMENT] ? (char *)TempArray[AA_COMMENT] : "");

	for (i=0; i<ADDR_MAX_FIELD_PERSONAL; i++) {
		if (TempArray[i]) {
			MemFree(TempArray[i]);
		}
	}

	V = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

	if (PersonalEntryID != 0) {  
		MDBValueStruct	*VTmp;	

		VTmp = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

		/* We've got to find and remove the old entry */
		MDBRead(Session->UserDN, MSGSRV_A_ADDRESSBOOK, VTmp);
		for (i = 0; i < VTmp->Used; i++) {
			if (MWQuickNCmp(VTmp->Value[i], FinalString, ADDR_ENTRY_ID_LEN)) {
				MDBAddValue(FinalString, V);
			} else {
				MDBAddValue(VTmp->Value[i], V);				
			}
		}
		MDBWrite(Session->UserDN, MSGSRV_A_ADDRESSBOOK, V);

		MDBDestroyValueStruct(VTmp);

	} else {
		/* This is new; Just write it to DS */
		MDBAdd(Session->UserDN , MSGSRV_A_ADDRESSBOOK, FinalString, V);
	}

	MDBDestroyValueStruct(V);

	MemFree(FinalString);

	MwMailAddrBookSearch(Client, Session, MailSession);

	return(TRUE);
}


BOOL
MwMailProcessAddrResultForm(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long *NewPage, unsigned long Page1, unsigned long Page2, unsigned long Page3)
{
	FILE				*IDFile=NULL;
	FILE				*AddrFile=NULL;
	FILE				*RecipFile=NULL;
	unsigned char	IDPath[XPL_MAX_PATH+1];
	unsigned char	AddrPath[XPL_MAX_PATH+1];
	unsigned char	RecipPath[XPL_MAX_PATH+1];
	unsigned long	ValueSize;
	unsigned char	FieldName[128];
	unsigned long	CurrentPage;
	BOOL				AddRecipients			= FALSE;
	BOOL				DelPersonal				= FALSE;
	BOOL				ActionProcessed		= FALSE;
	unsigned long	Offset=0;
	unsigned char	*ptr;
	unsigned char	*StartPtr;
	unsigned char	*EndPtr;
	unsigned long	RetVal;

	IDPath[0]		= '\0';
	AddrPath[0]		= '\0';
	RecipPath[0]	= '\0';

	CurrentPage=*NewPage;	
	while (MWGetFormName(Client, FieldName, 128)) {
		ModDebug("Got Name:%s\n", FieldName);

		ValueSize=BUFSIZE-Offset;
		while ((RetVal = MWGetFormValue(Client, Client->Temp+Offset, &ValueSize)) != FORMFIELD_NEXT) {
			/* GetFormValue will return the entire value or a full buffer */
			ValueSize+=Offset;
			switch (toupper(FieldName[0])) {

				/* Entry Info */
				case 'E': {
					switch (toupper(FieldName[6])) {

						/* entrypErsonal */
						case 'E': {
							if (!AddrFile) {
								snprintf(AddrPath, sizeof(AddrPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, ADDRESS_FILE_EXTENSION);

								AddrFile=fopen(AddrPath,"wb");
								if (!AddrFile) {
									if (IDFile) {	
										fclose(IDFile);
										IDFile=NULL;
										unlink(IDPath);
									}
									unlink(AddrPath);
									MailSession->Error = ERROR_BUILDING_RECIP_LIST__FOPEN;
									*NewPage = CurrentPage;
									return(FALSE);
								}
							}

							if (!IDFile) {
								snprintf(IDPath, sizeof(IDPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, ID_FILE_EXTENSION);
								IDFile = fopen(IDPath,"wb");
								if (!IDFile) {
									if (AddrFile) {
										fclose(AddrFile);
							  			AddrFile=NULL;
										unlink(AddrPath);										
									}
									unlink(IDPath);
									MailSession->Error = ERROR_SAVING_FORM_DATA__FOPEN;
									*NewPage = CurrentPage;
									return(FALSE);
								}
							}

							ptr = Client->Temp;
							StartPtr = Client->Temp;
							EndPtr = Client->Temp + ValueSize;

							/* The first Personal Address Book buffer should look like    "XXXXXXXX-<Address1> <Address2> . . . <AddressN>" */
							/* If it does not all fit in one buffer, subsequent buffers should look like this "<AddressX> . . . <AddressN>" */
							/* The 7 digit XXXXXXX is the Personal Address Book Entry ID */
							if ((ValueSize > 7) && (Client->Temp[8] == '-') && isdigit(Client->Temp[0])) {
								/* We have an ID. This must be the first buffer */
								Client->Temp[8] = '\0';
								fprintf(IDFile, "%s\r\n", Client->Temp);
								ptr += 9;
								StartPtr += 9;
							}

							/* Store the Address in the AddrFile */
							/* People might have more than one address in a Personal AddrBook Entry */
							while (ptr<EndPtr) {
								switch (*ptr) {
									case ',':
									case ';':
									case ' ': {
										fwrite(StartPtr, ptr-StartPtr, 1, AddrFile);
										fwrite("\r\n", 2, 1, AddrFile);
										while (ptr<EndPtr && !isalnum(*ptr)) {
											ptr++;
										}
										StartPtr=ptr;
										continue;
									}

									default: {
										ptr++;
										continue;
									}
								}
							}

							if (ptr-StartPtr) {
								if (RetVal==FORMFIELD_DONE) {
									fwrite(StartPtr, ptr-StartPtr, 1, AddrFile);
									fwrite("\r\n", 2, 1, AddrFile);
									Offset=0;
								} else {
									/*
										An address cannot be larger than MAXEMAILNAMESIZE, if we get something bigger it's some
										jerk trying to break us. We ignore data longer than that
									*/
									if (strlen(StartPtr)<MAXEMAILNAMESIZE) {
										Offset=ptr-StartPtr;
										memmove(Client->Temp, StartPtr, Offset);
									} else {
										Offset=0;
									}
								}
							}

							break;
						}

						/* entrysYstem */
						/* entrypUblic */
						case 'U':
						case 'Y': {
							if (!AddrFile) {
								snprintf(AddrPath, sizeof(AddrPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, ADDRESS_FILE_EXTENSION);

								AddrFile=fopen(AddrPath,"wb");
								if (!AddrFile) {
									if (IDFile) {
										fclose(IDFile);
										IDFile=NULL;
										unlink(IDPath);										
									}
									unlink(AddrPath);

									MailSession->Error = ERROR_BUILDING_RECIP_LIST__FOPEN;
									* NewPage = CurrentPage;
									return(FALSE);
								}
							}

							if (ValueSize > 0) {
								fprintf(AddrFile, "%s\r\n", Client->Temp);
							}
							break;
						}
					}
					break;
				}

				/* Action Request */
				case 'A': {
					switch (toupper(FieldName[6])) {

						/* actionAddto */
						case 'A': {

							if (!ActionProcessed) {
								/* These actions increment variables, Make sure we process them only once */
								ActionProcessed	= TRUE;

								/* Ensures that the same search results will be displayed */
								if ((MailSession->ResultsPerPage > 0) && (MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES)) {
									MailSession->Index = MailSession->FirstResultInList;

									if (MailSession->Index >= MailSession->Value->Used) {
										MailSession->InAddressList = NO;
									}
								}

								switch (toupper(FieldName[11])) {

									/* actionaddtoTo */
									case 'T': {
										snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
										AddRecipients		= TRUE;
										break;
									}

									/* actionaddtoCc */
									case 'C': {
										snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
										AddRecipients		= TRUE;
										break;
									}

									/* actionaddtoBcc */
									case 'B': {
										snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
										AddRecipients		= TRUE;
										break;
									}
								}
							}
							break;
						}

						/* actionDelete */
						case 'D': {
							DelPersonal = TRUE;
							/* List will change, reset display */
							MailSession->Index = 0;
							MailSession->InAddressList = NO;
							break;
						}

						/* actionLoadpage<1|2|3> */
						case 'L': {

							switch(FieldName[14]) {
								/* actionloadpage1 */
								case '1': {
									if (Page1 > 0) {
										*NewPage = Page1;
										MailSession->Index = 0;
										MailSession->InAddressList = NO;
									}
									break;
								}
								/* actionloadpage2 */
								case '2': {
									if (Page2 > 0) {
										*NewPage = Page2;
										MailSession->Index = 0;
										MailSession->InAddressList = NO;
									}
									break;
								}
								/* actionloadpage3 */
								case '3': {
									if (Page3 > 0) {
										*NewPage = Page3;
										MailSession->Index = 0;
										MailSession->InAddressList = NO;
									}
									break;
								}
							}
							snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
							AddRecipients = TRUE;

							break;
						}

						/* Warning: The Next, Previous, and Top actions can only be used when all results in the valuestruct are from the same address book	*/

						/* actionNext */
						case 'N': {
							if ((!ActionProcessed) && (MailSession->ResultsPerPage > 0) && (MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES)) {
								/* These actions increment variables, Make sure we process them only once */
								ActionProcessed = TRUE;

								MailSession->FirstResultInList += MailSession->ResultsPerPage;
								MailSession->Index = MailSession->FirstResultInList;

								if (MailSession->Index >= MailSession->Value->Used) {
									MailSession->InAddressList = NO;
								}
							}
							break;
						}

						/* actionPrevious */
						case 'P': {
							if ((!ActionProcessed) && (MailSession->ResultsPerPage > 0) && (MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES)) {
								/* These actions increment variables, Make sure we process them only once */
								ActionProcessed = TRUE;

								if (MailSession->FirstResultInList < MailSession->ResultsPerPage) {
									MailSession->FirstResultInList = 0;
									MailSession->Index = 0;
									MailSession->InAddressList = NO;
								} else {
									MailSession->FirstResultInList -= MailSession->ResultsPerPage;
									MailSession->Index = MailSession->FirstResultInList;

									if (MailSession->Index >= MailSession->Value->Used) {
										MailSession->InAddressList = NO;
									} else {
										/* We could have fallen off the end */
										MailSession->InAddressList = *MailSession->Value->Value[MailSession->Index];
									}
								}
							}
							break;
						}

						/* actionTop */
						case 'T': {
							if (MailSession->ResultsPerPage) {
									MailSession->FirstResultInList = 0;
									MailSession->Index = 0;
									MailSession->InAddressList = NO;
							}

							break;
						}
					}
					break;
				}
			}
			ValueSize=BUFSIZE-Offset;
		}
		Offset=0;
	}

	if (AddrFile) {
		fclose(AddrFile);
		AddrFile=NULL;
	}

	if (IDFile) {
		fclose(IDFile);
		IDFile = NULL;
	}

	if ((DelPersonal) && (IDPath[0] != '\0')) {
		MDBValueStruct *V;
		unsigned long	i;
		unsigned long	j;

		IDFile = fopen(IDPath,"rb");

		if (!IDFile) {
			MailSession->Error = ERROR_READING_FORM_DATA__FOPEN;
			*NewPage = CurrentPage;
			if ((AddrPath[0] != '\0')) {
				unlink(AddrPath);
			}
			unlink(IDPath);
			return(FALSE);
		}

		V = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

		MsgGetUserFeature(Session->UserDN, FEATURE_MWMAIL_ADDRBOOK_P, MSGSRV_A_ADDRESSBOOK, V);

		while (!feof(IDFile) && !ferror(IDFile)) {
			if (fgets(Client->Temp, MAXEMAILNAMESIZE, IDFile)) {
				ChopNL(Client->Temp);

				/* Find and Remove the ID from the DS Version of the Personal Address Book */
				for (i = 0; i < V->Used; i++) {
					if (MWQuickNCmp(V->Value[i], Client->Temp, ADDR_ENTRY_ID_LEN) == FALSE) {
						continue;
					} else {

						MDBFreeValue(i, V);

						/* Find and Remove the ID display list */
						if (MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES) {
							for (j = 0; j < MailSession->Value->Used; j++) {
								if (MWQuickNCmp(MailSession->Value->Value[j]+2, Client->Temp, ADDR_ENTRY_ID_LEN) == FALSE) {
									continue;
								} else {
									MDBFreeValue(j, MailSession->Value);
									break;
								}
							}
						}
						break;
					}
				}
			}
		}
		fclose(IDFile);
		MDBWrite(Session->UserDN, MSGSRV_A_ADDRESSBOOK, V);
		MDBDestroyValueStruct(V);
	}

	if (AddRecipients && (AddrPath[0] != '\0') && (access(AddrPath,0) == FILE_EXISTS)) {
		RecipFile = fopen(RecipPath, "ab");
		if (!RecipFile) {
			MailSession->Error = ERROR_BUILDING_RECIP_LIST__FOPEN;
			*NewPage = CurrentPage;
			unlink(AddrPath);
			if (IDPath[0] != '\0') {
				unlink(IDPath);
			}
			return (FALSE);
		}

		AddrFile=fopen(AddrPath,"rb");
		if (!AddrFile) {
			MailSession->Error = ERROR_BUILDING_RECIP_LIST__FOPEN;
			fclose(RecipFile);
			unlink(AddrPath);
			if (IDPath[0] != '\0') {
				unlink(IDPath);
			}
			*NewPage = CurrentPage;
			return (FALSE);
		}

		/* We have both files open - Do the update */
		while (!feof(AddrFile) && !ferror(AddrFile)) {
			if (fgets(Client->Temp, MAXEMAILNAMESIZE, AddrFile)) {
				fwrite(Client->Temp, strlen(Client->Temp), 1, RecipFile);
			}
		}

		fclose(AddrFile);
		fclose(RecipFile);
	}

	if (AddrPath[0] != '\0') {
		unlink(AddrPath);
	}
	if (IDPath[0] != '\0') {
		unlink(IDPath);
	}

	return(TRUE);
}


BOOL
MwMailProcessAddrSearchForm(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long *NewPage, unsigned long ResultsPage)
{
	unsigned char	FieldName[128];
	unsigned long	ValueSize;		
	unsigned long	Offset=0;
	unsigned long	CurrentPage;
	BOOL				SearchNow=FALSE;
	unsigned char  *ptr;
	CurrentPage=*NewPage;	
	MailSession->SearchScope		= 0;

	while (MWGetFormName(Client, FieldName, 128)) {
		ModDebug("Got Name:%s\n", FieldName);

		ValueSize=BUFSIZE-Offset;
		while ((MWGetFormValue(Client, Client->Temp+Offset, &ValueSize))!=FORMFIELD_NEXT) {
			ValueSize+=Offset;
			switch (toupper(FieldName[7])) {
				/* searchpErsonal */
				case 'E': {
					MailSession->SearchScope |= SEARCH_PERSONAL;
					break;
				}
				/* searchsYstem */
				case 'Y': {
					MailSession->SearchScope |= SEARCH_SYSTEM;
					break;
				}
				/* searchpUblic */
				case 'U': {
					MailSession->SearchScope |= SEARCH_PUBLIC;
					break;
				}
				/* searcheXecute */
				case 'X': {
					SearchNow=TRUE;
					*NewPage=ResultsPage;
					break;
				}

				/* searchsTring */
				case 'T': {
					if (MailSession->SearchString) {
						MemFree(MailSession->SearchString);
						MailSession->SearchString=NULL;
					}	

					/* Enforce LDAP SearchString Limit */
					Client->Temp[80] = '\0';

					/* whack leading spaces or asterisks */
					ptr = Client->Temp;
					while (isspace(*ptr) || (*ptr == '*')) {
						ptr++;
					}

					if (ptr > Client->Temp) {
						HulaStrNCpy(Client->Temp, ptr, sizeof(Client->Temp));
						ptr = Client->Temp;
					}

					/* truncate string if it contains spaces */
					while (*ptr != '\0') {
						*ptr=tolower(*ptr);
						if (isspace(*ptr)) {
							*ptr = '\0';
							break;
						}
						ptr++;
					}

					/* Remove any trailing asterisks */
					ptr--;
					while (ptr > Client->Temp) {
						if (*ptr != '*') {
							break;
						} 

						*ptr = '\0';
						ptr--;
					}
					if (*Client->Temp) {
						MailSession->SearchString = MemStrdup(Client->Temp);
					}
					break;
				}

			}
			ValueSize=BUFSIZE-Offset;
		}
		Offset=0;
	}

	if (SearchNow) {
		MwMailAddrBookSearch(Client, Session, MailSession);
	}

	return(TRUE);
}


BOOL


MwMailProcessAddrSelectForm(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long *NewPage)
{
	FILE				*RecipFile = NULL;
	FILE				*TempFile = NULL;	
	unsigned char	RecipPath[XPL_MAX_PATH+1];
   unsigned char	TempPath[XPL_MAX_PATH+1];
	unsigned long	ValueSize;
	unsigned char	FieldName[MAXEMAILNAMESIZE+1];
	unsigned long	CurrentPage;
	unsigned	long	len;
	BOOL				NeedFile = FALSE;

	CurrentPage=*NewPage;	
	while (MWGetFormName(Client, FieldName, MAXEMAILNAMESIZE+1)) {
		ModDebug("Got Name:%s\n", FieldName);

		ValueSize = BUFSIZE;
		while ((MWGetFormValue(Client, Client->Temp, &ValueSize))!=FORMFIELD_NEXT) {
			switch (toupper(FieldName[0])) {
				/* Remove */
				case 'R': {
					break;
				}
				/* To */
				case 'T': {
					NeedFile=TRUE;	
					snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
					break;
				}
				/* CC */
				case 'C': {
					NeedFile=TRUE;	
					snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
					break;
				}
				/* BCC */
				case 'B': {
					NeedFile=TRUE;	
					snprintf(RecipPath, sizeof(RecipPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
					break;
				}
			}

			/* We go through this for each address one at a time */
			if (NeedFile) {	 
				RecipFile = fopen(RecipPath, "rb");
				if (!RecipFile) {
					MailSession->Error = ERROR_BUILDING_RECIP_LIST__FOPEN;
					*NewPage = CurrentPage;
					return(FALSE);
				}
				snprintf(TempPath, sizeof(TempPath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, ADDRESS_FILE_EXTENSION);
				TempFile = fopen(TempPath, "wb");
				if (!TempFile) {
					MailSession->Error = ERROR_BUILDING_RECIP_LIST__FOPEN;
					fclose(RecipFile);
					*NewPage = CurrentPage;
					return(FALSE);
				}

				while (!feof(RecipFile) && !ferror(RecipFile)) {
					if (fgets(FieldName, MAXEMAILNAMESIZE, RecipFile)) {  /* Abuse FieldName to read in addresses */
						len = strlen(FieldName);
						/* The address from the file will have a \r\n at the end and will be 2 longer than the same address in Client->Temp */
						if (((len-2) != strlen(Client->Temp)) || (!MWQuickNCmp(Client->Temp, FieldName, len-2))) {
							fwrite(FieldName, len, 1, TempFile);				
						}													 						
					}
				}

				fclose(RecipFile);
				fclose(TempFile);

				unlink(RecipPath);
				rename(TempPath, RecipPath);
			}

			NeedFile=FALSE;
			ValueSize = BUFSIZE;	
		}		
	}

	return(TRUE);
}


BOOL 
MwMailAddrBookCleanUp(SessionStruct *Session, MailSessionStruct *MailSession) 
{
	MDBFreeValues(MailSession->Value);
	if (MailSession->ValueType == SESSION_VALUE_TYPE_ADDRESSES)	{
		MailSession->ValueType = 0;
	}
	if (MailSession->SearchString) {
		MemFree(MailSession->SearchString);
		MailSession->SearchString = NULL;
	}

	MwMailComposeCleanUp(Session, MailSession);
	return(TRUE);
}

