/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <connio.h>

unsigned char	*COMPOSE_EXT_TO_LIST[] = { "to", "cc", "bcc" };

#include <mdb.h>
#include <msgapi.h>
#include <msgaddr.h>
#include <nmap.h>
#include <libical.h>
#include <hulautil.h>
#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */

#include "mwmail.h"


#define	ProcessMemoryStream(Length)									\
	MemoryInCodec.StreamLength=(Length);								\
	MemoryInCodec.Codec(&MemoryInCodec, MemoryInCodec.Next);		\
	MWResetStreamChain(&MemoryInCodec);

#define	ProcessFileStream(File)											\
	FileInCodec.StreamData=(File);										\
	FileInCodec.Codec(&FileInCodec, FileInCodec.Next);				\
	MWResetStreamChain(&FileInCodec);

#define	ReleaseStreams														\
	if (RFC1522Codec) MWReleaseStream(RFC1522Codec);				\
	if (Base64Codec) MWReleaseStream(Base64Codec);					\
	if (BodyCharsetCodec) MWReleaseStream(BodyCharsetCodec);		\
	if (BodyQPCodec) MWReleaseStream(BodyQPCodec);					\
	if (MessageOutCodec) MWReleaseStream(MessageOutCodec);

#define ReleaseCalendarStreams											\
	if (RFC1522Codec) MWReleaseStream(RFC1522Codec);				\
	if (Base64Codec) MWReleaseStream(Base64Codec);					\
	if (RFC2445Codec) MWReleaseStream(RFC2445Codec);				\
	if (RFC2445MLCodec) MWReleaseStream(RFC2445MLCodec);			\
	if (MessageOutCodec) MWReleaseStream(MessageOutCodec);


#define	RestoreUserNMAP													\
	if (Session->NMAPs != ClientNMAPs) {								\
	Session->NMAPs=QueueNMAPs;											\
	MWSendNMAPServer(Session, "QUIT\r\n", 6);							\
	MWGetNMAPAnswer(Session, Buffer, BUFSIZE, FALSE);					\
	IPclose(Session->NMAPs);											\
	/* "flush" our input buffer */										\
	Session->NBufferPtr=0;												\
	Session->NMAPs=ClientNMAPs;											\
	Session->NMAPAddress.s_addr=NMAPCAddress; }							\


BOOL
MwMailSessionUserIsICalOrganizer(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession)
{
	int				ReplyInt;
	StreamStruct	NetworkInCodec;
	StreamStruct	*MemoryOutCodec;
	StreamStruct	*EncodingCodec;
	ICalObject				*ICal = NULL;
	unsigned char			*ICalStream;
	unsigned long			ICalStreamSize;
	unsigned long			Element;

	if (MailSession->CurrentMsgViewMessage > Session->NumOfMessages) {
		return(FALSE);
	}

	/* First, prep the MIME cache */
	if (!MwMailLoadMIMECache(MailSession->CurrentMsgViewMessage, Client, Session, MailSession)) {
		MailSession->Error = ERROR_ACCESSING_ORIGINAL_MESSAGE__MIME;

		return(FALSE);
	}

	/* Now we may parse the list */
	for (Element = 0; Element < MailSession->MIMEUsed; Element++) {
		if (MWQuickCmp(MailSession->MIME[Element].Type, "text/calendar")) {
			/* We parse the ICal Object to determine if we are the organizer */

			/* Now we retrieve the part into memory, so that we can parse it */
			memset(&NetworkInCodec, 0, sizeof(StreamStruct));
			NetworkInCodec.StreamData=Session;
			NetworkInCodec.Codec=MWStreamFromMWNMAP;
			/* Resolve the encoding of the message part */

			EncodingCodec=MWGetStream(NULL, MailSession->MIME[Element].Encoding, FALSE);

			/* Send the interpreted part to RAM */
			MemoryOutCodec=MWGetStream(NULL, NULL, FALSE);
			MemoryOutCodec->Codec=MWStreamToMemory;

			/* Request the body part; we're already connected to ClientNMAPs from the MwMailLoadMIMECache above */
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", 
								  MailSession->IDList[MailSession->CurrentMsgViewMessage - 1], 
								  MailSession->MIME[Element].PartStart, MailSession->MIME[Element].PartSize);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
			if (ReplyInt/10!=202) {
				MWReleaseStream(EncodingCodec);
				MWReleaseStream(MemoryOutCodec);
				return(FALSE);
			}

			/* Let the code know how much data to expect */
			NetworkInCodec.StreamLength=atol(Client->Temp);
			/* Chain up properly */
			if (EncodingCodec && EncodingCodec->Codec) {
				NetworkInCodec.Next=EncodingCodec;
				EncodingCodec->Next=MemoryOutCodec;
			} else {
				NetworkInCodec.Next=MemoryOutCodec;
			}

			Client->NMAPProtected=TRUE;
			NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);

			/* Read BRAW 1000 OK */
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
			Client->NMAPProtected=FALSE;

			ICalStream=MemoryOutCodec->StreamData;
			ICalStreamSize=MemoryOutCodec->StreamLength;
			ICalStream[ICalStreamSize]='\0';

			ICal=ICalParseObject(NULL, ICalStream, ICalStreamSize);
			MemFree(ICalStream);
			MWReleaseStream(EncodingCodec);
			MWReleaseStream(MemoryOutCodec);
			break;
		}
	}

	if (!ICal || !ICal->Organizer) {
		/* FIX-ME report error */
		if (ICal) {
			ICalFreeObjects(ICal);
		}
		return(FALSE);
	}
	
	/* Now we can finally determine whether or not we are the organizer */
	if (MWQuickCmp(ICal->Organizer->Address, Session->EMailAddress)) {
		ICalFreeObjects(ICal);
		return(TRUE);
	}

	ICalFreeObjects(ICal);
	return(FALSE);

}

static BOOL
RFC2822SendMIMEParam(FILE *MessageFile, unsigned char *Name, unsigned char *Value, unsigned char *UserCharset, StreamStruct **FileNameCodec)
{
	StreamStruct	*memoryInCodec;

	if (*FileNameCodec != NULL) {
		memoryInCodec = *FileNameCodec;

		if (MwMail.AttachmentNameEncodingSend == ANE_RESTRICT_TO_7BIT) {	 
			/* Mem -> RFC2231_Value_Decode -> quoted_string_encode -> File */
			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	

			/* send parameter */
			fprintf(MessageFile, "%s=", Name);
			memoryInCodec->Codec(memoryInCodec, memoryInCodec->Next);

			return(TRUE);

		} else if (MwMail.AttachmentNameEncodingSend == ANE_RFC2231) {
			/* Mem -> RFC2231_Value_Decode -> UserCharset -> RFC2231_Value_Encode -> File */
			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	
			memoryInCodec->Next->Next->Next->StreamData = Name;

			memoryInCodec->Codec(memoryInCodec, memoryInCodec->Next);

			return(TRUE);
		} else if (MwMail.AttachmentNameEncodingSend == ANE_RFC2047) {
			/* Mem -> RFC2231_Value_Decode -> RFC1522_Encode -> File */

			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	

			fprintf(MessageFile, "%s=", Name);
			memoryInCodec->Codec(memoryInCodec, memoryInCodec->Next);

			return(TRUE);
		} else {
			/* Mem -> RFC2231_Value_Decode -> UserCharset -> File */

			memoryInCodec->StreamData = Value;
			memoryInCodec->StreamLength = strlen(Value);	

			fprintf(MessageFile, "%s=\"", Name);
			memoryInCodec->Codec(memoryInCodec, memoryInCodec->Next);
			fwrite("\"", 1, 1, MessageFile);
			return(TRUE);
		}
	} else {
		if (MwMail.AttachmentNameEncodingSend == ANE_RESTRICT_TO_7BIT) {	 
			/* Mem -> RFC2231_Value_Decode -> quoted_string_encode -> File */
			memoryInCodec = MWGetStream(NULL, NULL, FALSE);
			if (memoryInCodec) {
				StreamStruct	*rfc2231ValueDecodeCodec;

				*FileNameCodec = memoryInCodec;

				memset(memoryInCodec, 0, sizeof(StreamStruct));
				memoryInCodec->Codec = MWStreamFromMemory;

				rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
				if (rfc2231ValueDecodeCodec) {
					StreamStruct	*quotedStringCodec;

					rfc2231ValueDecodeCodec->Charset = UserCharset;
					quotedStringCodec = MWGetStream(rfc2231ValueDecodeCodec, "quoted-string", TRUE);
					if (quotedStringCodec) {
						StreamStruct	*fileOutCodec;
						
						fileOutCodec = MWGetStream(quotedStringCodec, NULL, FALSE);
						if (fileOutCodec) {
							fileOutCodec->Codec = MWStreamToFile;
							fileOutCodec->StreamData = MessageFile;

							memoryInCodec->StreamData = Value;
							memoryInCodec->StreamLength = strlen(Value);	

							/* send parameter */
							fprintf(MessageFile, "%s=", Name);
							memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

							return(TRUE);
						}
						MWReleaseStream(quotedStringCodec);
					}
					MWReleaseStream(rfc2231ValueDecodeCodec);
				}
				MWReleaseStream(memoryInCodec);
			}
		} else if (MwMail.AttachmentNameEncodingSend == ANE_RFC2231) {
			/* Mem -> RFC2231_Value_Decode -> UserCharset -> RFC2231_Value_Encode -> File */
			memoryInCodec = MWGetStream(NULL, NULL, FALSE);
			if (memoryInCodec) {
				StreamStruct	*rfc2231ValueDecodeCodec;

				*FileNameCodec = memoryInCodec;

				memset(memoryInCodec, 0, sizeof(StreamStruct));
				memoryInCodec->Codec = MWStreamFromMemory;
				memoryInCodec->StreamData = Value;
				memoryInCodec->StreamLength = strlen(Value);	

				rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
				if (rfc2231ValueDecodeCodec) {
					StreamStruct	*charsetCodec;
					rfc2231ValueDecodeCodec->Charset = UserCharset;
					charsetCodec = MWGetStream(rfc2231ValueDecodeCodec, UserCharset, TRUE);
					if (charsetCodec) {
						StreamStruct	*rfc2231ValueEncodeCodec;
						rfc2231ValueEncodeCodec = MWGetStream(charsetCodec, "RFC2231_Param_Value", TRUE);
						if (rfc2231ValueEncodeCodec) {
							StreamStruct	*fileOutCodec;

							rfc2231ValueEncodeCodec->Charset = UserCharset;
							rfc2231ValueEncodeCodec->StreamData = (void *)Name;

							fileOutCodec = MWGetStream(rfc2231ValueEncodeCodec, NULL, FALSE);
							if (fileOutCodec) {
								fileOutCodec->Codec = MWStreamToFile;
								fileOutCodec->StreamData = MessageFile;

								/* send parameter */
								memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

								return(TRUE);
							}
							MWReleaseStream(rfc2231ValueEncodeCodec);
						}
						MWReleaseStream(charsetCodec);
					}
					MWReleaseStream(rfc2231ValueDecodeCodec);
				}
				MWReleaseStream(memoryInCodec);
			}
		} else if (MwMail.AttachmentNameEncodingSend == ANE_RFC2047) {
			/* Mem -> RFC2231_Value_Decode -> RFC2047_Encode -> File */
			memoryInCodec = MWGetStream(NULL, NULL, FALSE);
			if (memoryInCodec) {
				StreamStruct	*rfc2231ValueDecodeCodec;

				*FileNameCodec = memoryInCodec;

				memset(memoryInCodec, 0, sizeof(StreamStruct));
				memoryInCodec->Codec = MWStreamFromMemory;
				memoryInCodec->StreamData = Value;
				memoryInCodec->StreamLength = strlen(Value);	

				rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
				if (rfc2231ValueDecodeCodec) {
					StreamStruct	*rfc2047EncodeCodec;
					rfc2231ValueDecodeCodec->Charset = UserCharset;
					rfc2047EncodeCodec = MWGetStream(rfc2231ValueDecodeCodec, "RFC2047_Name", TRUE);
					if (rfc2047EncodeCodec) {
						StreamStruct	*fileOutCodec;

						rfc2047EncodeCodec->Charset = UserCharset;
						
						fileOutCodec = MWGetStream(rfc2047EncodeCodec, NULL, FALSE);
						if (fileOutCodec) {
							fileOutCodec->Codec = MWStreamToFile;
							fileOutCodec->StreamData = MessageFile;

							/* send parameter */
							fprintf(MessageFile, "%s=", Name);
							memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);

							return(TRUE);
						}
						MWReleaseStream(rfc2047EncodeCodec);
					}
					MWReleaseStream(rfc2231ValueDecodeCodec);
				}
				MWReleaseStream(memoryInCodec);
			}
		} else {
			/* Mem -> RFC2231_Value_Decode -> UserCharset -> File */
			memoryInCodec = MWGetStream(NULL, NULL, FALSE);
			if (memoryInCodec) {
				StreamStruct	*rfc2231ValueDecodeCodec;

				*FileNameCodec = memoryInCodec;

				memset(memoryInCodec, 0, sizeof(StreamStruct));
				memoryInCodec->Codec = MWStreamFromMemory;
				memoryInCodec->StreamData = Value;
				memoryInCodec->StreamLength = strlen(Value);	

				rfc2231ValueDecodeCodec = MWGetStream(memoryInCodec, "RFC2231_NMAP_Value", FALSE);
				if (rfc2231ValueDecodeCodec) {
					StreamStruct	*charsetCodec;
					rfc2231ValueDecodeCodec->Charset = UserCharset;
					charsetCodec = MWGetStream(rfc2231ValueDecodeCodec, UserCharset, TRUE);
					if (charsetCodec) {
						StreamStruct	*fileOutCodec;

						fileOutCodec = MWGetStream(charsetCodec, NULL, FALSE);
						if (fileOutCodec) {
							fileOutCodec->Codec = MWStreamToFile;
							fileOutCodec->StreamData = MessageFile;

							/* send parameter */
							fprintf(MessageFile, "%s=\"", Name);
							memoryInCodec->Codec(memoryInCodec, rfc2231ValueDecodeCodec);
							fwrite("\"", 1, 1, MessageFile);

							return(TRUE);
						}
						MWReleaseStream(charsetCodec);
					}
					MWReleaseStream(rfc2231ValueDecodeCodec);
				}
				MWReleaseStream(memoryInCodec);
			}
		}
	}

	*FileNameCodec = NULL;
	
	/* send parameter */
	fprintf(MessageFile, "%s=\"", Name);
	fprintf(MessageFile, "%s\"\r\n", Value);

	return(FALSE);
}

static BOOL
SendMailMessage(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long DSNFlags)
{
	unsigned char			Answer[BUFSIZE + 1];
	unsigned char			Buffer[BUFSIZE + 1];
	unsigned char			Boundary[BUFSIZE + 1];
	int						retVal;
	FILE						*FileHandle;
	FILE						*Message;
	unsigned long			NumRecipients=0;
	unsigned long			i;
	struct stat				sb;
	MDBValueStruct			*V;
	StreamStruct			FileInCodec;
	StreamStruct			MemoryInCodec;
	StreamStruct			*RFC1522Codec		= NULL;
	StreamStruct			*Base64Codec		= NULL;
	StreamStruct			*BodyCharsetCodec = NULL;
	StreamStruct			*BodyQPCodec		= NULL;
	StreamStruct			*MessageOutCodec	= NULL;
	StreamStruct			*CRLFCodec			= NULL;
	int						ClientNMAPs			= 0;
	int						QueueNMAPs			= 0;
	unsigned long			NMAPCAddress		= 0;

	if (Session->ReadOnly) {
		return(FALSE);
	}

	V = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);

	/* Connect to NMAP server */
	ClientNMAPs = Session->NMAPs;
	if ((MwMail.NmapQAddress != 0) && (MwMail.NmapQAddress != Session->NMAPAddress.s_addr)) {
		unsigned char	*User;

		ClientNMAPs=Session->NMAPs;
		Session->NMAPs=-1;
		Session->NBufferPtr=0;
		NMAPCAddress=Session->NMAPAddress.s_addr;
		Session->NMAPAddress.s_addr=MwMail.NmapQAddress;
		User=Session->User;
		Session->User=NULL;
		if (!MWConnectUserToNMAPServer(Session)) {
			Session->User=User;
			MailSession->Error = ERROR_CONNECTING_TO_QUEUE;
			Session->NMAPs=ClientNMAPs;
			MDBDestroyValueStruct(V);
			return(FALSE);
		}
		Session->User=User;
	} 
	QueueNMAPs=Session->NMAPs;

	/*XplConsolePrintf("[%d] Q:%x(%d) S:%x(%d), Current:%s\n", __LINE__, MwMail.NmapQAddress, QueueNMAPs, NMAPCAddress, ClientNMAPs, Session->NMAPs==ClientNMAPs ? "Session" : "Queue");*/

	/* Create Queue Entry */
	MWSendNMAPServer(Session, "QCREA\r\n", 7);
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
		MailSession->Error = ERROR_CREATING_MESSAGE__QCREA;
		MDBDestroyValueStruct(V);
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Start creating Header in file */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	Message=fopen(Buffer, "wb");
	if (!Message) {
		MailSession->Error = ERROR_CREATING_MESSAGE__FOPEN;
		MDBDestroyValueStruct(V);
		RestoreUserNMAP;
		return(FALSE);
	}
	
	/* Prepare all required streams */
	memset(&FileInCodec, 0, sizeof(StreamStruct));
	memset(&MemoryInCodec, 0, sizeof(StreamStruct));
	RFC1522Codec=MWGetStream(&MemoryInCodec, "RFC1522", TRUE);
	MessageOutCodec=MWGetStream(RFC1522Codec, NULL, TRUE);
	if (MailSession->Attachments) {
		Base64Codec=MWGetStream(NULL, "BASE64", TRUE);
		Base64Codec->Next=MessageOutCodec;
	}

	BodyCharsetCodec=MWGetStream(NULL, Session->Charset, TRUE);
	if (BodyCharsetCodec) {
		BodyQPCodec=MWGetStream(BodyCharsetCodec, "quoted-printable", TRUE);
	} else {
		BodyQPCodec=MWGetStream(NULL, "quoted-printable", TRUE);
	}
	BodyQPCodec->Next=MessageOutCodec;

	/* Set up the input & output */
	MemoryInCodec.StreamData=Buffer;
	MemoryInCodec.Codec=MWStreamFromMemory;
	MemoryInCodec.Next=RFC1522Codec;

	FileInCodec.Codec=MWStreamFromFile;
	FileInCodec.Next=RFC1522Codec;

	RFC1522Codec->Charset=Session->Charset;

	MessageOutCodec->StreamData=Message;
	MessageOutCodec->Codec=MWStreamToFile;

	/* Start building the message */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	i=sizeof(Client->cs);
	getpeername(Client->s, (struct sockaddr *)(&Client->cs), (socklen_t *)&i);
	fprintf(Message, "Received: from %s [%d.%d.%d.%d] by %s\r\n\twith %s; %s\r\n", 
						Session->User,
						Client->cs.sin_addr.s_net,
						Client->cs.sin_addr.s_host,
						Client->cs.sin_addr.s_lh,
						Client->cs.sin_addr.s_impno,
						Session->OfficialDomain, 
						PRODUCT_NAME,
						Buffer);

	/* Subject */
	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_SUBJECT);
	FileHandle=fopen(Buffer,"rb");

	if (FileHandle) {
		fprintf(Message, "Subject: ");

		fgets(Buffer, sizeof(Buffer), FileHandle);
		ProcessMemoryStream(strlen(Buffer));
		fclose(FileHandle);
		FileHandle=NULL;
	} else {
		fprintf(Message, "Subject: \r\n");
	}

	/* Reply-To:*/
	if (MDBRead(Session->UserDN, MSGSRV_A_EMAIL_ADDRESS, V) > 0) {
		fprintf(Message, "Reply-To: %s\r\n", V->Value[0]);
	} 														
	MDBFreeValues(V);

	/* From */
	fprintf(Message, "From: ");
	retVal = MWGetQuotedString(Session->DisplayName, Client->Temp, sizeof(Client->Temp));
	if (retVal == QP_STRING) {
		fputs(Client->Temp, Message);
		fprintf(Message, " <%s>\r\n", Session->EMailAddress);
	} else if (retVal == HAS_EXTENDED_CHARS) {													
		HulaStrNCpy(Buffer, Session->DisplayName, sizeof(Buffer));
		ProcessMemoryStream(strlen(Buffer));
		fprintf(Message, " <%s>\r\n", Session->EMailAddress);
	} else {
		fprintf(Message, "<%s>\r\n", Session->EMailAddress);
	}

	/* The envelope needs to have the real ID and Session->EmailAddess does not necessarily contain it */
	retVal = snprintf(Answer, sizeof(Answer), "QSTOR FROM %s %s\r\n", Session->EMailAddress, Session->User);
	MWSendNMAPServer(Session, Answer, retVal);

	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
		MailSession->Error = ERROR_CREATING_MESSAGE__QSTOR_FROM;
		fclose(Message);
		ReleaseStreams;
		RestoreUserNMAP;
		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	/* Send copy to our SentFolder */
	if (MailSession->SentFolder) {
		retVal = snprintf(Answer, sizeof(Answer), "QSTOR RAW %c%s %s %lu %s %lu\r\n", QUEUE_RECIP_MBOX_LOCAL, 
								 Session->User, Session->User, (long unsigned int)NO_FORWARD, MailSession->SentFolder, (long unsigned int)MSG_STATE_READ);
		MWSendNMAPServer(Session, Answer, retVal);
		if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
			MailSession->Error = ERROR_CREATING_MESSAGE__QSTOR_RAW;
			fclose(Message);
			ReleaseStreams;
			RestoreUserNMAP;
			MDBDestroyValueStruct(V);
			return(FALSE);
		}
	}
	
	/* Add recipients from the To:, CC: and BCC fields */
	for (i=0; i<3; i++) {
		BOOL	First;

		First=TRUE;

		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[i]);
		FileHandle=fopen(Buffer, "rb");

		if (FileHandle) {
			while (!feof(FileHandle) && !ferror(FileHandle)) {
				if (fgets(Buffer, MAXEMAILNAMESIZE, FileHandle)) {
					ChopNL(Buffer);
					if (Buffer[0]=='\0') {
						continue;
					}

					if (!strchr(Buffer, '@')) {
						strcat(Buffer, "@");
						strcat(Buffer, Session->OfficialDomain);
					}

					switch (i) {
						case COMPOSE_TO: {
							if (First) {
								First=FALSE;
								fprintf(Message, "To: %s", Buffer);
							} else {
								fprintf(Message, ",\r\n\t%s", Buffer);
							}
							break;
						}

						case COMPOSE_CC: {
							if (First) {
								First=FALSE;
								fprintf(Message, "CC: %s", Buffer);
							} else {
								fprintf(Message, ",\r\n\t%s", Buffer);
							}
							break;
						}

						case COMPOSE_BCC: {
							break;
						}
					}

					retVal = snprintf(Answer, sizeof(Answer), "QSTOR TO %s %s %d\r\n",
										Buffer, Buffer, (int)DSNFlags);
					MWSendNMAPServer(Session, Answer, retVal);

					if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
						MailSession->Error = ERROR_BUILDING_RECIP_LIST__QSTOR_TO;
						fclose(FileHandle);
						fclose(Message);
						ReleaseStreams;
						MDBDestroyValueStruct(V);
						return(FALSE);
					} else {
						NumRecipients++;
					}
				} 
			}
			if (!First) {
				fprintf(Message, "\r\n");
			}
			fclose(FileHandle);
		} 
	}

	if (NumRecipients==0) {
		MailSession->Error = ERROR_NO_RECIPIENTS;
		fclose(Message);
		ReleaseStreams;
		RestoreUserNMAP;
		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	if (MailSession->MaxRecipients>0 && (NumRecipients>MailSession->MaxRecipients)) {					 
		MailSession->Error = ERROR_TOO_MANY_RECIPIENTS;
		fclose(Message);
		ReleaseStreams;
		RestoreUserNMAP;
		MDBDestroyValueStruct(V);
		return(FALSE);
	}

	/* Date */
	MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
	fprintf(Message, "Date: %s\r\n", Buffer);

	/* Prorietary information */
	fprintf(Message, "X-Mailer: %s\r\n", PRODUCT_NAME);
	fprintf(Message, "X-Sender: %s\r\n", Session->User);

	/* Priority */
	if (MailSession->Priority != 0) {
		fprintf(Message, "X-Priority: %lu\r\n", MailSession->Priority);
		switch (MailSession->Priority) {
			case 1:
			case 2: {
				fprintf(Message, "X-MSMail-Priority: High\r\n");
				break;
			}
			case 3: {
				fprintf(Message, "X-MSMail-Priority: Normal\r\n");
				break;
			}
			case 4:
			case 5: {
				fprintf(Message, "X-MSMail-Priority: Low\r\n");
				break;
			}
			default: {
				break;
			}
		}
	}

	/* MIME */
	fprintf(Message, "MIME-Version: 1.0\r\n");

	/* Message-ID */
	fprintf(Message, "Message-ID: <%lu.%x%s>\r\n", time(NULL), (unsigned int)Session->User, Session->EMailAddress);
	

	/* Body */
	snprintf(Boundary, sizeof(Boundary), "------=_ModWebBOUNDARY_%x_%lu", (unsigned int)Session->User, time(NULL));

	if (MailSession->Attachments) {
		fprintf(Message, "Content-Type: multipart/mixed;\r\n\tboundary=\"%s\"\r\n\r\n", Boundary);
	} else {
		fprintf(Message, "Content-Type: text/plain; charset=\"%s\"\r\nContent-Transfer-Encoding: quoted-printable\r\n\r\n", Session->Charset);
	}

	/* Start creating Body in file */
	if (MailSession->Attachments) {
		fprintf(Message, "This is a multi-part message in MIME format.\r\n\r\n--%s\r\nContent-Type: text/plain; charset=\"%s\"\r\nContent-Transfer-Encoding: quoted-printable\r\n\r\n", Boundary, Session->Charset);
	}

	CRLFCodec=MWGetStream(NULL, "CRLF", TRUE);

	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_BODY);
	FileHandle=fopen(Buffer,"rb");
	if (FileHandle) {
		/* Stream from file to file */
		if (CRLFCodec) {
			FileInCodec.Next=CRLFCodec;
			if (BodyCharsetCodec) {
				CRLFCodec->Next=BodyCharsetCodec;
			} else {
				CRLFCodec->Next=BodyQPCodec;
			}
		} else {
			if (BodyCharsetCodec) {
				FileInCodec.Next=BodyCharsetCodec;
			} else {
				FileInCodec.Next=BodyQPCodec;
			}
		}
		ProcessFileStream(FileHandle);
		fclose(FileHandle);
	}

	if (MsgGetUserFeature(Session->UserDN, FEATURE_MODWEB, MSGSRV_A_SIGNATURE, V)) {
		BOOL				SignatureEnabled	= FALSE;
		unsigned char	*Signature			= NULL;
		
		i = 0;

		while (i < V->Used) {
			switch (*V->Value[i]) {
				case 'E': {
					if (*(V->Value[i] + 1) != '\0') {
						SignatureEnabled = (*(V->Value[i] + 2) == 'T') ? TRUE : FALSE;
					}
					break;
				}
				
				case 'S': {
					if (*(V->Value[i] + 1) != '\0') {
						Signature=V->Value[i]+2;
					}
					break;
				}
				default: {
					break;
				}
			}
			i++;
		}
		if ((SignatureEnabled) && (Signature)) {
			if (CRLFCodec) {
				MemoryInCodec.Next=CRLFCodec;
				if (BodyCharsetCodec) {
					CRLFCodec->Next=BodyCharsetCodec;
				} else {
					CRLFCodec->Next=BodyQPCodec;
				}
			} else {
				if (BodyCharsetCodec) {
					MemoryInCodec.Next=BodyCharsetCodec;
				} else {
					MemoryInCodec.Next=BodyQPCodec;
				}
			}
			/* Hand-process the stream since we want it to work on Signature instead of Buffer; reset to Buffer afterwards */
			MemoryInCodec.StreamData=Signature;
			MemoryInCodec.StreamLength=strlen(Signature);
			MemoryInCodec.Codec(&MemoryInCodec, MemoryInCodec.Next);
			MWResetStreamChain(&MemoryInCodec);
			MemoryInCodec.StreamData=Buffer;
			MemoryInCodec.Next=RFC1522Codec;
		}	
	}
	if (CRLFCodec) {
		MWReleaseStream(CRLFCodec);
	}

#if 0
	if (Licensed == FALSE) {
		fprintf(Message, "\r\n--\r\n");
		fprintf(Message, "%s", LicenseMessage);
		fprintf(Message, "\r\n");
	}
#endif

	fprintf(Message, "\r\n");

	MDBDestroyValueStruct(V);

	if (MailSession->Attachments) {
		StreamStruct *fileNameCodec = NULL;	

		for (i = 0; i < MailSession->Attachments; i++) {
			snprintf(Buffer, sizeof(Buffer), "%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)i);
			FileHandle=fopen(Buffer, "rb");
			if (!FileHandle) {
				fclose(Message);
				MailSession->Error = ERROR_CREATING_MESSAGE__FOPEN;
				ReleaseStreams;
				RestoreUserNMAP;
				return(FALSE);
			}

			/* Reading filename of attachment */
			fgets(Buffer, BUFSIZE, FileHandle);
			ChopNL(Buffer);

			/* Reading mime-type of attachment */
			fgets(Answer, BUFSIZE, FileHandle);
			ChopNL(Answer);
			if (MWQuickCmp(Answer, "message/rfc822")==FALSE) {
				FileInCodec.Next=Base64Codec;
				fprintf(Message, "\r\n--%s\r\nContent-Type: %s;\r\n\t", Boundary, Answer);
				RFC2822SendMIMEParam(Message, "name", Buffer, Session->Charset, &fileNameCodec);
				fprintf(Message, "\r\nContent-Transfer-Encoding: BASE64\r\nContent-Disposition: attachment;\r\n\t");
				RFC2822SendMIMEParam(Message, "filename", Buffer, Session->Charset, &fileNameCodec);
				fprintf(Message, "\r\n\r\n");
			} else {
				/* We don't need the BASE64 encoding */
				FileInCodec.Next=MessageOutCodec;
				fprintf(Message, "\r\n--%s\r\nContent-Type: %s;\r\n\t", Boundary, Answer);
				RFC2822SendMIMEParam(Message, "name", Buffer, Session->Charset, &fileNameCodec);
				fprintf(Message, "\r\nContent-Transfer-Encoding: 8bit\r\nContent-Disposition: inline\r\n\r\n");
			}

			ProcessFileStream(FileHandle);
			fclose(FileHandle);
		}
		MWReleaseStreams(fileNameCodec);
		fprintf(Message, "\r\n--%s--\r\n\r\n", Boundary);
	}
	fclose(Message);

	ReleaseStreams;

	snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
	if ((stat(Buffer, &sb)==-1) || (FileHandle=fopen(Buffer, "rb"))==NULL) {
		MailSession->Error = ERROR_CREATING_MESSAGE__STAT;
		RestoreUserNMAP;
		return(FALSE);
	}	

	if (MailSession->MaxSize != 0 && (unsigned)(sb.st_size) > MailSession->MaxSize) {
		MailSession->Error = ERROR_MESSAGE_EXCEEDS_SIZE_LIMIT;
		fclose(FileHandle);
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Tell NMAP the Message is coming */
	retVal=snprintf(Answer, sizeof(Answer), "QSTOR MESSAGE %d\r\n", (int)sb.st_size);
	MWSendNMAPServer(Session, Answer, retVal);

	while (!feof(FileHandle) && !ferror(FileHandle)) {
		if ((retVal=fread(Answer, 1, BUFSIZE, FileHandle))>0) {
			MWSendNMAPServer(Session, Answer, retVal);
		} 
	}
	fclose(FileHandle);
	
	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)!=1000) {
		MailSession->Error = ERROR_CREATING_MESSAGE__QSTOR_MESSAGE;
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Kick it */
	MWSendNMAPServer(Session, "QRUN\r\n", 6);

	if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)!=1000) {
		MailSession->Error = ERROR_CREATING_MESSAGE__QRUN;
		RestoreUserNMAP;
		return(FALSE);
	}

	RestoreUserNMAP;

	/* Whack our temp files */
	MwMailComposeCleanUp(Session, MailSession);
	return(TRUE);
}

BOOL
MwMailComposeCleanUp(SessionStruct *Session, MailSessionStruct *MailSession)
{
	unsigned char	Path[XPL_MAX_PATH+1];
	unsigned long	AttachNum;

	snprintf(Path, sizeof(Path), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_TEMP, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_BODY, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_SUBJECT, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_LOCATION, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_MESSAGE, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_RRULE, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_ICAL, MwMail.WorkDir, (unsigned int)Session->SessionID);
	unlink(Path);

	for (AttachNum=0; AttachNum < MailSession->Attachments; AttachNum++) {
		snprintf(Path, sizeof(Path),"%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)AttachNum);
		unlink(Path);
	}
	MailSession->Attachments = 0;

	MailSession->Priority = 0;
	MailSession->DSNFlags = DSN_DEFAULT;

	return(TRUE);
}

BOOL
MwMailSendAddressValues(ConnectionStruct *Client, SessionStruct *Session, unsigned char *FieldName)
{
	FILE				*FileHandle;
	unsigned long	count=0;

	snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, FieldName);

	MWProtectNMAP(Client, TRUE);
	FileHandle=fopen(Client->Temp,"rb");

	if (!FileHandle) {
		MWProtectNMAP(Client, FALSE);
		return (TRUE);
	} 

	while (!feof(FileHandle) && !ferror(FileHandle)) {
		if (fgets(Client->Temp, BUFSIZE, FileHandle)) {
			ChopNL(Client->Temp);
			if (count) {																// We've been here before, add separator
				MWSendClient(Client, ", ", 2);
			}
			count=strlen(Client->Temp);
			MWSendClient(Client, Client->Temp, count);
		}
	}

	fclose(FileHandle);
	MWProtectNMAP(Client, FALSE);

	return(TRUE);
}

BOOL
MwMailProcessComposeForm(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long *CurrentPage, unsigned long SendPage)
{
	FILE				*FileHandle=NULL;
	FILE				*AttachmentHandle=NULL;
	unsigned char	Path[XPL_MAX_PATH+1];
	unsigned long	ValueSize;
	unsigned char	FieldName[128];
	unsigned long	Offset=0;
	long RetVal;
	unsigned char	*ptr;
	unsigned char	*StartPtr;
	unsigned char	*EndPtr;
	unsigned long	DefaultPage;
	unsigned long	Action = 0xffffffff;
	unsigned char	Type[XPL_MAX_PATH+1];
	unsigned char	FilePath[XPL_MAX_PATH+1];
	unsigned long	FileSize;
	unsigned long	i;
	BOOL				FilesRemoved = FALSE;

	if (MailSession->DSNFlags == 0) {
		MailSession->DSNFlags = DSN_DEFAULT;
	}

	DefaultPage = *CurrentPage;
	Type[0]='\0';
	FilePath[0]='\0';

	MWProtectNMAP(Client, TRUE);
	while (MWGetFormNameEx(Client, FieldName, Type, FilePath, sizeof(FieldName))) {
		ModDebug("Got Name:%s\n", FieldName);
		ValueSize=BUFSIZE-Offset;
		FileSize=0;

		while ((RetVal=MWGetFormValue(Client, Client->Temp+Offset, &ValueSize))!=FORMFIELD_NEXT) {
			ValueSize+=Offset;
			switch (toupper(FieldName[0])) {
				/* Address fields & Actions */
				case 'A': {
					switch (toupper(FieldName[1])) {
						/* aCtion */
						case 'C': {
							switch(toupper(FieldName[6])) {
								/* actionLoadpage<next Page> */
								case 'L': {
								    long ret;
									i=strlen(FieldName);
									if (FieldName[i-2]=='.') {
										if (toupper(FieldName[i-1])=='X' || toupper(FieldName[i-1])=='Y') {
											FieldName[i-2]='\0';
										}
									}
									if ((ret = MWFindTemplatePage(FieldName+14, Session->TemplateID)) == -1) {
										*CurrentPage=DefaultPage;
									} else {
									    *CurrentPage = (unsigned long)ret;
									}
									break;	
								}
								/* actionS	*/
								case 'S': {
									switch (toupper(FieldName[9])) {
										/* actionsenDmsg */
										case 'D': {
											Action = ACTION_SEND_MESSAGE;
											*CurrentPage = SendPage;
												break;
										}
										/* actionseaRch<Field> */
										case 'R': {
										    long ret;
											switch (toupper(FieldName[12])) {
												/* actionsearchTo<next Page> */
												case 'T': {
													Action = ACTION_SEARCH_TO;
													break;
												}
												/* actionsearchCc<next Page> */
												case 'C': {
													Action = ACTION_SEARCH_CC;
													break;
												}
												/* actionsearchBc<next Page> */
												case 'B': {
													Action = ACTION_SEARCH_BCC;
													break;
												}
											}
											i=strlen(FieldName);
											if (FieldName[i-2]=='.') {
												if (toupper(FieldName[i-1])=='X' || toupper(FieldName[i-1])=='Y') {
													FieldName[i-2]='\0';
												}
											}
											if ((ret = MWFindTemplatePage(FieldName+14, Session->TemplateID))==-1) {
												*CurrentPage=DefaultPage;
											} else {
											    *CurrentPage = ret;
											}
											break;
										}
									}
									break;
								}
							}
							break;
						}
						/* aDdr */
						case 'D': {
							switch (toupper(FieldName[4])) {
								/* addrTo */
								case 'T': {
									if (!FileHandle) {
										snprintf(Path, sizeof(Path),"%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
									}
									break;
								}

								/* addrCc */
								case 'C': {
									if (!FileHandle) {
										snprintf(Path, sizeof(Path),"%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
									}
									break;
								}

								/* addrBcc */
								case 'B': {
									if (!FileHandle) {
										snprintf(Path, sizeof(Path),"%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_BCC]);
									}
									break;
								}

								default: {
									goto SkipWriteRecipients;
								}
							}

							if (!FileHandle) {
								FileHandle=fopen(Path, "wb");
								if (!FileHandle) {
									MailSession->Error=ERROR_SAVING_FORM_DATA__FOPEN;
									*CurrentPage=DefaultPage;
									return(FALSE);
								}
							}

							ptr=Client->Temp;
							StartPtr=Client->Temp;
							EndPtr=Client->Temp+ValueSize;

							while (ptr<EndPtr) {
								switch (*ptr) {
									case ',':
									case ';':
									case ' ': {
										fwrite(StartPtr, ptr-StartPtr, 1, FileHandle);

										/* validate email address is RFC compliant */
										if (MsgIsAddress(StartPtr,ptr-StartPtr, ",; ", NULL)){
											;
										} else {
											if (FieldName[4] == 'T'){
												MailSession->Error = ERROR_BAD_TO_ADDR;
											}else if (FieldName[4] == 'C') {
												MailSession->Error = ERROR_BAD_CC_ADDR;
											}
											else if (FieldName[4] == 'B') {
												MailSession->Error = ERROR_BAD_BCC_ADDR;
											}
										}

										fwrite("\r\n", 2, 1, FileHandle);
										while (ptr<EndPtr && !isalnum(*ptr)) {
											ptr++;
										}
										StartPtr=ptr;
										break;
									}

									default: {
										ptr++;
										break;
									}
								}
							}

							if (ptr-StartPtr) {
								if (RetVal==FORMFIELD_DONE) {

									fwrite(StartPtr, ptr-StartPtr, 1, FileHandle);

									/* validate email address is RFC compliant */
									if (MsgIsAddress(StartPtr,ptr-StartPtr, ",; ", NULL)){
										;
									} else {
										if (FieldName[4] == 'T'){
											MailSession->Error = ERROR_BAD_TO_ADDR;
										}else if (FieldName[4] == 'C') {
											MailSession->Error = ERROR_BAD_CC_ADDR;
										}
										else if (FieldName[4] == 'B') {
											MailSession->Error = ERROR_BAD_BCC_ADDR;
										}
									}

									fwrite("\r\n", 2, 1, FileHandle);
									Offset=0;
								} else {
									/*
										An address cannot be larger than MAXEMAILNAMESIZE, if we get something bigger it's some 
										!%@#%&* is trying to break us. We ignore data longer than that.
									*/
									if (strlen(StartPtr)<MAXEMAILNAMESIZE) {
										Offset=ptr-StartPtr;
										memmove(Client->Temp, StartPtr, Offset);
									} else {
										Offset=0;
									}
								}
							}
		SkipWriteRecipients:
							break;
						}
					}
					break;
				}

				/* File functions */
				case 'F': {
					switch (toupper(FieldName[4])) {
						/* FileRemove */
						case 'R': {
							if (isdigit(Client->Temp[0])) {
								i = atol(Client->Temp);
								FilesRemoved = TRUE;
								snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)i);
								unlink(Client->Temp);
							}
							break;
						}

						/* FileNew */
						case 'N': {
							if(ValueSize) {
								if (!AttachmentHandle) {
									snprintf(Path, sizeof(Path),"%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)MailSession->Attachments);
									AttachmentHandle = fopen(Path,"wb");
									if (!AttachmentHandle) {
										MailSession->Error = ERROR_STORING_ATTACHMENT__FOPEN;
										*CurrentPage = DefaultPage;
										return(FALSE);
									}

									if (FilePath[0] != '\0') {
										/* memoryIn -> rfc2231NMAPDecode -> filenameDecode -> rfc2231NMAPEncode -> fileOut */
										StreamStruct memoryInCodec;
										StreamStruct *rfc2231NMAPDecodeCodec;
										StreamStruct *filenameDecodeCodec;
										StreamStruct *rfc2231NMAPEncodeCodec;
										StreamStruct *fileOutCodec;

										memset(&memoryInCodec, 0, sizeof(StreamStruct));
										memoryInCodec.Codec = MWStreamFromMemory;
										memoryInCodec.StreamLength = strlen(FilePath);
										memoryInCodec.StreamData	= FilePath;

										rfc2231NMAPDecodeCodec = MWGetStream(&memoryInCodec, "RFC2231_NMAP_Value", FALSE);
										if (rfc2231NMAPDecodeCodec) {
											filenameDecodeCodec = MWGetStream(rfc2231NMAPDecodeCodec, "filename", FALSE);
											if (filenameDecodeCodec) {
												rfc2231NMAPEncodeCodec = MWGetStream(filenameDecodeCodec, "RFC2231_NMAP_Value", TRUE);
												if (rfc2231NMAPEncodeCodec) {
													fileOutCodec = MWGetStream(rfc2231NMAPEncodeCodec, NULL, TRUE);
													if (fileOutCodec) {
														fileOutCodec->StreamData = (void *)AttachmentHandle;
														fileOutCodec->Codec = MWStreamToFile;

														memoryInCodec.Codec(&memoryInCodec, rfc2231NMAPDecodeCodec);
														fwrite("\r\n", 2, 1, AttachmentHandle);

														MWReleaseStream(fileOutCodec);
													} else {
														fprintf(AttachmentHandle, "%s\r\n", FilePath);
													}
													MWReleaseStream(rfc2231NMAPEncodeCodec);
												} else {
													fprintf(AttachmentHandle, "%s\r\n", FilePath);
												}
												MWReleaseStream(filenameDecodeCodec);
											} else {
												fprintf(AttachmentHandle, "%s\r\n", FilePath);
											}
											MWReleaseStream(rfc2231NMAPDecodeCodec);
										} else {
											fprintf(AttachmentHandle, "%s\r\n", FilePath);
										} 
									} else {
										fprintf(AttachmentHandle, DEFAULT_ATTACHMENT_NAME"\r\n");
									}

									if (Type[0]!='\0') {
										fprintf(AttachmentHandle, "%s\r\n",Type);
									} else {
										fprintf(AttachmentHandle, DEFAULT_ATTACHMENT_CONTENT_TYPE"\r\n");
									}
								}

								FileSize+=ValueSize;
								fwrite(Client->Temp, ValueSize, 1, AttachmentHandle);

								if (((signed long)MwMail.TimeStamp - (signed long)Session->NMAPTimestamp) > (signed long)(NMAP_CONNECTION_TIMEOUT - (5 * 60))) {
									XplConsolePrintf("\nMODWEBD: MwMailProcessComposeForm() found NMAP time remaining less than 5 minutes.\r\n");
									XplConsolePrintf("         User:                       %s\r\n", Session->User);
									XplConsolePrintf("         MWMAILTimeStamp:            %lu\r\n", MwMail.TimeStamp);
									XplConsolePrintf("         Session->NMAPTimestamp:     %lu\r\n", Session->NMAPTimestamp);

									MWSendNMAPServer(Session, "NOOP\r\n", 6);
									MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);	
									/* Client->Temp can be used because we have already stored its payload to disk */
								}
							} else if (RetVal == FORMFIELD_ERROR) {
								if (AttachmentHandle) {
									fclose(AttachmentHandle);
									AttachmentHandle=NULL;
									/* remove attachment file here */
									snprintf(Path, sizeof(Path),"%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)MailSession->Attachments);
									unlink(Path);
								}
							}
							break;
						}

						default: {

							break;
						}
					}
					break;
				}

				/* Text Fields */
				case 'T': {
					switch (toupper(FieldName[4])) {
						/* TextBody */
						case 'B': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_BODY, MwMail.WorkDir, (unsigned int)Session->SessionID);
								FileHandle=fopen(Path,"wb");
								if (!FileHandle) {
									MailSession->Error = ERROR_STORING_BODY__FOPEN;
									*CurrentPage = DefaultPage;
									return(FALSE);
								}
							}

							if (ValueSize > 0) {
								fwrite(Client->Temp, ValueSize, 1, FileHandle);
							}
							if (RetVal==FORMFIELD_DONE) {
								fwrite("\r\n", 2, 1, FileHandle);
							}
							break;
						}

						/* TextSubject */
						case 'S': {
							if (!FileHandle) {
								snprintf(Path, sizeof(Path),"%s/%x."COMPOSE_EXT_SUBJECT, MwMail.WorkDir, (unsigned int)Session->SessionID);
								FileHandle=fopen(Path,"wb");
								if (!FileHandle) {
									MailSession->Error = ERROR_STORING_FORM_ELEMENT__FOPEN;
									*CurrentPage = DefaultPage;
									return(FALSE);
								}
							}
							if (ValueSize > 0) {
								fwrite(Client->Temp, ValueSize, 1, FileHandle);
							}
							if (RetVal==FORMFIELD_DONE) {
								fwrite("\r\n", 2, 1, FileHandle);
							}
							break;
						}
					}
					break;
				}

				/* Priority */
				case 'P': {
					MailSession->Priority = atol(Client->Temp);
				}

				/* DSN */
				case 'D': {
					switch (atol(Client->Temp)) {
						case 1:		MailSession->DSNFlags = DSN_HEADER | DSN_BODY | DSN_SUCCESS;	break;
						case 2:		MailSession->DSNFlags = DSN_DEFAULT | DSN_SUCCESS;					break;
						default:
						case 0:		MailSession->DSNFlags = DSN_DEFAULT;									break;
					}
				}
			}
			ValueSize=BUFSIZE-Offset;
		}
		Offset=0;

		if (AttachmentHandle) {
			fclose(AttachmentHandle);
			/* The only reason a file would be open is if an attachment had been uploaded successfully */
			MailSession->Attachments++;
			AttachmentHandle=NULL;
		}
		Type[0]='\0';
		FilePath[0]='\0';

		if (FileHandle) {
			fclose(FileHandle);
			FileHandle=NULL;
		}
	}

	/* If there was an error detected when reading in the compose form return here */
	if (MailSession->Error) {
		*CurrentPage = DefaultPage;
		return(FALSE);
	}

	MWProtectNMAP(Client, FALSE);
	
	if (FilesRemoved && (MailSession->Attachments > 0)) {
		/* Renumber the ones we still have */
		RetVal =0;
		for (i=0; i<MailSession->Attachments; i++) {
			snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)i);
			if (access(Client->Temp, 0)!=0)	{
				/* File is not there. It must have been wacked */
				RetVal++;
			} else {
				/* File is there. Check to see if it needs renamed */
				if (RetVal!=0) {
					snprintf(Path, sizeof(Path),"%s/%x.%d", MwMail.WorkDir, (unsigned int)Session->SessionID, (int)(i-RetVal));
					rename(Client->Temp, Path);
				}
			}
		}

		MailSession->Attachments=i-RetVal;
	}

	switch (Action) {
		case ACTION_SEND_MESSAGE: {
			if (!SendMailMessage(Client, Session, MailSession, MailSession->DSNFlags)) {
				*CurrentPage = DefaultPage;
				return(FALSE);
			}
			break;
		}
		
		case ACTION_SEARCH_TO:
		case ACTION_SEARCH_CC:
		case ACTION_SEARCH_BCC: {
			long	lines = 0;
			long	i;
			FILE	*TmpFileHandle;

			if (MailSession->SearchString) {
				MemFree (MailSession->SearchString);
				MailSession->SearchString = NULL;
			}

			/* Open the Address file, count lines and grab the last one */
			snprintf(Path, sizeof(Path), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[Action]);
			FileHandle=fopen(Path,"rb");
			if (!FileHandle) {
				MwMailAddrBookSearch(Client, Session, MailSession);
				return (FALSE);
			} 
	
			Client->Temp[0] = '\0';
			while (!feof(FileHandle) && !ferror(FileHandle)) {
				if (fgets(Client->Temp, BUFSIZE, FileHandle)) {
					lines++;
				}
			}

			/* If the last address has an @ sign, we assume you don't need to look it up. */
			if ((lines == 0) || strchr(Client->Temp, '@')) {
				fclose(FileHandle);
				MwMailAddrBookSearch(Client, Session, MailSession);
				return (TRUE);
			} else {
				/* Clean up the search string */
				ChopNL(Client->Temp);
				
				/* Enforce LDAP SearchString Limit */
				Client->Temp[80] = '\0';
				
				/* whack leading asterisks */
				ptr = Client->Temp;
				while (*ptr == '*') {
					ptr++;
				}

				if (ptr > Client->Temp) {
					HulaStrNCpy(Client->Temp, ptr, sizeof(Client->Temp));
					ptr = Client->Temp;
				}

				/* Go to the end of the string */
				while (*ptr != '\0') {
					ptr++;
				}

				/* Remove any trailing asterisks */
				ptr--;
				while (ptr > Client->Temp) {
					if (*ptr != '*') {
						break;
					}
				
					*ptr = '\0';
					ptr--;
				}
				if (*Client->Temp) {
					MailSession->SearchString = MemStrdup(Client->Temp);
				}
			}

			/* Remove partial entry line from end of file */
			snprintf(FilePath, sizeof(FilePath), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TEMP);  /* Abuse FilePath to save creating a buffer */
			TmpFileHandle = fopen(FilePath,"wb");
			if (!TmpFileHandle) {
				fclose(FileHandle);
				MwMailAddrBookSearch(Client, Session, MailSession);
				return (FALSE);
			} 

			rewind(FileHandle);

			lines--;
			if (lines < 1) {
				fputs("\r\n", TmpFileHandle);
			} else {
				for (i = 0; i < lines; i++) {
					if (fgets(Client->Temp, BUFSIZE, FileHandle)) {
						fputs(Client->Temp, TmpFileHandle);
					} else {
						fclose(FileHandle);
						fclose(TmpFileHandle);
						unlink(FilePath);
						MwMailAddrBookSearch(Client, Session, MailSession);
						return (FALSE);
					}
				}
			}

			fclose(FileHandle);
			FileHandle=NULL;

			fclose(TmpFileHandle);
			TmpFileHandle=NULL;

			unlink(Path);
			rename(FilePath, Path);
			
			MwMailAddrBookSearch(Client, Session, MailSession);
			return (TRUE);
		}

		default: {

		}
	}
 
	return(TRUE);
}




/*
 * The calling function must ensure that the proper message is in the HeaderCache
 */
BOOL
MwMailPrepareComposeAnswer(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long AnswerType, unsigned long DropOriginalMessage)
{
	FILE				*Handle = NULL;
	unsigned char	*ptr;
	unsigned char	ch;
	int				ReplyInt;
	StreamStruct	NetworkInCodec;
	StreamStruct	*FileOutCodec;
	StreamStruct	*EncodingCodec;
	StreamStruct	*CharsetCodec;
	unsigned char	*ReplyPrefix = "";
	unsigned char	*OriginalMessageString = "-----Original Message-----";
	unsigned char	*FromString = "From";
	unsigned char	*ToString = "To";
	unsigned char	*DateString = "Date";
	unsigned char	*SubjectString = "Subject";

	if (MailSession->CurrentMsgViewMessage > Session->NumOfMessages) {
		return(FALSE);
	}

	switch (AnswerType) {
		case COMPOSE_REPLYALL: {
			MwMailComposeCleanUp(Session, MailSession);

			/* Build Cc: file */
			snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_CC]);
			Handle = fopen(Client->Temp, "ab");
			if (Handle && ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "To:", &ptr)) > 0)) {
				unsigned char	*p2;

				ch = ptr[ReplyInt];
				ptr[ReplyInt] = '\0';
				p2 = ptr;
				do {
					p2 = MwMailParseRFC822Address(p2, NULL, 0, Client->Temp, BUFSIZE+1);
					if (MWQuickCmp(Client->Temp, Session->EMailAddress) == FALSE) {
						fprintf(Handle, "%s\r\n", Client->Temp);
					}
				} while (p2);
				ptr[ReplyInt] = ch;
			}


			if (Handle && ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Cc:", &ptr)) > 0)) {
				unsigned char	ch;
				unsigned char	*p2;

				ch = ptr[ReplyInt];
				ptr[ReplyInt] = '\0';
				p2 = ptr;
				do {
					p2 = MwMailParseRFC822Address(p2, NULL, 0, Client->Temp, BUFSIZE+1);
					if (MWQuickCmp(Client->Temp, Session->EMailAddress) == FALSE) {
						fprintf(Handle, "%s\r\n", Client->Temp);
					}
				} while (p2);
				ptr[ReplyInt] = ch;
			}

			if (Handle) {
				fclose(Handle);
				Handle = NULL;
			}
		}
		/* Fall-through */

		case COMPOSE_REPLY: {
			unsigned long	Element;

			if (AnswerType == COMPOSE_REPLY) {
				MwMailComposeCleanUp(Session, MailSession);
			}

			/* Build To: file */
			snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
			Handle = fopen(Client->Temp, "ab");
			if (Handle) {
				if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Reply-To:", &ptr)) > 0) {
					unsigned char	*p2;

					ch = ptr[ReplyInt];
					ptr[ReplyInt] = '\0';
					p2 = ptr;

					do {
						p2 = MwMailParseRFC822Address(p2, NULL, 0, Client->Temp, BUFSIZE+1);
						if (MWQuickCmp(Client->Temp, Session->EMailAddress) == FALSE) {
							fprintf(Handle, "%s\r\n", Client->Temp);
						}
					} while (p2);

					ptr[ReplyInt] = ch;

				} else if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "From:", &ptr)) > 0) {
					unsigned char	*p2;

					ch = ptr[ReplyInt];
					ptr[ReplyInt] = '\0';
					p2 = ptr;
					do {
						p2 = MwMailParseRFC822Address(p2, NULL, 0, Client->Temp, BUFSIZE+1);
						fprintf(Handle, "%s\r\n", Client->Temp);
					} while (p2);

					ptr[ReplyInt] = ch;
				}
			}
			if (Handle) {
				fclose(Handle);
				Handle = NULL;
			}

			if (!DropOriginalMessage) {
				/*
					Now for the tough part - generate the "Reply" message body
					We try to keep it simple, so we look for the first text/plain part in the message;
					if we don't find one (because it might have been just text/html we don't include a
					message in the reply
				*/


				/* First, prep the MIME cache */
				if (!MwMailLoadMIMECache(MailSession->CurrentMsgViewMessage, Client, Session, MailSession)) {
					MailSession->Error = ERROR_ACCESSING_ORIGINAL_MESSAGE__MIME;
					return(FALSE);
				}

				/* Now we may parse the list */
				for (Element=0; Element < MailSession->MIMEUsed; Element++) {
					if (MWQuickCmp(MailSession->MIME[Element].Type, "text/plain")) {
						/* Found our reply part... */
						break;
					}
				}

				/* If we found a reply-element Element will be less than MailSession->MIMEUsed */
				if (Element < MailSession->MIMEUsed) {
					/* Open the body message file */
					snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_BODY, MwMail.WorkDir, (unsigned int)Session->SessionID);
					Handle = fopen(Client->Temp, "ab");
					if (!Handle) {
						MailSession->Error = ERROR_CREATING_MESSAGE__FOPEN;
						return(FALSE);
					}

					/** Prepare the header **/
					fprintf(Handle, "\r\n\r\n\r\n%s\r\n", OriginalMessageString);
					/* From line */
					if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "From:", &ptr)) > 0) {
						ch = ptr[ReplyInt];
						ptr[ReplyInt] = '\0';

						fprintf(Handle, "%s: %s\r\n", FromString, ptr);

						ptr[ReplyInt] = ch;
					}

					/* To line */
					if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "To:", &ptr)) > 0) {
						ch = ptr[ReplyInt];
						ptr[ReplyInt] = '\0';

						fprintf(Handle, "%s: %s\r\n", ToString, ptr);

						ptr[ReplyInt] = ch;
					}

					/* Date line */
					if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Date:", &ptr)) > 0) {
						ch = ptr[ReplyInt];
						ptr[ReplyInt] = '\0';

						fprintf(Handle, "%s: %s\r\n", DateString, ptr);

						ptr[ReplyInt] = ch;
					}

					/* Subject line */
					if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Subject:", &ptr)) > 0) {
						ch = ptr[ReplyInt];
						ptr[ReplyInt] = '\0';

						fprintf(Handle, "%s: %s\r\n", SubjectString, ptr);

						ptr[ReplyInt] = ch;
					}
					fprintf(Handle, "\r\n");

					/* Now we need to pull the thing from NMAP in UTF-8*/

					/* Request the body part */
					ReplyInt = snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MailSession->CurrentMsgViewMessage - 1], MailSession->MIME[Element].PartStart, MailSession->MIME[Element].PartSize);
					MWSendNMAPServer(Session, Client->Temp, ReplyInt);
					ReplyInt = MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

					/** Prepare all streams for later use **/
					/* Read from NMAP */
					memset(&NetworkInCodec, 0, sizeof(StreamStruct));
					NetworkInCodec.StreamData = Session;
					NetworkInCodec.Codec = MWStreamFromMWNMAP;
					NetworkInCodec.StreamLength = atol(Client->Temp);

					EncodingCodec = MWGetStream(NULL, MailSession->MIME[Element].Encoding, FALSE);
					CharsetCodec = MWGetStream(NULL, MailSession->MIME[Element].Charset, FALSE);

					/* Send the interpreted part to the file */
					FileOutCodec = MWGetStream(NULL, NULL, FALSE);
					FileOutCodec->StreamData = (void *)Handle;
					FileOutCodec->Codec = MWStreamToFile;

					if (EncodingCodec) {
						NetworkInCodec.Next = EncodingCodec;
						if (CharsetCodec) {
							EncodingCodec->Next = CharsetCodec;
							CharsetCodec->Next = FileOutCodec;
						} else {
							EncodingCodec->Next = FileOutCodec;
						}
					} else {
						if (CharsetCodec) {
							NetworkInCodec.Next = CharsetCodec;
							CharsetCodec->Next = FileOutCodec;
						} else {
							NetworkInCodec.Next = FileOutCodec;
						}
					}

					Client->NMAPProtected = TRUE;
					NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);
					Client->NMAPProtected = FALSE;

					if (EncodingCodec) {
						MWReleaseStream(EncodingCodec);
					}

					if (CharsetCodec) {
						MWReleaseStream(CharsetCodec);
					}

					MWReleaseStream(FileOutCodec);

					/* Grab the BRAW DONE code and close the file */
					MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

					fclose(Handle);
					Handle = NULL;
				}
			}

			ReplyPrefix = "Re: ";
			break;
		}

		case COMPOSE_FORWARD: {
			FILE	*AttachHandle;

			MwMailComposeCleanUp(Session, MailSession);
			/* We attach the original as attachment */
			snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.0", MwMail.WorkDir, (unsigned int)Session->SessionID);
			AttachHandle=fopen(Client->Temp, "wb");
			if (!AttachHandle) {
				MailSession->Error = ERROR_CREATING_MESSAGE__FOPEN;
				return(FALSE);
			}

			/* Write attachment "name" */
			if ((ReplyInt=MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Subject:", &ptr))>0) {
				ch=ptr[ReplyInt];
				ptr[ReplyInt]='\0';
				fprintf(AttachHandle, "%s.eml\r\n", ptr);
				ptr[ReplyInt]=ch;
			} else {
				fprintf(AttachHandle, "forwarded message.eml\r\n");
			}

			/* Write attachment "type" */
			fprintf(AttachHandle, "message/rfc822\r\n");

			/* Get ready to get the data from NMAP */
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "LIST %lu\r\n", MailSession->IDList[MailSession->CurrentMsgViewMessage-1]);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

			if (ReplyInt/10!=202) {
				MailSession->Error = ERROR_CREATING_MESSAGE__LIST;
				fclose(AttachHandle);
				snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.0", MwMail.WorkDir, (unsigned int)Session->SessionID);
				unlink(Client->Temp);
				return(FALSE);
			}

			/* The response lists header & body size separate */
			ptr=strchr(Client->Temp, ' ');
			if (!ptr) {
				/* 
					Crap! Where did the space go??? 
					We're really screwed, NMAP is streaming and we don't know 
					how much, now way to handle this properly. We're officially
					out of sync if this happens
				*/
				fclose(AttachHandle);
				snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.0", MwMail.WorkDir, (unsigned int)Session->SessionID);
				unlink(Client->Temp);
				return(FALSE);
			}
			
			/** Prepare all streams for later use **/
			/* Read from NMAP */
			memset(&NetworkInCodec, 0, sizeof(StreamStruct));
			NetworkInCodec.StreamData=Session;
			NetworkInCodec.Codec=MWStreamFromMWNMAP;
			NetworkInCodec.StreamLength=atol(Client->Temp)+atol(ptr+1);

			/* Send the interpreted part to the file */
			FileOutCodec=MWGetStream(NULL, NULL, FALSE);
			FileOutCodec->StreamData=(void *)AttachHandle;
			FileOutCodec->Codec=MWStreamToFile;

			NetworkInCodec.Next=FileOutCodec;
			Client->NMAPProtected=TRUE;
			NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);
			Client->NMAPProtected=FALSE;
			MWReleaseStream(FileOutCodec);

			/* Grab the LIST DONE code and close the file */
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
			fclose(AttachHandle);

			/* Make a note that we've got an attachment */
			MailSession->Attachments=1;

			ReplyPrefix="Fw: ";
			break;
		}

		case COMPOSE_REDIRECT: {
			/* On redirect we send the message "as is" */
			ReplyPrefix="Redirected: ";
			break;
		}

		case COMPOSE_DELEGATE:
		case COMPOSE_DECLINE: {
			unsigned long			Element;

			MwMailComposeCleanUp(Session, MailSession);

			/* First, prep the MIME cache */
			if (!MwMailLoadMIMECache(MailSession->CurrentMsgViewMessage, Client, Session, MailSession)) {
				MailSession->Error = ERROR_ACCESSING_ORIGINAL_MESSAGE__MIME;

				return(FALSE);
			}

			/* Now we may parse the list */
			for (Element = 0; Element < MailSession->MIMEUsed; Element++) {
				if (MWQuickCmp(MailSession->MIME[Element].Type, "text/calendar")) {
					/* We create an iCal file so the actual compose can generate an iCal REPLY */

					snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_ICAL, MwMail.WorkDir, (unsigned int)Session->SessionID);
					Handle = fopen(Client->Temp, "wb");
					if (!Handle) {
						MailSession->Error = ERROR_CREATING_MESSAGE__FOPEN;
						return(FALSE);
					}

					/* Request the body part */
					MWSendNMAPServer(Session, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MailSession->CurrentMsgViewMessage - 1], MailSession->MIME[Element].PartStart, MailSession->MIME[Element].PartSize));
					MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

					/* Read from NMAP */
					memset(&NetworkInCodec, 0, sizeof(StreamStruct));
					NetworkInCodec.StreamData = Session;
					NetworkInCodec.Codec = MWStreamFromMWNMAP;
					NetworkInCodec.StreamLength = atol(Client->Temp);

					EncodingCodec = MWGetStream(NULL, MailSession->MIME[Element].Encoding, FALSE);
					CharsetCodec = MWGetStream(NULL, MailSession->MIME[Element].Charset, FALSE);

					/* Send the interpreted part to the file */
					FileOutCodec = MWGetStream(NULL, NULL, FALSE);
					FileOutCodec->StreamData = (void *)Handle;
					FileOutCodec->Codec = MWStreamToFile;

					if (EncodingCodec) {
						NetworkInCodec.Next = EncodingCodec;
						if (CharsetCodec) {
							EncodingCodec->Next = CharsetCodec;
							CharsetCodec->Next = FileOutCodec;
						} else {
							EncodingCodec->Next = FileOutCodec;
						}
					} else {
						if (CharsetCodec) {
							NetworkInCodec.Next = CharsetCodec;
							CharsetCodec->Next = FileOutCodec;
						} else {
							NetworkInCodec.Next = FileOutCodec;
						}
					}

					Client->NMAPProtected = TRUE;
					NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);
					Client->NMAPProtected = FALSE;

					if (EncodingCodec) {
						MWReleaseStream(EncodingCodec);
					}

					if (CharsetCodec) {
						MWReleaseStream(CharsetCodec);
					}

					MWReleaseStream(FileOutCodec);

					/* Grab the BRAW DONE code and close the file */
					MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

					fclose(Handle);
					Handle = NULL;

					break;
				}
			}

			if (AnswerType != COMPOSE_DELEGATE) {
				/* Build To: file */
				snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TO_LIST[COMPOSE_TO]);
				Handle = fopen(Client->Temp, "ab");
				if (Handle) {
					if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Reply-To:", &ptr)) > 0) {
						unsigned char	*p2;

						ch = ptr[ReplyInt];
						ptr[ReplyInt] = '\0';
						p2 = ptr;

						do {
							p2 = MwMailParseRFC822Address(p2, NULL, 0, Client->Temp, BUFSIZE+1);
							if (MWQuickCmp(Client->Temp, Session->EMailAddress) == FALSE) {
								fprintf(Handle, "%s\r\n", Client->Temp);
							}
						} while (p2);

						ptr[ReplyInt] = ch;
					} else if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "From:", &ptr)) > 0) {
						unsigned char	*p2;

						ch = ptr[ReplyInt];
						ptr[ReplyInt] = '\0';
						p2 = ptr;
						do {
							p2 = MwMailParseRFC822Address(p2, NULL, 0, Client->Temp, BUFSIZE+1);
							fprintf(Handle, "%s\r\n", Client->Temp);
						} while (p2);

						ptr[ReplyInt] = ch;
					}
				}
			}

			if (Handle) {
				fclose(Handle);
				Handle = NULL;
			}

			if (AnswerType == COMPOSE_DELEGATE) {
				ReplyPrefix = "Delegated: ";
			} else {
				ReplyPrefix = "Declined: ";
			}
			break;
		}

		default: {
			return(FALSE);
		}
	}

	/* Create the subject line */
	snprintf(Client->Temp, sizeof(Client->Temp), "%s/%x."COMPOSE_EXT_SUBJECT, MwMail.WorkDir, (unsigned int)Session->SessionID);
	Handle = fopen(Client->Temp, "wb");
	if (Handle) {
		if (MailSession->PrefixString != 0) {
			fprintf(Handle, "%s", Session->Strings[MailSession->PrefixString]);

			MailSession->PrefixString = 0;
		} else {
			fprintf(Handle, "%s", ReplyPrefix);
		}

		if ((ReplyInt = MwMailFindRFC822HeaderLine(MailSession->CachedHeader, "Subject:", &ptr)) > 0) {
			ch = ptr[ReplyInt];
			ptr[ReplyInt] = '\0';

			fprintf(Handle, "%s\r\n", ptr);

			ptr[ReplyInt] = ch;
		} else {
			fprintf(Handle, "\r\n");
		}

		fclose(Handle);
	}

	return(TRUE);
}


BOOL
MwMailHandleCalendarMessage(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long MessageID, BOOL Accept)
{
	FILE						*ICalFile;
	FILE						*ObjectFile;
	struct stat				sb;
	unsigned char			Answer[BUFSIZE + 1];
	unsigned char			Buffer[BUFSIZE + 1];
	unsigned char			*FullName;						/* Stores the user's fullname */
	MDBValueStruct			*User;							/* Used to read user information such as the full name */
	unsigned long			i;
	unsigned long			retVal;
	unsigned long			Day;
	unsigned long			Month;
	unsigned long			Year;
	unsigned long			Hour;
	unsigned long			Minute;
	StreamStruct			FileInCodec;
	StreamStruct			MemoryInCodec;
	StreamStruct			*RFC1522Codec		= NULL;
	StreamStruct			*Base64Codec		= NULL;
	StreamStruct			*MessageOutCodec	= NULL;
	StreamStruct			*RFC2445Codec		= NULL;
	StreamStruct			*RFC2445MLCodec	= NULL;
	BOOL						FoundPart=FALSE;
	unsigned long			Element;
	unsigned long			EndElement;
	unsigned long			PartNo = -1;
	ICalObject				*ICal;
	unsigned char			*ICalStream;
	unsigned long			ICalStreamSize;
	StreamStruct			NetworkInCodec;
	StreamStruct			*MemoryOutCodec;
	StreamStruct			*EncodingCodec;
	StreamStruct			*FileOutCodec;
	int						QueueNMAPs			= 0;
	int						ClientNMAPs			= 0;
	unsigned long			NMAPCAddress		= 0;
	ICalVAttendee            *Attendee;

	if (Session->ReadOnly) {
		return(FALSE);
	}

	/* An iCal reply needs the updated attendee information, the sequence number, the uid and a DTSTAMP */
	/* If we decline, we send a message back; if we accept we send a message back AND store the original in the calendar */

	/* Connect to NMAP server */
	ClientNMAPs=Session->NMAPs;
	if ((MwMail.NmapQAddress != 0) && (MwMail.NmapQAddress != Session->NMAPAddress.s_addr)) {
		unsigned char	*User;

		ClientNMAPs=Session->NMAPs;
		Session->NMAPs=-1;
		Session->NBufferPtr=0;
		NMAPCAddress=Session->NMAPAddress.s_addr;
		Session->NMAPAddress.s_addr=MwMail.NmapQAddress;
		User=Session->User;
		Session->User=NULL;
		if (!MWConnectUserToNMAPServer(Session)) {
			Session->User=User;
			MailSession->Error = ERROR_CONNECTING_TO_QUEUE;
			Session->NMAPs=ClientNMAPs;
			return(FALSE);
		}
		Session->User=User;
	}
	QueueNMAPs=Session->NMAPs;

	/*
		First, pull the ICal entry from NMAP and process it into an ICalObject;
		if no ical object contained set error return.
		Then check if we already have that entry in the store, set error and return if yes
		otherwise, generate new queue entry response
	*/


	/* Load the message */
	Session->NMAPs=ClientNMAPs;
	if (!MwMailLoadMIMECache(MessageID, Client, Session, MailSession)) {
		RestoreUserNMAP;
		return(FALSE);
	}

	MessageID--;

	/* Now find the text/calendar body part */
	for (Element=0; Element<MailSession->MIMEUsed && !FoundPart; Element++) {
		if (MWQuickCmp(MailSession->MIME[Element].Type, "multipart/alternative")) {
			Element++;
			for (EndElement=Element; EndElement<MailSession->MIMEUsed; EndElement++) {
				if (MailSession->MIME[Element].Depth>MailSession->MIME[EndElement].Depth) {
					break;
				}
			}
			/* We will overshoot above */
			EndElement--;

			for (i=EndElement; (i>=Element) && !FoundPart; i--) {
				if (MailSession->MIME[Element].Depth==MailSession->MIME[i].Depth) {
					if (MWQuickCmp(MailSession->MIME[i].Type, "text/calendar")) {
						FoundPart=TRUE;
						PartNo=i;
					}
				}
			}
		} else if (MWQuickCmp(MailSession->MIME[Element].Type, "multipart/mixed")) {
			Element++;
			for (i=Element; i<MailSession->MIMEUsed && !FoundPart; i++) {
				if (MailSession->MIME[Element].Depth==MailSession->MIME[i].Depth) {
					if (MWQuickCmp(MailSession->MIME[i].Type, "text/calendar")) {
						FoundPart=TRUE;
						PartNo=i;
					}
				}
			}
		} else if (MWQuickCmp(MailSession->MIME[Element].Type, "text/calendar")) {
			FoundPart=TRUE;
			PartNo=Element;
		}
	}

	if (!FoundPart) {
		RestoreUserNMAP;
		/* FIX-ME - report failure */
		return(FALSE);
	}

	/* Now we retrieve the part into memory, so that we can parse it */
	memset(&NetworkInCodec, 0, sizeof(StreamStruct));
	NetworkInCodec.StreamData=Session;
	NetworkInCodec.Codec=MWStreamFromMWNMAP;
	/* Resolve the encoding of the message part */

	EncodingCodec=MWGetStream(NULL, MailSession->MIME[PartNo].Encoding, FALSE);

	/* Send the interpreted part to RAM */
	MemoryOutCodec=MWGetStream(NULL, NULL, FALSE);
	MemoryOutCodec->Codec=MWStreamToMemory;

	/* Do the idiot user check */
	if (MessageID>=Session->NumOfMessages) {
		/* Shithead user */
		MWReleaseStream(EncodingCodec);
		MWReleaseStream(MemoryOutCodec);
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Request the body part; we're already connected to ClientNMAPs from the MwMailLoadMIMECache above */
	retVal=snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MessageID], MailSession->MIME[PartNo].PartStart, MailSession->MIME[PartNo].PartSize);
	MWSendNMAPServer(Session, Client->Temp, retVal);
	retVal=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
	if (retVal/10!=202) {
		MWReleaseStream(EncodingCodec);
		MWReleaseStream(MemoryOutCodec);
		RestoreUserNMAP;
		return(FALSE);
	}

	/* Let the code know how much data to expect */
	NetworkInCodec.StreamLength=atol(Client->Temp);
	/* Chain up properly */
	if (EncodingCodec && EncodingCodec->Codec) {
		NetworkInCodec.Next=EncodingCodec;
		EncodingCodec->Next=MemoryOutCodec;
	} else {
		NetworkInCodec.Next=MemoryOutCodec;
	}

	Client->NMAPProtected=TRUE;
	NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);
	Client->NMAPProtected=FALSE;

	/* Read BRAW 1000 OK */
	retVal=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
	Client->NMAPProtected=FALSE;

	ICalStream=MemoryOutCodec->StreamData;
	ICalStreamSize=MemoryOutCodec->StreamLength;
	ICalStream[ICalStreamSize]='\0';

	ICal=ICalParseObject(NULL, ICalStream, ICalStreamSize);


	MemFree(ICalStream);
	MWReleaseStream(EncodingCodec);
	MWReleaseStream(MemoryOutCodec);

	if (!ICal || ICal->Method!=ICAL_METHOD_REQUEST || !ICal->Organizer || !ICal->Attendee) {
		/* FIX-ME report error */
		RestoreUserNMAP;
		if (ICal) {
			ICalFreeObjects(ICal);
		}
		return(FALSE);
	}

	/* This seems like as good as place as any to update our attendee participation status */
	/* in the ICal object. This is required because we use ICalGenerateAttendeeFile which only */
	/* looks at the attendee object in the ICal Object. */

	/* put on Attendee list */
	for (Attendee = ICal->Attendee;
		  Attendee != NULL;
		  Attendee = Attendee->Next) {
		if (MWQuickCmp(Attendee->Address, Session->EMailAddress)==TRUE) {
			if (Accept)
				Attendee->State = ICAL_PARTSTAT_ACCEPTED;
			else
				Attendee->State = ICAL_PARTSTAT_DECLINED;
			
		}
	}


	/*
		We have the parsed object; now we need to do two things:
		a) Create a response to the original sender with our acceptance state
		b) Send it to our own calendar, if Accept is true
	*/

	/* Only send calendar message to organizer if there is an organizer and if we are not organizer, otherwise we are being weird  */
	if (ICal->Organizer && !MWQuickNCmp(ICal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress))) {

		/* Create Queue Entry */
		Session->NMAPs=QueueNMAPs;
		MWSendNMAPServer(Session, "QCREA\r\n", 7);
		if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		/* Start creating Header in file */
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
		ICalFile=fopen(Buffer, "wb");
		if (!ICalFile) {
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}
	
		/* Prepare all required streams */
		memset(&FileInCodec, 0, sizeof(StreamStruct));
		memset(&MemoryInCodec, 0, sizeof(StreamStruct));
		RFC1522Codec=MWGetStream(&MemoryInCodec, "RFC1522", TRUE);
		RFC2445Codec=MWGetStream(NULL, "RFC2445", TRUE);
		RFC2445MLCodec=MWGetStream(NULL, "RFC2445ML", TRUE);
		MessageOutCodec=MWGetStream(RFC1522Codec, NULL, TRUE);

		/* Set up the input & output */
		MemoryInCodec.StreamData=Buffer;
		MemoryInCodec.Codec=MWStreamFromMemory;
		MemoryInCodec.Next=RFC1522Codec;

		FileInCodec.Codec=MWStreamFromFile;
		FileInCodec.Next=RFC1522Codec;

		RFC2445MLCodec->Next=RFC2445Codec;
		RFC2445Codec->Next=MessageOutCodec;

		RFC1522Codec->Charset=Session->Charset;

		MessageOutCodec->StreamData=ICalFile;
		MessageOutCodec->Codec=MWStreamToFile;

		/* Start building the message */
		MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
		i=sizeof(Client->cs);
		getpeername(Client->s, (struct sockaddr *)&(Client->cs), (socklen_t *)&i);
		fprintf(ICalFile, "Received: from %s [%d.%d.%d.%d] by %s\r\n\twith %s; %s\r\n", 
				  Session->User,
				  Client->cs.sin_addr.s_net,
				  Client->cs.sin_addr.s_host,
				  Client->cs.sin_addr.s_lh,
				  Client->cs.sin_addr.s_impno,
				  Session->OfficialDomain, 
				  PRODUCT_NAME,
				  Buffer);

		/* Subject */
		fprintf(ICalFile, "Subject: ");
		if (ICal->Summary) {
			HulaStrNCpy(Buffer, ICal->Summary, sizeof(Buffer));
		} else {
			HulaStrNCpy(Buffer, "<no subject>", sizeof(Buffer));
		}
		ProcessMemoryStream(strlen(Buffer));

		if (Accept) {
			fprintf(ICalFile, " (Accepted)\r\n");
		} else {
			fprintf(ICalFile, " (Declined)\r\n");
		}

		User = MDBCreateValueStruct(MwMail.DirectoryHandle, NULL);
		/* Reply-To:*/
		if (MDBRead(Session->UserDN, MSGSRV_A_EMAIL_ADDRESS, User) > 0) {
			fprintf(ICalFile, "Reply-To: %s\r\n", User->Value[0]);
		}
		MDBDestroyValueStruct(User);

		/* From */
		fprintf(ICalFile, "From: ");
		retVal = MWGetQuotedString(Session->DisplayName, Client->Temp, sizeof(Client->Temp));
		if (retVal == QP_STRING) {
			fputs(Client->Temp, ICalFile);
			fprintf(ICalFile, " <%s>\r\n", Session->EMailAddress);
		} else if (retVal == HAS_EXTENDED_CHARS) {
			HulaStrNCpy(Buffer, Session->DisplayName, sizeof(Buffer));
			ProcessMemoryStream(strlen(Buffer));
			fprintf(ICalFile, " <%s>\r\n", Session->EMailAddress);
		} else {
			fprintf(ICalFile, "<%s>\r\n", Session->EMailAddress);
		}

		if ((FullName=MemStrdup(Client->Temp))==NULL) {
			RestoreUserNMAP;
			fclose(ICalFile);
			ICalFile=NULL;
			ReleaseCalendarStreams;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		retVal = snprintf(Answer, sizeof(Answer), "QSTOR FROM %s %s\r\n", Session->EMailAddress, Session->User);
		MWSendNMAPServer(Session, Answer, retVal);
		if ((MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) || !ICal->Organizer->Address) {
			RestoreUserNMAP;
			fclose(ICalFile);
			ICalFile=NULL;
			ReleaseCalendarStreams;
			ICalFreeObjects(ICal);
			MemFree(FullName);
			return(FALSE);
		}

		/* Add Organizer to the To: field */
		fprintf(ICalFile, "To: %s\r\n", ICal->Organizer->Address);
		retVal = snprintf(Answer, sizeof(Answer), "QSTOR TO %s %s %d\r\n", ICal->Organizer->Address, ICal->Organizer->Address, DSN_DEFAULT);
		MWSendNMAPServer(Session, Answer, retVal);

		if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE) != 1000) {
			RestoreUserNMAP;
			fclose(ICalFile);
			ICalFile=NULL;
			ReleaseCalendarStreams;
			ICalFreeObjects(ICal);
			MemFree(FullName);
			return(FALSE);
		}

		/* Date */
		MsgGetRFC822Date(Session->TimezoneOffset, 0, Buffer);
		fprintf(ICalFile, "Date: %s\r\n", Buffer);

		/* Prorietary information */
		fprintf(ICalFile, "X-Mailer: %s\r\n", PRODUCT_NAME);
		fprintf(ICalFile, "X-Sender: %s\r\n", Session->EMailAddress);

		/* MIME */
		fprintf(ICalFile, "MIME-Version: 1.0\r\nContent-Type: text/calendar; method=REPLY; charset=utf-8\r\n\r\n");

		FileInCodec.Next=RFC2445Codec;

		/* Now for the iCal object */
		fprintf(ICalFile, "BEGIN:VCALENDAR\r\nPRODID:-//Novell Inc//NetMail ModWeb//\r\nVERSION:2.0\r\nMETHOD:REPLY\r\n");

		switch(ICal->Type) {
			case ICAL_VEVENT:		fprintf(ICalFile, "BEGIN:VEVENT\r\n"); break;
			case ICAL_VTODO:		fprintf(ICalFile, "BEGIN:VTODO\r\n"); break;
			case ICAL_VJOURNAL:	fprintf(ICalFile, "BEGIN:VJOURNAL\r\n"); break;
		}

		ICalGenerateAttendeeFile(ICal, ICalFile);

		Attendee=ICal->Attendee;
		while (Attendee) {
			if (MWQuickCmp(Attendee->Address, Session->EMailAddress)) {
				break;
			}
			Attendee=Attendee->Next;
		}

		if (!Attendee) {
			/* If we are not the organizer and we are not in Attendee list then we are bcc */
			/* We must still let organizer know about our status update */
			fprintf(ICalFile, "ATTENDEE;CN=\"%s\";ROLE=\"NON-PARTICIPANT\";\r\n PARTSTAT=\"ACCEPTED\";CUTYPE=\"INDIVIDUAL\":MAILTO:%s\r\n",
			Session->DisplayName, Session->EMailAddress);			
		}

		MemFree(FullName);

		/* SEQUENCE */
		fprintf(ICalFile, "SEQUENCE:%d\r\n", (int)ICal->Sequence);

		/* UID */
		if (ICal->UID) {
		    ;
		} else {
		    snprintf(Buffer, sizeof(Buffer), "%d%s", (int)time(NULL), (ICal->Organizer->Address) ? (char *)ICal->Organizer->Address : "");
		    ICal->UID = MemStrdup(Buffer);
		}
		fprintf(ICalFile, "UID:%s\r\n", ICal->UID);
	
		/* DTSTAMP */
		MsgGetDMY(time(NULL), &Day, &Month, &Year, &Hour, &Minute, NULL);
		fprintf(ICalFile, "DTSTAMP:%04d%02d%02dT%02d%02d00Z\r\n", (int)Year, (int)Month, (int)Day, (int)Hour, (int)Minute);

		/* Close it off */

		switch(ICal->Type) {
			case ICAL_VEVENT:		fprintf(ICalFile, "END:VEVENT\r\nEND:VCALENDAR\r\n"); break;
			case ICAL_VTODO:		fprintf(ICalFile, "END:VTODO\r\nEND:VCALENDAR\r\n"); break;
			case ICAL_VJOURNAL:	fprintf(ICalFile, "END:VJOURNAL\r\nEND:VCALENDAR\r\n"); break;
		}

		fclose(ICalFile);
		ICalFile=NULL;
		ReleaseCalendarStreams;

		/* Send the sucker to NMAP */
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
		if ((stat(Buffer, &sb)==-1) || (ICalFile=fopen(Buffer, "rb"))==NULL) {
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		/* Tell NMAP the Message is coming */
		retVal=snprintf(Answer, sizeof(Answer), "QSTOR MESSAGE %d\r\n", (int)sb.st_size);
		MWSendNMAPServer(Session, Answer, retVal);

		while (!feof(ICalFile) && !ferror(ICalFile)) {
			if ((retVal=fread(Answer, 1, BUFSIZE, ICalFile))>0) {
				MWSendNMAPServer(Session, Answer, retVal);
			}
		}
		fclose(ICalFile);
		ICalFile=NULL;
	
		if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)!=1000) {
			ICalFreeObjects(ICal);
			RestoreUserNMAP;
			return(FALSE);
		}

		/* Kick it */
		MWSendNMAPServer(Session, "QRUN\r\n", 6);

		if (MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE)!=1000) {
			ICalFreeObjects(ICal);
			RestoreUserNMAP;
			return(FALSE);
		}
	}

	/* Now we need to let our calendar know if it has been accepted. */
	/* If it is being declined and we find our entry we need to update it */
	if (Accept) {

		ICalVAttendee	*Attendee;
		BOOL			SentAttendee=FALSE;


		/* If the object is already in our calendar we shouldn't put it there again */
		Session->NMAPs=ClientNMAPs;
		retVal = snprintf(Client->Temp, sizeof(Client->Temp), "CSFIND %lu %s\r\n", ICal->Sequence, ICal->UID);
		MWSendNMAPServer(Session, Client->Temp, retVal);
		retVal=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		if (retVal == 1000) {
			int sequence = atol(Client->Temp);

			/* object is in calendar don't create a new entry just update attendee status*/
			retVal = snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu A %s\r\n", (long unsigned int)sequence,Session->EMailAddress);
			MWSendNMAPServer(Session, Client->Temp, retVal);
			retVal = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);

			/* Now get rid of the mailbox calendar entry */
			retVal=snprintf(Answer, sizeof(Answer), "AFLG %lu %lu\r\n", (long unsigned int)MailSession->IDList[MessageID], (long unsigned int)MSG_STATE_DELETED);
			MWSendNMAPServer(Session, Answer, retVal);
			retVal=MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE);

			/* We need to let the cache know that things have changed */
			MwMailRefreshFolder(Session, MailSession);
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		/* Now comes the tricky part; we need to receive the ical object again, and filter the attendee lines, but keep everything else in */

		/* Prepare the attendee list... */
		Attendee=ICal->Attendee;
		while (Attendee) {
			if (MWQuickCmp(Attendee->Address, Session->EMailAddress)) {
				Attendee->State=ICAL_PARTSTAT_ACCEPTED;
			}
			Attendee=Attendee->Next;
		}

		/* Do the idiot user check */
		if (MessageID>=Session->NumOfMessages) {
			/* Shithead user */
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		/* We need to stream the part down, decode any QP/B64 and then read line by line to strip the ATTENDEE lines */
		memset(&NetworkInCodec, 0, sizeof(StreamStruct));
		NetworkInCodec.StreamData=Session;
		NetworkInCodec.Codec=MWStreamFromMWNMAP;
		/* Resolve the encoding of the message part */

		EncodingCodec=MWGetStream(NULL, MailSession->MIME[PartNo].Encoding, FALSE);

		/* Create ical read file */
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TEMP);
		ObjectFile=fopen(Buffer, "wb");
		if (!ObjectFile) {
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		/* Send the interpreted part to RAM */
		FileOutCodec=MWGetStream(NULL, NULL, FALSE);
		FileOutCodec->Codec=MWStreamToFile;
		FileOutCodec->StreamData=ObjectFile;

		/* Request the body part; we're already connected to ClientNMAPs from the MwMailLoadMIMECache above */
		retVal=snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MessageID], MailSession->MIME[PartNo].PartStart, MailSession->MIME[PartNo].PartSize);
		MWSendNMAPServer(Session, Client->Temp, retVal);
		retVal=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		if (retVal/10!=202) {
			MWReleaseStream(EncodingCodec);
			MWReleaseStream(FileOutCodec);
			ICalFreeObjects(ICal);
			RestoreUserNMAP;
			return(FALSE);
		}

		/* Let the code know how much data to expect */
		NetworkInCodec.StreamLength=atol(Client->Temp);

		/* Chain up properly */
		if (EncodingCodec && EncodingCodec->Codec) {
			NetworkInCodec.Next=EncodingCodec;
			EncodingCodec->Next=FileOutCodec;
		} else {
			NetworkInCodec.Next=FileOutCodec;
		}

		Client->NMAPProtected=TRUE;
		NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);
		Client->NMAPProtected=FALSE;

		/* Read BRAW 1000 OK */
		retVal=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		Client->NMAPProtected=FALSE;

		fclose(ObjectFile);
		MWReleaseStream(EncodingCodec);
		MWReleaseStream(FileOutCodec);

		/* Open ical read file */
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TEMP);
		ObjectFile=fopen(Buffer, "rb");
		if (!ObjectFile) {
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}
		/* Create ical send file */
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
		ICalFile=fopen(Buffer, "wb");
		if (!ICalFile) {
			RestoreUserNMAP;
			fclose(ObjectFile);
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		while (!feof(ObjectFile) && !ferror(ObjectFile)) {
			if (fgets(Client->Temp, BUFSIZE, ObjectFile)) {
				if (MWQuickNCmp(Client->Temp, "ATTENDEE", 8)!=TRUE) {
					if (!(i && isspace(Client->Temp[0]))) {
						fprintf(ICalFile, "%s", Client->Temp);
						i=0;
					}
				} else {
					i=1;
					if (!SentAttendee) {
						ICalGenerateAttendeeFile(ICal, ICalFile);
						SentAttendee=TRUE;
					}
				}
			}
		}
		fclose(ObjectFile);
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_TEMP);
		unlink(Buffer);
		fclose(ICalFile);
		ICalFile=NULL;

		/* We got the ICal object updated to reflect us accepting; now send it, back to queue connection */
		/* Send directly to the calendar */
		snprintf(Buffer, sizeof(Buffer), "%s/%x.%s", MwMail.WorkDir, (unsigned int)Session->SessionID, COMPOSE_EXT_MESSAGE);
		if ((stat(Buffer, &sb)==-1) || (ICalFile=fopen(Buffer, "rb"))==NULL) {
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}

		MWSendNMAPServer(Session, "CSSTOR MAIN\r\n", 13);
		if (MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE) != 2002) {
			fclose(ICalFile);
			RestoreUserNMAP;
			ICalFreeObjects(ICal);
			return(FALSE);
		}	

		while (!feof(ICalFile) && !ferror(ICalFile)) {
			if ((retVal=fread(Answer, 1, BUFSIZE, ICalFile))>0) {
				MWSendNMAPServer(Session, Answer, retVal);
			}
		}
		fclose(ICalFile);
		ICalFile=NULL;

		MWSendNMAPServer(Session, "\r\n.\r\n", 5);
		MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE);

	} else {
		/* If the object is already in our calendar we shouldn't put it there again */
		Session->NMAPs=ClientNMAPs;
		retVal = snprintf(Client->Temp, sizeof(Client->Temp), "CSFIND %lu %s\r\n", ICal->Sequence, ICal->UID);
		MWSendNMAPServer(Session, Client->Temp, retVal);
		retVal=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		if (retVal == 1000) {
			long unsigned int sequence = atol(Client->Temp);

			/* object is in calendar see if we need to delete it our just update the attendee status */
			/* remove from our calendar if there is no organizer or if we are not the organizer or if */
			/* there are not recipients or we are the only recipient for message */
			if (!ICal->Organizer || !MWQuickNCmp(ICal->Organizer->Address, Session->EMailAddress, strlen(Session->EMailAddress)) ||
				 !ICal->Attendee  || (!ICal->Attendee->Next && MWQuickNCmp(ICal->Organizer->Address, ICal->Attendee->Address, strlen(ICal->Organizer->Address)))) {

				/* Now get rid of the entry in the calendar */
				retVal = snprintf(Client->Temp, sizeof(Client->Temp), "CSDELE %lu\r\n", sequence);
				MWSendNMAPServer(Session, Client->Temp, retVal);
				/* We grab the response, but we wouldn't know how to handle an error, so we just continue... */
				retVal = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);
				MWSendNMAPServer(Session, "CSPURG\r\n", 9);
				retVal = MWGetNMAPAnswer(Session, Client->Temp, sizeof(Client->Temp), TRUE);

			} else {
				retVal = snprintf(Client->Temp, sizeof(Client->Temp), "CSATND %lu D %s\r\n", (long unsigned int)sequence, Session->EMailAddress);
				MWSendNMAPServer(Session, Client->Temp, retVal);
				retVal = MWGetNMAPAnswer(Session,Client->Temp, BUFSIZE, TRUE);
			}

			RestoreUserNMAP;
			ICalFreeObjects(ICal);

			/* We need to let the cache know that things have changed */
			MwMailRefreshFolder(Session, MailSession);
			return(FALSE);
		}


	}

	RestoreUserNMAP;

	/* Whack our temp files */
	MwMailComposeCleanUp(Session, MailSession);
	ICalFreeObjects(ICal);

	/* Now get rid of the entry in the calendar */
	retVal=snprintf(Answer, sizeof(Answer), "AFLG %lu %lu\r\n", (long unsigned int)MailSession->IDList[MessageID], (long unsigned int)MSG_STATE_DELETED);
	MWSendNMAPServer(Session, Answer, retVal);
	retVal=MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE);

	/* We need to let the cache know that things have changed */
	MwMailRefreshFolder(Session, MailSession);
	return(TRUE);
}

