/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include <xpl.h>
#include <memmgr.h>
#include <logger.h>
#include <hulautil.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>
#include <msgapi.h>
#include <streamio.h>
#include <openssl/md5.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <connio.h>

#include "forward.h"

#if defined(RELEASE_BUILD)
#define ForwardClientAlloc() MemPrivatePoolGetEntry(Forward.nmap.pool)
#else
#define ForwardClientAlloc() MemPrivatePoolGetEntryDebug(Forward.nmap.pool, __FILE__, __LINE__)
#endif

static void SignalHandler(int sigtype);

#define QUEUE_WORK_TO_DO(c, id, r) \
        { \
            XplWaitOnLocalSemaphore(Forward.nmap.semaphore); \
            if (XplSafeRead(Forward.nmap.worker.idle)) { \
                (c)->queue.previous = NULL; \
                if (((c)->queue.next = Forward.nmap.worker.head) != NULL) { \
                    (c)->queue.next->queue.previous = (c); \
                } else { \
                    Forward.nmap.worker.tail = (c); \
                } \
                Forward.nmap.worker.head = (c); \
                (r) = 0; \
            } else { \
                XplSafeIncrement(Forward.nmap.worker.active); \
                XplSignalBlock(); \
                XplBeginThread(&(id), HandleConnection, 24 * 1024, XplSafeRead(Forward.nmap.worker.active), (r)); \
                XplSignalHandler(SignalHandler); \
                if (!(r)) { \
                    (c)->queue.previous = NULL; \
                    if (((c)->queue.next = Forward.nmap.worker.head) != NULL) { \
                        (c)->queue.next->queue.previous = (c); \
                    } else { \
                        Forward.nmap.worker.tail = (c); \
                    } \
                    Forward.nmap.worker.head = (c); \
                } else { \
                    XplSafeDecrement(Forward.nmap.worker.active); \
                    (r) = -1; \
                } \
            } \
            XplSignalLocalSemaphore(Forward.nmap.semaphore); \
        }


ForwardGlobals Forward;

static BOOL 
ForwardClientAllocCB(void *buffer, void *data)
{
    register ForwardClient *c = (ForwardClient *)buffer;

    memset(c, 0, sizeof(ForwardClient));

    return(TRUE);
}

static void 
ForwardClientFree(ForwardClient *client)
{
    register ForwardClient *c = client;

    if (c->conn) {
        ConnClose(c->conn, 1);
        ConnFree(c->conn);
        c->conn = NULL;
    }

    MemPrivatePoolReturnEntry(c);

    return;
}

static void 
FreeClientData(ForwardClient *client)
{
    if (client && !(client->flags & FORWARD_CLIENT_FLAG_EXITING)) {
        client->flags |= FORWARD_CLIENT_FLAG_EXITING;

        if (client->conn) {
            ConnClose(client->conn, 1);
            ConnFree(client->conn);
            client->conn = NULL;
        }

        if (client->envelope) {
            MemFree(client->envelope);
        }
    }

    return;
}

static unsigned char *
ProcessSender(unsigned char *line)
{
    register unsigned char *ptr = line;
    register unsigned char c;
    register unsigned char *from;

    while (*ptr && !isspace(*ptr)) {
        ptr++;
    }

    c = *ptr;
    *ptr = '\0';
    from = MemStrdup(line);
    *ptr = c;

    return(from);
}

static int 
ProcessConnection(ForwardClient *client)
{
    int ccode;
    int count;
    int flags;
    int length;
    unsigned long used;
    unsigned long fwdFlags;
    unsigned char delim;
    unsigned char preserve;
    unsigned char *cur;
    unsigned char *ptr;
    unsigned char *eol;
    unsigned char *email = NULL;
    unsigned char *line;
    unsigned char *marker;
    unsigned char *domain;
    unsigned char *subject;
    unsigned char *from = NULL;
    MDBValueStruct *vs;
    MDBValueStruct *forwards;
    BOOL write;
    BOOL copy;

    if (((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) != -1) 
            && (ccode == 6020) 
            && ((ptr = strchr(client->line, ' ')) != NULL)) {
        *ptr++ = '\0';

        strcpy(client->queueID, client->line);

        length = atoi(ptr);
        client->envelope = MemMalloc(length + 3);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if (client->envelope) {
        sprintf(client->line, "Forward: %s", client->queueID);
        XplRenameThread(XplGetThreadID(), client->line);

        ccode = ConnRead(client->conn, client->envelope, length);
    } else {
        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    if ((ccode != -1) 
            && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 6021)) {
        client->envelope[length] = '\0';

        cur = client->envelope;
    } else {
        MemFree(client->envelope);
        client->envelope = NULL;

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);
        NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
        return(-1);
    }

    eol = NULL;
    while ((ccode != -1) && *cur) {
        copy = TRUE;

        if (eol) {
            *eol = preserve;
        }

        line = strchr(cur, 0x0A);
        if (line) {
            if (line[-1] == 0x0D) {
                eol = line - 1;
            } else {
                eol = line;
            }

            preserve = *eol;
            *eol = '\0';

            line++;
        } else {
            eol = NULL;
            line = cur + strlen(cur);
        }

        switch (cur[0]) {
            case QUEUE_FROM: {
                if (!from) {
                    from = ProcessSender(cur + 1);
                    if (from) {
                        copy = FALSE;

                        ccode = NMAPSendCommandF(client->conn, "QMOD RAW %s\r\n", cur);
                    }
                }

                break;
            }

            case QUEUE_CALENDAR_LOCAL: 
            case QUEUE_RECIP_LOCAL: {
                ptr = strrchr(cur, ' ');
                if (ptr) {
                    flags = atol(ptr + 1);
                } else {
                    flags = 0;
                }

                if (!(flags & NO_FORWARD)) {
                    domain = NULL;
                    marker = cur + 1;
                    while (*marker && !isspace(*marker)) {
                        if (*marker != '@') {
                            marker++;
                            continue;
                        }

                        domain = marker++;
                    }

                    delim = *marker;
                    *marker = '\0';

                    strcpy(client->recipient, cur + 1);

                    if (marker) {
                        *marker = delim;
                    }

                    vs = MDBCreateValueStruct(Forward.handle.directory, NULL);
                    if (vs) {
                        forwards = MDBShareContext(vs);
                    } else {
                        LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, NULL, sizeof(MDBValueStruct), __LINE__, NULL, 0);
                        break;
                    }

                    if (forwards) {
                        if (MsgFindObject(client->recipient, client->dn, NULL, NULL, vs)) {
                            strcpy(client->user, vs->Value[0]);

                            MDBFreeValues(vs);
                        } else {
                            LoggerEvent(Forward.handle.logging, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_UNKNOWN_USER, LOG_NOTICE, 0, client->recipient, "", XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);

                            MDBDestroyValueStruct(forwards);
                            MDBDestroyValueStruct(vs);
                            break;
                        }
                    } else {
                        LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, NULL, 16, __LINE__, NULL, 0);

                        MDBDestroyValueStruct(vs);
                        break;
                    }


                    if (MsgGetUserFeature(client->dn, FEATURE_FORWARD, MSGSRV_A_FORWARDING_ENABLED, vs)) {
                        fwdFlags = atol(vs->Value[0]);
                        if (fwdFlags 
                                && MsgGetUserFeature(client->dn, FEATURE_FORWARD, MSGSRV_A_FORWARDING_ADDRESS, forwards)) {
                            for (used = 0; (ccode != -1) && (used < forwards->Used); used++) {
                                LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_QUEUE, LOGGER_EVENT_FORWARD, LOG_INFO, 0, client->recipient, forwards->Value[used], 0, 0, NULL, 0);

                                if (domain) {
                                    ccode = NMAPSendCommandF(client->conn, "QMOD TO %s %s %d\r\n", forwards->Value[used], client->recipient, flags | NO_FORWARD);
                                } else {
                                    ccode = NMAPSendCommandF(client->conn, "QMOD TO %s %s@%s %d\r\n", forwards->Value[used], client->recipient, Forward.officialName, flags | NO_FORWARD);
                                }
                            }
                        }

                        if (fwdFlags == 1) {
                            copy = FALSE;
                        }

                        MDBFreeValues(vs);
                    }

                    if (MsgGetUserFeature(client->dn, FEATURE_AUTOREPLY, MSGSRV_A_AUTOREPLY_ENABLED, vs) 
                            && (atol(vs->Value[0]) != 0) 
                            && (flags & DSN_FAILURE)) {
                        MDBFreeValues(vs);

                        if (((subject = (unsigned char *)MemMalloc(32)) != NULL) 
                                && ((ccode = NMAPSendCommandF(client->conn, "QSRCH HEADER %s Precedence: bulk\r\n", client->queueID)) != -1) 
                                && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 4262) 
                                && ((ccode = NMAPSendCommandF(client->conn, "QGREP %s Subject:\r\n", client->queueID)) != -1)) {
                            strcpy(subject, "Subject: Re: ");

                            write = FALSE;
                            count = 13;

                            while ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == 2002) {
                                if (count == 13) {
                                    length = strlen(client->line);
                                    ptr = MemRealloc(subject, count + length + 1);
                                    if (ptr) {
                                        subject = ptr;
                                        write = TRUE;

                                        if (client->line[8] == ' ') {
                                            count = length - 9;
                                            memcpy(subject + 13, client->line + 9, count);

                                            count += 13;
                                        } else if (client->line[8] == '\0') {
                                            count = 13;
                                        } else {
                                            count = length - 8;
                                            memcpy(subject + 13, client->line + 8, count);

                                            count += 13;
                                        }

                                        subject[count] = '\0';

                                        continue;
                                    }

                                    LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, NULL, count + length + 1, __LINE__, NULL, 0);
                                    continue;
                                }

                                if (write) {
                                    length = strlen(client->line);
                                    ptr = MemRealloc(subject, count + length + 3);
                                    if (ptr) {
                                        subject = ptr;

                                        subject[count++] = '\r';
                                        subject[count++] = '\n';

                                        memcpy(subject + count, client->line, length);

                                        count += length;

                                        subject[count] = '\0';
                                        continue;
                                    }

                                    write = FALSE;

                                    LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, NULL, count + length + 3, __LINE__, NULL, 0);
                                    continue;
                                }
                            }

                            if ((ccode == NMAP_OK) 
                                    && ((ccode = NMAPSendCommand(client->conn, "QCREA\r\n", 7)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == NMAP_OK)) {
                                if (domain) {
                                    ccode = NMAPSendCommandF(client->conn, "QSTOR FROM %s %s -\r\n", client->recipient, client->recipient);
                                } else if ((email = MsgGetUserEmailAddress(client->dn, vs, NULL, 0)) != NULL) {
                                    ccode = NMAPSendCommandF(client->conn, "QSTOR FROM %s %s -\r\n", email, email);
                                } else if (MsgGetParentAttribute(client->dn, MSGSRV_A_DOMAIN, vs)) {
                                    ccode = NMAPSendCommandF(client->conn, "QSTOR FROM %s@%s %s@%s -\r\n", client->recipient, vs->Value[0], client->recipient, vs->Value[0]);
                                } else {
                                    ptr = strrchr(client->dn, '\\');
                                    if (ptr) {
                                        *ptr = '\0';
                                    }

                                    if (MDBRead(client->dn, MSGSRV_A_DOMAIN, vs)) {
                                        ccode = NMAPSendCommandF(client->conn, "QSTOR FROM %s@%s %s@%s -\r\n", client->recipient, vs->Value[0], client->recipient, vs->Value[0]);
                                    } else {
                                        ccode = NMAPSendCommandF(client->conn, "QSTOR FROM %s@%s %s@%s -\r\n", client->recipient, Forward.officialName, client->recipient, Forward.officialName);
                                    }

                                    if (ptr) {
                                        *ptr = '\\';
                                    }
                                }

                                if (count == 13) {
                                    strcpy(subject, "Subject: Automated Reply");
                                }
                            } else {
                                ccode = -1;
                            }

                            if ((ccode != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == NMAP_OK) 
                                    && ((ccode = NMAPSendCommandF(client->conn, "QSTOR TO %s %s %d\r\n", from, from, flags | NO_FORWARD)) != -1) 
                                    && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == NMAP_OK) 
                                    && ((ccode = ConnWrite(client->conn, "QSTOR MESSAGE\r\n", 15)) != -1)) {
                                if (domain) {
                                    ccode = ConnWriteF(client->conn, "From: %s\r\n", client->recipient);
                                } else if (email) {
                                    ccode = NMAPSendCommandF(client->conn, "From: %s\r\n", email);
                                } else if (vs->Used) {
                                    ccode = ConnWriteF(client->conn, "From: %s@%s\r\n", client->recipient, vs->Value[0]);
                                } else {
                                    ccode = ConnWriteF(client->conn, "From: %s@%s\r\n", client->recipient, Forward.officialName);
                                }
                            }

                            if ((ccode != -1) 
                                    && ((ccode = ConnWriteF(client->conn, "To: %s\r\n", from)) != -1) 
                                    && (MsgGetRFC822Date(-1, 0, client->line)) 
                                    && ((ccode = ConnWriteF(client->conn, "Date: %s\r\n", client->line)) != -1) 
                                    && ((ccode = ConnWriteF(client->conn, "%s\r\n", subject)) != -1) 
                                    && ((ccode = ConnWrite(client->conn, "Precedence: bulk\r\n", 18)) != -1) 
                                    && ((ccode = ConnWrite(client->conn, "X-Sender: Hula Auto-Reply Agent\r\n", 33)) != -1) 
                                    && ((ccode = ConnWrite(client->conn, "MIME-Version: 1.0\r\n", 19)) != -1) 
                                    && ((ccode = ConnWrite(client->conn, "Content-Type: text/plain; charset=\"UTF-8\"\r\n", 43)) != -1) 
                                    && ((ccode = ConnWrite(client->conn, "Content-Transfer-Encoding: 8bit\r\n\r\n", 35)) != -1)) {
                                MDBFreeValues(vs);

                                if (MsgGetUserFeature(client->dn, FEATURE_AUTOREPLY, MSGSRV_A_AUTOREPLY_MESSAGE, vs)) {
                                    ccode = ConnWrite(client->conn, vs->Value[0], strlen(vs->Value[0]));

                                    MDBFreeValues(vs);
                                } else {
                                    ccode = ConnWrite(client->conn, "--no auto-reply message set--\r\n", 31);
                                }
                            }

                            if (ccode != -1) {
                                if (((ccode = NMAPSendCommand(client->conn, "\r\n.\r\nQRUN\r\n", 11)) != -1) 
                                        && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == NMAP_OK) 
                                        && ((ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE)) == NMAP_OK)) {
                                    LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_QUEUE, LOGGER_EVENT_AUTO_REPLY, LOG_INFO, 0, client->recipient, from, 0, 0, NULL, 0);
                                }
                            } else {
                                NMAPSendCommand(client->conn, "\r\n.\r\nQABRT\r\n", 12);
                                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, TRUE);
                            }
                        } else if (!subject) {
                            LoggerEvent(Forward.handle.directory, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_ERROR, 0, __FILE__, NULL, 16, __LINE__, NULL, 0);
                        }

                        if (email) {
                            MemFree(email);
                        }

                        if (subject) {
                            MemFree(subject);
                        }
                    }

                    MDBDestroyValueStruct(forwards);
                    MDBDestroyValueStruct(vs);

                    break;
                }
            }

            case QUEUE_ADDRESS: 
            case QUEUE_BOUNCE: 
            case QUEUE_DATE: 
            case QUEUE_ID: 
            case QUEUE_RECIP_MBOX_LOCAL: 
            case QUEUE_RECIP_REMOTE: 
            case QUEUE_THIRD_PARTY: 
            case QUEUE_FLAGS: 
            default: {
                break;
            }
        }

        if (copy && (ccode != -1)) {
            ccode = NMAPSendCommandF(client->conn, "QMOD RAW %s\r\n", cur);
        }

        cur = line;
    }

    if (eol) {
        *eol = preserve;
    }

    if ((ccode != -1) 
            && ((ccode = NMAPSendCommand(client->conn, "QDONE\r\n", 7)) != -1)) {
        ccode = NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
    }

    if (client->envelope) {
        MemFree(client->envelope);
        client->envelope = NULL;
    }

    return(0);
}

static void 
HandleConnection(void *param)
{
    int ccode;
    long threadNumber = (long)param;
    time_t sleep = time(NULL);
    time_t wokeup;
    ForwardClient *client;

    if ((client = ForwardClientAlloc()) == NULL) {
        XplConsolePrintf("hulaforward: New worker failed to startup; out of memory.\r\n");

        NMAPSendCommand(client->conn, "QDONE\r\n", 7);

        XplSafeDecrement(Forward.nmap.worker.active);

        return;
    }

    do {
        XplRenameThread(XplGetThreadID(), "Forward Worker");

        XplSafeIncrement(Forward.nmap.worker.idle);

        XplWaitOnLocalSemaphore(Forward.nmap.worker.todo);

        XplSafeDecrement(Forward.nmap.worker.idle);

        wokeup = time(NULL);

        XplWaitOnLocalSemaphore(Forward.nmap.semaphore);

        client->conn = Forward.nmap.worker.tail;
        if (client->conn) {
            Forward.nmap.worker.tail = client->conn->queue.previous;
            if (Forward.nmap.worker.tail) {
                Forward.nmap.worker.tail->queue.next = NULL;
            } else {
                Forward.nmap.worker.head = NULL;
            }
        }

        XplSignalLocalSemaphore(Forward.nmap.semaphore);

        if (client->conn) {
            if (ConnNegotiate(client->conn, Forward.nmap.ssl.context)) {
                ccode = ProcessConnection(client);
            } else {
                NMAPSendCommand(client->conn, "QDONE\r\n", 7);
                NMAPReadAnswer(client->conn, client->line, CONN_BUFSIZE, FALSE);
            }
        }

        if (client->conn) {
            ConnFlush(client->conn);
        }

        FreeClientData(client);

        /* Live or die? */
        if (threadNumber == XplSafeRead(Forward.nmap.worker.active)) {
            if ((wokeup - sleep) > Forward.nmap.sleepTime) {
                break;
            }
        }

        sleep = time(NULL);

        ForwardClientAllocCB(client, NULL);
    } while (Forward.state == FORWARD_STATE_RUNNING);

    FreeClientData(client);

    ForwardClientFree(client);

    XplSafeDecrement(Forward.nmap.worker.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void 
ForwardServer(void *ignored)
{
    int i;
    int ccode;
    XplThreadID id;
    Connection *conn;

    XplSafeIncrement(Forward.server.active);

    XplRenameThread(XplGetThreadID(), "Forward Server");

    while (Forward.state < FORWARD_STATE_STOPPING) {
        if (ConnAccept(Forward.nmap.conn, &conn) != -1) {
            if (Forward.state < FORWARD_STATE_STOPPING) {
                conn->ssl.enable = FALSE;

                QUEUE_WORK_TO_DO(conn, id, ccode);
                if (!ccode) {
                    XplSignalLocalSemaphore(Forward.nmap.worker.todo);

                    continue;
                }
            }

            ConnWrite(conn, "QDONE\r\n", 7);
            ConnClose(conn, 0);

            ConnFree(conn);
            conn = NULL;

            continue;
        }

        switch (errno) {
            case ECONNABORTED:
#ifdef EPROTO
            case EPROTO: 
#endif
            case EINTR: {
                if (Forward.state < FORWARD_STATE_STOPPING) {
                    LoggerEvent(Forward.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);
                }

                continue;
            }

            default: {
                if (Forward.state < FORWARD_STATE_STOPPING) {
                    XplConsolePrintf("hulaforward: Exiting after an accept() failure; error %d\r\n", errno);

                    LoggerEvent(Forward.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);

                    Forward.state = FORWARD_STATE_STOPPING;
                }

                break;
            }
        }

        break;
    }

    /* Shutting down */
    Forward.state = FORWARD_STATE_UNLOADING;

    XplConsolePrintf("hulaforward: Shutting down.\r\n");

    id = XplSetThreadGroupID(Forward.id.group);

    if (Forward.nmap.conn) {
        ConnClose(Forward.nmap.conn, 1);
        Forward.nmap.conn = NULL;
    }

    if (Forward.nmap.ssl.enable) {
        Forward.nmap.ssl.enable = FALSE;

        if (Forward.nmap.ssl.conn) {
            ConnClose(Forward.nmap.ssl.conn, 1);
            Forward.nmap.ssl.conn = NULL;
        }

        if (Forward.nmap.ssl.context) {
            ConnSSLContextFree(Forward.nmap.ssl.context);
            Forward.nmap.ssl.context = NULL;
        }
    }

    ConnCloseAll(1);

    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (i = 0; (XplSafeRead(Forward.server.active) > 1) && (i < 60); i++) {
        XplDelay(1000);
    }

    for (i = 0; (ManagementState() != MANAGEMENT_STOPPED) && (i < 60); i++) {
        XplDelay(1000);
    }

    XplConsolePrintf("hulaforward: Shutting down %d queue threads\r\n", XplSafeRead(Forward.nmap.worker.active));

    XplWaitOnLocalSemaphore(Forward.nmap.semaphore);

    ccode = XplSafeRead(Forward.nmap.worker.idle);
    while (ccode--) {
        XplSignalLocalSemaphore(Forward.nmap.worker.todo);
    }

    XplSignalLocalSemaphore(Forward.nmap.semaphore);

    for (i = 0; XplSafeRead(Forward.nmap.worker.active) && (i < 60); i++) {
        XplDelay(1000);
    }

    if (XplSafeRead(Forward.server.active) > 1) {
        XplConsolePrintf("hulaforward: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(Forward.server.active) - 1);
    }

    if (XplSafeRead(Forward.nmap.worker.active)) {
        XplConsolePrintf("hulaforward: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(Forward.nmap.worker.active));
    }

    LoggerClose(Forward.handle.logging);
    Forward.handle.logging = NULL;

    XplCloseLocalSemaphore(Forward.nmap.worker.todo);
    XplCloseLocalSemaphore(Forward.nmap.semaphore);

    MsgShutdown();

    ConnShutdown();

    if (Forward.replyMessage) {
        MemFree(Forward.replyMessage);
        Forward.replyMessage = NULL;
    }

    MemPrivatePoolFree(Forward.nmap.pool);

    MemoryManagerClose(MSGSRV_AGENT_AUTOREPLY);

    XplConsolePrintf("hulaforward: Shutdown complete\r\n");

    XplSignalLocalSemaphore(Forward.sem.main);
    XplWaitOnLocalSemaphore(Forward.sem.shutdown);

    XplCloseLocalSemaphore(Forward.sem.shutdown);
    XplCloseLocalSemaphore(Forward.sem.main);

    XplSetThreadGroupID(id);

    return;
}

static BOOL 
ReadConfiguration(void)
{
    MDBValueStruct *vs;

    vs = MDBCreateValueStruct(Forward.handle.directory, MsgGetServerDN(NULL));
    if (vs) {
        if (MDBRead(MSGSRV_AGENT_AUTOREPLY, MSGSRV_A_AUTOREPLY_MESSAGE, vs)) {
            Forward.replyMessage = (unsigned char *)MemStrdup(vs->Value[0]);

            MDBFreeValues(vs);
        }

        if (MDBRead(MDB_CURRENT_CONTEXT, MSGSRV_A_OFFICIAL_NAME, vs)) {
            strcpy(Forward.officialName, vs->Value[0]);
            MDBFreeValues(vs);
        } else {
            Forward.officialName[0] = '\0';
        }

        MDBDestroyValueStruct(vs);

        return(TRUE);
    }

    return(FALSE);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
static int 
_NonAppCheckUnload(void)
{
    static BOOL    checked = FALSE;
    XplThreadID    id;

    if (!checked) {
        checked = TRUE;
        Forward.state = FORWARD_STATE_UNLOADING;

        XplWaitOnLocalSemaphore(Forward.sem.shutdown);

        id = XplSetThreadGroupID(Forward.id.group);
        ConnClose(Forward.nmap.conn, 1);
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(Forward.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (Forward.state < FORWARD_STATE_UNLOADING) {
                Forward.state = FORWARD_STATE_UNLOADING;
            }

            break;
        }

        case SIGINT:
        case SIGTERM: {
            if (Forward.state == FORWARD_STATE_STOPPING) {
                XplUnloadApp(getpid());
            } else if (Forward.state < FORWARD_STATE_STOPPING) {
                Forward.state = FORWARD_STATE_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

static int 
QueueSocketInit(void)
{
    Forward.nmap.conn = ConnAlloc(FALSE);
    if (Forward.nmap.conn) {
        memset(&(Forward.nmap.conn->socketAddress), 0, sizeof(Forward.nmap.conn->socketAddress));

        Forward.nmap.conn->socketAddress.sin_family = AF_INET;
        Forward.nmap.conn->socketAddress.sin_addr.s_addr = MsgGetAgentBindIPAddress();

        /* Get root privs back for the bind.  It's ok if this fails -
           the user might not need to be root to bind to the port */
        XplSetEffectiveUserId(0);

        Forward.nmap.conn->socket = ConnServerSocket(Forward.nmap.conn, 2048);
        if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
            XplConsolePrintf("hulaforward: Could not drop to unprivileged user '%s'\r\n", MsgGetUnprivilegedUser());
            ConnFree(Forward.nmap.conn);
            Forward.nmap.conn = NULL;
            return(-1);
        }

        if (Forward.nmap.conn->socket == -1) {
            XplConsolePrintf("hulaforward: Could not bind to dynamic port\r\n");
            ConnFree(Forward.nmap.conn);
            Forward.nmap.conn = NULL;
            return(-1);
        }

        if (NMAPRegister(MSGSRV_AGENT_AUTOREPLY, Forward.nmap.queue, Forward.nmap.conn->socketAddress.sin_port) != REGISTRATION_COMPLETED) {
            XplConsolePrintf("hulaforward: Could not register with hulanmap\r\n");
            ConnFree(Forward.nmap.conn);
            Forward.nmap.conn = NULL;
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaforward: Could not allocate connection.\r\n");
        return(-1);
    }

    return(0);
}

XplServiceCode(SignalHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int                ccode;
    XplThreadID        id;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaforward: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
        return(1);
    }

    XplSignalHandler(SignalHandler);

    Forward.id.main = XplGetThreadID();
    Forward.id.group = XplGetThreadGroupID();

    Forward.state = FORWARD_STATE_INITIALIZING;

    Forward.replyMessage = NULL;

    Forward.nmap.conn = NULL;
    Forward.nmap.queue = Q_FORWARD;
    Forward.nmap.pool = NULL;
    Forward.nmap.sleepTime = (5 * 60);
    Forward.nmap.ssl.conn = NULL;
    Forward.nmap.ssl.enable = FALSE;
    Forward.nmap.ssl.context = NULL;
    Forward.nmap.ssl.config.options = 0;

    Forward.handle.directory = NULL;
    Forward.handle.logging = NULL;

    strcpy(Forward.nmap.address, "127.0.0.1");

    XplSafeWrite(Forward.server.active, 0);

    XplSafeWrite(Forward.nmap.worker.idle, 0);
    XplSafeWrite(Forward.nmap.worker.active, 0);
    XplSafeWrite(Forward.nmap.worker.maximum, 100000);

    if (MemoryManagerOpen(MSGSRV_AGENT_AUTOREPLY) == TRUE) {
        Forward.nmap.pool = MemPrivatePoolAlloc("Forward Connections", sizeof(ForwardClient), 0, 3072, TRUE, FALSE, ForwardClientAllocCB, NULL, NULL);
        if (Forward.nmap.pool != NULL) {
            XplOpenLocalSemaphore(Forward.sem.main, 0);
            XplOpenLocalSemaphore(Forward.sem.shutdown, 1);
            XplOpenLocalSemaphore(Forward.nmap.semaphore, 1);
            XplOpenLocalSemaphore(Forward.nmap.worker.todo, 1);
        } else {
            MemoryManagerClose(MSGSRV_AGENT_AUTOREPLY);

            XplConsolePrintf("hulaforward: Unable to create connection pool; shutting down.\r\n");
            return(-1);
        }
    } else {
        XplConsolePrintf("hulaforward: Unable to initialize memory manager; shutting down.\r\n");
        return(-1);
    }

    ConnStartup(CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    Forward.handle.directory = (MDBHandle)MsgInit();
    if (Forward.handle.directory == NULL) {
        XplBell();
        XplConsolePrintf("hulaforward: Invalid directory credentials; exiting!\r\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_AUTOREPLY);

        return(-1);
    }

    NMAPInitialize(Forward.handle.directory);

    SetCurrentNameSpace(NWOS2_NAME_SPACE);
    SetTargetNameSpace(NWOS2_NAME_SPACE);

    Forward.handle.logging = LoggerOpen("hulaforward");
    if (!Forward.handle.logging) {
        XplConsolePrintf("hulaforward: Unable to initialize logging; disabled.\r\n");
    }

    ReadConfiguration();

    if (QueueSocketInit() < 0) {
        XplConsolePrintf("hulaforward: Exiting.\r\n");

        MemoryManagerClose(MSGSRV_AGENT_AUTOREPLY);

        return -1;
    }

    if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
        XplConsolePrintf("hulaforward: Could not drop to unprivileged user '%s', exiting.\r\n", MsgGetUnprivilegedUser());

        MemoryManagerClose(MSGSRV_AGENT_AUTOREPLY);

        return 1;
    }

    Forward.nmap.ssl.enable = FALSE;
    Forward.nmap.ssl.config.method = SSLv23_client_method;
    Forward.nmap.ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
    Forward.nmap.ssl.config.mode = SSL_MODE_AUTO_RETRY;
    Forward.nmap.ssl.config.cipherList = NULL;
    Forward.nmap.ssl.config.certificate.type = SSL_FILETYPE_PEM;
    Forward.nmap.ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
    Forward.nmap.ssl.config.key.type = SSL_FILETYPE_PEM;
    Forward.nmap.ssl.config.key.file = MsgGetTLSKeyPath(NULL);

    Forward.nmap.ssl.context = ConnSSLContextAlloc(&(Forward.nmap.ssl.config));
    if (Forward.nmap.ssl.context) {
        Forward.nmap.ssl.enable = TRUE;
    }

    NMAPSetEncryption(Forward.nmap.ssl.context);

    if ((ManagementInit(MSGSRV_AGENT_AUTOREPLY, Forward.handle.directory)) 
            && (ManagementSetVariables(GetForwardManagementVariables(), GetForwardManagementVariablesCount())) 
            && (ManagementSetCommands(GetForwardManagementCommands(), GetForwardManagementCommandsCount()))) {
        XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
    }


    if (ccode) {
        XplConsolePrintf("hulaforward: Unable to startup the management interface.\r\n");
    }

    Forward.state = FORWARD_STATE_RUNNING;

    XplStartMainThread(PRODUCT_SHORT_NAME, &id, ForwardServer, 8192, NULL, ccode);
    
    XplUnloadApp(XplGetThreadID());
    return(0);
}
