/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <logger.h>

#include "nmapdp.h"

void 
FreeClientData(NMAPClient *client)
{
    int i;
    unsigned int used;
    struct stat sb;
    sqlite3_stmt **stmt;
    NmapSqlCalComp *comp;

    if (!(client->states & NMAP_CLIENT_EXITING)) {
        client->states |= NMAP_CLIENT_EXITING;

        if (client->states & NMAP_CLIENT_QUEUE) {
            FreeQueueClientData(client);
            if (!(client->states & NMAP_CLIENT_USER)) {
                return;
            }
        }

        /* Check if new mail arrived or if the status of any message changed; 
           prevents loss of new messages since we call StoreMaildrop below */
        if (client->states & NMAP_CLIENT_MBOX) {
            if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
                sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
                if ((stat(client->path, &sb) == 0) && (client->mailbox.size != (unsigned long)sb.st_size)) {
                    ParseMaildrop(client);                                                                 
                }

                if (client->mailbox.fh) {
                    fclose(client->mailbox.fh);
                    client->mailbox.fh = NULL;
                }

                StoreMaildrop(client);
            } else {
                StoreSharedMaildrop(client);

                ConnClose(client->share.mailbox.conn, 0);
                ConnFree(client->share.mailbox.conn);
                client->share.mailbox.conn = NULL;
            }
        }


        if (client->mailbox.lock) {
            ReadNLockRelease(client->mailbox.lock);
            client->mailbox.lock = NULL;
        }

        if (client->mailbox.message.info) {
            MemFree(client->mailbox.message.info);
            client->mailbox.message.info = NULL;
        }

#if !defined(NMAP_SQL_CALENDARS)
        if (client->calendar.name[0] && !(client->share.flags & NMAP_SHARE_CALENDAR)) {
            if (client->calendar.fh) {
                fclose(client->calendar.fh);
                client->calendar.fh = NULL;
            }

            StoreCalendar(client);
        } else if (client->share.flags & NMAP_SHARE_CALENDAR) {
            StoreSharedCalendar(client);

            ConnClose(client->share.calendar.conn, 0);
            ConnFree(client->share.calendar.conn);
            client->share.calendar.conn = NULL;
        }

        if (client->calendar.lock) {
            ReadNLockRelease(client->calendar.lock);
            client->calendar.lock = NULL;
        }

        if (client->calendar.entries.info) {
            MemFree(client->calendar.entries.info);
            client->calendar.entries.info = NULL;
        }
#else /* !defined(NMAP_SQL_CALENDARS) */
        if (client->cal.objects) {
            used = client->cal.objects->used;
            comp = &(client->cal.objects->list[0]);

            while (used) {
                if (comp->organizer) {
                    MemFree(comp->organizer);
                }

                if (comp->summary) {
                    MemFree(comp->summary);
                }

                used--;
                comp++;
            }

            MemFree(client->cal.objects);
            client->cal.objects = NULL;
        }

        stmt = &(client->cal.sql.begin);
        for (i = 0; i < (sizeof(NMAPClientSQLStatements) / sizeof(sqlite3_stmt *)); i++, stmt++) {
            if (*stmt) {
                sqlite3_finalize(*stmt);
                *stmt = NULL;
            }
        }

        if (client->states & NMAP_CLIENT_CAL) {
            client->states &= ~(NMAP_CLIENT_CAL | NMAP_CLIENT_CALSEL);
            client->cal.id = 0;

            if (client->cal.handle) {
                sqlite3_close(client->cal.handle);
                client->cal.handle = NULL;
            }
        }
#endif /* !defined(NMAP_SQL_CALENDARS) */

        if (client->conn) {
            ConnClose(client->conn, 0);
            ConnFree(client->conn);
            client->conn = NULL;
        }
    }

    return;
}

int 
NmapCommandAuth(void *param)
{
    int i;
    unsigned char *ptr;
    unsigned char digest[16];
    unsigned char credential[33];
    MD5_CTX mdContext;
    NMAPClient *client = (NMAPClient *)param;

    if (!client->states) {
        if (client->buffer[4] == ' ') {
            ptr = client->buffer + 5;
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3240BADSTATE, sizeof(MSG3240BADSTATE) - 1));
    }

    MD5_Init(&mdContext);
    MD5_Update(&mdContext, client->user, strlen(client->user));
    MD5_Update(&mdContext, NMAP.server.hash, NMAP_HASH_SIZE);
    MD5_Final(digest, &mdContext);

    for (i = 0; i < 16; i++) {
        sprintf(credential + (i * 2), "%02x", digest[i]);
    }

    if (QuickCmp(credential, ptr)) {
        client->user[0] = '\0';
        client->states |= NMAP_CLIENT_AUTHORIZED;
        return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
    }

    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_WRONG_SYSTEM_AUTH, LOG_NOTICE, 0, NULL, NULL, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);

    ConnWrite(client->conn, MSG3242BADAUTH, sizeof(MSG3242BADAUTH) - 1);

    XplDelay(2000);

    FreeClientData(client);

    return(-1);
}

__inline static void 
ConfigureUser(NMAPClient *client)
{
    struct stat sb;
    FILE *handle;
    MDBValueStruct *vs;


    client->states |= NMAP_CLIENT_USER;
    client->store = MsgFindUserStore(client->user, NMAP.path.mail);

#if defined(NETWARE) || defined(LIBC)
    sprintf(client->line, "NMAP: %s", client->user);
    XplRenameThread(XplGetThreadID(), client->line);
#endif

    sprintf(client->path, "%s/%s/INBOX.box", client->store, client->user);
    if (stat(client->path, &sb) == 0) {
        sprintf(client->path, "%s/%s/MAIN.cal", client->store, client->user);
    } else {
        sprintf(client->path, "%s/%s", client->store, client->user);
        XplMakeDir(client->path);

        sprintf(client->path, "%s/%s/INBOX.box", client->store, client->user);
        handle = fopen(client->path, "wb");
        if (handle) {
            fclose(handle);

            sprintf(client->path, "%s/%s/INBOX.idx", client->store, client->user);
            handle = fopen(client->path, "wb");
            fprintf(handle, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);
            fclose(handle);

            sprintf(client->line, "%lu", NMAP.universalCounter++);
            vs = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
            MDBAddValue(client->line, vs);
            MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs);
            MDBDestroyValueStruct(vs);
        }

        sprintf(client->path, "%s/%s/MAIN.cal", client->store, client->user);
    }

    if (stat(client->path, &sb) != 0) {
        handle = fopen(client->path, "wb");
        if (handle != NULL) {
            fclose(handle);

            sprintf(client->path, "%s/%s/MAIN.idc",client->store, client->user);
            handle = fopen(client->path, "wb");
            fprintf(handle, "%s\r\n0000000000\r\n0000000000\r\n0000000000\r\n%010lu\r\n%010lu\r\n", NMAP_CAL_IDX_VERSION, NMAP.universalCounter++, NMAP.universalCounter++);
            fclose(handle);

            sprintf(client->line, "%lu", NMAP.universalCounter);
            vs = MDBCreateValueStruct(NMAP.handle.directory, NMAP.server.dn);
            MDBAddValue(client->line, vs);
            MDBWrite(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs);
            MDBDestroyValueStruct(vs);
        }
    }

    return;
}

int 
NmapCommandPass(void *param)
{
    unsigned char *ptr;
    unsigned char *pass;
    BOOL result;
    struct in_addr nmap;
    MDBValueStruct *vs;
    NMAPClient *client = (NMAPClient *)param;

    if (!client->states) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3240BADSTATE, sizeof(MSG3240BADSTATE) - 1));
    }

    if (*ptr++ == ' ') {
        if (       (toupper(ptr[0]) == 'U') 
                && (toupper(ptr[1]) == 'S') 
                && (toupper(ptr[2]) == 'E') 
                && (toupper(ptr[3]) == 'R') 
                && (ptr[4] == ' ') 
                && (ptr[5] != '\0') 
                && (!isspace(ptr[5])) 
                && ((pass = strchr(ptr + 5, ' ')) != NULL)) {
            ptr += 5;
            *pass++ = '\0';

            vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
            if (vs) {
                result = MsgFindObject(ptr, client->dn, NULL, &nmap, vs);
            } else {
                return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
            }

            if (result) {
                result = MDBVerifyPassword(client->dn, pass, vs);
            }

            if (result) {
                strcpy(client->user, vs->Value[0]);
            }

            MDBDestroyValueStruct(vs);

            if (result && (nmap.s_addr == MsgGetHostIPAddress())) {
                GET_STRING_HASH(client->user, client->userHash);

                client->flags |= NMAP_KEEPUSER;

                ConfigureUser(client);

                return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
            }

            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_WRONG_USER_AUTH, LOG_NOTICE, 0, ptr, NULL, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        } else if (    (toupper(ptr[0]) == 'S') 
                    && (toupper(ptr[1]) == 'Y') 
                    && (toupper(ptr[2]) == 'S') 
                    && (toupper(ptr[3]) == 'T') 
                    && (toupper(ptr[4]) == 'E') 
                    && (toupper(ptr[5]) == 'M') 
                    && (ptr[6] == ' ')) {
            ptr += 7;

#if defined(DEBUG)
            if (strcmp(NMAP.server.hash, ptr) == 0) {
                client->states |= NMAP_CLIENT_AUTHORIZED;

                return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
            }
#endif

            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_AUTH, LOGGER_EVENT_WRONG_SYSTEM_AUTH, LOG_NOTICE, 0, NULL, NULL, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
        }
    }

    XplDelay(2000);

    ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1);

    FreeClientData(client);
    return(-1);
}

int 
NmapCommandUlist(void *param)
{
    int ccode;
    unsigned long used;
    const unsigned char *name;
    const unsigned char *ptr;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;
    MDBEnumStruct *es;

    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_AUTHORIZED) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    if (vs) {
        es = MDBCreateEnumStruct(vs);
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    if (es) {
        MDBReadDN(NMAP.server.dn, MSGSRV_A_CONTEXT, vs);
    } else {
        MDBDestroyValueStruct(vs);

        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    ccode = 0;
    for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
        while ((name = MDBEnumerateObjectsEx(vs->Value[used], C_USER, NULL, FALSE, es, vs)) != NULL) {
            if ((ptr = strrchr(name, '\\')) != NULL) {
                ccode = ConnWriteF(client->conn, "2001-%s\r\n", ptr + 1);
            } else {
                ccode = ConnWriteF(client->conn, "2001-%s\r\n", name);
            }
        }
    }

    MDBDestroyEnumStruct(es, vs);
    MDBDestroyValueStruct(vs);

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandUser(void *param)
{
    unsigned long flags = 0;
    unsigned char *ptr;
    unsigned char *ptr2;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_AUTHORIZED) {
	ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    if ((*ptr++ == ' ') && (!isspace(*ptr))) {
        ptr2 = strchr(ptr, ' ');
        if (ptr2) {
            *ptr2++ = '\0';
            flags = atol(ptr2);
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    strcpy(client->user, ptr);

    GET_STRING_HASH(ptr, client->userHash);

#if 0
    /* Check if user exists ...

    if (User doesn't exist) {
        ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
        break;
    } */
#endif

    ConfigureUser(client);

    return(ConnWrite(client->conn, MSG1000USER, sizeof(MSG1000USER) - 1));
}
