/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <logger.h>

#include "nmapdp.h"

#if defined(WIN32)
#include <rpcdce.h>

void 
GuidReset(void)
{
    int i;
    unsigned long length;
    unsigned char salt[32];
    unsigned char name[MAX_COMPUTERNAME_LENGTH + 1];
    unsigned char digest[SHA_DIGEST_LENGTH];
    unsigned char *uid = NULL;
    unsigned char *ptr;
    FILETIME date;
    SHA_CTX context;
    UUID uuid;

    SHA1_Init(&context);

    GetSystemTimeAsFileTime(&date);
    sprintf(salt, "%010lu%010lu", date.dwHighDateTime, date.dwLowDateTime);
    SHA1_Update(&context, salt, strlen(salt));

    length = sizeof(name) - 1;
    if (GetComputerName(name, &length)) {
        SHA1_Update(&context, name, strlen(name));
    } else {
        RAND_bytes(name, sizeof(name) - 1);
        SHA1_Update(&context, name, sizeof(name) - 1);
    }

    if ((UuidCreate(&uuid) == RPC_S_OK) && (UuidToString(&uuid, &uid) == RPC_S_OK)) {
        SHA1_Update(&context, uid, strlen(uid));
    } else {
        RAND_bytes(name, sizeof(name) - 1);
        SHA1_Update(&context, name, sizeof(name) - 1);
    }

    if (uid) {
        RpcStringFree(&uid);
    }

    SHA1_Final(digest, &context);

    for (i = 0, ptr = NMAP.guid.next; i < SHA_DIGEST_LENGTH; i++, ptr += 2) {
        sprintf(ptr, "%02X", digest[i]);
    }

    memset(NMAP.guid.next + NMAP_GUID_PREFIX_LENGTH, '0', NMAP_GUID_SUFFIX_LENGTH);

    return;
}
#elif defined(NETWARE) || defined(LIBC)
#error GuidReset not implemented on this platform.

void 
GuidReset(void)
{
    return;
}
#elif defined(LINUX)

#if HAVE_SYS_SYSINFO_H
#include <sys/sysinfo.h>
#endif

void 
GuidReset(void)
{
    int i;
    char name[256 + 1];
    unsigned char salt[32];
    unsigned char digest[SHA_DIGEST_LENGTH];
    unsigned char *ptr;
#if HAVE_SYS_SYSINFO_H
    struct sysinfo si;
#endif
    struct timeval tv;
    SHA_CTX context;

    SHA1_Init(&context);

    gettimeofday(&tv, NULL);
    sprintf(salt, "%010lu%010lu", tv.tv_sec, tv.tv_usec);
    SHA1_Update(&context, salt, strlen(salt));

    if (!gethostname(name, sizeof(name) - 1)) {
        SHA1_Update(&context, name, strlen(name));
    } else {
        RAND_bytes(name, sizeof(name) - 1);
        SHA1_Update(&context, name, sizeof(name) - 1);
    }

#if HAVE_SYS_SYSINFO_H
    if (!sysinfo(&si)) {
        SHA1_Update(&context, &si, sizeof(si));
    } else {
        RAND_bytes(name, sizeof(name) - 1);
        SHA1_Update(&context, name, sizeof(name) - 1);
    }
#else 
    /* FIXME: should find a sysinfo equiv. for this */
    RAND_bytes(name, sizeof(name) - 1);
    SHA1_Update(&context, name, sizeof(name) - 1);
#endif

    SHA1_Final(digest, &context);

    for (i = 0, ptr = NMAP.guid.next; i < SHA_DIGEST_LENGTH; i++, ptr += 2) {
        sprintf(ptr, "%02X", digest[i]);
    }

    memset(NMAP.guid.next + NMAP_GUID_PREFIX_LENGTH, '0', NMAP_GUID_SUFFIX_LENGTH);

    return;
}
#else
#error GuidReset not implemented on this platform.
#endif

unsigned char *
GuidAlloc(void)
{
    int i;
    unsigned char c;
    unsigned char *ptr;
    unsigned char *id = (unsigned char *)MemMalloc(NMAP_GUID_LENGTH + 1);

    if (id) {
        XplWaitOnLocalSemaphore(NMAP.guid.semaphore);

        id[NMAP_GUID_LENGTH] = '\0';

        memcpy(id, NMAP.guid.next, sizeof(NMAP.guid.next));

        ptr = NMAP.guid.next + NMAP_GUID_PREFIX_LENGTH + NMAP_GUID_SUFFIX_LENGTH - 1;
        for (i = 0; i < NMAP_GUID_SUFFIX_LENGTH; i++) {
            c = *ptr + 1;
            if (c < ':') {
                *ptr = c; /* '0' through '9' */
                break;
            } else if (c != ':') {
                if (c < 'G') {
                    *ptr = c; /* 'B' through 'F' */
                    break;
                }

                *ptr = '0';
                ptr--;
                continue;
            }

            *ptr = 'A';
            break;
        }

        if (i < NMAP_GUID_SUFFIX_LENGTH) {
            XplSignalLocalSemaphore(NMAP.guid.semaphore);

            return(id);
        }

        GuidReset();
        memcpy(id, NMAP.guid.next, sizeof(NMAP.guid.next));

        XplSignalLocalSemaphore(NMAP.guid.semaphore);
    }

    return(id);
}

int 
NmapCommandGuid(void *param)
{
    int ccode;
    unsigned char *ptr;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & (NMAP_CLIENT_QUEUE | NMAP_CLIENT_AUTHORIZED)) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3012BADQSTATE, sizeof(MSG3012BADQSTATE) - 1));
    }

    /* GUID */
    if (*ptr == '\0') {
        XplWaitOnLocalSemaphore(NMAP.queue.semaphore);
        ptr = GuidAlloc();
        XplSignalLocalSemaphore(NMAP.queue.semaphore);

        if (ptr) {
            ccode = ConnWriteF(client->conn, "1000-%s\r\n", ptr);
            GuidFree(ptr);
        } else {
            ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1);
    }

    return(ccode);
}
