/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <msgapi.h>
#include "nmapdp.h"

#define TIMESLICE 2000

#define NUL 0
#define QUOTE 34
#define NIBBLE 4
#define ELEMENTSIZE (1)
#define BITCOUNT (8 * ELEMENTSIZE)
#define BVSET( bitvector, message) (bitvector[message / BITCOUNT] |= (0x80/*000000*/ /*(1 << BITCOUNT)*/ >> (message % BITCOUNT)))
#define LOOP_YIELD_CONTROL 100
#define NX_SEARCH_DATE  0x100
#define NX_SEARCH_FIELD 0x101
#define NX_SEARCH_FLAG  0x102
#define NX_SEARCH_SIZE  0x103

#define IsWhitespace(c) (WhiteSpaceTable[(c)])

BOOL WhiteSpaceTable[256] = {
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* 00 - 0f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* 10 - 1f */
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,            /* 20 - 2f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,            /* 30 - 3f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,            /* 40 - 4f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,            /* 50 - 5f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,            /* 60 - 6f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1,            /* 70 - 7f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* 80 - 8f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* 90 - 9f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* a0 - af */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* b0 - bf */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* c0 - cf */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* d0 - df */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,            /* e0 - ef */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1            /* f0 - ff */
};

/*
    -----------------------------------------------------------------------------
                            Internal Types and Structures
    -----------------------------------------------------------------------------
*/
typedef enum /* all symbols */
{
    HDRsy = 1,
    FLAGsy, ANSWEREDsy, DELETEDsy, DRAFTsy, FLAGGEDsy, SEENsy, RECENTsy, 
    BODYsy,
    TEXTsy,
    SIZEsy, LARGERsy, SMALLERsy, EQUALsy,
    RECEIVEDsy, SENTsy, BEFOREsy, ONsy, SINCEsy,
    endofbufsy, stringsy, errorsy, atomsy, HEXsy, BINsy, COUNTsy,
} SYMBOL;

/*typedef unsigned long BITVECTOR;    */
typedef unsigned char BITVECTOR;

#define IDENTSIZE 100
typedef struct
{
    SYMBOL sy; /* current symbol */
    unsigned char ch; /* current char */
    int offsetIn; /* current offset of bufferIn (Command) */
    int identStart; /* identifier offset start */
    int identLen; /* identifier length */
    unsigned char ident[ IDENTSIZE+1]; /* current identifier */
    unsigned char *Command; /* short cut to Client->Command */
    void *Client; /* JIC baggage */
    MessageInfoStruct *MsgInfo;
    unsigned long MsgUsed;
    SYMBOL ReturnAs;
} NMAPSEARCH;

typedef unsigned int UINT;
typedef long LPARAM;
typedef BOOL (*DYNASEARCH)( UINT msg, LPARAM lParam);

int NmapSearch( NMAPClient* Client);
static void NmapSearchDate( NMAPSEARCH *Search, BITVECTOR *bv, time_t time, SYMBOL sentReceived, SYMBOL beforeOnSince);
static void NmapSearchField( NMAPSEARCH *Search, BITVECTOR *bv, char *header, char *string, SYMBOL headerBodyText);
static void NmapSearchFlag( NMAPSEARCH *Search, BITVECTOR *bv, unsigned long flag);
static void NmapSearchInit( void);
static void NmapSearchSize( NMAPSEARCH *Search, BITVECTOR *bv, long size, SYMBOL smallerEqualLarger);

static void NextCh( NMAPSEARCH *Search) ;
static void InSymbol( NMAPSEARCH *Search);
static void search_key( NMAPSEARCH *Search, BITVECTOR *bv);
int PDBSearch(char *Doc, char *SearchString);

static char *g_key[] = /* all keywords */
{    "", 
    "FLAG", "ANSWERED", "DELETED", "DRAFT", "FLAGGED", "SEEN", "RECENT",
    "BODY",
    "TEXT",
    "HDR",
    "SIZE", "LARGER", "SMALLER", "EQUAL",
    "RECEIVED","SENT", "BEFORE", "ON", "SINCE",
    "HEX", "BIN",
    "COUNT",
};

static SYMBOL g_keysy[] = /* symbols corresponding to keywords */
{    -1,
    FLAGsy, ANSWEREDsy, DELETEDsy, DRAFTsy, FLAGGEDsy, SEENsy, RECENTsy,
    BODYsy,
    TEXTsy,
    HDRsy,
    SIZEsy, LARGERsy, SMALLERsy, EQUALsy,
    RECEIVEDsy, SENTsy, BEFOREsy, ONsy, SINCEsy,
    HEXsy, BINsy,
    COUNTsy,
};

static int g_keylen[sizeof( g_key) / sizeof(g_key[0])];

static int g_nkw = sizeof( g_key) / sizeof(g_key[0]);

BOOL       g_NmapSearchInit = FALSE;

/*----------------------------------------------------------------------------
;NmapSearch
Title:        
Globals:    
Return:        
Descrip:    search main and 
    search          ::= "SEARCH" ["BIN" / "HEX"]  1#search_key 
    are folded together

    NmapSearch searches the "selected" mailbox; the caller must guarantee
    the selected state.
------------------------------------------------------------------------------*/
int 
NmapSearch(NMAPClient *client)
{
    int ccode;
    size_t i, j, memorySize, bvAnswerSize;
    char *bvAnswer = 0;
    BITVECTOR *bv = 0;
    unsigned long bvSize;
    NMAPSEARCH Search;
    char *temp1;

    if( !g_NmapSearchInit) {
        NmapSearchInit();
    }

    memset( &Search, 0, sizeof( NMAPSEARCH)); 
    
    Search.Command  = client->buffer;
    Search.MsgUsed  = client->mailbox.message.used;
    Search.MsgInfo  = client->mailbox.message.info;
    Search.Client   = client;

    Search.ch = Search.Command[ 0]; 

    /* consume SEARCH token */
    InSymbol( &Search);

    /* get the next token */
    InSymbol( &Search); 

    switch( Search.sy) {
        case HEXsy: /* remember the way to return data */
        case BINsy:
            Search.ReturnAs = Search.sy;
            InSymbol( &Search);
        break;

        case COUNTsy: /* return msg count */
        {
            char szCount[20];
            sprintf( szCount, "1000 COUNT %lu\r\n", client->mailbox.message.count);
            ccode = ConnWrite(client->conn, szCount, strlen( szCount));
            goto done;
        }
        break;

        case endofbufsy:
            goto done;
        break;
    default:
      break;
    }

    bvSize = client->mailbox.message.used / BITCOUNT;
    if( client->mailbox.message.used % BITCOUNT)
    {
        bvSize++;
    }
    memorySize = bvSize * ELEMENTSIZE;

    bv = (BITVECTOR*)MemMalloc( memorySize);
    if( !bv)
    {
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
        goto done;
    }
    memset( bv, 0, memorySize);

    /* add 2 for \r\n plus 18 for other use */
    bvAnswerSize = (memorySize * 2) + 20;

    bvAnswer = (char*)MemMalloc( bvAnswerSize + 1);
    if( !bvAnswer) {
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
        goto done;
    }

    memset( bvAnswer, 0, bvAnswerSize);

    search_key( &Search, bv);

    temp1 = (char*)bv;

/*    sprintf( ControlInfo, "2002 %d\r\n", bvAnswerSize);        */
/*    ccode = ConnWrite(client->conn, ControlInfo, strlen( ControlInfo));    */


    switch( Search.ReturnAs) {
        case HEXsy: /* hex bv is returned to caller */
            for(i=0, j=0; i<memorySize; i++) {
                switch( (0xf0 & temp1[i]) >> NIBBLE) { /* high nibble */
                    case 0x0: bvAnswer[j] = '0'; break;
                    case 0x1: bvAnswer[j] = '1'; break;
                    case 0x2: bvAnswer[j] = '2'; break;
                    case 0x3: bvAnswer[j] = '3'; break;
                    case 0x4: bvAnswer[j] = '4'; break;
                    case 0x5: bvAnswer[j] = '5'; break;
                    case 0x6: bvAnswer[j] = '6'; break;
                    case 0x7: bvAnswer[j] = '7'; break;
                    case 0x8: bvAnswer[j] = '8'; break;
                    case 0x9: bvAnswer[j] = '9'; break;
                    case 0xA: bvAnswer[j] = 'A'; break;
                    case 0xB: bvAnswer[j] = 'B'; break;
                    case 0xC: bvAnswer[j] = 'C'; break;
                    case 0xD: bvAnswer[j] = 'D'; break;
                    case 0xE: bvAnswer[j] = 'E'; break;
                    case 0xF: bvAnswer[j] = 'F'; break;
                }

                j++;

                switch( 0x0f & temp1[i]) /* low nibble */
                {
                    case 0x0: bvAnswer[j] = '0'; break;
                    case 0x1: bvAnswer[j] = '1'; break;
                    case 0x2: bvAnswer[j] = '2'; break;
                    case 0x3: bvAnswer[j] = '3'; break;
                    case 0x4: bvAnswer[j] = '4'; break;
                    case 0x5: bvAnswer[j] = '5'; break;
                    case 0x6: bvAnswer[j] = '6'; break;
                    case 0x7: bvAnswer[j] = '7'; break;
                    case 0x8: bvAnswer[j] = '8'; break;
                    case 0x9: bvAnswer[j] = '9'; break;
                    case 0xA: bvAnswer[j] = 'A'; break;
                    case 0xB: bvAnswer[j] = 'B'; break;
                    case 0xC: bvAnswer[j] = 'C'; break;
                    case 0xD: bvAnswer[j] = 'D'; break;
                    case 0xE: bvAnswer[j] = 'E'; break;
                    case 0xF: bvAnswer[j] = 'F'; break;
                }

                j++;
            }
            ccode = ConnWriteF(client->conn, "2002 %d\r\n%s\r\n", j + strlen("\r\n"), bvAnswer);
        break;

        case BINsy: /* binary bv is returned to caller */
            break;

        default: {    /* buffer of decimal numbers is returned to caller */
            for (i=0; i<memorySize; i++) {
                for (j=0; j<8; j++) {
                    if (temp1[i] & (0x80>>j)) {
                        bvAnswerSize=sprintf(bvAnswer, "2002-%d\r\n", (i*8+j+1));
                        ccode = ConnWrite(client->conn, bvAnswer, bvAnswerSize);
                    }
                }
            }
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            break;
        }
    }

    /* end marker */

/*    sprintf( ControlInfo, "1000 OK\r\n"); */
/*    ccode = ConnWrite(client->conn, ControlInfo, strlen( ControlInfo)); */


done:
    MemFree( bvAnswer);
    MemFree( bv);

    return(ccode);
}

/*----------------------------------------------------------------------------
;search_key
Title:        
Globals:    
Return:        
Descrip:    
    search_key      ::=  see below for production rules

------------------------------------------------------------------------------*/
static void 
search_key( NMAPSEARCH *Search, BITVECTOR *bv)
{
    unsigned long flag;
    long date, size;
    SYMBOL sentReceived, beforeOnSince, smallerEqualLarger, headerBodyText;
    char fieldString[ IDENTSIZE+1];
    char searchString[ IDENTSIZE+1];

    switch( Search->sy)
    {
        case HDRsy:
        case BODYsy:
        case TEXTsy:
            switch( Search->sy)
            {
                case HDRsy:  /* "HDR"  headerString searchString */
                    headerBodyText = Search->sy;
                    InSymbol( Search);
                    strncpy( fieldString, Search->ident, sizeof( fieldString));
                    InSymbol( Search);
                    strncpy( searchString, Search->ident, sizeof( searchString));                    
                break;

                case BODYsy: /* "BODY" searchString */
                    headerBodyText = Search->sy;
                    strncpy( fieldString, Search->ident, sizeof( fieldString));
                    InSymbol( Search);
                    strncpy( searchString, Search->ident, sizeof( searchString));
                break;

                case TEXTsy: /* "TEXT" searchString */
                    headerBodyText = Search->sy;
                    strncpy( fieldString, Search->ident, sizeof( fieldString));
                    InSymbol( Search);
                    strncpy( searchString, Search->ident, sizeof( searchString));
                break;
            default: /* should not happen */
              break;
            }

            NmapSearchField( Search, bv, fieldString, searchString, headerBodyText);
        break;


        case FLAGsy: /* "FLAG" (  "ANSWERED" /   "DELETED" /   "DRAFT" /   "FLAGGED" /   "SEEN" /   "RECENT" ) */
            flag = 0;
            InSymbol( Search);
            switch( Search->sy)
            {
                case ANSWEREDsy:   flag = MSG_STATE_ANSWERED; break;
                case DELETEDsy:    flag = MSG_STATE_DELETED;  break;
                case DRAFTsy:      flag = MSG_STATE_DRAFT;    break;
                case FLAGGEDsy:    flag = MSG_STATE_PRIOHIGH; break;
                case SEENsy:       flag = MSG_STATE_READ;     break;
                case RECENTsy:     flag = MSG_STATE_RECENT;   break;
            default: /* should not happen */
              break;
            }

            NmapSearchFlag( Search, bv, flag);
        break;

        case SIZEsy: /* "SIZE" ( "LARGER" / "SMALLER" / "EQUAL" ) sizeString */
            InSymbol( Search);
            smallerEqualLarger = Search->sy;
            InSymbol( Search);

            size = atol( Search->ident);

            NmapSearchSize( Search, bv, size, smallerEqualLarger);
        break;

        case RECEIVEDsy: /* "RECEIVED" ( "BEFORE" / "ON" / "SINCE") dateString */
        case SENTsy:     /* "SENT"     ( "BEFORE" / "ON" / "SINCE") dateString */
            sentReceived = Search->sy;
            InSymbol( Search);
            beforeOnSince = Search->sy;
            InSymbol( Search);
            date = atol( Search->ident);

            NmapSearchDate( Search, bv, date, sentReceived, beforeOnSince);
        break;
    default: /* should not happen */
      break;
    }
}

/*----------------------------------------------------------------------------
;NmapSearchFlag
Title:        
Globals:    
Return:        
Descrip:    uses native index to search on flag(s)

------------------------------------------------------------------------------*/
static void 
NmapSearchFlag( NMAPSEARCH *Search, BITVECTOR *bv, unsigned long flag)
{
    unsigned long msg, i, end, StartTime;;

    StartTime=XplGetHighResolutionTimer();

    for(i=0; i<Search->MsgUsed; i+=1000) {

        end=min(Search->MsgUsed, i+1000);

        for( msg = i; msg < end; msg++) {
            if( Search->MsgInfo[msg].State & flag) {
                BVSET( bv, msg);
            }
        }

        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
            XplThreadSwitchWithDelay();
            StartTime=XplGetHighResolutionTimer();
        }
    }
}

/*----------------------------------------------------------------------------
;NmapSearchSize
Title:        
Globals:    
Return:        
Descrip:    uses native index to search on size

------------------------------------------------------------------------------*/
static void 
NmapSearchSize( NMAPSEARCH *Search, BITVECTOR *bv, long size, SYMBOL smallerEqualLarger)
{
    unsigned long msg, i, end, StartTime;;

    StartTime=XplGetHighResolutionTimer();

    switch( smallerEqualLarger) {
        case SMALLERsy: {
            for(i=0; i<Search->MsgUsed; i+=1000) {
                
                end=min(Search->MsgUsed, i+1000);

                for (msg = i; msg < end; msg++) {
                    if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].SSize < size)) {
                        BVSET( bv, msg);
                    }
                }

                if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                    XplThreadSwitchWithDelay();
                    StartTime=XplGetHighResolutionTimer();
                }
            }
            break;
        }

        case EQUALsy: {
            for(i=0; i<Search->MsgUsed; i+=1000) {
                
                end=min(Search->MsgUsed, i+1000);

                for (msg = i; msg < end; msg++) {
                    if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && ( Search->MsgInfo[msg].SSize == size)) {
                        BVSET( bv, msg);
                    }
                }

                if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                    XplThreadSwitchWithDelay();
                    StartTime=XplGetHighResolutionTimer();
                }
            }
            break;
        }

        case LARGERsy: {
            for(i=0; i<Search->MsgUsed; i+=1000) {
                
                end=min(Search->MsgUsed, i+1000);

                for (msg = i; msg < end; msg++) {
                    if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].SSize > size)) {
                        BVSET( bv, msg);
                    }
                }
                if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                    XplThreadSwitchWithDelay();
                    StartTime=XplGetHighResolutionTimer();
                }
            }
            break;
        default: /* should not happen */
          break;

        }
    }
}

/*----------------------------------------------------------------------------
;NmapSearchDate
Title:        
Globals:    
Return:        
Descrip:    uses native index to search on date

------------------------------------------------------------------------------*/
static void 
NmapSearchDate( NMAPSEARCH *Search, BITVECTOR *bv, time_t time, SYMBOL sentReceived, SYMBOL beforeOnSince)
{
    unsigned long msg, i, end, StartTime;

    StartTime=XplGetHighResolutionTimer();

    switch( sentReceived) {
        case SENTsy: {
            switch(beforeOnSince) {
                case BEFOREsy: {
                    for(i=0; i<Search->MsgUsed; i+=1000) {

                        end=min(Search->MsgUsed, i+1000);

                        for( msg = i; msg < end; msg++) {
                            if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].DateSent < time)) {
                                BVSET( bv, msg);
                            }
                        }
                        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                            XplThreadSwitchWithDelay();
                            StartTime=XplGetHighResolutionTimer();
                        }
                    }
                    break;
                }

                case ONsy: {
                    for(i=0; i<Search->MsgUsed; i+=1000) {
                        end=min(Search->MsgUsed, i+1000);

                        for( msg = i; msg < end; msg++) {
                            if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].DateSent == time)) {
                                BVSET( bv, msg);
                            }
                        }
                        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                            XplThreadSwitchWithDelay();
                            StartTime=XplGetHighResolutionTimer();
                        }
                    }
                    break;
                }

                case SINCEsy: {
                    for(i=0; i<Search->MsgUsed; i+=1000) {
                        end=min(Search->MsgUsed, i+1000);

                        for( msg = i; msg < end; msg++) {
                            if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].DateSent > time)) {
                                BVSET( bv, msg);
                            }
                        }
                        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                            XplThreadSwitchWithDelay();
                            StartTime=XplGetHighResolutionTimer();
                        }
                    }
                    break;
                }
            default: /* should not happen */
              break;
            }
            break;
        }

        case RECEIVEDsy: {
            switch( beforeOnSince) {
                case BEFOREsy: {
                    for(i=0; i<Search->MsgUsed; i+=1000) {
                        end=min(Search->MsgUsed, i+1000);

                        for( msg = i; msg < end; msg++) {
                            if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].DateReceived < time)) {
                                BVSET( bv, msg);
                            }
                        }

                        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                            XplThreadSwitchWithDelay();
                            StartTime=XplGetHighResolutionTimer();
                        }

                    }
                    break;
                }

                case ONsy: {
                    for(i=0; i<Search->MsgUsed; i+=1000) {
                        end=min(Search->MsgUsed, i+1000);

                        for( msg = i; msg < end; msg++) {
                            if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].DateReceived == time)) {
                                BVSET( bv, msg);
                            }
                        }

                        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                            XplThreadSwitchWithDelay();
                            StartTime=XplGetHighResolutionTimer();
                        }
                    }
                    break;
                }

                case SINCEsy: {
                    for(i=0; i<Search->MsgUsed; i+=1000) {
                        end=min(Search->MsgUsed, i+1000);

                        for( msg = i; msg < end; msg++) {
                            if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED) && (Search->MsgInfo[msg].DateReceived > time)) {
                                BVSET( bv, msg);
                            }
                        }

                        if (XplGetHighResolutionTimer()>(StartTime+TIMESLICE)) {
                            XplThreadSwitchWithDelay();
                            StartTime=XplGetHighResolutionTimer();
                        }
                    }
                    break;
                }
            default: /* should not happen */
              break;
            }
            break;
        }
    default: /* should not happen */
      break;
    }
}

/*----------------------------------------------------------------------------
;NmapSearchField
Title:        
Globals:    
Return:        
Descrip:    uses native index to search on text field
    TODO use fread instead of fgets and use buffer with overlapping read

------------------------------------------------------------------------------*/
static void 
NmapSearchField( NMAPSEARCH *Search, BITVECTOR *bv, char *fieldString, char *searchString, SYMBOL headerBodyText)
{
    char Path[XPL_MAX_PATH+1];
    char Answer[CONN_BUFSIZE+1];
    char newSearchString[ IDENTSIZE+3]; /* add 2 for the quotes we add + 1 for null terminator */
    unsigned long msg;
    long EndPos, CurPos;
    char *pColon;
    NMAPClient *client = Search->Client;
    FILE *SCMS;

    /* ------------- */
    /* sanity check */
    if( client->user[0]=='\0' || client->mailbox.name[0]=='\0') {
        return;
    }
    
    /* ----------------- */
    /* open the mailbox */
    sprintf( Path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
    client->mailbox.fh = fopen( Path, "rb");
    if (!client->mailbox.fh) {
        return;
    }

    /* transform the search string to be acceptable to PDBSearch */
    sprintf( newSearchString, "\"%s\"", searchString);

    switch( headerBodyText)    {
        case HDRsy: {
            /* examine each message */
            for( msg = 0; msg < client->mailbox.message.used; msg++) {
                /* check message state */
                if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED)) {
                    if (client->mailbox.message.info[msg].UseSCMS) {
                        sprintf(Path, "%s/%x/%lx",NMAP.path.scms, (unsigned int)client->mailbox.message.info[msg].SCMSID % 16, client->mailbox.message.info[msg].SCMSID);
                        SCMS=client->mailbox.fh;
                        client->mailbox.fh=fopen(Path, "rb");
                        if (!client->mailbox.fh) {
                            client->mailbox.fh=SCMS;
                            continue;
                        }
                    }

                    /* go to the start of the header for this message */
                    fseek(client->mailbox.fh, client->mailbox.message.info[msg].AuthPos, SEEK_SET);

                    /* boundary check                 */
                    while (NMAP.state < NMAP_STOPPING) {                                                                        /* The while loop is left via break */
                        CurPos=ftell(client->mailbox.fh);
                        if ((CurPos==-1) || (CurPos>=client->mailbox.message.info[msg].BodyPos)) {
                            break;
                        }
        
                        if (fgets( Answer, sizeof( Answer) - 1, client->mailbox.fh)!=NULL) {            /* get a line */
                            if (!isspace(Answer[0]) && ((pColon = strchr( Answer, ':'))!=NULL)) {
                                *pColon = 0;
                                if (XplStrCaseCmp(Answer, fieldString)==0) {
                                    if (PDBSearch( pColon+1, newSearchString)) {
                                        BVSET( bv, msg);
                                        break;
                                    }
                                }
                            }
                        } else {
                            break;
                        }
                    }

                    if (client->mailbox.message.info[msg].UseSCMS) {
                        fclose(client->mailbox.fh);
                        client->mailbox.fh=SCMS;
                    }
                }
            }
            break;
        }

        case BODYsy: {
            /* examine each message */
            for( msg = 0; msg < client->mailbox.message.used; msg++) {
                if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED)) {
                    if (client->mailbox.message.info[msg].UseSCMS) {
                        sprintf(Path, "%s/%x/%lx",NMAP.path.scms, (unsigned int)client->mailbox.message.info[msg].SCMSID % 16, client->mailbox.message.info[msg].SCMSID);
                        SCMS=client->mailbox.fh;
                        client->mailbox.fh=fopen(Path, "rb");
                    }

                    if (client->mailbox.fh) {
                        fseek( client->mailbox.fh, client->mailbox.message.info[msg].BodyPos, SEEK_SET);

                        EndPos = client->mailbox.message.info[msg].AuthPos + client->mailbox.message.info[msg].SSize;
                                            
                        while (NMAP.state < NMAP_STOPPING) {                                                                    /* This loop is left via break */
                            CurPos=ftell(client->mailbox.fh);
                            if ((CurPos==-1) || (CurPos>=EndPos)) {
                                break;
                            }
                            if (fgets( Answer, sizeof( Answer), client->mailbox.fh)!=NULL) {        /* get a line; this yields */
                                if( PDBSearch( Answer, newSearchString)) {                                /* was there a hit */
                                    BVSET( bv, msg);
                                    break;
                                }
                            } else {
                                break;
                            }
                        }
                        if (client->mailbox.message.info[msg].UseSCMS) {
                            fclose(client->mailbox.fh);
                            client->mailbox.fh=SCMS;
                        }
                    } else {
                        if (client->mailbox.message.info[msg].UseSCMS) {
                            client->mailbox.fh=SCMS;
                        }
                    }
                }
            }

            break;
        }

        case TEXTsy: { /* i.e. header and body */
            /* examine each message */
            for( msg = 0; msg < client->mailbox.message.used; msg++) {
                if (!(Search->MsgInfo[msg].State & MSG_STATE_PURGED)) {
                    if (client->mailbox.message.info[msg].UseSCMS) {
                        sprintf(Path, "%s/%x/%lx",NMAP.path.scms, (unsigned int)client->mailbox.message.info[msg].SCMSID % 16, client->mailbox.message.info[msg].SCMSID);
                        SCMS=client->mailbox.fh;
                        client->mailbox.fh=fopen(Path, "rb");
                    }

                    if (client->mailbox.fh) {
                        fseek( client->mailbox.fh, client->mailbox.message.info[msg].AuthPos, SEEK_SET);

                        EndPos=client->mailbox.message.info[msg].AuthPos+client->mailbox.message.info[msg].SSize;
                                            
                        while (NMAP.state < NMAP_STOPPING) {                                                                        /* The loop is left via break */
                            CurPos=ftell(client->mailbox.fh);
                            if ((CurPos==-1) || (CurPos>=EndPos)) {
                                break;
                            }
                            if (fgets( Answer, sizeof( Answer), client->mailbox.fh)!=NULL) {            /* get a line; this yields */
                                if( PDBSearch( Answer, newSearchString)) {                                /* was there a hit */
                                    BVSET( bv, msg);
                                    break;
                                }
                            } else {
                                break;
                            }
                        }
                        if (client->mailbox.message.info[msg].UseSCMS) {
                            fclose(client->mailbox.fh);
                            client->mailbox.fh=SCMS;
                        }
                    } else {
                        if (client->mailbox.message.info[msg].UseSCMS) {
                            client->mailbox.fh=SCMS;
                        }
                    }
                }
            }

            break;
        }
    default: /* should not happen */
      break;
    }

    fclose(client->mailbox.fh);
    client->mailbox.fh=NULL;
    return;
}

/*----------------------------------------------------------------------------
;NmapSearchInit
Title:        
Globals:    
Return:        
Descrip:    One time initialization of search data

------------------------------------------------------------------------------*/
static void 
NmapSearchInit( void)
{
    int i, len;

    g_NmapSearchInit = TRUE;

    len = sizeof( g_key) / sizeof( g_key[0]);
    for( i = 0; i < len; i++) {
        g_keylen[i] = strlen( g_key[i]);
    }
    
}

/*----------------------------------------------------------------------------
;NextCh
Title:        
Globals:    
Return:        
Descrip:    Get the next character in the buffer

------------------------------------------------------------------------------*/
static void 
NextCh( NMAPSEARCH *Search) 
{ 
    Search->offsetIn++; 
    Search->ch = Search->Command[Search->offsetIn];
}

/*----------------------------------------------------------------------------
;InSymbol
Title:        
Globals:    
Return:        
Descrip:    Tokenize a stream of characters
    dateString      ::= <"> *number <">
    headerString    ::= string
    searchString    ::= string
    string          ::= <"> *QUOTED_CHAR <">
    QUOTED_CHAR     ::= <any TEXT_CHAR except quoted_specials> "\" quoted_specials
    TEXT_CHAR       ::= <any CHAR except CR and LF>
    CR              ::= <ASCII CR, carriage return, 0x0D>
    LF              ::= <ASCII LF, line feed, 0x0A>
    quoted_specials ::= <"> / "\"
    number          ::= 1*digit // signed 32-bit integer 
    digit           ::= "0" / "1" / "2" / "3" / "4" / "5" / "6" / "7" / "8" / "9"

------------------------------------------------------------------------------*/

static void 
InSymbol( NMAPSEARCH *Search)
{
    int i;
    unsigned char lch;

    while (isspace(Search->ch)) {
        NextCh( Search);
    }
    
    lch = Search->ch;
    NextCh( Search);

    switch (lch) {
        case NUL: /* null */
            Search->sy = endofbufsy;
        break;

        case QUOTE: /* " start of string */
            for( i = 0;; i++ ) {
                /* the other quote */
                if( '"' == Search->ch) {
                     NextCh( Search);
                    break;
                }

                /* not allowed */
                if( (NUL == Search->ch) ) {
                    /*||  (CARRIAGE_RETURN == Search->ch)  */
                    /*||  (LINE_FEED == Search->ch)) */
                    /* error */
                    Search->sy = errorsy;
                    return;
                }

                /* quoted special */
                if( '\\' == Search->ch) {
                    /* next char has to be a quoted special */
                    NextCh( Search);
                    if( ('"' != Search->ch) && ('\\' != Search->ch)) {
                        /* error */
                        Search->sy = errorsy;
                        return;
                    }
                }

                if( i < IDENTSIZE) {
                    Search->ident[i] = Search->ch;
                }

                NextCh( Search);
            }

            if( i < IDENTSIZE) {
                Search->ident[i] = 0;
                Search->identLen = i;
            } else {
                Search->ident[IDENTSIZE] = 0;
                Search->identLen = IDENTSIZE;
            }

            Search->sy = stringsy;
        break;

        default:

            i = 0;
            if (Search->ch != ' ' && Search->ch != 0) {
                Search->ident[i] = lch;
                i++;
            } else {
                Search->sy = errorsy;
                break;
            }

            while (Search->ch != ' ' && Search->ch != 0) {
                if( i < IDENTSIZE) {
                    Search->ident[i] = Search->ch;
                    i++;
                }

                NextCh( Search);
            }

            if( i < IDENTSIZE) {
                Search->ident[i] = 0;
                Search->identLen = i;
            } else {
                Search->ident[IDENTSIZE] = 0;
                Search->identLen = IDENTSIZE;
            }

            Search->sy = atomsy;
            
            /* is it a keyword */
            for (i = 1; i < g_nkw; i++ ) {
                if( g_keylen[i] == Search->identLen  && 0 == XplStrNCaseCmp( Search->ident, g_key[i], g_keylen[i])) {
                    Search->sy = g_keysy[i];
                    break;
                }
            }
        break;
    }
}



/*----------------------------------------------------------------------------
;PDBSearch
Title:        
Globals:    
Return:        
Descrip:    search for text in a buffer

------------------------------------------------------------------------------*/
#define MAX_CHARS_IN_PDBSEARCH    512
int 
PDBSearch(char *Doc, char *SearchString)
{
    unsigned char *W[MAX_CHARS_IN_PDBSEARCH], *TextStart, *TextEnd, *text, *end;
    int WL[MAX_CHARS_IN_PDBSEARCH];
    unsigned char Expression[MAX_CHARS_IN_PDBSEARCH];
    long word, words=0, i,len;
    int Found=0;

    if (!Doc || !(*Doc))
        return(0);

    len=strlen(SearchString);
    if (len==0) {
        return(0);
    }

    if (len < MAX_CHARS_IN_PDBSEARCH) {
        strcpy(Expression, SearchString);
    } else {
        len = sizeof(Expression) - 1;
        strncpy(Expression, SearchString, len);
        Expression[MAX_CHARS_IN_PDBSEARCH - 1] = '\0';
    }

    for (i=0; i<len; i++) {
        Expression[i]=toupper(Expression[i]);
    }

    text=Expression;
    while (*text && IsWhitespace(*text)) {
        text++;
    }

    if ((*text=='\"') && ((end=strchr(text+1,'\"'))!=NULL)) {
        W[words++]=text+1;
        *end='\0';
        text=end+1;
    }
        
    while (*text && IsWhitespace(*text)) {
        text++;
    }

    if ((*text=='\0') && (!words))
        return(0);

    if (*text)
        W[words++]=text;

    for(i=text-Expression; (i<len) && (words<MAX_CHARS_IN_PDBSEARCH) ;i++) {
        if (IsWhitespace(Expression[i])) {
            if (!IsWhitespace(Expression[i+1])) {
                W[words++]=Expression+i+1;
                Expression[i]='\0';
            } else {
                Expression[i]='\0';
            }
        }
    }

    for (i=0; i<words;i++)
        WL[i]=strlen(W[i]);

    TextStart=Doc;
    TextEnd=Doc+strlen(Doc)-1;

    for (text=TextStart;(NMAP.state < NMAP_STOPPING) && (text!=TextEnd);text++) {
        for (word=0; word<words;word++) {
            i=0;
            while ((i!=WL[word]) && (W[word][i]==toupper(text[i]))) {
                i++;
            }
            if (WL[word]==i) {
                Found=1;

                if ((text!=TextStart && !IsWhitespace(*(text-1))) && ((text!=TextEnd) && !IsWhitespace(*(text+WL[word]))))
                    goto NoMatch;
            }
        }
NoMatch:
        ;
    }
    if (NMAP.state == NMAP_STOPPING) {
        return(0);
    }
    return(Found);
}

void
InitSearchModule(void)
{
    return;
}
