/****************************************************************************
 *
 * Copyright (c) 2004 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include "nmapdp.h"
#include "sql-lock.h"

#define NMAP_SQL_LOCKS_MASK 0x00000002
#define NMAP_SQL_LOCKS_ARRAY_SIZE (NMAP_SQL_LOCKS_MASK + 1)

NmapSqlLockChain NmapSqlLocks[NMAP_SQL_LOCKS_ARRAY_SIZE];

void *NmapSqlLocksPool = NULL;

BOOL 
NmapSqlLockInitCB(void *buffer, void *clientData)
{
    register NmapSqlLock *l = (NmapSqlLock *)buffer;

    if (l != NULL) {
        memset(l, 0, sizeof(NmapSqlLock));
        return(TRUE);
    }

    return(FALSE);
}

/* Init/Destroy */
NmapSqlLockErr 
NmapSqlLocksInit(unsigned long maxCached)
{
    unsigned long i;
    NmapSqlLockChain *chain;

    memset(&NmapSqlLocks, 0, sizeof(NmapSqlLocks));

    chain = &NmapSqlLocks[0];
    for (i = NMAP_SQL_LOCKS_ARRAY_SIZE; i; i--, chain++) {
        XplOpenLocalSemaphore(chain->sem, 1);
    }

    NmapSqlLocksPool = MemPrivatePoolAlloc("NMAP SQL Locks", sizeof(NmapSqlLock), 8, maxCached, TRUE, FALSE, NmapSqlLockInitCB, NULL, NULL);
    if (NmapSqlLocksPool) {
        return(NMAP_SQL_LOCK_OK);
    }

    return(NMAP_SQL_LOCK_OUT_OF_MEMORY);
}

void
NmapSqlLocksDestroy(void)
{
    unsigned long i;
    NmapSqlLock *lock;
    NmapSqlLock *next;
    NmapSqlLock *nextUid;
    NmapSqlLockChain *chain;

    chain = &NmapSqlLocks[0];
    for (i = NMAP_SQL_LOCKS_ARRAY_SIZE; i; i--, chain++) {
        XplCloseLocalSemaphore(chain->sem);

        lock = chain->head;
        while (lock) {
            next = lock->list.next;

            do {
                nextUid = next->uid.next;

                MemPrivatePoolReturnEntry((void *)lock);

                lock = nextUid;
            } while (lock);

            lock = next;
        }
    }

    MemPrivatePoolFree(NmapSqlLocksPool);

    return;
}

static NmapSqlLockErr 
NmapSqlLockLoad(NmapSqlLock **handle, unsigned long *hashHandle, unsigned char *user)
{
    int length;
    unsigned long slot;
    unsigned long hash;
    NmapSqlLock *lock;
    NmapSqlLock *last;
    NmapSqlLockChain *chain;

    length = strlen(user) + 1;

    if (hashHandle) {
        if (*hashHandle) {
            hash = *hashHandle;
        } else {
            GET_STRING_HASH(user, hash);
            *hashHandle = hash;
        }
    } else {
        GET_STRING_HASH(user, hash);
    }

    slot = hash & NMAP_SQL_LOCKS_MASK;
    chain = &(NmapSqlLocks[slot]);

    XplWaitOnLocalSemaphore(chain->sem);

    last = NULL;
    lock = chain->head;
    while (lock) {
        if (hash != lock->hash) {
            lock = lock->list.next;
            continue;
        }

        break;
    }

    while (lock) {
        if (memcmp(user, lock->user, length) != 0) {
            last = lock;
            lock = lock->uid.next;
            continue;
        }

        break;
    }

    if (lock) {
        lock->opened++;
        XplSignalLocalSemaphore(chain->sem);

        *handle = lock;
        return(NMAP_SQL_LOCK_OK);
    }

    lock = (NmapSqlLock *)MemPrivatePoolGetEntry(NmapSqlLocksPool);
    if (lock) {
        if (length < sizeof(lock->data)) {
            lock->user = lock->data;
        } else {
            lock->flags |= NMAP_SQL_FLAG_USER_ALLOCATED;
            lock->user = (unsigned char *)MemMalloc(length);
        }

        if (lock->user) {
            memcpy(lock->user, user, length);

            lock->opened = 1;

            lock->hash = hash;
            lock->chain = chain;

            lock->uid.next = NULL;
            lock->list.next = NULL;

            if (last) {
                lock->list.prev = NULL;

                last->uid.next = lock;
                lock->uid.prev = last;
            } else {
                lock->uid.prev = NULL;

                if ((lock->list.prev = chain->tail) != NULL) {
                    chain->tail->list.next = lock;
                } else {
                    chain->head = lock;
                }
                chain->tail = lock;
            }

            XplSignalLocalSemaphore(chain->sem);

            *handle = lock;
            return(NMAP_SQL_LOCK_OK);
        }
    }

    XplSignalLocalSemaphore(chain->sem);

    return(NMAP_SQL_LOCK_OUT_OF_MEMORY);
}

NmapSqlLockErr 
NmapSqlLockAcquire(NmapSqlLockType type, NmapSqlLock **handle, unsigned long *hashHandle, unsigned char *user)
{
    register NmapSqlLockErr err;

    err = NmapSqlLockLoad(handle, hashHandle, user);
    if (err == NMAP_SQL_LOCK_OK) {
        switch (type) {
            case NMAP_SQL_LOCK_OPEN: {
                return(err);
            }

            case NMAP_SQL_LOCK_READ: {
                break;
            }

            case NMAP_SQL_LOCK_WRITE: {
                break;
            }

            case NMAP_SQL_LOCK_PURGE: {
                break;
            }

            default: {
                return(NMAP_SQL_LOCK_UNKNOWN_TYPE);
            }
        }
    }

    return(err);
}

NmapSqlLockErr 
NmapSqlLockRelease(NmapSqlLockType type, NmapSqlLock *handle)
{
    register NmapSqlLock *lock = handle;

    XplWaitOnLocalSemaphore(handle->chain->sem);

    switch (type) {
        case NMAP_SQL_LOCK_OPEN: {
            lock->opened--;
            break;
        }

        case NMAP_SQL_LOCK_READ: {
            lock->readers--;
            return(NMAP_SQL_LOCK_OK);
        }

        case NMAP_SQL_LOCK_WRITE: {
            lock->writers--;
            return(NMAP_SQL_LOCK_OK);
        }

        case NMAP_SQL_LOCK_PURGE: {
            return(NMAP_SQL_LOCK_OK);
        }

        default: {
            return(NMAP_SQL_LOCK_UNKNOWN_TYPE);
        }
    }

    XplSignalLocalSemaphore(handle->chain->sem);

    return(NMAP_SQL_LOCK_OK);
}
