/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <openssl/ssl.h>
#include <openssl/err.h>
#include <hulautil.h>

#include <mdb.h>

#include "webadminp.h"

ConnectionStruct	*ConnectionCache[CONNECTION_CACHE_SIZE];
unsigned long		CCTop 		= 0;

/* Debug Statistics */
int					CCCached = 0;
int					CCCreate = 0;
int					CCFull = 0;
int					CCSpace = 0;

#define	URL_COMPRESSED_CHAR			'A'
#define	URL_COMPRESSION_PREFIX		'.'

static unsigned char HEX_TABLE[] = {"0123456789abcdef"};
static unsigned char URL_ENCODING_TABLE[] = {"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789_-"};
static unsigned char URL_DECODING_TABLE[256] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3F, 0xFE, 0xFF,
	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
	0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0xFF, 0xFF, 0xFF, 0xFF, 0x3E,
	0xFF, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};

static unsigned char BASE64_R[256] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3E, 0xFF, 0xFF, 0xFF, 0x3F,
	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
	0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};

/* This routine relies on the fact that base64 encoded data is always bigger than the unencoded form */
/* to decode in-place, with the unencoded data overwriting the encoded data. */
int
WADecodeBase64(unsigned char *Data, int Size, int *Remainder)
{
	int				DecodeIndex = 0;								/* Data[DecodeIndex] is position for next decoded byte */
	int				EncodeIndex = 0;								/* Data[EncodeIndex] is position of next encoded byte */
	int				i;
	unsigned char	Input[4];										/* Holds 4-byte encoded word while decoding */

	while(EncodeIndex < Size) {
		/* Get next 4 bytes to decode */
		Input[0]='\0';
		Input[1]='\0';
		Input[2]='\0';
		Input[3]='\0';

		for (i = 0; (i < 4) && (EncodeIndex < Size); ) {
			if ((Data[EncodeIndex] != '\r') && (Data[EncodeIndex] != '\n')) {
				Input[i++] = Data[EncodeIndex++];
			} else {
				EncodeIndex++;              /* Just skip CRLFs */
			}
		}
		if (i != 4) {							/* Ran out of Data */
			*Remainder = i;
			return(DecodeIndex);
		}

		/* 
			Decode block: Input is "6-bit bytes": top two bits of each 8-bit bytes are 0 and ignored
			out[0] = (Input[0] all 6 bits)  | (Input[1] First 2 bits)
			out[1] = (Input[1] last 4 bits) | (Input[2] First 4 bits)
			out[2] = (Input[2] last 2 bits) | (Input[3] all 6 bits)
		*/

		Data[DecodeIndex++] = ((BASE64_R[Input[0]] << 2) & 0xFC) | ((BASE64_R[Input[1]] >> 4) & 0x03);
		Data[DecodeIndex++] = ((BASE64_R[Input[1]] << 4) & 0xF0) | ((BASE64_R[Input[2]] >> 2) & 0x0F);
		Data[DecodeIndex++] = ((BASE64_R[Input[2]] << 6) & 0xC0) | ((BASE64_R[Input[3]])		  & 0x3F);
		if(Input[3] == '=') {				/* One padding byte */
			DecodeIndex--;
			if(Input[2] == '=') {			/* Two padding bytes */
				DecodeIndex--;
			}
		}
	}

	*Remainder = 0;
	return(DecodeIndex);
}


BOOL
WAQuickNCmp(unsigned char *str1, unsigned char *str2, int len)
{
	while (--len && *str1 && *str2 && toupper(*str1) == toupper(*str2)) {
		str1++;
		str2++;
	}
	return(toupper(*str1) == toupper(*str2));
}


BOOL
WAQuickCmp(unsigned char *str1, unsigned char *str2)
{
	while (*str1 && *str2 && toupper(*str1) == toupper(*str2)) {
		str1++;
		str2++;
	}
	return(toupper(*str1) == toupper(*str2));
}

static void
DebugConnCache(void)
{
	XplConsolePrintf("\rConnection Cache statistics:\nConnections retrieved from cache : %d\nConnections added to cache       : %d\n", CCCached, CCCreate);
	XplConsolePrintf("Connections destroyed            : %d\nConnections released into cache  : %d\n", CCFull, CCSpace);
}

ConnectionStruct
*GetCachedConnection(void)
{
	ConnectionStruct *RetVal;

	XplWaitOnLocalSemaphore(CacheSemaphore);

	if (CCTop > 0) {				/* We have a context in the cache to give out */
		CCTop--;
		RetVal = ConnectionCache[CCTop];
		XplSignalLocalSemaphore(CacheSemaphore);
		CCCached++;
	} else {							/* The cache is empty, we have to create a new context */
		XplSignalLocalSemaphore(CacheSemaphore);

		RetVal = malloc(sizeof(ConnectionStruct));
		CCCreate++;
	}

	return(RetVal);
}


BOOL
ReleaseCachedConnection(ConnectionStruct *Client)
{
	XplWaitOnLocalSemaphore(CacheSemaphore);

	if (CCTop == CONNECTION_CACHE_SIZE) {
		/* The cache is full; just destroy the memory */
		XplSignalLocalSemaphore(CacheSemaphore);
		free(Client);
		CCFull++;
	} else {
		/* We have space in the cache, stick the connection in there */
		ConnectionCache[CCTop] = Client;
		CCTop++;
		XplSignalLocalSemaphore(CacheSemaphore);
		CCSpace++;
	}
	return(TRUE);
}

BOOL
ClearConnectionCache(void)
{
	/* Clean up the connection cache */
	while (CCTop>0) {
		CCTop--;
		free(ConnectionCache[CCTop]);
	}
	return(TRUE);
}


BOOL
WARedirectClient(ConnectionStruct *Client, unsigned char *URL)
{
	int	i;

	i = strlen(URL);

	switch (Client->DeviceType) {
		case DEVICE_HTML: 
		default: {
			WASendClient(Client, "HTTP/1.1 302 Moved\r\nContent-type: text/html\r\nConnection: close\r\nLocation: ", 74);
			WASendClient(Client, URL, i);
			WASendClient(Client, "\r\n\r\n<H1>Document moved to <A HREF=\"", 35);
			WASendClient(Client, URL, i);
			WASendClient(Client, "\">", 2);
			WASendClient(Client, URL, i);
			WASendClient(Client, "</A></H1>", 9);
			Client->KeepAlive = FALSE;
			break;
		}

		case DEVICE_WML: {
			WASendClient(Client, "HTTP/1.1 200 OK\r\nContent-type: text/vnd.wap.wml\r\nConnection: close\r\n\r\n<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\"><wml><head><meta http-equiv=\"Cache-Control\" content=\"max-age=0\"/></head><card title=\"Session will be redirected\" onenterforward =\"", 332);
			WASendClient(Client, URL, i);
			WASendClient(Client, "\"></card></wml>", 15);
			Client->KeepAlive = FALSE;
			break;
		}
	}
	return(TRUE);	
}

BOOL
RedirectAfterLogin(ConnectionStruct *Client, unsigned char *URL)
{
	int	i;
	int	j;

 	j = strlen(URL);

	switch (Client->DeviceType) {
		case DEVICE_HTML: 
		default: {
			WASendClient(Client, "HTTP/1.1 302 Moved\r\nContent-type: text/html\r\nConnection: close\r\nLocation: ", 74);
			WASendClient(Client, URL, j);
			WASendClient(Client, "\r\nSet-Cookie: WAS-ID=", 21);
			i=snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Client->Session->SessionUID);
			WASendClient(Client, Client->Temp, i);
			WASendClient(Client, "; path=/\r\n\r\n<H1>Document moved to <A HREF=\"", 43);
			WASendClient(Client, URL, j);
			WASendClient(Client, "\">", 2);
			WASendClient(Client, URL, j);
			WASendClient(Client, "</A></H1>", 9);
			Client->KeepAlive = FALSE;
			break;
		}

		case DEVICE_WML: {
			WASendClient(Client, "HTTP/1.1 200 OK\r\nContent-type: text/vnd.wap.wml\r\nSet-Cookie: WAS-ID=", 68);
			i=snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Client->Session->SessionUID);
			WASendClient(Client, Client->Temp, i);
			WASendClient(Client, "; path=/\r\nConnection: close\r\n\r\n<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\"><wml><head><meta http-equiv=\"Cache-Control\" content=\"max-age=0\"/></head><card title=\"Session will be redirected\" onenterforward =\"", 293);
			WASendClient(Client, URL, j);
			WASendClient(Client, "\"></card></wml>", 15);
			Client->KeepAlive = FALSE;
			break;
		}
	}

	return(TRUE);	
}

#if 0
/* This function is currently not getting used */
static BOOL
SendCookie(ConnectionStruct *Client, unsigned char *URL)
{
	int	i;
	int	j;

	j = strlen(URL);

	switch (Client->DeviceType) {
		case DEVICE_HTML: 
		default: {
			WASendClient(Client, "HTTP/1.1 302 Moved\r\nContent-type: text/html\r\nConnection: close\r\nLocation: ", 74);
			WASendClient(Client, URL, j);
			WASendClient(Client, "\r\nSet-Cookie: WAS-ID=", 21);
 			i=snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Client->Session->SessionUID);
			WASendClient(Client, Client->Temp, i);
			WASendClient(Client, "; path=/\r\n\r\n<H1>Document moved to <A HREF=\"", 43);
			WASendClient(Client, URL, j);
			WASendClient(Client, "\">", 2);
			WASendClient(Client, URL, j);
			WASendClient(Client, "</A></H1>", 9);
			Client->KeepAlive = FALSE;
			break;
		}

		case DEVICE_WML: {
			WASendClient(Client, "HTTP/1.1 200 OK\r\nContent-type: text/vnd.wap.wml\r\nSet-Cookie: WAS-ID=", 68);
			i=snprintf(Client->Temp, sizeof(Client->Temp), "%lu", Client->Session->SessionUID);
			WASendClient(Client, Client->Temp, i);
			WASendClient(Client, "; path=/\r\nConnection: close\r\n\r\n<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\"><wml><head><meta http-equiv=\"Cache-Control\" content=\"max-age=0\"/></head><card title=\"Session will be redirected\" onenterforward =\"", 293);
			WASendClient(Client, URL, j);
			WASendClient(Client, "\"></card></wml>", 15);
			Client->KeepAlive = FALSE;
			break;
		}
	}
	return(TRUE);	
}
#endif

BOOL
SendTimeoutPage(ConnectionStruct *Client, URLStruct *URLData)
{
	unsigned long	*LanguageList;
	unsigned long	i;
	unsigned long	Template;
	unsigned long	Language = 0;

	switch (Client->DeviceType) {
		case DEVICE_HTML: 
		default: {
		    SessionStruct	*Session = malloc (sizeof(SessionStruct));
		    

			memset(Session, 0, sizeof(SessionStruct));

			/* Pick template; perform sanity check */
			if (URLData->Template < TemplateCount) {
				Template = URLData->Template;
			} else {
				Template = DefaultTemplate;
			}

			if (Templates[Template]->TimeoutTemplate == -1) {
				Template = DefaultTemplate;
			}

			if (URLData->Language>=Templates[Template]->LanguageCount) {
				URLData->Language=0;
			}

			/* Translate the language ID  */
			LanguageList = (unsigned long *)Templates[Template] + sizeof(TemplateStruct);
			for (i = 0; i < Templates[Template]->LanguageCount; i++) {
				if (i == LanguageList[URLData->Language]) {
					Language = i;
					break;
				}
			}
			if (i == Templates[Template]->LanguageCount) {
				Language = 0;
			}

			WASetSessionTemplate(Template, Language, Session);
			Client->KeepAlive = FALSE;

			/* We tell HandleTemplate not to send the header, so we can send a 404 ourselfs */
			WASendClient(Client, "HTTP/1.1 404 Document does not exist\r\nPragma: no-cache\r\nCache-Control: no-cache\r\nContent-type: text/html\r\nConnection: close\r\n\r\n", 127);

			WAHandleTemplate(Client, Session, Templates[Session->TemplateID]->TimeoutTemplate, FALSE);
			free(Session);
			break;
		}

		case DEVICE_WML: {
			WASendClient(Client, "HTTP/1.1 404 Not Found\r\nContent-type: text/vnd.wap.wml\r\nConnection: close\r\n\r\n<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\"><wml><head><meta http-equiv=\"Cache-Control\" content=\"max-age=0\"/></head><card title=\"Session timed out\"><p>Session timed out.</p><p><a href=\"/\">Log in?</a></p></card></wml>", 381);
			Client->KeepAlive = FALSE;
			break;
		}
	}
	return(TRUE);
}


BOOL
SendReloginPage(ConnectionStruct *Client, URLStruct *URLData)
{
	unsigned long	*LanguageList;
	unsigned long	i;
	SessionStruct	*Session;
	unsigned long	Template;
	unsigned long	Language = 0;

	Session = MemMalloc(sizeof(SessionStruct));
	memset(Session, 0, sizeof(SessionStruct));

	/* Pick template; perform sanity check */
	if (URLData->Template<TemplateCount) {
		Template = URLData->Template;
	} else {
		Template = DefaultTemplate;
	}

	if (URLData->Language >= Templates[Template]->LanguageCount) {
		URLData->Language = 0;
	}

	/* Translate the language ID  */
	LanguageList = (unsigned long *)Templates[Template] + sizeof(TemplateStruct);
	for (i = 0; i < Templates[Template]->LanguageCount; i++) {
		if (i == LanguageList[URLData->Language]) {
			Language = i;
			break;
		}
	}
	if (i == Templates[Template]->LanguageCount) {
		Language = 0;
	}

	WASetSessionTemplate(Template, Language, Session);


	switch (Client->DeviceType) {
		case DEVICE_HTML: 
		default: {
			WAHandleTemplate(Client, Session, Templates[Session->TemplateID]->ResumeTemplate, TRUE);
			Client->KeepAlive = FALSE;
			break;
		}

		case DEVICE_WML: {
			/* The following block of code is a copy of the  {Relogin} and {StoreData} tokens. */
			/* with some wml modifications.  This is a work around until we can used named templates */
			unsigned char				URL[256];
			unsigned char				Answer[BUFSIZE+1];
			int							count;

			/* First, accquire a TSession */
			Client->TSession = CreateTSession();

			/* Now store all data in the session */
			snprintf(Answer, sizeof(Answer), "%s/T%lu", WorkDir, Client->TSession->SessionID);
			Client->PostData = fopen(Answer, "wb");
			fprintf(Client->PostData, "%lu\r\n", Client->ContentLength+Client->BufferPtr);
			fprintf(Client->PostData, "%c%s\r\n", Client->EncodingType==FORM_DATA_ENCODED ? 'f' : 'u', Client->FormSeparator ? Client->FormSeparator : (unsigned char*)"");
			fprintf(Client->PostData, "%s\r\n", Client->URL);

			if (Client->BufferPtr > 0) {
				fwrite(Client->Buffer, 1, Client->BufferPtr, Client->PostData);
				Client->BufferPtr = 0;
			}
			while (Client->ContentLength > 0) {
				count = DoClientRead(Client, Client->Buffer+Client->BufferPtr, BUFSIZE-Client->BufferPtr);
				if ((count < 1) || (Exiting)) {
				    MemFree(Session);
					return(EndClientConnection(Client));
				}
				Client->BufferPtr += count;
				if (Client->BufferPtr > Client->ContentLength) {
					fwrite(Client->Buffer, 1, Client->ContentLength, Client->PostData);
					memmove(Client->Buffer, Client->Buffer + Client->ContentLength, Client->BufferPtr-Client->ContentLength);
					Client->BufferPtr -= Client->ContentLength;
					Client->ContentLength = 0;
				} else {
					fwrite(Client->Buffer, 1, Client->BufferPtr, Client->PostData);
					Client->ContentLength -= Client->BufferPtr;
					Client->BufferPtr = 0;
				}
			}
			fclose(Client->PostData);
			Client->PostData = NULL;
			Client->TSession->Timestamp = time(NULL);

			WAEncodeURL(Session, URL, 'x', 0, Client->TSession->SessionID, Client->TSession->SessionUID, 0, 0);
			/* End Block */
			/* The following lines has wml specific changes */
			ReleaseTSession(Client->TSession);
			Client->SessionUID = 0;
			Client->TSession = NULL;

			WASendClient(Client, "HTTP/1.1 200 OK\r\nContent-type: text/vnd.wap.wml\r\nConnection: close\r\n\r\n<?xml version=\"1.0\" encoding=\"UTF-8\"?><!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\"><wml><head><meta http-equiv=\"Cache-Control\" content=\"max-age=0\"/></head><template><do type=\"prev\" label=\"Back\"><prev/></do></template><card><p>The session has timed out, but can be resumed by logging in again.</p><p>Username: <input type=\"text\" name=\"UsernameValue\" value=\"\"/></p><p>Password: <input type=\"password\" name=\"PasswordValue\" value=\"\"/></p><p><anchor title=\"Submit\">Log In<go href=\"f\" method=\"post\"><postfield name=\"Username\" value=\"$(UsernameValue)\"/><postfield name=\"Password\" value=\"$(PasswordValue)\"/><postfield name=\"TSession\" value=\"", 752);
			WASendClient(Client, URL, strlen(URL));
			WASendClient(Client, "\"/></go></anchor></p></card></wml>", 34);
			break;
		}
	}
	
	MemFree(Session);
	return(TRUE);
}


#define CompressURL(Assignment)										\
		if ((Assignment) == URL_COMPRESSED_CHAR) {				\
			CompCount++;													\
		} else {																\
			if (!CompCount) {												\
				Target++;													\
			} else {															\
				/* Need to compress	 */								\
				URL[Target+2]=URL[Target];								\
				URL[Target++]=URL_COMPRESSION_PREFIX;				\
				URL[Target++]=URL_ENCODING_TABLE[CompCount];		\
				Target++;													\
				CompCount = 0;												\
			}																	\
		}

BOOL
WAEncodeURL(SessionStruct *Session, unsigned char *URL, unsigned char Type, unsigned long Request, unsigned long Arg1, unsigned long Arg2, unsigned long Arg3, unsigned long Arg4)
{
	URLStruct		URLData;
	unsigned long	Source;
	unsigned long	Target;
	unsigned char	*URLPtr=(unsigned char *)&URLData;
	unsigned long	CompCount;

	Target = 0;

	URL[Target++] = '/';
	URL[Target++] = Type;
	switch (Type) {
		case URL_TYPE_REDIRECT:
		case URL_TYPE_IMAGE: {
			URL[Target++] = '-';
			break;
		}

		default: {
			URL[Target++] = '?';
			break;
		}
	}

	URLData.Request = Request;
	URLData.Argument[0] = Arg1;
	URLData.Argument[1] = Arg2;
	URLData.Argument[2] = Arg3;
	URLData.Argument[3] = Arg4;
	URLData.Argument[4] = 0;
	URLData.ReloginOK = 0;
	URLData.SessionID = Session->SessionID;
	URLData.SessionUID = Session->SessionUID;
	URLData.Template = Session->TemplateID;
	URLData.Language = Session->Language;

	URLData.Checksum = 1234 + Arg1 + Arg2 + ~Arg3 + ~Arg4 + (Request << 2) - Session->TemplateID - Session->SessionUID;

#if !defined(UNCOMPRESSED_URLS)
	CompCount = 0;
	for (Source = 0; Source < sizeof(URLStruct); Source += 3) {
		CompressURL((URL[Target] = URL_ENCODING_TABLE[(URLPtr[Source] & 0xFC) >> 2]));
		CompressURL((URL[Target] = URL_ENCODING_TABLE[((URLPtr[Source] & 0x03) << 4) | ((URLPtr[Source+1] & 0xF0) >> 4)]));
		CompressURL((URL[Target] = URL_ENCODING_TABLE[((URLPtr[Source+1] & 0x0F) << 2) | ((URLPtr[Source+2] & 0xC0) >> 6)]));
		CompressURL((URL[Target] = URL_ENCODING_TABLE[URLPtr[Source+2] & 0x3F]));
	}
	if (CompCount) {
		URL[Target++]=URL_COMPRESSION_PREFIX;
		URL[Target++]=URL_ENCODING_TABLE[CompCount];
	}
	URL[Target] = '\0';
#else
	/* Watch for the ++ pre- and post-fixes :-) */
	for (Source = 0; Source < sizeof(URLStruct); Source += 3) {
		URL[Target++] = URL_ENCODING_TABLE[(URLPtr[Source] & 0xFC) >> 2];
		URL[Target++] = URL_ENCODING_TABLE[((URLPtr[Source] & 0x03) << 4) | ((URLPtr[Source+1] & 0xF0) >> 4)];
		URL[Target++] = URL_ENCODING_TABLE[((URLPtr[Source+1] & 0x0F) << 2) | ((URLPtr[Source+2] & 0xC0) >> 6)];
		URL[Target++] = URL_ENCODING_TABLE[URLPtr[Source+2] & 0x3F];
	}
	URL[Target++] = '\0';
#endif
	return(TRUE);
}

BOOL
WAEncodeURLEx(SessionStruct *Session, unsigned char *URL, unsigned char Type, unsigned long Request, unsigned long Arg1, unsigned long Arg2, unsigned long Arg3, unsigned long Arg4, BOOL ReloginOK)
{
	URLStruct		URLData;
	unsigned long	Source;
	unsigned long	Target;
	unsigned char	*URLPtr=(unsigned char *)&URLData;
	unsigned long	CompCount;

	Target = 0;

	URL[Target++] = '/';
	URL[Target++] = Type;
	switch (Type) {
		case URL_TYPE_REDIRECT:
		case URL_TYPE_IMAGE: {
			URL[Target++] = '-';
			break;
		}

		default: {
			URL[Target++] = '?';
			break;
		}
	}

	URLData.Request = Request;
	URLData.Argument[0] = Arg1;
	URLData.Argument[1] = Arg2;
	URLData.Argument[2] = Arg3;
	URLData.Argument[3] = Arg4;
	URLData.Argument[4] = 0;
	URLData.ReloginOK = ReloginOK;
	URLData.SessionID = Session->SessionID;
	URLData.SessionUID = Session->SessionUID;
	URLData.Template = Session->TemplateID;
	URLData.Language = Session->Language;

	URLData.Checksum = 1234 + Arg1 + Arg2 + ~Arg3 + ~Arg4 + (Request << 2) - Session->TemplateID - Session->SessionUID;

#if !defined(UNCOMPRESSED_URLS)
	CompCount = 0;
	for (Source = 0; Source < sizeof(URLStruct); Source += 3) {
		CompressURL((URL[Target] = URL_ENCODING_TABLE[(URLPtr[Source] & 0xFC) >> 2]));
		CompressURL((URL[Target] = URL_ENCODING_TABLE[((URLPtr[Source] & 0x03) << 4) | ((URLPtr[Source+1] & 0xF0) >> 4)]));
		CompressURL((URL[Target] = URL_ENCODING_TABLE[((URLPtr[Source+1] & 0x0F) << 2) | ((URLPtr[Source+2] & 0xC0) >> 6)]));
		CompressURL((URL[Target] = URL_ENCODING_TABLE[URLPtr[Source+2] & 0x3F]));
	}
	if (CompCount) {
		URL[Target++] = URL_COMPRESSION_PREFIX;
		URL[Target++] = URL_ENCODING_TABLE[CompCount];
	}
	URL[Target] = '\0';
#else
	/* Watch for the ++ pre- and post-fixes :-) */
	for (Source=0; Source<sizeof(URLStruct); Source+=3) {
		URL[Target++] = URL_ENCODING_TABLE[(URLPtr[Source] & 0xFC) >> 2];
		URL[Target++] = URL_ENCODING_TABLE[((URLPtr[Source] & 0x03) << 4) | ((URLPtr[Source+1] & 0xF0) >> 4)];
		URL[Target++] = URL_ENCODING_TABLE[((URLPtr[Source+1] & 0x0F) << 2) | ((URLPtr[Source+2] & 0xC0) >> 6)];
		URL[Target++] = URL_ENCODING_TABLE[URLPtr[Source+2] & 0x3F];
	}
	URL[Target++] = '\0';
#endif
	return(TRUE);
}

BOOL
DecodeURL(unsigned char *URL, URLStruct *URLData)
{
	unsigned long	Source;
	unsigned long	Target;
	unsigned char	*URLPtr = (unsigned char *)URLData;
	unsigned char	URLBuffer[(sizeof(URLStruct)*4)/3];
	unsigned long	len;

	WADebug("DecodeURL() called\n");
#if !defined(UNCOMPRESSED_URLS)
	Source = 0;
	for (Target = 0; Target < sizeof(URLBuffer);) {
		if (URL_DECODING_TABLE[URL[Source]] == 0xFF) {
			memset(URLData, 0, sizeof(URLStruct));
			URLData->Template = DefaultTemplate;

			return(FALSE);
		}

		if (URL[Source] == URL_COMPRESSION_PREFIX) {
			if ((len = URL_DECODING_TABLE[URL[++Source]]) > (sizeof(URLBuffer) - Target)) {
				memset(URLData, 0, sizeof(URLStruct));
				URLData->Template = DefaultTemplate;

				return(FALSE);
			}
			memset(URLBuffer+Target, URL_COMPRESSED_CHAR, len);
			Target += len;
			Source++;
		} else {
			URLBuffer[Target++] = URL[Source++];
		}
	}

	/* Base64 encoding always creates chunks of 4 chars */
	if (Target != sizeof(URLBuffer)) {
		memset(URLData, 0, sizeof(URLStruct));
		URLData->Template = DefaultTemplate;

		return(FALSE);
	}

	Source = 0;
	for (Target = 0; Target < sizeof(URLStruct); Source += 4) {
		URLPtr[Target++] = ((URL_DECODING_TABLE[URLBuffer[Source]] << 2)   & 0xFC) | ((URL_DECODING_TABLE[URLBuffer[Source+1]] >> 4) & 0x03);
		URLPtr[Target++] = ((URL_DECODING_TABLE[URLBuffer[Source+1]] << 4) & 0xF0) | ((URL_DECODING_TABLE[URLBuffer[Source+2]] >> 2) & 0x0F);
		URLPtr[Target++] = ((URL_DECODING_TABLE[URLBuffer[Source+2]] << 6) & 0xC0) | ((URL_DECODING_TABLE[URLBuffer[Source+3]]) & 0x3F);
	}
#else
	/* Base64 encoding always creates chunks of 4 chars */
	if ((strlen(URL) % 4) != 0) {
		memset(URLData, 0, sizeof(URLStruct));
		URLData->Template = DefaultTemplate;

		return(FALSE);
	}

	Source = 0;
	for (Target = 0; Target < sizeof(URLStruct); Source += 4) {
		if (URL_DECODING_TABLE[URL[Source]] == 0xFF) {
			memset(URLData, 0, sizeof(URLStruct));
			URLData->Template = DefaultTemplate;

			return(FALSE);
		}

		URLPtr[Target++] = ((URL_DECODING_TABLE[URL[Source]] << 2) & 0xFC) | ((URL_DECODING_TABLE[URL[Source+1]] >> 4) & 0x03);
		URLPtr[Target++] = ((URL_DECODING_TABLE[URL[Source+1]] << 4) & 0xF0) | ((URL_DECODING_TABLE[URL[Source+2]] >> 2) & 0x0F);
		URLPtr[Target++] = ((URL_DECODING_TABLE[URL[Source+2]] << 6) & 0xC0) | ((URL_DECODING_TABLE[URL[Source+3]]) & 0x3F);
	}
#endif

	/* Verify the checksum */
	if ((1234 + URLData->Argument[0] + URLData->Argument[1] + ~URLData->Argument[2] + ~URLData->Argument[3] + (URLData->Request << 2) - URLData->Template - URLData->SessionUID) != URLData->Checksum) {
		memset(URLData, 0, sizeof(URLStruct));
		URLData->Template = DefaultTemplate;

		return(FALSE);
	}
	return(TRUE);
}


BOOL
WAGetFormNameEx(ConnectionStruct *Client, unsigned char *Name, unsigned char *Type, unsigned char *Filename, unsigned long NameSize)
{
	unsigned char	*ptr;
	unsigned char	*Src;
	unsigned char	*Dest;
	unsigned char	*DestEnd;
	long				count;

	Client->FormState = FORMFIELD_NAME;

	/* Decide if we can read more data from the network */
	if (Client->ContentLength == 0) {
		/* We have nothing else to get */
		if (Client->BufferPtr == 0) {
			TemplateDebug("Client->ContentLength is 0 and buffer empty, no more data in the form\n");
			return(FALSE);
		}
	}

	/*
		This while loop will make sure we fill our buffer as much as possible, but at least 
		until we've read all ContentLength
	*/
	while ((Client->ContentLength > 0) && (Client->BufferPtr != BUFSIZE)) {
		if (!Client->PostData) {
			count = DoClientRead(Client, Client->Buffer+Client->BufferPtr, BUFSIZE-Client->BufferPtr);
			if ((count<1) || (Exiting)) {
				return(EndClientConnection(Client));
			}
			if (count > Client->ContentLength) {
				Client->ContentLength = 0;
			} else {
				Client->ContentLength -= count;
			}
			Client->BufferPtr += count;
			Client->Buffer[Client->BufferPtr] = '\0';
		} else {
			if (ferror(Client->PostData) || feof(Client->PostData) || Exiting) {
				return(EndClientConnection(Client));
			}
			count = fread(Client->Buffer + Client->BufferPtr, 1, BUFSIZE-Client->BufferPtr, Client->PostData);
			Client->BufferPtr += count;
			if (count > Client->ContentLength) {
				Client->ContentLength = 0;
			} else {
				Client->ContentLength -= count;
			}
		}
	}

	/* We now should have at least enough data to pull the name */

	if (Client->EncodingType == FORM_URL_ENCODED) {
		if (((ptr = strchr(Client->Buffer, '=')) == NULL) || ((ptr - Client->Buffer) > NameSize)) {
			/* Aw shit; who's got a name bigger than BUFSIZE??? */
			//XplConsolePrintf("Quick, let the NIMS team know that a FormName to large for BUFSIZE or NameSize (Client->Buffer:%s)\n", Client->Buffer);

			/*
				Make sure we bail in a way that allows us to be called again...
				The problem with the solution below is that if NameSize was too small, we'll still have 
				the value to be read, but no name to go with it... Oh well, if this function returns FALSE the
				caller should probably bail anyway...
			*/
			if (ptr) {
				Client->BufferPtr = strlen(ptr + 1);
				memmove(Client->Buffer, ptr + 1, Client->BufferPtr + 1);
			} else {
				Client->BufferPtr = 0;
			}
			return(FALSE);
		}

		Dest = Name;
		Src = Client->Buffer;
		DestEnd = Name + NameSize;

		while (Src != ptr && (Dest != DestEnd)) {
			switch (*Src) {
				case '%': {
					Src++;
					*Src |= 0x20;
					if (*Src > 64) {
						*Dest=(*Src-'a'+10)<<4;
					} else {
						*Dest=(*Src-'0')<<4;
					}
					Src++;
					*Src|=0x20;
					if (*Src > 64) {
						*Dest|=(*Src-'a'+10);
					} else {
						*Dest|=(*Src-'0');
					}
					Src++;
					if (*Dest!='\r') {
						Dest++;
					}
					break;
				}

				case '\n':
				case '\r': {
					Src=ptr;
					break;
				}

				case '+': {
					*Dest=' ';
					Src++;
					Dest++;
					break;
				}

				default: {
					*Dest=*Src;
					Src++;
					Dest++;
					break;
				}
			}
		}

		*Dest='\0';

		/* remove the form data from the input buffer */
		Client->BufferPtr=strlen(ptr+1);
		memmove(Client->Buffer, ptr+1, Client->BufferPtr+1);
	} else {
		/* FORM_DATA_ENCODED */
		if (!Client->FormSeparator) {
			Client->BufferPtr=0;
			return(FALSE);
		}

		if (WAReadClientLine(Client)==FALSE) {
			return(FALSE);
		}

		TemplateDebug("Processing %s\n", Client->Command);
		
		if (Client->Command[0]=='\0') {
			/*
				Crap, where's our separator??? Try to return so we can be called again 
				But, we might also have hit the end of the input stream...
			*/
			Client->BufferPtr=0;
			return(FALSE);
		}

		while (Client->Command[0]!='\0') {
			unsigned char	*ptr2;

			if (WAQuickNCmp(Client->Command, "content-disposition: ", 21)) {
				/* We have two things to grab, the name and the filename */
				TemplateDebug("Found disposition line:%s\n", Client->Command);
				ptr=strchr(Client->Command, '=');
				while (ptr) {
					if (WAQuickNCmp(ptr-8, "filename", 8)) {
						if (Filename) {
							ptr++;
							if (*ptr=='"') {
								ptr++;
								HulaStrNCpy(Filename, ptr, XPL_MAX_PATH);
								ptr2=strchr(Filename, '"');
								if (ptr2) {
									*ptr2='\0';
								}
							} else {
								HulaStrNCpy(Filename, ptr, XPL_MAX_PATH);
							}
						}
					} else if (WAQuickNCmp(ptr-4, "name", 4)) {
						ptr++;
						if (*ptr=='"') {
							ptr++;
							HulaStrNCpy(Name, ptr, NameSize);
							ptr2=strchr(Name, '"');
							if (ptr2) {
								*ptr2='\0';
							}
						} else {
						    HulaStrNCpy(Name, ptr, NameSize);
						}
					}
					ptr=strchr(ptr+1, '=');
				}
			} else if (Type && (WAQuickNCmp(Client->Command, "content-type: ", 14))) {
				ptr=strchr(Client->Command+14, ';');
				if (!ptr) {
					ptr=strchr(Client->Command+14, ' ');
				}
				if (ptr) {
					*ptr='\0';
				}
				HulaStrNCpy(Type, Client->Command+14, 128);
			} else {
				/* Find Teminating separator */
				count=strlen(Client->FormSeparator);
				if (WAQuickNCmp(Client->Command, Client->FormSeparator, count)) {
					if ((Client->Command[count]=='-') && (Client->Command[count+1]=='-')) {
						/* We found the end of the form data */
						/* Flush the buffer */
						Client->BufferPtr=0;
						return(FALSE);
					}
				} else {
					TemplateDebug("Unknown header line:%s\n", Client->Command);
				}
			}
			if (WAReadClientLine(Client)==FALSE) {
				return(FALSE);
			}
		}
	}
	TemplateDebug("Returning from WAGetFormName, Name:%s\n", Name);
	return(TRUE);
}

unsigned long
WAGetFormValue(ConnectionStruct *Client, unsigned char *Value, unsigned long *ValueSize)
{
	unsigned char	*ptr;
	unsigned char	*Src;
	unsigned char	*Dest;
	unsigned char	*DestEnd;
	long				count;

	if (Client->FormState==FORMFIELD_DONE) {
		Client->FormState=FORMFIELD_NAME;
		*ValueSize=0;
		return(FORMFIELD_NAME);
	}

	/* Decide if we can read more data from the network */
	if (Client->ContentLength==0) {
		/* We have nothing else to get */
		if (Client->BufferPtr==0) {
			TemplateDebug("Client->ContentLength is 0 and buffer empty, no more data in the form\n");
			Client->FormState=FORMFIELD_DONE;
			*ValueSize=0;
			return(FORMFIELD_NAME);
		}
	}

	/*
		This while loop will make sure we fill our buffer as much as possible, but at least 
		until we've read all ContentLength
	*/
	while ((Client->ContentLength>0) && (Client->BufferPtr!=BUFSIZE)) {
		if (!Client->PostData) {
			count=DoClientRead(Client, Client->Buffer+Client->BufferPtr, BUFSIZE-Client->BufferPtr);
			if ((count<1) || (Exiting)) {
				EndClientConnection(Client);
				Client->FormState=FORMFIELD_DONE;
				*ValueSize=0;
				return(FORMFIELD_NAME);
			}
			if (count>Client->ContentLength) {
				Client->ContentLength=0;
			} else {
				Client->ContentLength-=count;
			}
			Client->BufferPtr+=count;
			Client->Buffer[Client->BufferPtr]='\0';
		} else {
			if (ferror(Client->PostData) || feof(Client->PostData) || Exiting) {
				EndClientConnection(Client);
				Client->FormState=FORMFIELD_DONE;
				*ValueSize=0;
				return(FORMFIELD_NAME);
			}
			count=fread(Client->Buffer+Client->BufferPtr, 1, BUFSIZE-Client->BufferPtr, Client->PostData);
			Client->BufferPtr+=count;
			if (count>Client->ContentLength) {
				Client->ContentLength=0;
			} else {
				Client->ContentLength-=count;
			}
		}
	}

	if (Client->EncodingType==FORM_URL_ENCODED) {
		/* We now should have at least some data for the value */

		/* Find the end of our possible buffer */
		ptr=Client->Buffer;
		if (*ValueSize<Client->BufferPtr) {
			Src=Client->Buffer+*ValueSize;
		} else {
			Src=Client->Buffer+Client->BufferPtr;
		}
		while (ptr<Src && ptr[0]!='&') {
			ptr++;
		}

		/* We now either point to an & or to the end of the buffer that can fit into Value */
		Client->BufferPtr=strlen(ptr);
		if (ptr==Src) {
			Client->FormState=FORMFIELD_PARTIAL;
			if (Client->BufferPtr==0 && Client->ContentLength==0) {
				Client->FormState=FORMFIELD_DONE;
			}
		} else {
			Client->FormState=FORMFIELD_DONE;
		}

		Dest = Value;
		Src = Client->Buffer;
		DestEnd = Value + *ValueSize - 1;	/* Leave one for a terminator */

		while (Src != ptr && (Dest < DestEnd)) {
			switch (*Src) {
				case '%': {
					/* Make sure there's enough data left for us */
					if (Src+2>=ptr) {
						/* Not enough data, check if there's more to come */
						if (Client->FormState==FORMFIELD_PARTIAL) {
							Client->BufferPtr+=ptr-Src;
							ptr=Src;
						} else {
							/* We're in trouble, the guy sent an incomplete escape sequence */
							Src=ptr;
						}
						continue;
					}
					Src++;
					*Src|=0x20;
					if (*Src > 64) {
						*Dest=(*Src-'a'+10)<<4;
					} else {
						*Dest=(*Src-'0')<<4;
					}
					Src++;
					*Src|=0x20;
					if (*Src > 64) {
						*Dest|=(*Src-'a'+10);
					} else {
						*Dest|=(*Src-'0');
					}
					Src++;
					if (*Dest!='\r') {
						Dest++;
					}
					continue;
				}

				case '\n':
				case '\r': {
					Src=ptr;
					continue;
				}

				case '+': {
					*Dest=' ';
					Src++;
					Dest++;
					continue;
				}

				default: {
					*Dest=*Src;
					Src++;
					Dest++;
					continue;
				}
			}
		}

		*Dest='\0';
		*ValueSize=Dest-Value;
		/* remove the form data from the input buffer */
		if (ptr[0]=='&' && Client->FormState==FORMFIELD_DONE) {
			ptr++;
			Client->BufferPtr--;
		}
		memmove(Client->Buffer, ptr, Client->BufferPtr+1);
	} else {
		BOOL				Found=FALSE;
		unsigned long	i;
		unsigned long	Sep;
		unsigned long	SepLen;
		unsigned long	BufLen;
		unsigned long	DataLen;

		/* FORM_DATA_ENCODED */
		/* 
			We have as much data as possible in the buffer; 
			if there's not enough for another separator return 
			and wait for more data
		*/

		if (!Client->FormSeparator) {
			Client->BufferPtr=0;
			*ValueSize=0;
			return(FORMFIELD_NAME);
		}

		SepLen=strlen(Client->FormSeparator);

		/*
			When we get here, we're ready to read the body; we keep reading 'til we hit the MIME boundary.
			The tricky part is that we might get binary data, so we cannot use any string function or our
			usual routines...
		*/

		/* Temporary fix to get home */
		*ValueSize-=1;

		/* Determine how much data there is to check */
		if (*ValueSize>Client->BufferPtr) {
			BufLen=Client->BufferPtr;
		} else {
			BufLen=*ValueSize;
		}

		/* Calculate how much data we can return max */
		if (Client->ContentLength==0) {
			DataLen=BufLen;
		} else {
			DataLen=BufLen-SepLen;
			if (BufLen<SepLen) {
				//XplConsolePrintf("Serious form-trouble #1\n");
				Client->ContentLength=0;
				DataLen=BufLen;
			}
		}

		Sep=0;
		for(i = 0; (i < BufLen) && !Found; i++) {
			if (Client->Buffer[i] == Client->FormSeparator[Sep]) {
				if (++Sep == SepLen) {
					/* We found the separator in the stream; calculate length but add one since the index is 0 based */
					DataLen=i-Sep+1;
					Found=TRUE;
				}
			} else if (Sep) {
				i-=Sep;
				Sep=0;
			}
		}

		/* We now know how much data to send and if there's more for this entry */

		/* We could get DataLen=0 if the last thing in the buffer was the separator */
		memset(Value, 0, *ValueSize);
		if (DataLen>0) {
			memcpy(Value, Client->Buffer, DataLen);
			memmove(Client->Buffer, Client->Buffer+DataLen, Client->BufferPtr-DataLen);
			Client->BufferPtr-=DataLen;
			if (Found) {
				/* Strip trailing CR/LF */
				DataLen--;
				if (Value[DataLen]=='\n') {
					if (DataLen && Value[DataLen-1]=='\r') {
						DataLen--;
					}
				}
			}

			*ValueSize=DataLen;
			Value[DataLen]='\0';

			if (Found) {
				Client->FormState=FORMFIELD_DONE;
				return(FORMFIELD_DONE);
			} else {
				Client->FormState=FORMFIELD_PARTIAL;
				return(FORMFIELD_PARTIAL);
			}
		} else {
			*ValueSize=0;
			Value[0]='\0';
			Client->FormState=FORMFIELD_DONE;
			return(FORMFIELD_DONE);
		}
	}

	return(Client->FormState);
}


unsigned char
*GetURLFormName(unsigned char *URL, unsigned char *Name, unsigned long NameSize)
{
	unsigned char	*ptr;
	unsigned char	*Src;
	unsigned char	*Dest;
	unsigned char	*DestEnd;

	TemplateDebug("WAGetURLFormName() called\n");

	if (((ptr=strchr(URL, '='))==NULL) || ((ptr-URL) > NameSize)) {
		return(NULL);
	}

	Dest=Name;
	Src=URL;
	DestEnd=Name+NameSize;

	while (Src!=ptr && (Dest!=DestEnd)) {
		switch (*Src) {
			case '%': {
				Src++;
				*Src|=0x20;
				if (*Src > 64) {
					*Dest=(*Src-'a'+10)<<4;
				} else {
					*Dest=(*Src-'0')<<4;
				}
				Src++;
				*Src|=0x20;
				if (*Src > 64) {
					*Dest|=(*Src-'a'+10);
				} else {
					*Dest|=(*Src-'0');
				}
				Src++;
				if (*Dest!='\r') {
					Dest++;
				}
				break;
			}

			case '\n':
			case '\r': {
				Src=ptr;
				break;
			}

			case '+': {
				*Dest=' ';
				Src++;
				Dest++;
				break;
			}

			default: {
				*Dest=*Src;
				Src++;
				Dest++;
				break;
			}
		}
	}

	*Dest='\0';

	TemplateDebug("Returning from WAGetURLFormName, Name:%s\n", Name);
	return(ptr+1);
}

unsigned char
*GetURLFormValue(unsigned char *URL, unsigned char *Value, unsigned long *ValueSize)
{
	unsigned char	*ptr;
	unsigned char	*Src;
	unsigned char	*Dest;
	unsigned char	*DestEnd;
	long				count;

	TemplateDebug("GetURLFormValue() called\n");

	if (!URL) {
		return(NULL);
	}

	/* Find the end of our possible buffer */
	count=strlen(URL);
	ptr=URL;
	if (*ValueSize<count) {
		Src=URL+*ValueSize;
	} else {
		Src=URL+count;
	}
	while (ptr<Src && ptr[0]!='&') {
		ptr++;
	}

	Dest=Value;
	Src=URL;
	DestEnd=Value+*ValueSize;

	while (Src!=ptr && (Dest!=DestEnd)) {
		switch (*Src) {
			case '%': {
				/* Make sure there's enough data left for us */
				if (Src+2>=ptr) {
					/* We're in trouble, the guy sent an incomplete escape sequence */
					Src=ptr;
					continue;
				}
				Src++;
				*Src|=0x20;
				if (*Src > 64) {
					*Dest=(*Src-'a'+10)<<4;
				} else {
					*Dest=(*Src-'0')<<4;
				}
				Src++;
				*Src|=0x20;
				if (*Src > 64) {
					*Dest|=(*Src-'a'+10);
				} else {
					*Dest|=(*Src-'0');
				}
				Src++;
				if (*Dest!='\r') {
					Dest++;
				}
				continue;
			}

			case '\n':
			case '\r': {
				Src=ptr;
				continue;
			}

			case '+': {
				*Dest=' ';
				Src++;
				Dest++;
				continue;
			}

			default: {
				*Dest=*Src;
				Src++;
				Dest++;
				continue;
			}
		}
	}

	*Dest='\0';
	*ValueSize=Dest-Value;
	/* remove the form data from the input buffer */
	if (ptr[0]=='&') {
		ptr++;
	} else if (ptr[0]=='\0') {
		return(NULL);
	}

	TemplateDebug("Returning from WAGetFormValue\n");
	return(ptr);
}

int
WAMDBtoX500(const unsigned char *MDB, unsigned char *X500, MDBValueStruct *V)
{
	unsigned char	*PathElement;
	int				X500Ptr		= 0;
	int				PathElemLen;
	int				i;
	unsigned char	Tree[MDB_MAX_TREE_CHARS+2];
	unsigned char	*Restore = NULL;

	if (!MDB[0]) {
		X500[0] = '\0';
		return(0);
	}

	Tree[0]='\\';
	MDBGetServerInfo(NULL, Tree+1, V);

	X500[0] = '\0';
	do {
		/* Find the beginning of the next element (back-to-front) */
		PathElement = strrchr(MDB, '\\');
		if (PathElement) {
			if ((PathElement == MDB) && (XplStrCaseCmp(PathElement, Tree) == 0)) {
				break;								/* Leave loop; don't copy tree name */
			}
			PathElement++;
		} else {
			PathElement = (unsigned char *)MDB;
		}
		
		PathElemLen = strlen(PathElement);

		/* Copy the element and add the element separator */

		for (i = 0; i < PathElemLen; i++) {
			/*
			 *  A "." will appear in a name if it is a full email address, such as
			 *  Fink@biteme.com.  To display it, that "." must be represented as "\."
			 *  to delineate it from the standard Context Shifting "."
			 */
			if (PathElement[i] == '.') {
				X500[X500Ptr++] = '\\';
			}
			X500[X500Ptr++] = PathElement[i];
		}
		X500[X500Ptr++] = '.';
		X500[X500Ptr] = '\0';

		/* Prepare for the next loop: terminate MDB just before the beginning of this element */
		if(PathElement != MDB) {
			if (Restore) {
				*Restore = '\\';
			}

			*--PathElement = '\0';
			Restore = PathElement;
		}
	} while((PathElement != MDB) && (MDB[0] != '\0'));

	if (Restore) {
		*Restore = '\\';
	}

	if(X500Ptr > 0) {
		X500[--X500Ptr] = '\0';		/* Cut off trailing period */
	}

	return(X500Ptr);
}

int
WAX500toMDB(const unsigned char *X500, unsigned char *MDB, MDBValueStruct *V)
{
	unsigned char	*X500Element;
	int				MDBPtr;
	BOOL				First = TRUE;
	unsigned char	Tree[MDB_MAX_TREE_CHARS + 1];
	unsigned char	*Restore = NULL;

	if (!X500[0]) {
		MDB[0] = '\0';
		return(0);
	}

	MDBGetServerInfo(NULL, Tree, V);

	/* Make the MDB absolute */
	MDB[0] = '\\';
	HulaStrNCpy(MDB + 1, Tree, MDB_MAX_OBJECT_CHARS);

	MDBPtr = strlen(MDB);

	do {
		/* Find the beginning of the next element (back-to-front) */

		X500Element = (unsigned char *)X500 + strlen(X500) - 1;

		do {
			if (X500Element[0] == '.' && X500Element[-1] != '\\') {
				break;
			}

			X500Element--;
		} while (X500Element >= (unsigned char *)X500);

		if (X500Element) {
			if (First && (XplStrCaseCmp(X500Element + 1, Tree + 1) == 0)) { 		/* Check for trailing tree name */
				First = FALSE;
				*X500Element = '\0';
				continue;
			}

			X500Element++;
		} else {
			X500Element = (unsigned char *)X500;
		}

		/*
			Copy the element and add the element separator, but only if it won't walk
			past MDB_MAX_TREE_CHARS.  We don't want to generate an invalid DN.
		*/
		if ((MDBPtr + strlen(X500Element) + 2) < MDB_MAX_OBJECT_CHARS) {
			MDBPtr += snprintf(MDB + MDBPtr, MDB_MAX_OBJECT_CHARS - MDBPtr, "\\%s", X500Element);
		}

		/* Prepare for the next loop: terminate the X500 string just before the beginning of this element */
		if (X500Element != X500) {
			if (Restore) {
				*Restore = '.';
			}

			*--X500Element = '\0';
			Restore = X500Element;
		}
		First = FALSE;
	} while ((X500Element != X500) && (X500[0] != '\0'));

	if (Restore) {
		*Restore = '.';
	}

	/* Now we need to unescape any escaped periods */
	while ((X500Element = strstr(MDB, "\\."))) {
		memmove(X500Element, X500Element + 1, strlen(X500Element + 1) + 1);
	}

	return(MDBPtr);
}

#define	CLASSLIST_ALLOC_STEPS	20

BOOL
AddClass(SessionStruct *Session, const unsigned char *Class, unsigned long Description, unsigned long Image)
{
	if ((Session->ClassListCount+1)>=Session->ClassListAlloc) {
		Session->ClassList=realloc(Session->ClassList, (Session->ClassListAlloc+CLASSLIST_ALLOC_STEPS) * sizeof(ClassInfoStruct));
		if (!Session->ClassList) {
			return(FALSE);
		}
		memset((void *)((unsigned long)Session->ClassList+(unsigned long)((unsigned long)sizeof(ClassInfoStruct)*(unsigned long)Session->ClassListAlloc)), 0, sizeof(ClassInfoStruct)*CLASSLIST_ALLOC_STEPS);
		Session->ClassListAlloc+=CLASSLIST_ALLOC_STEPS;
	}

	Session->ClassList[Session->ClassListCount].Class=strdup(Class);
	Session->ClassList[Session->ClassListCount].Description=Session->Strings[Description];
	Session->ClassList[Session->ClassListCount].Image=Image;
	Session->ClassList[Session->ClassListCount].TemplateID=Session->TemplateID;

	Session->ClassListCount++;
	return(TRUE);
}

BOOL
FreeClassList(SessionStruct *Session)
{
	unsigned long	i;

	for (i=0; i<Session->ClassListCount; i++) {
		free(Session->ClassList[i].Class);
	}
	if (Session->ClassList) {
		free(Session->ClassList);
		Session->ClassList=NULL;
	}
	Session->ClassListAlloc=0;
	Session->ClassListCount=0;
	return(TRUE);
}

BOOL
WASendURLExtra(ConnectionStruct *Client, const unsigned char *Extra)
{
	unsigned char	Buffer[MDB_MAX_OBJECT_CHARS*3+1];
	unsigned char	*ptr;
	unsigned long	Pos=0;

	Buffer[Pos++]='+';

	ptr = (unsigned char *)Extra;
	while (ptr) {
	    if (Pos >= MDB_MAX_OBJECT_CHARS*3 + 1) {
            WASendClient(Client, (unsigned char *)Buffer, Pos);
            Pos = 0;
	    }
	    
		switch(ptr[0]) {
			case '\\': {
				Buffer[Pos++]='/';
				ptr++;
				continue;
			}

			case ' ': {
				Buffer[Pos++]='+';
				ptr++;
				continue;
			}

			case '\0': {
				WASendClient(Client, (unsigned char *)Buffer, Pos);
				return(TRUE);
			}

			default: {
				if ((ptr[0] < 0x30) || (ptr[0] > 0x7a) || ((ptr[0] > 0x39) && (ptr[0] < 0x41)) || ((ptr[0] > 0x5a) && (ptr[0] < 0x61))) {
					Buffer[Pos++]='%';
					Buffer[Pos++]=HEX_TABLE[(ptr[0] & 0xf0)>>4];
					Buffer[Pos++]=HEX_TABLE[ptr[0] & 0x0f];
				} else {
					Buffer[Pos++]=ptr[0];
				}
				ptr++;
				continue;
			}
		}
	}

	return(TRUE);
}

static __inline unsigned char 
HexToChar(signed char h)
{
   signed char c;

   c = h - '0';
   if((c > 9) || (c < 0)) {
      switch(h) {
         case 'A': case 'a':  c = 10;  break;
         case 'B': case 'b':  c = 11;  break;
         case 'C': case 'c':  c = 12;  break;
         case 'D': case 'd':  c = 13;  break;
         case 'E': case 'e':  c = 14;  break;
         case 'F': case 'f':  c = 15;  break;
      }
   }
   return(c);
}

unsigned long
WAURLExtraDecode(ConnectionStruct *Client, unsigned char *Target, unsigned long TargetSize)
{
	unsigned long	Pos=0;
	unsigned char	*ptr;

	if ((ptr=Client->URLExtra)==NULL) {
		Target[0]='\0';
		return(0);
	}

	while (TRUE) {
		if (Pos>=TargetSize) {
			return(Pos);
		}

		switch(ptr[0]) {
			case '+': {
				Target[Pos++]=' ';
				ptr++;
				continue;
			}

			case '/': {
				Target[Pos++]='\\';
				ptr++;
				continue;
			}

			case '%': {
				ptr++;
				ptr[0]|=0x20;
				if (ptr[0] > 64) {
					Target[Pos]=(ptr[0]-'a'+10)<<4;
				} else {
					Target[Pos]=(ptr[0]-'0')<<4;
				}
				ptr++;
				ptr[0]|=0x20;
				if (ptr[0] > 64) {
					Target[Pos]|=(ptr[0]-'a'+10);
				} else {
					Target[Pos]|=(ptr[0]-'0');
				}
				ptr++;
				Pos++;
				continue;
			}

			case '\0': {
				Target[Pos]='\0';
				return(Pos);
			}

			default: {
				Target[Pos++]=ptr[0];
				ptr++;
				continue;
			}
		}
	}
}

unsigned long
FindClassTemplate(SessionStruct *Session, unsigned char *Class, unsigned long Default)
{
	unsigned long	Template;

	Template=(unsigned long)WAFindTemplatePage(Class, Session->TemplateID);
	if (Template!=-1) {
		return(Template);
	} else {
		return(Default);
	}
}
