/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#define	PRODUCT_NAME		"Hula ModWeb Template Compiler"
#define	PRODUCT_VERSION	"$Revision: 1.2 $"

#define	BUFSIZE	1023

#include <config.h>
#include <xpl.h>
#include <hulautil.h>

#include "mwcomp.h"
#include "mwtempl.h"

#define	ALIGNMENT_BYTES					4
#define	CALC_ALIGNMENT_SIZE(Size)		((ALIGNMENT_BYTES - ((Size) % ALIGNMENT_BYTES)) % ALIGNMENT_BYTES);
#define	MAX_NESTING							16
#define	MAX_LANG_FILES						32
#define	STRING_ALLOC_SIZE					500
#define	LANGUAGE_ALLOC_SIZE				50
#define	NAME_ALLOC_SIZE					50
#define	PAGE_ALLOC_SIZE					50
#define	IMAGE_ALLOC_SIZE					50
#define	TEMPLATE_ALLOC_SIZE				50

TemplateStruct Template;
int				Verbose						= 0;
BOOL				ForceOverwrite				= FALSE;
BOOL				Warnings						= TRUE;
BOOL				NeedCompilerSupport		= FALSE;
BOOL				ShowUnused					= FALSE;
unsigned long	GlobalUID					= 0x7fffffff;
unsigned char	TemplateFilename[XPL_MAX_PATH]	= "template";
unsigned char	Description[XPL_MAX_PATH]	= "";
unsigned char	Workfile[XPL_MAX_PATH]		= "";
unsigned char	Targetfile[XPL_MAX_PATH]	= "";
unsigned char	TargetDir[XPL_MAX_PATH]		= "/hula/templates";
unsigned char	BaseDir[XPL_MAX_PATH]		= "/hula/templates";
unsigned char	LogFilename[XPL_MAX_PATH]	= "mwcomp.log";
FILE				*VerboseFile				= NULL;
#define			DEFAULT_INCLUDE_DIR		"/hula/include"
unsigned char	GlobalStrings[XPL_MAX_PATH]	= "global.txt";

#define	ChopNL(String)		{ char *pTr; pTr=strrchr((String), 0x0a);	if (pTr)	*pTr='\0'; pTr=strrchr((String), 0x0d); if (pTr) *pTr='\0'; }

#define	STATE_TOP					0
#define	STATE_PREFERENCES			1
#define	STATE_APPEARANCE			2
#define	STATE_IMAGES				3
#define	STATE_STRINGS				4
#define	STATE_PLUGINS				5
#define	STATE_LANGUAGE				6
#define	STATE_TEMPLATE				7
#define	STATE_DESIGNATE			8
#define	STATE_TOKENS				9
#define	STATE_ASSOCIATE			10
#define	STATE_NAMED_TEMPLATES	11
#define	STATE_COLORS				12
#define	STATE_GENERAL				13
#define	STATE_INCLUDE				14
#define	STATE_COMMONSTRINGS		15
#define	STATE_COMMONIMAGES		16
#define	STATE_COMMONSTRINGFILES	17
#define	STATE_DEFINES				18
#define	STATE_STATICOBJECTS			19

/* Template types */
typedef struct {
	unsigned char	Extension[4];
} TemplateExtensionStruct;

TemplateExtensionStruct TemplateExtension[] = {"ctp", "wat"};
unsigned long				TemplateType			= 0;


unsigned long		IncludeDirCount	= 0;
unsigned char		**IncludeDirs;

unsigned long		DefineCount			= 0;
unsigned char		**Defines;

/* Template globals */

#define	TOKEN_ALLOC_BLOCK				100
typedef struct {
	unsigned char	*Keyword;
	unsigned long	NumOfArguments;
	unsigned long	ArgType[5];
	unsigned	long	TokenID;
} TokenListStruct;

TokenListStruct	*Tokens				= NULL;
unsigned long		TokenCount			= 0;
unsigned long		TokenAllocated		= 0;

typedef struct {
	unsigned long	Number;
	unsigned char	*Name;
} TokenDefinitionStruct;

TokenDefinitionStruct	*TokenDefs	= NULL;
unsigned long		TokenDefCount		= 0;
unsigned long		TokenDefAllocated	= 0;

#define	ASSOCIATE_ALLOC_BLOCK				100
typedef struct {
	unsigned long	Number;
	unsigned char	*Name;
} AssociateDefinitionStruct;

AssociateDefinitionStruct	*AssociateDefs		= NULL;
unsigned long		AssociateDefCount			= 0;
unsigned long		AssociateDefAllocated	= 0;

typedef struct {
	unsigned long	FileCount;
	unsigned long	Number;
	unsigned char	*Name;
	unsigned char	*File[MAX_LANG_FILES];
} LanguageTableStruct;

unsigned long		LanguageAllocated	= 0;
LanguageTableStruct	*Languages		= NULL;

LanguageTableStruct	CommonStringInfo	= {0, 0, "", ""};

typedef struct {
    unsigned long Number;
    unsigned char *Name;
    unsigned char *DefaultValue;
    unsigned long UsedCount;
} StringTableStruct;

unsigned long		StringAllocated	= 0;
StringTableStruct	*Strings				= NULL;
unsigned char		***StringData		= NULL;

unsigned long		LocalizedStringCount			= 0;
unsigned long		CommonStringCount			= 0;
unsigned long		CommonStringAllocated	= 0;
StringTableStruct	*CommonStrings				= NULL;
unsigned char		**CommonStringData		= NULL;

typedef struct {
	unsigned long	Number;
	unsigned char	*Name;
	unsigned char	*File;
	unsigned long	UsedCount;
} ImageTableStruct;

unsigned long		ImageAllocated	= 0;
ImageTableStruct	*Images			= NULL;
unsigned char		***ImageData	= NULL;
unsigned long		**ImageLength	= NULL;

unsigned long		LocalizedImageCount		= 0;
unsigned long		CommonImageCount		= 0;
unsigned long		CommonImageAllocated	= 0;
ImageTableStruct	*CommonImages			= NULL;
unsigned char		**CommonImageData		= NULL;
unsigned long		*CommonImageLength	= NULL;

typedef struct {
	unsigned long	Number;
	unsigned char	*Name;
	unsigned char	*File;
	unsigned char	*MIMEType;
	unsigned long	UsedCount;
	unsigned char	**Compiled;
} DynamicObject;

typedef struct {
	unsigned char	*Name;
	unsigned char	*ObjectName;
	unsigned long	ObjectNumber;
	unsigned long	UsedCount;
} NamedDynamicObject;

unsigned long NamedDynamicObjectsAllocated = 0;
NamedDynamicObject *NamedDynamicObjects	= NULL;

unsigned long DynamicObjectsAllocated = 0;
DynamicObject *DynamicObjects = NULL;
unsigned long *DynamicObjectOffsets = NULL;
unsigned long DynamicObjectFixupFileOffset = 0;

unsigned long *StaticObjectOffsets = NULL;
unsigned long StaticObjectFixupFileOffset = 0;

unsigned char *InitialHTMLPage = NULL;
unsigned char *InitialWMLPage = NULL;
unsigned char *TimeOutPage = NULL;
unsigned char *ResumePage = NULL;

typedef struct {
	unsigned long	Number;
	unsigned char	*Name;
	unsigned char	*File;
	unsigned char	*MIMEType;
	unsigned long	UsedCount;
	unsigned char	*Data;
	unsigned long	Size;
} StaticObject;

unsigned long		StaticObjectAllocated = 0;
StaticObject	*StaticObjects		= NULL;

/* Prototypes */
unsigned long		ObjectNameToID(unsigned char *Name);
unsigned long		StaticObjectNameToID(unsigned char *Name);
unsigned long		StringNameToID(unsigned char *Name);

int
LogPrintf(const char *format, ...)
{
	va_list	arglist;

	va_start(arglist, format);
	vprintf(format, arglist);
	va_end(arglist);

	if (VerboseFile) {
		va_start(arglist, format);
		vfprintf(VerboseFile, format, arglist);
		va_end(arglist);
	}

	return(0);
}

BOOL
AddTokenDef(unsigned char *Filename, unsigned long SourceLine, unsigned char *TokenName, unsigned long TokenID)
{
	unsigned long	i;

	if (!TokenName || TokenName[0]=='\0') {
		return(FALSE);
	}

	for (i=0; i<TokenDefCount; i++) {
		if (TokenDefs[i].Number==TokenID) {
			LogPrintf("%s(%d): TokenID %d already defined\n", Filename, SourceLine, TokenID);
			return(FALSE);
		}
		if (QuickCmp(TokenDefs[i].Name, TokenName)) {
			LogPrintf("%s(%d): Token %s already defined\n", Filename, SourceLine, TokenName);
			return(FALSE);
		}
	}

	if ((TokenDefCount+1)>TokenDefAllocated) {
		TokenDefs=realloc(TokenDefs, (TokenDefAllocated+TOKEN_ALLOC_BLOCK)*sizeof(TokenDefinitionStruct));
		if (!TokenDefs) {
			LogPrintf("%s(%d): AddTokenDef(): Out of memory\n", Filename, SourceLine);
			return(FALSE);
		}
		TokenDefAllocated+=TOKEN_ALLOC_BLOCK;
	}
	TokenDefs[TokenDefCount].Name=strdup(TokenName);
	TokenDefs[TokenDefCount].Number=TokenID;

	if (Verbose) {
		LogPrintf("%s(%d): Add token definiton %s = %d\n", Filename, SourceLine, TokenDefs[TokenDefCount].Name, TokenDefs[TokenDefCount].Number);
	}
	TokenDefCount++;
	return(TRUE);
}

BOOL
AddAssociateDef(unsigned char *Filename, unsigned long SourceLine, unsigned char *AssociateName, unsigned long AssociateID)
{
	unsigned long	i;

	if (!AssociateName || AssociateName[0]=='\0') {
		return(FALSE);
	}

	for (i=0; i<AssociateDefCount; i++) {
#if 0
		if (AssociateDefs[i].Number==AssociateID) {
			LogPrintf("%s(%d): AssociateID %d already defined\n", Filename, SourceLine, AssociateID);
			return(FALSE);
		}
#endif
		if (QuickCmp(AssociateDefs[i].Name, AssociateName)) {
			LogPrintf("%s(%d): Associate %s already defined\n", Filename, SourceLine, AssociateName);
			return(FALSE);
		}
	}

	if ((AssociateDefCount+1)>AssociateDefAllocated) {
		AssociateDefs=realloc(AssociateDefs, (AssociateDefAllocated+ASSOCIATE_ALLOC_BLOCK)*sizeof(AssociateDefinitionStruct));
		if (!AssociateDefs) {
			LogPrintf("%s(%d): AddAssociateDef(): Out of memory\n", Filename, SourceLine);
			return(FALSE);
		}
		AssociateDefAllocated+=ASSOCIATE_ALLOC_BLOCK;
	}
	AssociateDefs[AssociateDefCount].Name=strdup(AssociateName);
	AssociateDefs[AssociateDefCount].Number=AssociateID;

	if (Verbose) {
		LogPrintf("%s(%d): Add associate definiton %s = %d\n", Filename, SourceLine, AssociateDefs[AssociateDefCount].Name, AssociateDefs[AssociateDefCount].Number);
	}
	AssociateDefCount++;
	return(TRUE);
}


BOOL
AddToken(unsigned char *Filename, unsigned long SourceLine, TokenListStruct *Token)
{
	if (!Token) {
		return(FALSE);
	}

	if ((TokenCount+1)>TokenAllocated) {
		Tokens=realloc(Tokens, (TokenAllocated+TOKEN_ALLOC_BLOCK)*sizeof(TokenListStruct));
		if (!Tokens) {
			LogPrintf("%s(%d): AddToken(): Out of memory\n", Filename, SourceLine);
			return(FALSE);
		}
		TokenAllocated+=TOKEN_ALLOC_BLOCK;
	}
	Tokens[TokenCount]=*Token;
	Tokens[TokenCount].Keyword=strdup(Token->Keyword);

	if (Verbose) {
		LogPrintf("%s(%d): Add token %s\n", Filename, SourceLine, Tokens[TokenCount].Keyword);
	}
	TokenCount++;
	return(TRUE);
}

unsigned long
TokenNameToID(unsigned char *Tokenname)
{
	unsigned long	i;

	for (i=0; i<TokenDefCount; i++) {
		if (QuickCmp(Tokenname, TokenDefs[i].Name)) {
			return(TokenDefs[i].Number);
		}
	}

	return(0xffffffff);
}

BOOL
IsDefined(unsigned char *Name)
{
	unsigned long	i;

	for (i=0; i<DefineCount; i++) {
		if (QuickCmp(Name, Defines[i])) {
			return(TRUE);
		}
	}

	return(FALSE);
}

BOOL
AddDefine(unsigned char *Filename, unsigned long SourceLine, unsigned char *Name)
{
	DefineCount++;
	Defines=realloc(Defines, sizeof(unsigned char *)*DefineCount);
	if (!Defines) {
		LogPrintf("%s(%d): Not enough memory to define %s\n", Filename, SourceLine, Name);
		DefineCount=0;
		return(FALSE);
	}
	Defines[DefineCount-1]=strdup(Name);
	return(TRUE);
}

BOOL
AddIncludeDir(unsigned char *Filename, unsigned long SourceLine, unsigned char *IncludeDir)
{
	if (access(IncludeDir, 0)!=0) {
		LogPrintf("%s(%d): Include directory %s does not exist\n", Filename, SourceLine, IncludeDir);
		return(FALSE);
	}
	IncludeDirCount++;
	IncludeDirs=realloc(IncludeDirs, sizeof(unsigned char *)*IncludeDirCount);
	if (!IncludeDirs) {
		LogPrintf("%s(%d): Not enough memory to add include directory %s\n", Filename, SourceLine, IncludeDir);
		IncludeDirCount=0;
		return(FALSE);
	}
	IncludeDirs[IncludeDirCount-1]=strdup(IncludeDir);
	return(TRUE);
}

void 
GetIncludePath(const char *Line, char *buffer)
{
    const char	*ptr2;
    char *ptr;
    
    /* We need to read the string names from the include file */
    ptr2=Line+9;
    while (isspace(*ptr2)) {
	ptr2++;
    }

    if (*ptr2 == '"' || *ptr2 == '<') {
	ptr2++;
    }
    
    strcpy (buffer, ptr2);
    
    ptr=strchr(buffer, '"');
    if (!ptr) {
	ptr = strchr(buffer, '>');
    }
    if (ptr) {
	*ptr = '\0';
    }
}

FILE
*IncludeFOpen(unsigned char *File, unsigned char *Rights)
{
	unsigned char	Name[XPL_MAX_PATH+1];
	FILE				*RetVal;
	unsigned long	i;

	/* First, try the standard path */
	sprintf(Name, "%s/%s/%s", BaseDir, TemplateFilename, File);
	RetVal=fopen(Name, Rights);
	if (RetVal) {
		return(RetVal);
	}

	/* Try as full path */
	RetVal=fopen(File, Rights);
	if (RetVal) {
		return(RetVal);
	}

	/* Now try our include directories */
	for (i=0; (i<IncludeDirCount) && !RetVal; i++) {
		sprintf(Name, "%s/%s", IncludeDirs[i], File);
		RetVal=fopen(Name, Rights);
		if (RetVal) {
			return(RetVal);
		}
	}
	return(NULL);
}


#define	SkipToStart(Start)		while (*(Start) && isspace(*(Start))) (Start)++;
BOOL
AddTokenFile(unsigned long SourceLine, unsigned char *Filename)
{
	int					i;
	FILE					*TokenFile;
	unsigned char		Line[1024];
	unsigned char		*ptr, *Start;
	unsigned long		TokenLine=0;
	TokenListStruct	Token;

	TokenFile=IncludeFOpen(Filename, "rb");
	if (!TokenFile) {
		LogPrintf("AddTokenFile(): Couldn't open token file %s\n", Filename);
		return(FALSE);
	}

	while (!feof(TokenFile) && !ferror(TokenFile)) {
		if (fgets(Line, sizeof(Line), TokenFile)!=NULL) {
			ChopNL(Line);
			TokenLine++;
			if (Verbose>2) {
				LogPrintf("%s(%d): %s\n", Filename, TokenLine, Line);
			}
			if (QuickNCmp(Line, "//", 2) || isspace(Line[0]) || Line[0]==';' || Line[0]=='\0' || QuickNCmp(Line, "/*", 2) || QuickNCmp(Line, "*/", 2)) {
				continue;
			}

			if (QuickNCmp(Line, "#define", 7)) {
				unsigned char	*Val;
				/* Add a value define */
				Start=Line+7;
				SkipToStart(Start);

				ptr=Start;
				while (!isspace(*ptr) && *ptr!='\0') {
					ptr++;
				}
				*ptr='\0';

				/* Start now points to the text definition; now get the value */
				Val=ptr+1;
				SkipToStart(Val);
				if (Verbose>1) {
					LogPrintf("%s(%d): TokenDefine %s=%d\n", Filename, TokenLine, Start, atol(Val));
				}
				AddTokenDef(Filename, TokenLine, Start, atol(Val));
				continue;
			}

			/*
				If we're here we assume we've got a token definition
				A token line must contain the following values in order:
				TokenName, NumberOfArguments, Arg[0]Type, Arg[1]Type, Arg[2]Type, Arg[3]Type, Arg[4]Type, TokenID

				The commas are required; whitespace is allowed; comments after tokenID are allowed

			*/

			/* First, do a syntax check and count commas */
			ptr=strchr(Line, ',');
			for (i=0; (i<7) && ptr; i++) {
				ptr=strchr(ptr+1, ',');
			}

			if (i!=7) {
				LogPrintf("%s(%d): Not enough arguments\n", Filename, TokenLine);
				continue;
			}

			/* We've got the commas, now start reading the stuff */
			/* Keyword */
			ptr=strchr(Line, ',');
			Token.Keyword=Line;
			*ptr='\0';

			if (Verbose>1) LogPrintf("%s(%d): Found keyword %s\n", Filename, TokenLine, Token.Keyword);

			/* Number of arguments */
			Start=ptr+1;
			SkipToStart(Start);
			ptr=strchr(Start, ',');
			Token.NumOfArguments=atol(Start);

			if (Verbose>1) LogPrintf("%s(%d): Found NumOfArguments %d\n", Filename, TokenLine, Token.NumOfArguments);
			/* Arguments */
			for (i=0; i<5; i++) {
				Start=ptr+1;
				SkipToStart(Start);
				ptr=strchr(Start, ',');		/* We're guaranteed a comma, we checked earlier */
				*ptr='\0';
				if (isdigit(*Start)) {
					Token.ArgType[i]=atol(Start);
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_UNKNOWN")) {
					Token.ArgType[i]=ARG_UNKNOWN;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_UNKNOWN %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_TEXT")) {
					Token.ArgType[i]=ARG_TEXT;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_TEXT %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_ASSOC")) {
					Token.ArgType[i]=ARG_ASSOC;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_ASSOC %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_NUM")) {
					Token.ArgType[i]=ARG_NUM;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_NUM %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_STRING")) {
					Token.ArgType[i]=ARG_STRING;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_STRING %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_URL")) {
					Token.ArgType[i]=ARG_URL;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_URL %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_PAGE")) {
					Token.ArgType[i]=ARG_PAGE;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_PAGE %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_PAGEID")) {
					Token.ArgType[i]=ARG_PAGEID;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_PAGEID %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_STATIC")) {
					Token.ArgType[i]=ARG_STATIC;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_STATIC %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else if (QuickCmp(Start, "ARG_IMAGE")) {
					Token.ArgType[i]=ARG_IMAGE;
					if (Verbose>1) LogPrintf("%s(%d): Argument[%d]: ARG_IMAGE %d\n", Filename, TokenLine, i, Token.ArgType[i]);
				} else {
					LogPrintf("%s(%d): Argument[%d]: Unknown argument type %s\n", Filename, TokenLine, i, Start);
					fclose(TokenFile);
					return(FALSE);
				}
			}
			/* TokenID */
			Start=ptr+1;
			SkipToStart(Start);

			if (isdigit(*Start)) {
				Token.TokenID=atol(Start);
				if (Verbose>1) LogPrintf("%s(%d): Token ID %d\n", Filename, TokenLine, Token.TokenID);
			} else {
				ptr=Start;
				while (!isspace(*ptr) && *ptr!='\0') {
					ptr++;
				}
				*ptr='\0';
				Token.TokenID=TokenNameToID(Start);
				if (Verbose>1) LogPrintf("%s(%d): Token ID %s=%d\n", Filename, TokenLine, Token.Keyword, Token.TokenID);
			}

			if (Token.TokenID!=0xffffffff) {
				if (!AddToken(Filename, TokenLine, &Token)) {
					continue;		// FIXME ? Should we abort? We will die if we need the failed definition later...
				}
			} else {
				LogPrintf("%s(%d): Did not recognize Token %s\n", Filename, TokenLine, Start);
				fclose(TokenFile);
				return(FALSE);
			}
		}
	}

	fclose(TokenFile);
	return(TRUE);
}


BOOL
AddAssociateFile(unsigned long SourceLine, unsigned char *Filename)
{
	FILE					*AssociateFile;
	unsigned char		Line[1024];
	unsigned char		*ptr, *Start;
	unsigned long		AssociateLine=0;

	AssociateFile=IncludeFOpen(Filename, "rb");
	if (!AssociateFile) {
		LogPrintf("AddAssociateFile(): Couldn't open associate file %s\n", Filename);
		return(FALSE);
	}

	while (!feof(AssociateFile) && !ferror(AssociateFile)) {
		if (fgets(Line, sizeof(Line), AssociateFile)!=NULL) {
			ChopNL(Line);
			AssociateLine++;
			if (Verbose>2) {
				LogPrintf("%s(%d): %s\n", Filename, AssociateLine, Line);
			}
			if (QuickNCmp(Line, "//", 2) || isspace(Line[0]) || Line[0]==';' || Line[0]=='\0' || QuickNCmp(Line, "/*", 2) || QuickNCmp(Line, "*/", 2)) {
				continue;
			}

			if (QuickNCmp(Line, "#define", 7)) {
				unsigned char	*Val;
				unsigned long	Value;
	
				/* Add a value define */
				Start=Line+7;
				SkipToStart(Start);

				ptr=Start;
				while (!isspace(*ptr) && *ptr!='\0') {
					ptr++;
				}
				*ptr='\0';

				/* Start now points to the text definition; now get the value */
				Val=ptr+1;
				SkipToStart(Val);
				if (Val[0]!='\'') {
					Value=atol(Val);
				} else {
					if (Val[1]!='\\') {
						Value=(unsigned long)Val[1];
					} else {
						switch(Val[2]) {
							case '\'': Value='\''; break;
							case '\\': Value='\\'; break;
							case 'n': Value='\n'; break;
							case 'r': Value='\r'; break;
							case 't': Value='\t'; break;
							case '"': Value='"'; break;
							default: Value=Val[2]; break;
						}
					}
				}
				if (Verbose>1) {
					LogPrintf("%s(%d): AssociateDefine %s=%d\n", Filename, AssociateLine, Start, Value);
				}
				if (QuickNCmp(Start, "AA_", 3)) {
					AddAssociateDef(Filename, AssociateLine, Start+3, Value);
				} else {
					AddAssociateDef(Filename, AssociateLine, Start, Value);
				}
				continue;
			}
		}
	}

	fclose(AssociateFile);
	return(TRUE);
}


/* It's plural because NW has a function call of the same name in singular (AddLanguages instead of AddLanguage) */
BOOL
AddLanguages(unsigned long SourceLine, unsigned char *Name, unsigned long ID, unsigned char *File)
{
	unsigned long	i;
	unsigned char	*ptr, *ptr2;
	unsigned char	*StartPtr;

	for (i=0; i<Template.LanguageCount; i++) {
		if (XplStrCaseCmp(Languages[i].Name, Name)==0) {
			LogPrintf("template.def(%d): Language '%s' already defined\n", SourceLine, Name);
			return(FALSE);
		}
		if (Languages[i].Number==ID) {
			LogPrintf("template.def(%d): Language-ID %d (%s) already defined\n", SourceLine, ID, Name);
			return(FALSE);
		}
	}
	if ((Template.LanguageCount+LANGUAGE_ALLOC_SIZE)>=LanguageAllocated) {
		Languages=realloc(Languages, (LanguageAllocated+LANGUAGE_ALLOC_SIZE)*sizeof(LanguageTableStruct));
		if (!Languages) {
			return(FALSE);
		}
		LanguageAllocated+=LANGUAGE_ALLOC_SIZE;
	}

	StartPtr=File;
	Languages[Template.LanguageCount].FileCount=0;
	do {
		ptr=strchr(StartPtr, ',');
		if (ptr) {
			*ptr='\0';
		}
		while (isspace(*StartPtr)) {
			StartPtr++;
		}
		if (ptr) {
			ptr2=ptr-1;
		} else {
			ptr2=StartPtr+strlen(StartPtr)-1;
		}
		while (isspace(*ptr2)) {
			ptr2--;
		}
		ptr2[1]='\0';
		if (Verbose) {
			LogPrintf("template.def(%d): Language %d (%s) from file %s\n", SourceLine, ID, Name, StartPtr);
		}
		Languages[Template.LanguageCount].File[Languages[Template.LanguageCount].FileCount]=strdup(StartPtr);
		Languages[Template.LanguageCount].FileCount++;
		StartPtr=ptr+1;
	} while (ptr && Languages[Template.LanguageCount].FileCount<MAX_LANG_FILES);

	Languages[Template.LanguageCount].Number=ID;
	Languages[Template.LanguageCount].Name=strdup(Name);

	Template.LanguageCount++;
	return(TRUE);
}


BOOL
AddName(unsigned long SourceLine, unsigned char *Name, unsigned char *TemplateFilename)
{
	unsigned long		i;

	for (i=0; i<Template.NameCount; i++) {
		if (XplStrCaseCmp(NamedDynamicObjects[i].Name, Name)==0) {
			LogPrintf("template.def(%d): Name '%s' already defined\n", SourceLine, Name);
			return(FALSE);
		}
	}
	if ((Template.NameCount+NAME_ALLOC_SIZE)>=NamedDynamicObjectsAllocated) {
		NamedDynamicObjects=realloc(NamedDynamicObjects, (NamedDynamicObjectsAllocated+NAME_ALLOC_SIZE)*sizeof(NamedDynamicObject));
		if (!NamedDynamicObjects) {
			return(FALSE);
		}
		NamedDynamicObjectsAllocated+=NAME_ALLOC_SIZE;
	}
	if (Verbose) {
		LogPrintf("template.def(%d): Added Name %s=Template %s\n", SourceLine, Name, TemplateFilename);
	}
	NamedDynamicObjects[Template.NameCount].ObjectName=strdup(TemplateFilename);
	NamedDynamicObjects[Template.NameCount].Name=strdup(Name);
	Template.NameCount++;
	return(TRUE);
}

BOOL
AddCommonStringFile(unsigned long SourceLine, unsigned char *Name, unsigned char *File)
{
	unsigned char	*ptr, *ptr2;
	unsigned char	*StartPtr;

	if (CommonStringInfo.FileCount>0) {
		return(FALSE);
	}

	StartPtr=File;
	CommonStringInfo.FileCount=0;
	do {
		ptr=strchr(StartPtr, ',');
		if (ptr) {
			*ptr='\0';
		}
		while (isspace(*StartPtr)) {
			StartPtr++;
		}
		if (ptr) {
			ptr2=ptr-1;
		} else {
			ptr2=StartPtr+strlen(StartPtr)-1;
		}
		while (isspace(*ptr2)) {
			ptr2--;
		}
		ptr2[1]='\0';
		if (Verbose) {
			LogPrintf("template.def(%d): Common strings from file %s\n", SourceLine, StartPtr);
		}
		CommonStringInfo.File[CommonStringInfo.FileCount]=strdup(StartPtr);
		CommonStringInfo.FileCount++;
		StartPtr=ptr+1;
	} while (ptr && CommonStringInfo.FileCount<MAX_LANG_FILES);

	CommonStringInfo.Name=strdup(Name);

	return(TRUE);
}


BOOL
AddCommonString(unsigned char *SourceFile, unsigned long SourceLine, unsigned char *Name, unsigned char *defaultValue, long ID)
{
	unsigned long	i;

	for (i=0; i<CommonStringCount; i++) {
		if (XplStrCaseCmp(CommonStrings[i].Name, Name)==0) {
			LogPrintf("%s(%d): CommonString '%s' already defined\n", SourceFile, SourceLine, Name);
			return(FALSE);
		}
		if (CommonStrings[i].Number==(unsigned long)ID) {
			LogPrintf("%s(%d): CommonString-ID %d (%s) already defined\n", SourceFile, SourceLine, ID, Name);
			return(FALSE);
		}
	}
	if ((CommonStringCount+STRING_ALLOC_SIZE)>=CommonStringAllocated) {
		CommonStrings=realloc(CommonStrings, (CommonStringAllocated+STRING_ALLOC_SIZE)*sizeof(StringTableStruct));
		if (!CommonStrings) {
			return(FALSE);
		}
		CommonStringAllocated+=STRING_ALLOC_SIZE;
	}
	if (ID==-1) {
		CommonStrings[CommonStringCount].Number=CommonStringCount;
	} else {
		CommonStrings[CommonStringCount].Number=ID;
	}
	if (Verbose) {
		LogPrintf("%s(%d): String %d (%s)\n", SourceFile, SourceLine, CommonStrings[CommonStringCount].Number, Name);
	}
	CommonStrings[CommonStringCount].Name=strdup(Name);
	CommonStrings[CommonStringCount].DefaultValue=strdup(defaultValue);
	CommonStrings[CommonStringCount].UsedCount=0;
	CommonStringCount++;
	return(TRUE);
}


BOOL
AddString(unsigned char *SourceFile, unsigned long SourceLine, unsigned char *Name, unsigned char *DefaultValue, long ID)
{
	unsigned long	i;

	for (i=0; i<LocalizedStringCount; i++) {
		if (XplStrCaseCmp(Strings[i].Name, Name)==0) {
			LogPrintf("%s(%d): String '%s' already defined\n", SourceFile, SourceLine, Name);
			return(FALSE);
		}
		if (Strings[i].Number==(unsigned long)ID) {
			LogPrintf("%s(%d): String-ID %d (%s) already defined\n", SourceFile, SourceLine, ID, Name);
			return(FALSE);
		}
	}

	if ((LocalizedStringCount + 1) >= StringAllocated) {
		Strings=realloc(Strings, (StringAllocated+STRING_ALLOC_SIZE)*sizeof(StringTableStruct));
		if (!Strings) {
			return(FALSE);
		}
		StringAllocated+=STRING_ALLOC_SIZE;
	}
	if (ID==-1) {
		Strings[LocalizedStringCount].Number=LocalizedStringCount;
	} else {
		Strings[LocalizedStringCount].Number=ID;
	}

	if (Verbose) {
		LogPrintf("%s(%d): String %d (%s)\n", SourceFile, SourceLine, Strings[LocalizedStringCount].Number, Name);
	}
	Strings[LocalizedStringCount].Name=strdup(Name);
	Strings[LocalizedStringCount].DefaultValue = strdup(DefaultValue);
	Strings[LocalizedStringCount].UsedCount=0;
	LocalizedStringCount++;
	return(TRUE);
}


BOOL
AddCommonImage(unsigned long SourceLine, unsigned char *Name, unsigned char *File)
{
	unsigned long	i;

	for (i=0; i<CommonImageCount; i++) {
		if (XplStrCaseCmp(CommonImages[i].Name, Name)==0) {
			LogPrintf("template.def(%d): CommonImage-name '%s' already defined\n", SourceLine, Name);
			return(FALSE);
		}
	}
	if ((CommonImageCount+IMAGE_ALLOC_SIZE)>=CommonImageAllocated) {
		CommonImages=realloc(CommonImages, (CommonImageAllocated+IMAGE_ALLOC_SIZE)*sizeof(ImageTableStruct));
		if (!CommonImages) {
			return(FALSE);
		}
		CommonImageAllocated+=IMAGE_ALLOC_SIZE;
	}
	if (Verbose) {
		LogPrintf("template.def(%d): Image %d (%s) from file %s\n", SourceLine, CommonImageCount, Name, File);
	}
	CommonImages[CommonImageCount].Number=CommonImageCount;
	CommonImages[CommonImageCount].Name=strdup(Name);
	CommonImages[CommonImageCount].File=strdup(File);
	CommonImages[CommonImageCount].UsedCount=0;
	CommonImageCount++;
	return(TRUE);
}


BOOL
AddImage(unsigned long SourceLine, unsigned char *Name, unsigned char *File)
{
	unsigned long	i;

	for (i=0; i<LocalizedImageCount; i++) {
		if (XplStrCaseCmp(Images[i].Name, Name)==0) {
			LogPrintf("template.def(%d): Image-name '%s' already defined\n", SourceLine, Name);
			return(FALSE);
		}
	}
	if ((LocalizedImageCount+IMAGE_ALLOC_SIZE)>=ImageAllocated) {
		Images=realloc(Images, (ImageAllocated+IMAGE_ALLOC_SIZE)*sizeof(ImageTableStruct));
		if (!Images) {
			return(FALSE);
		}
		ImageAllocated+=IMAGE_ALLOC_SIZE;
	}
	if (Verbose) {
		LogPrintf("template.def(%d): Image %d (%s) from file %s\n", SourceLine, LocalizedImageCount, Name, File);
	}
	Images[LocalizedImageCount].Number=LocalizedImageCount;
	Images[LocalizedImageCount].Name=strdup(Name);
	Images[LocalizedImageCount].File=strdup(File);
	Images[LocalizedImageCount].UsedCount=0;
	LocalizedImageCount++;
	return(TRUE);
}


BOOL
AddDynamicObjectFile(unsigned long SourceLine, unsigned char *Name, long ID, unsigned char *File, unsigned char *MIMEType)
{
	unsigned long	i;

	for (i=0; i<Template.TemplateCount; i++) {
		if (XplStrCaseCmp(DynamicObjects[i].Name, Name)==0) {
			LogPrintf("template.def(%d): Template-name '%s' already defined\n", SourceLine, Name);
			return(FALSE);
		}
		if (DynamicObjects[i].Number==(unsigned long)ID) {
			LogPrintf("template.def(%d): Template-ID %d (%s) already defined\n", SourceLine, ID, Name);
			return(FALSE);
		}
	}
	if ((Template.TemplateCount+TEMPLATE_ALLOC_SIZE)>=DynamicObjectsAllocated) {
		DynamicObjects=realloc(DynamicObjects, (DynamicObjectsAllocated+TEMPLATE_ALLOC_SIZE)*sizeof(DynamicObject));
		if (!DynamicObjects) {
			return(FALSE);
		}
		DynamicObjectsAllocated+=TEMPLATE_ALLOC_SIZE;
	}

	DynamicObjects[Template.TemplateCount].Number=ID;
	if (Verbose) {
		LogPrintf("template(%d): Template %d (%s) from file %s\n", SourceLine, DynamicObjects[Template.TemplateCount].Number, Name, File);
	}
	DynamicObjects[Template.TemplateCount].Name=strdup(Name);
	DynamicObjects[Template.TemplateCount].File=strdup(File);
	DynamicObjects[Template.TemplateCount].MIMEType=strdup(MIMEType);
	DynamicObjects[Template.TemplateCount].UsedCount=0;
	DynamicObjects[Template.TemplateCount].Compiled=NULL;
	Template.TemplateCount++;
	return(TRUE);
}


BOOL
AddStaticObject(unsigned long SourceLine, unsigned char *Name, long ID, unsigned char *File, unsigned char *MIMEType)
{
	unsigned long	i;

	for (i=0; i<Template.CommonStaticObjectCount; i++) {
		if (XplStrCaseCmp(StaticObjects[i].Name, Name)==0) {
			LogPrintf("template.def(%d): Static page-name '%s' already defined\n", SourceLine, Name);
			return(FALSE);
		}
		if (StaticObjects[i].Number==(unsigned long)ID) {
			LogPrintf("template.def(%d): Static page-ID %d (%s) already defined\n", SourceLine, ID, Name);
			return(FALSE);
		}
	}
	if ((Template.CommonStaticObjectCount+PAGE_ALLOC_SIZE)>=StaticObjectAllocated) {
		StaticObjects=realloc(StaticObjects, (StaticObjectAllocated+PAGE_ALLOC_SIZE)*sizeof(StaticObject));
		if (!StaticObjects) {
			return(FALSE);
		}
		StaticObjectAllocated+=PAGE_ALLOC_SIZE;
	}

	StaticObjects[Template.CommonStaticObjectCount].Number=ID;
	if (Verbose) {
		LogPrintf("template(%d): Static page %d (%s) from file %s\n", SourceLine, StaticObjects[Template.CommonStaticObjectCount].Number, Name, File);
	}
	StaticObjects[Template.CommonStaticObjectCount].Name=strdup(Name);
	StaticObjects[Template.CommonStaticObjectCount].File=strdup(File);
	StaticObjects[Template.CommonStaticObjectCount].MIMEType=strdup(MIMEType);
	StaticObjects[Template.CommonStaticObjectCount].UsedCount=0;
	StaticObjects[Template.CommonStaticObjectCount].Data=NULL;
	Template.CommonStaticObjectCount++;
	return(TRUE);
}

BOOL
AddStringDef(char *line, const char *file, long lineNum, int state)
{
    char *ptr;
    char *defaultValue;
    int id;
    BOOL ret;

    ptr = strchr(line, '|');
    if (!ptr) {
	LogPrintf("%s(%d): No default value specified for string '%s'\n", file, lineNum, line);
	return FALSE;
    }
    
    *ptr++ = '\0';
    defaultValue = ptr;

    ptr = strchr(line, '=');
    
    if (ptr) {
	*ptr = 0;
	id = atol(ptr + 1);
    } else {
	id = -1;
    }
    
    if (state == STATE_STRINGS) {
	return AddString(file, lineNum, line, defaultValue, id);
    } else {
	return AddCommonString(file, lineNum, line, defaultValue, id);
    }
}

BOOL
ParseDefinition(void)
{
	FILE				*DefinitionFile;
	unsigned char	Line[1024];
	int				State=STATE_TOP;
	unsigned char	*ptr;
	unsigned long	SourceLine=0;

	if (Verbose) {
		LogPrintf("Parsing template definition file\n");
	} else {
		LogPrintf("Phase I: Reading & interpreting template\n");
	}

	sprintf(Workfile, "%s/%s/template.def", BaseDir, TemplateFilename);
	DefinitionFile=fopen(Workfile, "rb");
	if (!DefinitionFile) {
		LogPrintf("\rCouldn't open template definition %s (%d)\n", Workfile, errno);
		return(FALSE);
	}

	while (!feof(DefinitionFile) && !ferror(DefinitionFile)) {
		if (fgets(Line, sizeof(Line), DefinitionFile)!=NULL) {
			SourceLine++;

			ChopNL(Line);

			if (Line[0]=='[') {
				State=STATE_TOP;
			} else if (Line[0]=='\0' || isspace(Line[0]) || (Line[0]=='#') || (Line[0]==';')) {
				if ((Line[0]!='#') || (!QuickNCmp(Line, "#include ", 9))) {
					continue;
				}
			}

			switch(State) {
				case STATE_TOP: {
					if (XplStrCaseCmp(Line, "[Preferences]")==0) {
						State=STATE_PREFERENCES;
						continue;
					}
					if (XplStrCaseCmp(Line, "[Images]")==0) {
						State=STATE_IMAGES;
						continue;
					}
					if (XplStrCaseCmp(Line, "[Includes]")==0) {
						State=STATE_INCLUDE;
						continue;
					}
					if (XplStrCaseCmp(Line, "[Strings]")==0) {
						State=STATE_STRINGS;
						continue;
					}
					if (XplStrCaseCmp(Line, "[Languages]")==0) {
						State=STATE_LANGUAGE;
						continue;
					}
					if (XplStrCaseCmp(Line, "[TemplateFiles]")==0) {
						State=STATE_TEMPLATE;
						continue;
					}
					if (XplStrCaseCmp(Line, "[NamedTemplates]")==0) {
						State=STATE_NAMED_TEMPLATES;
						continue;
					}
					if (XplStrCaseCmp(Line, "[TemplateDesignations]")==0) {
						State=STATE_DESIGNATE;
						continue;
					}
					if (XplStrCaseCmp(Line, "[TokenDefinitions]")==0) {
						State=STATE_TOKENS;
						continue;
					}
					if (XplStrCaseCmp(Line, "[AssociativeArrays]")==0) {
						State=STATE_ASSOCIATE;
						continue;
					}
					if (XplStrCaseCmp(Line, "[Colors]")==0) {
						State=STATE_COLORS;
						continue;
					}
					if (XplStrCaseCmp(Line, "[General]")==0) {
						State=STATE_GENERAL;
						continue;
					}
					if (XplStrCaseCmp(Line, "[CommonStrings]")==0) {
						State=STATE_COMMONSTRINGS;
						continue;
					}
					if (XplStrCaseCmp(Line, "[CommonStringFiles]")==0) {
						State=STATE_COMMONSTRINGFILES;
						continue;
					}
					if (XplStrCaseCmp(Line, "[CommonImages]")==0) {
						State=STATE_COMMONIMAGES;
						continue;
					}
					if (XplStrCaseCmp(Line, "[Defines]")==0) {
						State=STATE_DEFINES;
						continue;
					}
					if (XplStrCaseCmp(Line, "[StaticObjects]")==0) {
						State=STATE_STATICOBJECTS;
						continue;
					}
					break;
				}

				case STATE_COMMONIMAGES: {
					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					*ptr='\0';
					if (!AddCommonImage(SourceLine, Line, ptr+1)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}

				case STATE_IMAGES: {
					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					*ptr='\0';
					if (!AddImage(SourceLine, Line, ptr+1)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}

				case STATE_NAMED_TEMPLATES: {
					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					*ptr='\0';

					if (!AddName(SourceLine, Line, ptr+1)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}

				case STATE_COMMONSTRINGS:
				case STATE_STRINGS: {
				    if (QuickNCmp(Line, "#include ", 9)) {
					FILE *IncludeFile;
					unsigned long	IncludeLine=1;
					unsigned char	IncludePath[XPL_MAX_PATH];
					
					GetIncludePath(Line, IncludePath);
					IncludeFile = IncludeFOpen (IncludePath, "rb");
					
					if (!IncludeFile) {
					    LogPrintf("template.def(%d): Could not open file '%s'\n", SourceLine, Line);
					    fclose(DefinitionFile);
					    return(FALSE);
					}
					do {
					    if (fgets(Line, sizeof(Line), IncludeFile)!=NULL) {
						ChopNL(Line);
						if (Line[0]!='\0') {
						    if (!AddStringDef(Line, IncludePath, IncludeLine, State)) {
							fclose(DefinitionFile);
							return FALSE;
						    }
						}
						IncludeLine++;
					    }
					    IncludeLine++;
					} while (!feof(IncludeFile) && !ferror(IncludeFile));
					fclose(IncludeFile);
				    } else {
					if (!AddStringDef(Line, Workfile, SourceLine, State)) {
					    fclose(DefinitionFile);
					    return(FALSE);
					}
				    }
				    break;
				}

				case STATE_LANGUAGE: {
					unsigned char	*File;

					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					} else {
						*ptr='\0';
						File=ptr+1;
						ptr=strchr(Line, '[');
						if (!ptr) {
							*(File-1)='=';
							LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
							fclose(DefinitionFile);
							return(FALSE);
						}
						*ptr='\0';
						if (!AddLanguages(SourceLine, Line, atol(ptr+1), File)) {
							fclose(DefinitionFile);
							return(FALSE);
						}
					}
					break;
				}

				case STATE_COMMONSTRINGFILES: {
					unsigned char	*File;

					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					} else {
						*ptr='\0';
						File=ptr+1;
						ptr=strchr(Line, '[');
						if (!ptr) {
							if (!AddCommonStringFile(SourceLine, Line, File)) {
								*(File-1)='=';
								LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
								fclose(DefinitionFile);
								return(FALSE);
							}
						} else {
							*ptr='\0';
							if (!AddCommonStringFile(SourceLine, Line, File)) {
								fclose(DefinitionFile);
								return(FALSE);
							}
						}
					}
					break;
				}

							

				case STATE_STATICOBJECTS: {
					unsigned char	*File;
					unsigned char	*ptr2;

					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					} else {
						*ptr='\0';
						File=ptr+1;
						ptr=strchr(Line, '[');
						if (!ptr) {
							if (Verbose) {
								LogPrintf("Using ID %lu for template file %s\n", GlobalUID, File);
							}
							if (!AddStaticObject(SourceLine, Line, GlobalUID, File, "text/html")) {
								fclose(DefinitionFile);
								return(FALSE);
							}
							GlobalUID--;
						} else {
							*ptr='\0';
							ptr2=strchr(ptr+1, ']');
							if (ptr2) {
							    unsigned char * ptr3;

								*ptr2='\0';
								ptr3 = strchr(ptr + 1, '/');

								if (ptr3) {
								    if (!AddStaticObject(SourceLine, Line, GlobalUID, File, ptr+1)) {
									fclose(DefinitionFile);
									return(FALSE);
								    }
								    GlobalUID--;
								} else {
								    if (!AddStaticObject(SourceLine, Line, atol(ptr + 1), File, "text/html")) {
									fclose(DefinitionFile);
									return(FALSE);
								    }

								}
							} else {
							    fclose(DefinitionFile);
							    return(FALSE);
							}
						}
					}
					break;
				}


				case STATE_TEMPLATE: {
					unsigned char	*File;
					unsigned char	*ptr2;

					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					} else {
						*ptr='\0';
						File=ptr+1;
						ptr=strchr(Line, '[');
						if (!ptr) {
							if (Verbose) {
								LogPrintf("Using ID %lu for template file %s\n", GlobalUID, File);
							}
							if (!AddDynamicObjectFile(SourceLine, Line, GlobalUID, File, "text/html")) {
								fclose(DefinitionFile);
								return(FALSE);
							}
							GlobalUID--;
						} else {
							*ptr='\0';
							ptr2=strchr(ptr+1, ']');
							if (ptr2) {
							    unsigned char *ptr3;
							    *ptr2='\0';
							    ptr3 = strchr(ptr + 1, '/');
							    if (ptr3) {
								if (!AddDynamicObjectFile(SourceLine, Line, GlobalUID, File, ptr+1)) {
								    fclose(DefinitionFile);
								    return(FALSE);
								}
								GlobalUID--;
							    } else {
								if (!AddDynamicObjectFile(SourceLine, Line, atol(ptr+1), File, "text/html")) {
								    fclose(DefinitionFile);
								    return(FALSE);
								}
							    }

							} else {
							    fclose(DefinitionFile);
							    return(FALSE);
							}
						}
					}
					break;
				}

				case STATE_DESIGNATE: {
					if (XplStrNCaseCmp(Line, "Initial=", 8)==0) {
						InitialHTMLPage=strdup(Line+8);
					} else if (XplStrNCaseCmp(Line, "InitialWML=", 11)==0) {
						InitialWMLPage=strdup(Line+11);
					} else if (XplStrNCaseCmp(Line, "Timeout=", 8)==0) {
						TimeOutPage=strdup(Line+8);
					} else if (XplStrNCaseCmp(Line, "Resume=", 7)==0) {
						ResumePage=strdup(Line+7);
					} else {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
					}
					break;
				}

				case STATE_TOKENS: {
					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					if (!AddTokenFile(SourceLine, ptr+1)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}

				case STATE_ASSOCIATE: {
					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					if (!AddAssociateFile(SourceLine, ptr+1)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}

				case STATE_COLORS: {
					unsigned char	*FG;
					unsigned char	*BG;
					unsigned long	FGColor;
					unsigned long	BGColor;

					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					*ptr='\0';
					FG=ptr+1;
					while (isspace(*FG)) {
						FG++;
					}
					ptr=strchr(FG, ',');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					BG=ptr+1;
					while (isspace(*BG)) {
						FG++;
					}
					FGColor=strtol(FG, NULL, 16);
					BGColor=strtol(BG, NULL, 16);
					if (QuickNCmp(Line, "Page", 4)) {
						Template.Colors[0][0]=FGColor;
						Template.Colors[0][1]=BGColor;
					} else if (QuickNCmp(Line, "Border", 6)) {
						Template.Colors[1][0]=FGColor;
						Template.Colors[1][1]=BGColor;
					} else if (QuickNCmp(Line, "Section", 7)) {
						Template.Colors[2][0]=FGColor;
						Template.Colors[2][1]=BGColor;
					} else if (QuickNCmp(Line, "FieldName", 9)) {
						Template.Colors[3][0]=FGColor;
						Template.Colors[3][1]=BGColor;
					} else if (QuickNCmp(Line, "FieldBody", 9)) {
						Template.Colors[4][0]=FGColor;
						Template.Colors[4][1]=BGColor;
					}
					break;
				}

				case STATE_GENERAL: {
					unsigned char	*Value;

					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					*ptr='\0';
					Value=ptr+1;
					while (isspace(*Value)) {
						Value++;
					}
					if (QuickNCmp(Line, "Name", 4)) {
						strncpy(Template.Name, Value, TEMPLATE_NAME_SIZE);
						Template.Name[TEMPLATE_NAME_SIZE] = '\0';
					} else if (QuickNCmp(Line, "Description", 11)) {
						strcpy(Description, Value);
					}
					break;
				}

				case STATE_INCLUDE: {
					ptr=strchr(Line, '=');
					if (!ptr) {
						LogPrintf("template.def(%d): Syntax error '%s'\n", SourceLine, Line);
						fclose(DefinitionFile);
						return(FALSE);
					}
					if (!AddIncludeDir("template.def", SourceLine, ptr+1)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}

				case STATE_DEFINES: {
					if (!AddDefine("template.def", SourceLine, Line)) {
						fclose(DefinitionFile);
						return(FALSE);
					}
					break;
				}
			}
		}
	}
	fclose(DefinitionFile);
	return(TRUE);
}


BOOL
CheckDefinition(void)
{
	unsigned long	i;
	unsigned long	Top=0;
	unsigned long	TopIndex;

	if (Verbose) {
		LogPrintf("\rChecking template definition for errors...");
	} else {
		LogPrintf("\rPhase II: Checking template\n");
	}

	/* First, make sure that the number of strings makes sense */
	for (i=0; i<LocalizedStringCount; i++) {
		if (Strings[i].Number>Top) {
			Top=Strings[i].Number;
			TopIndex=i;
		}
	}

	if (Top>LocalizedStringCount) {
		LogPrintf("\nYou have gaps in the string definitions\nYou defined %d strings, but the highest string number is %d (%s)\n", LocalizedStringCount, Strings[TopIndex].Number, Strings[TopIndex].Name);
		return(FALSE);
	}

	/* No other sanity checks exist yet... */


	if (Verbose) {
		LogPrintf("none found!\n");
	}
	return(TRUE);
}

static int
GetStringIndex(const char *stringId)
{
    int i;
    
    for (i = 0; i < LocalizedStringCount; i++) {
	if (QuickCmp (stringId, Strings[i].Name)) {
	    return i;
	}
    }
    return -1;
}

static BOOL
ProcessLanguage(int language)
{
    int fileNo;
    int warned = FALSE;
    int stringNo;
    int j;
    
    if (Verbose) {
	LogPrintf("\rLoading strings for %s:\n", Languages[language].Name);
    }
    
    StringData[language] = malloc((LocalizedStringCount + 1) * sizeof(unsigned char *));		/* We need +1 to catch too many strings below! */
    if (!StringData[language]) {
	LogPrintf("Error: Not enough memory for language %s strings\n", Languages[language].Name);
	return(FALSE);
    }
    memset(StringData[language], 0, LocalizedStringCount * sizeof(unsigned char *));
    stringNo = 0;
    
    for (fileNo = 0; fileNo < Languages[language].FileCount; fileNo++) {
	FILE *data;
	
	data = IncludeFOpen(Languages[language].File[fileNo], "rb");
	if (!data) {
	    LogPrintf("Error: Could not open file %s\n", Languages[language].File[fileNo]);
	    return(FALSE);
	}
	
	do {
	    unsigned char line[10248];
	    
	    if (fgets(line, sizeof(line), data) != NULL) {
		ChopNL(line);
		if (stringNo < LocalizedStringCount) {
		    char *p;
		    int stringIndex;
		    
		    p = strchr (line, '|');
		    if (!p) {
			LogPrintf("Error: String name not specified for %s language string '%s'\n", Languages[language].Name, line);
			return FALSE;
		    }
		    
		    *p++ = 0;
		    
		    stringIndex = GetStringIndex(line);

		    if (stringIndex < 0) {
			LogPrintf("Error: Unknown string id %s in language %s\n", line, Languages[language].Name);
			fclose(data);
			return FALSE;
		    }

		    StringData[language][stringIndex] = strdup(p);
		    if (StringData[language][stringIndex] == NULL) {
			LogPrintf("Error: Not enough memory for %s language string %d\n", Languages[language].Name, stringNo);
			fclose(data);
			return(FALSE);
		    }
		    if (Verbose) {
			LogPrintf("Lang:%s, ID:%d > %s\n", Languages[language].Name, stringNo, line);
		    }
		    stringNo++;
		} else {
		    if (!warned) {
			LogPrintf("Warning: Too many strings for language %s\n", Languages[language].Name);
			warned=TRUE;
		    }
		}
	    }
	} while (!feof(data) && !ferror(data));
	
	fclose(data);
    }
   
    for (j = 0; j < LocalizedStringCount; j++) {
	int index = Strings[j].Number;
	
	if (StringData[language][index] == 0) {
	    /* Don't warn for the default language */
	    if (language != 0) {
		LogPrintf("Warning: Language %s did not define string '%s', using default.\n", Languages[language].Name, Strings[j].Name);
	    }
	    
	    StringData[language][index] = strdup(Strings[j].DefaultValue);
	}
    }

    return TRUE;
}


BOOL
LoadLanguageData(void)
{
	unsigned long	i, j;
	unsigned long StringNo;
	unsigned long	FileNo;
	unsigned long	Language;
	FILE *Data;
	struct stat		sb;
	BOOL				Warned;
	unsigned char Line[10248];

	if (Template.LanguageCount==0) {
		LogPrintf("\rNo languages defined\n");
		return(FALSE);
	}

	if (Verbose) {
		LogPrintf("\rLoading all language strings\n");
	} else {
		LogPrintf("\rPhase IIIa: Loading language strings\n");
	}

	/* Prepare common strings */
	if (Verbose) {
		LogPrintf("\rLoading common strings\n");
	}

	CommonStringData=malloc((CommonStringCount)*sizeof(unsigned char *));
	if (!CommonStringData) {
		LogPrintf("Error: Not enough memory for Common strings\n");
		return(FALSE);
	}
	memset(CommonStringData, 0, CommonStringCount*sizeof(unsigned char *));
	StringNo=0;
	Warned=FALSE;

	/* Fill common string data */
	for (i = 0; i < CommonStringCount; i++) {
	    CommonStringData[i] = strdup(CommonStrings[i].DefaultValue);
	}
	

	/* Prepare regular strings */
	StringData=malloc(Template.LanguageCount*sizeof(unsigned char *));
	if (!StringData) {
		LogPrintf("\rNot enough memory for language strings\n");
		return(FALSE);
	}
	memset(StringData, 0, Template.LanguageCount*sizeof(unsigned char *));

	for (Language=0; Language<Template.LanguageCount; Language++) {
	    if (!ProcessLanguage(Language)) {
		return FALSE;
	    }
	}

	if (Verbose) {
		LogPrintf("\rLoading all language Images\n");
	} else {
		LogPrintf("\rPhase IIIb: Loading language images\n");
	}

	/* Common images */
	if (Verbose) {
		LogPrintf("\rLoading common images\n");
	}
	CommonImageData=malloc(CommonImageCount*sizeof(unsigned char *));
	if (!CommonImageData) {
		LogPrintf("Error: Not enough memory for common images\n");
		return(FALSE);
	}
	memset(CommonImageData, 0, CommonImageCount*sizeof(unsigned char *));

	CommonImageLength=malloc(CommonImageCount*sizeof(unsigned long));
	if (!CommonImageLength) {
		LogPrintf("Error: Not enough memory for common image length data\n");
		return(FALSE);
	}
	memset(CommonImageLength, 0, CommonImageCount*sizeof(unsigned long));

	for (i=0; i<CommonImageCount; i++) {
		sprintf(Workfile, "%s/%s/common/%s", BaseDir, TemplateFilename, CommonImages[i].File);
		if (stat(Workfile, &sb)!=0) {
			LogPrintf("\rCannot find image %s (path:%s)\n", CommonImages[i].Name, Workfile);
			return(FALSE);
		}

		CommonImageData[i]=malloc(sb.st_size);
		if (CommonImageData[i]==NULL) {
			LogPrintf("\rNot enough memory for common image %s\n", CommonImages[i].Name);
			return(FALSE);
		}
		Data=fopen(Workfile, "rb");
		if (!Data) {
			LogPrintf("Error: Couldn't open common image file %s!\n", Images[i].Name);
			return(FALSE);
		}
		CommonImageLength[i]=sb.st_size;
		fread(CommonImageData[i], sb.st_size, 1, Data);
		fclose(Data);
		if (Verbose) {
			LogPrintf("\rLang:common, ID:%d, len:%d > %d/%s\n", i, CommonImageLength[i], Languages[Language].Number, CommonImages[i].File);
		}
	}

	if (i!=CommonImageCount) {
		LogPrintf("Error: Could not read all common images\n");
		for (j=0; j<i; j++) {
			free(CommonImageData[j]);
		}
		free(CommonImageData);
		free(CommonImageLength);
		CommonImageData=NULL;
		CommonImageLength=NULL;
		return(FALSE);
	}

	/* Regular images */
	ImageData=malloc(Template.LanguageCount*sizeof(unsigned char *));
	if (!ImageData) {
		LogPrintf("\rNot enough memory for language images\n");
		return(FALSE);
	}
	memset(ImageData, 0, Template.LanguageCount*sizeof(unsigned char *));

	ImageLength=malloc(Template.LanguageCount*sizeof(unsigned char *));
	if (!ImageLength) {
		LogPrintf("\rNot enough memory for language images\n");
		return(FALSE);
	}
	memset(ImageLength, 0, Template.LanguageCount*sizeof(unsigned char *));

	for (Language=0; Language<Template.LanguageCount; Language++) {
		if (Verbose) {
			LogPrintf("\rLoading images for %s:\n", Languages[Language].Name);
		}
		ImageData[Language]=malloc(LocalizedImageCount*sizeof(unsigned char *));
		if (!ImageData[Language]) {
			LogPrintf("Error: Not enough memory for language %s images\n", Languages[Language].Name);
			return(FALSE);
		}
		memset(ImageData[Language], 0, LocalizedImageCount*sizeof(unsigned char *));

		ImageLength[Language]=malloc(LocalizedImageCount*sizeof(unsigned long));
		if (!ImageLength[Language]) {
			LogPrintf("Error: Not enough memory for language %s image length data\n", Languages[Language].Name);
			return(FALSE);
		}
		memset(ImageLength[Language], 0, LocalizedImageCount*sizeof(unsigned long));

		for (i=0; i<LocalizedImageCount; i++) {
			sprintf(Workfile, "%s/%s/%d/%s", BaseDir, TemplateFilename, Languages[Language].Number, Images[i].File);
			if (stat(Workfile, &sb)!=0) {
				sprintf(Workfile, "%s/%s/%d/%s", BaseDir, TemplateFilename, Languages[0].Number, Images[i].File);
				if (stat(Workfile, &sb)!=0) {
					LogPrintf("\rCannot find image %s (path:%s)\n", Images[i].Name, Workfile);
					return(FALSE);
				} else {
					LogPrintf("\rCannot find image %s for language %d, defaulting to language %d\n", Images[i].Name, Languages[Language].Number, Languages[0].Number);
				}
			}

			ImageData[Language][i]=malloc(sb.st_size);
			if (ImageData[Language][i]==NULL) {
				LogPrintf("\rNot enough memory for %s language image %s\n", Languages[Language].Name, Images[i].Name);
				return(FALSE);
			}
			Data=fopen(Workfile, "rb");
			if (!Data) {
				LogPrintf("Error: Couldn't open image file %s for language %s!\n", Images[i].Name, Languages[Language].Name);
				return(FALSE);
			}
			ImageLength[Language][i]=sb.st_size;
			fread(ImageData[Language][i], sb.st_size, 1, Data);
			fclose(Data);
			if (Verbose) {
				LogPrintf("\rLang:%s, ID:%d, len:%d > %d/%s\n", Languages[Language].Name, i, ImageLength[Language][i], Languages[Language].Number, Images[i].File);
			}
		}

		if (i!=LocalizedImageCount) {
			LogPrintf("Error: Could not read all images for language %s\n", Languages[Language].Name);
			for (j=0; j<i; j++) {
				free(ImageData[Language][j]);
			}
			free(ImageData[Language]);
			free(ImageLength[Language]);
			ImageLength[Language]=NULL;
			return(FALSE);
		}
	}
	return(TRUE);
}


BOOL
CompileNames(void)
{
	unsigned long	i;
	unsigned long	ID;

	if (Verbose) {
		LogPrintf("\rParsing & compiling name tables\n");
	} else {
		LogPrintf("\rPhase IV: Compiling name tables\n");
	}

	for (i=0; i<Template.NameCount; i++) {
		if ((ID=ObjectNameToID(NamedDynamicObjects[i].ObjectName))==0xffffffff) {
			LogPrintf("\rCould not find template %s specified in [NamedTemplates] section\n", NamedDynamicObjects[i].ObjectName);
			return(FALSE);
		}
		NamedDynamicObjects[i].ObjectNumber=ID;
	}
	return(TRUE);
}

BOOL
FillOutTemplateStruct(TemplateStruct *Template)
{
    unsigned long i;
    unsigned long len;

    /* VersionID */

    /* Name */

    /* LanguageCount */
	
    /* StringCount */
    Template->StringCount = LocalizedStringCount + CommonStringCount;	

    /* ImageCount */
    Template->ImageCount = LocalizedImageCount + CommonImageCount;	

    /* TemplateCount */

    /* NameCount */

    /* CommonStaticObjectCount */

    /* FixupCount */
    Template->FixupCount = 
	Template->LanguageCount * (CommonStringCount + CommonImageCount + LocalizedStringCount + LocalizedImageCount) + 
	Template->TemplateCount + 
	Template->NameCount + 
	Template->CommonStaticObjectCount;

    /* InitialHTMLPage */
    if (InitialHTMLPage!=NULL) {
	for (i=0; i<Template->TemplateCount; i++) {
	    if (XplStrCaseCmp(DynamicObjects[i].Name, InitialHTMLPage)==0) {
		Template->InitialTemplate = i;
		DynamicObjects[i].UsedCount++;
		break;
	    }
	}

	if (i == Template->TemplateCount) {
	    LogPrintf("Error: Could not find initial template %s\n", InitialHTMLPage);
	    return(FALSE);
	}
    } else {
	Template->InitialTemplate = 0xffffffff;
    }

    /* InitialWMLPage */
    if (InitialWMLPage!=NULL) {
	for (i=0; i<Template->TemplateCount; i++) {
	    if (XplStrCaseCmp(DynamicObjects[i].Name, InitialWMLPage)==0) {
		Template->InitialWMLTemplate = i;
		DynamicObjects[i].UsedCount++;
		break;
	    }
	}

	if (i==Template->TemplateCount) {
	    LogPrintf("Error: Could not find initial WML display template %s\n", InitialWMLPage);
	    return(FALSE);
	}
    } else {
	Template->InitialWMLTemplate = 0;	/* Use the first template as default */
    }

    /* TimeOutPage */
    if (TimeOutPage!=NULL) {
	for (i=0; i<Template->TemplateCount; i++) {
	    if (XplStrCaseCmp(DynamicObjects[i].Name, TimeOutPage)==0) {
		Template->TimeoutTemplate = i;
		DynamicObjects[i].UsedCount++;
		break;
	    }
	}
	if (i==Template->TemplateCount) {
	    LogPrintf("Error: Could not find timeout template %s\n", TimeOutPage);
	    return(FALSE);
	}
    } else {
	Template->TimeoutTemplate = 0xffffffff;
    }

    /* ResumePage */
    if (ResumePage!=NULL) {
	for (i=0; i<Template->TemplateCount; i++) {
	    if (XplStrCaseCmp(DynamicObjects[i].Name, ResumePage)==0) {
		Template->ResumeTemplate = i;
		DynamicObjects[i].UsedCount++;
		break;
	    }
	}
	if (i==Template->TemplateCount) {
	    LogPrintf("Error: Could not find Resume template %s\n", ResumePage);
	    return(FALSE);
	}
    } else {
	Template->ResumeTemplate = 0xffffffff;
    }

    /* Colors */

    /* Write the Description field */
    if (Description[0]!='\0') {
	if ((Template->DescriptionString = StringNameToID(Description)) == 0xffffffff) {
	    LogPrintf("Error: Could not find Description string %s\n", Description);
	    return(0);
	}
    } else {
	Template->DescriptionString = 0xffffffff;
    }

    /* Spare fields */
    for (i = 0; i < 9; i++) {
	Template->Spare[i] = i;
    }

    /* SourcePath */
    len = strlen(BaseDir) + strlen(TemplateFilename) + 3;
    /* 2 for the slashes and 1 for the terminator */
    if (len < sizeof(Template->SourcePath)) {
	sprintf(Template->SourcePath, "%s/%s/", BaseDir, TemplateFilename);
    }
    return(TRUE);
}

BOOL
StoreStaticData(void)
{
	FILE				*Output;
	unsigned long	Base;
	unsigned long	Offset;
	unsigned long	Language;
	unsigned long	i;
	unsigned long	Padding[4]	=	{0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};
	BOOL				HaveCommonOffset=FALSE;
	unsigned long	CommonStringSave;
	unsigned long	CommonImageSave;
	unsigned long	CommonImageBase;

	if (Verbose) {
		LogPrintf("\rWriting static images and strings\n");
	} else {
		LogPrintf("\rPhase V: Storing static template data\n");
	}

	/* We write out all the data we already know about so that we can then add the compiled templates */
	Output=fopen(Targetfile, "wb");
	if (!Output) {
		LogPrintf("Error: Could not open targetfile %s for writing\n", Targetfile);
		return(FALSE);
	}

	/* write the structure */
	fwrite(&Template, sizeof(TemplateStruct), 1, Output);

	/* Now write the list of all language IDs */
	for (Language=0; Language<Template.LanguageCount; Language++) {
		fwrite(&Languages[Language].Number, sizeof(unsigned long), 1, Output);
	}

	/* Calculate where the fixup table will end, and how many items will be in it */	
	Base = 
	    sizeof(TemplateStruct) + 
	    sizeof(unsigned long) * Template.LanguageCount + 
	    sizeof(unsigned char *) * Template.FixupCount;

	/* Now write the complete fixup table; the offsets for the template list will be empty and filled out by the template parser */
	Offset=Base;

	if (Verbose) {
		LogPrintf("Now writing fixup table\n");
	}
	for (Language=0; Language<Template.LanguageCount; Language++) {
		/* Write "Common" Stringtable */
		CommonStringSave=Offset;
		Offset=Base;
		for (i=0; i<CommonStringCount; i++) {
			if (Verbose>1) {
				LogPrintf("Writing fixup address, Lang:%d, CommonString %02d, Offset:%08x\n", Language, i, Offset);
			}
			fwrite(&Offset, sizeof(unsigned long), 1, Output);
			Offset+=strlen(CommonStringData[i])+1;
			Offset+=(ALIGNMENT_BYTES - (strlen(CommonStringData[i])+1) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES;
		}
		if (Language!=0) {
			Offset=CommonStringSave;
		}
		for (i=0; i<LocalizedStringCount; i++) {
			if (Verbose>1) {
				LogPrintf("Writing fixup address, Lang:%d, String %02d, Offset:%08x\n", Language, i, Offset);
			}
			fwrite(&Offset, sizeof(unsigned long), 1, Output);
			Offset+=strlen(StringData[Language][i])+1;
			Offset+=(ALIGNMENT_BYTES - (strlen(StringData[Language][i])+1) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES;
		}

		if (Language==0) {
			CommonImageBase=Offset;
		}
		/* Write "Common" Imagetable */
		CommonImageSave=Offset;
		Offset=CommonImageBase;

		for (i=0; i<CommonImageCount; i++) {
			if (Verbose>1) {
				LogPrintf("Writing fixup address, Lang:%d, CommonImage %02d, Offset:%08x\n", Language, i, Offset);
			}
			fwrite(&Offset, sizeof(unsigned long), 1, Output);
			Offset+=CommonImageLength[i]+sizeof(unsigned long);
			Offset+=(ALIGNMENT_BYTES - CommonImageLength[i] % ALIGNMENT_BYTES) % ALIGNMENT_BYTES;
		}

		if (Language!=0) {
			Offset=CommonImageSave;
		}

		for (i=0; i<LocalizedImageCount; i++) {
			if (Verbose>1) {
				LogPrintf("Writing fixup address, Lang:%d, Image %02d, Offset:%08x\n", Language, i, Offset);
			}
			fwrite(&Offset, sizeof(unsigned long), 1, Output);
			Offset+=ImageLength[Language][i]+sizeof(unsigned long);
			Offset+=(ALIGNMENT_BYTES - ImageLength[Language][i] % ALIGNMENT_BYTES) % ALIGNMENT_BYTES;
		}
	}

	/* Write the name translation table */
	for (i=0; i<Template.NameCount; i++) {
		fwrite(&Offset, sizeof(unsigned long), 1, Output);
		Offset+=strlen(NamedDynamicObjects[i].Name)+1+sizeof(unsigned long);
		Offset+=(ALIGNMENT_BYTES - (strlen(NamedDynamicObjects[i].Name)+1) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES;
	}

	DynamicObjectFixupFileOffset=ftell(Output);
	for (i=0; i<Template.TemplateCount; i++) {
		/* We only write placeholders for the template fixup addresses for now */
		fwrite(&i, sizeof(unsigned long), 1, Output);
	}

	StaticObjectFixupFileOffset=ftell(Output);
	for (i=0; i<Template.CommonStaticObjectCount; i++) {
		/* We only write placeholders for the template fixup addresses for now */
		fwrite(&i, sizeof(unsigned long), 1, Output);
	}
	/* we've got the fixup table written, now write the actual data */
	for (Language=0; Language<Template.LanguageCount; Language++) {
		if (Language==0) {
			if (Verbose>1) {
				LogPrintf("Writing common strings, Offset:%08x\n", ftell(Output));
			}
			for (i=0; i<CommonStringCount; i++) {
				if (Verbose>1) {
					LogPrintf("Writing common string %02d , Offset:%08x\n", i, ftell(Output));
				}
				fwrite(CommonStringData[i], strlen(CommonStringData[i])+1, 1, Output);
				fwrite(&Padding, (ALIGNMENT_BYTES - (strlen(CommonStringData[i])+1) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES, 1, Output);
			}
		}
		if (Verbose>1) {
			LogPrintf("Writing regular strings, Offset:%08x\n", ftell(Output));
		}
		for (i=0; i<LocalizedStringCount; i++) {
			if (Verbose>1) {
				LogPrintf("Writing string %02d , Offset:%08x\n", i, ftell(Output));
			}
			fwrite(StringData[Language][i], strlen(StringData[Language][i])+1, 1, Output);
			fwrite(&Padding, (ALIGNMENT_BYTES - (strlen(StringData[Language][i])+1) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES, 1, Output);
		}

		if (Language==0) {
			if (Verbose>1) {
				LogPrintf("Writing common images, Offset:%08x\n", ftell(Output));
			}
			for (i=0; i<CommonImageCount; i++) {
				if (Verbose>1) {
					LogPrintf("Writing common image %02d , Offset:%08x\n", i, ftell(Output));
				}
				fwrite(&CommonImageLength[i], sizeof(unsigned long), 1, Output);
				fwrite(CommonImageData[i], CommonImageLength[i], 1, Output);
				fwrite(&Padding, (ALIGNMENT_BYTES - (CommonImageLength[i]) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES, 1, Output);
			}
		}
		if (Verbose>1) {
			LogPrintf("Writing regular images, Offset:%08x\n", ftell(Output));
		}
		for (i=0; i<LocalizedImageCount; i++) {
			if (Verbose>1) {
				LogPrintf("Writing image %02d , Offset:%08x\n", i, ftell(Output));
			}
			fwrite(&ImageLength[Language][i], sizeof(unsigned long), 1, Output);
			fwrite(ImageData[Language][i], ImageLength[Language][i], 1, Output);
			fwrite(&Padding, (ALIGNMENT_BYTES - (ImageLength[Language][i]) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES, 1, Output);
		}
	}

	/* Write the name translation table */
	for (i=0; i<Template.NameCount; i++) {
		fwrite(&NamedDynamicObjects[i].ObjectNumber, sizeof(unsigned long), 1, Output);
		fwrite(NamedDynamicObjects[i].Name, strlen(NamedDynamicObjects[i].Name)+1, 1, Output);
		fwrite(&Padding, (ALIGNMENT_BYTES - (strlen(NamedDynamicObjects[i].Name)+1) % ALIGNMENT_BYTES) % ALIGNMENT_BYTES, 1, Output);
	}
	
	fclose(Output);
	return(TRUE);
}

unsigned long
ObjectNameToID(unsigned char *Name)
{
	unsigned long	i;

	for (i=0; i<Template.TemplateCount; i++) {
		if (XplStrCaseCmp(DynamicObjects[i].Name, Name)==0) {
			DynamicObjects[i].UsedCount++;
			return(i);
		}
	}
	return(0xffffffff);
}

unsigned long
TemplateNumberToID(unsigned long Number)
{
	unsigned long	i;

	for (i=0; i<Template.TemplateCount; i++) {
		if (DynamicObjects[i].Number==Number) {
			DynamicObjects[i].UsedCount++;
			return(i);
		}
	}
	return(0xffffffff);
}

unsigned long
StaticObjectNameToID(unsigned char *Name)
{
	unsigned long	i;

	for (i=0; i<Template.CommonStaticObjectCount; i++) {
		if (XplStrCaseCmp(StaticObjects[i].Name, Name)==0) {
			StaticObjects[i].UsedCount++;
			return(i);
		}
	}
	return(0xffffffff);
}

unsigned long
StringNameToID(unsigned char *Name)
{
	unsigned long	i;

	/* First, check the common strings */
	for (i=0; i<CommonStringCount; i++) {
//		LogPrintf("\rComparing Common StringName %s with %s\n", CommonStrings[i].Name, Name);
		if (QuickCmp(CommonStrings[i].Name, Name)) {
			CommonStrings[i].UsedCount++;
			return(i);
		}
	}

	/* Now check regular strings */
	for (i=0; i<LocalizedStringCount; i++) {
//		LogPrintf("\rComparing StringName %s with %s\n", Strings[i].Name, Name);
		if (QuickCmp(Strings[i].Name, Name)) {
			Strings[i].UsedCount++;
			return(CommonStringCount+i);
		}
	}
	return(0xffffffff);
}

unsigned long
ImageNameToID(unsigned char *Name)
{
	unsigned long	i;

	/* First, check the common strings */
	for (i=0; i<CommonImageCount; i++) {
		if (XplStrCaseCmp(CommonImages[i].Name, Name)==0) {
			CommonImages[i].UsedCount++;
			return(i);
		}
	}

	/* Now check regular images */
	for (i=0; i<LocalizedImageCount; i++) {
		if (XplStrCaseCmp(Images[i].Name, Name)==0) {
			Images[i].UsedCount++;
			return(CommonImageCount+i);
		}
	}
	return(0xffffffff);
}


unsigned long
AssociateNameToID(unsigned char *Name)
{
	unsigned long	i;

	for (i=0; i<AssociateDefCount; i++) {
		if (QuickCmp(AssociateDefs[i].Name, Name)) {	/* Yes, found a keyword */
			return(AssociateDefs[i].Number);			
		}
	}

	return(0xffffffff);
}

unsigned long
StoreEndToken(FILE *Output, unsigned long PrevSize)
{

	TokenOverlayStruct	Overlay;
	unsigned long			Padding[4]	=	{0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};
	unsigned long			PadBytes;

	memset(&Overlay, 0, sizeof(TokenOverlayStruct));

	Overlay.TokenID=0;
	Overlay.Length=sizeof(TokenOverlayStruct);
	PadBytes=CALC_ALIGNMENT_SIZE(Overlay.Length);
	Overlay.Length+=PadBytes;
	Overlay.PrevSize=PrevSize;

	fwrite(&Overlay, sizeof(TokenOverlayStruct), 1, Output);
	fwrite(&Padding, PadBytes, 1, Output);

	return(Overlay.Length);
}

unsigned long
StoreTextToken(FILE *Output, unsigned char *Text, unsigned long Length, unsigned long PrevSize)
{
	TokenOverlayStruct	Overlay;
	unsigned long			Padding[4]	=	{0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};
	unsigned long			PadBytes;

	memset(&Overlay, 0, sizeof(TokenOverlayStruct));

	Overlay.TokenID=1;
	Overlay.Length=Length+sizeof(TokenOverlayStruct);
	PadBytes=CALC_ALIGNMENT_SIZE(Overlay.Length);
	Overlay.Length+=PadBytes;
	Overlay.PrevSize=PrevSize;
	Overlay.ArgumentSize[0]=Length;
	Overlay.ArgumentOffsetOrID[0]=0;
	fwrite(&Overlay, sizeof(TokenOverlayStruct), 1, Output);
	fwrite(Text, Length, 1, Output);
	fwrite(&Padding, PadBytes, 1, Output);
	
	return(Overlay.Length);
}

unsigned long
StoreHeaderToken(FILE *Output, unsigned char *MIMEType, BOOL Cacheable, unsigned long PrevSize)
{
	TokenOverlayStruct	Overlay;
	unsigned long			Padding[4]	=	{0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};
	unsigned long			PadBytes;
	unsigned long			Length=strlen(MIMEType)+1;

	memset(&Overlay, 0, sizeof(TokenOverlayStruct));

	Overlay.TokenID=1;
	Overlay.Length=Length+sizeof(TokenOverlayStruct);
	PadBytes=CALC_ALIGNMENT_SIZE(Overlay.Length);
	Overlay.Length+=PadBytes;
	Overlay.PrevSize=PrevSize;
	Overlay.ArgumentSize[0]=Length;
	Overlay.ArgumentOffsetOrID[0]=0;
	Overlay.ArgumentSize[0]=0;
	Overlay.ArgumentOffsetOrID[1]=Cacheable;

	fwrite(&Overlay, sizeof(TokenOverlayStruct), 1, Output);
	fwrite(MIMEType, Length, 1, Output);
	fwrite(&Padding, PadBytes, 1, Output);
	
	return(Overlay.Length);
}

unsigned long
ParseAndStoreToken(FILE *Output, unsigned char *Token, unsigned long PrevSize, unsigned char *TemplateFile, unsigned char *Filename, unsigned long Line)
{
	unsigned long			i;
	unsigned long			TokenLen;
	unsigned char			*Args;
	unsigned char			*ArgPtr;
	unsigned char			*ArgEndPtr;
	unsigned long			ArgCount;
	unsigned char			*ptr;
	unsigned long			ID;
	TokenOverlayStruct	Overlay;
	unsigned long			OverlayPos;
	unsigned long			ArgumentOffset;
	unsigned long			Padding[4]	=	{0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};
	unsigned long			PadBytes;

	/*
		This function is complicated; we need to write the data, calculate the size, then seek back and
		write the structure describing the data.
	*/

	memset(&Overlay, 0, sizeof(TokenOverlayStruct));

	TokenLen=strlen(Token);

	if (Verbose>1) {
		LogPrintf("ParseAndStoreToken:%s\n", Token);
	}

	Args=strchr(Token, ' ');
	if (Args) {
		*Args='\0';
		Args++;
		while (isspace(*Args)) {
			Args++;
		}
		ArgPtr=Args;
	} else {
		ArgPtr=Token+TokenLen;
	}

	for (i=0; i<TokenCount; i++) {
		if (QuickCmp(Token, Tokens[i].Keyword)) {
			/* We found a match */

			/* Remember the position, so we can write the TokenOverlay later (again) */
			OverlayPos=ftell(Output);

			fwrite(&Overlay, sizeof(TokenOverlayStruct), 1, Output);
			Overlay.Length=0;
			ArgumentOffset=0; 
			Overlay.TokenID=Tokens[i].TokenID;
			Overlay.PrevSize=PrevSize;

//			for (ArgCount=0; ArgCount<Tokens[i].NumOfArguments; ArgCount++) {
			ArgCount=0;
			while (ArgPtr[0]!='\0') {
				/* We need to find the end of the argument and NULL it out */
				if (*ArgPtr=='"') {
					ArgPtr++;
					ArgEndPtr=strchr(ArgPtr, '"');
CheckArgAgain:
					if (!ArgEndPtr) {
						ArgEndPtr=strchr(ArgPtr, ' ');
						if (ArgEndPtr) {
							*ArgEndPtr='\0';
						}
					} else {
						if (ArgEndPtr[-1]!='\\') {
							*ArgEndPtr='\0';
						} else {
							ArgEndPtr=strchr(ArgEndPtr+1, '"');
							goto CheckArgAgain;
						}
					}
				} else {
					ArgEndPtr=strchr(ArgPtr, ' ');
					if (ArgEndPtr) {
						*ArgEndPtr='\0';
					}
				}

				/* Check if we have escaped data in the string */
				ptr=ArgPtr;
				while ((ptr=strchr(ptr, '"'))!=NULL) {
					if (ptr[-1]=='\\') {
						memmove(ptr-1, ptr, strlen(ptr)+1);
					}
				}

				switch (Tokens[i].ArgType[ArgCount]) {
					case ARG_NUM: {
						long		ArgValue;

						ArgValue=atol(ArgPtr);
						Overlay.ArgumentSize[ArgCount]=sizeof(long);

						/* The next line is commented out; it's more efficient to store the number directly in ArgumentOffsetOrID */
//						Overlay.ArgumentOffsetOrID[ArgCount]=ArgumentOffset;
						Overlay.ArgumentOffsetOrID[ArgCount]=ArgValue;
						if (Overlay.ArgumentSize[ArgCount]>0) {
							fwrite(&ArgValue, Overlay.ArgumentSize[ArgCount], 1, Output);
						}
						Overlay.Length+=Overlay.ArgumentSize[ArgCount];
						PadBytes=CALC_ALIGNMENT_SIZE(Overlay.Length);
						Overlay.Length+=PadBytes;
						fwrite(&Padding, PadBytes, 1, Output);
						ArgumentOffset+=Overlay.ArgumentSize[ArgCount]+PadBytes;
						break;
					}

					case ARG_URL:
					case ARG_TEXT: {
						Overlay.ArgumentSize[ArgCount]=strlen(ArgPtr);
						Overlay.ArgumentOffsetOrID[ArgCount]=ArgumentOffset;/*-sizeof(TokenOverlayStruct);*/
						if (Overlay.ArgumentSize[ArgCount]>0) {
							fwrite(ArgPtr, Overlay.ArgumentSize[ArgCount], 1, Output);
						}
						Overlay.Length+=Overlay.ArgumentSize[ArgCount];
						PadBytes=CALC_ALIGNMENT_SIZE(Overlay.Length);
						Overlay.Length+=PadBytes;
						fwrite(&Padding, PadBytes, 1, Output);
						ArgumentOffset+=Overlay.ArgumentSize[ArgCount]+PadBytes;
						break;
					}

					case ARG_ASSOC:
					case ARG_PAGE:
					case ARG_PAGEID:
					case ARG_STATIC:
					case ARG_IMAGE:
					case ARG_STRING: {
						ptr=strchr(ArgPtr, ' ');
						if (ptr) {
							*ptr='\0';
						}
						switch (Tokens[i].ArgType[ArgCount]) {
							case ARG_STRING: {
								if ((ID=StringNameToID(ArgPtr))==0xffffffff) {
									if (ArgPtr[0]!='\0') {
										LogPrintf("\r%s(%d): %s: Referenced string '%s' unknown\n", Filename, Line, Tokens[i].Keyword, ArgPtr);
									} else {
										LogPrintf("\r%s(%d): %s: Required argument of type 'string' missing\n", Filename, Line, Tokens[i].Keyword);
									}
									return(0);
								}
								if (Verbose) LogPrintf("Token:%s String %s => ID:%d\n", Token, ArgPtr, ID);
								break;
							}

							case ARG_IMAGE: {
								if ((ID=ImageNameToID(ArgPtr))==0xffffffff) {
									if (ArgPtr[0]!='\0') {
										LogPrintf("\r%s(%d): %s: Referenced image '%s' unknown\n", Filename, Line, Tokens[i].Keyword, ArgPtr);
									} else {
										LogPrintf("\r%s(%d): %s: Required argument of type 'image' missing\n", Filename, Line, Tokens[i].Keyword);
									}
									return(0);
								}
								if (Verbose) LogPrintf("Token:%s Image %s => ID:%d\n", Token, ArgPtr, ID);
								break;
							}

							case ARG_PAGE: {
								if ((ID=ObjectNameToID(ArgPtr))==0xffffffff) {
									if (ArgPtr[0]!='\0') {
										LogPrintf("\r%s(%d): %s: Referenced template '%s' unknown\n", Filename, Line, Tokens[i].Keyword, ArgPtr);
									} else {
										LogPrintf("\r%s(%d): %s: Required argument of type 'template name' missing\n", Filename, Line, Tokens[i].Keyword);
									}
									return(0);
								}
								if (Verbose) LogPrintf("Token:%s Template %s => ID:%d\n", Token, ArgPtr, ID);
								break;
							}

							case ARG_PAGEID: {
								if ((ID=TemplateNumberToID(atol(ArgPtr)))==0xffffffff) {
									if (ArgPtr[0]!='\0') {
										LogPrintf("\r%s(%d): %s: Referenced template no '%s' unknown\n", Filename, Line, Tokens[i].Keyword, ArgPtr);
									} else {
										LogPrintf("\r%s(%d): %s: Required argument of type 'template number' missing\n", Filename, Line, Tokens[i].Keyword);
									}
									return(0);
								}
								if (Verbose) LogPrintf("Token:%s TemplateNo %d => ID:%d\n", Token, atol(ArgPtr), ID);
								break;
							}

							case ARG_STATIC: {
								if ((ID=StaticObjectNameToID(ArgPtr))==0xffffffff) {
									if (ArgPtr[0]!='\0') {
										LogPrintf("\r%s(%d): %s: Referenced static page name '%s' unknown\n", Filename, Line, Tokens[i].Keyword, ArgPtr);
									} else {
										LogPrintf("\r%s(%d): %s: Required argument of type 'template name' missing\n", Filename, Line, Tokens[i].Keyword);
									}
									return(0);
								}
								if (Verbose) LogPrintf("Token:%s TemplateNo %d => ID:%d\n", Token, atol(ArgPtr), ID);
								break;
							}

							case ARG_ASSOC: {
								if ((ID=AssociateNameToID(ArgPtr))==0xffffffff) {
									if (ArgPtr[0]!='\0') {
										LogPrintf("\r%s(%d): %s: Referenced associate name '%s' unknown\n", Filename, Line, Tokens[i].Keyword, ArgPtr);
									} else {
										LogPrintf("\r%s(%d): %s: Required argument of type 'associate name' missing\n", Filename, Line, Tokens[i].Keyword);
									}
									return(0);
								}
								if (Verbose) LogPrintf("Token:%s String %s => ID:%d\n", Token, ArgPtr, ID);
								break;
							}

						}
						Overlay.ArgumentSize[ArgCount]=sizeof(unsigned long);
						Overlay.ArgumentOffsetOrID[ArgCount]=ID;
						fwrite(&ID, sizeof(unsigned long), 1, Output);

						Overlay.Length+=Overlay.ArgumentSize[ArgCount];
						PadBytes=CALC_ALIGNMENT_SIZE(Overlay.Length);
						Overlay.Length+=PadBytes;
						fwrite(&Padding, PadBytes, 1, Output);
						ArgumentOffset+=Overlay.ArgumentSize[ArgCount]+PadBytes;

						if (ptr) {
							*ptr=' ';
						}
						break;
					}
				}

				if (ArgEndPtr) {
					ArgPtr=ArgEndPtr+1;
					while (*ArgPtr && isspace(*ArgPtr)) {
						ArgPtr++;
					}
				} else {
					ArgPtr=Token+TokenLen;
				}
				ArgCount++;
			}

			if (ArgCount<Tokens[i].NumOfArguments) {
				LogPrintf("\r%s(%d): Error: Not enough arguments for command '%s'\n", Filename, Line, Token);
				return(0);
			}

			/* We now have a filled-out overlay structure; seek back and write it... */
			ArgumentOffset=ftell(Output);
			Overlay.Length+=sizeof(TokenOverlayStruct);
			fseek(Output, OverlayPos, SEEK_SET);
			fwrite(&Overlay, sizeof(TokenOverlayStruct), 1, Output);
			fseek(Output, ArgumentOffset, SEEK_SET);
			return(Overlay.Length);
		}
	}

	LogPrintf("\r%s(%d): Error: Unknown token '%s'\n", Filename, Line, Token);
	return(0);
}

BOOL
ReadPageFile(unsigned char **PageBufferIn, unsigned char *Path, unsigned char *Filename, unsigned char *Name, unsigned long *PageSize, time_t *FileModTime)
{
	FILE				*pageFile;
	struct stat		sb;
	unsigned char	*pageBuffer;;

	pageBuffer=NULL;
	*PageBufferIn=NULL;

	
	pageFile=fopen(Path, "rb");
	if (!pageFile) {
		LogPrintf("\rCould not open template file %s (%s)\n", Filename, Path);
		return(FALSE);
	}

	if (Verbose) {
		LogPrintf("\rProcessing template %s (%s)\n", Filename, Name);
	}

	/* We read the whole template in a single slurp - allows for more efficient generated binaries */
	if (stat(Path, &sb)!=0) {
		LogPrintf("\rError getting size of template %s (%s)\n", Name, Workfile);
		fclose(pageFile);
		return(FALSE);
	}

	if (FileModTime) {
	    *FileModTime = sb.st_mtime;
	}
	
	pageBuffer=malloc(sb.st_size+sizeof(unsigned char));

	if (!pageBuffer) {
		LogPrintf("\rError getting memory for template %s (%s)\n", Name, Workfile);
		fclose(pageFile);
		return(FALSE);
	}

	if (fread(pageBuffer, sb.st_size, 1, pageFile)!=1) {
		LogPrintf("\rError reading template %s (%s)\n", Name, Workfile);
		fclose(pageFile);
		free(pageBuffer);
		return(FALSE);
	}
	pageBuffer[sb.st_size]='\0';

	fclose(pageFile);
	*PageBufferIn = pageBuffer;
	if (PageSize) {
		*PageSize = sb.st_size;
	}
	return(TRUE);
}

static BOOL
ParseAndStoreDynamicObject(FILE *outputFile, unsigned long *outputFileOffset, unsigned char *objectBuffer, unsigned char *objectPath, unsigned char *objectFileName, unsigned char *objectName, unsigned long objectID, unsigned long *PrevPos, BOOL TopLevel, int NestingLevel)
{
	unsigned char			*ptr;
	unsigned char			*prevptr;
	unsigned char			*endptr;
	unsigned char			*LastLinePtr;
	unsigned char			*LinePtr;
	unsigned long			Line=1;
	unsigned long			PrevPos2;
	unsigned long			PadBytes;
	unsigned long			Padding[4]	=	{0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};
	unsigned long			Len;
	unsigned long			i;

	/* This is the real parser */

	if (NestingLevel>MAX_NESTING) {
		LogPrintf("%s: Nesting too deep, possible #include loop, %d levels maximum supported\n", objectName, MAX_NESTING);
		return(FALSE);
	}

	if (Verbose) {
		LogPrintf("\rParsing the template\n");
	}

	prevptr=objectBuffer;
	ptr=prevptr;

	LinePtr=ptr;

	while (ptr) {
		ptr=strchr(prevptr, '{');

		/* Let's count lines */
		while (ptr && LinePtr!=ptr) {
			if (*LinePtr=='\n') {
				Line++;
			}
			LinePtr++;
		}
		LastLinePtr=LinePtr;

		/* Pass any text since the last token */
CheckAgain:
		if (!ptr) {
			Len=strlen(prevptr);
		} else {
			/* We need to make sure we've got one of our tokens and not a javascript page or something... */
			if (*(ptr+1)!='#' && *(ptr+1)!='!' && !isalpha(*(ptr+1))) {		/* No alpha character, can't be ours */
				ptr=strchr(ptr+1, '{');													/* } - to allow bracket matching */

				/* Let's count lines */
				while (ptr && LinePtr!=ptr) {
					if (*LinePtr=='\n') {
						Line++;
					}
					LinePtr++;
				}
				LastLinePtr=LinePtr;

				goto CheckAgain;
			}

			/* Check if it's our include statement */
			if (QuickNCmp(ptr, "{#include ", 10)) {
				/* Parse the rest and process the include */
				endptr=strchr(ptr, '}');
				if (!endptr) {
					if (Warnings) {
						unsigned char	TokenBuf[20];

						strncpy(TokenBuf, ptr+1, 19);									/* { - to allow bracket matching */
						endptr=strchr(TokenBuf, '}');
						if (!endptr) {
							endptr=strchr(TokenBuf, ' ');
						}
						if (endptr) {
							*endptr='\0';
						}
						LogPrintf("%s: Syntax error: %s\n", objectName, TokenBuf);
					}
				} else {
					unsigned char	*IncludeTemplate;
					unsigned char	Includefile[XPL_MAX_PATH]		= "";

					/* First, we write out the data collected so far... */
					Len=ptr-prevptr;
					PrevPos2=ftell(outputFile);
					*outputFileOffset+=StoreTextToken(outputFile, prevptr, Len, PrevPos2-*PrevPos);
					*PrevPos=PrevPos2;

					/* We need to read the include file, parse it, and then continue right here... */
					*endptr='\0';
					prevptr=endptr+1;
					ptr+=10;
					while (isspace(*ptr)) {
						ptr++;
					}
					
					if (*ptr=='"') {
						ptr++;
						endptr=strchr(ptr, '"');
						if (endptr) {
							*endptr='\0';
						}
					}
					/* ptr now points to the filename of the include file */
					
					sprintf(Includefile, "%s/%s/%s", BaseDir, TemplateFilename, ptr);
					if (Verbose) {
						LogPrintf("Reading include file %s, included by %s\n", ptr, objectName);
					}
					if (ReadPageFile(&IncludeTemplate, Includefile, ptr, "<include file>", NULL, NULL)) {
						if (Verbose) {
							LogPrintf("Parsing include file %s, included by %s\n", ptr, objectName);
						}
						if (!ParseAndStoreDynamicObject(outputFile, outputFileOffset, IncludeTemplate, Includefile, objectFileName, ptr, objectID, PrevPos, FALSE, ++NestingLevel)) {
							if (IncludeTemplate) {
								free(IncludeTemplate);
							}
							return(FALSE);
						}
					}
					if (IncludeTemplate) {
						free(IncludeTemplate);
					}
					continue;
				}
			} else if (QuickNCmp(ptr, "{#ifdef ", 8) || QuickNCmp(ptr, "{#ifndef ", 9)) {
				BOOL				Expect=FALSE;
				unsigned long	NestCount=0;
				unsigned char	*Check;
				unsigned long	DefinePos=8;

				/* Flip the logic? */
				if (toupper(ptr[4])=='N') {
					Expect=TRUE;
					DefinePos=9;
				}
				endptr=strchr(ptr, '}');
				if (!endptr) {
					if (Warnings) {
						unsigned char	TokenBuf[20];

						strncpy(TokenBuf, ptr+1, 19);
						endptr=strchr(TokenBuf, '}');
						if (!endptr) {
							endptr=strchr(TokenBuf, ' ');
						}
						if (endptr) {
							*endptr='\0';
						}
						LogPrintf("%s: Syntax error: %s\n", objectName, TokenBuf);
					}
					continue;
				}
				*endptr='\0';
				if (IsDefined(ptr+DefinePos)==Expect) {
					/* The ifdef/ifndef excludes, skip the data */
					Check=ptr+1;
					*endptr='}';
					while (*Check) {
						if (Check[0]=='{') {													/* } */
							if (QuickNCmp(Check, "{#ifdef ", 8) || QuickNCmp(Check, "{#ifndef ", 9)) {
								NestCount++;
							} else if (QuickNCmp(Check, "{#endif", 7)) {
								if (NestCount>0) {
									NestCount--;
								} else {
									/* We found the matching endif, continue */

									/* We write out the data collected so far... */
									Len=ptr-prevptr;
									PrevPos2=ftell(outputFile);
									*outputFileOffset+=StoreTextToken(outputFile, prevptr, Len, PrevPos2-*PrevPos);
									*PrevPos=PrevPos2;

									/* Now "reset" our pointers to after the conditional section */
									prevptr=Check;
									while(prevptr[0] && prevptr[0]!='}') {
										prevptr++;
									}
									if (prevptr[0]) {
										prevptr++;
									} else {
										ptr=NULL;
									}
									/* We're in a nested while loop, when we drop out we hit the continue for the outer loop */
									break;
								}
							}
						}
						Check++;
					}
				} else {
					*endptr='}';

					/* We write out the data collected so far... */
					Len=ptr-prevptr;
					PrevPos2=ftell(outputFile);
					*outputFileOffset+=StoreTextToken(outputFile, prevptr, Len, PrevPos2-*PrevPos);
					*PrevPos=PrevPos2;
					prevptr=endptr+1;
				}
				continue;
			} else if(QuickNCmp(ptr, "{#endif", 7)) {
				/* simply skip the token */
				endptr=strchr(ptr, '}');
				if (!endptr) {
					LogPrintf("%s: Syntax error missing '}' after '{#endif' \n", objectName);
					prevptr=ptr+1;
				} else {
					/* We write out the data collected so far... */
					Len=ptr-prevptr;
					PrevPos2=ftell(outputFile);
					*outputFileOffset+=StoreTextToken(outputFile, prevptr, Len, PrevPos2-*PrevPos);
					*PrevPos=PrevPos2;
					prevptr=endptr+1;
				}
				continue;
			}

			/* Check the list of known tokens */
			for (i=0; i<TokenCount; i++) {
				if (QuickNCmp(ptr+1, Tokens[i].Keyword, strlen(Tokens[i].Keyword))) {	/* Yes, found a keyword */
					break;
				}
			}
			if (ptr[1]!='!' && i==TokenCount) {
				/* We get here only if we didn't find one of our keywords */
				if (Warnings) {
					unsigned char	TokenBuf[128];

					strncpy(TokenBuf, ptr+1, 127);
					endptr=strchr(TokenBuf, '}');
					if (!endptr) {
						endptr=strchr(TokenBuf, ' ');
					}
					if (endptr) {
						*endptr='\0';
					}
					LogPrintf("%s(%d): Unknown token '%s' in template '%s'\n", objectPath, Line, TokenBuf, objectName);
				}
				ptr=strchr(ptr+1, '{');
				goto CheckAgain;
			}
			Len=ptr-prevptr;
		}

		if (Len>0) {
			PrevPos2=ftell(outputFile);
			*outputFileOffset+=StoreTextToken(outputFile, prevptr, Len, PrevPos2-*PrevPos);
			*PrevPos=PrevPos2;
		}
		if (ptr) {
			ptr++;
			endptr=strchr(ptr, '}');
			if (ptr[0]!='!') {		/* We skip comments */
				*endptr='\0';
				if (Verbose) {
					LogPrintf("\rTemplate %s (%d): Examining token %s\n", objectName, objectID, ptr);
				}
				PrevPos2=ftell(outputFile);
				Len=ParseAndStoreToken(outputFile, ptr, PrevPos2-*PrevPos, objectName, objectPath, Line);
				*PrevPos=PrevPos2;
				if (Len==0) {
					fclose(outputFile);
					return(FALSE);
				}
				*outputFileOffset+=Len;
				PadBytes=(ALIGNMENT_BYTES-(*outputFileOffset % ALIGNMENT_BYTES)) % ALIGNMENT_BYTES;
				if (PadBytes > 0) {
				    LogPrintf("\rError: mwcomp is producing tokens that are not byte aligned.\n");
				    fclose(outputFile);
				    return(FALSE);

                                    /* This code would realign */
				    /* fwrite(&Padding, PadBytes, 1, outputFile); */
				    /* *outputFileOffset += PadBytes; */
				}
			}
			prevptr=endptr+1;
		}
	}

	if (TopLevel) {
		*PrevPos=ftell(outputFile)-*PrevPos;
		*outputFileOffset+=StoreEndToken(outputFile, *PrevPos);
	}

	return(TRUE);
}

BOOL
StoreObject(FILE *outputFile, unsigned long *outputFileOffset, unsigned char *objectFileName, unsigned char *objectContentType, time_t objectModTime, unsigned char *objectStaticData, unsigned long objectStaticSize)
{
    TemplateObjectOverlay localObject;
    unsigned long overheadBytes;
    unsigned long padBytes;
    unsigned long padding[4] = {0xefbeadde, 0xefbeadde, 0xefbeadde, 0xefbeadde};

    /* object meta data */
    localObject.FileModTime = objectModTime;
    localObject.FileNameLen = strlen(objectFileName);
    localObject.ContentTypeLen = strlen(objectContentType);

    /* object fixup table */
    localObject.FileName = (unsigned char *)(sizeof(TemplateObjectOverlay));
    localObject.ContentType = (unsigned char *)(sizeof(TemplateObjectOverlay) + localObject.FileNameLen + 1); /* Remember the terminators */

    overheadBytes = sizeof(TemplateObjectOverlay) + localObject.FileNameLen + 1 + localObject.ContentTypeLen + 1;

    if (objectStaticData) {
	localObject.ObjectLen = objectStaticSize;
	padBytes = CALC_ALIGNMENT_SIZE(overheadBytes + objectStaticSize);

	/* fixup for the object itself */
	localObject.Object = (unsigned char *)overheadBytes;
    } else {
	/* dynamic object */
	localObject.ObjectLen = 0;
	padBytes = CALC_ALIGNMENT_SIZE(overheadBytes);

	/* fixup for the first token */
	localObject.Object = (unsigned char *)overheadBytes + padBytes;
    }    

    if (outputFileOffset) {
	if (objectStaticData) {
	    *outputFileOffset += (overheadBytes + padBytes + (sizeof(unsigned char) * objectStaticSize));
	} else {
	    *outputFileOffset += (overheadBytes + padBytes);
	}
    }

    if (fwrite(&localObject, sizeof(TemplateObjectOverlay), 1, outputFile) != 1) {
	return(FALSE);
    }
 
    if (fwrite(objectFileName, sizeof(unsigned char), localObject.FileNameLen + 1, outputFile) != (localObject.FileNameLen + 1)) {
	return(FALSE);
    }

    if (fwrite(objectContentType, sizeof(unsigned char), localObject.ContentTypeLen + 1, outputFile) != (localObject.ContentTypeLen + 1)) {
	return(FALSE);
    }

    if (objectStaticData) {
	if (fwrite(objectStaticData, sizeof(unsigned char), objectStaticSize, outputFile) != objectStaticSize) {
	    return(FALSE);
	}
    }

    fwrite(&padding, padBytes, 1, outputFile);

    return(TRUE);
}

BOOL
CompileTemplates(void)
{
    time_t modTime;
    FILE						*Output;
	unsigned char			*pageBuffer;
	unsigned long			TemplateID;
	unsigned long			Offset;
	unsigned long			PrevPos;

	if (Verbose) {
		LogPrintf("\rParsing & compiling template files\n");
	} else {
		LogPrintf("\rPhase VI: Compiling template files\n");
	}


	DynamicObjectOffsets=malloc(sizeof(unsigned long)*Template.TemplateCount);
	if (!DynamicObjectOffsets) {
		LogPrintf("\rError: No memory to allocate DynamicObjectOffsets\n");
		return(FALSE);
	}

	Output=fopen(Targetfile, "r+b");
	if (!Output) {
		LogPrintf("\rError: Could not re-open output file %s\n", Targetfile);
		return(FALSE);
	}

	fseek(Output, 0, SEEK_END);

	Offset=ftell(Output);

	for (TemplateID=0; TemplateID<Template.TemplateCount; TemplateID++) {
	    DynamicObjectOffsets[TemplateID]=Offset;
	    sprintf(Workfile, "%s/%s/%s", BaseDir, TemplateFilename, DynamicObjects[TemplateID].File);
	    if (ReadPageFile(&pageBuffer, Workfile, DynamicObjects[TemplateID].File, DynamicObjects[TemplateID].Name, NULL, &modTime)) {
		if (StoreObject(Output, &Offset, DynamicObjects[TemplateID].File, DynamicObjects[TemplateID].MIMEType, modTime, NULL, 0)) {
		    PrevPos=ftell(Output);
		    if (!ParseAndStoreDynamicObject(Output, &Offset, pageBuffer, Workfile, DynamicObjects[TemplateID].File, DynamicObjects[TemplateID].Name, TemplateID, &PrevPos, TRUE, 0)) {
			if (pageBuffer) {
			    free(pageBuffer);
			}
			return(FALSE);
		    }
		} else {
		    LogPrintf("\rError: Could not write to output file %s\n", Targetfile);
		    if (pageBuffer) {
			free(pageBuffer);
		    }
		    return(FALSE);

		}
	    }
	    if (pageBuffer) {
		free(pageBuffer);
	    }
	}

	/* Now go back and write the Fixup table for the templates */
	fseek(Output, DynamicObjectFixupFileOffset, SEEK_SET);
	for (TemplateID=0; TemplateID<Template.TemplateCount; TemplateID++) {
		fwrite(&(DynamicObjectOffsets[TemplateID]), sizeof(unsigned long), 1, Output);
	}
	fclose(Output);
	return(TRUE);
}

BOOL
StoreStaticObjects(void)
{
    TemplateObjectOverlay StaticObject;
	FILE						*Output;
	unsigned char			*Page;
	unsigned long			PageID;
	unsigned long			PrevPos;
	unsigned long			Size;
	time_t modTime;

	if (Verbose) {
		LogPrintf("\rStoring static pages\n");
	} else {
		LogPrintf("\rPhase VII: Storing static pages\n");
	}

	StaticObjectOffsets=malloc(sizeof(unsigned long)*Template.CommonStaticObjectCount);
	if (!StaticObjectOffsets) {
		LogPrintf("\rError: No memory to allocate StaticObjectOffsets\n");
		return(FALSE);
	}

	Output=fopen(Targetfile, "r+b");
	if (!Output) {
		LogPrintf("\rError: Could not re-open output file %s\n", Targetfile);
		return(FALSE);
	}

	fseek(Output, 0, SEEK_END);

	for (PageID=0; PageID<Template.CommonStaticObjectCount; PageID++) {
	    StaticObjectOffsets[PageID] = ftell(Output);
	
	    sprintf(Workfile, "%s/%s/%s", BaseDir, TemplateFilename, StaticObjects[PageID].File);

	    if (ReadPageFile(&Page, Workfile, StaticObjects[PageID].File, StaticObjects[PageID].Name, &Size, &modTime)) {
		if (!StoreObject(Output, NULL, StaticObjects[PageID].File, StaticObjects[PageID].MIMEType, modTime, Page, Size)) {
		    LogPrintf("\r Error: Could not write to output file %s\n", Targetfile);
		    return(FALSE);
		}
	    }

	    if (Page) {
		free(Page);
	    }
	}

	/* Now go back and write the Fixup table for the templates */
	fseek(Output, StaticObjectFixupFileOffset, SEEK_SET);
	for (PageID=0; PageID<Template.CommonStaticObjectCount; PageID++) {
		fwrite(&(StaticObjectOffsets[PageID]), sizeof(unsigned long), 1, Output);
	}
	fclose(Output);
	free(StaticObjectOffsets);
	return(TRUE);
}


BOOL
GenerateCompilerSupport(void)
{
	FILE				*StringFile;
	unsigned char	Path[XPL_MAX_PATH];
	unsigned long	i;
	
	if (Verbose) {
		LogPrintf("\rParsing & compiling template files\n");
	} else {
		LogPrintf("\rPhase VIII: Compiling template files\n");
	}

	if (TargetDir[0]!='\0') {
		sprintf(Path, "%s/%s.h", TargetDir, TemplateFilename);
	} else {
		sprintf(Path, "%s.h", TemplateFilename);
	}

	StringFile=fopen(Path, "wb");
	if (!StringFile) {
		LogPrintf("Couldn't create support file %s\n", Path);
		return(FALSE);
	}

	fprintf(StringFile, "/*\n");
	fprintf(StringFile, "\tDo not edit this file. It has been generated based\n");
	fprintf(StringFile, "\ton information read from the %s template.\n", TemplateFilename);
	fprintf(StringFile, "\tThis file is only for inclusion in MW & WA Agents\n");
	fprintf(StringFile, "*/\n\n");

	for (i=0; i<LocalizedStringCount; i++) {
		fprintf(StringFile, "#define\tSTRING_%-31s\t%05d\n", Strings[i].Name, Strings[i].Number);
	}

	fclose(StringFile);
	return(TRUE);
}


BOOL
DumpUnusedCommonImages(void)
{
	unsigned long	i;

	LogPrintf("\nList of unreferenced common images:\n");
	for (i=0; i<CommonImageCount; i++) {
		if (CommonImages[i].UsedCount==0) {
			LogPrintf("%s %s\n", CommonImages[i].Name, CommonImages[i].File);
		}
	}
	return(TRUE);
}


BOOL
DumpUnusedImages(void)
{
	unsigned long	i;

	LogPrintf("\nList of unreferenced images:\n");
	for (i=0; i<LocalizedImageCount; i++) {
		if (Images[i].UsedCount==0) {
			LogPrintf("%s %s\n", Images[i].Name, Images[i].File);
		}
	}
	return(TRUE);
}


BOOL
DumpUnusedStrings(void)
{
	unsigned long	i;

	LogPrintf("\nList of unreferenced strings:\n");
	for (i=0; i<LocalizedStringCount; i++) {
		if (Strings[i].UsedCount==0) {
			LogPrintf("[#%d] %s\n", i+1, Strings[i].Name);
		}
	}
	return(TRUE);
}


BOOL
DumpUnusedCommonStrings(void)
{
	unsigned long	i;

	LogPrintf("\nList of unreferenced common strings:\n");
	for (i=0; i<CommonStringCount; i++) {
		if (CommonStrings[i].UsedCount==0) {
			LogPrintf("[#%d] %s\n", i+1, CommonStrings[i].Name);
		}
	}
	return(TRUE);
}


BOOL
DumpUnusedTemplates(void)
{
	unsigned long	i;

	LogPrintf("\nList of unreferenced templates:\n");
	for (i=0; i<Template.TemplateCount; i++) {
		if (DynamicObjects[i].UsedCount==0) {
			LogPrintf("%s\n", DynamicObjects[i].Name);
		}
	}
	return(TRUE);
}


BOOL
FreeAllData(void)
{
	unsigned long	i;
	unsigned long	j;

	if (Verbose) {
		LogPrintf("\rFreeing all data\n");
	} else {
		LogPrintf("\rPhase IX: Cleaning up\n");
	}

	/* Clean up behind us */
	if (Template.LanguageCount>0) {
		for (i=0; i<Template.LanguageCount; i++) {
			free(Languages[i].Name);
			for (j=0; j<Languages[i].FileCount; j++) {
				free(Languages[i].File[j]);
			}
		}
		free(Languages);
	}

	if (Template.NameCount>0) {
		for (i=0; i<Template.NameCount; i++) {
			free(NamedDynamicObjects[i].Name);
			free(NamedDynamicObjects[i].ObjectName);
		}
		free(NamedDynamicObjects);
	}

	if (LocalizedStringCount>0) {
		for (i=0; i<LocalizedStringCount; i++) {
			free(Strings[i].Name);
		}
		free(Strings);
	}
 	if (LocalizedImageCount>0) {
		for (i=0; i<LocalizedImageCount; i++) {
			free(Images[i].Name);
			free(Images[i].File);
		}
		free(Images);
	}
 	if (CommonStringCount>0) {
		for (i=0; i<CommonStringCount; i++) {
			free(CommonStrings[i].Name);
		}
		free(CommonStrings);
	}
 	if (CommonImageCount>0) {
		for (i=0; i<CommonImageCount; i++) {
			free(CommonImages[i].Name);
			free(CommonImages[i].File);
		}
		free(CommonImages);
	}
	if (Template.TemplateCount>0) {
		for (i=0; i<Template.TemplateCount; i++) {
			free(DynamicObjects[i].Name);
			free(DynamicObjects[i].File);
			free(DynamicObjects[i].MIMEType);
		}
		free(DynamicObjects);
	}

	if (Template.CommonStaticObjectCount>0) {
		for (i=0; i<Template.CommonStaticObjectCount; i++) {
			free(StaticObjects[i].Name);
			free(StaticObjects[i].File);
			free(StaticObjects[i].MIMEType);
			free(StaticObjects[i].Data);
		}
		free(StaticObjects);
	}

	for (i=0; i<Template.LanguageCount; i++) {
		if (StringData) {
			if (StringData[i]) {
				for (j=0; j<LocalizedStringCount; j++) {
					free(StringData[i][j]);
				}
				free(StringData[i]);
			}
		}
		if (ImageData) {
			if (ImageData[i]) {
				for (j=0; j<LocalizedImageCount; j++) {
					free(ImageData[i][j]);
				}
				free(ImageData[i]);
			}
		}
		if (ImageLength) {
			if (ImageLength[i]) {
				free(ImageLength[i]);
			}
		}
	}
	if (StringData) {
		free(StringData);
	}
	if (ImageData) {
		free(ImageData);
	}
	if (ImageLength) {
		free(ImageLength);
	}

	if (CommonStringData) {
		if (CommonStringCount>0) {
			for (i=0; i<CommonStringCount; i++) {
				free(CommonStringData[i]);
			}
		}
		free(CommonStringData);
	}

	if (CommonImageData) {
		if (CommonImageCount>0) {
			for (i=0; i<CommonImageCount; i++) {
				free(CommonImageData[i]);
			}
		}
		free(CommonImageData);
	}
	if (CommonImageLength) {
		free(CommonImageLength);
	}

	if (InitialHTMLPage) {
		free(InitialHTMLPage);
	}
	if (InitialWMLPage) {
		free(InitialWMLPage);
	}
	if (TimeOutPage) {
		free(TimeOutPage);
	}
	if (ResumePage) {
		free(ResumePage);
	}
	if (DynamicObjectOffsets) {
		free(DynamicObjectOffsets);
	}

	if (Tokens) {
		for (i=0; i<TokenCount; i++) {
			free(Tokens[i].Keyword);
		}
		free(Tokens);
	}

	if (TokenDefCount>0) {
		for (i=0; i<TokenDefCount; i++) {
			free(TokenDefs[i].Name);
		}
		free(TokenDefs);
	}

	if (AssociateDefCount>0) {
		for (i=0; i<AssociateDefCount; i++) {
			free(AssociateDefs[i].Name);
		}
		free(AssociateDefs);
	}

	if (IncludeDirCount>0) {
		for (i=0; i<IncludeDirCount; i++) {
			free(IncludeDirs[i]);
		}
		free(IncludeDirs);
	}

	if (DefineCount>0) {
		for (i=0; i<DefineCount; i++) {
			free(Defines[i]);
		}
		free(Defines);
	}

	if (CommonStringInfo.FileCount>0) {
		for (i=0; i<CommonStringInfo.FileCount; i++) {
			free(CommonStringInfo.File[i]);
		}
		free(CommonStringInfo.Name);
	}
	return(TRUE);
}

int
main(int argc, char *argv[])
{
	int	Screen;

	SetCurrentNameSpace(NWOS2_NAME_SPACE);

	if (argc<2) {
		goto Usage;
	}

	/* Init Globals */
	Template.VersionID = TEMPLATE_CURRENT_VERSION;
	strncpy(Template.Name, "template", TEMPLATE_NAME_SIZE);
	Template.Name[TEMPLATE_NAME_SIZE] = '\0';
	Template.LanguageCount = 0;
	Template.TemplateCount = 0;
	Template.NameCount = 0;
	Template.CommonStaticObjectCount = 0;
	Template.SourcePath[0] = '\0';
	/* Parse any arguments */
	if (argc>1) {
		int	i;

		for (i=1; i<argc; i++)  {
			if (argv[i][0]!='-') {
				strcpy(TemplateFilename, argv[i]);
				strncpy(Template.Name, argv[i], TEMPLATE_NAME_SIZE);
				Template.Name[TEMPLATE_NAME_SIZE] = '\0';
			} else if (XplStrCaseCmp(argv[i],"-v")==0) {
				XplConsolePrintf("\r\n\n\n%s [%s]\n",PRODUCT_NAME, PRODUCT_VERSION);
				XplConsolePrintf("\r  Author:      peter dennis bartok [peter@novonyx.com]\n\n");

				return(0);
			}

			if ((XplStrCaseCmp(argv[i], "-?")==0) || (XplStrCaseCmp(argv[i], "-h")==0)) {
Usage:
				XplConsolePrintf("\r\n%s usage:\n", PRODUCT_NAME);
				XplConsolePrintf("\rMWCOMP Template [-b:<basedir>] [-t:<targetdir>] [-v] [-h | -?]\n");
				XplConsolePrintf("\r-b:<basedir>  - Base directory for templates (default:%s)\n", BaseDir);
				XplConsolePrintf("\r-d            - Display detailed progress information\n");
				XplConsolePrintf("\r-f            - Force overwrite of existing template\n");
				XplConsolePrintf("\r-g            - Generate template-specific strings header\n");
				XplConsolePrintf("\r-i:<incdir>   - Add include directory <incdir>\n");
				XplConsolePrintf("\r-l:<log>      - Set logfile pathname (default:%s)\n", LogFilename);
				XplConsolePrintf("\r-t:<targetd>  - Output directory for templates (default:%s)\n", TargetDir);
				XplConsolePrintf("\r-y:<type>     - Template type (0=modweb, 1=webadm) (default:modweb)\n");
				XplConsolePrintf("\r-u            - Log unused template components\n");
				XplConsolePrintf("\r-v            - Display version information\n");
				XplConsolePrintf("\r-w[x]         - Warning level, [0=no warnings, 1=warnings]\n");
				XplConsolePrintf("\r-? | -h       - This help screen\n\n");
				return(0);
			}

			if (XplStrNCaseCmp(argv[i], "-b:", 3)==0) {
				strcpy(BaseDir, argv[i]+3);
				continue;
			}

			if (XplStrNCaseCmp(argv[i], "-y:", 3)==0) {
				TemplateType=atol(argv[i]+3);
				if (TemplateType>1) {
					XplConsolePrintf("MWCOMP: Illegal templatetype (%s) specified\n", argv[i]);

					return(0);
				}
				continue;
			}

			if (XplStrNCaseCmp(argv[i], "-t:", 3)==0) {
				strcpy(TargetDir, argv[i]+3);
				continue;
			}

			if (XplStrNCaseCmp(argv[i], "-l:", 3)==0) {
				strcpy(LogFilename, argv[i]+3);
				continue;
			}

			if (XplStrNCaseCmp(argv[i], "-i:", 3)==0) {
				AddIncludeDir("commandline", 0, argv[i]+3);
				continue;
			}

			if (XplStrNCaseCmp(argv[i], "-d:", 3)==0) {
				AddDefine("commandline", 0, argv[i]+3);
				continue;
			}

			if (XplStrCaseCmp(argv[i], "-d")==0) {
				Verbose++;
				continue;
			}

			if (XplStrCaseCmp(argv[i], "-g")==0) {
				NeedCompilerSupport=TRUE;
				continue;
			}

			if (XplStrCaseCmp(argv[i], "-w0")==0) {
				Warnings=FALSE;
				continue;
			}

			if (XplStrCaseCmp(argv[i], "-f")==0) {
				ForceOverwrite=TRUE;
				continue;
			}

			if (XplStrCaseCmp(argv[i], "-u")==0) {
				ShowUnused=TRUE;
				continue;
			}

			if (XplStrNCaseCmp(argv[i], "-b:", 3)==0) {
				strcpy(BaseDir, argv[i]+3);
				continue;
			}
		}
	}

#if !defined(LIBC) /* fixme - NetWare requirement */
	Screen=XplCreateScreen("Modular WebAgent Compiler (MWComp)", 0);
	XplDisplayScreen(Screen);
	XplSetCurrentScreen(Screen);
#endif

	/* Set some reasonable defaults */
	if (IncludeDirs==0) {
		AddIncludeDir(__FILE__, __LINE__, DEFAULT_INCLUDE_DIR);
	}

	/*
		1) Parse .DEF file
		2) Locate all related images and build fixup table
		3) Locate all language strings files and build fixup table
		4) Locate all template files
		4) Parse template(s) and generate bytecode
	*/

	VerboseFile=fopen(LogFilename, "wb");
	if (Verbose) {
		if (BaseDir[0]!='\0') {
			LogPrintf("Parsing definition file %s/%s/template.def\n", BaseDir, TemplateFilename);
		} else {
			LogPrintf("Parsing definition file %s\n", TemplateFilename);
		}
	}

	if (TargetDir[0]!='\0') {
		sprintf(Targetfile, "%s/%s.%s", TargetDir, TemplateFilename, TemplateExtension[TemplateType].Extension);
	} else {
		sprintf(Targetfile, "%s.%s", TemplateFilename, TemplateExtension[TemplateType].Extension);
	}
	if ((access(Targetfile, 0)==0) && !ForceOverwrite) {
		LogPrintf("\rTargetfile %s exists, either use -f switch or rename template\n", Targetfile);

		return(-2);
	}

	if (!ParseDefinition()) goto Abort;
	if (!CheckDefinition()) goto Abort;
	if (!LoadLanguageData()) goto Abort;
	if (!CompileNames()) goto Abort;

	if (!FillOutTemplateStruct(&Template)) goto Abort;
	if (!StoreStaticData()) goto Abort;
	if (!CompileTemplates()) goto Abort;
	if (!StoreStaticObjects()) goto Abort;
	if (NeedCompilerSupport && !GenerateCompilerSupport()) goto Abort;

	if (ShowUnused) {
		DumpUnusedCommonImages();
		DumpUnusedImages();
		DumpUnusedCommonStrings();
		DumpUnusedStrings();
		DumpUnusedTemplates();
	}

	FreeAllData();
	LogPrintf("\nTemplate successfully compiled into %s\n", Targetfile);
	if (VerboseFile) {
		fclose(VerboseFile);
		if (!Verbose && !ShowUnused) {
			unlink(LogFilename);
		}
	}

	return(0);

Abort:
	FreeAllData();
	/* Remove the partial compiled file */
	unlink(Targetfile);
	if (VerboseFile) {
		LogPrintf("\rCompilation aborted due to errors, check %s for details\n", LogFilename);
	} else {
		LogPrintf("\rCompilation aborted due to errors, log could not be written\n");
	}
	if (VerboseFile) {
		fclose(VerboseFile);
	}
	return(-1);
}
