/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>

#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */

#include "mwcal.h"


/* Modified Base64 encoding/decoding tables; used in mUTF-7  */
static unsigned char MBASE64[] = {"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+,"};
static unsigned char MBASE64_R[256] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x3E, 0x3F, 0xFF, 0xFF, 0xFF,
	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
	0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};


__inline
static void Base64EncodeUnit(unsigned char *dest, unsigned char b64_buf[3], unsigned char table[]) {
	dest[0] = table[((b64_buf[0] & 0xFC) >> 2)];
	dest[1] = table[((b64_buf[0] & 0x03) << 4) | ((b64_buf[1] & 0xF0) >> 4)];
	dest[2] = table[((b64_buf[1] & 0x0F) << 2) | ((b64_buf[2] & 0xC0) >> 6)];
	dest[3] = table[((b64_buf[2] & 0x3F) << 0)];
}

static int
UTF8EncodeChar(unicode unichar, unsigned char *dest)
{
	if (unichar < 0x0080) {					/* 0000-007F */
		dest[0] = (char)(unichar);
		return(1);
	}
	if (unichar < 0x0800) {					/* 0080-07FF */
		dest[0] = (char)(0xC0 | ((unichar & 0x07C0) >> 6));
		dest[1] = (char)(0x80 | (unichar & 0x003F));
		return(2);
	}
														/* 0800-FFFF */
	dest[0] = (char)(0xE0 | ((unichar & 0xF000) >> 12));
	dest[1] = (char)(0x80 | ((unichar & 0x0FC0) >> 6));
	dest[2] = (char)(0x80 | (unichar & 0x003F));
	return(3);	
}

/* This function only handles UCS-2 */
static int
UTF8DecodeChar(const unsigned char *src, unicode *uchar)
{
	if (src[0] <= 0x7F) {			/* 0000-007F: one byte (0xxxxxxx) */
		*uchar = (unicode)src[0];
		return(1);
	}
	if (src[0] <= 0xDF) {			/* 0080-07FF: two bytes (110xxxxx 10xxxxxx) */
		*uchar = ((((unicode)src[0]) & 0x001F) << 6) |
					((((unicode)src[1]) & 0x003F) << 0);
		return(2);
	}
										/* 0800-FFFF: three bytes (1110xxxx 10xxxxxx 10xxxxxx) */
	*uchar = ((((unicode)src[0]) & 0x000F) << 12) |
				((((unicode)src[1]) & 0x003F) << 6) |
				((((unicode)src[2]) & 0x003F) << 0);
	return(3);
}

/*
	This function will translate an NMAP calendar name into UTF8;
	it will return the length of the translated string and it
	will NULL-terminate the UTF8 string
*/

int
CalendarUTF7toUTF8(const unsigned char *UTF7, unsigned char *UTF8)
{
	unsigned long	Src;
	unsigned long	Dst;
	BOOL				Done=FALSE;
	unsigned long	i;
	unsigned long	UDst;
	unsigned char	Unicode[6];
	unsigned char	Base64[4];

	Src=0;
	Dst=0;

	ModDebug("CalendarUTF7toUTF8(): Decoding %s\n", UTF7);

	while (!Done) {
		switch(UTF7[Src]) {
			case 0x7f: {
				UTF8[Dst]=' ';
				Dst++;
				Src++;
				break;
			}

			case '&': {
				Src++;
				if (UTF7[Src]!='-') {

					UDst=0;

					do {
						/* We get four encoded chars, translate them */
						for (i=0; i<4; i++) {
							if ((UTF7[Src]!='\0') && (UTF7[Src]!='-')) {
								Base64[i]=UTF7[Src++];
							} else {
								Base64[i]='A';
							}
						}

						/* We've got the encoded char in Base64, padded if neccessary, now translate to unicode */
						i=0;
						Unicode[UDst++] = ((MBASE64_R[Base64[0]] << 2) & 0xFC) | ((MBASE64_R[Base64[1]] >> 4) & 0x03);
						Unicode[UDst++] = ((MBASE64_R[Base64[1]] << 4) & 0xF0) | ((MBASE64_R[Base64[2]] >> 2) & 0x0F);
						Unicode[UDst++] = ((MBASE64_R[Base64[2]] << 6) & 0xC0) | ((MBASE64_R[Base64[3]] >> 0) & 0x3F);

						/* Convert to UTF-8 every 6 bytes (= 3 Unicode characters, = 2 loop iterations) */
						if (UDst==6) {
							for (i = 0; i < 6; i += 2) {
								if (((Unicode[i] << 8) | Unicode[i + 1]) != 0) {	/* May have padding */
									Dst += UTF8EncodeChar((unicode )(Unicode[i] << 8) | Unicode[i + 1], UTF8 + Dst);
								}
							}
							UDst=0;
						}						
					} while (UTF7[Src]!='\0' && UTF7[Src]!='-');

					if (UTF7[Src]=='-') {
						/* Whack the terminator */
						Src++;
					}

					if (UDst==3) {
						Dst += UTF8EncodeChar((unicode )(Unicode[0] << 8) | Unicode[1], UTF8 + Dst);
					}

				} else {
					UTF8[Dst]='&';
					Dst++;
					Src++;
				}

				break;
			}

			case '\0': {
				UTF8[Dst]='\0';
				Done=TRUE;
				break;
			}

			default: {
				UTF8[Dst++]=UTF7[Src++];
				break;
			}
		}
	}

	return(Dst);
}

/* Convert a UTF-8 encoded calendar name into the form used by NMAP. This form is modified UTF-7 as
 * described in IMAP (RFC 2060) with the further modification that spaces (0x20) are represented with 0x7F. */
int
CalendarUTF8toUTF7(const unsigned char *UTF8, int UTF8Len, unsigned char *UTF7, int UTF7Len)
{
	int				Src;										/* Index into UTF8 array												*/
	int				Dest = 0;								/* Index into UTF7 array												*/
	unicode			UniChar;									/* Current character in UTF-16										*/
	int				UTF8Size;								/* Size in bytes of last read UTF-8 character					*/
	int				EncodeStart = -1;						/* Index into UTF8 of first character in current sequence	*/
	int				Encode;									/* Index into UTF8 array used when encoding						*/
	unicode			EncodeUniChar;							/* Another UTF-16 character, used in encoding					*/
	int				UTF8SizeEncoded = -1;				/* Size in bytes of UTF-8 character currently being encoded */
	unsigned char	Base64Buffer[3];						/* Holds one 24-bit encoding unit									*/
	unsigned char	Base64Index;
	int				len;

	for (Src=0; Src<UTF8Len; Src+=UTF8Size) {
		UTF8Size = UTF8DecodeChar(UTF8+Src, &UniChar);	/* Assume we won't overflow... */

		/* 
			No encoding is needed for characters in the range 0x20-0x7e, except for 0x20 which
			must be encoded as 0x7f for NMAP, and 0x26 which gets encoded as "&-". 
		*/

		if ((UniChar >= 0x0020) && (UniChar <= 0x007e)) {
			/* Flush any pending encode sequence */
			if (EncodeStart>=0) {
				/* Check for overflow */
				len = Src-EncodeStart;					/* Number of bytes to be encoded */
				if (len % 3) {
					len+=3-(len % 3);						/* Round up to a multiple of three bytes */
				}
				len = ((len * 4) / 3) + 2;				/* Length of encoded data with escape characters */

				if((UTF7Len - Dest) < len) {
					return(Dest);							/* Overflow */
				}

				/* Encode the sequence */
				UTF7[Dest++]='&';
				Base64Index=0;
				len = 0;

				for (Encode=EncodeStart, len=0; Encode<Src; len++) {
					/* Add the next byte to the base64 encoding buffer */
					if (len % 2) {
						Base64Buffer[Base64Index++] = (unsigned char)(EncodeUniChar & 0x00FF);			/* Second byte of each UTF-16 character */
						Encode += UTF8SizeEncoded;
					} else {
						UTF8SizeEncoded = UTF8DecodeChar(UTF8+Encode, &EncodeUniChar);
						Base64Buffer[Base64Index++] = (unsigned char)((EncodeUniChar & 0xFF00) >> 8);	/* First byte of each UTF-16 character */
					}

					/* Encode the buffer every three bytes */
					if (Base64Index == 3) {	
						Base64EncodeUnit(UTF7 + Dest, Base64Buffer, MBASE64);
						Dest += 4;
						Base64Index = 0;
					}
				}

				if (Base64Index != 0) {														/* Partial encoding unit */
					for (; Base64Index<3; Base64Index++) {								/* Zero out remaining space */
						Base64Buffer[Base64Index] = 0;
					}
					Base64EncodeUnit(UTF7 + Dest, Base64Buffer, MBASE64);
					Dest += 4;
				}

				/* Terminate the encoded sequence */
				if (UTF7[Dest - 1] == MBASE64[0]) {		/* Strip off trailing 0 bytes */
					Dest--;
				}
				UTF7[Dest++] = '-';
				EncodeStart = -1;
			}

			len = UTF7Len - Dest;
			if(UniChar == 0x0020) {										/* Handle 0x20 ( ) special case */
				if (len==0) {
					 return(Dest);											/* Overflow */
				}
				UTF7[Dest++] = 0x7f;
			}  else if(UniChar == 0x0026) {							/* Handle 0x26 (&) special case */
				if(len < 2) return(Dest);								/* Overflow */
				UTF7[Dest++] = '&';
				UTF7[Dest++] = '-';
			} else if ((UniChar == '\\' || UniChar == '/')) {	/* Convert backslashes into slashes */
				if (Dest != 0) {
					if(len == 0) return(Dest);							/* Overflow */
					UTF7[Dest++] = '/';
				}
			} else { 														/* Generic case: just pass the character through */
				if (len == 0) {
					return(Dest);											/* Overflow */
				}
				UTF7[Dest++] = (unsigned char)UniChar;
			}

		/* All characters outside this range must be encoded */
		} else {
			if (EncodeStart < 0) {
				EncodeStart = Src;
			}
		}

		Encode += UTF8Size;
	}

	/* Check to see if we ended inside an encode sequence */
	if (EncodeStart >= 0) {
		/* Check for overflow */
		len = Src - EncodeStart;				/* Number of bytes to be encoded */
		if (len % 3) {
			len += 3 - (len % 3);				/* Round up to a multiple of three bytes */
		}

		len = ((len * 4) / 3) + 2;				/* Length of encoded data with escape characters */

		if((UTF7Len - Dest) < len) {
			return(Dest);							/* Overflow */
		}

		/* Encode the sequence */
		UTF7[Dest++] = '&';
		Base64Index = 0;
		len = 0;

		for (Encode=EncodeStart, len=0; Encode<Src; len++) {
			/* Add the next byte to the base64 encoding buffer */
			if (len % 2) {
				Base64Buffer[Base64Index++] = (unsigned char)(EncodeUniChar & 0x00FF);			/* Second byte of each UTF-16 character */
				Encode += UTF8SizeEncoded;
			} else {
				UTF8SizeEncoded = UTF8DecodeChar(UTF8 + Encode, &EncodeUniChar);
				Base64Buffer[Base64Index++] = (unsigned char)((EncodeUniChar & 0xFF00) >> 8);	/* First byte of each UTF-16 character */
			}

			/* Encode the buffer every three bytes */
			if (Base64Index == 3) {	
				Base64EncodeUnit(UTF7 + Dest, Base64Buffer, MBASE64);
				Dest += 4;
				Base64Index = 0;
			}
		}

		if (Base64Index != 0) {									/* Partial encoding unit */
			for (; Base64Index<3; Base64Index++) {		/* Zero out remaining space */
				Base64Buffer[Base64Index] = 0;
			}
			Base64EncodeUnit(UTF7 + Dest, Base64Buffer, MBASE64);
			Dest += 4;
		}

		/* Terminate the encoded sequence */
		if (UTF7[Dest - 1] == MBASE64[0]) {			/* Strip off trailing 0 bytes */
			Dest--;
		}
		UTF7[Dest++] = '-';
		EncodeStart = -1;
	}

	return(Dest);
}
