/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>
#include <hulautil.h>
#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */

#include "mwmail.h"

#define LOTUS_CALENDAR_FIX		1

#define ParseMIMELine(MIME, Type, Charset, Encoding, Name, StartPos, Size, HeaderSize)						\
{																																		\
	unsigned char	*p, *p2;																										\
																																		\
	/* Type */																														\
	p2=strchr((MIME), ' ');																										\
	if (p2) {																														\
		*p2='\0';																													\
		HulaStrNCpy(Type, (MIME), MIME_TYPE_LEN+MIME_SUBTYPE_LEN+1);																									\
		p=strchr(++p2, ' ');																										\
		if (p) {																														\
			*p='\0';																													\
			strcat(Type, "/");																									\
			strcat(Type, p2);																										\
			p2=strchr(++p, ' ');																									\
			if (p2) {																												\
				*p2='\0';																											\
				HulaStrNCpy(Charset, p, MIME_CHARSET_LEN+1);																								\
				p=strchr(++p2, '"');																								\
				if (p) {																												\
					*(p-1)='\0';																									\
					HulaStrNCpy(Encoding, p2, MIME_ENCODING_LEN + 1);																						\
					p2=strchr(++p, '"');																							\
					if (p2) {																										\
						*(p2++)='\0';																								\
						HulaStrNCpy(Name, p, MIME_NAME_LEN + 1);																							\
						p=strchr(++p2, ' ');																						\
						if (p) {																										\
							*p='\0';																									\
							StartPos=atol(p2);																					\
							p2=strchr(++p, ' ');																					\
							if (p2) {																								\
								*p2='\0';																							\
								Size=atol(p);																						\
								p=strchr(++p2, ' ');																				\
								if (p) {																								\
									*p='\0';																							\
									HeaderSize=atol(p2);																			\
								}																										\
							}																											\
						}																												\
					}																													\
				}																														\
			}																															\
		}																																\
	}																																	\
}

unsigned long
MwMailPrintFormatted(unsigned long Number, unsigned char *Buffer, int bufLen)
{
	if (Number<1024) {
		return(snprintf(Buffer, bufLen, "%d Bytes", (int)Number));
	} else if (Number<1024*1024) {
		return(snprintf(Buffer, bufLen, "%dKB", (int)(Number/1024)));
	} else {
		return(snprintf(Buffer, bufLen, "%dMB", (int)(Number/(1024*1024))));
	}
}


BOOL

MwMailSetMessageFlags(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long MessageID, unsigned long Flag)
{
	unsigned char	buffer[128];
	unsigned long	LastMessage = 0;

	if (!Session->ReadOnly) {
		switch (Flag) {
			case FLAG_READ: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_READ));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_DELETE: {
				if (!MailSession->PurgeDefault) {
					MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_DELETED));
					MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
				} else {
					MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PURGED));
					MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
					Session->NMAPChanged = TRUE;   /* We need to reset the msglist because a PURGE has just occured! */
				}

				break;
			}

			case FLAG_NORMPRIO: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOHIGH));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOLOW));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_LOWPRIO: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOHIGH));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOLOW));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}


			case FLAG_HIPRIO: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOLOW));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOHIGH));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_RECENT: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_RECENT));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_PRIVATE: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIVATE));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}
		}
	}

	if (MessageID > Session->NumOfMessages) {
		/* FIX ME error message reporting */

		return(FALSE);
	}

	/* Do we need to update the message chache? */
	if (MailSession->CachedMessageID == MessageID) {
		LastMessage = MailSession->CachedMessageID;

		MailSession->CachedMessageState = atol(buffer);
	}

	/* We need to let the cache now that things have changed */
	MwMailRefreshFolder(Session, MailSession);

	/* Did the folder refresh unselect our message? */
	if ((MailSession->CachedMessageID == 0) && (LastMessage != 0)) {
		MwMailLoadMessage(Client, LastMessage, Session, MailSession);
	}

	MDBFreeValues(MailSession->Value);
	return(TRUE);
}

BOOL
MwMailUnSetMessageFlags(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long MessageID, unsigned long Flag)
{
	unsigned char	buffer[128];
	unsigned long	LastMessage = 0;

	if (!Session->ReadOnly) {
		switch (Flag) {
			case FLAG_READ: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_READ));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_DELETE: {
				if (!MailSession->PurgeDefault) {
					MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_DELETED));
					MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
				} else {
					MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PURGED));
					MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);
					Session->NMAPChanged = TRUE;   /* We need to reset the msglist because a UNPURGE has just occured! */
				}

				break;
			}

			case FLAG_NORMPRIO:
			case FLAG_LOWPRIO:
			case FLAG_HIPRIO: {
				/* If you want to unset any priority we are just going to reset it to the default */

				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOHIGH));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIOLOW));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_RECENT: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_RECENT));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}

			case FLAG_PRIVATE: {
				MWSendNMAPServer(Session, buffer, snprintf(buffer, sizeof(buffer), "DFLG %lu %lu\r\n", MailSession->IDList[MessageID - 1], (long unsigned int)MSG_STATE_PRIVATE));
				MWGetNMAPAnswer(Session, buffer, sizeof(buffer), TRUE);

				break;
			}
		}
	}

	if (MessageID > Session->NumOfMessages) {
		/* FIX ME error message reporting */
		return(FALSE);
	}

	/* Do we need to update the message chache? */
	if (MailSession->CachedMessageID == MessageID) {
		LastMessage = MailSession->CachedMessageID;

		MailSession->CachedMessageState = atol(buffer);
	}

	/* We need to let the cache now that things have changed */
	MwMailRefreshFolder(Session, MailSession);

	/* Did the folder refresh unselect our message? */
	if ((MailSession->CachedMessageID == 0) && (LastMessage != 0)) {
		MwMailLoadMessage(Client, LastMessage, Session, MailSession);
	}

	MDBFreeValues(MailSession->Value);
	return(TRUE);
}

#define	MIME_REALLOC_SIZE		20
static BOOL
AddMIMECacheEntry(SessionStruct *Session, MailSessionStruct *MailSession, unsigned char *Type, unsigned char *Charset, unsigned char *Encoding, unsigned char *Name, unsigned long StartPos, unsigned long Size, unsigned long HeaderSize, unsigned long Depth)
{
	unsigned long	Used=MailSession->MIMEUsed;

	if (Used+1>=MailSession->MIMEAllocated) {
		MailSession->MIME=MemRealloc(MailSession->MIME, sizeof(MIMEStruct)*(MIME_REALLOC_SIZE+MailSession->MIMEAllocated));
		if (!MailSession->MIME) {
			return(FALSE);
		}
		memset(MailSession->MIME+MailSession->MIMEAllocated, 0, sizeof(MIMEStruct)*MIME_REALLOC_SIZE);
		MailSession->MIMEAllocated+=MIME_REALLOC_SIZE;
	}

	MailSession->MIME[Used].Type=MemStrdup(Type);
	if (Charset[0]=='-' || MWQuickCmp(Charset, "US-ASCII")) {
		/* Use user's default charset instead of US-ASCII; fix for hotmail and other systems generating broken headers */
		MailSession->MIME[Used].Charset=MemStrdup(Session->Charset);
	} else {
		MailSession->MIME[Used].Charset=MemStrdup(Charset);
	}
	MailSession->MIME[Used].Encoding=MemStrdup(Encoding);
	MailSession->MIME[Used].FileName=MemStrdup(Name);
	MailSession->MIME[Used].PartStart=StartPos;
	MailSession->MIME[Used].PartSize=Size;
	MailSession->MIME[Used].HeaderSize=HeaderSize;
	MailSession->MIME[Used].Depth=Depth;
	MailSession->MIMEUsed++;
	return(TRUE);
}


BOOL
MwMailClearMIMECache(MailSessionStruct *MailSession)
{
	unsigned long	i;

	if (!MailSession->MIME) {
		MailSession->MIMEUsed=0;
		MailSession->MIMEAllocated=0;
		return(TRUE);
	}

	for (i=0; i<MailSession->MIMEUsed; i++) {
		if (MailSession->MIME[i].Type) {
			MemFree(MailSession->MIME[i].Type);
		}
		if (MailSession->MIME[i].Charset) {
			MemFree(MailSession->MIME[i].Charset);
		}
		if (MailSession->MIME[i].Encoding) {
			MemFree(MailSession->MIME[i].Encoding);
		}
		if (MailSession->MIME[i].FileName) {
			MemFree(MailSession->MIME[i].FileName);
		}
	}

	MemFree(MailSession->MIME);
	MailSession->MIME=NULL;
	MailSession->MIMEUsed=0;
	MailSession->MIMEAllocated=0;
	MailSession->InAttachmentList=FALSE;

	return(TRUE);
}

BOOL
MwMailClearMIMECacheData(MailSessionStruct *MailSession)
{
	unsigned long	i;

	if (!MailSession->MIME) {
		MailSession->MIMEUsed=0;
		MailSession->MIMEAllocated=0;
		return(TRUE);
	}

	for (i=0; i<MailSession->MIMEUsed; i++) {
		if (MailSession->MIME[i].Type) {
			MemFree(MailSession->MIME[i].Type);
			MailSession->MIME[i].Type = NULL;
		}
		if (MailSession->MIME[i].Charset) {
			MemFree(MailSession->MIME[i].Charset);
			MailSession->MIME[i].Charset = NULL;
		}
		if (MailSession->MIME[i].Encoding) {
			MemFree(MailSession->MIME[i].Encoding);
			MailSession->MIME[i].Encoding = NULL;
		}
		if (MailSession->MIME[i].FileName) {
			MemFree(MailSession->MIME[i].FileName);
			MailSession->MIME[i].FileName	= NULL;
		}
	}

	MailSession->MIMEUsed = 0;
	MailSession->InAttachmentList = FALSE;

	return(TRUE);
}

BOOL
MwMailLoadMIMECache(unsigned long MessageID, ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession)
{
	unsigned long	ReplyInt;
	unsigned char	Type[MIME_TYPE_LEN+MIME_SUBTYPE_LEN+1];
	unsigned char	Charset[MIME_CHARSET_LEN+1];
	unsigned char	Encoding[MIME_ENCODING_LEN+1];
	unsigned char	Name[MIME_NAME_LEN+1];
	unsigned long	StartPos = -1;
	unsigned long	Size = -1;
	unsigned long	HeaderSize = -1;
	unsigned long	Depth=0;

	/* Is it already cached? */
	if (MessageID==MailSession->CachedMIMEMessageID && MailSession->MIMEUsed>0) {
		ModDebug("MwMailLoadMIMECache(): Message %d already cached\n", (int)MessageID);
		return(TRUE);
	}

	/* Clear the cache; but only the data; keep the structure(s) */
	MwMailClearMIMECacheData(MailSession);

	if (MessageID>Session->NumOfMessages) {
		return(FALSE);
	}

	MailSession->CachedMIMEMessageID=MessageID;

	MessageID--;		/* Our index is 0-based */

	ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "MIME %lu\r\n", MailSession->IDList[MessageID]);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);

	do {
		ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		switch(ReplyInt) {
			case 2002: {
				ParseMIMELine(Client->Temp, Type, Charset, Encoding, Name, StartPos, Size, HeaderSize);
				AddMIMECacheEntry(Session, MailSession, Type, Charset, Encoding, Name, StartPos, Size, HeaderSize, Depth);
				if (toupper(Type[0])=='M' && (MWQuickNCmp(Type, "multipart", 9) || MWQuickNCmp(Type, "message", 7))) {
					Depth++;
				}
				break;
			}

			case 2003:
			case 2004: {
					Depth--;
				break;
			}

			case 1000: {
				return(TRUE);
			}

			default: {


				MwMailClearMIMECacheData(MailSession);
				return(FALSE);
			}
		}
	} while (ReplyInt!=1000);
	

	return(TRUE);
}

static BOOL
LoadRFCHeader(unsigned long MessageID, unsigned long PartID, SessionStruct *Session, MailSessionStruct *MailSession)

{
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;
	StreamStruct	NetworkInCodec;
#ifdef HOTMAIL_CHARSET_FIX
	StreamStruct	*HotmailCodec;
#endif
	StreamStruct	*RFC1522Codec;
	StreamStruct	*MemoryOutCodec;

	/* Is it already cached? */
	if (MessageID==MailSession->CachedMIMEMessageID && PartID==MailSession->CachedPartID && MailSession->CachedPartHeader) {
		ModDebug("LoadRFCHeader(): Message %d part %d already cached\n", (int)MessageID, (int)PartID);
		return(TRUE);
	}

	if (MailSession->CachedPartHeader) {
		MemFree(MailSession->CachedPartHeader);
		MailSession->CachedPartHeader=NULL;
	}

	MailSession->CachedPartID=MessageID;
	if (MessageID>Session->NumOfMessages) {
		return(FALSE);
	}

	MessageID--;		/* Our index is 0-based */

	/* Request the header */
	ReplyInt=snprintf(Answer, sizeof(Answer), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MessageID], MailSession->MIME[PartID].PartStart, MailSession->MIME[PartID].HeaderSize);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);

	if (ReplyInt/10!=202) {
		MailSession->CachedPartHeader=NULL;
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);		
	}

	/* Read & Decode any RFC1522 header stuff */
	memset(&NetworkInCodec, 0, sizeof(StreamStruct));
	RFC1522Codec=MWGetStream(NULL, "RFC1522", FALSE);
#ifdef HOTMAIL_CHARSET_FIX
	HotmailCodec=MWGetStream(NULL, Session->Charset, FALSE);
#endif

	NetworkInCodec.StreamData=Session;
	NetworkInCodec.StreamLength=atoi(Answer);
	NetworkInCodec.Codec=MWStreamFromMWNMAP;
#ifdef HOTMAIL_CHARSET_FIX
	if (HotmailCodec) {
		NetworkInCodec.Next=HotmailCodec;
		HotmailCodec->Next=RFC1522Codec;
	} else 
#endif
	{
		NetworkInCodec.Next=RFC1522Codec;
	}

	MemoryOutCodec=MWGetStream(RFC1522Codec, NULL, FALSE);
	MemoryOutCodec->Codec=MWStreamToMemory;

#ifdef HOTMAIL_CHARSET_FIX
	NetworkInCodec.Codec(&NetworkInCodec, HotmailCodec);
#else
	NetworkInCodec.Codec(&NetworkInCodec, RFC1522Codec);
#endif

	if (MemoryOutCodec->StreamData == NULL) {
		if (NetworkInCodec.StreamLength!=0) {
			MWReleaseStream(RFC1522Codec);
#ifdef HOTMAIL_CHARSET_FIX
			MWReleaseStream(HotmailCodec);
#endif
			MWReleaseStream(MemoryOutCodec);
			return(FALSE);
		} else {
			/* We weren't supposed to read any data, so fake it */
			MemoryOutCodec->StreamData=MemStrdup("  ");
		}
	}

	MailSession->CachedPartHeader=MemoryOutCodec->StreamData;
	MailSession->CachedPartHeader[MemoryOutCodec->StreamLength]='\0';

	MWReleaseStream(RFC1522Codec);
#ifdef HOTMAIL_CHARSET_FIX
	MWReleaseStream(HotmailCodec);
#endif
	MWReleaseStream(MemoryOutCodec);

	/* Grab confirmation of BRAW */
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	if (ReplyInt!=1000) {
		if (MailSession->CachedPartHeader) {
			MemFree(MailSession->CachedPartHeader);
		}
		MailSession->CachedPartHeader=NULL;
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);
	}
	ReplyInt = MwMailMakeRFC822Header(MailSession->CachedPartHeader);

	return(TRUE);
}

#define	ChainBodyPartStreams							\
	if (EncodingCodec && EncodingCodec->Codec) {	\
		NetworkInCodec.Next=EncodingCodec;			\
		EncodingCodec->Next=NetworkOutCodec;		\
	} else {													\
		NetworkInCodec.Next=NetworkOutCodec;		\
	}															\


BOOL
MwMailSendBodyPart(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long MessageID, unsigned long Element)
{
	StreamStruct	NetworkInCodec;
	StreamStruct	*NetworkOutCodec;
	StreamStruct	*EncodingCodec;
	unsigned long	ReplyInt;

	/* First, prep the MIME cache */
	if (!MwMailLoadMIMECache(MessageID, Client, Session, MailSession)) {
		return(FALSE);
	}

   /* Sanity check for users who use multiple windows and don't refresh */
	if ((Element + 1) > MailSession->MIMEUsed) { 
		return(FALSE);
	}

	MessageID--;

	/** Prepare all streams for later use **/
	/* Read from NMAP */
	memset(&NetworkInCodec, 0, sizeof(StreamStruct));
	NetworkInCodec.StreamData=Session;
	NetworkInCodec.Codec=MWStreamFromMWNMAP;

	/* Resolve the encoding of the message part */
	EncodingCodec=MWGetStream(NULL, MailSession->MIME[Element].Encoding, FALSE);

	/* Send the interpreted part to the network */
	NetworkOutCodec=MWGetStream(NULL, NULL, FALSE);
	NetworkOutCodec->StreamData=(void *)Client;
	NetworkOutCodec->Codec=MWStreamToMWClient;

	/* Do the idiot user check */
	if (MessageID>=Session->NumOfMessages) {
		/* Shithead user */
		MWReleaseStream(EncodingCodec);
		MWReleaseStream(NetworkOutCodec);
		return(FALSE);		
	}

	/* Request the body part */
	ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MessageID], MailSession->MIME[Element].PartStart, MailSession->MIME[Element].PartSize);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

	if (ReplyInt/10!=202) {
		MWReleaseStream(EncodingCodec);
		MWReleaseStream(NetworkOutCodec);
		return(FALSE);		
	}

	/* Let the code know how much data to expect */
	NetworkInCodec.StreamLength=atol(Client->Temp);

	/* Prepare the HTTP headers */
	Client->KeepAlive=FALSE;

	MWSendHTTPHeader(Client, MailSession->MIME[Element].Type, 0, (MailSession->MIME[Element].FileName) ? (char *)MailSession->MIME[Element].FileName : "Unnamed", FALSE);

	/* Make sure we're chained up properly, depending on what codecs we have */
	ChainBodyPartStreams;

	Client->NMAPProtected=TRUE;
	NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);

	/* Grab the BRAW DONE code */
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
	Client->NMAPProtected=FALSE;

	MWReleaseStream(EncodingCodec);
	MWReleaseStream(NetworkOutCodec);

	return(TRUE);
}


#define ChainStreams										\
	if (CharsetCodec->Codec) {							\
		if (EncodingCodec->Codec) {					\
			NetworkInCodec.Next=EncodingCodec;		\
			EncodingCodec->Next=CharsetCodec;		\
			CharsetCodec->Next=MIMECodec;				\
		} else {												\
			NetworkInCodec.Next=CharsetCodec;		\
			CharsetCodec->Next=MIMECodec;				\
		}														\
	} else {													\
		if (EncodingCodec->Codec) {					\
			NetworkInCodec.Next=EncodingCodec;		\
			EncodingCodec->Next=MIMECodec;			\
		} else {												\
			NetworkInCodec.Next=MIMECodec;			\
		}														\
	}															\
	MIMECodec->Next=NetworkOutCodec;

#define	ResetStream(Stream)							\
	Stream->Len=0;											\
	Stream->Charset=NULL;								\
	Stream->URL=NULL;										\
	Stream->EOS=0;											\
	Stream->StreamLength=0;

BOOL
MwMailDisplayBody(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long RFCTemplateID, unsigned long ComposeID, unsigned char *LinkTarget, unsigned long MessageID, unsigned long MIMEElement, unsigned long Depth)
{
	StreamStruct	NetworkInCodec;
	StreamStruct	*NetworkOutCodec;
	StreamStruct	*CharsetCodec;
	StreamStruct	*EncodingCodec;
	StreamStruct	*MIMECodec;
	long				Element;
	unsigned long	EndElement;
	unsigned long	ReplyInt;
	unsigned long	i;
	BOOL				HaveCalendar=FALSE;

	/* First, prep the MIME cache */
	if (!MwMailLoadMIMECache(MessageID, Client, Session, MailSession) || MIMEElement>=MailSession->MIMEUsed) {
		return(FALSE);
	}

	MessageID--;

	/* 
		This function will be called for any "message/rfc822" toplevel. 
		This includes the actual message itself, even though it won't
		be marked as message/rfc822.
		We only parse from the specified MIME item, and never go higher 
		than the specifed depth

		We will try to find a handler for every type. If we don't find
		one we will ignore it and move on. we walk from first to last
		item, except if we encounter a multipart/alternative, in which
		case we find the end of the alternative section and walk back
		to the beginning, trying to find a handler.
		
	*/

	/** Prepare all streams for later use **/
	/* Read from NMAP */
	memset(&NetworkInCodec, 0, sizeof(StreamStruct));
	NetworkInCodec.StreamData=Session;
	NetworkInCodec.Codec=MWStreamFromMWNMAP;

	/* Resolve the charset of the message part */
	CharsetCodec=MWGetStream(NULL, NULL, FALSE);

	/* Resolve the encoding of the message part */
	EncodingCodec=MWGetStream(NULL, NULL, FALSE);

	/* Interpret the MIME type of the message part */
	MIMECodec=MWGetStream(NULL, NULL, FALSE);
	MIMECodec->Client=Client;
	
	/* Send the interpreted part to the network */
	NetworkOutCodec=MWGetStream(NULL, NULL, FALSE);
	NetworkOutCodec->StreamData=(void *)Client;
	NetworkOutCodec->Codec=MWStreamToMWClient;

	/** Loop through the elements, stop if we get a smaller depth than we're allowed or at the end **/
	for (Element=MIMEElement; (Element < (long)MailSession->MIMEUsed) && MailSession->MIME[Element].Depth>=Depth; Element++) {
		/* Reset so the logic below can work */
		MIMECodec->Codec=NULL;

		/* Check the type */
		if (MWQuickCmp(MailSession->MIME[Element].Type, "multipart/alternative")) {
			unsigned long RelatedElement;

			/* We need to find the end of the alternative section */
			RelatedElement=0xffffffff;
			Element++;
			for (EndElement=Element; EndElement<MailSession->MIMEUsed; EndElement++) {
				if (MailSession->MIME[Element].Depth>MailSession->MIME[EndElement].Depth) {
					break;
				}
			}
			/* We will overshoot above */
			EndElement--;

			/* Start at the first type after the multipart/alternative */
			/* Now go through in reverse, and try to find a proper codec for the type */
			for (i=EndElement; ((long)i >= Element) && !MIMECodec->Codec; i--) {
				/* We only care about stuff on the same level as our multipart thing */
				if (MailSession->MIME[Element].Depth==MailSession->MIME[i].Depth) {
#if 1
					if (MWQuickCmp(MailSession->MIME[i].Type, "multipart/related")) {
						if ((RelatedElement>(i+1)) && ((i+1)<MailSession->MIMEUsed)) {
							i++;
							RelatedElement=i;
						}
					}
#endif

#ifdef LOTUS_CALENDAR_FIX
					if (MWQuickCmp(MailSession->MIME[i].Type, "text/calendar")) {
						if (HaveCalendar) {
							continue;
						}
						HaveCalendar=TRUE;
					}
#endif
					if ((MIMECodec->Codec=MWFindCodec(MailSession->MIME[i].Type, FALSE))!=NULL) {
						CharsetCodec->Codec=MWFindCodec(MailSession->MIME[i].Charset, FALSE);
						if (CharsetCodec->Codec) {
							MIMECodec->Charset=(unsigned char *)CharsetCodec;
						} else {
							MIMECodec->Charset=NULL;
						}
						EncodingCodec->Codec=MWFindCodec(MailSession->MIME[i].Encoding, FALSE);
						EndElement=i;
					}
				}
			}
			Element=EndElement;
		} else if (MWQuickCmp(MailSession->MIME[Element].Type, "message/rfc822")) {
			/* Prep the part cache for display */
			Client->NMAPProtected=TRUE;
			if (LoadRFCHeader(MessageID+1, Element, Session, MailSession)) {
				MWHandleTemplate(Client, Session, RFCTemplateID); 
			}
			Client->NMAPProtected=FALSE;
#ifdef LOTUS_CALENDAR_FIX
		} else if (MWQuickCmp(MailSession->MIME[Element].Type, "text/calendar")) {
			if (!HaveCalendar) {
				/* Prep the part cache for display */
				if ((MIMECodec->Codec=MWFindCodec(MailSession->MIME[Element].Type, FALSE))!=NULL) {
					CharsetCodec->Codec=MWFindCodec(MailSession->MIME[Element].Charset, FALSE);
					if (CharsetCodec->Codec) {
						MIMECodec->Charset=(unsigned char *)CharsetCodec;
					} else {
						MIMECodec->Charset=NULL;
					}
					EncodingCodec->Codec=MWFindCodec(MailSession->MIME[Element].Encoding, FALSE);
					HaveCalendar=TRUE;
				}
			}
#endif
		} else {
			if (MwMail.DisplayTextAttachments || (!MWQuickCmp(MailSession->MIME[Element].Type, "text/plain")) || (MailSession->MIME[Element].FileName[0] == '\0')) {
				/* The previous line prevents plain text attachments from being displayed. */
				/* Unpredictable results occur when the charset is not clearly defined */
				if ((MIMECodec->Codec=MWFindCodec(MailSession->MIME[Element].Type, FALSE)) != NULL) {
					CharsetCodec->Codec=MWFindCodec(MailSession->MIME[Element].Charset, FALSE);
					if (CharsetCodec->Codec) {
						MIMECodec->Charset=(unsigned char *)CharsetCodec;
					} else {
						MIMECodec->Charset=NULL;
					}
					EncodingCodec->Codec=MWFindCodec(MailSession->MIME[Element].Encoding, FALSE);
				}
			}
		}

		/* If we have a handler, try to trigger the stream */
		if (MIMECodec->Codec) {
			/* Do the idiot user check */
			if (MessageID>=Session->NumOfMessages) {
				/* Shithead user */
				MWReleaseStream(EncodingCodec);
				MWReleaseStream(CharsetCodec);
				MWReleaseStream(NetworkOutCodec);
				MWReleaseStream(MIMECodec);
				return(FALSE);
			}
			/* Request the body part */
			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MessageID], MailSession->MIME[Element].PartStart, MailSession->MIME[Element].PartSize);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

			if (ReplyInt/10!=202) {
				MWReleaseStream(EncodingCodec);
				MWReleaseStream(CharsetCodec);
				MWReleaseStream(NetworkOutCodec);
				MWReleaseStream(MIMECodec);
				return(FALSE);		
			}

			/* Let the code know how much data to expect */
			NetworkInCodec.StreamLength=atol(Client->Temp);

			/* Most codecs want the redirect URL in URL, they use StreamData for their own purpose, and StreamData2 is for the LinkURL */
			Client->Temp[0]='\0';
			MWEncodeURL(Session, Client->Temp+strlen(Client->Temp)+1, URL_TYPE_LINK, REQUEST_COMPOSE_VIA_URL, ComposeID, 0, 0, 0);
			strcat(Client->Temp+strlen(Client->Temp)+1, "+");
			MIMECodec->URL=Client->Temp;
			if (LinkTarget && LinkTarget[0]) {
				MIMECodec->StreamData2=LinkTarget;
			} else {
				MIMECodec->StreamData2 = NULL;
			}

			/* Make sure we're chained up properly, depending on what codecs we have */
			ChainStreams;
			Client->NMAPProtected=TRUE;
			NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);
			ResetStream(EncodingCodec);
			ResetStream(CharsetCodec);
			ResetStream(MIMECodec);

			/* Grab the BRAW DONE code */
			ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
			Client->NMAPProtected=FALSE;
		}
	}

	MWReleaseStream(EncodingCodec);
	MWReleaseStream(CharsetCodec);
	MWReleaseStream(MIMECodec);
	MWReleaseStream(NetworkOutCodec);

	/* Mark the retrieved message as read! */
	if (!Session->ReadOnly) {
		MWSendNMAPServer(Session, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID], (long unsigned int)MSG_STATE_READ));
		MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
	}

	return(TRUE);
}


BOOL
MwMailDisplayBodySource(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long ComposeID, unsigned long MessageID)
{
	StreamStruct	NetworkInCodec;
	StreamStruct	*NetworkOutCodec;
	StreamStruct	*MIMECodec;
	unsigned char	*ptr;
	unsigned long	ReplyInt;

	/* First, prep the MIME cache */
	if (!MwMailLoadMIMECache(MessageID, Client, Session, MailSession)) {
		return(FALSE);
	}

	MessageID--;

	/** Prepare all streams for later use **/
	/* Read from NMAP */
	memset(&NetworkInCodec, 0, sizeof(StreamStruct));
	NetworkInCodec.StreamData=Session;
	NetworkInCodec.Codec=MWStreamFromMWNMAP;

	/* Interpret the MIME type of the message part */
	MIMECodec=MWGetStream(&NetworkInCodec, "text/plain", FALSE);
	
	/* Send the interpreted part to the network */
	NetworkOutCodec=MWGetStream(MIMECodec, NULL, FALSE);
	NetworkOutCodec->StreamData=(void *)Client;
	NetworkOutCodec->Codec=MWStreamToMWClient;

	MIMECodec->Charset="utf-8";

	/* Request the body part */
	ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "LIST %lu\r\n", MailSession->IDList[MessageID]);
	MWSendNMAPServer(Session, Client->Temp, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);

	if ((ReplyInt/10!=202) || ((ptr=strchr(Client->Temp, ' '))==NULL)) {
		MWReleaseStream(NetworkOutCodec);
		MWReleaseStream(MIMECodec);
		return(FALSE);		
	}

	/* Let the code know how much data to expect */
	NetworkInCodec.StreamLength=atol(Client->Temp)+atol(ptr+1);

	/* Most MIME codecs want the redirect URL */
	Client->Temp[0]='\0';
	MWEncodeURL(Session, Client->Temp+strlen(Client->Temp)+1, URL_TYPE_LINK, REQUEST_COMPOSE_VIA_URL, ComposeID, 0, 0, 0);
	strcat(Client->Temp+strlen(Client->Temp)+1, "+");
	MIMECodec->URL=Client->Temp;

	/* We should be chained properly */
	Client->NMAPProtected=TRUE;
	NetworkInCodec.Codec(&NetworkInCodec, NetworkInCodec.Next);

	/* Grab the LIST DONE code */
	ReplyInt=MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
	Client->NMAPProtected=FALSE;

	MWReleaseStream(MIMECodec);
	MWReleaseStream(NetworkOutCodec);

	return(TRUE);
}


BOOL
MwMailDisplayBodyText(ConnectionStruct *Client, SessionStruct *Session, MailSessionStruct *MailSession, unsigned long RFCTemplateID, unsigned long ComposeID, unsigned char *LinkTarget, unsigned long MessageID, unsigned long MIMEElement, unsigned long Depth, unsigned long *ChunkStart, unsigned long ChunkSize, BOOL *More)
{
	StreamStruct	MemoryInCodec;
	StreamStruct	*NetworkOutCodec;
	StreamStruct	*CharsetCodec;
	StreamStruct	*EncodingCodec;
	StreamStruct	*MIMECodec;
	long				Element;
	unsigned long	EndElement;
	unsigned long	ReplyInt;
	unsigned long	i;
#if 0	 /* This may be relevant to this function in the future */
	BOOL				HaveCalendar=FALSE;
#endif
	BOOL				PlainTextFound=FALSE;

	/* First, prep the MIME cache */
	if (!MwMailLoadMIMECache(MessageID, Client, Session, MailSession) || MIMEElement>=MailSession->MIMEUsed) {
		return(FALSE);
	}

	MessageID--;

	/* 
		This function will be called for any "message/rfc822" toplevel. 
		This includes the actual message itself, even though it won't
		be marked as message/rfc822.
		We only parse from the specified MIME item, and never go higher 
		than the specifed depth

		We will try to find a handler for every type. If we don't find
		one we will ignore it and move on. we walk from first to last
		item, except if we encounter a multipart/alternative, in which
		case we find the end of the alternative section and walk back
		to the beginning, trying to find a handler.
		
	*/

	/** Prepare all streams for later use **/
	/* Read from NMAP */
	memset(&MemoryInCodec, 0, sizeof(StreamStruct));
	MemoryInCodec.Codec=MWStreamFromMemory;

	/* Resolve the charset of the message part */
	CharsetCodec=MWGetStream(NULL, NULL, FALSE);

	/* Resolve the encoding of the message part */
	EncodingCodec=MWGetStream(NULL, NULL, FALSE);

	/* Interpret the MIME type of the message part */
	MIMECodec=MWGetStream(NULL, NULL, FALSE);
	MIMECodec->Client=Client;
	
	/* Send the interpreted part to the network */
	NetworkOutCodec=MWGetStream(NULL, NULL, FALSE);
	NetworkOutCodec->StreamData=(void *)Client;
	NetworkOutCodec->Codec=MWStreamToMWClient;

	/** Loop through the elements, stop if we get a smaller depth than we're allowed or at the end **/
	for (Element=MIMEElement; (Element < (long)MailSession->MIMEUsed) && MailSession->MIME[Element].Depth>=Depth && !PlainTextFound; Element++) {
		/* Reset so the logic below can work */
		MIMECodec->Codec=NULL;

		/* Check the type */
		if (MWQuickCmp(MailSession->MIME[Element].Type, "multipart/alternative")) {
			unsigned long	RelatedElement;

			RelatedElement=0xffffffff;

			/* We need to find the end of the alternative section */
			Element++;
			for (EndElement=Element; EndElement<MailSession->MIMEUsed; EndElement++) {
				if (MailSession->MIME[Element].Depth>MailSession->MIME[EndElement].Depth) {
					break;
				}
			}
			/* We will overshoot above */
			EndElement--;

			/* Start at the first type after the multipart/alternative */
			/* Now go through in reverse, and try to find a proper codec for the type */
			for (i=EndElement; ((long)i >= Element) && !MIMECodec->Codec; i--) {
				/* We only care about stuff on the same level as our multipart thing */
				if (MailSession->MIME[Element].Depth==MailSession->MIME[i].Depth) {

#if 1
					if (MWQuickCmp(MailSession->MIME[i].Type, "multipart/related")) {
						if ((RelatedElement>(i+1)) && ((i+1)<MailSession->MIMEUsed)) {
							i++;
							RelatedElement=i;
						}
					}
#endif 


#if 0	 /* This may be relevant to this function in the future */
#ifdef LOTUS_CALENDAR_FIX
					if (MWQuickCmp(MailSession->MIME[i].Type, "text/calendar")) {
						if (HaveCalendar) {
							continue;
						}
						HaveCalendar=TRUE;					}
#endif
#endif
					if (MWQuickCmp(MailSession->MIME[i].Type, "text/plain")) {
						PlainTextFound = TRUE;
						MIMECodec->Codec=MWFindCodec("text/wml", FALSE);
						CharsetCodec->Codec=MWFindCodec(MailSession->MIME[i].Charset, FALSE);
						if (CharsetCodec->Codec) {
							MIMECodec->Charset=(unsigned char *)CharsetCodec;
						} else {
							MIMECodec->Charset=NULL;
						}
						EncodingCodec->Codec=MWFindCodec(MailSession->MIME[i].Encoding, FALSE);
						EndElement=i;
					}
				}
			}
			Element=EndElement;
/* This may be relevant to this function in the future */
#if 0
		} else if (MWQuickCmp(MailSession->MIME[Element].Type, "message/rfc822")) {
			/* Prep the part cache for display */
			Client->NMAPProtected=TRUE;
			if (LoadRFCHeader(MessageID+1, Element, Session, MailSession)) {
				MWHandleTemplate(Client, Session, RFCTemplateID); 
			}
			Client->NMAPProtected=FALSE;
#ifdef LOTUS_CALENDAR_FIX
		} else if (MWQuickCmp(MailSession->MIME[Element].Type, "text/calendar")) {
			if (!HaveCalendar) {
				/* Prep the part cache for display */
				if ((MIMECodec->Codec=MWFindCodec(MailSession->MIME[Element].Type, FALSE))!=NULL) {
					CharsetCodec->Codec=MWFindCodec(MailSession->MIME[Element].Charset, FALSE);
					if (CharsetCodec->Codec) {
						MIMECodec->Charset=(unsigned char *)CharsetCodec;
					} else {
						MIMECodec->Charset=NULL;
					}
					EncodingCodec->Codec=MWFindCodec(MailSession->MIME[Element].Encoding, FALSE);
					HaveCalendar=TRUE;
				}
			}
#endif
#endif
		} else {
			if (MWQuickCmp(MailSession->MIME[Element].Type, "text/plain")) {
				PlainTextFound = TRUE;
				MIMECodec->Codec=MWFindCodec("text/wml", FALSE);
				CharsetCodec->Codec=MWFindCodec(MailSession->MIME[Element].Charset, FALSE);
				if (CharsetCodec->Codec) {
					MIMECodec->Charset=(unsigned char *)CharsetCodec;
				} else {
					MIMECodec->Charset=NULL;
				}
				EncodingCodec->Codec=MWFindCodec(MailSession->MIME[Element].Encoding, FALSE);
			}
		}

		/* Leave 100 bytes for NMAP overhead */
		if (ChunkSize > (BUFSIZE-100)) {
			ChunkSize = BUFSIZE-100;
		}

		/* If we have a handler, read the message and trigger the stream */
		if ((MIMECodec->Codec) && (*ChunkStart < MailSession->MIME[Element].PartSize)) {
			unsigned char *ptr;
			unsigned char *BodyStart;
			unsigned long BytesInBuffer = 0;		
		
			/* Is this the last chunk? */
			if (MailSession->MIME[Element].PartSize > (*ChunkStart + ChunkSize)) {
				*More = TRUE;
			} else {
				*More = FALSE;
			}

			/* Do the idiot user check */
			if (MessageID >= Session->NumOfMessages) {
				/* Shithead user */
				MWReleaseStream(EncodingCodec);
				MWReleaseStream(CharsetCodec);
				MWReleaseStream(NetworkOutCodec);
				MWReleaseStream(MIMECodec);
				return(FALSE);
			}
			/* Request the body part */

			ReplyInt=snprintf(Client->Temp, sizeof(Client->Temp), "BRAW %lu %lu %lu\r\n", MailSession->IDList[MessageID], MailSession->MIME[Element].PartStart + *ChunkStart, (*More) ? ChunkSize : MailSession->MIME[Element].PartSize - *ChunkStart);
			MWSendNMAPServer(Session, Client->Temp, ReplyInt);
			
			/* Get the NMAP response */
			do {
				ReplyInt= MWDirectNMAPRead(Session, Client->Temp + BytesInBuffer, BUFSIZE - BytesInBuffer);
				if (ReplyInt < 1) {
					MWReleaseStream(EncodingCodec);
					MWReleaseStream(CharsetCodec);
					MWReleaseStream(NetworkOutCodec);
					MWReleaseStream(MIMECodec);
					return(FALSE);		
				}
				BytesInBuffer += ReplyInt;
			} while ((BodyStart = strchr(Client->Temp, 0x0a)) == NULL);

			BodyStart++;

			/* Handle first line of the response */ 
			while (Client->Temp[0] == '6' && Client->Temp[1] == '0' && Client->Temp[2] == '0' && Client->Temp[3] == '0') {
				/* Mailbox has changed! Deal with it. */
				Session->NMAPChanged = TRUE;
				/* Reset Buffer, BytesInBuffer, and BodyStart so the rest of the code will work */
				memmove(Client->Temp, BodyStart, (Client->Temp + BytesInBuffer) - BodyStart);
				BytesInBuffer -= BodyStart - Client->Temp;
				while ((BodyStart = strchr(Client->Temp, 0x0a)) == NULL) {
					ReplyInt = MWDirectNMAPRead(Session, Client->Temp + BytesInBuffer, BUFSIZE - BytesInBuffer);
					if (ReplyInt < 1) {
						MWReleaseStream(EncodingCodec);
						MWReleaseStream(CharsetCodec);
						MWReleaseStream(NetworkOutCodec);
						MWReleaseStream(MIMECodec);
						return(FALSE);		
					}
					BytesInBuffer += ReplyInt;
				} 
				BodyStart++;
			}

			/* Check for NMAP Error */
			if (!MWQuickNCmp(Client->Temp, "202", 3)) {
				MWReleaseStream(EncodingCodec);
				MWReleaseStream(CharsetCodec);
				MWReleaseStream(NetworkOutCodec);
				MWReleaseStream(MIMECodec);
				return(FALSE);		
			}

			/* Find out how much data we should be looking for */
			ChunkSize = atol(Client->Temp + 5);
					   
  			/* Load the payload and the '1000 OK' into the buffer if it is not already there */
			while (((Client->Temp + BytesInBuffer) >= BodyStart) && (unsigned long)((Client->Temp + BytesInBuffer) - BodyStart) < (ChunkSize + 9)) {
				ReplyInt = MWDirectNMAPRead(Session, Client->Temp + BytesInBuffer, BUFSIZE - BytesInBuffer);

				if (ReplyInt < 1) {
					break;
				}
				
				BytesInBuffer += ReplyInt;
			}

			/* Set ptr to be at the end of the data and before the 1000 OK\r\n */
			ptr = Client->Temp + BytesInBuffer - 9;

			if (ptr <= BodyStart) {
				MWReleaseStream(EncodingCodec);
				MWReleaseStream(CharsetCodec);
				MWReleaseStream(NetworkOutCodec);
				MWReleaseStream(MIMECodec);
				return(FALSE);		
			}

			/* Look for a logical place to break up the message */
			if (*More) { 
				while (ptr > BodyStart) {
					if ((*ptr == ' ')  || (*ptr == '\r') || (*ptr == '\n') ) { /* CR, LF, SPACE */
						*(ptr+1)='\0';
						break;
					}
					ptr--;
				}
			}

			ChunkSize = ptr - BodyStart;

			*ChunkStart += ChunkSize;

			/* Let the codec know how much data to expect */
			MemoryInCodec.StreamLength = ChunkSize;
			MemoryInCodec.StreamData	= BodyStart;

			/* Make sure we're chained up properly, depending on what codecs we have */
			if (CharsetCodec->Codec) {							
				if (EncodingCodec->Codec) {					
					MemoryInCodec.Next=EncodingCodec;		
					EncodingCodec->Next=CharsetCodec;		
					CharsetCodec->Next=MIMECodec;				
				} else {												
					MemoryInCodec.Next=CharsetCodec;		
					CharsetCodec->Next=MIMECodec;				
				}														
			} else {													
				if (EncodingCodec->Codec) {					
					MemoryInCodec.Next=EncodingCodec;		
					EncodingCodec->Next=MIMECodec;			
				} else {												
					MemoryInCodec.Next=MIMECodec;			
				}														
			}															
			MIMECodec->Next=NetworkOutCodec;

			MemoryInCodec.Codec=MWStreamFromMemory;
			MemoryInCodec.Codec(&MemoryInCodec, MemoryInCodec.Next);		
			ResetStream(EncodingCodec);
			ResetStream(CharsetCodec);
			ResetStream(MIMECodec);

		}

		/* Mark the retrieved message as read! */
		if (!Session->ReadOnly) {
			MWSendNMAPServer(Session, Client->Temp, snprintf(Client->Temp, sizeof(Client->Temp), "AFLG %lu %lu\r\n", MailSession->IDList[MessageID], (long unsigned int)MSG_STATE_READ));
			MWGetNMAPAnswer(Session, Client->Temp, BUFSIZE, TRUE);
		}
	}

	MWReleaseStream(EncodingCodec);
	MWReleaseStream(CharsetCodec);
	MWReleaseStream(MIMECodec);
	MWReleaseStream(NetworkOutCodec);


	return(TRUE);
}
