/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: HydrogenApp.cpp,v 1.56 2004/02/20 14:09:13 comix Exp $
 *
 */

#include "HydrogenApp.h"
#include "FilePreview.h"
#include "lib/Hydrogen.h"

HydrogenApp* HydrogenApp::instance = NULL;

/**
 * Constructor
 */
HydrogenApp::HydrogenApp( MainForm *mainForm, Song *firstSong ) {
	instance = this;
	this->mainForm = mainForm;

	mixer = NULL;
	patternEditorPanel = NULL;
	statusBar = NULL;
	audioEngineInfoForm = NULL;
	helpBrowser = NULL;
	songEditorPanel = NULL;

	// Load the LADSPA plugin list
	m_pluginList = LadspaFX::getPluginList();
	m_pFXRootGroup = LadspaFX::getLadspaFXGroup();

	// Create the audio engine :)
	( Hydrogen::getInstance(this) );	// create the engine and set this class as engine listener

	(Hydrogen::getInstance())->setSong( firstSong );
	( PreferencesMng::getInstance() )->setLastSongFilename( firstSong->getFilename() );

	// set initial title
	QString songName( (firstSong->getName()).c_str() );
	mainForm->setCaption( ( "Hydrogen " + QString(VERSION) + QString( " - " ) + songName ) );



	PreferencesMng *pref = PreferencesMng::getInstance();
	if ( pref->getInterfaceMode() == "Child frame" ) {	// use the new MDI interface
		// create the workspace
		mainForm->workspace = new QWorkspace( mainForm );
		mainForm->workspace->setScrollBarsEnabled( true );
		string mdiBackground_path = string( IMG_PATH ) + string( "/img/mdiBackground.png" );
		mainForm->workspace->setBackgroundPixmap( QPixmap(mdiBackground_path.c_str()) );

		// restore mainForm properties
		WindowProperties mainFormProp = pref->getMainFormProperties();
//		mainForm->setMinimumSize( QSize( mainFormProp.width, mainFormProp.height ) );
		mainForm->resize( mainFormProp.width, mainFormProp.height );
		mainForm->move( mainFormProp.x, mainFormProp.y );
		mainForm->setCentralWidget( mainForm->workspace );

		// restore mixer properties
		mixer = new Mixer( mainForm->workspace );
		WindowProperties mixerProp = pref->getMixerProperties();
		mixer->resize( mixerProp.width, mixerProp.height );
		mixer->move( mixerProp.x, mixerProp.y );

		// restore song editor properties
		songEditorPanel = new SongEditorPanel( mainForm->workspace );
		WindowProperties songEditorProp = pref->getSongEditorProperties();
		songEditorPanel->resize( songEditorProp.width, songEditorProp.height );
		songEditorPanel->move( songEditorProp.x, songEditorProp.y );

		// restore pattern editor properties
		patternEditorPanel = new PatternEditorPanel( mainForm->workspace );
		WindowProperties patternEditorProp = pref->getPatternEditorProperties();
		patternEditorPanel->resize( patternEditorProp.width, patternEditorProp.height );
		patternEditorPanel->move( patternEditorProp.x, patternEditorProp.y );

		// restore audio engine form properties
		audioEngineInfoForm = new AudioEngineInfoForm( mainForm->workspace );
		audioEngineInfoForm->hide();
		WindowProperties audioEngineInfoProp = pref->getAudioEngineInfoProperties();
		audioEngineInfoForm->move( audioEngineInfoProp.x, audioEngineInfoProp.y );


		// restore drumkit manager properties
		drumkitManager = new DrumkitManager( mainForm->workspace );
		drumkitManager->hide();
		WindowProperties drumkitMngProp = pref->getDrumkitManagerProperties();
		drumkitManager->move( drumkitMngProp.x, drumkitMngProp.y );

		helpBrowser = new HelpBrowser( 0 );

		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			m_pLadspaFXProperties[nFX] = new LadspaFXProperties( mainForm->workspace, nFX );
			m_pLadspaFXProperties[nFX]->hide();
			WindowProperties prop = pref->getLadspaProperties(nFX);
			m_pLadspaFXProperties[nFX]->move( prop.x, prop.y );
		}
	}
	else {

		// restore mixer properties
		mixer = new Mixer(0);
		WindowProperties mixerProp = pref->getMixerProperties();
		mixer->resize( mixerProp.width, mixerProp.height );
		mixer->move( mixerProp.x, mixerProp.y );

		// restore pattern editor properties
		patternEditorPanel = new PatternEditorPanel( 0 );
		WindowProperties patternEditorProp = pref->getPatternEditorProperties();
		patternEditorPanel->resize( patternEditorProp.width, patternEditorProp.height );
		patternEditorPanel->move( patternEditorProp.x, patternEditorProp.y );

		// restore audio engine form properties
		audioEngineInfoForm = new AudioEngineInfoForm( 0 );
		WindowProperties audioEngineInfoProp = pref->getAudioEngineInfoProperties();
		audioEngineInfoForm->move( audioEngineInfoProp.x, audioEngineInfoProp.y );

		// restore song editor properties
		songEditorPanel = new SongEditorPanel( mainForm );
		mainForm->setCentralWidget( songEditorPanel );

		// restore drumkit manager properties
		drumkitManager = new DrumkitManager( 0 );
		WindowProperties drumkitMngProp = pref->getDrumkitManagerProperties();
		drumkitManager->move( drumkitMngProp.x, drumkitMngProp.y );

		helpBrowser = new HelpBrowser( 0 );

		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			m_pLadspaFXProperties[nFX] = new LadspaFXProperties( NULL, 0 );
			m_pLadspaFXProperties[nFX]->hide();
			WindowProperties prop = pref->getLadspaProperties(nFX);
			m_pLadspaFXProperties[nFX]->move( prop.x, prop.y );
		}
	}

	statusBar = new QStatusBar( mainForm );
	statusBar->setSizeGripEnabled( false );
}




/**
 * Destructor
 */
HydrogenApp::~HydrogenApp() {
	if (songEditorPanel) {
		delete songEditorPanel;
	}

	if (helpBrowser) {
		delete helpBrowser;
	}

	if (audioEngineInfoForm) {
		delete audioEngineInfoForm;
	}

	if (mixer) {
		mixer->updateStart(false);
		delete mixer;
	}

	if (patternEditorPanel) {
		patternEditorPanel->updateStart(false);
		delete patternEditorPanel;
	}

	if (drumkitManager) {
		delete drumkitManager;
	}

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		delete m_pLadspaFXProperties[nFX];
	}

	for (uint i = 0; i < m_pluginList.size(); i++) {
		delete m_pluginList[ i ];
	}

	delete m_pFXRootGroup;


	Hydrogen *engine = Hydrogen::getInstance();
	if (engine) {
		Song *song = engine->getSong();

		delete engine;
		engine = NULL;

		if (song) {
			delete song;
			song = NULL;
		}
	}
}




/**
 * Return an HydrogenApp instance
 */
HydrogenApp* HydrogenApp::getInstance() {
	if (instance == NULL) {
		cerr << "Error! HydrogenApp::getInstance (instance = NULL)" << endl;
	}
	return instance;
}




/**
 *
 */
void HydrogenApp::setSong(Song* song) {
	mixer->updateStart(false);
	patternEditorPanel->updateStart(false);

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		m_pLadspaFXProperties[nFX]->hide();
	}

	Song* oldSong = (Hydrogen::getInstance())->getSong();
	if (oldSong != NULL) {
		(Hydrogen::getInstance())->removeSong();
		delete oldSong;
		oldSong = NULL;
	}

	(Hydrogen::getInstance())->setSong(song);
	( PreferencesMng::getInstance() )->setLastSongFilename( song->getFilename() );

	if (mixer->isVisible()) {
		mixer->updateStart(true);
	}
	if (patternEditorPanel->isVisible()) {
		patternEditorPanel->updateStart(true);
	}
	songEditorPanel->updateAll();

	QString songName( (song->getName()).c_str() );
	mainForm->setCaption( ( "Hydrogen " + QString(VERSION) + QString( " - " ) + songName ) );

	mainForm->updateRecentUsedSongList();

	QMessageBox::information( mainForm, "Hydrogen", mainForm->m_pQApp->translate("HydrogenApp", "Song Info:     ") + QString("\n\n") + QString( song->getNotes().c_str() ) + QString("\n") );
}




/**
 *
 */
Song *HydrogenApp::getSong() {
	return (Hydrogen::getInstance())->getSong();
}




/**
 *
 */
void HydrogenApp::showMixer(bool show) {
	if (show) {
		mixer->show();
	}
	else {
		mixer->hide();
	}
}




/**
 *
 */
void HydrogenApp::showPatternEditor(bool show) {
	if (show) {
		patternEditorPanel->show();
	}
	else {
		patternEditorPanel->hide();
	}
}




/**
 *
 */
void HydrogenApp::showPreferencesDialog() {
	PreferencesDialog preferencesDialog(mainForm);
	if (preferencesDialog.exec() == QDialog::Accepted) {
//		(Hydrogen::getInstance())->restartDrivers();
	}
}





/**
 *
 */
void HydrogenApp::setStatusBarMessage(QString msg, int msec) {
	if (msec != 0) {
		statusBar->message(msg, msec);
	}
	else {
		statusBar->message(msg, msec);
	}
}




/**
 * Used in the mixer panel, shows the open file dialog and replace an existent instrument
 */
void HydrogenApp::loadNewInstrument(int instrNumber) {
	// FIXME: save the last used directory in preferences file
	static QString lastUsedDir = "";

	Hydrogen *engine = Hydrogen::getInstance();

	QFileDialog *fd = new QFileDialog(mainForm, "File Dialog", TRUE);
	fd->setMode(QFileDialog::ExistingFile);
	fd->setFilter( mainForm->m_pQApp->translate("HydrogenApp", "Audio files (*.wav *.WAV *.au *.AU *.aiff *.AIFF)") );
	fd->setCaption( mainForm->m_pQApp->translate("HydrogenApp", "Hydrogen - Load instrument") );
	fd->setDir( lastUsedDir );

	FilePreview *pPreview = new FilePreview();
	fd->setContentsPreviewEnabled( TRUE );
	fd->setContentsPreview( pPreview, pPreview );
	fd->setPreviewMode( QFileDialog::Contents );

	QString filename = "";
	if (fd->exec() == QDialog::Accepted) {
		filename = fd->selectedFile();
	}

	if (filename != "") {
		lastUsedDir = fd->dirPath();
		Sample *newSample = Sample::load( filename.latin1() );

		Instrument *instr = NULL;

		engine->lockEngine("HydrogenApp::loadNewInstrument");
		Song *song = engine->getSong();
		InstrumentList *instrList = song->getInstrumentList();
		instr = instrList->get( instrNumber );

		// delete old sample
		Sample *oldSample = instr->getSample();
		delete oldSample;

		// insert new sample from newInstrument
		instr->setSample( newSample );
		instr->setDrumkitName( "" );	// external sample, no drumkit info

		engine->unlockEngine();
	}
}




/**
 *
 */
void HydrogenApp::showAudioEngineInfoForm() {
	audioEngineInfoForm->hide();
	audioEngineInfoForm->show();
}





/**
 * This method is called from another thread
 */
void HydrogenApp::engineError( uint nErrorCode ) {
	QString msg;
	QApplication* pApp = mainForm->m_pQApp;

	switch (nErrorCode) {
		case Hydrogen::NO_OSS_SUPPORT:
			msg = pApp->translate( "HydrogenApp", "No OSS driver support" );
			break;

		case Hydrogen::NO_JACK_SUPPORT:
			msg = pApp->translate( "HydrogenApp", "No JACK driver support" );
			break;

		case Hydrogen::UNKNOWN_DRIVER:
			msg = pApp->translate( "HydrogenApp", "Unknown audio driver" );
			break;

		case Hydrogen::ERROR_STARTING_DRIVER:
			msg = pApp->translate( "HydrogenApp", "Error starting audio driver" );
			break;

		case Hydrogen::JACK_SERVER_SHUTDOWN:
			msg = pApp->translate( "HydrogenApp", "Jack driver: server shutdown" );
			break;

		case Hydrogen::JACK_CANNOT_ACTIVATE_CLIENT:
			msg = pApp->translate( "HydrogenApp", "Jack driver: cannot activate client" );
			break;

		case Hydrogen::JACK_CANNOT_CONNECT_OUTPUT_PORT:
			msg = pApp->translate( "HydrogenApp", "Jack driver: cannot connect output port" );
			break;

		case Hydrogen::JACK_ERROR_IN_PORT_REGISTER:
			msg = pApp->translate( "HydrogenApp", "Jack driver: error in port register" );
			break;

		default:
			msg = pApp->translate( "HydrogenApp", "Unknown error" );
	}

	H2TextEvent *ev = new H2TextEvent( "ErrorMsg", msg );
	QApplication::postEvent( mainForm, ev );

}







