/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: SongEditor.cpp,v 1.58 2004/02/25 13:29:01 comix Exp $
 *
 */


#include "SongEditor.h"
#include "lib/Song.h"
#include "lib/Hydrogen.h"


SongEditorPanel::SongEditorPanel(QWidget *parent) : QWidget(parent), Object( "SongEditorPanel" )
{
//	cout << "SongEditorPanel INIT" << endl;
	m_nInitialWidth = 600;
	m_nInitialHeight = 250;

	const uint nMinWidth = 300;
	const uint nMinHeight = 200;

	resize( QSize(m_nInitialWidth, m_nInitialHeight) );
	setMinimumSize( nMinWidth, nMinHeight );
//	setMaximumSize( width, height );

	setCaption( trUtf8( "Song Editor" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );


	// new pattern button
	string newPat_on_path = string( IMG_PATH ) + string( "/img/songEditor/newPatternBtn_on.png" );
	string newPat_off_path = string( IMG_PATH ) + string( "/img/songEditor/newPatternBtn_off.png" );
	string newPat_over_path = string( IMG_PATH ) + string( "/img/songEditor/newPatternBtn_over.png" );
	Button *newPatBtn = new Button(this, QSize(20, 20), newPat_on_path, newPat_off_path, newPat_over_path);
	newPatBtn->move(5, 5);
	QToolTip::add( newPatBtn, trUtf8("Create new pattern") );
	connect( newPatBtn, SIGNAL( clicked( Button* ) ), this, SLOT( newPatBtnClicked( Button* ) ) );

	// down button
	string downBtn_on_path = string( IMG_PATH ) + string( "/img/songEditor/downBtn_on.png" );
	string downBtn_off_path = string( IMG_PATH ) + string( "/img/songEditor/downBtn_off.png" );
	string downBtn_over_path = string( IMG_PATH ) + string( "/img/songEditor/downBtn_over.png" );
	downBtn = new Button(this, QSize(20, 20), downBtn_on_path, downBtn_off_path, downBtn_over_path);
	downBtn->move( 30, 5 );
	QToolTip::add( downBtn, trUtf8("Move the selected pattern down") );
	connect( downBtn, SIGNAL( clicked( Button* ) ), this, SLOT( downBtnClicked( Button* ) ) );

	// up button
	string upBtn_on_path = string( IMG_PATH ) + string( "/img/songEditor/upBtn_on.png" );
	string upBtn_off_path = string( IMG_PATH ) + string( "/img/songEditor/upBtn_off.png" );
	string upBtn_over_path = string( IMG_PATH ) + string( "/img/songEditor/upBtn_over.png" );
	upBtn = new Button(this, QSize(20, 20), upBtn_on_path, upBtn_off_path, upBtn_over_path);
	upBtn->move( 55, 5 );
	QToolTip::add( upBtn, trUtf8("Move the selected pattern up") );
	connect( upBtn, SIGNAL( clicked( Button* ) ), this, SLOT( upBtnClicked( Button* ) ) );


	// song button
	string songBtn_on_path = string( IMG_PATH ) + string( "/img/songEditor/songFuncBtn_on.png" );
	string songBtn_off_path = string( IMG_PATH ) + string( "/img/songEditor/songFuncBtn_off.png" );
	string songBtn_over_path = string( IMG_PATH ) + string( "/img/songEditor/songFuncBtn_over.png" );
	songBtn = new Button(this, QSize(20, 20), songBtn_on_path, songBtn_off_path, songBtn_over_path);
	songBtn->move( 80, 5 );
	QToolTip::add( songBtn, trUtf8("Song operations") );
	connect( songBtn, SIGNAL( clicked( Button* ) ), this, SLOT( songBtnClicked( Button* ) ) );

	// Song loop button
	string songLoop_on_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_loop_on.png" );
	string songLoop_off_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_loop_off.png" );
	string songLoop_over_path = string( IMG_PATH ) + string( "/img/playerControlPanel/btn_loop_over.png" );
	songLoopBtn = new ToggleButton(this, QSize(20, 20), songLoop_on_path, songLoop_off_path, songLoop_over_path);
	songLoopBtn->move( 105, 5 );
	songLoopBtn->setPressed( false );
	QToolTip::add( songLoopBtn, trUtf8("Song loop") );
	connect( songLoopBtn, SIGNAL( clicked(Button*) ), this, SLOT( songLoopBtnClicked(Button*) ) );

	// song editor
	sv1 = new QScrollView( this );
//	sv1->setFrameShape( QFrame::NoFrame );
	sv1->move( 105, 45 );
	sv1->resize( m_nInitialWidth - 110, m_nInitialHeight - 45 - 5 );

	songEditor = new SongEditor( sv1->viewport() );
	songEditor->move( 0, 0 );
	songEditor->show();
	sv1->addChild( songEditor );
	connect( sv1, SIGNAL( contentsMoving ( int , int ) ), this, SLOT( contentsMove( int, int ) ) );


	// song editor pattern list
	sv2 = new QScrollView( this );
//	sv2->setFrameShape( QFrame::NoFrame );
	sv2->move( 5, 45 );
	sv2->resize( 100, m_nInitialHeight - 45 - 5 );
	sv2->setVScrollBarMode( QScrollView::AlwaysOff );
	sv2->setHScrollBarMode( QScrollView::AlwaysOn );

	songEditorPatternList = new SongEditorPatternList( sv2->viewport() );
	songEditorPatternList->move( 0, 0 );
	songEditorPatternList->show();
	sv2->addChild( songEditorPatternList );


	// Position Ruler
	sv3 = new QScrollView( this );
//	sv3->setFrameShape( QFrame::NoFrame );
	sv3->setVScrollBarMode( QScrollView::AlwaysOff );
	sv3->move( 105, 30 );
	sv3->resize( m_nInitialWidth - 110, 15 );
	sv3->setHScrollBarMode( QScrollView::AlwaysOff );

	songEditorPositionRuler = new SongEditorPositionRuler( sv3->viewport() );
	songEditorPositionRuler->move( 0, 0 );
	sv3->addChild( songEditorPositionRuler );



	QPixmap patternIcon;
	string patternIcon_path = string( IMG_PATH ) + string( "/img/songEditor/patternIcon.png" );
	patternIcon.load( patternIcon_path.c_str() );

	songPopup = new QPopupMenu( this, "patternPopupMenu" );
	songPopup->insertItem( patternIcon, trUtf8("Song properties"),  this, SLOT( songPopup_songProperties() ) );
	songPopup->insertItem( patternIcon, trUtf8("Clear pattern sequence"),  this, SLOT( songPopup_clearSequence() ) );
	songPopup->insertItem( patternIcon, trUtf8("Delete all patterns"),  this, SLOT( songPopup_deleteAllPatterns() ) );



	// Background image
	string background_path = string( IMG_PATH ) + string( "/img/patternEditor/patternEditor_background.png" );
	QPixmap background;
	bool ok = background.load( background_path.c_str() );
	if( ok == false ){
		qWarning( "PatternEditorPanel: Error loading pixmap" );
	}
	setBackgroundPixmap( background );

	updateAll();

	(Hydrogen::getInstance())->addEngineListener(this);
}





/**
 * Destructor
 */
SongEditorPanel::~SongEditorPanel() {
//	cout << "SongEditorPanel destroy" << endl;
}





/**
 * Synchronize the patternlist with the patternsequence
 */
void SongEditorPanel::contentsMove( int x, int y) {
	sv2->verticalScrollBar()->setValue( sv1->verticalScrollBar()->value() );
	sv3->horizontalScrollBar()->setValue( sv1->horizontalScrollBar()->value() );
}



/**
 * Update and redraw all...
 */
void SongEditorPanel::updateAll() {
	Song *song = (Hydrogen::getInstance())->getSong();
	if ( song->isLoopEnabled() ) {
		songLoopBtn->setPressed( true );
	}
	else {
		songLoopBtn->setPressed( false );
	}

	songEditorPatternList->createBackground();
	songEditorPatternList->update();

	songEditor->createBackground();
	songEditor->update();
}




/**
 * Create a new pattern
 */
void SongEditorPanel::newPatBtnClicked( Button* btn) {
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	Pattern *emptyPattern = Pattern::getEmptyPattern();

	PatternPropertiesDialog *dialog = new PatternPropertiesDialog( this, emptyPattern );
	if ( dialog->exec() == QDialog::Accepted ) {
		patternList->add( emptyPattern );
		song->setModified( true );
		updateAll();
	}
	else {
		patternList->del( emptyPattern );
		delete emptyPattern;
		emptyPattern = NULL;
	}
	delete dialog;
	dialog = NULL;



}





/**
 * Show the song popup
 */
void SongEditorPanel::songBtnClicked( Button* btn) {
	songPopup->popup( songBtn->mapToGlobal( QPoint( 10, 10 ) ) );
}




/**
 * Song loop button clicked
 */
void SongEditorPanel::songLoopBtnClicked( Button* ref ) {
	bool value = songLoopBtn->isPressed();

	Hydrogen *pEngine = Hydrogen::getInstance();
	Song *song = pEngine->getSong();

/*	// verificare se si sta usando il jack-transport
	GenericDriver *pAudioDriver = pEngine->getAudioDriver();
	if ( pAudioDriver->getClassName() == "JackDriver" ) {
		if ( ( PreferencesMng::getInstance() )->isJackTransportSlave() ) {
//			cout << "non posso impostare il loop per la canzone (jack-transport)" << endl;
			song->setLoopEnabled( false );
			songLoopBtn->setPressed( false );
			QMessageBox::information( this, "Hydrogen", trUtf8("Can't set the loop mode while using Jack-Transport"));
		}
	}
	else {
		song->setLoopEnabled( value );
		song->setModified( true );
	}
*/
	song->setLoopEnabled( value );
	song->setModified( true );
}





/**
 * Move up a pattern in the patternList
 */
void SongEditorPanel::upBtnClicked( Button* btn ) {
	int selectedPatternPos = songEditorPatternList->getSelectedPatternPos();

	Hydrogen *engine = Hydrogen::getInstance();

	engine->lockEngine("SongEditorPanel::upBtnClicked");
	Song *song = engine->getSong();
	PatternList *list = song->getPatternList();

	if ( ( selectedPatternPos - 1 ) >= 0 ) {
		Pattern *temp = list->get( selectedPatternPos - 1 );
		list->replace( list->get( selectedPatternPos ), selectedPatternPos - 1 );
		list->replace( temp, selectedPatternPos );
		engine->unlockEngine();

		updateAll();
		song->setModified(true);
	}
	else {
		engine->unlockEngine();
	}
}




/**
 * Move down a pattern in the patternList
 */
void SongEditorPanel::downBtnClicked( Button* btn ) {
	int selectedPatternPos = songEditorPatternList->getSelectedPatternPos();

	Hydrogen *engine = Hydrogen::getInstance();
	engine->lockEngine("SongEditorPanel::downBtnClicked");
	Song *song = engine->getSong();
	PatternList *list = song->getPatternList();

	if ( ( selectedPatternPos + 1 ) < (int)list->getSize() ) {
		Pattern *temp = list->get( selectedPatternPos + 1 );
		list->replace( list->get( selectedPatternPos ), selectedPatternPos + 1 );
		list->replace( temp, selectedPatternPos );
		engine->unlockEngine();

		updateAll();
		song->setModified(true);
	}
	else {
		engine->unlockEngine();
	}

}




/**
 * Show the song properties dialog
 */
void SongEditorPanel::songPopup_songProperties() {
	SongPropertiesDialog *dialog = new SongPropertiesDialog( this );
	if ( dialog->exec() == QDialog::Accepted ) {
		updateAll();
		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		song->setModified( true );
	}
	delete dialog;
	dialog = NULL;
}




/**
 *
 */
void SongEditorPanel::songPopup_clearSequence() {
	Hydrogen *engine = Hydrogen::getInstance();

	engine->lockEngine("SongEditorPanel::songPopup_clearSequence");
	Song *song = engine->getSong();
	PatternList *sequence = song->getPatternSequence();
	sequence->clear();
	engine->unlockEngine();

	updateAll();
	song->setModified(true);

}




/**
 *
 */
void SongEditorPanel::songPopup_deleteAllPatterns() {
	Hydrogen *engine = Hydrogen::getInstance();
	engine->setCurrentPattern( NULL );

	engine->lockEngine("SongEditorPanel::songPopup_deleteAllPatterns");

	Song *song = engine->getSong();
	PatternList *sequence = song->getPatternSequence();
	sequence->clear();

	PatternList *patList = song->getPatternList();
	for (uint i = 0; i < patList->getSize(); i++) {
		Pattern *pat = patList->get(i);
		delete pat;
	}
	patList->clear();

	engine->unlockEngine();

	updateAll();
	song->setModified(true);
}




/**
 * This method is called from another thread (audio engine)
 */
void SongEditorPanel::patternChanged() {
	H2TextEvent *ev = new H2TextEvent( "patternChanged" );
	QApplication::postEvent( this, ev );
}




/**
 *
 */
void SongEditorPanel::customEvent( QCustomEvent *ev ) {
	if ( ev->type() != H2_TEXT_EVENT ) {	// Must be a H2TextEvent
		return;
	}
	QString message = ( (H2TextEvent*)ev )->getText();

	if ( message == QString( "patternChanged" ) ) {
		songEditorPatternList->createBackground();	// update the pattern highlight
		songEditorPatternList->update();

		// FIXME: update the song position ruler
		songEditorPositionRuler->updateRuler();
	}

}




/**
 *
 */
void SongEditorPanel::resizeEvent ( QResizeEvent *ev )
{
	// scrollview 1
	sv1->resize( width() - 110, height() - 45 - 5 );

	// scrollview 2
	sv2->resize( 100, height() - 45 - 5 );

	// scrollview 3
	sv3->resize( width() - 110, 15 );

}






///////////////////////////////////////////




/**
 * Constructor
 */
SongEditor::SongEditor( QWidget *parent ) : QWidget( parent ), Object( "SongEditor" )
{
//	cout << "song editor INIT" << endl;

	gridWidth = 16;
	gridHeight = (PreferencesMng::getInstance())->getPatternEditorGridHeight();
	initialWidth = maxPatternSequence * gridWidth;
	initialHeight = 10;

	changed = true;
	sequenceChanged = true;
	this->resize( QSize(initialWidth, initialHeight) );

	createBackground();	// create background pixmap

	update();
}





/**
 * Destructor
 */
SongEditor::~SongEditor() {
//	cout << "song editor destroy" << endl;
}





/**
 *
 */
void SongEditor::mousePressEvent( QMouseEvent *ev ) {
	int row = (ev->y() / gridHeight);
	int column = (ev->x() / gridWidth);

	Hydrogen *engine = Hydrogen::getInstance();
	engine->lockEngine("SongEditor::mousePressEvent");

	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();
	PatternList *patternSequence = song->getPatternSequence();

	if ( row >= (int)patternList->getSize() ) {
		return;
	}
	Pattern *pattern = patternList->get( row );

	// check the patternSequence size
	if ( column >= (int)patternSequence->getSize() ) {
		int distance = column - patternSequence->getSize();
		if (distance == 0) {
/*			for (int i = 0; i < distance; i++) {
				patternSequence->add( NULL );
			}
*/
			patternSequence->add( pattern );
			song->setModified(true);
		}
	}
	else {
		if ( ( patternSequence->get( column ) ) == pattern ) {	// the pattern exists yet, remove..

			//is the last pattern in the list?
			if ((int)(patternSequence->getSize() - 1) == column) {
//				cout << "LAST PATTERN" << endl;
				patternSequence->replace( NULL, column );
				if ( patternSequence->getSize() > 0) {
					// remove the trailing NULL patterns in patternSequence
					while ( patternSequence->getSize() > 0 ) {
						if ( (patternSequence->get( patternSequence->getSize() - 1 ) ) == NULL ) {
							patternSequence->del( patternSequence->getSize() - 1 );
						}
						else {
							break;
						}
					}
				}
			}

		}
		else {
			patternSequence->replace( pattern, column );
		}

		song->setModified(true);

	}
	engine->unlockEngine();

	// update
	changed = true;
	sequenceChanged = true;
	update();
}




/**
 *
 */
void SongEditor::paintEvent( QPaintEvent *ev ) {
	if (!isVisible()) {
		return;
	}

	if (changed) {
		changed = false;

		// ridisegno tutto solo se sono cambiate le note
		if (sequenceChanged) {
			bitBlt( &sequence, 0, 0, &background, 0, 0, width(), height(), CopyROP );
			drawSequence();
			sequenceChanged = false;
		}

		bitBlt( &temp, 0, 0, &sequence, 0, 0, width(), height(), CopyROP );
		setErasePixmap( temp );

		// copio l'immagine definitiva
		bitBlt(this, 0, 0, &temp, 0, 0, width(), height(), CopyROP);

	}

}






/**
 *
 */
void SongEditor::updateEditor() {
	if(!isVisible()) {
		return;
	}

//	createBackground();
	update();
}




/**
 *
 */
void SongEditor::createBackground() {
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();

	uint nPatterns = song->getPatternList()->getSize();

	static int oldHeight = -1;
	int newHeight = gridHeight * nPatterns;

	if (oldHeight != newHeight) {
		if (newHeight == 0) {
			newHeight = 1;	// the pixmap should not be empty
		}

		temp.resize( width(), newHeight );	// initialize the pixmap
		background.resize( width(), newHeight );	// initialize the pixmap
		sequence.resize( width(), newHeight );	// initialize the pixmap
		temp.fill();
		sequence.fill();
		this->resize( QSize( width(), newHeight ) );
	}

	background.fill();		// fill the pixmap with white color

	QPainter p( &background );
	p.setPen( QColor(0, 0, 0) );


	// horizontal lines (only fill color)
	for (uint i = 0; i < nPatterns; i++) {
		uint y = gridHeight * i;

		// fill line
		for (uint j = 1; j < gridHeight; j++) {
			if ( i % 2 != 0 ) {
				p.setPen( QColor( 240, 240, 255 ) );
				p.drawLine( 0, y + j, ( maxPatternSequence * gridWidth ), y + j );
			}
		}
	}

	// vertical lines
//	QColor res_1(0, 0, 0);
//	QColor res_2(100, 100, 100);
	QColor res_3(150, 150, 150);
	QColor res_4(200, 200, 200);
//	QColor res_5(230, 230, 230);

	for (uint i = 0; i < maxPatternSequence + 1; i++) {
		uint x = i * gridWidth;
		if ( (i % 4) == 0 ) {
			p.setPen( res_3 );
		}
		else {
			p.setPen( res_4 );
		}
		p.drawLine( x, 0, x, gridHeight * nPatterns );
	}

	// horizontal lines
	for (uint i = 0; i < nPatterns + 1; i++) {
		uint y = gridHeight * i;
		p.setPen(QColor(0, 0, 0));
		p.drawLine( 0, y, (maxPatternSequence * gridWidth), y);
	}

	changed = true;
	sequenceChanged = true;
}




/**
 *
 */
void SongEditor::drawSequence() {
	Song* song = (Hydrogen::getInstance())->getSong();

	PatternList *patList = song->getPatternList();
	PatternList *patSeq = song->getPatternSequence();

	uint listLenght = patList->getSize();
	uint seqLenght = patSeq->getSize();

	for (uint i = 0; i < seqLenght; i++) {
		Pattern *pat = patSeq->get( i );

		int position = -1;
		// find the position in pattern list
		for (uint j = 0; j < listLenght; j++) {
			Pattern *pat2 = patList->get( j );
			if (pat == pat2) {
				position = j;
				break;
			}
		}
		drawPattern( i, position );
	}
}





/**
 *
 */
void SongEditor::drawPattern( int pos, int number ) {
	QPainter p( &sequence );
	QColor black(0, 0, 0);

	p.setPen(black);

	int x = gridWidth * pos;
	int y = gridHeight * number;

	// draw the "dot"
	p.drawLine( x + 2, y + 2, x + gridWidth - 2, y + gridHeight - 2 );
	p.drawLine( x + 2, y + gridHeight - 2, x + gridWidth - 2, y + 2 );
}











///////////////////////////////////////////////







/**
 * Constructor
 */
SongEditorPatternList::SongEditorPatternList( QWidget *parent ) : QWidget( parent ), Object( "SongEditorPatternList" )
{
//	cout << "song editor PatternList INIT" << endl;

	m_nWidth = 150;
	m_nGridHeight = (PreferencesMng::getInstance())->getPatternEditorGridHeight();

	nSelectedPattern = 0;
	changed = true;
	this->resize( m_nWidth, initialHeight );

	QPixmap patternIcon;
	string patternIcon_path = string( IMG_PATH ) + string( "/img/songEditor/patternIcon.png" );
	patternIcon.load( patternIcon_path.c_str() );

	patternPopup = new QPopupMenu( this, "patternPopupMenu" );
	patternPopup->insertItem( patternIcon, trUtf8("Edit"),  this, SLOT( patternPopup_edit() ) );
	patternPopup->insertItem( patternIcon, trUtf8("Copy"),  this, SLOT( patternPopup_copy() ) );
	patternPopup->insertItem( patternIcon, trUtf8("Delete"),  this, SLOT( patternPopup_delete() ) );
	patternPopup->insertItem( patternIcon, trUtf8("Properties"),  this, SLOT( patternPopup_properties() ) );

	createBackground();
	update();

}






SongEditorPatternList::~SongEditorPatternList() {
//	cout << "song editor PatternList destroy" << endl;
}





/// Single click, select the next pattern
void SongEditorPatternList::mousePressEvent( QMouseEvent *ev ) {
	int row = (ev->y() / m_nGridHeight);

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	if ( row >= (int)patternList->getSize() ) {
		return;
	}
	nSelectedPattern = row;

	if ( song->getMode() == PATTERN_MODE ) {
		engine->setNextPattern( row );
		if (engine->getState() == READY) {
			engine->setCurrentPattern( patternList->get( row ) );
		}
	}
	else if ( song->getMode() == SONG_MODE ) {
		if (engine->getState() == READY) {
			engine->setCurrentPattern( patternList->get( row ) );
		}
	}

	if (ev->button() == RightButton) {
		patternPopup->popup( QPoint( ev->globalX(), ev->globalY() ) );
	}
}





/**
 * Double click, select the next pattern and opens the pattern editor
 */
void SongEditorPatternList::mouseDoubleClickEvent( QMouseEvent *ev ) {
	int row = (ev->y() / m_nGridHeight);

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	if ( row >= (int)patternList->getSize() ) {
		return;
	}

	nSelectedPattern = row;

	if ( song->getMode() == PATTERN_MODE ) {

		if (engine->getState() == READY) {
			engine->setCurrentPattern( patternList->get( row ) );
			engine->setNextPattern( row );
			HydrogenApp::getInstance()->showPatternEditor(true);
		}
		else if (engine->getState() == PLAYING) {
			engine->setNextPattern( row );
			HydrogenApp::getInstance()->showPatternEditor(true);
		}
	}
	else if ( song->getMode() == SONG_MODE ) {
		if (engine->getState() == READY) {
			engine->setCurrentPattern( patternList->get( row ) );
			HydrogenApp::getInstance()->showPatternEditor(true);
		}
		HydrogenApp::getInstance()->showPatternEditor(true);
	}
}





/**
 *
 */
void SongEditorPatternList::paintEvent( QPaintEvent *ev ) {
	if (!isVisible()) {
		return;
	}

	if (changed) {
		changed = false;
		bitBlt( &temp, 0, 0, &background, 0, 0, m_nWidth, height(), CopyROP );
		setErasePixmap( temp );
	}
}




void SongEditorPatternList::updateEditor()
{
	if(!isVisible()) {
		return;
	}

	update();
}




void SongEditorPatternList::createBackground()
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	uint nPatterns = song->getPatternList()->getSize();

	static int oldHeight = -1;
	int newHeight = m_nGridHeight * nPatterns;

	if (oldHeight != newHeight) {
		if (newHeight == 0) {
			newHeight = 1;	// the pixmap should not be empty
		}
		background.resize( m_nWidth, newHeight );	// initialize the pixmap
		temp.resize( m_nWidth, newHeight );	// initialize the pixmap
		temp.fill();
		this->resize( m_nWidth, newHeight );
	}

	background.fill();		// fill the pixmap with white color

	PreferencesMng *pref = PreferencesMng::getInstance();
	QString family = pref->getApplicationFontFamily().c_str();
	int size = pref->getApplicationFontPointSize();
	QFont font( family, size );


	QFont boldFont( family, size);
	boldFont.setBold( true );

	QPainter p( &background );
	p.setPen( QColor(0, 0, 0) );

	Pattern* currentPat = (Hydrogen::getInstance())->getCurrentPattern();
	int nCurrentPattern = -1;
	// find the position in pattern list
	for (uint i = 0; i < song->getPatternList()->getSize(); i++) {
		Pattern *pat2 = song->getPatternList()->get( i );
		if (currentPat == pat2) {
			nCurrentPattern = i;
			break;
		}
	}
	nSelectedPattern = nCurrentPattern;


	// horizontal lines (only fill color)
	for (uint i = 0; i < nPatterns; i++) {
		uint y = m_nGridHeight * i;

		if ( i == (uint)nCurrentPattern ) {
			// highlighted pattern line
			p.setPen( QColor( 219, 243, 255 ) );
			// fill line
			for (uint j = 1; j < m_nGridHeight; j++) {
				p.drawLine( 0, y + j, m_nWidth, y + j );
			}
		}
		else {
			// fill line
			for (uint j = 1; j < m_nGridHeight; j++) {
				if ( i % 2 != 0 ) {
					p.setPen( QColor( 240, 240, 255 ) );
					p.drawLine( 0, y + j, m_nWidth, y + j );
				}
			}
		}
	}

	// horizontal lines
	for (uint i = 0; i < nPatterns; i++) {
		uint y = m_nGridHeight * i;
		p.setPen(QColor(0, 0, 0));
		p.drawLine( 0, y, m_nWidth, y);

		if (i == (uint)nCurrentPattern) {
			p.setFont( boldFont );
		}
		else {
			p.setFont( font );
		}
		uint text_y = i * m_nGridHeight + m_nGridHeight;
		string patternName = song->getPatternList()->get(i)->getName();
		p.drawText( 5, text_y - 1 - m_nGridHeight, m_nWidth - 5, m_nGridHeight + 2, Qt::AlignVCenter, patternName.c_str() );
	}
	p.drawLine( 0, m_nGridHeight * nPatterns, m_nWidth, m_nGridHeight * nPatterns);

	p.drawLine( 0, 0, 0, height() );
	p.drawLine( m_nWidth - 1, 0, m_nWidth - 1, height() );

	changed = true;
}





void SongEditorPatternList::patternPopup_edit()
{
	HydrogenApp::getInstance()->showPatternEditor(true);
}





void SongEditorPatternList::patternPopup_properties()
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	if ( nSelectedPattern >= (int)patternList->getSize() ) {
		return;
	}
	Pattern *pattern = patternList->get( nSelectedPattern );

	PatternPropertiesDialog *dialog = new PatternPropertiesDialog(this, pattern);
	if (dialog->exec() == QDialog::Accepted) {
		Hydrogen *engine = Hydrogen::getInstance();
		Song *song = engine->getSong();
		song->setModified( true );
		createBackground();
		update();
	}
	delete dialog;
	dialog = NULL;
}






void SongEditorPatternList::patternPopup_delete()
{
	Hydrogen *engine = Hydrogen::getInstance();
	int state = engine->getState();

	if (state == PLAYING) {
		QMessageBox::information( this, "Hydrogen", trUtf8("Can't delete the pattern while the audio engine is playing"));
		return;
	}

	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();

	Pattern *pattern = patternList->get( nSelectedPattern );

	patternList->del(pattern);

	PatternList *patternSequence = song->getPatternSequence();
	patternSequence->del(pattern);

	// check if the pattern is the current pattern
	if (engine->getCurrentPattern() == pattern) {
		engine->setCurrentPattern( NULL );
	}

	delete pattern;
	pattern = NULL;

	( HydrogenApp::getInstance() )->getSongEditorPanel()->updateAll();

	song->setModified(true);
}






void SongEditorPatternList::patternPopup_copy()
{
	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();
	PatternList *patternList = song->getPatternList();
	Pattern *pattern = patternList->get( nSelectedPattern );

	Pattern *newPattern = pattern->copy();
	patternList->add( newPattern );

	// rename the copied pattern
	PatternPropertiesDialog *dialog = new PatternPropertiesDialog( this, newPattern );
	if ( dialog->exec() == QDialog::Accepted ) {
		song->setModified( true );
		if (song->getMode() == PATTERN_MODE) {
//			engine->setNextPattern(patternList->getSize() - 1);	// select the last pattern (the new copied pattern)
			engine->setCurrentPattern( patternList->get( patternList->getSize() - 1 ) );
		}
		else if (engine->getState() == READY) {
			engine->setCurrentPattern( patternList->get( patternList->getSize() - 1 ) );
		}
	}
	else {
		patternList->del( newPattern );
		delete newPattern;
		newPattern = NULL;
	}
	delete dialog;
	dialog = NULL;

	( HydrogenApp::getInstance() )->getSongEditorPanel()->updateAll();
}



////////////////////////////////////



SongEditorPositionRuler::SongEditorPositionRuler( QWidget *parent ) : QWidget( parent ), Object( "SongEditorPositionRuler" )
{
	changed = true;
	this->resize( initialWidth, height );

	temp.resize( initialWidth, height );	// initialize the pixmap
	temp.fill();

	background.resize( initialWidth, height );	// initialize the pixmap

	createBackground();	// create background pixmap

	// create tick position pixmap
	string tickPosition_path = string( IMG_PATH ) + string( "/img/patternEditor/tickPosition.png" );
	bool ok = tickPosition.load(tickPosition_path.c_str());
	if( ok == false ){
		qWarning( "PatternEditor: Error loading pixmap" );
	}

	update();

	m_pTimer = new QTimer(this);
	connect(m_pTimer, SIGNAL(timeout()), this, SLOT(updatePosition()));
	m_pTimer->start(500);

}



SongEditorPositionRuler::~SongEditorPositionRuler() {
	m_pTimer->stop();
}




void SongEditorPositionRuler::updateRuler() {
	update();
}




void SongEditorPositionRuler::createBackground() {
	QColor backgroundColor(230, 230, 230);
	background.fill( backgroundColor );

	PreferencesMng *pref = PreferencesMng::getInstance();
	QString family = pref->getApplicationFontFamily().c_str();
	int size = pref->getApplicationFontPointSize();
	QFont font( family, size );

	QPainter p( &background );
	p.setPen( QColor(0, 0, 0) );
	p.setFont( font );

	QColor res_2(100, 100, 100);
//	QColor res_3(150, 150, 150);
	QColor res_4(200, 200, 200);

	char tmp[10];
	for (uint i = 0; i < maxPatternSequence + 1; i++) {
		uint x = i * gridWidth;
		if ( (i % 4) == 0 ) {
			p.setPen( res_2 );
			sprintf( tmp, "%d", i );
			p.drawText( x - gridWidth, 0, gridWidth * 2, 15, Qt::AlignCenter, tmp );
		}
		else {
			p.setPen( res_4 );
			p.drawLine( x, 5, x, height - 5 );
		}
	}
}




void SongEditorPositionRuler::mousePressEvent( QMouseEvent *ev ) {
	int column = (ev->x() / gridWidth);

	( Hydrogen::getInstance() )->setPatternPos( column );
	update();
}





void SongEditorPositionRuler::paintEvent( QPaintEvent *ev ) {
	if (!isVisible()) {
		return;
	}

	static int oldPosition = -1000;
	int pos = ( Hydrogen::getInstance() )->getPatternPos();
	if ( oldPosition != pos ) {
		changed = true;
		oldPosition = pos;
	}

	if (changed) {
		changed = false;
		bitBlt( &temp, 0, 0, &background, 0, 0, initialWidth, height, CopyROP );

		if (pos != -1) {
			uint x = pos * gridWidth;
			bitBlt( &temp, x + 3, 4, &tickPosition, 0, 0, 11, 8, CopyROP );
		}

		setErasePixmap( temp );
	}
}



void SongEditorPositionRuler::updatePosition()
{
//	infoLog( "[updatePosition]" );
	update();
}













