/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: AlsaMidiDriver.cpp,v 1.10 2004/03/01 10:33:34 comix Exp $
 *
 */

#include "AlsaMidiDriver.h"

// check if ALSA_SEQ support is enabled
#ifdef USE_ALSA_SEQ


#include <pthread.h>

pthread_t midiDriverThread;
pthread_mutex_t midiDriver_mutex;

bool isMidiDriverRunning = false;


snd_seq_t *seq_handle = NULL;
int npfd;
struct pollfd *pfd;
int portId;
int clientId;


/**
 *
 */
void* alsaMidiDriver_thread(void* param)
{
	pthread_mutex_lock( &midiDriver_mutex );

	AlsaMidiDriver *instance = ( AlsaMidiDriver* )param;
	instance->infoLog("alsaMidiDriver_thread() starting");

	// FIXME: gestire gli errori

	/* make sure the other threads can cancel this thread any time */
	if ( pthread_setcancelstate( PTHREAD_CANCEL_ENABLE, NULL ) ) {
		instance->errorLog( "Failed to set the cancel state of the midi thread" );
		pthread_mutex_unlock( &midiDriver_mutex );
		pthread_exit( NULL );
	}
	if ( pthread_setcanceltype( PTHREAD_CANCEL_ASYNCHRONOUS, NULL ) ) {
		instance->errorLog( "Failed to set the cancel state of the midi thread" );
		pthread_mutex_unlock( &midiDriver_mutex );
		pthread_exit( NULL );
	}

	// debug
	if ( seq_handle != NULL ) {
		instance->errorLog( "alsaMidiDriver_thread(): seq_handle != NULL" );
		cout << seq_handle << endl;
		pthread_mutex_unlock( &midiDriver_mutex );
		pthread_exit( NULL );
	}

	if ( snd_seq_open( &seq_handle, "hw", SND_SEQ_OPEN_DUPLEX, 0 ) < 0 ) {
		instance->errorLog( "Error opening ALSA sequencer" );
		pthread_mutex_unlock( &midiDriver_mutex );
		pthread_exit( NULL );
	}

	snd_seq_set_client_name( seq_handle, "Hydrogen" );


	if ( (portId = snd_seq_create_simple_port( 	seq_handle,
									"Hydrogen Midi-In",
									SND_SEQ_PORT_CAP_WRITE |
									SND_SEQ_PORT_CAP_SUBS_WRITE,
									SND_SEQ_PORT_TYPE_APPLICATION
									)
		) < 0) {
		instance->errorLog("Error creating sequencer port.");
		pthread_mutex_unlock( &midiDriver_mutex );
		pthread_exit(NULL);
	}
	clientId = snd_seq_client_id( seq_handle );

	int m_local_addr_port = portId;
	int m_local_addr_client = clientId;
	int m_dest_addr_port = ( PreferencesMng::getInstance() )->getMidiDest_port();
	int m_dest_addr_client = ( PreferencesMng::getInstance() )->getMidiDest_client();

	if ( ( m_dest_addr_port != -1 ) && ( m_dest_addr_client != -1 ) ) {
		snd_seq_port_subscribe_t *subs;
		snd_seq_port_subscribe_alloca( &subs );
		snd_seq_addr_t sender, dest;

		sender.client = m_dest_addr_client;
		sender.port = m_dest_addr_port;
		dest.client = m_local_addr_client;
		dest.port = m_local_addr_port;

		/* set in and out ports */
		snd_seq_port_subscribe_set_sender( subs, &sender );
		snd_seq_port_subscribe_set_dest( subs, &dest );

		/* subscribe */
		int ret = snd_seq_subscribe_port(seq_handle, subs);
		if ( ret < 0 ){
			instance->errorLog( "snd_seq_connect_from(" + toString(m_dest_addr_client) + ":" + toString(m_dest_addr_port) +" error" );
		}
	}

	instance->infoLog( "Midi input port at " + toString(clientId) + ":" + toString(portId) );

	npfd = snd_seq_poll_descriptors_count( seq_handle, POLLIN );
	pfd = ( struct pollfd* )alloca( npfd * sizeof( struct pollfd ) );
	snd_seq_poll_descriptors( seq_handle, pfd, npfd, POLLIN );

	instance->infoLog("MIDI Thread INIT");
	while( isMidiDriverRunning ) {
		if ( poll( pfd, npfd, 100 ) > 0 ) {
			instance->midi_action( seq_handle );
		}
	}
	snd_seq_close ( seq_handle );
	seq_handle = NULL;
	instance->infoLog("MIDI Thread DESTROY");
	pthread_mutex_unlock( &midiDriver_mutex );

	pthread_exit(NULL);
	return NULL;
}





AlsaMidiDriver::AlsaMidiDriver() : Object( "AlsaMidiDriver" )
{
//	infoLog("INIT");
	active = false;
	pthread_mutex_init( &midiDriver_mutex, NULL);
}




AlsaMidiDriver::~AlsaMidiDriver()
{
	if ( isMidiDriverRunning ) {
		close();
	}
//	infoLog("DESTROY");
}




void AlsaMidiDriver::open()
{
	// start main thread
	isMidiDriverRunning = true;
	pthread_attr_t attr;
	pthread_attr_init(&attr);
	pthread_create(&midiDriverThread, &attr, alsaMidiDriver_thread, (void*)this);
}




void AlsaMidiDriver::close()
{
	isMidiDriverRunning = false;

	// the mutex is locked until the midiDriverThread is running
//	infoLog("close(): lock");
	pthread_mutex_lock( &midiDriver_mutex );

//	infoLog("join");
	pthread_join(midiDriverThread, NULL);

	pthread_mutex_unlock( &midiDriver_mutex );
//	infoLog("close(): unlock");
}





void AlsaMidiDriver::midi_action(snd_seq_t *seq_handle)
{
	Hydrogen *engine = Hydrogen::getInstance();
	int nState = engine->getState();
	if ( (nState != READY) && (nState != PLAYING) ) {
		errorLog( "[midi_action] Skipping midi event! Audio engine not ready." );
		return;
	}


	bool useMidiTransport = true;

	snd_seq_event_t *ev;
	do {
		if (!seq_handle) {
			break;
		}
		snd_seq_event_input(seq_handle, &ev);

		if (active) {
			switch ( ev->type ) {
			case SND_SEQ_EVENT_NOTEON:
				noteOnEvent( ev );
				break;

			case SND_SEQ_EVENT_NOTEOFF:
				noteOffEvent( ev );
				break;

			case SND_SEQ_EVENT_CONTROLLER:
				controllerEvent( ev );
				break;

			case SND_SEQ_EVENT_SYSEX: {
				// midi dump
					snd_midi_event_t *seq_midi_parser;
					if( snd_midi_event_new( 32, &seq_midi_parser ) ) {
						errorLog( "[midi_action] error creating midi event parser" );
					}
					unsigned char midi_event_buffer[ 256 ];
					int _bytes_read = snd_midi_event_decode( seq_midi_parser, midi_event_buffer, 32, ev );

					sysexEvent( midi_event_buffer, _bytes_read );
					engine->raiseMidiActivityEvent();
				}
				break;

			case SND_SEQ_EVENT_QFRAME:
//				cout << "quarter frame" << endl;
				if ( useMidiTransport ) {
					playEvent();
				}
				break;

			case SND_SEQ_EVENT_CLOCK:
//				cout << "Midi CLock" << endl;
				break;

			case SND_SEQ_EVENT_SONGPOS:
				warningLog("[midi_action] SND_SEQ_EVENT_SONGPOS not implemented yet");	// 20
				break;

			case SND_SEQ_EVENT_START:
				infoLog("[midi_action] SND_SEQ_EVENT_START");	// 30
				if ( useMidiTransport ) {
					engine->raiseMidiActivityEvent();
					playEvent();
				}
				break;

			case SND_SEQ_EVENT_CONTINUE:
				warningLog("[midi_action] SND_SEQ_EVENT_CONTINUE not implemented yet");	// 31
				break;

			case SND_SEQ_EVENT_STOP:
				infoLog( "[midi_action] SND_SEQ_EVENT_STOP" );	// 32
				if ( useMidiTransport ) {
					engine->raiseMidiActivityEvent();
					stopEvent();
				}
				break;

			case SND_SEQ_EVENT_PITCHBEND:
				break;

			case SND_SEQ_EVENT_PGMCHANGE:
				break;

			case SND_SEQ_EVENT_CLIENT_EXIT:
				infoLog( "[midi_action] SND_SEQ_EVENT_CLIENT_EXIT" );
				break;

			case SND_SEQ_EVENT_PORT_SUBSCRIBED:
				infoLog( "[midi_action] SND_SEQ_EVENT_PORT_SUBSCRIBED" );
				break;

			case SND_SEQ_EVENT_PORT_UNSUBSCRIBED:
				infoLog( "[midi_action] SND_SEQ_EVENT_PORT_UNSUBSCRIBED" );
				break;

			case SND_SEQ_EVENT_SENSING:
				break;

			default:
				warningLog( "[midi_action] AlsaMidiDriver: Unknown MIDI Event. type = " + toString((int)ev->type) );
			}
		}
		snd_seq_free_event( ev );
	} while ( snd_seq_event_input_pending( seq_handle, 0 ) > 0 );
}




void AlsaMidiDriver::midiDump( unsigned char *buffer, int bufSize )
{
	cout << "[midiDump] (" << bufSize << " bytes)\t: ";
	for (int i = 0; i < bufSize; i++) {
		cout << " " << (int)buffer[ i ];
	}
	cout << endl;
}





void AlsaMidiDriver::noteOnEvent( snd_seq_event_t* ev ) {

/*
	not working

	float nDelay = 0;
	switch (ev->flags & SND_SEQ_TIME_STAMP_MASK) {
		case SND_SEQ_TIME_STAMP_TICK:
			cout << "time = " << ev->time.tick << "xxx = " << ev->time.tick / 192 << endl;
			nDelay = (ev->time.tick / 192.0);
			break;

		case SND_SEQ_TIME_STAMP_REAL:
			printf(", time = %d.%09d", (int)ev->time.time.tv_sec, (int)ev->time.time.tv_nsec);
			break;
	}
	cout << "Note on, delay = " << nDelay << endl;
*/

	int midiPortChannel = (PreferencesMng::getInstance())->getMidiPortChannel();
	int channel = ev->data.control.channel;
	int note = ev->data.note.note;
	float velocity = ev->data.note.velocity / 127.0;

	if (velocity == 0) {
		noteOffEvent( ev );
		return;
	}

	bool isChannelValid = true;
	if ( midiPortChannel != -1 ) {
		isChannelValid = ( channel == midiPortChannel );
	}

	Hydrogen *engine = Hydrogen::getInstance();

	bool patternSelect = false;

	if ( isChannelValid ) {
		if ( patternSelect ) {
			int patternNumber = note - 36;
			infoLog( "[noteOnEvent] next pattern = " + toString(patternNumber) );

			engine->setNextPattern( patternNumber );
		}
		else 	{
			// FIXME: calcolare il pan
			float pan_L = 1.0;
			float pan_R = 1.0;

			int voice = note - 36;
			//int position = engine->getRealtimeTickPosition();
			if ( voice < 0 ) voice = 0;
			if ( voice > (MAX_TRACKS -1 ) ) voice = MAX_TRACKS - 1;

			//Instrument *instr = ( song->getInstrumentList() )->get( voice ); // da usare in base alla nota
// 			Note *newNote = new Note( position, velocity, pan_L, pan_R );
// 			newNote->setInstrument( instr );

			engine->addRealtimeNote (voice, velocity, pan_L, pan_R, 0.0, true);

// 			engine->noteOn( newNote );
// 			engine->unlockEngine();
		}
	}
	engine->raiseMidiActivityEvent();
}



void AlsaMidiDriver::noteOffEvent( snd_seq_event_t* ev )
{
//	cout << "note off" << endl;
	if ( (PreferencesMng::getInstance() )->isIgnoringMidiNoteOff() ) {
		return;
	}

	Hydrogen *engine = Hydrogen::getInstance();
	Song *song = engine->getSong();

	int note = ev->data.note.note;
	int voice = note - 36;
	if ( voice < 0 ) voice = 0;
	if ( voice > (MAX_TRACKS -1 ) ) voice = MAX_TRACKS - 1;
	Instrument *instr = ( song->getInstrumentList() )->get( voice ); // da usare in base alla nota
	Note *newNote = new Note( 0, 0, 0, 0 );
	newNote->setInstrument( instr );

	engine->noteOff( newNote );
}




void AlsaMidiDriver::controllerEvent( snd_seq_event_t* ev )
{
	int channel = ev->data.control.channel;
	uint param = ev->data.control.param;
	uint value = ev->data.control.value;
	uint controllerParam = ev->data.control.param;

	switch ( controllerParam ) {
		case 1:	// ModWheel
			infoLog( "[controllerEvent] ModWheel control event" );
			break;

		case 7:	// Volume
			infoLog( "[controllerEvent] Volume control event" );
			break;

		case 10:	// Pan
			infoLog( "[controllerEvent] Pan control event" );
			break;

		case 11:	// Expression
			infoLog( "[controllerEvent] Expression control event" );
			break;

		case 64:	// Sustain
			infoLog( "[controllerEvent] Sustain control event" );
			break;

		case 66:	// Sostenuto
			infoLog( "[controllerEvent] Sostenuto control event" );
			break;

		case 91:	// reverb
			infoLog( "[controllerEvent] Reverb control event" );
			break;

		case 93:	// chorus
			infoLog( "[controllerEvent] Chorus control event" );
			break;

		default:
			warningLog( "[controllerEvent] Unhandled Control event" );
			warningLog( "[controllerEvent] controller param = " + toString(param) );
			warningLog( "[controllerEvent] controller channel = " + toString(channel) );
			warningLog( "[controllerEvent] controller value = " + toString(value) );
	}
}





void AlsaMidiDriver::sysexEvent( unsigned char *buf, int nBytes )
{
//	infoLog( "[sysexEvent]" );

	/*
		General MMC message
		0	1	2	3	4	5
		240	127	id	6	cmd	247

		cmd:
		1	stop
		2	play
		3	Deferred play
		4	Fast Forward
		5	Rewind
		6	Record strobe (punch in)
		7	Record exit (punch out)
		9	Pause


		Goto MMC message
		0	1	2	3	4	5	6	7	8	9	10	11	12
		240	127	id	6	68	6	1	hr	mn	sc	fr	ff	247
	*/


	if ( nBytes == 6 ) {
		if ( ( buf[0] == 240 ) && ( buf[1] == 127 ) && ( buf[2] == 127 ) && ( buf[3] == 6 ) ) {
			switch (buf[4] ) {
				case 1:	// STOP
					infoLog( "[sysexEvent] MMC STOP" );
					stopEvent();
					break;

				case 2:	// PLAY
					warningLog( "[sysexEvent] MMC PLAY not implemented yet." );
					break;

				case 3:	//DEFERRED PLAY
					warningLog( "[sysexEvent] MMC DEFERRED PLAY not implemented yet." );
					break;

				case 4:	// FAST FWD
					warningLog( "[sysexEvent] MMC FAST FWD not implemented yet." );
					break;

				case 5:	// REWIND
					warningLog( "[sysexEvent] MMC REWIND not implemented yet." );
					break;

				case 6:	// RECORD STROBE (PUNCH IN)
					warningLog( "[sysexEvent] MMC PUNCH IN not implemented yet." );
					break;

				case 7:	// RECORD EXIT (PUNCH OUT)
					warningLog( "[sysexEvent] MMC PUNCH OUT not implemented yet." );
					break;

				case 9:	//PAUSE
					warningLog( "[sysexEvent] MMC PAUSE not implemented yet." );
					break;

				default:
					warningLog( "[sysexEvent] Unknown MMC Command" );
					midiDump( buf, nBytes );
			}
		}
	}
	else if ( nBytes == 13 ) {
		warningLog( "[sysexEvent] MMC GOTO Message not implemented yet" );
		midiDump( buf, nBytes );
		int id = buf[2];
		int hr = buf[7];
		int mn = buf[8];
		int sc = buf[9];
		int fr = buf[10];
		int ff = buf[11];
		char tmp[200];
		sprintf( tmp, "[sysexEvent] GOTO %d:%d:%d:%d:%d", hr, mn, sc, fr, ff );
		infoLog( tmp );

	}
	else {
		warningLog( "[sysexEvent] Unknown SysEx message" );
		midiDump( buf, nBytes );
	}

	( Hydrogen::getInstance() )->raiseMidiActivityEvent();
}





void AlsaMidiDriver::playEvent()
{
	if ( ( Hydrogen::getInstance() )->getState() != PLAYING ) {
		( Hydrogen::getInstance() )->start();
	}
}



void AlsaMidiDriver::stopEvent()
{
	if ( ( Hydrogen::getInstance() )->getState() == PLAYING ) {
		( Hydrogen::getInstance() )->stop();
	}
}


vector<MidiPortInfo> AlsaMidiDriver::listAlsaOutputPorts()
{
	vector<MidiPortInfo> list;

	if ( seq_handle == NULL ) {
		return list;
	}

	snd_seq_client_info_t *cinfo;	// client info
	snd_seq_port_info_t *pinfo;	// port info

	snd_seq_client_info_alloca( &cinfo );
	snd_seq_client_info_set_client( cinfo, -1 );

	/* while the next client one the sequencer is avaiable */
	while ( snd_seq_query_next_client( seq_handle, cinfo ) >= 0 ) {
		// get client from cinfo
		int client = snd_seq_client_info_get_client( cinfo );

		// fill pinfo
		snd_seq_port_info_alloca( &pinfo );
		snd_seq_port_info_set_client( pinfo, client );
		snd_seq_port_info_set_port( pinfo, -1 );

		// while the next port is avail
		while ( snd_seq_query_next_port( seq_handle, pinfo ) >= 0 ) {

			/* get its capability */
			int cap =  snd_seq_port_info_get_capability(pinfo);

			if ( snd_seq_client_id( seq_handle ) != snd_seq_port_info_get_client(pinfo) && snd_seq_port_info_get_client(pinfo) != 0 ) {
				// output ports
				if 	(
					(cap & SND_SEQ_PORT_CAP_SUBS_WRITE) != 0 &&
					snd_seq_client_id( seq_handle ) != snd_seq_port_info_get_client(pinfo)
					) {
					MidiPortInfo info;
					info.name = snd_seq_port_info_get_name(pinfo);
					info.client = snd_seq_port_info_get_client(pinfo);
					info.port = snd_seq_port_info_get_port(pinfo);

					list.push_back( info );
				}
/*
				// input ports
				if 	(
					(cap & SND_SEQ_PORT_CAP_SUBS_READ) != 0 &&
					snd_seq_client_id( seq_handle ) != snd_seq_port_info_get_client(pinfo)
					)
				{
					cout << "INPUT" << endl;
					cout << "\tclient = " << snd_seq_port_info_get_client(pinfo) << endl;
					cout << "\tport = " << snd_seq_port_info_get_port(pinfo) << endl;
					cout << "\tname(client) = " << snd_seq_client_info_get_name(cinfo) << endl;
					cout << "\tname(port) = " << snd_seq_port_info_get_name(pinfo) << endl;
				}
*/
			}
		} // while
	} // while

	return list;
}




#endif // USE_ALSA_SEQ





