/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: LocalFileMng.cpp,v 1.18 2004/02/03 11:20:37 comix Exp $
 *
 */

#include "LocalFileMng.h"

#include "tinyxml.h"
#include "../config.h"

//-----------------------------------------------------------------------------
/**
 * Costruttore.
 */
LocalFileMng::LocalFileMng() : Object( "LocalFileMng" )
{
//	infoLog("INIT");
}




//-----------------------------------------------------------------------------
/**
 * Distruttore.
 */
LocalFileMng::~LocalFileMng(){
//	infoLog("DESTROY");
}




/**
 *
 */
vector<string> LocalFileMng::listUserDrumkits() {
	vector<string> list;

	string directory = (PreferencesMng::getInstance())->getUserHome() + "/.hydrogen/data";
	DIR* dirp = opendir( directory.c_str() );
	if( dirp == NULL ) {
		errorLog( "Directory " + string(directory) + " not found" );
	}
	else{
		while( true ) {
			dirent *dirInfo = readdir( dirp );
			if( dirInfo == NULL ){
				break;
			}
			if ( ( dirInfo->d_type != DT_DIR ) && ( dirInfo->d_type != DT_UNKNOWN ) ) {
				warningLog( "listUserDrumkits: file " + directory + "/" + string(dirInfo->d_name) + " is not a directory" );
				continue;
			}

			string filename( dirInfo->d_name );
			if( ( filename == "." ) || ( filename == ".." ) ){
				continue;
			}
			list.push_back( filename );
		}
		closedir( dirp );
	}
	return list;

}




/**
 *
 */
vector<string> LocalFileMng::listSystemDrumkits() {
	vector<string> list;

	string directory = string( DATA_DIR ) + "/drumkits";
	DIR* dirp = opendir( directory.c_str() );
	if( dirp == NULL ) {
		errorLog( "Directory " + string(directory) + " not found" );
	}
	else{
		while( true ) {
			dirent *dirInfo = readdir( dirp );
			if( dirInfo == NULL ){
				break;
			}
			if ( ( dirInfo->d_type != DT_DIR ) && ( dirInfo->d_type != DT_UNKNOWN ) ) {
				warningLog( "listSystemDrumkits: file " + directory + "/" + string(dirInfo->d_name) + " is not a directory" );
				continue;
			}
			string filename( dirInfo->d_name );
			if( ( filename == "." ) || ( filename == ".." ) ){
				continue;
			}
			list.push_back( filename );
		}
		closedir( dirp );
	}
	return list;

}




/**
 *
 */
string LocalFileMng::getDrumkitDirectory( string drumkitName ) {
	// search in system drumkit
	vector<string> systemDrumkits = listSystemDrumkits();
	for ( uint i = 0; i < systemDrumkits.size(); i++ ) {
		if ( systemDrumkits[ i ] == drumkitName ) {
			string path = string( DATA_DIR ) + "/drumkits/";
			return path;
		}
	}

	// search in user drumkit
	vector<string> userDrumkits = listUserDrumkits();
	for ( uint i = 0; i < userDrumkits.size(); i++ ) {
		if ( userDrumkits[ i ] == drumkitName ) {
			string path = (PreferencesMng::getInstance())->getUserHome() + "/.hydrogen/data/";
			return path;
		}
	}

	return "";	// FIXME
}



/**
 *
 */
DrumkitInfo* LocalFileMng::loadDrumkit( string directory ) {
	infoLog( "loadDrumkit " + directory );

	// che if the drumkit.xml file exists
	string drumkitInfoFile = directory + "/drumkit.xml";
	std::ifstream verify( drumkitInfoFile.c_str() , std::ios::in | std::ios::binary );
	if (verify == NULL){
		errorLog( "Load Instrument: Data file " + drumkitInfoFile + " not found." );
		return NULL;
	}

	TiXmlDocument doc( drumkitInfoFile.c_str() );
	doc.LoadFile();

	// root element
	TiXmlNode* drumkitNode;	// root element
	if ( !( drumkitNode = doc.FirstChild( "drumkit_info" ) ) ) {
		errorLog( "Error reading drumkit: drumkit_info node not found" );
		return NULL;
	}

	// Name
	string name = readXmlString( this, drumkitNode, "name", "");
	if (name == "") {
		errorLog( "Error reading drumkit: name node not found" );
		return NULL;
	}

	string author = readXmlString( this, drumkitNode, "author", "undefined author" );
	string info = readXmlString( this, drumkitNode, "info", "defaultInfo" );

	DrumkitInfo *drumkitInfo = new DrumkitInfo();
	drumkitInfo->setName( name );
	drumkitInfo->setAuthor( author );
	drumkitInfo->setInfo( info );

	InstrumentList *instrumentList = new InstrumentList();

	TiXmlNode* instrumentListNode;
	if ( (instrumentListNode = drumkitNode->FirstChild( "instrumentList" ) ) ) {
		// INSTRUMENT NODE
		int instrumentList_count = 0;
		TiXmlNode* instrumentNode = 0;
		for( instrumentNode = instrumentListNode->FirstChild("instrument"); instrumentNode; instrumentNode = instrumentNode->NextSibling("instrument")) {
			instrumentList_count++;

			string id = readXmlString( this, instrumentNode, "id", "" );
			string filename = readXmlString( this, instrumentNode, "filename", "" );
			string name = readXmlString( this, instrumentNode, "name", "" );
			float volume = readXmlFloat( this, instrumentNode, "volume", 1.0f );
			bool isMuted = readXmlBool( this, instrumentNode, "isMuted", false );
			float pan_L = readXmlFloat( this, instrumentNode, "pan_L", 1.0f );
			float pan_R = readXmlFloat( this, instrumentNode, "pan_R", 1.0f );

			Instrument *instr = new Instrument( id, name, "not used", 0, volume );

			// exclude-notes vector
			TiXmlNode* excludeNode = instrumentNode->FirstChild( "exclude" );
			if ( excludeNode ) {
				TiXmlNode* idNode = 0;
				for( idNode = excludeNode->FirstChild( "id"); idNode; idNode = idNode->NextSibling( "id" )) {
					int id = atoi( idNode->FirstChild()->Value() );
					instr->excludeVectId.push_back( id );
				}
			}
			else {
//				warningLog( "Error reading drumkit: exclude node not found" );
			}


			Sample *sample = new Sample( 0, filename );	// used only for filename

			instr->setMuted( isMuted );
			instr->setPan_L( pan_L );
			instr->setPan_R( pan_R );
			instr->setSample( sample );
			instr->setDrumkitName( drumkitInfo->getName() );

			instrumentList->add( instr );
		}
		// prepare the exclude vector
		for (uint nInstr = 0; nInstr < instrumentList->getSize(); nInstr++) {
			Instrument* pInstr = instrumentList->get( nInstr );
			for (uint i = 0; i < pInstr->excludeVectId.size(); i++) {
				int id = pInstr->excludeVectId[ i ];
				Instrument* pExcluded = instrumentList->get( id );
				pInstr->excludeVect.push_back( pExcluded );
			}
		}

	}
	else {
		warningLog( "Error reading drumkit: instrumentList node not found" );
	}
	drumkitInfo->setInstrumentList( instrumentList );

	return drumkitInfo;
}





/**
 *
 */
int LocalFileMng::saveDrumkit( DrumkitInfo *info ) {
	infoLog( "Saving drumkit " + info->getName() );

	// Debug
	info->dump();

	// crete the drumkit directory
	string drumkitDir = ( PreferencesMng::getInstance() )->getUserHome() + string( "/.hydrogen/data/" ) + info->getName();
	mkdir( drumkitDir.c_str(), S_IRWXU );

	// clear all old files in the directory
	string clearCmd = "rm -f " + drumkitDir + "/*";
	system( clearCmd.c_str() );

	// create the drumkit.xml file
	string drumkitXmlFilename = drumkitDir + string( "/drumkit.xml" );

	TiXmlDocument doc( drumkitXmlFilename.c_str() );

	TiXmlElement rootNode( "drumkit_info" );

	writeXmlString( &rootNode, "name", info->getName() );		// name
	writeXmlString( &rootNode, "author", info->getAuthor() );	// author
	writeXmlString( &rootNode, "info", info->getInfo() );		// info

	TiXmlElement instrumentListNode( "instrumentList" );		// instrument list
	uint nInstrument = info->getInstrumentList()->getSize();
	// INSTRUMENT NODE
	for (uint i = 0; i < nInstrument; i++) {
		Instrument *instr = info->getInstrumentList()->get(i);

		// ricavo il nome senza path
		string filename = instr->getSample()->getFilename();
		int index = 0;

		index = filename.rfind( '/' );

		string test = filename;
		filename = filename.substr( index + 1, test.size() - index + 1 );

		string absFilename = drumkitDir + "/" + filename;
		instr->save( absFilename );

		TiXmlElement instrumentNode( "instrument" );

		// id
		writeXmlString( &instrumentNode, "id", instr->getId() );

		// filename
		writeXmlString( &instrumentNode, "filename", filename );

		// name
		writeXmlString( &instrumentNode, "name", instr->getName() );

		// volume
		writeXmlString( &instrumentNode, "volume", toString( instr->getVolume() ) );

		// FIXME: usare writeBool?
		// is muted?
		string value = "false";
		if (instr->isMuted()) {
			value = "true";
		}
		writeXmlString( &instrumentNode, "isMuted", value );

		// pan L
		writeXmlString( &instrumentNode, "pan_L", toString( instr->getPan_L() ) );

		// pan R
		writeXmlString( &instrumentNode, "pan_R", toString( instr->getPan_R() ) );

		// exclude vector
		TiXmlElement excludeNode( "exclude" );
		if (instr->excludeVectId.size() != 0) {
			for (uint i = 0; i <instr->excludeVectId.size(); i++) {
				writeXmlString( &excludeNode, "id" , toString( instr->excludeVectId[ i ] ) );
			}
		}
		instrumentNode.InsertEndChild( excludeNode);

		instrumentListNode.InsertEndChild(instrumentNode);
	}

	rootNode.InsertEndChild(instrumentListNode);

	doc.InsertEndChild( rootNode );
	doc.SaveFile();

	return 0; // ok
}





/**
 *
 */
void LocalFileMng::installDrumkit( string filename ) {
	infoLog( "install drumkit " + filename );

	string dataDir = ( PreferencesMng::getInstance() )->getUserHome() + string( "/.hydrogen/data/" );

	// unpack the drumkit
	string cmd = string( "cd " ) + dataDir + string( "; tar xzf " ) + filename;
	system( cmd.c_str() );
}




/**
 *
 */
int LocalFileMng::uninstallDrumkit( string drumkitName ) {
	infoLog( "uninstall drumkit " + drumkitName );

	// verificare che non sia un drumkit di sistema

	return 0;	// OK
}




//-----------------------------------------------------------------------------
/**
 * Carica la canzone specificata.
 */
Song* LocalFileMng::loadSong(string filename) {

	Song *song = NULL;

	SongReader reader;
	song = reader.readSong(filename);

	return song;
}







//-----------------------------------------------------------------------------
/**
 * Save the song
 */
void LocalFileMng::saveSong(Song *song, string filename) {
	SongWriter writer;

	writer.writeSong(song, filename);
}













//-----------------------------------------------------------------------------
//	Implementation of SongReader class
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
/**
 * Constructor
 */
SongReader::SongReader() : Object( "SongReader" )
{
//	infoLog("init");
}





//-----------------------------------------------------------------------------
/**
 * Destructor
 */
SongReader::~SongReader() {
//	infoLog("destroy");
}



//-----------------------------------------------------------------------------
/**
 * Read a song.
 * return NULL = error reading song file
 */
Song* SongReader::readSong(string filename) {
	infoLog("loading song " + filename);
	Song* song = NULL;
	bool convertTo081Format = false;

	std::ifstream verify(filename.c_str() , std::ios::in | std::ios::binary);	// song file exists??
	if (verify == NULL) {
		errorLog("Song file " + filename + " not found.");
		return NULL;
	}


	TiXmlDocument doc(filename.c_str());
	doc.LoadFile();

	TiXmlNode* songNode;	// root element
	if ( !( songNode = doc.FirstChild("song") ) ) {
		errorLog("Error reading song: song node not found");
		return NULL;
	}


	string sVersion = "";	// version
	sVersion = LocalFileMng::readXmlString( this, songNode, "version", sVersion );
	if ( sVersion != VERSION ) {
		warningLog( "Trying to load a song created with a different version of hydrogen.");
		warningLog( "Song [" + filename + "] saved with version " + sVersion );
		if ( ( sVersion == "0.8.0") || ( sVersion == "0.7.6") || ( sVersion == "0.8.0beta3") || ( sVersion == "0.8.0beta2") ) {
			convertTo081Format = true;
		}
	}

	float fBpm = 120;	// Bpm
	fBpm = LocalFileMng::readXmlFloat( this, songNode, "bpm", fBpm );

	float fVolume = 0.5;	// Volume
	fVolume = LocalFileMng::readXmlFloat( this, songNode, "volume", fVolume );

	float fMetronomeVolume = 0.50f;		// Metronome Volume
	fMetronomeVolume = LocalFileMng::readXmlFloat( this, songNode, "metronomeVolume", fMetronomeVolume );

	string sName = "Untitled Song";	// Name
	sName = LocalFileMng::readXmlString( this, songNode, "name", sName );

	string sAuthor = "Unknown Author";	// Author
	sAuthor = LocalFileMng::readXmlString( this, songNode, "author", sAuthor );

	string sNotes = "";	// Notes
	sNotes = LocalFileMng::readXmlString( this, songNode, "notes", sNotes );

	bool bLoopEnabled = false;		// Loop enabled
	bLoopEnabled = LocalFileMng::readXmlBool( this, songNode, "loopEnabled", bLoopEnabled );

	int nMode = PATTERN_MODE;	// Mode (song/pattern)
	string sMode = "pattern";
	sMode = LocalFileMng::readXmlString( this, songNode, "mode", sMode );
	if ( sMode == "song" ) {
		nMode = SONG_MODE;
	}

	bool bHumanizeTimeEnabled = false;	// humanize time enabled
	bHumanizeTimeEnabled = LocalFileMng::readXmlBool( this, songNode, "humanizeTimeEnabled", bHumanizeTimeEnabled );

	float fHumanizeTimeValue = 0.0;		// humanize time
	fHumanizeTimeValue = LocalFileMng::readXmlFloat( this, songNode, "humanize_time", fHumanizeTimeValue );

	bool bHumanizeVelocityEnabled = false;	// humanize velocity enabled
	bHumanizeVelocityEnabled = LocalFileMng::readXmlBool( this, songNode, "humanizeVelocityEnabled", bHumanizeVelocityEnabled );

	float fHumanizeVelocityValue = 0.0;	// humanize velocity
	fHumanizeVelocityValue = LocalFileMng::readXmlFloat( this, songNode, "humanize_velocity", fHumanizeVelocityValue );

	bool bSwingEnabled = false;		// swing enabled
	bSwingEnabled = LocalFileMng::readXmlBool( this, songNode, "swingEnabled", bSwingEnabled );

	float fSwingFactor = 0.0;	// swing factor
	fSwingFactor = LocalFileMng::readXmlFloat( this, songNode, "swing_factor", fSwingFactor );

	song = new Song( "id", sName, sAuthor, fBpm, fVolume );
	song->setMetronomeVolume( fMetronomeVolume );
	song->setNotes( sNotes );
	song->setLoopEnabled( bLoopEnabled );
	song->setMode( nMode );
	song->setHumanizeTimeEnabled( bHumanizeTimeEnabled );
	song->setHumanizeTimeValue( fHumanizeTimeValue );
	song->setHumanizeVelocityEnabled( bHumanizeVelocityEnabled );
	song->setHumanizeVelocityValue( fHumanizeVelocityValue );
	song->setSwingEnabled( bSwingEnabled );
	song->setSwingFactor( fSwingFactor );


	//  Instrument List
	LocalFileMng localFileMng;
	InstrumentList *instrumentList = new InstrumentList();

	TiXmlNode* instrumentListNode;
	if ( (instrumentListNode = songNode->FirstChild("instrumentList") ) ) {
		// INSTRUMENT NODE
		int instrumentList_count = 0;
		TiXmlNode* instrumentNode = 0;
		for( instrumentNode = instrumentListNode->FirstChild("instrument"); instrumentNode; instrumentNode = instrumentNode->NextSibling("instrument")) {
			instrumentList_count++;

			string sId = "";	// instrument id
			sId = LocalFileMng::readXmlString( this, instrumentNode, "id", sId );

			string sDrumkit = "";	// drumkit
			sDrumkit = LocalFileMng::readXmlString( this, instrumentNode, "drumkit", sDrumkit );

			string sFilename = "";	// filename
			sFilename = LocalFileMng::readXmlString( this, instrumentNode, "filename", sFilename );

			string sName = "";	// name
			sName = LocalFileMng::readXmlString( this, instrumentNode, "name", sName );

			float fVolume = 1.0;	// volume
			fVolume = LocalFileMng::readXmlFloat( this, instrumentNode, "volume", fVolume );

			bool bIsMuted = false;	// is muted
			bIsMuted = LocalFileMng::readXmlBool( this, instrumentNode, "isMuted", bIsMuted );

			float fPan_L = 1.0;	// pan L
			fPan_L = LocalFileMng::readXmlFloat( this, instrumentNode, "pan_L", fPan_L );

			float fPan_R = 1.0;	// pan R
			fPan_R = LocalFileMng::readXmlFloat( this, instrumentNode, "pan_R", fPan_R );

			float fFX1Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX1Level", 0.0 );		// FX level
			float fFX2Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX2Level", 0.0 );		// FX level
			float fFX3Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX3Level", 0.0 );		// FX level
			float fFX4Level = LocalFileMng::readXmlFloat( this, instrumentNode, "FX4Level", 0.0 );		// FX level


			string drumkitPath = "";
			if ( sDrumkit != "" ) {
				drumkitPath = localFileMng.getDrumkitDirectory( sDrumkit ) + sDrumkit + "/";
			}
			Instrument *instr = Instrument::load( drumkitPath + sFilename );
			if (instr) { // is instrument valid?
				instr->getSample()->setFilename( sFilename );
			}
			else {
				errorLog( "Error Loading song: NULL instrument, now using /emptySample.wav" );
				instr = Instrument::load( string(DATA_DIR) + "/emptySample.wav" );
			}

			// exclude-notes vector
			TiXmlNode* excludeNode = instrumentNode->FirstChild( "exclude" );
			if ( excludeNode ) {
				TiXmlNode* idNode = 0;
				for( idNode = excludeNode->FirstChild( "id"); idNode; idNode = idNode->NextSibling( "id" )) {
					int id = atoi( idNode->FirstChild()->Value() );
					instr->excludeVectId.push_back( id );
				}
			}
			else {
				warningLog( "Error loading song: exclude node not found" );
			}


			instr->setId( sId );
			instr->setVolume( fVolume );
			instr->setName( sName );
			instr->setMuted( bIsMuted );
			instr->setPan_L( fPan_L );
			instr->setPan_R( fPan_R );
			if ( sDrumkit != "" ) {
				instr->setDrumkitName( sDrumkit );
			}
			instr->setFXLevel( 0, fFX1Level );
			instr->setFXLevel( 1, fFX2Level );
			instr->setFXLevel( 2, fFX3Level );
			instr->setFXLevel( 3, fFX4Level );
			instrumentList->add(instr);
		}
		// prepare the exclude vector
		for (uint nInstr = 0; nInstr < instrumentList->getSize(); nInstr++) {
			Instrument* pInstr = instrumentList->get( nInstr );
			if (pInstr) {
				for (uint i = 0; i < pInstr->excludeVectId.size(); i++) {
					int id = pInstr->excludeVectId[ i ];
					Instrument* pExcluded = instrumentList->get( id );
					pInstr->excludeVect.push_back( pExcluded );
				}
			}
			else {
				errorLog( "[SongReader::readSong] pInstr == NULL" );
			}
		}


		song->setInstrumentList( instrumentList );
		if ( instrumentList_count != MAX_TRACKS ) {
			warningLog( "Instrument number != MAX_INSTRUMENTS");

			// create the missing instruments with empty sample
			uint nInstrMissing = MAX_TRACKS - instrumentList_count;
			for (uint i = 0; i < nInstrMissing; i++) {
				int instrId = instrumentList_count + i;
				warningLog( "Creating empty instrument");

				Instrument *instr = Instrument::load( string(DATA_DIR) + "/emptySample.wav" );
				char idStr[10];
				sprintf(idStr,"%d", instrId);
				instr->setId( idStr );
				instr->setVolume( 0.1 );
				instr->setName( "Empty" );
				instr->setMuted( false );
				instr->setPan_L( 1.0 );
				instr->setPan_R( 1.0 );
				instrumentList->add( instr );
			}
		}
	}
	else {
		errorLog("Error reading song: instrumentList node not found");
		delete song;
		return NULL;
	}





	// Pattern list
	TiXmlNode* patterns = songNode->FirstChild("patternList");

	PatternList *patternList = new PatternList();
	int pattern_count = 0;
	TiXmlNode* patternNode = 0;
	for (patternNode = patterns->FirstChild("pattern"); patternNode; patternNode = patternNode->NextSibling( "pattern" )) {
		pattern_count++;
		Pattern *pat = getPattern(patternNode, instrumentList);
		if (pat) {
			patternList->add(pat);
		}
		else {
			errorLog( "Error loading pattern" );
			delete patternList;
			delete song;
			return NULL;
		}
	}
	song->setPatternList(patternList);


	// Pattern sequence
	TiXmlNode* patternSequenceNode = songNode->FirstChild("patternSequence");

	PatternList *patternSequence = new PatternList();

	int patternSequence_count = 0;
	TiXmlNode* patternId = 0;
	for (patternId = patternSequenceNode->FirstChild("patternID"); patternId; patternId = patternId->NextSibling("patternID")) {
		patternSequence_count++;
		string patId = patternId->FirstChild()->Value();

		Pattern *pat = NULL;

		for (uint i = 0; i < patternList->getSize(); i++) {
			Pattern *tmp = patternList->get(i);
			if (tmp) {
				if (tmp->getName() == patId) {
					pat = tmp;
					break;
				}
			}
		}
		patternSequence->add( pat );
	}
	song->setPatternSequence(patternSequence);


	// LADSPA FX
	TiXmlNode* ladspaNode = songNode->FirstChild( "ladspa" );
	if (ladspaNode) {
		int nFX = 0;
		TiXmlNode* fxNode;
		for (fxNode = ladspaNode->FirstChild("fx"); fxNode; fxNode = fxNode->NextSibling("fx")) {
			string sName = LocalFileMng::readXmlString( this, fxNode, "name", "" );
			string sFilename = LocalFileMng::readXmlString( this, fxNode, "filename", "" );
			bool bEnabled = LocalFileMng::readXmlBool( this, fxNode, "enabled", false );
			float fVolume = LocalFileMng::readXmlFloat( this, fxNode, "volume", 1.0 );

			if (sName != "no plugin" ) {
				// FIXME: il caricamento va fatto fare all'engine, solo lui sa il samplerate esatto
				LadspaFX* pFX = LadspaFX::load( sFilename, sName, 44100 );
				song->setLadspaFX( nFX, pFX );
				if (pFX) {
					pFX->setEnabled( bEnabled );
					pFX->setVolume( fVolume );
					TiXmlNode* inputControlNode;
					for ( inputControlNode = fxNode->FirstChild("inputControlPort"); inputControlNode; inputControlNode = inputControlNode->NextSibling("inputControlPort")) {
						string sName = LocalFileMng::readXmlString( this, inputControlNode, "name", "" );
						float fValue = LocalFileMng::readXmlFloat( this, inputControlNode, "value", 0.0 );

						for (uint nPort = 0; nPort < pFX->inputControlPorts.size(); nPort++) {
							LadspaControlPort *port = pFX->inputControlPorts[ nPort ];
							if ( string(port->sName) == sName) {
								port->fControlValue = fValue;
							}
						}
					}

					TiXmlNode* outputControlNode;
					for ( outputControlNode = fxNode->FirstChild("outputControlPort"); outputControlNode; outputControlNode = outputControlNode->NextSibling("outputControlPort")) {
					}
				}
			}
			nFX++;
		}
	}
	else {
		warningLog( "[SongReader::readSong()] ladspa node not found" );
	}


	song->setModified(false);
	song->setFilename(filename);


	if (convertTo081Format) {
		warningLog("Converting from old song format...");
		for (uint nPattern = 0; nPattern < song->getPatternList()->getSize(); nPattern++) {
			Pattern *pat = song->getPatternList()->get(nPattern);
			pat->setSize( pat->getSize() * 192 / 64 );

			for (uint nSequence = 0; nSequence < MAX_TRACKS; nSequence++) {
				Sequence *sequence = pat->getSequenceList()->get(nSequence);
				for (uint nNote = 0; nNote < MAX_NOTES; nNote++) {
					Note *note = sequence->noteList[nNote];
					if (note) {
						int nPos = note->getPosition() * 192 / 64;
						note->setPosition( nPos );
					}
				}
			}
		}
		warningLog("Save song with new format");
		SongWriter writer;
		writer.writeSong(song, filename);
		delete song;
		return readSong(filename);
	}

	return song;
}



//-----------------------------------------------------------------------------
/**
 *
 */
Pattern* SongReader::getPattern(TiXmlNode* pattern, InstrumentList* instrList){
	Pattern *pat = NULL;

	string sName = "";	// name
	sName = LocalFileMng::readXmlString( this, pattern, "name", sName );

	int nSize = MAX_NOTES;
	nSize = LocalFileMng::readXmlInt( this, pattern, "size", nSize );
	if (nSize > MAX_NOTES) {
		errorLog( "[SongReader::getPattern] Pattern size > MAX_NOTES" );
		return NULL;
	}


	pat = new Pattern( sName, nSize );

	SequenceList *sequenceList = new SequenceList();

	TiXmlNode* sequenceListNode = pattern->FirstChild("sequenceList");

	int sequence_count = 0;
	TiXmlNode* sequenceNode = 0;
	for (sequenceNode = sequenceListNode->FirstChild("sequence"); sequenceNode; sequenceNode = sequenceNode->NextSibling("sequence")) {
		sequence_count++;
		Sequence *seq = getSequence(sequenceNode, instrList);

		sequenceList->add(seq);
	}
	pat->setSequenceList(sequenceList);
	if ( sequence_count != MAX_TRACKS ) {
		warningLog( "sequence number != MAX_INSTRUMENTS" );

		uint nMissingSequences = MAX_TRACKS - sequence_count;
		for (uint i = 0; i < nMissingSequences; i++) {
			warningLog( "Creating empty Sequence");
			Sequence *seq = new Sequence( "empty seq" );
			sequenceList->add(seq);
		}
	}
	return pat;
}





//-----------------------------------------------------------------------------
/**
 *
 */
Sequence* SongReader::getSequence(TiXmlNode* sequence, InstrumentList* instrList) {
	Sequence* seq = NULL;

	string sName = "";
	sName = LocalFileMng::readXmlString( this, sequence, "name", sName );

	seq = new Sequence( sName );

	TiXmlNode* noteListNode = sequence->FirstChild( "noteList" );

	TiXmlNode* noteNode = 0;
	for (noteNode = noteListNode->FirstChild("note"); noteNode; noteNode = noteNode->NextSibling("note")) {
		Note *note = getNote(noteNode, instrList);
		seq->noteList[note->getPosition()] = note;
	}

	return seq;
}




//-----------------------------------------------------------------------------
/**
 *
 */
Note* SongReader::getNote(TiXmlNode* noteNode, InstrumentList *instrList){
	Note* note = NULL;

	uint nPosition = LocalFileMng::readXmlInt( this, noteNode, "position", 0 );
	float fVelocity = LocalFileMng::readXmlFloat( this, noteNode, "velocity", 0.8 );
	float fPan_L = LocalFileMng::readXmlFloat( this, noteNode, "pan_L", 1.0 );
	float fPan_R = LocalFileMng::readXmlFloat( this, noteNode, "pan_R", 1.0 );
	float nPitch = LocalFileMng::readXmlFloat( this, noteNode, "pitch", 0.0 );

	TiXmlNode* instrNode = noteNode->FirstChild("instrument");
	string instrId = instrNode->FirstChild()->Value();

	Instrument *instrRef = NULL;
	// search instrument by ref
	for (uint i = 0; i < instrList->getSize(); i++) {
		Instrument *instr = instrList->get(i);
		if (instrId == instr->getId()) {
			instrRef = instr;
			break;
		}
	}
	if (instrRef == NULL) {
		errorLog("instrRef NULL");
	}

	note = new Note( nPosition, fVelocity, fPan_L, fPan_R, nPitch);
	note->setInstrument(instrRef);

	return note;
}






//-----------------------------------------------------------------------------
//	Implementation of SongWriter class
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
/**
 * Constructor
 */
SongWriter::SongWriter() : Object( "SongWriter" )
{
//	infoLog("init");
}





//-----------------------------------------------------------------------------
/**
 * Destructor
 */
SongWriter::~SongWriter() {
//	infoLog("destroy");
}





//-----------------------------------------------------------------------------
/**
 *
 */
void SongWriter::writeSong(Song *song, string filename) {
	infoLog( "Saving song " + filename );

	// FIXME: verificare se e' possibile scrivere il file
	// FIXME: verificare che il file non sia gia' esistente
	// FIXME: effettuare copia di backup per il file gia' esistente

	TiXmlDocument doc(filename);

	TiXmlElement songNode("song");

	LocalFileMng::writeXmlString( &songNode, "version", string(VERSION) );
	LocalFileMng::writeXmlString( &songNode, "bpm", toString( song->getBpm() ) );
	LocalFileMng::writeXmlString( &songNode, "volume", toString( song->getVolume() ) );
	LocalFileMng::writeXmlString( &songNode, "metronomeVolume", toString( song->getMetronomeVolume() ) );
	LocalFileMng::writeXmlString( &songNode, "name", song->getName() );
	LocalFileMng::writeXmlString( &songNode, "author", song->getAuthor() );
	LocalFileMng::writeXmlString( &songNode, "notes", song->getNotes() );
	LocalFileMng::writeXmlBool( &songNode, "loopEnabled", song->isLoopEnabled() );

	if (song->getMode() == SONG_MODE ) {
		LocalFileMng::writeXmlString( &songNode, "mode", string("song") );
	}
	else {
		LocalFileMng::writeXmlString( &songNode, "mode", string("pattern") );
	}

	LocalFileMng::writeXmlBool( &songNode, "humanizeTimeEnabled", song->isHumanizeTimeEnabled() );
	LocalFileMng::writeXmlString( &songNode, "humanize_time", toString( song->getHumanizeTimeValue() ) );
	LocalFileMng::writeXmlBool( &songNode, "humanizeVelocityEnabled", song->isHumanizeVelocityEnabled() );
	LocalFileMng::writeXmlString( &songNode, "humanize_velocity", toString( song->getHumanizeVelocityValue() ) );
	LocalFileMng::writeXmlBool( &songNode, "swingEnabled", song->isSwingEnabled() );
	LocalFileMng::writeXmlString( &songNode, "swing_factor", toString( song->getSwingFactor() ) );

	// instrument list
	TiXmlElement instrumentListNode("instrumentList");
	uint nInstrument = song->getInstrumentList()->getSize();

	// INSTRUMENT NODE
	for (uint i = 0; i < nInstrument; i++) {
		Instrument *instr = song->getInstrumentList()->get(i);

		TiXmlElement instrumentNode("instrument");

		LocalFileMng::writeXmlString( &instrumentNode, "id", instr->getId() );
		LocalFileMng::writeXmlString( &instrumentNode, "drumkit", instr->getDrumkitName() );
		LocalFileMng::writeXmlString( &instrumentNode, "filename", instr->getSample()->getFilename() );
		LocalFileMng::writeXmlString( &instrumentNode, "name", instr->getName() );
		LocalFileMng::writeXmlString( &instrumentNode, "volume", toString( instr->getVolume() ) );
		LocalFileMng::writeXmlBool( &instrumentNode, "isMuted", instr->isMuted() );
		LocalFileMng::writeXmlString( &instrumentNode, "pan_L", toString( instr->getPan_L() ) );
		LocalFileMng::writeXmlString( &instrumentNode, "pan_R", toString( instr->getPan_R() ) );

		LocalFileMng::writeXmlString( &instrumentNode, "FX1Level", toString( instr->getFXLevel(0) ) );
		LocalFileMng::writeXmlString( &instrumentNode, "FX2Level", toString( instr->getFXLevel(1) ) );
		LocalFileMng::writeXmlString( &instrumentNode, "FX3Level", toString( instr->getFXLevel(2) ) );
		LocalFileMng::writeXmlString( &instrumentNode, "FX4Level", toString( instr->getFXLevel(3) ) );

		TiXmlElement excludeNode( "exclude" );
		if (instr->excludeVectId.size() != 0) {
			for (uint i = 0; i <instr->excludeVectId.size(); i++) {
				LocalFileMng::writeXmlString( &excludeNode, "id" , toString( instr->excludeVectId[ i ] ) );
			}
		}
		instrumentNode.InsertEndChild( excludeNode);
		instrumentListNode.InsertEndChild(instrumentNode);
	}
	songNode.InsertEndChild(instrumentListNode);


	// pattern list
	TiXmlElement patternListNode("patternList");

	uint nPatterns = song->getPatternList()->getSize();
	for (uint i = 0; i < nPatterns; i++) {
		Pattern *pat = song->getPatternList()->get(i);

		// pattern
		TiXmlElement patternNode("pattern");
		LocalFileMng::writeXmlString( &patternNode, "name", pat->getName() );
		LocalFileMng::writeXmlString( &patternNode, "size", toString( pat->getSize() ) );


		// sequence list
		TiXmlElement sequenceListNode("sequenceList");

		uint nSequences = pat->getSequenceList()->getSize();
		for (uint j = 0; j < nSequences; j++) {
			Sequence *seq = pat->getSequenceList()->get(j);

			// Sequence
			TiXmlElement sequenceNode("sequence");
			LocalFileMng::writeXmlString( &sequenceNode, "name", seq->getName() );

			// Note List
			TiXmlElement noteListNode("noteList");

			uint nNotes = MAX_NOTES;
			for (uint y = 0; y < nNotes; y++) {
				Note *note = seq->noteList[y];
				if (note != NULL) {
					// note
					TiXmlElement noteNode("note");
					LocalFileMng::writeXmlString( &noteNode, "position", toString( note->getPosition() ) );
					LocalFileMng::writeXmlString( &noteNode, "velocity", toString( note->getVelocity() ) );
					LocalFileMng::writeXmlString( &noteNode, "pan_L", toString( note->getPan_L() ) );
					LocalFileMng::writeXmlString( &noteNode, "pan_R", toString( note->getPan_R() ) );
					LocalFileMng::writeXmlString( &noteNode, "pitch", toString( note->getPitch() ) );
					LocalFileMng::writeXmlString( &noteNode, "instrument", note->getInstrument()->getId() );
					noteListNode.InsertEndChild(noteNode);
				}
			}
			sequenceNode.InsertEndChild(noteListNode);
			sequenceListNode.InsertEndChild(sequenceNode);
		}
		patternNode.InsertEndChild(sequenceListNode);
		patternListNode.InsertEndChild(patternNode);
	}
	songNode.InsertEndChild(patternListNode);



	// pattern sequence
	TiXmlElement patternSequenceNode( "patternSequence" );
	uint nPatternSequence = song->getPatternSequence()->getSize();
	for ( uint i = 0; i < nPatternSequence; i++ ) {
		Pattern *pat = song->getPatternSequence()->get( i );
		string patName = "NULL";
		if (pat) {
			patName = pat->getName();
		}
		LocalFileMng::writeXmlString( &patternSequenceNode, "patternID", patName );
	}
	songNode.InsertEndChild( patternSequenceNode );


	// LADSPA FX
	TiXmlElement ladspaFxNode( "ladspa" );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		TiXmlElement fxNode( "fx" );

		LadspaFX *pFX = song->getLadspaFX(nFX);

		if ( pFX ) {
			LocalFileMng::writeXmlString( &fxNode, "name", pFX->getPluginLabel() );
			LocalFileMng::writeXmlString( &fxNode, "filename", pFX->getLibraryPath() );
			LocalFileMng::writeXmlBool( &fxNode, "enabled", pFX->isEnabled() );
			LocalFileMng::writeXmlString( &fxNode, "volume", toString( pFX->getVolume() ) );
			for (uint nControl = 0; nControl < pFX->inputControlPorts.size(); nControl++) {
				LadspaControlPort *pControlPort = pFX->inputControlPorts[ nControl ];
				TiXmlElement controlPortNode( "inputControlPort" );
				LocalFileMng::writeXmlString( &controlPortNode, "name", pControlPort->sName );
				LocalFileMng::writeXmlString( &controlPortNode, "value", toString( pControlPort->fControlValue ) );
				fxNode.InsertEndChild( controlPortNode );
			}
			for (uint nControl = 0; nControl < pFX->outputControlPorts.size(); nControl++) {
				LadspaControlPort *pControlPort = pFX->inputControlPorts[ nControl ];
				TiXmlElement controlPortNode( "outputControlPort" );
				LocalFileMng::writeXmlString( &controlPortNode, "name", pControlPort->sName );
				LocalFileMng::writeXmlString( &controlPortNode, "value", toString( pControlPort->fControlValue ) );
				fxNode.InsertEndChild( controlPortNode );
			}
		}
		else {
			LocalFileMng::writeXmlString( &fxNode, "name", string("no plugin") );
			LocalFileMng::writeXmlString( &fxNode, "filename", string("-") );
			LocalFileMng::writeXmlBool( &fxNode, "enabled", false );
			LocalFileMng::writeXmlString( &fxNode, "volume", toString( 0.0 ) );
		}

		ladspaFxNode.InsertEndChild( fxNode );
	}

	songNode.InsertEndChild( ladspaFxNode );




	doc.InsertEndChild(songNode);
        doc.SaveFile();

	song->setModified(false);
	song->setFilename(filename);
}





/**
 *
 */
string LocalFileMng::readXmlString( Object* obj, TiXmlNode* parent, string nodeName, string defaultValue )
{
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			return node->FirstChild()->Value();
		}
		else {
			obj->warningLog( "Using default value in " + nodeName );
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}


float LocalFileMng::readXmlFloat( Object* obj, TiXmlNode* parent, string nodeName, float defaultValue )
{
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			//float res = atof( node->FirstChild()->Value() );
			float res = stringToFloat( node->FirstChild()->Value() );
			return res;
		}
		else {
			obj->warningLog( "Using default value in " + nodeName );
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}


int LocalFileMng::readXmlInt( Object* obj, TiXmlNode* parent, string nodeName, int defaultValue )
{
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			return atoi( node->FirstChild()->Value() );
		}
		else {
			obj->warningLog( "Using default value in " + nodeName );
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}


/**
 *
 */
bool LocalFileMng::readXmlBool( Object* obj, TiXmlNode* parent, string nodeName, bool defaultValue ) {
	TiXmlNode* node;
	if ( ( node = parent->FirstChild( nodeName.c_str() ) ) ) {
		if ( node->FirstChild() ) {
			if ( string( node->FirstChild()->Value() ) == "true" ) {
				return true;
			}
			else {
				return false;
			}
		}
		else {
			obj->warningLog( "Using default value in " + nodeName );
			return defaultValue;
		}
	}
	else {
		obj->warningLog( "Error reading XML node: " + nodeName + " node not found" );
		return defaultValue;
	}
}





/**
 *
 */
void LocalFileMng::writeXmlString( TiXmlNode *parent, string name, string text ) {
	TiXmlElement versionNode( name );
	TiXmlText versionText( text );
	versionNode.InsertEndChild( versionText );
	parent->InsertEndChild( versionNode );
}





/**
 *
 */
void LocalFileMng::writeXmlBool( TiXmlNode *parent, string name, bool value ) {
	if (value) {
		writeXmlString( parent, name, string("true") );
	}
	else {
		writeXmlString( parent, name, string("false") );
	}
}


