/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PreferencesMng.cpp,v 1.17 2004/03/01 10:33:34 comix Exp $
 *
 */

#include <stdlib.h>
#include "PreferencesMng.h"

#include "tinyxml.h"
#include "config.h"
#include "LocalFileMng.h"

PreferencesMng* PreferencesMng::instance = NULL;



//----------------------------------------------------------------------------
/**
 * Return an instance of PreferencesMng
 */
PreferencesMng* PreferencesMng::getInstance() {
	if (instance == NULL) {
		instance = new PreferencesMng();
	}

	return instance;
}




//----------------------------------------------------------------------------
/**
 * Constructor
 */
PreferencesMng::PreferencesMng() : Object( "PreferencesMng" )
{
	infoLog( "INIT" );

	demoPath = string(DATA_DIR) + "/demo_songs/";

	// General properties
	restoreLastSong = true;
	lastSongFilename = "";
	hearNewNotes = true;

	char * ladpath = getenv("LADSPA_PATH");
	if (ladpath) {
	   m_sLadspaPath = ladpath;
	}
	else {
	   m_sLadspaPath = "/usr/lib/ladspa";
	}

	recordEvents = false;
	quantizeEvents = true;

	// Audio engine properties
	setAudioDriver( "Oss" );
	useMetronome = false;
	metronomeVolume = 0.5f;
	maxNotes = 32;

	// OSS Driver properties
	bufferSize = 1024;
	sampleRate = 44100;
	ossDevice = "/dev/dsp";

	// Jack driver properties
	jackPortName1 = "alsa_pcm:playback_1";
	jackPortName2 = "alsa_pcm:playback_2";
	m_jackTransportMode = NO_JACK_TRANSPORT;
	jackTrackOuts = false;
	jackConnectDefaults = true;

	// Alsa MIDI driver properties
	midiPortChannel = -1;
	midiDest_name = "-";
	midiDest_client = -1;
	midiDest_port = -1;
	m_bIgnoreMidiNoteOff = true;

	// GUI Properties
	m_sQTStyle = "";
	interfaceMode = "Child frame";
	applicationFontFamily = "Helvetica";
	applicationFontPointSize = 10;
	mixerFontFamily = "Helvetica";
	mixerFontPointSize = 8;
	m_nPatternEditorGridHeight = 16;
	m_nPatternEditorGridWidth = 3;
	mixerFalloffSpeed = FALLOFF_NORMAL;
	m_nPatternEditorGridResolution = 8;
	m_bPatternEditorUsingTriplets = false;
	m_bShowInstrumentPeaks = true;

	// window properties
	mainFormProperties.x = 0;
	mainFormProperties.y = 0;
	mainFormProperties.width = 723;
	mainFormProperties.height = 645;

	mixerProperties.x = 10;
	mixerProperties.y = 350;
	mixerProperties.width = 829;
	mixerProperties.height = 220;

	patternEditorProperties.x = 280;
	patternEditorProperties.y = 100;
	patternEditorProperties.width = 706;
	patternEditorProperties.height = 302;

	songEditorProperties.x = 10;
	songEditorProperties.y = 10;
	songEditorProperties.width = 600;
	songEditorProperties.height = 250;

	drumkitManagerProperties.x = 500;
	drumkitManagerProperties.y = 20;

	audioEngineInfoProperties.x = 720;
	audioEngineInfoProperties.y = 120;

	m_ladspaProperties[0].x = 100;
	m_ladspaProperties[0].y = 100;

	m_ladspaProperties[1].x = 110;
	m_ladspaProperties[1].y = 110;

	m_ladspaProperties[2].x = 120;
	m_ladspaProperties[2].y = 120;

	m_ladspaProperties[3].x = 130;
	m_ladspaProperties[3].y = 130;

	loadPreferences();
}





//----------------------------------------------------------------------------
/**
 * Destructor
 */
PreferencesMng::~PreferencesMng() {
	savePreferences();
	infoLog( "DESTROY" );
	instance = NULL;
}




//----------------------------------------------------------------------------
/**
 * Get user's home directory.
 */
string PreferencesMng::getUserHome() {
	string home;
	struct passwd *pwdbuf;

	pwdbuf = getpwuid(getuid());
	home = pwdbuf->pw_dir;

	return home;
}




//----------------------------------------------------------------------------
/**
 * Load the preferences file
 */
void PreferencesMng::loadPreferences() {
	bool recreate = false;	// configuration file must be recreated?
	string prefDir = getUserHome() + "/.hydrogen";
	string dataDir = getUserHome() + "/.hydrogen/data";
	string filename = prefDir + "/hydrogen.conf";

	infoLog("Loading preferences file in " + filename);

	// pref directory exists?
	DIR* dirp = opendir( prefDir.c_str() );
	if ( dirp == NULL ) {
		warningLog( "Configuration directory not found." );
		createPreferencesDirectory();
	}

	// data directory exists?
	DIR* dirp2 = opendir( dataDir.c_str() );
	if ( dirp2 == NULL ) {
		warningLog( "Data directory not found." );
		createDataDirectory();
	}


	// pref file exists?
	std::ifstream input(filename.c_str() , std::ios::in | std::ios::binary);
	if (input){
		// read preferences file
		TiXmlDocument doc(filename.c_str());
		doc.LoadFile();

		TiXmlNode* rootNode;
		if ( (rootNode = doc.FirstChild("hydrogen_preferences")) ) {

			// version
			string version = LocalFileMng::readXmlString( this, rootNode, "version", "" );
			if ( version == "" ) {
				recreate = true;
			}

			//////// GENERAL ///////////
			m_sLadspaPath = LocalFileMng::readXmlString( this, rootNode, "ladspaPath", m_sLadspaPath );
			restoreLastSong = LocalFileMng::readXmlBool( this, rootNode, "restoreLastSong", restoreLastSong );
			hearNewNotes = LocalFileMng::readXmlBool( this, rootNode, "hearNewNotes", hearNewNotes );
			recordEvents = LocalFileMng::readXmlBool( this, rootNode, "recordEvents", recordEvents );
			quantizeEvents = LocalFileMng::readXmlBool( this, rootNode, "quantizeEvents", quantizeEvents );

			TiXmlNode* pRecentUsedSongsNode = rootNode->FirstChild( "recentUsedSongs" );
			if ( pRecentUsedSongsNode ) {
				TiXmlNode* pSongNode = 0;
				for( pSongNode = pRecentUsedSongsNode->FirstChild("song"); pSongNode; pSongNode = pSongNode->NextSibling( "song" ) ) {
					string sFilename = pSongNode->FirstChild()->Value();
					m_recentFiles.push_back( sFilename );
				}
			}
			else {
				warningLog( "Error reading configuration file: recentUsedSongs node not found" );
			}


			/////////////// AUDIO ENGINE //////////////
			TiXmlNode* audioEngineNode;
			if ( !(audioEngineNode = rootNode->FirstChild( "audio_engine" ) ) ) {
				warningLog( "Error reading configuration file: audio_engine node not found" );
				recreate = true;
			}
			else {
				// Audio driver
				audioDriver = LocalFileMng::readXmlString( this, audioEngineNode, "audio_driver", audioDriver );

				// Use metronome
				useMetronome = LocalFileMng::readXmlBool( this, audioEngineNode, "use_metronome", useMetronome );

				// Metronome volume
				metronomeVolume = LocalFileMng::readXmlFloat( this, audioEngineNode, "metronome_volume", metronomeVolume );

				// Max notes
				maxNotes = LocalFileMng::readXmlInt( this, audioEngineNode, "maxNotes", maxNotes );

				//// OSS DRIVER ////
				TiXmlNode* ossDriverNode;
				if ( !(ossDriverNode = audioEngineNode->FirstChild( "oss_driver" ) ) ) {
					warningLog("Error reading configuration file: oss_driver node not found");
					recreate = true;
				}
				else {
					// Buffer size
					bufferSize = LocalFileMng::readXmlInt( this, ossDriverNode, "buffer_size", bufferSize );

					// Sample rate
					sampleRate = LocalFileMng::readXmlInt( this, ossDriverNode, "samplerate", sampleRate );

					ossDevice = LocalFileMng::readXmlString( this, ossDriverNode, "ossDevice", ossDevice );
				}

				//// JACK DRIVER ////
				TiXmlNode* jackDriverNode;
				if ( !(jackDriverNode = audioEngineNode->FirstChild( "jack_driver" ) ) ) {
					warningLog("Error reading configuration file: jack_driver node not found");
					recreate = true;
				}
				else {
					// Jack port name 1
					jackPortName1 = LocalFileMng::readXmlString( this, jackDriverNode, "jack_port_name_1", jackPortName1 );

					// Jack port name 2
					jackPortName2 = LocalFileMng::readXmlString( this, jackDriverNode, "jack_port_name_2", jackPortName2 );

					// Jack transport mode
					string sMode = LocalFileMng::readXmlString( this, jackDriverNode, "jack_transport_mode", "NO_JACK_TRANSPORT" );
					if (sMode == "NO_JACK_TRANSPORT") {
						m_jackTransportMode = NO_JACK_TRANSPORT;
					}
					else if (sMode == "USE_JACK_TRANSPORT") {
						m_jackTransportMode = USE_JACK_TRANSPORT;
					}


					// Jack track outs?
					jackTrackOuts = LocalFileMng::readXmlBool( this, jackDriverNode, "jack_track_outs", jackTrackOuts );

					// Connect to default jack outputs?
					jackConnectDefaults = LocalFileMng::readXmlBool( this, jackDriverNode, "jack_connect_defaults", jackConnectDefaults );
				}

				//// ALSA MIDI DRIVER ////
				TiXmlNode* alsaMidiDriverNode;
				if ( !(alsaMidiDriverNode = audioEngineNode->FirstChild( "alsa_midi_driver" ) ) ) {
					warningLog( "Error reading configuration file: alsa_midi_driver node not found" );
					recreate = true;
				}
				else {
					// midi port channel
					midiPortChannel = LocalFileMng::readXmlInt( this, alsaMidiDriverNode, "midi_port_channel", midiPortChannel );

					// midi destination name
					midiDest_name = LocalFileMng::readXmlString( this, alsaMidiDriverNode, "midi_dest_name", midiDest_name );

					// midi destination client
					midiDest_client = LocalFileMng::readXmlInt( this, alsaMidiDriverNode, "midi_dest_client", midiDest_client );

					// midi destination port
					midiDest_port = LocalFileMng::readXmlInt( this, alsaMidiDriverNode, "midi_dest_port", midiDest_port );

					// ignore note off
					m_bIgnoreMidiNoteOff = LocalFileMng::readXmlBool( this, alsaMidiDriverNode, "ignoreMidiNoteOff", m_bIgnoreMidiNoteOff );
				}
			}

			/////////////// GUI //////////////
			TiXmlNode* guiNode;
			if ( !(guiNode = rootNode->FirstChild( "gui" ) ) ) {
				warningLog("Error reading configuration file: gui node not found");
				recreate = true;
			}
			else {
				// QT Style
				m_sQTStyle = LocalFileMng::readXmlString( this, guiNode, "QTStyle", m_sQTStyle );

				// Interface mode
				interfaceMode = LocalFileMng::readXmlString( this, guiNode, "interface_mode", interfaceMode );

				// Application font family
				applicationFontFamily = LocalFileMng::readXmlString( this, guiNode, "application_font_family", applicationFontFamily );

				// Application font pointSize
				applicationFontPointSize = LocalFileMng::readXmlInt( this, guiNode, "application_font_pointsize", applicationFontPointSize );

				// mixer font family
				mixerFontFamily = LocalFileMng::readXmlString( this, guiNode, "mixer_font_family", mixerFontFamily );

				// mixer font pointSize
				mixerFontPointSize = LocalFileMng::readXmlInt( this, guiNode, "mixer_font_pointsize", mixerFontPointSize );

				// Mixer falloff speed
				mixerFalloffSpeed = LocalFileMng::readXmlFloat( this, guiNode, "mixer_falloff_speed", mixerFalloffSpeed );

				// pattern editor grid resolution
				m_nPatternEditorGridResolution = LocalFileMng::readXmlInt( this, guiNode, "patternEditorGridResolution", m_nPatternEditorGridResolution );
				m_bPatternEditorUsingTriplets = LocalFileMng::readXmlBool( this, guiNode, "patternEditorUsingTriplets", m_bPatternEditorUsingTriplets );
				m_bShowInstrumentPeaks = LocalFileMng::readXmlBool( this, guiNode, "showInstrumentPeaks", m_bShowInstrumentPeaks );

				// pattern editor grid height
				m_nPatternEditorGridHeight = LocalFileMng::readXmlInt( this, guiNode, "patternEditorGridHeight", m_nPatternEditorGridHeight );

				// pattern editor grid width
				m_nPatternEditorGridWidth = LocalFileMng::readXmlInt( this, guiNode, "patternEditorGridWidth", m_nPatternEditorGridWidth );

				// mainForm window properties
				setMainFormProperties( readWindowProperties( guiNode, "mainForm_properties", mainFormProperties ) );
				setMixerProperties( readWindowProperties( guiNode, "mixer_properties", mixerProperties ) );
				setPatternEditorProperties( readWindowProperties( guiNode, "patternEditor_properties", patternEditorProperties ) );
				setSongEditorProperties( readWindowProperties( guiNode, "songEditor_properties", songEditorProperties ) );
				setDrumkitManagerProperties( readWindowProperties( guiNode, "drumkitManager_properties", drumkitManagerProperties ) );
				setAudioEngineInfoProperties( readWindowProperties( guiNode, "audioEngineInfo_properties", audioEngineInfoProperties ) );

				for (uint nFX = 0; nFX < MAX_FX; nFX++) {
					string sNodeName = "ladspaFX_properties" + toString(nFX);
					setLadspaProperties(nFX, readWindowProperties( guiNode, sNodeName, m_ladspaProperties[nFX] ) );
				}
			}

			/////////////// FILES //////////////
			TiXmlNode* filesNode;
			if ( !(filesNode = rootNode->FirstChild( "files" ) ) ) {
				warningLog("Error reading configuration file: files node not found");
				recreate = true;
			}
			else {
				// last used song
				lastSongFilename = LocalFileMng::readXmlString( this, filesNode, "lastSongFilename", lastSongFilename );
			}

		} // rootNode
		else {
			warningLog("Error reading configuration file: hydrogen_preferences node not found");
			recreate = true;
		}
	}
	else {
		warningLog("Configuration file not found.");
		recreate = true;
	}


	// The preferences file should be recreated?
	if (recreate == true) {
		infoLog("Recreating configuration file");
		savePreferences();
	}

}





//----------------------------------------------------------------------------
/**
 * Save the preferences file
 */
void PreferencesMng::savePreferences() {
	infoLog("Saving preferences file");
	string prefDir = getUserHome() + "/.hydrogen";
	string filename = prefDir + "/hydrogen.conf";

	TiXmlDocument doc(filename.c_str());

	TiXmlElement rootNode("hydrogen_preferences");

	// hydrogen version
	LocalFileMng::writeXmlString( &rootNode, "version", string(VERSION) );

	////// GENERAL ///////
	LocalFileMng::writeXmlString( &rootNode, "ladspaPath", m_sLadspaPath );

	string restoreLastSongStr = "false";
	if ( restoreLastSong ) {
		restoreLastSongStr = "true";
	}
	LocalFileMng::writeXmlString( &rootNode, "restoreLastSong", restoreLastSongStr );

	// hear new notes in the pattern editor
	string hearNewNotesStr = "false";
	if ( hearNewNotes ) {
		hearNewNotesStr = "true";
	}
	LocalFileMng::writeXmlString( &rootNode, "hearNewNotes", hearNewNotesStr );

	// key/midi event prefs
	LocalFileMng::writeXmlString( &rootNode, "recordEvents", recordEvents ? "true": "false" );
	LocalFileMng::writeXmlString( &rootNode, "quantizeEvents", quantizeEvents ? "true": "false" );


	// Recent used songs
	TiXmlElement recentUsedSongsNode( "recentUsedSongs" );
	{
		uint nSongs = 5;
		if ( m_recentFiles.size() < 5 ) {
			nSongs = m_recentFiles.size();
		}
		for ( uint i = 0; i < nSongs; i++ ) {
			LocalFileMng::writeXmlString( &recentUsedSongsNode, "song", m_recentFiles[ i ] );
		}
	}
	rootNode.InsertEndChild( recentUsedSongsNode );


	//---- AUDIO ENGINE ----
	TiXmlElement audioEngineNode( "audio_engine" );
	{
		// audio driver
		LocalFileMng::writeXmlString( &audioEngineNode, "audio_driver", audioDriver );

		// use metronome
		string useMetronomeStr = "false";
		if ( useMetronome ) {
			useMetronomeStr = "true";
		}
		LocalFileMng::writeXmlString( &audioEngineNode, "use_metronome", useMetronomeStr );


		// Metronome volume
		LocalFileMng::writeXmlString( &audioEngineNode, "metronome_volume", toString( metronomeVolume ) );

		// Max notes
		LocalFileMng::writeXmlString( &audioEngineNode, "maxNotes", toString( maxNotes ) );


		//// OSS DRIVER ////
		TiXmlElement ossDriverNode("oss_driver");
		{
			// buffer size
			LocalFileMng::writeXmlString( &ossDriverNode, "buffer_size", toString( bufferSize ) );

			// sample rate
			LocalFileMng::writeXmlString( &ossDriverNode, "samplerate", toString( sampleRate ) );

			LocalFileMng::writeXmlString( &ossDriverNode, "ossDevice", ossDevice );
		}
		audioEngineNode.InsertEndChild( ossDriverNode );

		//// JACK DRIVER ////
		TiXmlElement jackDriverNode( "jack_driver" );
		{
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_port_name_1", jackPortName1 );	// jack port name 1
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_port_name_2", jackPortName2 );	// jack port name 2

			// jack transport slave
			string sMode;
			if ( m_jackTransportMode == NO_JACK_TRANSPORT) {
				sMode = "NO_JACK_TRANSPORT";
			}
			else if ( m_jackTransportMode == USE_JACK_TRANSPORT) {
				sMode = "USE_JACK_TRANSPORT";
			}
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_transport_mode", sMode );

			// jack default connection
			string jackConnectDefaultsString = "false";
			if (jackConnectDefaults) {
				jackConnectDefaultsString = "true";
			}
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_connect_defaults", jackConnectDefaultsString );

			// jack track outs
			string jackTrackOutsString = "false";
			if (jackTrackOuts) {
				jackTrackOutsString = "true";
			}
			LocalFileMng::writeXmlString( &jackDriverNode, "jack_track_outs", jackTrackOutsString );

		}
		audioEngineNode.InsertEndChild( jackDriverNode );

		//// ALSA MIDI DRIVER ////
		TiXmlElement alsaMidiDriverNode( "alsa_midi_driver" );
		{
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_port_channel", toString( midiPortChannel ) );	// Midi port channel
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_dest_name", midiDest_name );		// Midi destination name
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_dest_client", toString( midiDest_client ) );		// Midi destination client
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "midi_dest_port", toString( midiDest_port ) );		// Midi destination port

			// Ignore midi note off
			string sIgnore = "false";
			if (m_bIgnoreMidiNoteOff) {
				sIgnore = "true";
			}
			LocalFileMng::writeXmlString( &alsaMidiDriverNode, "ignoreMidiNoteOff", sIgnore );
		}
		audioEngineNode.InsertEndChild( alsaMidiDriverNode );

	}
	rootNode.InsertEndChild( audioEngineNode );

	//---- GUI ----
	TiXmlElement guiNode("gui");
	{
		LocalFileMng::writeXmlString( &guiNode, "QTStyle", m_sQTStyle );

		// Interface mode
		if (interfaceMode == "Top level") {
			LocalFileMng::writeXmlString( &guiNode, "interface_mode", "Top level" );
		}
		else {
			LocalFileMng::writeXmlString( &guiNode, "interface_mode", "Child frame" );
		}

		// Application font family
		LocalFileMng::writeXmlString( &guiNode, "application_font_family", applicationFontFamily );

		// Application font pointsize
		LocalFileMng::writeXmlString( &guiNode, "application_font_pointsize", toString( applicationFontPointSize ) );

		// Mixer font family
		LocalFileMng::writeXmlString( &guiNode, "mixer_font_family", mixerFontFamily );

		// Mixer font pointsize
		LocalFileMng::writeXmlString( &guiNode, "mixer_font_pointsize", toString( mixerFontPointSize ) );

		// Mixer falloff speed
		LocalFileMng::writeXmlString( &guiNode, "mixer_falloff_speed", toString( mixerFalloffSpeed ) );

		// pattern editor grid resolution
		LocalFileMng::writeXmlString( &guiNode, "patternEditorGridResolution", toString( m_nPatternEditorGridResolution ) );

		// pattern editor grid height
		LocalFileMng::writeXmlString( &guiNode, "patternEditorGridHeight", toString( m_nPatternEditorGridHeight ) );

		// pattern editor grid width
		LocalFileMng::writeXmlString( &guiNode, "patternEditorGridWidth", toString( m_nPatternEditorGridWidth ) );

		// Pattern editor using triplets?
		string sTripletsString = "false";
		if (m_bPatternEditorUsingTriplets) {
			sTripletsString = "true";
		}
		LocalFileMng::writeXmlString( &guiNode, "patternEditorUsingTriplets", sTripletsString );

		// show instrument peaks?
		string sInstrPeaksString = "false";
		if (m_bShowInstrumentPeaks) {
			sInstrPeaksString = "true";
		}
		LocalFileMng::writeXmlString( &guiNode, "showInstrumentPeaks", sInstrPeaksString );

		// MainForm window properties
		writeWindowProperties( &guiNode, "mainForm_properties", mainFormProperties );
		writeWindowProperties( &guiNode, "mixer_properties", mixerProperties );
		writeWindowProperties( &guiNode, "patternEditor_properties", patternEditorProperties );
		writeWindowProperties( &guiNode, "songEditor_properties", songEditorProperties );
		writeWindowProperties( &guiNode, "drumkitManager_properties", drumkitManagerProperties );
		writeWindowProperties( &guiNode, "audioEngineInfo_properties", audioEngineInfoProperties );
		for (uint nFX = 0; nFX < MAX_FX; nFX++) {
			string sNode = "ladspaFX_properties" + toString(nFX);
			writeWindowProperties( &guiNode, sNode, m_ladspaProperties[nFX] );
		}
	}
	rootNode.InsertEndChild( guiNode );

	//---- FILES ----
	TiXmlElement filesNode( "files" );
	{
		// last used song
		LocalFileMng::writeXmlString( &filesNode, "lastSongFilename", lastSongFilename );
	}
	rootNode.InsertEndChild( filesNode );

	doc.InsertEndChild(rootNode);
	doc.SaveFile();
}




//----------------------------------------------------------------------------
/**
 * Create preferences directory
 */
void PreferencesMng::createPreferencesDirectory() {
	string prefDir = getUserHome() + "/.hydrogen";

	warningLog("Creating preference file directory in " + prefDir);

	mkdir(prefDir.c_str(),S_IRWXU);

}




//----------------------------------------------------------------------------
/**
 * Create data directory
 */
void PreferencesMng::createDataDirectory() {
	string dir = getUserHome() + "/.hydrogen/data";

	warningLog("Creating data directory in " + dir);

	mkdir(dir.c_str(),S_IRWXU);
}




void PreferencesMng::setRecentFiles( vector<string> recentFiles )
{
	// find single filenames. (skip duplicates)
	vector<string> temp;
	for (uint i = 0; i < recentFiles.size(); i++) {
		string sFilename = recentFiles[ i ];

		bool bExists = false;
		for (uint j = 0; j < temp.size(); j++) {
			if ( sFilename == temp[ j ] ) {
				bExists = true;
				break;
			}
		}
		if ( !bExists ) {
			temp.push_back( sFilename );
		}
	}

	m_recentFiles = temp;
}



/**
 * Read the xml nodes related to window properties
 */
WindowProperties PreferencesMng::readWindowProperties( TiXmlNode *parent, string windowName, WindowProperties defaultProp ) {
	WindowProperties prop = defaultProp;

	TiXmlNode* windowPropNode;
	if ( !(windowPropNode = parent->FirstChild( windowName.c_str() ) ) ) {
		warningLog( "Error reading configuration file: " + windowName + " node not found" );
	}
	else {
		// x
		prop.x = LocalFileMng::readXmlInt( this, windowPropNode, "x", prop.x );

		// y
		prop.y = LocalFileMng::readXmlInt( this, windowPropNode, "y", prop.y );

		// width
		prop.width = LocalFileMng::readXmlInt( this, windowPropNode, "width", prop.width );

		// height
		prop.height = LocalFileMng::readXmlInt( this, windowPropNode, "height", prop.height );
	}

	return prop;
}






/**
 * Write the xml nodes related to window properties
 */
void PreferencesMng::writeWindowProperties(TiXmlNode *parent, string windowName, WindowProperties prop ) {
	TiXmlElement windowPropNode( windowName.c_str() );
		LocalFileMng::writeXmlString( &windowPropNode, "x", toString( prop.x ) );
		LocalFileMng::writeXmlString( &windowPropNode, "y", toString( prop.y ) );
		LocalFileMng::writeXmlString( &windowPropNode, "width", toString( prop.width ) );
		LocalFileMng::writeXmlString( &windowPropNode, "height", toString( prop.height ) );
	parent->InsertEndChild( windowPropNode );
}





/**
 * Constructor
 */
WindowProperties::WindowProperties() : Object( "WindowProperties" )
{
//	infoLog( "INIT" );
	x = 0;
	y = 0;
	width = 0;
	height = 0;
}



/**
 * Destructor
 */
WindowProperties::~WindowProperties() {
//	infoLog( "DESTROY" );
}


