/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: Song.cpp,v 1.9 2004/02/20 14:09:20 comix Exp $
 *
 */

#include "Song.h"
#include "config.h"

//----------------------------------------------------------------------------
//	Implementation of SequenceList class
//----------------------------------------------------------------------------
/**
 * Constructor
 */
SequenceList::SequenceList() : Object( "SequenceList" )
{
//	infoLog("Init");
}




/**
 * Destructor
 */
SequenceList::~SequenceList() {
//	infoLog("destroy");
	for (unsigned int i = 0; i < list.size(); i++) {
		if( list[i] != NULL ){
			delete list[i];
			list[i] = NULL;
		}
	}
}




/**
 *
 */
void SequenceList::add(Sequence* newSequence) {
	list.push_back(newSequence);
}




/**
 *
 */
Sequence* SequenceList::get(unsigned int pos) {
	return list[pos];
}




/**
 *
 */
unsigned int SequenceList::getSize() {
	return list.size();
}



/**
 *
 */
SequenceList* SequenceList::copy() {
	SequenceList *newSequenceList = new SequenceList();

	for (uint nSeq = 0; nSeq < this->getSize(); nSeq++) {
		Sequence *oldSequence = this->get(nSeq);
		Sequence *newSequence = oldSequence->copy();
		newSequenceList->add(newSequence);
	}

	return newSequenceList;
}




//----------------------------------------------------------------------------
//	Implementation of PatternList class
//----------------------------------------------------------------------------
/**
 * Constructor
 */
PatternList::PatternList() : Object( "PatternList" )
{
//	infoLog("Init");
}




/**
 * Destructor
 */
PatternList::~PatternList() {
//	infoLog("destroy");

	// find single patterns. (skip duplicates)
	vector<Pattern*> temp;
	for (unsigned int i = 0; i < list.size(); i++) {
		Pattern *pat = list[i];

		// pat exists in temp?
		bool exists = false;
		for(unsigned int j = 0; j < temp.size(); j++) {
			if (pat == temp[j]) {
				exists = true;
				break;
			}
		}
		if (!exists) {
			temp.push_back(pat);
		}
	}

	// delete patterns
	for (unsigned int i = 0; i < temp.size(); i++) {
		Pattern *pat = temp[i];
		if (pat != NULL) {
			delete pat;
			pat = NULL;
		}
	}
}




/**
 *
 */
void PatternList::add(Pattern* newPattern) {
	list.push_back(newPattern);
}





/**
 *
 */
Pattern* PatternList::get(unsigned int pos) {
	if (pos > list.size()) {
		char tmp[200];
		sprintf( tmp, "Pattern index out of bounds. index = %d", pos );
		errorLog( tmp );
		return NULL;
	}
	return list[pos];
}




/**
 *
 */
unsigned int PatternList::getSize() {
	return list.size();
}




/**
 *
 */
void PatternList::clear() {
	list.clear();
}




/**
 * Replace an existent pattern with another one
 */
void PatternList::replace( Pattern* newPattern, unsigned int pos ) {
	list.insert( list.begin() + pos, newPattern );	// insert the new pattern
	// remove the old pattern
	list.erase( list.begin() + pos + 1 );
}




/**
 * Remove a pattern from the list (every instance in the list)
 * Note: the pattern is not deleted!!!
 */
void PatternList::del(Pattern *pattern) {
//	infoLog("Delete pattern " + pattern->getName());
	// create the new pattern list
	vector<Pattern*> newList;
	for (uint i = 0; i < list.size(); i++) {
		Pattern *pat = list[i];
		if (pat == pattern) {
			continue;
		}
		newList.push_back(pat);
	}

	// copy the new list element over the old vector reference
	list.clear();
	for (uint i = 0; i < newList.size(); i++) {
		list.push_back(newList[i]);
	}
}





/**
 * Remove one pattern from the list
 * Note: the pattern is not deleted!!!
 */
void PatternList::del(uint index) {
	// create the new pattern list
	vector<Pattern*> newList;
	for (uint i = 0; i < list.size(); i++) {
		if (i == index) {
			continue;
		}
		Pattern *pat = list[i];
		newList.push_back(pat);
	}

	// copy the new list element over the old vector reference
	list.clear();
	for (uint i = 0; i < newList.size(); i++) {
		list.push_back(newList[i]);
	}
}





//----------------------------------------------------------------------------
//	Implementation of InstrumentList class
//----------------------------------------------------------------------------
/**
 * Constructor
 */
InstrumentList::InstrumentList() : Object( "InstrumentList" )
{
//	infoLog("Init");
}



/**
 * Destructor
 */
InstrumentList::~InstrumentList() {
//	infoLog("destroy");
	for (unsigned int i = 0; i < list.size(); i++) {
		if( list[i] != NULL ){
			delete list[i];
			list[i] = NULL;
		}
	}
}




/**
 *
 */
void InstrumentList::add(Instrument* newInstrument) {
	list.push_back(newInstrument);
	posmap[newInstrument] = list.size() - 1;
}



/**
 *
 */
Instrument* InstrumentList::get(unsigned int pos) {
	return list[pos];
}

/**
 * returns index of instrument in list
 * if instrument not found, returns -1
 */
int InstrumentList::getPos(Instrument * inst) {
	if (posmap.find(inst) == posmap.end())
		return -1;
	return posmap[inst];
}


/**
 *
 */
unsigned int InstrumentList::getSize() {
	return list.size();
}




/**
 * Replace the instrument sample
 */
void InstrumentList::replaceSample(Instrument* newInstrument, int pos) {
	Instrument *oldInstr = list[pos];

	// delete old sample
	Sample *oldSample = oldInstr->getSample();
	delete oldSample;

	// insert new sample from newInstrument
	Sample *newSample = Sample::load(newInstrument->getSample()->getFilename());
	oldInstr->setSample(newSample);

	delete newInstrument;
	newInstrument = NULL;
}




//----------------------------------------------------------------------------
//	IMPLEMENTATION OF SONG CLASS
//----------------------------------------------------------------------------



//----------------------------------------------------------------------------
/**
 * Constructor
 */
Song::Song(string id, string name, string author, float bpm, float volume) : Object( "Song     " )
{
	infoLog( "init \"" + name + "\"" );

	instrumentList = NULL;
	patternList = NULL;
	patternSequence = NULL;

	this->id = id;
	this->name = name;
	this->author = author;
	this->resolution = 48;	// number of ticks per quarter
	this->humanizeTimeValue = 0.0;
	this->humanizeVelocityValue = 0.0;

	setBpm( bpm) ;
	setVolume( volume );
	setMetronomeVolume( 0.50f );	// default
	setNotes( "Song info" );
	setFilename( "" );
	setModified( false );
	setLoopEnabled( false );
	setSwingFactor( 0.0 );
	setMode( PATTERN_MODE );
	setHumanizeTimeEnabled( false );
	setHumanizeVelocityEnabled( false );
	setSwingEnabled( false );
	setPatternList( NULL );
	setPatternSequence( NULL );

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		setLadspaFX( nFX, NULL );
	}
}




//----------------------------------------------------------------------------
/**
 * Destructor
 */
Song::~Song(){
	// delete all patterns
	delete patternList;

	if (patternSequence) {
		patternSequence->clear();
		delete patternSequence;
	}

	delete instrumentList;

	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		delete m_pLadspaFX[nFX];
	}

	infoLog( "destroy \"" + name + "\"" );
}




//----------------------------------------------------------------------------
/**
 * Set bpm.
 */
void Song::setBpm(float bpm) {
	this->bpm = bpm;
}




//----------------------------------------------------------------------------
/**
 * Load a song form file
 */
Song* Song::load(string filename) {
	Song *song = NULL;
	LocalFileMng mng;
	song = mng.loadSong(filename);

	return song;
}



//----------------------------------------------------------------------------
/**
 * Save a song to file
 */
void Song::save(string filename) {
	LocalFileMng mng;
	mng.saveSong(this, filename);
}




//----------------------------------------------------------------------------
/**
 *
 */
unsigned int Song::estimateSongSize() {
	unsigned int nPatterns = patternSequence->getSize();	// number of patterns in the song
	unsigned int ticks = MAX_NOTES * nPatterns;

	return ticks;
}





/**
 * Return an empty song
 */
Song* Song::getEmptySong() {
	string dataDir = DATA_DIR;
	string filename = dataDir + "/DefaultSong.h2song";
	Song *song = Song::load( filename );

	return song;
}




float Song::getHumanizeTimeValue() {
	return humanizeTimeValue;
}



void Song::setHumanizeTimeValue(float value) {
	humanizeTimeValue = value;
}


float Song::getHumanizeVelocityValue() {
	return humanizeVelocityValue;
}



void Song::setHumanizeVelocityValue(float value) {
	humanizeVelocityValue = value;
}


void Song::setSwingFactor( float factor ) {
	if (factor < 0.0) {
		factor = 0.0;
	}
	else if (factor > 1.0) {
		factor = 1.0;
	}

	swingFactor = factor;
}



//----------------------------------------------------------------------------
//	IMPLEMENTATION OF SEQUENCE CLASS
//----------------------------------------------------------------------------



//----------------------------------------------------------------------------
/**
 * Constructor
 */
Sequence::Sequence(string name) : Object( "Sequence" )
{
//	logger.info(this, "INIT " + name);
	setName( name );

	for (int i = 0; i < MAX_NOTES; i++) {
		noteList[i] = NULL;
	}
}





//----------------------------------------------------------------------------
/**
 * Destructor
 */
Sequence::~Sequence(){
	// delete all notes
	for( unsigned int i = 0; i < MAX_NOTES; i++ ){
		if( noteList[i] != NULL ){
			delete noteList[i];
			noteList[i] = NULL;
		}
	}

//	logger.info(this, "DESTROY " + name);

}





Sequence* Sequence::copy() {
	Sequence *newSequence = new Sequence(this->getName());

	for (uint nNote = 0; nNote < MAX_NOTES; nNote++) {
		Note *oldNote = this->noteList[nNote];
		if (oldNote == NULL) {
			newSequence->noteList[nNote] = NULL;
		}
		else {
			Note *newNote = oldNote->copy();
			newSequence->noteList[nNote] = newNote;
		}
	}

	return newSequence;
}







//----------------------------------------------------------------------------
//	IMPLEMENTATION OF INSTRUMENT CLASS
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
/**
 * Constructor
 */
Instrument::Instrument( string id, string name, string author, int type, float volume ) : Object( "Instrument" )
{

//	logger.info(this, "INIT " + name);
	this->id = id;
	this->name = name;
	this->author = author;
	this->volume = volume;
	sample = NULL;
	setMuted( false );
	setPeak_L( 0.0 );
	setPeak_R( 0.0 );
	setPan_L( 1.0 );
	setPan_R( 1.0 );
	setDrumkitName( "" );
	for (uint nFX = 0; nFX < MAX_FX; nFX++) {
		setFXLevel( nFX, 0.0 );
	}
}






//----------------------------------------------------------------------------
/**
 * Destructor
 */
Instrument::~Instrument(){
	if (sample != NULL) {
		delete sample;
		sample = NULL;
	}
//	logger.info(this, "DESTROY " + name);
}






//----------------------------------------------------------------------------
/**
 * Load an instrument from file
 */
Instrument* Instrument::load(string filename) {
	Logger logger;
//	logger.log("Instrument::load() loading " + filename);

	Instrument *instr = NULL;

	string id = filename;	// FIXME: per ora l'ID e' il filename, bisogna calcolarlo in maniera relativa
	string name = "undefined instrument name";
	string author = "undefined author";
	uint type = 0;	// not used
	float volume = 0.8;

	std::ifstream verify(filename.c_str() , std::ios::in | std::ios::binary);
	if (verify == NULL){
		logger.log( "Instrument::load() Load Instrument: Data file " + filename + " not found." );
		return NULL;
	}

	instr = new Instrument(id, name, author, type, volume);

	Sample *sample = Sample::load(filename);
	if (sample == NULL) {
		logger.log( "Instrument::load() Load Instrument: Sample NULL" );
	}
	instr->setSample(sample);

	return instr;
}





/**
 *
 */
void Instrument::setVolume(float volume) {
	if (volume > 1.0) {
		volume = 1.0;
	}
	else if (volume < 0.0) {
		volume = 0.0;
	}
	this->volume = volume;
}





//----------------------------------------------------------------------------
/**
 * Save an instrument to file
 */
void Instrument::save(string filename) {
	sample->save( filename );
}




/**
 *
 */
void Instrument::setPan_L( float val ) {
	if (val < 0.0) {
		val = 0.0;
		errorLog("setPan_L() val < 0.0");
	}
	else if (val > 1.0) {
		errorLog("setPan_L() val > 1.0");
		val = 1.0;
	}
	pan_L = val;
}





/**
 *
 */
void Instrument::setPan_R( float val ) {
	if (val < 0.0) {
		val = 0.0;
		errorLog("setPan_R() val < 0.0");
	}
	else if (val > 1.0) {
		val = 1.0;
		errorLog("setPan_R() val > 1.0");
	}
	pan_R = val;
}



void Instrument::setPeak_L( float fPeak )
{
	if (fPeak < 0.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_L] Error: peak < 0 (peak = %f)", fPeak );
		errorLog( tmp );

		fPeak = 0.0f;
	}
	else if( fPeak > 1.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_L] Error: peak > 1 (peak = %f)", fPeak );
		errorLog( tmp );
		fPeak = 1.0f;
	}

	this->peak_L = fPeak;
}

void Instrument::setPeak_R( float fPeak )
{
	if (fPeak < 0.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_R] Error: peak < 0 (peak = %f)", fPeak );
		errorLog( tmp );

		fPeak = 0.0f;
	}
	else if( fPeak > 1.0f) {
		char tmp[200];
		sprintf( tmp, "[setPeak_R] Error: peak > 1 (peak = %f)", fPeak );
		errorLog( tmp );
		fPeak = 1.0f;
	}

	this->peak_R = fPeak;
}





//----------------------------------------------------------------------------
//	IMPLEMENTATION OF NOTE CLASS
//----------------------------------------------------------------------------

/**
 * Constructor
 */
Note::Note(	unsigned int position,
			float velocity,
			float pan_L,
			float pan_R,
			float nPitch
		) : Object( "Note" )
{
	//infoLog("INIT");

	this->position = position;
	this->instrument = NULL;
	setVelocity( velocity );
	setPan_L( pan_L );
	setPan_R( pan_R );
	setPitch( nPitch );


	samplePosition = 0.0;
	humanizeDelay = 0;
}






//----------------------------------------------------------------------------
/**
 * Destructor
 */
Note::~Note(){
	//infoLog("DESTROY");
}





/**
 *
 */
void Note::setInstrument(Instrument* instrument) {
	this->instrument = instrument;
}




/**
 *
 */
void Note::setVelocity( float val ) {
	if (val < 0.0) {
		char tmp[100];
		sprintf(tmp, "setVelocity(): %f < 0.0", val);
		errorLog(tmp);
		val = 0.0;
	}
	else if (val > 1.0) {
		char tmp[100];
		sprintf(tmp, "setVelocity(): %f > 1.0", val);
		errorLog(tmp);
		val = 1.0;
	}

	velocity = val;
}




/**
 *
 */
void Note::setPan_L( float val ) {
	if (val < 0.0) {
		char tmp[100];
		sprintf(tmp, "setPan_L(): %f < 0.0", val);
		errorLog(tmp);
		val = 0.0;
	}
	else if (val > 1.0) {
		char tmp[100];
		sprintf(tmp, "setPan_L(): %f > 1.0", val);
		errorLog(tmp);
		val = 1.0;
	}

	pan_L = val;
}





/**
 *
 */
void Note::setPan_R( float val ) {
	if (val < 0.0) {
		char tmp[100];
		sprintf(tmp, "setPan_R(): %f < 0.0", val);
		errorLog(tmp);
		val = 0.0;
	}
	else if (val > 1.0) {
		char tmp[100];
		sprintf(tmp, "setPan_L(): %f > 1.0", val);
		errorLog(tmp);
		val = 1.0;
	}

	pan_R = val;
}




/**
 *
 */
Instrument* Note::getInstrument() {
	return instrument;
}



Note* Note::copy() {
	Note* newNote = new Note(
		this->getPosition(),
		this->getVelocity(),
		this->getPan_L(),
		this->getPan_R(),
		this->getPitch()
	);
	newNote->setInstrument(this->getInstrument());

	return newNote;
}


//----------------------------------------------------------------------------
//	IMPLEMENTATION OF PATTERN CLASS
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
/**
 * Constructor
 */
Pattern::Pattern( string name, uint nSize ) : Object( "Pattern" )
{
//	infoLog("init");
	setName( name );
	setSize( nSize );
}




//----------------------------------------------------------------------------
/**
 * Destructor
 */
Pattern::~Pattern() {
//	infoLog("destroy");

	// delete all Sequences
	if (sequenceList != NULL) {
		delete sequenceList;
		sequenceList = NULL;
	}

}





//----------------------------------------------------------------------------
/**
 *
 */
void Pattern::setName(string name) {
	this->name = name;
}





//----------------------------------------------------------------------------
/**
 *
 */
string Pattern::getName() {
	return name;
}





//----------------------------------------------------------------------------
/**
 * Return an empty Pattern
 */
Pattern* Pattern::getEmptyPattern()
{
	SequenceList *sequenceList = new SequenceList();
	for (uint i = 0; i < MAX_TRACKS; i++) {
		Sequence *trk0 = new Sequence("sequence");
		sequenceList->add(trk0);
	}

	Pattern *pat = new Pattern("Empty pattern");
	pat->setSequenceList(sequenceList);

	return pat;
}




Pattern* Pattern::copy()
{
	Pattern *newPat = new Pattern(this->getName());
	newPat->setSize(this->getSize());
	SequenceList *newSeqList = this->getSequenceList()->copy();
	newPat->setSequenceList(newSeqList);

	return newPat;
}






//////////////////////////





/**
 *
 */
DrumkitInfo::DrumkitInfo() : Object( "DrumkitInfo" )
{
//	infoLog( "init" );
	instrumentList = NULL;
}



/**
 *
 */
DrumkitInfo::~DrumkitInfo()
{
//	infoLog( "Destroy" );

	if (instrumentList) {
		delete instrumentList;
		instrumentList = NULL;
	}

}


/**
 *
 */
void DrumkitInfo::dump()
{
	cout << "Drumkit dump" << endl;
	cout << "\tName = " << name << endl;
	cout << "\tAuthor = " << author << endl;
	cout << "\tInfo = " << info << endl;

	cout << "\tInstrument list" << endl;
	for ( uint i = 0; i < instrumentList->getSize(); i++) {
		Instrument *instr = instrumentList->get( i );
		cout << "\t\t" << i << " - " << instr->getSample()->getFilename() << endl;

	}

}




