/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: PreferencesDialog.cpp,v 1.3 2004/11/12 14:34:36 comix Exp $
 *
 */


#include "PreferencesDialog.h"
#include "HydrogenApp.h"
#include "MainForm.h"

#include "qmessagebox.h"
#include "qstylefactory.h"

#include "lib/Hydrogen.h"
#include "lib/Preferences.h"
#include "lib/drivers/AlsaMidiDriver.h"

PreferencesDialog::PreferencesDialog(QWidget* parent) : PreferencesDialog_UI(parent, 0, true), Object( "PreferencesDialog" )
{
	setCaption( trUtf8( "Preferences" ) );
	setIcon( QPixmap( QString(IMG_PATH) + QString( "/img/icon32.png") ) );

	setMinimumSize( width(), height() );
	setMaximumSize( width(), height() );

	Preferences *pPreferencesMng = Preferences::getInstance();
	pPreferencesMng->loadPreferences( false );	// reload user's preferences

	// Selected audio Driver
	string audioDriver = pPreferencesMng->getAudioDriver();
	if (audioDriver == "Oss") {
		driverComboBox->setCurrentItem(0);
	}
	else if (audioDriver == "Jack") {
		driverComboBox->setCurrentItem(1);
	}
	else if ( audioDriver == "Alsa" ) {
		driverComboBox->setCurrentItem( 2 );
	}
	else {
		errorLog( "Unknown audio driver from preferences [" + audioDriver + "]" );
	}
	updateDriverInfo();

	// TODO: eliminare il metronomo, usare i dati della canzone
	// Use metronome
	if ( pPreferencesMng->isMetronomeEnabled() ) {
		metronomeEnabledCheckBox->setChecked(true);
	}
	else {
		metronomeEnabledCheckBox->setChecked(false);
	}

	// metronome volume
	uint metronomeVol = (uint)( pPreferencesMng->getMetronomeVolume() * 100.0 );
	metronomeVolumeSpinBox->setValue(metronomeVol);

	// max voices
	maxVoicesTxt->setValue( pPreferencesMng->getMaxNotes() );

	// JACK
	trackOutsCheckBox->setChecked( pPreferencesMng->isJackTrackOuts() );
	connectDefaultsCheckBox->setChecked( pPreferencesMng->isJackConnectDefaults() );
	//~ JACK

	// ALSA
	m_pAlsaBufferSizeTxt->setValue( pPreferencesMng->getAlsaBufferSize() );
	m_pAlsaDeviceEdit->setText( pPreferencesMng->getAlsaAudioDevice() );
	switch ( pPreferencesMng->getAlsaAudioDeviceSampleRate() ) {
		case 44100:
			m_pAlsaSampleRateComboBox->setCurrentItem( 0 );
			break;
		case 48000:
			m_pAlsaSampleRateComboBox->setCurrentItem( 1 );
			break;
		default:
			errorLog( "Wrong Alsa audio device samplerate: " + toString( pPreferencesMng->getAlsaAudioDeviceSampleRate() ) );
	}
	//~ ALSA

	// OSS
	m_pOSSBufferSizeTxt->setValue( pPreferencesMng->getBufferSize() );

	switch ( pPreferencesMng->getSampleRate() ) {
		case 44100:
			m_pOSSSampleRateComboBox->setCurrentItem( 0 );
			break;
		case 48000:
			m_pOSSSampleRateComboBox->setCurrentItem( 1 );
			break;
		default:
			errorLog( "Wrong OSS audio device samplerate: " + toString( pPreferencesMng->getSampleRate() ) );
	}

	ossDeviceTxt->setText( QString( pPreferencesMng->getOssDevice().c_str() ) );
	//~ OSS


	// Appearance tab
	QString applicationFamily = (pPreferencesMng->getApplicationFontFamily()).c_str();
	int applicationPointSize = pPreferencesMng->getApplicationFontPointSize();

	QFont applicationFont( applicationFamily, applicationPointSize );
	applicationFontLbl->setFont( applicationFont );
	applicationFontLbl->setText( applicationFamily + QString("  %1").arg( applicationPointSize ) );

	QString mixerFamily = ( pPreferencesMng->getMixerFontFamily()).c_str();
	int mixerPointSize = pPreferencesMng->getMixerFontPointSize();
	QFont mixerFont( mixerFamily, mixerPointSize );
	mixerFontLbl->setFont( mixerFont );
	mixerFontLbl->setText( mixerFamily + QString("  %1").arg( mixerPointSize ) );


	float falloffSpeed = pPreferencesMng->getMixerFalloffSpeed();
	if (falloffSpeed == FALLOFF_SLOW) {
		mixerFalloffComboBox->setCurrentItem(0);
	}
	else if (falloffSpeed == FALLOFF_NORMAL) {
		mixerFalloffComboBox->setCurrentItem(1);
	}
	else if (falloffSpeed == FALLOFF_FAST) {
		mixerFalloffComboBox->setCurrentItem(2);
	}
	else {
		errorLog( "PreferencesDialog: wrong mixerFalloff value = " + toString(falloffSpeed) );
	}

	switch ( pPreferencesMng->getInterfaceMode() ) {
		case Preferences::MDI:
			interfaceModeComboBox->setCurrentItem( 0 );
			break;

		case Preferences::TOP_LEVEL:
			interfaceModeComboBox->setCurrentItem( 1 );
			break;

		case Preferences::SINGLE_PANED:
			interfaceModeComboBox->setCurrentItem( 2 );
			break;

		default:
			errorLog( "[INIT] Wrong interfaceMode: " + toString( pPreferencesMng->getInterfaceMode() ) );
	}

	// Style
	QStringList list = QStyleFactory::keys();
	uint i = 0;
	for ( QStringList::Iterator it = list.begin(); it != list.end(); it++) {
		m_styleComboBox->insertItem( *it );
		//infoLog( "QT Stile: " + *it   );
		//string sStyle = (*it).latin1();
		string sStyle = (*it).latin1();
		if (sStyle == pPreferencesMng->getQTStyle() ) {
			m_styleComboBox->setCurrentItem( i );
		}
		i++;
	}


	// midi tab
	#ifndef USE_ALSA_SEQ
		midiPortChannelComboBox->setEnabled( false );
		midiPortComboBox->setEnabled( false );
	#endif

	#ifdef USE_ALSA_SEQ
		// list alsa output ports
		vector<MidiPortInfo> midiOutList = ( ( Hydrogen::getInstance() )->getMidiDriver() )->listAlsaOutputPorts();
		midiPortComboBox->insertItem( "None" );
		for (uint i = 0; i < midiOutList.size(); i++) {
			MidiPortInfo info = midiOutList[ i ];
			char tmp[200];
			sprintf( tmp, "%s [%d:%d]",
					info.name.c_str(),
					info.client,
					info.port
			);

			midiPortComboBox->insertItem( QString( tmp ) );

			if ( ( info.client == pPreferencesMng->getMidiDest_client() ) && ( info.port == pPreferencesMng->getMidiDest_port() ) ) {
				midiPortComboBox->setCurrentItem( i + 1 );
			}
		}
	#endif

	int midiPortChannel = pPreferencesMng->getMidiPortChannel();
	if ( midiPortChannel == -1 ) {
		midiPortChannelComboBox->setCurrentItem( 0 );
	}
	else {
		midiPortChannelComboBox->setCurrentItem( midiPortChannel + 1 );
	}

	// General tab
	restoreLastUsedSongCheckbox->setChecked( pPreferencesMng->isRestoreLastSongEnabled() );

	m_bNeedDriverRestart = false;
}




PreferencesDialog::~PreferencesDialog() {
//	cout << "PreferencesDIalog Destroy" << endl;
}



/**
 * Cancel Button clicked
 */
void PreferencesDialog::cancelBtnClicked() {
	Preferences *preferencesMng = Preferences::getInstance();
	preferencesMng->loadPreferences( false );	// reload old user's preferences
	reject();
}


/**
 * Ok button clicked
 */
void PreferencesDialog::okBtnClicked() {
	m_bNeedDriverRestart = true;

	Preferences *pPreferencesMng = Preferences::getInstance();

	// Selected audio driver
	switch ( driverComboBox->currentItem() ) {
		case 0:
			pPreferencesMng->setAudioDriver( "Oss" );
			break;

		case 1:
			pPreferencesMng->setAudioDriver( "Jack" );
			break;

		case 2:
			pPreferencesMng->setAudioDriver( "Alsa" );
			break;

		default:
			string selectedDriver = (driverComboBox->currentText()).latin1();
			errorLog( "Unknown driver = " + selectedDriver );
	}

	// JACK
	pPreferencesMng->setJackTrackOuts( trackOutsCheckBox->isChecked() );
	pPreferencesMng->setJackConnectDefaults( connectDefaultsCheckBox->isChecked() );
	//~ JACK

	// ALSA
	pPreferencesMng->setAlsaBufferSize( m_pAlsaBufferSizeTxt->value() );
	if ( m_pAlsaSampleRateComboBox->currentText() == QString( "44100" ) ) {
		pPreferencesMng->setAlsaAudioDeviceSampleRate( 44100 );
	}
	else if ( m_pAlsaSampleRateComboBox->currentText() == QString( "48000" ) ) {
		pPreferencesMng->setAlsaAudioDeviceSampleRate( 48000 );
	}

	if ( m_pAlsaDeviceEdit->text().latin1() != pPreferencesMng->getAlsaAudioDevice() ) {
		pPreferencesMng->setAlsaAudioDevice( m_pAlsaDeviceEdit->text().latin1() );
		m_bNeedDriverRestart = true;
	}
	//~ ALSA

	// OSS
	pPreferencesMng->setBufferSize( m_pOSSBufferSizeTxt->value() );

	if ( m_pOSSSampleRateComboBox->currentText() == QString( "44100" ) ) {
		pPreferencesMng->setSampleRate( 44100 );
	}
	else if ( m_pOSSSampleRateComboBox->currentText() == QString( "48000" ) ) {
		pPreferencesMng->setSampleRate( 48000 );
	}

	if (ossDeviceTxt->text().latin1() != pPreferencesMng->getOssDevice() ) {
		m_bNeedDriverRestart = true;
	}
	pPreferencesMng->setOssDevice( ossDeviceTxt->text().latin1() );
	//~ OSS



	// metronome
	bool isMetronomeEnabled = metronomeEnabledCheckBox->isChecked();
	pPreferencesMng->setUseMetronome(isMetronomeEnabled);

	float metronomeVolume = (metronomeVolumeSpinBox->value()) / 100.0;
	pPreferencesMng->setMetronomeVolume(metronomeVolume);

	// maxVoices
	pPreferencesMng->setMaxNotes(maxVoicesTxt->value() );



	// Mixer falloff
	QString falloffStr = mixerFalloffComboBox->currentText().latin1();
	if ( falloffStr== trUtf8("Slow") ) {
		pPreferencesMng->setMixerFalloffSpeed(FALLOFF_SLOW);
	}
	else if ( falloffStr == trUtf8("Normal") ) {
		pPreferencesMng->setMixerFalloffSpeed(FALLOFF_NORMAL);
	}
	else if ( falloffStr == trUtf8("Fast") ) {
		pPreferencesMng->setMixerFalloffSpeed(FALLOFF_FAST);
	}
	else {
		errorLog( "[okBtnClicked()] Unknown mixerFallOffSpeed: " + string(falloffStr.latin1()) );
	}

	// interface mode
	switch( interfaceModeComboBox->currentItem() ) {
		case Preferences::TOP_LEVEL:
			infoLog( "[okBtnClicked] TOP LEVEL interface selected" );
			pPreferencesMng->setInterfaceMode( Preferences::TOP_LEVEL );
			break;

		case Preferences::MDI:
			infoLog( "[okBtnClicked] MDI interface selected" );
			pPreferencesMng->setInterfaceMode( Preferences::MDI );
			break;

		case Preferences::SINGLE_PANED:
			infoLog( "[okBtnClicked] SINGLE PANED interface selected" );
			pPreferencesMng->setInterfaceMode( Preferences::SINGLE_PANED );
			break;

		default:
			errorLog( "[okBtnClicked] Unknown interface mode: " + toString( interfaceModeComboBox->currentItem() ) + ", " + interfaceModeComboBox->currentText().latin1() );
	}



	// Midi channel
	if ( midiPortComboBox->currentItem() == 0 ) {
		pPreferencesMng->setMidiDest_name( "-" );
		pPreferencesMng->setMidiDest_client( -1 );
		pPreferencesMng->setMidiDest_port( -1 );
		m_bNeedDriverRestart = true;
	}
	else {
		string newMidiDest = midiPortComboBox->currentText().latin1();

		char midiDest_client_old[200];
		sprintf( midiDest_client_old, "%s [%d:%d]",
				pPreferencesMng->getMidiDest_name().c_str(),
				pPreferencesMng->getMidiDest_client(),
				pPreferencesMng->getMidiDest_port()
		);

		#ifdef USE_ALSA_SEQ
			if ( string(midiDest_client_old) != newMidiDest ) {
				// find the selected midi port in the list
				vector<MidiPortInfo> midiOutList = ( ( Hydrogen::getInstance() )->getMidiDriver() )->listAlsaOutputPorts();
				for (uint i = 0; i < midiOutList.size(); i++) {
					MidiPortInfo info = midiOutList[ i ];
					char tmp[200];
					sprintf( tmp, "%s [%d:%d]", info.name.c_str(), info.client, info.port );

					if ( string(tmp) == newMidiDest ) {
						pPreferencesMng->setMidiDest_name( info.name );
						pPreferencesMng->setMidiDest_client( info.client );
						pPreferencesMng->setMidiDest_port( info.port );
					}
				}
				m_bNeedDriverRestart = true;
			}
		#endif

	}
	int midiPortChannel = midiPortChannelComboBox->currentItem() - 1;
	if ( pPreferencesMng->getMidiPortChannel() != midiPortChannel ) {
		m_bNeedDriverRestart = true;
	}
	pPreferencesMng->setMidiPortChannel( midiPortChannel );


	// General tab
	pPreferencesMng->setRestoreLastSongEnabled( restoreLastUsedSongCheckbox->isChecked() );

	pPreferencesMng->savePreferences();

	if (m_bNeedDriverRestart) {
		(Hydrogen::getInstance())->restartDrivers();
	}
	accept();
}





/**
 *
 */
void PreferencesDialog::driverChanged() {
	string selectedDriver = (driverComboBox->currentText()).latin1();
	updateDriverInfo();
	m_bNeedDriverRestart = true;
}




void PreferencesDialog::updateDriverInfo() {
	QString info = "";

	bool bJack_support = false;
	#ifdef JACK_SUPPORT
		bJack_support = true;
	#endif

	bool bAlsa_support = false;
	#ifdef ALSA_AUDIO_SUPPORT
		bAlsa_support = true;
	#endif

	bool bOss_support = false;
	#ifdef OSS_SUPPORT
		bOss_support = true;
	#endif


	int nDriver = driverComboBox->currentItem();
	if ( nDriver == 0 ) {	// OSS
		info += trUtf8("<b>Open Sound System</b><br>Simple audio driver [/dev/dsp]");
		if ( !bOss_support ) {
			info += trUtf8("<br><b><font color=\"red\">Not compiled</font></b>");
		}
		m_pJackDriverFrame->hide();
		m_pAlsaDriverFrame->hide();
		m_pOssDriverFrame->show();

	}
	else if ( nDriver == 1 ) {	// JACK
		info += trUtf8("<b>Jack Audio Connection Kit Driver</b><br>Low latency audio driver");
		if ( !bJack_support ) {
			info += trUtf8("<br><b><font color=\"red\">Not compiled</font></b>");
		}
		m_pJackDriverFrame->show();
		m_pAlsaDriverFrame->hide();
		m_pOssDriverFrame->hide();
	}
	else if ( nDriver == 2 ) {	// ALSA
		info += trUtf8("<b>ALSA Driver</b><br>");
		if ( !bAlsa_support ) {
			info += trUtf8("<br><b><font color=\"red\">Not compiled</font></b>");
		}

		m_pJackDriverFrame->hide();
		m_pAlsaDriverFrame->show();
		m_pOssDriverFrame->hide();
	}
	else {
		string selectedDriver = (driverComboBox->currentText()).latin1();
		errorLog( "Unknown driver = " + selectedDriver );
	}

	driverInfoLbl->setText(info);
}




void PreferencesDialog::useMetronomeCheckboxChanged() {
	metronomeVolumeSpinBox->setEnabled(metronomeEnabledCheckBox->isChecked());
}



void PreferencesDialog::selectApplicationFont() {
	Preferences *preferencesMng = Preferences::getInstance();

	QString family = (preferencesMng->getApplicationFontFamily()).c_str();
	int pointSize = preferencesMng->getApplicationFontPointSize();


	bool ok;
	QFont font = QFontDialog::getFont( &ok, QFont( family, pointSize ), this );
	if ( ok ) {
		// font is set to the font the user selected
		family = font.family();
		pointSize = font.pointSize();
		string familyStr = family.latin1();
		preferencesMng->setApplicationFontFamily(familyStr);
		preferencesMng->setApplicationFontPointSize(pointSize);


	} else {
		// the user cancelled the dialog; font is set to the initial
		// value, in this case Times, 12.
	}

	QFont newFont(family, pointSize);
	applicationFontLbl->setFont(newFont);
	applicationFontLbl->setText(family + QString("  %1").arg(pointSize));
}




void PreferencesDialog::bufferSizeChanged() {
	m_bNeedDriverRestart = true;
}




void PreferencesDialog::sampleRateChanged() {
	m_bNeedDriverRestart = true;
}



void PreferencesDialog::restartDriverBtnClicked() {
	// update audio driver preferences

	Preferences *preferencesMng = Preferences::getInstance();

	// Audio buffer size
	preferencesMng->setBufferSize( m_pOSSBufferSizeTxt->value() );

	// Selected audio driver
	string selectedDriver = (driverComboBox->currentText()).latin1();
	preferencesMng->setAudioDriver(selectedDriver);

	// sample rate
	uint sampleRate = 44100;
	string sampleRateTxt = (m_pOSSSampleRateComboBox->currentText()).latin1();
	if (sampleRateTxt == "44100") {
		sampleRate = 44100;
	}
	else if (sampleRateTxt == "48000") {
		sampleRate = 48000;
	}
	preferencesMng->setSampleRate(sampleRate);

	(Hydrogen::getInstance())->restartDrivers();
	m_bNeedDriverRestart = false;
}



void PreferencesDialog::selectMixerFont() {
	Preferences *preferencesMng = Preferences::getInstance();

	QString family = (preferencesMng->getMixerFontFamily()).c_str();
	int pointSize = preferencesMng->getMixerFontPointSize();

	bool ok;
	QFont font = QFontDialog::getFont( &ok, QFont( family, pointSize ), this );
	if ( ok ) {
		// font is set to the font the user selected
		family = font.family();
		pointSize = font.pointSize();
		string familyStr = family.latin1();
		preferencesMng->setMixerFontFamily(familyStr);
		preferencesMng->setMixerFontPointSize(pointSize);
	}
	QFont newFont(family, pointSize);
	mixerFontLbl->setFont(newFont);
	mixerFontLbl->setText(family + QString("  %1").arg(pointSize));
}





void PreferencesDialog::midiPortChannelChanged()
{
	m_bNeedDriverRestart = true;
}


void PreferencesDialog::guiStyleChanged()
{
//	infoLog( "[guiStyleChanged]" );
	QApplication *pQApp = (HydrogenApp::getInstance())->getMainForm()->m_pQApp;
	QString sStyle = m_styleComboBox->currentText();
	pQApp->setStyle( sStyle );

	Preferences *pPref = Preferences::getInstance();
	pPref->setQTStyle( sStyle.latin1() );
}


