/*************************************************************************************************
 * Implementation of the core API
 *                                                      Copyright (C) 2004-2005 Mikio Hirabayashi
 * This file is part of Hyper Estraier.
 * Hyper Estraier is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License or any later version.  Hyper Estraier is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with Hyper
 * Estraier; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA.
 *************************************************************************************************/


#include "estraier.h"
#include "myconf.h"

#define ESTNUMBUFSIZ   32                /* size of a buffer for a number */
#define ESTPATHBUFSIZ  4096              /* size of a buffer for a path */
#define ESTIOBUFSIZ    8192              /* size of a buffer for I/O */
#define ESTALLOCUNIT   1024              /* unit number of memory allocation */
#define ESTMINIBNUM    31                /* bucket number of map for attributes */
#define ESTSCANWNUM    256               /* number of words for scaning check */
#define ESTSIGNUM      64                /* number of signals */
#define ESTREGSUBMAX   32                /* maximum number of substrings for regex */

#define ESTMETADBNAME  "_meta"           /* name of the meta database */
#define ESTKEYIDXNUM   "_idxnum"         /* key for the number of inverted indexes */
#define ESTKEYDSEQ     "_dseq"           /* key for the sequence for document IDs */
#define ESTKEYDNUM     "_dnum"           /* key for the number of documents */
#define ESTKEYAMODE    "_amode"          /* key for the mode of text analyzer */
#define ESTKEYMETA     "_meta"           /* key for meta data */

#define ESTIDXDBNAME   "_idx"            /* name of the inverted index */
#define ESTIDXDBLRM    55                /* records in a leaf node of the inverted index */
#define ESTIDXDBNIM    160               /* records in a non-leaf node of the inverted index */
#define ESTIDXDBLCN    16                /* number of leaf cache of the inverted index */
#define ESTIDXDBNCN    16                /* number of non-leaf cache of the inverted index */
#define ESTIDXDBRLCN   128               /* number of leaf cache of the index reader */
#define ESTIDXDBRNCN   64                /* number of non-leaf cache of the index reader */
#define ESTIDXDBMIN    (1048576*768)     /* minimum size of a database file */
#define ESTIDXDBMAX    (1048576*1536)    /* maximum size of a database file */

#define ESTFWMDBNAME   "_fwm"            /* name of the database for forward matching */
#define ESTFWMDBLRM    111               /* records in a leaf node of forward matching DB */
#define ESTFWMDBNIM    110               /* records in a non-leaf node of forward matching DB */
#define ESTFWMDBLCN    32                /* number of leaf cache of forward matching DB */
#define ESTFWMDBNCN    16                /* number of non-leaf cache of forward matching DB */

#define ESTATTRDBNAME  "_attr"           /* name of the database for attrutes */
#define ESTATTRDBBNUM  245759            /* bucket number of the database for attrutes */
#define ESTATTRDBDNUM  3                 /* division number of the database for attrutes */
#define ESTATTRDBALN   -5                /* alignment of the database for attrutes */

#define ESTTEXTDBNAME  "_text"           /* name of the database of texts */
#define ESTTEXTDBBNUM  61417             /* bucket number of the database for texts */
#define ESTTEXTDBDNUM  7                 /* division number of the database for texts */
#define ESTTEXTDBALN   -5                /* alignment of the database for texts */

#define ESTKWDDBNAME   "_kwd"            /* name of the database of keywords */
#define ESTKWDDBBNUM   163819            /* bucket number of the database for keywords */
#define ESTKWDDBDNUM   3                 /* division number of the database for keywords */
#define ESTKWDDBALN    -5                /* alignment of the database for keywords */

#define ESTLISTDBNAME  "_list"           /* name of the database of document list */
#define ESTLISTDBLRM   99                /* records in a leaf node of document list DB */
#define ESTLISTDBNIM   200               /* records in a non-leaf node of document list DB */
#define ESTLISTDBLCN   32                /* number of leaf cache of document list DB */
#define ESTLISTDBNCN   16                /* number of non-leaf cache of document list DB */

#define ESTIDXCCBNUM   524288            /* bucket number of cache for the inverted index */
#define ESTIDXCCMAX    (1048576*64)      /* max size of the cache */
#define ESTOUTCCBNUM   131072            /* bucket number of cache for deleted documents */
#define ESTKEYCCMNUM   65536             /* bucket number of cache for keys for TF-IDF */
#define ESTATTRCCMNUM  8192              /* number of cache for attributes */
#define ESTTEXTCCMNUM  1024              /* number of cache for texts */
#define ESTRESCCMNUM   256               /* number of cache for results */
#define ESTCCIRSLOT    256               /* slot timing for interruption */
#define ESTCCCBFREQ    10000             /* frequency of callback for flushing words */

#define ESTDIRMODE     00755             /* permission of a creating directory */
#define ESTICCHECKSIZ  32768             /* size of checking character code */
#define ESTICMISSMAX   256               /* allowance number of missing characters */
#define ESTICALLWRAT   0.001             /* allowance ratio of missing characters */
#define ESTOCPOINT     10                /* point per occurrence */
#define ESTJHASHNUM    251               /* hash number for a junction */
#define ESTWORDMAXLEN  48                /* maximum length of a word */
#define ESTWORDAVGLEN  8                 /* average length of a word */
#define ESTATTRALW     1.5               /* allowance ratio of attribute narrowing */
#define ESTKEYSCALW    3                 /* allowance ratio of TF-IDF for keywords */
#define ESTMEMIRATIO   1.1               /* incremental ratio of memory allocation */

#define ESTECLKNUM     32                /* number of keywords to esclipse candidates */
#define ESTSMLRKNUM    16                /* number of keywords to get candidates */
#define ESTSMLRUNUM    1024              /* number of adopted documents for a keyword */
#define ESTSMLRNMIN    0.5               /* the minimum value for narrowing */

enum {                                   /* enumeration for character categories */
  ESTSPACECHR,                           /* space characters */
  ESTDELIMCHR,                           /* delimiter characters */
  ESTWESTALPH,                           /* west alphabets */
  ESTEASTALPH                            /* east alphabets */
};

enum {                                   /* enumeration for text analizer modes */
  ESTAMNORMAL,                           /* normal */
  ESTAMPERFNG                            /* perfect N-gram */
};

typedef struct {                         /* type of structure for a hitting object */
  int id;                                /* ID of a document */
  int score;                             /* score tuned by TF-IDF */
  char *value;                           /* value of an attribute for sorting */
} ESTSCORE;

typedef struct {                         /* type of structure for a conditional attribute */
  char *name;                            /* name */
  int nsiz;                              /* size of the name */
  char *oper;                            /* operator */
  char *val;                             /* value */
  int vsiz;                              /* size of the value */
  const char *cop;                       /* canonical operator */
  int sign;                              /* positive or negative */
  char *sval;                            /* value of small cases */
  int ssiz;                              /* size of the small value */
  void *regex;                           /* compiled regular expressions */
  time_t num;                            /* numeric value */
} ESTCATTR;

typedef struct {                         /* type of structure for a hitting object */
  const char *word;                      /* face of keyword */
  int wsiz;                              /* size of the keyword */
  int pt;                                /* score tuned by TF-IDF */
} ESTKEYSC;


/* private function prototypes */
static int est_enc_miss(const char *ptr, int size, const char *icode, const char *ocode);
static void est_normalize_text(unsigned char *utext, int size, int *sp);
static void est_canonicalize_text(unsigned char *utext, int size, int funcspc);
static int est_char_category(int c);
static int est_char_category_perfng(int c);
static char *est_phrase_from_thumb(const char *sphrase);
static void est_snippet_add_text(const unsigned char *rtext, const unsigned char *ctext,
                                 int size, int awsiz, CBDATUM *res, const CBLIST *rwords);
static int est_str_fwmatch_wide(const unsigned char *haystack, int hsiz,
                                const unsigned char *needle, int nsiz);
static ESTIDX *est_idx_open(const char *name, int omode, int dnum);
static int est_idx_close(ESTIDX *idx);
static void est_idx_set_tuning(ESTIDX *idx, int lrecmax, int nidxmax, int lcnum, int ncnum);
static void est_idx_increment(ESTIDX *idx);
static int est_idx_dnum(ESTIDX *idx);
static int est_idx_add(ESTIDX *idx, const char *word, int wsiz, const char *vbuf, int vsiz);
static int est_idx_put_one(ESTIDX *idx, int inum, const char *word, int wsiz,
                           const char *vbuf, int vsiz);
static int est_idx_out(ESTIDX *idx, const char *word, int wsiz);
static char *est_idx_get(ESTIDX *idx, const char *word, int wsiz, int *sp);
static char *est_idx_get_one(ESTIDX *idx, int inum, const char *word, int wsiz, int *sp);
static int est_idx_vsiz(ESTIDX *idx, const char *word, int wsiz);
static int est_idx_num(ESTIDX *idx);
static double est_idx_size(ESTIDX *idx);
static int est_idx_size_current(ESTIDX *idx);
static int est_idx_sync(ESTIDX *idx);
static int est_idx_optimize(ESTIDX *idx);
static void est_idx_set_current(ESTIDX *idx);
static int est_db_write_meta(ESTDB *db);
static void est_db_inform(ESTDB *db, const char *info);
static void est_db_prepare_meta(ESTDB *db);
static CBLIST *est_phrase_terms(const char *phrase);
static int est_score_compare_by_id(const void *ap, const void *bp);
static int est_score_compare_by_score(const void *ap, const void *bp);
static int est_score_compare_by_str_asc(const void *ap, const void *bp);
static int est_score_compare_by_str_desc(const void *ap, const void *bp);
static int est_score_compare_by_num_asc(const void *ap, const void *bp);
static int est_score_compare_by_num_desc(const void *ap, const void *bp);
static ESTSCORE *est_search_uvset(ESTDB *db, int *nump, CBMAP *hints, int add);
static void est_expand_word_bw(ESTDB *db, const char *word, CBLIST *list);
static void est_expand_word_ew(ESTDB *db, const char *word, CBLIST *list);
static void est_expand_word_rx(ESTDB *db, const char *word, CBLIST *list);
static ESTSCORE *est_search_union(ESTDB *db, const char *term, int gstep,
                                  int *nump, CBMAP *hints, int add);
static const ESTSCORE *est_rescc_get(ESTDB *db, const char *word, int size, int *nump);
static void est_rescc_put(ESTDB *db, const char *word, int size, ESTSCORE *scores, int num);
static int est_narrow_scores(ESTDB *db, const CBLIST *attrs, const char *order,
                             ESTSCORE *scores, int snum, int limit, int *restp);
static int est_eclipse_scores(ESTDB *db, ESTSCORE *scores, int snum, int num,
                              int vnum, int tfidf, double limit, CBMAP *shadows);
static int est_match_attr(const char *tval, int tsiz, const char *cop, int sign,
                          const char *oval, int osiz, const char *sval, int ssiz,
                          const void *regex, int onum);
static int est_check_strand(const char *tval, const char *oval);
static int est_check_stror(const char *tval, const char *oval);
static int est_check_numbt(const char *tval, const char *oval);
static int est_keysc_compare(const void *ap, const void *bp);
static ESTSCORE *est_search_similar(ESTDB *db, CBMAP *svmap, int *nump,
                                    int knum, int unum, int tfidf, double nmin);
static CBMAP *est_phrase_vector(const char *phrase);
static CBMAP *est_get_tvmap(ESTDB *db, int id, int vnum, int tfidf);
static void est_random_fclose(void);
static int est_signal_dispatch(int signum);



/*************************************************************************************************
 * common settings
 *************************************************************************************************/


/* version of Hyper Estraier */
const char *est_version = _EST_VERSION;



/*************************************************************************************************
 * API for document
 *************************************************************************************************/


/* Create a document object. */
ESTDOC *est_doc_new(void){
  ESTDOC *doc;
  CB_MALLOC(doc, sizeof(ESTDOC));
  doc->id = -1;
  doc->attrs = NULL;
  doc->dtexts = NULL;
  return doc;
}


/* Create a document object made from draft data. */
ESTDOC *est_doc_new_from_draft(const char *draft){
  ESTDOC *doc;
  CBLIST *lines;
  const char *line;
  char *pv;
  int i;
  assert(draft);
  doc = est_doc_new();
  lines = cbsplit(draft, -1, "\n");
  for(i = 0; i < CB_LISTNUM(lines); i++){
    line = CB_LISTVAL(lines, i, NULL);
    while(*line > '\0' && *line <= ' '){
      line++;
    }
    if(*line == '\0'){
      i++;
      break;
    }
    if(*line == '%') continue;
    if((pv = strchr(line, '=')) != NULL){
      *(pv++) = '\0';
      est_doc_add_attr(doc, line, pv);
    }
  }
  for(; i < CB_LISTNUM(lines); i++){
    line = CB_LISTVAL(lines, i, NULL);
    if(*line == '\t'){
      est_doc_add_hidden_text(doc, line + 1);
    } else {
      est_doc_add_text(doc, line);
    }
  }
  cblistclose(lines);
  return doc;
}


/* Destroy a document object. */
void est_doc_delete(ESTDOC *doc){
  assert(doc);
  if(doc->dtexts) cblistclose(doc->dtexts);
  if(doc->attrs) cbmapclose(doc->attrs);
  free(doc);
}


/* Add an attribute to a document object. */
void est_doc_add_attr(ESTDOC *doc, const char *name, const char *value){
  char *rbuf, *wp;
  assert(doc && name);
  if(name[0] == '\0' || name[0] == '%') return;
  if(!doc->attrs) doc->attrs = cbmapopenex(ESTMINIBNUM);
  if(value){
    rbuf = cbmemdup(value, -1);
    for(wp = rbuf; *wp != '\0'; wp++){
      if(*wp > 0 && *wp < ' ') *wp = ' ';
    }
    cbstrsqzspc(rbuf);
    cbmapputvbuf(doc->attrs, name, strlen(name), rbuf, strlen(rbuf));
  } else {
    cbmapout(doc->attrs, name, -1);
  }
}


/* Add a sentence of text to a document object. */
void est_doc_add_text(ESTDOC *doc, const char *text){
  unsigned char *utext;
  char *rtext, *wp;
  int size;
  assert(doc && text);
  while(*text > '\0' && *text <= ' '){
    text++;
  }
  if(text[0] == '\0') return;
  if(!doc->dtexts) doc->dtexts = cblistopen();
  utext = (unsigned char *)est_uconv_in(text, strlen(text), &size);
  est_normalize_text(utext, size, &size);
  rtext = est_uconv_out((char *)utext, size, NULL);
  for(wp = rtext; *wp != '\0'; wp++){
    if(*wp > 0 && *wp < ' ') *wp = ' ';
  }
  cbstrsqzspc(rtext);
  if(rtext[0] != '\0'){
    cblistpushbuf(doc->dtexts, rtext, strlen(rtext));
  } else {
    free(rtext);
  }
  free(utext);
}


/* Add a hidden sentence to a document object. */
void est_doc_add_hidden_text(ESTDOC *doc, const char *text){
  unsigned char *utext;
  char *rtext, *wp;
  int size;
  assert(doc && text);
  while(*text > '\0' && *text <= ' '){
    text++;
  }
  if(text[0] == '\0') return;
  utext = (unsigned char *)est_uconv_in(text, strlen(text), &size);
  est_normalize_text(utext, size, &size);
  rtext = est_uconv_out((char *)utext, size, NULL);
  for(wp = rtext; *wp != '\0'; wp++){
    if(*wp > 0 && *wp < ' ') *wp = ' ';
  }
  cbstrsqzspc(rtext);
  if(rtext[0] != '\0'){
    if(!doc->attrs) doc->attrs = cbmapopenex(ESTMINIBNUM);
    if(cbmapget(doc->attrs, "", 0, NULL)) cbmapputcat(doc->attrs, "", 0, " ", 1);
    cbmapputcat(doc->attrs, "", 0, rtext, -1);
  }
  free(rtext);
  free(utext);
}


/* Get the ID number of a document object. */
int est_doc_id(ESTDOC *doc){
  assert(doc);
  return doc->id;
}


/* Get a list of attribute names of a document object. */
CBLIST *est_doc_attr_names(ESTDOC *doc){
  CBLIST *names;
  const char *kbuf;
  int ksiz;
  assert(doc);
  if(!doc->attrs) return cblistopen();
  names = cblistopen();
  cbmapiterinit(doc->attrs);
  while((kbuf = cbmapiternext(doc->attrs, &ksiz)) != NULL){
    if(ksiz > 0) cblistpush(names, kbuf, ksiz);
  }
  cblistsort(names);
  return names;
}


/* Get the value of an attribute of a document object. */
const char *est_doc_attr(ESTDOC *doc, const char *name){
  assert(doc && name);
  if(!doc->attrs || name[0] == '\0') return NULL;
  return cbmapget(doc->attrs, name, -1, NULL);
}


/* Get a list of sentences of the text of a document object. */
const CBLIST *est_doc_texts(ESTDOC *doc){
  assert(doc);
  if(!doc->dtexts) doc->dtexts = cblistopen();
  return doc->dtexts;
}


/* Concatenate sentences of the text of a document object. */
char *est_doc_cat_texts(ESTDOC *doc){
  CBDATUM *datum;
  const char *elem;
  int i, size;
  if(!doc->dtexts) return cbmemdup("", 0);
  datum = cbdatumopen("", 0);
  for(i = 0; i < CB_LISTNUM(doc->dtexts); i++){
    elem = CB_LISTVAL2(doc->dtexts, i, &size);
    if(i > 0) cbdatumcat(datum, " ", 1);
    cbdatumcat(datum, elem, size);
  }
  return cbdatumtomalloc(datum, NULL);
}


/* Dump draft data of a document object. */
char *est_doc_dump_draft(ESTDOC *doc){
  CBLIST *list;
  CBDATUM *datum;
  const char *kbuf, *vbuf;
  int i, ksiz, vsiz;
  assert(doc);
  datum = cbdatumopen("", 0);
  if(doc->attrs){
    list = est_doc_attr_names(doc);
    for(i = 0; i < CB_LISTNUM(list); i++){
      kbuf = CB_LISTVAL2(list, i, &ksiz);
      vbuf = cbmapget(doc->attrs, kbuf, ksiz, &vsiz);
      cbdatumcat(datum, kbuf, ksiz);
      cbdatumcat(datum, "=", 1);
      cbdatumcat(datum, vbuf, vsiz);
      cbdatumcat(datum, "\n", 1);
    }
    cblistclose(list);
  }
  cbdatumcat(datum, "\n", 1);
  if(doc->dtexts){
    for(i = 0; i < CB_LISTNUM(doc->dtexts); i++){
      kbuf = CB_LISTVAL2(doc->dtexts, i, &ksiz);
      cbdatumcat(datum, kbuf, ksiz);
      cbdatumcat(datum, "\n", 1);
    }
  }
  if(doc->attrs && (vbuf = cbmapget(doc->attrs, "", 0, &vsiz)) != NULL){
    cbdatumcat(datum, "\t", 1);
    cbdatumcat(datum, vbuf, vsiz);
    cbdatumcat(datum, "\n", 1);
  }
  return cbdatumtomalloc(datum, NULL);
}


/* Make a snippet of the body text of a document object. */
char *est_doc_make_snippet(ESTDOC *doc, const CBLIST *words, int wwidth, int hwidth, int awidth){
  CBDATUM *res, *sbuf;
  CBMAP *counts;
  CBLIST *rwords;
  const char *text, *word, *cval;
  const unsigned char *rword;
  unsigned char *rtext, *ctext;
  int i, j, k, bi, size, wsiz, rwsiz, mywidth, awsiz, csiz;
  assert(doc && words && wwidth >= 0 && hwidth >= 0 && awidth >= 0);
  if(!doc->dtexts) doc->dtexts = cblistopen();
  res = cbdatumopen("", 0);
  rwords = cblistopen();
  for(i = 0; i < CB_LISTNUM(words); i++){
    word = CB_LISTVAL2(words, i, &wsiz);
    if(wsiz < 1 || !strcmp(word, ESTOPUVSET)) continue;
    rtext = (unsigned char *)est_uconv_in(word, wsiz, &size);
    est_canonicalize_text(rtext, size, TRUE);
    cblistpushbuf(rwords, (char *)rtext, size);
  }
  sbuf = cbdatumopen("", 0);
  for(i = 0; i < CB_LISTNUM(doc->dtexts); i++){
    text = CB_LISTVAL2(doc->dtexts, i, &size);
    if(i > 0) cbdatumcat(sbuf, " ", 1);
    cbdatumcat(sbuf, text, size);
  }
  rtext = (unsigned char *)est_uconv_in(CB_DATUMPTR(sbuf), CB_DATUMSIZE(sbuf), &size);
  ctext = (unsigned char *)cbmemdup((char *)rtext, size);
  est_canonicalize_text(ctext, size, FALSE);
  mywidth = hwidth;
  if(CB_LISTNUM(rwords) < 1) mywidth *= 3;
  if(mywidth > wwidth) mywidth = wwidth;
  for(i = 0; i < size && mywidth > 0; i += 2){
    mywidth -= est_char_category(rtext[i] * 0x100 + rtext[i+1]) == ESTEASTALPH ? 2 : 1;
  }
  awsiz = size - i;
  if(awsiz > ESTWORDMAXLEN) awsiz = ESTWORDMAXLEN;
  est_snippet_add_text(rtext, ctext, i, awsiz, res, rwords);
  wwidth -= hwidth;
  bi = i + 2;
  cbdatumcat(res, "\n", 1);
  if(awidth > 0){
    counts = cbmapopenex(ESTMINIBNUM);
    for(i = bi; i < size && wwidth >= 0; i += 2){
      for(j = 0; j < CB_LISTNUM(rwords); j++){
        rword = (unsigned char *)CB_LISTVAL2(rwords, j, &rwsiz);
        if(est_str_fwmatch_wide(ctext + i, size - i, rword, rwsiz) > 0 &&
           (!(cval = cbmapget(counts, (char *)rword, rwsiz, &csiz)) ||
            csiz < (wwidth > awidth * 1.2 ? 2 : 1))){
          cbmapputcat(counts, (char *)rword, rwsiz, "*", 1);
          if(cbmaprnum(counts) >= CB_LISTNUM(rwords)){
            cbmapclose(counts);
            counts = cbmapopenex(ESTMINIBNUM);
          }
          mywidth = awidth / 2 + 1;
          for(k = i - 2; k >= bi && mywidth >= 0; k -= 2){
            mywidth -= est_char_category(rtext[k] * 0x100 + rtext[k+1]) == ESTEASTALPH ? 2 : 1;
          }
          bi = k;
          mywidth = awidth / 2 + 1;
          for(k = i + rwsiz + 2; k < size && mywidth >= 0; k += 2){
            mywidth -= est_char_category(rtext[k] * 0x100 + rtext[k+1]) == ESTEASTALPH ? 2 : 1;
          }
          if(k > size) k = size;
          est_snippet_add_text(rtext + bi, ctext + bi, k - bi, 0, res, rwords);
          wwidth -= awidth + rwsiz / 2;
          bi = k + 2;
          i = bi - 2;
          cbdatumcat(res, "\n", 1);
          break;
        }
      }
    }
    cbmapclose(counts);
  }
  free(ctext);
  free(rtext);
  cbdatumclose(sbuf);
  cblistclose(rwords);
  return cbdatumtomalloc(res, NULL);
}



/*************************************************************************************************
 * API for search conditions
 *************************************************************************************************/


/* Create a condition object. */
ESTCOND *est_cond_new(void){
  ESTCOND *cond;
  CB_MALLOC(cond, sizeof(ESTCOND));
  cond->phrase = NULL;
  cond->gstep = 2;
  cond->tfidf = TRUE;
  cond->simple = FALSE;
  cond->attrs = NULL;
  cond->order = NULL;
  cond->max = -1;
  cond->scfb = FALSE;
  cond->scores = NULL;
  cond->snum = 0;
  cond->opts = 0;
  cond->ecllim = -1.0;
  cond->shadows = NULL;
  return cond;
}


/* Destroy a condition object. */
void est_cond_delete(ESTCOND *cond){
  assert(cond);
  if(cond->shadows) cbmapclose(cond->shadows);
  if(cond->scores) free(cond->scores);
  if(cond->order) free(cond->order);
  if(cond->attrs) cblistclose(cond->attrs);
  if(cond->phrase) free(cond->phrase);
  free(cond);
}


/* Set a search phrase to a condition object. */
void est_cond_set_phrase(ESTCOND *cond, const char *phrase){
  assert(cond && phrase);
  if(cond->phrase) free(cond->phrase);
  while(*phrase > '\0' && *phrase <= ' '){
    phrase++;
  }
  cond->phrase = cbmemdup(phrase, -1);
}


/* Add a condition of an attribute fo a condition object. */
void est_cond_add_attr(ESTCOND *cond, const char *expr){
  assert(cond && expr);
  while(*expr > '\0' && *expr <= ' '){
    expr++;
  }
  if(*expr == '\0') return;
  if(!cond->attrs) cond->attrs = cblistopen();
  cblistpush(cond->attrs, expr, -1);
}


/* Set the order of a condition object. */
void est_cond_set_order(ESTCOND *cond, const char *expr){
  assert(cond && expr);
  while(*expr > '\0' && *expr <= ' '){
    expr++;
  }
  if(*expr == '\0') return;
  if(cond->order) free(cond->order);
  cond->order = cbmemdup(expr, -1);
}


/* Set the maximum number of retrieval of a condition object. */
void est_cond_set_max(ESTCOND *cond, int max){
  assert(cond && max >= 0);
  cond->max = max;
}


/* Set options of retrieval of a condition object. */
void est_cond_set_options(ESTCOND *cond, int options){
  assert(cond);
  if(options & ESTCONDSURE) cond->gstep = 1;
  if(options & ESTCONDUSUAL) cond->gstep = 2;
  if(options & ESTCONDFAST) cond->gstep = 3;
  if(options & ESTCONDAGITO) cond->gstep = 4;
  if(options & ESTCONDNOIDF) cond->tfidf = FALSE;
  if(options & ESTCONDSIMPLE) cond->simple = TRUE;
  if(options & ESTCONDSCFB) cond->scfb = TRUE;
  cond->opts |= options;
}


/* Set the upper limit of similarity for document eclipse. */
void est_cond_set_eclipse(ESTCOND *cond, double limit){
  assert(cond);
  if(limit > 0.0 && limit < 1.0) cond->ecllim = limit;
}



/*************************************************************************************************
 * API for database
 *************************************************************************************************/


/* Get the string of an error code. */
const char *est_err_msg(int ecode){
  switch(ecode){
  case ESTENOERR: return "no error";
  case ESTEINVAL: return "invalid argument";
  case ESTEACCES: return "access forbidden";
  case ESTELOCK: return "lock failure";
  case ESTEDB: return "database problem";
  case ESTEIO: return "I/O problem";
  case ESTENOITEM: return "no such item";
  default: break;
  }
  return "miscellaneous";
}


/* Open a database. */
ESTDB *est_db_open(const char *name, int omode, int *ecp){
  ESTDB *db;
  DEPOT *metadb;
  ESTIDX *idxdb;
  CURIA *attrdb, *textdb, *kwddb;
  VILLA *fwmdb, *listdb;
  char path[ESTPATHBUFSIZ], vbuf[ESTNUMBUFSIZ];
  int domode, comode, vomode, idxnum, dseq, dnum, amode, vsiz;
  assert(name && ecp);
  *ecp = ESTENOERR;
  if((omode & ESTDBWRITER) && (omode & ESTDBCREAT) && !est_mkdir(name)){
    switch(errno){
    case EACCES:
      *ecp = ESTEACCES;
      return NULL;
    case EEXIST:
      break;
    default:
      *ecp = ESTEIO;
      return NULL;
    }
  }
  domode = DP_OREADER;
  comode = CR_OREADER;
  vomode = VL_OREADER;
  if(omode & ESTDBWRITER){
    domode = DP_OWRITER;
    comode = CR_OWRITER;
    vomode = VL_OWRITER | (ESTUSEZLIB ? VL_OZCOMP : 0);
    if(omode & ESTDBCREAT){
      domode |= DP_OCREAT;
      comode |= CR_OCREAT;
      vomode |= VL_OCREAT;
    }
    if(omode & ESTDBTRUNC){
      domode |= DP_OTRUNC;
      comode |= CR_OTRUNC;
      vomode |= VL_OTRUNC;
    }
  }
  if(omode & ESTDBNOLCK){
    domode |= DP_ONOLCK;
    comode |= CR_ONOLCK;
    vomode |= VL_ONOLCK;
  }
  if(omode & ESTDBLCKNB){
    domode |= DP_OLCKNB;
    comode |= CR_OLCKNB;
    vomode |= VL_OLCKNB;
  }
  idxnum = 0;
  dseq = 0;
  dnum = 0;
  amode = ESTAMNORMAL;
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTMETADBNAME);
  if((metadb = dpopen(path, domode, ESTMINIBNUM)) != NULL){
    if((vsiz = dpgetwb(metadb, ESTKEYIDXNUM, -1, 0, ESTNUMBUFSIZ - 1, vbuf)) > 0){
      vbuf[vsiz] = '\0';
      idxnum = atoi(vbuf);
    }
    if((vsiz = dpgetwb(metadb, ESTKEYDSEQ, -1, 0, ESTNUMBUFSIZ - 1, vbuf)) > 0){
      vbuf[vsiz] = '\0';
      dseq = atoi(vbuf);
    }
    if((vsiz = dpgetwb(metadb, ESTKEYDNUM, -1, 0, ESTNUMBUFSIZ - 1, vbuf)) > 0){
      vbuf[vsiz] = '\0';
      dnum = atoi(vbuf);
    }
    if((vsiz = dpgetwb(metadb, ESTKEYAMODE, -1, 0, ESTNUMBUFSIZ - 1, vbuf)) > 0){
      vbuf[vsiz] = '\0';
      amode = atoi(vbuf);
    } else if(omode & ESTDBPERFNG){
      amode = ESTAMPERFNG;
    }
  }
  if(!metadb){
    *ecp = dpecode == DP_ELOCK ? ESTELOCK : ESTEDB;
    return NULL;
  }
  if(idxnum < 1) idxnum = 1;
  if(dseq < 0) dseq = 0;
  if(dnum < 0) dnum = 0;
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTIDXDBNAME);
  idxdb = est_idx_open(path, vomode, idxnum);
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTFWMDBNAME);
  fwmdb = vlopen(path, vomode, VL_CMPLEX);
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTATTRDBNAME);
  attrdb = cropen(path, comode, ESTATTRDBBNUM, ESTATTRDBDNUM);
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTTEXTDBNAME);
  textdb = cropen(path, comode, ESTTEXTDBBNUM, ESTTEXTDBDNUM);
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTKWDDBNAME);
  kwddb = cropen(path, comode, ESTKWDDBBNUM, ESTKWDDBDNUM);
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTLISTDBNAME);
  listdb = vlopen(path, vomode, VL_CMPLEX);
  if(!metadb || !idxdb || !fwmdb || !attrdb ||!textdb || !listdb){
    if(listdb) vlclose(listdb);
    if(kwddb) crclose(kwddb);
    if(textdb) crclose(textdb);
    if(attrdb) crclose(attrdb);
    if(fwmdb) vlclose(fwmdb);
    if(idxdb) est_idx_close(idxdb);
    dpclose(metadb);
    *ecp = ESTEDB;
    return NULL;
  }
  if(omode & ESTDBWRITER){
    crsetalign(attrdb, ESTATTRDBALN);
    crsetalign(textdb, ESTTEXTDBALN);
    crsetalign(kwddb, ESTKWDDBALN);
    est_idx_set_tuning(idxdb, ESTIDXDBLRM, ESTIDXDBNIM, ESTIDXDBLCN, ESTIDXDBNCN);
    est_idx_set_current(idxdb);
    vlsettuning(fwmdb, ESTFWMDBLRM, ESTFWMDBNIM, ESTFWMDBLCN, ESTFWMDBNCN);
    vlsettuning(listdb, ESTLISTDBLRM, ESTLISTDBNIM, ESTLISTDBLCN, ESTLISTDBNCN);
  } else {
    est_idx_set_tuning(idxdb, -1, -1, ESTIDXDBRLCN, ESTIDXDBRNCN);
    vlsettuning(fwmdb, -1, -1, ESTFWMDBLCN, ESTFWMDBNCN);
    vlsettuning(listdb, -1, -1, ESTLISTDBLCN, ESTLISTDBNCN);
  }
  CB_MALLOC(db, sizeof(ESTDB));
  db->name = cbmemdup(name, -1);
  db->metadb = metadb;
  db->idxdb = idxdb;
  db->fwmdb = fwmdb;
  db->attrdb = attrdb;
  db->textdb = textdb;
  db->kwddb = kwddb;
  db->listdb = listdb;
  db->ecode = ESTENOERR;
  db->fatal = FALSE;
  db->dseq = dseq;
  db->dnum = dnum;
  db->amode = amode;
  if(omode & ESTDBWRITER){
    db->idxcc = cbmapopenex(ESTIDXCCBNUM);
    db->icsiz = 0;
    db->icmax = ESTIDXCCMAX;
    db->outcc = cbmapopenex(ESTOUTCCBNUM);
  } else {
    db->idxcc = cbmapopenex(1);
    db->icsiz = 0;
    db->icmax = 0;
    db->outcc = cbmapopenex(1);
  }
  db->keycc = cbmapopenex(ESTKEYCCMNUM + 1);
  db->kcmnum = ESTKEYCCMNUM;
  db->attrcc = cbmapopenex(ESTATTRCCMNUM + 1);
  db->acmnum = ESTATTRCCMNUM;
  db->textcc = cbmapopenex(ESTTEXTCCMNUM + 1);
  db->tcmnum = ESTTEXTCCMNUM;
  db->veccc = cbmapopenex(ESTATTRCCMNUM / 2 + 1);
  db->vcmnum = ESTATTRCCMNUM / 2;
  db->rescc = cbmapopenex(ESTRESCCMNUM * 2 + 1);
  db->rcmnum = ESTRESCCMNUM;
  db->spacc = NULL;
  db->scmnum = 0;
  db->scname = NULL;
  db->cbinfo = NULL;
  db->dfdb = NULL;
  db->metacc = NULL;
  db->intflag = FALSE;
  return db;
}


/* Close a database. */
int est_db_close(ESTDB *db, int *ecp){
  int err;
  assert(db && ecp);
  *ecp = ESTENOERR;
  err = FALSE;
  if(dpwritable(db->metadb)){
    if(!est_db_flush(db, -1) || !est_db_write_meta(db)) err = TRUE;
  }
  est_db_inform(db, "closing");
  if(db->metacc) cbmapclose(db->metacc);
  if(db->spacc){
    free(db->scname);
    cbmapclose(db->spacc);
  }
  cbmapclose(db->rescc);
  cbmapclose(db->veccc);
  cbmapclose(db->textcc);
  cbmapclose(db->attrcc);
  cbmapclose(db->keycc);
  cbmapclose(db->outcc);
  cbmapclose(db->idxcc);
  if(!vlclose(db->listdb)) err = TRUE;
  if(!crclose(db->kwddb)) err = TRUE;
  if(!crclose(db->textdb)) err = TRUE;
  if(!crclose(db->attrdb)) err = TRUE;
  if(!vlclose(db->fwmdb)) err = TRUE;
  if(!est_idx_close(db->idxdb)) err = TRUE;
  if(!dpclose(db->metadb)) err = TRUE;
  free(db->name);
  if(db->fatal){
    *ecp = db->ecode;
    err = TRUE;
  } else if(err){
    *ecp = ESTEDB;
  }
  free(db);
  return err ? FALSE : TRUE;
}


/* Get the last happended error code of a database. */
int est_db_error(ESTDB *db){
  assert(db);
  return db->ecode;
}


/* Check whether a database has a fatal error. */
int est_db_fatal(ESTDB *db){
  assert(db);
  return db->fatal;
}


/* Flush index words in the cache of a database. */
int est_db_flush(ESTDB *db, int max){
  CBMAP *ids;
  CBLIST *keys;
  CBDATUM *nval;
  const char *kbuf, *vbuf, *rp, *pv;
  char *tbuf;
  int i, j, inc, err, ksiz, vsiz, rnum, id, dnum, tsiz;
  assert(db);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  if(cbmaprnum(db->idxcc) < 1 && cbmaprnum(db->outcc) < 1) return TRUE;
  db->intflag = FALSE;
  inc = est_db_used_cache_size(db) > db->icmax;
  err = FALSE;
  keys = cblistopen();
  cbmapiterinit(db->idxcc);
  while((kbuf = cbmapiternext(db->idxcc, &ksiz)) != NULL){
    cblistpush(keys, kbuf, ksiz);
  }
  rnum = CB_LISTNUM(keys);
  cblistsort(keys);
  if(max > 0){
    while(CB_LISTNUM(keys) > max){
      free(cblistpop(keys, NULL));
    }
  }
  for(i = 0; i < CB_LISTNUM(keys); i++){
    kbuf = CB_LISTVAL2(keys, i, &ksiz);
    vbuf = cbmapget(db->idxcc, kbuf, ksiz, &vsiz);
    if(!est_idx_add(db->idxdb, kbuf, ksiz, vbuf, vsiz) ||
       (!vlput(db->fwmdb, kbuf, ksiz, "", 0, VL_DKEEP) && dpecode != DP_EKEEP)){
      err = TRUE;
      break;
    }
    cbmapout(db->idxcc, kbuf, ksiz);
    db->icsiz -= vsiz;
    if(i % ESTCCCBFREQ == 0){
      est_db_inform(db, "flushing index words");
      if(est_idx_size_current(db->idxdb) >= ESTIDXDBMAX){
        est_db_inform(db, "adding a new database file");
        est_idx_increment(db->idxdb);
        inc = FALSE;
      }
    }
    if(max > 0 && db->intflag && i > 0 && i % ESTCCIRSLOT == 0) break;
  }
  cblistclose(keys);
  if(cbmaprnum(db->idxcc) < 1){
    cbmapclose(db->idxcc);
    db->idxcc = cbmapopenex(rnum > ESTIDXCCBNUM ? rnum * 1.5 : ESTIDXCCBNUM);
  }
  if(max < 1 && cbmaprnum(db->outcc) > 0){
    ids = cbmapopen();
    keys = cblistopen();
    cbmapiterinit(db->outcc);
    while((kbuf = cbmapiternext(db->outcc, &ksiz)) != NULL){
      if(*kbuf == '\t'){
        id = atoi(kbuf + 1);
        cbmapput(ids, (char *)&id, sizeof(int), "", 0, FALSE);
      } else {
        cblistpush(keys, kbuf, ksiz);
      }
    }
    cblistsort(keys);
    dnum = est_idx_dnum(db->idxdb);
    for(i = 0; i < CB_LISTNUM(keys); i++){
      kbuf = CB_LISTVAL2(keys, i, &ksiz);
      for(j = 0; j < dnum; j++){
        if((tbuf = est_idx_get_one(db->idxdb, j, kbuf, ksiz, &tsiz)) != NULL){
          nval = cbdatumopen("", 0);
          rp = tbuf;
          while(rp < tbuf + tsiz){
            pv = rp;
            rp += 5;
            while(*rp != 0x0){
              rp += 2;
            }
            rp++;
            if(!cbmapget(ids, pv, sizeof(int), NULL)) cbdatumcat(nval, pv, rp - pv);
          }
          if(!est_idx_put_one(db->idxdb, j, kbuf, ksiz, CB_DATUMPTR(nval), CB_DATUMSIZE(nval)))
            err = TRUE;
          cbdatumclose(nval);
          free(tbuf);
        }
      }
      cbmapout(db->outcc, kbuf, ksiz);
      if(i % ESTCCCBFREQ == 0) est_db_inform(db, "cleaning dispensable keys");
      if(max > 0 && db->intflag && i > 0 && i % ESTCCIRSLOT == 0) break;
    }
    cblistclose(keys);
    cbmapclose(ids);
    if(!(max > 0 && db->intflag)){
      cbmapclose(db->outcc);
      db->outcc = cbmapopenex(ESTOUTCCBNUM);
    }
  }
  cbmapclose(db->keycc);
  db->keycc = cbmapopenex(ESTKEYCCMNUM + 1);
  db->kcmnum = ESTKEYCCMNUM;
  if(!(max > 0 && db->intflag) && inc && est_idx_size_current(db->idxdb) >= ESTIDXDBMIN){
    est_db_inform(db, "adding a new database file");
    est_idx_increment(db->idxdb);
  }
  if(max > 0 && db->intflag) est_db_inform(db, "flushing interrupted");
  db->intflag = FALSE;
  if(err){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    return FALSE;
  }
  return TRUE;
}


/* Synchronize updating contents of a database. */
int est_db_sync(ESTDB *db){
  int err;
  assert(db);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  err = FALSE;
  if(!est_db_flush(db, -1) || !est_db_write_meta(db)) err = TRUE;
  est_db_inform(db, "synchronizing the database for meta information");
  if(!dpsync(db->metadb)) err = TRUE;
  est_db_inform(db, "synchronizing the inverted index");
  if(!est_idx_sync(db->idxdb)) err = TRUE;
  est_db_inform(db, "synchronizing the database for forward matching");
  if(!vlsync(db->fwmdb)) err = TRUE;
  est_db_inform(db, "synchronizing the database for attrutes");
  if(!crsync(db->attrdb)) err = TRUE;
  est_db_inform(db, "synchronizing the database for texts");
  if(!crsync(db->textdb)) err = TRUE;
  est_db_inform(db, "synchronizing the database for keywords");
  if(!crsync(db->kwddb)) err = TRUE;
  est_db_inform(db, "synchronizing the database for document list");
  if(!vlsync(db->listdb)) err = TRUE;
  if(err){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
  }
  return err ? FALSE : TRUE;
}


/* Optimize a database. */
int est_db_optimize(ESTDB *db, int options){
  CBMAP *dmap;
  CBLIST *words;
  CBDATUM *nval;
  const char *word, *rp, *pv;
  char *kbuf, *vbuf;
  int i, err, id, ksiz, vsiz, wsiz;
  assert(db);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  if(!est_db_flush(db, -1)) return FALSE;
  err = FALSE;
  if(!(options & ESTOPTNOPURGE)){
    dmap = cbmapopenex(vlrnum(db->listdb) + 1);
    vlcurfirst(db->listdb);
    while((vbuf = vlcurval(db->listdb, NULL)) != NULL){
      id = atoi(vbuf);
      cbmapput(dmap, (char *)&id, sizeof(int), "", 0, FALSE);
      free(vbuf);
      vlcurnext(db->listdb);
    }
    words = cblistopen();
    vlcurfirst(db->fwmdb);
    while((kbuf = vlcurkey(db->fwmdb, &ksiz)) != NULL){
      cblistpushbuf(words, kbuf, ksiz);
      vlcurnext(db->fwmdb);
    }
    for(i = 0; i < CB_LISTNUM(words); i++){
      if(i % (ESTIDXDBLRM * 4) == 0) est_idx_set_current(db->idxdb);
      word = CB_LISTVAL2(words, i, &wsiz);
      if((vbuf = est_idx_get(db->idxdb, word, wsiz, &vsiz)) != NULL){
        nval = cbdatumopen("", 0);
        rp = vbuf;
        while(rp < vbuf + vsiz){
          pv = rp;
          rp += 5;
          while(*rp != 0x0){
            rp += 2;
          }
          rp++;
          if(cbmapget(dmap, pv, sizeof(int), NULL)) cbdatumcat(nval, pv, rp - pv);
        }
        if(!est_idx_out(db->idxdb, word, wsiz)) err = TRUE;
        if(CB_DATUMSIZE(nval) > 0){
          if(!est_idx_add(db->idxdb, word, wsiz, CB_DATUMPTR(nval), CB_DATUMSIZE(nval)))
            err = TRUE;
        } else {
          if(!vlout(db->fwmdb, word, wsiz)) err = TRUE;
        }
        cbdatumclose(nval);
        free(vbuf);
      } else {
        err = TRUE;
      }
      free(kbuf);
      if(i % ESTCCCBFREQ == 0) est_db_inform(db, "cleaning dispensable keys");
    }
    cblistclose(words);
    cbmapclose(dmap);
  }
  if(!(options & ESTOPTNODBOPT)){
    est_db_inform(db, "optimizing the inverted index");
    if(!est_idx_optimize(db->idxdb)) err = TRUE;
    est_db_inform(db, "optimizing the database for forward matching");
    if(!vloptimize(db->fwmdb)) err = TRUE;
    est_db_inform(db, "optimizing the database for attrutes");
    if(!croptimize(db->attrdb, -1)) err = TRUE;
    est_db_inform(db, "optimizing the database for texts");
    if(!croptimize(db->textdb, -1)) err = TRUE;
    est_db_inform(db, "optimizing the database for keywords");
    if(!croptimize(db->kwddb, -1)) err = TRUE;
    est_db_inform(db, "optimizing the database for document list");
    if(!vloptimize(db->listdb)) err = TRUE;
  }
  cbmapclose(db->rescc);
  db->rescc = cbmapopenex(db->rcmnum * 2 + 1);
  if(err){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
  }
  return err ? FALSE : TRUE;
}


/* Add a document to a database. */
int est_db_put_doc(ESTDB *db, ESTDOC *doc, int options){
  CBMAP *ocmap, *fmap, *qmap;
  CBLIST *words;
  CBDATUM *ocbuf;
  md5_state_t ms;
  const char *uri, *ndig, *text, *word, *fnext, *snext, *kbuf, *vbuf;
  unsigned char junc[2], c;
  char dobuf[32], dsbuf[64], *wp, *odig, wbuf[ESTWORDMAXLEN+3], *sbuf, *zbuf, nbuf[ESTNUMBUFSIZ];
  int i, j, id, err, wnum, wsiz, fnsiz, snsiz, *np, num, ksiz, vsiz, ssiz, zsiz;
  double tune;
  assert(db && doc);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  if(!doc->attrs || !(uri = cbmapget(doc->attrs, ESTDATTRURI, -1, NULL)) || uri[0] == '\0'){
    db->ecode = ESTEINVAL;
    return FALSE;
  }
  if(!doc->dtexts) doc->dtexts = cblistopen();
  if(!(ndig = cbmapget(doc->attrs, ESTDATTRDIGEST, -1, NULL))){
    md5_init(&ms);
    for(i = 0; i < cblistnum(doc->dtexts); i++){
      vbuf = cblistval(doc->dtexts, i, &vsiz);
      md5_append(&ms, (md5_byte_t *)vbuf, vsiz);
      md5_append(&ms, (md5_byte_t *)"\n", 1);
    }
    if((vbuf = cbmapget(doc->attrs, "", 0, &vsiz)) != NULL){
      md5_append(&ms, (md5_byte_t *)"\t", 1);
      md5_append(&ms, (md5_byte_t *)vbuf, vsiz);
      md5_append(&ms, (md5_byte_t *)"\n", 1);
    }
    md5_finish(&ms, (md5_byte_t *)dobuf);
    wp = dsbuf;
    for(i = 0; i < 16; i++){
      wp += sprintf(wp, "%02x", ((unsigned char *)dobuf)[i]);
    }
    ndig = dsbuf;
    cbmapput(doc->attrs, ESTDATTRDIGEST, -1, ndig, -1, FALSE);
  }
  if((id = est_db_uri_to_id(db, uri)) > 0){
    if((odig = est_db_get_doc_attr(db, id, ESTDATTRDIGEST)) != NULL){
      if(!strcmp(odig, ndig)){
        free(odig);
        doc->id = id;
        sprintf(nbuf, "%d", id);
        cbmapput(doc->attrs, ESTDATTRID, -1, nbuf, -1, TRUE);
        return est_db_edit_doc(db, doc);
      }
      free(odig);
    }
    if(!est_db_out_doc(db, id, (options & ESTPDCLEAN) ? ESTODCLEAN : 0)) return FALSE;
  }
  doc->id = ++(db->dseq);
  sprintf(nbuf, "%d", doc->id);
  cbmapput(doc->attrs, ESTDATTRID, -1, nbuf, -1, TRUE);
  ocmap = cbmapopen();
  fmap = cbmapopen();
  qmap = cbmapopen();
  wnum = 0;
  for(i = -1; i < CB_LISTNUM(doc->dtexts); i++){
    if(i < 0){
      if(!(text = cbmapget(doc->attrs, "", 0, NULL))) continue;
    } else {
      text = CB_LISTVAL(doc->dtexts, i, NULL);
    }
    words = cblistopen();
    switch(db->amode){
    case ESTAMPERFNG:
      est_break_text_perfng(text, words, FALSE, TRUE);
      break;
    default:
      est_break_text(text, words, FALSE, TRUE);
      break;
    }
    wnum += CB_LISTNUM(words);
    for(j = 0; j < CB_LISTNUM(words); j++){
      word = CB_LISTVAL2(words, j, &wsiz);
      if(wsiz > ESTWORDMAXLEN) continue;
      fnext = cblistval(words, j + 1, &fnsiz);
      snext = cblistval(words, j + 2, &snsiz);
      junc[0] = fnext ? dpinnerhash(fnext, fnsiz) % ESTJHASHNUM + 1: 0xff;
      junc[1] = snext ? dpouterhash(snext, snsiz) % ESTJHASHNUM + 1: 0xff;
      memcpy(wbuf, word, wsiz);
      memcpy(wbuf + wsiz, "\t", 1);
      memcpy(wbuf + wsiz + 1, junc, 2);
      np = (int *)cbmapget(fmap, word, wsiz, NULL);
      num = np ? *(int *)np : 0;
      num += ESTOCPOINT;
      cbmapput(fmap, word, wsiz, (char *)&num, sizeof(int), TRUE);
      if(cbmapput(qmap, wbuf, wsiz + 3, "", 0, FALSE))
        cbmapputcat(ocmap, word, wsiz, (char *)junc, 2);
    }
    cblistclose(words);
  }
  tune = log(wnum + 32);
  tune = (tune * tune) / 12.0;
  cbmapiterinit(ocmap);
  while((kbuf = cbmapiternext(ocmap, &ksiz)) != NULL){
    vbuf = cbmapget(ocmap, kbuf, ksiz, &vsiz);
    ocbuf = cbdatumopen("", 0);
    cbdatumcat(ocbuf, (char *)&(doc->id), sizeof(int));
    num = *(int *)cbmapget(fmap, kbuf, ksiz, NULL) / tune;
    if(num >= 0x80) num += (0x80 - num) * 0.75;
    if(num >= 0xc0) num += (0xc0 - num) * 0.75;
    c = num < 0xff ? num : 0xff;
    cbdatumcat(ocbuf, (char *)&c, 1);
    cbdatumcat(ocbuf, vbuf, vsiz);
    c = 0x00;
    cbdatumcat(ocbuf, (char *)&c, 1);
    cbmapputcat(db->idxcc, kbuf, ksiz, CB_DATUMPTR(ocbuf), CB_DATUMSIZE(ocbuf));
    db->icsiz += CB_DATUMSIZE(ocbuf);
    cbdatumclose(ocbuf);
  }
  cbmapclose(qmap);
  cbmapclose(fmap);
  cbmapclose(ocmap);
  err = FALSE;
  sbuf = cbmapdump(doc->attrs, &ssiz);
  if(!crput(db->attrdb, (char *)&(doc->id), sizeof(int), sbuf, ssiz, CR_DKEEP)){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    err = TRUE;
  }
  free(sbuf);
  sbuf = cblistdump(doc->dtexts, &ssiz);
  if(!(zbuf = est_deflate(sbuf, ssiz, &zsiz))){
    CB_MALLOC(zbuf, 1);
    zsiz = 0;
    db->ecode = ESTEMISC;
    db->fatal = TRUE;
    err = TRUE;
  }
  if(!crput(db->textdb, (char *)&(doc->id), sizeof(int), zbuf, zsiz, CR_DKEEP)){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    err = TRUE;
  }
  free(zbuf);
  free(sbuf);
  sprintf(nbuf, "%d", doc->id);
  if(!vlput(db->listdb, uri, -1, nbuf, -1, VL_DKEEP)){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    err = TRUE;
  }
  db->dnum++;
  if(est_db_used_cache_size(db) > db->icmax && !est_db_flush(db, INT_MAX)) err = TRUE;
  return err ? FALSE : TRUE;
}


/* Remove a document from a database. */
int est_db_out_doc(ESTDB *db, int id, int options){
  ESTDOC *doc;
  CBLIST *words;
  const char *uri, *text, *word;
  char numbuf[ESTNUMBUFSIZ];
  int i, j, len, wsiz;
  assert(db && id > 0);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  if(!(doc = est_db_get_doc(db, id, 0))) return FALSE;
  if(!doc->attrs || !(uri = cbmapget(doc->attrs, ESTDATTRURI, -1, NULL))){
    est_doc_delete(doc);
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    return FALSE;
  }
  if(!crout(db->attrdb, (char *)&id, sizeof(int)) ||
     !crout(db->textdb, (char *)&id, sizeof(int)) ||
     (!crout(db->kwddb, (char *)&id, sizeof(int)) && dpecode != DP_ENOITEM) ||
     !vlout(db->listdb, uri, -1)){
    est_doc_delete(doc);
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    return FALSE;
  }
  cbmapout(db->attrcc, (char *)&id, sizeof(int));
  cbmapout(db->textcc, (char *)&id, sizeof(int));
  cbmapout(db->veccc, (char *)&id, sizeof(int));
  if(db->spacc) cbmapout(db->spacc, (char *)&id, sizeof(int));
  if((options & ESTODCLEAN) && doc->dtexts){
    len = sprintf(numbuf, "\t%d", doc->id);
    cbmapput(db->outcc, numbuf, len, "", 0, FALSE);
    for(i = -1; i < CB_LISTNUM(doc->dtexts); i++){
      if(i < 0){
        if(!(text = cbmapget(doc->attrs, "", 0, NULL))) continue;
      } else {
        text = CB_LISTVAL(doc->dtexts, i, NULL);
      }
      words = cblistopen();
      switch(db->amode){
      case ESTAMPERFNG:
        est_break_text_perfng(text, words, FALSE, TRUE);
        break;
      default:
        est_break_text(text, words, FALSE, TRUE);
        break;
      }
      for(j = 0; j < CB_LISTNUM(words); j++){
        word = CB_LISTVAL2(words, j, &wsiz);
        cbmapput(db->outcc, word, wsiz, "", 0, FALSE);
      }
      cblistclose(words);
    }
  }
  est_doc_delete(doc);
  if(!est_db_set_doc_entity(db, id, NULL, -1) && db->ecode != ESTENOITEM) return FALSE;
  db->dnum--;
  return TRUE;
}


/* Edit attributes of a document object in a database. */
int est_db_edit_doc(ESTDB *db, ESTDOC *doc){
  const char *uri, *tmp;
  char *ouri, numbuf[ESTNUMBUFSIZ], *text, *sbuf;
  int err, id, oid, ssiz;
  assert(db && doc);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  id = -1;
  uri = NULL;
  if(doc->attrs){
    if((tmp = cbmapget(doc->attrs, ESTDATTRID, -1, NULL)) != NULL) id = atoi(tmp);
    if((tmp = cbmapget(doc->attrs, ESTDATTRURI, -1, NULL)) != NULL) uri = tmp;
  }
  if(id < 1 || (doc->id > 0 && doc->id != id) || !uri || uri[0] == '\0'){
    db->ecode = ESTEINVAL;
    return FALSE;
  }
  err = FALSE;
  if((oid = est_db_uri_to_id(db, uri)) == -1){
    if(!(ouri = est_db_get_doc_attr(db, id, ESTDATTRURI))){
      db->ecode = ESTEINVAL;
      return FALSE;
    }
    sprintf(numbuf, "%d", id);
    if(!vlout(db->listdb, ouri, -1) || !vlput(db->listdb, uri, -1, numbuf, -1, VL_DKEEP))
      err = TRUE;
    free(ouri);
  } else if(oid != id){
    db->ecode = ESTEINVAL;
    return FALSE;
  }
  if((text = est_db_get_doc_attr(db, id, "")) != NULL){
    cbmapput(doc->attrs, "", 0, text, -1, TRUE);
    free(text);
  }
  sbuf = cbmapdump(doc->attrs, &ssiz);
  if(!crput(db->attrdb, (char *)&id, sizeof(int), sbuf, ssiz, CR_DOVER)){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    err = TRUE;
  }
  free(sbuf);
  cbmapout(db->attrcc, (char *)&id, sizeof(int));
  if(db->spacc) cbmapout(db->spacc, (char *)&id, sizeof(int));
  return err ? FALSE : TRUE;
}


/* Retrieve a document in a database. */
ESTDOC *est_db_get_doc(ESTDB *db, int id, int options){
  ESTDOC *doc;
  const char *cbuf;
  char *vbuf, *zbuf;
  int i, csiz, vsiz, zsiz, num;
  assert(db && id > 0);
  cbuf = NULL;
  if(options & ESTGDNOATTR){
    if(!crvsiz(db->attrdb, (char *)&id, sizeof(int))){
      if(dpecode == DP_ENOITEM){
        db->ecode = ESTENOITEM;
        return NULL;
      } else {
        db->ecode = ESTEDB;
        db->fatal = TRUE;
        return NULL;
      }
    }
    vbuf = NULL;
  } else if((cbuf = cbmapget(db->attrcc, (char *)&id, sizeof(int), &csiz)) != NULL){
    cbmapmove(db->attrcc, (char *)&id, sizeof(int), FALSE);
    vbuf = NULL;
  } else if(!(vbuf = crget(db->attrdb, (char *)&id, sizeof(int), 0, -1, &vsiz))){
    if(dpecode == DP_ENOITEM){
      db->ecode = ESTENOITEM;
      return NULL;
    } else {
      db->ecode = ESTEDB;
      db->fatal = TRUE;
      return NULL;
    }
  }
  doc = est_doc_new();
  doc->id = id;
  if(cbuf){
    doc->attrs = cbmapload(cbuf, csiz);
  } else if(vbuf){
    doc->attrs = cbmapload(vbuf, vsiz);
    cbmapputvbuf(db->attrcc, (char *)&id, sizeof(int), vbuf, vsiz);
    if(cbmaprnum(db->attrcc) > db->acmnum){
      num = cbmaprnum(db->attrcc) * 0.1 + 1;
      cbmapiterinit(db->attrcc);
      for(i = 0; i < num && (cbuf = cbmapiternext(db->attrcc, NULL)) != NULL; i++){
        cbmapout(db->attrcc, cbuf, sizeof(int));
      }
    }
  } else {
    doc->attrs = NULL;
  }
  if(!(options & ESTGDNOTEXT)){
    if((cbuf = cbmapget(db->textcc, (char *)&id, sizeof(int), &csiz)) != NULL){
      cbmapmove(db->textcc, (char *)&id, sizeof(int), FALSE);
      doc->dtexts = cblistload(cbuf, csiz);
    } else {
      if(!(zbuf = crget(db->textdb, (char *)&id, sizeof(int), 0, -1, &zsiz))){
        db->ecode = ESTEDB;
        db->fatal = TRUE;
        est_doc_delete(doc);
        return NULL;
      }
      if(!(vbuf = est_inflate(zbuf, zsiz, &vsiz))){
        db->ecode = ESTEDB;
        db->fatal = TRUE;
        free(zbuf);
        est_doc_delete(doc);
        return NULL;
      }
      doc->dtexts = cblistload(vbuf, vsiz);
      cbmapputvbuf(db->textcc, (char *)&id, sizeof(int), vbuf, vsiz);
      if(cbmaprnum(db->textcc) > db->tcmnum){
        num = cbmaprnum(db->textcc) * 0.1 + 1;
        cbmapiterinit(db->textcc);
        for(i = 0; i < num &&(cbuf = cbmapiternext(db->textcc, NULL)) != NULL; i++){
          cbmapout(db->textcc, cbuf, sizeof(int));
        }
      }
      free(zbuf);
    }
  }
  return doc;
}


/* Retrieve the value of an attribute of a document in a database. */
char *est_db_get_doc_attr(ESTDB *db, int id, const char *name){
  const char *cbuf;
  char *mbuf, *vbuf;
  int cb, csiz, msiz, vsiz;
  assert(db && id > 0 && name);
  cb = db->spacc && !strcmp(name, db->scname);
  if(cb && (cbuf = cbmapget(db->spacc, (char *)&id, sizeof(int), &csiz)) != NULL){
    cbmapmove(db->spacc, (char *)&id, sizeof(int), FALSE);
    return cbmemdup(cbuf, csiz);
  }
  if(!(mbuf = crget(db->attrdb, (char *)&id, sizeof(int), 0, -1, &msiz))){
    db->ecode = dpecode == DP_ENOITEM ? ESTENOITEM : ESTEDB;
    return NULL;
  }
  if(!(vbuf = cbmaploadone(mbuf, msiz, name, -1, &vsiz))){
    db->ecode = ESTENOITEM;
    free(mbuf);
    return NULL;
  }
  if(cb) cbmapput(db->spacc, (char *)&id, sizeof(int), vbuf, vsiz, FALSE);
  free(mbuf);
  return vbuf;
}


/* Get the ID of a document spacified by URI. */
int est_db_uri_to_id(ESTDB *db, const char *uri){
  char *vbuf;
  int id;
  assert(db && uri);
  if(!(vbuf = vlget(db->listdb, uri, -1, NULL))){
    db->ecode = ESTENOITEM;
    return -1;
  }
  id = atoi(vbuf);
  free(vbuf);
  return id;
}


/* Get the name of a database. */
const char *est_db_name(ESTDB *db){
  assert(db);
  return db->name;
}


/* Get the number of documents in a database. */
int est_db_doc_num(ESTDB *db){
  assert(db);
  return db->dnum;
}


/* Get the number of words in a database. */
int est_db_word_num(ESTDB *db){
  assert(db);
  return vlrnum(db->fwmdb);
}


/* Get the size of a database. */
double est_db_size(ESTDB *db){
  assert(db);
  return (double)dpfsiz(db->metadb) + est_idx_size(db->idxdb) + (double)vlfsiz(db->fwmdb) +
    crfsizd(db->attrdb) + crfsizd(db->textdb) + crfsizd(db->kwddb) + (double)vlfsiz(db->listdb);
}


/* Search documents corresponding a condition for a database. */
int *est_db_search(ESTDB *db, ESTCOND *cond, int *nump, CBMAP *hints){
  ESTSCORE *scores, *tscores;
  CBMAP *svmap;
  CBLIST *terms;
  const char *term, *rp;
  char *tmp, numbuf[ESTNUMBUFSIZ];
  int i, j, snum, pcnum, ncnum, tsnum, add, nnum, id, score, hnum, len, rest, *rval;
  double tune;
  assert(db && cond && nump);
  scores = NULL;
  snum = 0;
  if(cond->phrase){
    if(cbstrfwmatch(cond->phrase, ESTOPID)){
      if((id = atoi(cond->phrase + strlen(ESTOPID))) > 0){
        CB_MALLOC(scores, sizeof(ESTSCORE));
        scores[0].id = id;
        scores[0].score = 0;
        snum = 1;
      } else {
        CB_MALLOC(scores, 1);
        snum = 0;
      }
    } else if(cbstrfwmatch(cond->phrase, ESTOPURI)){
      rp = cond->phrase + strlen(ESTOPURI);
      while(*rp > '\0' && *rp <= ' '){
        rp++;
      }
      if((id = est_db_uri_to_id(db, rp)) > 0){
        CB_MALLOC(scores, sizeof(ESTSCORE));
        scores[0].id = id;
        scores[0].score = 0;
        snum = 1;
      } else {
        CB_MALLOC(scores, 1);
        snum = 0;
      }
    } else if(cbstrfwmatch(cond->phrase, ESTOPSIMILAR)){
      rp = cond->phrase + strlen(ESTOPSIMILAR);
      while(*rp > '\0' && *rp <= ' '){
        rp++;
      }
      svmap = est_phrase_vector(rp);
      scores = est_search_similar(db, svmap, &snum, ESTSMLRKNUM, ESTSMLRUNUM, cond->tfidf,
                                  cond->order ? ESTSMLRNMIN : 0.0);
      cbmapclose(svmap);
    } else {
      if(cond->simple){
        tmp = est_phrase_from_thumb(cond->phrase);
        terms = est_phrase_terms(tmp);
        free(tmp);
      } else {
        terms = est_phrase_terms(cond->phrase);
      }
      pcnum = 0;
      ncnum = 0;
      add = TRUE;
      for(i = 0; i < CB_LISTNUM(terms); i++){
        term = CB_LISTVAL(terms, i, NULL);
        if(!strcmp(term, ESTOPISECT)){
          add = TRUE;
        } else if(!strcmp(term, ESTOPDIFF)){
          add = FALSE;
        } else {
          if(!strcmp(term, ESTOPUVSET)){
            tscores = est_search_uvset(db, &tsnum, hints, add);
          } else {
            tscores = est_search_union(db, term, cond->gstep, &tsnum, hints, add);
          }
          if(add){
            if(cond->tfidf){
              tune = log(tsnum + 3);
              tune =  tune * tune * tune;
              if(tune < 8.0) tune = 8.0;
              for(j = 0; j < tsnum; j++){
                tscores[j].score *= 10000 / tune;
              }
            }
            pcnum++;
          } else {
            ncnum++;
          }
          if(scores){
            CB_REALLOC(scores, (snum + tsnum) * sizeof(ESTSCORE) + 1);
            for(j = 0; j < tsnum; j++){
              scores[snum+j].id = tscores[j].id;
              scores[snum+j].score = add ? tscores[j].score : -1;
            }
            snum += tsnum;
            free(tscores);
          } else {
            scores = tscores;
            snum = tsnum;
          }
        }
      }
      if(scores){
        if(pcnum > 1 || ncnum > 0){
          qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_id);
          nnum = 0;
          for(i = 0; i < snum; i++){
            id = scores[i].id;
            score = scores[i].score;
            hnum = score >= 0 ? 1 : 0;
            for(j = i + 1; j < snum && scores[j].id == id; j++){
              if(score >= 0 && scores[j].score >= 0){
                score += scores[j].score;
                hnum++;
              } else {
                score = -1;
              }
            }
            if(score >= 0 && hnum >= pcnum){
              scores[nnum].id = id;
              scores[nnum].score = score;
              nnum++;
            }
            i = j - 1;
          }
          snum = nnum;
        }
      } else {
        CB_MALLOC(scores, 1);
        snum = 0;
      }
      cblistclose(terms);
    }
  } else if(cond->attrs){
    scores = est_search_uvset(db, &snum, hints, TRUE);
  } else {
    CB_MALLOC(scores, 1);
    snum = 0;
  }
  if(cbmaprnum(db->outcc) > 0){
    tsnum = 0;
    for(i = 0; i < snum; i++){
      len = sprintf(numbuf, "\t%d", scores[i].id);
      if(cbmapget(db->outcc, numbuf, len, NULL)) continue;
      scores[tsnum++] = scores[i];
    }
    snum = tsnum;
  }
  if(cond->max > 0 && cond->max * ESTATTRALW + 1 < snum && cond->attrs && !cond->order){
    qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_score);
    nnum = est_narrow_scores(db, cond->attrs, cond->order, scores, snum,
                             cond->max * ESTATTRALW + 1, &rest);
    if(hints){
      sprintf(numbuf, "%d",
              rest > cond->max / 2 ? (int)(snum * (nnum / (double)(snum - rest))) : nnum);
      cbmapput(hints, "", 0, numbuf, -1, FALSE);
    }
    snum = nnum;
  } else {
    if(cond->attrs || cond->order)
      snum = est_narrow_scores(db, cond->attrs, cond->order, scores, snum, INT_MAX, &rest);
    if(!cond->order) qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_score);
    if(hints){
      sprintf(numbuf, "%d", snum);
      cbmapput(hints, "", 0, numbuf, -1, FALSE);
    }
  }
  if(cond->shadows) cbmapclose(cond->shadows);
  if(cond->ecllim >= 0.0){
    cond->shadows = cbmapopenex(snum + 1);
    snum = est_eclipse_scores(db, scores, snum, cond->max > 0 ? cond->max : snum,
                              ESTECLKNUM, cond->tfidf, cond->ecllim, cond->shadows);
  } else {
    cond->shadows = NULL;
  }
  if(cond->max >= 0 && cond->max < snum) snum = cond->max;
  CB_MALLOC(rval, snum * sizeof(int) + 1);
  for(i = 0; i < snum; i++){
    rval[i] = scores[i].id;
  }
  if(cond->scfb){
    CB_REALLOC(cond->scores, snum * sizeof(int) + 1);
    for(i = 0; i < snum; i++){
      cond->scores[i] = scores[i].score;
    }
    cond->snum = snum;
  }
  *nump = snum;
  if(*nump < 1) db->ecode = ESTENOITEM;
  free(scores);
  return rval;
}


/* Check whether a document object matches the phrase of a search condition object definitely. */
int est_db_scan_doc(ESTDB *db, ESTDOC *doc, ESTCOND *cond){
  struct { char *word; int num; } wsets[ESTSCANWNUM];
  CBLIST *terms, *words;
  const char *term, *text;
  unsigned char *rbuf;
  char *tmp;
  int i, j, k, wsnum, add, rsiz, hit;
  assert(db && doc && cond);
  if(!cond->phrase || cbstrfwmatch(cond->phrase, ESTOPSIMILAR)) return FALSE;
  if(!doc->dtexts) doc->dtexts = cblistopen();
  if(cond->simple){
    tmp = est_phrase_from_thumb(cond->phrase);
    terms = est_phrase_terms(tmp);
    free(tmp);
  } else {
    terms = est_phrase_terms(cond->phrase);
  }
  wsnum = 0;
  add = TRUE;
  for(i = 0; i < CB_LISTNUM(terms); i++){
    term = CB_LISTVAL(terms, i, NULL);
    if(!strcmp(term, ESTOPISECT)){
      add = TRUE;
    } else if(!strcmp(term, ESTOPDIFF)){
      add = FALSE;
    } else if(add && strcmp(term, ESTOPUVSET)){
      if(term[0] == ' '){
        term++;
        if(term[0] == 'b'){
          term++;
        } else  if(term[0] == 'e'){
          term++;
        }
      }
      words = cbsplit(term, -1, "\t");
      while(wsnum < ESTSCANWNUM && CB_LISTNUM(words) > 0){
        wsets[wsnum].word = cblistshift(words, NULL);
        wsets[wsnum].num = i;
        wsnum++;
      }
      cblistclose(words);
    }
  }
  for(i = -1; i < CB_LISTNUM(doc->dtexts); i++){
    if(i < 0){
      if(!doc->attrs || !(text = cbmapget(doc->attrs, "", 0, NULL))) continue;
    } else {
      text = CB_LISTVAL(doc->dtexts, i, NULL);
    }
    rbuf = (unsigned char *)est_uconv_in(text, strlen(text), &rsiz);
    est_canonicalize_text(rbuf, rsiz, FALSE);
    tmp = est_uconv_out((char *)rbuf, rsiz, &rsiz);
    for(j = 0; j < wsnum; j++){
      if(!wsets[j].word) continue;
      if(strstr(tmp, wsets[j].word)){
        for(k = 0; k < wsnum; k++){
          if(!wsets[k].word) continue;
          if(wsets[k].num == wsets[j].num){
            free(wsets[k].word);
            wsets[k].word = NULL;
          }
        }
      }
    }
    free(tmp);
    free(rbuf);
  }
  hit = TRUE;
  for(i = 0; i < wsnum; i++){
    if(!wsets[i].word) continue;
    free(wsets[i].word);
    hit = FALSE;
  }
  cblistclose(terms);
  return hit;
}


/* Set the maximum size of the cache memory of a database. */
void est_db_set_cache_size(ESTDB *db, size_t size, int anum, int tnum, int rnum){
  assert(db);
  if(dpwritable(db->metadb) && size > 0) db->icmax = size;
  if(anum > 0) db->acmnum = anum;
  if(tnum > 0) db->tcmnum = tnum;
  if(rnum > 0) db->rcmnum = rnum;
  db->vcmnum = db->acmnum / 2 + 1;
}



/*************************************************************************************************
 * features for experts
 *************************************************************************************************/


/* Handle to the file of random number generator. */
FILE *est_random_ifp = NULL;


/* POSIX signal handlers. */
void (*est_signal_handlers[ESTSIGNUM])(int);


/* Break a sentence of text and extract words. */
void est_break_text(const char *text, CBLIST *list, int norm, int tail){
  CBLIST *words;
  const unsigned char *word, *next;
  unsigned char *utext;
  char *tmp;
  int i, j, k, size, cc, wsiz, nsiz, tsiz;
  assert(text);
  utext = (unsigned char *)est_uconv_in(text, strlen(text), &size);
  if(norm) est_normalize_text(utext, size, &size);
  est_canonicalize_text(utext, size, FALSE);
  words = cblistopen();
  for(i = 0; i < size; i += 2){
    cc = est_char_category(utext[i] * 0x100 + utext[i+1]);
    for(j = i + 2; j < size; j += 2){
      if(est_char_category(utext[j] * 0x100 + utext[j+1]) != cc) break;
    }
    switch(cc){
    case ESTDELIMCHR:
    case ESTWESTALPH:
      cblistpush(words, (char *)(utext + i), j - i);
      break;
    case ESTEASTALPH:
      for(k = i; k < j; k += 2){
        if(j - k >= 4){
          cblistpush(words, (char *)(utext + k), 4);
        } else {
          cblistpush(words, (char *)(utext + k), 2);
        }
      }
      break;
    default:
      break;
    }
    i = j - 2;
  }
  for(i = 0; i < CB_LISTNUM(words); i++){
    word = (unsigned char *)CB_LISTVAL2(words, i, &wsiz);
    if(est_char_category(word[0] * 0x100 + word[1]) == ESTEASTALPH && wsiz == 2 &&
       i < CB_LISTNUM(words) - 1){
      next = (unsigned char *)cblistval(words, i + 1, &nsiz);
      if(nsiz > 4) nsiz = 4;
      if(est_char_category(next[0] * 0x100 + next[1]) == ESTEASTALPH && nsiz > 2) nsiz = 2;
      CB_MALLOC(tmp, wsiz + nsiz + 1);
      memcpy(tmp, word, wsiz);
      memcpy(tmp + wsiz, next, nsiz);
      cblistover(words, i, tmp, wsiz + nsiz);
      free(tmp);
    }
  }
  for(i = 0; i < CB_LISTNUM(words); i++){
    word = (unsigned char *)CB_LISTVAL2(words, i, &wsiz);
    if(!tail && wsiz == 2 && i == CB_LISTNUM(words) - 1){
      if(est_char_category(word[0] * 0x100 + word[1]) == ESTEASTALPH) continue;
    }
    tmp = est_uconv_out((char *)word, wsiz, &tsiz);
    cblistpushbuf(list, tmp, tsiz);
  }
  cblistclose(words);
  free(utext);
}


/* Break a sentence of text and extract words using perfect N-gram analyzer. */
void est_break_text_perfng(const char *text, CBLIST *list, int norm, int tail){
  CBLIST *words;
  const unsigned char *word, *next;
  unsigned char *utext;
  char *tmp;
  int i, j, k, size, cc, wsiz, nsiz, tsiz;
  assert(text);
  utext = (unsigned char *)est_uconv_in(text, strlen(text), &size);
  if(norm) est_normalize_text(utext, size, &size);
  est_canonicalize_text(utext, size, FALSE);
  words = cblistopen();
  for(i = 0; i < size; i += 2){
    cc = est_char_category_perfng(utext[i] * 0x100 + utext[i+1]);
    for(j = i + 2; j < size; j += 2){
      if(est_char_category_perfng(utext[j] * 0x100 + utext[j+1]) != cc) break;
    }
    switch(cc){
    case ESTEASTALPH:
      for(k = i; k < j; k += 2){
        if(j - k >= 4){
          cblistpush(words, (char *)(utext + k), 4);
        } else {
          cblistpush(words, (char *)(utext + k), 2);
        }
      }
      break;
    default:
      break;
    }
    i = j - 2;
  }
  for(i = 0; i < CB_LISTNUM(words); i++){
    word = (unsigned char *)CB_LISTVAL2(words, i, &wsiz);
    if(est_char_category_perfng(word[0] * 0x100 + word[1]) == ESTEASTALPH && wsiz == 2 &&
       i < CB_LISTNUM(words) - 1){
      next = (unsigned char *)cblistval(words, i + 1, &nsiz);
      if(nsiz > 4) nsiz = 4;
      if(est_char_category_perfng(next[0] * 0x100 + next[1]) == ESTEASTALPH && nsiz > 2) nsiz = 2;
      CB_MALLOC(tmp, wsiz + nsiz + 1);
      memcpy(tmp, word, wsiz);
      memcpy(tmp + wsiz, next, nsiz);
      cblistover(words, i, tmp, wsiz + nsiz);
      free(tmp);
    }
  }
  for(i = 0; i < CB_LISTNUM(words); i++){
    word = (unsigned char *)CB_LISTVAL2(words, i, &wsiz);
    if(!tail && wsiz == 2 && i == CB_LISTNUM(words) - 1){
      if(est_char_category_perfng(word[0] * 0x100 + word[1]) == ESTEASTALPH) continue;
    }
    tmp = est_uconv_out((char *)word, wsiz, &tsiz);
    cblistpushbuf(list, tmp, tsiz);
  }
  cblistclose(words);
  free(utext);
}


/* Convert the character encoding of a string. */
char *est_iconv(const char *ptr, int size,
                const char *icode, const char *ocode, int *sp, int *mp){
  iconv_t ic;
  char *obuf, *wp, *rp;
  size_t isiz, osiz;
  int miss;
  assert(ptr && icode && ocode);
  if(size < 0) size = strlen(ptr);
  if(icode[0] == 'x' && icode[1] == '-'){
    if(!cbstricmp(icode, "x-sjis")){
      icode = "Shift_JIS";
    } else if(!cbstricmp(icode, "x-ujis")){
      icode = "EUC-JP";
    } else if(!cbstricmp(icode, "x-euc-jp")){
      icode = "EUC-JP";
    }
  } else if(icode[0] == 'w' || icode[0] == 'W'){
    if(!cbstricmp(icode, "windows-31j")){
      icode = "CP932";
    }
  }
  if(ocode[0] == 'x' && ocode[1] == '-'){
    if(!cbstricmp(ocode, "x-sjis")){
      ocode = "Shift_JIS";
    } else if(!cbstricmp(ocode, "x-ujis")){
      ocode = "EUC-JP";
    } else if(!cbstricmp(ocode, "x-euc-jp")){
      ocode = "EUC-JP";
    }
  } else if(ocode[0] == 'w' || ocode[0] == 'W'){
    if(!cbstricmp(ocode, "windows-31j")){
      ocode = "CP932";
    }
  }
  if((ic = iconv_open(ocode, icode)) == (iconv_t)-1) return NULL;
  isiz = size;
  osiz = isiz * 5;
  CB_MALLOC(obuf, osiz + 1);
  wp = obuf;
  rp = (char *)ptr;
  miss = 0;
  while(isiz > 0){
    if(iconv(ic, (void *)&rp, &isiz, &wp, &osiz) == -1){
      if(errno == EILSEQ && (*rp == 0x5c || *rp == 0x7e)){
        *wp = *rp;
        wp++;
        rp++;
        isiz--;
      } else if(errno == EILSEQ || errno == EINVAL){
        rp++;
        isiz--;
        miss++;
      } else {
        break;
      }
    }
  }
  *wp = '\0';
  if(sp) *sp = wp - obuf;
  if(mp) *mp = miss;
  if(iconv_close(ic) == -1){
    free(obuf);
    return NULL;
  }
  return obuf;
}


/* Detect the encoding of a string automatically. */
const char *est_enc_name(const char *ptr, int size, int plang){
  const char *hypo;
  int i, lim, miss, ascii, cr;
  assert(ptr);
  if(size < 0) size = strlen(ptr);
  if(size > ESTICCHECKSIZ) size = ESTICCHECKSIZ;
  if(size >= 2 && (!memcmp(ptr, "\xfe\xff", 2) || !memcmp(ptr, "\xff\xfe", 2))) return "UTF-16";
  ascii = TRUE;
  cr = FALSE;
  lim = size - 1;
  for(i = 0; i < lim; i += 2){
    if(ptr[i] == 0x0) return "UTF-16BE";
    if(ptr[i+1] == 0x0) return "UTF-16LE";
    if(ptr[i] < 0x0 || ptr[i] == 0x1b){
      ascii = FALSE;
    } else if(ptr[i] == 0xd){
      cr = TRUE;
    }
  }
  if(ascii) return "US-ASCII";
  switch(plang){
  case ESTLANGEN:
    if(est_enc_miss(ptr, size, "UTF-8", "UTF-16BE") < 1) return "UTF-8";
    return "ISO-8859-1";
  case ESTLANGJA:
    lim = size - 3;
    for(i = 0; i < lim; i++){
      if(ptr[i] == 0x1b){
        i++;
        if(ptr[i] == '(' && strchr("BJHI", ptr[i+1])) return "ISO-2022-JP";
        if(ptr[i] == '$' && strchr("@B(", ptr[i+1])) return "ISO-2022-JP";
      }
    }
    if(est_enc_miss(ptr, size, "UTF-8", "UTF-16BE") < 1) return "UTF-8";
    hypo = NULL;
    if(cr){
      if((miss = est_enc_miss(ptr, size, "Shift_JIS", "EUC-JP")) < 1) return "Shift_JIS";
      if(!hypo && miss / (double)size <= ESTICALLWRAT) hypo = "Shift_JIS";
      if((miss = est_enc_miss(ptr, size, "EUC-JP", "UTF-16BE")) < 1) return "EUC-JP";
      if(!hypo && miss / (double)size <= ESTICALLWRAT) hypo = "EUC-JP";
    } else {
      if((miss = est_enc_miss(ptr, size, "EUC-JP", "UTF-16BE")) < 1) return "EUC-JP";
      if(!hypo && miss / (double)size <= ESTICALLWRAT) hypo = "EUC-JP";
      if((miss = est_enc_miss(ptr, size, "Shift_JIS", "EUC-JP")) < 1) return "Shift_JIS";
      if(!hypo && miss / (double)size <= ESTICALLWRAT) hypo = "Shift_JIS";
    }
    if((miss = est_enc_miss(ptr, size, "UTF-8", "UTF-16BE")) < 1) return "UTF-8";
    if(!hypo && miss / (double)size <= ESTICALLWRAT) hypo = "UTF-8";
    if((miss = est_enc_miss(ptr, size, "CP932", "UTF-16BE")) < 1) return "CP932";
    if(!hypo && miss / (double)size <= ESTICALLWRAT) hypo = "CP932";
    return hypo ? hypo : "ISO-8859-1";
  case ESTLANGZH:
    if(est_enc_miss(ptr, size, "UTF-8", "UTF-16BE") < 1) return "UTF-8";
    if(est_enc_miss(ptr, size, "EUC-CN", "UTF-16BE") < 1) return "EUC-CN";
    if(est_enc_miss(ptr, size, "BIG5", "UTF-16BE") < 1) return "BIG5";
    return "ISO-8859-1";
  case ESTLANGKO:
    if(est_enc_miss(ptr, size, "UTF-8", "UTF-16BE") < 1) return "UTF-8";
    if(est_enc_miss(ptr, size, "EUC-KR", "UTF-16BE") < 1) return "EUC-KR";
    return "ISO-8859-1";
  default:
    break;
  }
  return "ISO-8859-1";
}


/* Convert a UTF-8 string into UTF-16BE. */
char *est_uconv_in(const char *ptr, int size, int *sp){
  const unsigned char *rp;
  char *rbuf, *wp;
  assert(ptr && size >= 0 && sp);
  rp = (unsigned char *)ptr;
  CB_MALLOC(rbuf, size * 2 + 1);
  wp = rbuf;
  while(rp < (unsigned char *)ptr + size){
    if(*rp < 0x7f){
      *(wp++) = 0x00;
      *(wp++) = *rp;
      rp += 1;
    } else if(*rp < 0xdf){
      if(rp >= (unsigned char *)ptr + size - 1) break;
      *(wp++) = (rp[0] & 0x1f) >> 2;
      *(wp++) = (rp[0] << 6) | (rp[1] & 0x3f);
      rp += 2;
    } else if(*rp < 0xf0){
      if(rp >= (unsigned char *)ptr + size - 2) break;
      *(wp++) = (rp[0] << 4) | ((rp[1] & 0x3f) >> 2);
      *(wp++) = (rp[1] << 6) | (rp[2] & 0x3f);
      rp += 3;
    } else if(*rp < 0xf8){
      if(rp >= (unsigned char *)ptr + size - 3) break;
      *(wp++) = 0x00;
      *(wp++) = '?';
      rp += 4;
    } else if(*rp < 0xfb){
      if(rp >= (unsigned char *)ptr + size - 4) break;
      *(wp++) = 0x00;
      *(wp++) = '?';
      rp += 5;
    } else if(*rp < 0xfd){
      if(rp >= (unsigned char *)ptr + size - 5) break;
      *(wp++) = 0x00;
      *(wp++) = '?';
      rp += 6;
    } else {
      break;
    }
  }
  *wp = '\0';
  *sp = wp - rbuf;
  return rbuf;
}


/* Convert a UTF-16BE string into UTF-8. */
char *est_uconv_out(const char *ptr, int size, int *sp){
  const unsigned char *rp;
  char *rbuf, *wp;
  int c;
  assert(ptr && size >= 0);
  if(size % 2 != 0) size--;
  rp = (unsigned char *)ptr;
  CB_MALLOC(rbuf, size * 2 + 1);
  wp = rbuf;
  while(rp < (unsigned char *)ptr + size){
    c = rp[0] * 0x100 + rp[1];
    if(c < 0x0080){
      *(wp++) = rp[1];
    } else if(c < 0x0900){
      *(wp++) = 0xc0 | (rp[0] << 2) | ((rp[1] >> 6) & 0x03);
      *(wp++) = 0x80 | (rp[1] & 0x3f);
    } else {
      *(wp++) = 0xe0 | ((rp[0] >> 4) & 0x0f);
      *(wp++) = 0x80 | ((rp[0] & 0x0f) << 2) | ((rp[1] >> 6) & 0x03);
      *(wp++) = 0x80 | (rp[1] & 0x3f);
    }
    rp += 2;
  }
  *wp = '\0';
  if(sp) *sp = wp - rbuf;
  return rbuf;
}


/* Compress a serial object with ZLIB. */
char *est_deflate(const char *ptr, int size, int *sp){
  z_stream zs;
  char *buf;
  unsigned char obuf[ESTIOBUFSIZ];
  int rv, asiz, bsiz, osiz;
  assert(ptr && sp);
  if(size < 0) size = strlen(ptr);
  if(!ESTUSEZLIB){
    if(sp) *sp = size;
    return cbmemdup(ptr, size);
  }
  zs.zalloc = Z_NULL;
  zs.zfree = Z_NULL;
  zs.opaque = Z_NULL;
  if(deflateInit2(&zs, 5, Z_DEFLATED, -15, 7, Z_DEFAULT_STRATEGY) != Z_OK) return NULL;
  asiz = ESTIOBUFSIZ;
  CB_MALLOC(buf, asiz);
  bsiz = 0;
  zs.next_in = (unsigned char *)ptr;
  zs.avail_in = size;
  zs.next_out = obuf;
  zs.avail_out = ESTIOBUFSIZ;
  while((rv = deflate(&zs, Z_FINISH)) == Z_OK){
    osiz = ESTIOBUFSIZ - zs.avail_out;
    if(bsiz + osiz > asiz){
      asiz = asiz * 2 + osiz;
      CB_REALLOC(buf, asiz);
    }
    memcpy(buf + bsiz, obuf, osiz);
    bsiz += osiz;
    zs.next_out = obuf;
    zs.avail_out = ESTIOBUFSIZ;
  }
  if(rv != Z_STREAM_END){
    free(buf);
    deflateEnd(&zs);
    return NULL;
  }
  osiz = ESTIOBUFSIZ - zs.avail_out;
  if(bsiz + osiz + 1 > asiz){
    asiz = asiz * 2 + osiz;
    CB_REALLOC(buf, asiz);
  }
  memcpy(buf + bsiz, obuf, osiz);
  bsiz += osiz;
  buf[bsiz++] = '\0';
  *sp = bsiz;
  deflateEnd(&zs);
  return buf;
}


/* Decompress a serial object compressed with ZLIB. */
char *est_inflate(const char *ptr, int size, int *sp){
  z_stream zs;
  char *buf;
  unsigned char obuf[ESTIOBUFSIZ];
  int rv, asiz, bsiz, osiz;
  assert(ptr && size >= 0 && sp);
  if(!ESTUSEZLIB){
    *sp = size;
    return cbmemdup(ptr, size);
  }
  zs.zalloc = Z_NULL;
  zs.zfree = Z_NULL;
  zs.opaque = Z_NULL;
  if(inflateInit2(&zs, -15) != Z_OK) return NULL;
  asiz = ESTIOBUFSIZ;
  CB_MALLOC(buf, asiz);
  bsiz = 0;
  zs.next_in = (unsigned char *)ptr;
  zs.avail_in = size;
  zs.next_out = obuf;
  zs.avail_out = ESTIOBUFSIZ;
  while((rv = inflate(&zs, Z_NO_FLUSH)) == Z_OK){
    osiz = ESTIOBUFSIZ - zs.avail_out;
    if(bsiz + osiz >= asiz){
      asiz = asiz * 2 + osiz;
      CB_REALLOC(buf, asiz);
    }
    memcpy(buf + bsiz, obuf, osiz);
    bsiz += osiz;
    zs.next_out = obuf;
    zs.avail_out = ESTIOBUFSIZ;
  }
  if(rv != Z_STREAM_END){
    free(buf);
    inflateEnd(&zs);
    return NULL;
  }
  osiz = ESTIOBUFSIZ - zs.avail_out;
  if(bsiz + osiz >= asiz){
    asiz = asiz * 2 + osiz;
    CB_REALLOC(buf, asiz);
  }
  memcpy(buf + bsiz, obuf, osiz);
  bsiz += osiz;
  buf[bsiz] = '\0';
  if(sp) *sp = bsiz;
  inflateEnd(&zs);
  return buf;
}


/* Get the border string for draft data of documents. */
const char *est_border_str(void){
  static int first = TRUE;
  static char border[ESTPATHBUFSIZ];
  int t, p;
  if(first){
    t = (int)(time(NULL) + est_random() * INT_MAX);
    p = (int)(getpid() + est_random() * INT_MAX);
    sprintf(border, "--------[%08X%08X]--------",
            dpouterhash((char *)&t, sizeof(int)), dpouterhash((char *)&p, sizeof(int)));
    first = FALSE;
  }
  return border;
}


/* Get the real random number. */
double est_random(void){
  static int first = TRUE;
  int num;
  if(first && !est_random_ifp){
    if((est_random_ifp = fopen("/dev/urandom", "rb")) != NULL){
      atexit(est_random_fclose);
    } else {
      srand(getpid());
    }
    first = FALSE;
  }
  if(est_random_ifp){
    fread(&num, sizeof(int), 1, est_random_ifp);
    return (num & 0x7fffffff) / (double)0x7fffffff;
  }
  return rand() / (double)RAND_MAX;
}


/* Get the random number in normal distribution. */
double est_random_nd(void){
  double d;
  d = (sqrt(-2 * log(1.0 - est_random())) * cos(3.1415926535 * 2 * est_random()) + 6.0) / 12.0;
  if(d > 1.0) d = 1.0;
  if(d < 0.0) d = 0.0;
  return d;
}


/* Get an MD5 hash string of a key string. */
char *est_make_crypt(const char *key){
  md5_state_t ms;
  char digest[32], str[64], *wp;
  int i;
  assert(key);
  md5_init(&ms);
  md5_append(&ms, (md5_byte_t *)key, strlen(key));
  md5_finish(&ms, (md5_byte_t *)digest);
  wp = str;
  for(i = 0; i < 16; i++){
    wp += sprintf(wp, "%02x", ((unsigned char *)digest)[i]);
  }
  return cbmemdup(str, -1);
}


/* Check whether a key matches an MD5 hash string. */
int est_match_crypt(const char *key, const char *hash){
  char *khash;
  int rv;
  assert(key && hash);
  khash = est_make_crypt(key);
  rv = !strcmp(khash, hash);
  free(khash);
  return rv;
}


/* Create a regular expression object. */
void *est_regex_new(const char *str){
  regex_t regex;
  assert(str);
  if(regcomp(&regex, str, REG_EXTENDED | REG_NOSUB) != 0) return NULL;
  return cbmemdup((char *)&regex, sizeof(regex_t));
}


/* Delete a regular expression object. */
void est_regex_delete(void *regex){
  assert(regex);
  regfree(regex);
  free(regex);
}


/* Check whether a regular expression matches a string. */
int est_regex_match(const void *regex, const char *str){
  assert(regex && str);
  return regexec(regex, str, 0, NULL, 0) == 0;
}


/* Replace each substring matching a regular expression string. */
char *est_regex_replace(const char *str, const char *bef, const char *aft){
  regex_t regex;
  regmatch_t subs[256];
  CBDATUM *datum;
  const char *sp;
  int first;
  assert(str && bef && aft);
  if(bef[0] == '\0' || regcomp(&regex, bef, REG_EXTENDED) != 0) return cbmemdup(str, -1);
  if(regexec(&regex, str, ESTREGSUBMAX, subs, 0) != 0){
    regfree(&regex);
    return cbmemdup(str, -1);
  }
  sp = str;
  datum = cbdatumopen("", 0);
  first = TRUE;
  while(sp[0] != '\0' && regexec(&regex, sp, 1, subs, first ? 0 : REG_NOTBOL) == 0){
    first = FALSE;
    if(subs[0].rm_so == -1) break;
    cbdatumcat(datum, sp, subs[0].rm_so);
    cbdatumcat(datum, aft, -1);
    sp += subs[0].rm_eo;
    if(subs[0].rm_eo < 1) break;
  }
  cbdatumcat(datum, sp, -1);
  regfree(&regex);
  return cbdatumtomalloc(datum, NULL);
}


/* Set the ID number of a document object. */
void est_doc_set_id(ESTDOC *doc, int id){
  assert(doc);
  doc->id = id;
}


/* Get the hidden texts of a document object. */
const char *est_doc_hidden_texts(ESTDOC *doc){
  const char *rv;
  assert(doc);
  rv = doc->attrs ? cbmapget(doc->attrs, "", 0, NULL) : NULL;
  return rv ? rv : "";
}


/* Check whether a docuemnt object is empty. */
int est_doc_is_empty(ESTDOC *doc){
  assert(doc);
  if((!doc->dtexts || cblistnum(doc->dtexts) < 1) &&
     (!doc->attrs || !cbmapget(doc->attrs, "", 0, NULL))) return TRUE;
  return FALSE;
}


/* Get the phrase of a condition object. */
const char *est_cond_phrase(ESTCOND *cond){
  assert(cond);
  return cond->phrase;
}


/* Get a list object of attribute expressions of a condition object. */
const CBLIST *est_cond_attrs(ESTCOND *cond){
  assert(cond);
  return cond->attrs;
}


/* Get the order expression of a condition object. */
const char *est_cond_order(ESTCOND *cond){
  assert(cond);
  return cond->order;
}


/* Get the maximum number of retrieval of a condition object. */
int est_cond_max(ESTCOND *cond){
  assert(cond);
  return cond->max;
}


/* Get the options of a condition object. */
int est_cond_options(ESTCOND *cond){
  assert(cond);
  return cond->opts;
}


/* Get the score of a document corresponding to a condition object. */
int est_cond_score(ESTCOND *cond, int index){
  assert(cond);
  if(!cond->scores || index < 0 || index >= cond->snum) return -1;
  return cond->scores[index];
}


/* Get an array of ID numbers of eclipsed docuemnts of a document in a condition object. */
const int *est_cond_shadows(ESTCOND *cond, int id, int *np){
  const char *vbuf;
  int vsiz;
  assert(cond && id > 0 && np);
  if(!cond->shadows || !(vbuf = cbmapget(cond->shadows, (char *)&id, sizeof(int), &vsiz))){
    *np = 0;
    return (int *)"";
  }
  *np = vsiz / sizeof(int);
  return (int *)vbuf;
}


/* Set the error code of a database. */
void est_db_set_ecode(ESTDB *db, int ecode){
  assert(db);
  db->ecode = ecode;
}


/* Set the entity data of a document in a database. */
int est_db_set_doc_entity(ESTDB *db, int id, const char *ptr, int size){
  int err;
  assert(db && id > 0);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  err = FALSE;
  if(ptr){
    if(!crputlob(db->textdb, (char *)&id, sizeof(int), ptr, size, CR_DOVER)){
      db->ecode = ESTEDB;
      err = TRUE;
    }
  } else {
    if(!croutlob(db->textdb, (char *)&id, sizeof(int))){
      db->ecode = dpecode == DP_ENOITEM ? ESTENOITEM : ESTEDB;
      err = TRUE;
    }
  }
  return err ? FALSE : TRUE;
}


/* Get the entity data of a document in a database. */
char *est_db_get_doc_entity(ESTDB *db, int id, int *sp){
  char *ptr;
  assert(db && id > 0 && sp);
  if(!(ptr = crgetlob(db->textdb, (char *)&id, sizeof(int), 0, -1, sp))){
    db->ecode = dpecode == DP_ENOITEM ? ESTENOITEM : ESTEDB;
    return NULL;
  }
  return ptr;
}


/* Add a piece of meta data to a database. */
void est_db_add_meta(ESTDB *db, const char *name, const char *value){
  assert(db && name);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return;
  }
  if(!db->metacc) est_db_prepare_meta(db);
  if(value){
    cbmapput(db->metacc, name, -1, value, -1, TRUE);
  } else {
    cbmapout(db->metacc, name, -1);
  }
}


/* Get a list of names of meta data of a database. */
CBLIST *est_db_meta_names(ESTDB *db){
  assert(db);
  if(!db->metacc) est_db_prepare_meta(db);
  return cbmapkeys(db->metacc);
}


/* Get the value of a piece of meta data of a database. */
char *est_db_meta(ESTDB *db, const char *name){
  const char *vbuf;
  int vsiz;
  assert(db && name);
  if(!db->metacc) est_db_prepare_meta(db);
  if(!(vbuf = cbmapget(db->metacc, name, -1, &vsiz))) return NULL;
  return cbmemdup(vbuf, vsiz);
}


/* Extract keywords of a document object. */
CBMAP *est_db_etch_doc(ESTDB *db, ESTDOC *doc, int max){
  ESTKEYSC *scores;
  CBMAP *keys, *umap;
  CBLIST *words;
  const char *text, *word, *vbuf;
  const unsigned char *uword;
  char numbuf[ESTNUMBUFSIZ];
  int i, wsiz, num, smax, snum, vsiz;
  assert(doc && max >= 0);
  if(!doc->dtexts) return cbmapopenex(1);
  keys = cbmapopenex(max * 1.5);
  words = cblistopen();
  for(i = -1; i < CB_LISTNUM(doc->dtexts); i++){
    if(i < 0){
      if(!doc->attrs || !(text = cbmapget(doc->attrs, "", 0, NULL))) continue;
    } else {
      text = CB_LISTVAL(doc->dtexts, i, NULL);
    }
    if(db){
      switch(db->amode){
      case ESTAMPERFNG:
        est_break_text_perfng(text, words, FALSE, TRUE);
        break;
      default:
        est_break_text(text, words, FALSE, TRUE);
        break;
      }
    } else {
      est_break_text(text, words, FALSE, TRUE);
    }
  }
  umap = cbmapopenex(CB_LISTNUM(words) + 1);
  for(i = 0; i < CB_LISTNUM(words); i++){
    word = CB_LISTVAL2(words, i, &wsiz);
    if(wsiz > ESTWORDMAXLEN) continue;
    num = (vbuf = cbmapget(umap, word, wsiz, NULL)) ? *(int *)vbuf + 1 : 1;
    cbmapput(umap, word, wsiz, (char *)&num, sizeof(int), TRUE);
  }
  CB_MALLOC(scores, cbmaprnum(umap) * sizeof(ESTKEYSC) + 1);
  snum = 0;
  cbmapiterinit(umap);
  while((uword = (unsigned char *)cbmapiternext(umap, &wsiz)) != NULL){
    scores[snum].word = (char *)uword;
    scores[snum].wsiz = wsiz;
    scores[snum].pt = (vbuf = cbmapget(umap, (char *)uword, wsiz, NULL)) ? *(int *)vbuf : 0;
    if(uword[0] >= 0xe3){
      if(wsiz <= 3){
        scores[snum].pt /= 2;
      } else {
        if(uword[0] == 0xe3 && (uword[1] == 0x80 || uword[1] == 0x81 ||
                                (uword[1] == 0x82 && uword[2] <= 0x9f))) scores[snum].pt /= 2;
        if(uword[3] == 0xe3 && (uword[4] == 0x80 || uword[4] == 0x81 ||
                                (uword[4] == 0x82 && uword[5] <= 0x9f))) scores[snum].pt /= 2;
      }
    } else if(wsiz <= 1){
      scores[snum].pt /= 3;
    } else if(wsiz <= 2){
      scores[snum].pt /= 2;
    }
    snum++;
  }
  qsort(scores, snum, sizeof(ESTKEYSC), est_keysc_compare);
  smax = max * (db ? ESTKEYSCALW : 1) + 1;
  snum = snum > smax ? smax : snum;
  if(db){
    for(i = 0; i < snum; i++){
      if((vbuf = cbmapget(db->keycc, scores[i].word, scores[i].wsiz, NULL)) != NULL){
        cbmapmove(db->keycc, scores[i].word, scores[i].wsiz, FALSE);
        vsiz = *(int*)vbuf;
      } else {
        if(db->dfdb){
          if((vsiz = dpgetwb(db->dfdb, scores[i].word, scores[i].wsiz,
                             0, ESTNUMBUFSIZ - 1, numbuf)) > 0){
            numbuf[vsiz] = '\0';
            vsiz = atoi(numbuf);
          } else {
            vsiz = 0;
          }
        } else {
          vsiz = est_idx_vsiz(db->idxdb, scores[i].word, scores[i].wsiz);
          if(cbmapget(db->idxcc, scores[i].word, scores[i].wsiz, &num)) vsiz += num;
        }
        cbmapput(db->keycc, scores[i].word, scores[i].wsiz, (char *)&vsiz, sizeof(int), FALSE);
      }
      scores[i].pt *= 400000.0 / (vsiz + 64);
    }
    if(db->kcmnum >= 0 && cbmaprnum(db->keycc) > db->kcmnum){
      num = db->kcmnum * 0.1 + 1;
      cbmapiterinit(db->keycc);
      for(i = 0; i < num && (word = cbmapiternext(db->keycc, &wsiz)) != NULL; i++){
        cbmapout(db->keycc, word, wsiz);
      }
    }
    qsort(scores, snum, sizeof(ESTKEYSC), est_keysc_compare);
  }
  for(i = 0; i < snum &&  i < max; i++){
    vsiz = sprintf(numbuf, "%d", scores[i].pt);
    cbmapput(keys, scores[i].word, scores[i].wsiz, numbuf, vsiz, FALSE);
  }
  free(scores);
  cbmapclose(umap);
  cblistclose(words);
  return keys;
}


/* Retrieve a map object of keywords. */
int est_db_put_keywords(ESTDB *db, int id, CBMAP *kwords){
  char *mbuf;
  int msiz, err;
  assert(db && id > 0 && kwords);
  if(!dpwritable(db->metadb)){
    db->ecode = ESTEACCES;
    return FALSE;
  }
  mbuf = cbmapdump(kwords, &msiz);
  err = FALSE;
  if(!crput(db->kwddb, (char *)&id, sizeof(int), mbuf, msiz, CR_DOVER)){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
    err = TRUE;
  }
  free(mbuf);
  return err ? FALSE : TRUE;
}


/* Retrieve a map object of keywords. */
CBMAP *est_db_get_keywords(ESTDB *db, int id){
  CBMAP *kwords;
  const char *cbuf;
  char *mbuf;
  int i, csiz, msiz, num;
  assert(db && id > 0);
  if((cbuf = cbmapget(db->veccc, (char *)&id, sizeof(int), &csiz)) != NULL){
    cbmapmove(db->veccc, (char *)&id, sizeof(int), FALSE);
    return cbmapload(cbuf, csiz);
  }
  if(!(mbuf = crget(db->kwddb, (char *)&id, sizeof(int), 0, -1, &msiz))) return NULL;
  kwords = cbmapload(mbuf, msiz);
  cbmapputvbuf(db->veccc, (char *)&id, sizeof(int), mbuf, msiz);
  if(cbmaprnum(db->veccc) > db->vcmnum){
    num = cbmaprnum(db->veccc) * 0.1 + 1;
    cbmapiterinit(db->veccc);
    for(i = 0; i < num && (cbuf = cbmapiternext(db->veccc, NULL)) != NULL; i++){
      cbmapout(db->veccc, cbuf, sizeof(int));
    }
  }
  return kwords;
}


/* Mesure the total size of each inner records of a stored document. */
int est_db_measure_doc(ESTDB *db, int id, int parts){
  int sum, num;
  assert(db && id > 0);
  sum = 0;
  if((parts & ESTMDATTR) && (num = crvsiz(db->attrdb, (char *)&id, sizeof(int))) > 0) sum += num;
  if((parts & ESTMDTEXT) && (num = crvsiz(db->textdb, (char *)&id, sizeof(int))) > 0) sum += num;
  if((parts & ESTMDKWD) && (num = crvsiz(db->kwddb, (char *)&id, sizeof(int))) > 0) sum += num;
  return sum;
}


/* Initialize the iterator of a database. */
int est_db_iter_init(ESTDB *db, const char *prev){
  char *vbuf;
  assert(db);
  if(prev){
    if(!vlcurjump(db->listdb, prev, -1, VL_JFORWARD)) return dpecode == DP_ENOITEM;
    if((vbuf = vlcurkey(db->listdb, NULL)) != NULL){
      if(strcmp(prev, vbuf) >= 0) vlcurnext(db->listdb);
      free(vbuf);
    }
    return TRUE;
  }
  return vlcurfirst(db->listdb) || dpecode == DP_ENOITEM;
}


/* Get the next ID of the iterator of a database. */
int est_db_iter_next(ESTDB *db){
  char *vbuf;
  int id;
  assert(db);
  if(!(vbuf = vlcurval(db->listdb, NULL))){
    if(dpecode == DP_ENOITEM){
      db->ecode = ESTENOITEM;
      return 0;
    } else {
      db->ecode = ESTEDB;
      db->fatal = TRUE;
      return -1;
    }
  }
  id = atoi(vbuf);
  free(vbuf);
  vlcurnext(db->listdb);
  return id;
}


/* Initialize the word iterator of a database. */
int est_db_word_iter_init(ESTDB *db){
  assert(db);
  return vlcurfirst(db->fwmdb) || dpecode == DP_ENOITEM;
}


/* Get the next word of the word iterator of a database. */
char *est_db_word_iter_next(ESTDB *db){
  char *word;
  assert(db);
  if(!(word = vlcurkey(db->fwmdb, NULL))) return NULL;
  vlcurnext(db->fwmdb);
  return word;
}


/* Get the size of the record of a word. */
int est_db_word_rec_size(ESTDB *db, const char *word){
  assert(db && word);
  return est_idx_vsiz(db->idxdb, word, strlen(word));
}


/* Get the number of records in the cache memory of a database. */
int est_db_cache_num(ESTDB *db){
  assert(db);
  return cbmaprnum(db->idxcc);
}


/* Get the size of used cache region. */
int est_db_used_cache_size(ESTDB *db){
  assert(db);
  return (db->icsiz + cbmaprnum(db->idxcc) * (sizeof(CBMAPDATUM) + ESTWORDAVGLEN)) * ESTMEMIRATIO;
}


/* Set the special cache for narrowing and sorting with document attributes. */
void est_db_set_special_cache(ESTDB *db, const char *name, int num){
  assert(db && name && num >= 0);
  if(db->spacc){
    free(db->scname);
    cbmapclose(db->spacc);
  }
  db->spacc = cbmapopenex(num + 1);
  db->scmnum = num;
  db->scname = cbmemdup(name, -1);
}


/* Set the callback function for database events. */
void est_db_set_informer(ESTDB *db, void (*func)(const char *)){
  assert(db && func);
  db->cbinfo = func;
  est_db_inform(db, "status");
}


/* Fill the cache for keys for TF-IDF. */
void est_db_fill_key_cache(ESTDB *db){
  char *kbuf, *msg;
  int i, ksiz, vsiz;
  assert(db);
  vlcurfirst(db->fwmdb);
  for(i = 0; (kbuf = vlcurkey(db->fwmdb, &ksiz)) != NULL; i++){
    vsiz = est_idx_vsiz(db->idxdb, kbuf, ksiz);
    cbmapput(db->keycc, kbuf, ksiz, (char *)&vsiz, sizeof(int), TRUE);
    free(kbuf);
    vlcurnext(db->fwmdb);
    if(i % ESTCCCBFREQ == 0){
      msg = cbsprintf("filling the key cache for TF-IDF (%d)", i + 1);
      est_db_inform(db, msg);
      free(msg);
    }
  }
  db->kcmnum = -1;
}


/* Set the database of document frequency. */
void est_db_set_dfdb(ESTDB *db, DEPOT *dfdb){
  assert(db);
  db->dfdb = dfdb;
}


/* Clear the result cache. */
void est_db_refresh_rescc(ESTDB *db){
  ESTSCORE sc;
  const char *word;
  int size;
  assert(db);
  sc.id = -1;
  sc.score = 0;
  sc.value = NULL;
  cbmapiterinit(db->rescc);
  while((word = cbmapiternext(db->rescc, &size)) != NULL){
    cbmapput(db->rescc, word, size, (char *)&sc, sizeof(ESTSCORE), TRUE);
  }
}


/* Charge the result cache. */
void est_db_charge_rescc(ESTDB *db, int max){
  CBLIST *words;
  const char *word, *vbuf;
  int i, num, size, vsiz;
  assert(db);
  if(max < 0) max = INT_MAX;
  words = cblistopen();
  cbmapiterinit(db->rescc);
  while((word = cbmapiternext(db->rescc, &size)) != NULL){
    vbuf = cbmapget(db->rescc, word, size, &vsiz);
    if(vsiz == sizeof(ESTSCORE) && ((ESTSCORE *)vbuf)->id == -1) cblistpush(words, word, size);
  }
  num = CB_LISTNUM(words);
  for(i = 0; i < max && i < num; i++){
    word = cblistval(words, num - i - 1, &size);
    free(est_search_union(db, word, 1, &size, NULL, TRUE));
  }
  cblistclose(words);
}


/* Get list of words in the result cache. */
CBLIST *est_db_list_rescc(ESTDB *db){
  CBLIST *words;
  const char *word;
  int size;
  assert(db);
  words = cblistopen();
  cbmapiterinit(db->rescc);
  while((word = cbmapiternext(db->rescc, &size)) != NULL){
    cblistunshift(words, word, size);
  }
  return words;
}


/* Interrupt long time processing. */
void est_db_interrupt(ESTDB *db){
  assert(db);
  db->intflag = TRUE;
}


/* Extract words for snippet from hints of search. */
CBLIST *est_hints_to_words(CBMAP *hints){
  CBLIST *words;
  const char *kbuf;
  int ksiz;
  assert(hints);
  words = cblistopen();
  cbmapiterinit(hints);
  while((kbuf = cbmapiternext(hints, &ksiz)) != NULL){
    if(ksiz < 1 || atoi(cbmapget(hints, kbuf, ksiz, NULL)) < 1) continue;
    cblistpush(words, kbuf, ksiz);
  }
  return words;
}


/* Make a directory. */
int est_mkdir(const char *path){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  return mkdir(path) == 0 ? TRUE : FALSE;
#else
  assert(path);
  return mkdir(path, ESTDIRMODE) == 0 ? TRUE : FALSE;
#endif
}


/* Remove a directory and its contents recursively. */
int est_rmdir_rec(const char *path){
  CBLIST *files;
  const char *file;
  char pbuf[ESTPATHBUFSIZ];
  int i;
  assert(path);
  if((files = cbdirlist(path)) != NULL){
    for(i = 0; i < CB_LISTNUM(files); i++){
      file = CB_LISTVAL(files, i, NULL);
      if(!strcmp(file, ESTCDIRSTR) || !strcmp(file, ESTPDIRSTR)) continue;
      sprintf(pbuf, "%s%c%s", path, ESTPATHCHR, file);
      if(unlink(pbuf) == -1) est_rmdir_rec(pbuf);
    }
    cblistclose(files);
  }
  return rmdir(path) == 0;
}


/* Get the canonicalized absolute pathname of a file. */
char *est_realpath(const char *path){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  char pbuf[ESTPATHBUFSIZ*2], *p;
  if(GetFullPathName(path, ESTPATHBUFSIZ, pbuf, &p) == 0){
    if((((path[0] >= 'A' && path[0] <= 'Z') || (path[0] >= 'a' && path[0] <= 'z')) &&
        path[1] == ':' && path[2] == ESTPATHCHR) || path[0] == ESTPATHCHR ||
       GetFullPathName(ESTCDIRSTR, ESTPATHBUFSIZ, pbuf, &p) == 0){
      sprintf(pbuf, "%s", path);
    } else {
      sprintf(pbuf + strlen(pbuf), "%c%s", ESTPATHCHR, path);
    }
  }
  return cbmemdup(pbuf, -1);
#else
  char pbuf[ESTPATHBUFSIZ*2];
  assert(path);
  if(!realpath(path, pbuf)){
    if(path[0] == ESTPATHCHR || !realpath(ESTCDIRSTR, pbuf)){
      sprintf(pbuf, "%s", path);
    } else {
      sprintf(pbuf + strlen(pbuf), "%c%s", ESTPATHCHR, path);
    }
  }
  return cbmemdup(pbuf, -1);
#endif
}


/* Get the time of day in milliseconds. */
double est_gettimeofday(void){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  SYSTEMTIME st;
  struct tm ts;
  GetLocalTime(&st);
  memset(&ts, 0, sizeof(ts));
  ts.tm_year = st.wYear - 1900;
  ts.tm_mon = st.wMonth - 1;
  ts.tm_mday = st.wDay;
  ts.tm_hour = st.wHour;
  ts.tm_min = st.wMinute;
  ts.tm_sec = st.wSecond;
  return (double)mktime(&ts) * 1000 + (double)st.wMilliseconds;
#else
  struct timeval tv;
  struct timezone tz;
  if(gettimeofday(&tv, &tz) == -1) return 0.0;
  return (double)tv.tv_sec * 1000 + (double)tv.tv_usec / 1000;
#endif
}


/* Suspend execution for microsecond intervals. */
void est_usleep(unsigned long usec){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  Sleep(usec / 1000);
#else
  usleep(usec);
#endif
}


/* Set a signal handler. */
void est_signal(int signum, void (*sighandler)(int)){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  static int first = TRUE;
  int i;
  assert(signum >= 0 && sighandler);
  if(first){
    for(i = 1; i < ESTSIGNUM; i++){
      est_signal_handlers[i] = NULL;
    }
    SetConsoleCtrlHandler((PHANDLER_ROUTINE)est_signal_dispatch, TRUE);
    first = FALSE;
  }
  if(signum >= ESTSIGNUM) return;
  if(sighandler == SIG_IGN){
    signal(signum, SIG_IGN);
  } else if(sighandler == SIG_DFL){
    signal(signum, SIG_DFL);
  } else {
    signal(signum, (void (*)(int))est_signal_dispatch);
    est_signal_handlers[signum] = sighandler;
  }
#else
  static int first = TRUE;
  struct sigaction act;
  int i;
  assert(signum >= 0 && sighandler);
  if(first){
    for(i = 1; i < ESTSIGNUM; i++){
      est_signal_handlers[i] = NULL;
    }
    first = FALSE;
  }
  if(signum >= ESTSIGNUM) return;
  memset(&act, 0, sizeof(act));
  if(sighandler == SIG_IGN){
    act.sa_handler = SIG_IGN;
  } else if(sighandler == SIG_DFL){
    act.sa_handler = SIG_DFL;
  } else {
    act.sa_handler = (void (*)(int))est_signal_dispatch;
    est_signal_handlers[signum] = sighandler;
  }
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  sigaction(signum, &act, NULL);
#endif
}


/* Send a signal to a process. */
int est_kill(int pid, int sig){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  return FALSE;
#else
  return kill(pid, sig) == 0;
#endif
}


/* get the media type of an extention */
const char *est_ext_type(const char *ext){
  static const char *list[] = {
    ".txt", "text/plain", ".txt.en", "text/plain",
    ".txt.ja", "text/plain", ".asc", "text/plain",
    ".in", "text/plain", ".c", "text/plain",
    ".h", "text/plain", ".cc", "text/plain",
    ".java", "text/plain", ".sh", "text/plain",
    ".pl", "text/plain", ".py", "text/plain",
    ".rb", "text/plain", ".idl", "text/plain",
    ".csv", "text/plain", ".log", "text/plain",
    ".conf", "text/plain", ".rc", "text/plain",
    ".ini", "text/plain", ".html", "text/html",
    ".htm", "text/html", ".xhtml", "text/html",
    ".xht", "text/html", ".css", "text/css",
    ".js", "text/javascript", ".tsv", "text/tab-separated-values",
    ".eml", "message/rfc822", ".mime", "message/rfc822",
    ".mht", "message/rfc822", ".mhtml", "message/rfc822",
    ".sgml", "application/sgml", ".sgm", "application/sgml",
    ".xml", "application/xml", ".xsl", "application/xml",
    ".xslt", "application/xslt+xml", ".xhtml", "application/xhtml+xml",
    ".xht", "application/xhtml+xml", ".rdf", "application/rdf+xml",
    ".rss", "application/rss+xml", ".dtd", "application/xml-dtd",
    ".rtf", "application/rtf", ".pdf", "application/pdf",
    ".ps", "application/postscript", ".eps", "application/postscript",
    ".doc", "application/msword", ".xls", "application/vnd.ms-excel",
    ".ppt", "application/vnd.ms-powerpoint", ".xdw", "application/vnd.fujixerox.docuworks",
    ".swf", "application/x-shockwave-flash", ".zip", "application/zip",
    ".tar", "application/x-tar", ".gz", "application/x-gzip",
    ".bz2", "application/octet-stream", ".z", "application/octet-stream",
    ".lha", "application/octet-stream", ".lzh", "application/octet-stream",
    ".cab", "application/octet-stream", ".rar", "application/octet-stream",
    ".sit", "application/octet-stream", ".bin", "application/octet-stream",
    ".o", "application/octet-stream", ".so", "application/octet-stream",
    ".exe", "application/octet-stream", ".dll", "application/octet-stream",
    ".class", "application/octet-stream", ".png", "image/png",
    ".gif", "image/gif", ".jpg", "image/jpeg",
    ".jpeg", "image/jpeg", ".tif", "image/tiff",
    ".tiff", "image/tiff", ".bmp", "image/bmp",
    ".au", "audio/basic", ".snd", "audio/basic",
    ".mid", "audio/midi", ".midi", "audio/midi",
    ".mp2", "audio/mpeg", ".mp3", "audio/mpeg",
    ".wav", "audio/x-wav", ".mpg", "video/mpeg",
    ".mpeg", "video/mpeg", ".qt", "video/quicktime",
    ".mov", "video/quicktime", ".avi", "video/x-msvideo",
    NULL
  };
  int i;
  assert(ext);
  for(i = 0; list[i]; i++){
    if(!cbstricmp(ext, list[i])) return list[i+1];
  }
  return "application/octet-stream";
}


/* Set a seed vector from a map object. */
void est_vector_set_seed(CBMAP *svmap, int *svec, int vnum){
  const char *kbuf;
  int i, ksiz;
  assert(svmap && svec && vnum > 0);
  cbmapiterinit(svmap);
  for(i = 0; i < vnum; i++){
    if((kbuf = cbmapiternext(svmap, &ksiz)) != NULL){
      svec[i] = atoi(cbmapget(svmap, kbuf, ksiz, NULL));
    } else {
      svec[i] = 0;
    }
  }
}


/* Set a target vector from a map object. */
void est_vector_set_target(CBMAP *svmap, CBMAP *tvmap, int *tvec, int vnum){
  const char *kbuf, *vbuf;
  int i, ksiz;
  assert(svmap && tvmap && tvec && vnum > 0);
  cbmapiterinit(svmap);
  for(i = 0; i < vnum; i++){
    if((kbuf = cbmapiternext(svmap, &ksiz)) != NULL){
      vbuf = cbmapget(tvmap, kbuf, ksiz, NULL);
      tvec[i] = vbuf ? atoi(vbuf) : 0;
    } else {
      tvec[i] = 0;
    }
  }
}


/* Get the cosine of the angle of two vectors. */
double est_vector_cosine(const int *avec, const int *bvec, int vnum){
  int i;
  double iprod, aabs, babs;
  assert(avec && bvec && vnum >= 0);
  iprod = 0.0;
  for(i = 0; i < vnum; i++){
    iprod += (double)avec[i] * (double)bvec[i];
  }
  aabs = 0.0;
  for(i = 0; i < vnum; i++){
    aabs += (double)avec[i] * (double)avec[i];
  }
  aabs = sqrt(aabs);
  babs = 0.0;
  for(i = 0; i < vnum; i++){
    babs += (double)bvec[i] * (double)bvec[i];
  }
  babs = sqrt(babs);
  if(iprod <= 0.0 || aabs < 1.0 || babs < 1.0) return 0.0;
  return iprod / (aabs * babs);
}



/*************************************************************************************************
 * private objects
 *************************************************************************************************/


/* Count the number of missing characters when converting.
   `ptr' specifies the pointer to a region.
   `size' specifies the size of the region.
   `icode' specifies the name of encoding of the input string.
   `ocode' specifies the name of encoding of the output string.
   The return value is the number of missing characters. */
static int est_enc_miss(const char *ptr, int size, const char *icode, const char *ocode){
  iconv_t ic;
  char obuf[ESTICCHECKSIZ], *wp, *rp;
  size_t isiz, osiz;
  int miss;
  assert(ptr && size >= 0 && icode && ocode);
  isiz = size;
  if((ic = iconv_open(ocode, icode)) == (iconv_t)-1) return ESTICMISSMAX;
  miss = 0;
  rp = (char *)ptr;
  while(isiz > 0){
    osiz = ESTICCHECKSIZ;
    wp = obuf;
    if(iconv(ic, (void *)&rp, &isiz, &wp, &osiz) == -1){
      if(errno == EILSEQ || errno == EINVAL){
        rp++;
        isiz--;
        miss++;
        if(miss >= ESTICMISSMAX) break;
      } else {
        break;
      }
    }
  }
  if(iconv_close(ic) == -1) return ESTICMISSMAX;
  return miss;
}


/* Normalize a text.
   `utext' specifies a text whose encoding is UTF-16BE.
   `size' specifies the size of the text.
   `sp' specifies the pointer to a variable to which the size of the result is assigned. */
static void est_normalize_text(unsigned char *utext, int size, int *sp){
  int i, wi, b1, b2;
  assert(utext && size >= 0 && sp);
  wi = 0;
  for(i = 0; i < size - 1; i += 2){
    b1 = utext[i];
    b2 = utext[i+1];
    if(b1 == 0x0 && (b2 <= 0x8 || (b2 >= 0x0e && b2 <= 0x1f))){
      /* control characters */
      utext[wi] = 0x0;
      utext[wi+1] = 0x20;
    } else if(b1 == 0x0 && b2 == 0xa0){
      /* no-break space */
      utext[wi] = 0x0;
      utext[wi+1] = 0x20;
    } else if(b1 == 0x20 && b2 == 0x2){
      /* en space */
      utext[wi] = 0x0;
      utext[wi+1] = 0x20;
    } else if(b1 == 0x20 && b2 == 0x3){
      /* em space */
      utext[wi] = 0x0;
      utext[wi+1] = 0x20;
    } else if(b1 == 0x20 && b2 == 0x9){
      /* thin space */
      utext[wi] = 0x0;
      utext[wi+1] = 0x20;
    } else if(b1 == 0x30 && b2 == 0x0){
      /* fullwidth space */
      utext[wi] = 0x0;
      utext[wi+1] = 0x20;
    } else if(b1 == 0xff){
      if(b2 >= 0x21 && b2 <= 0x3a){
        /* fullwidth alphabets */
        utext[wi] = 0x0;
        utext[wi+1] = b2 - 0x21 + 0x41;
      } else if(b2 >= 0x41 && b2 <= 0x5a){
        /* fullwidth small alphabets */
        utext[wi] = 0x0;
        utext[wi+1] = b2 - 0x41 + 0x61;
      } else if(b2 >= 0x10 && b2 <= 0x19){
        /* fullwidth numbers */
        utext[wi] = 0x0;
        utext[wi+1] = b2 - 0x10 + 0x30;
      } else if(b2 == 0x61){
        /* halfwidth full stop */
        utext[wi] = 0x30;
        utext[wi+1] = 0x2;
      } else if(b2 == 0x62){
        /* halfwidth left corner */
        utext[wi] = 0x30;
        utext[wi+1] = 0xc;
      } else if(b2 == 0x63){
        /* halfwidth right corner */
        utext[wi] = 0x30;
        utext[wi+1] = 0xd;
      } else if(b2 == 0x64){
        /* halfwidth comma */
        utext[wi] = 0x30;
        utext[wi+1] = 0x1;
      } else if(b2 == 0x65){
        /* halfwidth middle dot */
        utext[wi] = 0x30;
        utext[wi+1] = 0xfb;
      } else if(b2 == 0x66){
        /* halfwidth wo */
        utext[wi] = 0x30;
        utext[wi+1] = 0xf2;
      } else if(b2 >= 0x67 && b2 <= 0x6b){
        /* halfwidth small a-o */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x67) * 2 + 0xa1;
      } else if(b2 >= 0x6c && b2 <= 0x6e){
        /* halfwidth small ya-yo */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x6c) * 2 + 0xe3;
      } else if(b2 == 0x6f){
        /* halfwidth small tu */
        utext[wi] = 0x30;
        utext[wi+1] = 0xc3;
      } else if(b2 == 0x70){
        /* halfwidth prolonged mark */
        utext[wi] = 0x30;
        utext[wi+1] = 0xfc;
      } else if(b2 >= 0x71 && b2 <= 0x75){
        /* halfwidth a-o */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x71) * 2 + 0xa2;
        if(i + 2 < size - 1 && b2 == 0x73 && utext[i+2] == 0xff && utext[i+3] == 0x9e){
          utext[wi+1] = 0xf4;
          i += 2;
        }
      } else if(b2 >= 0x76 && b2 <= 0x7a){
        /* halfwidth ka-ko */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x76) * 2 + 0xab;
        if(i + 2 < size - 1 && utext[i+2] == 0xff && utext[i+3] == 0x9e){
          utext[wi+1] += 1;
          i += 2;
        }
      } else if(b2 >= 0x7b && b2 <= 0x7f){
        /* halfwidth sa-so */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x7b) * 2 + 0xb5;
        if(i + 2 < size - 1 && utext[i+2] == 0xff && utext[i+3] == 0x9e){
          utext[wi+1] += 1;
          i += 2;
        }
      } else if(b2 >= 0x80 && b2 <= 0x84){
        /* halfwidth ta-to */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x80) * 2 + 0xbf + (b2 >= 0x82 ? 1 : 0);
        if(i + 2 < size - 1 && utext[i+2] == 0xff && utext[i+3] == 0x9e){
          utext[wi+1] += 1;
          i += 2;
        }
      } else if(b2 >= 0x85 && b2 <= 0x89){
        /* halfwidth na-no */
        utext[wi] = 0x30;
        utext[wi+1] = b2 - 0x85 + 0xca;
      } else if(b2 >= 0x8a && b2 <= 0x8e){
        /* halfwidth ha-ho */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x8a) * 3 + 0xcf;
        if(i + 2 < size - 1){
          if(utext[i+2] == 0xff && utext[i+3] == 0x9e){
            utext[wi+1] += 1;
            i += 2;
          } else if(utext[i+2] == 0xff && utext[i+3] == 0x9f){
            utext[wi+1] += 2;
            i += 2;
          }
        }
      } else if(b2 >= 0x8f && b2 <= 0x93){
        /* halfwidth ma-mo */
        utext[wi] = 0x30;
        utext[wi+1] = b2 - 0x8f + 0xde;
      } else if(b2 >= 0x94 && b2 <= 0x96){
        /* halfwidth ya-yo */
        utext[wi] = 0x30;
        utext[wi+1] = (b2 - 0x94) * 2 + 0xe4;
      } else if(b2 >= 0x97 && b2 <= 0x9b){
        /* halfwidth ra-ro */
        utext[wi] = 0x30;
        utext[wi+1] = b2 - 0x97 + 0xe9;
      } else if(b2 == 0x9c){
        /* halfwidth wa */
        utext[wi] = 0x30;
        utext[wi+1] = 0xef;
      } else if(b2 == 0x9d){
        /* halfwidth wo */
        utext[wi] = 0x30;
        utext[wi+1] = 0xf3;
      } else {
        utext[wi] = b1;
        utext[wi+1] = b2;
      }
    } else {
      utext[wi] = b1;
      utext[wi+1] = b2;
    }
    wi += 2;
  }
  *sp = wi;
}


/* Canonicalize a text for search keys.
   `utext' specifies a text whose encoding is UTF-16BE.
   `size' specifies the size of the text.
   `funcspc' specifies whether to allow functional space characters. */
static void est_canonicalize_text(unsigned char *utext, int size, int funcspc){
  int i;
  for(i = 0; i < size; i += 2){
    if(utext[i] == 0x0){
      if(utext[i+1] >= 'A' && utext[i+1] <= 'Z'){
        /* ascii */
        utext[i+1] += 'a' - 'A';
      } else if((utext[i+1] >= 0xc0 && utext[i+1] <= 0xd6) ||
                (utext[i+1] >= 0xd8 && utext[i+1] <= 0xde)){
        /* latin-1 supplement */
        utext[i+1] += 0x20;
      } else if(!funcspc && utext[i+1] < ' '){
        /* functional spaces */
        utext[i+1] = ' ';
      }
    } else if(utext[i] == 0x1){
      if((utext[i+1] <= 0x36 && utext[i+1] % 2 == 0) ||
         (utext[i+1] >= 0x39 && utext[i+1] <= 0x47 && utext[i+1] % 2 == 1) ||
         (utext[i+1] >= 0x4a && utext[i+1] <= 0x76 && utext[i+1] % 2 == 0) ||
         (utext[i+1] >= 0x79 && utext[i+1] <= 0x7d && utext[i+1] % 2 == 1)){
        /* latin extended-a */
        utext[i+1] += 0x1;
      } else if(utext[i+1] == 0x78){
        /* y with umlaut */
        utext[i] = 0x0;
        utext[i+1] = 0xff;
      }
    } else if(utext[i] == 0x3){
      if(utext[i+1] >= 0x91 && utext[i+1] <= 0xa9){
        /* greek */
        utext[i+1] += 0x20;
      }
    } else if(utext[i] == 0x4){
      if(utext[i+1] >= 0x10 && utext[i+1] <= 0x2f){
        /* cyrillic */
        utext[i+1] += 0x20;
      } else if(utext[i+1] <= 0x0f){
        /* cyrillic with mark */
        utext[i+1] += 0x50;
      }
    } else if(utext[i] == 0xff){
      if(utext[i+1] >= 0xf0){
        /* special */
        utext[i] = 0x0;
        utext[i+1] = ' ';
      }
    }
  }
}


/* Categorize a character.
   `c' specifies the UCS number of a character.
   The return value is the category of the character. */
static int est_char_category(int c){
  /* ascii space */
  if(c <= 0x0020) return ESTSPACECHR;
  /* ascii alnum */
  if((c >= 0x0030 && c <= 0x0039) || (c >= 0x0041 && c <= 0x005a) ||
     (c >= 0x0061 && c <= 0x007a)) return ESTWESTALPH;
  /* latin */
  if((c >= 0x00c0 && c <= 0x00ff && c != 0x00d7 && c != 0x00f7) || (c >= 0x0100 && c <= 0x017f))
    return ESTWESTALPH;
  /* arabic and syrian */
  if(c >= 0x0600 && c <= 0x08ff) return ESTEASTALPH;
  /* south and south east asia */
  if((c >= 0x0900 && c <= 0x109f) || (c >= 0x1700 && c <= 0x1cff)) return ESTEASTALPH;
  /* cjk */
  if((c >= 0x1100 && c <= 0x11ff) || (c >= 0x2e80 && c <= 0xd7af) ||
     (c >= 0xf900 && c <= 0xfaff) || (c >= 0xff00 && c <= 0xffef)) return ESTEASTALPH;
  /* asian presentation forms */
  if((c >= 0xfb50 && c <= 0xfdff) || (c >= 0xfe30 && c <= 0xfe4f) ||
     (c >= 0xfe70 && c <= 0xfeff)) return ESTEASTALPH;
  /* others */
  return ESTDELIMCHR;
}


/* Categorize a character for perfect N-gram analyzer.
   `c' specifies the UCS number of a character.
   The return value is the category of the character. */
static int est_char_category_perfng(int c){
  if(c <= 0x0020) return ESTSPACECHR;
  return ESTEASTALPH;
}


/* Convert a simplified phrase into complete form.
   `sphrase' specifies a simplified phrase.
   The return value is the complete form of the phrase. */
static char *est_phrase_from_thumb(const char *sphrase){
  CBDATUM *datum;
  const char *oper, *rp, *pv;
  unsigned char *utext;
  char *rtext;
  int size, quote, lw;
  assert(sphrase);
  datum = cbdatumopen("", 0);
  utext = (unsigned char *)est_uconv_in(sphrase, strlen(sphrase), &size);
  est_normalize_text(utext, size, &size);
  est_canonicalize_text(utext, size, FALSE);
  rtext = est_uconv_out((char *)utext, size, NULL);
  cbstrsqzspc(rtext);
  quote = FALSE;
  oper = NULL;
  lw = FALSE;
  for(rp = rtext; *rp != '\0'; rp++){
    if(*rp == '"'){
      if(oper){
        cbdatumcat(datum, oper, -1);
        oper = NULL;
      }
      quote = !quote;
      continue;
    }
    if(quote){
      cbdatumcat(datum, rp, 1);
      continue;
    }
    switch(*rp){
    case ' ':
      if(!oper) oper = " AND ";
      lw = FALSE;
      break;
    case '&':
      oper = " AND ";
      lw = FALSE;
      break;
    case '|':
      oper = " OR ";
      lw = FALSE;
      break;
    case '!':
      oper = " ANDNOT ";
      lw = FALSE;
      break;
    default:
      if(oper){
        cbdatumcat(datum, oper, -1);
        oper = NULL;
      }
      if(!lw){
        pv = rp;
        while(*pv != '\0' && *pv != ' '){
          pv++;
        }
        if(pv > rp + 1 && pv[-1] == '*'){
          if(rp[0] == '*'){
            cbdatumcat(datum, ESTOPWCRX " ",  -1);
          } else {
            cbdatumcat(datum, ESTOPWCBW " ",  -1);
          }
        } else if(pv > rp + 1 && rp[0] == '*'){
          if(pv[-1] == '*'){
            cbdatumcat(datum, ESTOPWCRX " ",  -1);
          } else {
            cbdatumcat(datum, ESTOPWCEW " ",  -1);
          }
        }
      }
      if(*rp != '*' || (lw && rp[1] != '\0' && rp[1] != ' ')) cbdatumcat(datum, rp, 1);
      lw = TRUE;
    }
  }
  free(rtext);
  free(utext);
  return cbdatumtomalloc(datum, NULL);
}


/* Add a string to a snippet.
   `rtext' specifies a raw text.
   `ctext' specifies a canonicalized text.
   `size' specifies the size of the raw text and the canonicalized text.
   `awsiz' specifies the size of allowance for matching words.
   `res' specifies a datum object for the result.
   `rwords' specifies a list object of raw words. */
static void est_snippet_add_text(const unsigned char *rtext, const unsigned char *ctext,
                                 int size, int awsiz, CBDATUM *res, const CBLIST *rwords){
  const unsigned char *rword;
  char *orig;
  int i, j, bi, rwsiz, step, osiz;
  bi = 0;
  for(i = 0; i < size; i += 2){
    for(j = 0; j < CB_LISTNUM(rwords); j++){
      rword = (unsigned char *)CB_LISTVAL2(rwords, j, &rwsiz);
      if((step = est_str_fwmatch_wide(ctext + i, size + awsiz - i, rword, rwsiz)) > 0){
        if(i - bi > 0){
          orig = est_uconv_out((char *)rtext + bi, i - bi, &osiz);
          cbdatumcat(res, orig, osiz);
          cbdatumcat(res, "\n", 1);
          free(orig);
        }
        orig = est_uconv_out((char *)rtext + i, step, &osiz);
        cbdatumcat(res, orig, osiz);
        free(orig);
        cbdatumcat(res, "\t", 1);
        orig = est_uconv_out((char *)rword, rwsiz, &osiz);
        cbdatumcat(res, orig, osiz);
        free(orig);
        cbdatumcat(res, "\n", 1);
        bi = i + step;
        i = bi - 2;
        break;
      }
    }
  }
  if(i - bi > 0){
    orig = est_uconv_out((char *)rtext + bi, i - bi, &osiz);
    cbdatumcat(res, orig, osiz);
    cbdatumcat(res, "\n", 1);
    free(orig);
  }
}


/* Check whether a string begins with a key.
   `string' specifies a target string whose encoding is UTF-16BE.
   `size' specifies the size of the target string.
   `key' specifies a key string whose encoding is UTF-16BE.
   `ksiz' specifies the size of the key string.
   `key' specifies the pointer
   The return value is the number of characters of the corresponding string, or 0 if the target
   string does not begin with the key. */
static int est_str_fwmatch_wide(const unsigned char *str, int size,
                                const unsigned char *key, int ksiz){
  int si, ki;
  assert(str && size >= 0 && key && ksiz >= 0);
  if(size < 2 || ksiz < 2 || (str[0] == 0x0 && str[1] <= 0x20)) return 0;
  si = 0;
  ki = 0;
  while(ki < ksiz){
    if(si >= size) return 0;
    if(str[si] == 0x0 && str[si+1] <= 0x20){
      si += 2;
      continue;
    }
    if(key[ki] == 0x0 && key[ki+1] <= 0x20){
      ki += 2;
      continue;
    }
    if(str[si] != key[ki] || str[si+1] != key[ki+1]) return 0;
    si += 2;
    ki += 2;
  }
  return si;
}


/* Open the inverted index.
   `name' specifies the name of a directory.
   `omode' specifies an open mode of Villa.
   `dnum' specifies the number of database files.
   The return value is a database object of the database. */
static ESTIDX *est_idx_open(const char *name, int omode, int dnum){
  ESTIDX *idx;
  CBLIST *files;
  char path[ESTPATHBUFSIZ];
  int i;
  assert(name && dnum > 0);
  if(dnum > ESTIDXDMAX) dnum = ESTIDXDMAX;
  CB_MALLOC(idx, sizeof(ESTIDX));
  if((omode & VL_OCREAT) && !est_mkdir(name) && errno != EEXIST) return NULL;
  if((omode & VL_OTRUNC) && (files = cbdirlist(name)) != NULL){
    for(i = 0; i < CB_LISTNUM(files); i++){
      sprintf(path, "%s%c%s", name, ESTPATHCHR, CB_LISTVAL(files, i, NULL));
      unlink(path);
    }
    cblistclose(files);
  }
  for(i = 0; i < dnum; i++){
    sprintf(path, "%s%c%04d", name, ESTPATHCHR, i + 1);
    if(!(idx->dbs[i] = vlopen(path, omode, VL_CMPLEX))){
      while(--i >= 0){
        vlclose(idx->dbs[i]);
      }
      return NULL;
    }
  }
  idx->name = cbmemdup(name, -1);
  idx->omode = omode;
  idx->dnum = dnum;
  idx->cdb = idx->dbs[dnum-1];
  return idx;
}


/* Close the inverted index.
   `idx' specifies an object of the inverted index.
   The return value is true if success, else it is false. */
static int est_idx_close(ESTIDX *idx){
  int i, err;
  assert(idx);
  err = FALSE;
  for(i = 0; i < idx->dnum; i++){
    if(!vlclose(idx->dbs[i])) err = TRUE;
  }
  free(idx->name);
  free(idx);
  return err ? FALSE : TRUE;
}


/* Set the tuning parameters of the inverted index.
   `idx' specifies an object of the inverted index.
   Other parameters are same with `vlsettuning' of Villa. */
static void est_idx_set_tuning(ESTIDX *idx, int lrecmax, int nidxmax, int lcnum, int ncnum){
  int i;
  assert(idx);
  for(i = 0; i < idx->dnum; i++){
    vlsettuning(idx->dbs[i], lrecmax, nidxmax, lcnum, ncnum);
  }
}


/* Increment the inverted index.
   `idx' specifies an object of the inverted index. */
static void est_idx_increment(ESTIDX *idx){
  char path[ESTPATHBUFSIZ];
  int i, min, size;
  assert(idx);
  min = INT_MAX;
  for(i = 0; i < idx->dnum; i++){
    size = vlfsiz(idx->cdb);
    if(size < min) min = size;
  }
  if(idx->dnum >= ESTIDXDMAX || (idx->dnum >= ESTIDXDSTD && min < ESTIDXDBMAX)){
    est_idx_set_current(idx);
    return;
  }
  sprintf(path, "%s%c%04d", idx->name, ESTPATHCHR, idx->dnum + 1);
  if((idx->dbs[idx->dnum] = vlopen(path, idx->omode | VL_OCREAT | VL_OTRUNC, VL_CMPLEX)) != NULL){
    idx->cdb = idx->dbs[idx->dnum];
    idx->dnum++;
  }
}


/* Get the number of files of the inverted index.
   The return the number of files of the inverted index. */
static int est_idx_dnum(ESTIDX *idx){
  assert(idx);
  return idx->dnum;
}


/* Add a record to the inverted index.
   `idx' specifies an object of the inverted index.
   `word' specifies a word.
   `vbuf' specifies the pointer to the value of a record.
   `vsiz' specifies the size of the value.
   The return value is true if success, else it is false. */
static int est_idx_add(ESTIDX *idx, const char *word, int wsiz, const char *vbuf, int vsiz){
  assert(idx && word && wsiz >= 0 && vbuf && vsiz >= 0);
  return vlput(idx->cdb, word, wsiz, vbuf, vsiz, VL_DDUP);
}


/* Store a record to a file of the inverted index.
   `idx' specifies an object of the inverted index.
   `inum' specifies the index of a file of the inverted index.
   `word' specifies a word.
   `vbuf' specifies the pointer to the value of a record.
   `vsiz' specifies the size of the value.
   The return value is true if success, else it is false. */
static int est_idx_put_one(ESTIDX *idx, int inum, const char *word, int wsiz,
                           const char *vbuf, int vsiz){
  assert(idx && inum >= 0 && word && wsiz >= 0 && vbuf && vsiz >= 0);
  if(!vloutlist(idx->dbs[inum], word, wsiz) && dpecode != DP_ENOITEM) return FALSE;
  return vsiz < 1 ? TRUE : vlput(idx->dbs[inum], word, wsiz, vbuf, vsiz, VL_DDUP);
}


/* Remove a record from the inverted index.
   `idx' specifies an object of the inverted index.
   `word' specifies a word.
   `wsiz' specifies the size of the word.
   The return value is true if success, else it is false.  Even if no item correspongs, it is
   success. */
static int est_idx_out(ESTIDX *idx, const char *word, int wsiz){
  int i, err;
  assert(idx && word && wsiz >= 0);
  err = FALSE;
  for(i = 0; i < idx->dnum; i++){
    if(!vloutlist(idx->dbs[i], word, wsiz) && dpecode != DP_ENOITEM) err = TRUE;
  }
  return err ? FALSE : TRUE;
}


/* Get a record from the inverted index.
   `idx' specifies an object of the inverted index.
   `word' specifies a word.
   `wsiz' specifies the size of the word.
   `sp' specifies the pointer to a variable to which the size of the region of the return value
   is assigned.
   The return value is the pointer to the region of the value of the corresponding record.
   if no item correspongs, empty region is returned. */
static char *est_idx_get(ESTIDX *idx, const char *word, int wsiz, int *sp){
  CBDATUM *datum;
  char *vbuf;
  int i, vsiz;
  assert(idx && word && wsiz >= 0 && sp);
  datum = cbdatumopen("", 0);
  for(i = 0; i < idx->dnum; i++){
    if(!(vbuf = vlgetcat(idx->dbs[i], word, wsiz, &vsiz))) continue;
    cbdatumcat(datum, vbuf, vsiz);
    free(vbuf);
  }
  return cbdatumtomalloc(datum, sp);
}


/* Get a record from a file of the inverted index.
   `idx' specifies an object of the inverted index.
   `inum' specifies the index of a file of the inverted index.
   `word' specifies a word.
   `wsiz' specifies the size of the word.
   `sp' specifies the pointer to a variable to which the size of the region of the return value
   is assigned.
   The return value is the pointer to the region of the value of the corresponding record.
   if no item correspongs, `NULL' is returned. */
static char *est_idx_get_one(ESTIDX *idx, int inum, const char *word, int wsiz, int *sp){
  assert(idx && inum >= 0 && word && wsiz >= 0 && sp);
  return vlgetcat(idx->dbs[inum], word, wsiz, sp);
}


/* Get the size of the value of a record in the inverted index.
   `idx' specifies an object of the inverted index.
   `word' specifies a word.
   `wsiz' specifies the size of the word.
   The return value is the size of the value of the corresponding record.
   if no item correspongs, 0 is returned. */
static int est_idx_vsiz(ESTIDX *idx, const char *word, int wsiz){
  char *vbuf;
  int i, sum, vsiz;
  assert(idx && word && wsiz >= 0);
  sum = 0;
  for(i = 0; i < idx->dnum; i++){
    if(!(vbuf = vlgetcat(idx->dbs[i], word, wsiz, &vsiz))) continue;
    sum += vsiz;
    free(vbuf);
  }
  return sum;
}


/* Get the number of division of the inverted index.
   `idx' specifies an object of the inverted index.
   The return value is the number of division of the inverted index. */
static int est_idx_num(ESTIDX *idx){
  assert(idx);
  return idx->dnum;
}


/* Get the size of the inverted index.
   `idx' specifies an object of the inverted index.
   The return value is the size of the inverted index. */
static double est_idx_size(ESTIDX *idx){
  int i;
  double size;
  assert(idx);
  size = 0;
  for(i = 0; i < idx->dnum; i++){
    size += vlfsiz(idx->dbs[i]);
  }
  return size;
}


/* Get the size of the current file of the inverted index.
   `idx' specifies an object of the inverted index.
   The return value is the size of the current file of the inverted index. */
static int est_idx_size_current(ESTIDX *idx){
  assert(idx);
  return vlfsiz(idx->cdb);
}


/* Syncronize the inverted index.
   `idx' specifies an object of the inverted index.
   The return value is the size of the inverted index. */
static int est_idx_sync(ESTIDX *idx){
  int i;
  assert(idx);
  for(i = 0; i < idx->dnum; i++){
    if(!vlsync(idx->dbs[i])) return FALSE;
  }
  return TRUE;
}


/* Optimize the inverted index.
   `idx' specifies an object of the inverted index.
   The return value is the size of the inverted index. */
static int est_idx_optimize(ESTIDX *idx){
  int i;
  assert(idx);
  for(i = 0; i < idx->dnum; i++){
    if(!vloptimize(idx->dbs[i])) return FALSE;
  }
  return TRUE;
}


/* Set the current database to the smallest one in the inverted index.
   `idx' specifies an object of the inverted index. */
static void est_idx_set_current(ESTIDX *idx){
  int i, size, min;
  assert(idx);
  min = vlfsiz(idx->cdb);
  for(i = 0; i < idx->dnum; i++){
    if((size = vlfsiz(idx->dbs[i])) < min){
      idx->cdb = idx->dbs[i];
      min = size;
    }
  }
}


/* Write meta data to the database.
   `db' specifies a database object.
   The return value is true if success, else it is false. */
static int est_db_write_meta(ESTDB *db){
  char vbuf[ESTNUMBUFSIZ], *sbuf;
  int err, ssiz;
  assert(db);
  err = FALSE;
  sprintf(vbuf, "%d", est_idx_num(db->idxdb));
  if(!dpput(db->metadb, ESTKEYIDXNUM, -1, vbuf, -1, DP_DOVER)) err = TRUE;
  sprintf(vbuf, "%d", db->dseq);
  if(!dpput(db->metadb, ESTKEYDSEQ, -1, vbuf, -1, DP_DOVER)) err = TRUE;
  sprintf(vbuf, "%d", db->dnum);
  if(!dpput(db->metadb, ESTKEYDNUM, -1, vbuf, -1, DP_DOVER)) err = TRUE;
  sprintf(vbuf, "%d", db->amode);
  if(!dpput(db->metadb, ESTKEYAMODE, -1, vbuf, -1, DP_DOVER)) err = TRUE;
  if(db->metacc){
    sbuf = cbmapdump(db->metacc, &ssiz);
    if(!dpput(db->metadb, ESTKEYMETA, -1, sbuf, ssiz, DP_DOVER)) err = TRUE;
    free(sbuf);
  }
  if(err){
    db->ecode = ESTEDB;
    db->fatal = TRUE;
  }
  return err ? FALSE : TRUE;
}


/* Call the callback function of a database.
   `db' specifies a database object.
   `info' specifies an extra message. */
static void est_db_inform(ESTDB *db, const char *info){
  char *msg;
  assert(db);
  if(!db->cbinfo) return;
  msg = cbsprintf("%s: name=%s dnum=%d wnum=%d fsiz=%.0f crnum=%d csiz=%d dknum=%d",
                  info, db->name, db->dnum, vlrnum(db->fwmdb), (double)est_db_size(db),
                  cbmaprnum(db->idxcc), est_db_used_cache_size(db), cbmaprnum(db->outcc));
  db->cbinfo(msg);
  free(msg);
}


/* Prepare cache for meta data.
   `db' specifies a database object. */
static void est_db_prepare_meta(ESTDB *db){
  char *sbuf;
  int ssiz;
  assert(db);
  if((sbuf = dpget(db->metadb, ESTKEYMETA, -1, 0, -1, &ssiz)) != NULL){
    db->metacc = cbmapload(sbuf, ssiz);
    free(sbuf);
  } else {
    db->metacc = cbmapopenex(ESTMINIBNUM);
  }
}


/* Create a list of terms for search.
   `phrase' specifies a search phrase.
   The return value is a list object of the terms of the phrase. */
static CBLIST *est_phrase_terms(const char *phrase){
  CBLIST *terms, *elems;
  CBDATUM *datum;
  const char *elem;
  char *tbuf, *pbuf;
  int i, tsiz, psiz, lw;
  assert(phrase);
  terms = cblistopen();
  tbuf = est_uconv_in(phrase, strlen(phrase), &tsiz);
  est_normalize_text((unsigned char *)tbuf, tsiz, &tsiz);
  pbuf = est_uconv_out(tbuf, tsiz, &psiz);
  elems = cbsplit(pbuf, psiz, "\a\b\t\n\v\f\r ");
  datum = cbdatumopen("", 0);
  lw = FALSE;
  for(i = 0; i < CB_LISTNUM(elems); i++){
    elem = CB_LISTVAL(elems, i, NULL);
    if(elem[0] == '\0') continue;
    if(!strcmp(elem, ESTOPUNION)){
      if(CB_DATUMSIZE(datum) < 1) continue;
      if(lw) cbdatumcat(datum, "\t", -1);
      lw = FALSE;
    } else if(!strcmp(elem, ESTOPWCBW)){
      if(!lw) cbdatumcat(datum, " b", 2);
    } else if(!strcmp(elem, ESTOPWCEW)){
      if(!lw) cbdatumcat(datum, " e", 2);
    } else if(!strcmp(elem, ESTOPWCRX)){
      if(!lw) cbdatumcat(datum, " r", 2);
    } else if(!strcmp(elem, ESTOPISECT) || !strcmp(elem, ESTOPDIFF)){
      if(CB_DATUMSIZE(datum) < 1) continue;
      cblistpush(terms, CB_DATUMPTR(datum), CB_DATUMSIZE(datum));
      cbdatumsetsize(datum, 0);
      cblistpush(terms, elem, -1);
      lw = FALSE;
    } else {
      if(CB_DATUMSIZE(datum) > 0 && lw) cbdatumcat(datum, " ", 1);
      cbdatumcat(datum, elem, -1);
      lw = TRUE;
    }
  }
  if(CB_DATUMSIZE(datum) > 0) cblistpush(terms, CB_DATUMPTR(datum), CB_DATUMSIZE(datum));
  cbdatumclose(datum);
  cblistclose(elems);
  free(pbuf);
  free(tbuf);
  for(i = 0; i < CB_LISTNUM(terms); i++){
    elem = CB_LISTVAL(terms, i, NULL);
    if(!strcmp(elem, ESTOPUVSET) || !strcmp(elem, ESTOPISECT) ||
       !strcmp(elem, ESTOPDIFF)) continue;
    tbuf = est_uconv_in(elem, strlen(elem), &tsiz);
    est_canonicalize_text((unsigned char *)tbuf, tsiz, TRUE);
    pbuf = est_uconv_out(tbuf, tsiz, &psiz);
    cblistover(terms, i, pbuf, -1);
    free(pbuf);
    free(tbuf);
  }
  for(i = CB_LISTNUM(terms) - 1; i >= 0; i--){
    elem = CB_LISTVAL(terms, i, NULL);
    if(strcmp(elem, ESTOPISECT) && strcmp(elem, ESTOPDIFF)) break;
    free(cblistpop(terms, NULL));
  }
  return terms;
}


/* Compare two scores by each ID.
   `ap' specifies the pointer to one score.
   `bp' specifies the pointer to the other score.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_score_compare_by_id(const void *ap, const void *bp){
  assert(ap && bp);
  return ((ESTSCORE *)ap)->id - ((ESTSCORE *)bp)->id;
}


/* Compare two scores by each score point.
   `ap' specifies the pointer to one score.
   `bp' specifies the pointer to the other score.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_score_compare_by_score(const void *ap, const void *bp){
  assert(ap && bp);
  return ((ESTSCORE *)bp)->score - ((ESTSCORE *)ap)->score;
}


/* Compare two scores by attributes of strings for ascending order.
   `ap' specifies the pointer to one score.
   `bp' specifies the pointer to the other score.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_score_compare_by_str_asc(const void *ap, const void *bp){
  assert(ap && bp);
  return strcmp(((ESTSCORE *)ap)->value, ((ESTSCORE *)bp)->value);
}


/* Compare two scores by attributes of strings for descending order.
   `ap' specifies the pointer to one score.
   `bp' specifies the pointer to the other score.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_score_compare_by_str_desc(const void *ap, const void *bp){
  assert(ap && bp);
  return strcmp(((ESTSCORE *)bp)->value, ((ESTSCORE *)ap)->value);
}


/* Compare two scores by attributes of numbers for ascending order.
   `ap' specifies the pointer to one score.
   `bp' specifies the pointer to the other score.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_score_compare_by_num_asc(const void *ap, const void *bp){
  assert(ap && bp);
  return (time_t)((ESTSCORE *)ap)->value - (time_t)((ESTSCORE *)bp)->value;
}


/* Compare two scores by attributes of numbers for descending order.
   `ap' specifies the pointer to one score.
   `bp' specifies the pointer to the other score.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_score_compare_by_num_desc(const void *ap, const void *bp){
  assert(ap && bp);
  return (time_t)((ESTSCORE *)bp)->value - (time_t)((ESTSCORE *)ap)->value;
}


/* Get the universal set of documents in a database.
   `db' specifies a database object.
   `nump' specifies the pointer to which the number of elements in the result is assigned.
   `hints' specifies a list object.  If it is `NULL', it is not used.
   `add' specifies whether the result to be treated in union or difference.
   The return value is an array whose elements are ID numbers of corresponding documents. */
static ESTSCORE *est_search_uvset(ESTDB *db, int *nump, CBMAP *hints, int add){
  ESTSCORE *scores;
  char *vbuf, numbuf[ESTNUMBUFSIZ];
  int snum, smax;
  assert(db && nump);
  smax = ESTALLOCUNIT;
  CB_MALLOC(scores, smax * sizeof(ESTSCORE));
  snum = 0;
  vlcurfirst(db->listdb);
  while((vbuf = vlcurval(db->listdb, NULL)) != NULL){
    if(snum >= smax){
      smax *= 2;
      CB_REALLOC(scores, smax * sizeof(ESTSCORE));
    }
    scores[snum].id = atoi(vbuf);
    scores[snum].score = 0;
    snum++;
    free(vbuf);
    vlcurnext(db->listdb);
  }
  *nump = snum;
  if(hints){
    sprintf(numbuf, "%d", snum * (add ? 1 : -1));
    cbmapput(hints, ESTOPUVSET, -1, numbuf, -1, FALSE);
  }
  return scores;
}


/* Expand a word to words which begins with it.
   `db' specifies a database object.
   `word' specifies a word.
   `list' specifies a list object to contain the results. */
static void est_expand_word_bw(ESTDB *db, const char *word, CBLIST *list){
  char *kbuf;
  int ksiz;
  assert(db && word && list);
  vlcurjump(db->fwmdb, word, -1, VL_JFORWARD);
  while((kbuf = vlcurkey(db->fwmdb, &ksiz)) != NULL){
    if(!cbstrfwmatch(kbuf, word)){
      free(kbuf);
      break;
    }
    cblistpushbuf(list, kbuf, ksiz);
    vlcurnext(db->fwmdb);
  }
}


/* Expand a word to words which ends with it.
   `db' specifies a database object.
   `word' specifies a word.
   `list' specifies a list object to contain the results. */
static void est_expand_word_ew(ESTDB *db, const char *word, CBLIST *list){
  char *kbuf;
  int wsiz, ksiz;
  assert(db && word && list);
  wsiz = strlen(word);
  vlcurfirst(db->fwmdb);
  while((kbuf = vlcurkey(db->fwmdb, &ksiz)) != NULL){
    if(ksiz >= wsiz && !memcmp(kbuf + ksiz - wsiz, word, wsiz)){
      cblistpushbuf(list, kbuf, ksiz);
    } else {
      free(kbuf);
    }
    vlcurnext(db->fwmdb);
  }
}


/* Expand regular expressios to words which matches them.
   `db' specifies a database object.
   `word' specifies regular expressions.
   `list' specifies a list object to contain the results. */
static void est_expand_word_rx(ESTDB *db, const char *word, CBLIST *list){
  void *regex;
  char *kbuf;
  int ksiz;
  assert(db && word && list);
  if(!(regex = est_regex_new(word))) return;
  vlcurfirst(db->fwmdb);
  while((kbuf = vlcurkey(db->fwmdb, &ksiz)) != NULL){
    if(est_regex_match(regex, kbuf)){
      cblistpushbuf(list, kbuf, ksiz);
    } else {
      free(kbuf);
    }
    vlcurnext(db->fwmdb);
  }
  est_regex_delete(regex);
}


/* Get a correspinding set of documents in a database.
   `db' specifies a database object.
   `term' specifies a union term.
   `gstep' specifies number of steps of N-gram.
   `nump' specifies the pointer to which the number of elements in the result is assigned.
   `hints' specifies a list object.  If it is `NULL', it is not used.
   `add' specifies whether the result to be treated in union or difference.
   The return value is an array whose elements are ID numbers of corresponding documents. */
static ESTSCORE *est_search_union(ESTDB *db, const char *term, int gstep,
                                  int *nump, CBMAP *hints, int add){
  const ESTSCORE *cscores;
  ESTSCORE *scores, *tscores;
  CBLIST *words, *grams;
  const char *ckey, *word, *gram, *rp, *fnext, *snext, *cbuf;
  char *vbuf, numbuf[ESTNUMBUFSIZ];
  int i, j, k, snum, smax, cksiz, single, tsmax, tsnum, vsiz, gcnum, gsiz, csiz, wgstep, nnum;
  int west, wild, mfsiz, mssiz, mfhash, mshash, tfhash, tshash, id, score, hit, hnum;
  assert(db && term && gstep > 0 && nump);
  smax = ESTALLOCUNIT;
  CB_MALLOC(scores, smax * sizeof(ESTSCORE));
  snum = 0;
  words = cbsplit(term, -1, "\t");
  for(i = 0; i < CB_LISTNUM(words); i++){
    ckey = CB_LISTVAL2(words, i, &cksiz);
    word = ckey;
    if((cscores = est_rescc_get(db, ckey, cksiz, &tsnum)) != NULL){
      if(hints){
        sprintf(numbuf, "%d", tsnum * (add ? 1 : -1));
        cbmapput(hints, word, -1, numbuf, -1, FALSE);
      }
      for(j = 0; j < tsnum; j++){
        if(snum >= smax){
          smax *= 2;
          CB_REALLOC(scores, smax * sizeof(ESTSCORE));
        }
        scores[snum].id = cscores[j].id;
        scores[snum].score = cscores[j].score;
        snum++;
      }
    } else {
      wild = '\0';
      if(word[0] == ' '){
        word++;
        if(word[0] == 'b'){
          wild = 'b';
          word++;
        } else  if(word[0] == 'e'){
          wild = 'e';
          word++;
        } else  if(word[0] == 'r'){
          wild = 'r';
          word++;
        }
      }
      west = ((unsigned char *)word)[0] <= 0xdf;
      if(!west || db->amode) wild = '\0';
      single = FALSE;
      grams = cblistopen();
      switch(wild){
      case 'b':
        est_break_text(word, grams, TRUE, FALSE);
        cblistpush(grams, word, -1);
        while(CB_LISTNUM(grams) > 1){
          free(cblistpop(grams, NULL));
        }
        word = CB_LISTVAL(grams, 0, NULL);
        est_expand_word_bw(db, word, grams);
        single = TRUE;
        break;
      case 'e':
        est_break_text(word, grams, TRUE, FALSE);
        cblistunshift(grams, word, -1);
        while(CB_LISTNUM(grams) > 1){
          free(cblistshift(grams, NULL));
        }
        word = CB_LISTVAL(grams, 0, NULL);
        est_expand_word_ew(db, word, grams);
        single = TRUE;
        break;
      case 'r':
        est_break_text(word, grams, TRUE, FALSE);
        while(CB_LISTNUM(grams) > 0){
          free(cblistshift(grams, NULL));
        }
        est_expand_word_rx(db, word, grams);
        single = TRUE;
        break;
      default:
        switch(db->amode){
        case ESTAMPERFNG:
          est_break_text_perfng(word, grams, TRUE, FALSE);
          break;
        default:
          est_break_text(word, grams, TRUE, FALSE);
          break;
        }
        if(CB_LISTNUM(grams) < 1){
          est_expand_word_bw(db, word, grams);
          single = TRUE;
        }
        break;
      }
      tsmax = ESTALLOCUNIT;
      CB_MALLOC(tscores, tsmax * sizeof(ESTSCORE));
      tsnum = 0;
      gcnum = 0;
      wgstep = !single && (CB_LISTNUM(grams) > 2 || gstep > 2) ? gstep : 1;
      if(west && gstep <= 2) wgstep = 1;
      for(j = 0; j < CB_LISTNUM(grams); j += wgstep){
        gcnum++;
        gram = CB_LISTVAL2(grams, j, &gsiz);
        fnext = cblistval(grams, j + 1, &mfsiz);
        snext = cblistval(grams, j + 2, &mssiz);
        mfhash = fnext ? dpinnerhash(fnext, mfsiz) % ESTJHASHNUM + 1: 0xff;
        mshash = snext ? dpouterhash(snext, mssiz) % ESTJHASHNUM + 1: 0xff;
        vbuf = est_idx_get(db->idxdb, gram, gsiz, &vsiz);
        if((cbuf = cbmapget(db->idxcc, gram, gsiz, &csiz)) != NULL){
          if(vbuf){
            CB_REALLOC(vbuf, vsiz + csiz + 100);
            memcpy(vbuf + vsiz, cbuf, csiz);
            vsiz += csiz;
          } else {
            vbuf = cbmemdup(cbuf, csiz);
            vsiz = csiz;
          }
        }
        if(!vbuf) continue;
        rp = vbuf;
        while(rp < vbuf + vsiz){
          memcpy(&id, rp, sizeof(int));
          rp += sizeof(int);
          score = *(unsigned char *)rp + 1;
          rp++;
          hit = mfhash == 0xff && mshash == 0xff;
          while(rp < vbuf + vsiz){
            tfhash = *(unsigned char *)rp;
            rp++;
            tshash = *(unsigned char *)rp;
            rp++;
            if((mfhash == 0xff || mfhash == tfhash) && (mshash == 0xff || mshash == tshash))
              hit = TRUE;
            if(*(unsigned char *)rp == 0x00){
              rp++;
              break;
            }
          }
          if(hit || single){
            if(tsnum >= tsmax){
              tsmax *= 2;
              CB_REALLOC(tscores, tsmax * sizeof(ESTSCORE));
            }
            tscores[tsnum].id = id;
            tscores[tsnum].score = score * 100;
            tsnum++;
          }
        }
        free(vbuf);
      }
      if(gcnum > 1){
        qsort(tscores, tsnum, sizeof(ESTSCORE), est_score_compare_by_id);
        nnum = 0;
        for(j = 0; j < tsnum; j++){
          id = tscores[j].id;
          score = tscores[j].score;
          hnum = 1;
          for(k = j + 1; k < tsnum && tscores[k].id == id; k++){
            score += tscores[k].score;
            hnum++;
          }
          if(hnum >= gcnum || single){
            tscores[nnum].id = id;
            tscores[nnum].score = score / hnum;
            nnum++;
          }
          j = k - 1;
        }
        tsnum = nnum;
      }
      if(hints){
        sprintf(numbuf, "%d", tsnum * (add ? 1 : -1));
        cbmapput(hints, word, -1, numbuf, -1, FALSE);
      }
      cblistclose(grams);
      for(j = 0; j < tsnum; j++){
        if(snum >= smax){
          smax *= 2;
          CB_REALLOC(scores, smax * sizeof(ESTSCORE));
        }
        scores[snum].id = tscores[j].id;
        scores[snum].score = tscores[j].score;
        snum++;
      }
      est_rescc_put(db, ckey, cksiz, tscores, tsnum);
    }
  }
  cblistclose(words);
  qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_id);
  nnum = 0;
  for(i = 0; i < snum; i++){
    id = scores[i].id;
    score = scores[i].score;
    hnum = 1;
    for(j = i + 1; j < snum && scores[j].id == id; j++){
      score += scores[j].score;
      hnum++;
    }
    scores[nnum].id = id;
    scores[nnum].score = score / hnum;
    nnum++;
    i = j - 1;
  }
  *nump = nnum;
  return scores;
}


/* Get scores in the result cache.
   `db' specifies a database object.
   `word' specifies a search word.
   `size' specifies the size of the word.
   `nump' specifies the pointer to which the number of elements in the result is assigned.
   The return value is an array whose elements are ID numbers of corresponding documents. */
static const ESTSCORE *est_rescc_get(ESTDB *db, const char *word, int size, int *nump){
  const char *vbuf;
  int vsiz;
  assert(db && word && size >= 0 && nump);
  if(!(vbuf = cbmapget(db->rescc, word, size, &vsiz))) return NULL;
  if(vsiz == sizeof(ESTSCORE) && ((ESTSCORE *)vbuf)->id == -1) return NULL;
  cbmapmove(db->rescc, word, size, FALSE);
  *nump = vsiz / sizeof(ESTSCORE);
  return (ESTSCORE *)vbuf;
}


/* Add scores into the result cache.
   `db' specifies a database object.
   `word' specifies a search word.
   `size' specifies the size of the word.
   `scores' specifies an array of scores.  It is released in this function.
   `num' specifies the number of elements of the score array. */
static void est_rescc_put(ESTDB *db, const char *word, int size, ESTSCORE *scores, int num){
  int i;
  assert(db && word && size >= 0 && scores && num >= 0);
  cbmapputvbuf(db->rescc, word, size, (char *)scores, num * sizeof(ESTSCORE));
  if(cbmaprnum(db->rescc) > db->rcmnum){
    num = db->rcmnum * 0.1 + 1;
    cbmapiterinit(db->rescc);
    for(i = 0; i < num && (word = cbmapiternext(db->rescc, &size)) != NULL; i++){
      cbmapout(db->rescc, word, size);
    }
  }
}


/* Narrow and sort scores of search candidates.
   `db' specifies a database object.
   `attrs' specifies a list object of narrowing attributes.
   `order' specifies an expression for sorting.
   `scores' specifies an array of scores of search candidates.
   `snum' specifies the number of the array.
   `limit' specifies the limit number to check.
   `restp' specifies the pointer to a variable to which rest number to be checked is assigned.
   The return value is the new number of the array. */
static int est_narrow_scores(ESTDB *db, const CBLIST *attrs, const char *order,
                             ESTSCORE *scores, int snum, int limit, int *restp){
  ESTCATTR *list;
  const char *otype, *cbuf, *rp, *pv, *ibuf;
  unsigned char *utmp;
  char *oname, *wp, *mbuf, *vbuf;
  int i, j, k, ci, oi, anum, tsiz, nnum, csiz, msiz, miss, vsiz, num, isiz, onlen;
  time_t tval;
  assert(db && scores && snum >= 0 && restp);
  *restp = 0;
  ci = -1;
  oi = -1;
  oname = NULL;
  otype = NULL;
  if(order){
    oname = cbmemdup(order, -1);
    cbstrtrim(oname);
    otype = ESTORDSTRA;
    if((wp = strchr(oname, ' ')) != NULL){
      *wp = '\0';
      rp = wp + 1;
      while(*rp == ' '){
        rp++;
      }
      otype = rp;
    }
  }
  if(attrs){
    anum = CB_LISTNUM(attrs);
    CB_MALLOC(list, sizeof(ESTCATTR) * anum + 1);
    for(i = 0; i < anum; i++){
      list[i].name = NULL;
      list[i].oper = NULL;
      list[i].val = NULL;
      rp = CB_LISTVAL(attrs, i, NULL);
      while(*rp > 0 && *rp <= ' '){
        rp++;
      }
      if((pv = strchr(rp, ' ')) != NULL){
        list[i].nsiz = pv - rp;
        list[i].name = cbmemdup(rp, list[i].nsiz);
        rp = pv;
        while(*rp > 0 && *rp <= ' '){
          rp++;
        }
        if((pv = strchr(rp, ' ')) != NULL){
          list[i].oper = cbmemdup(rp, pv - rp);
          rp = pv;
          while(*rp > 0 && *rp <= ' '){
            rp++;
          }
          list[i].vsiz = strlen(rp);
          list[i].val = cbmemdup(rp, list[i].vsiz);
        } else {
          list[i].oper = cbmemdup(rp, -1);
        }
      } else {
        list[i].nsiz = strlen(rp);
        list[i].name = cbmemdup(rp, list[i].nsiz);
      }
      if(!list[i].oper){
        list[i].oper = cbmemdup("", 0);
      }
      if(!list[i].val){
        list[i].vsiz = 0;
        list[i].val = cbmemdup("", 0);
      }
    }
    for(i = 0; i < anum; i++){
      rp = list[i].oper;
      if(*rp == '!'){
        list[i].sign = FALSE;
        rp++;
      } else {
        list[i].sign = TRUE;
      }
      if(*rp == 'I' || *rp == 'i'){
        utmp = (unsigned char *)est_uconv_in(list[i].val, list[i].vsiz, &tsiz);
        est_normalize_text(utmp, tsiz, &tsiz);
        est_canonicalize_text(utmp, tsiz, FALSE);
        list[i].sval = (char *)est_uconv_out((char *)utmp, tsiz, &(list[i].ssiz));
        free(utmp);
        rp++;
      } else {
        list[i].sval = NULL;
        list[i].ssiz = 0;
      }
      list[i].regex = NULL;
      list[i].num = cbstrmktime(list[i].val);
      if(!cbstricmp(rp, ESTOPSTREQ)){
        list[i].cop = ESTOPSTREQ;
      } else if(!cbstricmp(rp, ESTOPSTRNE)){
        list[i].cop = ESTOPSTRNE;
      } else if(!cbstricmp(rp, ESTOPSTRINC)){
        list[i].cop = ESTOPSTRINC;
      } else if(!cbstricmp(rp, ESTOPSTRBW)){
        list[i].cop = ESTOPSTRBW;
      } else if(!cbstricmp(rp, ESTOPSTREW)){
        list[i].cop = ESTOPSTREW;
      } else if(!cbstricmp(rp, ESTOPSTRAND)){
        list[i].cop = ESTOPSTRAND;
      } else if(!cbstricmp(rp, ESTOPSTROR)){
        list[i].cop = ESTOPSTROR;
      } else if(!cbstricmp(rp, ESTOPSTRRX)){
        list[i].cop = ESTOPSTRRX;
        list[i].regex = list[i].sval ? est_regex_new(list[i].sval) : est_regex_new(list[i].val);
      } else if(!cbstricmp(rp, ESTOPNUMEQ)){
        list[i].cop = ESTOPNUMEQ;
      } else if(!cbstricmp(rp, ESTOPNUMNE)){
        list[i].cop = ESTOPNUMNE;
      } else if(!cbstricmp(rp, ESTOPNUMGT)){
        list[i].cop = ESTOPNUMGT;
      } else if(!cbstricmp(rp, ESTOPNUMGE)){
        list[i].cop = ESTOPNUMGE;
      } else if(!cbstricmp(rp, ESTOPNUMLT)){
        list[i].cop = ESTOPNUMLT;
      } else if(!cbstricmp(rp, ESTOPNUMLE)){
        list[i].cop = ESTOPNUMLE;
      } else if(!cbstricmp(rp, ESTOPNUMBT)){
        list[i].cop = ESTOPNUMBT;
      } else {
        list[i].cop = NULL;
      }
    }
    if(db->spacc){
      for(i = 0; i < anum; i++){
        if(!strcmp(list[i].name, db->scname)){
          ci = i;
          break;
        }
      }
    }
    if(oname){
      for(i = 0; i < anum; i++){
        if(!strcmp(list[i].name, oname)){
          oi = i;
          break;
        }
      }
    }
    nnum = 0;
    for(i = 0; i < snum; i++){
      if(nnum >= limit){
        *restp = snum - i;
        break;
      }
      scores[i].value = NULL;
      if(ci >= 0){
        if((cbuf = cbmapget(db->spacc, (char *)&(scores[i].id), sizeof(int), &csiz)) != NULL)
          cbmapmove(db->spacc, (char *)&(scores[i].id), sizeof(int), FALSE);
      } else {
        cbuf = NULL;
        csiz = 0;
      }
      mbuf = NULL;
      if((cbuf && anum == 1) ||
         (mbuf = crget(db->attrdb, (char *)&(scores[i].id), sizeof(int), 0, -1, &msiz)) != NULL){
        miss = FALSE;
        for(j = 0; !miss && j < anum; j++){
          if(list[j].nsiz < 1) continue;
          if(mbuf){
            vbuf = cbmaploadone(mbuf, msiz, list[j].name, list[j].nsiz, &vsiz);
          } else if(csiz != 1 || cbuf[0] != '\0'){
            vbuf = cbmemdup(cbuf, csiz);
            vsiz = csiz;
          } else {
            vbuf = NULL;
          }
          if(list[j].oper[0] == '\0'){
            if(!vbuf) miss = TRUE;
          } else {
            if(!vbuf){
              vbuf = cbmemdup("", 0);
              vsiz = 0;
            }
            if(!est_match_attr(vbuf, vsiz, list[j].cop, list[j].sign, list[j].val, list[j].vsiz,
                               list[j].sval, list[j].ssiz, list[j].regex, list[j].num))
              miss = TRUE;
          }
          if(j == ci && !cbuf){
            if(vbuf){
              cbmapput(db->spacc, (char *)&(scores[i].id), sizeof(int), vbuf, vsiz, FALSE);
            } else {
              cbmapput(db->spacc, (char *)&(scores[i].id), sizeof(int), "", 1, FALSE);
            }
            if(cbmaprnum(db->spacc) > db->scmnum){
              num = db->scmnum * 0.1 + 1;
              cbmapiterinit(db->spacc);
              for(k = 0; k < num && (ibuf = cbmapiternext(db->spacc, &isiz)) != NULL; k++){
                cbmapout(db->spacc, ibuf, isiz);
              }
            }
          }
          if(j == oi){
            scores[i].value = vbuf;
          } else {
            free(vbuf);
          }
        }
        if(miss){
          free(scores[i].value);
        } else {
          scores[nnum++] = scores[i];
        }
      }
      free(mbuf);
    }
    snum = nnum;
    for(i = 0; i < anum; i++){
      if(list[i].regex) est_regex_delete(list[i].regex);
      free(list[i].sval);
      free(list[i].val);
      free(list[i].oper);
      free(list[i].name);
    }
    free(list);
  } else {
    for(i = 0; i < snum; i++){
      scores[i].value = NULL;
    }
  }
  if(oname){
    ci = db->spacc && !strcmp(oname, db->scname);
    onlen = strlen(oname);
    for(i = 0; i < snum; i++){
      if(scores[i].value) continue;
      if(ci && (cbuf = cbmapget(db->spacc, (char *)&(scores[i].id), sizeof(int), &csiz)) != NULL){
        cbmapmove(db->spacc, (char *)&(scores[i].id), sizeof(int), FALSE);
        if(csiz == 1 && cbuf[0] == '\0'){
          scores[i].value = cbmemdup("", 0);
        } else {
          scores[i].value = cbmemdup(cbuf, csiz);
        }
        continue;
      }
      if((mbuf = crget(db->attrdb, (char *)&(scores[i].id), sizeof(int), 0, -1, &msiz)) != NULL){
        if((vbuf = cbmaploadone(mbuf, msiz, oname, onlen, &vsiz)) != NULL){
          if(ci) cbmapput(db->spacc, (char *)&(scores[i].id), sizeof(int), vbuf, vsiz, FALSE);
          scores[i].value = vbuf;
        } else {
          if(ci) cbmapput(db->spacc, (char *)&(scores[i].id), sizeof(int), "", 1, FALSE);
          scores[i].value = cbmemdup("", 0);
        }
        if(ci && cbmaprnum(db->spacc) > db->scmnum){
          num = db->scmnum * 0.1 + 1;
          cbmapiterinit(db->spacc);
          for(j = 0; j < num && (ibuf = cbmapiternext(db->spacc, &isiz)) != NULL; j++){
            cbmapout(db->spacc, ibuf, isiz);
          }
        }
        free(mbuf);
      } else {
        scores[i].value = cbmemdup("", 0);
      }
    }
    if(!cbstricmp(otype, ESTORDSTRA)){
      qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_str_asc);
    } else if(!cbstricmp(otype, ESTORDSTRD)){
      qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_str_desc);
    } else if(!cbstricmp(otype, ESTORDNUMA)){
      for(i = 0; i < snum; i++){
        tval = cbstrmktime(scores[i].value);
        free(scores[i].value);
        scores[i].value = (void *)tval;
      }
      qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_num_asc);
      for(i = 0; i < snum; i++){
        scores[i].value = NULL;
      }
    } else if(!cbstricmp(otype, ESTORDNUMD)){
      for(i = 0; i < snum; i++){
        tval = cbstrmktime(scores[i].value);
        free(scores[i].value);
        scores[i].value = (void *)tval;
      }
      qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_num_desc);
      for(i = 0; i < snum; i++){
        scores[i].value = NULL;
      }
    }
    for(i = 0; i < snum; i++){
      free(scores[i].value);
    }
    free(oname);
  }
  return snum;
}


/* Narrow and sort scores of search candidates.
   `db' specifies a database object.
   `scores' specifies an array of scores of search candidates.
   `snum' specifies the number of the array.
   `num' specifies the number of documents to be shown.
   `max' specifies the maximum number of shown documents.
   `vnum' specifies the number of dimensions of the vector.
   `tfidf' specifies whether to perform TF-IDF tuning.
   `limit' specifies the upper limit of similarity for documents to survive.
   `shadows' specifies a map object to store shadow document information.
   The return value is the new number of the array. */
static int est_eclipse_scores(ESTDB *db, ESTSCORE *scores, int snum, int num,
                              int vnum, int tfidf, double limit, CBMAP *shadows){
  CBMAP *svmap, *tvmap;
  int i, j, max, *svec, *tvec, pair[2], nnum;
  double dval;
  assert(db && scores && snum >= 0 && num >= 0 && vnum > 0 && limit > 0.0 && shadows);
  max = limit < 0.1 ? snum : num * ((2.0 / limit) + 0.5);
  if(max > snum) max = snum;
  CB_MALLOC(svec, vnum * sizeof(int));
  CB_MALLOC(tvec, vnum * sizeof(int));
  for(i = 0; i < max; i++){
    svmap = est_get_tvmap(db, scores[i].id, vnum, tfidf);
    scores[i].value = (char *)svmap;
  }
  for(i = 0; i < max; i++){
    svmap = (CBMAP *)(scores[i].value);
    if(!svmap || cbmaprnum(svmap) < 1) continue;
    if(num-- < 1) continue;
    est_vector_set_seed(svmap, svec, vnum);
    for(j = i + 1; j < max; j++){
      tvmap = (CBMAP *)(scores[j].value);
      if(!tvmap || cbmaprnum(tvmap) < 1) continue;
      est_vector_set_target(svmap, tvmap, tvec, vnum);
      dval = est_vector_cosine(svec, tvec, vnum);
      if(dval > limit){
        cbmapclose(tvmap);
        scores[j].value = NULL;
        pair[0] = scores[j].id;
        pair[1] = (int)(dval * 10000.0);
        cbmapputcat(shadows, (char *)&(scores[i].id), sizeof(int),
                    (char *)pair, sizeof(int) * 2);
      }
    }
  }
  nnum = 0;
  for(i = 0; i < max; i++){
    if(scores[i].value){
      cbmapclose((CBMAP *)(scores[i].value));
      scores[nnum++] = scores[i];
    }
  }
  for(i = max; i < snum; i++){
    scores[nnum++] = scores[i];
  }
  free(tvec);
  free(svec);
  return nnum;
}


/* Check whether a score matches an attribute condition.
   `tval' specifies the target value;
   `tsiz' specifies the size of the target value
   `cop' specifies the pointer to the operator.
   `sign' specifies the sign of operation.
   `oval' specifies the operation value;
   `osiz' specifies the size of the operation value
   `sval' specifies the operation value of small cases;
   `ssiz' specifies the size of the operation value of small cases.
   `regex' specifies the regular expressions.
   `onum' specifies the numeric value.
   The return value is true if it does match, else it is false. */
static int est_match_attr(const char *tval, int tsiz, const char *cop, int sign,
                          const char *oval, int osiz, const char *sval, int ssiz,
                          const void *regex, int onum){
  unsigned char *eval;
  char *cval;
  int csiz, esiz, hit;
  assert(tval && tsiz >= 0 && oval && osiz >= 0);
  cval = NULL;
  if(sval){
    eval = (unsigned char *)est_uconv_in(tval, tsiz, &esiz);
    est_normalize_text(eval, esiz, &esiz);
    est_canonicalize_text(eval, esiz, FALSE);
    cval = (char *)est_uconv_out((char *)eval, esiz, &csiz);
    free(eval);
    tval = cval;
    tsiz = csiz;
    oval = sval;
    osiz = ssiz;
  }
  if(cop == ESTOPSTREQ){
    hit = !strcmp(tval, oval);
  } else if(cop == ESTOPSTRNE){
    hit = strcmp(tval, oval) != 0;
  } else if(cop == ESTOPSTRINC){
    hit = strstr(tval, oval) != NULL;
  } else if(cop == ESTOPSTRBW){
    hit = cbstrfwmatch(tval, oval);
  } else if(cop == ESTOPSTREW){
    hit = cbstrbwmatch(tval, oval);
  } else if(cop == ESTOPSTRAND){
    hit = est_check_strand(tval, oval);
  } else if(cop == ESTOPSTROR){
    hit = est_check_stror(tval, oval);
  } else if(cop == ESTOPSTRRX){
    hit = regex ? est_regex_match(regex, tval) : FALSE;
  } else if(cop == ESTOPNUMEQ){
    hit = cbstrmktime(tval) == onum;
  } else if(cop == ESTOPNUMNE){
    hit = cbstrmktime(tval) != onum;
  } else if(cop == ESTOPNUMGT){
    hit = cbstrmktime(tval) > onum;
  } else if(cop == ESTOPNUMGE){
    hit = cbstrmktime(tval) >= onum;
  } else if(cop == ESTOPNUMLT){
    hit = cbstrmktime(tval) < onum;
  } else if(cop == ESTOPNUMLE){
    hit = cbstrmktime(tval) <= onum;
  } else if(cop == ESTOPNUMBT){
    hit = est_check_numbt(tval, oval);
  } else {
    hit = FALSE;
  }
  free(cval);
  return sign ? hit : !hit;
}


/* Check whether a string includes all tokens in another string.
   `tval' specifies the target value;
   `oval' specifies the operation value;
   The return value is the result of the check. */
static int est_check_strand(const char *tval, const char *oval){
  const char *sp, *ep, *rp, *pp, *qp;
  int hit;
  assert(tval && oval);
  sp = oval;
  while(*sp != '\0'){
    while(*sp == ' '){
      sp++;
    }
    ep = sp;
    while(*ep != '\0' && *ep != ' '){
      ep++;
    }
    if(ep > sp){
      hit = FALSE;
      for(rp = tval; *rp != '\0'; rp++){
        for(pp = sp, qp = rp; pp < ep; pp++, qp++){
          if(*pp != *qp) break;
        }
        if(pp == ep){
          hit = TRUE;
          break;
        }
      }
      if(!hit) return FALSE;
    }
    sp = ep;
  }
  return TRUE;
}


/* Check whether a string includes at least one token in another string.
   `tval' specifies the target value;
   `oval' specifies the operation value;
   The return value is the result of the check. */
static int est_check_stror(const char *tval, const char *oval){
  const char *sp, *ep, *rp, *pp, *qp;
  int hit;
  assert(tval && oval);
  sp = oval;
  while(*sp != '\0'){
    while(*sp == ' '){
      sp++;
    }
    ep = sp;
    while(*ep != '\0' && *ep != ' '){
      ep++;
    }
    if(ep > sp){
      hit = FALSE;
      for(rp = tval; *rp != '\0'; rp++){
        for(pp = sp, qp = rp; pp < ep; pp++, qp++){
          if(*pp != *qp) break;
        }
        if(pp == ep){
          hit = TRUE;
          break;
        }
      }
      if(hit) return TRUE;
    }
    sp = ep;
  }
  return FALSE;
}


/* Check whether a decimal string is between two tokens in another string.
   `tval' specifies the target value;
   `oval' specifies the operation value;
   The return value is the result of the check. */
static int est_check_numbt(const char *tval, const char *oval){
  time_t val, lower, upper, swap;
  char numbuf[ESTNUMBUFSIZ];
  int i;
  for(i = 0; i < ESTNUMBUFSIZ && oval[i] != '\0' && oval[i] != ' ' && oval[i] != '\t'; i++){
    numbuf[i] = oval[i];
  }
  numbuf[i] = '\0';
  oval += i;
  while(*oval == ' ' || *oval == '\t'){
    oval++;
  }
  if(*oval == '\0') return FALSE;
  val = cbstrmktime(tval);
  lower = cbstrmktime(numbuf);
  upper = cbstrmktime(oval);
  if(lower > upper){
    swap = lower;
    lower = upper;
    upper = swap;
  }
  return val >= lower && val <= upper;
}


/* Compare two keywords by scores in descending order.
   `ap' specifies the pointer to one keyword.
   `bp' specifies the pointer to the other keyword.
   The return value is negative if one is small, positive if one is big, 0 if both are equal. */
static int est_keysc_compare(const void *ap, const void *bp){
  assert(ap && bp);
  return ((ESTKEYSC *)bp)->pt - ((ESTKEYSC *)ap)->pt;
}


/* Get a similar set of documents in a database.
   `db' specifies a database object.
   `svmap' specifies a map object of a seed vector.
   `nump' specifies the pointer to which the number of elements in the result is assigned.
   `knum' specifies the number of keywords to get candidates.
   `unum' specifies the number of adopted documents for a keyword.
   `tfidf' specifies whether to perform TF-IDF tuning.
   `nmin' specifies the minimum value for narrowing.
   The return value is an array whose elements are ID numbers of similar documents. */
static ESTSCORE *est_search_similar(ESTDB *db, CBMAP *svmap, int *nump,
                                    int knum, int unum, int tfidf, double nmin){
  ESTSCORE *scores, *tscores;
  CBMAP *tvmap;
  const char *word;
  int i, j, vnum, snum, tmax, tsnum, nnum, lid, *svec, *tvec;
  double dval;
  assert(db && svmap && nump && knum >= 0 && unum >= 0 && nmin >= 0.0);
  CB_MALLOC(scores, sizeof(ESTSCORE) * unum * knum);
  snum = 0;
  if((vnum = cbmaprnum(svmap)) < 1) vnum = 1;
  cbmapiterinit(svmap);
  tmax = unum;
  for(i = 0; i < knum && (word = cbmapiternext(svmap, NULL)) != NULL; i++){
    tscores = est_search_union(db, word, 1, &tsnum, NULL, TRUE);
    qsort(tscores, tsnum, sizeof(ESTSCORE), est_score_compare_by_score);
    for(j = 0; j < tmax && j < tsnum; j++){
      scores[snum].id = tscores[j].id;
      scores[snum].score = tscores[j].score;
      snum++;
    }
    free(tscores);
    tmax -= unum / knum / 1.25;
  }
  qsort(scores, snum, sizeof(ESTSCORE), est_score_compare_by_id);
  nnum = 0;
  lid = -1;
  CB_MALLOC(svec, vnum * sizeof(int));
  CB_MALLOC(tvec, vnum * sizeof(int));
  est_vector_set_seed(svmap, svec, vnum);
  for(i = 0; i < snum; i++){
    if(scores[i].id != lid){
      tvmap = est_get_tvmap(db, scores[i].id, vnum, tfidf);
      if(tvmap){
        est_vector_set_target(svmap, tvmap, tvec, vnum);
        if((dval = est_vector_cosine(svec, tvec, vnum)) >= nmin){
          scores[nnum].id = scores[i].id;
          scores[nnum].score = (int)(dval * 10000);
          if(scores[nnum].score == 9999) scores[nnum].score = 10000;
          nnum++;
        }
        cbmapclose(tvmap);
      }
    }
    lid = scores[i].id;
  }
  free(tvec);
  free(svec);
  snum = nnum;
  *nump = snum;
  return scores;
}


/* Create a map object of a vector for similar search from a phrase.
   `phrase' specifies a search phrase for similar search.
   The return value is a map object of the seed vector. */
static CBMAP *est_phrase_vector(const char *phrase){
  CBMAP *svmap;
  CBLIST *list;
  const char *pv, *rp;
  char *utext, *rtext;
  int i, num, len, size;
  svmap = cbmapopenex(ESTMINIBNUM);
  list = cblistopen();
  while(*phrase != '\0'){
    if(*phrase == ESTOPWITH[0] && cbstrfwmatch(phrase, ESTOPWITH)){
      phrase += strlen(ESTOPWITH);
      pv = phrase;
      while(*phrase != '\0'){
        if(*phrase <= ' ' && cbstrfwmatch(phrase + 1, ESTOPWITH)){
          phrase++;
          break;
        }
        phrase++;
      }
      cblistpush(list, pv, phrase - pv);
    } else {
      phrase++;
    }
  }
  for(i = 0; i < CB_LISTNUM(list); i++){
    pv = CB_LISTVAL(list, i, NULL);
    while(*pv > '\0' && *pv <= ' '){
      pv++;
    }
    num = strtol(pv, (char **)&rp, 10);
    if(rp && (len = rp - pv) > 0 && num >= 0){
      utext = est_uconv_in(rp, strlen(rp), &size);
      est_normalize_text((unsigned char *)utext, size, &size);
      est_canonicalize_text((unsigned char *)utext, size, FALSE);
      rtext = est_uconv_out(utext, size, NULL);
      cbstrsqzspc(rtext);
      if(rtext[0] != '\0') cbmapput(svmap, rtext, -1, pv, len, FALSE);
      free(rtext);
      free(utext);
    }
  }
  cblistclose(list);
  return svmap;
}


/* Get the target vector of a document dynamically.
   `db' specifies a database object.
   `id' specifies the ID of a document.
   `vnum' specifies the number of dimensions of the vector.
   `tfidf' specifies whether to perform TF-IDF tuning.
   The return value is a map object of the target vector. */
static CBMAP *est_get_tvmap(ESTDB *db, int id, int vnum, int tfidf){
  ESTDOC *doc;
  CBMAP *tvmap;
  assert(db && id > 0);
  if((tvmap = est_db_get_keywords(db, id)) != NULL) return tvmap;
  if(!(doc = est_db_get_doc(db, id, 0))) return NULL;
  tvmap = est_db_etch_doc(tfidf ? db : NULL, doc, vnum);
  est_doc_delete(doc);
  if(dpwritable(db->metadb)) est_db_put_keywords(db, id, tvmap);
  return tvmap;
}


/* Close the handle to the file of random number generator. */
static void est_random_fclose(void){
  if(est_random_ifp) fclose(est_random_ifp);
}


/* Dispatch a signal to the corresponding handler.
   Signum specifies the number of catched signal. */
static int est_signal_dispatch(int signum){
#if defined(_SYS_MSVC_) || defined(_SYS_MINGW_)
  switch(signum){
  case CTRL_C_EVENT: case CTRL_BREAK_EVENT: case CTRL_CLOSE_EVENT:
    signum = 2;
    break;
  case CTRL_LOGOFF_EVENT: case CTRL_SHUTDOWN_EVENT:
    signum = 15;
    break;
  }
  if(est_signal_handlers[signum]) est_signal_handlers[signum](signum);
  return TRUE;
#else
  assert(signum >= 0);
  if(est_signal_handlers[signum]) est_signal_handlers[signum](signum);
  return TRUE;
#endif
}



/* END OF FILE */
