#!/usr/bin/perl -w

#    mkpatch - Create patches against the Linux kernel
#    Copyright (c) 1999  Frodo Looijaard <frodol@dds.nl>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use strict;

use vars qw($temp);
$temp = "mkpatch/.temp";

# Generate a diff between the old kernel file and the new i2c file. We
# arrange the headers to tell us the old tree was under directory
# `linux-old', and the new tree under `linux'.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
# $_[2]: Name of the kernel file
# $_[3]: Name of the patched file
sub print_diff
{
  my ($package_root,$kernel_root,$kernel_file,$package_file) = @_;
  my ($diff_command,$package_mtime,$kernel_mtime);

  $diff_command = "diff -u";
  if ( -e "$kernel_root/$kernel_file") {
    $diff_command .= " $kernel_root/$kernel_file";
    $kernel_mtime = (stat("$kernel_root/$kernel_file"))[9];
  } else {
    $diff_command .= " /dev/null";
    $kernel_mtime = 0;
  }
  if ( -e "$package_root/$package_file") {
    $diff_command .= " $package_root/$package_file";
    $package_mtime = (stat("$package_root/$package_file"))[9];
  } else {
    $diff_command .= " /dev/null";
    $package_mtime = 0;
  }
  open INPUT, "$diff_command|" or die "Can't execute `$diff_command'";
  if (<INPUT>) {
    <INPUT>;
    print "--- linux-old/$kernel_file\t".gmtime($kernel_mtime)."\n".
          "+++ linux/$kernel_file\t".gmtime($package_mtime)."\n";

    print while <INPUT>;
  }
  close INPUT;
}

# Find all the lm_sensors code in a file
# $_[0]: Linux kernel tree (like /usr/src/linux)
# $_[1]: Name of the kernel file
# Returns a list of strings with the sensors codes
sub find_sensors_code
{
  my ($kernel_root,$kernel_file) = @_;
  my @res;
  open INPUT, "$kernel_root/$kernel_file" 
       or return @res;
  while (<INPUT>) {
    if (m@sensors code starts here@) {
      push @res,"";
      while (<INPUT>) {
        last if m@sensors code ends here@;
        $res[$#res] .= $_;
      }
    }
  }
  return @res;    
} 

# This generates diffs for kernel file Documentation/Configure.help. This
# file contains the help texts that can be displayed during `make *config'
# for the kernel.
# The generated diff will replace the whole I2C section with the one we
# have in our (partial) Configure.help file.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_Documentation_Configure_help
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "Documentation/Configure.help";
  my $package_file = "mkpatch/Configure.help";

# First, we generate a temporary file which is the same as the kernel's
# Configure.help file, except that we replace the I2C section (starts with
# "I2C support" included and ends with "Bus Mouse Support" excluded).

  open INPUT,"$kernel_root/$kernel_file"
        or die "Can't open `$kernel_root/$kernel_file'";
  open INPUTI,"$package_root/$package_file"
        or die "Can't open `$package_root/$package_file'";
  open OUTPUT,">$package_root/$temp"
        or die "Can't open `$package_root/$temp'";

  while(<INPUT>) {
    unless (m/^I2C support$/) {
      print OUTPUT;
      next;
    }
    print OUTPUT while <INPUTI>;
    while(<INPUT>) {
      if (m/^Bus Mouse Support$/) {
        print OUTPUT;
        last;
      }
    }
  }
   
  close INPUT;
  close OUTPUT;
  close INPUTI;

# Now we can generate the diff between the original Configure.help
# and the temporary file.
  print_diff $package_root,$kernel_root,$kernel_file,$temp;
}

# This generates diffs for drivers/i2c/Makefile
# for either 'old style' or 'new style'.
# Don't bother putting in 'old style' support for new architectures
# like 8xx, IBM405, or StrongARM since they aren't supported
# in old kernels anyway.
sub gen_drivers_i2c_Makefile
{
  my ($package_root,$kernel_root) = @_;
  my $kernel_file = "drivers/i2c/Makefile";
  my $package_file = $temp;
  my $use_new_format = 0;
  if (-e "$kernel_root/$kernel_file") {
    `grep -q -s 'i2c\.o' "$kernel_root/$kernel_file"`;
     $use_new_format = ! $?;
  }

  open OUTPUT,">$package_root/$package_file"
        or die "Can't open $package_root/$package_file";
  if ($use_new_format) {
    print OUTPUT <<'EOF';
#
# Makefile for the kernel i2c bus driver.
#

O_TARGET := i2c.o

export-objs	:= i2c-core.o i2c-algo-bit.o i2c-algo-pcf.o \
                   i2c-algo-8xx.o i2c-proc.o i2c-algo-ibm_ocp.o

obj-$(CONFIG_I2C)		+= i2c-core.o
obj-$(CONFIG_I2C_CHARDEV)	+= i2c-dev.o
obj-$(CONFIG_I2C_ALGOBIT)	+= i2c-algo-bit.o
obj-$(CONFIG_I2C_PHILIPSPAR)	+= i2c-philips-par.o
obj-$(CONFIG_I2C_ELV)		+= i2c-elv.o
obj-$(CONFIG_I2C_VELLEMAN)	+= i2c-velleman.o
obj-$(CONFIG_I2C_PPORT)		+= i2c-pport.o
obj-$(CONFIG_I2C_ALGOPCF)	+= i2c-algo-pcf.o
obj-$(CONFIG_I2C_ELEKTOR)	+= i2c-elektor.o
obj-$(CONFIG_I2C_PCFEPP)	+= i2c-pcf-epp.o
obj-$(CONFIG_I2C_PROC)		+= i2c-proc.o
obj-$(CONFIG_I2C_ALGO8XX)	+= i2c-algo-8xx.o
obj-$(CONFIG_I2C_RPXLITE)	+= i2c-rpx.o
obj-$(CONFIG_I2C_IBM_OCP_ALGO)	+= i2c-algo-ibm_ocp.o
obj-$(CONFIG_I2C_IBM_OCP_ADAP)	+= i2c-adap-ibm_ocp.o
obj-$(CONFIG_I2C_FRODO)		+= i2c-frodo.o

# This is needed for automatic patch generation: sensors code starts here
# This is needed for automatic patch generation: sensors code ends here

include $(TOPDIR)/Rules.make

EOF
  } else {
    print OUTPUT <<'EOF';
#
# Makefile for the kernel i2c bus driver.
#

SUB_DIRS     :=
MOD_SUB_DIRS := $(SUB_DIRS)
ALL_SUB_DIRS := $(SUB_DIRS)
MOD_LIST_NAME := I2C_MODULES

L_TARGET := i2c.a
MX_OBJS :=  
M_OBJS  := 
LX_OBJS :=
L_OBJS  := 

# -----
# i2c core components
# -----

ifeq ($(CONFIG_I2C),y)
  LX_OBJS += i2c-core.o
else
  ifeq ($(CONFIG_I2C),m)
    MX_OBJS += i2c-core.o
  endif
endif

ifeq ($(CONFIG_I2C_CHARDEV),y)
  L_OBJS += i2c-dev.o
else
  ifeq ($(CONFIG_I2C_CHARDEV),m)
    M_OBJS += i2c-dev.o
  endif
endif

ifeq ($(CONFIG_I2C_PROC),y)
  LX_OBJS += i2c-proc.o
else
  ifeq ($(CONFIG_I2C_PROC),m)
    MX_OBJS += i2c-proc.o
  endif
endif

# -----
# Bit banging adapters...
# -----

ifeq ($(CONFIG_I2C_ALGOBIT),y)
  LX_OBJS += i2c-algo-bit.o
else
  ifeq ($(CONFIG_I2C_ALGOBIT),m)
    MX_OBJS += i2c-algo-bit.o
  endif
endif

ifeq ($(CONFIG_I2C_PHILIPSPAR),y)
  L_OBJS += i2c-philips-par.o
else
  ifeq ($(CONFIG_I2C_PHILIPSPAR),m)
    M_OBJS += i2c-philips-par.o
  endif
endif

ifeq ($(CONFIG_I2C_ELV),y)
  L_OBJS += i2c-elv.o
else
  ifeq ($(CONFIG_I2C_ELV),m)
    M_OBJS += i2c-elv.o
  endif
endif

ifeq ($(CONFIG_I2C_VELLEMAN),y)
  L_OBJS += i2c-velleman.o
else
  ifeq ($(CONFIG_I2C_VELLEMAN),m)
    M_OBJS += i2c-velleman.o
  endif
endif



# -----
# PCF components
# -----

ifeq ($(CONFIG_I2C_ALGOPCF),y)
  LX_OBJS += i2c-algo-pcf.o
else
  ifeq ($(CONFIG_I2C_ALGOPCF),m)
    MX_OBJS += i2c-algo-pcf.o
  endif
endif

ifeq ($(CONFIG_I2C_ELEKTOR),y)
  L_OBJS += i2c-elektor.o
else
  ifeq ($(CONFIG_I2C_ELEKTOR),m)
    M_OBJS += i2c-elektor.o
  endif
endif

# This is needed for automatic patch generation: sensors code starts here
# This is needed for automatic patch generation: sensors code ends here

include $(TOPDIR)/Rules.make

EOF
  }
  close OUTPUT;
  print_diff $package_root,$kernel_root,$kernel_file,$package_file;
}
 
# This generates diffs for kernel drivers/i2c/ directory, except Config.in
# and Makefile.
# The lists of files to be handled is read in FILES, the list of headers to
# be translated is read in INCLUDES.
# $_[0]: i2c package root (like /tmp/i2c)
# $_[1]: Linux kernel tree (like /usr/src/linux)
sub gen_drivers_i2c_core
{
  my ($package_root,$kernel_root) = @_;
  my (%files,%includes,$package_file,$kernel_file);
  my ($dummy,$data0,$data1,$sedscript,@sensors_subs);

  # --> Read FILES
  open INPUT, "$package_root/mkpatch/FILES" 
        or die "Can't open `$package_root/mkpatch/FILES'";
  while (<INPUT>) {
    ($data0,$data1) = /(\S+)\s+(\S+)/;
    $files{$data0} = $data1;
  } 
  close INPUT;

  # --> Read INCLUDES
  open INPUT, "$package_root/mkpatch/INCLUDES" 
        or die "Can't open `$package_root/mkpatch/INCLUDES'";
  while (<INPUT>) {
    ($data0,$data1) = /(\S+)\s+(\S+)/;
    $includes{$data0} = $data1;
    $sedscript .= 's,(#\s*include\s*)'.$data0.'(\s*),$1'."$data1".'$2, ; ';
  } 
  close INPUT;

  # --> Start generating
  foreach $package_file (sort keys %files) {
    $kernel_file = $files{$package_file};
    @sensors_subs = find_sensors_code "$kernel_root","$kernel_file";
    open INPUT, "$package_root/$package_file"
      or open INPUT, "/dev/null"
         or die "Can't open `$package_root/$package_file'";
    open OUTPUT, ">$package_root/$temp"
         or die "Can't open `$package_root/$temp'";
    while (<INPUT>) {
      eval $sedscript;
      if (m@sensors code starts here@) {
        print OUTPUT;
        while (<INPUT>) {
           last if m@sensors code ends here@;
        }
        print OUTPUT $sensors_subs[0];
        shift @sensors_subs
      }
      print OUTPUT;
    }
    close INPUT;
    close OUTPUT;
    print_diff "$package_root","$kernel_root","$kernel_file","$temp";
  }
}

sub usage
{
  print "Usage: $0 package_root kernel_root\n";
  exit 1;
}

# Read the command line
my $package_root = $ARGV[0];
my $kernel_root = $ARGV[1];

usage()
  unless -d "$package_root/mkpatch"
  and -f "$kernel_root/Rules.make";

gen_Documentation_Configure_help $package_root, $kernel_root;
#gen_drivers_i2c_Makefile $package_root, $kernel_root;
gen_drivers_i2c_core $package_root, $kernel_root;

# Clear temporary file
unlink("$package_root/$temp");
