#define ENABLE_V17
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t31.c - A T.31 compatible class 1 FAX modem interface.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Special thanks to Lee Howard <faxguy@howardsilvan.com>
 * for his great work debugging and polishing this code.
 *
 * Copyright (C) 2004, 2005, 2006 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t31.c,v 1.56 2006/05/25 04:45:44 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <memory.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <assert.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/dc_restore.h"
#include "spandsp/queue.h"
#include "spandsp/power_meter.h"
#include "spandsp/complex.h"
#include "spandsp/tone_generate.h"
#include "spandsp/async.h"
#include "spandsp/hdlc.h"
#include "spandsp/fsk.h"
#include "spandsp/v29rx.h"
#include "spandsp/v29tx.h"
#include "spandsp/v27ter_rx.h"
#include "spandsp/v27ter_tx.h"
#if defined(ENABLE_V17)
#include "spandsp/v17rx.h"
#include "spandsp/v17tx.h"
#endif

#include "spandsp/at_interpreter.h"
#include "spandsp/t31.h"

#define ms_to_samples(t)    (((t)*SAMPLE_RATE)/1000)

typedef const char *(*at_cmd_service_t)(t31_state_t *s, const char *cmd);

#define ETX 0x03
#define DLE 0x10
#define SUB 0x1A

enum
{
    T31_FLUSH,
    T31_SILENCE,
    T31_CED_TONE,
    T31_CNG_TONE,
    T31_NOCNG_TONE,
    T31_V21_TX,
    T31_V17_TX,
    T31_V27TER_TX,
    T31_V29_TX,
    T31_V21_RX,
    T31_V17_RX,
    T31_V27TER_RX,
    T31_V29_RX
};

static int restart_modem(t31_state_t *s, int new_modem);

static int t31_modem_control_handler(at_state_t *s, void *user_data, int op, const char *num)
{
    t31_state_t *t;
    
    t = (t31_state_t *) user_data;
    switch (op)
    {
    case AT_MODEM_CONTROL_ANSWER:
        t->call_samples = 0;
        break;
    case AT_MODEM_CONTROL_CALL:
        t->call_samples = 0;
        break;
    case AT_MODEM_CONTROL_RESTART:
        restart_modem(t, (intptr_t) num);
        return 0;
    }
    return t->modem_control_handler(t, t->modem_control_user_data, op, num);
}
/*- End of function --------------------------------------------------------*/

static void fast_putbit(void *user_data, int bit)
{
    t31_state_t *s;

    s = (t31_state_t *) user_data;
    if (bit < 0)
    {
        /* Special conditions */
        switch (bit)
        {
        case PUTBIT_TRAINING_FAILED:
            break;
        case PUTBIT_TRAINING_SUCCEEDED:
            /* The modem is now trained */
            at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
            s->at_state.rx_signal_present = TRUE;
            break;
        case PUTBIT_CARRIER_UP:
            break;
        case PUTBIT_CARRIER_DOWN:
            if (s->at_state.rx_signal_present)
            {
                s->at_state.rx_data[s->at_state.rx_data_bytes++] = DLE;
                s->at_state.rx_data[s->at_state.rx_data_bytes++] = ETX;
                s->at_state.at_tx_handler(&s->at_state, s->at_state.at_tx_user_data, s->at_state.rx_data, s->at_state.rx_data_bytes);
                s->at_state.rx_data_bytes = 0;
                at_put_response_code(&s->at_state, AT_RESPONSE_CODE_NO_CARRIER);
                s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
            }
            s->at_state.rx_signal_present = FALSE;
            break;
        default:
            if (s->at_state.p.result_code_format)
                span_log(&s->logging, SPAN_LOG_FLOW, "Eh!\n");
            break;
        }
        return;
    }
    s->current_byte = (s->current_byte >> 1) | (bit << 7);
    if (++s->bit_no >= 8)
    {
        if (s->current_byte == DLE)
            s->at_state.rx_data[s->at_state.rx_data_bytes++] = (uint8_t) s->current_byte;
        s->at_state.rx_data[s->at_state.rx_data_bytes++] = (uint8_t) s->current_byte;
        if (s->at_state.rx_data_bytes >= 250)
        {
            s->at_state.at_tx_handler(&s->at_state, s->at_state.at_tx_user_data, s->at_state.rx_data, s->at_state.rx_data_bytes);
            s->at_state.rx_data_bytes = 0;
        }
        s->bit_no = 0;
        s->current_byte = 0;
    }
}
/*- End of function --------------------------------------------------------*/

static int fast_getbit(void *user_data)
{
    t31_state_t *s;
    int bit;

    s = (t31_state_t *) user_data;
    if (s->bit_no <= 0)
    {
        if (s->tx_out_bytes != s->tx_in_bytes)
        {
            s->current_byte = s->tx_data[s->tx_out_bytes++];
            if (s->tx_out_bytes > T31_TX_BUF_LEN - 1)
            {
                s->tx_out_bytes = T31_TX_BUF_LEN - 1;
                fprintf(stderr, "End of transmit buffer reached!\n");
            }
            if (s->tx_holding)
            {
                /* See if the buffer is approaching empty. It might be time to release flow control. */
                if (s->tx_out_bytes > 1024)
                {
                    s->tx_holding = FALSE;
                    /* Tell the application to release further data */
                    at_modem_control(&s->at_state, AT_MODEM_CONTROL_CTS, (void *) 1);
                }
            }
        }
        else
        {
            if (s->data_final)
            {
                s->data_final = FALSE;
                /* This will put the modem into its shutdown sequence. When
                   it has finally shut down, an OK response will be sent. */
                return 3;
            }
            /* Fill with 0xFF bytes. This is appropriate at the start of transmission,
               as per T.30. If it happens in the middle of data, it is bad. What else
               can be done, though? */
            s->current_byte = 0xFF;
        }
        s->bit_no = 8;
    }
    s->bit_no--;
    bit = s->current_byte & 1;
    s->current_byte >>= 1;
    return bit;
}
/*- End of function --------------------------------------------------------*/

static void hdlc_tx_underflow(void *user_data)
{
    t31_state_t *s;

    s = (t31_state_t *) user_data;
    if (s->hdlc_final)
    {
        at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
        s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
        s->hdlc_final = FALSE;
        restart_modem(s, T31_SILENCE);
    }
    else
    {
        s->last_dtedata_samples = s->call_samples;
        at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
    }
}
/*- End of function --------------------------------------------------------*/

static void hdlc_accept(void *user_data, int ok, const uint8_t *msg, int len)
{
    uint8_t buf[256];
    t31_state_t *s;
    int i;

    s = (t31_state_t *) user_data;
    if (len < 0)
    {
        /* Special conditions */
        switch (len)
        {
        case PUTBIT_CARRIER_UP:
            if (s->modem == T31_CNG_TONE  ||  s->modem == T31_NOCNG_TONE  ||  s->modem == T31_V21_RX)
            {
                s->at_state.rx_signal_present = TRUE;
                s->rx_message_received = FALSE;
            }
            break;
        case PUTBIT_CARRIER_DOWN:
            if (s->rx_message_received)
            {
                if (s->at_state.dte_is_waiting)
                {
                    if (s->at_state.ok_is_pending)
                    {
                        at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
                        s->at_state.ok_is_pending = FALSE;
                    }
                    else
                    {
                        at_put_response_code(&s->at_state, AT_RESPONSE_CODE_NO_CARRIER);
                    }
                    s->at_state.dte_is_waiting = FALSE;
                }
                else
                {
                    buf[0] = AT_RESPONSE_CODE_NO_CARRIER;
                    queue_write_msg(&(s->rx_queue), buf, 1);
                }
            }
            s->at_state.rx_signal_present = FALSE;
            break;
        case PUTBIT_FRAMING_OK:
            if (s->modem == T31_CNG_TONE  ||  s->modem == T31_NOCNG_TONE)
            {
                /* Once we get any valid HDLC the CNG tone stops, and we drop
                   to the V.21 receive modem on its own. */
                s->modem = T31_V21_RX;
                s->at_state.transmit = FALSE;
            }
            if (s->modem != T31_V21_RX)
            {
                /* V.21 has been detected while expecting a different carrier.
                   If +FAR=0 then result +FCERROR and return to command-mode.
                   If +FAR=1 then report +FRH:3 and CONNECT, switching to
                   V.21 receive mode. */
                if (s->at_state.p.adaptive_receive)
                {
                    s->at_state.rx_signal_present = TRUE;
                    s->rx_message_received = TRUE;
                    s->modem = T31_V21_RX;
                    s->at_state.transmit = FALSE;
                    s->at_state.dte_is_waiting = TRUE;
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_FRH3);
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
                }
                else
                {
                    s->modem = T31_SILENCE;
                    s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
                    s->rx_message_received = FALSE;
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_FCERROR);
                }
            }
            else
            {
                if (!s->rx_message_received)
                {
                    /* Report CONNECT as soon as possible to avoid a timeout. */
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
                    s->rx_message_received = TRUE;
                }
            }
            break;
        case PUTBIT_ABORT:
            /* Just ignore these */
            break;
        default:
            span_log(&s->logging, SPAN_LOG_WARNING, "Unexpected HDLC special length - %d!\n", len);
            break;
        }
        return;
    }
    if (s->at_state.dte_is_waiting)
    {
        /* Send straight away */
        /* It is safe to look at the two bytes beyond the length of the message,
           and expect to find the FCS there. */
        for (i = 0;  i < len + 2;  i++)
        {
            if (msg[i] == DLE)
                s->at_state.rx_data[s->at_state.rx_data_bytes++] = DLE;
            s->at_state.rx_data[s->at_state.rx_data_bytes++] = msg[i];
        }
        s->at_state.rx_data[s->at_state.rx_data_bytes++] = DLE;
        s->at_state.rx_data[s->at_state.rx_data_bytes++] = ETX;
        s->at_state.at_tx_handler(&s->at_state, s->at_state.at_tx_user_data, s->at_state.rx_data, s->at_state.rx_data_bytes);
        s->at_state.rx_data_bytes = 0;
        if (msg[1] == 0x13  &&  ok)
        {
            /* This is the last frame.  We don't send OK until the carrier drops to avoid
               redetecting it later. */
            s->at_state.ok_is_pending = TRUE;
        }
        else
        {
            at_put_response_code(&s->at_state, (ok)  ?  AT_RESPONSE_CODE_OK  :  AT_RESPONSE_CODE_ERROR);
            s->at_state.dte_is_waiting = FALSE;
            s->rx_message_received = FALSE;
        }
    }
    else
    {
        /* Queue it */
        buf[0] = (ok)  ?  AT_RESPONSE_CODE_OK  :  AT_RESPONSE_CODE_ERROR;
        s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
        memcpy(buf + 1, msg, len);
        queue_write_msg(&(s->rx_queue), buf, len + 1);
    }
    s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
}
/*- End of function --------------------------------------------------------*/

static void t31_v21_rx(t31_state_t *s)
{
    hdlc_rx_init(&(s->hdlcrx), FALSE, TRUE, 5, hdlc_accept, s);
    s->at_state.ok_is_pending = FALSE;
    s->hdlc_final = FALSE;
    s->hdlc_len = 0;
    s->dled = FALSE;
    fsk_rx_init(&(s->v21rx), &preset_fsk_specs[FSK_V21CH2], TRUE, (put_bit_func_t) hdlc_rx_bit, &(s->hdlcrx));
    s->at_state.transmit = TRUE;
}
/*- End of function --------------------------------------------------------*/

static int restart_modem(t31_state_t *s, int new_modem)
{
    tone_gen_descriptor_t tone_desc;

    span_log(&s->logging, SPAN_LOG_FLOW, "Restart modem %d\n", new_modem);
    if (s->modem == new_modem)
        return 0;
    queue_flush(&(s->rx_queue));
    s->modem = new_modem;
    s->data_final = FALSE;
    s->at_state.rx_signal_present = FALSE;
    s->rx_message_received = FALSE;
    switch (s->modem)
    {
    case T31_CED_TONE:
        s->silent_samples += ms_to_samples(200);
        make_tone_gen_descriptor(&tone_desc,
                                 2100,
                                 -11,
                                 0,
                                 0,
                                 2600,
                                 75,
                                 0,
                                 0,
                                 FALSE);
        tone_gen_init(&(s->tone_gen), &tone_desc);
        s->at_state.transmit = TRUE;
        break;
    case T31_CNG_TONE:
        /* CNG is special, since we need to receive V.21 HDLC messages while sending the
           tone. Everything else in FAX processing sends only one way at a time. */
        /* 0.5s of 1100Hz + 3.0s of silence repeating */
        make_tone_gen_descriptor(&tone_desc,
                                 1100,
                                 -11,
                                 0,
                                 0,
                                 500,
                                 3000,
                                 0,
                                 0,
                                 TRUE);
        tone_gen_init(&(s->tone_gen), &tone_desc);
        /* Do V.21/HDLC receive in parallel. The other end may send its
           first message at any time. The CNG tone will continue until
           we get a valid preamble. */
        t31_v21_rx(s);
        s->at_state.transmit = TRUE;
        break;
    case T31_NOCNG_TONE:
        t31_v21_rx(s);
        s->at_state.transmit = FALSE;
        break;
    case T31_V21_TX:
        hdlc_tx_init(&(s->hdlctx), FALSE, hdlc_tx_underflow, s);
        hdlc_tx_preamble(&(s->hdlctx), 43);
        s->hdlc_final = FALSE;
        s->hdlc_len = 0;
        s->dled = FALSE;
        fsk_tx_init(&(s->v21tx), &preset_fsk_specs[FSK_V21CH2], (get_bit_func_t) hdlc_tx_getbit, &(s->hdlctx));
        s->at_state.transmit = TRUE;
        break;
    case T31_V21_RX:
        t31_v21_rx(s);
        break;
#if defined(ENABLE_V17)
    case T31_V17_TX:
        v17_tx_restart(&(s->v17tx), s->bit_rate, FALSE, s->short_train);
        s->tx_out_bytes = 0;
        s->at_state.transmit = TRUE;
        break;
    case T31_V17_RX:
        /* Allow for +FCERROR/+FRH:3 */
        t31_v21_rx(s);
        v17_rx_restart(&(s->v17rx), s->bit_rate, s->short_train);
        s->at_state.transmit = FALSE;
        break;
#endif
    case T31_V27TER_TX:
        v27ter_tx_restart(&(s->v27ter_tx), s->bit_rate, FALSE);
        s->tx_out_bytes = 0;
        s->at_state.transmit = TRUE;
        break;
    case T31_V27TER_RX:
        /* Allow for +FCERROR/+FRH:3 */
        t31_v21_rx(s);
        v27ter_rx_restart(&(s->v27ter_rx), s->bit_rate, FALSE);
        s->at_state.transmit = FALSE;
        break;
    case T31_V29_TX:
        v29_tx_restart(&(s->v29tx), s->bit_rate, FALSE);
        s->tx_out_bytes = 0;
        s->at_state.transmit = TRUE;
        break;
    case T31_V29_RX:
        /* Allow for +FCERROR/+FRH:3 */
        t31_v21_rx(s);
        v29_rx_restart(&(s->v29rx), s->bit_rate, FALSE);
        s->at_state.transmit = FALSE;
        break;
    case T31_SILENCE:
        s->at_state.transmit = FALSE;
        break;
    case T31_FLUSH:
        /* Send 200ms of silence to "push" the last audio out */
        s->modem = T31_SILENCE;
        s->silent_samples += ms_to_samples(200);
        s->at_state.transmit = TRUE;
        break;
    }
    s->bit_no = 0;
    s->current_byte = 0xFF;
    s->tx_in_bytes = 0;
    s->tx_out_bytes = 0;
    return 0;
}
/*- End of function --------------------------------------------------------*/

static __inline__ void dle_unstuff_hdlc(t31_state_t *s, const char *stuffed, int len)
{
    int i;

    for (i = 0;  i < len;  i++)
    {
        if (s->dled)
        {
            s->dled = FALSE;
            if (stuffed[i] == ETX)
            {
                hdlc_tx_preamble(&(s->hdlctx), 2);
                hdlc_tx_frame(&(s->hdlctx), s->hdlc_buf, s->hdlc_len);
                hdlc_tx_preamble(&(s->hdlctx), 2);
                s->hdlc_final = (s->hdlc_buf[1] & 0x10);
                s->hdlc_len = 0;
            }
            else if (stuffed[i] == SUB)
            {
                s->hdlc_buf[s->hdlc_len++] = DLE;
                s->hdlc_buf[s->hdlc_len++] = DLE;
            }
            else
            {
                s->hdlc_buf[s->hdlc_len++] = stuffed[i];
            }
        }
        else
        {
            if (stuffed[i] == DLE)
                s->dled = TRUE;
            else
                s->hdlc_buf[s->hdlc_len++] = stuffed[i];
        }
    }
}
/*- End of function --------------------------------------------------------*/

static __inline__ void dle_unstuff(t31_state_t *s, const char *stuffed, int len)
{
    int i;
    
    for (i = 0;  i < len;  i++)
    {
        if (s->dled)
        {
            s->dled = FALSE;
            if (stuffed[i] == ETX)
            {
                s->data_final = TRUE;
                s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
                return;
            }
        }
        else if (stuffed[i] == DLE)
        {
            s->dled = TRUE;
            continue;
        }
        s->tx_data[s->tx_in_bytes++] = stuffed[i];
        if (s->tx_in_bytes > T31_TX_BUF_LEN - 1)
        {
            fprintf(stderr, "No room in buffer for new data!\n");
            return;
        }
    }
    if (!s->tx_holding)
    {
        /* See if the buffer is approaching full. We might need to apply flow control. */
        if (s->tx_in_bytes > T31_TX_BUF_LEN - 1024)
        {
            s->tx_holding = TRUE;
            /* Tell the application to hold further data */
            at_modem_control(&s->at_state, AT_MODEM_CONTROL_CTS, (void *) 0);
        }
    }
}
/*- End of function --------------------------------------------------------*/

static int process_class1_cmd(at_state_t *t, void *user_data, int direction, int operation, int val)
{
    int new_modem;
    int new_transmit;
    int i;
    int len;
    int immediate_response;
    t31_state_t *s;
    uint8_t msg[256];

    s = (t31_state_t *) user_data;
    new_transmit = direction;
    immediate_response = TRUE;
    switch (operation)
    {
    case 'S':
        s->at_state.transmit = new_transmit;
        s->modem = T31_SILENCE;
        if (new_transmit)
        {
            /* Send some silence to space transmissions. */
            s->silent_samples += val*80;
        }
        else
        {
            /* Wait until we have received a specified period of silence. */
            queue_flush(&(s->rx_queue));
            s->silence_awaited = val*80;
            s->at_state.at_rx_mode = AT_MODE_DELIVERY;
        }
        immediate_response = FALSE;
        span_log(&s->logging, SPAN_LOG_FLOW, "Silence %dms\n", val*10);
        break;
    case 'H':
        switch (val)
        {
        case 3:
            new_modem = (new_transmit)  ?  T31_V21_TX  :  T31_V21_RX;
            s->short_train = FALSE;
            s->bit_rate = 300;
            break;
        default:
            return -1;
        }
        span_log(&s->logging, SPAN_LOG_FLOW, "HDLC\n");
        if (new_modem != s->modem)
        {
            restart_modem(s, new_modem);
            immediate_response = FALSE;
        }
        s->at_state.transmit = new_transmit;
        if (new_transmit)
        {
            s->last_dtedata_samples = s->call_samples;
            s->at_state.at_rx_mode = AT_MODE_HDLC;
            at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
        }
        else
        {
            /* Send straight away, if there is something queued. */
            s->at_state.at_rx_mode = AT_MODE_DELIVERY;
            s->rx_message_received = FALSE;
            do
            {
                if (!queue_empty(&(s->rx_queue)))
                {
                    len = queue_read_msg(&(s->rx_queue), msg, 256);
                    if (len > 1)
                    {
                        if (msg[0] == AT_RESPONSE_CODE_OK)
                            at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
                        for (i = 1;  i < len;  i++)
                        {
                            if (msg[i] == DLE)
                                s->at_state.rx_data[s->at_state.rx_data_bytes++] = DLE;
                            s->at_state.rx_data[s->at_state.rx_data_bytes++] = msg[i];
                        }
                        /* Fake CRC */
                        s->at_state.rx_data[s->at_state.rx_data_bytes++] = 0;
                        s->at_state.rx_data[s->at_state.rx_data_bytes++] = 0;
                        s->at_state.rx_data[s->at_state.rx_data_bytes++] = DLE;
                        s->at_state.rx_data[s->at_state.rx_data_bytes++] = ETX;
                        s->at_state.at_tx_handler(&s->at_state, s->at_state.at_tx_user_data, s->at_state.rx_data, s->at_state.rx_data_bytes);
                        s->at_state.rx_data_bytes = 0;
                    }
                    at_put_response_code(&s->at_state, msg[0]);
                }
                else
                {
                    s->at_state.dte_is_waiting = TRUE;
                    break;
                }
            }
            while (msg[0] == AT_RESPONSE_CODE_CONNECT);
        }
        immediate_response = FALSE;
        break;
    default:
        switch (val)
        {
        case 24:
            new_modem = (new_transmit)  ?  T31_V27TER_TX  :  T31_V27TER_RX;
            s->short_train = FALSE;
            s->bit_rate = 2400;
            break;
        case 48:
            new_modem = (new_transmit)  ?  T31_V27TER_TX  :  T31_V27TER_RX;
            s->short_train = FALSE;
            s->bit_rate = 4800;
            break;
        case 72:
            new_modem = (new_transmit)  ?  T31_V29_TX  :  T31_V29_RX;
            s->short_train = FALSE;
            s->bit_rate = 7200;
            break;
        case 96:
            new_modem = (new_transmit)  ?  T31_V29_TX  :  T31_V29_RX;
            s->short_train = FALSE;
            s->bit_rate = 9600;
            break;
#if defined(ENABLE_V17)
        case 73:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = FALSE;
            s->bit_rate = 7200;
            break;
        case 74:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = TRUE;
            s->bit_rate = 7200;
            break;
        case 97:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = FALSE;
            s->bit_rate = 9600;
            break;
        case 98:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = TRUE;
            s->bit_rate = 9600;
            break;
        case 121:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = FALSE;
            s->bit_rate = 12000;
            break;
        case 122:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = TRUE;
            s->bit_rate = 12000;
            break;
        case 145:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = FALSE;
            s->bit_rate = 14400;
            break;
        case 146:
            new_modem = (new_transmit)  ?  T31_V17_TX  :  T31_V17_RX;
            s->short_train = TRUE;
            s->bit_rate = 14400;
            break;
#endif
        default:
            return -1;
        }
        span_log(&s->logging, SPAN_LOG_FLOW, "Short training = %d, bit rate = %d\n", s->short_train, s->bit_rate);
        if (new_transmit)
        {
            at_put_response_code(&s->at_state, AT_RESPONSE_CODE_CONNECT);
            s->at_state.at_rx_mode = AT_MODE_STUFFED;
        }
        else
        {
            s->at_state.at_rx_mode = AT_MODE_DELIVERY;
        }
        restart_modem(s, new_modem);
        immediate_response = FALSE;
        break;
    }
    return immediate_response;
}
/*- End of function --------------------------------------------------------*/

void t31_call_event(t31_state_t *s, int event)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Call event %d received\n", event);
    at_call_event(&s->at_state, event);
}
/*- End of function --------------------------------------------------------*/

int t31_at_rx(t31_state_t *s, const char *t, int len)
{
    s->last_dtedata_samples = s->call_samples;

    switch (s->at_state.at_rx_mode)
    {
    case AT_MODE_ONHOOK_COMMAND:
    case AT_MODE_OFFHOOK_COMMAND:
        at_interpreter(&s->at_state, t, len);
        break;
    case AT_MODE_DELIVERY:
        /* Data from the DTE in this state returns us to command mode */
        if (len)
        {
            s->at_state.rx_data_bytes = 0;
            s->at_state.transmit = FALSE;
            s->modem = T31_SILENCE;
            s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
            at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
        }
        break;
    case AT_MODE_HDLC:
        dle_unstuff_hdlc(s, t, len);
        break;
    case AT_MODE_STUFFED:
        if (s->tx_out_bytes)
        {
            /* Make room for new data in existing data buffer. */
            s->tx_in_bytes = &(s->tx_data[s->tx_in_bytes]) - &(s->tx_data[s->tx_out_bytes]);
            memmove(&(s->tx_data[0]), &(s->tx_data[s->tx_out_bytes]), s->tx_in_bytes);
            s->tx_out_bytes = 0;
        }
        dle_unstuff(s, t, len);
        break;
    }
    return len;
}
/*- End of function --------------------------------------------------------*/

int t31_rx(t31_state_t *s, int16_t *buf, int len)
{
    int i;
    int32_t power;

    /* Monitor for received silence.  Maximum needed detection is AT+FRS=255 (255*10ms). */
    /* We could probably only run this loop if (s->modem == T31_SILENCE), however,
       the spec says "when silence has been present on the line for the amount of
       time specified".  That means some of the silence may have occurred before
       the AT+FRS=n command. This condition, however, is not likely to ever be the
       case.  (AT+FRS=n will usually be issued before the remote goes silent.) */
    for (i = 0;  i < len;  i++)
    {
        /* Clean up any DC influence. */
        power = power_meter_update(&(s->rx_power), buf[i] - s->last_sample);
        s->last_sample = buf[i];
        if (power > s->silence_threshold_power)
        {
            s->silence_heard = 0;
        }
        else
        {        
            if (s->silence_heard <= ms_to_samples(255*10))
                s->silence_heard++;
        }
    }

    /* Time is determined by counting audio packets come in. */
    s->call_samples += len;
    /* In HDLC transmit mode if 5 seconds elapse without data from the DTE
       then we must result in ERROR, returning to command-mode. */
    if ((s->at_state.at_rx_mode == AT_MODE_HDLC || s->at_state.at_rx_mode == AT_MODE_STUFFED)
        &&
        s->call_samples > s->last_dtedata_samples + ms_to_samples(5000))
    {
        s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
        at_put_response_code(&s->at_state, AT_RESPONSE_CODE_ERROR);
        restart_modem(s, T31_SILENCE);
    }

    if ((s->modem == T31_CNG_TONE  ||  s->modem == T31_NOCNG_TONE)
        &&
        (s->call_samples > ms_to_samples(s->at_state.p.s_regs[7]*1000)))
    {
        /* After calling S7 has elapsed... no carrier found. */
        at_put_response_code(&s->at_state, AT_RESPONSE_CODE_NO_CARRIER);
        restart_modem(s, T31_SILENCE);
        at_modem_control(&s->at_state, AT_MODEM_CONTROL_HANGUP, NULL);
        s->at_state.at_rx_mode = AT_MODE_ONHOOK_COMMAND;
    }
    if (!s->at_state.transmit  ||  s->modem == T31_CNG_TONE)
    {
        switch (s->modem)
        {
        case T31_CED_TONE:
            break;
        case T31_CNG_TONE:
        case T31_NOCNG_TONE:
        case T31_V21_RX:
            fsk_rx(&(s->v21rx), buf, len);
            break;
#if defined(ENABLE_V17)
        case T31_V17_RX:
            v17_rx(&(s->v17rx), buf, len);
            if (!(s->at_state.rx_signal_present))
                fsk_rx(&(s->v21rx), buf, len);
            break;
#endif
        case T31_V27TER_RX:
            v27ter_rx(&(s->v27ter_rx), buf, len);
            if (!(s->at_state.rx_signal_present))
                fsk_rx(&(s->v21rx), buf, len);
            break;
        case T31_V29_RX:
            v29_rx(&(s->v29rx), buf, len);
            if (!(s->at_state.rx_signal_present))
                fsk_rx(&(s->v21rx), buf, len);
            break;
        case T31_SILENCE:
            if (s->silence_awaited  &&  s->silence_heard >= s->silence_awaited)
            {
                s->silence_heard = 0;
                s->silence_awaited = 0;
                at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
                s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
            }
            break;
        default:
            /* Absorb the data, but ignore it. */
            break;
        }
    }
    return  0;
}
/*- End of function --------------------------------------------------------*/

int t31_tx(t31_state_t *s, int16_t *buf, int max_len)
{
    int len;
    int lenx;

    len = 0;
    if (s->at_state.transmit)
    {
        if (s->silent_samples)
        {
            len = s->silent_samples;
            if (len > max_len)
                len = max_len;
            s->silent_samples -= len;
            max_len -= len;
            memset(buf, 0, len*sizeof(int16_t));
            if ((max_len > 0  ||  !s->silent_samples)
                &&
                s->modem == T31_SILENCE)
            {
                at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
                max_len = 0;
                if (s->at_state.do_hangup)
                {
                    at_modem_control(&s->at_state, AT_MODEM_CONTROL_HANGUP, NULL);
                    s->at_state.at_rx_mode = AT_MODE_ONHOOK_COMMAND;
                    s->at_state.do_hangup = FALSE;
                }
                else
                {
                    s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
                }
            }
        }
        if (max_len > 0)
        {
            switch (s->modem)
            {
            case T31_CED_TONE:
                if ((lenx = tone_gen(&(s->tone_gen), buf + len, max_len)) <= 0)
                {
                    /* Go directly to V.21/HDLC transmit. */
                    restart_modem(s, T31_V21_TX);
                    s->at_state.at_rx_mode = AT_MODE_HDLC;
                }
                len += lenx;
                break;
            case T31_CNG_TONE:
                len += tone_gen(&(s->tone_gen), buf + len, max_len);
                break;
            case T31_V21_TX:
                len += fsk_tx(&(s->v21tx), buf + len, max_len);
                break;
#if defined(ENABLE_V17)
            case T31_V17_TX:
                if ((lenx = v17_tx(&(s->v17tx), buf + len, max_len)) <= 0)
                {
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
                    s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
                    restart_modem(s, T31_SILENCE);
                }
                len += lenx;
                break;
#endif
            case T31_V27TER_TX:
                if ((lenx = v27ter_tx(&(s->v27ter_tx), buf + len, max_len)) <= 0)
                {
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
                    s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
                    restart_modem(s, T31_SILENCE);
                }
                len += lenx;
                break;
            case T31_V29_TX:
                if ((lenx = v29_tx(&(s->v29tx), buf + len, max_len)) <= 0)
                {
                    at_put_response_code(&s->at_state, AT_RESPONSE_CODE_OK);
                    s->at_state.at_rx_mode = AT_MODE_OFFHOOK_COMMAND;
                    restart_modem(s, T31_SILENCE);
                }
                len += lenx;
                break;
            }
        }
    }
    return len;
}
/*- End of function --------------------------------------------------------*/

t31_state_t *t31_init(t31_state_t *s,
                      at_tx_handler_t *at_tx_handler,
                      void *at_tx_user_data,
                      t31_modem_control_handler_t *modem_control_handler,
                      void *modem_control_user_data)
{
    if (at_tx_handler == NULL  ||  modem_control_handler == NULL)
        return NULL;
    memset(s, 0, sizeof(*s));

    span_log_init(&s->logging, SPAN_LOG_NONE, NULL);
    span_log_set_protocol(&s->logging, "T.31");

    s->modem_control_handler = modem_control_handler;
    s->modem_control_user_data = modem_control_user_data;
#if defined(ENABLE_V17)
    v17_rx_init(&(s->v17rx), 14400, fast_putbit, s);
    v17_tx_init(&(s->v17tx), 14400, FALSE, fast_getbit, s);
#endif
    v29_rx_init(&(s->v29rx), 9600, fast_putbit, s);
    v29_tx_init(&(s->v29tx), 9600, FALSE, fast_getbit, s);
    v27ter_rx_init(&(s->v27ter_rx), 4800, fast_putbit, s);
    v27ter_tx_init(&(s->v27ter_tx), 4800, FALSE, fast_getbit, s);
    power_meter_init(&(s->rx_power), 4);
    s->last_sample = 0;
    s->silence_threshold_power = power_meter_level_dbm0(-30);
    s->at_state.rx_signal_present = FALSE;

    s->at_state.do_hangup = FALSE;
    s->at_state.line_ptr = 0;
    s->silent_samples = 0;
    s->silence_heard = 0;
    s->silence_awaited = 0;
    s->call_samples = 0;
    s->modem = -1;
    s->at_state.transmit = -1;
    if (queue_create(&(s->rx_queue), 4096, QUEUE_WRITE_ATOMIC | QUEUE_READ_ATOMIC) < 0)
        return NULL;
    at_init(&s->at_state, at_tx_handler, at_tx_user_data, t31_modem_control_handler, s);
    at_set_class1_handler(&s->at_state, process_class1_cmd, s);
    return s;
}
/*- End of function --------------------------------------------------------*/

int t31_release(t31_state_t *s)
{
    at_reset_call_info(&s->at_state);
    free(s);
    return 0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
