#define ENABLE_V17
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t38.c - An implementation of a T.38 gateway, less the packet exchange part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_gateway.c,v 1.28 2006/05/19 12:23:32 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <tgmath.h>
#include <math.h>
#include <assert.h>
#include <tiffio.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/queue.h"
#include "spandsp/bit_operations.h"
#include "spandsp/alaw_ulaw.h"
#include "spandsp/power_meter.h"
#include "spandsp/complex.h"
#include "spandsp/tone_generate.h"
#include "spandsp/async.h"
#include "spandsp/hdlc.h"
#include "spandsp/fsk.h"
#include "spandsp/v29rx.h"
#include "spandsp/v29tx.h"
#include "spandsp/v27ter_rx.h"
#include "spandsp/v27ter_tx.h"
#if defined(ENABLE_V17)
#include "spandsp/v17rx.h"
#include "spandsp/v17tx.h"
#endif
#include "spandsp/t4.h"

#include "spandsp/t30_fcf.h"
#include "spandsp/t35.h"
#include "spandsp/t30.h"

#include "spandsp/t38.h"

#define INDICATOR_TX_COUNT  3

#define ms_to_samples(t)    (((t)*SAMPLE_RATE)/1000)

#define DISBIT1     0x01
#define DISBIT2     0x02
#define DISBIT3     0x04
#define DISBIT4     0x08
#define DISBIT5     0x10
#define DISBIT6     0x20
#define DISBIT7     0x40
#define DISBIT8     0x80

enum
{
    T38_SILENCE,
    T38_CED_TONE,
    T38_CNG_TONE,
    T38_V21_RX,
    T38_V27TER_RX,
    T38_V29_RX,
    T38_V17_RX
};

#define SEND_INTERVAL   20

static int restart_rx_modem(t38_state_t *s);
static void add_to_non_ecm_tx_buffer(t38_state_t *s, const uint8_t *buf, int len);
static void t38_hdlc_tx(t38_state_t *s, const uint8_t *frame, int len);
static void t38_hdlc_tx_preamble(t38_state_t *s, int len);
static int t38_hdlc_tx_getbit(t38_state_t *t);
static void t38_hdlc_tx_init(t38_state_t *s);

static int set_next_tx_type(t38_state_t *s)
{
    tone_gen_descriptor_t tone_desc;

    if (s->next_rx_indicator == s->current_rx_indicator)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Changing to no signal - %d %d\n", s->next_rx_indicator, s->current_rx_indicator);
        if (s->next_rx_indicator == T38_IND_NO_SIGNAL)
        {
            s->silence = 0;
        }
        else
        {
            s->silence = ms_to_samples(75);
            s->next_rx_indicator =
            s->current_rx_indicator = T38_IND_NO_SIGNAL;
        }
        return 0;
    }
    s->silence = -1;
    span_log(&s->logging, SPAN_LOG_FLOW, "Changing to %d\n", s->next_rx_indicator);

    switch (s->next_rx_indicator)
    {
    case T38_IND_NO_SIGNAL:
        /* Impose 75ms minimum on transmitted silence */
        s->silence = ms_to_samples(75);
        break;
    case T38_IND_CNG:
        /* 0.5s of 1100Hz + 3.0s of silence repeating */
        make_tone_gen_descriptor(&tone_desc,
                                 1100,
                                 -11,
                                 0,
                                 0,
                                 500,
                                 3000,
                                 0,
                                 0,
                                 TRUE);
        tone_gen_init(&(s->tone_gen), &tone_desc);
        break;
    case T38_IND_CED:
        make_tone_gen_descriptor(&tone_desc,
                                 2100,
                                 -11,
                                 0,
                                 0,
                                 2600,
                                 75,
                                 0,
                                 0,
                                 FALSE);
        tone_gen_init(&(s->tone_gen), &tone_desc);
        break;
    case T38_IND_V21_PREAMBLE:
        t38_hdlc_tx_init(s);
        t38_hdlc_tx_preamble(s, 32);
        s->data_final = FALSE;
        s->hdlc_len = 0;
        fsk_tx_init(&(s->v21tx), &preset_fsk_specs[FSK_V21CH2], (get_bit_func_t) t38_hdlc_tx_getbit, s);
        break;
    case T38_IND_V27TER_2400_TRAINING:
        v27ter_tx_restart(&(s->v27ter_tx), 2400, s->use_tep);
        break;
    case T38_IND_V27TER_4800_TRAINING:
        v27ter_tx_restart(&(s->v27ter_tx), 4800, s->use_tep);
        break;
    case T38_IND_V29_7200_TRAINING:
        v29_tx_restart(&(s->v29tx), 7200, s->use_tep);
        break;
    case T38_IND_V29_9600_TRAINING:
        v29_tx_restart(&(s->v29tx), 9600, s->use_tep);
        break;
#if defined(ENABLE_V17)
    case T38_IND_V17_7200_SHORT_TRAINING:
        v17_tx_restart(&(s->v17tx), 7200, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_7200_LONG_TRAINING:
        v17_tx_restart(&(s->v17tx), 7200, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_9600_SHORT_TRAINING:
        v17_tx_restart(&(s->v17tx), 9600, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_9600_LONG_TRAINING:
        v17_tx_restart(&(s->v17tx), 9600, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_12000_SHORT_TRAINING:
        v17_tx_restart(&(s->v17tx), 12000, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_12000_LONG_TRAINING:
        v17_tx_restart(&(s->v17tx), 12000, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_14400_SHORT_TRAINING:
        v17_tx_restart(&(s->v17tx), 14400, s->use_tep, s->short_train);
        break;
    case T38_IND_V17_14400_LONG_TRAINING:
        v17_tx_restart(&(s->v17tx), 14400, s->use_tep, s->short_train);
        break;
#endif
    case T38_IND_V8_ANSAM:
        break;
    case T38_IND_V8_SIGNAL:
        break;
    case T38_IND_V34_CNTL_CHANNEL_1200:
        break;
    case T38_IND_V34_PRI_CHANNEL:
        break;
    case T38_IND_V34_CC_RETRAIN:
        break;
    case T38_IND_V33_12000_TRAINING:
        break;
    case T38_IND_V33_14400_TRAINING:
        break;
    default:
        break;
    }
    s->bit_no = 0;
    s->current_byte = 0xFF;
    s->tx_out_bytes = 0;
    s->tx_in_bytes = 0;
    s->tx_latest_eol_byte = 0;
    s->flow_control_fill_octet = 0xFF;
    s->at_initial_all_ones = TRUE;
    s->bit_stream = 0;
    if (s->flow_control_fill_octets)
    {
        span_log(&s->logging, SPAN_LOG_WARNING, "Flow control generated %d octets\n", s->flow_control_fill_octets);
        s->flow_control_fill_octets = 0;
    }
    s->current_rx_indicator = s->next_rx_indicator;
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t38_gateway_process_rx_missing(t38_state_t *s, int seq_no)
{
    span_log(&s->logging, SPAN_LOG_WARNING, "IFP seq no %d missing\n", seq_no);
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t38_gateway_process_rx_indicator(t38_state_t *s, int indicator)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Rx indicator %s\n", t38_indicator(indicator));
    s->next_rx_indicator = indicator;

    if (s->silence == 0)
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Immediate change - %d -> %d\n", s->current_rx_indicator, s->next_rx_indicator);
        set_next_tx_type(s);
    }
    else
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Delayed change - (%d) %d -> %d\n", s->silence, s->current_rx_indicator, s->next_rx_indicator);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void pump_out_hdlc(t38_state_t *s, int previous)
{
    /* Don't start pumping data into the actual output stream until there is
       enough backlog to create some elasticity for jitter tolerance. */
    if (s->current_rx_indicator == T38_IND_V21_PREAMBLE  &&  s->hdlc_len > 5)
    {
        if (previous <= 5)
            previous = 0;
        t38_hdlc_tx(s, s->hdlc_buf + previous, s->hdlc_len - previous);
    }
}
/*- End of function --------------------------------------------------------*/

static void pump_out_final_hdlc(t38_state_t *s, int good_fcs)
{
    uint16_t crc;

    crc = crc_itu16_calc(s->hdlc_buf, s->hdlc_len, 0xFFFF) ^ 0xFFFF;
    /* If the incoming CRC was wrong, corrupt the outgoing one */
    if (!good_fcs)
        crc = ~crc;
    s->hdlc_buf[s->hdlc_len++] = (uint8_t) (crc & 0xFF);
    s->hdlc_buf[s->hdlc_len++] = (uint8_t) ((crc >> 8) & 0xFF);
    if (s->current_rx_indicator == T38_IND_V21_PREAMBLE)
    {
        if (s->hdlc_len <= 5)
            t38_hdlc_tx(s, s->hdlc_buf, s->hdlc_len);
        else
            t38_hdlc_tx(s, s->hdlc_buf + s->hdlc_len - 2, 2);
        t38_hdlc_tx_preamble(s, 2);
    }
}
/*- End of function --------------------------------------------------------*/

static void constrain_fast_modem(t38_state_t *s, uint8_t *buf, int len)
{
    /* We may need to adjust the capabilities, so they do not exceed our own */
    /* TODO: fiddle the contents */
    switch (buf[4] & (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3))
    {
    case 0:
    case DISBIT4:
        /* V.27ter only */
        break;
    case DISBIT3:
    case (DISBIT4 | DISBIT3):
        /* V.27ter and V.29 */
        break;
    case (DISBIT6 | DISBIT4 | DISBIT3):
        /* V.27ter, V.29 and V.17 */
#if !defined(ENABLE_V17)
        buf[4] &= ~DISBIT6;
#endif
        break;
    case (DISBIT5 | DISBIT4):
    case (DISBIT6 | DISBIT4):
    case (DISBIT6 | DISBIT5 | DISBIT4):
    case (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3):
        /* Reserved */
        buf[4] &= ~(DISBIT6 | DISBIT5);
        buf[4] |= (DISBIT4 | DISBIT3);
        break;
    default:
        /* Not used */
        buf[4] &= ~(DISBIT6 | DISBIT5);
        buf[4] |= (DISBIT4 | DISBIT3);
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void set_fast_modem(t38_state_t *s, uint8_t *buf, int len)
{
    /* We need to check which modem type is about to be used. */
    switch (buf[4] & (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3))
    {
    case 0:
        s->bit_rate = 2400;
        s->fast_modem = T38_V27TER_RX;
        s->short_train = FALSE;
        break;
    case DISBIT4:
        s->bit_rate = 4800;
        s->fast_modem = T38_V27TER_RX;
        s->short_train = FALSE;
        break;
    case DISBIT3:
        s->bit_rate = 9600;
        s->fast_modem = T38_V29_RX;
        s->short_train = FALSE;
        break;
    case (DISBIT4 | DISBIT3):
        s->bit_rate = 7200;
        s->fast_modem = T38_V29_RX;
        s->short_train = FALSE;
        break;
#if defined(ENABLE_V17)
    case DISBIT6:
        s->bit_rate = 14400;
        s->fast_modem = T38_V17_RX;
        s->short_train = FALSE;
        break;
    case (DISBIT6 | DISBIT4):
        s->bit_rate = 12000;
        s->fast_modem = T38_V17_RX;
        s->short_train = FALSE;
        break;
    case (DISBIT6 | DISBIT3):
        s->bit_rate = 9600;
        s->fast_modem = T38_V17_RX;
        s->short_train = FALSE;
        break;
    case (DISBIT6 | DISBIT4 | DISBIT3):
        s->bit_rate = 7200;
        s->fast_modem = T38_V17_RX;
        s->short_train = FALSE;
        break;
#endif
    case (DISBIT5 | DISBIT3):
    case (DISBIT5 | DISBIT4 | DISBIT3):
    case (DISBIT6 | DISBIT5):
    case (DISBIT6 | DISBIT5 | DISBIT3):
    case (DISBIT6 | DISBIT5 | DISBIT4):
    case (DISBIT6 | DISBIT5 | DISBIT4 | DISBIT3):
        /* Reserved */
        s->bit_rate = 0;
        s->fast_modem = T38_SILENCE;
        break;
    default:
        /* Not used */
        s->bit_rate = 0;
        s->fast_modem = T38_SILENCE;
        break;
    }
}
/*- End of function --------------------------------------------------------*/

int t38_gateway_process_rx_data(t38_state_t *s, int data_type, int field_type, const uint8_t *buf, int len)
{
    int i;
    int previous;

    span_log(&s->logging, SPAN_LOG_FLOW, "Rx data type %s/%s\n", t38_data_type(data_type), t38_field_type(field_type));
    switch (field_type)
    {
    case T38_FIELD_HDLC_DATA:
        previous = s->hdlc_len;
        for (i = 0;  i < len;  i++)
        {
            s->hdlc_buf[s->hdlc_len++] = bit_reverse8(buf[i]);
            if (s->hdlc_len == 4)
            {
                /* Check if we need to corrupt this message */
                if (s->hdlc_buf[2] == T30_NSF
                    ||
                    s->hdlc_buf[2] == T30_NSC
                    ||
                    s->hdlc_buf[2] == T30_NSS)
                {
                    /* Corrupt the message, so it will be ignored */
                    s->hdlc_buf[3] = 0;
                }
            }
            else if (s->hdlc_len == 6)
            {
                switch (s->hdlc_buf[2])
                {
                case T30_DIS:
                    /* We may need to adjust the capabilities, so they do not exceed our own */
                    constrain_fast_modem(s, s->hdlc_buf, s->hdlc_len);
                    break;
                default:
                    break;
                }
            }
        }
        pump_out_hdlc(s, previous);
        break;
    case T38_FIELD_HDLC_SIG_END:
        /* There should be no data */
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_SIG_END!\n");
        break;
    case T38_FIELD_HDLC_FCS_OK:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_OK!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC good\n", t30_frametype(s->hdlc_buf[2]));
        pump_out_final_hdlc(s, TRUE);
        s->hdlc_len = 0;
        break;
    case T38_FIELD_HDLC_FCS_BAD:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_BAD!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC bad\n", t30_frametype(s->hdlc_buf[2]));
        pump_out_final_hdlc(s, FALSE);
        s->hdlc_len = 0;
        break;
    case T38_FIELD_HDLC_FCS_OK_SIG_END:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_OK_SIG_END!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC OK, sig end\n", t30_frametype(s->hdlc_buf[2]));
        switch (s->hdlc_buf[2])
        {
        case T30_CFR:
            /* Successful training means we should change to short training */
            s->short_train = TRUE;
            break;
        case T30_DTC:
        case T30_DCS:
        case T30_DCS + 1:
            /* We need to check which modem type is about to be used. */
            set_fast_modem(s, s->hdlc_buf, s->hdlc_len);
            break;
        default:
            break;
        }
        pump_out_final_hdlc(s, TRUE);
        s->data_final = TRUE;
        s->hdlc_len = 0;
        break;
    case T38_FIELD_HDLC_FCS_BAD_SIG_END:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_HDLC_FCS_BAD_SIG_END!\n");
        span_log(&s->logging, SPAN_LOG_FLOW, "Type %s - CRC bad, sig end\n", t30_frametype(s->hdlc_buf[2]));
        pump_out_final_hdlc(s, FALSE);
        s->data_final = TRUE;
        s->hdlc_len = 0;
        break;
    case T38_FIELD_T4_NON_ECM_DATA:
        add_to_non_ecm_tx_buffer(s, buf, len);
        break;
    case T38_FIELD_T4_NON_ECM_SIG_END:
        if (len > 0)
            span_log(&s->logging, SPAN_LOG_WARNING, "There is data in a T38_FIELD_NON_ECM_SIG_END!\n");
        /* Don't flow control the data any more */
        s->tx_latest_eol_byte = s->tx_in_bytes;
        s->data_final = TRUE;
        break;
    case T38_FIELD_CM_MESSAGE:
    case T38_FIELD_JM_MESSAGE:
    case T38_FIELD_CI_MESSAGE:
    case T38_FIELD_V34RATE:
    default:
        break;
    }

#if 0
    if (span_log_test(&s->logging, SPAN_LOG_FLOW))
    {
        int i;

        if (len > 0)
        {
            span_log(&s->logging, SPAN_LOG_FLOW, "Data: ");
            for (i = 0;  i < len;  i++)
                span_log(&s->logging, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, " %02X", buf[i]);
        }
    }
    span_log(&s->logging, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "\n");
#endif
    return 0;
}
/*- End of function --------------------------------------------------------*/

#if defined(ENABLE_V17)
static void non_ecm_putbit_v17(void *user_data, int bit)
{
    t38_state_t *s;

    s = (t38_state_t *) user_data;
    if (bit < 0)
    {
        /* Special conditions */
        switch (bit)
        {
        case PUTBIT_TRAINING_FAILED:
            break;
        case PUTBIT_TRAINING_SUCCEEDED:
            /* The modem is now trained */
            s->rx_signal_present = TRUE;
            s->v21_rx_active = FALSE;
            switch (s->bit_rate)
            {
            case 7200:
                t38_send_indicator(s, T38_IND_V17_7200_LONG_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V17_7200;
                break;
            case 9600:
                t38_send_indicator(s, T38_IND_V17_9600_LONG_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V17_9600;
                break;
            case 12000:
                t38_send_indicator(s, T38_IND_V17_12000_LONG_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V17_12000;
                break;
            case 14400:
                t38_send_indicator(s, T38_IND_V17_14400_LONG_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V17_14400;
                break;
            }
            s->since_last_tx_packet = 0;
            s->rx_data_bytes = 0;
            break;
        case PUTBIT_CARRIER_UP:
            break;
        case PUTBIT_CARRIER_DOWN:
            switch (s->current_tx_data)
            {
            case T38_DATA_V17_7200:
            case T38_DATA_V17_9600:
            case T38_DATA_V17_12000:
            case T38_DATA_V17_14400:
                t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_SIG_END, "", 0);
                t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
                s->rx_signal_present = FALSE;
                restart_rx_modem(s);
                break;
            }
            break;
        default:
            span_log(&s->logging, SPAN_LOG_WARNING, "Eh!\n");
            break;
        }
        return;
    }
    s->current_byte = (s->current_byte << 1) | (bit & 1);
    if (++s->bit_no >= 8)
    {
        s->rx_data[s->rx_data_bytes++] = s->current_byte;
        if (s->since_last_tx_packet >= ms_to_samples(SEND_INTERVAL))
        {
            t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_DATA, s->rx_data, s->rx_data_bytes);
            s->rx_data_bytes = 0;
            s->since_last_tx_packet = 0;
        }
        s->bit_no = 0;
        s->current_byte = 0;
    }
}
/*- End of function --------------------------------------------------------*/
#endif

static void non_ecm_putbit_v29(void *user_data, int bit)
{
    t38_state_t *s;

    s = (t38_state_t *) user_data;
    if (bit < 0)
    {
        /* Special conditions */
        switch (bit)
        {
        case PUTBIT_TRAINING_FAILED:
            break;
        case PUTBIT_TRAINING_SUCCEEDED:
            /* The modem is now trained */
            s->rx_signal_present = TRUE;
            s->v21_rx_active = FALSE;
            switch (s->bit_rate)
            {
            case 7200:
                t38_send_indicator(s, T38_IND_V29_7200_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V29_7200;
                break;
            case 9600:
                t38_send_indicator(s, T38_IND_V29_9600_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V29_9600;
                break;
            }
            s->since_last_tx_packet = 0;
            s->rx_data_bytes = 0;
            break;
        case PUTBIT_CARRIER_UP:
            break;
        case PUTBIT_CARRIER_DOWN:
            switch (s->current_tx_data)
            {
            case T38_DATA_V29_7200:
            case T38_DATA_V29_9600:
                t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_SIG_END, (uint8_t *) "", 0);
                t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
                s->rx_signal_present = FALSE;
                restart_rx_modem(s);
                break;
            }
            break;
        default:
            span_log(&s->logging, SPAN_LOG_WARNING, "Eh!\n");
            break;
        }
        return;
    }
    s->current_byte = (s->current_byte << 1) | (bit & 1);
    if (++s->bit_no >= 8)
    {
        s->rx_data[s->rx_data_bytes++] = (uint8_t) s->current_byte;
        if (s->since_last_tx_packet >= ms_to_samples(SEND_INTERVAL))
        {
            t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_DATA, s->rx_data, s->rx_data_bytes);
            s->rx_data_bytes = 0;
            s->since_last_tx_packet = 0;
        }
        s->bit_no = 0;
        s->current_byte = 0;
    }
}
/*- End of function --------------------------------------------------------*/

static void non_ecm_putbit_v27ter(void *user_data, int bit)
{
    t38_state_t *s;

    s = (t38_state_t *) user_data;
    if (bit < 0)
    {
        /* Special conditions */
        switch (bit)
        {
        case PUTBIT_TRAINING_FAILED:
            break;
        case PUTBIT_TRAINING_SUCCEEDED:
            /* The modem is now trained */
            s->rx_signal_present = TRUE;
            s->v21_rx_active = FALSE;
            switch (s->bit_rate)
            {
            case 2400:
                t38_send_indicator(s, T38_IND_V27TER_2400_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V27TER_2400;
                break;
            case 4800:
                t38_send_indicator(s, T38_IND_V27TER_4800_TRAINING, INDICATOR_TX_COUNT);
                s->current_tx_data = T38_DATA_V27TER_4800;
                break;
            }
            s->since_last_tx_packet = 0;
            s->rx_data_bytes = 0;
            break;
        case PUTBIT_CARRIER_UP:
            break;
        case PUTBIT_CARRIER_DOWN:
            switch (s->current_tx_data)
            {
            case T38_DATA_V27TER_2400:
            case T38_DATA_V27TER_4800:
                t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_SIG_END, (uint8_t *) "", 0);
                t38_send_indicator(s, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
                s->rx_signal_present = FALSE;
                restart_rx_modem(s);
                break;
            }
            break;
        default:
            span_log(&s->logging, SPAN_LOG_WARNING, "Eh!\n");
            break;
        }
        return;
    }
    s->current_byte = (s->current_byte << 1) | (bit & 1);
    if (++s->bit_no >= 8)
    {
        s->rx_data[s->rx_data_bytes++] = (uint8_t) s->current_byte;
        if (s->since_last_tx_packet >= ms_to_samples(SEND_INTERVAL))
        {
            t38_send_data(s, s->current_tx_data, T38_FIELD_T4_NON_ECM_DATA, s->rx_data, s->rx_data_bytes);
            s->rx_data_bytes = 0;
            s->since_last_tx_packet = 0;
        }
        s->bit_no = 0;
        s->current_byte = 0;
    }
}
/*- End of function --------------------------------------------------------*/

static int non_ecm_getbit(void *user_data)
{
    t38_state_t *s;
    int bit;

    s = (t38_state_t *) user_data;
    if (s->bit_no <= 0)
    {
        /* We need another byte */
        if (s->tx_out_bytes != s->tx_latest_eol_byte)
        {
            s->current_byte = s->tx_data[s->tx_out_bytes];
            s->tx_out_bytes = (s->tx_out_bytes + 1) & (T38_TX_BUF_LEN - 1);
        }
        else
        {
            if (s->data_final)
            {
                s->data_final = FALSE;
                return 2;
            }
            s->current_byte = s->flow_control_fill_octet;
            s->flow_control_fill_octets++;
        }
        s->bit_no = 8;
    }
    s->bit_no--;
    bit = (s->current_byte >> 7) & 1;
    s->current_byte <<= 1;
    return bit;
}
/*- End of function --------------------------------------------------------*/

static void add_to_non_ecm_tx_buffer(t38_state_t *s, const uint8_t *buf, int len)
{
    int i;
    int upper;
    int lower;

    /* A rate adapting data stuffer for non-ECM image data */
    i = 0;
    if (s->at_initial_all_ones)
    {
        /* Dump initial 0xFF bytes. We will add enough of our own to makes things flow
           smoothly. If we don't strip these off we might end up delaying the start of
           forwarding by a large amount, as we could end up with a large block of 0xFF
           bytes before the real data begins. This is especially true with PC FAX
           systems. */
        for (  ;  i < len;  i++)
        {
            if (buf[i] != 0xFF)
            {
                s->at_initial_all_ones = FALSE;
                break;
            }
        }
    }
    if (s->at_tcf)
    {
        for (  ;  i < len;  i++)
        {
            /* Check for zero bytes, as we can pause and punp out zeros while waiting for more
               incoming data. */
            if (buf[i] == 0)
            {
                s->tx_latest_eol_byte = s->tx_in_bytes;
                s->flow_control_fill_octet = 0x00;
            }
            s->tx_data[s->tx_in_bytes] = buf[i];
            s->tx_in_bytes = (s->tx_in_bytes + 1) & (T38_TX_BUF_LEN - 1);
        }
    }
    else
    {
        for (  ;  i < len;  i++)
        {
            /* Check for EOLs, because at an EOL we can pause and pump out zeros while
               waiting for more incoming data. */
            if (buf[i])
            {
                /* There might be an EOL here. Look for at least 11 zeros, followed by a one. */
                upper = bottom_bit((s->bit_stream | 0x800) & 0xFFF);
                lower = top_bit(buf[i] & 0xFF);
                if (lower > 0  &&  upper - lower >= 3)
                {
                    s->tx_latest_eol_byte = s->tx_in_bytes;
                    s->flow_control_fill_octet = 0x00;
                }
            }
            s->bit_stream = (s->bit_stream << 8) | buf[i];
            s->tx_data[s->tx_in_bytes] = buf[i];
            s->tx_in_bytes = (s->tx_in_bytes + 1) & (T38_TX_BUF_LEN - 1);
        }
    }
}
/*- End of function --------------------------------------------------------*/

static void t38_hdlc_tx_init(t38_state_t *s)
{
    s->hdlctx_num_bits = 0;
    s->hdlctx_len = 0;
    s->hdlctx_pos = 0;
    s->hdlctx_byte = 0;
    s->hdlctx_bits = 0;
    s->hdlctx_idle_byte = 0x7E;
    s->hdlctx_buffer[s->hdlctx_len++] = (uint8_t) s->hdlctx_idle_byte;
    s->hdlctx_buffer[s->hdlctx_len++] = (uint8_t) s->hdlctx_idle_byte;
    s->hdlctx_buffer[s->hdlctx_len] = (uint8_t) s->hdlctx_idle_byte;
    s->hdlc_underflow_reported = FALSE;
}
/*- End of function --------------------------------------------------------*/

static void t38_hdlc_tx(t38_state_t *s, const uint8_t *frame, int len)
{
    int i;
    int byte_in_progress;
    int forming;

    forming = s->hdlctx_buffer[s->hdlctx_len];
    while (len--)
    {
        byte_in_progress = (int) *frame++;
        for (i = 0;  i < 8;  i++)
        {
            forming = (forming << 1) | (byte_in_progress & 0x01);
            byte_in_progress >>= 1;
            if ((forming & 0x1F) == 0x1F)
            {
                /* There are 5 ones - stuff */
                forming <<= 1;
                s->hdlctx_num_bits++;
            }
        }
        /* An input byte will generate between 8 and 10 output bits */
        s->hdlctx_buffer[s->hdlctx_len++] = (uint8_t) (forming >> s->hdlctx_num_bits);
        if (s->hdlctx_num_bits >= 8)
        {
            s->hdlctx_num_bits -= 8;
            s->hdlctx_buffer[s->hdlctx_len++] = (uint8_t) (forming >> s->hdlctx_num_bits);
        }
    }
    s->hdlctx_buffer[s->hdlctx_len] = (uint8_t) forming;
    s->hdlctx_idle_byte = (0x7E7E >> s->hdlctx_num_bits) & 0xFF;
}
/*- End of function --------------------------------------------------------*/

static void t38_hdlc_tx_preamble(t38_state_t *s, int len)
{
    s->hdlctx_idle_byte = (0x7E7E >> s->hdlctx_num_bits) & 0xFF;
    if (len-- > 0)
    {
        s->hdlctx_buffer[s->hdlctx_len] <<= (8 - s->hdlctx_num_bits);
        s->hdlctx_buffer[s->hdlctx_len] |= (0x7E >> s->hdlctx_num_bits);
        s->hdlctx_len++;
    }
    while (len-- > 0)
        s->hdlctx_buffer[s->hdlctx_len++] = (uint8_t) s->hdlctx_idle_byte;
    s->hdlctx_buffer[s->hdlctx_len] = 0x7E;
}
/*- End of function --------------------------------------------------------*/

static int t38_hdlc_tx_getbit(t38_state_t *s)
{
    int txbit;

    if (s->hdlctx_bits-- == 0)
    {
        if (s->hdlctx_len != s->hdlctx_pos)
        {
            s->hdlctx_byte = s->hdlctx_buffer[s->hdlctx_pos++];
            if (s->hdlctx_pos >= s->hdlctx_len)
            {
                s->hdlctx_pos = 0;
                s->hdlctx_len = 0;
                s->hdlctx_buffer[0] = 0x7E;
            }
            s->hdlc_underflow_reported = FALSE;
        }
        else
        {
            s->hdlctx_byte = s->hdlctx_idle_byte;
            if (!s->hdlc_underflow_reported)
            {
                if (s->data_final)
                {
                    set_next_tx_type(s);
                    s->data_final = FALSE;
                }
                s->hdlc_underflow_reported = TRUE;
            }
        }
        s->hdlctx_bits = 7;
    }
    txbit = (s->hdlctx_byte >> 7) & 0x01;
    s->hdlctx_byte <<= 1;
    return  txbit;
}
/*- End of function --------------------------------------------------------*/

static void t38_hdlc_rx_bit(hdlc_rx_state_t *s, int new_bit)
{
    t38_state_t *t;
    int final;
    uint8_t buf[2];

    t = (t38_state_t *) s->user_data;
    if (new_bit < 0)
    {
        /* Special conditions */
        switch (new_bit)
        {
        case PUTBIT_CARRIER_UP:
            /* Reset the HDLC receiver. */
            s->len = 0;
            s->num_bits = 0;
            s->flags_seen = 0;
            s->framing_ok_announced = FALSE;
            t->rx_signal_present = TRUE;
            break;
        case PUTBIT_CARRIER_DOWN:
            if (t->current_tx_indicator == T38_IND_V21_PREAMBLE)
            {
                t38_send_data(t, T38_DATA_V21, T38_FIELD_HDLC_SIG_END, (uint8_t *) "", 0);
                t38_send_indicator(t, T38_IND_NO_SIGNAL, INDICATOR_TX_COUNT);
                t->rx_signal_present = FALSE;
                restart_rx_modem(t);
            }
            break;
        default:
            span_log(&t->logging, SPAN_LOG_WARNING, "Unexpected HDLC special bit - %d!\n", new_bit);
            break;
        }
        return;
    }
    s->bit_buf |= (new_bit & 1);
    if ((s->bit_buf & 0x7F) == 0x7E)
    {
        if ((s->bit_buf & 0x80))
        {
            /* Hit HDLC abort */
            s->rx_aborts++;
        }
        else
        {
            /* Hit HDLC flag */
            if (s->flags_seen >= s->framing_ok_threshold)
            {
                if (s->len)
                {
                    if (s->len >= 2)
                    {
                        if (crc_itu16_check(s->buffer, s->len))
                        {
                            s->rx_frames++;
                            s->rx_bytes += s->len - 2;
                            t->rx_data_bytes = 0;
                            final = (s->len - 2 >= 2  &&  s->buffer[0] == 0xFF  &&  s->buffer[1] == 0x13);
                            span_log(&t->logging, SPAN_LOG_FLOW, "E Type %s\n", t30_frametype(s->buffer[2]));
                            switch (s->buffer[2])
                            {
                            case T30_CFR:
                                /* We are changing from TCF exchange to image exchange */
                                t->at_tcf = FALSE;
                                /* Successful training means we should change to short training */
                                t->short_train = TRUE;
                                break;
                            case T30_RTP:
                                /* We are going back to the exchange of fresh TCF */
                                t->at_tcf = TRUE;
                                break;
                            case T30_DTC:
                            case T30_DCS:
                            case T30_DCS + 1:
                                /* We need to check which modem type is about to be used. */
                                set_fast_modem(t, s->buffer, s->len);
                                break;
                            default:
                                break;
                            }
                            t38_send_data(t, T38_DATA_V21, (final)  ?  T38_FIELD_HDLC_FCS_OK_SIG_END  :  T38_FIELD_HDLC_FCS_OK, (uint8_t *) "", 0);
                        }
                        else
                        {
                            s->rx_crc_errors++;
                            t->rx_data_bytes = 0;
                            final = (s->len - 2 >= 2  &&  s->buffer[0] == 0xFF  &&  s->buffer[1] == 0x13);
                            span_log(&t->logging, SPAN_LOG_FLOW, "F Type %s\n", t30_frametype(s->buffer[2]));
                            t38_send_data(t, T38_DATA_V21, (final)  ?  T38_FIELD_HDLC_FCS_BAD_SIG_END  :  T38_FIELD_HDLC_FCS_BAD, (uint8_t *) "", 0);
                        }
                    }
                    else
                    {
                        /* Frame too short */
                        s->rx_length_errors++;
                    }
                }
            }
            else
            {
                if (++s->flags_seen >= s->framing_ok_threshold  &&  !s->framing_ok_announced)
                {
                    t38_send_indicator(t, T38_IND_V21_PREAMBLE, INDICATOR_TX_COUNT);
                    t->fast_rx_active = FALSE;
                    if (t->current_rx_indicator == T38_IND_CNG)
                    {
                        t->next_rx_indicator = T38_IND_NO_SIGNAL;
                        set_next_tx_type(t);
                    }
                    s->framing_ok_announced = TRUE;
                }
            }
        }
        s->len = 0;
        s->num_bits = 0;
        t->corrupt_the_frame = FALSE;
    }
    else
    {
        if (s->flags_seen >= s->framing_ok_threshold  &&  (s->bit_buf & 0x3F) != 0x3E)
        {
            s->byte_in_progress = (s->byte_in_progress >> 1) | ((s->bit_buf & 0x01) << 7);
            if (++s->num_bits == 8)
            {
                if (s->len >= (int) sizeof(s->buffer))
                {
                    /* Frame too long */
                    s->rx_length_errors++;
                    s->flags_seen = s->framing_ok_threshold - 1;
                    s->len = 0;
                }
                else
                {
                    s->buffer[s->len] = (uint8_t) s->byte_in_progress;
                    if (s->len >= 2)
                    {
                        /* Make the transmission lag by two bytes, so we do not send the CRC, and
                           do not report the CRC result too late. */
                        buf[0] = (t->corrupt_the_frame)  ?  0  :  bit_reverse8(s->buffer[s->len - 2]);
                        switch (s->len)
                        {
                        case 4:
                            switch (s->buffer[2])
                            {
                            case T30_NSF:
                            case T30_NSC:
                            case T30_NSS:
                                /* We need to corrupt the rest of this message */
                                t->corrupt_the_frame = TRUE;
                                break;
                            default:
                                break;
                            }
                            break;
                        case 6:
                            switch (s->buffer[2])
                            {
                            case T30_DIS:
                                /* We may need to adjust the capabilities, so they do not exceed our own */
                                constrain_fast_modem(t, s->buffer, s->len);
                                break;
                            default:
                                break;
                            }
                        }
                        t38_send_data(t, T38_DATA_V21, T38_FIELD_HDLC_DATA, buf, 1);
                    }
                    s->len++;
                }
                s->num_bits = 0;
            }
        }
    }
    s->bit_buf <<= 1;
}
/*- End of function --------------------------------------------------------*/

static int restart_rx_modem(t38_state_t *s)
{
    span_log(&s->logging, SPAN_LOG_FLOW, "Restart modem\n");
    hdlc_rx_init(&(s->hdlcrx), FALSE, TRUE, 5, NULL, s);

    s->rx_signal_present = FALSE;
    s->data_final = FALSE;
    fsk_rx_init(&(s->v21rx), &preset_fsk_specs[FSK_V21CH2], TRUE, (put_bit_func_t) t38_hdlc_rx_bit, &(s->hdlcrx));
    s->v21_rx_active = TRUE;
    switch (s->fast_modem)
    {
#if defined(ENABLE_V17)
    case T38_V17_RX:
        v17_rx_restart(&(s->v17rx), s->bit_rate, s->short_train);
        s->fast_rx_active = T38_V17_RX;
        break;
#endif
    case T38_V27TER_RX:
        v27ter_rx_restart(&(s->v27ter_rx), s->bit_rate, FALSE);
        s->fast_rx_active = T38_V27TER_RX;
        break;
    case T38_V29_RX:
        v29_rx_restart(&(s->v29rx), s->bit_rate, FALSE);
        s->fast_rx_active = T38_V29_RX;
        break;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t38_rx(t38_state_t *s, const int16_t *amp, int len)
{
    s->since_last_tx_packet += len;
    if (s->v21_rx_active)
        fsk_rx(&(s->v21rx), amp, len);
    switch (s->fast_rx_active)
    {
#if defined(ENABLE_V17)
    case T38_V17_RX:
        v17_rx(&(s->v17rx), amp, len);
        break;
#endif
    case T38_V27TER_RX:
        v27ter_rx(&(s->v27ter_rx), amp, len);
        break;
    case T38_V29_RX:
        v29_rx(&(s->v29rx), amp, len);
        break;
    }
    return  0;
}
/*- End of function --------------------------------------------------------*/

int t38_tx(t38_state_t *s, int16_t *amp, int max_len)
{
    int len;

    len = 0;
    if (s->silence == 0)
        return len;
    switch (s->current_rx_indicator)
    {
    case T38_IND_NO_SIGNAL:
        if (s->silence > 0)
        {
            len = max_len;
            if (len > s->silence)
                len = s->silence;
            s->silence -= len;
            memset(amp, 0, len*sizeof(int16_t));
            if (s->silence == 0)
                set_next_tx_type(s);
        }
        break;
    case T38_IND_CED:
        if ((len = tone_gen(&(s->tone_gen), amp + len, max_len)) <= 0)
        {
            s->silence = 0;
            set_next_tx_type(s);
        }
        break;
    case T38_IND_CNG:
        len = tone_gen(&(s->tone_gen), amp + len, max_len);
        break;
    case T38_IND_V21_PREAMBLE:
        len = fsk_tx(&(s->v21tx), amp + len, max_len);
        break;
#if defined(ENABLE_V17)
    case T38_IND_V17_7200_LONG_TRAINING:
    case T38_IND_V17_9600_LONG_TRAINING:
    case T38_IND_V17_12000_LONG_TRAINING:
    case T38_IND_V17_14400_LONG_TRAINING:
    case T38_IND_V17_7200_SHORT_TRAINING:
    case T38_IND_V17_9600_SHORT_TRAINING:
    case T38_IND_V17_12000_SHORT_TRAINING:
    case T38_IND_V17_14400_SHORT_TRAINING:
        if ((len = v17_tx(&(s->v17tx), amp + len, max_len)) <= 0)
        {
            s->silence = 0;
            set_next_tx_type(s);
        }
        break;
#endif
    case T38_IND_V27TER_2400_TRAINING:
    case T38_IND_V27TER_4800_TRAINING:
        if ((len = v27ter_tx(&(s->v27ter_tx), amp + len, max_len)) <= 0)
        {
            s->silence = 0;
            set_next_tx_type(s);
        }
        break;
    case T38_IND_V29_7200_TRAINING:
    case T38_IND_V29_9600_TRAINING:
        if ((len = v29_tx(&(s->v29tx), amp + len, max_len)) <= 0)
        {
            s->silence = 0;
            set_next_tx_type(s);
        }
        break;
    }
    return len;
}
/*- End of function --------------------------------------------------------*/

t38_state_t *t38_gateway_init(t38_state_t *s,
                              int calling_party,
                              t38_tx_packet_handler_t *tx_packet_handler,
                              void *tx_packet_user_data)
{
    memset(s, 0, sizeof(*s));
    if (tx_packet_handler == NULL)
        return NULL;

    s->gateway = TRUE;
#if defined(ENABLE_V17)
    v17_rx_init(&(s->v17rx), 14400, non_ecm_putbit_v17, s);
    v17_tx_init(&(s->v17tx), 14400, FALSE, non_ecm_getbit, s);
#endif
    v29_rx_init(&(s->v29rx), 9600, non_ecm_putbit_v29, s);
    v29_tx_init(&(s->v29tx), 9600, FALSE, non_ecm_getbit, s);
    v27ter_rx_init(&(s->v27ter_rx), 4800, non_ecm_putbit_v27ter, s);
    v27ter_tx_init(&(s->v27ter_tx), 4800, FALSE, non_ecm_getbit, s);
    s->rx_signal_present = FALSE;

    s->data_rate_management_method = 2;
    s->data_transport_protocol = T38_TRANSPORT_UDPTL;
    s->fill_bit_removal = FALSE;
    s->mmr_transcoding = FALSE;
    s->jbig_transcoding = FALSE;
    s->max_buffer_size = 400;
    s->max_datagram_size = 100;
    s->t38_version = 0;
    s->iaf = FALSE;

    s->samples = 0;
    s->next_send_samples = 0;

    s->silence = 0;
    s->at_tcf = TRUE;
    restart_rx_modem(s);
    s->tx_packet_handler = tx_packet_handler;
    s->tx_packet_user_data = tx_packet_user_data;
    return s;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
