/*
 * Copyright 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

package sun.security.jca;

import java.lang.ref.*;

import java.security.*;

/**
 * Collection of static utility methods used by the security framework.
 *
 * @version 1.9, 05/05/07
 * @author  Andreas Sterbenz
 * @since   1.5
 */
public final class JCAUtil {
    
    private JCAUtil() {
	// no instantiation
    }
    
    // lock to use for synchronization
    private static final Object LOCK = JCAUtil.class;
    
    // cached SecureRandom instance
    private static volatile SecureRandom secureRandom;
    
    // size of the temporary arrays we use. Should fit into the CPU's 1st 
    // level cache and could be adjusted based on the platform
    private final static int ARRAY_SIZE = 4096;
    
    /**
     * Get the size of a temporary buffer array to use in order to be
     * cache efficient. totalSize indicates the total amount of data to 
     * be buffered. Used by the engineUpdate(ByteBuffer) methods.
     */
    public static int getTempArraySize(int totalSize) {
	return Math.min(ARRAY_SIZE, totalSize);
    }
    
    /**
     * Get a SecureRandom instance. This method should me used by JDK
     * internal code in favor of calling "new SecureRandom()". That needs to
     * iterate through the provider table to find the default SecureRandom
     * implementation, which is fairly inefficient.
     */
    public static SecureRandom getSecureRandom() {
	// we use double checked locking to minimize synchronization
	// works because we use a volatile reference
	SecureRandom r = secureRandom;
	if (r == null) {
	    synchronized (LOCK) {
		r = secureRandom;
		if (r == null) {
		    r = new SecureRandom();
		    secureRandom = r;
		}
	    }
	}
	return r;
    }
    
}

