# -*- coding: utf-8 -*-
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# Library modules import
#-------------------------------------------------------------------------------

import sys, unittest

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.XMLvariables import Model
from Base.XMLmodel import ModelTest
from Base.XMLengine import *
from Pages.DefineUserScalarsModel import DefineUserScalarsModel
from Pages.ThermalScalarModel import ThermalScalarModel

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

class Boundary(object) :
    """
    Abstract class 
    """
    def __new__(cls, nature , label, case) :
        """
        Factory
        """
        if nature == 'inlet':
            return InletBoundary.__new__(InletBoundary, label, case)
        elif nature == 'coal_inlet':
            from CoalCombustionModel import CoalCombustionModel
            Model().isNotInList(CoalCombustionModel(case).getCoalCombustionModel(), ("off",))
            return CoalInletBoundary.__new__(CoalInletBoundary, label, case)
        elif nature == 'outlet':
            return OutletBoundary.__new__(OutletBoundary, label, case)
        elif nature == 'symmetry':
            return SymmetryBoundary.__new__(SymmetryBoundary, label, case)
        elif nature == 'wall':
            return WallBoundary.__new__(WallBoundary, label, case)
        elif nature == 'radiative_wall':
            Model().isNotInList(case.xmlInitNode('radiative_transfer')['model'], ("off",))
            return RadiativeWallBoundary.__new__(RadiativeWallBoundary, label, case)
        else :
            raise ValueError, "Unknown boundary nature"


    def __init__(self, nature , label, case) :
        """
        """
        self._label = label
        self._nature = nature
        self._case = case
        self._XMLBoundaryConditionsNode = self._case.xmlGetNode('boundary_conditions')
        self._thermalLabelsList = ('temperature_celsius', 'temperature_kelvin', 'enthalpy')

        self.sca_model = DefineUserScalarsModel(self._case)

        #
        # Create nodes
        if nature != "coal_inlet" and nature != "radiative_wall":
            self.boundNode = self._XMLBoundaryConditionsNode.xmlInitNode(nature, label = label)
        elif nature == "coal_inlet":
            self.boundNode = self._XMLBoundaryConditionsNode.xmlInitNode('inlet', label = label)
        elif nature == "radiative_wall":
            self.boundNode = self._XMLBoundaryConditionsNode.xmlInitNode('wall', label = label)
        self._initBoundary()


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the boundary node (vitual method)
        """
        pass


    def updateScalarTypeAndName(self, scalarNode, label):
        """
        Check and update type and name of scalar labelled label for boundary conditions for wall
        """
        #update name and type of scalar

        if self.sca_model.getScalarType(label) == 'thermal':
            Model().isInList(self.sca_model.getScalarName(label), self._thermalLabelsList)
        elif self.sca_model.getScalarType(label) == 'user':
            Model().isInList(self.sca_model.getScalarName(label)[:6], ('scalar')) 
        scalarNode['name'] = self.sca_model.getScalarName(label)
        scalarNode['type'] = self.sca_model.getScalarType(label)


    def delete(self):
        """
        Delete Boundary
        """
        self.boundNode.xmlRemoveNode()


class InletBoundary(Boundary) :
    """
    """
    def __new__(cls, label, case) :
        """
        Constructor
        """
        return object.__new__(cls)


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the boundary node
        """
        self.__velocityChoices = [ 'norm', 'flow1', 'flow2', 
                                   'norm+direction', 'flow1+direction', 'flow2+direction']
        self.__turbulenceChoices = ['hydraulic_diameter', 'turbulent_intensity']
        self.__flowTagList = ['flow1', 'flow2', 'flow1+direction', 'flow2+direction']
        self.__componentsList = ['direction_x', 'direction_y', 'direction_z']

        self.th_model = ThermalScalarModel(self._case)

        # Initialize nodes if necessary
        self.getVelocityChoice()

        # Turbulence
        self.getTurbulenceChoice()

        # Scalars
        for label in self.sca_model.getScalarLabelsList():
            self.getScalar(label)

        from CoalCombustionModel import CoalCombustionModel
        if CoalCombustionModel(self._case).getCoalCombustionModel() =="off":
            self.boundNode.xmlRemoveChild('coal_flow')
            n = self.boundNode.xmlGetNode('temperature')
            if n:
                n.xmlRemoveNode()


    def __deleteNormNodes(self, VelocityNode):
        """ Delete nodes which depends of norm"""
        node = VelocityNode.xmlGetChildNode('norm')
        if node :
            node.xmlRemoveNode()


    def __deleteFlow1Nodes(self, VelocityNode):
        """ Delete nodes which depends of mass flow"""
        node = VelocityNode.xmlGetChildNode('flow1')
        if node :
            node.xmlRemoveNode()


    def __deleteFlow2Nodes(self, VelocityNode):
        """ Delete nodes which depends of volumic flow"""
        node = VelocityNode.xmlGetChildNode('flow2')
        if node :
            node.xmlRemoveNode()


    def isDirection(self):
        """ verify if direction is selected or not"""
        dir = 'off'
        node = self.boundNode.xmlGetNode('velocity_pressure')

        for component in self.__componentsList:
            value = node.xmlGetChildDouble(component)
            if value != None :
                dir = 'on'

        return dir


    def defaultValues(self):
        """
        Default values
        """
        dico = {}
        dico['velocityChoice'] = 'norm'
        dico['turbulenceChoice'] = 'hydraulic_diameter'
        dico['hydraulic_diameter'] = 1
        dico['turbulent_intensity'] = 2
        dico['velocity'] = 0.0
        dico['flow'] = 1
        dico['norm'] = 1
        dico['scalar'] = 0.0
        dico['scalarChoice'] = 'dirichlet'

        return dico


    def deleteDirectionNodes(self):
        """ Delete nodes which depends of directions"""
        self.updateVelocityChoiceForDirection(0)
        VelocityNode = self.boundNode.xmlInitNode('velocity_pressure')
        node = VelocityNode.xmlGetChildNode('direction_x')
        if node :
            node.xmlRemoveNode()
        node = VelocityNode.xmlGetChildNode('direction_y')
        if node :
            node.xmlRemoveNode()
        node = VelocityNode.xmlGetChildNode('direction_z')
        if node :
            node.xmlRemoveNode()

        # update choice of velocity
        choice = self.getVelocityChoice()
        choice = string.split(choice, '+')[0]
        self.setVelocityChoice(choice)


    def getVelocity(self):
        """
        Get value of velocity beyond choice - Global method for the view
        """
        choice = self.getVelocityChoice()
        Model().isInList(choice, self.__velocityChoices)
        if choice in ('norm', 'norm+direction'):
            value = self.getNorm()
        elif choice in ('flow1', 'flow1+direction'):
            value = self.getFlow('flow1')
        elif choice in ('flow2', 'flow2+direction'):
            value = self.getFlow('flow2')

        return value


    def setVelocity(self, value):
        """
        Set value of velocity beyond choice - Global method for the view
        """
        Model().isFloat(value)

        choice = self.getVelocityChoice()
        Model().isInList(choice, self.__velocityChoices)

        if choice in ('norm', 'norm+direction'):
            self.setNorm(value)
        elif choice in ('flow1', 'flow1+direction'):
            self.setFlow('flow1', value)

        elif choice in ('flow2', 'flow2+direction'):
            self.setFlow('flow2', value)


    def getVelocityChoice(self):
        """
        Get the choice velocity
        """
        node = self.boundNode.xmlInitNode('velocity_pressure', 'choice')
        choice = node['choice']
        if not node['choice']:
            choice = self.defaultValues()['velocityChoice']
            self.setVelocityChoice(choice)

        return choice


    def updateVelocityChoiceForDirection(self, dir):
        """
        Update choice of velocity beyond dir=1 (with direction) or dir=0 (without direction)
        """
        Model().isInList(dir, (0,1))
        if dir == 1:
            choice = self.getVelocityChoice()
            if choice[-9:] != 'direction':
                self.setVelocityChoice(choice + "+" + 'direction')
        elif dir == 0:
            choice = self.getVelocityChoice()
            if choice[-9:] == 'direction':
                self.setVelocityChoice(choice[:-10])


    def setVelocityChoice(self, value):
        """
        Set the velocity definition according to choice
        """
        Model().isInList(value, self.__velocityChoices)

        if self.isDirection() == 'on' and value[-9:] != 'direction':
            value = value + "+" + 'direction'
        #
        # Check if value is a new velocity choice value
        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')
        if XMLVelocityNode['choice'] != None :
            if XMLVelocityNode['choice'] == value:
                return
        #
        # Update velocity choice
        XMLVelocityNode['choice'] = value
        #
        # Velocities updating
        if value == 'norm' :
            #
            # Create norm node if necessary
            self.getNorm()

            # Delete 'flow1', 'flow2' and 'direction' nodes
            self.__deleteFlow1Nodes(XMLVelocityNode)
            self.__deleteFlow2Nodes(XMLVelocityNode)
            self.deleteDirectionNodes()

        elif value == 'norm+direction' :
            #
            # Create norm node if necessary
            self.getNorm()
            #
            # Create direction nodes if necessary
            self.getDirection('direction_x')
            self.getDirection('direction_y')
            self.getDirection('direction_z')
            #
            # Delete 'flow1' and 'flow2'" nodes
            self.__deleteFlow1Nodes(XMLVelocityNode)
            self.__deleteFlow2Nodes(XMLVelocityNode)

        elif value == 'flow1' :
            # Create flow1 node if necessary
            self.getFlow('flow1')

            # Delete 'norm', 'flow2' and 'directio'n nodes
            self.__deleteNormNodes(XMLVelocityNode)
            self.__deleteFlow2Nodes(XMLVelocityNode)
            self.deleteDirectionNodes()

        elif value == 'flow1+direction' :
            # Create flow1 node if necessary
            self.getFlow('flow1')

            # Create direction nodes if necessary
            self.getDirection('direction_x')
            self.getDirection('direction_y')
            self.getDirection('direction_z')
            #
            # Delete 'norm', 'flow2'nodes
            self.__deleteNormNodes(XMLVelocityNode)
            self.__deleteFlow2Nodes(XMLVelocityNode)

        elif value == 'flow2' :
            # Create flow2 node if necessary
            self.getFlow('flow2')

            # Delete 'norm', 'flow1' and 'directio'n nodes
            self.__deleteNormNodes(XMLVelocityNode)
            self.__deleteFlow1Nodes(XMLVelocityNode)
            self.deleteDirectionNodes()

        elif value == 'flow2+direction' :
            # Create flow2 node if necessary
            self.getFlow('flow2')

            # Create direction nodes if necessary
            self.getDirection('direction_x')
            self.getDirection('direction_y')
            self.getDirection('direction_z')
            #
            # Delete 'norm', 'flow2'nodes
            self.__deleteNormNodes(XMLVelocityNode)
            self.__deleteFlow1Nodes(XMLVelocityNode)


    def getTurbulenceChoice(self):
        """
        Get the turbulence choice
        """
        XMLTurbulenceNode = self.boundNode.xmlInitNode('turbulence')

        choice = XMLTurbulenceNode['choice']
        if choice not in self.__turbulenceChoices :
            choice = self.defaultValues()['turbulenceChoice']
            self.setTurbulenceChoice(choice)

        return choice


    def setTurbulenceChoice(self, value):
        """
        Set the choice turbulence
        """
        Model().isInList(value, self.__turbulenceChoices)

        XMLTurbulenceNode = self.boundNode.xmlInitNode('turbulence')

        if XMLTurbulenceNode['choice'] != None:
            if XMLTurbulenceNode['choice'] == value:
                return

        XMLTurbulenceNode['choice'] = value
        #
        # Update values
        if value == 'hydraulic_diameter' :
            self.getHydraulicDiameter()
            #
            # Delete intensity node
            node = XMLTurbulenceNode.xmlGetChildNode('turbulent_intensity')
            if node :
                node.xmlRemoveNode()

        elif value == 'turbulent_intensity' :
            #
            # Create velocities nodes if necessary
            self.getHydraulicDiameter()
            self.getTurbulentIntensity()


    def getHydraulicDiameter(self):
        """
        Get hydraulic diameter
        """
        XMLTurbulenceNode = self.boundNode.xmlInitNode('turbulence')
        Model().isInList(XMLTurbulenceNode['choice'],  self.__turbulenceChoices)
        value = XMLTurbulenceNode.xmlGetDouble('hydraulic_diameter')
        if value == None :
            value = self.defaultValues()['hydraulic_diameter']
            self.setHydraulicDiameter(value)
        return value


    def setHydraulicDiameter(self, value):
        """
        Set hydraulic diameter
        """
        Model().isStrictPositiveFloat(value)

        XMLTurbulenceNode = self.boundNode.xmlInitNode('turbulence')
        Model().isInList(XMLTurbulenceNode['choice'],  self.__turbulenceChoices)
        XMLTurbulenceNode.xmlSetData('hydraulic_diameter', value)


    def getTurbulentIntensity(self):
        """
        Get turbulent intensity
        """
        XMLTurbulenceNode = self.boundNode.xmlInitNode('turbulence')
        Model().isInList(XMLTurbulenceNode['choice'], ('turbulent_intensity',))
        value = XMLTurbulenceNode.xmlGetDouble('turbulent_intensity')
        if value == None :
            value = self.defaultValues()['turbulent_intensity']
            self.setTurbulentIntensity(value)

        return value


    def setTurbulentIntensity(self, value):
        """
        Set turbulent intensity
        """
        Model().isStrictPositiveFloat(value)

        XMLTurbulenceNode = self.boundNode.xmlInitNode('turbulence')
        Model().isInList(XMLTurbulenceNode['choice'], ('turbulent_intensity',))
        XMLTurbulenceNode.xmlSetData('turbulent_intensity', value)


    def getFlow(self, tag):
        """
        Get value for mass flow ('flow1') or volumic flow('flow2')
        """
        Model().isInList(tag, self.__flowTagList)
        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')

        flow = XMLVelocityNode.xmlGetChildDouble(tag)

        if flow == None:
            flow = self.defaultValues()['flow']
            self.setFlow(tag, flow)
        return flow


    def setFlow(self, tag, value):
        """
        Set value for mass flow ('flow1') or volumic flow('flow2')
        """
        Model().isInList(tag, self.__flowTagList)
        Model().isFloat(value)
        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')

        # Flow updating
        XMLVelocityNode.xmlSetData(tag, value)


    def getNorm(self):
        """
        Get norm for velocity
        """
        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(XMLVelocityNode['choice'], ('norm', 'norm+direction'))

        norm = XMLVelocityNode.xmlGetDouble('norm')
        if norm == None:
            norm = self.defaultValues()['norm']
            self.setNorm(norm)
        return norm


    def setNorm(self, value):
        """
        Set norm for velocity
        """
        Model().isFloat(value)

        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(XMLVelocityNode['choice'], ('norm', 'norm+direction'))

        # Flow updating
        XMLVelocityNode.xmlSetData('norm', value)


    def getDirection(self, component):
        """
        Get the component velocity
        """
        Model().isInList(component, self.__componentsList)

        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(XMLVelocityNode['choice'], ('norm+direction', 'flow1+direction', 'flow2+direction'))

        # Get velocity
        value = XMLVelocityNode.xmlGetChildDouble(component)
        if value == None :
            value = self.defaultValues()['velocity']
            self.setDirection(component, value)
        return value


    def setDirection(self, component, value):
        """
        Set the component velocity for fieldLabel
        """
        Model().isInList(component, self.__componentsList)
        Model().isFloat(value)

        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(XMLVelocityNode['choice'], ('norm+direction', 'flow1+direction', 'flow2+direction'))

        # Update velocity
        XMLVelocityNode.xmlSetData(component, value)


    def getScalar(self, scalarLabel) :
        """
        Get scalar value
        """
        Model().isInList(scalarLabel, self.sca_model.getScalarLabelsList())

        scalarNode = self.boundNode.xmlInitNode('scalar', choice="dirichlet", label=scalarLabel)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, scalarLabel)

        value = scalarNode.xmlGetChildDouble('dirichlet')
        if value == None :
            value = self.defaultValues()['scalar']
            self.setScalar(scalarLabel, value)
        return value


    def setScalar(self, scalarLabel, value):
        """
        Set scalar value
        """
        Model().isInList(scalarLabel, self.sca_model.getScalarLabelsList())
        Model().isFloat(value)

        scalarNode = self.boundNode.xmlInitNode('scalar', choice="dirichlet", label=scalarLabel)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, scalarLabel)

        scalarNode.xmlSetData('dirichlet', value)


    def getScalarImposedValue(self, label):
        return self.getScalar(label)


    def setScalarImposedValue(self, label, value):
        self.setScalar(label, value)


class CoalInletBoundary(Boundary) :
    """
    """
    def __new__(cls, label, case) :
        """
        Constructor
        """
        return object.__new__(cls)


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the boundary node
        """
        self.typeList = ['airflow', 'coalflow']

        from CoalThermoChemistry import CoalThermoChemistryModel
        coalThermoChModel = CoalThermoChemistryModel("dp_FCP", self._case)
        self.coalNumber = coalThermoChModel.getCoals().getCoalNumber()
        self.coalClassesNumber = []
        for coal in range(0, self.coalNumber):
            self.coalClassesNumber.append(coalThermoChModel.getCoals().getCoal(coal+1).getClassesNumber())

        # Initialize nodes if necessary
        self.getCoalType()


    def _deleteCoalNodes(self):
        """
        Delete all nodes udes for coal. Private method
        """
        node = self.boundNode.xmlInitNode('velocity_pressure')
        for n in node.xmlGetChildNodeList('coal_flow'):
            n.xmlRemoveNode()


    def _getClassCoalRatio(self, coal, classe):
        """
        Return ratio for classe for coal. Private method
        """
        Model().isInt(coal)
        Model().isInt(classe)

        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        nc = n.xmlInitNode('coal_flow', name="coal"+num)
        num = '%2.2i' % (classe+1)
        nratio = nc.xmlGetChildNode('ratio', name="class"+num)
        if nratio:
            ratio = nc.xmlGetChildDouble('ratio', name="class"+num)
        else:
            if self.coalClassesNumber[coal] > 1:
                if classe == 0:
                    ratio = 100.
                    self._setClassCoalRatio(ratio, coal, classe)
                else:
                    ratio = self.defaultValues()['ratio']
                    self._setClassCoalRatio(ratio, coal, classe)
            else:
                ratio = 100.
                self._setClassCoalRatio(ratio, coal, classe)

        return ratio


    def _setClassCoalRatio(self, value, coal, classe):
        """
        Put value of ratio when several classes for coal. Private method
        """
        Model().isFloat(value)
        Model().isLowerOrEqual(value, 100.)
        Model().isInt(coal)

        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        nc = n.xmlInitNode('coal_flow', name="coal"+ num)

        num = '%2.2i' % (classe+1)
        nc.xmlSetData('ratio', value, name="class"+ num)


    def _verifySumRatios(self, coal, list):
        """
        Verify the addition of all ratios for one coal is equal to 100. Private method
        """
        Model().isInt(coal)
        Model().isList(list)

        som = 0.
        for i in range(0, self.coalClassesNumber[coal]):
            som += list[i]
        Model().isFloatEqual(som, 100.)


    def defaultValues(self):
        """
        Default values
        """
        dico = {}
        dico['flow'] = 1
        dico['ratio'] = 0.0
        from ReferenceValuesModel import ReferenceValuesModel
        dico['temperature'] = ReferenceValuesModel(self._case).getTemperature()

        return dico


    def getCoalType(self):
        """
        Return type (air or air+coal) for velocities's boundary conditions for inlet coal flow.
        """
        node = self.boundNode.xmlInitNode('velocity_pressure')
        n = node.xmlGetChildNodeList('coal_flow')
        if n:
            type = "coalflow"
        else:
            type = "airflow"
        return type


    def setCoalType(self, type):
        """
        Set type (air or air+coal) for velocities's boundary conditions for inlet coal flow.
        """
        Model().isInList(type, self.typeList)
        node = self.boundNode.xmlInitNode('velocity_pressure')

        self.getAirTemperature()
        if type == "airflow":
            self._deleteCoalNodes()
        else:
            for coal in range(0, self.coalNumber):
                self.getCoalFlow(coal)
                self.getCoalTemperature(coal)
                self.getCoalRatios(coal)


    def getCoalFlow(self, coal):
        """
        Return value of flow for coal
        """
        Model().isInt(coal)
        
        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        n2 = n.xmlInitNode('coal_flow', name="coal"+num)
        flow = n2.xmlGetDouble('flow1')
        if flow == None:
            flow = self.defaultValues()['flow']
            self.setCoalFlow(flow, coal)
            
        return flow


    def setCoalFlow(self, value, coal):
        """
        Put value of flow for coal
        """
        Model().isFloat(value)
        Model().isInt(coal)
        
        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        n2 = n.xmlGetNode('coal_flow', name="coal"+num)
        if n2:
            n2.xmlSetData('flow1',value)


    def getAirTemperature(self) :
        """
        Return value of the temperature for air for coal choice
        """
        n = self.boundNode.xmlInitNode('velocity_pressure')
        temperature = n.xmlGetChildDouble('temperature')
        if temperature == None:
            temperature = self.defaultValues()['temperature']
            self.setAirTemperature(temperature)

        return temperature


    def setAirTemperature(self, value) :
        """
        Set value of the temperature for air for coal choice
        """
        Model().isFloat(value)
        
        n = self.boundNode.xmlInitNode('velocity_pressure')
        n.xmlSetData('temperature',value)


    def getCoalTemperature(self, coal):
        """
        Return value of temperature for coal for coal choice
        """
        Model().isInt(coal)
        
        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        nt = n.xmlInitNode('coal_flow', name="coal"+ num)
        temperature = nt.xmlGetChildDouble('temperature')
        if temperature == None:
            temperature = self.defaultValues()['temperature']
            self.setCoalTemperature(temperature, coal)
    
        return temperature


    def setCoalTemperature(self, value, coal):
        """
        Put value of temperature for coal for coal choice
        """
        Model().isFloat(value)
        Model().isInt(coal)
        
        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        nc = n.xmlGetNode('coal_flow', name="coal"+ num)
        if nc:
            nc.xmlSetData('temperature',value)


    def getCoalRatios(self, coal):
        """
        Put list of values of classe's ratio for one coal
        """
        Model().isInt(coal)
        list = []

        for i in range(0, self.coalClassesNumber[coal]):
            list.append(self._getClassCoalRatio(coal, i))
        self._verifySumRatios(coal, list)

        return list


    def setCoalRatios(self, coal, list):
        """
        Put list of values of classe's ratio for one coal
        """
        Model().isInt(coal)
        Model().isIntEqual(len(list), self.coalClassesNumber[coal])
        som = 0.
        for i in range(0, self.coalClassesNumber[coal]):
            som += list[i]
        Model().isFloatEqual(som, 100.)
        
        n = self.boundNode.xmlInitNode('velocity_pressure')
        num = '%2.2i' % (coal+1)
        nc = n.xmlInitNode('coal_flow', name="coal"+ num)
        
        for i in range(0, len(list)):
            num = '%2.2i' % (i+1)
            nc.xmlSetData('ratio', list[i], name="class"+ num)

   
    # *********  the following methods are only used by CoalCombustionView *****
    def deleteCoalFlow(self, coal, nbCoals):
        """ 
        delete coal_flow with name = coaln
        """
        Model().isInt(coal)
        Model().isInt(nbCoals)
        
        n = self.boundNode.xmlGetNode('velocity_pressure')
        if n:
            num = '%2.2i' % (coal+1)
            n2 = n.xmlGetNode('coal_flow', name="coal"+ num)
            # delete coal
            if n2:
                n2.xmlRemoveNode()
                # rename other coals
                for icoal in range(coal+1, nbCoals):
                    self._renameCoalFlow(icoal)


    def _renameCoalFlow(self, coal):
        """ 
        coaln become coaln-1. Private method
        """ 
        Model().isInt(coal)
        
        n = self.boundNode.xmlGetNode('velocity_pressure')
        if n:
            num = '%2.2i' % (coal+1)
            newNum = '%2.2i' % coal
            n2 = n.xmlGetNode('coal_flow', name="coal"+ num)
            if n2:
                n2['name'] = "coal"+newNum 


    def deleteClassRatio(self, coal, classe, nbClasses):
        """ 
        delete ratio with name = classe
        """ 
        Model().isInt(coal)
        Model().isInt(classe)
        Model().isInt(nbClasses)
        
        n = self.boundNode.xmlGetNode('velocity_pressure')
        if n:
            num = '%2.2i' % (coal+1)
            n2 = n.xmlGetNode('coal_flow', name="coal"+num)
            if n2:
                n2.xmlRemoveChild('ratio')
                self.getCoalRatios(coal)


class OutletBoundary(Boundary) :
    """
    """
    def __new__(cls, label, case) :
        """
        Constructor
        """
        return object.__new__(cls)


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the boundary node
        """
        self._scalarChoicesList = ['dirichlet', 'neumann']
        
        self.getReferencePressure()
        
        for label in self.sca_model.getScalarLabelsList():
            self.getScalar(label)


    def __deleteScalarNodes(self, label, tag):
        """
        Delete nodes of scalars 
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(tag, self._scalarChoicesList)
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
        
        for tt in self._scalarChoicesList:
            if tt != tag:
                scalarNode.xmlRemoveChild(tt)


    def defaultValues(self):
        """
        Default values
        """
        dico = {}
        from ReferenceValuesModel import ReferenceValuesModel
        dico['reference_pressure'] = ReferenceValuesModel(self._case).getPressure()
        dico['scalarChoice'] = 'neumann'
        dico['scalar'] = 0.

        return dico


    def getScalarChoice(self, label):
        """
        Get scalar choice
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())

        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
        
        #update type and name of scalar
        self.updateScalarTypeAndName(scalarNode, label)

        choice = scalarNode['choice']
        if not choice:
            choice = self.defaultValues()['scalarChoice']
            self.setScalarChoice(label, choice)
            
        return choice


    def setScalarChoice(self, label, choice) :
        """
        Set scalar choice
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(choice, self._scalarChoicesList)
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        if scalarNode['choice'] == choice:
            return  
            
        scalarNode['choice'] = choice
        if choice == 'dirichlet':
            self.getScalar(label)
            self.__deleteScalarNodes(label, 'dirichlet')
        elif choice == 'neumann':
            self.setScalar(label, 0.)
            self.__deleteScalarNodes(label, 'neumann')


    def getScalar(self, label) :
        """
        Get variableName variable
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
        
        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        choice = self.getScalarChoice(label)

        value = scalarNode.xmlGetChildDouble(choice)
        if value == None :
            value = self.defaultValues()['scalar']
            self.setScalar(label, value)
        return value


    def setScalar(self, label, value) :
        """
        Set variableName variable
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isFloat(value)
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        choice = self.getScalarChoice(label)
        if choice == 'neumann':
            Model().isInList(value, (0.,))
        Model().isInList(choice, self._scalarChoicesList)

        scalarNode.xmlSetData(choice, value)


    def getScalarImposedValue(self, label):
        Model().isInList(self.getScalarChoice(label), ('dirichlet',))
        return self.getScalar(label)


    def setScalarImposedValue(self, label, value):
        Model().isInList(self.getScalarChoice(label), ('dirichlet',))
        self.setScalar(label, value)


    def getScalarImposedFlux(self, label):
        Model().isInList(self.getScalarChoice(label), ('neumann',))
        return self.getScalar(label)


    def setScalarImposedFlux(self, label, value):
        Model().isInList(self.getScalarChoice(label), ('neumann',))
        self.setScalar(label, value)


    def getPressureChoice(self) :
        """
        Return if the value of pressure exist or not of boundary conditions for outlet.
        """
        choice = "off"
        if self.boundNode.xmlGetChildNode('dirichlet', name='reference_pressure'):
            choice = "on"

        return choice


    def setPressureChoice(self, choice) :
        """
        Set balise of pressure beyond the choice for boundary conditions for outlet
        """
        Model().isOnOff(choice)
        if choice == 'off':
            self.setReferencePressure(self, 0.0)
        else:
            if node.xmlGetDouble('dirichlet', name='pressure') == None:
                self.setReferencePressure(self.defaultValues()['pressure'])


    def getReferencePressure(self) :
        """
        Get reference pressure
        """
        pressure = self.boundNode.xmlGetDouble('dirichlet', name='pressure')
        if pressure == None: 
            return 0
            
        return pressure


    def setReferencePressure(self, value) :
        """
        Set reference pressure
        """
        Model().isPositiveFloat(value)
        
        node = self.boundNode.xmlInitNode('dirichlet', name='pressure')
        if value == 0:
            node.xmlRemoveNode()
        else:
            self.boundNode.xmlSetData('dirichlet', value, name='pressure')


class SymmetryBoundary(Boundary) :
    """
    """
    def __new__(cls, label, case) :
        """
        Constructor
        """
        return object.__new__(cls)


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the boundary node
        """
        pass


class WallBoundary(Boundary) :
    """
    """
    def __new__(cls, label, case) :
        """
        Constructor
        """
        return object.__new__(cls)


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the boundary node
        """
        self._fluxChoices=['temperature', 'flux']
        self._scalarChoices = ['dirichlet', 'neumann', 'exchange_coefficient']
        
        # Initialize nodes if necessary
        self.getVelocityChoice()
        self.getRoughnessChoice()

        # Scalars
        for label in self.sca_model.getScalarLabelsList():
            self.getScalarChoice(label)


    def __deleteVelocities(self, node):
        """
        Delete nodes of velocity 
        """
        node.xmlRemoveChild('dirichlet', name='velocity_U')
        node.xmlRemoveChild('dirichlet', name='velocity_V')
        node.xmlRemoveChild('dirichlet', name='velocity_W')


    def __deleteScalarNodes(self, label, tag):
        """
        Delete nodes of scalars 
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(tag, self._scalarChoices)
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
        
        if tag == 'exchange_coefficient':
            scalarNode.xmlRemoveChild('dirichlet')
        else:
            for tt in ('dirichlet', 'neumann'):
                if tt != tag:
                    scalarNode.xmlRemoveChild(tt)
                    scalarNode.xmlRemoveChild('exchange_coefficient')


    def defaultValues(self):
        """
        Default values
        """
        dico = {}
        dico['velocityChoice'] = "off"
        dico['velocityValue']  = 0.
        dico['scalarChoice']   = "neumann"
        dico['scalarValue']   = 0.
        dico['roughness'] = 0.01
        dico['flux'] = 0
        return dico


    def getVelocityChoice(self):
        """
        Get the velocity choice 
        """
        node = self.boundNode.xmlInitNode('velocity_pressure', 'choice')
        choice = node['choice']
        if not choice:
            choice = self.defaultValues()['velocityChoice']
            self.setVelocityChoice(choice)
        return node['choice']


    def setVelocityChoice(self, choice):
        """
        Set the velocity choice
        """
        Model().isOnOff(choice)
        #
        # Check if value is a new velocity choice value
        XMLVelocityNode = self.boundNode.xmlInitNode('velocity_pressure')

        if XMLVelocityNode['choice'] == choice:
            return

        # Update velocity choice
        XMLVelocityNode['choice'] = choice
        #
        # Velocities updating
        if choice == 'on':
            # Create norm node if necessary
            self.getVelocities()
        else:
            # Delete 'flow1', 'flow2' and 'direction' nodes
            self.__deleteVelocities(XMLVelocityNode)


    def getVelocities(self):
        """
        Set the velocity definition according to choice
        """
        node = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(node['choice'],('on',))
        
        n = node.xmlGetChildNode('dirichlet', name='velocity_U')
        if n:
            u = node.xmlGetChildDouble('dirichlet', name='velocity_U')
        else:
            u = self.defaultValues()['velocityValue']
        n = node.xmlGetChildNode('dirichlet', name='velocity_V')
        if n:
            v = node.xmlGetChildDouble('dirichlet', name='velocity_V')
        else:
            v = self.defaultValues()['velocityValue']
        n = node.xmlGetChildNode('dirichlet', name='velocity_W')
        if n:
            w = node.xmlGetChildDouble('dirichlet', name='velocity_W')
        else:
            w = self.defaultValues()['velocityValue']
        self.setVelocities(u, v, w)

        return u, v, w


    def setVelocities(self, u, v, w):
        """
        Set the velocity definition according to choice
        """
        Model().isFloat(u)
        Model().isFloat(v)
        Model().isFloat(w)
        
        node = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(node['choice'],('on',))
        
        node.xmlSetData('dirichlet', u, name='velocity_U')
        node.xmlSetData('dirichlet', v, name='velocity_V')
        node.xmlSetData('dirichlet', w, name='velocity_W')


    def setVelocityComponent(self, val, component):
        """
        Set the value of component of the velocity - Method for the view
        """
        Model().isFloat(val)
        Model().isInList(component, ('velocity_U', 'velocity_V', 'velocity_W'))
        node = self.boundNode.xmlInitNode('velocity_pressure')
        Model().isInList(node['choice'], ('on', 'off'))
        
        node.xmlSetData('dirichlet', val, name=component)


    def getRoughnessChoice(self):
        """
        Return if the value of roughness height exist or not of boundary conditions for wall.
        """
        choice = "off"
        node = self.boundNode.xmlInitNode('velocity_pressure')
        if node.xmlGetChildNode('roughness'):
            choice = "on"

        return choice


    def setRoughnessChoice(self, choice):
        """
        Update balise of roughness beyond the choice for boundary conditions for wall.
        """
        Model().isOnOff(choice)
        node = self.boundNode.xmlInitNode('velocity_pressure')
        if choice == 'off':
            self.setRoughness(0.0)
        else:
            if node.xmlGetDouble('roughness') == None:
                self.setRoughness(self.defaultValues()['roughness'])


    def getRoughness(self):
        """
        Get the value of roughness height if it's exist of boundary conditions for wall.
        """
        node = self.boundNode.xmlInitNode('velocity_pressure')

        val = node.xmlGetDouble('roughness')
        if val == None:
            return 0

        return val


    def setRoughness(self, value):
        """
        Put value of roughness height in xmlfile
        """
        Model().isGreaterOrEqual(value, 0.)
        
        node = self.boundNode.xmlInitNode('velocity_pressure', 'choice')
        if value == 0.:
            node.xmlRemoveChild('roughness')
        else:
            node.xmlSetData('roughness', value)


    def getScalarChoice(self, label):
        """
        Get scalar choice
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
        
        #update type and name of scalar
        self.updateScalarTypeAndName(scalarNode, label)

        choice = scalarNode['choice']
        if not choice:
            choice = self.defaultValues()['scalarChoice']
            self.setScalarChoice(label, choice)
            
        return choice


    def setScalarChoice(self, label, choice) :
        """
        Set scalar choice
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(choice, self._scalarChoices)
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        if scalarNode['choice'] == choice:
            return

        scalarNode['choice'] = choice
        if choice == 'dirichlet':
            self.getScalarImposedValue(label)
            self.__deleteScalarNodes(label, 'dirichlet')
        elif choice == 'neumann':
            self.getScalarImposedFlux(label)
            self.__deleteScalarNodes(label, 'neumann')
        elif choice == 'exchange_coefficient':
            self.getScalarImposedFlux(label)
            self.getScalarExchangeCoefficient(label)
            self.__deleteScalarNodes(label, 'exchange_coefficient')


    def getScalar(self, label) :
        """
        Get variableName variable
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('dirichlet', 'neumann', 'exchange_coefficient'))

        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        choice = self.getScalarChoice(label)
        if choice == 'exchange_coefficient':
            choice = 'neumann'

        value = scalarNode.xmlGetChildDouble(choice)
        if value == None :
            value = self.defaultValues()['scalar']
            self.setScalar(label, value)
        return value


    def setScalar(self, label, value) :
        """
        Set variableName variable
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isFloat(value)
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        choice = self.getScalarChoice(label)
        if choice == 'exchange_coefficient':
            choice = 'neumann'

        scalarNode.xmlSetData(choice, value)


    def getScalarImposedValue(self, label):
        """
        Get scalar dirichlet value
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('dirichlet',))
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        value = scalarNode.xmlGetChildDouble('dirichlet')
        if not value:
            value = self.defaultValues()['scalarValue']
            self.setScalarImposedValue(label, value)

        return value


    def setScalarImposedValue(self, label, value):
        """
        Set scalar dirichlet value
        """
        Model().isFloat(value)
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('dirichlet',))
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
            
        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        scalarNode.xmlSetData('dirichlet', value)


    def getScalarImposedFlux(self, label):
        """
        Get scalar neumann value
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('neumann', 'exchange_coefficient'))
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
        
        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        value = scalarNode.xmlGetChildDouble('neumann')
        if not value:
            value = self.defaultValues()['scalarValue']
            self.setScalarImposedFlux(label, value)

        return value


    def setScalarImposedFlux(self, label, value):
        """
        Set scalar neumann value
        """
        Model().isFloat(value)
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('neumann', 'exchange_coefficient'))
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        scalarNode.xmlSetData('neumann', value)


    def getScalarExchangeCoefficient(self, label):
        """
        Get scalar values for exchange coefficient
        """
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('exchange_coefficient',))
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)

        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        value = scalarNode.xmlGetChildDouble('exchange_coefficient')
        if not value:
            value = self.defaultValues()['scalarValue']
            self.setScalarExchangeCoefficient(label, value)

        return value


    def setScalarExchangeCoefficient(self, label, value):
        """
        Set scalar values for exchange coefficient
        """
        Model().isFloat(value)
        Model().isInList(label, self.sca_model.getScalarLabelsList())
        Model().isInList(self.getScalarChoice(label), ('exchange_coefficient',))
        
        scalarNode = self.boundNode.xmlInitNode('scalar', label=label)
            
        #update name and type of scalar
        self.updateScalarTypeAndName(scalarNode, label)
        
        scalarNode.xmlSetData('exchange_coefficient', value)


class RadiativeWallBoundary(Boundary) :
    """
    """
    def __new__(cls, label, case) :
        """
        Constructor
        """
        return object.__new__(cls)


    def _initBoundary(self):
        """
        Initialize the boundary, add nodes in the radiative boundary node
        """
        self._radiativeChoices = ['itpimp', 'ipgrno', 'ifgrno']

        self.head_list = ['emissivity', 'thermal_conductivity', 'thickness', 'flux', 
                          'external_temperature_profile','internal_temperature_profile', 
                          'output_zone']


    def _getListValRay(self, choice):
        """
        Return list of radiative variables according to choice's
        """
        Model().isInList(choice, self._radiativeChoices)
        list = []
        if choice == 'itpimp': 
            list = ('emissivity', 'internal_temperature_profile', 'output_zone')
        elif choice == 'ipgrno':
            list = ('emissivity', 'thermal_conductivity', 'thickness',
                    'external_temperature_profile',
                    'internal_temperature_profile', 'output_zone')
        elif choice == 'ifgrno':
            list = ('emissivity', 'flux', 'internal_temperature_profile', 'output_zone')

        return list


    def defaultValues(self):
        """
        Default values
        """
        dico = {}
        dico['emissivity']  = 0.8
        dico['thermal_conductivity']  = 3.0
        dico['thickness']  = 0.10
        dico['flux']  = 0.
        dico['external_temperature_profile']  = 300.
        dico['internal_temperature_profile']  = 300.
        dico['choice_condition']  = 'itpimp'
        dico['output_zone']  = 1
        return dico


    def getRadiativeChoice(self):
        """
        Return variables according to choice of type of condition for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        choice = nod_ray_cond['choice']
        if not choice: 
            choice = self.defaultValues()['choice_condition']
            self.setRadiativeChoice(choice)
        return choice


    def setRadiativeChoice(self, choice):
        """
        Put variables according to choice of type of condition for the radiative wall
        """
        Model().isInList(choice, self._radiativeChoices)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond['choice'] = choice
        list = self._getListValRay(choice)
        for i in list:
            if not nod_ray_cond.xmlGetChildNode(i):
                nod_ray_cond.xmlSetData(i, self.defaultValues()[i])


    def getEmissivity(self):
        """
        Return value of emissivity for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        val = nod_ray_cond.xmlGetChildDouble('emissivity')
        if not val:
            val = self.defaultValues()['emissivity']
            self.setEmissivity(val)
        
        return val


    def setEmissivity(self, val):
        """
        Put value of emissivity for the radiative wall
        """
        Model().isGreaterOrEqual(val, 0.)
        Model().isLowerOrEqual(val, 1.)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData('emissivity', val)


    def getThermalConductivity(self):
        """
        Return value of thermal conductivity for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        val = nod_ray_cond.xmlGetChildDouble('thermal_conductivity')
        if not val:
            val = self.defaultValues()['thermal_conductivity']
            self.setThermalConductivity(val)
            
        return val


    def setThermalConductivity(self, val):
        """
        Put value of thermal conductivity for the radiative wall
        """
        Model().isGreaterOrEqual(val, 0.)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData('thermal_conductivity', val)


    def getThickness(self):
        """
        Return value of thickness for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        val = nod_ray_cond.xmlGetChildDouble('thickness')
        if not val:
            val = self.defaultValues()['thickness']
            self.setThickness(val)
            
        return val


    def setThickness(self, val):
        """
        Put value of thickness for the radiative wall
        """
        Model().isGreaterOrEqual(val, 0.)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData('thickness', val)


    def getExternalTemperatureProfile(self):
        """
        Return value of external temperature profile for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        val = nod_ray_cond.xmlGetChildDouble('external_temperature_profile')
        if not val:
            val = self.defaultValues()['external_temperature_profile']
            self.setExternalTemperatureProfile(val)

        return val


    def setExternalTemperatureProfile(self, val):
        """
        Put value of external temperature profile for the radiative wall
        """
        Model().isGreaterOrEqual(val, 0.)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData('external_temperature_profile',val)


    def getInternalTemperatureProfile(self):
        """
        Return value of internal temperature profile for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        val = nod_ray_cond.xmlGetChildDouble('internal_temperature_profile')
        if not val:
            val = self.defaultValues()['internal_temperature_profile']
            self.setInternalTemperatureProfile(val)

        return val


    def setInternalTemperatureProfile(self, val):
        """
        Put value of internal temperature profile for the radiative wall
        """
        Model().isGreaterOrEqual(val, 0.)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData('internal_temperature_profile',val)


    def getFlux(self):
        """
        Return value of flux for the radiative wall
        """
##        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
##        val = nod_ray_cond.xmlGetChildDouble('flux')
##        if not val:
##            val = self.defaultValues()['flux']
##            self.setFlux(val)
        val = self.getValRay('flux')
        
        return val


    def setFlux(self, val):
        """
        Put value of flux for the radiative wall
        """
        Model().isGreaterOrEqual(val, 0.)
        
##        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
##        nod_ray_cond.xmlSetData('flux', val)
        self.setValRay(val, 'flux')


    def getOutputRadiativeZone(self):
        """
        Return value of output radiative zone for the radiative wall
        """
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        ival = nod_ray_cond.xmlGetInt('output_zone')
        if not ival:
            ival = self.defaultValues()['output_zone']
            self.setOutputRadiativeZone(ival)

        return ival


    def setOutputRadiativeZone(self, ival):
        """
        Put value of output radiative zone for the radiative wall
        """
        Model().isInt(ival)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData('output_zone', ival)


    def getValRay(self, rayvar):
        """
        Return value of radiative variable named 'var' for the radiative wall
        """
        Model().isInList(rayvar, self.head_list)
        
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        if rayvar == "output_zone":
            val = nod_ray_cond.xmlGetInt(rayvar)
        else:
            val = nod_ray_cond.xmlGetDouble(rayvar)
        if not val:
            val = self.defaultValues()[rayvar]
            self.setValRay(val, rayvar)

        return val

    def setValRay(self, val, rayvar):
        """
        Put value of radiative variable named 'rayvar' for the radiative wall
        """
        Model().isInList(rayvar, self.head_list)
        if rayvar == "output_zone":
            Model().isInt(val)
        else:
            Model().isFloat(val)
        nod_ray_cond = self.boundNode.xmlInitChildNode('radiative_data')
        nod_ray_cond.xmlSetData(rayvar, val)


##
#-------------------------------------------------------------------------------
# InletBoundaryModel test case for inlet boundaries conditions
#-------------------------------------------------------------------------------


class InletBoundaryTestCase(ModelTest):
    """
    Unittest.
    """
    def checkInletBoundaryInstantiation(self):
        """
        Check whether the InletBoundary class could be instantiated 
        """
        model = None
        model = Boundary("inlet", "entree1", self.case)
        assert model != None, 'Could not instantiate InletBoundary'


    def checkSetAndGetVelocityChoice(self):
        """Check whether the velocity choice could be set and get for inlet boundary."""
        model = Boundary("inlet", "entree1", self.case)
        model.setVelocityChoice('flow1+direction')
        node =  model._XMLBoundaryConditionsNode

        doc = '''<boundary_conditions>
                    <inlet label="entree1">
                        <velocity_pressure choice="flow1+direction">
                            <flow1>1</flow1>
                            <direction_x>0</direction_x>
                            <direction_y>0</direction_y>
                            <direction_z>0</direction_z>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                 </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set velocity choice for inlet boundary'

        assert model.getVelocityChoice() == "flow1+direction",\
           'Could not get velocity choice for inlet boundary'


    def checkSetAndGetFlowAndDirection(self):
        """Check whether the mass or volumic flow could be set and get for inlet boundary."""
        model = Boundary("inlet", "entree1", self.case)
        model.setVelocityChoice('flow1+direction')
        node =  model._XMLBoundaryConditionsNode
        
        model.setFlow('flow1', 3.5)
##        model.setFlow('flow2', 3.5)
        model.setDirection('direction_z', 2.0)
        
        doc = '''<boundary_conditions>
                    <inlet label="entree1">
                        <velocity_pressure choice="flow1+direction">
                            <flow1>3.5</flow1>
                            <direction_x>0</direction_x>
                            <direction_y>0</direction_y>
                            <direction_z>2.0</direction_z>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc),\
           'Could not set mass flow or volumic flow and directions for inlet boundary'

        assert model.getVelocityChoice() == "flow1+direction",\
           'Could not get mass flow or volumic flow and directions for inlet boundary'


    def checkSetAndGetNorm(self):
        """Check whether the velocity norm could be set and get for inlet boundary."""
        model = Boundary("inlet", "entree1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('flow1+direction')
        model.setVelocityChoice('norm')
        #il faut explicietement supprimer direction si on ne dirige pas la vitesse
        model.deleteDirectionNodes()

        model.setNorm(999.99)
        doc = '''<boundary_conditions>
                        <inlet label="entree1">
                            <velocity_pressure choice="norm">
                                    <norm>999.99</norm>
                            </velocity_pressure>
                            <turbulence choice="hydraulic_diameter">
                                <hydraulic_diameter>1</hydraulic_diameter>
                            </turbulence>
                        </inlet>
                 </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc),\
           'Could not set velocity norm for inlet boundary'

        assert model.getVelocityChoice() == "norm",\
           'Could not get velocity norm for inlet boundary'


    def checkSetAndGetTurbulenceChoice(self):
        """Check whether the turbulence choice could be set and get for inlet boundary."""
        model = Boundary("inlet", "entree1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('norm')
        model.setTurbulenceChoice('turbulent_intensity')
        
        doc = '''<boundary_conditions>
                    <inlet label="entree1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                        </velocity_pressure>
                        <turbulence choice="turbulent_intensity">
                            <hydraulic_diameter>1</hydraulic_diameter>
                            <turbulent_intensity>2</turbulent_intensity>
                        </turbulence>
                    </inlet>
                 </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set turbulence choice for inlet boundary'

        assert model.getTurbulenceChoice() == "turbulent_intensity",\
           'Could not get turbulence choice for inlet boundary'


    def checkSetAndGetHydraulicDiameterAndTurbulentIntensity(self):
        """
        Check whether the hydraulic_diameter and turbulent_intensity could be 
        set and get for inlet boundary.
        """
        model = Boundary("inlet", "entree1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('norm')
        model.setTurbulenceChoice('turbulent_intensity')
        
        #testé pour bonne prise en compte
##        model.setHydraulicDiameter(-120.)
##        model.setTurbulentIntensity(-0.005)

        model.setHydraulicDiameter(120.)
        model.setTurbulentIntensity(0.005)
        
        doc = '''<boundary_conditions>
                    <inlet label="entree1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                        </velocity_pressure>
                        <turbulence choice="turbulent_intensity">
                            <hydraulic_diameter>120.</hydraulic_diameter>
                            <turbulent_intensity>0.005</turbulent_intensity>
                        </turbulence>
                    </inlet>
                 </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set hydraulic_diameter and turbulent_intensity for inlet boundary'

        assert model.getHydraulicDiameter() == 120,\
           'Could not get hydraulic_diameter for turbulence for inlet boundary'
        assert model.getTurbulentIntensity() == 0.005,\
           'Could not get turbulent_intensity for inlet boundary'


    def checkSetAndGetThermalScalar(self):
        """Check whether the thermal scalar could be set and get for inlet boundary."""
        model = Boundary("inlet", "entree1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('norm')
        model.setTurbulenceChoice('hydraulic_diameter')
        model.th_model.setThermalModel('temperature_celsius')
        model.setScalar("Temp.C", 15.)

        doc = '''<boundary_conditions>
                    <inlet label="entree1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                        <scalar label="Temp.C" name="temperature_celsius" type="thermal">
                            <dirichlet>15</dirichlet>
                        </scalar>
                    </inlet>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc),\
           'Could not set thermal scalar value for inlet boundary'

        assert model.getScalar("Temp.C") == 15,\
           'Could not get thermal scalar value for inlet boundary'


    def checkSetAndGetScalar(self):
        """Check whether the user scalar could be set and get for inlet boundary."""
        model = Boundary("inlet", "entree1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('norm')
        model.setTurbulenceChoice('hydraulic_diameter')
        model.sca_model.addUserScalar('1', 'sca1')
        model.sca_model.addUserScalar('1', 'sca2')
        model.setScalar('sca1', 11.)
        model.setScalar('sca2', 22.)

        doc = '''<boundary_conditions>
                    <inlet label="entree1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                        <scalar label="sca1" name="scalar1" type="user">
                            <dirichlet>11</dirichlet>
                        </scalar>
                        <scalar label="sca2" name="scalar2" type="user">
                            <dirichlet>22</dirichlet>
                        </scalar>
                    </inlet>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set user scalar value for inlet boundary'

        assert model.getScalar('sca2') == 22,\
           'Could not get user scalar value for inlet boundary'
           
           
           
def suite():
    testSuite = unittest.makeSuite(InletBoundaryTestCase, "check")
    return testSuite


def runTest():
    print "InletBoundaryTestCase"
    runner = unittest.TextTestRunner()
    runner.run(suite())


#-------------------------------------------------------------------------------
# CoalInletBoundaryModel test case for coal inlet boundaries conditions
#-------------------------------------------------------------------------------


class CoalInletBoundaryTestCase(ModelTest):
    """
    Unittest.
    """
    def checkCoalInletBoundaryInstantiation(self):
        """
        Check whether the CoalInletBoundary class could be instantiated 
        """
        from CoalCombustionModel import CoalCombustionModel
        CoalCombustionModel(self.case).setCoalCombustionModel('coal_homo')
        model = None
        model = Boundary("coal_inlet", "entree1", self.case)
        assert model != None, 'Could not instantiate CoalInletBoundary'


    def checkSetAndGetCoalType(self):
        """Check whether the type of inlet coal could be set and get for coal inlet boundary."""
        from CoalCombustionModel import CoalCombustionModel
        CoalCombustionModel(self.case).setCoalCombustionModel('coal_homo')

        model = Boundary("inlet", "charb1", self.case)
        coal_model = Boundary("coal_inlet", "charb1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('norm')
        coal_model.setCoalType('coalflow')

        doc = '''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                            <temperature>1273.15</temperature>
                            <coal_flow name="coal01">
                                <flow1>1</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
        
        assert node == self.xmlNodeFromString(doc),\
           'Could not set coalflow type for coal inlet boundary'

        assert coal_model.getCoalType() == "coalflow",\
           'Could not get coalflow type for coal inlet boundary'
           
        coal_model.setCoalType('airflow')
        doc1 = '''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                            <temperature>1273.15</temperature>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
        assert node == self.xmlNodeFromString(doc1),\
           'Could not set airflow type for coal inlet boundary'

        assert coal_model.getCoalType() == "airflow",\
           'Could not get airflow type for coal inlet boundary'


    def checkSetAndGetAirAndCoalTemperature(self):
        """Check whether the temperature of air and coal could be set and get for coal inlet boundary."""
        from CoalCombustionModel import CoalCombustionModel
        CoalCombustionModel(self.case).setCoalCombustionModel('coal_homo')

        model = Boundary("inlet", "charb1", self.case)
        coal_model = Boundary("coal_inlet", "charb1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('norm')
        coal_model.setCoalType('coalflow')
        coal_model.setAirTemperature(500.)
        coal_model.setCoalTemperature(999.99, 0)
        
        doc = '''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="norm">
                            <norm>1</norm>
                            <temperature>500.</temperature>
                            <coal_flow name="coal01">
                                <flow1>1</flow1>
                                <temperature>999.99</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
        
        assert node == self.xmlNodeFromString(doc),\
           'Could not set air temperature for coal inlet boundary'

        assert coal_model.getAirTemperature() == 500.,\
           'Could not get air temperature for coal inlet boundary'
        
        assert coal_model.getCoalTemperature(0) == 999.99,\
           'Could not get coal temperature for coal inlet boundary'


    def checkSetAndGetCoalFlow(self):
        """Check whether the flow of inlet coal could be set and get for coal inlet boundary."""
        from CoalCombustionModel import CoalCombustionModel
        CoalCombustionModel(self.case).setCoalCombustionModel('coal_homo')

        model = Boundary("inlet", "charb1", self.case)
        coal_model = Boundary("coal_inlet", "charb1", self.case)
        node =  model._XMLBoundaryConditionsNode
        model.setVelocityChoice('flow1')
        coal_model.setCoalType('coalflow')
        coal_model.setCoalFlow(123.5, 0)

        doc = '''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="flow1">
                            <temperature>1273.15</temperature>
                            <flow1>1</flow1>
                            <coal_flow name="coal01">
                                <flow1>123.5</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set flow for coal inlet boundary'

        assert coal_model.getCoalFlow(0) == 123.5,\
           'Could not get flow for coal inlet boundary'


    def checkSetAndGetCoalRatios(self):
        """Check whether the ratio of classes could be set and get for coal inlet boundary."""
        os.remove("dp_FCP")        
        from CoalCombustionModel import CoalCombustionModel
        m = CoalCombustionModel(self.case)
        m.setCoalCombustionModel('coal_homo')
        
        # creation du fichier dp_FCP avec 2 charbons et 3 classes
        self.case['data_path'] = "."
        from CoalThermoChemistry import CoalThermoChemistryModel, Coal
        coalThermoChModel = CoalThermoChemistryModel("dp_FCP", self.case)
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalThermoChModel.save()

        coalThermoChModel.getCoals().addCoal(Coal())
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalThermoChModel.save()
        coalNumber = 2
        coal = coalThermoChModel.getCoals().getCoal(coalNumber)
        coal.addInitDiameterClasses(0.5)
        coalThermoChModel.getCoals().updateCoal(coalNumber, coal)
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalThermoChModel.save()
        # fin de la creation du fichier dp_FCP

        model = Boundary("inlet", "charb1", self.case)
        coal_model = Boundary("coal_inlet", "charb1", self.case)
        
        model.setVelocityChoice('flow1')
        model.setVelocity(12.5)
        coal_model.setCoalType('coalflow')
        coal_model.setCoalFlow(123.5, 0)
        coal_model.setAirTemperature(500.)
        coal_model.setCoalTemperature(999.99, 0)
##        coal_model.setCoalRatios(0, (45,))
        coal_model.setCoalRatios(1, (45, 55))

        node =  model._XMLBoundaryConditionsNode

        doc = '''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="flow1">
                            <temperature>500</temperature>
                            <flow1>12.5</flow1>
                            <coal_flow name="coal01">
                                <flow1>123.5</flow1>
                                <temperature>999.99</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                            <coal_flow name="coal02">
                                <flow1>1</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">45</ratio>
                                <ratio name="class02">55</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set ratios for classes for coal inlet boundary'

        assert coal_model.getCoalRatios(1) == [45.0, 55.0],\
           'Could not get ratios for classes for coal inlet boundary'


    def checkDeleteCoalAndClassesRatios(self):
        """Check whether coal or classes could be deleted for coal inlet boundary."""
        os.remove("dp_FCP")        
        from CoalCombustionModel import CoalCombustionModel
        m = CoalCombustionModel(self.case)
        m.setCoalCombustionModel('coal_homo')
        
        # creation du fichier dp_FCP avec 3 charbons et 6 classes
        self.case['data_path'] = "."
        from CoalThermoChemistry import CoalThermoChemistryModel, Coal
        coalThermoChModel = CoalThermoChemistryModel("dp_FCP", self.case)
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalThermoChModel.save()

        coalThermoChModel.getCoals().addCoal(Coal())
        coalThermoChModel.getCoals().addCoal(Coal())
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalThermoChModel.save()
        coalNumber = 2
        coal = coalThermoChModel.getCoals().getCoal(coalNumber)
        coal.addInitDiameterClasses(0.5)
        coalThermoChModel.getCoals().updateCoal(coalNumber, coal)
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalNumber = 3
        coal = coalThermoChModel.getCoals().getCoal(coalNumber)
        coal.addInitDiameterClasses(9.9)
        coal.addInitDiameterClasses(21.)
        coalThermoChModel.getCoals().updateCoal(coalNumber, coal)
        m.createCoalModelScalarsAndProperties(coalThermoChModel)
        coalThermoChModel.save()
        # fin de la creation du fichier dp_FCP
        model = Boundary("inlet", "charb1", self.case)
        coal_model = Boundary("coal_inlet", "charb1", self.case)
        
        model.setVelocityChoice('flow1')
        model.setVelocity(12.5)
        coal_model.setCoalType('coalflow')
        coal_model.setCoalFlow(123.5, 0)
        coal_model.setAirTemperature(500.)
        coal_model.setCoalTemperature(999.99, 0)
        coal_model.setCoalRatios(1, (45, 55))
        coal_model.setCoalRatios(2, (10, 20, 70))

        node =  model._XMLBoundaryConditionsNode
        doc = '''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="flow1">
                            <temperature>500</temperature>
                            <flow1>12.5</flow1>
                            <coal_flow name="coal01">
                                <flow1>123.5</flow1>
                                <temperature>999.99</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                            <coal_flow name="coal02">
                                <flow1>1</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">45</ratio>
                                <ratio name="class02">55</ratio>
                            </coal_flow>
                            <coal_flow name="coal03">
                                <flow1>1</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">10</ratio>
                                <ratio name="class02">20</ratio>
                                <ratio name="class03">70</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc),\
           'Could not set ratios for classes for coal inlet boundary'

        coal_model.deleteCoalFlow(1,3)
        doc1 ='''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="flow1">
                            <temperature>500</temperature>
                            <flow1>12.5</flow1>
                            <coal_flow name="coal01">
                                <flow1>123.5</flow1>
                                <temperature>999.99</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                            <coal_flow name="coal02">
                                <flow1>1</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">10</ratio>
                                <ratio name="class02">20</ratio>
                                <ratio name="class03">70</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc1),\
           'Could not delete one coal for coal inlet boundary'
           
        coal_model.deleteClassRatio(1, 1, 3)
        doc2 ='''<boundary_conditions>
                    <inlet label="charb1">
                        <velocity_pressure choice="flow1">
                            <temperature>500</temperature>
                            <flow1>12.5</flow1>
                            <coal_flow name="coal01">
                                <flow1>123.5</flow1>
                                <temperature>999.99</temperature>
                                <ratio name="class01">100</ratio>
                            </coal_flow>
                            <coal_flow name="coal02">
                                <flow1>1</flow1>
                                <temperature>1273.15</temperature>
                                <ratio name="class01">100</ratio>
                                <ratio name="class02">0</ratio>
                            </coal_flow>
                        </velocity_pressure>
                        <turbulence choice="hydraulic_diameter">
                            <hydraulic_diameter>1</hydraulic_diameter>
                        </turbulence>
                    </inlet>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc2),\
           'Could not delete one class of one coal for coal inlet boundary'
           

def suite2():
    testSuite = unittest.makeSuite(CoalInletBoundaryTestCase, "check")
    return testSuite


def runTest2():
    print "CoalInletBoundaryTestCase"
    runner = unittest.TextTestRunner()
    runner.run(suite2())

#-------------------------------------------------------------------------------
# WallBoundaryModel test case for wall boundaries conditions
#-------------------------------------------------------------------------------


class WallBoundaryTestCase(ModelTest):
    """
    Unittest.
    """
    def checkWallBoundaryInstantiation(self):
        """
        Check whether the WallBoundary class could be instantiated WallBoundary
        """
        model = None
        model = Boundary("wall", "paroi", self.case)
        assert model != None, 'Could not instantiate '


    def checkSetAndGetVelocityChoice(self):
        """Check whether the velocity choice could be set and get for wall boundary."""
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('on')
        model = Boundary("wall", "fenetre", self.case)
        model.setVelocityChoice('off')
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="on">
                            <dirichlet name="velocity_U">0.0</dirichlet>
                            <dirichlet name="velocity_V">0.0</dirichlet>
                            <dirichlet name="velocity_W">0.0</dirichlet>
                        </velocity_pressure>
                    </wall>
                    <wall label="fenetre">
                        <velocity_pressure choice="off"/>
                    </wall>
                </boundary_conditions>'''
        
        assert node == self.xmlNodeFromString(doc),\
           'Could not set choice of velocity for wall boundary'

        assert model.getVelocityChoice() == 'off',\
           'Could not get set choice of velocity for wall boundary'
           
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')

        doc2 = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="fenetre">
                        <velocity_pressure choice="off"/>
                    </wall>
                </boundary_conditions>'''
           
        assert node == self.xmlNodeFromString(doc2),\
           'Could not set choice of velocity for wall boundary'


    def checkSetAndGetVelocityValue(self):
        """Check whether the velocity value could be set and get for wall boundary."""
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('on')
        model.setVelocities(1., 2., 3.)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="on">
                            <dirichlet name="velocity_U">1</dirichlet>
                            <dirichlet name="velocity_V">2</dirichlet>
                            <dirichlet name="velocity_W">3</dirichlet>
                        </velocity_pressure>
                    </wall>
                </boundary_conditions>'''
        
        assert node == self.xmlNodeFromString(doc),\
           'Could not set values of velocity for wall boundary'

        assert model.getVelocities() == (1, 2 ,3),\
           'Could not get set values of velocity for wall boundary'       


    def checkSetAndGetRoughnessChoiceAndValue(self):
        """Check whether the roughness could be set and get for wall boundary."""
        model = Boundary("wall", "mur", self.case)
        model.setRoughnessChoice('on')
        node =  model._XMLBoundaryConditionsNode
      
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off">
                            <roughness>0.01</roughness>
                        </velocity_pressure>
                    </wall>
                </boundary_conditions>'''
        
        assert node == self.xmlNodeFromString(doc),\
           'Could not update roughness choice for wall boundary'
           
        model.setRoughness(15.33)
        doc1 = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off">
                            <roughness>15.33</roughness>
                        </velocity_pressure>
                    </wall>
                 </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc1),\
           'Could not set roughness value for wall boundary'

        assert model.getRoughness() == 15.33,\
           'Could not get roughness value for wall boundary'


    def checkSetAndGetScalarChoice(self):
        """Check whether the scalar choice could be set and get for wall boundary."""
        model = Boundary("wall", "mur", self.case)
        model.sca_model.addUserScalar('1', 'sca1')
        model.sca_model.addUserScalar('1', 'sca2')
        model.setScalarChoice('sca2', 'dirichlet')
        node =  model._XMLBoundaryConditionsNode

        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                        <scalar choice="dirichlet" label="sca2" name="scalar2" type="user">
                            <dirichlet>0.0</dirichlet>
                        </scalar>
                    </wall>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc),\
           'Could not set scalar choice for wall boundary'

        assert model.getScalarChoice('sca2') == 'dirichlet',\
           'Could not get scalar choice for wall boundary'

        assert model.getScalarChoice('sca1') == 'neumann',\
           'Could not get scalar choice for wall boundary'
           
        model.setScalarChoice('sca1', 'exchange_coefficient')
        doc1 = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                        <scalar choice="exchange_coefficient" label="sca1" name="scalar1" type="user">
                            <neumann>0.0</neumann>
                            <exchange_coefficient>0.0</exchange_coefficient>
                        </scalar>
                        <scalar choice="dirichlet" label="sca2" name="scalar2" type="user">
                            <dirichlet>0.0</dirichlet>
                        </scalar>
                    </wall>
                 </boundary_conditions>'''
                 
        assert node == self.xmlNodeFromString(doc1),\
           'Could not set scalar choice for wall boundary'
           
        assert model.getScalarChoice('sca1') == 'exchange_coefficient',\
           'Could not get scalar choice for wall boundary'
           
           
    def checkSetAndGetScalarImposedValueFluxAndExchangeCoefficient(self):
        """Check whether the scalar values could be set and get for wall boundary."""
        model = Boundary("wall", "mur", self.case)
        model.sca_model.addUserScalar('1', 'sca1')
        model.sca_model.addUserScalar('1', 'sca2')
        model.setScalarChoice('sca1', 'exchange_coefficient')
        model.setScalarChoice('sca2', 'dirichlet')
        model.setScalarImposedFlux('sca1', 130.)
        model.setScalarExchangeCoefficient('sca1', 0.130)
        model.setScalarImposedValue('sca2', 55.)
        node =  model._XMLBoundaryConditionsNode

        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                        <scalar choice="exchange_coefficient" label="sca1" name="scalar1" type="user">
                            <neumann>130.0</neumann>
                            <exchange_coefficient>0.130</exchange_coefficient>
                        </scalar>
                        <scalar choice="dirichlet" label="sca2" name="scalar2" type="user">
                            <dirichlet>55.</dirichlet>
                        </scalar>
                    </wall>
                 </boundary_conditions>'''
                 
        assert node == self.xmlNodeFromString(doc),\
           'Could not set scalar imposed value, flux and exchange_coefficient for wall boundary'

        assert model.getScalarImposedValue('sca2') == 55.,\
           'Could not get scalar imposed value for wall boundary'

        assert model.getScalarImposedFlux('sca1') == 130.,\
            'Could not get scalar imposed value for wall boundary'
            
        assert model.getScalarExchangeCoefficient('sca1') == 0.130,\
            'Could not get scalar imposed value for wall boundary'
 
      
def suite3():
    testSuite = unittest.makeSuite(WallBoundaryTestCase, "check")
    return testSuite


def runTest3():
    print "WallBoundaryTestCase"
    runner = unittest.TextTestRunner()
    runner.run(suite3())


#-------------------------------------------------------------------------------
# RadiativeWallBoundaryModel test case for radiative boundaries conditions
#-------------------------------------------------------------------------------


class RadiativeWallBoundaryTestCase(ModelTest):
    """
    Unittest.
    """
    def checkRadiativeWallBoundaryInstantiation(self):
        """
        Check whether the RadiativeWallBoundary class could be instantiated RadiativeWallBoundary
        """
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = None
        model = Boundary("radiative_wall", "paroi", self.case)
        assert model != None, 'Could not instantiate '


    def checkSetAndgetRadiativeChoice(self):
        """Check whether the type of condition could be set and get for radiative wall boundary."""
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ipgrno')
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ipgrno">
                            <emissivity>0.8</emissivity>
                            <thermal_conductivity>3</thermal_conductivity>
                            <thickness>0.1</thickness>
                            <external_temperature_profile>300</external_temperature_profile>
                            <internal_temperature_profile>300</internal_temperature_profile>
                            <output_zone>1</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set type of condition for radiative wall boundary'

        assert model.getRadiativeChoice() == "ipgrno",\
           'Could not get type of condition for radiative wall boundary'


    def checkSetAndGetEmissivity(self):
        """Check whether the emissivity could be set and get for radiative wall boundary."""
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ipgrno')
        model.setEmissivity(0.22)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ipgrno">
                            <emissivity>0.22</emissivity>
                            <thermal_conductivity>3</thermal_conductivity>
                            <thickness>0.1</thickness>
                            <external_temperature_profile>300</external_temperature_profile>
                            <internal_temperature_profile>300</internal_temperature_profile>
                            <output_zone>1</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set emissivity for radiative wall boundary'

        assert model.getEmissivity() == 0.22,\
           'Could not get emissivity for radiative wall boundary'


    def checkSetAndGetThermalConductivity(self):
        """Check whether the thermal conductivity could be set and get for radiative wall boundary."""
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ipgrno')
        model.setThermalConductivity(5.6)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ipgrno">
                            <emissivity>0.8</emissivity>
                            <thermal_conductivity>5.6</thermal_conductivity>
                            <thickness>0.1</thickness>
                            <external_temperature_profile>300</external_temperature_profile>
                            <internal_temperature_profile>300</internal_temperature_profile>
                            <output_zone>1</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set thermal conductivity for radiative wall boundary'

        assert model.getThermalConductivity() == 5.6,\
           'Could not get thermal conductivity for radiative wall boundary'
    

    def checkSetAndGetThickness(self):
        """Check whether the thickness could be set and get for radiative wall boundary."""
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ipgrno')
        model.setThickness(2.)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ipgrno">
                            <emissivity>0.8</emissivity>
                            <thermal_conductivity>3</thermal_conductivity>
                            <thickness>2.</thickness>
                            <external_temperature_profile>300</external_temperature_profile>
                            <internal_temperature_profile>300</internal_temperature_profile>
                            <output_zone>1</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set thickness for radiative wall boundary'

        assert model.getThickness() == 2.0,\
           'Could not get thickness for radiative wall boundary'
    
    
    def checkSetAndGetExternalAndInternalTemperatureProfile(self):
        """
        Check whether the external and internal temperature profile 
        could be set and get for radiative wall boundary.
        """
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ipgrno')
        model.setExternalTemperatureProfile(55.55)
        model.setInternalTemperatureProfile(987.)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ipgrno">
                            <emissivity>0.8</emissivity>
                            <thermal_conductivity>3</thermal_conductivity>
                            <thickness>0.1</thickness>
                            <external_temperature_profile>55.55</external_temperature_profile>
                            <internal_temperature_profile>987.</internal_temperature_profile>
                            <output_zone>1</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set external or internal temperature profile for radiative wall boundary'

        assert model.getExternalTemperatureProfile() == 55.55,\
           'Could not get external temperature profile for radiative wall boundary'
           
        assert model.getInternalTemperatureProfile() == 987.,\
           'Could not get internal temperature profile for radiative wall boundary'
           
        
    
    def checkSetAndGetOutputRadiativeZone(self):
        """
        Check whether the output radiative zone could be set and get for 
        radiative wall boundary.
        """
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ipgrno')
        model.setOutputRadiativeZone(21)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ipgrno">
                            <emissivity>0.8</emissivity>
                            <thermal_conductivity>3</thermal_conductivity>
                            <thickness>0.1</thickness>
                            <external_temperature_profile>300.</external_temperature_profile>
                            <internal_temperature_profile>300.</internal_temperature_profile>
                            <output_zone>21</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set output radiative zone for radiative wall boundary'

        assert model.getOutputRadiativeZone() == 21,\
           'Could not get output radiative zone for radiative wall boundary'
    
    
    def checkSetAndGetFlux(self):
        """Check whether the flux could be set and get for radiative wall boundary."""
        from ThermalRadiationModel import ThermalRadiationModel
        ThermalRadiationModel(self.case).setRadiativeModel('dom')
        
        model = Boundary("wall", "mur", self.case)
        model.setVelocityChoice('off')
        model = Boundary("radiative_wall", "radiateur", self.case)
        model.setRadiativeChoice('ifgrno')
        model.setFlux(5.65)
        node =  model._XMLBoundaryConditionsNode
        
        doc = '''<boundary_conditions>
                    <wall label="mur">
                        <velocity_pressure choice="off"/>
                    </wall>
                    <wall label="radiateur">
                        <radiative_data choice="ifgrno">
                            <emissivity>0.8</emissivity>
                            <flux>5.65</flux>
                            <internal_temperature_profile>300.</internal_temperature_profile>
                            <output_zone>1</output_zone>
                        </radiative_data>
                    </wall>
                </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set flux for radiative wall boundary'

        assert model.getFlux() == 5.65,\
           'Could not get flux for radiative wall boundary'
           
def suite4():
    testSuite = unittest.makeSuite(RadiativeWallBoundaryTestCase, "check")
    return testSuite


def runTest4():
    print "RadiativeWallBoundaryTestCase"
    runner = unittest.TextTestRunner()
    runner.run(suite4())

#-------------------------------------------------------------------------------
# OutletBoundaryModel test case for outlet boundaries conditions
#-------------------------------------------------------------------------------

class OutletBoundaryTestCase(ModelTest):
    """
    Unittest.
    """
    def checkOutletBoundaryInstantiation(self):
        """
        Check whether the OutletBoundary class could be instantiated OutletBoundary
        """
        model = None
        model = Boundary("outlet", "sortie", self.case)
        assert model != None, 'Could not instantiate '


    def checkSetAndGetPressure(self):
        """Check whether the reference pressure could be set and get for outlet boundary."""
        model = Boundary("outlet", "sortie", self.case)
        model.setReferencePressure(111333.)

        node =  model._XMLBoundaryConditionsNode

        doc = '''<boundary_conditions>
                    <outlet label="sortie">
                        <dirichlet name="pressure">111333</dirichlet>
                    </outlet>
                </boundary_conditions>'''
                
        assert node == self.xmlNodeFromString(doc),\
           'Could not set reference pressure for outlet boundary'

        assert model.getReferencePressure() == 111333,\
           'Could not get reference pressure for outlet boundary'


    def checkSetAndGetScalarChoiceAndValue(self):
        """Check whether the scalar choice and value could be set and get for outlet boundary."""
        model = Boundary("outlet", "sortie", self.case)
        model.sca_model.addUserScalar('1', 'sca1')
        model.sca_model.addUserScalar('1', 'sca2')
        model.setScalarChoice('sca1', 'dirichlet')
        model.setScalar('sca1', 10.10)
        model.setScalarChoice('sca2', 'neumann')
        node =  model._XMLBoundaryConditionsNode

        doc = '''<boundary_conditions>
                    <outlet label="sortie">
                        <scalar choice="dirichlet" label="sca1" name="scalar1" type="user">
                            <dirichlet>10.1</dirichlet>
                        </scalar>
                        <scalar choice="neumann" label="sca2" name="scalar2" type="user">
                            <neumann>0</neumann>
                        </scalar>
                    </outlet>
            </boundary_conditions>'''

        assert node == self.xmlNodeFromString(doc),\
           'Could not set choice and value at scalars for outlet boundary'

        assert model.getScalarChoice('sca1') == 'dirichlet',\
           'Could not get choice of scalar for outlet boundary'
           
        assert model.getScalar('sca2') == 0,\
           'Could not get choice of scalar for outlet boundary'


def suite5():
    testSuite = unittest.makeSuite(OutletBoundaryTestCase, "check")
    return testSuite


def runTest5():
    print "OutletBoundaryTestCase"
    runner = unittest.TextTestRunner()
    runner.run(suite5())

#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
