/*
#   idjcserver.c: IDJC code that communicates with the radio server 
#   Copyright (C) 2005-2006 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "../config.h"
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <jack/jack.h>
#include <jack/transport.h>
#include <getopt.h>
#include <string.h>
#include <pthread.h>
#include <fcntl.h>
#include <math.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <jack/ringbuffer.h>
#include <assert.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisenc.h>
#include <time.h>
#include "kvpparse.h"

#if HAVE_LIBSHOUT
#if   defined HAVE_SHOUT_SHOUT_H
#include <shout/shout.h>
#elif defined HAVE_SHOUT_H
#include <shout.h>
#endif  /*HAVE_SHOUT_SHOUT_H*/
/* other versions of libshout define SHOUT_FORMAT_VORBIS instead */
#ifndef SHOUT_FORMAT_OGG
#define SHOUT_FORMAT_OGG SHOUT_FORMAT_VORBIS
#endif  /*SHOUT_FORMAT_OGG*/
#define CONNECTING 1
#define CONNECTED 2
#endif  /*HAVE_LIBSHOUT*/

#define DISCONNECTED 0
#define TRUE 1
#define FALSE 0
#define RB_SIZE 262144
#define RSB_SIZE 262144
#define MP3_BUF_SIZ 4096
#define SRC_SIZE 8192
#define OGG_META_SRC_SIZE 1024

#define AP_ACTUATION_DELAY 0.25F
#define AP_CANCEL_THRESH 0.01F

pthread_mutex_t pm = PTHREAD_MUTEX_INITIALIZER;

char *reply[] = {"disconnected", "connecting", "connected"};

typedef jack_default_audio_sample_t sample_t;
	 
jack_client_t *client;
jack_port_t *stream_left_port;
jack_port_t *stream_right_port;
unsigned long sr;

pid_t encoder_pid, saver_pid;

/* part of CTRL+C handling */
sigset_t serversigset;

long offset = 0, total;
int transport_aware = 0, connection = DISCONNECTED;
jack_transport_state_t transport_state;

jack_ringbuffer_t *leftring, *rightring, *leftsring, *rightsring;
/* inter-thread control variables */
int ringbuffer_can_write = FALSE, sringbuffer_can_write = FALSE, ringbuffer_flush = FALSE, sringbuffer_flush = FALSE, disconnect_f = FALSE, is_saving = FALSE;
int keep_connection, keep_saving, keep_watchdog, newmeta;
/* data passed from the GUI pointed to by these variables */
char *server_type, *host, *port, *password, *name, *metatext, *format, *bandwidth, *action, *streamname, *streamurl, *streamdesc, *streamgenre, *mount, *status_tag, *save_file,  *save_artist, *save_title, *save_format, *save_bitrate, *public_s, *meta_s, *save_pause, *auto_pause, *ap_actuation_delay, *ap_cancel_thresh_log, *meta_artist, *meta_title, *abr_s, *stereo_s, *high_quality;
int public_f, meta_f, pause_f, paused, abr_f, stereo_f;
/* autopause stuff */
int auto_pause_f, auto_pause_actuate_f, ap_actuation_delay_samples;
uint64_t ap_quiet_samples_count;
float ap_cancel_thresh = AP_CANCEL_THRESH;
/* used for pipes */
int reader[2], writer[2], sreader[2], swriter[2];
/* thread handles */
pthread_t server_thread_h, codec_feeder_h, codec_streamer_ogg_h, watchdog_h;
pthread_t saver_thread_h, save_codec_feeder_h;
/* these strings contain stream audio parameters */
char ai_bitrate[4];
char ai_samplerate[8];
char ai_channels[2];
/* stops flush from blocking when jack is not running */
int jack_closed_f;
/* set when cpu overloaded or network is sluggish to indicate frames have been dropped */
int stream_buffer_full;

struct kvpdict kvpdict[] = {
         { "SERV", &server_type },
         { "RECF", &save_file },
         { "RECP", &save_pause },
         { "AUTP", &auto_pause },
         { "APOD", &ap_actuation_delay },
         { "APTH", &ap_cancel_thresh_log },
         { "ARTI", &save_artist },
         { "TITL", &save_title },
         { "SFMT", &save_format },
         { "SRAT", &save_bitrate },
         { "HOST", &host },
         { "PORT", &port },
         { "PASS", &password },
         { "NAME", &name },
         { "FORM", &format },
         { "BAND", &bandwidth },
         { "ACTN", &action },
         { "SNAM", &streamname },
         { "SURL", &streamurl },
         { "SDES", &streamdesc },
         { "SGEN", &streamgenre },
         { "MOUN", &mount },
         { "STAG", &status_tag },
         { "PUBF", &public_s },
         { "METF", &meta_s },
         { "MART", &meta_artist },
         { "MTIT", &meta_title },
         { "ABRF", &abr_s },
         { "STER", &stereo_s },
         { "QUAL", &high_quality },
         { "META", &metatext, &pm},	/* use a mutex lock on this one */
         { "", NULL }};

#if HAVE_LIBSHOUT
shout_t *shout;
#endif
      
void process_silence(jack_nframes_t nframes)
   {
   jack_nframes_t space_avail = jack_ringbuffer_write_space(rightring) / sizeof (sample_t);
   jack_nframes_t todo = (space_avail > nframes ? nframes : space_avail);
   sample_t *buffer = (sample_t *) calloc(sizeof (sample_t), todo);
   if (buffer == NULL)
      fprintf(stderr, "Error, unable to allocate buffer in process silence\n");
   jack_ringbuffer_write(leftring, (char *)buffer, todo * sizeof (sample_t));
   jack_ringbuffer_write(rightring, (char *)buffer, todo * sizeof (sample_t));  
   jack_ringbuffer_write(leftsring, (char *)buffer, todo * sizeof (sample_t));
   jack_ringbuffer_write(rightsring, (char *)buffer, todo * sizeof (sample_t));
   free(buffer);
   }

void process_audio(jack_nframes_t nframes)
   {
   sample_t *ptrl, *ptrr;
   sample_t *ls_buffer = (sample_t *) jack_port_get_buffer (stream_left_port, nframes);
   sample_t *rs_buffer = (sample_t *) jack_port_get_buffer (stream_right_port, nframes);
#if HAVE_LIBSHOUT
   jack_nframes_t space_avail, todo;
#endif
   jack_nframes_t sspace_avail, stodo;
   
   /* audio level analysis for the autopause feature */
   ptrl = ls_buffer + nframes;
   ptrr = rs_buffer + nframes;
   while (ptrl-- != ls_buffer)
      {
      if (fabsf(*ptrl) > ap_cancel_thresh || fabsf(*--ptrr) > ap_cancel_thresh)
         ap_quiet_samples_count = 0L;
      else
         ap_quiet_samples_count++;
      }

#if HAVE_LIBSHOUT
   space_avail = jack_ringbuffer_write_space(rightring) / sizeof (sample_t);
   todo = (space_avail > nframes ? nframes : space_avail);    
    
   if (ringbuffer_can_write == TRUE)
      {
      if (todo)
         {
         jack_ringbuffer_write(leftring, (char *)ls_buffer, todo * sizeof (sample_t));
         jack_ringbuffer_write(rightring, (char *)rs_buffer, todo * sizeof (sample_t));
         }
      else
         stream_buffer_full = TRUE;
      }

   if (ringbuffer_flush == TRUE)
      {
      jack_ringbuffer_reset(leftring);
      jack_ringbuffer_reset(rightring);
      ringbuffer_flush = FALSE;
      }   
#endif /* HAVE_LIBSHOUT */

   sspace_avail = jack_ringbuffer_write_space(rightsring) / sizeof (sample_t);
   stodo = (sspace_avail > nframes ? nframes : sspace_avail);

   if (sringbuffer_flush == TRUE)
      {
      jack_ringbuffer_reset(leftsring);
      jack_ringbuffer_reset(rightsring);
      sringbuffer_flush = FALSE;
      }

   if (sringbuffer_can_write == TRUE)
      {
      jack_ringbuffer_write(leftsring, (char *)ls_buffer, stodo * sizeof (sample_t));
      jack_ringbuffer_write(rightsring, (char *)rs_buffer, stodo * sizeof (sample_t));
      }
   }

int process(jack_nframes_t nframes, void *arg)
   {
   if(transport_aware)
      {
      jack_position_t pos;

      if(jack_transport_query(client, &pos) != JackTransportRolling)
         {
	 process_silence(nframes);
	 return 0;
	 }
      }
   process_audio(nframes);
   return 0;
   }

int sample_rate_change()
   {
   fprintf(stderr, "The sample rate changed!  Exiting...\n");
   exit(-1);
   }

int zctoo()
   {
   fprintf(stderr, "idjcserver: zero-cross timeout expired\n");
   return 1;
   }

void *save_codec_feeder(void *args)
   {   
   size_t frames;
   sample_t *lbuff, *rbuff, *l, *r;
   unsigned char *p, *pcmbuff;
   int32_t pcmvalue;
   float scaled_value;
   int i, leftmute = FALSE, rightmute = FALSE, lefthcp = 0, righthcp = 0, begin = 0, end = 0, zc_timeout = 0;
   enum crosstype { ZC_NORMAL, ZC_CLOSING, ZC_PAUSED, ZC_OPENING };
   enum crosstype zerocross = ZC_NORMAL;
   
   usleep(20000);
   if(!(lbuff = malloc(sizeof (sample_t) * SRC_SIZE)))
      {
      fprintf(stderr, "Unable to allocate left buffer in save_codec_feeder\n");
      keep_saving = FALSE;
      }
   if(!(rbuff = malloc(sizeof (sample_t) * SRC_SIZE)))
      {
      fprintf(stderr, "Unable to allocate right buffer in save_codec_feeder\n");
      keep_saving = FALSE;
      }
   if(!(pcmbuff = (unsigned char *)malloc(8 * SRC_SIZE)))
      {
      fprintf(stderr, "Unable to allocate space for pcm buffer\n");
      keep_saving = FALSE;
      }
      
   sringbuffer_can_write = TRUE;	/* Turn on the flow of data from JACK */
   
   while (keep_saving == TRUE)
      {
      usleep(10000);
      auto_pause_actuate_f = ap_quiet_samples_count  > ap_actuation_delay_samples;
      if (zerocross != ZC_PAUSED)
         paused |= pause_f | (auto_pause_f & auto_pause_actuate_f);
      else
         paused = pause_f | (auto_pause_f & auto_pause_actuate_f);
      if ((!(frames = jack_ringbuffer_read_space(rightsring) / sizeof (sample_t) & ~ 0xF)) && zerocross == ZC_NORMAL)
         continue;
      frames = (frames > SRC_SIZE) ? SRC_SIZE : frames;
      jack_ringbuffer_read(leftsring, (char *)lbuff, frames * sizeof (sample_t));
      jack_ringbuffer_read(rightsring, (char *)rbuff, frames * sizeof (sample_t));
      
      switch (zerocross)	/* a stereo zero-cross switch to eliminate the click upon pause */
         {			/* works much better with a mono sound source however */
         case ZC_NORMAL:
            if (paused)
               {
               zerocross = ZC_CLOSING;
               lefthcp = lbuff[0] >= 0.0F;			/* left half cycle is positive */
               righthcp = rbuff[0] >= 0.0F;
               zc_timeout = sr/12;
               }
            else
               {
               begin = 0;
               end = frames;
               break;
               }
         case ZC_CLOSING:
            if (frames > 0)				/* determine if data is still flowing */
               {
               for (i = 0; i < frames; i++)
                  {
                  if (leftmute == FALSE && lefthcp == TRUE && lbuff[i] <= 0.00001F)
                     leftmute = TRUE;
                  if (rightmute == FALSE && righthcp == TRUE && lbuff[i] <= 0.00001F)
                     rightmute = TRUE;
                  if (leftmute == TRUE)
                     lbuff[i] = 0.0F;			/* mute this channel from here on in */
                  else
                     lefthcp = lbuff[i] > 0.0F;		/* determine if half cycle is positive */
                  if (rightmute == TRUE)
                     rbuff[i] = 0.0F;
                  else
                     righthcp = rbuff[i] > 0.0F;
                  if ((leftmute && rightmute) || (zc_timeout-- == 0 && zctoo()))
                     {
                     sringbuffer_can_write = FALSE;
                     sringbuffer_flush = TRUE;
                     while (sringbuffer_flush == TRUE && jack_closed_f == FALSE)
                        usleep(5000);
                     break;
                     }
                  }
               end = i;
               }
            else
               zerocross = ZC_PAUSED;
            break;
         case ZC_PAUSED:
            if ((end = frames) == 0)
               {
               sringbuffer_can_write |= !paused;
               break;
               }
            lefthcp = lbuff[0] > 0.0F;
            righthcp = rbuff[0] > 0.0F;
            zc_timeout = sr/12;
            zerocross = ZC_OPENING;
         case ZC_OPENING:
            for (i = 0; i < frames; i++)
               {
               if (leftmute == TRUE && lefthcp == FALSE && lbuff[i] >= -0.00001F)	/* detect zero cross */
                  leftmute = FALSE;
               if (rightmute == TRUE && righthcp == FALSE && rbuff[i] >= -0.00001F)
                  rightmute = FALSE;
               if ((!leftmute && !rightmute) || (zc_timeout-- == 0 && zctoo()))
                  {
                  zerocross = ZC_NORMAL;
                  break;
                  }
               lefthcp = lbuff[i] > 0.0F;
               if (leftmute == TRUE)
                  lbuff[i] = 0.0F;
               righthcp = rbuff[i] > 0.0F;
               if (rightmute == TRUE)
                  rbuff[i] = 0.0F;
               }
            begin = i;
         }
      
      l=lbuff;
      r=rbuff;
      p=pcmbuff;
      
      
      if (!strcmp(save_format, "mp3"))
         {
         for (i = begin; i < end; i++)
            {
            scaled_value = *l++ * 2147483648.0F;
            pcmvalue = (int32_t)scaled_value;
            if (scaled_value > 2147483647.0F)
               pcmvalue = 0x7FFFFFFF;
            if (scaled_value < -2147483648.0F)
               pcmvalue = 0xFFFFFFFF;
            *p++ = pcmvalue;
            *p++ = pcmvalue >> 8;
            *p++ = pcmvalue >> 16;
            *p++ = pcmvalue >> 24;
   
            scaled_value = *r++ * 2147483648.0F;
            pcmvalue = (int32_t)scaled_value;
            if (scaled_value > 2147483647.0F)
               pcmvalue = 0x7FFFFFFF;
            if (scaled_value < -2147483648.0F)
               pcmvalue = 0xFFFFFFFF;
            *p++ = pcmvalue;
            *p++ = pcmvalue >> 8;
            *p++ = pcmvalue >> 16;
            *p++ = pcmvalue >> 24;
            }
         write(swriter[1], pcmbuff, frames * 8);
         }
      else
         {
         for (i = begin; i < end; i++)
            {
            scaled_value = *l++ * 32768;
            pcmvalue = (int32_t)scaled_value;
            if (scaled_value > 32767)
               scaled_value = 32767;
            if (scaled_value < -32768)
               scaled_value = -32768;
            *p++ = pcmvalue;
            *p++ = pcmvalue >> 8;
            
            scaled_value = *r++ * 32768;
            pcmvalue = (int32_t)scaled_value;
            if (scaled_value > 32767)
               scaled_value = 32767;
            if (scaled_value < -32768)
               scaled_value = -32768;
            *p++ = pcmvalue;
            *p++ = pcmvalue >> 8;
            }
         write(swriter[1], pcmbuff, frames * 4);
         }
      }
   
   paused = 0;
   sringbuffer_can_write = FALSE;
   sringbuffer_flush = TRUE;
   while (sringbuffer_flush == TRUE && jack_closed_f == FALSE)
      usleep(5000);
   close(swriter[1]);
   return NULL;
   }

#if HAVE_LIBSHOUT
/* This is how the disconnection from the server is done should the connection hang */
void *watchdog(void *args)
   {
   int count;
   
   while (keep_watchdog)
      {
      usleep(100000);
      count = 0;
      while (disconnect_f == TRUE && keep_watchdog)
         {
	 usleep(100000);
	 if (count++ > 3)	/* number of 10ths of a second before forcible disconnection */
	    {
	    if(pthread_cancel(server_thread_h) == 0)
	       {
               shout_close(shout);
	       shout_free(shout);
	       shout_shutdown();
	       close(reader[0]);
	       pthread_join(codec_feeder_h, NULL); 
	       connection = DISCONNECTED;
	       disconnect_f = FALSE;
	       keep_watchdog = FALSE;
               newmeta = FALSE;
	       fprintf(stderr, "Forcible disconnection by watchdog\n");
	       }
            if(pthread_cancel(codec_streamer_ogg_h) == 0)
               {
               shout_close(shout);
               shout_free(shout);
               shout_shutdown();
               ringbuffer_can_write = FALSE;
               ringbuffer_flush = TRUE;
               while (ringbuffer_flush == TRUE && jack_closed_f == FALSE)
                  usleep(5000);
	       connection = DISCONNECTED;
	       disconnect_f = FALSE;
	       keep_watchdog = FALSE;
               newmeta = FALSE;
	       fprintf(stderr, "Forcible disconnection by watchdog\n");
               }
	    }
	 }
      }
   fprintf(stderr, "watchdog exiting\n");
   return NULL;
   }
      
void *server_thread(void *args)
   {
   int portnum = atoi(port);
   int protocol, fmt;
   unsigned char *mp3buff;
   long qty_read, ret;
   shout_metadata_t *meta;

   usleep(20000);
   shout_init();
   if (!(mp3buff = (unsigned char *)malloc(MP3_BUF_SIZ)))
      {
      fprintf(stderr, "Unable to allocate space for mp3 buffer\n");
      goto error2;
      }   
   if (!(shout = shout_new()))
      {
      fprintf(stderr, "Could not allocate shout_t\n");
      goto error2;
      }
   if (!(meta = shout_metadata_new()))
      {
      fprintf(stderr, "Could not allocate metadata structure\n");
      goto error;
      }
   if (shout_set_host(shout, host) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting hostname: %s\n", shout_get_error(shout));
      goto error;
      }

   if (!strcmp(server_type, "Shoutcast"))
      protocol = SHOUT_PROTOCOL_ICY;
   else if (!strcmp(server_type, "Icecast 2"))
      protocol = SHOUT_PROTOCOL_HTTP;
   else if (!strcmp(server_type, "Icecast"))
      protocol = SHOUT_PROTOCOL_XAUDIOCAST;
   else
      {
      fprintf(stderr, "Protocol not recognised\n");		/* This should never ever happen */
      goto error;
      }
   if (shout_set_protocol(shout, protocol) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting protocol: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_port(shout, portnum) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting port: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_password(shout, password) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting password: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_mount(shout, mount) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting mount: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_user(shout, name) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting user: %s\n", shout_get_error(shout));
      goto error;
      }
   if (!strcmp(format, "mp3"))
      fmt = SHOUT_FORMAT_MP3;
   else
      {
      fmt = SHOUT_FORMAT_OGG;
      fprintf(stderr, "Using ogg format\n");
      }   
   if (shout_set_format(shout, fmt) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting format: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_name(shout, streamname) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream name %s\n", shout_get_error(shout));
      goto error;
      }  
   if (shout_set_url(shout, streamurl) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream url %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_description(shout, streamdesc) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream description %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_genre(shout, streamgenre) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream description %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_public(shout, public_f) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting public flag %s\n", shout_get_error(shout));
      goto error;
      }   
   if (shout_set_audio_info(shout, SHOUT_AI_BITRATE, ai_bitrate) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error declaring the bitrate %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_audio_info(shout, SHOUT_AI_SAMPLERATE, ai_samplerate) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error declaring the samplerate %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_audio_info(shout, SHOUT_AI_CHANNELS, ai_channels) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error declaring the number of channels %s\n", shout_get_error(shout));
      goto error;
      }
   if (pthread_create(&watchdog_h, NULL, watchdog, NULL))
      {
      fprintf(stderr, "Failed to start watchdog thread\n");
      goto error;
      }
      
   if (shout_open(shout) == SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Connected to server...\n");
      total = 0;
      usleep(20000);
      while (keep_connection == TRUE)
         {
         qty_read = read(reader[0], mp3buff, MP3_BUF_SIZ);
         total = total + qty_read;

         if (qty_read > 0)
	    {
	    ret = shout_send(shout, mp3buff, qty_read);
	    if (ret != SHOUTERR_SUCCESS)
	       {
	       fprintf(stderr, "DEBUG: Send error: %s\n", shout_get_error(shout));
	       break;
	       }
	    connection = CONNECTED;
            }
	 else
	    break;
	 if (newmeta == TRUE)
	    {
            if (meta_f)
               {
               pthread_mutex_lock(&pm);
               shout_metadata_add(meta, "song", metatext);
               newmeta = FALSE;
               pthread_mutex_unlock(&pm);
               shout_set_metadata(shout, meta);
               }
	    else
               newmeta = FALSE;
            }   
	 usleep(50000);
         }
      shout_close(shout);
      }
   else
      fprintf(stderr, "Error connecting to server: %s\n", shout_get_error(shout));
      
   pthread_cancel(watchdog_h);
   error:
   shout_free(shout);
   error2:
   shout_shutdown();
   keep_connection = FALSE;
   close(reader[0]);
   pthread_join(codec_feeder_h, NULL);
   fprintf(stderr, "Disconnected\n");
   connection = DISCONNECTED;
   disconnect_f = FALSE;  
   newmeta = FALSE;
   return NULL;
   }

void *codec_feeder(void *args)
   {   
   size_t frames;
   sample_t *lbuff, *rbuff, *l, *r;
   unsigned char *p, *pcmbuff;
   int32_t pcmvalue;
   float scaled_value;
   int i;
   
   usleep(20000);
   if(!(lbuff = malloc(sizeof (sample_t) * SRC_SIZE)))
      {
      fprintf(stderr, "Unable to allocate left buffer in codec_feeder\n");
      keep_connection = FALSE;
      }
   if(!(rbuff = malloc(sizeof (sample_t) * SRC_SIZE)))
      {
      fprintf(stderr, "Unable to allocate right buffer in codec_feeder\n");
      keep_connection = FALSE;
      }
   if(!(pcmbuff = (unsigned char *)malloc(8 * SRC_SIZE)))
      {
      fprintf(stderr, "Unable to allocate space for pcm buffer\n");
      keep_connection = FALSE;
      }

   ringbuffer_can_write = TRUE;		/* Turn on the flow of data from JACK */     
   
   while (keep_connection == TRUE)
      {
      usleep(10000);
      if (!(frames = jack_ringbuffer_read_space(rightring) / sizeof (sample_t)))
         continue;
      frames = (frames > SRC_SIZE) ? SRC_SIZE : frames;
      jack_ringbuffer_read(leftring, (char *)lbuff, frames * sizeof (sample_t));
      jack_ringbuffer_read(rightring, (char *)rbuff, frames * sizeof (sample_t));
      l=lbuff;
      r=rbuff;
      p=pcmbuff;
      for (i = 0; i < frames; i++)
         {
         scaled_value = *l++ * 2147483648.0F;	/* as below but retaining at least 24 bits of precision */
         pcmvalue = (int32_t)scaled_value;
         if (scaled_value > 2147483647.0F)
            pcmvalue = 0x7FFFFFFF;
         if (scaled_value < -2147483648.0F)
            pcmvalue = 0xFFFFFFFF;
         *p++ = pcmvalue;
         *p++ = pcmvalue >> 8;
         *p++ = pcmvalue >> 16;
         *p++ = pcmvalue >> 24;

         scaled_value = *r++ * 2147483648.0F;
         pcmvalue = (int32_t)scaled_value;
         if (scaled_value > 2147483647.0F)
            pcmvalue = 0x7FFFFFFF;
         if (scaled_value < -2147483648.0F)
            pcmvalue = 0xFFFFFFFF;
         *p++ = pcmvalue;
         *p++ = pcmvalue >> 8;
         *p++ = pcmvalue >> 16;
         *p++ = pcmvalue >> 24;
         }
      write(writer[1], pcmbuff, frames * 8);
      }
   
   ringbuffer_can_write = FALSE;
   ringbuffer_flush = TRUE;
   while (ringbuffer_flush == TRUE && jack_closed_f == FALSE)
      usleep(5000);
   close(writer[1]);
   return NULL;
   }

int oe_write_page_shout(ogg_page *page, shout_t *shout)
   {
   int ret1, ret2;
   
   ret1 = shout_send(shout, page->header, page->header_len);
   ret2 = shout_send(shout, page->body, page->body_len);
   if (ret1 == SHOUTERR_SUCCESS && ret2 == SHOUTERR_SUCCESS)
      return page->header_len + page->body_len;
   else
      return 0;
   }

void *codec_streamer_ogg(void *args)
   {   
   ogg_stream_state os;
   ogg_page 		 og;
   ogg_packet 		 op;
   
   vorbis_dsp_state vd;
   vorbis_block     vb;
   vorbis_info      vi;
   vorbis_comment   vc;
   ogg_packet header_main;
   ogg_packet header_comments;
   ogg_packet header_codebooks;
   int result;
   int ret=0;
   int breakout;
   int protocol;
   int portnum = atoi(port);
   int fmt;
   int eos;
   int i;
   int streamid, oldstreamid=0;
   jack_nframes_t frames;
   int channels, bitrate, min_bitrate, max_bitrate;
   sample_t *left_dm_buffer=NULL, *right_dm_buffer=NULL, *ldmptr, *rdmptr;
   float **buffer, *bptr;
      
   usleep(20000);
   /* Shoutcast server initialisation */
   shout_init();
   if (!(shout = shout_new()))
      {
      fprintf(stderr, "Could not allocate shout_t\n");
      goto error2;
      }
   if (shout_set_host(shout, host) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting hostname: %s\n", shout_get_error(shout));
      goto error;
      }

   if (!strcmp(server_type, "Shoutcast"))
      protocol = SHOUT_PROTOCOL_ICY;
   else if (!strcmp(server_type, "Icecast 2"))
      protocol = SHOUT_PROTOCOL_HTTP;
   else if (!strcmp(server_type, "Icecast"))
      protocol = SHOUT_PROTOCOL_XAUDIOCAST;
   else
      {
      fprintf(stderr, "Protocol not recognised\n");		/* This should never ever happen */
      goto error;
      }
   if (shout_set_protocol(shout, protocol) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting protocol: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_port(shout, portnum) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting port: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_password(shout, password) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting password: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_mount(shout, mount) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting mount: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_user(shout, name) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting user: %s\n", shout_get_error(shout));
      goto error;
      }
   if (!strcmp(format, "mp3"))
      {
      fprintf(stderr, "WTF!!!!!!!! using mp3 format\n");
      fmt = SHOUT_FORMAT_MP3;
      }
   else
      {
      fmt = SHOUT_FORMAT_OGG;
      fprintf(stderr, "Using ogg format\n");
      }   
   if (shout_set_format(shout, fmt) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting format: %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_name(shout, streamname) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream name %s\n", shout_get_error(shout));
      goto error;
      }  
   if (shout_set_url(shout, streamurl) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream url %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_description(shout, streamdesc) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream description %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_genre(shout, streamgenre) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting the stream description %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_public(shout, public_f) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error setting public flag %s\n", shout_get_error(shout));
      goto error;
      }   
   if (shout_set_audio_info(shout, SHOUT_AI_BITRATE, ai_bitrate) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error declaring the bitrate %s\n", shout_get_error(shout));
      goto error;
      }
   min_bitrate = max_bitrate = bitrate = atoi(ai_bitrate) * 1000;
   if (abr_f)
      {
      min_bitrate *= 0.5;
      max_bitrate *= 1.5;
      fprintf(stderr, "Using average bitrate settings\n");
      }
   if (shout_set_audio_info(shout, SHOUT_AI_SAMPLERATE, ai_samplerate) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error declaring the samplerate %s\n", shout_get_error(shout));
      goto error;
      }
   if (shout_set_audio_info(shout, SHOUT_AI_CHANNELS, ai_channels) != SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Error declaring the number of channels %s\n", shout_get_error(shout));
      goto error;
      }
   channels = atoi(ai_channels);
   if (pthread_create(&watchdog_h, NULL, watchdog, NULL))
      {
      fprintf(stderr, "Failed to start watchdog thread\n");
      goto error;
      }
      
   if (shout_open(shout) == SHOUTERR_SUCCESS)
      {
      fprintf(stderr, "Connected to server...\n");
      ringbuffer_can_write = TRUE;		/* Turn on the flow of data from JACK */
      total = 0;
      usleep(10000);
      while (keep_connection == TRUE)
         {
         vorbis_info_init(&vi);
         if (vorbis_encode_init(&vi, channels, sr, max_bitrate, bitrate, min_bitrate))
            {
            fprintf(stderr, "vorbis_encode_init error\n");
            goto cleanup;
            }
         vorbis_analysis_init(&vd,&vi);
         vorbis_block_init(&vd,&vb);
         vorbis_comment_init(&vc);
         if (meta_f && meta_artist && meta_title)
            {
            vorbis_comment_add_tag(&vc, "ARTIST", meta_artist);
            vorbis_comment_add_tag(&vc, "TITLE", meta_title);
            vorbis_comment_add_tag(&vc, "COMMENT", streamname);
            }
         streamid = rand();
         while (streamid == oldstreamid)
            streamid = rand();
         fprintf(stderr, "stream id = %d\n", streamid);
         ogg_stream_init(&os, oldstreamid = streamid);
         
         /* Build the packets */
         vorbis_analysis_headerout(&vd,&vc,&header_main,&header_comments,&header_codebooks);

         /* And prepare to stream them out */
         ogg_stream_packetin(&os,&header_main);
         ogg_stream_packetin(&os,&header_comments);
         ogg_stream_packetin(&os,&header_codebooks); 
         while((result = ogg_stream_flush(&os, &og)))
            {
            if(!result) break;
            ret = oe_write_page_shout(&og, shout);
            if(ret != og.header_len + og.body_len)
               {
               fprintf(stderr, "Failed writing header to output stream\n");
               ret = 1;
               goto cleanup; /* Bail and try to clean up stuff */
               }
            else
               connection = CONNECTED;
            }
         fprintf(stderr, "wrote header to stream\n");
         breakout = FALSE;
         eos = FALSE;
         newmeta = FALSE;
         while (keep_connection == TRUE && breakout == FALSE)
            {
            if (!meta_f || newmeta == FALSE)
               {
               if ((frames = jack_ringbuffer_read_space(rightring) / sizeof (sample_t)) < 2000)
                  {
                  usleep(10000);
                  continue;
                  }
               frames = (frames > OGG_META_SRC_SIZE) ? OGG_META_SRC_SIZE : frames;
               buffer = vorbis_analysis_buffer(&vd, frames);
               if (channels == 2)
                  {
                  jack_ringbuffer_read(leftring, (char *)buffer[0], frames * sizeof (sample_t));
                  jack_ringbuffer_read(rightring, (char *)buffer[1], frames * sizeof (sample_t));
                  }
               else
                  {
                  /* downmix to mono */
                  ldmptr = left_dm_buffer = (sample_t *) realloc(left_dm_buffer, frames * sizeof (sample_t));
                  rdmptr = right_dm_buffer = (sample_t *) realloc(right_dm_buffer, frames * sizeof (sample_t));
                  jack_ringbuffer_read(leftring, (char *)left_dm_buffer, frames * sizeof (sample_t));
                  jack_ringbuffer_read(rightring, (char *)right_dm_buffer, frames * sizeof (sample_t));
                  bptr = buffer[0];
                  for (i = 0; i < frames ; i++)
                     *bptr++ = ((float)(*ldmptr++ + *rdmptr++)) * 0.5F;
                  }
               vorbis_analysis_wrote(&vd, frames);
	       }
            else
               {
               vorbis_analysis_wrote(&vd, 0);
               breakout = TRUE;
               }
            while (vorbis_analysis_blockout(&vd, &vb)==1)
               {
               vorbis_analysis(&vb, NULL);
               vorbis_bitrate_addblock(&vb);
               while(vorbis_bitrate_flushpacket(&vd, &op))
                  {
                  ogg_stream_packetin(&os, &op);
                  while(!eos)
                     {
                     int result = ogg_stream_pageout(&os, &og);
                     if(!result) break;
   
                     total += (ret = oe_write_page_shout(&og, shout));
                     if (ret != og.header_len + og.body_len)
                        {
                        fprintf(stderr, "failed writing data to stream\n");
                        ret = 1;
                        goto cleanup;
                        }

                     if (ogg_page_eos(&og))
                        eos = 1;
                     }
                  }
               }
            }
         if (newmeta == TRUE)
            {
            ogg_stream_clear(&os);
            vorbis_block_clear(&vb);
            vorbis_dsp_clear(&vd);
            vorbis_info_clear(&vi);
	    }
         }
      }
   ret = 0; 
cleanup:
   fprintf(stderr, "closing server connection\n");
   shout_close(shout);
   fprintf(stderr, "flushing ringbuffer\n");
   ringbuffer_can_write = FALSE;
   ringbuffer_flush = TRUE;
   while (ringbuffer_flush == TRUE)
      usleep(5000);
   fprintf(stderr, "running ogg encoder cleanup code\n");
   ogg_stream_clear(&os);
   vorbis_block_clear(&vb);
   vorbis_dsp_clear(&vd);
   vorbis_info_clear(&vi);
   fprintf(stderr, "conditionally freeing mono downmix buffers\n");
   if (left_dm_buffer)
      free(left_dm_buffer);
   if (right_dm_buffer)
      free(right_dm_buffer);
error:
   fprintf(stderr, "calling shout_free\n");
   shout_free(shout);   
error2:
   fprintf(stderr, "calling shout_shutdown\n");
   shout_shutdown();
   keep_connection = FALSE;
   connection = DISCONNECTED;
   keep_watchdog = FALSE;
   disconnect_f = FALSE;
   newmeta = FALSE;
   fprintf(stderr, "ogg stream thread exiting\n");
   return NULL;
   }

int stream_connect()
   {
   connection = CONNECTING;
   if (pipe(writer) || pipe(reader))
      {
      perror("Failed to open pipe: ");
      return 5;
      }

   snprintf(ai_bitrate, sizeof (ai_bitrate), "%s", bandwidth);
   snprintf(ai_samplerate, sizeof (ai_samplerate), "%lu", sr);
   if (stereo_f)
      strcpy(ai_channels, "2");
   else
      strcpy(ai_channels, "1");		/* indicate stereo on bandwidths other than 32 */

   if(!(encoder_pid=fork()))
      {
      char *samplerate;
      char *resamplerate;
      int bitrate;
      char *lamepath;
      char *qualval;
      int  highquality;

      close(writer[1]);			/* we will not be using these 2 */
      close(reader[0]);
      close(swriter[1]);		/* and certainly not using these */
      close(sreader[0]);
      dup2(writer[0], STDIN_FILENO);	/* re-open stdin as getting from the writer pipe */
      close(writer[0]);
      dup2(reader[1], STDOUT_FILENO);	/* Use stdin and stdout to access the pipe, now */
      close(reader[1]);			/* so we won't be needing this either now */

      usleep(20000);
      
      bitrate = atoi(bandwidth);
      lamepath = getenv("lame");
      
      if (sr == 44100)			/* Only these two sample rates are supported */
         {
         samplerate = "44.1";
         resamplerate = "22.05";
         }
      else
         {
         samplerate = "48";
         resamplerate = "24";
         }
      if (bitrate <= 24)
         resamplerate = "16";
      
      highquality = high_quality != NULL && high_quality[0] == '1';
      
      if (!strcmp(format, "mp3"))
         {
         usleep(10000);
         if (stereo_f)
            {
            if (bitrate >= 64)
               {
               qualval = highquality ? "1" : "2";
               execlp(lamepath, "lame", "-r", "-x", "-q", qualval, "-s", samplerate, "--cbr", "--noreplaygain", "-b", bandwidth, "--bitwidth", "32", "-", NULL);
               }
            else
               {
               qualval = highquality ? "0" : "2";
               execlp(lamepath, "lame", "-r", "-x", "-q", qualval, "-s", samplerate, "--cbr", "--noreplaygain", "-b", bandwidth, "--resample", resamplerate, "--bitwidth", "32", "-", NULL);
               }
	    }
         else
	    {
            if (bitrate >= 64)
               {
               qualval = highquality ? "1" : "2";
               execlp(lamepath, "lame", "-r", "-x", "-q", qualval, "-s", samplerate, "--cbr", "--noreplaygain", "-b", bandwidth, "-m", "s", "-a", "--bitwidth", "32", "-", NULL);
               }
            else
               {
               qualval = highquality ? "0" : "2";
               execlp(lamepath, "lame", "-r", "-x", "-q", qualval, "-s", samplerate, "--cbr", "--noreplaygain", "-b", bandwidth, "-m", "s", "-a", "--resample", resamplerate, "--bitwidth", "32", "-", NULL);
               }
	    }
	 fprintf(stderr, "Error: execlp failed to spawn lame process or bad option\n");
	 }
      else
         fprintf(stderr, "stream_connect: unsupported mode %s\n", format);
      }
   close(writer[0]);		/* we are not using these 2 at this end */
   close(reader[1]);            
   if (encoder_pid == -1)
      {
      fprintf(stderr, "Failed to create child process for mp3/ogg encoding: ");
      goto error;
      }
   signal(SIGCLD, SIG_IGN);

   usleep(50000);
      
   keep_connection = TRUE;	/* stops our read and write threads from exiting */
   keep_watchdog = TRUE;	/* stops our watchdog from exiting */
   if(pthread_create(&server_thread_h, NULL, server_thread, NULL))
      {
      fprintf(stderr, "Failed to start server thread\n");
      goto error;
      }
      
   usleep(10000);
      
   if (pthread_create(&codec_feeder_h, NULL, codec_feeder, NULL))
      {
      fprintf(stderr, "Failed to start codec feeder thread\n");
      goto error;
      }
      
   usleep(10000);

   return 0;
   error:
   close(writer[1]);
   close(reader[0]);
   return 5;
   }
   
int stream_connect_ogg()
   {
   connection = CONNECTING;

   snprintf(ai_bitrate, sizeof (ai_bitrate), "%s", bandwidth);
   snprintf(ai_samplerate, sizeof (ai_samplerate), "%lu", sr);
   if (stereo_f)
      strcpy(ai_channels, "2");
   else
      strcpy(ai_channels, "1");		/* indicate stereo on bandwidths other than 32 */
   keep_connection = TRUE;	/* stops our read and write threads from exiting */
   keep_watchdog = TRUE;	/* stops our watchdog from exiting */
   usleep(20000);
   if (pthread_create(&codec_streamer_ogg_h, NULL, codec_streamer_ogg, NULL))
      {
      fprintf(stderr, "Failed to start ogg codec/streamer thread\n");
      return 5;
      }
   usleep(20000);
   return 0;
   }   
   
void disconnect()
   {
   if (keep_connection == TRUE)
      {
      fprintf(stderr, "Disconnect... waiting for server threads to finish\n");
      keep_connection = FALSE;
      disconnect_f = TRUE;		/* enable disconnection via the watchdog thread */
      if (strcmp(format, "mp3") != 0)
         {
         pthread_join(codec_streamer_ogg_h, NULL);
         }
      else
         {
         pthread_join(server_thread_h, NULL);
         pthread_join(codec_feeder_h, NULL);
         }
      pthread_join(watchdog_h, NULL);
      fprintf(stderr, "all threads joined\n");
      }
   else
      connection = DISCONNECTED;		/* Just in case */
   }   

#endif /*HAVE_LIBSHOUT*/

void *saver_thread(void *args)
   {
   long qty_read;
   unsigned char *mp3buff;
   FILE *fp;

   usleep(20000);
   if (!(mp3buff = (unsigned char *)malloc(MP3_BUF_SIZ)))
      {
      fprintf(stderr, "Unable to allocate space for mp3 buffer\n");
      close(sreader[0]);
      return NULL;
      }

   fp = fopen(save_file, "w");
   if (fp == NULL)
      {
      fprintf(stderr, "Unable to open file %s for writing\n", save_file);
      close(sreader[0]);
      free(mp3buff);
      return NULL;
      }

   is_saving = TRUE;

   while (TRUE)
      {
      usleep(50000);
      qty_read = read(sreader[0], mp3buff, MP3_BUF_SIZ);
      if (qty_read > 0)
         fwrite(mp3buff, 1, qty_read, fp);
      else
         break;
      }
   fclose (fp);
   close(sreader[0]);
   free(mp3buff);
   return NULL;
   }
   
void stop_saving()
   {
   if (keep_saving == TRUE)
      {
      keep_saving = FALSE;
      pthread_join(saver_thread_h, NULL);
      is_saving = FALSE;
      fprintf(stderr, "Saving stopped\n");
      }
   }

void shutdown(void *arg)
   {
   jack_closed_f = TRUE;
   }

int start_saving()
   {
   if (pipe(swriter) || pipe(sreader))
      {
      perror("Failed to open pipe: ");
      return 5;
      }
   usleep(10000);
      
   if(!(saver_pid=fork()))
      {
      char *samplerate;
      char *resamplerate;
      char oggsr[20];
      int bitrate;
      
      close(swriter[1]);		/* we will not be using these 2 */
      close(sreader[0]);
      close(writer[1]);			/* and we certainly are not using these */
      close(reader[0]);
      dup2(swriter[0], STDIN_FILENO);	/* re-open stdin as getting from the writer pipe */
      close(swriter[0]);
      dup2(sreader[1], STDOUT_FILENO);	/* Use stdin and stdout to access the pipe, now */
      close(sreader[1]);			/* so we won't be needing this either now */
      
      usleep(20000);
      
      if (sr == 44100)			/* Only these two sample rates are supported */
         {
         samplerate = "44.1";
         resamplerate = "22.05";
         }
      else
         {
         samplerate = "48";
         resamplerate = "24";
         }
      sprintf(oggsr, "%lu", sr);
      bitrate = atoi(save_bitrate);
      if (bitrate <= 32)
         resamplerate = "16";
      
      if (!strcmp(save_format, "mp3"))
         {
         fprintf(stderr, "mp3 chosen\n");
	 if (bitrate >= 64)
            {
            execlp(getenv("lame"), "lame", "--bitwidth", "32", "-r", "-x", "-h", "-s", samplerate, "--cbr", "-b", save_bitrate, "--tt", save_title, "--ta", save_artist, "-", NULL);
            }
         else
            {
            execlp(getenv("lame"), "lame", "--bitwidth", "32", "-r", "-x", "-h", "-s", samplerate, "--cbr", "-b", save_bitrate, "--tt", save_title, "--ta", save_artist, "--resample", resamplerate, "-", NULL);
            }
	 }
      else if (!strcmp(save_format, "ogg"))
         {
         fprintf(stderr, "ogg chosen\n");
         usleep(10000);
	 execlp(getenv("oggenc"), "oggenc", "-r", "-R", oggsr, "-q", save_bitrate, "-t", save_title, "-a", save_artist, "-", NULL);
	 }
      else
         fprintf(stderr, "Unknown format %s chosen\n", save_format);
      exit(-5);
      }
   close(swriter[0]);		/* we are not using these 2 at this end */
   close(sreader[1]);
   
   if (saver_pid == -1)
      {
      fprintf(stderr, "Failed to create child process for mp3/ogg encoding: ");
      goto error;
      }
   signal(SIGCLD, SIG_IGN);
   usleep(30000);
   keep_saving = TRUE;	/* stops our read and write threads from exiting */
   if(pthread_create(&saver_thread_h, NULL, saver_thread, NULL))
      {
      fprintf(stderr, "Failed to start saver thread\n");
      goto error;
      }
   usleep(30000);
   if (pthread_create(&save_codec_feeder_h, NULL, save_codec_feeder, NULL))
      {
      fprintf(stderr, "Failed to start save codec feeder thread\n");
      goto error;
      }

   return 0;
   error:
   close(swriter[1]);
   close(sreader[0]);
   return 5;
   }

void interrupt_handler(int data)	/* eats the first ^C */
   {						/* the parent program will handle it */
   static int count = 0;

   if (count++ > 9)
      {
      fprintf(stderr, "Server exiting due to ^C interrupt.\n");
      exit(130);
      }
   else
      {
      if (count == 1)
         fprintf(stderr, "Server got ^C interrupt.\n");
      sigdelset(&serversigset, SIGINT);
      }
   }

void my_jack_error_callback(const char *message)
   {
   printf("JACK: %s\n", message);
   fflush(stdout);
   }

int main(int argc, char **argv)
   {
   FILE *fp = stdin;
   char *client_name;

   setenv("LC_ALL", "C", 1);

   if(sigfillset(&serversigset))
      {
      fprintf(stderr, "sigfillset failed\n");
      return 1;
      }
   signal(SIGINT, interrupt_handler);
   
   srand(time(NULL));
   jack_set_error_function(my_jack_error_callback);
   if(!(client_name = (char *) malloc(9 * sizeof (char))))
      {
      fprintf(stderr, "malloc failure\n");
      return 1;
      }
   strcpy(client_name, "idjc-srv");
   if((client = jack_client_new(client_name)) == 0)
      {
      fprintf(stderr, "jack server not running?\n");
      return 1;
      }
   
   jack_set_process_callback (client, process, NULL);
   jack_on_shutdown(client, shutdown, NULL);
   
   stream_left_port = jack_port_register(client, "lt_in", JACK_DEFAULT_AUDIO_TYPE, 
   					JackPortIsInput, 0);
   stream_right_port = jack_port_register(client, "rt_in", JACK_DEFAULT_AUDIO_TYPE, 
   					JackPortIsInput, 0);
   
   sr = jack_get_sample_rate(client);
   ap_actuation_delay_samples = AP_ACTUATION_DELAY * (float)sr;
   
#if HAVE_LIBSHOUT
   if (!(leftring = jack_ringbuffer_create(RB_SIZE)))
      {
      fprintf(stderr, "Unable to allocate space for left ringbuffer\n");
      return 0;
      }
      
   if (!(rightring = jack_ringbuffer_create(RB_SIZE)))
      {
      fprintf(stderr, "Unable to allocate space for right ringbuffer\n");
      return 0;
      }
#endif /*HAVE_LIBSHOUT*/

   if (!(leftsring = jack_ringbuffer_create(RSB_SIZE)))
      {
      fprintf(stderr, "Unable to allocate space for left save ringbuffer\n");
      return 0;
      }
      
   if (!(rightsring = jack_ringbuffer_create(RSB_SIZE)))
      {
      fprintf(stderr, "Unable to allocate space for right save ringbuffer\n");
      return 0;
      }
     
   if (jack_activate(client))
      {
      fprintf(stderr, "cannot activate client\n");
      return 1;
      }

   printf("up\n");
   fflush(stdout);

   jack_connect(client, "idjc-mx:str_lt", "idjc-srv:lt_in");
   jack_connect(client, "idjc-mx:str_rt", "idjc-srv:rt_in"); 
      
   while (kvp_parse(kvpdict, fp)) 
      {
      if (jack_closed_f == TRUE)
         break;
#if HAVE_LIBSHOUT
      if (!(strcmp(action, "connect")) && keep_connection == FALSE && disconnect_f == FALSE)
         {
         if (meta_s)
            meta_f = (meta_s[0] == '1') ? 1 : 0;
         if (public_s)
            public_f = (public_s[0] == '1') ? 1 : 0;
         if (abr_s)
            abr_f = (abr_s[0] == '1') ? 1 : 0;
         if (stereo_s)
            stereo_f = (stereo_s[0] == '1') ? 1 : 0;
         if (!strcmp(format, "mp3"))
            {
            if (stream_connect() != 0)
               connection = DISCONNECTED;
            }
         else
            {
            if (stream_connect_ogg() != 0)
               connection = DISCONNECTED;
            }
         }
      if(!(strcmp(action, "disconnect")))
	 disconnect();
      if(!(strcmp(action, "newmeta")))
         {
	 while (newmeta == TRUE)
	    usleep(10000);
	 if (keep_connection == TRUE && connection != DISCONNECTED)
	    newmeta = TRUE;
	 }
#endif /*HAVE_LIBSHOUT*/      
      if(!(strcmp(action, "RecordStart")))
         {
         fprintf(stderr, "Recording: %s\nArtist: %s\nTitle: %s\nFormat: %s %skb/s\n", save_file, save_artist, save_title, save_format, save_bitrate);
         start_saving();
	 }
      if(!(strcmp(action, "RecordStop")))
         {
         stop_saving();
         fprintf(stderr, "Recording finished\n");
         }
      if(!(strcmp(action, "Pause")))
         {
         if (save_pause)
            pause_f = (save_pause[0] == '1') ? 1 : 0;
         if (auto_pause)
            auto_pause_f = (auto_pause[0] == '1') ? 1 : 0;
         if (ap_actuation_delay)
            ap_actuation_delay_samples = atof(ap_actuation_delay) * sr;
         if (ap_cancel_thresh_log)
            ap_cancel_thresh = powf(10.0F, (float)atoi(ap_cancel_thresh_log) / 20.0F);
         }
      if(!(strcmp(action, "status")))
         {
	 printf("IDJCserver: %s %s %lu %d %d\n", reply[connection], status_tag, total, is_saving, paused);
	 fflush(stdout);
	 }
      if (stream_buffer_full)
         {
         stream_buffer_full = FALSE;
         fprintf(stderr, "idjcserver: audio data lost - check CPU load and network connection\n");
         }
      }
   jack_client_close(client);
#if HAVE_LIBSHOUT
   disconnect();
#endif
   stop_saving();
   fprintf(stderr, "Server module has closed\n");
   return 0;
   }
