#   idjcgui.py: Main python code of IDJC
#   Copyright (C) 2005 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

appname= "Internet DJ Console"
copyright = "Copyright 2006 Stephen Fairchild"
copyright2 = u"\xA9 2006 Stephen Fairchild"	# As above but uses copyright character (c)
license = "Released under the GNU General Public License V2.0"
# version is set in the configure.in file

import locale, os, sys

def locale_encoding():
   read = os.popen("locale -a 2>/dev/null")# I have to do all this because an exception 
   validlocales = read.read()		# refuses to be trapped for some reason
   read.close()				# therefore I need to prevent it from occurring
   validlocales = validlocales.splitlines()
   lang = os.environ.get("LANG")
   if lang is not None and lang is not "":
      # I wouldn't need to do this if I could use try/except but for some reason I can't trap any locale exceptions hence this code here.
      lang = lang[:5] + lang[5:].replace("-", "").lower().replace("\x00", "").strip()	
      for each in validlocales:
         each = each[:5] + each[5:].replace("-", "").lower().replace("\x00", "").strip()
         if each == lang:		# almost tempted to add a strip on this line too
            break
      else:
         print "The LANG environment variable points to an invalid locale:", lang
         print "Type 'locale -a' at a console for a list of valid settings"
         del os.environ["LANG"]
   os.environ["LANG"] = locale.setlocale(locale.LC_ALL, "")
   try:
      locale.getpreferredencoding()
   except locale.Error:
      del os.environ["LANG"] 
      locale.setlocale(locale.LC_ALL, "")
      de = "iso8859-1"
   else:
      de = locale.getpreferredencoding()
   if de == "ANSI_X3.4-1968":
      # Avoid using ASCII encoding.  The terminal is most likely latin1 or utf8
      # The ASCII fallback is there for windows compatibility!
      de = "iso-8859-1"
   fe = os.environ.get("G_FILENAME_ENCODING")
   if fe == None or fe == "":
      if os.environ.get("G_BROKEN_FILENAMES") == "1":
         fe = sys.getfilesystemencoding()
         os.environ["G_FILENAME_ENCODING"]=fe
      else:
         fe = "UTF-8"
   elif fe.lower() == "@locale" or fe.lower() == "locale":
      fe = de
      os.environ["G_FILENAME_ENCODING"]=fe
   return de, fe		# note that fe can contain a comma separated list

display_enc, file_enc = locale_encoding()

import pygtk
pygtk.require('2.0')
import gtk, gobject, time, fcntl, signal, stat

# import the python modules for my app
try:
   from idjc_config import *
   from IDJCmedia import *
   from IDJCservergui import *
   from IDJCmixprefs import *
   from IDJCjingles import *
   from langheader import *
except:
   sys.path.append(os.environ.get("pkgpyexecdir"))
   from idjc_config import *
   from IDJCmedia import *
   from IDJCservergui import *
   from IDJCmixprefs import *
   from IDJCjingles import *
   from langheader import *

def make_limiter_scale():
   scalebox = gtk.VBox()
   label = gtk.Label("30")
   alignment = gtk.Alignment(0, 0)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("25")
   alignment = gtk.Alignment(0, .1)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("20")
   alignment = gtk.Alignment(0, .3)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("15")
   alignment = gtk.Alignment(0, .5)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("10")
   alignment = gtk.Alignment(0, .7)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label(" 5")
   alignment = gtk.Alignment(0, .9)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label(" 0")
   alignment = gtk.Alignment(0, 1)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   return scalebox  

def make_meter_scale():		# A logarithmic meter scale for a 'VU' meter
   scalebox = gtk.VBox()
   label = gtk.Label("  0")
   alignment = gtk.Alignment(0, 0)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label(" -6")
   alignment = gtk.Alignment(0, 0)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("-12")
   alignment = gtk.Alignment(0, 0.25)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("-18")
   alignment = gtk.Alignment(0, 0.5)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("-24")
   alignment = gtk.Alignment(0, 0.75)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("-30")
   alignment = gtk.Alignment(0, 1)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   label = gtk.Label("-36")
   alignment = gtk.Alignment(0, 1)
   alignment.add(label)
   label.show()
   scalebox.add(alignment)
   alignment.show()
   return scalebox
   
def make_meter_unit(text, l_meter, r_meter):   
   mic_peak_box = gtk.VBox()
   mic_peak_box.set_border_width(3)
   frame = gtk.Frame()
   frame.set_border_width(4)
   hbox = gtk.HBox()
   hbox.set_border_width(1)
   frame.add(hbox)
   label = gtk.Label(text)
   labelbox = gtk.HBox()
   labelbox.add(label)
   label.show()
   mic_peak_box.pack_start(labelbox, False, False, 0)
   labelbox.show()
   mic_peak_box.add(frame)
   frame.show()
   hbox.show()
   l_meter.set_size_request(20, -1)
   hbox.add(l_meter)
   scalebox = make_meter_scale()
   hbox.add(scalebox)
   scalebox.show()
   r_meter.set_size_request(20, -1)
   hbox.add(r_meter)
   l_meter.show()
   r_meter.show()
   return mic_peak_box
 
class nice_mic_togglebutton(gtk.ToggleButton):
   def __init__(self, label = None, use_underline = True):
      gtk.ToggleButton.__init__(self, label, use_underline)
   def __str__(self):
      return gtk.ToggleButton.__str__() + " wrapped by a prelight remover class"
   def set_sensitive(self, bool):
      if bool is False:
         gtk.ToggleButton.set_sensitive(self, False)
         gtk.ToggleButton.set_active(self, True)
         gtk.ToggleButton.set_inconsistent(self, True)
         gtk.ToggleButton.set_inconsistent(self, False)
      else:
         gtk.ToggleButton.set_sensitive(self, True)
         
# Base class for an audio meter.
class rawmeter(gtk.ProgressBar): # can add stuff here to change the appearance later
   def set_meter_value(self, newdata):
      if newdata > self.scale:
         newdata = self.scale
      if newdata < 0:
         newdata = 0
      if newdata != self.olddata:
         self.olddata = newdata
         self.set_fraction(1.0 - float(newdata) / self.scale)
   def get_meter_value(self):
      return int (((1.0 - self.get_fraction()) * self.scale) + 0.5)
   def __init__(self):
      gtk.ProgressBar.__init__(self)
      gtk.ProgressBar.set_orientation(self, gtk.PROGRESS_BOTTOM_TO_TOP)
      self.scale = 36		# the default number of dB in the scale
      self.olddata = -1
      
class limitermeter(rawmeter):
   def set_meter_value(self, newvalue):
      if newvalue > self.scale:
         newvalue = self.scale
      newvalue = - newvalue + self.scale
      rawmeter.set_meter_value(self, newvalue)
   def __init__(self):
      rawmeter.__init__(self)
      self.scale = 30

class vumeter(rawmeter):	# this meter is fed rms values at 50ms intervals
   def set_meter_value(self, newvalue):
      if newvalue > self.scale:
         newvalue = self.scale
	 
      self.gen6 = self.gen5
      self.gen5 = self.gen4
      self.gen4 = self.gen3
      self.gen3 = self.gen2
      self.gen2 = self.gen1
      self.gen1 = newvalue
      # take a mean average of rms values for 300ms weighted towards recent sounds
      newvalue = (5 * self.gen1 + 6 * self.gen2 + 4 * self.gen3 + 3 * self.gen4 + 2 * self.gen5 + self.gen6 ) / 21
      rawmeter.set_meter_value(self, newvalue)
   def __init__(self):
      rawmeter.__init__(self)
      self.gen1 = self.gen2 = self.gen3 = self.gen4 = self.gen5 = self.scale

class peakholdmeter(rawmeter):
   def set_meter_value(self, newvalue):
      oldvalue = self.get_meter_value()
      if newvalue <= oldvalue + 1:
	 self.lastpeak = 0
	 oldvalue = newvalue
      else:
         self.lastpeak = self.lastpeak + 1
	 if self.lastpeak > self.peakholditers:
	    decfactor = (self.lastpeak - self.peakholditers) * pow(self.lastpeak - self.peakholditers, 1.2)
	    newvalue = oldvalue + decfactor
	 else:
	    newvalue = oldvalue
      rawmeter.set_meter_value(self, newvalue)    
   def __init__(self):
      rawmeter.__init__(self)
      self.lastpeak = 0
      self.oldvalue = 0
      self.peakholditers = 3	# number of calls before the meter starts to decay
  
# A dialog window to appear when shutdown is selected while still streaming.
class idjc_shutdown_dialog:
   def window_attn(self, widget, event):
      if event.new_window_state | gtk.gdk.WINDOW_STATE_ICONIFIED:
         widget.set_urgency_hint(True)
      else:
         widget.set_urgency_hint(False)
   
   def respond(self, dialog, response, actionyes, actionno):
      if response == gtk.RESPONSE_OK:
         print "Dialog quit"
         if actionyes is not None:
            actionyes()
      if response == gtk.RESPONSE_DELETE_EVENT or response == gtk.RESPONSE_CANCEL:
         print "Dialog keep running"
         if actionno is not None:
            actionno()
      dialog.destroy()

   def __init__(self, window_group = None, actionyes = None, actionno = None, additional_text = None):
      dialog = gtk.Dialog(idjc_shutdown_text, None, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT, (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_QUIT, gtk.RESPONSE_OK))
      if window_group is not None:
         window_group.add_window(dialog)
      dialog.set_icon_from_file(pkgdatadir + "icon" + gfext)
      dialog.set_resizable(False)
      dialog.connect("close", self.respond, actionyes, actionno)
      dialog.connect("response", self.respond, actionyes, actionno)
      dialog.connect("window-state-event", self.window_attn)
      
      hbox = gtk.HBox(False, 20)
      hbox.set_border_width(20)
      dialog.vbox.pack_start(hbox, True, True, 0)
      hbox.show()
      image = gtk.Image()
      image.set_from_stock(gtk.STOCK_DIALOG_QUESTION, gtk.ICON_SIZE_DIALOG)
      hbox.pack_start(image, True, True, 0)
      image.show()
      vbox = gtk.VBox()
      vbox.set_spacing(8)
      hbox.pack_start(vbox, True, True, 0)
      vbox.show()
      
      if additional_text is not None:
         if type(additional_text) is str:
            additional_text = additional_text.splitlines()
         for each in additional_text:
            label = gtk.Label()
            attrlist = pango.AttrList()
            attrlist.insert(pango.AttrSize(12500, 0, len(each)))
            label.set_attributes(attrlist)
            label.set_text(each)
            vbox.add(label)
            label.show()
      dialog.show()
  
class MainWindow:
   def send_new_mixer_stats(self):
      string_to_send = ":%03d:%03d:%03d:%03d:%03d:%d:%d%d%d%d%d:%d%d:%d%d%d:%d%d%d%d:%d:%d:%d:%d:%d:" % (
						self.deckadj.get_value(),
						self.crossadj.get_value(),
                                                self.jingles.deckadj.get_value(),
                                                self.jingles.interadj.get_value(),
                                                self.mixbackadj.get_value(),
                                                self.jingles.playing,
						self.player_left.stream.get_active(),
						self.player_left.listen.get_active(),
						self.player_right.stream.get_active(),
						self.player_right.listen.get_active(),
						self.listen.get_active(),
						self.player_left.pause.get_active(),
						self.player_right.pause.get_active(),
						self.mic_lselect.get_active(),
						self.mic_rselect.get_active(),
						self.aux_select.get_active(),
						self.player_left.flush,
						self.player_right.flush,
						self.jingles.flush,
                                                self.jingles.interludeflush,
						self.simplemixer,
                                                self.alarm,
                                                self.mixermode,
                                                self.prefs_window.fadeout.get_active(),
                                                self.player_left.play.get_active() or
                                                self.player_right.play.get_active())
      self.mixer_write("MIXR=%s\nACTN=mixstats\nend\n" % string_to_send, True)

      self.alarm = False
      iteration = 0
      while self.player_left.flush or self.player_right.flush or self.jingles.flush or self.jingles.interludeflush:
         time.sleep(0.05)
         self.vu_update(False)
         self.jingles.interludeflush = self.jingles.interludeflush & self.interlude_playing.value
         self.jingles.flush = self.jingles.flush & self.jingles_playing.value
         self.player_left.flush = self.player_left.flush & self.player_left.mixer_playing.value
         self.player_right.flush = self.player_right.flush & self.player_right.mixer_playing.value
      
      if self.crossadj.get_value() < 50:
         self.songname = self.player_left.songname
         self.artist = self.player_left.artist
         self.title = self.player_left.title
      else:
         self.songname = self.player_right.songname
         self.artist = self.player_right.artist
         self.title = self.player_right.title
         
      if self.metadata != self.songname:
         self.metadata = self.songname			# now we need to send the new metadata
	 print "song title: %s\n" % self.metadata.encode(self.denc, "replace")
	 if self.metadata != u"":
	    self.window.set_title("%s :: IDJC" % self.metadata)
            tm = time.localtime()
            ts = "%02d:%02d :: " % (tm[3], tm[4])	# hours and minutes
            self.history_buffer.place_cursor(self.history_buffer.get_end_iter())
            self.history_buffer.insert_at_cursor(ts + self.metadata + "\n")
            adjustment = self.history_window.get_vadjustment()
            adjustment.set_value(adjustment.upper)
            try:
               file = open(self.idjc + "history.log", "a")
            except IOError:
               print "unable to open history.log for writing"
            else:
               try:
                  file.write(time.strftime("%x %X :: ") + self.metadata.encode("utf-8") + "\n")
               except IOError:
                  print "unable to append to file \"history.log\""
               file.close()

            try: # The song the DJ is listening to for the xchat /listening trigger
               file = open(self.idjc + "songtitle", "w")
	       file.write(self.metadata)		# <--- This is wrong
	       file.close()
	    except IOError:
	       print "Failed to write the song title to disk"
	       
            # Write out the x-chat track announcement to a file
            # note how the announcement is done on a time delay
            if self.prefs_window.announce_enable.get_active() and self.server_window.connect_button.get_active():
	       message_text = "d" + str(len(self.prefs_window.nickentry.get_text().encode("utf-8"))) + ":" +			self.prefs_window.nickentry.get_text().encode("utf-8") + "d" +	str(len(self.prefs_window.channelsentry.get_text().encode("utf-8"))) + ":" + 			self.prefs_window.channelsentry.get_text().encode("utf-8")
               cookedmessage = self.cook(self.prefs_window.announcemessageentry.get_text())
	       message_text = message_text + "d" + str(len(cookedmessage)) + ":" + cookedmessage
	       timestr = str(int(time.time() + self.prefs_window.announcedelayadj.get_value()))
	       message_text = message_text +  "d" + str(len(timestr)) + ":" + timestr
	       gobject.timeout_add(int(self.prefs_window.announcedelayadj.get_value() * 1000 - 500), self.announce_the_track, message_text)
            self.server_window.new_meta(self.metadata, self.artist, self.title)    
         else:
	    self.window.set_title(self.appname)
	 
         
   def announce_the_track(self, message):
      try:
	 file = open(self.idjc + "announce.xchat", "w")
	 fcntl.flock(file.fileno(), fcntl.LOCK_EX)
         file.write(message)
	 fcntl.flock(file.fileno(), fcntl.LOCK_UN)
	 file.close()
      except IOError:
	 print "Problem writing the announcement file to disk"
      return False  
      
   def cook(self, rawtext):
      rawtext = rawtext.replace("%s", self.metadata)
      rawtext = rawtext.replace("%n", self.server_window.name_entry.get_text())
      rawtext = rawtext.replace("%u", self.server_window.url_entry.get_text())
      rawtext = rawtext.replace("%d", self.server_window.description_entry.get_text())
      return rawtext.encode("utf-8")
	 
   def callback(self, widget, data):
      print "%s was pressed" % data
      if data == "Advance":
         if self.crossfade.get_value() < 50:
            self.player_left.advance()
         else:
            self.player_right.advance()
      if data == "Server":
         self.server_window.window.present()
         self.server_window.password_entry.grab_focus()
      if data == "Prefs":
         self.prefs_window.window.present()
      if data == "Jingles":
         self.jingles.window.present()
         self.jingles.entry.grab_focus()
      if data == "cfleft":
         if self.crosspass:
            gobject.source_remove(self.crosspass)
            self.crosspass = 0
         self.crossfade.set_value(0)
      if data == "cfright":
         if self.crosspass:
            gobject.source_remove(self.crosspass)
            self.crosspass = 0
         self.crossfade.set_value(100)
      if data == "pass-crossfader":
         if self.crosspass:
	    self.crossdirection = not self.crossdirection
	 else:
	    self.crossdirection = (self.crossadj.get_value() <= 50)
	    self.crosspass = gobject.timeout_add(int(self.passspeed_adj.get_value() * 10), self.cb_crosspass)
      if data == "Clear History":
         self.history_buffer.set_text("")

   def expandercallback(self, expander, param_spec, user_data=None): 
      if expander.get_expanded():
         self.history_vbox.show()
      else:
         self.history_vbox.hide()
      if self.player_left.is_playing:
         self.player_left.reselect_cursor_please = True
      if self.player_right.is_playing:
         self.player_right.reselect_cursor_please = True

   def cb_crosspass(self):
      gtk.gdk.threads_enter()
      x = self.crossadj.get_value()
      if x == 100 * self.crossdirection:
         self.crosspass = 0
         gtk.gdk.threads_leave()
	 return False
      if self.crossdirection:
	 self.crossfade.set_value(x+1)
      else:
	 self.crossfade.set_value(x-1)
      gtk.gdk.threads_leave()
      return True
	    
   def cb_toggle(self, widget, data):
      print "Hello there, %s was toggled %s" % (data, ("OFF","ON")[widget.get_active()])
      if data == "Aux Open":
         self.send_new_mixer_stats()
      if data == "Mic L Open":
         self.new_mixermode(self.mixermode)
      if data == "Mic R Open":
         self.new_mixermode(self.mixermode)
      if data == "Mic Open":
         x = self.mic_select.get_active()
         self.mic_lselect.set_active(x)
	 self.mic_rselect.set_active(x)
      if data == "stream-mon":
         self.send_new_mixer_stats()
      if data == "Greenphone":
         mode = self.mixermode
         if widget.get_active() == True:
            if self.mixermode == self.PRIVATE_PHONE:
               self.mixermode = self.PUBLIC_PHONE
               self.redphone.set_active(False)
            self.mixermode = self.PUBLIC_PHONE
         else:
            if self.mixermode == self.PUBLIC_PHONE:
               self.mixermode = self.NO_PHONE
         if self.mixermode != mode:
            self.new_mixermode(self.mixermode)
      if data == "Redphone":
         mode = self.mixermode
         if widget.get_active() == True:
            if self.mixermode == self.PUBLIC_PHONE:
               self.mixermode = self.PRIVATE_PHONE
               self.greenphone.set_active(False)
            self.mixermode = self.PRIVATE_PHONE
         else:
            if self.mixermode == self.PRIVATE_PHONE:
               self.mixermode = self.NO_PHONE
         if self.mixermode != mode:
            self.new_mixermode(self.mixermode)
            
   def new_mixermode(self, mode):
      mic = self.mic_lselect.get_active() or self.mic_rselect.get_active()
      sens = (mode == self.NO_PHONE or mode == self.PUBLIC_PHONE or mic == True)
      self.player_left.listen.set_sensitive(sens)
      self.player_right.listen.set_sensitive(sens)
      sens = mode != self.PUBLIC_PHONE
      self.mic_select.set_sensitive(sens)
      self.mic_lselect.set_sensitive(sens)
      self.mic_rselect.set_sensitive(sens)
      if mode == self.PRIVATE_PHONE:
         self.spacerbox.show()
         self.pbphoneimage.show()
         self.mixback.show()
      else:
         self.spacerbox.hide()
         self.pbphoneimage.hide()
         self.mixback.hide()
      self.send_new_mixer_stats()

   def cb_crossfade(self, fader):
      # expire song title data on players that are not on due the crossfader position
      if self.crossadj.get_value() < 50:
         if self.player_right.is_playing == False:
            self.player_right.songname = u""
      else:
         if self.player_left.is_playing == False:
            self.player_left.songname = u""
      # update the mixer of the new crossfader setting and also the metadata if need be
      self.send_new_mixer_stats()
   
   def cb_deckvol(self, gain):
      #print "Hello there, the volume control was moved, value = %d" % gain.value
      self.send_new_mixer_stats()

   def mic_click(self, widget, event):
      if event.button == 3:
         if self.mic_select.flags() & gtk.VISIBLE:
	    self.mic_select.hide()
	    self.mic_lselect.show()
	    self.mic_rselect.show()
	 else:
	    x = self.mic_lselect.get_active()
	    if x == self.mic_rselect.get_active():
	       self.mic_lselect.hide()
	       self.mic_rselect.hide()
	       self.mic_select.set_active(x)
	       self.mic_select.show()
         return True
      return False

   def cb_menu_select(self, widget, data):
      print ("Hello, %s was chosen from the menu" % data)   
            
   def save_session(self):
      try:
         fh = open(self.session_filename, "w")
         fh.write("deckvol=" + str(self.deckadj.get_value()) + "\n")
         fh.write("crossfade=" + str(self.crossadj.get_value()) + "\n")
         fh.write("stream_mon="+ str(int(self.listen.get_active())) + "\n")
         fh.write("tracks_played=" + str(int(self.history_expander.get_expanded())) + "\n")
         fh.write("pass_speed=" + str(self.passspeed_adj.get_value()) + "\n")
         fh.write("prefs=" + str(int((self.prefs_window.window.flags() & gtk.VISIBLE) != 0)) + "\n")
         fh.write("server=" + str(int((self.server_window.window.flags() & gtk.VISIBLE) != 0)) + "\n")
         fh.write("jingles=" + str(int((self.jingles.window.flags() & gtk.VISIBLE) != 0)) + "\n")
         fh.write("prefspage=" + str(self.prefs_window.notebook.get_current_page()) + "\n")
         fh.close()
      except:
         print "Error writing out main session data"
      try:
         fh = open(self.session_filename + "_tracks", "w")
         start, end = self.history_buffer.get_bounds()
         text = self.history_buffer.get_text(start, end)
         fh.write(text)
         fh.close()
      except:
         print "Error writing out tracks played data"
      
   def restore_session(self):
      try:
         fh = open(self.session_filename, "r")
      except:
         return
      while 1:
         try:
            line = fh.readline()
            if line == "":
               break
         except:
               break
         if line.startswith("deckvol="):
            self.deckadj.set_value(float(line[8:-1]))
         if line.startswith("crossfade="):
            self.crossadj.set_value(float(line[10:-1]))
         if line.startswith("stream_mon="):
            self.listen.set_active(int(line[11:-1]))
         if line.startswith("tracks_played="):
            if int(line[14:-1]):
               self.history_expander.emit("activate")
         if line.startswith("pass_speed="):
            self.passspeed_adj.set_value(float(line[11:-1]))
         if line.startswith("prefs=1"):
            self.prefs_window.window.show()
         if line.startswith("server=1"):
            self.server_window.window.show()
         if line.startswith("jingles=1"):
            self.jingles.window.show()
         if line.startswith("prefspage="):
            self.prefs_window.notebook.set_current_page(int(line[10:-1]))
      stat = os.stat(self.session_filename + "_tracks")
      if stat.st_ctime + 21600 > time.time():
         try:
            fh = open(self.session_filename + "_tracks", "r")
         except:
            return
         text = fh.read()
         fh.close()
         self.history_buffer.set_text(text)
      else:
         print "Track history text is more than six hours old.  Disregarding.\n"

   def destroy_hard(self, widget=None, data=None):
      try:
         gtk.main_quit()
      except:
         pass
      gtk.gdk.threads_leave()
      sys.exit(0)

   def destroy(self, widget=None, data=None):
      signal.signal(signal.SIGINT, self.destroy_hard)
      if self.crosspass:
         gobject.source_remove(self.crosspass)
      if self.server_window.server_on_f == True:
         print "Shutdown: awaiting disconnection from server"
         self.server_window.disconnect()
         while self.server_window.server_on_f:
            gtk.gdk.threads_leave()
            self.server_window.server_monitor()
            gtk.gdk.threads_enter()
            time.sleep(0.1)
      if self.server_window.is_recording == True:
         print "Shutdown: stopping recording"
         self.server_window.stop_recording()
         while self.server_window.is_saving:
            gtk.gdk.threads_leave()
            self.server_window.server_monitor()
            gtk.gdk.threads_enter()
            time.sleep(0.1)
      gobject.source_remove(self.server_window.servmon_source_id)

      self.mic_select.set_active(False)
      self.aux_select.set_active(False)
      self.player_left.cleanup()
      self.player_right.cleanup()
      self.jingles.cleanup()
      self.player_left.flush = True
      self.player_right.flush = True
      self.send_new_mixer_stats()
      gobject.source_remove(self.statstimeout)
      gobject.source_remove(self.vutimeout)
      self.save_session()
      self.prefs_window.save_player_prefs()
      self.server_window.serv_cmd.close()
      self.mixer_ctrl.close()
      self.server_window.save(self.idjc)
 
      gtk.main_quit()
      time.sleep(0.35)	# Allow time for all subthreads/programs time to exit 
      gtk.gdk.threads_leave()
      sys.exit(0)

   def delete_event(self, widget, event, data=None):
      if self.server_window.server_on_f == True:
         idjc_shutdown_dialog(self.window_group, self.destroy, None, (is_streaming_text, question_quit_text))
         return True
      elif self.server_window.is_recording:
         idjc_shutdown_dialog(self.window_group, self.destroy, None, (is_recording_text, question_quit_text))
         return True
      self.destroy()
      return False

   # the all singing all dancing error trapping mixer restarting mixer communications method
   # all mixer write operations should go through here so that broken pipes can be trapped properly
   def mixer_write(self, message, flush = False):
      try:
         self.mixer_ctrl.write(message)
         if flush:
            self.mixer_ctrl.flush()
      except (IOError, ValueError):
         print "*** Mixer has likely crashed ***"
         self.mixer_ctrl, self.mixer_rply = os.popen2([ libexecdir + "idjcmixer" ], 4096)
         if self.mixer_ctrl == 0 or self.mixer_rply == 0:
            print "Unable to open a pipe to the mixer"
            self.destroy_hard()
         rply = self.mixer_read()
         while rply[:17] != "IDJC: Sample rate":
            if rply == "":
               if self.last_chance == True:
                  print "mixer crashed a third time -- exiting"
                  self.destroy_hard()
               self.last_chance = True
               print "mixer has crashed a second time"
               mixer_write(self, message, flush)
            rply = self.mixer_read()
            if rply[:6] == "JACK: ":
               print rply
               self.destroy_hard()
         self.samplerate = rply[18:].strip()
         print "Sample rate is %s" % self.samplerate
         if self.samplerate != "44100" and self.samplerate !="48000":
            print "Sample rate not supported or garbled reply from mixer"
            self.destroy_hard()
         self.mixer_write("ACTN=nothing\nend\nACTN=sync\nend\n", True)
         print "attempting to sync"
         while 1:
            rply = self.mixer_read()
            if rply == "IDJC: sync reply\n":
               break;
            if rply == "":
               print "mixer has crashed a second time"
               self.destroy_hard()
            print "attempting to sync"
         print "got sync"
         
         self.send_new_mixer_stats()			# restore previous settings to the mixer
         self.prefs_window.send_new_compressor_stats()
         self.prefs_window.send_new_noisegate_stats()
         self.prefs_window.send_new_microphone_stats()
         self.prefs_window.send_new_normalizer_stats()
         self.prefs_window.bind_jack_ports()
         self.mixer_write("ACTN=serverbind\n", True)	# tell mixer to bind audio with the server module
         self.player_left.next.clicked()
         self.player_right.next.clicked()
         self.jingles.stop.clicked()
         if self.jingles.interlude_player_track != "":
            self.jingles.start_interlude_player(self.jingles.interlude_player_track)
         self.mixer_write(message, flush)	# resume what we were doing
      else:
         self.last_chance = False

   def mixer_read(self):
      line = self.mixer_rply.readline()
      if line == "Segmentation Fault\n":
         line = ""
         print "Mixer reports a segmentation fault"
         self.mixer_rply.close()
         self.mixer_ctrl.close()
      return line

   def vu_update(self, locking = True):
      if locking:
         gtk.gdk.threads_enter()
      try:
         self.mixer_write("ACTN=requestlevels\nend\n", True)
      except ValueError, IOError:
         if locking:
            gtk.gdk.threads_leave()
         return True

      while 1:
         line = self.mixer_read()
         if line == "":
            if locking:
               gtk.gdk.threads_leave()
            return True
         if line == "end\n":
	    break
         if not line.count("="):
            print "vu_update - invalid line from mixer:", line
            continue
	 line = line.split("=")
	 key = line[0].strip()
         try:
	    value = int(line[1].strip())
         except ValueError:
            print "vu_update - invalid value - not an int", line[0] + "=" + line[1]
            continue
	 try:
            self.vumap[key].set_meter_value(value)
         except KeyError:
	    print "key value", key, "missing from vumap"
      if self.jingles.playing == True and int(self.jingles_playing) == 0:
         self.jingles.stop_player(False)

      if locking:
         gtk.gdk.threads_leave()
      return True
      
   def stats_update(self):
      gtk.gdk.threads_enter()
      if not self.player_left.player_is_playing:
         self.player_left.update_time_stats()
      else:
         if self.prefs_window.fadeout.get_active():
            self.player_left.check_mixer_signal()
      if not self.player_right.player_is_playing:
         self.player_right.update_time_stats()
      else:
         if self.prefs_window.fadeout.get_active():
            self.player_right.check_mixer_signal()
      gtk.gdk.threads_leave()
      return True

   def cb_history_populate(self, textview, menu):
      menusep = gtk.SeparatorMenuItem()
      menu.append(menusep)
      menusep.show()
      menuitem = gtk.MenuItem(track_history_clear_text)
      menuitem.connect_object("activate", gtk.Button.clicked, self.history_clear)
      menu.append(menuitem)
      menuitem.show()

   def configure_event(self, widget, event):
      if self.player_left.is_playing:
         self.player_left.reselect_cursor_please = True
      if self.player_right.is_playing:
         self.player_right.reselect_cursor_please = True
      if not self.simplemixer:
         self.fullwinx.set_value(event.width)
         self.fullwiny.set_value(event.height)
      else:
         self.minwinx.set_value(event.width)
         self.minwiny.set_value(event.height)

   class initfailed:
      def __init__(self, errormessage = "Something bad happened and IDJC could not continue"):
         print errormessage
         
   class initcleanexit:
      pass

   def __init__(self, clargs):
      signal.signal(signal.SIGINT, self.destroy_hard)
      
      self.appname = appname
      self.version = version
      self.copyright = copyright2
      self.license = license
      
      print appname, "Version", version, "\n" + copyright, "\n" + license + "\n"
      for each in clargs:
         if each == "-v" or each == "--version":
            raise self.initcleanexit
      if shoutenabled == 0:
         print "Streaming is disabled.  To enable it install libshout then reinstall IDJC."
      self.denc = display_enc
      self.fenc = file_enc
      print "display encoding:", self.denc, "filesystem encoding:", self.fenc
      home = os.environ.get("HOME")
      if home == "":
         raise self.initfailed("Error, HOME environment variable is not set")
      self.home = home
      if not os.path.isdir(home + "/.idjc"):
         os.mkdir(home + "/.idjc")
      if not os.path.isdir(home + "/.idjc/jingles"):
         os.mkdir(home + "/.idjc/jingles")
      self.idjc = home + "/.idjc/"
      
      self.mixer_ctrl, self.mixer_rply = os.popen2([ libexecdir + "idjcmixer" ], 4096)
      if self.mixer_ctrl == 0 or self.mixer_rply == 0:
         print "Unable to open a pipe to the mixer"
	 self.init_failed = True
         return
  
      self.window_group = gtk.WindowGroup()
      self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
      self.window_group.add_window(self.window)
      self.window.set_title(self.appname)
      self.window.set_border_width(8)
      self.window.connect("delete_event",self.delete_event)
      self.window.set_icon_from_file(pkgdatadir + "icon" + gfext)
      
      self.hbox10 = gtk.HBox(False)
      self.hbox10.set_border_width(1)
      self.hbox10.set_spacing(5)
      self.hbox9 = gtk.HBox(False, 0)
      self.hbox9.set_spacing(6)
      self.vbox8 = gtk.VBox(False, 0)
      self.hbox9.pack_start(self.vbox8, True, True ,0)
      self.window.add(self.hbox9)
      self.hbox9.show()
      # add box 6 to box 8
      self.vbox6 = gtk.VBox(False, 0)
      self.vbox8.pack_start(self.vbox6, True, True, 0)
      # add box 7 to box 8
      self.hbox7 = gtk.HBox(True)
      self.hbox7.set_spacing(5)
      self.hbox7.set_border_width(3)
      
      hbox = gtk.HBox()
      hbox.set_border_width(6)
      frame = gtk.Frame()
      frame.set_border_width(0)
      frame.set_shadow_type(gtk.SHADOW_IN)
      frame.add(self.hbox7)
      hbox.pack_start(frame, True, True, 0)
      frame.show()
      
      indicatorbox = gtk.VBox()
      indicatorbox.set_border_width(1)
      indicatorbox.set_spacing(1)
      hbox.pack_start(indicatorbox, False, False, 6)
      indicatorbox.show()
      
      self.frame2 = gtk.Frame()
      self.frame2.set_border_width(0)
      self.frame2.set_shadow_type(gtk.SHADOW_IN)
      self.frame2.add(self.hbox10)
      self.hbox10.show()
      hbox.pack_start(self.frame2, True, True, 0)
      self.frame2.show()
      
      buttonbargroup = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)
      buttonbargroup.add_widget(frame)
      buttonbargroup.add_widget(self.frame2)
      
      self.vbox8.pack_start(hbox, False, False, 0)
      hbox.show()
      
      # show box 8 now that it's finished
      self.vbox8.show()            
      
      # do box 7 before the others since it is nice and simple
      # add a playlist button to open the playlist window
      self.button = gtk.Button(prefs_button_text)
      self.button.connect("clicked", self.callback, "Prefs")
      self.hbox7.pack_start(self.button, True, True, 0)
      self.button.show()
      
      # add a server button to open the server window
      # this incorporates some led type indicators
      leftunlitpixbuf = gtk.gdk.pixbuf_new_from_file_at_size(pkgdatadir + "led_unlit_clear_border_64x64" + gfext, 14, 14)
      leftamberpixbuf = gtk.gdk.pixbuf_new_from_file_at_size(pkgdatadir +
      "led_lit_amber_black_border_64x64" + gfext, 14, 14)
      rightunlitpixbuf = leftunlitpixbuf.copy()
      leftredpixbuf = gtk.gdk.pixbuf_new_from_file_at_size(pkgdatadir + "led_lit_red_black_border_64x64" + gfext, 14, 14)
      rightamberpixbuf = leftamberpixbuf.copy()
      rightgreenpixbuf = gtk.gdk.pixbuf_new_from_file_at_size(pkgdatadir +
      "led_lit_green_black_border_64x64" + gfext, 14, 14)
      
      self.left_unlit_image = gtk.Image()
      self.left_unlit_image.set_from_pixbuf(leftunlitpixbuf)
      self.right_unlit_image = gtk.Image()
      self.right_unlit_image.set_from_pixbuf(rightunlitpixbuf)
      self.left_red_image = gtk.Image()
      self.left_red_image.set_from_pixbuf(leftredpixbuf)
      self.left_amber_image = gtk.Image()
      self.left_amber_image.set_from_pixbuf(leftamberpixbuf)
      self.right_amber_image = gtk.Image()
      self.right_amber_image.set_from_pixbuf(rightamberpixbuf)
      self.right_green_image = gtk.Image()
      self.right_green_image.set_from_pixbuf(rightgreenpixbuf)
      self.server_button = gtk.Button(server_button_text)
      self.server_button.connect("clicked", self.callback, "Server")
      self.hbox7.pack_start(self.server_button, True, True, 0)
      self.server_button.show()
      
      indicatorbox.add(self.right_unlit_image)
      indicatorbox.add(self.right_amber_image)
      indicatorbox.add(self.right_green_image)
      indicatorbox.add(self.left_unlit_image)
      indicatorbox.add(self.left_amber_image)
      indicatorbox.add(self.left_red_image)
      self.left_unlit_image.show()
      self.right_unlit_image.show()
      
      # add a jingles button to open the jingles window
      self.jingles_button = gtk.Button(jingles_button_text)
      self.jingles_button.connect("clicked", self.callback, "Jingles")
      self.hbox7.pack_start(self.jingles_button, True, True, 0)
      self.jingles_button.show()
      
      phonebox = gtk.HBox()
      
      pixbuf4 = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "greenphone" + gfext)
      pixbuf4 = pixbuf4.scale_simple(25, 20, gtk.gdk.INTERP_BILINEAR)
      image = gtk.Image()
      image.set_from_pixbuf(pixbuf4)
      image.show()
      self.greenphone = gtk.ToggleButton()
      self.greenphone.add(image)
      self.greenphone.connect("toggled", self.cb_toggle, "Greenphone")
      phonebox.pack_start(self.greenphone, False, False, 0)
      self.greenphone.show()

      pixbuf5 = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "redphone" + gfext)
      pixbuf5 = pixbuf5.scale_simple(25, 20, gtk.gdk.INTERP_BILINEAR)
      image = gtk.Image()
      image.set_from_pixbuf(pixbuf5)
      image.show()
      self.redphone = gtk.ToggleButton()
      self.redphone.add(image)
      self.redphone.connect("toggled", self.cb_toggle, "Redphone")
      phonebox.pack_start(self.redphone, False, False, 0)
      self.redphone.show()
 
      self.hbox10.pack_start(phonebox, False, False, 0)
      phonebox.show()
 
      pixbuf3 = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "jack2" + gfext)
      pixbuf3 = pixbuf3.scale_simple(32, 20, gtk.gdk.INTERP_BILINEAR)
      image = gtk.Image()
      image.set_from_pixbuf(pixbuf3)
      image.show()
      self.aux_select = gtk.ToggleButton()
      self.aux_select.add(image)
      self.aux_select.connect("toggled", self.cb_toggle, "Aux Open")
      self.hbox10.pack_start(self.aux_select, False, False, 0)
      self.aux_select.show()
      
      # add a microphone open button to do - guess what?
      pixbuf = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "mic5" + gfext)
      pixbuf = pixbuf.scale_simple(75, 20, gtk.gdk.INTERP_HYPER)
      image = gtk.Image()
      image.set_from_pixbuf(pixbuf)
      image.show()
      self.mic_select_box = gtk.HBox()
      self.mic_select = nice_mic_togglebutton()
      self.mic_select.add(image)
      self.mic_select_box.pack_start(self.mic_select, True, True, 0)
      self.mic_select.connect("toggled", self.cb_toggle, "Mic Open")
      self.mic_select.connect("button-press-event", self.mic_click)
      self.mic_select.show()
      
      pixbuf = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "mic4" + gfext)
      pixbuf1 = pixbuf.scale_simple(49, 21, gtk.gdk.INTERP_BILINEAR)
      pixbuf2 = pixbuf.scale_simple(49, 21, gtk.gdk.INTERP_BILINEAR)
      image1 = gtk.Image()
      image1.set_from_pixbuf(pixbuf1)
      image1.show()
      image2 = gtk.Image()
      image2.set_from_pixbuf(pixbuf2)
      image2.show()
      
      self.mic_lselect = gtk.ToggleButton()
      self.mic_lselect.add(image1)
      self.mic_lselect.connect("toggled", self.cb_toggle, "Mic L Open")
      self.mic_lselect.connect("button-press-event", self.mic_click)
      self.mic_select_box.pack_start(self.mic_lselect, True, True, 0)
      self.mic_rselect = gtk.ToggleButton()
      self.mic_rselect.add(image2)
      self.mic_rselect.connect("toggled", self.cb_toggle, "Mic R Open")
      self.mic_rselect.connect("button-press-event", self.mic_click)
      self.mic_select_box.pack_start(self.mic_rselect, True, True, 0)
      
      self.hbox10.pack_start(self.mic_select_box, True, True, 0)
      self.mic_select_box.show()
      
      pixbuf = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "advance" + gfext)
      pixbuf = pixbuf.scale_simple(32, 14, gtk.gdk.INTERP_BILINEAR)
      image = gtk.Image()
      image.set_from_pixbuf(pixbuf)
      self.advance = gtk.Button()
      self.advance.add(image)
      image.show()
      self.advance.connect("clicked", self.callback, "Advance")
      self.hbox10.pack_end(self.advance, False, False, 0)
      self.advance.show()
      
      # we are done messing with hbox7 so lets show it
      self.hbox7.show()
      # ditto
      self.hbox10.show()
      
      # Now to the interesting stuff - the mixing decks.

      self.hbox4 = gtk.HBox(False, 0)
      self.vbox6.pack_start(self.hbox4, True, True, 0)
      
      # Boxes 3L and 3R contain our "decks" basically media players
      self.vbox3L = gtk.VBox(False, 0)
      self.vbox3L.set_border_width(2)
      self.hbox4.pack_start(self.vbox3L, True, True, 0)

      # A vertical box for our main volume control
      self.vboxvol = gtk.VBox(False, 0)
      self.vboxvol.set_border_width(2)
      self.volframe = gtk.Frame()
      self.volframe.set_border_width(5)
      self.volframe.set_shadow_type(gtk.SHADOW_ETCHED_IN)
      self.volframe.add(self.vboxvol)
      self.volframe.show()
      self.hbox4.pack_start(self.volframe, False, True, 4)
           
      # A pictoral volume label
      image = gtk.Image()
      image.set_from_file(pkgdatadir + "volume2" + gfext)
      self.vboxvol.pack_start(image, False, False, 0)
      image.show()
      
      # The decks common volume controller.
      self.deckadj = gtk.Adjustment(0.0, 0.0, 100.0, 1.0, 6.0, 0.0)
      self.deckadj.connect("value_changed", self.cb_deckvol)
      self.deckvol = gtk.VScale(self.deckadj)
      self.deckvol.set_update_policy(gtk.UPDATE_CONTINUOUS)
      self.deckvol.set_value_pos(gtk.POS_RIGHT)
      self.deckvol.set_draw_value(False)
      self.vboxvol.pack_start(self.deckvol, True, True, 0)
      self.deckvol.show()
      
      self.spacerbox = gtk.VBox()
      self.spacerbox.set_size_request(1, 5)
      self.vboxvol.pack_start(self.spacerbox, False, False, 0)
       
      pixbuf = gtk.gdk.pixbuf_new_from_file(pkgdatadir + "pbphone" + gfext)
      pixbuf = pixbuf.scale_simple(20, 17, gtk.gdk.INTERP_HYPER)
      self.pbphoneimage = gtk.Image()
      self.pbphoneimage.set_from_pixbuf(pixbuf)
      self.vboxvol.pack_start(self.pbphoneimage, False, False, 0)
      
      self.mixbackadj = gtk.Adjustment(50.0, 0.0, 105.0, 1.0, 6.0, 5.0)
      self.mixbackadj.connect("value_changed", self.cb_deckvol)
      self.mixback = gtk.VScale(self.mixbackadj)
      self.mixback.set_update_policy(gtk.UPDATE_CONTINUOUS)
      self.mixback.set_digits(1)
      self.mixback.set_value_pos(gtk.POS_TOP)
      self.mixback.set_draw_value(False)
      self.vboxvol.pack_start(self.mixback, True, True, 0)
      
      self.vboxvol.show()
      
      # A box for the second deck.
      self.vbox3R = gtk.VBox(False, 0)
      self.vbox3R.set_border_width(2)
      self.hbox4.pack_start(self.vbox3R, True, True, 0)
    
      # hbox4 is full now so let's show it.
      self.hbox4.show()
      
      # The contents of the two player panes 3L and 3R are next up
      # The two identical players have been moved into one class
      
      self.player_left = IDJC_Media_Player(self.vbox3L, "left", self)
      self.vbox3L.show()
      
      self.player_right = IDJC_Media_Player(self.vbox3R, "right", self)
      self.vbox3R.show()
      
      # A track history window to help with announcements

      history_expander_hbox = gtk.HBox()
      self.history_expander = gtk.expander_new_with_mnemonic(tracks_played_text)
      history_expander_hbox.pack_start(self.history_expander, True, True, 6)
      self.history_expander.connect("notify::expanded", self.expandercallback)
      self.history_expander.show()
      self.vbox6.pack_start(history_expander_hbox, False, False, 0)
      history_expander_hbox.show()
      
      self.history_vbox = gtk.VBox()
      history_hbox = gtk.HBox()
      self.history_vbox.pack_start(history_hbox, True, True, 0)
      self.vbox6.pack_start(self.history_vbox, True, True, 0)
      #self.history_vbox.show()
      history_hbox.show()
      history_frame = gtk.Frame()
      history_hbox.pack_start(history_frame, True, True, 6)
      history_frame.show()
      history_frame.set_border_width(0)
      self.history_window = gtk.ScrolledWindow()
      history_frame.add(self.history_window)
      self.history_window.set_border_width(4)
      self.history_window.show()
      self.history_window.set_size_request(-1, 81)
      self.history_window.set_shadow_type(gtk.SHADOW_IN)
      self.history_window.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
      
      history_clear_box = gtk.HBox()
      self.history_clear = gtk.Button(" " + track_history_clear_text + " ")
      self.history_clear.connect("clicked", self.callback, "Clear History") 
      history_clear_box.pack_start(self.history_clear, True, False, 0)
      self.history_clear.show()
      self.history_vbox.pack_start(history_clear_box, False, False, 1)
      #history_clear_box.show()
      
      spacer = gtk.VBox()
      self.history_vbox.pack_start(spacer, False, False, 1)
      spacer.show()
      
      self.history_textview = gtk.TextView()
      self.history_textview.connect("populate-popup", self.cb_history_populate)
      self.history_window.add(self.history_textview)
      self.history_textview.show()
      self.history_textview.set_cursor_visible(False)
      self.history_textview.set_editable(False)
      self.history_textview.set_wrap_mode(gtk.WRAP_CHAR)
      self.history_buffer = self.history_textview.get_buffer()
      
      self.abox = gtk.HBox()		# If there is a better way of making a spacer I would like to know
      self.abox.set_border_width(2)
      self.vbox6.pack_start(self.abox, False, False, 0)
      self.abox.show()
      
      # The crossfader.  No DJ should be without one. ;)
      self.outercrossbox = gtk.HBox()
      crossframe = gtk.Frame()
      self.outercrossbox.pack_start(crossframe, True, True, 6)
      self.outercrossbox.show()
      crossframe.set_border_width(0)
      self.crossbox = gtk.HBox()
      crossframe.add(self.crossbox)
      crossframe.show()
      self.crossbox.set_border_width(2)
      self.crossbox.set_spacing(3)
      
      cross_sizegroup = gtk.SizeGroup(gtk.SIZE_GROUP_HORIZONTAL)
            
      self.listen = gtk.ToggleButton(stream_monitor_text)
      cross_sizegroup.add_widget(self.listen)
      self.listen.connect("toggled", self.cb_toggle, "stream-mon")
      
      self.crossbox.pack_start(self.listen, False, False, 0)
      self.listen.show()
      
      self.passleft = make_arrow_button(self, gtk.ARROW_LEFT, gtk.SHADOW_NONE, "cfleft")
      self.crossbox.pack_start(self.passleft, False, False, 0)
      self.passleft.show()
      
      self.crossadj = gtk.Adjustment(0.0, 0.0, 100.0, 1.0, 3.0, 0.0)
      self.crossadj.connect("value_changed", self.cb_crossfade)      
      self.crossfade = gtk.HScale(self.crossadj)
      self.crossfade.set_update_policy(gtk.UPDATE_CONTINUOUS)
      self.crossfade.set_draw_value(False)
      self.crossbox.pack_start(self.crossfade, True, True, 0)
      self.crossfade.show()
      self.vbox6.pack_start(self.outercrossbox, False, False, 2)

      self.passright = make_arrow_button(self, gtk.ARROW_RIGHT, gtk.SHADOW_NONE, "cfright")
      self.crossbox.pack_start(self.passright, False, False, 0)
      self.passright.show()
      
      passbox = gtk.HBox()
      passbox.set_spacing(2)
      
      self.passspeed_adj = gtk.Adjustment(1.0, 0.25, 8.0, 0.25, 0.25, 0.25)
      self.passspeed = gtk.SpinButton(self.passspeed_adj, 0, 2)
      passbox.pack_start(self.passspeed, False, False, 0)
      self.passspeed.show()
      
      image = gtk.Image()
      image.set_from_file(pkgdatadir + "pass" + gfext)
      image.show()
      self.passbutton = gtk.Button()
      self.passbutton.set_size_request(53, -1)
      self.passbutton.add(image)
      self.passbutton.connect("clicked", self.callback, "pass-crossfader")
      passbox.pack_start(self.passbutton, True, True, 0)
      self.passbutton.show()
      
      self.crossbox.pack_start(passbox, False, False, 0)
      cross_sizegroup.add_widget(passbox)
      passbox.show()
      self.crossbox.show()
      
      abox = gtk.HBox()
      abox.set_border_width(1)
      self.vbox6.pack_start(abox, False, False, 0)
      abox.show()
      
      # We are done with vbox6 so lets show it
      self.vbox6.show()
      
      # Pause here for a reply from the mixer.
      self.last_chance = False
      rply = jackrply = self.mixer_read()
      while rply[:6] != "IDJC: ":
         if rply == "":
            print "mixer crashed"
            message_dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE, "The mixer module crashed during initialisation.\nThe most likely cause is a corrupt xine config file.\nDeleting the xine config files should solve the problem.\ne.g. rm -fdr ~/.xine/")
            message_dialog.set_icon_from_file(pkgdatadir + "icon" + gfext)
            message_dialog.run()
            message_dialog.destroy()
            raise self.initfailed()
         rply = self.mixer_read()
         if rply[:6] == "JACK: ":
            jackrply = rply
      if rply[:17] == "IDJC: Sample rate":
         self.samplerate = rply[18:].strip()
         self.mixer_write("ACTN=sync\nend\n", True)
         self.mixer_read()
      else:
         if jackrply == "JACK: could not attach as client (duplicate client name?)\n":
            message_dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE, "At present only one instance of idjc can be running per user.\n\nIf idjc was running and it crashed it is possible that part of the program is still running.  Try the following in a console before attempting to run it again:\n\n   			$ killall idjcmixer idjcserver")
         else:
            errormessage = "The JACK sound server needs to be running in order to run IDJC.\nIn order to manually start it try something like:\n\n		$ jackd -d alsa -r 44100 -p 2048\n\nIf you would like JACK to start automatically when you start IDJC try this:\n\n\techo \"/usr/bin/jackd -d alsa -r 44100\" >~/.jackdrc\n\n" + "If you have already done this it is possible another application or non-JACK sound server is using the sound card.\n\nPossible remedies would be to close the other audio app or configure the sound server to go into suspend mode after a brief amount of idle time"
            if rply.strip() !="" and rply.strip() != "IDJC: Error":
               errormessage = errormessage + "\nThe error message reported by JACK is:\n\n" + rply
            message_dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE, errormessage)
         message_dialog.set_icon_from_file(pkgdatadir + "icon" + gfext)
         message_dialog.run()
         message_dialog.destroy()
         raise self.initfailed()
      
      self.songname = u""
      self.newmetadata = False
      self.server_window = server_window(self)
      self.server_window.load(self.idjc)
      self.jingles = jingles(self)
            
      # Set variables
      self.showing_left_file_requester = False
      self.showing_right_file_requester = False
      self.metadata = ""
      self.simplemixer = False
      self.crosspass = 0
      self.alarm = False
      self.NO_PHONE = 0
      self.PUBLIC_PHONE = 1
      self.PRIVATE_PHONE = 2
      self.mixermode = self.NO_PHONE
      self.jingles_playing = int_object(0)
      self.interlude_playing = int_object(0)
      self.player_left.playtime_elapsed = int_object(0)
      self.player_right.playtime_elapsed = int_object(0)
      self.player_left.mixer_playing = int_object(0)
      self.player_right.mixer_playing = int_object(0)
      self.player_left.mixer_signal_f = int_object(0)
      self.player_right.mixer_signal_f = int_object(0)
      self.player_left.mixer_cid = int_object(0)
      self.player_right.mixer_cid = int_object(0)
      self.jingles.mixer_jingles_cid = int_object(0)
      self.jingles.mixer_interlude_cid = int_object(0)
      self.player_left.runout = int_object(0)
      self.player_right.runout = int_object(0)
      self.fullwinx = int_object(1)
      self.fullwiny = int_object(1)
      self.minwinx = int_object(1)
      self.minwiny = int_object(1)
      self.in_vu_timeout = False
      self.vucounter = 0
      self.session_filename = self.idjc + "main_session"

      self.meterbox = gtk.HBox()
      self.hbox9.pack_start(self.meterbox, False, False, 0)
      self.meterbox.show()
      peakbox = gtk.VBox()
      self.meterbox.add(peakbox)
      peakbox.show()
      self.vubox = gtk.VBox()
      self.meterbox.add(self.vubox)
      self.vubox.show()
      
      self.str_l_peak = peakholdmeter()      
      self.str_r_peak = peakholdmeter() 
      self.stream_peak_box = make_meter_unit(str_peak_text, self.str_l_peak, self.str_r_peak)
      peakbox.pack_start(self.stream_peak_box)
      self.stream_peak_box.show()
                  
      self.str_l_rms_vu = vumeter()
      self.str_r_rms_vu = vumeter()
      self.stream_vu_box = make_meter_unit(str_vu_text, self.str_l_rms_vu, self.str_r_rms_vu)
      self.vubox.pack_start(self.stream_vu_box)
      self.stream_vu_box.show()
       
      self.mic_l_peak = peakholdmeter()      
      self.mic_r_peak = peakholdmeter() 
      self.mic_peak_box = make_meter_unit(mic_peak_text, self.mic_l_peak, self.mic_r_peak)     
      peakbox.pack_start(self.mic_peak_box)

      self.mic_peak_box.show()
                  
      self.mic_l_rms_vu = vumeter()
      self.mic_r_rms_vu = vumeter()
      self.mic_vu_box = make_meter_unit(mic_vu_text, self.mic_l_rms_vu, self.mic_r_rms_vu)
      self.vubox.pack_start(self.mic_vu_box)
      self.mic_vu_box.show()
      
      self.limiter_box = gtk.VBox()
      self.limiter_box.set_border_width(3)
      label = gtk.Label(comp_bar_text)
      self.limiter_box.pack_start(label, False, False, 0)
      label.show()
      frame = gtk.Frame()
      hbox = gtk.HBox()
      hbox.set_border_width(1)
      frame.add(hbox)
      frame.set_border_width(4)
      self.limiter_box.pack_start(frame, True, True, 0)
      frame.show()
      self.left_compressor_gain = limitermeter()
      self.left_compressor_gain.set_size_request(15, -1)
      hbox.pack_start(self.left_compressor_gain, False, False, 0)
      self.left_compressor_gain.show()
      self.limiter_scale = make_limiter_scale()
      hbox.pack_start(self.limiter_scale, False, False, 0)
      self.limiter_scale.show()
      self.right_compressor_gain = limitermeter()
      self.right_compressor_gain.set_size_request(16, -1)
      hbox.pack_start(self.right_compressor_gain, False, False, 0)
      self.right_compressor_gain.show()
      hbox.show()
      self.meterbox.pack_start(self.limiter_box, False, False, 0)
      self.limiter_box.show()
                  
      # Originally designed to get the stats for the meters from the mixer, it now pulls
      # out much more info than that now.
      self.vumap = {
         "mic_l_peak"	: self.mic_l_peak,
	 "mic_r_peak"	: self.mic_r_peak,
	 "mic_l_rms"	: self.mic_l_rms_vu,
	 "mic_r_rms"	: self.mic_r_rms_vu,
	 "str_l_peak"	: self.str_l_peak,
	 "str_r_peak"	: self.str_r_peak,
	 "str_l_rms"	: self.str_l_rms_vu,
	 "str_r_rms"	: self.str_r_rms_vu,
	 "left_compressor_gain"  : self.left_compressor_gain,
	 "right_compressor_gain" : self.right_compressor_gain,
         "jingles_playing"	 : self.jingles_playing,
         "left_elapsed"		 : self.player_left.playtime_elapsed,
         "right_elapsed"	 : self.player_right.playtime_elapsed,
         "left_playing"		 : self.player_left.mixer_playing,
         "right_playing"	 : self.player_right.mixer_playing,
         "interlude_playing"	 : self.interlude_playing,
         "left_signal"		 : self.player_left.mixer_signal_f,
         "right_signal"		 : self.player_right.mixer_signal_f,
         "left_cid"		 : self.player_left.mixer_cid,
         "right_cid"		 : self.player_right.mixer_cid,
         "jingles_cid"		 : self.jingles.mixer_jingles_cid,
         "interlude_cid"	 : self.jingles.mixer_interlude_cid,
         "left_audio_runout"	 : self.player_left.runout,
         "right_audio_runout"	 : self.player_right.runout
	 }

      self.prefs_window = mixprefs(self)
      self.prefs_window.load_player_prefs()
      self.prefs_window.apply_player_prefs()
      self.vutimeout = gobject.timeout_add(50, self.vu_update)
      self.statstimeout = gobject.timeout_add(500, self.stats_update)

      signal.signal(signal.SIGINT, self.destroy)
      if not self.simplemixer:
         self.window.resize(self.fullwinx, self.fullwiny)
      else:
         self.window.resize(self.minwinx, self.minwiny)
      self.window.connect("configure_event", self.configure_event)
      self.jingles.window.resize(self.jingles.jingleswinx, self.jingles.jingleswiny)

      if self.prefs_window.restore_session_option.get_active():
         print "Restoring previous session"
         self.player_left.restore_session()
         self.player_right.restore_session()
         self.restore_session()
 
      self.window.show()
   
   def main(self):
      gtk.main()

def environment_variables_okay():
   # Shows a dialog box when one or more of the helper programs is missing
   df = "/bin/true"		# default program to use in place of the missing
   environvars = (
      ("metaflac from the flac package needed to play flac files\n", ("metaflac", "missing")),
      ("lame needed for the encoding of mp3 streams\n", ("lame", df)),
      ("oggenc needed for the encoding of ogg/vorbis\n", ("oggenc", df)),
      ("ogginfo needed for ogg track length and tags\n", ("ogginfo", "missing")),
      ("jackd from the jack-audio-connection-kit package\n", ("jackd", "missing")),
      ("vorbiscomment from the vorbistools package\n", ("vorbiscomment", "missing")) )
   error_message = "The following programs were found to be missing...\n\n"
   for each in environvars:
      if os.environ.get(each[1][0], each[1][1]) == each[1][1]:
         error_message = error_message + each[0]	# append the appropriate error message
         os.environ[each[1][0]] = each[1][1]		# set the actual environment variable to the chosen default
   if error_message[-2:] != "\n\n":
      error_message = error_message + "\n\t\tDo you want to continue?"
      message_dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_INFO, gtk.BUTTONS_YES_NO, error_message)
      message_dialog.set_icon_from_file(pkgdatadir + "icon" + gfext)
      message_dialog.set_default_response(gtk.RESPONSE_YES)
      if message_dialog.run() == gtk.RESPONSE_NO:
         message_dialog.destroy()
         return False
      else:
         message_dialog.destroy()
   return True

# Let's run this...
if __name__ == "__main__":
   gtk.gdk.threads_init()
   gtk.gdk.threads_enter()
   retcode = 0
   if environment_variables_okay() == True:
      try:
         run_instance = MainWindow(sys.argv[1:])
      except MainWindow.initfailed:
         retcode = 5
      except MainWindow.initcleanexit:
         pass
      else:
         try:
            run_instance.main()
         except KeyboardInterrupt:
            pass
   gtk.gdk.threads_leave()
   sys.exit(retcode)
