/*
#   mp4decode.c: decodes mp4 file format for xlplayer
#   Copyright (C) 2007 Stephen Fairchild (s-fairchild@users.sourceforge.net)
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program in the file entitled COPYING.
#   If not, see <http://www.gnu.org/licenses/>.
*/

#include "../config.h"
#ifdef HAVE_FAAD

#include <stdio.h>
#include <stdlib.h>
#include <mp4ff.h>
#include <faad.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "xlplayer.h"
#include "mp4decode.h"

#define TRUE 1
#define FALSE 0
#define ACCEPTED 1
#define REJECTED 0

static uint32_t cbmp4_read(void *user_data, void *buffer, uint32_t length)
   {
   struct mp4decode_vars *self = user_data;

   return (uint32_t)read(self->mp4filehandle, buffer, length);
   }

static uint32_t cbmp4_seek(void *user_data, uint64_t position)
   {
   struct mp4decode_vars *self = user_data;
   
   return lseek(self->mp4filehandle, position, SEEK_SET);
   }

static uint32_t cbmp4_write(void *user_data, void *buffer, uint32_t length)
   {
   return 0;
   }

static uint32_t cbmp4_truncate(void *user_data)
   {
   return 0;
   }

static void mp4decode_eject(struct xlplayer *xlplayer)
   {
   struct mp4decode_vars *self = xlplayer->dec_data;
   
   faacDecClose(self->faachandle);
   mp4ff_close(self->mp4handle);
   close(self->mp4filehandle);
   if (self->mp4buffer)
      free(self->mp4buffer);
   if (self->mp4flbuf)
      free(self->mp4flbuf);
   if (self->faacresample_f)
      {
      if (xlplayer->src_data.data_out)
         free(xlplayer->src_data.data_out);
      xlplayer->src_state = src_delete(xlplayer->src_state);
      }
   free(self);
   }

static void mp4decode_init(struct xlplayer *xlplayer)
   {
   struct mp4decode_vars *self = xlplayer->dec_data;
   int src_error;
   
   self->mp4buffer = NULL;
   self->mp4flbuf = NULL;
   if ((self->faacresample_f = self->faacsamplerate != xlplayer->samplerate))
      {
      fprintf(stderr, "Configuring resampler\n");
      xlplayer->src_data.output_frames = 0;
      xlplayer->src_data.data_out = NULL;
      xlplayer->src_data.src_ratio = (double)xlplayer->samplerate / (double)self->faacsamplerate;
      xlplayer->src_data.end_of_input = 0;
      xlplayer->src_state = src_new(xlplayer->rsqual, self->faacnumchannels, &src_error);
      if (src_error)
         {
         fprintf(stderr, "mp4decode_init: src_new reports %s\n", src_strerror(src_error));
         self->faacresample_f = 0;
         mp4decode_eject(xlplayer);
         xlplayer->playmode = PM_STOPPED;
         xlplayer->command = CMD_COMPLETE;
         return;
         }
      }
   if (xlplayer->seek_s)
      {
      self->mp4_sample_number = (int)((double)xlplayer->seek_s * self->faacsamplerate / mp4ff_get_sample_duration(self->mp4handle, 0, 0) + 0.5);
      }
   else
      self->mp4_sample_number = 0;
   }
   
static void mp4decode_play(struct xlplayer *xlplayer)
   {
   struct mp4decode_vars *self = xlplayer->dec_data;
   int32_t buffersize;
   void *data;
   faacDecFrameInfo frame_info;
   SRC_DATA *src_data = &(xlplayer->src_data);
   
   if (self->mp4_sample_number >= self->mp4_numsamples || (buffersize = mp4ff_read_sample_getsize(self->mp4handle, 0, self->mp4_sample_number)) == 0)
      goto finished;
   if (!(self->mp4buffer = realloc(self->mp4buffer, buffersize)))
      {
      fprintf(stderr, "mp4decode_play: malloc failure\n");
      goto finished;
      }
   if(!mp4ff_read_sample_v2(self->mp4handle, 0, self->mp4_sample_number++, self->mp4buffer))
      {
      fprintf(stderr, "mp4decode_play: read_sample_v2 returned zero indicating an error\n");
      goto finished;
      }
   if (!(data = faacDecDecode(self->faachandle, &frame_info, self->mp4buffer, buffersize)))
      {
      fprintf(stderr, "mp4decode_play: possible corrupt media file\n");
      goto finished;
      }
   self->mp4flbuf = realloc(self->mp4flbuf, frame_info.samples * sizeof (float));
   xlplayer_make_audio_to_float(xlplayer, self->mp4flbuf, data, frame_info.samples / frame_info.channels, 16, frame_info.channels); 
   if (self->faacresample_f)
      {
      src_data->end_of_input = (frame_info.samples == 0);
      src_data->input_frames = frame_info.samples / frame_info.channels;
      src_data->data_in = self->mp4flbuf;
      src_data->output_frames = (int)(src_data->input_frames * src_data->src_ratio) + 2 + (512 * src_data->end_of_input);
      src_data->data_out = realloc(src_data->data_out, src_data->output_frames * frame_info.channels * sizeof (float));
      if (src_process(xlplayer->src_state, src_data))
         {
         fprintf(stderr, "mp4decode_play: error occured during resampling\n");
         goto finished;
         }
      xlplayer_demux_channel_data(xlplayer, src_data->data_out, src_data->output_frames_gen, frame_info.channels, 1.f);
      }
   else
      xlplayer_demux_channel_data(xlplayer, self->mp4flbuf, frame_info.samples / frame_info.channels, frame_info.channels, 1.f);
   xlplayer_write_channel_data(xlplayer);
   return;
finished:
   xlplayer->playmode = PM_EJECTING;
   return;
   }

int mp4decode_reg(struct xlplayer *xlplayer)
   {
   struct mp4decode_vars *self;
   unsigned char *ppBuf;
   unsigned int pBufSize;
   
   if(!(self = xlplayer->dec_data = malloc(sizeof (struct mp4decode_vars))))
      {
      fprintf(stderr, "mp4decode_reg: malloc failure\n");
      return REJECTED;
      }
   if(!(self->mp4filehandle = open(xlplayer->pathname, O_RDONLY)))
      {
      fprintf(stderr, "mp4decode_reg: could not open file %s\n", xlplayer->pathname);
      return REJECTED;
      }
   self->mp4cbstruct.read = cbmp4_read;
   self->mp4cbstruct.write = cbmp4_write;
   self->mp4cbstruct.seek = cbmp4_seek;
   self->mp4cbstruct.truncate = cbmp4_truncate;
   self->mp4cbstruct.user_data = self;
   if (!(self->mp4handle = mp4ff_open_read(&(self->mp4cbstruct))))
      {
      close(self->mp4filehandle);
      return REJECTED;
      }
   if (mp4ff_total_tracks(self->mp4handle) != 1)
      goto error;
   if (mp4ff_get_audio_type(self->mp4handle, 0) != 64)
      goto error;
   if(!(self->mp4_numsamples = mp4ff_num_samples(self->mp4handle, 0)))
      goto error;
   mp4ff_get_decoder_config(self->mp4handle, 0, &ppBuf, &pBufSize);
   if (!(self->faachandle = faacDecOpen()))
      {
      fprintf(stderr, "mp4decode_reg: failed to initialise faac decoder\n");
      goto error;
      }
   faacDecInit2(self->faachandle, ppBuf, pBufSize, &(self->faacsamplerate), &(self->faacnumchannels));
   xlplayer->dec_init = mp4decode_init;
   xlplayer->dec_play = mp4decode_play;
   xlplayer->dec_eject = mp4decode_eject;
   return ACCEPTED;
error:
   mp4ff_close(self->mp4handle);
   close(self->mp4filehandle);
   return REJECTED;
   }

#endif /* HAVE_FAAD */
